package com.bokesoft.distro.tech.bootsupport.starter.utils;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.bokesoft.distro.tech.bootsupport.starter.datasource.DataSourceConfig;
import com.bokesoft.distro.tech.bootsupport.starter.deployment.SpringResourceMultiSolutionMetaResolverFactory;
import com.bokesoft.distro.tech.commons.basis.jdbc.DBTypeUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.unbescape.properties.PropertiesEscape;
import org.unbescape.xml.XmlEscape;

import com.bokesoft.distro.tech.bootsupport.starter.api.YigoRawConfiger;
import com.bokesoft.distro.tech.bootsupport.starter.api.ctx.ConfigerContext;
import com.bokesoft.distro.tech.bootsupport.starter.config.YigoCoreConfig;
import com.bokesoft.distro.tech.commons.basis.StringTemplate;
/**
 * 用于实现 Yigo 启动阶段的配置文件的准备
 */
public class YigoPropPreparationHelper {
	private static final Logger log = LoggerFactory.getLogger(YigoPropPreparationHelper.class);

	/** properties 文件需要进行特别处理(主要是避免 "\"、非 ISO8859 字符等) */
	private static final String FILE_EXT_PROPERTIES = ".properties";
	/** xml 文件需要进行特别处理(XML encoding) */
	private static final String FILE_EXT_XML = ".xml";
	/** resource 文件类行的url匹配规则 */
	private static final Pattern FILE_PATH_PATTERN=Pattern.compile("(^file://)(\\w+/)?(\\w+:)(/.*)?");

	private static ValueProvider valueProvider = null;
	private static List<String> pathVariableList = new ArrayList<>();
	private static final String SYSTEM_TYPE=System.getProperty("os.name").toLowerCase();

	/**
	 * 设置参数值的提供者
	 *
	 * @param valueProvider 赋值接口实现
	 */
	public static void setValueProvider(ValueProvider valueProvider) {
		YigoPropPreparationHelper.valueProvider = valueProvider;
	}

	/**
	 * 设置 path 类型的变量名列表, 具体对这些变量的值的处理参见 {@link #formatPath(String)}
	 *
	 * @param pathVariableList path 类型的变量名列表
	 */
	public static void setPathVariableList(List<String> pathVariableList) {
		YigoPropPreparationHelper.pathVariableList = pathVariableList;
	}

	/**
	 * 替换运行参数
	 *
	 * @param loader         Spring {@link ResourceLoader}
	 * @param resources      资源列表
	 * @param configers      额外的配置文件修改组件
	 * @param yigoCoreConfig yml参数定义的{@link YigoCoreConfig}配置,主要提供core.properties设置
	 * @param dataSourceConfig yml参数定义的{@link DataSourceConfig}配置,主要提供数据库连接设置
	 * @return 存放处理后的资源文件的临时目录
	 * @throws IOException
	 */
	public static PrepareResult prepare(ResourceLoader loader, List<String> resources, List<YigoRawConfiger> configers,
								 YigoCoreConfig yigoCoreConfig, DataSourceConfig dataSourceConfig)throws  IOException{

		final Map<String, String> filesToWrite = readSpringResources(loader, resources);
		//静默启动不需要准备core.properties和dsn.properties相关配置
		if (isSilentStart(filesToWrite,yigoCoreConfig)) {
			String workPath = prepare(loader, filesToWrite);
			return new PrepareResult(workPath, true);
		}
		String corePropertiesStr = filesToWrite.get("core.properties");
		if (StringUtils.isBlank(corePropertiesStr)) {
			//创建默认core.properties
			buildDefaultCoreProperties(filesToWrite, yigoCoreConfig);
		}
		ConfigerContext context = new ConfigerContext() {
			@Override
			public String getFile(String fileName) {
				return filesToWrite.get(fileName);
			}
		};
		addCoreConfigSolutions(context, yigoCoreConfig.getSolutions(), filesToWrite);
		for (YigoRawConfiger configer : configers) {
			configer.prepare(context);
			filesToWrite.putAll(context.getAdditionalFiles());
		}
		correctDsnProperties(loader, filesToWrite, yigoCoreConfig, dataSourceConfig);

		String workPath = prepare(loader, filesToWrite);
		return new PrepareResult(workPath, filesToWrite, false);
	}

	private static boolean isSilentStart(Map<String, String> filesToWrite, YigoCoreConfig yigoCoreConfig) {
		String corePropertiesStr = filesToWrite.get("core.properties");
		if (StringUtils.isBlank(corePropertiesStr)) {
			if(null == yigoCoreConfig.getSolutions() || yigoCoreConfig.getSolutions().isEmpty()){
				return true;
			}
		}
		return false;
	}

	/**
	 * 添加 solutions  默认第一个是主 solution
	 * @param context 配置上下文
	 */
	private static void addCoreConfigSolutions(ConfigerContext context, List<String> slnList
			,Map<String, String> filesToWriteMap ) {
		if(CollectionUtils.isNotEmpty(slnList)){
			for (int number = 0; number < slnList.size(); number++) {
				String slnRes = unifyFileResourceUrl(slnList.get(number));
				String slnName = "sln-" + number;
				context.addSolution(slnName, SpringResourceMultiSolutionMetaResolverFactory.class,
						SpringResourceMultiSolutionMetaResolverFactory.buildParas(slnRes));
			}
		}
		// ConfigerContext.addSolution()的修改结果传递到当前栈内存中
		filesToWriteMap.putAll(context.getAdditionalFiles());
	}

	/**
	 * 设置 默认初始化的CoreProperties
	 * 默认的core.properties 中 没有solution 需要后续方法添加
	 * @param filesToWrite 内存缓存
	 * @param yigoCoreConfig yml参数定义的{@link YigoCoreConfig}配置,主要提供core.properties设置
	 * @throws IOException
	 */
    private static void buildDefaultCoreProperties(Map<String, String> filesToWrite, YigoCoreConfig yigoCoreConfig) throws IOException {
            Properties coreProps = new Properties();
            coreProps.setProperty("CACHE", yigoCoreConfig.getCache());
            coreProps.setProperty("KICK", yigoCoreConfig.getKick());
            coreProps.setProperty("DSN", yigoCoreConfig.getDsn());
            coreProps.setProperty("LOGSVR", yigoCoreConfig.getLogsvr());
            coreProps.setProperty("DEBUG", yigoCoreConfig.getDebug());
		    coreProps.setProperty("ExternalBaseUrl",yigoCoreConfig.getExternalBaseUrl());
			coreProps.setProperty("_CREATED_BY",YigoPropPreparationHelper.class.getName());
			try(StringWriter sw=new StringWriter()){
				coreProps.store(sw,"Created by "+YigoPropPreparationHelper.class.getName());
				filesToWrite.put("core.properties",sw.toString());
			}

    }

	/**
	 * *
	 * @param loader 资源加载器
	 * @param filesToWrite 内存缓存map
	 * @param yigoCoreConfig yml参数定义的{@link YigoCoreConfig}配置,主要提供dbfactory设置
	 * @param dataSourceConfig yml参数定义的{@link DataSourceConfig}配置,主要提供数据库连接设置
	 * @throws IOException
	 */
	private static void correctDsnProperties(ResourceLoader loader, Map<String, String> filesToWrite,
			YigoCoreConfig yigoCoreConfig,DataSourceConfig dataSourceConfig) throws IOException {
		String corePropertiesStr = filesToWrite.get("core.properties");
		corePropertiesStr = processTmpl(loader,"core.properties",corePropertiesStr,valueProvider);
		Properties coreProps = new Properties();
		try(StringReader sw=new StringReader(corePropertiesStr)){
			coreProps.load(sw);
			String dsnFile = coreProps.getProperty("DSN");
			String dsnFileContent = filesToWrite.get(dsnFile+".properties");
			if(StringUtils.isBlank(dsnFileContent)){
				buildDefaultDsnProperties(filesToWrite, dsnFile, yigoCoreConfig, dataSourceConfig);
			}
		}
	}

	/**
	 * * 根据yml参数创建默认DSN.properties
	 * @param filesToWrite 内存缓存map
	 * @param dnsFileName 对应DSN.properties的文件名
	 * @param yigoCoreConfig yml参数定义的{@link YigoCoreConfig}配置,主要提供dbfactory设置等
	 * @param dataSourceConfig yml参数定义的{@link DataSourceConfig}配置,主要提供数据库连接设置
	 * @throws IOException
	 */
	private static void buildDefaultDsnProperties(Map<String, String> filesToWrite,
		String dnsFileName, YigoCoreConfig yigoCoreConfig, DataSourceConfig dataSourceConfig) throws IOException{
		Properties dsnProps = new Properties();
		String url = dataSourceConfig.getUrl();
		setDsnDriver(url,dsnProps, dataSourceConfig);
		//处理 dsn.properties 赋值
		dsnProps.setProperty("Name", dnsFileName);
		dsnProps.setProperty("ConnectionType",yigoCoreConfig.getConnectionType());
		dsnProps.setProperty("URL", url);
		dsnProps.setProperty("User", dataSourceConfig.getUsername());
		dsnProps.setProperty("Password", dataSourceConfig.getPassword());
		dsnProps.setProperty("DBFactory",yigoCoreConfig.getDbFactory());
		try(StringWriter sw=new StringWriter()){
			dsnProps.store(sw,"Created by "+YigoPropPreparationHelper.class.getName());
			filesToWrite.put(dnsFileName+".properties",sw.toString());
		}
	}

	/**
	 * 替换运行参数
	 *
	 * @param filesToWrite key:文件名, value:文件内容字符串, 其中内容包含可被替换的变量
	 * @return 存放处理后的资源文件的临时目录
	 * @throws IOException
	 */
	private static String prepare(ResourceLoader loader, Map<String, String> filesToWrite) throws IOException {
		File tmpPath = buildYigoProperties(loader, filesToWrite);
		return tmpPath.getCanonicalPath();
	}

	private static Map<String, String> readSpringResources(ResourceLoader loader, List<String> resources)
			throws IOException {
		Map<String, String> filesToWrite = new HashMap<>();
		for (String resource : resources) {
			resource = unifyFileResourceUrl(resource);
			Resource res = loader.getResource(resource);
			if (null != res && res.exists()) {
				String fileName = res.getFilename();
				if (null == fileName) {
					throw new UnsupportedOperationException("不支持的资源 '" + resource + "' - 无法得到 file name");
				}
				String text = IOUtils.toString(res.getInputStream(), StandardCharsets.UTF_8);
				filesToWrite.put(fileName, text);
			} else {
				throw new IOException("无法加载资源 '" + resource + "' - 文件不存在");
			}
		}
		return filesToWrite;
	}

	private static File buildYigoProperties(ResourceLoader loader, Map<String, String> filesToWrite) throws IOException {
		File tmpPath = createTempClasspath();
		for (Map.Entry<String, String> en : filesToWrite.entrySet()) {
			String fileName = en.getKey();
			
			String content = processTmpl(loader, fileName, en.getValue(), valueProvider);
			filesToWrite.put(fileName, content);

			File f = new File(tmpPath, fileName);
			FileUtils.writeStringToFile(f, content, "UTF-8");
		}
		return tmpPath;
	}

	private static void setDsnDriver(String url, Properties dsnProps, DataSourceConfig yigoDsConfig){
		dsnProps.setProperty("DBType", DBTypeUtils.getDBType(url));
		dsnProps.setProperty("Driver", DBTypeUtils.getDBDriver(url));
	}

	private static File createTempClasspath() throws IOException {
		File tmpPath = Files
				.createTempDirectory(
						YigoPropPreparationHelper.class.getName() + "-" + Long.toString(System.currentTimeMillis()))
				.toFile();
		log.info("Yigo 属性设置文件临时存放目录: " + tmpPath);
		return tmpPath;
	}

	private static String processTmpl(ResourceLoader loader, String fileName, String tmplString, ValueProvider valueProvider)
			throws IOException {
		if (null == tmplString) {
			return "";
		}

		if (null == valueProvider) {
			return tmplString;
		}

		StringTemplate st = new StringTemplate(tmplString, StringTemplate.REGEX_PATTERN_JAVA_STYLE);
		String[] vars = st.getVariables();
		for (int i = 0; i < vars.length; i++) {
			String var = vars[i];
			String value;
			//在资源文件中出现了${include xxx}标记时根据,xxx的resource-path获取文本内容,然后整个文本替换标记
			if (var.indexOf("include") == 0) {
				value = includeAsContent(loader, var.substring(8));
			} else {
				value = valueProvider.getValue(var);
				if (pathVariableList.contains(var)) {
					value = formatPath(value);
				}
				if (fileName.endsWith(FILE_EXT_PROPERTIES)) {
					value = formatProperies(value);
				} else if (fileName.endsWith(FILE_EXT_XML)) {
					value = formatXml(value);
				}
			}
			st.setVariable(var, value);
		}

		return st.getParseResult();
	}

	/**
	 * 在资源文件中出现了${include xxx}标记时根据,xxx的resource-path获取文本内容,然后整个文本替换标记
	 *
	 * @param resourcePath 资源路径
	 * @throws IOException
	 */
	private static String includeAsContent(ResourceLoader loader, String resourcePath) throws IOException {
		String result = null;
		Resource res = loader.getResource(resourcePath);
		if (null != res && res.exists()) {
			if (null == res.getFilename()) {
				throw new UnsupportedOperationException("不支持的资源 '" + resourcePath + "' - 无法得到 file name");
			}
			String text = IOUtils.toString(res.getInputStream(), StandardCharsets.UTF_8);
			result = processTmpl(loader, res.getFilename(), text, valueProvider);
		} else {
			throw new IOException("无法加载资源 '" + resourcePath + "' - 文件不存在");
		}
		return result;
	}

	private static String formatPath(String pathValue) {
		if (StringUtils.isBlank(pathValue)) {
			return pathValue;
		}

		// 尝试当作一个文件/目录进行处理
		try {
			File f = new File(pathValue);
			Path p = Paths.get(f.toURI());
			pathValue = p.normalize().toString();
		} catch (Exception ex) {
			// Ignore it
		}
		pathValue = unifyFileResourceUrl(pathValue);
		return pathValue;
	}

	static String unifyFileResourceUrl(String pathValue) {
		//如果是file协议进行下一步斜杠纠正
		if(pathValue.startsWith("file://")) {
			// 为了保证 Windows 路径在 properties 中正常生效，"\" 需要替换为 "/"
			pathValue = pathValue.replaceAll("\\\\", "/");
			// 如果是windows环境.则准备补位“/”
			boolean isWin = SYSTEM_TYPE.startsWith("windows");
			Matcher matcher = FILE_PATH_PATTERN.matcher(pathValue);
			if (isWin && matcher.matches()) {
				if (matcher.group(2) == null) {
					pathValue = matcher.group(1) + "/" + matcher.group(3) + (matcher.group(4) == null ? "" : matcher.group(4));
				}
			}
			return pathValue;
		}
		return pathValue;
	}

	private static String formatProperies(String value) {
		if (StringUtils.isBlank(value)) {
			return value;
		}
		value = PropertiesEscape.escapePropertiesValue(value);
		return value;
	}

	private static String formatXml(String value) {
		if (StringUtils.isBlank(value)) {
			return value;
		}
		value = XmlEscape.escapeXml11(value);
		return value;
	}

	public static interface ValueProvider {
		public String getValue(String variable);
	}

	public static class PrepareResult {
		private String workPath;
		private boolean silentStart;
		private Map<String, String> filesToWrite;

		public PrepareResult(String workPath, boolean silentStart) {
			this(workPath, null, silentStart);
		}

		public PrepareResult(String workPath, Map<String, String> filesToWrite, boolean silentStart) {
			this.workPath = workPath;
			this.filesToWrite = filesToWrite;
			this.silentStart = silentStart;
		}

		public String getWorkPath() {
			return workPath;
		}

		public boolean isSilentStart() {
			return silentStart;
		}

		public Map<String, String> getFilesToWrite() {
			return filesToWrite;
		}
	}
}
