package com.bokesoft.distro.tech.bootsupport.starter.utils;

import com.bokesoft.distro.tech.bootsupport.starter.wrapper.WrappedConnection;
import com.bokesoft.distro.tech.bootsupport.starter.wrapper.WrappedDBManager;
import com.bokesoft.distro.tech.commons.basis.MiscUtil;
import com.bokesoft.distro.tech.commons.basis.flightrecord.utils.YFRUtils;
import com.bokesoft.yes.mid.connection.ConnectionProfile;
import com.bokesoft.yes.mid.connection.dbmanager.*;
import com.bokesoft.yigo.mid.connection.DBType;
import com.bokesoft.yigo.mid.connection.IConnectionProfile;
import com.bokesoft.yigo.mid.connection.IDBManager;
import org.slf4j.event.Level;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.UUID;

public class DataSourceAwareDBManagerBuilder {
	/**
	 *  dbManager的获取
	 * @param dbType 数据库类型
	 * @param dataSource 数据源
	 * @param logLevel 日志等级
	 * @param dbTraceCacheEnabled Trace开关
	 * @param execTimeoutControlEnabled  超时开关
	 * @return dbManager
	 * @throws Throwable
	 */
	public static IDBManager getDBManager(int  dbType,DataSource dataSource,Level logLevel,
	              boolean dbTraceCacheEnabled, boolean execTimeoutControlEnabled ) throws Throwable{
		return getDBManager(buildConnectionProfile(dbType),dataSource,logLevel,dbTraceCacheEnabled,execTimeoutControlEnabled);
	}

	/**
	 *  dbManager的获取
	 * @param connectionProfile yigo 数据库连接信息
	 * @param dataSource 数据源
	 * @param logLevel 日志等级
	 * @param dbTraceCacheEnabled Trace开关
	 * @param execTimeoutControlEnabled  超时开关
	 * @return dbManager
	 * @throws Throwable
	 */
	public static IDBManager getDBManager(IConnectionProfile connectionProfile,DataSource dataSource,Level logLevel,
	                                      boolean dbTraceCacheEnabled, boolean execTimeoutControlEnabled ) throws Throwable {
		String instanceId = UUID.randomUUID().toString();
		Connection connection = getConnection(instanceId, dataSource, dbTraceCacheEnabled, execTimeoutControlEnabled);
		IDBManager dbManager = getDBManager(connectionProfile, connection);
		dbManager.setKey(connectionProfile.getKey());
		WrappedDBManager wdbm = new WrappedDBManager(dbManager, instanceId, execTimeoutControlEnabled);
		wdbm.setLogLevel(logLevel);
		//FIXME 暂时没有配置项用于设置 IConnectionRecoveryFactory
		//wdbm.setRecoveryFactory(???);
		return wdbm;
	}

	/**
	 * 获取connection
	 * @param instanceId 数据库实例id
	 * @param datasource 数据源
	 * @param dbTraceCacheEnabled trace
	 * @param execTimeoutControlEnabled timeout
	 * @return conn
	 * @throws SQLException 数据库连接异常
	 */
	private static Connection getConnection(String instanceId,DataSource datasource, boolean dbTraceCacheEnabled, boolean execTimeoutControlEnabled) throws SQLException {
		Connection conn = null;
		MiscUtil.$assert(null == datasource, "未配置有效的数据源,系统无法使用 ");
		conn = new WrappedConnection(YFRUtils.getDatasourceId(datasource), datasource.getConnection(), instanceId, dbTraceCacheEnabled, execTimeoutControlEnabled);
		try {
			conn.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
		} catch (Exception ex) {
			//如果是 SQLite 就忽略这个错误: SQLite supports only TRANSACTION_SERIALIZABLE and TRANSACTION_READ_UNCOMMITTED.
			conn.setTransactionIsolation(Connection.TRANSACTION_READ_UNCOMMITTED);
		}
		return conn;
	}


	/**
	 * 依据connectionfile Connection (+dbType)创建DBManager
	 * @param connectionProfile 数据库连接信息
	 * @param conn 数据库连接
	 * @return
	 * @throws SQLException
	 */
	private static IDBManager getDBManager(IConnectionProfile connectionProfile,Connection conn) throws Throwable {
		conn.setAutoCommit(false);
		IDBManager dbm = null;
		int dbType = -1;
		if(connectionProfile!=null&&connectionProfile.getDBType()!=0){
			dbType = connectionProfile.getDBType();
		}
		switch (dbType) {
			case DBType.SqlServer:
				dbm = new SqlServerDBManager(connectionProfile, conn);
				break;
			case DBType.Oracle:
				dbm = new OracleDBManager(connectionProfile, conn);
				break;
			case DBType.DB2:
				dbm = new DB2DBManager(connectionProfile, conn);
				break;
			case DBType.MySql:
				dbm = new MySqlDBManager(connectionProfile, conn);
				break;
			case DBType.MySqlCluster:
				dbm = new MySqlClusterDBManager(connectionProfile, conn);
				break;
			case DBType.Sqlite:
				dbm = new SqliteDBManager(connectionProfile, conn);
				break;
			case DBType.DM:
				dbm = new DMDBManager(connectionProfile, conn);
				break;
			case DBType.MyCat:
				dbm = new MyCatDBManager(connectionProfile, conn);
				break;
			case DBType.HighGo:
				dbm = new HighGoDBManager(connectionProfile, conn);
				break;
			case DBType.GaussDB:
				dbm = new GaussDBManager(connectionProfile, conn);
				break;
			case DBType.Kingbase:
				dbm = new KingbaseManager(connectionProfile, conn);
				break;
			case DBType.EsgynDB:
				dbm = new EsgynDBManager(connectionProfile, conn);
				break;
			case DBType.ShenTong:
				dbm = new ShenTongDBManager(connectionProfile, conn);
				break;
			case DBType.PostgreSQL:
				dbm = new PostgreSQLDBManager(connectionProfile, conn);
				break;
			default:
				throw new UnsupportedOperationException("DBType:" + dbType + ", no match! Unsupport db connection!");
		}
		dbm.setKey(connectionProfile.getKey());
		return dbm;
	}

	private static IConnectionProfile buildConnectionProfile(int dbType){
		ConnectionProfile cp =new ConnectionProfile();
		cp.setDBType(dbType);
		cp.setKey("DBTYPE_AUTO_"+dbType);
		return cp;
	}

}
