package com.bokesoft.distro.tech.bootsupport.starter.filter;

import com.bokesoft.distro.tech.bootsupport.starter.api.YigoExtendJavascriptProvider;
import jodd.util.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.web.util.ContentCachingResponseWrapper;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;


/**
 * 用于处理 Yigo 的扩展 js (/project/extend.js).<br/>
 * 考虑以 "分组" 的方式注册不同模块的 js 文件, 分组可以定义先后依赖顺序
 */
public class YigoProjectExtendJSFilter implements Filter{

    private static final Logger logger = LoggerFactory.getLogger(YigoProjectExtendJSFilter.class);

    private final List<YigoExtendJavascriptProvider> providers;

    public YigoProjectExtendJSFilter(List<YigoExtendJavascriptProvider> providers) {
        this.providers = providers;
    }

    @Override
    public void doFilter(ServletRequest req, ServletResponse resp, FilterChain chain) throws IOException, ServletException {
        if(null != providers && !providers.isEmpty()) {
            PatchTo200StatusContentResponseWrapper response = new PatchTo200StatusContentResponseWrapper(new ContentCachingResponseWrapper((HttpServletResponse) resp));
            IgnoreAcceptEncodingServletRequestWrapper request = new IgnoreAcceptEncodingServletRequestWrapper((HttpServletRequest)req);
            //设置 UTF-8 编码
            response.setCharacterEncoding("UTF-8");
            response.setContentType("application/x-javascript;charset=UTF-8");//禁用缓存
            response.setHeader("Cache-Control", "no-cache, no-store, must-revalidate"); // HTTP 1.1.
            response.setHeader("Pragma", "no-cache"); // HTTP 1.0.
            response.setDateHeader("Expires", 0); // Proxies.
            chain.doFilter(request, response);
            // 原extendJs 文件内容
            String originalJS;
            if (response.getRealStatus() == HttpServletResponse.SC_OK) {
                originalJS = new String(response.getContentAsByteArray(), "utf-8");
            }else {
                logger.warn("请求系统原生extend.js失败, 原生extend.js将被忽略。");
                originalJS = "";
            }
            //清除原extent.js文件内容
            response.resetBuffer();
            //扩展模块 js 文件内容
            String js = buildExtendJS();
            PrintWriter printWriter = response.getWriter();
            printWriter.write(js);
            printWriter.write(System.lineSeparator());
            printWriter.write(System.lineSeparator());
            printWriter.write(wrapOriginalJS(originalJS));
            response.copyBodyToResponse();
        }else{
            chain.doFilter(req, resp);
        }
    }

    private String wrapOriginalJS(String originalJS) {
        return new StringBuilder("/** \n"
                        +" * Build from static resource: yigo/project/extend.js\n"
                        +" * Created by: " + this.getClass().getSimpleName() + "(" + Base64.encodeToString(this.getClass().getName()) + ") \n"
                        +" */")
                .append("\n")
                .append("(function(){\n")
                .append(originalJS)
                .append("\n})();").toString();
    }

    private String buildExtendJS() throws IOException {
        List<String> parts = new ArrayList<>();
        parts.add("/** \n"
                +" * Created by: " + this.getClass().getSimpleName() + "(" + Base64.encodeToString(this.getClass().getName()) + ") \n"
                +" */");

        for(YigoExtendJavascriptProvider p: providers) {
            List<Resource> resources = p.getResources();
            if (resources == null || resources.isEmpty()) {
                continue;
            }
            for(Resource r: resources) {
                String head = "/** \n * " + r.getFilename() + "(" + Base64.encodeToString(r.getURL().toString())+")\n */";
                try(InputStream is = r.getInputStream()){
                    String js = IOUtils.toString(is, "UTF-8");
                    parts.add(head + '\n' + "(function(){\n" + js + "\n})();");
                }
            }
        }

        return StringUtils.join(parts, "\n\n");
    }

    static class IgnoreAcceptEncodingServletRequestWrapper extends HttpServletRequestWrapper {

        public IgnoreAcceptEncodingServletRequestWrapper(HttpServletRequest request) {
            super(request);
        }

        @Override
        public String getHeader(String name) {
            //设置请求接收的编码类型 主要是屏蔽GzipFilter 带来的影响
            //屏蔽到可能包含的 gzip
            if ("accept-encoding".equals(name)) {
                return null;
            }
            return super.getHeader(name);
        }

    }

    static class PatchTo200StatusContentResponseWrapper extends HttpServletResponseWrapper {

        private final ContentCachingResponseWrapper responseWrapper;

        private int realStatus = -1;

        public PatchTo200StatusContentResponseWrapper(ContentCachingResponseWrapper response) {
            super(response);
            responseWrapper = response;
        }

        @Override
        public int getStatus() {
            return SC_OK;
        }

        @Override
        public void sendError(int sc){
            realStatus = sc;
        }

        @Override
        public void sendError(int sc, String msg){
            realStatus = sc;
            // 强制作为200的相应，此处不需要错误信息 忽略 msg
        }

        @Override
        public void setStatus(int sc) {
            realStatus = sc;
        }

        public int getRealStatus(){
            return realStatus == -1 ? responseWrapper.getStatus() : realStatus;
        }

        public byte[] getContentAsByteArray(){
            return responseWrapper.getContentAsByteArray();
        }

        public void copyBodyToResponse() throws IOException {
            responseWrapper.copyBodyToResponse();
        }


    }

}
