package com.bokesoft.distro.tech.bootsupport.starter.deployment;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ResourceLoader;
import org.springframework.stereotype.Component;

import com.bokesoft.distro.tech.bootsupport.starter.api.YigoRawConfiger;
import com.bokesoft.distro.tech.bootsupport.starter.api.ctx.ConfigerContext;
import com.bokesoft.distro.tech.bootsupport.starter.config.YigoModuleConfig;
import com.bokesoft.distro.tech.yigosupport.deployment.resource.AbstractResourceMultiSolutionMetaResolverFactory;
import com.bokesoft.distro.tech.yigosupport.deployment.resource.intf.IResourceIO;

/**
 * 基于 Spring 资源加载的 MetaResolverFactory
 */
@Component
public class SpringResourceMultiSolutionMetaResolverFactory extends AbstractResourceMultiSolutionMetaResolverFactory {
    private static final Logger log = LoggerFactory.getLogger(SpringResourceMultiSolutionMetaResolverFactory.class);

    public static final String PARA_KEY_ROOT_RESOURCE = "ROOT_RESOURCE";
	public static final String PARA_KEY_IGNORE_RESOURCES = "IGNORE_RESOURCES";
	
	/** Spring resource loader, 用于支持以 resource 的方式 (classpath://... 、file://... 、 http://... ) 加载资源 */
	private static ResourceLoader RESOURCE_LOADER;
	/** 来自 {@link YigoModuleConfig#getIgnoreSolutionResources()}, 用于处理全局配置中的忽略资源定义 */
	private static List<String> IGNORE_SOLUTION_RESOURCES;

	/** 资源根路径 */
	private String rootResourcePath;
	/** 需要忽略的资源路径(相对路径) */
	private List<String> ignoreResourcePaths;

	@Override
	public void initParas(Map<String, Object> paras) {
		this.rootResourcePath = (String)paras.get(PARA_KEY_ROOT_RESOURCE);
        if (StringUtils.isBlank(rootResourcePath)){
            throw new IllegalArgumentException(
                "Initial parameter invalid: '"+PARA_KEY_ROOT_RESOURCE+"' is not defined.");
        }

		String ignoresStr = (String) paras.get(PARA_KEY_IGNORE_RESOURCES);
		if (StringUtils.isNotBlank(ignoresStr)) {
			this.ignoreResourcePaths = Arrays.asList(StringUtils.split(ignoresStr, ","));
		}

		super.initParas(paras);
	}

    @Autowired
    public void setResourceLoader(ResourceLoader resourceLoader) {
    	RESOURCE_LOADER = resourceLoader;
    }

	@Autowired
	public void setYigoModuleConfig(YigoModuleConfig yigoModuleConfig){
		IGNORE_SOLUTION_RESOURCES = yigoModuleConfig.getIgnoreSolutionResources();
        if (null!=IGNORE_SOLUTION_RESOURCES && (! IGNORE_SOLUTION_RESOURCES.isEmpty())){
            log.warn("已定义全局 Yigo Solution 资源路径忽略通配列表: 来自 '{}' 配置项 , 包括如下 Path Wildcard 模式: '{}'",
                    YigoModuleConfig.PROP_PREFIX, IGNORE_SOLUTION_RESOURCES);
        }
	}
    
	@Override
	protected IResourceIO getResourceIO() {
        return new SpringResourceIO(
            rootResourcePath, RESOURCE_LOADER, ignoreResourcePaths, IGNORE_SOLUTION_RESOURCES);
	}

	/**
	 * 构造 {@link YigoRawConfiger#prepare(ConfigerContext)} 执行
	 * {@link ConfigerContext#addSolution(String, Class, Map)} 的 属性参数 .
	 * 是{@link SpringResourceMultiSolutionMetaResolverFactory#buildParas(String, List)}的重载方法,补充参数ignoreResourcePaths默认为null
	 * @param rootResourcePath 要加载的solution的资源根路径
	 * @return
	 */
	public static final Map<String, String> buildParas(String rootResourcePath){
		return buildParas(rootResourcePath, (List<String>) null);
	}

	/**
	 * 构造 {@link YigoRawConfiger#prepare(ConfigerContext)} 执行
	 * {@link ConfigerContext#addSolution(String, Class, Map)} 的 属性参数.
	 * @param rootResourcePath 要加载的solution的资源根路径
	 * @param ignoreResourcePaths 资源忽略路径的列表,支持通配符匹配，例如：/BK_Basic/Entry.xml、/&ast;/Form/&ast;&ast;.xml
	 * @return
	 */
	public static final Map<String, String> buildParas(String rootResourcePath,List<String> ignoreResourcePaths){
		Map<String, String> result = new HashMap<String, String>();
		result.put(PARA_KEY_ROOT_RESOURCE, rootResourcePath);
		if (null != ignoreResourcePaths && ignoreResourcePaths.size() != 0) {
			result.put(PARA_KEY_IGNORE_RESOURCES, String.join(",", ignoreResourcePaths));
		}
		return result;
	}
    
}