package com.bokesoft.yes.webcollections.erpbase;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.reflect.MethodUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.bokesoft.erp.filter.ProcessErrorFilter;
import com.bokesoft.erp.function.ERPMidFunctionCluster;
import com.bokesoft.erp.para.ParaDefines;
import com.bokesoft.erp.view.CreateViewByXml;
import com.bokesoft.yes.erp.config.ERPMetaFactory;
import com.bokesoft.yes.erp.lock.ERPBusinessLockService;
import com.bokesoft.yes.mid.base.MidGlobalEnv;
import com.bokesoft.yes.mid.cmd.ERPRowExpandService;
import com.bokesoft.yes.mid.cmd.ERPService;
import com.bokesoft.yes.mid.cmd.richdocument.strut.DictionaryService;
import com.bokesoft.yes.mid.cmd.richdocument.strut.OrgDictService;
import com.bokesoft.yes.mid.cmd.richdocument.strut.RichDocumentService;
import com.bokesoft.yes.mid.cmd.richdocument.strut.RichServiceFilterImpl;
import com.bokesoft.yes.mid.module.IModulePlugin;
import com.bokesoft.yes.mid.service.filter.ServiceFilterFactory;
import com.bokesoft.yes.mid.web.services.HeadInfoFieldService;
import com.bokesoft.yes.mid.web.services.OperatorService;
import com.bokesoft.yigo.meta.enhance.MetaEnhance;
import com.bokesoft.yigo.meta.enhance.MetaExtService;
import com.bokesoft.yigo.meta.enhance.MetaService;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.factory.MetaFactory;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.mid.parser.util.MidFunctionUtil;
import com.bokesoft.yigo.mid.service.IExtService;
import com.bokesoft.yigo.mid.service.IServiceFilter;
import com.bokesoft.yigo.mid.service.provider.ServiceProviderFactory;
import com.bokesoft.yigo.mid.util.ContextBuilder;
import com.bokesoft.yigo.tools.wildcard.WildcardUtil;

/**
 * ERP 服务初始化.
 * 参考 bokeerp/erp-backend/erp-backend-base com.bokesoft.erp.all.initiator.ERPServiceInitiator 构建.
 */
public class ERPServiceInitiator implements IModulePlugin{
    private static Logger logger = LoggerFactory.getLogger(ERPServiceInitiator.class);

    public ERPServiceInitiator(){
        DefaultContext context = null;
        try {
            context = ContextBuilder.create();

            logger.info("{} - 开始初始化过程 ...", ERPServiceInitiator.class);
            init(context);
            logger.info("{} - 初始化完成 .", ERPServiceInitiator.class);

            context.commit();
        } catch (Throwable ex) {
            logger.error(ERPServiceInitiator.class + " - 初始化错误: " + ex.getMessage(), ex);
            if (null != context) {
                try {
                    context.rollback();
                } catch (Throwable e) {
                    //Ignore it
                }
            }
            ExceptionUtils.rethrow(ex);
        } finally {
            if (null != context) {
                try {
                    context.close();
                } catch (Throwable e) {
                    //Ignore it
                }
            }
        }
    }

    private void init(DefaultContext context) throws Throwable {
        initERPMetaFactory(context);

        MidFunctionUtil midFunctionUtil = new MidFunctionUtil();
        midFunctionUtil.regFunctionProvider(new ERPMidFunctionCluster());

        ServiceProviderFactory.registerProvider(this.getERPServices());
        ServcieFilterProviderFactory.registerProvider(this.getERPServiceFilters());

        //FIXME: 这里要有个ExtService的注册机制, 先采用修改solutionEnhance对象的方式处理
        IMetaFactory metaFactory = context.getVE().getMetaFactory();
        MetaEnhance enhance = metaFactory.getEnhance(null);
        if (enhance != null) {
            MetaExtService metaExtService = enhance.getMetaExtService();
            if (metaExtService == null) {
                metaExtService = new MetaExtService();
                enhance.setMetaExtService(metaExtService);
            }
            addExtService(metaExtService, ERPRowExpandService.class.getSimpleName(), ERPRowExpandService.class);
            addExtService(metaExtService, HeadInfoFieldService.class.getSimpleName(), HeadInfoFieldService.class);
        }

        //FIXME [PATCH] 怀疑 yes 和 erp 的前端代码对 paras 的处理不同
        //    导致 RichDocumentContext#checkScope(paraKey) 中 ParaDefines.instance.containsKey(paraKey) 检查失败
        ParaDefines.setParaCheck(false);

        //FIXME 暂时简单的加入 CreateView 机制
        logger.info("开始处理 CreateView.xml ...");
        CreateViewByXml createViewByXml = new CreateViewByXml();
        createViewByXml.invoke(context);
        logger.info("完成处理 CreateView.xml .");
    }

    private void initERPMetaFactory(DefaultContext context) {
        IMetaFactory oldMetaFactory = context.getVE().getMetaFactory();
        ERPMetaFactory erpMetaFactory = new ERPMetaFactory(oldMetaFactory);
        MetaFactory.setGlobalInstance(erpMetaFactory);
        MidGlobalEnv.getInstance().setMetaFactory(erpMetaFactory);
    }

    private <T extends IExtService> void addExtService(MetaExtService metaExtService, String serviceName,
        Class<T> serviceClass) {
        if (metaExtService.containsKey(serviceName)) {
            return;
        }
        String impl = serviceClass.getName();
        MetaService service = new MetaService();
        service.setName(serviceName);
        service.setImpl(impl);
        metaExtService.add(service);
    }

    private Object[][] getERPServices() {
        Object[][] result = new Object[][] {{RichDocumentService.ServiceName, new RichDocumentService()},
            {OrgDictService.ServiceName, new OrgDictService()},
            {DictionaryService.ServiceName, new DictionaryService()},
            {ERPBusinessLockService.ServiceName, new ERPBusinessLockService()},
            {ERPService.ServiceName, new ERPService()}, {"OperatorService", new OperatorService()},
            //{"AutoRecordService", new AutoRecordService()}
        };

        return fillExtServices(result);
    }

    private Object[][] fillExtServices(Object[][] services){
        List<Object[]> serviceList = new ArrayList<>(Arrays.asList(services));

        String className = "com.bokesoft.erp.record.cmd.AutoRecordService";
        try {
            Class<?> clazz = Class.forName(className);
            Object inst = clazz.newInstance();
            String serviceName = (String) MethodUtils.invokeMethod(inst, true, "getServiceName");
            serviceList.add(new Object[]{serviceName, inst});
        } catch (ReflectiveOperationException e) {
            logger.warn("忽略加载 "+className+", 原因: "+e.getMessage(), e);
        }
        
        return serviceList.toArray(new Object[0][0]);
    }

    private List<ServiceFilterItem> getERPServiceFilters() {
        //FIXME 这些 Service Filter 的作用需要进一步讨论和拆分
        return new ArrayList<ServiceFilterItem>() {{
            add(new ServiceFilterItem("*", ProcessErrorFilter.class));
            add(new ServiceFilterItem("RichDocument/*", RichServiceFilterImpl.class));
//            add(new ServiceFilterItem("*", AutoRecordFilter.class));
            // add(new ServiceFilterItem("RichDocument/*", MessageFilter.class));
            // add(new ServiceFilterItem("OrgDictService/*", MessageFilter.class));
            // add(new ServiceFilterItem("ERPBusinessLock/*", MessageFilter.class));
            // add(new ServiceFilterItem("DictionaryService/*", MessageFilter.class));
            // add(new ServiceFilterItem("BPM/*", MessageFilter.class));
            // add(new ServiceFilterItem("DTS/*", MessageFilter.class));
            // add(new ServiceFilterItem("InvokeService/*", MessageFilter.class));
            // add(new ServiceFilterItem("InvokeUnsafe/InvokeExtServicWe2/ERPWebService", MessageFilter.class));
        }};
    }

    private static class ServcieFilterProviderFactory {
        private static Map<String, List<Class<? extends IServiceFilter>>> filters = new ConcurrentHashMap<>();

        static {
            ServiceFilterFactory.getInstance().setMatcher((service, arguments) -> {
                Set<Map.Entry<String, List<Class<? extends IServiceFilter>>>> entries = null;
                entries = ServcieFilterProviderFactory.filters.entrySet();
                List<IServiceFilter> filters = new ArrayList<>();
                for (Map.Entry<String, List<Class<? extends IServiceFilter>>> entry : entries) {
                    String key = entry.getKey();
                    if (!WildcardUtil.isMatch(service, key)) {
                        continue;
                    }
                    List<Class<? extends IServiceFilter>> filterClasses = entry.getValue();
                    for (Class<? extends IServiceFilter> filterClass : filterClasses) {
                        filters.add(filterClass.newInstance());
                    }
                }
                return filters;
            });
        }

        public static void registerProvider(List<ServiceFilterItem> items) {
            if (items == null) {
                return;
            }
            for (ServiceFilterItem item : items) {
                String name = item.getName();
                Class<? extends IServiceFilter> filterClass = item.getFilterClass();
                filters.computeIfAbsent(name, key -> new ArrayList<>()).add(filterClass);
            }
        }
    }

    private static class ServiceFilterItem{
        private final String name;
        private final Class<? extends IServiceFilter> filterClass;

        public ServiceFilterItem(String name, Class<? extends IServiceFilter> filterClass) {
            this.name = name;
            this.filterClass = filterClass;
        }

        public String getName() {
            return name;
        }

        public Class<? extends IServiceFilter> getFilterClass() {
            return filterClass;
        }
    }

}
