package com.bokesoft.yes.excel.cmd.normal;

import java.io.ByteArrayOutputStream;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import com.bokesoft.yes.excel.document.ExcelDataTable;
import com.bokesoft.yes.excel.document.ExcelDetailDataRow;
import com.bokesoft.yes.excel.document.ExcelDocumentProcess;
import com.bokesoft.yes.excel.document.ExcelExpandDataRow;
import com.bokesoft.yes.excel.document.ExcelGroupDetailRow;
import com.bokesoft.yes.excel.document.GroupContext;
import com.bokesoft.yes.excel.document.IExcelDataRow;
import com.bokesoft.yes.excel.parser.ExcelContext;
import com.bokesoft.yes.excel.template.ColumnExpandItem;
import com.bokesoft.yes.excel.template.ExcelCell;
import com.bokesoft.yes.excel.template.ExcelRow;
import com.bokesoft.yes.excel.template.ExcelSheet;
import com.bokesoft.yes.excel.template.ExcelWorkbook;
import com.bokesoft.yes.excel.template.util.ExcelUtil;
import com.bokesoft.yes.excel.transformer.ExcelTransformerFactory;
import com.bokesoft.yes.excel.transformer.IExcelTransformer;
import com.bokesoft.yes.meta.i18n.StringTable;
import com.bokesoft.yes.struct.filedata.FileData;
import com.bokesoft.yigo.common.def.ScriptType;
import com.bokesoft.yigo.common.def.TableMode;
import com.bokesoft.yigo.common.util.FileUtil;
import com.bokesoft.yigo.common.util.SimpleStringFormat;
import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.meta.base.MetaException;
import com.bokesoft.yigo.meta.dataobject.MetaColumn;
import com.bokesoft.yigo.meta.dataobject.MetaDataObject;
import com.bokesoft.yigo.meta.dataobject.MetaTable;
import com.bokesoft.yigo.meta.dataobject.MetaTableCollection;
import com.bokesoft.yigo.meta.exceltemplate.ExcelTemplateRowType;
import com.bokesoft.yigo.meta.exceltemplate.MetaExcelWorkbook;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.form.MetaForm;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.struct.condition.ConditionParas;
import com.bokesoft.yigo.struct.datatable.DataTable;
import com.bokesoft.yigo.struct.datatable.DataTableMetaData;
import com.bokesoft.yigo.struct.document.Document;
import com.bokesoft.yigo.struct.document.FilterMap;
import com.bokesoft.yigo.util.ExcelUtils;
import com.bokesoft.yigo.util.StampExcelUtil;

public class ExportExcelWithMultiPageTemplate implements IExport {
	/** 上下文环境*/
	private DefaultContext context = null;
	/** Excel导出上下文环境 */
	private ExcelContext excelContext = null;
	/** 要导出的数据集 */
	private Document document = null;
	/** Excel对象*/
	private Workbook workBook = null;
	/** Excel模板配置对象*/
	private MetaExcelWorkbook metaWorkbook = null;
	/** Excel虚拟对象*/
	private ExcelWorkbook virtualWorkbook = null;
	/** 值类型转换工具类*/
	private ExcelTransformerFactory transformerFactory = null;
	/** Excel单元格格式工具类*/
	private ExcelStyleFactory styleFactory = null;
	/** 重新加载数据是的过滤条件*/
	private FilterMap filterMap = null;
	/** 重新加载数据是的过滤条件*/
	private ConditionParas condParameters = null;
	/** 导出后扩展服务名*/
	private String postExportServiceName = null;
	/** 自定义导出后的文件名(不包含后缀)*/
	private String exportFileName=null;
	
	/**
	 * 
	 * @param context 上下文环境
	 * @param metaWorkbook Excel模板配置对象
	 * @param filterMap 载入数据所需的OID条件
	 * @param condParameters 载入数据所需的查询条件
	 * @throws Throwable
	 */
	public ExportExcelWithMultiPageTemplate(DefaultContext context, MetaExcelWorkbook metaWorkbook, FilterMap filterMap,
			ConditionParas condParameters, String postExportServiceName, String exportFileName) throws Throwable {
		this.context = context;
		this.metaWorkbook = metaWorkbook;
		
		this.filterMap = filterMap;
		this.condParameters = condParameters;
		this.postExportServiceName = postExportServiceName;
		this.exportFileName=exportFileName;
		this.excelContext = new ExcelContext(context);
	}
	
	/**
	 * 导出数据并返回FileData
	 * @return FileData 
	 */
	@Override
	public FileData exportData() throws Throwable {
		String formKey = metaWorkbook.getFormKey();
		MetaForm metaForm = context.getVE().getMetaFactory().getMetaForm(formKey);
		MetaDataObject metaDataObject = metaForm.getDataSource().getDataObject();
		
		LoadMultiPageDocument loadMultiPageDocument = new LoadMultiPageDocument(context, filterMap, condParameters);
		document = loadMultiPageDocument.reloadDocument(metaForm);

		// 将cache中的数据一次性取出放到document中
		processDataTableToDocument();

		context.setDocument(document);
		context.setFormKey(formKey);
		context.setDataObject(metaDataObject);
		excelContext.setDocument(document);

		// 获取所有的表头数据源，定位第一行
		for (MetaTable metaTable : metaDataObject.getTableCollection()) {
			if (metaTable.getTableMode() == TableMode.HEAD) {
				document.get(metaTable.getKey()).first();
			}
		}
		
		// 通过配置文件生成Excel模型
		virtualWorkbook = new ExcelWorkbook(metaDataObject, metaWorkbook, context.getMidParser());
		virtualWorkbook.calcExpandColumn(context.getMidParser(), document, metaDataObject);

		// 处理明细数据集,扩展和汇总
		ExcelDocumentProcess excelDocumentProcess = new ExcelDocumentProcess(metaDataObject, virtualWorkbook, document,
				excelContext);
		excelDocumentProcess.process();

		workBook = new SXSSFWorkbook(500);
		transformerFactory = new ExcelTransformerFactory();
		styleFactory = new ExcelStyleFactory(workBook);
		HashMap<Integer, Integer> columnWidths = null;
		
		ExcelSheet excelSheet = null;
		ExcelRow excelRow = null;

		Sheet sheet = null;
		int rowType = -1;
		
		String fileName = "";
		ByteArrayOutputStream bos = null;
		
		Iterator<ExcelSheet> itExeclSheet = virtualWorkbook.iterator();
		while (itExeclSheet.hasNext()) {
			columnWidths = new HashMap<Integer, Integer>();
			excelSheet = itExeclSheet.next();
			sheet = ExcelUtil.getSheet(workBook, excelSheet.getName());

			int offset = 0;
			Iterator<ExcelRow> itExeclRow = excelSheet.iterator();
			while (itExeclRow.hasNext()) {
				excelRow = itExeclRow.next();
				rowType = excelRow.getType();
				switch (rowType) {
				case ExcelTemplateRowType.Head:
				case ExcelTemplateRowType.Fix:
				case ExcelTemplateRowType.DetailHead:
					exportFixRow(document, metaDataObject, sheet, excelRow, offset,columnWidths);
					break;
				case ExcelTemplateRowType.Detail:
					// 导出明细数据，连带汇总数据一起
					offset = exportDetailRow(excelDocumentProcess, document, metaDataObject, sheet, excelRow, offset,
							columnWidths);
					break;
				}
			}
			HashMap<Integer, Integer> templateColumnWidths = excelSheet.getColumnWidths();
			ExcelUtil.calcFinalColumnWith(templateColumnWidths,columnWidths);
			ExcelUtil.setSheetColumnWidth(sheet, columnWidths);
		}
		
		IMetaFactory metaFactory = context.getVE().getMetaFactory();
		
		String filePath = ExcelUtils.getExportFilePath(metaFactory, formKey);
		if(exportFileName!=null && !exportFileName.isEmpty()){
			fileName = exportFileName + "." + FileUtil.getSuffixName(filePath);
		}else{
			fileName = metaForm.getKey() + "." + FileUtil.getSuffixName(filePath);
		}
		try{
			bos = new ByteArrayOutputStream();
			workBook.write(bos);
			FileData fileData = new FileData();
			fileData.setData(bos.toByteArray());
			fileData.setFileName(fileName);
			return fileData;
		}finally {
			if(bos != null){
				bos.close();
			}
		}
	}

	private void processDataTableToDocument() {
		MetaTableCollection tableCollection = document.getMetaDataObject().getTableCollection();
		String tableKey = "";
		DataTable dataTable = null;
		DataTable tempDataTable = null;

		DataTableMetaData metaData = null;
		int columnCount = 0;

		for (MetaTable table : tableCollection) {
			tableKey = table.getKey();
			dataTable = document.get(tableKey);
			metaData = dataTable.getMetaData();
			columnCount = metaData.getColumnCount();

			tempDataTable = new DataTable(metaData);
			dataTable.beforeFirst();
			while (dataTable.next(true)) {
				tempDataTable.append();
				for (int index = 0; index < columnCount; index++) {
					tempDataTable.setObject(index, dataTable.getObject(index));
				}
			}
			tempDataTable.first();
			document.put(tableKey, tempDataTable);
		}
	}

	/**
	 * 导出明细数据 并返回行偏移量
	 * 
	 * @param document
	 * @param metaForm
	 * @param sheet
	 * @param excelRow
	 * @return
	 * @throws Throwable
	 */
	private int exportDetailRow(ExcelDocumentProcess process, Document document, MetaDataObject metaDataObject,
			Sheet sheet, ExcelRow excelRow, int offset, HashMap<Integer, Integer> columnWidths) throws Throwable {
		excelContext.setExcelRow(excelRow);
		String tableKey = excelRow.getTableKey();
		ExcelDataTable excelDataTable = process.getExcelDataTable(tableKey);
		if (excelDataTable == null) {
			// gridKey对应的表不存在
			return offset;
		}
		
		Row row = null;
		int startRowIndex = excelRow.getRowIndex() - 1;
		
		DataTable dataTable = document.get(tableKey);
		MetaTable metaTable = metaDataObject.getMetaTable(tableKey);
		excelContext.setDataTable(dataTable);
		excelContext.setMetaTable(metaTable);
		Iterator<IExcelDataRow> itExcelDataRows = excelDataTable.iterator();
		int rowCount = 0;
		// 记录动态合并单元格所需要的上下文
		HashMap<Integer, GroupContext> groupContextMap = new HashMap<Integer, GroupContext>();
		int groupContextCount = 0;
		while (itExcelDataRows.hasNext()) {
			int rowIndex = startRowIndex + offset;
			row = ExcelUtil.getRow(sheet, rowIndex);
			row.setHeightInPoints((short) excelRow.getHeight());

			IExcelDataRow excelDataRow = itExcelDataRows.next();
			String type = excelDataRow.getType();
			if ("Expand".equalsIgnoreCase(type)) {
				//TODO　分页报表不允许导出扩展列,走到这里说明模版配置错误
				fillExpandData(row, (ExcelExpandDataRow) excelDataRow, dataTable, excelRow, metaTable, columnWidths);
			} else if ("Detail".equalsIgnoreCase(type)) {
				fillDetailData(sheet, rowIndex, (ExcelDetailDataRow) excelDataRow, dataTable, excelRow, metaTable,
						columnWidths, groupContextMap, groupContextCount);
			} else if ("GroupDetail".equalsIgnoreCase(type)) {
				fillGroupData(sheet, rowIndex, (ExcelGroupDetailRow) excelDataRow, dataTable, excelRow, metaTable,
						columnWidths, groupContextMap, groupContextCount);
			}

			offset++;
			rowCount++;
			groupContextCount++;
		}

		groupContextMap.clear();

		if (rowCount > 0) {
			offset -= 1;
		}
		return offset;
	}

	private void fillGroupData(Sheet sheet, int rowIndex, ExcelGroupDetailRow groupDetailDataRow, DataTable dataTable,
			ExcelRow excelRow, MetaTable metaTable, HashMap<Integer, Integer> columnWidths,
			HashMap<Integer, GroupContext> groupContextMap, int groupContextCount) throws Throwable {
		Iterator<ExcelCell> itExcelCell = null;
		ExcelCell excelCell = null;

		GroupContext groupContext = new GroupContext();
		groupContext.addDefinition(groupDetailDataRow.getGroupLevel(), groupDetailDataRow.getDefinition());

		groupContextMap.put(groupContextCount, groupContext);
		GroupContext lastGroupContext = groupContextMap.get(groupContextCount - 1);

		// 用于计算分组行汇总值
		excelContext.setDetailRows(groupDetailDataRow.getDetailRows());

		itExcelCell = groupDetailDataRow.getExcelRow().iterator();
		while (itExcelCell.hasNext()) {
			excelCell = itExcelCell.next();
			if (excelCell.getCellIndex() == -1) {
				continue;
			}
			fillCellData(dataTable, metaTable, sheet, rowIndex, excelCell, columnWidths, groupContext,
					lastGroupContext);
		}
	}

	private void fillExpandData(Row row, ExcelExpandDataRow expandDataRow, DataTable dataTable, ExcelRow excelRow,
			MetaTable metaTable, HashMap<Integer, Integer> columnWidths) throws Throwable {
		Iterator<ExcelCell> itExcelCell = null;
		ExcelCell excelCell = null;
		Cell cell = null;
		
		String definition = null;
		IExcelDataRow refRow = null;
		IExcelTransformer transformer = null;
		
		MetaColumn metaColumn = null;

		itExcelCell = excelRow.iterator();
		while(itExcelCell.hasNext()) {
			excelCell = itExcelCell.next();
			definition = excelCell.getDefinition();
			String sourceType = excelCell.getSourceType();
			
			int colIndex = excelCell.getCellIndex() - 1;
			cell = ExcelUtil.getCell(row, colIndex);
			
			Object value = null;
			if (sourceType.equalsIgnoreCase("field")) {
				// 确定数据行
				if (excelCell.getColumnExpandTarget()) {
					refRow = findRowInExpandRow(dataTable, expandDataRow, excelCell);
				} else {
					refRow = expandDataRow.getRow(0);
				}

				if (refRow != null) {
					// 找到对应列
					metaColumn = metaTable.get(definition);
					value = dataTable.getObject(refRow.getRowIndex(), metaColumn.getKey());

					transformer = transformerFactory.getTransformer(context, excelCell);
					if (transformer != null) {
						value = transformer.transform(excelCell, value);
					}
				}
			} else if (sourceType.equalsIgnoreCase("const")) {
				value = definition;
			} else if (sourceType.equalsIgnoreCase("formula")) {
				excelContext.setExcelExpandDataRow(expandDataRow);
				value = excelContext.getMidParser().eval(ScriptType.Formula, definition, excelContext, null);
			}

			if (value != null) {
				//对一些0值的处理
				if (value instanceof BigDecimal) {
					BigDecimal tempValue = (BigDecimal) value;
					if (tempValue.compareTo(BigDecimal.ZERO) != 0) {
						ExcelUtil.setCellValue(cell, value);
					}
				} else {
					ExcelUtil.setCellValue(cell, value);
				}
			}
			
			// 设置单元格样式
			cell.setCellStyle(styleFactory.getCellStyle(excelCell));
			ExcelUtil.calcMaxColumnWidth(cell, columnWidths, colIndex);
		}
	}

	private IExcelDataRow findRowInExpandRow(DataTable dataTable, ExcelExpandDataRow expandDataRow,
			ExcelCell excelCell) {
		IExcelDataRow row = null;
		ExcelDetailDataRow tmpRow = null;
		ArrayList<ColumnExpandItem> columnExpandItems = excelCell.getColumnExpandItems();
		int size = expandDataRow.size();
		for (int i = 0; i < size; ++i) {
			tmpRow = (ExcelDetailDataRow) expandDataRow.getRow(i);
			ColumnExpandItem columnExpandItem = null;
			boolean found = true;
			Iterator<ColumnExpandItem> it = columnExpandItems.iterator();
			while (it.hasNext()) {
				columnExpandItem = it.next();
				Object value = dataTable.getObject(tmpRow.getRowIndex(), columnExpandItem.getKey());
				if (!ExcelDataUtil.equals(columnExpandItem.getValue(), value)) {
					found = false;
					break;
				}
			}
			if (found) {
				row = tmpRow;
				break;
			}
		}
		return row;
	}

	private void fillDetailData(Sheet sheet, int rowIndex, ExcelDetailDataRow detailDataRow, DataTable dataTable,
			ExcelRow excelRow, MetaTable metaTable, HashMap<Integer, Integer> columnWidths,
			HashMap<Integer, GroupContext> groupContextMap, int groupContextCount) throws Throwable {
		Iterator<ExcelCell> itExcelCell = null;
		ExcelCell excelCell = null;
		excelContext.setExcelDetailDataRow(detailDataRow);

		GroupContext groupContext = new GroupContext();
		groupContext.setDefinitionMap(detailDataRow.getDefinitionMap());

		groupContextMap.put(groupContextCount, groupContext);
		GroupContext lastGroupContext = groupContextMap.get(groupContextCount - 1);

		dataTable.setPos(detailDataRow.getRowIndex());
		itExcelCell = excelRow.iterator();
		while (itExcelCell.hasNext()) {
			excelCell = itExcelCell.next();
			fillCellData(dataTable, metaTable, sheet, rowIndex, excelCell, columnWidths, groupContext,
					lastGroupContext);
		}
	}

	private void exportFixRow(Document document, MetaDataObject metaDataObject, Sheet sheet, ExcelRow excelRow,
			int offset, HashMap<Integer, Integer> columnWidths) throws Throwable {
		int rowIndex = excelRow.getRowIndex() -1  + offset;
		Row row = ExcelUtil.getRow(sheet, rowIndex);
		row.setHeightInPoints((short) excelRow.getHeight());

		ExcelCell excelCell = null;
		excelContext.setExcelRow(excelRow);
		Iterator<ExcelCell> itExeclCell = excelRow.iterator();
		while (itExeclCell.hasNext()) {
			excelCell = itExeclCell.next();
			String tableKey = excelCell.getTableKey();

			MetaTable metaTable = metaDataObject.getMetaTable(tableKey);
			DataTable dataTable = document.get(tableKey);
			fillCellData(dataTable, metaTable, sheet, rowIndex, excelCell, columnWidths, null, null);
		}
	}

	private void fillCellData(DataTable dataTable, MetaTable metaTable, Sheet sheet, int rowIndex, ExcelCell excelCell,
			HashMap<Integer, Integer> columnWidths, GroupContext groupContext, GroupContext lastGroupContext)
			throws Throwable {
		int colIndex = excelCell.getCellIndex() - 1;
		int columnSpan = excelCell.getMergedColumnSpan() - 1;
		int rowSpan = excelCell.getMergedRowSpan() - 1;

		String definition = excelCell.getDefinition();
		if (definition == null || definition.equals("")) {
			throw new MetaException(MetaException.TEMPLATE_DATAOBJECT_NOT_DEFINED, SimpleStringFormat
					.format(StringTable.getString(null, "", StringTable.TemplateDataobjectNotDefined)));
		}
		Row row = ExcelUtil.getRow(sheet, rowIndex);
		Cell cell = ExcelUtil.getCell(row, colIndex);
		String sourceType = excelCell.getSourceType();

		Object value = null;
		if (sourceType.equalsIgnoreCase("field")) {
			if (excelCell.getColumnExpandTarget()) {
				value = excelCell.getExpandCaption() == null ? excelCell.getExpandValue()
						: excelCell.getExpandCaption();
			} else {
				value = dataTable.getObject(definition);
			}
		} else if (sourceType.equalsIgnoreCase("const")) {
			value = definition;
		} else if (sourceType.equalsIgnoreCase("formula")) {
			value = excelContext.getMidParser().eval(ScriptType.Formula, definition, excelContext, null);

			String[] formula = definition.split("\\(");
			String SumExpandDirect = formula[0];
			if ("SumExpandPortrait".equalsIgnoreCase(SumExpandDirect)) {
				int expandCount = excelContext.getExpandCount();

				String para1 = formula[1].substring(formula[1].indexOf("{") + 1, formula[1].lastIndexOf("}"));
				String[] expandFieldIndex = para1.split(",");
				if (expandCount < expandFieldIndex.length - 1) {
					expandCount = expandCount + 1;
					excelContext.setExpandCount(expandCount);
				} else {
					excelContext.setExpandCount(0);
				}
			}
		}

		short formatType = StampExcelUtil.getExcelCellStyle(excelCell, workBook);
		if (formatType != -1) {
			styleFactory.getCellStyle(excelCell).setDataFormat(formatType);
		}
		
		if (!StampExcelUtil.isNumericCell(excelCell)) {
			IExcelTransformer transformer = transformerFactory.getTransformer(context, excelCell);
			if (transformer != null) {
				value = transformer.transform(excelCell, value);
			}
		}
		
		if (value != null) {
			//对一些0值的处理
			if (value instanceof BigDecimal) {
				BigDecimal tempValue = (BigDecimal) value;
				if (tempValue.compareTo(BigDecimal.ZERO) != 0) {
					ExcelUtil.setCellValue(cell, value);
				}
			} else {
				ExcelUtil.setCellValue(cell, value);
			}
		}

		if (groupContext != null) {
			groupContext.putDisplayValue(excelCell.getCellIndex(), TypeConvertor.toString(value));
		}

		// 设置单元格样式
		CellStyle cellStyle = styleFactory.getCellStyle(excelCell);
		if (columnSpan > 0 || rowSpan > 0) {
			for (int c = 0; c <= columnSpan; c++) {
				for (int r = 0; r <= rowSpan; r++) {
					cell = ExcelUtil.getCell(sheet, rowIndex + r, colIndex + c);
					cell.setCellStyle(cellStyle);
				}
			}
			sheet.addMergedRegion(new CellRangeAddress(rowIndex, rowIndex + rowSpan, colIndex, colIndex + columnSpan));
		} else {
			cell.setCellStyle(cellStyle);
		}

		ExcelUtil.calcMaxColumnWidth(cell, columnWidths, colIndex);
	}
}
