package com.bokesoft.yes.excel.cmd.normal;

import java.awt.Color;
import java.util.HashMap;

import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.DefaultIndexedColorMap;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFColor;
import org.apache.poi.xssf.usermodel.XSSFDataFormat;
import org.apache.poi.xssf.usermodel.XSSFFont;
import org.openxmlformats.schemas.spreadsheetml.x2006.main.CTColor;

import com.bokesoft.yes.common.util.StringUtil;
import com.bokesoft.yes.excel.template.ExcelBorder;
import com.bokesoft.yes.excel.template.ExcelCell;
import com.bokesoft.yes.excel.template.ExcelDisplay;
import com.bokesoft.yes.excel.template.ExcelFont;
import com.bokesoft.yigo.common.def.HAlignment;
import com.bokesoft.yigo.common.def.VAlignment;
import com.bokesoft.yigo.meta.exceltemplate.ExcelCellBorderType;
import com.bokesoft.yigo.meta.form.component.MetaComponent;
import com.bokesoft.yigo.meta.form.component.MetaFont;
import com.bokesoft.yigo.meta.form.component.MetaFormat;

/**
 * Excel单元格样式工具类
 */
public class ExcelStyleFactory {
	private Workbook workbook = null;

	private HashMap<String, CellStyle> styleMap = new HashMap<String, CellStyle>();
	
	private CellStyle defaultStyle = null;
	
	public ExcelStyleFactory(Workbook workbook) {
		this.workbook = workbook;
		
		defaultStyle = workbook.createCellStyle();
		defaultStyle.setVerticalAlignment(VerticalAlignment.CENTER);
		defaultStyle.setAlignment(HorizontalAlignment.CENTER);
		defaultStyle.setFont(createDefaultFont(workbook));
	}

	/**
	 * @param excelCell 单元格虚拟结构对象 
	 * @return 返回单元格样式对象
	 */
	public CellStyle getCellStyle(ExcelCell excelCell) {
		CellStyle cellStyle = null;
		
		ExcelDisplay display = excelCell.getDisplay();
		if (display == null) {
			return defaultStyle;
		}
		
		String key = display.createStyleKey();
		if (styleMap.containsKey(key)) {
			cellStyle = styleMap.get(key);
		}

		if (cellStyle == null) {
			cellStyle = workbook.createCellStyle();
			styleMap.put(key, cellStyle);
			cellStyle.setAlignment(tranExcelHAlignment(display.getHAlign()));
			cellStyle.setVerticalAlignment(tranExcelVAlignment(display.getVAlign()));
			cellStyle.setWrapText(display.isWrapText());
			
			if (display.getFormat() != null){
				String formatString = "0";
				if (display.getFormat().getFormatString() != null){
					formatString = display.getFormat().getFormatString();
				}
				XSSFDataFormat df = (XSSFDataFormat) workbook.createDataFormat();
				cellStyle.setDataFormat(df.getFormat(formatString));
			}

			Font font = createDefaultFont(workbook);
			if (display.getFont() != null) {
				font = getExcelFont(display.getFont());
			}
			
			String foreColor = display.getForeColor();
			if (!foreColor.isEmpty()) {
				font.setColor(IndexedColors.valueOf(foreColor.toUpperCase()).index);
			}
			cellStyle.setFont(font);
			
			String backColor = display.getBackColor();
			if (!backColor.isEmpty()) {
				cellStyle.setFillForegroundColor(IndexedColors.valueOf(backColor.toUpperCase()).index);
				cellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
			}
			
			if (display.getBorder() != null) {
				ExcelBorder excelBorder = display.getBorder();
				cellStyle.setBorderLeft(tranExcelBorder(excelBorder.getLeftStyle()));
				cellStyle.setBorderTop(tranExcelBorder(excelBorder.getTopStyle()));
				cellStyle.setBorderRight(tranExcelBorder(excelBorder.getRightStyle()));
				cellStyle.setBorderBottom(tranExcelBorder(excelBorder.getBottomStyle()));
				
				String sColor = excelBorder.getLeftColor();
				if (!sColor.isEmpty()) {
					cellStyle.setLeftBorderColor(IndexedColors.valueOf(sColor.toUpperCase()).index);
				}
				
				sColor = excelBorder.getTopColor();
				if (!sColor.isEmpty()) {
					cellStyle.setTopBorderColor(IndexedColors.valueOf(sColor.toUpperCase()).index);
				}
				
				sColor = excelBorder.getRightColor();
				if (!sColor.isEmpty()) {
					cellStyle.setRightBorderColor(IndexedColors.valueOf(sColor.toUpperCase()).index);
				}
				
				sColor = excelBorder.getBottomColor();
				if (!sColor.isEmpty()) {
					cellStyle.setBottomBorderColor(IndexedColors.valueOf(sColor.toUpperCase()).index);
				}
			}
		}

		return cellStyle;
	}
	
	/**
	 * @param excelCell 单元格结构对象 
	 * @return 返回单元格样式对象
	 */
	public CellStyle getCellStyle(MetaComponent comp) {
		String cellStyleKey = this.createCellStyleKey(comp);
				
		if(styleMap.containsKey(cellStyleKey)){
			return styleMap.get(cellStyleKey);
		}
		
		XSSFCellStyle cellStyle = (XSSFCellStyle) workbook.createCellStyle();
		setBorderStyle(comp,cellStyle);
		
		MetaFormat format = comp.getFormat();
		if(format == null){
			styleMap.put(cellStyleKey, cellStyle);
			return cellStyle;
		}
			
		cellStyle.setAlignment(tranExcelHAlignment(format.getHAlign()));
		cellStyle.setVerticalAlignment(tranExcelVAlignment(format.getVAlign()));
		
		XSSFFont font = createDefaultXSSFFont(workbook);
		
		String foreColor = format.getForeColor();
					
		if (!StringUtil.isBlankOrNull(foreColor)) {
			XSSFColor rgbFontColor = setfontStyle(foreColor);
			font.setColor(rgbFontColor);
		}
		if (format.getFont() != null){
			getExcelFont(format.getFont(),font);
		}			
		cellStyle.setFont(font);
					
		String backColor = format.getBackColor();
		if (!StringUtil.isBlankOrNull(backColor)) {			
			XSSFColor rgbBackColor = setfontStyle(backColor);
			cellStyle.setFillForegroundColor(rgbBackColor);
			cellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
		}

		styleMap.put(cellStyleKey, cellStyle);
		return cellStyle;
	}

	private String createCellStyleKey(MetaComponent comp){
		
		String cellStyleKey = "";
		cellStyleKey += comp.getBorderStyle() == null ? "" : comp.getBorderStyle();
		cellStyleKey += "_" + comp.getBorderColor() == null ? "" : comp.getBorderColor();
		
		MetaFormat format = comp.getFormat();
		if(comp.getFormat() == null){
			return cellStyleKey;
		}
		
		cellStyleKey += "_" + format.getHAlign();
		cellStyleKey += "_" + format.getVAlign();
		cellStyleKey += "_" + comp.getForeColor() == null ? "" : comp.getForeColor();
		cellStyleKey += "_" + comp.getBackColor() == null ? "" : comp.getBackColor();
		
		MetaFont font = format.getFont();
		if (font == null){
			return cellStyleKey;
		}
		
		cellStyleKey += "_" + (font.getSize() == -1 ? "12" : font.getSize());
		cellStyleKey += "_" + (StringUtil.isBlankOrNull(font.getName())? "宋体" : font.getName());
		cellStyleKey += "_" + font.isBold() == null ? "" : font.isBold();
		cellStyleKey += "_" + font.isItalic() == null ? "" : font.isItalic();
		
		return cellStyleKey;
	}


	private void setBorderStyle(MetaComponent comp, XSSFCellStyle cellStyle) {
		String borderStyle = comp.getBorderStyle();
		String borderColor = comp.getBorderColor();
		
		if (!StringUtil.isBlankOrNull(borderStyle)){
			String[] style = borderStyle.split(" ");
			//设置一个即为四条边框全设置，设置两个极为第一个表示上下边框第二个代表左右边框，设置四个极为上右下左的顺序设置边框。
			if (style.length == 1){
				cellStyle.setBorderTop(tranExcelBorder(style[0]));
				cellStyle.setBorderRight(tranExcelBorder(style[0]));
				cellStyle.setBorderBottom(tranExcelBorder(style[0]));
				cellStyle.setBorderLeft(tranExcelBorder(style[0]));	
				if (!StringUtil.isBlankOrNull(borderColor) && tranExcelBorder(style[0]) != BorderStyle.NONE){
					XSSFColor rgbBackColor = setfontStyle(borderColor);
					cellStyle.setTopBorderColor(rgbBackColor);
					cellStyle.setRightBorderColor(rgbBackColor);
					cellStyle.setBottomBorderColor(rgbBackColor);
					cellStyle.setLeftBorderColor(rgbBackColor);	
				}
			}else if (style.length == 2){
				cellStyle.setBorderTop(tranExcelBorder(style[0]));
				cellStyle.setBorderRight(tranExcelBorder(style[1]));
				cellStyle.setBorderBottom(tranExcelBorder(style[0]));
				cellStyle.setBorderLeft(tranExcelBorder(style[1]));	
				if (!StringUtil.isBlankOrNull(borderColor)){
					XSSFColor rgbBackColor = setfontStyle(borderColor);
					if (tranExcelBorder(style[0]) != BorderStyle.NONE){
						cellStyle.setTopBorderColor(rgbBackColor);
						cellStyle.setBottomBorderColor(rgbBackColor);
					}
					if (tranExcelBorder(style[1]) != BorderStyle.NONE){
						cellStyle.setRightBorderColor(rgbBackColor);
						cellStyle.setLeftBorderColor(rgbBackColor);	
					}					
				}
			}else if (style.length == 4){
				cellStyle.setBorderTop(tranExcelBorder(style[0]));
				cellStyle.setBorderRight(tranExcelBorder(style[1]));
				cellStyle.setBorderBottom(tranExcelBorder(style[2]));
				cellStyle.setBorderLeft(tranExcelBorder(style[3]));	
				if (!StringUtil.isBlankOrNull(borderColor)){
					XSSFColor rgbBackColor = setfontStyle(borderColor);
					if (tranExcelBorder(style[0]) != BorderStyle.NONE){
						cellStyle.setTopBorderColor(rgbBackColor);
					}
					if (tranExcelBorder(style[1]) != BorderStyle.NONE){
						cellStyle.setRightBorderColor(rgbBackColor);
					}
					if (tranExcelBorder(style[2]) != BorderStyle.NONE){
						cellStyle.setBottomBorderColor(rgbBackColor);
					}
					if (tranExcelBorder(style[3]) != BorderStyle.NONE){
						cellStyle.setLeftBorderColor(rgbBackColor);	
					}					
				}
			}
		}
	}

	private XSSFColor setfontStyle(String styleColor) {
		int r = Integer.valueOf(styleColor.substring(1, 3),16);
		int g = Integer.valueOf(styleColor.substring(3, 5),16);
		int b = Integer.valueOf(styleColor.substring(5, 7),16);
		String bufferedImage = String.format("0xFF%02X%02X%02X", r,g,b);
		String subString = String.valueOf(bufferedImage.substring(3));
		int colorInt = Integer.parseInt(subString,16);
		Color color = new Color(colorInt);
		XSSFColor rgbColor = new XSSFColor(color, new DefaultIndexedColorMap());
		return rgbColor;
	}


	private HorizontalAlignment tranExcelHAlignment(int hAlignment) {
		HorizontalAlignment excelHAlignment = HorizontalAlignment.CENTER;
		switch (hAlignment) {
		case HAlignment.CENTER:
			excelHAlignment = HorizontalAlignment.CENTER;
			break;
		case HAlignment.LEFT:
			excelHAlignment = HorizontalAlignment.LEFT;
			break;
		case HAlignment.RIGHT:
			excelHAlignment = HorizontalAlignment.RIGHT;
			break;
		}

		return excelHAlignment;
	}

	private VerticalAlignment tranExcelVAlignment(int vAlignment) {
		VerticalAlignment excelVAlignment = VerticalAlignment.CENTER;
		switch (vAlignment) {
		case VAlignment.CENTER:
			excelVAlignment = VerticalAlignment.CENTER;
			break;
		case VAlignment.BOTTOM:
			excelVAlignment = VerticalAlignment.BOTTOM;
			break;
		case VAlignment.TOP:
			excelVAlignment = VerticalAlignment.TOP;
			break;
		}
		return excelVAlignment;
	}
	
	private BorderStyle tranExcelBorder(int boder) {
		BorderStyle boderType = BorderStyle.NONE;
		switch (boder) {
		case ExcelCellBorderType.Border_None:
			boderType = BorderStyle.NONE;
			break;
		case ExcelCellBorderType.Border_Double:
			boderType = BorderStyle.DOUBLE;
			break;
		case ExcelCellBorderType.Border_Thin:
			boderType = BorderStyle.THIN;
			break;
		case ExcelCellBorderType.Border_Medium:
			boderType = BorderStyle.MEDIUM;
			break;
		case ExcelCellBorderType.Border_Dashed:
			boderType = BorderStyle.DASHED;
			break;
		case ExcelCellBorderType.Border_Hair:
			boderType = BorderStyle.HAIR;
			break;
		case ExcelCellBorderType.Border_Thick:
			boderType = BorderStyle.THICK;
			break;
		}

		return boderType;
	}
	
	private BorderStyle tranExcelBorder(String boder) {
		BorderStyle boderType = BorderStyle.NONE;
		switch (boder) {
		case ExcelCellBorderType.STR_Border_None:
			boderType = BorderStyle.NONE;
			break;
		case ExcelCellBorderType.STR_Border_Double:
			boderType = BorderStyle.DOUBLE;
			break;
		case ExcelCellBorderType.STR_Border_Thin:
			boderType = BorderStyle.THIN;
			break;
		case ExcelCellBorderType.STR_Border_Medium:
			boderType = BorderStyle.MEDIUM;
			break;
		case ExcelCellBorderType.STR_Border_Dashed:
			boderType = BorderStyle.DASHED;
			break;
		case ExcelCellBorderType.STR_Border_Hair:
			boderType = BorderStyle.HAIR;
			break;
		case ExcelCellBorderType.STR_Border_Thick:
			boderType = BorderStyle.THICK;
			break;
		}

		return boderType;
	}
	
	
	private Font getExcelFont(ExcelFont excelFont) {
		Font cellFont = workbook.createFont();
		cellFont.setBold(excelFont.isBold());
		cellFont.setItalic(excelFont.isItalic());
		cellFont.setFontName(excelFont.getName());
		cellFont.setFontHeightInPoints((short) excelFont.getSize());

		return cellFont;
	}
	
	private Font getExcelFont(MetaFont metaFont,Font font) {
		if (metaFont.isBold() != null){
			font.setBold(metaFont.isBold());
		}	
		if (metaFont.isBold() != null){
			font.setItalic(metaFont.isItalic());
		}		
		if (!StringUtil.isBlankOrNull(metaFont.getName())){
			font.setFontName(metaFont.getName());
		}else{
			font.setFontName("宋体");
		}	
		if (metaFont.getSize() == -1){
			font.setFontHeightInPoints((short) 12);
		}else{
			font.setFontHeightInPoints(metaFont.getSize().shortValue());
		}
		return font;
	}
	
	
	private Font createDefaultFont(Workbook workbook) {
		Font font = workbook.createFont();
		font.setBold(false);
		font.setItalic(false);
		font.setFontName("宋体");
		font.setFontHeightInPoints((short) 12);
		return font;
	}
	
	private XSSFFont createDefaultXSSFFont(Workbook workbook) {
		XSSFFont font = (XSSFFont) workbook.createFont();
		font.setBold(false);
		font.setItalic(false);
		font.setFontName("宋体");
		font.setFontHeightInPoints((short) 12);
		return font;
	}
	
	/**
	 * 获取默认格式
	 * @return 返回单元格格式对象
	 */
	public CellStyle getDefaultStyle() {
		return defaultStyle;
	}
}
