package com.bokesoft.yes.excel.cmd.normal;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;

import com.bokesoft.yes.csv.cmd.normal.IExportPostProcess;
import com.bokesoft.yes.data.service.DocumentServiceManager;
import com.bokesoft.yes.data.service.IDocumentService;
import com.bokesoft.yes.data.service.ModuleNameConstants;
import com.bokesoft.yes.excel.template.ExcelCell;
import com.bokesoft.yes.excel.template.ExcelRow;
import com.bokesoft.yes.excel.template.ExcelSheet;
import com.bokesoft.yes.excel.template.ExcelWorkbook;
import com.bokesoft.yes.excel.template.util.ExcelUtil;
import com.bokesoft.yes.excel.transformer.ExcelTransformerFactory;
import com.bokesoft.yes.excel.transformer.IExcelTransformer;
import com.bokesoft.yes.excel.utils.ExcelFileUtil;
import com.bokesoft.yes.struct.filedata.FileData;
import com.bokesoft.yigo.common.def.ScriptType;
import com.bokesoft.yigo.common.def.TableMode;
import com.bokesoft.yigo.meta.dataobject.MetaDataObject;
import com.bokesoft.yigo.meta.dataobject.MetaTable;
import com.bokesoft.yigo.meta.exceltemplate.ExcelTemplateRowType;
import com.bokesoft.yigo.meta.exceltemplate.MetaExcelWorkbook;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.form.MetaForm;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.struct.datatable.DataTable;
import com.bokesoft.yigo.struct.document.Document;
import com.bokesoft.yigo.util.ExcelUtils;
import com.bokesoft.yigo.util.StampExcelUtil;

public class BatchExportExcelWithTemplate implements IExport {
	/** 上下文环境*/
	private DefaultContext context = null;
	/** 批量导出界面的数据集*/
	private DataTable srcDataTable = null;
	/** 批量导出的数据源表中对应的OID字段*/
	private String OIDFieldKey = null;
	/** Excel对象*/
	private Workbook workbook = null;
	/** Excel模板配置对象*/
	private MetaExcelWorkbook metaWorkbook = null;
	/** Excel模板处理对象*/
	private ExcelWorkbook virtualWorkbook = null;
	/** 待导出界面的配置对象*/
	private MetaForm metaForm = null;
	/** 待导出界面的配置对象*/
	private MetaDataObject exportMetaDataObject = null;
	/** 值转换工具类*/
	private ExcelTransformerFactory transformerFactory = null;
	/** Excel单元格格式工具类*/
	private ExcelStyleFactory styleFactory = null;
	/** 导出后扩展服务名*/
	private String postServiceName = null;
	/**自定义导出后的文件名(不包含后缀)*/
	private String exportFileName=null;
	/** 记录sheet的行数*/
	private HashMap<String, Integer> sheetRowIndex = new HashMap<String, Integer>();
	/** 记录每个sheet的列宽*/
	private HashMap<String, HashMap<Integer, Integer>> sheetColumnWidths = new HashMap<String, HashMap<Integer, Integer>>();

    private boolean needDownload = false;
	
	public BatchExportExcelWithTemplate(DefaultContext context, MetaExcelWorkbook metaWorkbook, DataTable dataTable, String OIDFieldKey, String exportFormKey, String postExportServiceName, String exportFileName,boolean needDownload) throws Throwable {
		this.context = context;
		this.metaWorkbook = metaWorkbook;
		this.postServiceName = postExportServiceName;
		this.srcDataTable = dataTable;
		this.OIDFieldKey = OIDFieldKey;
		this.exportFileName=exportFileName;
		this.needDownload = needDownload;
		metaForm = context.getVE().getMetaFactory().getMetaForm(exportFormKey);
		exportMetaDataObject = metaForm.getDataSource().getDataObject();
	}
	
	@SuppressWarnings("deprecation")
	@Override
	public FileData exportData() throws Throwable {
		
		// 通过配置文件生成Excel模型
		virtualWorkbook = new ExcelWorkbook(exportMetaDataObject, metaWorkbook, context.getMidParser());
		virtualWorkbook.calcExpandColumn(context.getMidParser(), null, exportMetaDataObject);
		
		workbook = new SXSSFWorkbook(500);
		transformerFactory = new ExcelTransformerFactory();
		styleFactory = new ExcelStyleFactory(workbook);
		
		boolean isFirst = true;
		Document document = null;
		List<Long> oIDs = ExcelUtils.getOIDs(srcDataTable, OIDFieldKey);
		IDocumentService docService = DocumentServiceManager.getInstance()
				.getDocumentService(context, ModuleNameConstants.EXCEL, "BatchExportExcelWithTemplate", metaForm.getKey(), metaWorkbook.getKey());
		docService.init(context, metaForm.getKey(), oIDs, true);
		while (docService.hasNext()) {
			document = docService.next();
			if (document != null) {
				DefaultContext newContext = new DefaultContext(context);
				newContext.setDocument(document);
				exportSingleDocument(newContext, isFirst, document);
				isFirst = false;
			}
		}
		
		IMetaFactory metaFactory = context.getVE().getMetaFactory();
		
		String filePath = ExcelUtils.getExportFilePath(metaFactory, metaWorkbook.getFormKey());
		
		FileData fileData = new FileData();
		context.setDocument(document);
		if (!needDownload){				
			fileData = ExcelFileUtil.writeExcel(workbook, filePath);
		}else{
			fileData = ExcelFileUtil.convertWorkbookToFileData(metaWorkbook.getFormKey(), exportFileName, workbook);
		}
		IExportPostProcess process = ExcelUtils.getPostExtProcess(context, postServiceName);
		if( process != null ) {
			process.process(context, fileData);
		}
		if(needDownload){
			return fileData;
		}else{
			return new FileData();
		}
	}

	private void exportSingleDocument(DefaultContext context, boolean isFirst, Document exportDocument) throws Throwable {
		int rowType = -1;
		Sheet sheet = null;
		ExcelRow excelRow = null;
		ExcelSheet excelSheet = null;
		HashMap<Integer, Integer> columnWidths = null;
		
		Iterator<ExcelSheet> itExeclSheet = virtualWorkbook.iterator();
		while (itExeclSheet.hasNext()) {
			excelSheet = itExeclSheet.next();
			String sheetName = excelSheet.getName();
			sheet = ExcelUtil.getSheet(workbook, sheetName);
			
			if (!sheetColumnWidths.containsKey(sheetName)) {
				sheetColumnWidths.put(sheetName, new HashMap<Integer, Integer>());
			}
			if (!sheetRowIndex.containsKey(sheetName)) {
				sheetRowIndex.put(sheetName, 0);
			}
			
			int offset = sheetRowIndex.get(sheetName);
			columnWidths = sheetColumnWidths.get(excelSheet.getName());
			
			Iterator<ExcelRow> itExeclRow = excelSheet.iterator();
			while (itExeclRow.hasNext()) {
				excelRow = itExeclRow.next();
				rowType = excelRow.getType();
				switch (rowType) {
				case ExcelTemplateRowType.Head:
				case ExcelTemplateRowType.Fix:
				case ExcelTemplateRowType.DetailHead:
					if (isFirst) {
						// 表头只需要第一次导入
						exportFixRow(exportDocument, sheet, excelRow, offset, columnWidths);
					}
					break;
				case ExcelTemplateRowType.Detail:
					// 导出明细数据，连带汇总数据一起
					offset = exportDetailRow(context, exportDocument, sheet, excelRow, offset, columnWidths);
					break;
				}
			}
			
			sheetRowIndex.put(excelSheet.getName(), offset);
		}
	}
	
	@SuppressWarnings("deprecation")
	private int exportDetailRow(DefaultContext context, Document document, Sheet sheet, ExcelRow excelRow, int offset, HashMap<Integer, Integer> columnWidths) throws Throwable {
		String tableKey = excelRow.getTableKey();
		DataTable dataTable = document.get(tableKey);
		if (dataTable == null) {
			return offset;
		}
		
		Row row = null;
		int startRowIndex = excelRow.getRowIndex() - 1;
		
		dataTable.beforeFirst();
		while(dataTable.next(true)) {
			int rowIndex = startRowIndex + offset;
			row = ExcelUtil.getRow(sheet, rowIndex);
			row.setHeightInPoints((short) excelRow.getHeight());
			
			fillDetailData(context, row, document, excelRow);
			
			offset++;
		}

		return offset;
	}
	
	private void fillDetailData(DefaultContext context, Row row, Document document, ExcelRow excelRow) throws Throwable {
		Iterator<ExcelCell> itExcelCell = null;
		ExcelCell excelCell = null;
		Cell cell = null;
		
		String definition = null;
		String rowTableKey = excelRow.getTableKey();
		DataTable rowDataTable = document.get(rowTableKey);
		
		itExcelCell = excelRow.iterator();
		while(itExcelCell.hasNext()) {
			excelCell = itExcelCell.next();
			definition = excelCell.getDefinition();
			String sourceType = excelCell.getSourceType();
			
			int colIndex = excelCell.getCellIndex() - 1;
			cell = ExcelUtil.getCell(row, colIndex);
			
			Object value = null;
			if (sourceType.equalsIgnoreCase("field")) {
				String tableKey = excelCell.getTableKey();
				if ( tableKey == null || tableKey.isEmpty()) {
					value = rowDataTable.getObject(definition);
				} else {
					MetaTable metaTable = exportMetaDataObject.getTable(tableKey);
					DataTable dataTable = document.get(tableKey);
					if (metaTable.getTableMode() == TableMode.HEAD) {
						dataTable.first();
					}
					value = dataTable.getObject(definition);
				}
			} else if (sourceType.equalsIgnoreCase("const")) {
				value = definition;
			} else if (sourceType.equalsIgnoreCase("formula")) {
				 value = context.getMidParser().eval(ScriptType.Formula, definition);
			}
			
			// 设置单元格样式
			CellStyle cellStyle = styleFactory.getCellStyle(excelCell);
			cell.setCellStyle(cellStyle);
			
			
			short formatType = StampExcelUtil.getExcelCellStyle(excelCell, workbook);
			if (formatType != -1) {
				cellStyle.setDataFormat(formatType);
			} 
			
			if (!StampExcelUtil.isNumericCell(excelCell)) {
				IExcelTransformer transformer = transformerFactory.getTransformer(context.getVE(), excelCell);
				if (transformer != null) {
					value = transformer.transform(excelCell, value);
				}
			}
			
			if (value != null) {
				ExcelUtil.setCellValue(cell, value);
			}
		}
	}
	
	private void exportFixRow(Document document, Sheet sheet, ExcelRow excelRow, int offset, HashMap<Integer, Integer> columnWidths) throws Throwable {
		int rowIndex = excelRow.getRowIndex() - 1 + offset;
		Row row = ExcelUtil.getRow(sheet, rowIndex);
		row.setHeightInPoints((short) excelRow.getHeight());
		Cell cell = null;

		ExcelCell excelCell = null;

		String tableKey = null;
		DataTable dataTable = null;
		Object value = null;
		
		Iterator<ExcelCell> itExeclCell = excelRow.iterator();
		while (itExeclCell.hasNext()) {
			excelCell = itExeclCell.next();
			int colIndex = excelCell.getCellIndex() - 1;
			int columnSpan = excelCell.getMergedColumnSpan() - 1;
			int rowSpan = excelCell.getMergedRowSpan() - 1;

			String definition = excelCell.getDefinition();
			
			cell = ExcelUtil.getCell(row, colIndex);
			String sourceType = excelCell.getSourceType();
			
			if (sourceType.equalsIgnoreCase("field")) {
				if (excelCell.getColumnExpandTarget()) {
					value = excelCell.getExpandCaption();
				} else {
					tableKey = excelCell.getTableKey();
					
					dataTable = document.get(tableKey);
					value = dataTable.getObject(definition);
				}
			} else if (sourceType.equalsIgnoreCase("const")) {
				value = definition;
			} else if (sourceType.equalsIgnoreCase("formula")) {
				value = context.getMidParser().eval(ScriptType.Formula, definition);
			}

			// 设置单元格样式
			CellStyle cellStyle = styleFactory.getCellStyle(excelCell);
			if (columnSpan > 0 || rowSpan > 0) {
				for (int c = 0; c <= columnSpan; c++) {
					for (int r = 0; r <= rowSpan; r++) {
						cell = ExcelUtil.getCell(sheet, rowIndex + r, colIndex + c);
						cell.setCellStyle(cellStyle);
					}
				}
				
				sheet.addMergedRegionUnsafe(new CellRangeAddress(rowIndex, rowIndex + rowSpan, colIndex, colIndex + columnSpan));
			} else {
				cell.setCellStyle(cellStyle);
			}
			
			short formatType = StampExcelUtil.getExcelCellStyle(excelCell, workbook);
			if (formatType != -1) {
				cellStyle.setDataFormat(formatType);
			} 
			
			if (!StampExcelUtil.isNumericCell(excelCell)) {
				IExcelTransformer transformer = transformerFactory.getTransformer(context.getVE(), excelCell);
				if (transformer != null) {
					value = transformer.transform(excelCell, value);
				}
			}
			
			if (value != null) {
				ExcelUtil.setCellValue(cell, value);
			}
		}
	}
}