package com.bokesoft.yigo.meta.form.anim;

import java.util.LinkedList;

import com.bokesoft.yes.meta.persist.dom.form.MetaConstants;
import com.bokesoft.yigo.common.def.AnimInterpolatorType;
import com.bokesoft.yigo.common.def.AnimSizeType;
import com.bokesoft.yigo.common.def.LayoutAnimOrderType;
import com.bokesoft.yigo.meta.base.AbstractMetaObject;
import com.bokesoft.yigo.meta.base.GenericNoKeyCollectionWithKey;
import com.bokesoft.yigo.meta.base.IMetaEnv;
import com.bokesoft.yigo.meta.base.MetaException;

/***
 * 动画定义配置对象
 *
 */
public class MetaAnimItem extends GenericNoKeyCollectionWithKey<MetaAnimItem> {
	
	public static final String TAG_NAME = "AnimItem";

    /** 动画标识，最外层的AnimItem需要该属性 */
    private String key = "";
    
    /** 动画类型 */
    private Integer type = -1;
    
    /** 动画属性集合*/
    private MetaAnimPropertyCollection properties;

    /** 单次播放持续时间，单位是毫秒，负值无效 */
    private Long duration = -1L;

    /** 最长播放时间，单位是毫秒，非正值无效 */
    private Long maxDuration = -1L;

    /** 重复模式，忽略大小写 */
    private Integer repeatMode = -1;

    /** 重复次数，小于0表示无限循环，0表示播放1次，
     * 正数表示重复播放次数（例如为3时代表播放后又重复3次，总计4次） */
    private Integer repeatCount;

    /** 延迟播放的时间，单位是毫秒，负值无效 */
    private Long startOffset = -1L;

    /** 表示当动画播放完后控件是否需要保持最后显示的状态，
     * 注意当动画为子动画时，需要等到最外层父动画播放完 */
    private Boolean fillAfter;

    /** 表示当动画播放前（在延迟播放时间段内）控件是否需要保持动画最初的状态 */
    private Boolean fillBefore;

    /** 插值器类型，用于控制动画速率
     * @see AnimInterpolatorType
     */
    private Integer interpolator = AnimInterpolatorType.ACCELERATE_DECELERATE;
    
    /**
     * 子View执行Anim顺序，取值范围为{@link LayoutAnimOrderType},在动画作为布局动画时使用
     */
    private Integer order = LayoutAnimOrderType.NORMAL;
    
    /**
     * 用于延迟的每个子View的动画开始动画间隔时间，是所执行动画时间的倍数，取值范围为[0,1]
     * 在动画作为布局动画时使用
     */
    private Float delay = 0f;
    
    public MetaAnimItem(){
    	this.properties = new MetaAnimPropertyCollection();
    }

    public void setKey(String key) {
        this.key = key;
    }

    @Override
    public String getKey() {
        return key;
    }
    
    /**
     * 设置动画类型
     * @param type 动画类型
     */
    public void setType(Integer type) {
		this.type = type;
	}
    
    /**
     * 获取动画类型
     * @return 动画类型
     */
    public Integer getType() {
		return type;
	}
    
    /**
     * 设置动画属性集合
     * @param properties 动画属性集合
     */
    public void setProperties(MetaAnimPropertyCollection properties) {
		this.properties = properties;
	}
    
    /**
     * 获取动画属性集合
     * @return 动画属性集合
     */
    public MetaAnimPropertyCollection getProperties() {
		return properties;
	}

    /**
     * 获取单次播放持续时间
     * @return 单次播放持续时间
     */
    public Long getDuration() {
        return duration;
    }

    /**
     * 设置单次播放持续时间
     * @param duration 单次播放持续时间
     */
    public void setDuration(Long duration) {
        this.duration = duration;
    }

    /**
     * 获取最长播放时间
     * @return 最长播放时间
     */
    public Long getMaxDuration() {
        return maxDuration;
    }

    /**
     * 设置最长播放时间
     * @param maxDuration 最长播放时间
     */
    public void setMaxDuration(Long maxDuration) {
        this.maxDuration = maxDuration;
    }

    /**
     * 获取重复模式
     * @return 重复模式
     */
    public Integer getRepeatMode() {
        return repeatMode;
    }

    /**
     * 设置重复模式
     * @param repeatMode 重复模式
     */
    public void setRepeatMode(Integer repeatMode) {
        this.repeatMode = repeatMode;
    }

    /**
     * 获取重复次数
     * @return 重复次数
     */
    public Integer getRepeatCount() {
        return repeatCount;
    }

    /**
     * 设置重复次数
     * @param repeatCount 重复次数
     */
    public void setRepeatCount(Integer repeatCount) {
        this.repeatCount = repeatCount;
    }

    /**
     * 获取延迟播放的时间
     * @return 延迟播放的时间
     */
    public Long getStartOffset() {
        return startOffset;
    }

    /**
     * 设置延迟播放的时间
     * @param startOffset 延迟播放的时间
     */
    public void setStartOffset(Long startOffset) {
        this.startOffset = startOffset;
    }

    /**
     * 获取是否需要保持最后的状态
     * @return 是否需要保持最后的状态
     */
    public Boolean getFillAfter() {
        return fillAfter;
    }

    /**
     * 设置是否需要保持最后的状态
     * @param fillAfter 是否需要保持最后的状态
     */
    public void setFillAfter(Boolean fillAfter) {
        this.fillAfter = fillAfter;
    }

    /**
     * 获取当动画播放前（在延迟播放时间段内）控件是否需要保持动画最初的状态
     * @return 当动画播放前（在延迟播放时间段内）控件是否需要保持动画最初的状态
     */
    public Boolean getFillBefore() {
        return fillBefore;
    }

    /**
     * 设置当动画播放前（在延迟播放时间段内）控件是否需要保持动画最初的状态
     * @param fillBefore 当动画播放前（在延迟播放时间段内）控件是否需要保持动画最初的状态
     */
    public void setFillBefore(Boolean fillBefore) {
        this.fillBefore = fillBefore;
    }

    /** 设置插值器类型
     *
     * @return 插值器类型
     * @see AnimInterpolatorType
     */
    public Integer getInterpolator() {
        return interpolator;
    }

    /** 获取插值器类型
     *
     * @param interpolator 插值器类型
     * @see AnimInterpolatorType
     */
    public void setInterpolator(Integer interpolator) {
        this.interpolator = interpolator;
    }
    
    /**
     * 获取属性值
     * @param key 属性标识
     * @return 属性值
     */
    private String getPropertiesValue(String key){
    	MetaAnimPropertyCollection collection = getProperties();
    	if(collection == null){
    		return null;
    	}
    	MetaAnimProperty property = collection.get(key);
    	if(property == null){
    		return null;
    	}
    	return property.getValue();
    }
    
    /**
     * 获取boolean类型的属性值
     * @param key 属性标识
     * @param defaultValue 默认值
     * @return 属性值
     */
    private boolean getPropertiesValue(String key, boolean defaultValue){
    	String string = getPropertiesValue(key);
    	if(string != null && !string.isEmpty()){
    		try{
    			return Boolean.parseBoolean(string);
    		}catch(Exception e){
    		}
    	}
    	return defaultValue;
    }
    
    /**
     * 获取float类型的属性值
     * @param key 属性标识
     * @param defaultValue 默认值
     * @return 属性值
     */
    private float getPropertiesValue(String key, float defaultValue){
    	String string = getPropertiesValue(key);
    	if(string != null && !string.isEmpty()){
    		try{
    			return Float.parseFloat(string);
    		}catch(Exception e){
    		}
    	}
    	return defaultValue;
    }
    
    /**
     * 获取值类型
     * @param key 值类型标识
     * @param defaultValue 默认值
     * @return 值类型
     * @see AnimSizeType
     */
    private int getPropertiesSizeType(String key, int defaultValue){
    	int ret = -1;
    	String str = getPropertiesValue(key);
    	if(str != null){
    		ret = AnimSizeType.parse(str);
    	}
    	if(ret < 0){
    		ret = defaultValue;
    	}
    	return ret;
    }
    
    public void setPropertiesValue(String key, String value){
    	MetaAnimProperty property = new MetaAnimProperty();
    	property.setKey(key == null ? "" : key);
    	property.setValue(value == null ? "" : value);
    	this.properties.add(property);
    }
    
    /**
     * 获取是否共享插值器{@link #type}为{@link AnimType#ANIMSET}时使用
     * 
     * @return 是否共享插值器，为true时其全部子AnimItem、子子AnimItem... ...使用改AnimSet的插值器
     * 
     * 默认值为false
     */
    public boolean isShareInterpolator(){
    	return getPropertiesValue(MetaConstants.ANIMSET_SHAREINTERPOLATOR, false);
    }
    
    /**
     * 获取起始透明度，{@link #type}为{@link AnimType#ALPHAANIM}时使用
     * @return 起始透明度，取值范围为[0,1],默认值为0
     */
    public float getFromAlpha(){
    	return getPropertiesValue(MetaConstants.ALPHAANIM_FROMALPHA, 0f);
    }
    
    /**
     * 获取最终透明度，{@link #type}为{@link AnimType#ALPHAANIM}时使用
     * @return 起始透明度，取值范围为[0,1],默认值为0
     */
    public float getToAlpha(){
    	return getPropertiesValue(MetaConstants.ALPHAANIM_TOALPHA, 0f);
    }
    
    /**
     * 获取起始角度，{@link #type}为{@link AnimType#ROTATEANIM}时使用
     * @return 起始角度，0为没有旋转，正数表示顺时针旋转的角度，默认为0
     */
    public float getFromDegrees(){
    	return getPropertiesValue(MetaConstants.ROTATEANIM_FROMDEGREES, 0f);
    }
    
    /**
     * 获取最终角度，{@link #type}为{@link AnimType#ROTATEANIM}时使用
     * @return 起始角度，0为没有旋转，正数表示顺时针旋转的角度，默认为0
     */
    public float getToDegrees(){
    	return getPropertiesValue(MetaConstants.ROTATEANIM_TODEGREES, 0f);
    }

    /**
     * 获取动画中心X值类型，{@link #type}为{@link AnimType#ROTATEANIM}或{@link AnimType#SCALEANIM}时使用
     * @return 动画中心X值类型，取值范围为{@link AnimSizeType},默认值为{@link AnimSizeType#Absolute}
     */
    public int getPivotXType(){
    	return getPropertiesSizeType(MetaConstants.ROTATEANIM_PIVOTXTYPE, AnimSizeType.Absolute);
    }
    
    /**
     * 获取动画中心X值大小，{@link #type}为{@link AnimType#ROTATEANIM}或{@link AnimType#SCALEANIM}时使用
     * @return 动画中心X值大小，默认值为0
     */
    public float getPivotXValue(){
    	return getPropertiesValue(MetaConstants.ROTATEANIM_PIVOTXVALUE, 0f);
    }

    /**
     * 获取动画中心Y值类型，{@link #type}为{@link AnimType#ROTATEANIM}或{@link AnimType#SCALEANIM}时使用
     * @return 动画中心X值类型，取值范围为{@link AnimSizeType},默认值为{@link AnimSizeType#Absolute}
     */
    public int getPivotYType(){
    	return getPropertiesSizeType(MetaConstants.ROTATEANIM_PIVOTYTYPE, AnimSizeType.Absolute);
    }
    
    /**
     * 获取动画中心Y值大小，{@link #type}为{@link AnimType#ROTATEANIM}或{@link AnimType#SCALEANIM}时使用
     * @return 动画中心X值大小，默认值为0
     */
    public float getPivotYValue(){
    	return getPropertiesValue(MetaConstants.ROTATEANIM_PIVOTYVALUE, 0f);
    }
    
    /**
     * 获取控件本身的宽度的起始缩放倍数，{@link #type}为{@link AnimType#SCALEANIM}时使用
     * @return 控件本身的宽度的起始缩放倍数，默认为0
     */
    public float getFromX(){
    	return getPropertiesValue(MetaConstants.SCALEANIM_FROMX, 0f);
    }
    
    /**
     * 获取控件本身的宽度的最终缩放倍数，{@link #type}为{@link AnimType#SCALEANIM}时使用
     * @return 控件本身的宽度的最终缩放倍数，默认为0
     */
    public float getToX(){
    	return getPropertiesValue(MetaConstants.SCALEANIM_TOX, 0f);
    }
    
    /**
     * 获取控件本身的高度的起始缩放倍数，{@link #type}为{@link AnimType#SCALEANIM}时使用
     * @return 控件本身的高度的起始缩放倍数，默认为0
     */
    public float getFromY(){
    	return getPropertiesValue(MetaConstants.SCALEANIM_FROMY, 0f);
    }
    
    /**
     * 获取控件本身的高度的最终缩放倍数，{@link #type}为{@link AnimType#SCALEANIM}时使用
     * @return 控件本身的高度的最终缩放倍数，默认为0
     */
    public float getToY(){
    	return getPropertiesValue(MetaConstants.SCALEANIM_TOY, 0f);
    }
    
    /**
     * 获取X轴起始的坐标值类型，{@link #type}为{@link AnimType#TRANSLATEANIM}时使用
     * @return X轴起始的坐标值类型，取值范围为{@link AnimSizeType},默认值为{@link AnimSizeType#Absolute}
     */
    public int getFromXType(){
    	return getPropertiesSizeType(MetaConstants.TRANSLATE_FROMXTYPE, AnimSizeType.Absolute);
    }
    
    /**
     * 获取X轴起始的坐标值大小，{@link #type}为{@link AnimType#TRANSLATEANIM}时使用
     * @return X轴起始的坐标值大小，默认值为0
     */
    public float getFromXValue(){
    	return getPropertiesValue(MetaConstants.TRANSLATE_FROMXVALUE, 0f);
    }
    
    /**
     * 获取X轴最终的坐标值类型，{@link #type}为{@link AnimType#TRANSLATEANIM}时使用
     * @return X轴最终的坐标值类型，取值范围为{@link AnimSizeType},默认值为{@link AnimSizeType#Absolute}
     */
    public int getToXType(){
    	return getPropertiesSizeType(MetaConstants.TRANSLATE_TOXTYPE, AnimSizeType.Absolute);
    }
    
    /**
     * 获取X轴最终的坐标值大小，{@link #type}为{@link AnimType#TRANSLATEANIM}时使用
     * @return X轴最终的坐标值大小，默认值为0
     */
    public float getToXValue(){
    	return getPropertiesValue(MetaConstants.TRANSLATE_TOXVALUE, 0f);
    }
    
    /**
     * 获取Y轴起始的坐标值类型，{@link #type}为{@link AnimType#TRANSLATEANIM}时使用
     * @return Y轴起始的坐标值类型，取值范围为{@link AnimSizeType},默认值为{@link AnimSizeType#Absolute}
     */
    public int getFromYType(){
    	return getPropertiesSizeType(MetaConstants.TRANSLATE_FROMYTYPE, AnimSizeType.Absolute);
    }
    
    /**
     * 获取Y轴起始的坐标值大小，{@link #type}为{@link AnimType#TRANSLATEANIM}时使用
     * @return Y轴起始的坐标值大小，默认值为0
     */
    public float getFromYValue(){
    	return getPropertiesValue(MetaConstants.TRANSLATE_FROMYVALUE, 0f);
    }
    
    /**
     * 获取Y轴最终的坐标值类型，{@link #type}为{@link AnimType#TRANSLATEANIM}时使用
     * @return Y轴最终的坐标值类型，取值范围为{@link AnimSizeType},默认值为{@link AnimSizeType#Absolute}
     */
    public int getToYType(){
    	return getPropertiesSizeType(MetaConstants.TRANSLATE_TOYTYPE, AnimSizeType.Absolute);
    }
    
    /**
     * 获取Y轴最终的坐标值大小，{@link #type}为{@link AnimType#TRANSLATEANIM}时使用
     * @return Y轴最终的坐标值大小，默认值为0
     */
    public float getToYValue(){
    	return getPropertiesValue(MetaConstants.TRANSLATE_TOYVALUE, 0f);
    }
    
    /**
     * 获取子view执行anim的顺序
     * 在动画作为布局动画时使用
     * 
     * @return 子view执行anim的顺序，取值范围为{@link LayoutAnimOrderType},默认值为{@link #order}
     */
    public Integer getOrder() {
		return order;
	}
    
    /**
     * 设置子view执行anim的顺序
     * 在动画作为布局动画时使用
     * 
     * @param order 子view的执行顺序，取值范围为{@link LayoutAnimOrderType},默认值为{@link #order}
     */
    public void setOrder(Integer order) {
		this.order = order;
	}
    
    /**
     * 获取用于延迟的每个子view的动画开始动画的间隔时间，是所执行动画时间的倍数。
     * 在动画作为布局动画时使用
     * 
     * @return 用于延迟的每个子view的动画开始动画的间隔时间，取值范围为[0,1],默认值为{@link #delay}
     */
    public Float getDelay() {
		return delay;
	}
    
    /**
     * 设置用于延迟的每个子view的动画开始动画的间隔时间，是所执行动画时间的倍数。
     * 在动画作为布局动画时使用
     * 
     * @param delay用于延迟的每个子view的动画开始动画的间隔时间，取值范围为[0,1],默认值为{@link #delay}
     */
    public void setDelay(Float delay) {
		this.delay = delay;
	}
    
    @Override
    public String getTagName() {
    	return TAG_NAME;
    }
    
    @Override
    public void getChildMetaObjects(LinkedList<Object> list) {
    	addAll(list, new AbstractMetaObject[]{properties});
    }
    
    @Override
    public AbstractMetaObject newInstance() {
    	return new MetaAnimItem();
    }

	@Override
	public AbstractMetaObject createChildMetaObject(IMetaEnv env, Object doc, Object node, String tagName, Object flag,
			int runType) throws MetaException {
		AbstractMetaObject resultMetaObject = null;
		if(TAG_NAME.equals(tagName)){
			MetaAnimItem newObject = new MetaAnimItem();
			newObject.preProcessMetaObject(env, doc, node, flag, runType);
			add(newObject);
			resultMetaObject = newObject;
		}else if(MetaAnimPropertyCollection.TAG_NAME.equals(tagName)){
			MetaAnimPropertyCollection collection = new MetaAnimPropertyCollection();
			collection.preProcessMetaObject(env, doc, node, flag, runType);
			properties = collection;
			resultMetaObject = collection;
		}
		return resultMetaObject;
	}

    @Override
    public MetaAnimItem clone() {
        MetaAnimItem newObject = (MetaAnimItem) super.clone();
        newObject.setKey(key);
        newObject.setType(type);
        newObject.setProperties(properties == null ? null : (MetaAnimPropertyCollection)properties.clone());
        newObject.setDuration(duration);
        newObject.setMaxDuration(maxDuration);
        newObject.setRepeatMode(repeatMode);
        newObject.setRepeatCount(repeatCount);
        newObject.setStartOffset(startOffset);
        newObject.setFillAfter(fillAfter);
        newObject.setFillBefore(fillBefore);
        newObject.setInterpolator(interpolator);
        newObject.setOrder(order);
        newObject.setDelay(delay);
        return newObject;
    }
}
