package com.bokesoft.yes.meta.json.util;

import java.util.ArrayList;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONObject;

import com.bokesoft.yes.common.json.JSONConstants;
import com.bokesoft.yes.meta.json.AbstractJSONHandler;
import com.bokesoft.yes.meta.json.ISerializeContext;
import com.bokesoft.yes.meta.json.UIJSONHandlerMap;
import com.bokesoft.yigo.meta.base.AbstractMetaObject;
import com.bokesoft.yigo.meta.base.GenericKeyCollection;
import com.bokesoft.yigo.meta.base.GenericKeyCollectionWithKey;
import com.bokesoft.yigo.meta.base.GenericNoKeyCollection;
import com.bokesoft.yigo.meta.base.GenericNoKeyCollectionWithKey;

public class UIJSONHandlerUtil {
	
	/**
	 * 序列化一个配置对象
	 * @param meta 配置对象
	 * @param hook 上下文
	 * @return
	 * @throws Throwable
	 */
	public static JSONObject build(AbstractMetaObject meta, ISerializeContext hook) throws Throwable {
		String controlType = meta.getTagName();
		AbstractJSONHandler<AbstractMetaObject,ISerializeContext> handler = UIJSONHandlerMap.getHandler(controlType);
		JSONObject model = null;
		if (handler != null) {
			model = handler.toJSON(meta, hook);
		}
		return model;
	}
	
	/**
	 * 反序列化一个已知类型对象
	 * @param clazz 类型
	 * @param json json对象
	 * @return
	 * @throws Throwable
	 */
	@SuppressWarnings("unchecked")
	public static <T> T unbuild(Class<T> clazz, JSONObject json) throws Throwable {
		AbstractMetaObject meta = (AbstractMetaObject)clazz.newInstance();
		String controlType = meta.getTagName();
		AbstractJSONHandler<AbstractMetaObject,ISerializeContext> handler = UIJSONHandlerMap.getHandler(controlType);
		if (handler != null) {
			handler.fromJSONImpl(meta, json);
		}
		return (T) meta;
	}
	
	/**
	 * 反序列化一个未知对象
	 */
	public static AbstractMetaObject unbuild(JSONObject json) throws Throwable {
		String tagName = json.optString(JSONConstants.COMMON_TAGNAME);
		AbstractMetaObject meta = null;
		AbstractJSONHandler<AbstractMetaObject,ISerializeContext> handler = UIJSONHandlerMap.getHandler(tagName);
		if (handler != null) {
			meta = handler.newInstance();
			handler.fromJSONImpl(meta, json);
		}
		return meta;
	}
	
	/**
	 * 反序列化一个已知内部类型集合
	 * @param clazz 内部类型
	 * @param array 集合json
	 */	
	public static <T> List<T> unbuild(Class<T> clazz, JSONArray array) throws Throwable {
		List<T> list = new ArrayList<T>();
		for( int i = 0,size = array.length();i < size;i++ ) {
			list.add(unbuild(clazz, array.getJSONObject(i)));
		}
		return list;
	}
	
	/**
	 * 序列化GenericNoKeyCollection
	 */
	public static JSONArray buildNoKeyCollection(ISerializeContext hook, 
			GenericNoKeyCollection<? extends AbstractMetaObject> collection) throws Throwable {
		JSONArray array = new JSONArray();
		AbstractJSONHandler<AbstractMetaObject,ISerializeContext> handler = null;
		for( AbstractMetaObject c : collection ) {
			handler = UIJSONHandlerMap.getHandler(c.getTagName());
			if( handler != null ) {
				array.put(handler.toJSON(c, hook));
			}
		}
		return array;
	}
	
	/**
	 * 序列化GenericKeyCollection,可能有少数需要JSONObject类型,前台处理
	 */
	public static JSONArray buildKeyCollection(ISerializeContext hook, 
			GenericKeyCollection<? extends AbstractMetaObject> collection) throws Throwable {
		JSONArray array = new JSONArray();
		AbstractJSONHandler<AbstractMetaObject,ISerializeContext> handler = null;
		for( AbstractMetaObject c : collection ) {
			handler = UIJSONHandlerMap.getHandler(c.getTagName());
			if( handler != null ) {
				array.put(handler.toJSON(c, hook));
			}
		}
		return array;
	}
	
	/**
	 * 序列化GenericKeyCollectionWithKey
	 */
	public static JSONArray buildKeyWithKeyCollection(ISerializeContext hook, 
			GenericKeyCollectionWithKey<? extends AbstractMetaObject> collection) throws Throwable {
		JSONArray array = new JSONArray();
		AbstractJSONHandler<AbstractMetaObject,ISerializeContext> handler = null;
		for( AbstractMetaObject c : collection ) {
			handler = UIJSONHandlerMap.getHandler(c.getTagName());
			if( handler != null ) {
				array.put(handler.toJSON(c, hook));
			}
		}
		return array;
	}
	
	/**
	 * 序列化GenericNoKeyCollectionWithKey
	 */
	public static JSONArray buildNoKeyWithKeyCollection(ISerializeContext hook, 
			GenericNoKeyCollectionWithKey<? extends AbstractMetaObject> collection) throws Throwable {
		JSONArray array = new JSONArray();
		AbstractJSONHandler<AbstractMetaObject,ISerializeContext> handler = null;
		for( AbstractMetaObject c : collection ) {
			handler = UIJSONHandlerMap.getHandler(c.getTagName());
			if( handler != null ) {
				array.put(handler.toJSON(c, hook));
			}
		}
		return array;
	}
}
