package com.bokesoft.yigo.tools.dict;

import java.util.List;

import com.bokesoft.yigo.struct.dict.Item;
import com.bokesoft.yigo.struct.dict.ItemData;

/**
 * 字典缓存接口
 * @author zhufw
 *
 */
public interface IDictCache {
	
	/**
	 * 删除缓存
	 * @param itemKey 字典标识
	 * @param oid 字典数据标识
	 * @throws Throwable 处理异常
	 */
	public void removeItem(String itemKey, long oid) throws Throwable;
	
	/**
	 * 获取一个缓存
	 * @param itemKey 字典标识
	 * @param oid 字典数据标识
	 * @return 字典项
	 * @throws Throwable 处理异常
	 */
	public Item getItem(String itemKey, long oid) throws Throwable;

	/**
	 * 根据字典Code获取缓存
	 * @param itemKey 字典标识
	 * @param code 编码
	 * @return 字典项
	 * @throws Throwable 处理异常
	 */
	public Item getItemByCode(String itemKey, String code) throws Throwable;
	
	/**
	 * 获取多个缓存
	 * @param itemKey 字典标识
	 * @param oids 字典数据标识集合
	 * @return 字典项集合
	 * @throws Throwable 处理异常
	 */
	public List<Item> getItems(String itemKey, List<Long> oids) throws Throwable;
	
	/**
	 * 刷新一个缓存
	 * @param itemKey 字典标识
	 * @param oid 字典数据标识
	 * @throws Throwable 处理异常
	 */
	public void refreshItem(String itemKey, long oid ) throws Throwable;
	
	/**
	 * 刷新多个缓存
	 * @param itemKey 字典标识
	 * @param oids 字典数据标识集合
	 * @throws Throwable 处理异常
	 */
	public void refreshItems(String itemKey, List<Long> oids) throws Throwable;
	
	/**
	 * 查找复合条件的数据
	 * @param itemKey 字典标识
	 * @param fuzzyFields 模糊检查字段定义
	 * @param fuzzyValue 值
	 * @param filter 过滤
	 * @param root 根过滤
	 * @param stateMask 状态过滤
	 * @param queryMatchType 模糊匹配类型
	 * @return 字典项数据集合
	 * @throws Throwable 处理异常
	 */
	public List<ItemData> lookup(String itemKey, String fuzzyFields, String fuzzyValue, IItemFilter filter, ItemData root, int formState, int stateMask, int queryMatchType) throws Throwable;
	
	/**
	 * 精确定位一个值
	 * 存在多个的情况会报错
	 * @param itemKey 字典标识
	 * @param field 字段
	 * @param value 值
	 * @param filter 过滤
	 * @param root 根过滤
	 * @param stateMask 状态过滤
	 * @param formKey 表单Key
	 * @param fieldKey 字段Key
	 * @return 字典项
	 * @throws Throwable 处理异常
	 */
	public Item locate(String itemKey, String field, Object value, IItemFilter filter, ItemData root, int stateMask, String formKey, String fieldKey, int formState) throws Throwable;
	
	/**
	 * 清空一个字典缓存
	 * @param itemKey 字典标识
	 */
	public void removeDictCache(String itemKey);
	
	/**
	 * 标记字典缓存需更新（一般用于数据同步后， 字典树状结构发生变化的情况）
	 * @param itemKey 字典标识
	 * @param oid 字典数据标识
	 * @return OID
	 * @throws Throwable 处理异常
	 */
	public long updateCache(String itemKey, long oid) throws Throwable;
	
}
