package com.bokesoft.yigo.struct.rights;

import java.util.ArrayList;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.bokesoft.yes.common.json.JSONHelper;
import com.bokesoft.yes.common.json.SerializationException;
import com.bokesoft.yigo.common.json.JSONSerializable;

public class BaseFormRights implements JSONSerializable {
	
	/** 表单标识*/
	protected String formKey = null;
	
	/** 操作权限,集合中的操作为有权限的操作*/
	protected List<String> optRights = new ArrayList<String>();
	
	/** 字段可见权限,集合中的字段为没有权限的字段*/
	protected List<String> visibleRights = new ArrayList<String>();

	/** 字段可编辑权限,集合中的字段为没有权限的字段*/
	protected List<String> enableRights = new ArrayList<String>();

	/** 是否有所有操作权限*/
	protected boolean hasAllOptRights = false;

	/** 是否有所有可见性权限*/
	protected boolean hasAllEnableRights = true;

	/** 是否有所有可用性权限*/
	protected boolean hasAllVisibleRights = true;
	
	/** 是否需要缓存*/
	protected boolean needCache = true;
	
	public BaseFormRights(){
		
	}
	
	public BaseFormRights(String formKey){
		this.formKey = formKey;
	}
	
	/**
	 * 设置表单key
	 * 
	 * @param formKey
	 */
	public void setFormKey(String formKey) {
		this.formKey = formKey;
	}

	/**
	 * 获取表单key
	 * 
	 * @return
	 */
	public String getFormKey() {
		return this.formKey;
	}
	
	/**
	 * 是否有字段的可见性权限
	 * 
	 * @param fieldKey
	 * @return
	 */
	public boolean hasVisibleRights(String fieldKey) {
		if (hasAllVisibleRights) {
			return true;
		}
		return !visibleRights.contains(fieldKey);
	}
	
	/**
	 * 添加无可见性权限的字段
	 * 
	 * @param fieldKey 字段标识
	 */
	public void addVisibleRights(String fieldKey) {
		if( hasAllVisibleRights ) {
			hasAllVisibleRights = false;
		}
		visibleRights.add(fieldKey);
	}
	
	/**
	 * 移除无可见性权限的字段
	 * 
	 * @param fieldKey 字段标识
	 */
	public void removeVisibleRights(String fieldKey) {
		visibleRights.remove(fieldKey);
		
		hasAllVisibleRights = visibleRights.isEmpty();
	}
	
	/**
	 * 批量添加无可见性权限的字段
	 * @param visibleRightsList 无可见性权限字段集合
	 */
	public void addVisibleRights(List<String> visibleRightsList) {
		if(visibleRightsList != null){
			visibleRights.addAll(visibleRightsList);
		}
		hasAllVisibleRights = visibleRights.isEmpty();
	}
	
	/**
	 * 获取无可见性权限字段集合
	 * @return 无可见性权限字段集合
	 */
	public List<String> getVisibleRights() {
		return visibleRights;
	}
	
	/**
	 * 是否有字段的可编辑权限
	 * 
	 * @param fieldKey 字段标识
	 * @return
	 */
	public boolean hasEnableRights(String fieldKey) {
		if (hasAllEnableRights) {
			return true;
		}
		return !enableRights.contains(fieldKey);
	}
	
	/**
	 * 添加无可用性条件的字段
	 * @param fieldKey 字段标识
	 */
	public void addEnableRights(String fieldKey) {
		if( hasAllEnableRights ) {
			hasAllEnableRights = false;
		}
		enableRights.add(fieldKey);
	}
	
	/**
	 * 移除无可用性权限的字段
	 * 
	 * @param fieldKey 字段标识
	 */
	public void removeEnableRights(String fieldKey) {
		enableRights.remove(fieldKey);
		
		hasAllEnableRights = enableRights.isEmpty();
	}
	
	/**
	 * 批量添加无可用性权限的字段
	 * @param enableRightsList 无可用性权限字段集合
	 */
	public void addEnableRights(List<String> enableRightsList) {
		if(enableRightsList != null){
			enableRights.addAll(enableRightsList);
		}
		this.hasAllEnableRights = enableRights.isEmpty();
	}
	
	/**
	 * 是否有操作权限
	 * 
	 * @param optKey 操作标识
	 * @return 是否有操作权限
	 */
	public boolean hasOptRights(String optKey) {
		if (hasAllOptRights) {
			return true;
		}
		return optRights.contains(optKey);
	}
	
	/**
	 * 获取无可用性权限字段集合
	 * 
	 * @return 可用性权限字段集合
	 */
	public List<String> getEnableRights() {
		return enableRights;
	}
	
	/**
	 * 批量添加操作权限
	 * @param optRights 可用操作集合
	 */
	public void addOptRights(List<String> optRights){
		if( optRights != null ) {
			this.optRights.addAll(optRights);
		}
	}

	/**
	 * 这里添加的字段 是有权限的
	 * 
	 * @param optKey 操作标识
	 */
	public void addOptRights(String optKey) {
		if(optKey.equalsIgnoreCase("*")){
			hasAllOptRights = true;
		}else if(!optRights.contains(optKey)){
			optRights.add(optKey);
		}
	}
	
	/**
	 * 移除无权限操作
	 * 
	 * @param optKey 操作标识
	 */
	public void removeOptRights(String optKey) {
		if( hasAllOptRights ) {
			hasAllOptRights = false;
		}
		optRights.remove(optKey);
	}
	
	/**
	 * 获取有权限的操作列表集合
	 * @return 有权限的操作列表集合
	 */
	public List<String> getOptRights() {
		return optRights;
	}
	
	/**
	 * 设置是否有所有可见性权限
	 * 
	 * @param b
	 */
	public void setAllVisibleRights(boolean b) {
		if( b ){
			this.visibleRights.clear();	
		}
		this.hasAllVisibleRights = b;
	}
	
	/**
	 * 设置是否有所有可用性权限
	 * 
	 * @param b
	 */
	public void setAllEnableRights(boolean b) {
		if( b ){
			this.enableRights.clear();	
		}
		this.hasAllEnableRights = b;
	}

	/**
	 * 设置是否有所有操作权限
	 * 
	 * @param b
	 */
	public void setAllOptRights(boolean b) {
		if(!b){
			this.optRights.clear();	
		}
		this.hasAllOptRights = b;
	}

	/**
	 * 是否有所有字段可见性权限
	 * 
	 * @return
	 */
	public boolean hasAllVisibleRights() {
		return this.hasAllVisibleRights;
	}

	/**
	 * 是否有所有字段可编辑性权限
	 * 
	 * @return
	 */
	public boolean hasAllEnableRights() {
		return this.hasAllEnableRights;
	}

	/**
	 * 是否有所有操作权限
	 * 
	 * @return
	 */
	public boolean hasAllOptRights() {
		return this.hasAllOptRights;
	}

	/**
	 * 是否需要缓存
	 * @return
	 */
	public boolean needCache() {
		return needCache;
	}

	/**
	 * 设置是否需要缓存
	 * @param needCache
	 */
	public void setNeedCache(boolean needCache) {
		this.needCache = needCache;
	}

	public JSONObject toJSON() throws SerializationException, JSONException {
		JSONObject jsonObj = new JSONObject();
		JSONHelper.writeToJSON(jsonObj, RightsJSONConstants.FORMRIGHTS_FORMKEY,
				formKey, "");
		
		jsonObj.put(RightsJSONConstants.RIGHTS_NEEDCACHE, needCache);

		if (this.hasAllOptRights) {
			jsonObj.put(RightsJSONConstants.FORMRIGHTS_ALLOPTRIGHTS, true);
		} else {
			jsonObj.put(RightsJSONConstants.FORMRIGHTS_ALLOPTRIGHTS, false);
			JSONArray optArray = new JSONArray();
			for (String opt : optRights) {
				optArray.put(opt);
			}

			jsonObj.put(RightsJSONConstants.FORMRIGHTS_OPTRIGHTS, optArray);
		}

		if (this.hasAllEnableRights) {
			jsonObj.put(RightsJSONConstants.FORMRIGHTS_ALLENABLERIGHTS, true);
		} else {
			jsonObj.put(RightsJSONConstants.FORMRIGHTS_ALLENABLERIGHTS, false);
			JSONArray enableArray = new JSONArray();
			for (String field : enableRights) {
				enableArray.put(field);
			}

			jsonObj.put(RightsJSONConstants.FORMRIGHTS_ENABLERIGHTS,
					enableArray);
		}

		if (this.hasAllVisibleRights) {
			jsonObj.put(RightsJSONConstants.FORMRIGHTS_ALLVISIBLERIGHTS, true);
		} else {
			jsonObj.put(RightsJSONConstants.FORMRIGHTS_ALLVISIBLERIGHTS, false);
			JSONArray visibleArray = new JSONArray();
			for (String field : visibleRights) {
				visibleArray.put(field);
			}

			jsonObj.put(RightsJSONConstants.FORMRIGHTS_VISIBLERIGHTS,
					visibleArray);
		}

		return jsonObj;
	}

	@Override
	public void fromJSON(JSONObject jsonObj) throws Throwable {
		formKey = JSONHelper.readFromJSON(jsonObj,
				RightsJSONConstants.FORMRIGHTS_FORMKEY, "");
		
		needCache = jsonObj.getBoolean(RightsJSONConstants.RIGHTS_NEEDCACHE);

		hasAllOptRights = jsonObj.getBoolean(RightsJSONConstants.FORMRIGHTS_ALLOPTRIGHTS);
		if (!hasAllOptRights) {
			JSONArray optArray = jsonObj.getJSONArray(RightsJSONConstants.FORMRIGHTS_OPTRIGHTS);
			optRights.clear();
			for (int i = 0; i < optArray.length(); i++) {
				String opt = optArray.getString(i);
				optRights.add(opt);
			}
		}

		hasAllEnableRights = jsonObj.getBoolean(RightsJSONConstants.FORMRIGHTS_ALLENABLERIGHTS);
		if (!hasAllEnableRights) {
			JSONArray enableArray = jsonObj.getJSONArray(RightsJSONConstants.FORMRIGHTS_ENABLERIGHTS);
			enableRights.clear();
			for (int i = 0; i < enableArray.length(); i++) {
				String field = enableArray.getString(i);
				enableRights.add(field);
			}
		}

		hasAllVisibleRights = jsonObj.getBoolean(RightsJSONConstants.FORMRIGHTS_ALLVISIBLERIGHTS);
		if (!hasAllVisibleRights) {
			JSONArray visibleArray = jsonObj.getJSONArray(RightsJSONConstants.FORMRIGHTS_VISIBLERIGHTS);
			visibleRights.clear();
			for (int i = 0; i < visibleArray.length(); i++) {
				String field = visibleArray.getString(i);
				visibleRights.add(field);
			}
		}
	}
	

	/**
	 * 是否为默认有所有权限，是界面界面将不打勾，不默认进行合并 ，数据库中没有数据
	 */
	protected boolean defStatus = true;
	
	public boolean isDefStatus() {
		return defStatus;
	}

	public void setDefStatus(boolean defStatus) {
		this.defStatus = defStatus;
	}
}
