package com.bokesoft.yigo.struct.datatable;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Vector;

import com.bokesoft.yes.common.struct.HashMapIgnoreCaseFastGet;
import com.bokesoft.yes.struct.document.IJSONHandler;
import com.bokesoft.yigo.struct.exception.StructException;

/**
 * 表格的元数据。包含表格中所有列的信息，并提供列信息的查询和修改方法。
 * 
 * @author 刘翔翔
 * @see DataTable
 */
public class DataTableMetaData implements Externalizable, IJSONHandler {

	/** 列信息的数组 */
	private Vector<ColumnInfo> tableColumns;
 
	/** 列信息的映射表 */
	private HashMapIgnoreCaseFastGet<ColumnInfo> tableColumnMap = null;

	/** DataTableMetaData的无参构造器 */
	public DataTableMetaData() {
		tableColumns = new Vector<ColumnInfo>();
		tableColumnMap = new HashMapIgnoreCaseFastGet<ColumnInfo>();
	}

	/**
	 * 在末尾添加一列
	 * 
	 * @param column 列信息
	 * @throws StructException 结构异常
	 */
	public void addColumn(ColumnInfo column) throws StructException {
		addColumn(tableColumns.size(), column);
	}
	
	/**
	 * 判断某列是否存在
	 * @param key 列标识
	 * @return 如果key指定的列存在则返回true，否则返回false
	 */
	public boolean constains(String key) {
		return tableColumnMap.containsKey(key);
	}

	/**
	 * 在指定位置添加一列
	 * 
	 * @param index 序号
	 * @param column 列信息
	 * @throws StructException 结构异常
	 */
	public void addColumn(int index, ColumnInfo column) throws StructException {
		String columnKey = column.getColumnKey();

		if (tableColumnMap.get(columnKey) != null)
			throw new StructException(StructException.COLUMN_KEY_ERROR, 
					StructException.formatMessage(null, StructException.COLUMN_KEY_ERROR) + ",列" + columnKey + "已经添加,请勿重复添加！");

		column.setColumnIndex(index);
		tableColumns.add(index, column);
		tableColumnMap.put(column.getColumnKey(), column);

		while (index < tableColumns.size() - 1) {
			index++;
			tableColumns.get(index).setColumnIndex(index);
		}
	}

	/**
	 * 移除指定列
	 * 
	 * @param columnKey
	 *            指定列的主键
	 */
	public void removeColumn(String columnKey) {
		ColumnInfo column = tableColumnMap.remove(columnKey);
		if (column == null)
			return;
		int index = column.getColumnIndex();
		tableColumns.remove(index);
		while (index < tableColumns.size()) {
			tableColumns.get(index).setColumnIndex(index);
			index++;
		}
	}

	void clear() {
		tableColumns.clear();
		tableColumnMap.clear();
	}

	/**
	 * 给定列的键值查询列的序号，若列不存在即返回-1
	 * 
	 * @param columnKey 列标识
	 * @return 如果columnKey指定的列存在则返回列序号，否则返回-1
	 * @throws StructException 结构异常
	 */
	public int findColumnIndexByKey(String columnKey) {
//		ColumnInfo column = tableColumnMap.get(columnKey);
//		if (column != null)
//			return column.getColumnIndex();

		return findColumnIndexByKey(columnKey, false);
	}
	
	public int findColumnIndexByKey(String columnKey, boolean throwErr) {
		ColumnInfo column = tableColumnMap.get(columnKey);
		if (column != null) {
			return column.getColumnIndex();
		}

		if (throwErr) {
			throw new StructException(StructException.UNFINDED_COLUMN,
					StructException.formatMessage(null, StructException.UNFINDED_COLUMN, columnKey));
		}

		return -1;
	}
	
	/**
	 * 给定列的键值查询列的序号，若列不存在即返回-1，columnKey需要为全小写，内部实现，二次开发中请勿使用
	 * @param columnKey 列标识
	 * @return 如果columnKey指定的列存在则返回其序号，否则返回-1
	 * @throws StructException 结构异常
	 */
	public int impl_findColumnIndexByKey(String columnKey) {
//		ColumnInfo column = tableColumnMap.get(columnKey);
//		if (column != null)
//			return column.getColumnIndex();

		return impl_findColumnIndexByKey(columnKey, false);
	}
	public int impl_findColumnIndexByKey(String columnKey, boolean throwErr) {
		ColumnInfo column = tableColumnMap.get(columnKey);
		if (column != null)
			return column.getColumnIndex();
		
		if (throwErr) {
			throw new StructException(StructException.UNFINDED_COLUMN,
					StructException.formatMessage(null, StructException.UNFINDED_COLUMN, columnKey));
		}

		return -1;
	}


	/**
	 * 给定某列的键值查询列的信息
	 * 
	 * @param columnKey
	 *            列的键值
	 * @return 列的信息
	 * @throws StructException
	 *             列的键值为空或者没有对应列
	 */
	public ColumnInfo getColumnInfo(String columnKey) throws StructException {
		ColumnInfo column = tableColumnMap.get(columnKey);
		if (column != null)
			return column;
		throw new StructException(StructException.COLUMN_KEY_ERROR,
				StructException.formatMessage(null, StructException.COLUMN_KEY_ERROR) + ",列" + columnKey + "不存在！");
	}

	// /**
	// * 列的键值的重复性检查
	// *
	// * @param columnKey
	// * @throws StructException
	// */
	// void columnKeyNameSpaceCheck(String columnKey) throws StructException {
	// if (columnKey == null || columnKey.length() == 0)
	// throw new StructException(StructException.COLUMN_ERROR, "列的键值为空!");
	//
	// ColumnInfo column = tableColumnMap.get(columnKey);
	// if (column != null)
	// throw new StructException(StructException.COLUMN_ERROR, "列的键值重复!");
	// }

	/**
	 * 给定某列的序号查询列的信息
	 * 
	 * @param index
	 *            列的序号
	 * @return 列的信息
	 */
	public ColumnInfo getColumnInfo(int index) {
		return tableColumns.elementAt(index);
	}

	/**
	 * 对象序列化输出流
	 */
	public void writeExternal(ObjectOutput out) throws IOException {
	}

	/**
	 * 输入流序列化对象
	 */
	public void readExternal(ObjectInput in) throws IOException,
			ClassNotFoundException {
	}

	/**
	 * 返回按值复制的对象
	 * 
	 * @return 按值复制的对象
	 */
	public DataTableMetaData deepClone() {
		DataTableMetaData clone = new DataTableMetaData();
		for (ColumnInfo c : tableColumns)
			clone.tableColumns.add(c.deepClone());
		clone.tableColumnMap.putAll(tableColumnMap);
		return clone;
	}

	/**
	 * 查询列的总数
	 * 
	 * @return 列的总数
	 */
	public int getColumnCount() {
		return tableColumns.size();
	}

	private ColumnInfo tempColumnInfo = null;
	@Override
	public IJSONHandler newHandler(String token) {
		this.tempColumnInfo = new ColumnInfo();
		return this.tempColumnInfo;
	}

	@Override
	public void putAttr(String name, String value) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public void endHandler() {
	}

	@Override
	public void endChildHandler(IJSONHandler handler) {
		int index = this.tableColumns.size();
		this.tableColumns.add(this.tempColumnInfo);
		this.tempColumnInfo.setColumnIndex(index);
		this.tableColumnMap.put(this.tempColumnInfo.getColumnKey(), this.tempColumnInfo);
		this.tempColumnInfo = null;
	}

}