package com.bokesoft.yigo.struct.datatable;

import org.json.JSONException;
import org.json.JSONObject;

import com.bokesoft.yes.common.json.JSONHelper;
import com.bokesoft.yes.common.json.SerializationException;
import com.bokesoft.yes.common.util.DBTypeUtil;
import com.bokesoft.yes.struct.datatype.DataTypeAction;
import com.bokesoft.yes.struct.datatype.DataTypeActionFactory;
import com.bokesoft.yes.struct.document.DocumentJSONConstants;
import com.bokesoft.yes.struct.document.IJSONHandler;
import com.bokesoft.yigo.common.def.DataType;
import com.bokesoft.yigo.common.json.JSONSerializable;
import com.bokesoft.yigo.struct.exception.StructException;

/**
 * 表示DataTable中一列的信息。列的信息包含：列的键值、 列的数据类型。
 * 
 * @author 刘翔翔
 * @see DataTable
 * @see DataTableMetaData
 */
public class ColumnInfo implements JSONSerializable, IJSONHandler {

	/** 是否有读的权限 */
	private boolean hasReadRights=true;
	/** 是否有写的权限 */
	private boolean hasWriteRights=true;
	
	/** 列的序号 */
	private int columnIndex = -1;

	/**
	 * 列的键值
	 */
	private String columnKey;

	/**
	 * 列数据使用的Java类型
	 * 
	 * @see com.bokesoft.yigo.common.def.JavaDataType
	 */
	private int userDataType;

	/**
	 * 列的数据类型
	 * 
	 * @see com.bokesoft.yigo.common.def.DataType
	 */
	private int dataType;

	/**
	 * 当前列的数据类型行为对象
	 */
	private DataTypeAction dataTypeAction = null;

	/**
	 * 访问控制标志
	 */
	private boolean accessControl = false;
	
	/** 
	 * 默认值 
	 */
	private String defaultValue = null;
	/** 
	 * 是否主键字段
	 */
	private boolean isPrimary = false;
	/** 
	 * 小数位数
	 */
	private int scale = -1;
	
	/** 字符长度或二进制大小 */
	private int length = 0;
	
	/** 文本大小写 @see com.bokesoft.yigo.common.def.CaseType */
	private int caseType = 0;
	
	/**
	 * 字典列绑定的code列
	 */
	private String codeColumnKey = "";

	private boolean checkLength = false;
	
	/**
	 * 无数据源字段对应的列
	 */
	private boolean ignoreChangeState = false;

	public void setCheckLength(boolean checkLength) {
		this.checkLength = checkLength;
	}

	public boolean getCheckLength( ) {
		return this.checkLength;
	}

	
	/**
	 * ColumnInfo的构造器
	 * 
	 * @param columnKey
	 *            列的键值
	 * @param dataType
	 *            列的数据类型
	 * @throws StructException
	 *             不存在的数据类型
	 */
	public ColumnInfo(String columnKey, int dataType) throws StructException {
		this.columnKey = columnKey.intern();
		this.dataType = dataType;
		if (dataType == DataType.STRING) {
			this.length = 255;
		}
		this.userDataType = DBTypeUtil.dataType2JavaDataType(dataType);
		this.dataTypeAction = DataTypeActionFactory.getDataType(userDataType);
	}

	ColumnInfo() {
	}

	/**
	 * 查询此列的键值
	 * 
	 * @return 列的键值
	 */
	public String getColumnKey() {
		return columnKey;
	}

	/**
	 * 设置此列的键值
	 * @param columnKey 此列的键值
	 */
	public void setColumnKey(String columnKey) {
		this.columnKey = columnKey;
	}

	/**
	 * 获取当前列的数据类型行为对象
	 * 
	 * @return 当前列的数据类型行为对象
	 * @throws StructException
	 *             类型不存在
	 */
	public DataTypeAction getDataTypeAction() {
		return dataTypeAction;
	}

	public ColumnInfo deepClone() {
		ColumnInfo clone = new ColumnInfo();
		clone.dataType = dataType;
		clone.columnKey = columnKey;
		clone.userDataType = userDataType;
		clone.dataTypeAction = dataTypeAction;
		clone.columnIndex = columnIndex;
		clone.accessControl = accessControl;
		clone.scale = scale;
		clone.length = length;
		clone.codeColumnKey = codeColumnKey;
		clone.caseType = caseType;
		clone.ignoreChangeState = ignoreChangeState;
		return clone;
	}

	public JSONObject toJSON() throws SerializationException, JSONException {
		JSONObject jsonObj = new JSONObject();
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_KEY, columnKey, "");
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_DATATYPE, dataType, -1);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_USER_DATATYPE, userDataType, -1);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_INDEX, columnIndex, -1);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_ACCESSCONTROL, accessControl, false);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_DEFAULTVALUE, defaultValue, null);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_ISPRIMARY, this.isPrimary, false);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_SCALE, this.scale, -1);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_LEGTH, this.length, 0);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_CODECOLUMNKEY, this.codeColumnKey, "");
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_CASETYPE, this.caseType, 0);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_CHECKLENGTH, this.checkLength, false);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DATATABLE_IGNORE_CHANGE_STATE, this.ignoreChangeState, false);
		return jsonObj;
	}

	public void fromJSON(JSONObject jsonObj) throws SerializationException, JSONException {
		columnKey = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_KEY, "");
		dataType = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_DATATYPE, -1);
		userDataType = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_USER_DATATYPE, -1);
		dataTypeAction = DataTypeActionFactory.getDataType(userDataType);
		columnIndex = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_INDEX, -1);
		accessControl = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_ACCESSCONTROL, false);
		defaultValue = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_DEFAULTVALUE, null);
		isPrimary = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_ISPRIMARY, false);
		scale = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_SCALE, -1);
		length = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_LEGTH, 0);
		codeColumnKey = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_CODECOLUMNKEY, "");
		caseType = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_COLUMN_CASETYPE, 0);
		checkLength = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_CHECKLENGTH, false);
		ignoreChangeState = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DATATABLE_IGNORE_CHANGE_STATE, false);
	}

	public int getDataType() {
		return dataType;
	}

	public int getUserDataType() {
		return userDataType;
	}

	public int getColumnIndex() {
		return columnIndex;
	}

	void setColumnIndex(int columnIndex) {
		this.columnIndex = columnIndex;
	}

	public boolean isAccessControl() {
		return this.accessControl;
	}

	public void setAccessControl(boolean accessControl) {
		this.accessControl = accessControl;
	}

	public void setAccessControl(Boolean accessControl) {
		if(accessControl != null) {
			this.accessControl = accessControl;	
		}
	}
	/**
	 * 设置默认值
	 * @param defaultValue 默认值
	 */
	public void setDefaultValue(String defaultValue) {
		if (defaultValue != null) {
			defaultValue = defaultValue.intern();
		}
		this.defaultValue = defaultValue;
	}
	
	/**
	 * 取得默认值
	 * @return 默认值
	 */
	public String getDefaultValue() {
		return this.defaultValue;
	}
	
	/**
	 * 获取精度
	 * @return 精度
	 */
	public int getScale() {
		return this.scale;
	}

	/**
	 * 设置精度
	 * @param scale 精度
	 */
	public void setScale(int scale) {
		this.scale = scale;
	}
	
	public void setScale(Integer scale) {
		if(scale != null) {
			this.scale = scale;
		}
	}
	
	public boolean isPrimary() {
		return isPrimary;
	}

	public void setPrimary(boolean isPrimary) {
		this.isPrimary = isPrimary;
	}
	
	public void setPrimary(Boolean isPrimary) {
		if(isPrimary != null) {
			this.isPrimary = isPrimary;
		}
	}
	public String getCodeColumnKey() {
		return codeColumnKey;
	}

	public void setCodeColumnKey(String codeColumnKey) {
		if (codeColumnKey != null) {
			codeColumnKey = codeColumnKey.intern();
		}
		this.codeColumnKey = codeColumnKey;
	}

	@Override
	public IJSONHandler newHandler(String token) {
		return null;
	}

	@Override
	public void putAttr(String name, String value) {
		if ("user_type".equals(name)) {
			this.userDataType = Integer.parseInt(value);
			this.dataTypeAction = DataTypeActionFactory.getDataType(userDataType);
		} else if ("data_type".equals(name)) {
			this.dataType = Integer.parseInt(value);
		} else if ("key".equals(name)) {
			this.columnKey = value;
		} else if ("index".equals(name)) {
			this.columnIndex = Integer.parseInt(value);
		} else if ("accesscontrol".equals(name)) {
			this.accessControl = Boolean.parseBoolean(value);
		}
	}

	@Override
	public void endHandler() {

	}

	@Override
	public void endChildHandler(IJSONHandler handler) {

	}

	public boolean isHasReadRights() {
		return hasReadRights;
	}

	public void setHasReadRights(boolean hasReadRights) {
		this.hasReadRights = hasReadRights;
	}

	public boolean isHasWriteRights() {
		return hasWriteRights;
	}

	public void setHasWriteRights(boolean hasWriteRights) {
		this.hasWriteRights = hasWriteRights;
	}

	public int getLength() {
		return length;
	}

	public void setLength(int length) {
		this.length = length;
	}
	
	public void setLength(Integer length) {
		if(length != null) {
			this.length = length;
		}
	}
	
	public int getCaseType() {
		return caseType;
	}
	
	public void setCaseType(Integer caseType) {
		if(caseType != null) {
			this.caseType = caseType;
		}
	}
	
	public void setIgnoreChangeState(boolean b) {
		this.ignoreChangeState = b;
	}
	
	public boolean getIgnoreChangeState() {
		return this.ignoreChangeState;
	}
}
