package com.bokesoft.yigo.struct.rights;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.bokesoft.yes.common.json.JSONHelper;
import com.bokesoft.yes.common.json.SerializationException;
import com.bokesoft.yigo.common.json.JSONSerializable;

/**
 * 字典数据权限
 * 
 * @author zhufw
 *
 */
public class DictRights implements JSONSerializable {
	/**
	 * 有权限的字典id
	 */
	private List<Long> ids = new ArrayList<Long>();

	private List<Long> readRightIds = new ArrayList<Long>();

	private List<Long> writeRightIds = new ArrayList<Long>();

	private String itemKey = null;

	private boolean hasAllRights = false;
	
	private boolean hasEmptRights = false;

	public DictRights() {

	}

	public DictRights(String itemKey) {
		this.itemKey = itemKey;
	}

	/**
	 * 字典节点是否有权限
	 * 
	 * @param oid 字典oid
	 * @return 是否有权限
	 */
	public boolean hasRights(long oid) {

		if (hasAllRights) {
			return true;
		} else if (oid == 0) {
			return true;
		}
		return ids.contains(oid);
	}

	public boolean hasRead(long oid) {
		if (hasAllRights) {
			return true;
		} else if (oid == 0) {
			return true;
		}
		return readRightIds.contains(oid);
	}

	public boolean hasWrite(long oid) {
		if (hasAllRights) {
			return true;
		} else if (oid == 0) {
			return true;
		}
		return writeRightIds.contains(oid);
	}
	public List<Long> getIDs() {
		return ids;
	}

	public List<Long> getReadRightIds() {
		return readRightIds;
	}
	public List<Long> getWriteRightIds() {
		return writeRightIds;
	}
	public int size() {
		return ids.size();
	}

	/**
	 * 转换为json对象
	 * 
	 * @return json对象
	 */
	@Override
	public JSONObject toJSON() throws SerializationException, JSONException {
		JSONObject jsonObj = new JSONObject();
		JSONHelper.writeToJSON(jsonObj, RightsJSONConstants.DICTRIGHTS_ITEMKEY,
				itemKey, "");

		jsonObj.put(RightsJSONConstants.DICTRIGHTS_EMPTRIGHTS, hasEmptRights);
		if (this.hasAllRights) {
			jsonObj.put(RightsJSONConstants.DICTRIGHTS_ALLRIGHTS, true);
		} else {
			jsonObj.put(RightsJSONConstants.DICTRIGHTS_ALLRIGHTS, false);
			// 循环put 性能高于 直接构造函数创建
			// JSONArray idsArray = new JSONArray(ids);
			JSONArray idsArray = new JSONArray();
			for (long id : ids) {
				idsArray.put(id);
			}

			jsonObj.put(RightsJSONConstants.DICTRIGHTS_IDS, idsArray);
		}
		return jsonObj;

	}

	/**
	 * 从JSON对象获得对应的属性
	 * 
	 * @param jsonObj
	 *            json对象
	 * @throws SerializationException 序列化异常
	 */
	@Override
	public void fromJSON(JSONObject jsonObj) throws SerializationException,
			JSONException {
		itemKey = JSONHelper.readFromJSON(jsonObj,
				RightsJSONConstants.DICTRIGHTS_ITEMKEY, "");

		hasEmptRights = jsonObj.getBoolean(RightsJSONConstants.DICTRIGHTS_EMPTRIGHTS);
		
		hasAllRights = jsonObj
				.getBoolean(RightsJSONConstants.DICTRIGHTS_ALLRIGHTS);
		if (!hasAllRights) {
			JSONArray idsArray = jsonObj
					.getJSONArray(RightsJSONConstants.DICTRIGHTS_IDS);
			ids.clear();
			for (int i = 0; i < idsArray.length(); i++) {
				long id = idsArray.getLong(i);
				ids.add(id);
			}
		}

	}

	public void addWriteRightsDictID(long dictID) {

		if (hasAllRights) {
			return;
		}
		if (!writeRightIds.contains(dictID)) {
			writeRightIds.add(dictID);
		}
	}

	/**
	 * 添加有权限的字典节点
	 *
	 * @param dictID 有权限的字典oid
	 */
	public void addDictID(long dictID) {

		if (hasAllRights) {
			return;
		}

		if (!ids.contains(dictID)) {
			ids.add(dictID);
		}
	}

	public void addReadRightsDictID(long dictID) {

		if (hasAllRights) {
			return;
		}
		if (!readRightIds.contains(dictID)) {
			readRightIds.add(dictID);
		}
	}

	public void addDictIDs(Collection<Long> collection) {

		if (hasAllRights) {
			return;
		}

		this.ids.removeAll(collection);
		this.ids.addAll(collection);
	}

	public void addReadRightsDictIDs(Collection<Long> collection) {

		if (hasAllRights) {
			return;
		}

		this.readRightIds.removeAll(collection);
		this.readRightIds.addAll(collection);
	}

	public void addWriteRightsDictIDs(Collection<Long> collection) {

		if (hasAllRights) {
			return;
		}

		this.writeRightIds.removeAll(collection);
		this.writeRightIds.addAll(collection);
	}
	/**
	 * 设置有该字典的所有权限
	 * 
	 * @param b 是否有所有权限
	 */
	public void setAllRights(boolean b) {
		this.hasAllRights = b;
	}

	/**
	 * 是否有该字典的所有权限
	 * 
	 * @return 是否有所有权限
	 */
	public boolean hasAllRights() {
		return this.hasAllRights;
	}
	
	/**
	 * 是否为空值权限
	 * 
	 * @return 是否空值权限
	 */
	public boolean hasEmptRights() {
		return hasEmptRights;
	}

	/**
	 * 设置空值权限 
	 * 
	 * @param b 空值权限
	 */
	public void setEmptRights(boolean b) {
		this.hasEmptRights = b;
	}

	/**
	 * 返回字典标识
	 * @return 字典标识
	 */
	public String getItemKey(){
		return this.itemKey;
	}
	
	/**
	 * 数据权限合并
	 * @param dr 数据权限
	 */
	public void merge(DictRights dr) {
		
		if(dr == null){
			return;
		}
		
		if (!this.itemKey.equals(dr.getItemKey())) {
			throw new RuntimeException("不同字典的权限无法合并.");
		}
		
		if(dr.hasEmptRights) {
			this.hasEmptRights = true;
		}
		
		//自身有所有权限　，　则不合并
		if(hasAllRights){
			return ;
		}
		
		if(dr.hasAllRights()){
			this.hasAllRights = true;
			return ;
		}
		
		this.ids.removeAll(dr.getIDs());
		this.ids.addAll(dr.getIDs());
		this.readRightIds.removeAll(dr.getReadRightIds());
		this.readRightIds.addAll(dr.getReadRightIds());
		this.writeRightIds.removeAll(dr.getWriteRightIds());
		this.writeRightIds.addAll(dr.getWriteRightIds());
	}
	
	public DictRights clone(){
		DictRights clone = new DictRights(this.itemKey);
		clone.merge(this);
		return clone;
	}

}
