package com.bokesoft.yigo.struct.expand.work;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.bokesoft.yigo.struct.expand.IExpandSourceField;
import com.bokesoft.yigo.struct.expand.IExpandWork;
import com.bokesoft.yigo.struct.expand.processor.DimensionItem;
import com.bokesoft.yes.common.util.StringUtil;
import com.bokesoft.yigo.common.struct.IPairItem;
import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.struct.datatable.ColumnInfo;
import com.bokesoft.yigo.struct.datatable.DataTable;
import com.bokesoft.yigo.struct.document.Document;
import com.bokesoft.yigo.struct.expand.IExpandDimensionField;

/**
 * 对Docuemnt中需要扩展的DataTable进行扩展处理,生成新的扩展DataTable
 * 
 * @author chenbb
 *
 * @param <S>
 * @param <F>
 */
public class DataTableExpandWork<S extends IExpandSourceField<F>, F extends IExpandDimensionField> implements IExpandWork<S, F> {
	
	private Document document = null;
	
	private DataTable baseDataTable = null;
	
	public DataTableExpandWork(DataTable baseDataTable, Document document) {
		this.document = document;
		this.baseDataTable = baseDataTable;
	}

	@Override
	public void doWork(List<String> fixedFieldKeys, List<DimensionItem<IPairItem, S, F>> colRootDimensionItems,
			List<DimensionItem<IPairItem, S, F>> rowRootDimensionItems) {
		String targetKey = getExpandTargetKey(baseDataTable.getKey());
		DataTable targetDataTable = createTargetTableStruct(colRootDimensionItems, fixedFieldKeys, targetKey, baseDataTable);
		fillExpandData(baseDataTable, fixedFieldKeys, colRootDimensionItems, targetDataTable);
		this.document.remove(targetKey);
		this.document.add(targetKey, targetDataTable);
	}

	/**
	 * 填入扩展数据
	 * @param sourceDataTable
	 * @param fixedFieldKeys
	 * @param colRootDimensionItems
	 * @param targetDataTable
	 */
	private void fillExpandData(DataTable sourceDataTable, List<String> fixedFieldKeys,
			List<DimensionItem<IPairItem, S, F>> colRootDimensionItems, DataTable targetDataTable) {
		HashMap<String, Object> fixedValues = new HashMap<>();
		initFixedValues(fixedValues, fixedFieldKeys);
		
		String fixValuesKey = null;
		HashMap<String, Integer> recorder = new HashMap<String, Integer>();
		sourceDataTable.beforeFirst();
		while (sourceDataTable.next()) {
			// 将未扩展无关的数据先行处理, 数据发生变化时触发新增数据行
			if (isFixedValueChanged(fixedValues, sourceDataTable)) {
				fixValuesKey = getFixValuesKey(fixedFieldKeys, sourceDataTable);
				if (recorder.containsKey(fixValuesKey)) {
					int rowIndex = recorder.get(fixValuesKey);
					targetDataTable.setPos(rowIndex);
				} else {
					Object fixedValue = null;
					int rowIndex = targetDataTable.append();
					for (String fixedFieldKey : fixedFieldKeys) {
						fixedValue = sourceDataTable.getObject(fixedFieldKey);
						targetDataTable.setObject(fixedFieldKey, fixedValue);
					}
					recorder.put(fixValuesKey, rowIndex);
				}
				resetFixedValues(fixedValues, sourceDataTable);
			}
			
			//创建目标中的动态扩展列
			Object curTargetValue;
			for (DimensionItem<IPairItem, S, F> expandRoot : colRootDimensionItems) {
				for (DimensionItem<IPairItem, S, F> item : expandRoot.getAllLeafs()) {
					if (isMatchedItem(sourceDataTable, item)) {
						String expandFieldKey = item.getPathKey();
						String sourceFieldKey = item.getSourceField().getKey();
						curTargetValue = targetDataTable.getObject(expandFieldKey);
						// 若扩展数据源中的值依然是初始值，那么说明没有进行过赋值，那么进行赋值(目前只考虑了 数值类型为0, 以及字符类型为 空的情况)
						boolean canSetValue = StringUtil.isNumeric(curTargetValue) && TypeConvertor.toInteger(curTargetValue) == 0;
						canSetValue = canSetValue | StringUtil.isBlankOrNull(curTargetValue);
						if (canSetValue) {
							targetDataTable.setObject(expandFieldKey, sourceDataTable.getObject(sourceFieldKey));
						}
						break;
					}
				}
			}
		}
	}
	
	/**
	 * 判断当前扩展项，是否适合当前数据行
	 * @param dataTable
	 * @param leafItem
	 * @return
	 */
	private boolean isMatchedItem(DataTable dataTable, DimensionItem<IPairItem, S, F> leafItem) {
		if (leafItem == null) {
			return true;
		}
		
		F dimenField = leafItem.getDimensionField();
		if (dimenField == null) {
			return true;
		}
		
		ColumnInfo columnInfo = dataTable.getMetaData().getColumnInfo(dimenField.getKey());
		Object dimenValue = dataTable.getObject(dimenField.getKey());
		Object itemValue = leafItem.getData().getValue();
		Object leafItemValue = itemValue == null ? null :TypeConvertor.toDataType(columnInfo.getDataType(), itemValue);
		if (!isEqual(leafItemValue, dimenValue)) {
			return false;
		} else {
			return isMatchedItem(dataTable, leafItem.getParent());
		}
	}
	
	/**
	 * 创建扩展目标表结构
	 * 
	 * @param colRootDimensionItems
	 * @param targetKey
	 * @param sourceDataTable
	 * @return
	 */
	private DataTable createTargetTableStruct(List<DimensionItem<IPairItem, S, F>> colRootDimensionItems, List<String> fixedFieldKeys, String targetKey, DataTable sourceDataTable) {
		DataTable targetDataTable = new DataTable();
		targetDataTable.setKey(targetKey);
		
		//创建目标中的固定列
		ColumnInfo tmpColumn = null;
		for (String fixFieldKey : fixedFieldKeys) {
			tmpColumn = sourceDataTable.getMetaData().getColumnInfo(fixFieldKey);
			targetDataTable.addColumn(tmpColumn.deepClone());
		}

		//创建目标中的动态扩展列
		for (DimensionItem<IPairItem, S, F> expandRoot : colRootDimensionItems) {
			calcTargetTableColumn(expandRoot, targetDataTable, sourceDataTable);
		}
		return targetDataTable;
	}
	
	/**
	 * 计算扩展列
	 * 
	 * @param expandRoot
	 * @param targetDataTable
	 */
	private void calcTargetTableColumn(DimensionItem<IPairItem, S, F> expandRoot, DataTable targetDataTable, DataTable sourceDataTable) {
		S sourceField = expandRoot.getSourceField();
		List<DimensionItem<IPairItem, S, F>> allLeafs = expandRoot.getAllLeafs();
		ColumnInfo expandSourceColumn = sourceDataTable.getMetaData().getColumnInfo(sourceField.getKey());
		for (DimensionItem<IPairItem, S, F> item : allLeafs) {
			String expandKey = item.getPathKey();
			ColumnInfo columnInfo = new ColumnInfo(expandKey, expandSourceColumn.getDataType());
			targetDataTable.addColumn(columnInfo);
		}
	}
	
	/**
	 * 判断当前的固定值记录是否发生了变化， 发生变化时候需要在目标中新增1行
	 * @param fixedValues
	 * @param dataTable
	 * @return
	 */
	private boolean isFixedValueChanged(HashMap<String, Object> fixedValues, DataTable dataTable) {
		String fixedFieldKey;
		Object curFixedFieldValue, nextFixedFieldValue;
		for (Map.Entry<String, Object> entry : fixedValues.entrySet()) {
			fixedFieldKey = entry.getKey();
			curFixedFieldValue = entry.getValue();
			nextFixedFieldValue = dataTable.getObject(fixedFieldKey);
			if (!isEqual(curFixedFieldValue, nextFixedFieldValue)) {
				return true;
			}
		}
		return false;
	}
	
	
	/**
	 * 初始化扩展无关字段的临时记录
	 * @param fixedFieldKeys
	 */
	private void initFixedValues(HashMap<String, Object> fixedValues, List<String> fixedFieldKeys) {
		for (String fixedFieldKey : fixedFieldKeys) {
			fixedValues.put(fixedFieldKey, null);
		}
	}
	
	/**
	 * 更新扩展无关字段临时记录值
	 * @param fixedValues
	 * @param sourceDataTable
	 */
	private void resetFixedValues(HashMap<String, Object> fixedValues, DataTable sourceDataTable) {
		Object fixedValue = null;
		for (String fixedFieldKey : fixedValues.keySet()) {
			fixedValue = sourceDataTable.getObject(fixedFieldKey);
			fixedValues.put(fixedFieldKey, fixedValue);
		}
	}
	
	private String getFixValuesKey(List<String> fixedFieldKeys, DataTable dataTable) {
		StringBuilder sb = new StringBuilder();
		Object fixedValue = null;
		for (String fixedFieldKey : fixedFieldKeys) {
			fixedValue = dataTable.getObject(fixedFieldKey);
			sb.append(fixedValue).append("__");
		}
		return sb.toString();
	}
	
	
	/**
	 * 简易判断两个值是否相等
	 * 
	 * @param obj1
	 * @param obj2
	 * @return
	 */
	private boolean isEqual(Object obj1, Object obj2) {
		if (obj1 == null && obj2 == null) return true;
		
		if (obj1 == null || obj2 == null) return false;
		
		return obj1.equals(obj2);
	}
}