package com.bokesoft.yigo.struct.document;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.bokesoft.yes.common.json.JSONHelper;
import com.bokesoft.yes.common.struct.StringHashMap;
import com.bokesoft.yes.struct.datatable.json.DataTableListImpl;
import com.bokesoft.yes.struct.datatable.json.ExpandDataClassImpl;
import com.bokesoft.yes.struct.datatable.json.ExpandDataImpl;
import com.bokesoft.yes.struct.datatable.json.ExpandDataTypeImpl;
import com.bokesoft.yes.struct.document.DocumentJSONConstants;
import com.bokesoft.yes.struct.document.ExpandDataType;
import com.bokesoft.yes.struct.document.IJSONHandler;
import com.bokesoft.yigo.common.def.SystemField;
import com.bokesoft.yigo.common.def.TableMode;
import com.bokesoft.yigo.common.json.JSONSerializable;
import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.meta.dataobject.MetaColumn;
import com.bokesoft.yigo.meta.dataobject.MetaDataObject;
import com.bokesoft.yigo.meta.dataobject.MetaTable;
import com.bokesoft.yigo.meta.dataobject.MetaTableCollection;
import com.bokesoft.yigo.struct.datatable.DataTable;
import com.bokesoft.yigo.struct.datatable.DataTableMetaData;
import com.bokesoft.yigo.struct.document.track.DocumentTrack;
import com.bokesoft.yigo.struct.exception.StructException;

/**
 * 抽象文档（Document）提供描述业务数据的服务。抽象文档包括表头和若干张明细表的数据，其是以一个数据表（DataTable）的数组来表示的。
 * 其中第一个数据表就是表头的数据
 * ，其它的数据表每一个对应一个明细。每一个数据表都有有一个KEY对应，在抽象文档中添加一个数据表时就要设置。例如，创建一个抽象文档
 * ，在其中添加一个KEY为AA的数据表：
 * 
 * <pre>
 * Document doc = new Document(1);
 * doc.appendDataTable(&quot;AA&quot;, new DataTable());
 * </pre>
 * 
 * 可以通过KEY或者序号为参数从抽象文档中获取指定的数据表对象，然后可以使用这个对象进行相关操作(数据表对象的操作参见DataTable）。
 * 
 * @author 刘翔翔
 * @see DataTable
 */
public class Document implements JSONSerializable, IJSONHandler {

	public static final int NORMAL = 0;

	public static final int NEW = 1;

	public static final int MODIFIED = 2;

	public static final int DELETE = 3;

	/** 数据对象的标识，程序内部使用，不要在代码中使用 */
	private String objectKey = "";

	/** 抽象文档的唯一标号 */
	private Long OID = -1l;

	/** 父数据节点编号 */
	private Long POID = null;

	/** 同步管理用到的版本ID */
	private Integer VERID = 0;

	/** 标记明细是否更改的明细版本ID */
	private Integer DVERID = 0;

	/** 文档对象的状态 */
	private int state = NORMAL;

	/** 文档的类型 */
	private int documentType = DocumentType.DATAOBJECT;

	/** 文档中的数据 */
	private ArrayList<DataTable> tableList = new ArrayList<DataTable>();

	/** 表的标识与表的映射 */
	private StringHashMap<DataTable> dataTableMap = StringHashMap.newInstance();
	
	/** 影子表的映射*/
	private StringHashMap<DataTable> shadowTableMap = StringHashMap.newInstance();

	/** 自定义的扩展数据 **/
	private StringHashMap<Object> expandData = StringHashMap.newInstance();

	/** 文档元数据对象 */
	private MetaDataObject metaDataObject = null;
	
	/** 是否需要json影子表到后台*/
	private boolean withShadowTable = false;

	/** 文档的执行轨迹 */
	private DocumentTrack documentTrack = null;

	/**
	 * 文档构造函数
	 * 
	 * @param metaDataObject
	 *            文档元数据对象
	 * @param OID
	 *            文档对象OID
	 */
	public Document(MetaDataObject metaDataObject, long OID) {
		this.metaDataObject = metaDataObject;
		if (metaDataObject != null) {
			this.objectKey = metaDataObject.getKey();
		}
		this.OID = OID;
	}

	/**
	 * 载入数据之前清掉tableList以及dataTableMap的内容
	 */
	public void clear() {
		tableList.clear();
		dataTableMap.clear();
		shadowTableMap.clear();
	}

	/**
	 * 添加一张数据表
	 * 
	 * @param key
	 *            表的标识
	 * @param table
	 *            数据表
	 */
	public void add(String key, DataTable table) {
		tableList.add(table);
		table.setKey(key);
		dataTableMap.put(key, table);
	}

	/**
	 * 删除一张数据表
	 * 
	 * @param key
	 *            表的标识
	 */
	public void remove(String key) {
		DataTable table = dataTableMap.get(key);
		tableList.remove(table);
		dataTableMap.remove(key);
	}

	/**
	 * 获取通过标识指定的数据表对象
	 * 
	 * @param key
	 *            表的标识
	 * @return 指定的数据表对象
	 */
	public DataTable get(String key) {
		return dataTableMap.get(key);
	}
	
	/**
	 * 放入一个数据表对应的影子表
	 * @param key 表的标识
	 * @param shadowTable 影子表
	 */
	public void addShadowTable(String key,DataTable shadowTable) {
		shadowTableMap.put(key, shadowTable);
	}
	
	/**
	 * 获取数据表对应的影子表
	 * @param key 表的标识
	 * @return 表对象
	 */
	public DataTable getShadowTable(String key) {
		return shadowTableMap.get(key);
	}
	
	/**
	 * 获取所有影子表标识
	 * @return 所有影子表标识
	 */
	public Set<String> shadowKeys() {
		return this.shadowTableMap.keySet();
	}
	
	/**
	 * 清空所有的影子表数据
	 * 
	 */
	public void clearAllShadow() {
		shadowTableMap.clear();
	}

	/**
	 * 设置key所在的表，如果key在Document存在，则替换，否则增加
	 * 
	 * @param key
	 *            表标识
	 * @param table
	 *            数据表对象
	 */
	public void put(String key, DataTable table) {
		if (this.dataTableMap.containsKey(key)) {
			DataTable oldTable = this.dataTableMap.get(key);
			for (int i = 0; i < this.tableList.size(); ++i) {
				DataTable tmpTable = this.tableList.get(i);
				if (tmpTable == oldTable) {
					this.tableList.set(i, table);
					break;
				}
			}
			this.dataTableMap.put(key, table);
		} else {
			this.add(key, table);
		}
	}

	/**
	 * 取得表的数量
	 * 
	 * @return 表的数量
	 */
	public int size() {
		return this.tableList.size();
	}

	/**
	 * 获取通过序号指定的数据表对象
	 * 
	 * @param index
	 *            序号
	 * @return 通过序号指定的数据表对象
	 */
	public DataTable get(int index) {
		return tableList.get(index);
	}

	public void reSetIDValue(DataTable head) {
		if (metaDataObject != null && OID != -1 && head != null && head.first()) {
			DataTableMetaData meta = head.getMetaData();
			if (meta.constains(SystemField.POID_SYS_KEY)) {
				POID = head.getLong(0, SystemField.POID_SYS_KEY);
			}
			if (meta.constains(SystemField.VERID_SYS_KEY)) {
				VERID = head.getInt(0, SystemField.VERID_SYS_KEY);
			}
			if (meta.constains(SystemField.DVERID_SYS_KEY)) {
				DVERID = head.getInt(0, SystemField.DVERID_SYS_KEY);				
			}
		}
	}

	public void reSetIDValue() throws StructException {
		if (metaDataObject != null) {
			MetaTable table = metaDataObject.getMainTable();
			DataTable head = get(table.getKey());
			reSetIDValue(head);
		}
	}

	/**
	 * @return 同步管理用到的版本ID
	 */
	public int getVERID() {
		return VERID;
	}

	/**
	 * 设置同步管理用到的版本ID
	 * 
	 * @param VERID
	 *            同步管理用到的版本ID
	 */
	public void setVERID(int VERID) {
		this.VERID = VERID;
	}

	/**
	 * @return 标记明细是否更改的明细版本ID
	 */
	public int getDVERID() {
		return DVERID;
	}

	/**
	 * 设置标记明细是否更改的明细版本ID
	 * 
	 * @param DVERID
	 *            标记明细是否更改的明细版本ID
	 * 
	 */
	public void setDVERID(int DVERID) {
		this.DVERID = DVERID;
	}

	/**
	 * 设置抽象文档的唯一标识
	 * 
	 * @param OID
	 *            文档的唯一标识
	 */
	public void setOID(long OID) {
		this.OID = OID;
	}

	/**
	 * @return 抽象文档的唯一标识
	 */
	public long getOID() {
		return OID;
	}

	/**
	 * 设置数据对象标识，不应在二次开发中使用
	 * 
	 * @param objectKey
	 *            数据对象标识
	 */
	@Deprecated
	public void setObjectKey(String objectKey) {
		this.objectKey = objectKey;
	}

	/**
	 * 返回数据对象标识，不应在二次开发中使用
	 * 
	 * @return 数据对象标识
	 */
	@Deprecated
	public String getObjectKey() {
		return this.objectKey;
	}

	/**
	 * 设置抽象文档的父数据节点编号
	 * 
	 * @param POID
	 *            父数据的唯一标识
	 */
	public void setPOID(Long POID) {
		this.POID = POID;
	}

	/**
	 * 取得父数据对象的唯一标识
	 * 
	 * @return 父数据对象的唯一标识
	 */
	public Long getPOID() {
		return POID;
	}

	/**
	 * 当前抽象文档是否为新增
	 * 
	 * @return 如果文档状态为新增则返回true，否则返回false
	 */
	public boolean isNew() {
		return state == NEW;
	}

	/**
	 * 设置文档为新增状态
	 */
	public void setNew() {
		this.state = NEW;
	}

	/**
	 * 设置文档为普通状态
	 */
	public void setNormal() {
		this.state = NORMAL;
	}

	/**
	 * 设置文档为修改状态
	 */
	public void setModified() {
		if (state == NORMAL)
			state = MODIFIED;
	}

	/**
	 * 取得文档是否为修改状态
	 * 
	 * @return 如果文档为修改状态则返回true，否则返回false
	 */
	public boolean isModified() {
		return state == MODIFIED;
	}

	/**
	 * 设置文档为删除状态
	 */
	public void setDelete() {
		state = DELETE;
	}

	/**
	 * 取得文档是否为修改状态
	 * 
	 * @return 如果文档为修改状态则返回true，否则返回false
	 */
	public boolean isDelete() {
		return state == DELETE;
	}

	/**
	 * 取得文档的状态
	 * 
	 * @return 文档状态
	 * @see #NEW
	 * @see #NORMAL
	 * @see #MODIFIED
	 */
	public int getState() {
		return this.state;
	}

	/**
	 * 取得文档的类型
	 * 
	 * @return 文档的类型，取值为{@link DocumentType}
	 */
	public int getDocumentType() {
		return documentType;
	}

	/**
	 * 设置文档的类型
	 * 
	 * @param documentType
	 *            文档的类型，取值为{@link DocumentType}
	 */
	public void setDocumentType(int documentType) {
		this.documentType = documentType;
	}

	/**
	 * 返回文档所有表集合的迭代器
	 * 
	 * @return 表集合迭代器
	 */
	public Iterator<DataTable> iterator() {
		return this.tableList.iterator();
	}

	/**
	 * 获取所有的拓展数据
	 * @return 所有的拓展数据
	 */
	public StringHashMap<Object> getAllExpandData(){
		return this.expandData;
	}
	
	/**
	 * 给定KEY，添加一个自定义扩展数据
	 * 
	 * @param key
	 *            扩展数据标识
	 * @param value
	 *            扩展数据值
	 */
	public void putExpandData(String key, Object value) {
		expandData.put(key, value);
	}

	/**
	 * 根据给定的KEY，返回一个自定义扩展数据对象
	 * 
	 * @param key
	 *            标识
	 * @return 如果存在key指定的扩展数据则返回其值，否则返回null
	 */
	public Object getExpandData(String key) {
		return expandData.get(key);
	}

	/**
	 * 移除给定key的扩展数据
	 * 
	 * @param key
	 *            扩展数据标识
	 */
	public void removeExpandData(String key) {
		expandData.remove(key);
	}

	/**
	 * 获取当前抽象文档的轨迹对象，如果对象为空则创建一个
	 * 
	 * @return 文档轨迹对象
	 */
	public DocumentTrack getDocumentTrack() {
		if (documentTrack == null)
			documentTrack = new DocumentTrack();
		return documentTrack;
	}

	/**
	 * 设置文档元数据对象
	 * 
	 * @param metaDataObject
	 *            文档元数据对象
	 */
	public void setMetaDataObject(MetaDataObject metaDataObject) {
		this.metaDataObject = metaDataObject;
		if (metaDataObject != null) {
			this.objectKey = metaDataObject.getKey();
		}
	}

	/**
	 * 取得文档元数据对象
	 * 
	 * @return 文档元数据对象
	 */
	public MetaDataObject getMetaDataObject() {
		return this.metaDataObject;
	}

	/**
	 * 是否需要序列化影子表到服务端
	 * @return 标志
	 */
	public boolean withShadow() {
		return withShadowTable;
	}

	/**
	 * 设置是否需要序列化影子表到服务端
	 * @param withShadowTable 是否序列化影子表
	 */
	public void setWithShadow(boolean withShadowTable) {
		this.withShadowTable = withShadowTable;
	}

	public String toString() {
		StringBuilder sb = new StringBuilder("<Document OID=\"");
		sb.append(OID);
		sb.append("\" POID=\"");
		sb.append(POID);
		sb.append("\" VERID=\"");
		sb.append(VERID);
		sb.append("\" DVERID=\"");
		sb.append(DVERID);
		sb.append("\" isNew=\"");
		sb.append(isNew());
		sb.append("\" documentType=\"");
		sb.append(DocumentType.toString(documentType));
		sb.append("\">\n");

		for (DataTable table : tableList)
			table.appendString(sb, true);

		sb.append("</Document>\n");
		return sb.toString();

	}

	public JSONObject headTableToJSON() throws Throwable {
		if (this.metaDataObject == null)
			return null;

		JSONObject jsonObj = new JSONObject();
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_OBJECTKEY, objectKey, "");
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_OID, OID, -1l);
		if (POID != null)
			JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_POID, POID, -1l);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_VERID, VERID, 0);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_DVERID, DVERID, 0);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_STATE, state, NORMAL);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_TYPE, documentType, DocumentType.DATAOBJECT);
		JSONArray tableArray = new JSONArray();

		Iterator<Entry<String, DataTable>> it = dataTableMap.entrySet().iterator();
		Entry<String, DataTable> entry = null;

		while (it.hasNext()) {
			entry = it.next();
			String key = entry.getKey();
			if (metaDataObject.getTable(key).getTableMode() != TableMode.HEAD)
				continue;
			DataTable dt = entry.getValue();
			JSONObject dataTableObj = dt.toJSON();
			JSONHelper.writeToJSON(dataTableObj, DocumentJSONConstants.DATATABLE_TABLEKEY, key, "");
			tableArray.put(dataTableObj);
		}

		jsonObj.put(DocumentJSONConstants.DOCUMENT_TABLELIST, tableArray);
		return jsonObj;
	}
	
	public JSONObject toJSON(boolean headOnly) throws Throwable {
		return this.toJSONImpl(headOnly);
	}
	
	public JSONObject toJSON() throws Throwable {
		return this.toJSONImpl(false);
	}

	/**
	 * 转换为json对象
	 * 
	 * @return json对象
	 * @throws JSONException JSON异常
	 */
	private JSONObject toJSONImpl(boolean headOnly) throws Throwable {
		JSONObject jsonObj = new JSONObject();
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_OBJECTKEY, objectKey, "");
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_OID, OID, -1l);
		if (POID != null)
			JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_POID, POID, -1l);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_VERID, VERID, 0);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_DVERID, DVERID, 0);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_STATE, state, NORMAL);
		JSONHelper.writeToJSON(jsonObj, DocumentJSONConstants.DOCUMENT_TYPE, documentType, DocumentType.DATAOBJECT);
		
		if (this.metaDataObject != null) {
			jsonObj.put(DocumentJSONConstants.DOCUMENT_VERSION, metaDataObject.getVersion());
			jsonObj.put("mainTableKey", this.metaDataObject.getMainTableKey());
		}
		JSONArray tableArray = new JSONArray();

		Iterator<Entry<String, DataTable>> it = dataTableMap.entrySet().iterator();
		Entry<String, DataTable> entry = null;

		// 如果需要序列化影子表且对应有影子表,那么源表不用序列化,否则序列化源表
		while ( it.hasNext() ) {
			entry = it.next();
			String key = entry.getKey();
			
			if( headOnly ) { // 只序列化头表
				String mainTableKey = metaDataObject == null ? "" : metaDataObject.getMainTableKey();
				if( !mainTableKey.equals(key) ) {
					continue;
				}
			}
			
			DataTable srcTable = entry.getValue();
			DataTable shadowTable = shadowTableMap.get(key);
			DataTable jsonTable = withShadowTable ? (shadowTable != null ? shadowTable : srcTable) : srcTable;
			setDefaultValue(key, jsonTable);
						
			JSONObject tableJSON = jsonTable.toJSON();
			JSONHelper.writeToJSON(tableJSON, DocumentJSONConstants.DATATABLE_TABLEKEY, key, "");
			if (metaDataObject != null) {
				MetaTable metaTable = metaDataObject.getMetaTable(key);
				if( metaTable != null ) {
					tableJSON.put("parentKey", metaTable.getParentKey());
					tableJSON.put("isT", metaTable.isT());
					tableJSON.put("tableMode", metaTable.getTableMode());
					tableJSON.put("isPersist", metaTable.isPersist());
				}
			}
			tableArray.put(tableJSON);
		}

		// 数据表
		jsonObj.put(DocumentJSONConstants.DOCUMENT_TABLELIST, tableArray);

		// 扩展数据
		JSONObject jsonExpandData = new JSONObject();
		jsonObj.put(DocumentJSONConstants.DOCUMENT_EXPAND_DATA, jsonExpandData);
		JSONObject jsonExpandDataType = new JSONObject();
		jsonObj.put(DocumentJSONConstants.DOCUMENT_EXPAND_DATATYPE, jsonExpandDataType);
		JSONObject jsonExpandDataClass = new JSONObject();
		jsonObj.put(DocumentJSONConstants.DOCUMENT_EXPAND_CLASS, jsonExpandDataClass);

		for (Entry<String, Object> e : expandData.entrySet()) {
			String key = e.getKey();
			Object value = e.getValue();
			if (value instanceof String) {
				jsonExpandDataType.put(key, ExpandDataType.STRING);
				jsonExpandData.put(key, value);
			} else if (value instanceof Integer) {
				jsonExpandDataType.put(key, ExpandDataType.INT);
				jsonExpandData.put(key, value);
			} else if (value instanceof Long) {
				jsonExpandDataType.put(key, ExpandDataType.LONG);
				jsonExpandData.put(key, value);
			} else if (value instanceof Map){
				jsonExpandDataType.put(key, ExpandDataType.MAP);
				JSONObject o = new JSONObject((Map<?,?>) value);
				jsonExpandData.put(key, o);
			} else if (value instanceof JSONObject){
				jsonExpandDataType.put(key, ExpandDataType.JSONOBJ);
				jsonExpandData.put(key, value);
			} else if (value instanceof JSONSerializable) {
				jsonExpandDataType.put(key, ExpandDataType.JSON_OBJECT);
				JSONSerializable js = (JSONSerializable) value;
				jsonExpandData.put(key, js.toJSON());
				jsonExpandDataClass.put(key, value.getClass().getName());
			}
		}
		jsonObj.put("isDoc", true);
		return jsonObj;
	}

	/**
	 * 设置列的默认值
	 * @param key
	 * @param dataTable
	 */
	private void setDefaultValue(String key, DataTable dataTable){
		if(this.metaDataObject == null){
			return;
		}
		
		MetaTable metaTable = this.metaDataObject.getMetaTable(key);
		if(metaTable == null){
			return;
		}
		
		Iterator<MetaColumn> itColumn = metaTable.iterator();
		int index = 0;
		String defaultValue = null;

		// 临时纠错方案，因为客户端传到中间层的DataTable会少一些后期MetaTable处理加入的列，所以
		// 如果超出DataTable列数的时候直接跳出处理
		DataTableMetaData tableMetaData = dataTable.getMetaData();
		int columnCount = tableMetaData.getColumnCount();
		while ( itColumn.hasNext() ) {
			MetaColumn metaColumn = itColumn.next();
			
			if ( index >= columnCount ) {
				break;
			}
			defaultValue = metaColumn.getDefaultValue();
			if(defaultValue != null && !defaultValue.isEmpty()){
				dataTable.getMetaData().getColumnInfo(index).setDefaultValue(defaultValue);
			}
			++index;
		}		
	}
	/**
	 * 从JSON对象获得对应的属性
	 * 
	 * @param jsonObj
	 *            json对象
	 * @throws Throwable 处理异常
	 */
	public void fromJSON(JSONObject jsonObj) throws Throwable {
		objectKey = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DOCUMENT_OBJECTKEY, "");
		OID = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DOCUMENT_OID, -1l);
		if (!jsonObj.has(DocumentJSONConstants.DOCUMENT_POID))
			POID = null;
		else
			POID = jsonObj.getLong(DocumentJSONConstants.DOCUMENT_POID);
		VERID = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DOCUMENT_VERID, 0);
		DVERID = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DOCUMENT_DVERID, 0);
		state = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DOCUMENT_STATE, NORMAL);
		documentType = JSONHelper.readFromJSON(jsonObj, DocumentJSONConstants.DOCUMENT_TYPE, DocumentType.DATAOBJECT);
		JSONArray tableArray = jsonObj.getJSONArray(DocumentJSONConstants.DOCUMENT_TABLELIST);
		tableList.clear();
		dataTableMap.clear();
		shadowTableMap.clear();
		for (int i = 0; i < tableArray.length(); i++) {
			JSONObject tableObj = tableArray.getJSONObject(i);
			String tableKey = JSONHelper.readFromJSON(tableObj, DocumentJSONConstants.DATATABLE_TABLEKEY, "");
			DataTable dataTable = new DataTable();
			dataTable.fromJSON(tableObj);
			tableList.add(dataTable);
			dataTableMap.put(tableKey, dataTable);
		}

		if (!jsonObj.isNull(DocumentJSONConstants.DOCUMENT_EXPAND_DATA)) {
			JSONObject jsonExpandData = jsonObj.getJSONObject(DocumentJSONConstants.DOCUMENT_EXPAND_DATA);
			JSONObject jsonExpandDataType = jsonObj.getJSONObject(DocumentJSONConstants.DOCUMENT_EXPAND_DATATYPE);
			JSONObject jsonExpandDataClass = jsonObj.getJSONObject(DocumentJSONConstants.DOCUMENT_EXPAND_CLASS);

			/** 为Android做类库兼容 */
			@SuppressWarnings("rawtypes")
			Iterator it = jsonExpandData.keys();
			while (it.hasNext()) {
				String key = it.next().toString();
				int type = jsonExpandDataType.getInt(key);
				switch (type) {
				case ExpandDataType.INT:
					expandData.put(key, jsonExpandData.getInt(key));
					break;
				case ExpandDataType.LONG:
					expandData.put(key, jsonExpandData.getLong(key));
					break;
				case ExpandDataType.STRING:
					expandData.put(key, jsonExpandData.getString(key));
					break;
				case ExpandDataType.MAP:
					JSONObject json = jsonExpandData.optJSONObject(key);
					Map<String, Object> params = null;
					
					if(json != null){
						params = new HashMap<String, Object>();
						Iterator<?> v = json.keys();
						while (v.hasNext()) {
						    Object o = v.next();
					
					        String k = TypeConvertor.toString(o);
					    	params.put(k,json.opt(k)); 
						}
					}
					expandData.put(key, params);
					break;
				case ExpandDataType.JSONOBJ:
					expandData.put(key, jsonExpandData.optJSONObject(key));
					break;
				case ExpandDataType.JSON_OBJECT:
					Object value = jsonExpandData.optJSONObject(key);
					if (value == null) {
						expandData.put(key, value);
					} else {
						String className = jsonExpandDataClass.getString(key);
						JSONSerializable js = (JSONSerializable) Class.forName(className).newInstance();
						js.fromJSON(jsonExpandData.getJSONObject(key));
						expandData.put(key, js);
					}
					break;
				}
			}
		}

	}

	/**
	 * 批量更新文档中所有表的状态
	 * @throws Throwable 处理异常
	 */
	public void batchUpdate() throws Throwable {
		batchUpdate(false);
	}
	
	/**
	 * 批量更新文档中所有表的状态，内部函数，请勿在二次开发中使用
	 * @param reserveOrgData 是否保留行的原值标志
	 * @throws Throwable 处理异常
	 */
	@Deprecated
	public void batchUpdate(boolean reserveOrgData) throws Throwable {
		for (int i = 0; i < tableList.size(); i++) {
			DataTable table = tableList.get(i);
			table.setShowDeletedFlag(false);
			table.batchUpdate(reserveOrgData);
		}
	}
	
	/**
	 * 重置文档中所有表的原值数据列表为null，内部函数，请勿在二次开发中使用
	 */
	@Deprecated
	public void clearOriginalData() {
		for (int i = 0; i < tableList.size(); i++) {
			DataTable table = tableList.get(i);
			table.clearOriginalData();
		}
	}

	/**
	 * 将所有表移到第一行
	 */
	public void moveFirst() {
		for (int i = 0; i < tableList.size(); i++) {
			DataTable table = tableList.get(i);
			table.first();
		}
	}

	public Document deepClone() {
		Document clone = new Document(metaDataObject, this.OID);
		clone.setObjectKey(objectKey);
		clone.setPOID(this.POID);
		clone.setVERID(this.VERID);
		clone.setDVERID(this.DVERID);

		MetaTableCollection metaTableCollection = metaDataObject.getTableCollection();
		Iterator<MetaTable> it = metaTableCollection.iterator();
		MetaTable metaTable = null;
		int index = 0;
		DataTable table = null;
		while (it.hasNext()) {
			metaTable = it.next();
			table = this.get(index);
			clone.add(metaTable.getKey(), table.deepClone());
			index++;
		}

		clone.state = state;
		clone.setDocumentType(this.documentType);

		String key = null;
		Object value = null;
		for (Entry<String, Object> e : expandData.entrySet()) {
			key = e.getKey();
			value = e.getValue();
			clone.putExpandData(key, value);
		}

		return clone;
	}

	@Override
	public IJSONHandler newHandler(String token) {
		if ("table_list".equals(token)) {
			return new DataTableListImpl(this);
		} else if ("expand_data_class".equals(token)) {
			return new ExpandDataClassImpl();
		} else if ("expand_data".equals(token)) {
			return new ExpandDataImpl();
		} else if ("expand_data_type".equals(token)) {
			return new ExpandDataTypeImpl();
		}
		return this;
	}

	@Override
	public void putAttr(String name, String value) {
		if ("object_key".equals(name)) {
			this.objectKey = value;
		} else if ("state".equals(name)) {
			this.state = Integer.parseInt(value);
		}
	}

	@Override
	public void endHandler() {
		// TODO Auto-generated method stub

	}

	@Override
	public void endChildHandler(IJSONHandler handler) {
		// TODO Auto-generated method stub

	}

	/**
	 * 关闭，用于关闭缓冲
	 */
	public void close() {
		Iterator<DataTable> it = this.tableList.iterator();
		DataTable t = null;
		while (it.hasNext()) {
			t = it.next();
			t.close();
		}
	}
}