import AbstractDomElement from "../common/dom/AbstractDomElement";
import Div from "../common/dom/element/Div";
import IVBarListener from "./IVBarListener";

/**
 * @author chenbb
 * 
 * 纵向滚动条，用于动态载入html元素，向外部提供位置信息，
 * 用于显示和删除html中的可见和不可见元素
 * 
 * 注:仅用于行高固定的情况
 */
export default class VScrollBar extends AbstractDomElement {

    public static CLS_DYNAMIC_SCROLL_BAR = "dynamic-scroll-bar";

    /** 实际内容的容器，此处主要模拟实际高度 */
    private content: Div;
    /** 行高 */
    private rowHeight: number;
    /** 滚动条宽度 */
    private barWidth: number;
    /** 视窗显示行数 */
    private visibleRowCount: number = 0;
    /** 界面上的可见元素，用于绑定当前滚动条的可见高度 */
    private $viewAreaEl: any;
    /** 滚动条当前位置 */
    private position: number = 0;
    /** 总行数 */
    private totalRowCount: number = 0;
    /** 监听事件 */
    private vbarListener: IVBarListener;
    /** 原本的 overflowY 属性 */
    private oldOverflowY!: any;

    constructor(rowHeight: number, viewAreaEl: HTMLElement, vbarListener: IVBarListener) {
        super();
        this.$viewAreaEl = $(viewAreaEl);
        
        this.rowHeight = rowHeight;
        this.barWidth = VScrollBar.getScrollBarWidth(this.$viewAreaEl);
        this.vbarListener = vbarListener;
        
        this.content = new Div();
        this.content.setWidth(this.barWidth);
        this.addChild(this.content);
        this.setStyle('position', 'absolute');
        this.setStyle('z-index', 9999);
        this.setStyle('overflow-y', 'auto');
        this.setStyle('overflow-x', 'hidden');
        this.setStyle('background-color', '#EEE');
        this.addClass(VScrollBar.CLS_DYNAMIC_SCROLL_BAR);
        this.setWidth(this.barWidth);
        this.resetViewAreaEl();
        this.update();

        var self = this;
        this.addEventListener("scroll", function(e: any) {
            var startIndex = Math.ceil(self.getEl().scrollTop / rowHeight);
            self.position = startIndex;
            self.ensurePosition(self.visibleRowCount);
            self.vbarListener && self.vbarListener.scroll(self.position, self.visibleRowCount);
        });
    }
   
    /**
     * 重置视窗
     */
    private resetViewAreaEl(): void {
        var self = this;
        this.oldOverflowY = this.$viewAreaEl[0].style.overflowY;
        this.$viewAreaEl[0].style.overflowY = 'hidden';
        this.$viewAreaEl.on("mousewheel DOMMouseScroll", function (event: any) {
            event.preventDefault();
            var orgEvent = event.originalEvent;
            var wheel = orgEvent.wheelDelta || -orgEvent.detail;
            var delta = Math.max(-1, Math.min(1, wheel));
            var visibleHeight = self.$viewAreaEl.height();
            if(delta<0){//向下滚动
                var total = self.totalRowCount * self.rowHeight;
                if (self.getEl().scrollTop < total - visibleHeight - self.rowHeight) {
                    self.getEl().scrollTop = self.getEl().scrollTop + self.rowHeight;
                } else {
                    self.getEl().scrollTop = total - visibleHeight;
                }
            }else{//向上滚动
                if (self.getEl().scrollTop > self.rowHeight) {
                    self.getEl().scrollTop = self.getEl().scrollTop - self.rowHeight;
                } else {
                    self.getEl().scrollTop = 0;
                }
            } 
        });

        var $el = <JQuery> this.$viewAreaEl;
        $el.scroll( e => {
            this.repositionLeft();
        });
        $el.resize(e => {
            this.repositionLeft();
        });
    }

    repositionLeft() {
        var left = this.$viewAreaEl.width() - this.barWidth + this.$viewAreaEl.scrollLeft();
        this.setStyle("left", left + "px");
    }

    /**
     * 设置全部行数量
     * 
     * @param totalCount 
     */
    setTotalCount(totalRowCount: number): void {
         // 多加一行的高度，防止内容被底部滚动条覆盖
        this.totalRowCount = totalRowCount + 1;
        this.content.setHeight(totalRowCount * this.rowHeight);
    }

    /**
     * 设置顶点位置
     * 
     * @param top 
     */
    setTopPosition(top: number) {
        this.position = top;
        this.getEl().scrollTop = top * this.rowHeight;
    }

    /**
     * 获取顶点位置
     * 
     * @returns 
     */
    getTopPosition(): number {
        return this.position;
    }

    /**
     * 更新滚动条
    */
    update(): void {
        var height = this.$viewAreaEl[0].clientHeight;
        this.setHeight(height);
        // 横向滚动出现时的处理
        var isHBarVisible = this.$viewAreaEl.height() > height;
        var opacity =  isHBarVisible ? 0.5 : 1
        this.content.setStyle('opacity', opacity);
        this.setStyle('opacity', opacity);

        var curVisibleRowCount = this.getCurVisibleRowCount(height);
        if (curVisibleRowCount != this.visibleRowCount) {
            this.visibleRowCount = curVisibleRowCount;
            this.ensurePosition(this.visibleRowCount);
            this.vbarListener && this.vbarListener.init(this.position, curVisibleRowCount);
            this.vbarListener && this.vbarListener.scroll(this.position, curVisibleRowCount);
        }
        this.setStyle("top", "0px");
        this.repositionLeft();
    }

    private ensurePosition(curVisibleRowCount: number): void {
        // 改变窗口大小时，如何position已经超出，则需要重新适配最后一行位置
        // setTotalCount中多加了最后一行空白行，这里判断时需要减掉
        if (this.position + curVisibleRowCount > this.totalRowCount - 1) {
            this.position = this.totalRowCount - curVisibleRowCount - 1;
            this.position = (this.position < 0 ? 0 : this.position);
        }
    }

    /**
     * 卸载当前滚动条
     */
    unInstall(): void {
        this.$viewAreaEl[0].style.overflowY = this.oldOverflowY;
    }

    /**
     * 计算并获取当前视窗的可见行数
     * @param curHeight 
     * @returns 
     */
    private getCurVisibleRowCount(curHeight: number): number {
        return Math.floor(curHeight / this.rowHeight);
    }

    /**
     * 计算当前滚动条的宽度
     * 
     * @param viewAreaEl 
     * @returns 
     */
     static getScrollBarWidth($viewAreaEl: any): number {
        var noScroll = $viewAreaEl.width();
        var viewAreaEl = $viewAreaEl[0];
        var old = viewAreaEl.style.overflowY;
        viewAreaEl.style.overflowY = 'scroll';
        var scroll = viewAreaEl.clientWidth;
        viewAreaEl.style.overflowY = old;
        var width = noScroll - scroll;
        return width > 3 ? width : 17;
    }
}