/**
 * 表格变式
 */

import SizeInfo from "../../../common/struct/SizeInfo";
import TypeUtil from "../../../common/util/TypeUtil";
import { EVariantType } from "../../common/struct/variant/EVariantType";
import { GridColumnVariant, GridSettingVariantData } from "../../common/struct/variant/VariantData";
import GridUtils from "../ui/util/GridUtils";
import VariantUtils from "./VariantUtils";

export default class GridSettingVariant {

    private grid: YIUI.Control.Grid;

    /** 表格变式初始设置 */
    private initVariant?: GridSettingVariantData;
    
    constructor(grid: YIUI.Control.Grid) {
        this.grid = grid;
        this.initVariant = this.prepareInitGridSettingVariant();
    }

    /**
     * 应用表格变式
     * @param grid 
     * @param gridVariant 
     */
    async loadGridSettingVariant(bNeedLoadFromServer: boolean): Promise<void> {
        var variantData = GridUtils.getGridVariantCache(this.grid);
        if (!variantData || bNeedLoadFromServer) {
            variantData = await this.getGridSettingVariantOnServer(this.grid);
            GridUtils.cacheGridVariantCache(this.grid, variantData);
        } 
        variantData && this.applyGridSettingVariant(variantData);
    }

    /**
     * 保存表格变式
     */
     async saveGridSettingVariant() {
        var gridSetting = this.parseGridSettingVariant();
        var formID = this.grid.ofFormID;
        var form = YIUI.FormStack.getForm(formID);
        if (!form) return;
        var variantPrefix = this.getGridSettingPrefix(form);
        var jsonData = JSON.stringify(gridSetting);
        await VariantUtils.saveVariant(form.getFormKey(), variantPrefix, this.grid.variantName, jsonData, JSON.stringify(this.initVariant));
    }

    resetGridSettingVariant(variantName: string) {
        var formID = this.grid.ofFormID;
        var form = YIUI.FormStack.getForm(formID);
        if (!form) return;
        var variantPrefix = this.getGridSettingPrefix(form);
        this.initVariant && VariantUtils.saveVariant(form.getFormKey(), variantPrefix, variantName, JSON.stringify(this.initVariant), JSON.stringify(this.initVariant));
    }

    getInitVariant(): GridSettingVariantData | undefined{
        return this.initVariant;
    }
    
    /**
     * 应用表格变式设置表格
     * @param gridVariant 
     */
     applyGridSettingVariant(gridVariant: GridSettingVariantData) {
        if (TypeUtil.isEmptyObject(gridVariant)) return;

        if (this.initVariant) {
           for (let initColumnVariant of this.initVariant.columns) {
                var columnInfo = GridUtils.getGridColumnInfo(initColumnVariant.columnkey, this.grid.dataModel.columns);
                if (columnInfo) {
                    var columnVariant = this.getColumnVariant(columnInfo.key, gridVariant);
                    this.configGridColumnByVariant(columnInfo, columnVariant, initColumnVariant);
                }

                if(this.grid.impl && this.grid.impl.settings) {
                    columnInfo = GridUtils.getGridColumnInfo(initColumnVariant.columnkey, this.grid.impl.settings.columns);
                    if (columnInfo) {
                        var columnVariant = this.getColumnVariant(columnInfo.key, gridVariant);
                        this.configGridColumnByVariant(columnInfo, columnVariant, initColumnVariant);
                    }
                }
           }
           this.grid.variantName = gridVariant.name;
        }
    }

    private configGridColumnByVariant(gridColumn: YIUI.ColumnInfo, columnVariant: GridColumnVariant | null, initColumnVariant: GridColumnVariant): void {
        if (columnVariant) {
            // 动态扩展列中的Value值一样，Caption不一样的情况，如独立计划需求，计划周期选择，日或周时，生成的扩展列Key是一样的，但是caption不一样
            // 暂时不做caption的处理，后续有这方面需求再说，主要需考虑设计器中的列caption修改后，以变式中还是设计器中修改为准。
            // columnInfo.label = columnInfo.expandColumnKey ? columnInfo.label : columnVariant.caption; 
            //gridColumn.hidden = gridColumn.hidden || !TypeUtil.toBoolean(columnVariant.visible);
            gridColumn.visible = TypeUtil.toBoolean(initColumnVariant.visible) && TypeUtil.toBoolean(columnVariant.visible);
            gridColumn.width = columnVariant.width;
        }
    }

    /**
     * 将当前表格状态转化成变式
     */
     public parseGridSettingVariant(): GridSettingVariantData {
        var columns: GridColumnVariant[] = [];
        var gridSetting: GridSettingVariantData = {columns: columns};
        !this.initVariant && this.prepareInitGridSettingVariant();
        if (this.initVariant) {
            for (let initColumnVariant of this.initVariant.columns) {
                var columnInfo = GridUtils.getGridColumnInfo(initColumnVariant.columnkey, this.grid.dataModel.columns);
                if (columnInfo) {
                    var column = this.parseViewColumnVariant(columnInfo);
                    columns.push(column);
                }
            }
         }
         return gridSetting;
    }

    private prepareInitGridSettingVariant() {
        var columns: GridColumnVariant[] = [];
        var gridSetting: GridSettingVariantData = {columns: columns};
        var modelColumns = this.grid.dataModel.colModel ? this.grid.dataModel.colModel.columns : this.grid.dataModel.columns;
        for (let i = 0; i < modelColumns.length; i++) {
            if (modelColumns[i].visible) {
                var column = this.parseModelColumnVariant(modelColumns[i]);
                columns.push(column);
            }
        }
        return gridSetting;
    }

    parseModelColumnVariant(viewColumnInfo: YIUI.ColumnInfo): GridColumnVariant {
        var columnVariant: GridColumnVariant = {
            index: viewColumnInfo.index,
            columnkey: viewColumnInfo.key, 
            caption: viewColumnInfo.label || viewColumnInfo.caption, 
            visible: viewColumnInfo.visible ? 1 : 0,
            width: SizeInfo.getNumFromPX(viewColumnInfo.width),
            orgcaption: viewColumnInfo.label || viewColumnInfo.caption
        };
        return columnVariant;
    }

    parseViewColumnVariant(viewColumnInfo: YIUI.ColumnInfo): GridColumnVariant {
        var columnVariant: GridColumnVariant = {
            index: viewColumnInfo.index,
            columnkey: viewColumnInfo.key, 
            caption: viewColumnInfo.label || viewColumnInfo.caption, 
            visible: viewColumnInfo.visible ? 1 : 0,
            width: SizeInfo.getNumFromPX(viewColumnInfo.width),
            orgcaption: viewColumnInfo.label || viewColumnInfo.caption
        };
        return columnVariant;
    }

    private getColumnVariant(columnKey: string, gridVariant:GridSettingVariantData): GridColumnVariant | null {
        if (gridVariant.columns) {
            for (let column of gridVariant.columns) {
                if (column.columnkey == columnKey) {
                    return column;
                }
            }
        }
        return null;
    }

    private async getGridSettingVariantOnServer(grid: YIUI.Control.Grid): Promise<GridSettingVariantData|null> {
        if (!this.initVariant) return null;
        var formID = this.grid.ofFormID;
        var form = YIUI.FormStack.getForm(formID);
        if (!form) return null;
        var variantPrefix = this.getGridSettingPrefix(form);
        return await VariantUtils.getVariantContent(variantPrefix, JSON.stringify(this.initVariant));
    }

    private getGridSettingPrefix(form: YIUI.UIForm): string {
        return `${EVariantType.GridSetting}.${form.getFormKey()}.${this.grid.key}`;
    }
    
    static newInstance(grid: YIUI.Control.Grid): GridSettingVariant {
        return new GridSettingVariant(grid);
    }
}