import TypeUtil from "../../../../../common/util/TypeUtil";
import FormUtils from "../../util/FormUtils";
import { EErrorLocation, IComponentError } from "./ErrorStruct";

/**
 * 表单检查控件收集器
 * 
 */
export default class FormErrorCollector {

    /**
     * 收集界面所有检查不通过组件
     * 
     * @param form 
     */
    static collectAllComponent(form: YIUI.UIForm): IComponentError[] {
        var errorInfos: IComponentError[] = [];
        if (!form || !form.getRoot()) return errorInfos;
        var root = form.getRoot();
        FormErrorCollector.collectComponent(root, errorInfos, form, false);
        return errorInfos;
    }

    /**
     * 判断当界面是否有检查不通过控件
     * @param form 
     * @returns 
     */
    static hasError(form: YIUI.UIForm): boolean {
        if (YIUI.FormStack.getActiveForm() != form) return false;
        if (!form || !form.getRoot()) return false;
        var root = form.getRoot();
        var errorInfos: IComponentError[] = [];
        FormErrorCollector.collectComponent(root, errorInfos, form, true);
        return errorInfos.length > 0;
    }

    /**
     * 从一个基点开始遍历子组件收集检查信息
     * @param base 
     * @param errorInfos 
     * @param form
     * @param onlyCheck 
     */
    static collectComponent(base: YIUI.Component, errorInfos: IComponentError[], form: YIUI.UIForm, onlyCheck: boolean): void {
        if (!base.visible) return;
        base.accept((comp: YIUI.Component) => {
            if(!comp.visible) return false;  //不可见的控件不收集(若面板不可见，则其所有子组件都不收集)
            if (onlyCheck &&　errorInfos.length > 0) return false; 
            FormErrorCollector.collectErrorComponent(comp, errorInfos, form);
            return true;
        });
    }

    /**
     * 收集控件中的错误信息
     * 
     * @param comp 
     * @param errorInfos 
     */
     public static collectErrorComponent(comp: YIUI.Component, errorInfos: IComponentError[], form: YIUI.UIForm): void {
        if(comp.isDestroyed) return;
        if (FormUtils.isMirror(comp.key)) return; // 镜像字段不收集，定位组件时，根据原字段来定位(原组件可见可编辑定位到原组件，原组件不可见或不可编辑再找其镜像组件)
        if (!comp.visible) return; 
        if (comp.type == YIUI.CONTROLTYPE.GRID) {
            var grid = <YIUI.Control.Grid> comp;
            FormErrorCollector.collectErrorCellInGrid(grid, errorInfos, form);
        } else {
            if (!comp.enable) return;  //不可编辑的控件不收集
            if (comp.errorInfo.error || comp.isRequired()) {
                errorInfos.push({
                    key: comp.key,
                    caption: comp.caption,
                    enable: comp.enable,
                    visible: comp.visible,
                    error: comp.isRequired() ? `${comp.caption} 【必填】` : `${comp.caption} 【${comp.errorInfo.errorMsg}】`,
                    location: EErrorLocation.HEAD
                });
            }
        }
    }

    /**
     * 收集表格控件的错误信息
     * 
     * @param grid 
     * @param errorInfos 
     */
    public static collectErrorCellInGrid(grid: YIUI.Control.Grid, errorInfos: IComponentError[], form: YIUI.UIForm): void {
        var rowData;
        for (var ri = 0, size = grid.getRowCount(); ri < size; ri++) {
            rowData = grid.getRowDataAt(ri);
            if (YIUI.GridUtil.isEmptyRow(rowData)) {
                continue;
            }
            if (grid.impl && FormErrorCollector.hasErrorInRow(ri, grid, form)) {
                errorInfos.push({
                    key: grid.key,
                    caption: grid.caption,
                    enable: true,
                    visible: true,
                    error: `${grid.caption} 行号:${ri + 1}【检查不通过】`,
                    row: ri,
                    location: EErrorLocation.ROW
                });
                break; // 只收集能找到的第一行错误行
            }
        }
    }

    /**
     * 判断表格行中是否存在检查不通过
     * @param row 
     * @param grid 
     * @param form 
     * @returns 
     */
    private static hasErrorInRow(row: number, grid: YIUI.Control.Grid, form: YIUI.UIForm): boolean {
        var rowData = grid.getRowDataAt(row);
        var rowError = rowData.error ? rowData.error.error : undefined;
        if (rowError && !TypeUtil.isEmptyObject(rowError)) { return true; }

        var cellData;
        for (var ci = 0; ci < grid.getColumnCount(); ci ++) {
            var column = grid.getColumnAt(ci);
            var metaCell = grid.getCellEditOpt(row, ci);
            if (!column.visible && !FormUtils.hasMirror(metaCell.key, form)) continue; // 不可见列且没有镜像字段的单元格不收集
            cellData = grid.getCellDataAt(row, ci);
            if (cellData[3] || (cellData[4] && cellData[5])) {
                return true;
            }
        }
        return false;
    }

    /**
     * 收集某一行中的单元格错误信息
     * 
     * @param row 
     * @param grid 
     * @param errorInfos 
     * @param form 
     */
    public static collectErrorCellInRow(row: number, grid: YIUI.Control.Grid, errorInfos: IComponentError[], form: YIUI.UIForm): void {
        var cellData;
        for (var ci = 0; ci < grid.getColumnCount(); ci ++) {
            var column = grid.getColumnAt(ci);
            var metaCell = grid.getCellEditOpt(row, ci);
            if (!column.visible && !FormUtils.hasMirror(metaCell.key, form)) continue; // 不可见列且没有镜像字段的单元格不收集

            cellData = grid.getCellDataAt(row, ci);
            
            if (cellData[3] || (cellData[4] && cellData[5])) {
                
                errorInfos.push({
                    key: grid.key,
                    caption: grid.caption,
                    cellkey: metaCell.key,
                    enable: cellData[2],
                    visible: column.visible,
                    error: cellData[3] ? `${grid.caption} 【${metaCell.caption} 必填】` :
                                            `${grid.caption} 【${cellData[5]}】`,
                    row: row,
                    col: ci,
                    location: EErrorLocation.CELL
                });
            }
        }
    }

}

//if (!cellData[2] && !FormUtils.hasMirror(metaCell.key, form)) continue;  //不可编辑且没有镜像字段的单元格不收集 (采购订单净值不可编辑，但是设置了检查..)
//var cellLoc = `${grid.caption ? grid.caption : grid.key}-${ri}-${ci}`;
/*
var getFormError = function () {
    var msg = form.getErrorMsg();
    if( !msg ) {
        msg = YIUI.ViewException.formatMessage(YIUI.ViewException.FORM_CHECK_ERROR, form.caption);
    }
    return msg;
}

var getRowError = function (grid,rowData,lineNo) {
    var msg = grid.getErrorMsg(rowData.error.error);
    if( !msg ) {
        msg = YIUI.ViewException.formatMessage(YIUI.ViewException.GRID_ROW_ERROR, grid.caption, lineNo);
    }
    return msg;
}
*/