import PopupPanel from "../../../../../common/component/container/panel/PopupPanel";
import JTreeItem from "../../../../../common/component/tree/base/item/JTreeItem";
import TreeItemModel from "../../../../../common/component/tree/base/model/TreeItemModel";
import IXElement from "../../../../../common/dom/xelement/IXElement";
import Events from "../../../../../common/event/Events";
import ArrayList from "../../../../../common/struct/ArrayList";
import Stack from "../../../../../common/struct/Stack";
import FormUtils from "../../util/FormUtils";
import TableUtils from "../../util/TableUtils";
import { EErrorLocation, IComponentError } from "./ErrorStruct";
import FormErrorCollector from "./FormErrorCollector";
import ComponentLocatorTree from "./tree/ComponentLocatorTree";

/**
 * 控件定位视图
 */
export default class ComponentLocatorView extends PopupPanel {

    private bMouseOver: boolean = false;

    private bDoingItemClick: boolean = false;

    //private list: JList;
    private list: ComponentLocatorTree;

    private curForm: YIUI.UIForm | undefined;

    private curTimer: any;

    constructor() {
        super();
        this.addClass('yg-component-locateview');
        this.setStyle("overflow", "auto");

        this.list = new ComponentLocatorTree();
        this.setContent(this.list.getDomElement());

        this.addEventListener(Events.MouseEnter, (event: any) => {
            this.bMouseOver = true;
        }, true);

        this.addEventListener(Events.MouseLeave, (event: any) => {
            if (this.bDoingItemClick || this.getEl() != event.target) return;
            // 等待窗口的鼠标事件被莫名其妙的触发，这里禁止鼠标移出到等待窗口上...
            if (event.toElement && event.toElement.classList.contains('loading', 'mask')) return;
            this.bMouseOver = false;
            this.hide();
        }, true);

        this.list.setOnItemClick(async (node: JTreeItem) => {
            if (this.curTimer) clearTimeout(this.curTimer);
            this.bDoingItemClick = true;
            try {
                if (node) {
                    var nodeModel = node.getModel();
                    var itemData = nodeModel.getItemData();
                    if (this.curForm && itemData) {
                        await FormUtils.locateComponentAndCell(itemData.data , this.curForm);
                        this.curTimer = setTimeout(() => {
                            // 若是行检查错误，定位到具体行后，继续收集其具体单元格检查信息，以及其明细表格检查信息
                            var info = <IComponentError> itemData.data;
                            if (this.curForm && info.row != undefined && info.location == EErrorLocation.ROW) {
                                var grid = <YIUI.Control.Grid> this.curForm.getComponent(info.key);
                                this.loadSubNodesByRow(info.row, nodeModel, grid, this.curForm);
                            }
                            if (!node.isLeaf()) {
                                node.toggle();
                            }
                            this.curTimer = undefined;
                        }, 500);
                    }
                }
            } finally {
                this.bDoingItemClick = false;
            }
        });
    }

    private loadSubNodesByRow(row: number, nodeModel: TreeItemModel, grid: YIUI.Control.Grid, form: YIUI.UIForm): void {
        // 收集当前行的检查信息
        nodeModel.clearChildren();
        var cellErrorInfos: IComponentError[] = [];
        FormErrorCollector.collectErrorCellInRow(row, grid, cellErrorInfos, form);
        var treeModel = this.list.getModel();
        for (let cellError of cellErrorInfos) {
            var childItemModel =  treeModel.createTreeItemModel({caption: cellError.error, data: cellError, branch: false});
            treeModel.addChildItem(nodeModel, childItemModel);
        }

        var record = new ArrayList<String>();
        var subGridKeys = form.getSubDetail(grid.key);
        if (subGridKeys) {
            for (let subKey of subGridKeys) {
                // 收集子表格中存在检查错误
                var subGridRowErrors: IComponentError[] = [];
                var subComp =  form.getComponent(subKey);
                if (subComp.type == YIUI.CONTROLTYPE.GRID) {
                    var subGrid = <YIUI.Control.Grid> subComp;
                    FormErrorCollector.collectErrorCellInGrid(subGrid, subGridRowErrors, form);
                } else {
                    FormErrorCollector.collectComponent(subComp, subGridRowErrors, form, false);
                }
                for (let rowError of subGridRowErrors) {
                    if (!record.contain(rowError.key)) {
                        var childItemModel =  treeModel.createTreeItemModel({caption: rowError.error, data: rowError, branch: true});
                        treeModel.addChildItem(nodeModel, childItemModel);
                        record.add(rowError.key);
                    }
                }
            }
        }
    }

    hide(): void {
        if (!this.bMouseOver) {
            if (this.curTimer) clearTimeout(this.curTimer);
            super.hide();
            this.clearModel();
        }
    }

    showLocatorView(form: YIUI.UIForm, left: number, top: number, width: number, height: number, parent: HTMLElement): void {
        if (!this.isShowing()) {
            this.loadView(form);
            this.show(left, top, width, height, parent);
        }
    }

    /**
     * 清除视图内容
     */
    private clearModel(): void {
        this.list && this.list.clearModel();
    }

    /**
     * 加载视图内容
     * @param form 
     */
    private loadView(form: YIUI.UIForm): void {
        if (this.curTimer) clearTimeout(this.curTimer);
        this.clearModel()
        var errorInfos: IComponentError[] = FormErrorCollector.collectAllComponent(form);
        for (let info of errorInfos) {
            this.list.addItem({caption: info.error, data: info, branch: info.location == EErrorLocation.ROW})
        }
        this.list.updateUI();
        this.curForm = form;
    }

    private locateParentGrid(row: number, grid: YIUI.Control.Grid, form: YIUI.UIForm): void {
        // 如果存在父表格，需要先定位到父表明细对应的行
        var pGridKey = grid.parentGridKey;
        if (!pGridKey) return;

        var doc = form.getDocument();
        var table = doc.getByKey(grid.tableKey);
        if (!table) return;
        table.setPos(row);
        var curOID = table.getByKey(YIUI.SystemField.OID_SYS_KEY);

        var stack = new Stack();
        while (grid && table && curOID > 0) {
            var poid = TableUtils.getValueByOID(YIUI.SystemField.POID_SYS_KEY, curOID, table);
            stack.push({key: grid.parentGridKey, oid: poid});
            grid = form.getComponent(grid.parentGridKey);
            table = grid ? doc.getByKey(grid.tableKey) : undefined;
            curOID =  poid;
        }
    }
}