import PopupPanel from "../../../../common/component/container/panel/PopupPanel";
import ContentSearchBox from "../../../../common/component/control/ContentSearchBox";
import { IGridFindResult } from "../../../../common/struct/Structs";
import GridUtils from "../util/GridUtils";

export default class PopupSearchView extends PopupPanel {

    private grid: YIUI.Control.Grid;

    private searchbox: ContentSearchBox;

    private result?: IGridFindResult;

    constructor(searchbox: ContentSearchBox, grid: YIUI.Control.Grid) {
        super();
        this.grid = grid;
        this.searchbox = searchbox;
        this.setContent(searchbox.getDomElement());

        this.searchbox.setOnCloseListener((e: MouseEvent) => {
            this.hide();
        });

        this.searchbox.setOnNextListener((e: MouseEvent) => {
            this.locNext();
        });

        this.searchbox.setOnPreviousListener((e: MouseEvent) => {
            this.locPrevious();
        });

        this.searchbox.setOnEnterListener((e: KeyboardEvent) => {
            e.stopPropagation();
            if (e.key === 'Enter') {
                this.locNext();
                setTimeout(function () {
                    searchbox.caretLast();
                }, 100);
            }
        });
    }

    hide(): void {
        this.setStyle('display', 'none');
        this.reset();
    }

    show(): void {
        this.setStyle('display', 'block');
        this.searchbox && this.searchbox.caretLast();
    }

    async locNext() {
        if (this.needResearch()) this.relocAll();
        if (!this.result) return;
        if (this.result.pos < this.result.cells.length - 1) {
            this.result.pos ++;
        } else if (this.result.cells.length > 0) {
            this.result.pos = 0; // 返回起始位置，循环定位
        }
        await this.updateFocusFindStyle();
        this.updateResultLabel();
        this.resetLocStatus();
    }

    async locPrevious() {
        if (this.needResearch()) this.relocAll();
        if (!this.result) return;
        if (this.result.pos > 0) {
            this.result.pos --;
        } else if (this.result.cells.length > 0) {
            this.result.pos = this.result.cells.length - 1;
        }
        await this.updateFocusFindStyle();
        this.updateResultLabel();
        this.resetLocStatus();
    }

    resetLocStatus(): void {
        this.searchbox.setPreviousEnable(!!this.result && this.result.pos > 0);
        this.searchbox.setNextEnable(!!this.result && this.result.pos < this.result.cells.length);
    }

    /**
     * 第一次查找时，保留所有的查找结果
     */
    relocAll(): void {
        var findText = this.getFindText();
        this.result = GridUtils.findResult(findText, this.grid);
        this.updateAllFindStyle();
    }

    needResearch(): boolean {
        return !this.result || this.result.content != this.getFindText();
    }

    updateAllFindStyle() {
        if (!this.result) return;
        GridUtils.clearFindStyle(this.grid);
        GridUtils.clearFindFocusStyle(this.grid);
        var $grid = this.grid.getEl();
        var pageInfo = this.grid.getPageInfo();
        for (var i = 0; i < this.result.cells.length; i++) {
            var cellPosition = this.result.cells[i];
            if (this.isNormalPagination() && this.getResultPageIndex(cellPosition.row) != pageInfo.curPageIndex) {
                continue;
            }
            var $cell = $($grid.getGridCellAt(cellPosition.row + 1, this.grid.showRowHead ? cellPosition.col + 1 : cellPosition.col));
            $cell.addClass("find-cell-style");
            $cell.closest('tr').addClass("find-row-style");
        }
    }

    async updateFocusFindStyle(): Promise<void> {
        if (!this.result) return;
        if (this.isOutRangePos()) return;
        GridUtils.clearFindFocusStyle(this.grid);
        var $grid = this.grid.getEl();
        var cellPosition = this.result.cells[this.result.pos];
        await this.locPage(cellPosition.row);

        var $cell = $($grid.getGridCellAt(cellPosition.row + 1, this.grid.showRowHead ? cellPosition.col + 1 : cellPosition.col));
        $cell.addClass("find-focusCell-style");
        this.grid.setFocusCell(cellPosition.row, cellPosition.col);
    }

    private isNormalPagination() {
        return this.grid.getMetaObj().pageLoadType != YIUI.PageLoadType.NONE;
    }

    private getResultPageIndex(gRowPos: number): number {
        var pageInfo = this.grid.getPageInfo();
        if (pageInfo && pageInfo.pageRowCount > 0) {
            return Math.floor(gRowPos / pageInfo.pageRowCount);
        }
        return 0;
    }

    /**
     * 根据当前行号，翻页
     * @param gRowPos 全局行号
     * @returns 当前页行号
     */
    private async locPage(gRowPos: number): Promise<void> {
        var pageInfo = this.grid.getPageInfo();
        var bPage = this.isNormalPagination();
        if (pageInfo && pageInfo.pageRowCount > 0) {
            var resultPageIndex = Math.floor(gRowPos / pageInfo.pageRowCount);
            var curPageIndex = GridUtils.getCurPageIndex(this.grid);   //TypeUtil.toNum(this.grid.el[0].p.page);
            if (resultPageIndex != curPageIndex) {
                if (!bPage && resultPageIndex > curPageIndex) { // 此处处理滚动分页，对于页码分页的情况未做处理，因为表格数据模型中的数据不全
                    GridUtils.populate(resultPageIndex + 1, this.grid); // populate 此方法参数传入的是页码，页码是从1开始
                    GridUtils.scrollVisibleCell(gRowPos + 1, 0, this.grid);
                    GridUtils.populateVisible(this.grid);
                }
                this.updateAllFindStyle();
            }
        }
    }

    updateResultLabel(): void {
        if (this.isOutRangePos()) return;
        var resultInfo = "No Result";
        if (this.result && this.result.cells.length > 0) {
            resultInfo = `${this.result.pos + 1} of ${this.result.cells.length}`;
        }
        this.searchbox && this.searchbox.updateResultLabel(resultInfo);
    }

    getFindText(): string {
        return this.searchbox.getSearchText();
    }

    reset(): void {
        this.result = undefined;
        this.updateResultLabel();
        this.searchbox && this.searchbox.reset();
        this.grid && GridUtils.clearFindStyle(this.grid);
        this.grid && GridUtils.clearFindFocusStyle(this.grid);
    }

    private isOutRangePos(): boolean {
        if (!this.result) return true;
        var pos = this.result.pos;
        return pos < 0 || pos > this.result.cells.length - 1;
    }
}
