import StringUtil from "../../../../common/util/StringUtil";
import TypeUtil from "../../../../common/util/TypeUtil";
import { EErrorLocation, IComponentError } from "../view/locator/ErrorStruct";
import GridUtils from "./GridUtils";

export default class FormUtils {

    /**
     * 判断当前字段Key 在表单中是否存在镜像字段
     * @param key 
     * @param form 
     * @returns 
     */
    static hasMirror(key: string, form: YIUI.UIForm): boolean {
        var mirrorKey = FormUtils.getMirrorKey(key);
        var mirrorComp = form.getComponent(mirrorKey);
        return !!mirrorComp;
    }

    static isMirror(key: string): boolean {
        if (!key) return false;
        if (!key.endsWith("_mirror")) {
            if (!/_mirror[1-9]$/.test(key)){
                return false;
            }
        }
        return true;
    }

    /**
     * 根据普通字段Key,获取镜像字段Key
     * @param key 
     * @returns 
     */
    static getMirrorKey(key: string): string {
        return key + '__mirror';
    }

    /**
     * 根据普通字段Key,获取镜像字段
     * @param key 
     * @returns 
     */
    static getMirror(key: string, form: YIUI.UIForm) : YIUI.Component | undefined {
        if (!key) return undefined;
        var mirrorKey = FormUtils.getMirrorKey(key);
        var mirrorComp = form.getComponent(mirrorKey);
        return mirrorComp ? <YIUI.Component> mirrorComp : undefined;
    }

    /**
     * 定位组件
     * @param 
     */
     static async locateComponentAndCell(info: IComponentError, form: YIUI.UIForm): Promise<void> {
        if (info && form) {
            var ctx = new YIUI.ViewContext(form);
            if (info.location == EErrorLocation.CELL) {
                if (info.row !== undefined && info.col !== undefined) {
                    var grid = <YIUI.Control.Grid> form.getComponent(info.key);
                    await form.eval(`LocateComponent('${grid.key}')`, ctx, []); // 先定位到表格，表格可能在tab中
                    GridUtils.scrollVisibleCell(info.row, info.col, grid);
                    GridUtils.populateVisible(grid);
                    GridUtils.setCellFocus(info.row, info.col, grid)
                    if (info.cellkey && (!info.enable || !info.visible)) {  //定位组件时，根据原字段来定位(原组件可见可编辑定位到原组件，原组件不可见或不可编辑再找其镜像组件)
                        var mirror = FormUtils.getMirror(info.cellkey, form);
                        if (mirror && mirror.enable && mirror.visible) {
                            await form.eval(`LocateComponent('${mirror.key}')`, ctx, []);
                        }
                    }
                }
            } else if (info.location == EErrorLocation.ROW) {
                var grid = <YIUI.Control.Grid> form.getComponent(info.key);
                var row = info.row;
                if (row != undefined) {
                    GridUtils.scrollVisibleCell(row, 0, grid);
                    GridUtils.populateVisible(grid);
                    //GridUtils.setCellFocus(row, 0, grid)
                    await GridUtils.selectRow(row, grid);
                }
            } else {
                await form.eval(`LocateComponent('${info.key}')`, ctx, []);
            }
        }
    }

    /**
     * 定位父表格中的行
     */
    static async locateParentRow(gridKey: string, parentRowIndex: number, form: YIUI.UIForm): Promise<void> {
        var uiGrid = <YIUI.Control.Grid> form.getComponent(gridKey);
        var pGridKey = uiGrid.parentGridKey;
        await FormUtils.locateGridRow(pGridKey, parentRowIndex, form);
    }

    /**
     * 定位表格中的行
     */
    static async locateGridRow(gridKey: string, rowIndex: number, form: YIUI.UIForm): Promise<void> {
        if (StringUtil.isBlankOrNull(gridKey)) return;
        var pInfo = {
            key: gridKey,
            caption: gridKey,
            enable: true,
            visible: true,
            error: '',
            row: rowIndex,
            location: EErrorLocation.ROW
        }
        await FormUtils.locateComponentAndCell(pInfo, form);
    }

    static async locateHeadComponent(compKey: string, form: YIUI.UIForm) {
        var info = {
            key: compKey,
            caption: compKey,
            enable: true,
            visible: true,
            error: '',
            location: EErrorLocation.HEAD
        }
        await FormUtils.locateComponentAndCell(info, form);
    }

    static async locateGridCell(gridKey: string, row: number, col: number, form: YIUI.UIForm) {
        var grid = <YIUI.Control.Grid> form.getComponent(gridKey);
        var meta = grid.getCellEditOpt(row, col);
        var visible = GridUtils.isGridColumnVisible(meta.key, grid);
        var info = {
            key: gridKey,
            cellkey: meta.key,
            caption: meta.key,
            enable: true,
            visible: visible,
            error: '',
            row: row,
            col: col,
            location: EErrorLocation.CELL
        }
        await FormUtils.locateComponentAndCell(info, form);
    }

    static fireCustomRelayout(): void {
        //$(document).trigger('custom.relayout');
        var event = new CustomEvent("custom.relayout", {
            detail: {p1:'', p2: ''}
        });
        document.dispatchEvent(event);
        //定义
        //document.addEventListener("custom.relayout", function (e) {console.log(e,'触发custom.relayout事件');
    }

    static installCustomRelayoutListener(): void {
        document.addEventListener("custom.relayout", e => {
            var activeForm = YIUI.FormStack.getActiveForm(); //堆栈中所记录的当前form, 可能是Modal弹出窗口
            if (activeForm) {
                var rootPanel = activeForm.getRoot();
                if (FormUtils.hasCollapse(activeForm)) {
                    var rootHeight = FormUtils.calcRootHeight(activeForm);
                    FormUtils.resetModalDialogHeight(activeForm, rootHeight);
                    rootPanel.doLayout(rootPanel.getWidth(), rootHeight);
                } else {
                    rootPanel.doLayout(rootPanel.getWidth(), rootPanel.getHeight());
                }
            }/* else {
                _this.layout(_this.getWidth(), _this.getHeight());
            }*/
        });
    }

    static hasCollapse(activePopForm: YIUI.UIForm): boolean {
        //var $dialogEl = activePopForm.getModalDialogEl();
        var dialogEl = FormUtils.getModalDialogEl(activePopForm);
        if (dialogEl) {
            return !! dialogEl.querySelector('.yg_collapse');
        }
        return false;
    }

    static calcRootHeight(activePopForm: YIUI.UIForm): number | string {
        var rootPanel = activePopForm.getRoot();
        if (activePopForm.target != YIUI.FormTarget.MODAL) {
            return rootPanel.getHeight();
        }
        var autoHeight = rootPanel.calcAutoHeight();
        var metaPopHeight = TypeUtil.getReal(activePopForm.popHeight);
        var maxPopHeight = metaPopHeight ? metaPopHeight : 800;
        var realHeight = maxPopHeight;
        if (autoHeight > 30 && typeof maxPopHeight === 'number' && autoHeight < maxPopHeight) {
            realHeight = autoHeight;
        }
        return realHeight;
    }
    
    static resetModalDialogHeight(activePopForm: YIUI.UIForm, contentHeight: any): void {
        if (activePopForm.target != YIUI.FormTarget.MODAL) {
            return;
        }
        var dialogEl = FormUtils.getModalDialogEl(activePopForm);
        if (dialogEl) {
            var headerEl = dialogEl.querySelector('.jsPanel-hdr');
            var headerHeight = headerEl ? headerEl.clientHeight : 0;
            var dialogHeight = (contentHeight == "auto" ? "auto" : contentHeight + headerHeight);
            dialogEl.style.height = dialogHeight + "px";

            var contentEl: any = dialogEl.querySelector('.jsPanel-content');
            contentEl && (contentEl.style.height = dialogHeight + 'px');
            //var headerHeight = $(".dialog-header", $dialogEl).outerHeight();
            //var dialogHeight = (contentHeight == "auto" ? "auto" : contentHeight + headerHeight);
            //$dialogEl.css('height', dialogHeight);
            //$('.dialog-content', $dialogEl).css('height', contentHeight);
            //$('.dialog-content-inner', $dialogEl).css('height', contentHeight);
        }
    }

    static getModalDialogEl(popForm: YIUI.UIForm): HTMLElement | undefined {
        if (popForm && popForm.dialog) {
            return popForm.dialog.dialog;
        }
        return undefined;
    }
}