import StringBuilder from "../../../common/util/StringBuilder";
import AbstractYigoFuns from "../../common/funs/AbstractYigoFuns";
import PanelUtils from "../ui/util/PanelUtils";
import VariantUtils from "../variant/VariantUtils";
import GridSettingVariant from "../variant/GridSettingVariant";
import { EVariantType } from "../../common/struct/variant/EVariantType";
import FormUtils from "../ui/util/FormUtils";
import { EErrorLocation, IComponentError } from "../ui/view/locator/ErrorStruct";
import GridUtils from "../ui/util/GridUtils";

export default class YigoWebFuns extends AbstractYigoFuns {

    protected initFunctions(): void {
        this.addFun('ToggleArea', this.toggleArea);
        this.addFun('ParseGridSettingVariant', this.parseGridSettingVariant);
        this.addFun('LoadGridSettingVariant', this.loadGridSettingVariant);
        this.addFun('ResetGridSettingVariant', this.resetGridSettingVariant);
        this.addFun('ConvertControlToJSON', this.convertControlToJSON);
        this.addFun('GetGridSettingDefaultVariantName', this.getGridSettingDefaultVariantName);
        this.addFun('GetInitGridSettingVariant', this.getInitGridSettingVariant);
        this.addFun('LocateComponentAndCell', this.locateComponentAndCell);
        this.addFun('LocateGridRow', this.locateGridRow);
    }

    /**
     * 定位到表格行
     * @param args[0] 表格标志
     * @param args[1] 行序号
     */
    private async locateGridRow(evaluator: any, name: string, ctx: any, args: any[]): Promise<any> {
        var form = <YIUI.UIForm> ctx.form;
        var gridKey = args[0];
        var row = args.length > 1 ? args[1] : 0;
        await FormUtils.locateGridRow(gridKey, row, form);
    }

    /**
     * 定位表单中的组件/单元格
     * args[0]: 组件标志 | 单元格标志
     * args[1]: 行序号 (可选，定位单元格时必填)
     */
    private async locateComponentAndCell(evaluator: any, name: string, ctx: any, args: any[]): Promise<any> {
        var form = <YIUI.UIForm> ctx.form;
        var componentKey = args[0];
        var row = args.length > 1 ? args[1] : 0;
        var parentRowIndex = args.length > 2 ? args[2] : -1;

        var info: IComponentError;
        var loc = form.getCellLocation(componentKey);
        if (loc) {
            var uiGrid = <YIUI.Control.Grid> form.getComponent(loc.key);
            var visible = GridUtils.isGridColumnVisible(componentKey, uiGrid);
            info = {
                key: loc.key,
                cellkey: componentKey,
                caption: componentKey,
                enable: true,
                visible: visible,
                error: '',
                row: row,
                col: loc.column,
                location: EErrorLocation.CELL
            }
            if (parentRowIndex > -1) { // 如果设置了父明细行，先定位到父明细行
                await FormUtils.locateParentRow(loc.key, parentRowIndex, form);
                setTimeout(async () => {
                    await FormUtils.locateComponentAndCell(info, form);
                }, 500);
            }
        } else {
            info = {
                key: componentKey,
                caption: componentKey,
                enable: true,
                visible: true,
                error: '',
                location: EErrorLocation.HEAD
            }
        }
        await FormUtils.locateComponentAndCell(info, form);
    }

    private getInitGridSettingVariant(evaluator: any, name: string, ctx: any, args: any[]): any {
        var form = <YIUI.UIForm> ctx.form;
        var gridKey = args[0];
        var uiGrid = <YIUI.Control.Grid> form.getComponent(gridKey);
        if (!uiGrid.variant) return "";
        return (<GridSettingVariant> uiGrid.variant).getInitVariant();
    }

    /**
     * 重置表格变式
     * @param name 
     * @param ctx 
     * @param args 
     * @returns 
     */
    private async resetGridSettingVariant(evaluator: any, name: string, ctx: any, args: any[]): Promise<any> {
        var form = <YIUI.UIForm> ctx.form;
        var gridKey = args[0];
        var variantName = args[1];
        var uiGrid = <YIUI.Control.Grid> form.getComponent(gridKey);
        if (!uiGrid.variant) return;
        (<GridSettingVariant> uiGrid.variant).resetGridSettingVariant(variantName);
        uiGrid.needReloadVariant = true;
    } 

    /**
    * 获取当前表格变式名称
    * @param name 
    * @param ctx 
    * @param args 
    * @returns 
    */
    private async getGridSettingDefaultVariantName(evaluator: any, name: string, ctx: any, args: any[]): Promise<any> {
        var formKey = args[0];
        var gridKey = args[1];
        var variantPrefix = `${EVariantType.GridSetting}.${formKey}.${gridKey}`;
        return await VariantUtils.getDefaultVariantName(variantPrefix);
    } 

    /**
     * 获取表格当前设置
     * @param name
     * @param ctx
     * @param args
     * @returns
     */
     private parseGridSettingVariant(evaluator: any, name: string, ctx: any, args: any[]): any {
        var form = <YIUI.UIForm> ctx.form;
        var gridKey = args[0];
        var uiGrid = <YIUI.Control.Grid> form.getComponent(gridKey);
        if (!uiGrid.variant) return "";
        return (<GridSettingVariant> uiGrid.variant).parseGridSettingVariant();
    }

    /**
     * 载入当前表格变式
     * @param name 
     * @param ctx 
     * @param args 
     * @returns 
     */
    private async loadGridSettingVariant(evaluator: any, name: string, ctx: any, args: any[]): Promise<void> {
        var form = <YIUI.UIForm> ctx.form;
        var gridKey = args[0];
        var uiGrid = <YIUI.Control.Grid> form.getComponent(gridKey);
        if (!uiGrid.variant) return;
        await (<GridSettingVariant> uiGrid.variant).loadGridSettingVariant(true);
        uiGrid.needReloadVariant = true;
    }

    /**
     * 将控件内容转换为JSON
     * @param name
     * @param ctx
     * @param args
     * @returns
     */
    private convertControlToJSON(evaluator: any, name: string, ctx: any, args: any[]): string {
        var form = <YIUI.UIForm>ctx.form;
        var sbSetting = new StringBuilder();
        for (let componentKey of args) {
            var comp = form.getComponent(componentKey);
            if (comp.type == YIUI.ControlType.GRID) {
                var uiGrid = <YIUI.Control.Grid>comp;
                var columnCount = uiGrid.getColumnCount();
                var rowCount = uiGrid.getRowCount();
                var sbColumns = new StringBuilder();
                for (var row = 0; row < rowCount; row++) {
                    var sb = new StringBuilder();
                    for (var col = 0; col < columnCount; col++) {
                        var column = uiGrid.getColumnAt(col);
                        if (!column.key.startsWith('_')) {
                            var value = uiGrid.getValueAt(row, col);
                            var item = `"${column.key}":"${value}"`;
                            sb.append(item);
                        }
                    }
                    if (sb.getSize() > 0) {
                        var strJSON = "{" + sb.toStringWithSep(",") + "}";
                        sbColumns.append(strJSON);
                    }
                }
                if (sbColumns.getSize() > 0) {
                    var strColumnsJSON = "[" + sbColumns.toStringWithSep(",") + "]";
                    sbSetting.append(`"${componentKey}":${strColumnsJSON}`);
                }
            } else {
                var uiControl = <YIUI.Control> comp;
                sbSetting.append(`"${componentKey}":"${uiControl.getValue()}"`);
            }
        }
        if (sbSetting.getSize() > 0) {
            return "{" + sbSetting.toStringWithSep(",") + "}";
        } else {
            return "";
        }
    }

    private toggleArea(evaluator: any, name: string, ctx: any, args: any[]): any {
        var form = <YIUI.UIForm> ctx.form;
        var gridPanelKey = args[0];
        var groupKey = args[1];
        var value = <boolean> args[2];
        var gridPanel = <YIUI.Panel.GridLayoutPanel> form.getComponent(gridPanelKey);
        var handler = PanelUtils.createCollapsibleHandler(gridPanel);
        handler.updateAreaCollapse(groupKey, value);
        return value;
    }

}