import PrintConstants from "../base/PrintConstants";
import ITagNode from "../../../common/xml/node/ITagNode";
import {ENodeType} from "../../../common/enum/Enums";

/**
 * 存储了每个节点类型的检查方法
 */
export default class PrintNodeCheckUtil {

    public static checkFuns: any;

    /**
     * 错误信息
     */
    public static msg: string;

    /**
     * 初始化检查方法
     * 在保存前执行
     */
    static init() {
        this.checkFuns = {}
        this.msg = "";
        this.checkFuns[PrintConstants.NODE_Report] = ((node: ITagNode) => {
            this.ReportNodeCheck(node);
        })
        this.checkFuns[PrintConstants.NODE_Section] = ((node: ITagNode) => {
            this.SectionNodeCheck(node);
        })
        this.checkFuns[PrintConstants.NODE_Cell] = ((node: ITagNode) => {
            this.CellNodeCheck(node);
        })
        this.checkFuns[PrintConstants.NODE_Row] = ((node:ITagNode) => {
            this.RowNodeCheck(node);
        })
    }

    static nodeCheck(node: ITagNode): void {
        if (node) {
            if (node.getNodeType() == ENodeType.TAG) {
                let tagName = node.getTagName();
                if (this.checkFuns[tagName]) {
                    this.checkFuns[tagName](node);
                }
                // 递归检查节点下的子节点
                for (const child of node.getChildren()) {
                    this.nodeCheck(<ITagNode>child);
                }
            }
        }
    }


    private static ReportNodeCheck(node: ITagNode) {

    }

    private static SectionNodeCheck(node: ITagNode) {
        // row下cell和column的检查, 先删column会报样式的问题 但是删cell不会 导致打印时出现问题
        let columns = node.getChild("Columns");
        let rows = node.getChild("Rows");
        if (columns && rows) {
            // 删除所有的row也不会报错 但是打印时报错
            if (rows.getChildren().length == 0) {
                return;
            }
            for (let i = 0; i < rows.getChildren().length; i++) {
                let row = <ITagNode>rows.getChildren()[i];
                if (columns.getChildren().length != row.getChildren().length) {
                    this.msg += this.getDescription(node) + "第" + (i + 1) + "行" + "下行列数不匹配, 请检查;<br/>";
                }
            }
        }
    }

    private static RowNodeCheck(node:ITagNode) {
        // row type为分组时 需要填写分组标识
        let type = node.getAttributeValue("Type");
        let groupKey = node.getAttributeValue("GroupKey");
        if(type == "Group" && !groupKey) {
            this.msg += this.getDescription(node) + "未填写分组标识;<br/>"
        }
    }

    private static CellNodeCheck(node: ITagNode) {
        let key = node.getAttributeValue("Key");
        let sourceType = node.getAttributeValue("SourceType");
        // 如果类型是字段 并且没有key的话需要提示
        if (sourceType == "Field" && !key) {
            this.msg += this.getDescription(node) + "处标识为空, 请检查;<br/>";
        }
    }

    /**
     * 位置描述
     * @param node
     */
    static getDescription(node: ITagNode): string {
        let locationDescription: string = "位置: ";
        let ld: string = "";
        while (node.getTagName() != PrintConstants.NODE_Grid) {
            let tagName = node.getTagName();
            let index = node.getParent()!.getChildren().indexOf(node) + 1;
            if (tagName == PrintConstants.NODE_Section) { // section
                ld =  "第" + index + "段" + ld + " ";
            }
            if (tagName == PrintConstants.NODE_Row) { // row
                ld = "第" + index + "行" + ld + " ";
            }
            if (tagName == PrintConstants.NODE_Cell) { // column
                ld = "第" + index +"列" + ld + " ";
            }
            node = node.getParent()!;
        }
        return locationDescription + ld;
    }


}