import AbstractWorkSpace from "../base/AbstractWorkSpace";
import {EDesignMode} from "../../common/enum/Enums";
import AbstractDesignView from "../base/AbstractDesignView";
import ComplexDesignView from "../base/ComplexDesignView";
import PrintDesignView from "./view/PrintDesignView";
import OperationDispatcher from "../../common/operation/OperationDispatcher";
import PrintGraphicView from "./view/PrintGraphicView";
import PrintToolbar from "./view/PrintToolbar";
import Button from "../../common/component/control/Button";
import PrintButton from "./view/PrintButton";
import Label from "../../common/component/control/Label";
import SizeInfo from "../../common/struct/SizeInfo";
import SetAlignmentCmd from "./cmd/SetAlignmentCmd";
import CmdQueue from "../../common/cmd/CmdQueue";
import SetBorderCmd from "./cmd/SetBorderCmd";
import SetSectionCmd from "./cmd/SetSectionCmd";
import ITagNode from "../../common/xml/node/ITagNode";
import SetRowCmd from "./cmd/SetRowCmd";
import PrintH5HFlexPanel from "./view/PrintH5HFlexPanel";
import TypeUtil from "../../common/util/TypeUtil";
import PrintCombobox from "./view/PrintCombobox";
import PrintButtonBox from "./view/PrintButtonBox";
import Toolbar from "../../common/component/toolbar/Toolbar";
import JComponent from "../../common/component/JComponent";
import Events from "../../common/event/Events";
import PrintConstants from "./base/PrintConstants";
import PrintActionUtil from "./util/PrintActionUtil";
import P18N from "./base/P18N";
import NewRowElementCmd from "./cmd/NewRowElementCmd";
import NewColumnElementCmd from "./cmd/NewColumnElementCmd";
import NewSectionElementCmd from "./cmd/NewSectionElementCmd";
import DeleteElementCmd from "./cmd/DeleteElementCmd";
import DivisionCellCmd from "./cmd/DivisionCellCmd";
import MergeCellCmd from "./cmd/MergeCellCmd";
import PRINT from "./base/PRINT";
import {IPropertyIO} from "../plugin/property/base/PropertyDefine";
import {PrintPropertyIO} from "./property/PrintPropertyIO";
import SetStyleBorderCmd from "./cmd/SetStyleBorderCmd";
import SetBorderColorCmd from "./cmd/SetBorderColorCmd";
import PrintNodeUtil from "./util/PrintNodeUtil";
import PrintColorEditor from "../../common/component/control/PrintColorEditor";
import PrintNodeCheckUtil from "./util/PrintNodeCheckUtil";
import StringUtil from "../../common/util/StringUtil";
import AbstractDomElement from "../../common/dom/AbstractDomElement";

export default class PrintWorkSpace extends AbstractWorkSpace<PrintGraphicView, Toolbar> {

    constructor(mode: EDesignMode) {
        super(mode, new Toolbar());
        this.addClass("print-workspace");
    }

    createDesignView(viewContainer: ComplexDesignView<any, any>, operationDispatcher: OperationDispatcher): AbstractDesignView<any> {
        return new PrintDesignView(viewContainer, operationDispatcher);
    }

    createGraphicView(mode: EDesignMode) {
        var printGraphicView = new PrintGraphicView();
        for (var action of PrintActionUtil.ACTION_ON_NODE) {
            var menuItem = printGraphicView.getToolBar().addMenuItem(action, P18N.getString(action))
            menuItem.setStyle("font-size", "12px");
            menuItem.setStyle("padding","1px 8px")
            this.initToolOptHandler(menuItem, printGraphicView);
        }
        let menuItem1 = printGraphicView.getPageOpt().addMenuItem(PrintConstants.Node_Page_Opt, P18N.getString(PrintConstants.Node_Page_Opt));
        let menuItem2 = printGraphicView.getPageOpt().addMenuItem(PrintConstants.Node_Paper_Opt,P18N.getString(PrintConstants.Node_Paper_Opt));
        this.initToolOptHandler(menuItem1, printGraphicView);
        this.initToolOptHandler(menuItem2, printGraphicView);
        this.installPrintToolbar(printGraphicView.getPrintToolBar());

        let innerTemplateView = printGraphicView.getTemplateView();
        this.initViewMouseEvent(innerTemplateView.getDomElement());

        return printGraphicView;
    }

    load(xml: string) {
        super.load(xml ? xml : this.DEMO_XML);
        this.getXmlEntity().addEventListener(this.getGraphicView().getPrintToolBar());
        PrintNodeUtil.reloadReport(this.getXmlEntity().getTagRoot());
    }

    /**
     * 初始化打印模板三角点击事件
     */
    private initViewMouseEvent(domEl: AbstractDomElement): void {
        domEl.addEventListener('mousedown', (e: MouseEvent) => {
            var target = <HTMLElement>e.target;
            if (target.nodeName == "IMG" && target.className == 'selected-td-img') {
                this.getGraphicView().getOptStateDelegate().getCurrentState().mousedown(e)
                this.operationDispatcher.eval("Macro_ShowColumnList()");
            } else {
                this.getGraphicView().getOptStateDelegate().getCurrentState().mousedown(e)
            }
        });
        domEl.addEventListener('mousemove', (e: MouseEvent) => this.getGraphicView().getOptStateDelegate().getCurrentState().mousemove(e));
        domEl.addEventListener('mouseup', (e: MouseEvent) => this.getGraphicView().getOptStateDelegate().getCurrentState().mouseup(e));
    }


    private initToolOptHandler(item: JComponent<any>, printGraphicView: PrintGraphicView): void {
        item.addEventListener(Events.MouseDown, (event: any) => {
            event = event || window.event;
            var action = item.getKey();
            switch (action) {
                case PrintConstants.NODE_Paragraph_Delete:
                    this.doDeleteElementCmd(action, printGraphicView);
                    break;
                case PrintConstants.NODE_Row_Delete:
                    this.doDeleteElementCmd(action, printGraphicView);
                    break;
                case PrintConstants.NODE_Column_Delete:
                    this.doDeleteElementCmd(action, printGraphicView);
                    break;
                case PrintConstants.NODE_AddParagraph_After:
                    this.doAddNewSectionElementCmd(action, printGraphicView);
                    break;
                case  PrintConstants.NODE_AddParagraph_Before :
                    this.doAddNewSectionElementCmd(action, printGraphicView);
                    break;
                case  PrintConstants.NODE_AddRow_Before :
                    this.doAddNewRowElementCmd(action, printGraphicView);
                    break;
                case  PrintConstants.NODE_AddRow_After :
                    this.doAddNewRowElementCmd(action, printGraphicView);
                    break;
                case  PrintConstants.NODE_AddColumn_After :
                    this.doAddNewColumnElementCmd(action, printGraphicView);
                    break;
                case  PrintConstants.NODE_AddColumn_Before :
                    this.doAddNewColumnElementCmd(action, printGraphicView);
                    break;
                case PrintConstants.NODE_Column_Porporty:
                    var exp = "Macro_ShowColumnView();"
                    this.operationDispatcher.eval(exp)
                    break;
                case PrintConstants.NODE_Row_Porporty:
                    var exp = "Macro_ShowRowView();"
                    this.operationDispatcher.eval(exp)
                    break;
                case PrintConstants.NODE_Para_Porporty:
                    var exp = "Macro_ShowSectionView();"
                    this.operationDispatcher.eval(exp)
                    break;
                case PrintConstants.NODE_Cell_Data:
                    var exp = "Macro_ShowReportCellInfo();"
                    this.operationDispatcher.eval(exp)
                    break;
                case PrintConstants.NODE_Cell_DataStyle:
                    var exp = "Macro_ShowReportCellFormat();"
                    this.operationDispatcher.eval(exp)
                    break;
                case PrintConstants.NODE_Cell_ViewStyle:
                    var exp = "Macro_ShowReportCellDisplay();"
                    this.operationDispatcher.eval(exp)
                    break;
                case PrintConstants.Node_Page_Opt:
                    var exp ="ShowModal('D_ReportSetting',{FormKey:{GetPrintAttrValue('FormKey')},ReportKey:{GetPrintAttrValue('Key')}, XML:{GetReportXML()}, ShowPage:{'true'}});"
                    this.operationDispatcher.eval(exp)
                    break;
                case PrintConstants.Node_Paper_Opt:
                    var exp ="ShowModal('D_ReportSetting',{FormKey:{GetPrintAttrValue('FormKey')},ReportKey:{GetPrintAttrValue('Key')}, XML:{GetReportXML()}, ShowPage:{'false'}});"
                    this.operationDispatcher.eval(exp)
                    break;
            }

            printGraphicView.hideToolbar();
        }, true);
    }

    //删除
    public doDeleteElementCmd(action: string, printGraphicView: PrintGraphicView): any {
        var cmd = new DeleteElementCmd(printGraphicView, this.getSelectionModel().doCacheTagNodes(), action);
        this.getCmdQueue().doCmd(cmd);
    }

    //新增段
    public doAddNewSectionElementCmd(action: string, printGraphicView: PrintGraphicView): any {
        var cmd = new NewSectionElementCmd(printGraphicView, this.getSelectionModel().doCacheTagNodes(), action);
        this.getCmdQueue().doCmd(cmd);
    }

    //新增行
    public doAddNewRowElementCmd(action: string, printGraphicView: PrintGraphicView): any {
        var cmd = new NewRowElementCmd(printGraphicView, this.getSelectionModel().doCacheTagNodes(), action);
        this.getCmdQueue().doCmd(cmd);
    }

    //新增行
    doAddNewColumnElementCmd(action: string, printGraphicView: PrintGraphicView): any {
        var cmd = new NewColumnElementCmd(printGraphicView, this.getSelectionModel().doCacheTagNodes(), action);
        this.getCmdQueue().doCmd(cmd);
    }

    installToolbar(toolbar: Toolbar, viewContainer: ComplexDesignView<any, any>): void {
        var btnUndo = new Button("撤销");
        //btnUndo.setTitle("快捷键 Ctrl+z");
        btnUndo.setOnClickListener((e: MouseEvent) => viewContainer.undo());
        toolbar.addItem(btnUndo);
        var btnRedo = new Button("重做");
        //btnRedo.setTitle("快捷键 Ctrl+y");
        btnRedo.setOnClickListener((e: MouseEvent) => viewContainer.redo());
        toolbar.addItem(btnRedo);
        // 分隔符
        toolbar.addSeparator();
        var btnSwitch = new Button("源码");
        btnSwitch.setOnClickListener((e: MouseEvent) => {
            viewContainer.toggleSourceVisible();
            btnSwitch.setCaption(viewContainer.isSourceVisible() ? "视图" : "源码");
        });
        toolbar.addItem(btnSwitch);
        var btnDataSource = new Button("数据源");
        btnDataSource.setOnClickListener((e: MouseEvent) => {
            var exp = "Macro_ShowReportDataSource();"
            this.operationDispatcher.eval(exp)
        });
        toolbar.addItem(btnDataSource);
    }

    installPrintToolbar(toolbar: PrintToolbar): void {
        //文本操作
        var h5VFlexPanel = new PrintH5HFlexPanel(toolbar, "TestPanel");
        toolbar.addComponent(h5VFlexPanel, 1, 1, 5, 1)
        var pointer = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/print_pointer.png") center no-repeat', "指针", "pointer");
        //h5VFlexPanel.addComponent(pointer);
        var insertText = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/set_text_field.png") center no-repeat', "插入文本", "insertText");
        //h5VFlexPanel.addComponent(insertText);
        var insertPicture = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/insert_picture.png") center no-repeat', "插入图片", "insertPicture");
       // h5VFlexPanel.addComponent(insertPicture);
        var insertChart = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/insert_chart.png") center no-repeat', "插入图表", "insertChart");
     //   h5VFlexPanel.addComponent(insertChart);

        var merge_cell = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/merge_cell.png") center no-repeat', "合并", "merge_cell");
        merge_cell.setOnClickListener((e: MouseEvent) => this.mergeCell());
        h5VFlexPanel.addComponent(merge_cell);
        var split_cell = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/split_cell.png") center no-repeat', "分割", "split_cell");
        split_cell.setOnClickListener((e: MouseEvent) => this.divisionCell());
        h5VFlexPanel.addComponent(split_cell);
        var left_align = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/left_align.png") center no-repeat', "左对齐", "left_align");
        left_align.setOnClickListener((e: MouseEvent) => this.align_click("Display", "HAlign", "Left","Center"));
        h5VFlexPanel.addComponent(left_align);
        var center_align = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/center_align.png") center no-repeat', "水平居中", "center_align");
        center_align.setOnClickListener((e: MouseEvent) => this.align_click("Display", "HAlign", "Center","Center"));
        h5VFlexPanel.addComponent(center_align);
        var right_align = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/right_align.png") center no-repeat', "右对齐", "right_align");
        right_align.setOnClickListener((e: MouseEvent) => this.align_click("Display", "HAlign", "Right","Center"));
        h5VFlexPanel.addComponent(right_align);
        var top_align = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/top_align.png") center no-repeat', "顶对齐", "top_align");
        top_align.setOnClickListener((e: MouseEvent) => this.align_click("Display", "VAlign", "Top","Center"));
        h5VFlexPanel.addComponent(top_align);
        var vcenter_align = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/vcenter_align.png") center no-repeat', "顶居中", "vcenter_align");
        vcenter_align.setOnClickListener((e: MouseEvent) => this.align_click("Display", "VAlign", "Center","Center"));
        h5VFlexPanel.addComponent(vcenter_align);
        var bottom_align = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/bottom_align.png") center no-repeat', "底对齐", "bottom_align");
        bottom_align.setOnClickListener((e: MouseEvent) => this.align_click("Display", "VAlign", "Bottom","Center"));
        h5VFlexPanel.addComponent(bottom_align);
        h5VFlexPanel.setStyle("position", "relative");
        h5VFlexPanel.addClass("bpm-toolbar")
        //字体
        var font = new PrintCombobox(130, "字体", "Name");
        font.setOnValueChanged((e: MouseEvent) => this.align_click("Display;Font", "Name", font.getValue()));
        font.initControl("宋体:宋体;新宋体:新宋体;仿宋:仿宋;黑体:黑体;微软雅黑:微软雅黑;隶书:隶书;楷体:楷体;Agency FB:Agency" +
            " FB;Algerian:Algerian;Arial:Arial;Arial Black:Arial Black;Arial Narrow:Arial Narrow;Arial:Arial;Arial Black:Arial Black;Arial Narrow:Arial Narrow;Book Antiqua:Book Antiqua;Bookman Old Style:Bookman Old Style;Bookshelf Symbol 7:Bookshelf Symbol 7;Bradley Hand ITC:Bradley Hand ITC;Calibri:Calibri;Calibri Light:Calibri Light;Cambria:Cambria;Cambria Math:Cambria Math;Candara:Candara;Candara Light:Candara Light;Century:Century;Century Gothic:Century Gothic;Comic Sans MS:Comic Sans MS;Consolas:Consolas;Constantia:Constantia;Corbel:Corbel;Corbel Light:Corbel Light;Courier New:Courier New;DejaVu Sans Mono:DejaVu Sans Mono;DengXian:DengXian;DengXian Light:DengXian Light;Dubai:Dubai;Dubai Light:Dubai Light;Dubai Medium:Dubai Medium;Ebrima:Ebrima;FangSong:FangSong;Franklin Gothic Medium:Franklin Gothic Medium;Freestyle Script:Freestyle Script;French Script MT:French Script MT;Gabriola:Gabriola;Gadugi:Gadugi;Garamond:Garamond;Georgia:Georgia;HoloLens MDL2 Assets:HoloLens MDL2 Assets;Impact:Impact;Ink Free:Ink Free;Javanese Text:Javanese Text;Juice ITC:Juice ITC;KaiTi:KaiTi;Kristen ITC:Kristen ITC;Leelawadee UI:Leelawadee UI;Leelawadee UI Semilight:Leelawadee UI Semilight;Lucida Bright:Lucida Bright;Lucida Console:Lucida Console;Lucida Handwriting:Lucida Handwriting;华文中宋:华文中宋;华文仿宋:华文仿宋;华文宋体:华文宋体;华文彩云:华文彩云;华文新魏:华文新魏;华文楷体:华文楷体;华文琥珀:华文琥珀;华文细黑:华文细黑;华文行楷:华文行楷;华文隶书:华文隶书;幼圆:幼圆;方正姚体:方正姚体;方正舒体:方正舒体;隶书:隶书            ")
        h5VFlexPanel.addComponent(font)
        //字体大小
        var fontSize = new PrintCombobox(50, "字体大小", "Size");
        fontSize.setOnValueChanged((e: MouseEvent) => this.align_click("Display;Font", "Size", fontSize.getValue(),"12"));
        fontSize.initControl("8;9;10;11;12;14;16;18;20;22;24;26;28;36;48;72")
        h5VFlexPanel.addComponent(fontSize)


        //加粗

        var bold = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/bold.png") center no-repeat', "粗体", "Bold");
        bold.setOnClickListener((e: MouseEvent) => this.align_click("Display;Font", "Bold", String(!TypeUtil.toBoolean(bold.getValue())),"false"));
        h5VFlexPanel.addComponent(bold);
        var italic = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/italic.png") center no-repeat', "斜体", "Italic");
        italic.setOnClickListener((e: MouseEvent) => this.align_click("Display;Font", "Italic", String(!TypeUtil.toBoolean(italic.getValue())),"false"));
        h5VFlexPanel.addComponent(italic);
        //下划线
        var underline = new PrintButtonBox("UnderLine", ['url("' + PRINT.PREFIX + '/css/img/border_none.png") center no-repeat',
                'url("' + PRINT.PREFIX + '/css/img/border_style1.png") center no-repeat',
                'url("' + PRINT.PREFIX + '/css/img/border_style_dashed.png") center no-repeat',
                'url("' + PRINT.PREFIX + '/css/img/border_style_medium.png") center no-repeat'],
            SizeInfo.valueOfPX(70), SizeInfo.valueOfPX(100));
        underline.setTitle("下划线");
        underline.setWidth(SizeInfo.valueOfPX(50))
        underline.setHeight(SizeInfo.valueOfPX(26))
        underline.setValue("1");
        underline.buttonText.setStyle("background", 'url("' + PRINT.PREFIX + '/css/img/underline.png") center no-repeat');
        underline.buttonText.setStyle(" background-size", '100%');
        underline.buttonText.setOnClickListener((e: MouseEvent) => {
            var value = underline.buttonText.getValue();
            switch (value) {
                case "0":
                    value = "None";
                    break;
                case "1":
                    value = "SolidNormal";
                    break;
                case "2":
                    value = "Dashed";
                    break;
                case "3":
                    value = "SolidBold";
                    break;

            }
            this.align_click("Display;Font", "UnderLine", value,"None")
        });
        underline.setOnValueChanged((e: MouseEvent) => {
            //underline.buttonText.setValue(underline.button.getValue());
            var value = underline.buttonText.getValue();
            switch (value) {
                case "0":
                    value = "None";
                    break;
                case "1":
                    value = "SolidNormal";
                    break;
                case "2":
                    value = "Dashed";
                    break;
                case "3":
                    value = "SolidBold";
                    break;

            }
            this.align_click("Display;Font", "UnderLine", value,"None");
        });
        h5VFlexPanel.addComponent(underline);

        //显示类型
        var showType = new PrintCombobox(130, "显示类型", "ShowType");
        showType.setOnValueChanged((e: MouseEvent) => this.align_click("Display", "ShowType", showType.getValue()));
        showType.initControl("TEXT:TEXT;AZTEC:AZTEC;CODABAR:CODABAR;CODE_39:CODE_39;CODE_93:CODE_93;CODE_128:CODE_128;DATA_MATRIX:DATA_MATRIX;EAN_8:EAN_8;EAN_13:EAN_13;ITF:ITF;MAXICODE:MAXICODE;PDF_417:PDF_417;QR_CODE:QR_CODE;RSS_14:RSS_14;RSS_EXPANDED:RSS_EXPANDED;UPC_A:UPC_A;UPC_E:UPC_E;UPC_EAN_EXTENSION:UPC_EAN_EXTENSION")
        h5VFlexPanel.addComponent(showType)

        //编码格式
        var utf = new PrintCombobox(130, "编码格式", "EncodingType");
        utf.setOnValueChanged((e: MouseEvent) => this.align_click("Display", "EncodingType", utf.getValue(),"UTF-8"));
        utf.initControl("UTF-8:UTF-8;utf8:utf8;GBK:GBK;GBK2312:GBK2312;ISO8859_11:ISO8859_11;Cp437:Cp437;ISO8859_13:ISO8859_13;ISO8859_15:ISO8859_15;ASCII:ASCII;Cp1256:Cp1256;EUC_CN:EUC_CN;Cp1251:Cp1251;Cp125:Cp125;2Cp1250:2Cp1250;EUC_KR:EUC_KR;Shift_JIS:Shift_JIS;windows-1251:windows-1251;windows-1250:windows-1250;ISO-8859-13:ISO-8859-13;windows-1252:windows-1252;ISO-8859-11:ISO-8859-11;windows-1256:windows-1256;lSO-8859-15:lSO-8859-15;US-ASCII:US-ASCII;ISO8859_4:ISO8859_4;ISO8859_5:ISO8859_5;ISO8859_2:ISO8859_2;lSO8859_3:lSO8859_3;lSO8859_1:lSO8859_1;UTF8:UTF8;Big5:Big5;UnicodeBig:UnicodeBig;ISO8859_8:ISO8859_8;ISO8859_9:ISO8859_9;GB18030:GB18030;ISO8859_6:ISO8859_6;UnicodeBigUnmarked:UnicodeBigUnmarked;ISO8859_7:ISO8859_7;lSO-8859-4:lSO-8859-4;ISO-8859-3:ISO-8859-3;lSO-8859-2:lSO-8859-2;ISO-8859-1:ISO-8859-1;ISO-8859-8:ISO-8859-8;lSO-8859-7:lSO-8859-7;ISO-8859-6:ISO-8859-6;ISO-8859-5:ISO-8859-5;SJIS:SJIS;EUC-KR:EUC-KR;ISO-8859-9:ISO-8859-9;UTF-16BE:UTF-16BE")
        h5VFlexPanel.addComponent(utf)
        //编码格式
        var errorCorrectionLevel = new PrintCombobox(60, "错误更正级别", "ErrorCorrectionLevel");
        errorCorrectionLevel.setOnValueChanged((e: MouseEvent) => this.align_click("Display", "ErrorCorrectionLevel", errorCorrectionLevel.getValue(),"L"));

        errorCorrectionLevel.initControl("L:L;M:M;Q:Q;H:H")
        h5VFlexPanel.addComponent(errorCorrectionLevel)
        //编码格式
        var margin = new PrintCombobox(40, "Margin", "Margin");
        margin.setOnValueChanged((e: MouseEvent) => this.align_click("Display", "Margin", margin.getValue(),"0"));
        margin.initControl("0:0;1:1;2:2;3:3;4:4")
        h5VFlexPanel.addComponent(margin)
        var h5VFlexPanel1 = new PrintH5HFlexPanel(toolbar, "StylePanel");
        toolbar.addComponent(h5VFlexPanel1, 1, 2, 5, 1)
        h5VFlexPanel1.setStyle("position", "relative");
        h5VFlexPanel1.addClass("bpm-toolbar")

        //段类型
        var colLabel = new Label("段类型");
        colLabel.setWidth(SizeInfo.valueOfPX(50));
        colLabel.addClass("print-label");
        h5VFlexPanel1.addComponent(colLabel);
        var colType = new PrintCombobox(80, "段类型", "colType");
        colType.initControl("PageHead:页眉;Paragraph:段落;Table:表格;PageTail:页尾");
        colType.setOnValueChanged((e: MouseEvent) => this.section_click("Type", colType.getValue()));

        h5VFlexPanel1.addComponent(colType);
        //行类型
        var rowLabel = new Label("行类型");
        rowLabel.setWidth(SizeInfo.valueOfPX(50));
        rowLabel.addClass("print-label");
        h5VFlexPanel1.addComponent(rowLabel);
        var rowType = new PrintCombobox(80, "行类型", "rowType");
        rowType.initControl("TableTitleHead:表格标题头;DetailHead:明细行头;TableParagraph:段落;LinkPrev:承上行;Detail:明细行;Group:分组行;LinkNext:启下行;DetailTail:明细尾;TableTitleTail:表格标题尾")
        rowType.setOnValueChanged((e: MouseEvent) => this.row_click("Type", rowType.getValue()));
        h5VFlexPanel1.addComponent(rowType)
        var append_row = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/append_row.png") center no-repeat', "添加行", "append_row");
        append_row.setOnClickListener((e: MouseEvent) => this.doAddNewRowElementCmd(PrintConstants.NODE_AddRow_Before, this.viewContainer.getGraphicView()));
        h5VFlexPanel1.addComponent(append_row);

        var insert_row = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/insert_row.png") center no-repeat', "插入行", "insert_row");
        h5VFlexPanel1.addComponent(insert_row);
        insert_row.setOnClickListener((e: MouseEvent) => this.doAddNewRowElementCmd(PrintConstants.NODE_AddRow_After, this.viewContainer.getGraphicView()));
        var delete_row = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/delete_row.png") center no-repeat', "删除行", "delete_row");
        h5VFlexPanel1.addComponent(delete_row);
        delete_row.setOnClickListener((e: MouseEvent) => this.doDeleteElementCmd(PrintConstants.NODE_Row_Delete, this.viewContainer.getGraphicView()));


        var append_column = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/append_column.png") center no-repeat', "添加列", "append_column");
        h5VFlexPanel1.addComponent(append_column);
        append_column.setOnClickListener((e: MouseEvent) => this.doAddNewColumnElementCmd(PrintConstants.NODE_AddColumn_Before, this.viewContainer.getGraphicView()));
        var insert_column = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/insert_column.png") center no-repeat', "插入列", "insert_column");
        h5VFlexPanel1.addComponent(insert_column);
        insert_column.setOnClickListener((e: MouseEvent) => this.doAddNewColumnElementCmd(PrintConstants.NODE_AddColumn_After, this.viewContainer.getGraphicView()));
        var delete_column = new PrintButton('', 'url("' + PRINT.PREFIX + '/css/img/delete_column.png") center no-repeat', "删除列", "delete_column");
        h5VFlexPanel1.addComponent(delete_column);
        delete_column.setOnClickListener((e: MouseEvent) => this.doDeleteElementCmd(PrintConstants.NODE_Column_Delete, this.viewContainer.getGraphicView()));

        var images1 = ['url("' + PRINT.PREFIX + '/css/img/border_none.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/border_style1.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/border_style_dashed.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/border_style_medium.png") center no-repeat']
        var border_style = new PrintButtonBox("border_style", images1,
            SizeInfo.valueOfPX(70), SizeInfo.valueOfPX(100));
        border_style.setTitle("边框类型");
        border_style.setValue("1");
        border_style.setStyle("margin-top", "2px");
        border_style.setWidth(SizeInfo.valueOfPX(95))
        border_style.setHeight(SizeInfo.valueOfPX(26))
        border_style.buttonText.setStyle("background", 'url("' + PRINT.PREFIX + '/css/img/border_style1.png") center no-repeat');
        border_style.buttonText.setStyle(" background-size", '100%');
        border_style.setOnValueChanged((e: MouseEvent) => {
            var value = border_style.buttonText.getValue();
            this.border_click("Display;Border", ["TopStyle", "BottomStyle", "LeftStyle", "RightStyle"], value,"0");
            border_style.buttonText.setStyle("background", images1[value]);
        });
        border_style.buttonText.setOnClickListener((e: MouseEvent) => {
            var value = border_style.buttonText.getValue();
            this.border_click("Display;Border", ["TopStyle", "BottomStyle", "LeftStyle", "RightStyle"], value,"0");
        });
        h5VFlexPanel1.addComponent(border_style);
        //颜色
        var colorLabel = new Label("边框颜色");
        colorLabel.setWidth(SizeInfo.valueOfPX(70));
        colorLabel.addClass("print-label")
        colorLabel.setKey("BorderColor")
        h5VFlexPanel1.addComponent(colorLabel);


        var colorEditor = new PrintColorEditor();
        colorEditor.addClass("print-colorEditor");
        colorEditor.setWidth(SizeInfo.valueOfPX(100));
        colorEditor.setHeight(SizeInfo.valueOfPX(23));

        //边框颜色
        colorEditor.setOnValueChanged((e: MouseEvent) => {
            var border_style_value = border_style.buttonText.getValue() || "1";
            if (border_style_value == "-1") {//无线样式
                return;
            }
            this.border_color_click("Display;Border", ["TopColor", "BottomColor", "LeftColor", "RightColor"], colorEditor.getValue());
        })
        h5VFlexPanel1.addComponent(colorEditor);
        var images = [
            'url("' + PRINT.PREFIX + '/css/img/cell_full_border.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/cell_no_border.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/cell_full_out_border.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/cell_left_out_border.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/cell_right_out_border.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/cell_top_out_border.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/cell_bottom_out_border.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/cell_left_border.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/cell_right_border.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/cell_top_border.png") center no-repeat',
            'url("' + PRINT.PREFIX + '/css/img/cell_bottom_border.png") center no-repeat'
        ];

        var cell_full_border = new PrintButtonBox("cell_full_border", images, SizeInfo.valueOfPX(51), SizeInfo.valueOfPX(286));
        cell_full_border.setTitle("边框模式")
        cell_full_border.setKey("BorderStyle");
        cell_full_border.setValue("1");
        cell_full_border.setWidth(SizeInfo.valueOfPX(50));
        cell_full_border.setHeight(SizeInfo.valueOfPX(26));
        cell_full_border.buttonText.setStyle("background", 'url("' + PRINT.PREFIX + '/css/img/cell_no_border.png") center no-repeat');
        cell_full_border.buttonText.setStyle(" background-size", '100%');
        //设置边框模式
        cell_full_border.setOnValueChanged((e: MouseEvent) => {
            var border_style_value = border_style.buttonText.getValue() || "1";
            var border_color_value = colorEditor.getValue() || "#000000";
            var cell_full_border_value = cell_full_border.buttonText.getValue() || "1";
            if (border_style_value == "-1") {//无线样式
                return;
            }
            this.style_border_click("Display;Border", ["TopColor", "BottomColor", "LeftColor", "RightColor", "TopStyle", "BottomStyle", "LeftStyle", "RightStyle"], border_style_value, border_color_value, cell_full_border_value,"0");
            cell_full_border.buttonText.setStyle("background", images[Number(cell_full_border_value)]);
        });
        cell_full_border.buttonText.setOnClickListener((e: MouseEvent) => {
            var border_style_value = border_style.buttonText.getValue() || "1";
            var border_color_value = colorEditor.getValue() || "#000000";
            var cell_full_border_value = cell_full_border.buttonText.getValue() || "1";
            if (border_style_value == "-1") {//无线样式
                return;
            }
            this.style_border_click("Display;Border", ["TopColor", "BottomColor", "LeftColor", "RightColor", "TopStyle", "BottomStyle", "LeftStyle", "RightStyle"], border_style_value, border_color_value, cell_full_border_value,"0");
        });
        h5VFlexPanel1.addComponent(cell_full_border);
        var qcolorLabel = new Label("前景色");
        qcolorLabel.setKey("qcolorLabel");
        qcolorLabel.setWidth(SizeInfo.valueOfPX(50));
        qcolorLabel.addClass("print-label")


        h5VFlexPanel1.addComponent(qcolorLabel);
        var qcolorEditor = new PrintColorEditor();
        qcolorEditor.setKey("ForeColor")
        qcolorEditor.setWidth(SizeInfo.valueOfPX(100));
        qcolorEditor.setHeight(SizeInfo.valueOfPX(23));
        qcolorEditor.addClass("print-colorEditor")
        qcolorEditor.setOnValueChanged((e: MouseEvent) => {
            this.align_click("Display", "ForeColor", qcolorEditor.getValue());

        })

        h5VFlexPanel1.addComponent(qcolorEditor);


        var bcolorLabel = new Label("背景色");
        bcolorLabel.setKey("bcolorLabel");
        bcolorLabel.setWidth(SizeInfo.valueOfPX(50));
        bcolorLabel.addClass("print-label")
        h5VFlexPanel1.addComponent(bcolorLabel);

        var bcolorEditor = new PrintColorEditor();
        bcolorEditor.setKey("BackColor")
        bcolorEditor.setWidth(SizeInfo.valueOfPX(100));
        bcolorEditor.setHeight(SizeInfo.valueOfPX(23));
        bcolorEditor.addClass("print-colorEditor")
        bcolorEditor.setOnValueChanged((e: MouseEvent) => {
            this.align_click("Display", "BackColor", bcolorEditor.getValue());

        })
        h5VFlexPanel1.addComponent(bcolorEditor);

    }


    save(): void {
        var exp = `SavePrint()`;
        // 对页面元素进行检查
        PrintNodeCheckUtil.init();
        let root = <ITagNode>this.getXmlEntity().getRoot().getChildren()[1];
        PrintNodeCheckUtil.nodeCheck(root);
        if(!StringUtil.isBlankOrNull(PrintNodeCheckUtil.msg)){
            //@ts-ignore
            layui.layer.msg(PrintNodeCheckUtil.msg, {icon: 5, time: 5000,})
            throw new Error(PrintNodeCheckUtil.msg);
        }
        this.operationDispatcher.eval(exp);
    }



    getCmdQueue(): CmdQueue {
        return this.viewContainer.getGraphicView().getXmlEntity().getCmdQueue();
    }

    align_click(tagName: string, type: string, alignmentValue: string,defaultValue?:string): void {
        this.getSelectionModel().getSingleSeletionItem();
        var iTagNodes = this.getSelectionModel().doCacheTagNodes();
        var cmd = new SetAlignmentCmd(tagName, this.getXmlEntity(), iTagNodes, alignmentValue, [""], type,defaultValue);
        this.getCmdQueue().doCmd(cmd);
    }

    divisionCell(): void {
        this.getSelectionModel().getSingleSeletionItem();
        var iTagNodes = this.getSelectionModel().doCacheTagNodes();
        var cmd = new DivisionCellCmd(this.getXmlEntity(), iTagNodes);
        this.getCmdQueue().doCmd(cmd);
    }

    mergeCell(): void {
        this.getSelectionModel().getSingleSeletionItem();
        var iTagNodes = this.getSelectionModel().doCacheTagNodes();
        var cmd = new MergeCellCmd(this.getXmlEntity(), iTagNodes, this);
        this.getCmdQueue().doCmd(cmd);
    }

    border_click(tagName: string, type: string[], alignmentValue: string,defaultValue?:string): void {
        this.getSelectionModel().getSingleSeletionItem();
        var iTagNodes = this.getSelectionModel().doCacheTagNodes();
        var cmd = new SetBorderCmd(tagName, this.getXmlEntity(), iTagNodes, alignmentValue, type,defaultValue);
        this.getCmdQueue().doCmd(cmd);
    }

    style_border_click(tagName: string, type: string[], style_border: string, border_color: string, border_style: string,defaultValue:string): void {
        this.getSelectionModel().getSingleSeletionItem();
        var iTagNodes = this.getSelectionModel().doCacheTagNodes();
        var cmd = new SetStyleBorderCmd(this.getGraphicView(), tagName, this.getXmlEntity(), iTagNodes, style_border, border_color, border_style, [[]], type,defaultValue);
        this.getCmdQueue().doCmd(cmd);
    }

    border_color_click(tagName: string, type: string[], alignmentValue: string): void {
        this.getSelectionModel().getSingleSeletionItem();
        var iTagNodes = this.getSelectionModel().doCacheTagNodes();
        var cmd = new SetBorderColorCmd(tagName, this.getXmlEntity(), iTagNodes, alignmentValue, type);
        this.getCmdQueue().doCmd(cmd);
    }

    section_click(type: string, alignmentValue: string): void {
        var iTagNode = this.getSelectionModel().getSingleSeletionItem();
        var cmd = new SetSectionCmd(this.getXmlEntity(), <ITagNode>iTagNode, alignmentValue, "", type);
        this.getCmdQueue().doCmd(cmd);
    }

    row_click(type: string, alignmentValue: string): void {
        var iTagNode = this.getSelectionModel().getSingleSeletionItem();
        var cmd = new SetRowCmd(this.getXmlEntity(), <ITagNode>iTagNode, alignmentValue, "", type);
        this.getCmdQueue().doCmd(cmd);
    }

    delete(): void {
        var exp = `DeleteBpm()`;
        this.operationDispatcher.eval(exp);
    }

    deleteElement(): void {
        throw new Error("Method not implemented.");
    }

    private DEMO_XML: string = `<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<Report BottomMargin="15" Caption="test112" FormKey="FormulaDefine" Key="test112" LeftMargin="25" PageHeight="842" PageID="A4" PageWidth="595" PaperHeight="842" PaperID="A4" PaperWidth="595" RightMargin="25" TopMargin="15">
    <DataSource>
        <Table Key="ERP_FormulaDefine" Caption="公式定义主表" SourceType="DataObject">
            <Field Caption="对象标识" Key="OID"/>
            <Field Caption="主对象标识" Key="SOID"/>
            <Field Caption="父对象标识" Key="POID"/>
            <Field Caption="对象版本" Key="VERID"/>
            <Field Caption="对象明细版本" Key="DVERID"/>
            <Field Caption="启用标记" Key="Enable"/>
            <Field Caption="节点类型" Key="NodeType"/>
            <Field Caption="TLeft" Key="TLeft"/>
            <Field Caption="TRight" Key="TRight"/>
            <Field Caption="上级节点" Key="ParentID"/>
            <Field Caption="代码" Key="Code"/>
            <Field Caption="名称" Key="Name"/>
            <Field Caption="名称(外文)" Key="ForeignName"/>
            <Field Caption="创建人员" Key="Creator"/>
            <Field Caption="创建时间" Key="CreateTime"/>
            <Field Caption="修改人员" Key="Modifier"/>
            <Field Caption="修改时间" Key="ModifyTime"/>
            <Field Caption="备注" Key="Notes"/>
            <Field Caption="集团" Key="ClientID"/>
            <Field Caption="Define" Key="Define"/>
            <Field Caption="公式类型" Key="FormulaType"/>
            <Field Caption="公式来源类型" Key="FormulaSourceType"/>
        </Table>
        <Table Key="ERP_FormulaParaDtl" Caption="参数列表" SourceType="DataObject">
            <Field Caption="对象标识" Key="OID"/>
            <Field Caption="主对象标识" Key="SOID"/>
            <Field Caption="父对象标识" Key="POID"/>
            <Field Caption="对象版本" Key="VERID"/>
            <Field Caption="对象明细版本" Key="DVERID"/>
            <Field Caption="ParaKey" Key="ParaKey"/>
            <Field Caption="ParaNote" Key="ParaNote"/>
            <Field Caption="IsFormulaPara" Key="IsFormulaPara"/>
        </Table>
        <Table Key="ERP_FormulaOperatorDtl" Caption="操作员列表" SourceType="DataObject">
            <Field Caption="对象标识" Key="OID"/>
            <Field Caption="主对象标识" Key="SOID"/>
            <Field Caption="父对象标识" Key="POID"/>
            <Field Caption="对象版本" Key="VERID"/>
            <Field Caption="对象明细版本" Key="DVERID"/>
            <Field Caption="Operator" Key="Operator"/>
        </Table>
    </DataSource>
    <Grid>
        <Section Key="FormulaDefine" Type="Paragraph">
            <Columns>
                <Column Width="540"/>
            </Columns>
            <Rows>
                    <Row Height="50">
                        <Cell Caption="公式定义">
                            <Display>
                                <Font Bold="true" Name="SimSun" Size="14"/>
                                <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                        <CellText>
                            <![CDATA[公式定义]]>
                        </CellText>
                    </Cell>
                </Row>
            </Rows>
        </Section>
        <Section Key="body_basic" Type="Paragraph">
            <Columns>
                <Column Width="67"/>
                <Column Width="67"/>
                <Column Width="67"/>
                <Column Width="67"/>
                <Column Width="67"/>
                <Column Width="67"/>
                <Column Width="67"/>
                <Column Width="71"/>
            </Columns>
            <Rows>
                <Row Height="30">
                    <Cell Caption="代码" Key="Inner_Code">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="3" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="代码" FieldKey="Code" Key="Code" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="名称" Key="Inner_Name" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="名称" FieldKey="Name" IsMerged="true" IsMergedHead="true" Key="Name" MergedColumnSpan="3" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="2">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="分组模块" Key="L_ParentID" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="分组模块" FieldKey="ParentID" Key="ParentID" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Format DataType="Dict" FieldKeys="Name" ItemKey="FormulaDefine"/>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
                <Row Height="30">
                    <Cell Caption="公式类型" Key="Lab_FormulaType">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="3" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="公式类型" FieldKey="FormulaType" Key="FormulaType" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Format DataType="List">
                                <ListItem Text="取值" Value="VALUE"/>
                                <ListItem Text="条件" Value="VALUE_CONDITION"/>
                                <ListItem Text="行为" Value="ACTION"/>
                                <ListItem Text="参与者" Value="PARTICIPATOR"/>
                            </Format>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="表达式" Key="Formula" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="SQL" Key="SQL" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="固定值" Key="Const" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="表格" Key="Grid" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="FormulaSourceType" FieldKey="FormulaSourceType" Key="FormulaSourceType" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell>
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
                <Row Height="30">
                    <Cell Caption="定义" Key="Lab_Define">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell FieldKey="Define" IsMerged="true" IsMergedHead="true" Key="Define" MergedColumnSpan="7" MergedRowSpan="2" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                        <CellText>
                            <![CDATA[定义]]>
                        </CellText>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="2">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="3">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="4">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="5">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="6">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
                <Row Height="30">
                    <Cell>
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="2" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="3" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="4" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="5" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell IsMerged="true" MergedColumnSpan="6" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
                <Row Height="30">
                    <Cell Caption="名称(外文)" Key="Inner_ForeignName" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="名称(外文)" FieldKey="ForeignName" Key="ForeignName" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="节点类型" Key="Inner_NodeType" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="节点类型" FieldKey="NodeType" Key="NodeType" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Format DataType="List">
                                <ListItem Text="明细节点" Value="0"/>
                                <ListItem Text="汇总节点" Value="1"/>
                            </Format>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell>
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell>
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell>
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell>
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
            </Rows>
        </Section>
        <Section Key="FormulaPara" Type="Table">
            <Columns>
                <Column Width="180"/>
                <Column Width="180"/>
                <Column Width="180"/>
            </Columns>
            <Rows>
                <Row Height="30" Type="DetailHead">
                    <Cell Caption="公式">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="参数">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="参数说明">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
                <Row Height="30" TableKey="ERP_FormulaParaDtl" Type="Detail">
                    <Cell Caption="公式" FieldKey="IsFormulaPara" Key="IsFormulaPara" SourceType="Field" TableKey="ERP_FormulaParaDtl">
                        <Display>
                            <Format DataType="List">
                                <ListItem Text="公式 (是) " Value="1"/>
                                <ListItem Text="公式 (否) " Value="0"/>
                            </Format>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell FieldKey="ParaKey" Key="Para" SourceType="Field" TableKey="ERP_FormulaParaDtl">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                        <CellText>
                            <![CDATA[参数]]>
                        </CellText>
                    </Cell>
                    <Cell Caption="参数说明" FieldKey="ParaNote" Key="ParaNote" SourceType="Field" TableKey="ERP_FormulaParaDtl">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
            </Rows>
        </Section>
        <Section Key="OperatorGrid" Type="Table">
            <Columns>
                <Column Width="540"/>
            </Columns>
            <Rows>
                <Row Height="30" Type="DetailHead">
                    <Cell Caption="操作员">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
                <Row Height="31" TableKey="ERP_FormulaOperatorDtl" Type="Detail">
                    <Cell Caption="操作员" FieldKey="Operator" Key="Operator" SourceType="Field" TableKey="ERP_FormulaOperatorDtl">
                        <Display>
                            <Format DataType="Dict" FieldKeys="Name" ItemKey="Operator"/>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
            </Rows>
        </Section>
         <!--<Image Key="Image1" Caption="测试Image" X="60" Y="60" Width="50" Height="50"/>
    <Chart Key="Chart1" Caption="测试Chart" X="70" Y="70" Width="50" Height="50"/>-->
        <Section Key="body_system" Type="Paragraph">
            <Columns>
                <Column Width="286"/>
                <Column Width="255"/>
            </Columns>
            <Rows>
                <Row Height="30">
                    <Cell Caption="集团" Key="Inner_ClientID" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell FieldKey="ClientID" Key="ClientID" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Font UnderLine="SolidNormal"/>
                            <Format DataType="Dict" FieldKeys="Name" ItemKey="Client"/>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                        <CellText>
                            <![CDATA[集团]]>
                        </CellText>
                    </Cell>
                </Row>
                <Row Height="30">
                    <Cell Caption="节点状态" Key="Inner_Enable" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="节点状态" FieldKey="Enable" Key="Enable" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Format DataType="List">
                                <ListItem Text="作废" Value="-1"/>
                                <ListItem Text="停用" Value="0"/>
                                <ListItem Text="启用" Value="1"/>
                            </Format>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
                <Row Height="33">
                    <Cell Caption="创建人员" Key="Inner_Creator" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="3" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="创建人员" FieldKey="Creator" Key="Creator" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Format DataType="Dict" FieldKeys="Name" ItemKey="Operator"/>
                            <Border BottomColor="#000000" BottomStyle="3" LeftColor="#000000" LeftStyle="3" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
                <Row Height="18">
                    <Cell Caption="创建时间" Key="Inner_CreateTime" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="3" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="创建时间" FieldKey="CreateTime" Key="CreateTime" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Format DataType="Date" FormatString="yyyy-MM-dd HH:mm:ss"/>
                            <Border BottomColor="#000000" BottomStyle="3" LeftColor="#000000" LeftStyle="3" RightColor="#000000" RightStyle="3" TopColor="#000000" TopStyle="3"/>
                        </Display>
                    </Cell>
                </Row>
                <Row Height="30">
                    <Cell Caption="修改人员" Key="Inner_Modifier" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="修改人员" FieldKey="Modifier" Key="Modifier" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Format DataType="Dict" FieldKeys="Name" ItemKey="Operator"/>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="3"/>
                        </Display>
                    </Cell>
                </Row>
                <Row Height="30">
                    <Cell Caption="修改时间" Key="Inner_ModifyTime" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="修改时间" FieldKey="ModifyTime" Key="ModifyTime" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Format DataType="Date" FormatString="yyyy-MM-dd HH:mm:ss"/>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    
                </Row>
                <Row Height="30">
                    <Cell Caption="备注" Key="Inner_Notes" MergedColumnSpan="1" MergedRowSpan="1">
                        <Display>
                            <Font UnderLine="SolidNormal"/>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                    <Cell Caption="备注" FieldKey="Notes" Key="Notes" MergedColumnSpan="1" MergedRowSpan="1" SourceType="Field" TableKey="ERP_FormulaDefine">
                        <Display>
                            <Font UnderLine="SolidNormal"/>
                            <Border BottomColor="#000000" BottomStyle="1" LeftColor="#000000" LeftStyle="1" RightColor="#000000" RightStyle="1" TopColor="#000000" TopStyle="1"/>
                        </Display>
                    </Cell>
                </Row>
            </Rows>
        </Section>
    </Grid>
      <Embed>
        <Text Key="Text1" Caption="测试Text" SourceType="Const" X="150" Y="150" Width="50" Height="50"/>
        <Text Key="Text2" Caption="测试试试试试试试试试" SourceType="Const" X="200" Y="300" Width="100" Height="100"/>
        <!--<Image Key="Image1" Caption="测试Image" X="60" Y="60" Width="50" Height="50"/>
    <Chart Key="Chart1" Caption="测试Chart" X="70" Y="70" Width="50" Height="50"/>-->
    </Embed>
</Report>
`;

    protected prepareToolbar(toolbar: Toolbar): void {
        var btnSave = new Button("保存");
        btnSave.setOnClickListener((e: MouseEvent) => {
            this.save();
        });
        toolbar.addItem(btnSave);

        var btnDelete = new Button("删除");
        btnDelete.setOnClickListener((e: MouseEvent) => {
            this.delete();
        });
        toolbar.addItem(btnDelete);
        this.installToolbar(toolbar, this.viewContainer);
    }

    protected createPropertyIO(): IPropertyIO {
        return new PrintPropertyIO(this.getXmlEntity(), this.getGraphicView());
    }
}