import DivElement from "../../../../common/dom/element/DivElement";
import { ENodeType} from "../../../../common/enum/Enums";
import StringBuilder from "../../../../common/util/StringBuilder";
import INode from "../../../../common/xml/node/INode";
import ITagNode from "../../../../common/xml/node/ITagNode";
import XmlUtil from "../../../../common/xml/util/XmlUtil";
import AbstractXmlEditor from "./AbstractXmlEditor";
import AbstractPropertyDefine from "../../property/base/AbstractPropertyDefine";

//@ts-ignore
export default class CodeMirrorXmlEditor extends AbstractXmlEditor<DivElement> {

    private xmlEditor: CodeMirror.Editor;

    private enableCursorActivity = false;

    private bChanged = false;

    private propertyDefine: AbstractPropertyDefine;

    constructor(propertyDefine: AbstractPropertyDefine) {
        super(new DivElement());
        this.propertyDefine = propertyDefine;

        //@ts-ignore
        this.xmlEditor = CodeMirror(this.getEl(), {
            mode: 'text/xml',
            'lineNumbers': true,
            'lineWrapping': true, //换行
            gutters: ["CodeMirror-linenumbers", "CodeMirror-foldgutter"],
            readOnly: true,
            // @ts-ignore
            foldGutter: true,
            styleActiveLine: true
        });

        this.xmlEditor.on('change', (cm, changeObj) => {
            if (!this.enableCursorActivity) return;
            this.bChanged = true;
            this.commit();
        });

        this.xmlEditor.on('cursorActivity', (cm) => {
            if (!this.enableCursorActivity) return;
            if (cm.somethingSelected()) return;
            var line = cm.getCursor().line;
            var curLineNode = this.xmlEntity.findNodeByLine(line);
            if (!curLineNode) return;
            var curTagNode = this.findPropertyTagNode(curLineNode);
            if (curTagNode) this.xmlEntity.getSelectionModel().select(curTagNode);
        });
    }

    findPropertyTagNode(node: INode | undefined): ITagNode | null {
        if (!node) return null;
        //var PD = WfDevPropertyDefine.getInstance();
        if (node.getNodeType() == ENodeType.TAG) {
            var tagName = (<ITagNode> node).getTagName();
            var groups = this.propertyDefine.getPropertyDefines(tagName);
            if (groups.length > 0) return <ITagNode> node;
        }
        return this.findPropertyTagNode(node.getParent());
    }

    commit(): boolean {
        if (!this.isVisible()) return false;
        if (!this.bChanged) return false;
        var xml = this.xmlEditor.getValue();
        if (!XmlUtil.checkXml(xml)) return false;
        this.xmlEntity.reload(xml);
        this.xmlEntity.fireCommit();
        return true;
    }

    reload(): void {
        if (!this.isVisible()) return;
        this.bChanged = false;
        this.enableCursorActivity = false;
        var orgSelectedItem = this.xmlEntity.getSelectionModel().getSingleSeletionItem();
        var xml = this.xmlEntity.getXml(false);
        this.xmlEditor.setValue(xml);
        this.xmlEditor.focus();
        this.xmlEditor.refresh();
        this.xmlEditor.clearHistory();
        this.enableCursorActivity = true;
        orgSelectedItem && this.locate(orgSelectedItem.getTagNode());
    }

    locate(tagNode: ITagNode): void {
        var beginLine = tagNode.getStartLine();
        if (beginLine < 0) return;
        var startPos = {ch:0, line: beginLine};
        var lineStr = this.xmlEditor.getLine(beginLine);
        if(!lineStr) return;
        var endPos = {ch:lineStr.length, line: beginLine};
        this.xmlEditor.setSelection(startPos, endPos);
        this.xmlEditor.scrollIntoView(startPos, 200);
    }
/* 
    private getSearchCursor(tagNode: ITagNode): any {
        if (!tagNode) return;
        var primaryKey = "ID";
        if (!tagNode.hasAttribute(primaryKey)) {
            primaryKey = "Key"
            if (!tagNode.hasAttribute(primaryKey)) return;
        }
        var primaryValue = tagNode.getAttributeValue(primaryKey);
        var search = ` ${primaryKey}="${primaryValue}"`;
        return this.xmlEditor.getSearchCursor(search, null, {caseFold: false}); 
    } */

    setEditable(editable: boolean): void {
		this.xmlEditor.setOption("readOnly", !editable);
    }

    refresh(): void {
        this.xmlEditor.refresh();
    }

    update(tagNode: ITagNode, bUpdateSelectOnly: boolean, bUpdateAttrOnly: boolean, bNeedUpdateNodeXml: boolean): void {
        // 非源码状态下该方法不可靠，涉及到新增节点时，节点的getLine是不可靠的
        if (!bNeedUpdateNodeXml) return;
        if (!this.isVisible()) return;
        var beginLine = tagNode.getStartLine();
       /*  if (beginLine < 0) return;
        var endLine = tagNode.getEndLine();
        if (endLine < 0) return;
        var startPos = {ch:0, line: beginLine};
        var lineStr = this.xmlEditor.getLine(endLine);
        var endPos = {ch:lineStr.length, line: endLine}; */

        var startPos = this.findTagStartPos(beginLine, tagNode.getTagName());
        if (!startPos) return;
        var endPos = this.findTagEndPos(startPos, tagNode.getTagName());
        if (!endPos) return;

        this.enableCursorActivity = false;
        var pos = this.xmlEditor.getCursor();
        var xml = tagNode.toXml(false).trim();
        this.xmlEditor.replaceRange(xml, startPos, endPos);
        this.xmlEditor.setCursor(pos);
        this.enableCursorActivity = true;

        /*var searchCursor = this.getSearchCursor(tagNode);
        if (searchCursor && searchCursor.findNext()) {
            var lineIndex = searchCursor.pos.from.line;
            var lineStr = this.xmlEditor.getLine(lineIndex);
            var tagName = XmlUtil.getTagName(lineStr);
            if (!tagName) return;
            var startPos = {ch:0, line: lineIndex};
            var endPos;
            if (lineStr.endsWith("/>")) {
                endPos = {ch:lineStr.length, line: lineIndex};
            } else {
                var endTagSymbo = `</${tagName}>`;
                for (var index=lineIndex; index <= this.xmlEditor.lastLine(); index ++ ) {
                    lineStr = this.xmlEditor.getLine(index);
                    var chIndex = lineStr.indexOf(endTagSymbo);
                    if (chIndex >= 0) {
                        endPos = {ch:chIndex + endTagSymbo.length , line: index};
                        break;
                    }
                }
            }
            if (endPos) {
                this.enableCursorActivity = false;
                var pos = this.xmlEditor.getCursor();
                var xml = tagNode.toXml(false);
                this.xmlEditor.replaceRange(xml, startPos, endPos);
                this.xmlEditor.setCursor(pos);
                this.enableCursorActivity = true;
            }
        }*/
    }

    //@ts-ignore
    private findTagStartPos(startLine: number, tagName: string): any | undefined {
        var lineTokens = this.xmlEditor.getLineTokens(startLine);
        for (let token of lineTokens) {
            if (token.string == tagName) {
                return {ch:token.start - 1, line: startLine};
            }
        }
        return undefined;
    }

    //@ts-ignore
    private findTagEndPos(startPos: any, tagName: string): any | undefined {
        var startLine = startPos.line;
        var lastLine = this.xmlEditor.lineCount();
        var stackTags = [tagName];
        for (var line = startPos.line; line <= lastLine; line ++) {
            var lineTokens = this.xmlEditor.getLineTokens(line);
            for (let token of lineTokens) {
                if (line == startLine && token.start <= startPos.ch + 1) {
                    continue;
                }
                if (token.type == ECMTokenType.tag) {
                    var curTag = token.string;
                    if (stackTags[stackTags.length - 1] == curTag) {
                        stackTags.pop();
                    } else {
                        stackTags.push(curTag);
                    }
                } else if (token.string == '/>') {
                    stackTags.pop();
                }
                if (stackTags.length == 0) {
                    return {ch:token.end + 1, line: line};
                } 
            }
        }
        return undefined;
    }

    out(level: number, node: INode, bStart: boolean): void {
        let sb = new StringBuilder();
        node.writeXml(level, sb, true);
        this.xmlEditor.setValue(sb.toString());
        this.xmlEditor.clearHistory();
    }

    undo(): void {
        this.xmlEditor.undo();
    }

    redo(): void {
        this.xmlEditor.redo();
    }

    isChanged(): boolean {
        return this.xmlEditor.historySize().undo > 0;
    }

    resize(width: any, height: any): void {
        this.xmlEditor.setSize(width, height);
    }
}

enum ECMTokenType {
    tag = 'tag',
    tagbracket = 'tag bracket'
}