import JComponent from "../../../common/component/JComponent";
import {Container, DataMapTableLayout} from "../layout/DataMapTableLayout";
import ITagNode from "../../../common/xml/node/ITagNode";
import {CommonConstant} from "../base/CommonConstant";
import {DataMapGraphicView} from "./DataMapGraphicView";
import {FieldRow} from "./element/FieldRow";
import Span from "../../../common/component/control/Span";
import TextEditor from "../../../common/component/control/TextEditor";
import InputElement from "../../../common/dom/element/InputElement";
import ArrayList from "../../../common/struct/ArrayList";
import OperationDispatcher from "../../../common/operation/OperationDispatcher";
import SizeInfo from "../../../common/struct/SizeInfo";
import {CommonFunction} from "../base/CommonFunction";
import Menu from "../../../common/component/menu/Menu";


export class DataMapSourceTableView extends JComponent<DataMapTableLayout> {
    private tagNode: ITagNode;
    private tableLayout: DataMapTableLayout;
    private icon: Span | undefined;
    private input: Span | undefined;
    private textEditor: TextEditor | undefined;
    private tableName: string | undefined;
    private addButton: InputElement | undefined;
    private menu: Menu = new Menu();

    // textEditor是否事件初始化
    private initTextEditor : boolean = false;

    constructor(tagNode:ITagNode, srcItems : Map<string, any> , dataMapGraphicView:DataMapGraphicView){
        super(new DataMapTableLayout(0,0));
        this.tagNode = tagNode;
        this.tableLayout = this.getDomElement();
        if(tagNode.getTagName() == CommonConstant.NODE_SourceTable){
            this.tableLayout.init(tagNode.getParent()!.getAttributeValueInt(CommonConstant.ATTR_Width, 0),CommonConstant.ROW_HEIGHT);
            this.buildTableRow(srcItems,dataMapGraphicView);
        }else if(tagNode.getTagName() == CommonConstant.NODE_SourceTableCollection){
            this.tableLayout.init(tagNode.getAttributeValueInt(CommonConstant.ATTR_Width, 0) , CommonConstant.ROW_HEIGHT);
            this.buildFormRow(srcItems, dataMapGraphicView);
        }
        this.tableLayout.addClass('layout');
    }

    buildFormRow(srcItems : Map<string, any>, dataMapGraphicView: DataMapGraphicView){
        let field = this.tableLayout.addField(CommonConstant.Type_Source_SourceTable);
        let parent = this.tagNode.getParent();
        if(!parent) return;
        let fieldView = new FieldRow(this.tagNode);
        field.addClass("tablehead");
        field.addChild(fieldView);

        let srcFormKey =dataMapGraphicView.getSrcFormKey();
        this.tableName = srcFormKey;
        fieldView.buildFormRow(srcItems.get(srcFormKey),srcFormKey);
        let container = this.tableLayout.addSpan();
        this.addTitleComponent(container,CommonConstant.NODE_SrcFormKey,srcFormKey);
    }

    buildTableRow(srcItems : Map<string, any> , dataMapGraphicView:DataMapGraphicView){
        let field = this.tableLayout.addField(CommonConstant.Type_Source_SourceTable);
        let fieldRow = new FieldRow(this.tagNode);
        field.addChild(fieldRow);
        field.addClass("tablehead");

        let root = this.tagNode.getParent()!.getParent()!;
        let srcFormKey = root.getAttributeValue(CommonConstant.NODE_SrcFormKey);
        let tableKey = this.tagNode.getAttributeValue(CommonConstant.NODE_Table_Key);
        let caption = srcItems.get(tableKey) ? srcItems.get(tableKey).Caption : "";
        fieldRow.buildTableRow(tableKey, caption, dataMapGraphicView);

        let container = this.tableLayout.addSpan();
        let type = fieldRow.getTagNode().getAttributeValue(CommonConstant.ATTR_Type);
        if(type != CommonConstant.NODE_Table){
            let dataSource = fieldRow.getTagNode()?.getAttributeValue(CommonConstant.ATTR_DataSource);
            let entityFormKey = fieldRow.getTagNode()?.getAttributeValue(CommonConstant.ATTR_EntityFormKey);
            this.tableName = entityFormKey + "|" + dataSource;
            this.addTitleComponent(container,CommonConstant.ATTR_DataSource,this.tableName);
            fieldRow.setText(dataSource);
        }else{
            this.tableName = srcFormKey + "|" + tableKey;
            this.addTitleComponent(container,CommonConstant.NODE_Table_Key,this.tableName);
            fieldRow.setText("(表" + srcItems.get(tableKey)?.index + ")  " + tableKey + " " + caption);
        }
        fieldRow.addClass("tableRow");

        this.buildFieldRow(srcFormKey, tableKey, srcItems, dataMapGraphicView);

    }
    buildFieldRow(srcDataObjectKey: string, tableKey: string, srcItems : Map<string, any>,  dataMapGraphicView:DataMapGraphicView){
        let fieldList = new ArrayList();
        let children = this.tagNode.getChildren();
        for (let child of children) {
            let caption = "";
            let dataType = "";
            let childTag =  <ITagNode>child;
            let key = CommonFunction.getSourceFieldValue(childTag);
            let type = childTag.getAttributeValue(CommonConstant.NODE_Type);
            if(fieldList.contain(key)) continue;
            let field = this.tableLayout.addField(CommonConstant.Type_Source_SourceTable);
            let fieldRow = new FieldRow(childTag);
            field.addChild(fieldRow);
            if(!type){
                if(srcItems.get(key)){
                    caption = srcItems.get(key).Caption;
                    dataType = srcItems.get(key).DataType;
                }
            }
            fieldRow.buildFieldRow(key, caption, dataType, dataMapGraphicView, type);
            fieldList.add(key);
        }
    }

    addTitleComponent(container: Container,attrKey: string,attrValue: string){
        this.icon = container.getIcon();
        this.addButton = container.getAddButton();
        this.input = container.getInput();
        this.textEditor = container.getTextEditor();
        this.icon.getDomElement().setAttr(attrKey,attrValue);
        this.addButton.setAttr(attrKey,attrValue);
        this.input.getDomElement().setAttr(attrKey,attrValue);
    }

    isTableTitle(){
        return this.tagNode.getTagName() == CommonConstant.NODE_SourceTableCollection;
    }

    setInitTextEditor(value : boolean) {
        this.initTextEditor = value;
    }

    getInitTextEditor(){
        return this.initTextEditor;
    }

    getAddIcon(){
        return this.icon!;
    }

    getInputButton(): InputElement{
        return this.addButton!;
    }

    getInputPromptSpan(){
        return this.input;
    }

    getInputTextEditor(){
        return this.textEditor;
    }

    getMenu(): Menu {
        this.menu.clearChildren();
        return this.menu;
    }

    getTableName(): string {
        return this.tableName!;
    }

    getWidth(): SizeInfo {
        return SizeInfo.valueOfPX(this.tableLayout.getWidth()!);
    }

    getHeight(): SizeInfo {
        return SizeInfo.valueOfPX(this.tableLayout.getHeight()!);
    }

    containPoint(x: number,y: number): boolean{
        return CommonFunction.containPoint(x,y,this.getEl());
    }

    locRow(x: number, y: number): FieldRow | null{
        return CommonFunction.locRow(x, y, this.tableLayout);
    }

    getTagNode(): ITagNode {
        return this.tagNode;
    }
}