import JComponent from "../../../common/component/JComponent";
import {ENodeType } from "../../../common/enum/Enums";
import INode from "../../../common/xml/node/INode";
import ITagNode from "../../../common/xml/node/ITagNode";
import ViewElementFactory from "./ViewElementFactory";
import AbstractLink from "./element/link/AbstractLink";
import AbstractWfElement from "./element/AbstractWfElement";
import SelectionModel from "../../../common/selection/SelectionModel";
import CmdQueue from "../../../common/cmd/CmdQueue";
import WfDefines from "../base/WfDefines";
import AbstractNode from "./element/node/AbstractNode";
import OptStateDelegate from "../state/OptStateDelegate";
import MoveNodeCmd from "../cmd/MoveNodeCmd";
import NewNodeCmd from "../cmd/NewNodeCmd";
import NewLinkCmd from "../cmd/NewLinkCmd";
import GPaper from "../../plugin/graphic/snap/GPaper";
import IGElement from "../../../common/ui/graphic/IGElement";
import DeleteElementCmd from "../cmd/DeleteElementCmd";
import ChangeLineFromNodeCmd from "../cmd/ChangeLineFromNodeCmd";
import ChangeLineToNodeCmd from "../cmd/ChangeLineToNodeCmd";
import IXmlEntity from "../../../common/xml/IXmlEntity";
import ActionNames from "../../../common/event/ActionNames";
import Paras from "../../../common/struct/Paras";
import SizeInfo from "../../../common/struct/SizeInfo";
import XmlEntity from "../../../common/xml/XmlEntity";
import WfConstants from "../base/WfConstants";
import I18N from "../base/I18N";
import Events from "../../../common/event/Events";
import Menu from "../../../common/component/menu/Menu";
import AddNodeOnLineCmd from "../cmd/AddNodeOnLineCmd";
import PopupPanel from "../../../common/component/container/panel/PopupPanel";
import InputElement from "../../../common/dom/element/InputElement";
import ChangeCaptionCmd from "../cmd/ChangeCaptionCmd";
import WfActionUtil from './../util/WfActionUtil';
import GlobalConstants from "../../../common/struct/GlobalConstants";
import DenyFlow from "./element/link/DenyFlow";
import SequenceFlow from "./element/link/SequenceFlow";
import Custom from '../../../common/component/control/Custom';
import DomElement from "../../../common/dom/element/DomElement";
import WfToolbox from "./WfToolbox";
import ContainerPanel from '../../../common/component/container/panel/ContainerPanel';
import AbstractGraphicView from "../../base/AbstractGraphicView";
import AbstractPropertyDefine from "../../plugin/property/base/AbstractPropertyDefine";
import WfDevPropertyDefine from "../base/WfDevPropertyDefine";

/**
 * 工作流设计器图形化视图
 * 
 * @author chenbb
 */
 export default class WfGraphicView extends AbstractGraphicView<OptStateDelegate> {

    private paper: GPaper;

    private xmlEntity: IXmlEntity;

    private mapNodes: Map<String, AbstractNode>;

    private mapLinks: Map<String, AbstractLink>;

    private vSnapLine: IGElement;

    private hSnapLine: IGElement;

    private linePopupMenu: Menu;

    private nodePopupMenu: Menu;

    private popupInput: PopupPanel;

    private inputEl: InputElement;

    private beginNode?: AbstractNode;

    private endNode?: AbstractNode;

    private bShowDenyLinks = false;

    private bShowSequenceLinks = true;

    private paperContainer: ContainerPanel;

    constructor(toolbox?: WfToolbox) {
        super(toolbox ? WfDefines.DEV_TOOLBOX_WIDTH : 0, 0, 0, 0);
        this.setStyle('overflow', 'auto');
        this.xmlEntity = new XmlEntity();

        this.mapNodes = new Map();
        this.mapLinks = new Map();
        this.paper = new GPaper(WfDefines.PAPER_W, WfDefines.PAPER_H);
        this.paper.attr({'tabindex': 0});
        this.paper.addClass('bpm-view');
        this.initViewMouseEvent(this.paper);
        toolbox && this.setLeft(toolbox);
        toolbox && toolbox.bindGraphicView(this);
        
        this.paperContainer = new ContainerPanel();
        var paperControl = new Custom(new DomElement(this.paper.getHtmlElement()));
        this.paperContainer.addComponent(paperControl);
        this.setCenter(this.paperContainer);

        var snapLineAttr = {
            'stroke-width': '1px',
             stroke: '#0095ff',
            'fill-opacity': '0.5',
            'strokeWidth': 1,
            'stroke-dasharray': '8, 8'
        }
        this.hSnapLine = this.paper.line(0, -1, WfDefines.PAPER_W, -1).attr(snapLineAttr);
        this.vSnapLine = this.paper.line(-1, 0, -1, WfDefines.PAPER_H).attr(snapLineAttr);

        this.linePopupMenu = new Menu();
        for (var action of WfActionUtil.ACTION_ON_LINE) {
            var menuItem = this.linePopupMenu.addMenuItem(action, I18N.getString(action));
            this.initLinkOpHandler(menuItem);
        }

        this.nodePopupMenu = new Menu();
        for (var action of WfActionUtil.ACTION_ON_NODE) {
            var menuItem = this.nodePopupMenu.addMenuItem(action, I18N.getString(action));
            this.initNodeOpHandler(menuItem);
        }

        this.popupInput = new PopupPanel();
        this.inputEl = new InputElement();
        this.popupInput.setContent(this.inputEl);

        this.initKeyListener(); 
    }

    createOptStateDelegate(): OptStateDelegate {
        return new OptStateDelegate(this);
    }

    setEditable(editable: boolean) {
        super.setEditable(editable);
        editable ? this.getOptStateDelegate().reset() : this.getOptStateDelegate().setUnableState();
    }

    resize(width: number, height: number) {
        this.paperContainer.resize(this.hasLeft() ? (width - WfDefines.DEV_TOOLBOX_WIDTH) : width, height);
    }

    private initKeyListener(): void {
        this.getEl().addEventListener('keyup', (e) => {
            if (e.key === 'Enter') {
                this.doUpdateCaption();
                e.stopPropagation();
            } 
        });
    }

    private doUpdateCaption() {
        if (this.popupInput.isShowing()) {
            this.popupInput.hide();
            var curSeletable = this.getSelectionModel().getSingleSeletionItem();
            if (curSeletable) {
                var input: HTMLInputElement = <HTMLInputElement> this.inputEl.getEl();
                this.doChangeCaptionCmd(curSeletable.getTagNode(), input.value);
                this.getSelectionModel().reset();
                this.getSelectionModel().select(curSeletable.getTagNode());
            }
        }
    }

    private initNodeOpHandler(item: JComponent<any>): void {
        item.addEventListener(Events.MouseDown, (event: any) => {
            event = event || window.event;
            var action = item.getKey();
            if (ActionNames.model_node_remove == action) {
                this.doDeleteElementCmd();
            } else if (WfConstants.NODE_Flow_Sequence == action) {
                var linkState = this.getOptStateDelegate().setNewLinkState(WfConstants.NODE_Flow_Sequence);
                var selectable = this.getSelectionModel().getSingleSeletionItem();
                var wfNode = this.findElement(selectable?.getTagNode());
                if (wfNode instanceof AbstractNode) {
                    linkState.setFromNode(wfNode);
                }
            } else {
                // 新增节点加连线
                var selectable = this.getSelectionModel().getSingleSeletionItem();
                var wfNode = this.findElement(selectable?.getTagNode());
                if (wfNode instanceof AbstractNode) {
                    this.getOptStateDelegate().setNewLinkAndNodeState(wfNode, action, WfConstants.NODE_Flow_Sequence);
                }
            }
            this.hidePopMenu();
        }, true);
    }

    private initLinkOpHandler(item: JComponent<any>): void {
        item.addEventListener(Events.MouseDown, (event: any) => {
            event = event || window.event;
            var action = item.getKey();
            if (action == ActionNames.model_node_remove) {
                this.doDeleteElementCmd();
            } else {
                this.doAddNodeOnLineCmd(action);
            }
            this.hidePopMenu();
        }, true);
    }

    /**
     * 初始化视图面板的鼠标事件
     * @param paper 
     */
    private initViewMouseEvent(paper: IGElement): void {
        paper.mousedown(e => this.getOptStateDelegate().getCurrentState().mousedown(e));
        paper.mousemove(e => this.getOptStateDelegate().getCurrentState().mousemove(e));
        paper.mouseup(e => this.getOptStateDelegate().getCurrentState().mouseup(e));
    }    

    public getXmlEntity(): IXmlEntity | undefined {
        return this.xmlEntity;
    }

    public focus(): void {
        this.paper.getHtmlElement().focus();
    }

    showNodePopMenu(x: number, y: number): void {
        if (!this.isEditable()) return;
        var top = this.paperContainer.getEl().scrollTop;
        var left = this.paperContainer.getEl().scrollLeft;
        this.nodePopupMenu.show(x - left, y - top, 90, 220, this.paperContainer.getEl());
    }

    showLinePopMenu(x: number, y: number): void {
        if (!this.isEditable()) return;
        var top = this.paperContainer.getEl().scrollTop;
        var left = this.paperContainer.getEl().scrollLeft;
        this.linePopupMenu.show(x - left, y - top, 90, 180, this.paperContainer.getEl());
    }

    showPopupInput(x: number, y: number, width: number, height:number, text: string): void {
        if (!this.isEditable()) return;

        var input: HTMLInputElement = <HTMLInputElement> this.inputEl.getEl();
        input.value = text;
        this.inputEl.setStyleWidth(SizeInfo.valueOfPX(width + 12));
        this.inputEl.setStyleHeight(SizeInfo.valueOfPX(height + 6));
        this.setCaretPosition(input, text ? text.length : 0);

        var top = this.paperContainer.getEl().scrollTop;
        var left = this.paperContainer.getEl().scrollLeft;
        this.popupInput.show(x - left, y - top, width, height, this.paperContainer.getEl())
    }

    private setCaretPosition(control: any, pos: number): void {
        if(control.setSelectionRange) {
            control.focus();
            control.setSelectionRange(pos, pos);
        }
    }

    hidePopMenu(): void {
        this.nodePopupMenu.hide();
        this.linePopupMenu.hide();
    }

    getPaper(): GPaper {
        return this.paper;
    }

    update(tagNode: ITagNode): void {
        var node = this.findNode(tagNode);
        if (node) {
            node.updataView();
            if (node instanceof AbstractLink) {
                this.updateLink(<AbstractLink> node);
            } else {
                this.updateLinkByNode(node);
            }
        }
    }

    private findNode(tagNode: ITagNode): AbstractNode | null {
        var node = this.findNodeInCollection(tagNode, this.mapNodes.values());
        if (node) return node;
        node = this.findNodeInCollection(tagNode, this.mapLinks.values());
        return node;
    }

    private findNodeInCollection(tagNode: ITagNode, nodes: IterableIterator<AbstractNode>): AbstractNode | null {
        if (!tagNode) return null;
        for (let node of nodes) {
            if (node.getTagNode().getTagName() == tagNode.getTagName()
                && node.getID() == tagNode.getAttributeValue("ID")) {
                  return node;
            }
        }
        return null;
    }

    getSelectionModel(): SelectionModel {
        return this.xmlEntity.getSelectionModel();
    }

    getBeginNode(): AbstractNode | undefined {
        return this.beginNode;
    }

    getEndNode(): AbstractNode | undefined {
        return this.endNode;
    }

    load(xmlEntity: IXmlEntity): void {
        //this.maxID = 0;
        this.xmlEntity = xmlEntity;
        var node: INode = xmlEntity.getRoot();
        this.clearElements();
        this.loadNodes(node);
        this.loadLinks();
        this.initDenyLinks();

        var selectItem = this.xmlEntity.getSelectionModel().getSingleSeletionItem();
        if (selectItem) {
            var selectNode = this.findNode(selectItem.getTagNode());
            selectNode && this.xmlEntity.getSelectionModel().select(selectNode.getTagNode());
        }
    }

    getPropertyDefine(): AbstractPropertyDefine {
        return WfDevPropertyDefine.getInstance();
    }

    private loadLinks(): void {
        for (let linkView of this.mapLinks.values()) {
            this.updateLink(linkView);
        }
    }

    private isMultiNode(node: INode | undefined): boolean {
        if (!node) return false;
        if (node.getNodeType() != ENodeType.TAG) return false;
        var tagNode = <ITagNode> node;
        return tagNode.getTagName() == WfConstants.NODE_AssistanceCollection;
    }

    private loadNodes(node: INode): void {
        if (this.isMultiNode(node.getParent())) return;
        if (node.getNodeType() == ENodeType.ROOT
            || node.getNodeType() == ENodeType.TAG) {
            var tagNode = <ITagNode> node;
            var wfNode = this.addViewElement(tagNode);
            if (tagNode.getTagName() == WfConstants.NODE_Event_Begin) {
                this.beginNode = wfNode;
            } else if (tagNode.getTagName() == WfConstants.NODE_Event_End) {
                this.endNode = wfNode;
            }
            //this.updateMaxID(wfNode);
            for (let child of tagNode.getChildren()) {
                this.loadNodes(child);
            }
        }
    }

    public addViewElement(node: ITagNode): AbstractWfElement | undefined {
        var viewElement = ViewElementFactory.create(node);
        if (!viewElement) return undefined;

        viewElement.setOnDblClick((e) => {
            var curSeletable = this.getSelectionModel().getSingleSeletionItem();
            if (curSeletable) {
                var wfElement = this.findElement(curSeletable.getTagNode());
                var gCaption = wfElement?.getGCaption();
                if (wfElement && gCaption) {
                    var area = wfElement.getCaptionArea();
                    this.showPopupInput(area.x, area.y, area.width, area.height, wfElement ? wfElement.getCaption() : "");
                } 
            }
        });

        viewElement.initGSection(this.paper.section());
        viewElement.initXmlNode(node);
        
        this.initNodeEvent(viewElement);
        if (viewElement instanceof AbstractNode) {
            this.mapNodes.set(viewElement.getID(), <AbstractNode> viewElement);
        } else if (viewElement instanceof AbstractLink) {
            this.initLinkEvent(viewElement);
            this.mapLinks.set(viewElement.getID(), <AbstractLink> viewElement);
        }
        return viewElement;
    }

    updateLink(linkView: AbstractLink): void {
        var fromID = linkView.getFromID();
        var toKey = linkView.getToKey();
        linkView.setFromNode(this.getViewNode(fromID));
        linkView.setToNode(this.findViewNode(toKey));
    }

    /**
     * 节点位置变化后，重新绘制相关联的线
     * 
     */
     public updateLinkByNode(updatedNode: AbstractNode): void {
        var nodeKey = updatedNode?.getKey();
        for (let transition of this.mapLinks.values()) {
            var fromID = transition.getFromID();
            var node = this.getViewNode(fromID);
            var toKey = transition.getToKey();
            if (nodeKey == node?.getKey() || nodeKey == toKey) {
                transition.drawLink();
            }
        }
        this.updateDenyLink(updatedNode.getTagNode());
    }

    private updateDenyLink(tagNode: ITagNode): void {
        var nodeID = tagNode.getAttributeValue(GlobalConstants.S_ID);
        var bDenyToLastNode = tagNode.getAttributeValueBoolean(WfConstants.ATTR_DenyToLastNode, false);
        var denyNodeKey = tagNode.getAttributeValue(WfConstants.ATTR_DenyNodeKey, '');
        var denyKey = this.parseDenyKey(nodeID);
        var denyLink = this.mapLinks.get(denyKey);
        if (!bDenyToLastNode && denyNodeKey) {
            this.ensureDenyLink(nodeID, denyNodeKey);
       } else if (denyLink) {
            this.removeWfNode(denyLink);
       }
       this.drawDenyFlow();
    }

    private drawDenyFlow(): void {
        for (let denyFlow of this.mapLinks.values()) {
            if (denyFlow instanceof DenyFlow) {
                denyFlow.drawLink();
            }
       }
    }

    private ensureDenyLink(formNodeID: string, denyNodeKey: string): DenyFlow {
        var denyLinkKey = this.parseDenyKey(formNodeID);
        var denyFlow;
        if (this.mapLinks.has(denyLinkKey)) {
            denyFlow = <DenyFlow> this.mapLinks.get(denyLinkKey);
        } else {
            denyFlow = new DenyFlow();
            denyFlow.setID(denyLinkKey);
            denyFlow.setKey(denyLinkKey);
            denyFlow.initGSection(this.paper.section());
            denyFlow.setVisiable(this.bShowDenyLinks);
            this.mapLinks.set(denyLinkKey, denyFlow);
        }
        var fromNode = this.getViewNode(formNodeID);
        var toNode = this.findViewNode(denyNodeKey);
        denyFlow.setFromNode(fromNode);
        denyFlow.setToNode(toNode);
        return denyFlow;
    }

    private initDenyLinks(): void {
        for (let node of this.mapNodes.values()) {
            this.initDenyLink(node);
        }
    }

    private initDenyLink(node: AbstractNode) {
        var bDenyToLastNode = node.getTagNode().getAttributeValueBoolean(WfConstants.ATTR_DenyToLastNode, false);
        var denyNodeKey = node.getTagNode().getAttributeValue(WfConstants.ATTR_DenyNodeKey, '');
        if (!bDenyToLastNode && denyNodeKey) {
             this.ensureDenyLink(node.getID(), denyNodeKey);
        }
    }

    setDenyFlowDisplay(b: boolean) {
        this.bShowDenyLinks = b;
        for (let link of this.mapLinks.values()) {
            if (link instanceof DenyFlow) {
                link.setVisiable(b);
            }
        }
    }

    setSequenceFlowDisplay(b: boolean) {
        this.bShowSequenceLinks = b;
        for (let link of this.mapLinks.values()) {
            if (link instanceof SequenceFlow) {
                link.setVisiable(b);
            }
        }
    }

    private parseDenyKey(curNodeID: string) {
        return `${curNodeID}__deny`;
    }

    private initLinkEvent(link: AbstractLink): void {
        var selectSymobo = link.getSelectSymbo();
        var children = selectSymobo?.getChildren();
        if (children && children.length >= 2) {
            var formHandle = children[0];
            var toHandle = children[1];
            formHandle.mousedown( e => {
                if (!this.isEditable()) return;

                this.getOptStateDelegate().setChangeLineFromNodeState(link);
                e.stopPropagation();
            });
            toHandle.mousedown( e => {
                if (!this.isEditable()) return;
                
                this.getOptStateDelegate().setChangeLineToNodeState(link);
                e.stopPropagation();
            });
            var onDragMove = (dx: number, dy: number, x: number, y: number, event: MouseEvent) => {
                this.getOptStateDelegate().getCurrentState().dragmove(dx, dy, x, y, event);
            };
            var onDragStart = (x: number, y: number, event: MouseEvent) => {
                this.getOptStateDelegate().getCurrentState().dragstart(x, y, event);
            };
            var onDragEnd = (event: MouseEvent) => {
                this.getOptStateDelegate().getCurrentState().dragend(event);
            };
            formHandle.drag(onDragMove, onDragStart, onDragEnd);
            toHandle.drag(onDragMove, onDragStart, onDragEnd);
        }

        link.setOptHandler((e) => {
            //this.optStateDelegate.setOptOnLineState(link);
            var gEL =  link.getOptSymbo();
            var x = gEL.getBBox().cx;
            var y = gEL.getBBox().cy;
            this.showLinePopMenu(x + 15, y + 10);
            //this.showLinePopMenu(e.clientX, e.clientY);
            e.stopPropagation();
        });
    }

    /**
     * 初始化节点事件
     * 
     * @param wfNode
     */
    public initNodeEvent(wfNode?: AbstractWfElement): void {
        if (!wfNode) return;
        var controlNode = wfNode.getControlNode();
        controlNode.mousedown((e) => {
            this.focus();
            this.getOptStateDelegate().getCurrentState().mousedown(e);
            e.stopPropagation();
        });

        controlNode.hover(() => wfNode.markHovered(), () => wfNode.markUnHovered());

        var onDragMove = (dx: number, dy: number, x: number, y: number, event: MouseEvent) => {
            this.getOptStateDelegate().getCurrentState().dragmove(dx, dy, x, y, event);
        };
        var onDragStart = (x: number, y: number, event: MouseEvent) => {
            this.getOptStateDelegate().getCurrentState().dragstart(x, y, event);
        };
        var onDragEnd = (event: MouseEvent) => {
            this.getOptStateDelegate().getCurrentState().dragend(event);
        };
        controlNode.drag(onDragMove, onDragStart, onDragEnd);

        wfNode.setOptHandler((e) => {
            this.showNodePopMenu(e.offsetX, e.offsetY);
            e.stopPropagation();
        });
    }

    public getViewLink(id?: string): AbstractLink | undefined {
        if (!id) return undefined;
        return this.mapLinks.get(id);
    }

    public getViewNode(id?: string): AbstractNode | undefined {
        if (!id) return undefined;
        return this.mapNodes.get(id);
    }

    public getNodeKeys(): IterableIterator<String> {
        return this.mapNodes.keys();
    }

    public getNodes(): IterableIterator<AbstractNode> {
        return this.mapNodes.values();
    }

    public getLinks(): IterableIterator<AbstractLink> {
        return this.mapLinks.values();
    }

    public findViewNode(nodeKey: string): AbstractNode | undefined {
        for (let node of this.mapNodes.values())  {
            if (node.getKey() == nodeKey) {
                return node;
            }
        }
        return undefined;
    }

    public drawDragVirtual(elems: IGElement[], dx: number, dy: number) : void {
        var selectItems = this.getSelectionModel().getSelectionItems();
        if (selectItems.length == 0) {
            return;
        }
        var firstItem = selectItems[0];
        var firstNode = this.findElement(firstItem.getTagNode());
        if (firstNode instanceof AbstractNode) {
            this.snap(elems[0]);
        }
        
        for (let elem of elems) {
            elem.attr({transform: 'T' + [dx, dy]});
        }
    }

    public snap(snapElem: IGElement): void {
        for (let value of this.mapNodes.values()) {
            if (this.snapToVNode(snapElem, value)) {
                break;
            }
        }
        for (let value of this.mapNodes.values()) {
            if (this.snapToHNode(snapElem, value)) {
                break;
            }
        }
    }

    doMoveNodeCmd(offsetX: number, offsetY: number) {
        var firstSelectItem = this.getSelectionModel().getSingleSeletionItem();
        var wfElement = this.findElement(firstSelectItem?.getTagNode());
        if (!wfElement) return;

        var snapX = this.vSnapLine.getBBox().x;
        var snapY = this.hSnapLine.getBBox().y;
        if (snapX > 0) {
            offsetX = snapX - (<AbstractWfElement> wfElement).getCenter().x;
        }
        if (snapY > 0) {
            offsetY = snapY - (<AbstractWfElement> wfElement).getCenter().y;
        }
        var cmd = new MoveNodeCmd(this, offsetX, offsetY);
        this.getCmdQueue().doCmd(cmd);
        this.resetVSnapLine();
        this.resetHSnapLine();
    }

    doNewNodeCmd(type: string, x: number, y: number): any {
        if (!this.xmlEntity) return;
        var cmd = new NewNodeCmd(this.xmlEntity, this.newID(), type, x, y);
        this.getCmdQueue().doCmd(cmd);
    }

    doNewLinkCmd(type: string, fromTag: ITagNode | undefined, toKey: string | undefined): any {
        if (!this.xmlEntity || !fromTag || !toKey) return;
        var cmd = new NewLinkCmd(this.xmlEntity, this.newID(), type, fromTag, toKey);
        this.getCmdQueue().doCmd(cmd);
    }

    doDeleteElementCmd(): any {
        var cmd = new DeleteElementCmd(this, this.getSelectionModel().doCacheTagNodes());
        this.getCmdQueue().doCmd(cmd);
    }

    doChangeLineFromNodeCmd(fromTag: ITagNode, orgFromTag: ITagNode, toLinkTag: ITagNode): any {
        if (!this.xmlEntity || !fromTag || !toLinkTag) return;
        var cmd = new ChangeLineFromNodeCmd(this.xmlEntity, fromTag, orgFromTag, toLinkTag);
        this.getCmdQueue().doCmd(cmd);
    }

    doChangeLineToNodeCmd(linkTag: ITagNode | undefined, toKey: string): any {
        if (!this.xmlEntity || !linkTag) return;
        var cmd = new ChangeLineToNodeCmd(this.xmlEntity, linkTag, toKey);
        this.getCmdQueue().doCmd(cmd);
    }

    doAddNodeOnLineCmd(type: string): void {
        if (!this.xmlEntity) return;
        var selectable = this.getSelectionModel().getSingleSeletionItem();
        var tagLink = selectable?.getTagNode();
        var wfElement = this.findElement(tagLink);
        if (wfElement) {
            var cmd = new AddNodeOnLineCmd(this, this.newID(), type, <AbstractLink> wfElement);
            this.getCmdQueue().doCmd(cmd);
        }
    }

    doChangeCaptionCmd(tagNode: ITagNode, caption: string): any {
        if (!this.xmlEntity || !tagNode) return;
        var cmd = new ChangeCaptionCmd(this.xmlEntity, tagNode, caption);
        this.getCmdQueue().doCmd(cmd);
    }

    undoCmd(): void {
        this.getCmdQueue().undoCmd();
    }

    redoCmd(): void {
        this.getCmdQueue().redoCmd();
    }

    getCmdQueue(): CmdQueue {
        return this.xmlEntity.getCmdQueue();
    }

    hitHtmlElement(el: HTMLElement): AbstractWfElement | undefined {
        if (el == this.paper.getHtmlElement()) {
            return undefined;
        }
        for (let node of this.mapNodes.values()) {
            if (node.hitHtmlElement(el)) {
                return node;
            }
        }
        for (let node of this.mapLinks.values()) {
            if (node.hitHtmlElement(el)) {
                return node;
            }
        }
        return undefined;
    }

    hitNode(x: number, y: number): AbstractNode | undefined {
        var selected;
        var tempForests = Number.MAX_SAFE_INTEGER;
        for (let node of this.mapNodes.values()) {
            var bounds = node.getGSection().getBBox();
            if (x > bounds.x && y > bounds.y
                 && x < bounds.x + bounds.width && y < bounds.y + bounds.height) {
                var forests = bounds.width * bounds.height;
                if (forests < tempForests) {
                    selected = node;
                    tempForests = forests;
                } 
            }
        }
        return selected;   
    }

    getRelationLinks(node: AbstractNode): AbstractLink[] {
        var links = [];
        var nodeKey = node.getKey();
        for (let transition of this.mapLinks.values()) {
            var fromID = transition.getFromID();
            var fromNode = this.getViewNode(fromID);
            var toKey = transition.getToKey();
            if (nodeKey == fromNode?.getKey() || nodeKey == toKey) {
                links.push(transition);
            }
        }
        return links;
    }

    removeWfNode(node?: AbstractWfElement) {
        if (!node) return;

        node.getGSection().remove();
        this.mapNodes.delete(node.getID());
        this.mapLinks.delete(node.getID());

        var denyKey = this.parseDenyKey(node.getID());
        var denyLink = this.mapLinks.get(denyKey);
        denyLink && this.removeWfNode(denyLink);
    }

    newID(): number {
        return this.xmlEntity.genNextNodeID();
    }

    notifyEvent(cmd: string, paras: Paras): void {
        this.hidePopMenu();
        var node = paras.getElement();
        switch (cmd) {
            case ActionNames.model_node_add:
                var element = this.addViewElement(node);
                if (element instanceof AbstractLink) this.updateLink(<AbstractLink> element);
                element?.updataView();
                this.getSelectionModel().select(node);
                break;
            case ActionNames.model_node_remove:
                var wfElement = this.findElement(node);
                this.removeWfNode(wfElement);
                break;
            case ActionNames.model_node_update:
                this.update(node);
                break;
            case ActionNames.model_node_update_select:
                this.doUpdateCaption()
                this.update(node);
                break;
        }
    }

    findElement(xmlNode: ITagNode | undefined): AbstractWfElement | undefined {
        if (!xmlNode) return undefined;
        for (let node of this.mapNodes.values()) {
           if (xmlNode == node.getTagNode()) {
                return node;
           }
        }
        for (let node of this.mapLinks.values()) {
            if (xmlNode == node.getTagNode()) {
                return node;
            }
        }
        return undefined;
    }

    /**
     * 清除所有驳回线的选中状态（驳回线在xml模型中并不存在，所以不能通过模型监听的方式实现状态变化控制）
     */
    unSelectAllDenyFlows(): void {
        for (let node of this.mapLinks.values()) {
            if (node instanceof DenyFlow) {
                node.getTagNode().setSelected(false);
                node.markUnSelected();
            }
        }
    }

    private snapToVNode(snapElem: IGElement, snapToNode: AbstractNode): boolean {
        if (!snapElem) return false;
        var box = snapElem.getBBox();
        var center = snapToNode.getCenter();
        var toX = center.x;
        var dx = Math.abs(toX - box.cx);
        if (dx <= 20) {
            this.vSnapLine.attr({x1: toX, x2: toX});
            return true;
        } else {
            this.resetVSnapLine();
        }
        return false;
    }

    private snapToHNode(snapElem: IGElement, snapToNode: AbstractNode): boolean {
        if (!snapElem) return false;
        var box = snapElem.getBBox();
        var center = snapToNode.getCenter();
        var toY = center.y;
        var dy = Math.abs(toY - box.cy);
        if (dy <= 20) {
            this.hSnapLine.attr({y1: toY, y2: toY});
            return true;
        } else {
            this.resetHSnapLine();
        }
        return false;
    }

    private resetVSnapLine(): void {
        this.vSnapLine.attr({x1: -1, x2: -1}); 
    }

    private resetHSnapLine(): void {
        this.hSnapLine.attr({y1: -1, y2: -1}); 
    }

    private clearElements(): void {
        for (let node of this.mapNodes.values()) {
            node.remove();
        }
        for (let node of this.mapLinks.values()) {
            node.remove();
        }
        this.mapLinks.clear();
        this.mapNodes.clear();
    }

   /*  private updateMaxID(wfElement?: AbstractWfElement): void {
        if (!wfElement) return;
        var id = wfElement.getID() ? parseInt(wfElement.getID()): 0;
        this.maxID = Math.max(this.maxID, id);
    } */
}
