import BorderPanel from "../../common/component/container/panel/BorderPanel";
import Button from "../../common/component/control/Button";
import Toolbar from "../../common/component/toolbar/Toolbar";
import IXmlEntity from "../../common/xml/IXmlEntity";
import AbstractGraphicView from "./AbstractGraphicView";
import AbstractXmlEditor from "../plugin/editor/xmleditor/AbstractXmlEditor";
import CodeMirrorXmlEditor from "../plugin/editor/xmleditor/CodeMirrorXmlEditor";
import OptStateDelegate from "../bpm/state/OptStateDelegate";
import JComponent from "../../common/component/JComponent";
import { EDesignMode } from "../../common/enum/Enums";
/**
 * 复合设计界面用于整合源码视图
 */
export default class ComplexDesignView<V extends AbstractGraphicView<any>, B extends JComponent<any>> extends BorderPanel {

    private xmlEntity!: IXmlEntity;

    private graphicView: V;

    private sourceView: AbstractXmlEditor<any>;

    private bSourceVisible = false;

    private toolbar: B;

    constructor(graphicView: V, toolbar: B) {
        super(0, 0, toolbar.getHeightValue(), 0);
        this.toolbar = toolbar;
        this.addClass("complex-design-view");
        this.setTop(toolbar);
        this.graphicView = graphicView;
        this.sourceView = new CodeMirrorXmlEditor(graphicView.getPropertyDefine());
        this.displayView();
    }

    installToolbar(toolbar: Toolbar, mode: EDesignMode): void {
        var btnUndo = new Button("撤销");
        //btnUndo.setTitle("快捷键 Ctrl+z");
        btnUndo.setOnClickListener((e: MouseEvent) => this.undo());
        toolbar.addItem(btnUndo);
        var btnRedo = new Button("重做");
        //btnRedo.setTitle("快捷键 Ctrl+y");
        btnRedo.setOnClickListener((e: MouseEvent) => this.redo());
        toolbar.addItem(btnRedo);
        // 分隔符
        if (mode != EDesignMode.User) {
            toolbar.addSeparator();
            var btnSwitch = new Button("源码");
            btnSwitch.setOnClickListener((e: MouseEvent) => {
                this.toggleSourceVisible();
                btnSwitch.setCaption(this.bSourceVisible ? "视图" : "源码");
            });
            toolbar.addItem(btnSwitch);
        }
    }

    loadXml(xmlEntity: IXmlEntity): void {
        this.xmlEntity = xmlEntity;
        this.sourceView.load(this.xmlEntity);
        this.xmlEntity.addEventListener(this.sourceView);
        this.graphicView.load(this.xmlEntity);
        this.xmlEntity.addEventListener(this.graphicView);
    }

    setEditable(editable: boolean) {
        this.graphicView.setEditable(editable);
        this.sourceView.setEditable(editable)
    }

    reloadSource() {
        this.sourceView.reload();
    }

    getOptStateDelegate(): OptStateDelegate {
        return this.graphicView.getOptStateDelegate();
    }

    getGraphicView(): V {
        return this.graphicView;
    }

    getSourceView(): AbstractXmlEditor<any> {
        return this.sourceView;
    }

    toggleSourceVisible(): void {
        // 若当前为源码视图，且做过修改，切换前先提交源码
        if (this.bSourceVisible) {
            this.sourceView.commit();
            this.graphicView.load(this.xmlEntity);
        } 
        this.bSourceVisible = ! this.bSourceVisible;
        this.sourceView.setVisible(this.bSourceVisible);
        this.displayView();
    }

    redo(): void {
        this.bSourceVisible ? this.sourceView.redo() : this.xmlEntity.redo(); 
    }

    undo(): void {
        this.bSourceVisible ? this.sourceView.undo() : this.xmlEntity.undo(); 
    }

    private displayView(): void {
        this.setCenter(this.bSourceVisible ? this.sourceView : this.graphicView);
        this.display();
        this.bSourceVisible && this.sourceView.reload();
    }

    isSourceVisible() {
        return this.bSourceVisible;
    }

    resize(width: number, height: number) {
        super.resize(width, height);
        var contentHeight = height - this.toolbar.getHeightValue();
        this.sourceView.resize(width, contentHeight);
        this.graphicView.resize(width, contentHeight);
    }
}