import { ETokenType } from "../enum/Enums";
import StringBuilder from "../util/StringBuilder";
import StringUtil from "../util/StringUtil";
import IXmlLexicalAnalysisListener from "./IXmlLexicalAnalysisListener";
import Token from "./Token";
import XmlUtil from "./util/XmlUtil";

/**
 * xml词法分析器
 * 
 * @author: chenbb
 */
export default class XmlLexicalAnalysis {
    
    // 行位置
    private posInLine: number = 0;
    
    private line: number = 0;
    
    // 全局文档指针位置
    private position: number = 0;
    
    private source: string;
    
    private curChar: string;
    
    private totalSize: number = 0;
    
    private list: Token[] = [];

	private analysisListener?: IXmlLexicalAnalysisListener;
    
	public constructor(source: string) {
        this.posInLine = 0;
        this.position = 0;
        this.source = source;
        this.source = this.source + StringUtil.CONTENT_END;
        this.totalSize = this.source.length;
        this.curChar = source.charAt(0);
    }
	
	public analysis(): Token[] {
		while (this.position < this.totalSize) {
			this.calcToken();
			this.incNext();
		}
		return this.list;
	}

	public setAnalysisListener(analysisListener?: IXmlLexicalAnalysisListener): void {
		this.analysisListener = analysisListener;
	}
	
	private calcToken(): void {
		
        while (StringUtil.isBlank(this.curChar)) {
			this.analysisListener && this.analysisListener.fireSkipBlank(this.curChar);
            this.incNext();
        }
        
		if (StringUtil.isContentEnd(this.curChar)) {
			this.analysisListener && this.analysisListener.fireFinishedAnalysis();
			return;
		}
        
        switch (this.curChar) {
        case '<':
        	let nextChar: string = this.incNext();
        	if (nextChar == '?') {
        		this.record(this.follows(Token.CHAR_XML, Token.TK_HEAD_START, Token.TK_UNKNOW));
        	} else if (nextChar == '!') {
        		var char1 = this.incNext();
        		var char2 = this.incNext();
        		if (char1 == '-' && char2 == '-') { //<!-- -->
        			this.record(Token.TK_COMMENT_START);
        			this.record(this.comment());
        		} else if (char1 == '[' && char2 == 'C') {
        			if (this.checkAndIncNextString("DATA[")) { // <![CDATA[]]>
        				this.record(Token.TK_CDATA_START);
        				this.record(this.cdata());
        			} else {
        				XmlUtil.error(this.line, this.posInLine, "error on comment");
        			}
        		} else {
        			XmlUtil.error(this.line, this.posInLine, "error on comment");
        		}
        	} else if (nextChar == '/') {
        		this.record(Token.TK_LABS); 
        	} else {
        		//回退一个字符
        		this.decPrevious();
        		this.record(Token.TK_LAB); 
        	}
        	break;
        //case ']': record(follow(']', TokenType.CDATA_END, TokenType.ERROR)); break;
        case '?': this.record(this.follow('>', Token.TK_HEAD_END, Token.TK_UNKNOW)); break;
        case '/': 
			this.record(this.follow('>', Token.TK_SRAB, Token.TK_UNKNOW));
			break;
        case '>': this.record(Token.TK_RAB); break;
        case '=': this.record(Token.TK_EQ); break;
		case '\'':
        case '"': this.record(this.string(this.curChar, this.line, this.posInLine)); break;
        default:  this.record(this.text()); break;
        }
	}
	
	private record(token: Token): void {
 		if (token.getType() == ETokenType.ERROR) {
			throw new Error(token.getValue());
		}
		this.analysisListener && this.analysisListener.fireAppendedToken(token);
		this.list.push(token);
	}
	
    private follows(expects: string[], ifyes: Token, ifno: Token): Token {
    	let bChecked: boolean = true;
    	for (let expect of expects) {
	        if (this.incNext() != expect) {
	        	bChecked = false;
	        	break;
	        }
    	}
    	if (bChecked) {
    		return ifyes;
    	}
        return ifno;
    }
	
    private follow(expect: string, ifyes: Token, ifno: Token): Token {
        if (this.incNext() == expect) {
            //this.incNext();
            return ifyes;
        }
        return ifno;
    }
    
    private string(start: string, line: number, pos: number): Token {
        let sb = new StringBuilder();
        while (this.incNext() != start) {
            if (this.curChar == '\u0000') {
                XmlUtil.error(line, pos, "EOF while scanning string literal");
            }
            if (this.curChar == '\n') {
                XmlUtil.error(line, pos, "EOL while scanning string literal");
            }
            sb.append(this.curChar);
        }
        return new Token(ETokenType.STRING, sb.toString(), this.line);
    }

	private text(): Token {
		let sb = new StringBuilder();
        while (StringUtil.isAlpha(this.curChar) || StringUtil.isDigit(this.curChar) || this.curChar == '_') {
        	sb.append(this.curChar);
            this.incNext();
        }
 
        if (sb.isEmpty()) {
            XmlUtil.error(this.line, this.posInLine, `identifer_or_integer unrecopgnized character: (${this.curChar}) ${this.curChar}`);
        }
        this.decPrevious();
        return new Token(ETokenType.TEXT, sb.toString(), this.line);
    }
	
	/**
	 * 获取注释
	 * @return
	 */
	private comment(): Token {
		let chs: string[] = [this.incNext(), this.incNext(), this.incNext()];
		let sb = new StringBuilder();
		sb.appends(chs);
		let c: string;
		while (this.position < this.totalSize) {
			if (chs[0] == '-' && chs[1] == '-' && chs[2] == '>') {
				return new Token(ETokenType.COMMENT_END, sb.toString(), this.line);
			}
			c = this.incNext();
			sb.append(c);
			chs[0] = chs[1];
			chs[1] = chs[2];
			chs[2] = c;
		}
		return Token.TK_UNKNOW;
	}
	
	/**
	 * 获取 cdata
	 * @return
	 */
	private cdata(): Token {
		let chs: string[] = [this.incNext(), this.incNext(), this.incNext()];
		let sb = new StringBuilder();
		sb.appends(chs);
		let c: string;
		while (this.position < this.totalSize) {
			if (chs[0] == ']' && chs[1] == ']' && chs[2] == '>') {
				return new Token(ETokenType.CDATA_END, sb.toString(), this.line);
			}
			c = this.incNext();
			sb.append(c);
			chs[0] = chs[1];
			chs[1] = chs[2];
			chs[2] = c;
		}
		return Token.TK_UNKNOW;
	}
	
	
	private incNext(): string {
        this.posInLine ++;
        this.position++;
        if (this.position >= this.totalSize) {
            this.curChar = '\u0000';
            return this.curChar;
        }
        this.curChar = this.source.charAt(this.position);
        if (this.curChar == '\n') {
            this.line ++;
            this.posInLine = 0;
			this.analysisListener && this.analysisListener.fireNextLine();
        }
        return this.curChar;
	}
	
	private decPrevious(): void {
        this.posInLine --;
        this.position --;
        this.curChar = this.source.charAt(this.position);
	}
	
	/**
	 * 检查后续字符串
	 * @param s
	 * @return
	 */
	private checkAndIncNextString(s: string): boolean {
		let len = s.length;
		let sb = new StringBuilder();
		for (let index=0; index<len; index ++) {
			sb.append(this.incNext());
		}
		return s == sb.toString();
	}
}
