import { ESizeType } from "../enum/Enums";
import GlobalSetting from "../GlobalSetting";
import TypeUtil from "../util/TypeUtil";
/**
 * 尺寸封装
 */
export default class SizeInfo {

    public static AUTO_SIZE: SizeInfo = new SizeInfo(0, ESizeType.Auto);
    public static PX2: SizeInfo = new SizeInfo(2, ESizeType.px);
    public static PX5: SizeInfo = new SizeInfo(5, ESizeType.px);
    public static PX30: SizeInfo = new SizeInfo(30, ESizeType.px);
    public static PX200: SizeInfo = new SizeInfo(200, ESizeType.px);
    public static PERCENT100: SizeInfo = new SizeInfo(100, ESizeType.Percent);
    public static VH100: SizeInfo = new SizeInfo(100, ESizeType.vh);
    public static FLEXSCALE1: SizeInfo = new SizeInfo(1, ESizeType.FlexScale);
    public static NONE: SizeInfo = new SizeInfo(-1, ESizeType.none);

    private value: number;
    private type: ESizeType;
    constructor(value: number, type: ESizeType) {
        this.value = value;
        this.type = type;
    }

    public getValue(): number {
        return this.value;
    }

    public getType(): ESizeType {
        return this.type;
    }

    public toString(): string {
        switch (this.type) {
            case ESizeType.none:
                return "";
            case ESizeType.mm:
                return this.value + "mm";
            case ESizeType.Scale:
                return this.value.toString();
            case ESizeType.px:
                return this.value + "px";
            case ESizeType.vh:
                return this.value + "vh";
            case ESizeType.Percent:
                return this.value + "%";
            case ESizeType.Auto:
                return ESizeType.Auto;
            default:
                return this.value + this.type.toString();
        }
    }

    public toStyleString(item: string): string {
        if (ESizeType.FlexScale == this.type) {
            return  "flex:" + this.value;
        }
        return item + ":" + this.value + this.type.toString();
    }

    public static valueOf(value: {type: ESizeType; value?: number;}): SizeInfo;
    public static valueOf(value:number, type:ESizeType): SizeInfo;
    public static valueOf(value:any, type?:ESizeType): SizeInfo {
        if (type) {
            return new SizeInfo(value, type);
        } else {
            return new SizeInfo(value.value, value.type);
        }
    }

    public static valueOfPX(value: number): SizeInfo {
        return new SizeInfo(value, ESizeType.px);
    }

    public static valueOfMM(value: number): SizeInfo {
        return new SizeInfo(value, ESizeType.mm);
    }

    public static valueOfPercent(value: number): SizeInfo {
        return new SizeInfo(value, ESizeType.Percent);
    }
 
    public static isFixSize(size: SizeInfo) {
        return size.getType() == ESizeType.px;
    }

    public static isPercentSize(size: SizeInfo) {
        return size.getType() == ESizeType.Percent;
    }

    static mm2Px(mm: number): number {
        GlobalSetting.checkMMPXSetting();
        return mm * GlobalSetting.MM_PX;
    }

    static getDPIRatio(WIDTH:number,MM:number): number {
        return WIDTH / SizeInfo.mm2Px(MM);
    }

    /**
     * 将px类型的尺寸字符，转成数值
     * @param strValue 
     * @returns 
     */
    static getNumFromPX(strValue: string | number): number {
        if (typeof strValue === 'number') return strValue;
        if (!strValue) return 0;
        if (strValue.toLowerCase().endsWith('px')) {
            return TypeUtil.toNum(strValue.replace('px', ''));
        } else {
            return TypeUtil.toNum(strValue);
        }
    }

}