import AbstractDomElement from "../AbstractDomElement";
import SpanElement from "./SpanElement";
import SizeInfo from "../../struct/SizeInfo";
import TableCellLocation from "../../struct/TableCellLocation";
import IXElement from "../xelement/IXElement";

export default class TableLayout extends AbstractDomElement {

    private tBody!: TBody;

    constructor(widths: number[], lineWidth: number = 1) {
        super();
        this.init(widths, lineWidth);
    }

    init(widths: number[], lineWidth: number = 1) {
        this.clearChildren();
        this.tBody = new TBody(widths, lineWidth);
        this.addChild(this.tBody);
    }

    protected createEl(tag:string): HTMLElement {
        return document.createElement("table");
    }

    initRows(heights: number[]): void {
        for (let height of heights) {
            this.addRow(height);
        }
    }

    resetRowsTop(): void {
        this.tBody.resetRowsTop();
    }

    setCellElement(row: number, col: number, element: AbstractDomElement): void {
        var cell = this.tBody.getRow(row).getCell(col);
        cell.addChild(element);
    }

    getCell(cellLocation: TableCellLocation): Cell {
        return this.tBody.getRow(cellLocation.getRow()).getCell(cellLocation.getCol());
    }

    getCellAt(row: number, col: number): Cell {
        return this.tBody.getRow(row).getCell(col);
    }

    addRow(height: number): Row {
        return this.tBody.addRow(height);
    }

    getLastColumnIndex(): number {
        return this.tBody.getLastColumnIndex();
    }

    getLastRowIndex(): number {
        return this.tBody.getLastRowIndex();
    }

    hitCell(x: number, y: number): TableCellLocation | null {
        this.resetRowsTop();
        var hitRowIndex = -1;
        for (var rowIndex = 0; rowIndex < this.tBody.getRowCount(); rowIndex++) {
            var row = this.tBody.getRow(rowIndex);
            if (y > row.getTop() && y < row.getTop() + row.getHeight()) {
                hitRowIndex = rowIndex;
                break;
            }
        }

        var hitColIndex = -1;
        for (var colIndex = 0; colIndex < this.tBody.getColumnCount(); colIndex++) {
            var column = this.tBody.getColumn(colIndex);
            if (x > column.getLeft() && x < column.getLeft() + column.getWidth()) {
                hitColIndex = colIndex;
                break;
            }
        }

        if (hitRowIndex == -1 || hitColIndex == -1) return null;
        return new TableCellLocation(hitRowIndex, hitColIndex);
    }

    hitCellByEl(el: HTMLElement) : TableCellLocation | null {
        this.resetRowsTop();
        for (var rowIndex = 0; rowIndex < this.tBody.getRowCount(); rowIndex++) {
            var row = this.tBody.getRow(rowIndex);
            for (var colIndex = 0; colIndex < this.tBody.getColumnCount(); colIndex ++) {
                if (row.getCell(colIndex).getEl() == el) {
                    return new TableCellLocation(rowIndex, colIndex);
                }
            }
        }
        return null;
    }

    /**
    * 某列的左边线位置
    * @param colIndex 
    * @returns 
    */
    getLeft(colIndex: number): number {
        return this.tBody.getColumn(colIndex).getLeft();
    }

    /**
     * 某列的右边线位置
     * @param colIndex 
     * @returns 
     */
    getRight(colIndex: number): number {
        var column = this.tBody.getColumn(colIndex);
        return column.getLeft() + column.getWidth();
    }

    getColumnWidth(colIndex: number): number {
        var column = this.tBody.getColumn(colIndex);
        return column.getWidth();
    }

    /**
     * 某行的上边线位置
     * 
     * @param rowIndex 
     * @returns 
     */
    getTop(rowIndex: number): number {
        this.resetRowsTop();
        return this.tBody.getRow(rowIndex).getTop();
    }

    /**
     * 某行的下边线位置
     * @param rowIndex 
     * @returns 
     */
    getBottom(rowIndex: number): number {
        this.resetRowsTop();
        var row = this.tBody.getRow(rowIndex);
        return row.getTop() + row.getHeight();
    }

    getRowHeight(rowIndex: number): number {
        var row = this.tBody.getRow(rowIndex);
        return row.getHeight();
    }

    getHeightBetween(row1: number, row2: number) {
        if (row1 > row2) {
            // 交换行序号
            var tmp = row1; row1 = row2; row1 = tmp;
        }
        this.resetRowsTop();
        return this.getBottom(row2) - this.getTop(row1);
    }

    getWidth(): number {
        var lastCol = this.tBody.getLastColumnIndex();
        return this.getRight(lastCol);
    }

    getHeight(): number {
        this.resetRowsTop();
        var lastRow = this.tBody.getRowCount() - 1;
        return this.getBottom(lastRow);
    }

    isMarkedCell(row: number, col: number): boolean {
        var cell = this.tBody.getRow(row).getCell(col);
        return cell.isMarked();
    }

    markRange(startRow: number, endRow: number) {
        var endCol = this.getLastColumnIndex();
        for (var rowIndex = startRow; rowIndex <= endRow; rowIndex ++) {
            var row = this.tBody.getRow(rowIndex);
            for (var colIndex = 0; colIndex <= endCol; colIndex ++) {
                var cell = row.getCell(colIndex);
                cell.setMarked(true);
            }
        }
    }

    markCell(row: number, col: number): void {
        this.tBody.getRow(row).getCell(col).setMarked(true);
    }

    clearMarked(): void {
        var endRow = this.getLastRowIndex();
        var endCol = this.getLastColumnIndex();
        for (var rowIndex = 0; rowIndex <= endRow; rowIndex ++) {
            var row = this.tBody.getRow(rowIndex);
            for (var colIndex = 0; colIndex <= endCol; colIndex ++) {
                var cell = row.getCell(colIndex);
                cell.setMarked(false);
            }
        }
    }
}

class TBody extends AbstractDomElement {

    private lineWidth = 1;

    private columns: Column[] = [];

    constructor(widths: number[], lineWidth: number) {
        super();
        this.lineWidth = lineWidth;
        for (let width of widths) {
            var column = new Column(width);
            this.columns.push(column);
        }
        this.resetColumnsLeft();
        var row = this.addRow(0.5);
        row.addClass('first');
        this.initFirstRow(row);
    }

    resetColumnsLeft() {
        var left = 0;
        for (let column of this.columns) {
            left += this.lineWidth;
            column.setLeft(left);
            left += column.getWidth();
        }
    }

    resetRowsTop() {
        var top = 0;
        for (var rowIndex = 0; rowIndex<this.getRowCount(); rowIndex++) {
            var row = this.getRow(rowIndex);
            row.setTop(top);
            top += row.getHeight();
        }
    }

    private initFirstRow(row: Row) {
        for (var i=0; i<this.columns.length; i++) {
            var span = new SpanElement("");
            span.setStyleWidth(SizeInfo.valueOfPX(this.columns[i].getWidth()));
            span.setStyle('display', 'block');
            var cell = row.getCell(i);
            cell.setStyleWidth(SizeInfo.valueOfPX(this.columns[i].getWidth()));
            cell.addChild(span);
        }
    }

    protected createEl(tag:string): HTMLElement {
        return document.createElement("tbody");
    }

    addRow(height: number): Row {
        var row = new Row(height);
        for (let column of this.columns) {
            var cell = new Cell();
            row.addCell(cell);
        }
        this.addChild(row);
        return row;
    }

    /**
     * 获取行
     * @param index 
     * @returns 
     */
    getRow(index: number): Row {
        return <Row> this.getChildAt(index + 1); // 第一行不算
    }

    /**
     * 返回行数
     * @returns 
     */
    getRowCount(): number {
        return this.getChildCount() - 1; // 第一行不算
    }

    getColumn(index: number): Column {
        return this.columns[index];
    }

    getColumnCount(): number {
        return this.columns.length;
    }

    getLastColumnIndex(): number {
        return this.getColumnCount() - 1;
    }

    getLastRowIndex(): number {
        return this.getRowCount() - 1;
    }
}

class Column {
    private left: number = 0;

    private width: number;

    constructor(width: number) {
        this.width = width;
    }

    setLeft(left: number) {
        this.left = left;
    }

    getLeft() {
        return this.left;
    }

    getWidth(): number {
        return this.width;
    }
}

class Row extends AbstractDomElement {

    private top: number = 0;

    constructor(height: number) {
        super();
        //this.height = height;
        if (height > 0) this.setStyleHeight(SizeInfo.valueOfPX(height));
    }

    protected createEl(tag:string): HTMLElement {
        return document.createElement("tr");
    }

    setTop(top: number) {
        this.top = top;
    }

    getTop() {
        return this.top;
    }

    getHeight(): number {
        var height = $(this.getEl()).height();
        return height ? height : 0;
        //this.getEl().clientHeight; // 此处不用配置中的行height，是因为erp中有多个控件放在一个grid cell中的用法，会将row高度撑大
    }

    addCell(cell: Cell) {
        this.addChild(cell);
    }

    getCell(col: number): Cell {
        return <Cell> this.getChildAt(col);
    }

    setCellValue(col: number, value: any): void {
        var cell = this.getCell(col);
        cell.setText(value);
    }

    setCellElement(col: number, element: AbstractDomElement): Cell {
        var cell = this.getCell(col);
        cell.setContent(element);
        return cell;
    }
}

class Cell extends AbstractDomElement {

    private marked: boolean = false;

    protected createEl(tag:string): HTMLElement {
        return document.createElement("td");
    }

    setContent(element: AbstractDomElement): void {
        this.addChild(element);
    }

    getContent(): AbstractDomElement | null {
        if (this.getChildCount() == 0) return null;
        return this.getChildAt(0);
    }

    setMarked(b: boolean) {
        this.marked = b;
        //b ? this.addClass('cell_marked') : this.removeClass('cell_marked');
    }

    isMarked(): boolean {
        return this.marked;
    }
}


/****************************
 * 
 * 

.designingForm table.layout > tbody > tr > td {
    border-bottom: 1px dashed rgb(224, 224, 224);
    border-right: 1px dashed rgb(224, 224, 224);
}

tr {
    display: table-row;
    vertical-align: inherit;
    border-color: inherit;
}
.ui-glp .layout {
    border-collapse: separate;
    border-spacing: 0px;
    table-layout: fixed;
}

.designingForm table.layout {
    border-top: 1px dashed rgb(224, 224, 224);
    border-left: 1px dashed rgb(224, 224, 224);
}
.ui-glp .layout {
    border-collapse: separate;
    border-spacing: 0px;
    table-layout: fixed;
}

 */