import SizeInfo from "../struct/SizeInfo";
import IStyleFactory from "../style/IStyleFactory";
import StyleFactoryMananger from "../style/StyleFactoryMananger";
import IXElement from "./xelement/IXElement";
import HtmlXElement from "./xelement/html/HtmlXElement";

export default abstract class AbstractDomElement {

    protected styleFactory: IStyleFactory =  StyleFactoryMananger.getInstance().getStyleFactory();

    //protected el: HTMLElement;

    protected xEl: IXElement;

    protected children: AbstractDomElement[] = [];

    constructor(initEl?: HTMLElement | IXElement | null) {
        let tmpEl = initEl || this.createEl();
        if (tmpEl instanceof Element) {
            this.xEl = new HtmlXElement(tmpEl);
        } else {
            this.xEl = tmpEl;
        }
    }

    protected createEl(tag:string='div'): IXElement | HTMLElement {
        return new HtmlXElement(tag);
    }
 
    setID(id: string): void {
        this.setStyle("id", id);
    }

    setText(caption: string): void {
        this.xEl.setInnerText(caption);
    }

    getText(): string {
        return this.xEl.getInnerText();
    }

    setAttr(name: string, value: string) {
        this.xEl.setAttribute(name, value);
    }

    getAttr(name: string) {
        return this.xEl.getAttribute(name);
    }

    setStyleHeight(height: SizeInfo) {
        this.setStyle("height", height.toString());
    }

    setStyleWidth(width: SizeInfo) {
        this.setStyle("width", width.toString())
    }

    setStyle(key: any, value: any): void {
        this.xEl.setStyle(key, value);
    }

    setStyles(styles: any) {
        for (var key in styles) {
            this.setStyle(key, styles[key]) ;
        }
    }

    addClass(cls:string):void {
        if (cls) {
            this.xEl.addClass(cls);
        }
    }

    addClassArray(clsArray: string[] = []) {
        for (let c of clsArray) {
            this.addClass(c);
        }
    }

    removeClass(cls:string):void {
        this.xEl.removeClass(cls);
    }

    toggleClass(cls:string): void {
        this.xEl.toggleClass(cls);
    }

    hasClass(cls: string): boolean {
        return this.xEl.hasClass(cls);
    }

    getEl(): HTMLElement {
        return this.xEl.getHtmlElement();
    }

    parentEl(): HTMLElement | null {
        if (!this.getEl().parentElement) return null;
        return this.getEl().parentElement;
    }

    addChild(child?: AbstractDomElement | null): void {
        if (child) {
            this.children.push(child);
            this.mountChild(child);
        }
    }

    public removeChild(child?: AbstractDomElement | null): void {
        if (child) {
            var index = this.children.indexOf(child); 
            if (index>= 0) { 
                this.children.splice(index, 1); 
            }
            this.unmountChild(child);
        }
    }

    public getChildAt(index: number): AbstractDomElement {
        return this.children[index];
    }

    public getChildren(): AbstractDomElement[] {
        return this.children;
    }

    public getChildCount(): number {
        return this.children.length;
    }

    public clearChildren(): void {
        for (let child of this.children) {
            this.unmountChild(child);
        }
        this.children = [];
    }

    public render(parentEl: HTMLElement) {
        if (!parentEl.contains(this.getEl())) {
            parentEl.appendChild(this.getEl());
        }
    }

    public hasChild(child: AbstractDomElement): boolean {
        if (this.children == null) return false;
        return this.children.indexOf(child) > -1;
    }

    protected mountChild(child: AbstractDomElement | null) {
        if (child && !this.xEl.getHtmlElement().contains(child.getEl())) {
            this.xEl.getHtmlElement().appendChild(child.getEl());
        } 
    }

    private unmountChild(child: AbstractDomElement) {
        if (child && this.xEl.getHtmlElement().contains(child.getEl())) {
            this.xEl.getHtmlElement().removeChild(child.getEl());
        }
    }

    addEventListener(name: string, fun: any, useCapture: boolean = false) {
        this.xEl.addEventListener(name, fun, useCapture);
    }

    removeEventListener(name: string, fun: any, useCapture: boolean = false) {
        this.xEl.removeEventListener(name, fun, useCapture);
    }

    hover(funOver: any, funOut: any): void {
        this.xEl.addEventListener("mouseover", funOver, false);
        this.xEl.addEventListener("mouseout", funOut, false);
    }

    findElement(el: HTMLElement | null): AbstractDomElement | null {
        if (!el) {
            return null;
        }
        if (this.getEl() == el) {
            return this;
        }
        if (this.getChildren() != null) {
            for (let child of this.getChildren()) {
                var find = child.findElement(el);
                if (find != null) return find;
            }
        }
        return null;
    }

    markSelect(b: boolean): void {
        b ? this.addClass("selected") : this.removeClass("selected");
    }

    resize(width:number, height:number) {}

    fireResized(width:number, height:number) {}

    blur(): void {
        this.getEl().blur();
    }

    getScrollTop(): number {
        return this.xEl.getScrollTop();
    }

    setScrollTop(top: number): void {
        this.xEl.setScrollTop(top);
    }

    getScrollLeft(): number {
        return this.xEl.getScrollLeft();
    }

    setScrollLeft(left: number): void {
        this.xEl.setScrollLeft(left);
    }

    getNodeType(): number {
        return this.xEl.getNodeType();
    }

    getOffsetLeft(): number {
        return this.xEl.getOffsetLeft();
    }
    
    getOffsetTop(): number {
        return this.xEl.getOffsetTop();
    }

    getHtmlClientHeight(): number {
        return this.xEl.getHtmlClientHeight();
    }

    getHtmlClientWidth(): number {
        return this.xEl.getHtmlClientWidth();
    }

    getHtmlOffsetWidth(): number {
        return this.xEl.getHtmlOffsetWidth();
    }

    getBoundingClientRect(): any {
        return this.xEl.getBoundingClientRect();
    }

    setOncontextmenu(fn: any): void {
        this.xEl.setOncontextmenu(fn);
    }

    appendTo(parent: HTMLElement): void {
        parent.appendChild(this.getEl());
    }

    display(parent: AbstractDomElement): void {}
}