import ArrayList from "../../struct/ArrayList";
import SizeInfo from "../../struct/SizeInfo";
import JComponent from "../JComponent";
import Tab from "./Tab";

import TabbedPaneElement from "../../dom/element/TabbedPaneElement";
import DomElement from "../../dom/element/DomElement";
import Events from "../../event/Events";
import EventHub from "../../event/EventHub";
import { ESizeType } from "../../enum/Enums";

export default class JTabbedPane extends JComponent<TabbedPaneElement> {

    protected navGroup!: DomElement;

    protected contentGroup!: DomElement;

    private tabs: ArrayList<Tab>;

    private emitter: EventHub<Tab> = new EventHub();

    constructor() {
        super(new TabbedPaneElement());
        this.tabs = new ArrayList<Tab>();
        this.navGroup = new DomElement(<HTMLElement> this.getEl().querySelector('ul.tab-nav-group'));
        this.contentGroup = new DomElement(<HTMLElement> this.getEl().querySelector('div.tab-content-group'));
        var skin = this.styleFactory.getTabPaneSkin();
        this.addClass(skin.tab);
        this.navGroup.addClass(skin.tab_title);
        this.contentGroup.addClass(skin.tab_content);
        
        var self = this;
        this.emitter.on(Events.Tab.NAV_SELECT, function(tab: Tab) {
            self.selectTab(tab);
        });

        this.emitter.on(Events.Tab.NAV_CLOSE, function(tab: Tab) {
            self.removeTab(tab);
        });
    }

    addTab(key: string, caption: string): Tab {
        var tab = new Tab(key, caption, this.emitter);
        this.navGroup.addChild(tab.getNav());
        this.contentGroup.addChild(tab.getContent());
        this.tabs.add(tab);
        return tab;
    }

    removeTabByKey(key: string) {
        var tab = this.getTabByKey(key);
        this.removeTab(tab);
    }

    selectTab(tab: Tab) {
        if (tab) {
            this.unSelectAll();
            tab.select();
        }
    }

    getSelectedTab(): Tab | null {
        for (let tab of this.tabs.getList()) {
            if (tab.isSelected()) return tab;
        }
        return null;
    }

    select(index: number): void {
        var tab = this.tabs.get(index);
        this.selectTab(tab);
    }

    close(index: number): void {
        var tab = this.tabs.get(index);
        this.removeTab(tab);
    }

    closeAll(): void {
        for (let tab of this.tabs.getList()) {
            this.navGroup.removeChild(tab.getNav());
            this.contentGroup.removeChild(tab.getContent());
        }
        this.tabs.clearAll();
    }

    getTabByKey(key: string): Tab | null {
        for (let tab of this.tabs.getList()) {
            if (tab.getKey() == key) {
                return tab;
            }
        }
        return null;
    }

    setNavHeight(height: number) {
        this.navGroup.setStyleHeight(SizeInfo.valueOf(height, ESizeType.px));
    }

    removeTab(tab: Tab | null) {
        if (tab != null) {
            this.navGroup.removeChild(tab.getNav());
            this.contentGroup.removeChild(tab.getContent());
            var index = this.tabs.remove(tab);
            if (index < this.tabs.getSize()) {
                this.select(index);
            } else if (index > 0) {
                this.select(index - 1);
            }
        }
    }

    display(): void {
        var tab = this.getSelectedTab();
        if (tab) {
            tab.getContent().display();
        }

    }

    private unSelectAll() {
        for (let tab of this.tabs.getList()) {
            tab.unselect();
        }
    }

}


