package com.bokesoft.erp.desigerfunction.datasource;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.bokesoft.entity.BaseContextAction;
import com.bokesoft.erp.WebDesignerConfiguration;
import com.bokesoft.yes.common.log.LogSvr;
import com.bokesoft.yes.design.vo.Exp;
import com.bokesoft.yes.design.vo.Para;
import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.javadoc.Javadoc;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.reflections.Reflections;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

/***
 * 生成后端EntityContextAction子类所有函数的JSON文件
 *
 * @author ZHSY
 * @Date 2021-10-30
 */
public class GenerateFunctionsJsonFile {

    /*** 日志类 */
    private static final Logger logger = Logger.getLogger(GenerateFunctionsJsonFile.class.getName());

    private static final Integer POOL_SIZE = 5;

    private static ThreadPoolExecutor executor;

    /*** 后端函数json文件 */
    public static final String END_JSON_FILE = "/json/endFuns.json";
    public static String FunctionsJson;

    public static String getFunctionsJson() throws IOException {
        if (StringUtils.isBlank(FunctionsJson)) {
            String jsonContent = readFromFile();
            if (StringUtils.isNotEmpty(jsonContent)) {
                FunctionsJson = jsonContent;
            } else {
                GenerateFunctionsJsonFile.generateFunctionsJsonFile();
                writeToFile(FunctionsJson);
            }
        }
        return FunctionsJson;
    }

    private static String readFromFile() throws IOException {
        StringBuilder jsonFileContent = new StringBuilder();
        String designerJsonFilePath = WebDesignerConfiguration.getDesignerJsonFilePath();
        File file = new File(designerJsonFilePath);
        if (file.exists()) {
            BufferedReader bufferedReader = new BufferedReader(new FileReader(file));
            String read = "";
            while ((read = bufferedReader.readLine()) != null) {
                jsonFileContent.append(read).append("\n");
            }
        } else {
            InputStream inputStream = GenerateFunctionsJsonFile.class.getResourceAsStream(END_JSON_FILE);
            if (Objects.nonNull(inputStream)) {
                InputStreamReader reader = new InputStreamReader(inputStream, StandardCharsets.UTF_8);
                BufferedReader bufferedReader = new BufferedReader(reader);
                String read = "";
                while ((read = bufferedReader.readLine()) != null) {
                    jsonFileContent.append(read).append("\n");
                }
            }
        }

        return jsonFileContent.toString();
    }

    /**
     * 写入文件
     */
    public static void writeToFile(String functionsJson) {
        try {
            // 如果是jar包 不写
            if (isJar(WebDesignerConfiguration.class)) {
                return;
            }
            // 指定文件路径
//            String resourcePath = getJsonFilePath();
            String resourcePath = WebDesignerConfiguration.getDesignerJsonFilePath();

            // 创建文件写入器
            FileWriter fileWriter = new FileWriter(resourcePath);

            // 将 JSON 对象写入文件
            fileWriter.write(functionsJson);

            // 关闭文件写入器
            fileWriter.close();

            logger.info("JSON 写入文件成功。");

        } catch (IOException e) {
            LogSvr.getInstance().error(e.getMessage(), e);
        }
    }

    private static String getJsonFilePath() {
        String classPath = getAbsolutePath(WebDesignerConfiguration.class);
        // 指定文件路径
        String resourcePath = classPath.substring(0, classPath.indexOf("java"));
        return resourcePath + "resources" + File.separator + END_JSON_FILE;
    }

    @SuppressWarnings("rawtypes")
    public static void generateFunctionsJsonFile() {
        Map<String, Object> map = new HashMap<>();
        StringBuilder buf = new StringBuilder();
        try {
            Reflections reflections = new Reflections("com.bokesoft");
            Set<Class<? extends BaseContextAction>> classes = reflections.getSubTypesOf(BaseContextAction.class);
            // 分段分多线程去处理
            ArrayList<Set> sets = splitClasses(classes, POOL_SIZE);
            CountDownLatch latch = new CountDownLatch(sets.size());
            ArrayList<Future> futures = new ArrayList<>();
            executor = new ThreadPoolExecutor(POOL_SIZE, POOL_SIZE, 0L, TimeUnit.MILLISECONDS,
                    new LinkedBlockingQueue<Runnable>(1024), new ThreadPoolExecutor.AbortPolicy());

            for (Set set : sets) {
                Future<String> future = executor.submit(new Callable<String>() {
                    @Override
                    public String call() throws Exception {
                        StringBuilder keywords = new StringBuilder();
                        generateFunctionMap(set, keywords, map);
                        latch.countDown();
                        return keywords.toString();
                    }
                });
                futures.add(future);
            }

            latch.await();

            for (Future future : futures) {
                buf.append(future.get());
            }

            map.put("keyWords", buf.toString());
            // 关闭线程池
            executor.shutdown();
            // 保存json数据
            FunctionsJson = JSON.toJSONString(map, SerializerFeature.DisableCircularReferenceDetect, SerializerFeature.PrettyFormat);

        } catch (Throwable e) {
            logger.warning(ExceptionUtils.getStackTrace(e));
        }
    }

    /**
     * 对 classes 进行分段 subNum 为分成的段数
     *
     * @param classes
     * @param subNum
     * @return
     */
    private static ArrayList<Set> splitClasses(Set<Class<? extends BaseContextAction>> classes, Integer subNum) {
        ArrayList<Set> classesArr = new ArrayList<>();
        int size = classes.size();
        int perSize = size / subNum;
        int start = 0;
        int end = 0;
        for (int i = 0; i < subNum; i++) {
            start = i * perSize;
            end = (i + 1) * perSize;
            if (i == subNum - 1) {
                end = size;
            }
            Set<Class<? extends BaseContextAction>> subClasses = new HashSet<>();
            Iterator<Class<? extends BaseContextAction>> iterator = classes.iterator();
            int j = 0;
            while (iterator.hasNext()) {
                Class<? extends BaseContextAction> next = iterator.next();
                if (j >= start && j < end) {
                    subClasses.add(next);
                }
                j++;
            }
            classesArr.add(subClasses);
        }
        return classesArr;
    }

    private static void generateFunctionMap(Set<Class<? extends BaseContextAction>> classes, StringBuilder buf, Map<String, Object> map) throws FileNotFoundException {
        for (Class<? extends BaseContextAction> clazz : classes) {
            Method[] methods = clazz.getDeclaredMethods();
            if (methods.length > 0) {
                Exp exp = null;
                Para para = null;
                List<Para> paras = null;
                // 生成类下所有函数的信息 该方法非常耗时间
                Map<String, String> methodExpMap = extSetDescDetail(clazz);
                for (Method method : methods) {
                    // 只有public权限的方法保留,其他权限的方法过滤掉
                    if (!method.toString().startsWith("public ")) {
                        continue;
                    }
                    exp = new Exp();
                    if ("com.bokesoft.erp.ShortNameFunction".equals(clazz.getName())) {
                        exp.setKey(method.getName());
                    } else {
                        exp.setKey(clazz.getName() + "." + method.getName());
                    }
                    exp.setDesc("");

                    buf.append(exp.getKey());
                    buf.append(" ");

                    paras = new ArrayList<>();
                    StringBuilder paraSBuilder = new StringBuilder();

                    // 获取本方法所有参数类型，存入数组
                    Class<?>[] getTypeParameters = method.getParameterTypes();
                    if (getTypeParameters.length == 0) {
                        exp.setDescDetail("此方法无参数");
                    }

                    StringBuilder params = new StringBuilder();
                    Parameter[] getParameters = method.getParameters();
                    if (null != getParameters && getParameters.length > 0) {
                        String parameterTypeName = "";// 参数类型名
                        for (int i = 0; i < getParameters.length; i++) {
                            para = new Para();
                            if (getTypeParameters.length > 0) {
                                parameterTypeName = getTypeParameters[i].getName();
                            }
                            // 参数名
                            para.setKey(getParameters[i].getName());
                            para.setOptional("true");
                            if (StringUtils.isNotEmpty(parameterTypeName)) {
                                para.setType(parameterTypeName.replaceAll("\\[L", "").replaceAll("\\;", ""));
                                para.setDesc("参数" + (i + 1) + "：" + getParameters[i].getName() + ", 参数类型:"
                                        + para.getType() + "; ");
                                paraSBuilder.append(para.getDesc());
                                params.append(getParameters[i].getName()).append(",");
                            } else {
                                para.setType("");
                                para.setDesc("");
                            }
                            paras.add(para);
                        }
                    }

                    // 返回值
                    Type t = method.getAnnotatedReturnType().getType();
                    exp.setReturnValueType(t.getTypeName());
                    exp.setPlatform("All");
                    String descDetail;
                    if (methodExpMap.containsKey(exp.getKey()) && StringUtils.isNotEmpty(descDetail = methodExpMap.get(exp.getKey()))) {
                        exp.setDescDetail(descDetail);
                    } else {
                        exp.setDescDetail(paraSBuilder + "返回类型：" + t.getTypeName());
                    }
                    exp.setParas(paras);

                    String functionName = "";
                    if ("com.bokesoft.erp.ShortNameFunction".equals(clazz.getName())) {
                        functionName = method.getName();
                    } else {
                        functionName = clazz.getName() + "." + method.getName();
                    }
                    if (params.length() > 0) {
                        map.put(functionName + "("
                                + params.substring(0, params.toString().lastIndexOf(",")) + ")", exp);
                    } else {
                        map.put(functionName + "()", exp);
                    }
                }
            }
        }
    }

    /**
     * 设置函数描述
     */
    public static Map<String, String> extSetDescDetail(Class<?> cls) throws FileNotFoundException {
        String absolutePath = getAbsolutePath(cls);
        String name = cls.getName();
        Map<String, String> methodMap = new HashMap<>();
        File file;
        if (StringUtils.isNotBlank(absolutePath) && (file = new File(absolutePath)).exists()) {
            // 耗时
            CompilationUnit cu = StaticJavaParser.parse(file);
            cu.findAll(MethodDeclaration.class).forEach(method -> {
                // 获取方法名
                // 获取方法名
                String methodName = name+"."+method.getName().asString();
                Javadoc javadoc = method.getJavadoc().orElse(null);
                String comment = "";
                if (javadoc != null) {
                    comment = javadoc.getDescription().toText().replaceAll("[\\n\\r\\t/*]", "").trim();
                }
                methodMap.put(methodName, comment);
                methodMap.put(method.getName().asString(), comment);
            });
        }
        return methodMap;

    }

    public static String getAbsolutePath(Class<?> cls) {
        try {
            String classFileParentPath = new File(Objects.requireNonNull(cls.getResource(cls.getSimpleName() + ".class")).getFile()).getParent();
            String sourceFileParentPath = classFileParentPath.replace("target\\classes", "src\\main\\java");
            File dir = new File(sourceFileParentPath);
            if (dir.exists()) {
                return sourceFileParentPath + File.separator + cls.getSimpleName() + ".java";
            }
            // 如果这是jar包下的class，直接退出
            if (isJar(cls) || !classFileParentPath.contains("target")) {
                return "";
            }
            // 寻找源文件的绝对路径 针对erp-business 下的一些包分为多个类型的java文件夹
            // 1. 截取 target 之前的路径 这个路径是module 名
            String modulePath = classFileParentPath.substring(0, classFileParentPath.indexOf("target"));
            // 2. 找 module 下的到 com.bokesoft 的所有路径
            List<String> parentPaths = findFilePaths(modulePath, "com.bokesoft");
            // 3. 对 src/ 路径下的所有可能路径进行判断，如果存在则返回
            for (String parentPath : parentPaths) {
                String sourcePath = parentPath + File.separator + cls.getName().replace(".", File.separator) + ".java";
                File sourceFile = new File(sourcePath);
                if (sourceFile.exists()) {
                    return sourcePath;
                }
            }
        } catch (Exception e) {
            LogSvr.getInstance().error(e.getMessage(), e);
        }
        return "";
    }

    public static Boolean isJar(Class<?> cls) {
        String classFileParentPath = new File(Objects.requireNonNull(cls.getResource(cls.getSimpleName() + ".class")).getFile()).getParent();
        return classFileParentPath.contains(".jar");
    }

    public static List<String> findFilePaths(String modulePath, String packageName) {
        List<String> filePaths = new ArrayList<>();
        File[] modulePaths = new File(modulePath + "src\\main").listFiles();
        if (Objects.isNull(modulePaths)) {
            return filePaths;
        }
        for (File module : modulePaths) {
            // 如果不是文件夹或者是resources文件夹则跳过
            if (!module.isDirectory() || StringUtils.equals(module.getName(), "resources")) {
                continue;
            }
            String packagePath = module + File.separator + packageName.replace(".", File.separator);
            File packageDir = new File(packagePath);
            if (packageDir.exists() && packageDir.isDirectory()) {
                filePaths.add(module.getAbsolutePath());
            }
        }
        return filePaths;
    }

}
