package com.bokesoft.erp;

import javax.annotation.PostConstruct;

import com.bokesoft.erp.designerxmlattr.GenerateXmlAttrsJsonFile;
import com.bokesoft.yes.common.log.LogSvr;
import com.bokesoft.yes.design.function.DesignActionUtil;
import com.bokesoft.yes.design.io.LoadFileTree;
import com.bokesoft.yes.helper.FilePathHelper;
import com.bokesoft.yigo.meta.base.IMetaResolver;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.factory.IMetaResolverFactory;
import com.bokesoft.yigo.meta.factory.MetaFactory;
import com.bokesoft.yigo.meta.intf.IMetaSolution;
import com.bokesoft.yigo.meta.solution.MetaSolution;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;

import com.bokesoft.erp.all.initiator.InitializingDesigner;
import com.bokesoft.yes.design.utils.ReloadForm;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

@Configuration
@ComponentScan
public class WebDesignerConfiguration {
    private static Logger logger = LoggerFactory.getLogger(WebDesignerConfiguration.class);

    /** 确定是否启用 Web Designer */
    @Value("${application-erp.webdesigner.enable:false}")
    private boolean enable;
    /**
     * 设置 "设计器目标 Solution" 的 URL("file:..." 或者 "classpath:...", 要求能够对应到实际的文件路径);
     * 设计器可以修改的内容包括目标 Solution 父目录下所有的 Solution;
     * 如果不指定, 设计器会将系统的 Primary Solution 视作目标 Solution.
     */
    @Value("${application-erp.webdesigner.target-solution-location:}")
    private String targetSolutionLocation;

    private boolean delayedProcessScope = false;
    
    @Value("${yigo.configuration.solution.data-path:}")
    private String dataPath;

    @Autowired
    private ResourceLoader resourceLoader;

    private static WebDesignerConfiguration instance;
    public WebDesignerConfiguration() {
        instance = this;
    }

    @PostConstruct
    public void postConstruct() throws IOException {
        InitializingDesigner.setEnable(enable);

        setupTargetSolutionPath();
        
        String isDelayedProcessScope=DesignActionUtil.getPreferencesInfo("DelayedProcessScope");
        if(isDelayedProcessScope!=null){
            delayedProcessScope = Boolean.parseBoolean(isDelayedProcessScope);
        }
        ReloadForm.setDelayedProcessScope(delayedProcessScope);
        logger.info("WebDesignerConfiguration inited!");
    }

    public void setupTargetSolutionPath() throws IOException{
        if (StringUtils.isBlank(targetSolutionLocation)){
            FilePathHelper.setTargetSolutionPath(null);
            return;
        }

        String locError = null;
        Resource res = resourceLoader.getResource(targetSolutionLocation);
        if (res.isFile()){
            File f = res.getFile();
            String locPath = f.getCanonicalPath();
            if (! f.exists()){
                locError = "设计器目标 Solution '"+targetSolutionLocation+"' 对应的文件路径 '"+locPath+"' 不存在";
            }else if (! f.isDirectory()){
                locError = "设计器目标 Solution '"+targetSolutionLocation+"' 对应的文件路径 '"+locPath+"' 不是有效目录";
            }else{
                FilePathHelper.setTargetSolutionPath(locPath);
                logger.info("设计器目标 Solution '{}' 的实际路径为 '{}' .", targetSolutionLocation, locPath);
                return;
            }
        }else{
            locError = "设计器目标 Solution '"+targetSolutionLocation+"' 不是文件系统路径";
        }
        if (null!=locError){
            logger.error("Web Designer 配置错误: "+locError);
            throw new RuntimeException(locError);
        }
    }

    public static String getDesignerDataPath() {
        return FilePathHelper.toBackFilePath(instance.dataPath)+ File.separator+"designer";
    }
    public static String getDesignerSolutionPathPath(String projectKey) throws Throwable {
        String solutionPath = null;
        IMetaFactory globalInstance = MetaFactory.getGlobalInstance();
//        for (MetaSolution metaSolution : globalInstance.getMetaSolutions()) {
            MetaSolution solution = (MetaSolution) globalInstance.getSolution().getProject(projectKey).getSolution();
//            if (metaSolution.getProjectCollection().containsKey(projectKey)){
                solutionPath = LoadFileTree.getSolutionPath(solution.getKey());
//                break;

//        }
        if (solutionPath == null) {
            throw new Exception("solution not found!");
        }
        return FilePathHelper.toBackFilePath(solutionPath);
    }
    public static Set<String> getDesignerSolutionTmpDataPaths() throws Throwable {
        Set<String> paths = new HashSet<>();
        IMetaFactory globalInstance = MetaFactory.getGlobalInstance();
        for (MetaSolution metaSolution : globalInstance.getMetaSolutions()) {
            String solutionPath = LoadFileTree.getSolutionPath(metaSolution.getKey());
            String path = solutionPath + File.separator+ "Data" + File.separator+ "designer" + File.separator + "tmp";
            paths.add(FilePathHelper.toBackFilePath(path));
        }
        return paths;
    }
    public static String getDesignerSolutionTmpDataPath(String projectKey) throws Throwable {
        String solutionPath = getDesignerSolutionPathPath(projectKey) + File.separator + "Data" + File.separator + "designer" + File.separator + "tmp";
        return PathNormalization(solutionPath.replaceAll("/","\\\\").replace("\\\\",File.separator));
    }
    public static String getDesignerImportSolutionsPath() {
        return FilePathHelper.toBackFilePath(instance.dataPath)+ File.separator+"designer"+ File.separator+ "ImportSolutions.xml";
    }
    public static String getDesignerSolutionDataPath() {
        return FilePathHelper.toBackFilePath(instance.dataPath)+ File.separator+"designer"+ File.separator+"solutions";
    }
    public static String  getDesignerJsonFilePath (){
        return FilePathHelper.toBackFilePath(instance.dataPath)+ File.separator+"designer"+ File.separator+ "endFuns.json";

    }
    public static String  getxmlAttrJsonFilePath (){
        return FilePathHelper.toBackFilePath(instance.dataPath)+ File.separator+"designer"+ File.separator+ "xmlAttrs.json";
    }

    public static  String PathNormalization(String path) {
        Path path1 = Paths.get(path);
        return FilePathHelper.toBackFilePath(path1.normalize().toString());
    }
}
