/*!
 *  decimal.js v10.4.3
 *  An arbitrary-precision Decimal type for JavaScript.
 *  https://github.com/MikeMcl/decimal.js
 *  Copyright (c) 2022 Michael Mclaughlin <M8ch88l@gmail.com>
 *  MIT Licence
 */
var EXP_LIMIT = 9e15, MAX_DIGITS = 1e9, NUMERALS = "0123456789abcdef", LN10 = "2.3025850929940456840179914546843642076011014886287729760333279009675726096773524802359972050895982983419677840422862486334095254650828067566662873690987816894829072083255546808437998948262331985283935053089653777326288461633662222876982198867465436674744042432743651550489343149393914796194044002221051017141748003688084012647080685567743216228355220114804663715659121373450747856947683463616792101806445070648000277502684916746550586856935673420670581136429224554405758925724208241314695689016758940256776311356919292033376587141660230105703089634572075440370847469940168269282808481184289314848524948644871927809676271275775397027668605952496716674183485704422507197965004714951050492214776567636938662976979522110718264549734772662425709429322582798502585509785265383207606726317164309505995087807523710333101197857547331541421808427543863591778117054309827482385045648019095610299291824318237525357709750539565187697510374970888692180205189339507238539205144634197265287286965110862571492198849978748873771345686209167058", PI$9 = "3.1415926535897932384626433832795028841971693993751058209749445923078164062862089986280348253421170679821480865132823066470938446095505822317253594081284811174502841027019385211055596446229489549303819644288109756659334461284756482337867831652712019091456485669234603486104543266482133936072602491412737245870066063155881748815209209628292540917153643678925903600113305305488204665213841469519415116094330572703657595919530921861173819326117931051185480744623799627495673518857527248912279381830119491298336733624406566430860213949463952247371907021798609437027705392171762931767523846748184676694051320005681271452635608277857713427577896091736371787214684409012249534301465495853710507922796892589235420199561121290219608640344181598136297747713099605187072113499999983729780499510597317328160963185950244594553469083026425223082533446850352619311881710100031378387528865875332083814206171776691473035982534904287554687311595628638823537875937519577818577805321712268066130019278766111959092164201989380952572010654858632789", DEFAULTS$1 = {
  // These values must be integers within the stated ranges (inclusive).
  // Most of these values can be changed at run-time using the `Decimal.config` method.
  // The maximum number of significant digits of the result of a calculation or base conversion.
  // E.g. `Decimal.config({ precision: 20 });`
  precision: 20,
  // 1 to MAX_DIGITS
  // The rounding mode used when rounding to `precision`.
  //
  // ROUND_UP         0 Away from zero.
  // ROUND_DOWN       1 Towards zero.
  // ROUND_CEIL       2 Towards +Infinity.
  // ROUND_FLOOR      3 Towards -Infinity.
  // ROUND_HALF_UP    4 Towards nearest neighbour. If equidistant, up.
  // ROUND_HALF_DOWN  5 Towards nearest neighbour. If equidistant, down.
  // ROUND_HALF_EVEN  6 Towards nearest neighbour. If equidistant, towards even neighbour.
  // ROUND_HALF_CEIL  7 Towards nearest neighbour. If equidistant, towards +Infinity.
  // ROUND_HALF_FLOOR 8 Towards nearest neighbour. If equidistant, towards -Infinity.
  //
  // E.g.
  // `Decimal.rounding = 4;`
  // `Decimal.rounding = Decimal.ROUND_HALF_UP;`
  rounding: 4,
  // 0 to 8
  // The modulo mode used when calculating the modulus: a mod n.
  // The quotient (q = a / n) is calculated according to the corresponding rounding mode.
  // The remainder (r) is calculated as: r = a - n * q.
  //
  // UP         0 The remainder is positive if the dividend is negative, else is negative.
  // DOWN       1 The remainder has the same sign as the dividend (JavaScript %).
  // FLOOR      3 The remainder has the same sign as the divisor (Python %).
  // HALF_EVEN  6 The IEEE 754 remainder function.
  // EUCLID     9 Euclidian division. q = sign(n) * floor(a / abs(n)). Always positive.
  //
  // Truncated division (1), floored division (3), the IEEE 754 remainder (6), and Euclidian
  // division (9) are commonly used for the modulus operation. The other rounding modes can also
  // be used, but they may not give useful results.
  modulo: 1,
  // 0 to 9
  // The exponent value at and beneath which `toString` returns exponential notation.
  // JavaScript numbers: -7
  toExpNeg: -7,
  // 0 to -EXP_LIMIT
  // The exponent value at and above which `toString` returns exponential notation.
  // JavaScript numbers: 21
  toExpPos: 21,
  // 0 to EXP_LIMIT
  // The minimum exponent value, beneath which underflow to zero occurs.
  // JavaScript numbers: -324  (5e-324)
  minE: -EXP_LIMIT,
  // -1 to -EXP_LIMIT
  // The maximum exponent value, above which overflow to Infinity occurs.
  // JavaScript numbers: 308  (1.7976931348623157e+308)
  maxE: EXP_LIMIT,
  // 1 to EXP_LIMIT
  // Whether to use cryptographically-secure random number generation, if available.
  crypto: false
  // true/false
}, inexact, quadrant, external = true, decimalError = "[DecimalError] ", invalidArgument = decimalError + "Invalid argument: ", precisionLimitExceeded = decimalError + "Precision limit exceeded", cryptoUnavailable = decimalError + "crypto unavailable", tag = "[object Decimal]", mathfloor = Math.floor, mathpow = Math.pow, isBinary = /^0b([01]+(\.[01]*)?|\.[01]+)(p[+-]?\d+)?$/i, isHex$1 = /^0x([0-9a-f]+(\.[0-9a-f]*)?|\.[0-9a-f]+)(p[+-]?\d+)?$/i, isOctal = /^0o([0-7]+(\.[0-7]*)?|\.[0-7]+)(p[+-]?\d+)?$/i, isDecimal = /^(\d+(\.\d*)?|\.\d+)(e[+-]?\d+)?$/i, BASE = 1e7, LOG_BASE = 7, MAX_SAFE_INTEGER$1 = 9007199254740991, LN10_PRECISION = LN10.length - 1, PI_PRECISION = PI$9.length - 1, P$1 = { toStringTag: tag };
P$1.absoluteValue = P$1.abs = function() {
  var x2 = new this.constructor(this);
  if (x2.s < 0)
    x2.s = 1;
  return finalise(x2);
};
P$1.ceil = function() {
  return finalise(new this.constructor(this), this.e + 1, 2);
};
P$1.clampedTo = P$1.clamp = function(min3, max3) {
  var k2, x2 = this, Ctor = x2.constructor;
  min3 = new Ctor(min3);
  max3 = new Ctor(max3);
  if (!min3.s || !max3.s)
    return new Ctor(NaN);
  if (min3.gt(max3))
    throw Error(invalidArgument + max3);
  k2 = x2.cmp(min3);
  return k2 < 0 ? min3 : x2.cmp(max3) > 0 ? max3 : new Ctor(x2);
};
P$1.comparedTo = P$1.cmp = function(y2) {
  var i2, j2, xdL, ydL, x2 = this, xd = x2.d, yd = (y2 = new x2.constructor(y2)).d, xs = x2.s, ys = y2.s;
  if (!xd || !yd) {
    return !xs || !ys ? NaN : xs !== ys ? xs : xd === yd ? 0 : !xd ^ xs < 0 ? 1 : -1;
  }
  if (!xd[0] || !yd[0])
    return xd[0] ? xs : yd[0] ? -ys : 0;
  if (xs !== ys)
    return xs;
  if (x2.e !== y2.e)
    return x2.e > y2.e ^ xs < 0 ? 1 : -1;
  xdL = xd.length;
  ydL = yd.length;
  for (i2 = 0, j2 = xdL < ydL ? xdL : ydL; i2 < j2; ++i2) {
    if (xd[i2] !== yd[i2])
      return xd[i2] > yd[i2] ^ xs < 0 ? 1 : -1;
  }
  return xdL === ydL ? 0 : xdL > ydL ^ xs < 0 ? 1 : -1;
};
P$1.cosine = P$1.cos = function() {
  var pr, rm, x2 = this, Ctor = x2.constructor;
  if (!x2.d)
    return new Ctor(NaN);
  if (!x2.d[0])
    return new Ctor(1);
  pr = Ctor.precision;
  rm = Ctor.rounding;
  Ctor.precision = pr + Math.max(x2.e, x2.sd()) + LOG_BASE;
  Ctor.rounding = 1;
  x2 = cosine(Ctor, toLessThanHalfPi(Ctor, x2));
  Ctor.precision = pr;
  Ctor.rounding = rm;
  return finalise(quadrant == 2 || quadrant == 3 ? x2.neg() : x2, pr, rm, true);
};
P$1.cubeRoot = P$1.cbrt = function() {
  var e2, m2, n2, r2, rep, s2, sd, t3, t32, t3plusx, x2 = this, Ctor = x2.constructor;
  if (!x2.isFinite() || x2.isZero())
    return new Ctor(x2);
  external = false;
  s2 = x2.s * mathpow(x2.s * x2, 1 / 3);
  if (!s2 || Math.abs(s2) == 1 / 0) {
    n2 = digitsToString(x2.d);
    e2 = x2.e;
    if (s2 = (e2 - n2.length + 1) % 3)
      n2 += s2 == 1 || s2 == -2 ? "0" : "00";
    s2 = mathpow(n2, 1 / 3);
    e2 = mathfloor((e2 + 1) / 3) - (e2 % 3 == (e2 < 0 ? -1 : 2));
    if (s2 == 1 / 0) {
      n2 = "5e" + e2;
    } else {
      n2 = s2.toExponential();
      n2 = n2.slice(0, n2.indexOf("e") + 1) + e2;
    }
    r2 = new Ctor(n2);
    r2.s = x2.s;
  } else {
    r2 = new Ctor(s2.toString());
  }
  sd = (e2 = Ctor.precision) + 3;
  for (; ; ) {
    t3 = r2;
    t32 = t3.times(t3).times(t3);
    t3plusx = t32.plus(x2);
    r2 = divide(t3plusx.plus(x2).times(t3), t3plusx.plus(t32), sd + 2, 1);
    if (digitsToString(t3.d).slice(0, sd) === (n2 = digitsToString(r2.d)).slice(0, sd)) {
      n2 = n2.slice(sd - 3, sd + 1);
      if (n2 == "9999" || !rep && n2 == "4999") {
        if (!rep) {
          finalise(t3, e2 + 1, 0);
          if (t3.times(t3).times(t3).eq(x2)) {
            r2 = t3;
            break;
          }
        }
        sd += 4;
        rep = 1;
      } else {
        if (!+n2 || !+n2.slice(1) && n2.charAt(0) == "5") {
          finalise(r2, e2 + 1, 1);
          m2 = !r2.times(r2).times(r2).eq(x2);
        }
        break;
      }
    }
  }
  external = true;
  return finalise(r2, e2, Ctor.rounding, m2);
};
P$1.decimalPlaces = P$1.dp = function() {
  var w2, d2 = this.d, n2 = NaN;
  if (d2) {
    w2 = d2.length - 1;
    n2 = (w2 - mathfloor(this.e / LOG_BASE)) * LOG_BASE;
    w2 = d2[w2];
    if (w2)
      for (; w2 % 10 == 0; w2 /= 10)
        n2--;
    if (n2 < 0)
      n2 = 0;
  }
  return n2;
};
P$1.dividedBy = P$1.div = function(y2) {
  return divide(this, new this.constructor(y2));
};
P$1.dividedToIntegerBy = P$1.divToInt = function(y2) {
  var x2 = this, Ctor = x2.constructor;
  return finalise(divide(x2, new Ctor(y2), 0, 1, 1), Ctor.precision, Ctor.rounding);
};
P$1.equals = P$1.eq = function(y2) {
  return this.cmp(y2) === 0;
};
P$1.floor = function() {
  return finalise(new this.constructor(this), this.e + 1, 3);
};
P$1.greaterThan = P$1.gt = function(y2) {
  return this.cmp(y2) > 0;
};
P$1.greaterThanOrEqualTo = P$1.gte = function(y2) {
  var k2 = this.cmp(y2);
  return k2 == 1 || k2 === 0;
};
P$1.hyperbolicCosine = P$1.cosh = function() {
  var k2, n2, pr, rm, len2, x2 = this, Ctor = x2.constructor, one = new Ctor(1);
  if (!x2.isFinite())
    return new Ctor(x2.s ? 1 / 0 : NaN);
  if (x2.isZero())
    return one;
  pr = Ctor.precision;
  rm = Ctor.rounding;
  Ctor.precision = pr + Math.max(x2.e, x2.sd()) + 4;
  Ctor.rounding = 1;
  len2 = x2.d.length;
  if (len2 < 32) {
    k2 = Math.ceil(len2 / 3);
    n2 = (1 / tinyPow(4, k2)).toString();
  } else {
    k2 = 16;
    n2 = "2.3283064365386962890625e-10";
  }
  x2 = taylorSeries(Ctor, 1, x2.times(n2), new Ctor(1), true);
  var cosh2_x, i2 = k2, d8 = new Ctor(8);
  for (; i2--; ) {
    cosh2_x = x2.times(x2);
    x2 = one.minus(cosh2_x.times(d8.minus(cosh2_x.times(d8))));
  }
  return finalise(x2, Ctor.precision = pr, Ctor.rounding = rm, true);
};
P$1.hyperbolicSine = P$1.sinh = function() {
  var k2, pr, rm, len2, x2 = this, Ctor = x2.constructor;
  if (!x2.isFinite() || x2.isZero())
    return new Ctor(x2);
  pr = Ctor.precision;
  rm = Ctor.rounding;
  Ctor.precision = pr + Math.max(x2.e, x2.sd()) + 4;
  Ctor.rounding = 1;
  len2 = x2.d.length;
  if (len2 < 3) {
    x2 = taylorSeries(Ctor, 2, x2, x2, true);
  } else {
    k2 = 1.4 * Math.sqrt(len2);
    k2 = k2 > 16 ? 16 : k2 | 0;
    x2 = x2.times(1 / tinyPow(5, k2));
    x2 = taylorSeries(Ctor, 2, x2, x2, true);
    var sinh2_x, d5 = new Ctor(5), d16 = new Ctor(16), d20 = new Ctor(20);
    for (; k2--; ) {
      sinh2_x = x2.times(x2);
      x2 = x2.times(d5.plus(sinh2_x.times(d16.times(sinh2_x).plus(d20))));
    }
  }
  Ctor.precision = pr;
  Ctor.rounding = rm;
  return finalise(x2, pr, rm, true);
};
P$1.hyperbolicTangent = P$1.tanh = function() {
  var pr, rm, x2 = this, Ctor = x2.constructor;
  if (!x2.isFinite())
    return new Ctor(x2.s);
  if (x2.isZero())
    return new Ctor(x2);
  pr = Ctor.precision;
  rm = Ctor.rounding;
  Ctor.precision = pr + 7;
  Ctor.rounding = 1;
  return divide(x2.sinh(), x2.cosh(), Ctor.precision = pr, Ctor.rounding = rm);
};
P$1.inverseCosine = P$1.acos = function() {
  var halfPi, x2 = this, Ctor = x2.constructor, k2 = x2.abs().cmp(1), pr = Ctor.precision, rm = Ctor.rounding;
  if (k2 !== -1) {
    return k2 === 0 ? x2.isNeg() ? getPi(Ctor, pr, rm) : new Ctor(0) : new Ctor(NaN);
  }
  if (x2.isZero())
    return getPi(Ctor, pr + 4, rm).times(0.5);
  Ctor.precision = pr + 6;
  Ctor.rounding = 1;
  x2 = x2.asin();
  halfPi = getPi(Ctor, pr + 4, rm).times(0.5);
  Ctor.precision = pr;
  Ctor.rounding = rm;
  return halfPi.minus(x2);
};
P$1.inverseHyperbolicCosine = P$1.acosh = function() {
  var pr, rm, x2 = this, Ctor = x2.constructor;
  if (x2.lte(1))
    return new Ctor(x2.eq(1) ? 0 : NaN);
  if (!x2.isFinite())
    return new Ctor(x2);
  pr = Ctor.precision;
  rm = Ctor.rounding;
  Ctor.precision = pr + Math.max(Math.abs(x2.e), x2.sd()) + 4;
  Ctor.rounding = 1;
  external = false;
  x2 = x2.times(x2).minus(1).sqrt().plus(x2);
  external = true;
  Ctor.precision = pr;
  Ctor.rounding = rm;
  return x2.ln();
};
P$1.inverseHyperbolicSine = P$1.asinh = function() {
  var pr, rm, x2 = this, Ctor = x2.constructor;
  if (!x2.isFinite() || x2.isZero())
    return new Ctor(x2);
  pr = Ctor.precision;
  rm = Ctor.rounding;
  Ctor.precision = pr + 2 * Math.max(Math.abs(x2.e), x2.sd()) + 6;
  Ctor.rounding = 1;
  external = false;
  x2 = x2.times(x2).plus(1).sqrt().plus(x2);
  external = true;
  Ctor.precision = pr;
  Ctor.rounding = rm;
  return x2.ln();
};
P$1.inverseHyperbolicTangent = P$1.atanh = function() {
  var pr, rm, wpr, xsd, x2 = this, Ctor = x2.constructor;
  if (!x2.isFinite())
    return new Ctor(NaN);
  if (x2.e >= 0)
    return new Ctor(x2.abs().eq(1) ? x2.s / 0 : x2.isZero() ? x2 : NaN);
  pr = Ctor.precision;
  rm = Ctor.rounding;
  xsd = x2.sd();
  if (Math.max(xsd, pr) < 2 * -x2.e - 1)
    return finalise(new Ctor(x2), pr, rm, true);
  Ctor.precision = wpr = xsd - x2.e;
  x2 = divide(x2.plus(1), new Ctor(1).minus(x2), wpr + pr, 1);
  Ctor.precision = pr + 4;
  Ctor.rounding = 1;
  x2 = x2.ln();
  Ctor.precision = pr;
  Ctor.rounding = rm;
  return x2.times(0.5);
};
P$1.inverseSine = P$1.asin = function() {
  var halfPi, k2, pr, rm, x2 = this, Ctor = x2.constructor;
  if (x2.isZero())
    return new Ctor(x2);
  k2 = x2.abs().cmp(1);
  pr = Ctor.precision;
  rm = Ctor.rounding;
  if (k2 !== -1) {
    if (k2 === 0) {
      halfPi = getPi(Ctor, pr + 4, rm).times(0.5);
      halfPi.s = x2.s;
      return halfPi;
    }
    return new Ctor(NaN);
  }
  Ctor.precision = pr + 6;
  Ctor.rounding = 1;
  x2 = x2.div(new Ctor(1).minus(x2.times(x2)).sqrt().plus(1)).atan();
  Ctor.precision = pr;
  Ctor.rounding = rm;
  return x2.times(2);
};
P$1.inverseTangent = P$1.atan = function() {
  var i2, j2, k2, n2, px, t3, r2, wpr, x2, x3 = this, Ctor = x3.constructor, pr = Ctor.precision, rm = Ctor.rounding;
  if (!x3.isFinite()) {
    if (!x3.s)
      return new Ctor(NaN);
    if (pr + 4 <= PI_PRECISION) {
      r2 = getPi(Ctor, pr + 4, rm).times(0.5);
      r2.s = x3.s;
      return r2;
    }
  } else if (x3.isZero()) {
    return new Ctor(x3);
  } else if (x3.abs().eq(1) && pr + 4 <= PI_PRECISION) {
    r2 = getPi(Ctor, pr + 4, rm).times(0.25);
    r2.s = x3.s;
    return r2;
  }
  Ctor.precision = wpr = pr + 10;
  Ctor.rounding = 1;
  k2 = Math.min(28, wpr / LOG_BASE + 2 | 0);
  for (i2 = k2; i2; --i2)
    x3 = x3.div(x3.times(x3).plus(1).sqrt().plus(1));
  external = false;
  j2 = Math.ceil(wpr / LOG_BASE);
  n2 = 1;
  x2 = x3.times(x3);
  r2 = new Ctor(x3);
  px = x3;
  for (; i2 !== -1; ) {
    px = px.times(x2);
    t3 = r2.minus(px.div(n2 += 2));
    px = px.times(x2);
    r2 = t3.plus(px.div(n2 += 2));
    if (r2.d[j2] !== void 0)
      for (i2 = j2; r2.d[i2] === t3.d[i2] && i2--; )
        ;
  }
  if (k2)
    r2 = r2.times(2 << k2 - 1);
  external = true;
  return finalise(r2, Ctor.precision = pr, Ctor.rounding = rm, true);
};
P$1.isFinite = function() {
  return !!this.d;
};
P$1.isInteger = P$1.isInt = function() {
  return !!this.d && mathfloor(this.e / LOG_BASE) > this.d.length - 2;
};
P$1.isNaN = function() {
  return !this.s;
};
P$1.isNegative = P$1.isNeg = function() {
  return this.s < 0;
};
P$1.isPositive = P$1.isPos = function() {
  return this.s > 0;
};
P$1.isZero = function() {
  return !!this.d && this.d[0] === 0;
};
P$1.lessThan = P$1.lt = function(y2) {
  return this.cmp(y2) < 0;
};
P$1.lessThanOrEqualTo = P$1.lte = function(y2) {
  return this.cmp(y2) < 1;
};
P$1.logarithm = P$1.log = function(base2) {
  var isBase10, d2, denominator, k2, inf, num, sd, r2, arg = this, Ctor = arg.constructor, pr = Ctor.precision, rm = Ctor.rounding, guard = 5;
  if (base2 == null) {
    base2 = new Ctor(10);
    isBase10 = true;
  } else {
    base2 = new Ctor(base2);
    d2 = base2.d;
    if (base2.s < 0 || !d2 || !d2[0] || base2.eq(1))
      return new Ctor(NaN);
    isBase10 = base2.eq(10);
  }
  d2 = arg.d;
  if (arg.s < 0 || !d2 || !d2[0] || arg.eq(1)) {
    return new Ctor(d2 && !d2[0] ? -1 / 0 : arg.s != 1 ? NaN : d2 ? 0 : 1 / 0);
  }
  if (isBase10) {
    if (d2.length > 1) {
      inf = true;
    } else {
      for (k2 = d2[0]; k2 % 10 === 0; )
        k2 /= 10;
      inf = k2 !== 1;
    }
  }
  external = false;
  sd = pr + guard;
  num = naturalLogarithm(arg, sd);
  denominator = isBase10 ? getLn10(Ctor, sd + 10) : naturalLogarithm(base2, sd);
  r2 = divide(num, denominator, sd, 1);
  if (checkRoundingDigits(r2.d, k2 = pr, rm)) {
    do {
      sd += 10;
      num = naturalLogarithm(arg, sd);
      denominator = isBase10 ? getLn10(Ctor, sd + 10) : naturalLogarithm(base2, sd);
      r2 = divide(num, denominator, sd, 1);
      if (!inf) {
        if (+digitsToString(r2.d).slice(k2 + 1, k2 + 15) + 1 == 1e14) {
          r2 = finalise(r2, pr + 1, 0);
        }
        break;
      }
    } while (checkRoundingDigits(r2.d, k2 += 10, rm));
  }
  external = true;
  return finalise(r2, pr, rm);
};
P$1.minus = P$1.sub = function(y2) {
  var d2, e2, i2, j2, k2, len2, pr, rm, xd, xe, xLTy, yd, x2 = this, Ctor = x2.constructor;
  y2 = new Ctor(y2);
  if (!x2.d || !y2.d) {
    if (!x2.s || !y2.s)
      y2 = new Ctor(NaN);
    else if (x2.d)
      y2.s = -y2.s;
    else
      y2 = new Ctor(y2.d || x2.s !== y2.s ? x2 : NaN);
    return y2;
  }
  if (x2.s != y2.s) {
    y2.s = -y2.s;
    return x2.plus(y2);
  }
  xd = x2.d;
  yd = y2.d;
  pr = Ctor.precision;
  rm = Ctor.rounding;
  if (!xd[0] || !yd[0]) {
    if (yd[0])
      y2.s = -y2.s;
    else if (xd[0])
      y2 = new Ctor(x2);
    else
      return new Ctor(rm === 3 ? -0 : 0);
    return external ? finalise(y2, pr, rm) : y2;
  }
  e2 = mathfloor(y2.e / LOG_BASE);
  xe = mathfloor(x2.e / LOG_BASE);
  xd = xd.slice();
  k2 = xe - e2;
  if (k2) {
    xLTy = k2 < 0;
    if (xLTy) {
      d2 = xd;
      k2 = -k2;
      len2 = yd.length;
    } else {
      d2 = yd;
      e2 = xe;
      len2 = xd.length;
    }
    i2 = Math.max(Math.ceil(pr / LOG_BASE), len2) + 2;
    if (k2 > i2) {
      k2 = i2;
      d2.length = 1;
    }
    d2.reverse();
    for (i2 = k2; i2--; )
      d2.push(0);
    d2.reverse();
  } else {
    i2 = xd.length;
    len2 = yd.length;
    xLTy = i2 < len2;
    if (xLTy)
      len2 = i2;
    for (i2 = 0; i2 < len2; i2++) {
      if (xd[i2] != yd[i2]) {
        xLTy = xd[i2] < yd[i2];
        break;
      }
    }
    k2 = 0;
  }
  if (xLTy) {
    d2 = xd;
    xd = yd;
    yd = d2;
    y2.s = -y2.s;
  }
  len2 = xd.length;
  for (i2 = yd.length - len2; i2 > 0; --i2)
    xd[len2++] = 0;
  for (i2 = yd.length; i2 > k2; ) {
    if (xd[--i2] < yd[i2]) {
      for (j2 = i2; j2 && xd[--j2] === 0; )
        xd[j2] = BASE - 1;
      --xd[j2];
      xd[i2] += BASE;
    }
    xd[i2] -= yd[i2];
  }
  for (; xd[--len2] === 0; )
    xd.pop();
  for (; xd[0] === 0; xd.shift())
    --e2;
  if (!xd[0])
    return new Ctor(rm === 3 ? -0 : 0);
  y2.d = xd;
  y2.e = getBase10Exponent(xd, e2);
  return external ? finalise(y2, pr, rm) : y2;
};
P$1.modulo = P$1.mod = function(y2) {
  var q2, x2 = this, Ctor = x2.constructor;
  y2 = new Ctor(y2);
  if (!x2.d || !y2.s || y2.d && !y2.d[0])
    return new Ctor(NaN);
  if (!y2.d || x2.d && !x2.d[0]) {
    return finalise(new Ctor(x2), Ctor.precision, Ctor.rounding);
  }
  external = false;
  if (Ctor.modulo == 9) {
    q2 = divide(x2, y2.abs(), 0, 3, 1);
    q2.s *= y2.s;
  } else {
    q2 = divide(x2, y2, 0, Ctor.modulo, 1);
  }
  q2 = q2.times(y2);
  external = true;
  return x2.minus(q2);
};
P$1.naturalExponential = P$1.exp = function() {
  return naturalExponential(this);
};
P$1.naturalLogarithm = P$1.ln = function() {
  return naturalLogarithm(this);
};
P$1.negated = P$1.neg = function() {
  var x2 = new this.constructor(this);
  x2.s = -x2.s;
  return finalise(x2);
};
P$1.plus = P$1.add = function(y2) {
  var carry2, d2, e2, i2, k2, len2, pr, rm, xd, yd, x2 = this, Ctor = x2.constructor;
  y2 = new Ctor(y2);
  if (!x2.d || !y2.d) {
    if (!x2.s || !y2.s)
      y2 = new Ctor(NaN);
    else if (!x2.d)
      y2 = new Ctor(y2.d || x2.s === y2.s ? x2 : NaN);
    return y2;
  }
  if (x2.s != y2.s) {
    y2.s = -y2.s;
    return x2.minus(y2);
  }
  xd = x2.d;
  yd = y2.d;
  pr = Ctor.precision;
  rm = Ctor.rounding;
  if (!xd[0] || !yd[0]) {
    if (!yd[0])
      y2 = new Ctor(x2);
    return external ? finalise(y2, pr, rm) : y2;
  }
  k2 = mathfloor(x2.e / LOG_BASE);
  e2 = mathfloor(y2.e / LOG_BASE);
  xd = xd.slice();
  i2 = k2 - e2;
  if (i2) {
    if (i2 < 0) {
      d2 = xd;
      i2 = -i2;
      len2 = yd.length;
    } else {
      d2 = yd;
      e2 = k2;
      len2 = xd.length;
    }
    k2 = Math.ceil(pr / LOG_BASE);
    len2 = k2 > len2 ? k2 + 1 : len2 + 1;
    if (i2 > len2) {
      i2 = len2;
      d2.length = 1;
    }
    d2.reverse();
    for (; i2--; )
      d2.push(0);
    d2.reverse();
  }
  len2 = xd.length;
  i2 = yd.length;
  if (len2 - i2 < 0) {
    i2 = len2;
    d2 = yd;
    yd = xd;
    xd = d2;
  }
  for (carry2 = 0; i2; ) {
    carry2 = (xd[--i2] = xd[i2] + yd[i2] + carry2) / BASE | 0;
    xd[i2] %= BASE;
  }
  if (carry2) {
    xd.unshift(carry2);
    ++e2;
  }
  for (len2 = xd.length; xd[--len2] == 0; )
    xd.pop();
  y2.d = xd;
  y2.e = getBase10Exponent(xd, e2);
  return external ? finalise(y2, pr, rm) : y2;
};
P$1.precision = P$1.sd = function(z2) {
  var k2, x2 = this;
  if (z2 !== void 0 && z2 !== !!z2 && z2 !== 1 && z2 !== 0)
    throw Error(invalidArgument + z2);
  if (x2.d) {
    k2 = getPrecision$1(x2.d);
    if (z2 && x2.e + 1 > k2)
      k2 = x2.e + 1;
  } else {
    k2 = NaN;
  }
  return k2;
};
P$1.round = function() {
  var x2 = this, Ctor = x2.constructor;
  return finalise(new Ctor(x2), x2.e + 1, Ctor.rounding);
};
P$1.sine = P$1.sin = function() {
  var pr, rm, x2 = this, Ctor = x2.constructor;
  if (!x2.isFinite())
    return new Ctor(NaN);
  if (x2.isZero())
    return new Ctor(x2);
  pr = Ctor.precision;
  rm = Ctor.rounding;
  Ctor.precision = pr + Math.max(x2.e, x2.sd()) + LOG_BASE;
  Ctor.rounding = 1;
  x2 = sine(Ctor, toLessThanHalfPi(Ctor, x2));
  Ctor.precision = pr;
  Ctor.rounding = rm;
  return finalise(quadrant > 2 ? x2.neg() : x2, pr, rm, true);
};
P$1.squareRoot = P$1.sqrt = function() {
  var m2, n2, sd, r2, rep, t3, x2 = this, d2 = x2.d, e2 = x2.e, s2 = x2.s, Ctor = x2.constructor;
  if (s2 !== 1 || !d2 || !d2[0]) {
    return new Ctor(!s2 || s2 < 0 && (!d2 || d2[0]) ? NaN : d2 ? x2 : 1 / 0);
  }
  external = false;
  s2 = Math.sqrt(+x2);
  if (s2 == 0 || s2 == 1 / 0) {
    n2 = digitsToString(d2);
    if ((n2.length + e2) % 2 == 0)
      n2 += "0";
    s2 = Math.sqrt(n2);
    e2 = mathfloor((e2 + 1) / 2) - (e2 < 0 || e2 % 2);
    if (s2 == 1 / 0) {
      n2 = "5e" + e2;
    } else {
      n2 = s2.toExponential();
      n2 = n2.slice(0, n2.indexOf("e") + 1) + e2;
    }
    r2 = new Ctor(n2);
  } else {
    r2 = new Ctor(s2.toString());
  }
  sd = (e2 = Ctor.precision) + 3;
  for (; ; ) {
    t3 = r2;
    r2 = t3.plus(divide(x2, t3, sd + 2, 1)).times(0.5);
    if (digitsToString(t3.d).slice(0, sd) === (n2 = digitsToString(r2.d)).slice(0, sd)) {
      n2 = n2.slice(sd - 3, sd + 1);
      if (n2 == "9999" || !rep && n2 == "4999") {
        if (!rep) {
          finalise(t3, e2 + 1, 0);
          if (t3.times(t3).eq(x2)) {
            r2 = t3;
            break;
          }
        }
        sd += 4;
        rep = 1;
      } else {
        if (!+n2 || !+n2.slice(1) && n2.charAt(0) == "5") {
          finalise(r2, e2 + 1, 1);
          m2 = !r2.times(r2).eq(x2);
        }
        break;
      }
    }
  }
  external = true;
  return finalise(r2, e2, Ctor.rounding, m2);
};
P$1.tangent = P$1.tan = function() {
  var pr, rm, x2 = this, Ctor = x2.constructor;
  if (!x2.isFinite())
    return new Ctor(NaN);
  if (x2.isZero())
    return new Ctor(x2);
  pr = Ctor.precision;
  rm = Ctor.rounding;
  Ctor.precision = pr + 10;
  Ctor.rounding = 1;
  x2 = x2.sin();
  x2.s = 1;
  x2 = divide(x2, new Ctor(1).minus(x2.times(x2)).sqrt(), pr + 10, 0);
  Ctor.precision = pr;
  Ctor.rounding = rm;
  return finalise(quadrant == 2 || quadrant == 4 ? x2.neg() : x2, pr, rm, true);
};
P$1.times = P$1.mul = function(y2) {
  var carry2, e2, i2, k2, r2, rL, t3, xdL, ydL, x2 = this, Ctor = x2.constructor, xd = x2.d, yd = (y2 = new Ctor(y2)).d;
  y2.s *= x2.s;
  if (!xd || !xd[0] || !yd || !yd[0]) {
    return new Ctor(!y2.s || xd && !xd[0] && !yd || yd && !yd[0] && !xd ? NaN : !xd || !yd ? y2.s / 0 : y2.s * 0);
  }
  e2 = mathfloor(x2.e / LOG_BASE) + mathfloor(y2.e / LOG_BASE);
  xdL = xd.length;
  ydL = yd.length;
  if (xdL < ydL) {
    r2 = xd;
    xd = yd;
    yd = r2;
    rL = xdL;
    xdL = ydL;
    ydL = rL;
  }
  r2 = [];
  rL = xdL + ydL;
  for (i2 = rL; i2--; )
    r2.push(0);
  for (i2 = ydL; --i2 >= 0; ) {
    carry2 = 0;
    for (k2 = xdL + i2; k2 > i2; ) {
      t3 = r2[k2] + yd[i2] * xd[k2 - i2 - 1] + carry2;
      r2[k2--] = t3 % BASE | 0;
      carry2 = t3 / BASE | 0;
    }
    r2[k2] = (r2[k2] + carry2) % BASE | 0;
  }
  for (; !r2[--rL]; )
    r2.pop();
  if (carry2)
    ++e2;
  else
    r2.shift();
  y2.d = r2;
  y2.e = getBase10Exponent(r2, e2);
  return external ? finalise(y2, Ctor.precision, Ctor.rounding) : y2;
};
P$1.toBinary = function(sd, rm) {
  return toStringBinary(this, 2, sd, rm);
};
P$1.toDecimalPlaces = P$1.toDP = function(dp, rm) {
  var x2 = this, Ctor = x2.constructor;
  x2 = new Ctor(x2);
  if (dp === void 0)
    return x2;
  checkInt32(dp, 0, MAX_DIGITS);
  if (rm === void 0)
    rm = Ctor.rounding;
  else
    checkInt32(rm, 0, 8);
  return finalise(x2, dp + x2.e + 1, rm);
};
P$1.toExponential = function(dp, rm) {
  var str, x2 = this, Ctor = x2.constructor;
  if (dp === void 0) {
    str = finiteToString(x2, true);
  } else {
    checkInt32(dp, 0, MAX_DIGITS);
    if (rm === void 0)
      rm = Ctor.rounding;
    else
      checkInt32(rm, 0, 8);
    x2 = finalise(new Ctor(x2), dp + 1, rm);
    str = finiteToString(x2, true, dp + 1);
  }
  return x2.isNeg() && !x2.isZero() ? "-" + str : str;
};
P$1.toFixed = function(dp, rm) {
  var str, y2, x2 = this, Ctor = x2.constructor;
  if (dp === void 0) {
    str = finiteToString(x2);
  } else {
    checkInt32(dp, 0, MAX_DIGITS);
    if (rm === void 0)
      rm = Ctor.rounding;
    else
      checkInt32(rm, 0, 8);
    y2 = finalise(new Ctor(x2), dp + x2.e + 1, rm);
    str = finiteToString(y2, false, dp + y2.e + 1);
  }
  return x2.isNeg() && !x2.isZero() ? "-" + str : str;
};
P$1.toFraction = function(maxD) {
  var d2, d0, d1, d22, e2, k2, n2, n0, n1, pr, q2, r2, x2 = this, xd = x2.d, Ctor = x2.constructor;
  if (!xd)
    return new Ctor(x2);
  n1 = d0 = new Ctor(1);
  d1 = n0 = new Ctor(0);
  d2 = new Ctor(d1);
  e2 = d2.e = getPrecision$1(xd) - x2.e - 1;
  k2 = e2 % LOG_BASE;
  d2.d[0] = mathpow(10, k2 < 0 ? LOG_BASE + k2 : k2);
  if (maxD == null) {
    maxD = e2 > 0 ? d2 : n1;
  } else {
    n2 = new Ctor(maxD);
    if (!n2.isInt() || n2.lt(n1))
      throw Error(invalidArgument + n2);
    maxD = n2.gt(d2) ? e2 > 0 ? d2 : n1 : n2;
  }
  external = false;
  n2 = new Ctor(digitsToString(xd));
  pr = Ctor.precision;
  Ctor.precision = e2 = xd.length * LOG_BASE * 2;
  for (; ; ) {
    q2 = divide(n2, d2, 0, 1, 1);
    d22 = d0.plus(q2.times(d1));
    if (d22.cmp(maxD) == 1)
      break;
    d0 = d1;
    d1 = d22;
    d22 = n1;
    n1 = n0.plus(q2.times(d22));
    n0 = d22;
    d22 = d2;
    d2 = n2.minus(q2.times(d22));
    n2 = d22;
  }
  d22 = divide(maxD.minus(d0), d1, 0, 1, 1);
  n0 = n0.plus(d22.times(n1));
  d0 = d0.plus(d22.times(d1));
  n0.s = n1.s = x2.s;
  r2 = divide(n1, d1, e2, 1).minus(x2).abs().cmp(divide(n0, d0, e2, 1).minus(x2).abs()) < 1 ? [n1, d1] : [n0, d0];
  Ctor.precision = pr;
  external = true;
  return r2;
};
P$1.toHexadecimal = P$1.toHex = function(sd, rm) {
  return toStringBinary(this, 16, sd, rm);
};
P$1.toNearest = function(y2, rm) {
  var x2 = this, Ctor = x2.constructor;
  x2 = new Ctor(x2);
  if (y2 == null) {
    if (!x2.d)
      return x2;
    y2 = new Ctor(1);
    rm = Ctor.rounding;
  } else {
    y2 = new Ctor(y2);
    if (rm === void 0) {
      rm = Ctor.rounding;
    } else {
      checkInt32(rm, 0, 8);
    }
    if (!x2.d)
      return y2.s ? x2 : y2;
    if (!y2.d) {
      if (y2.s)
        y2.s = x2.s;
      return y2;
    }
  }
  if (y2.d[0]) {
    external = false;
    x2 = divide(x2, y2, 0, rm, 1).times(y2);
    external = true;
    finalise(x2);
  } else {
    y2.s = x2.s;
    x2 = y2;
  }
  return x2;
};
P$1.toNumber = function() {
  return +this;
};
P$1.toOctal = function(sd, rm) {
  return toStringBinary(this, 8, sd, rm);
};
P$1.toPower = P$1.pow = function(y2) {
  var e2, k2, pr, r2, rm, s2, x2 = this, Ctor = x2.constructor, yn = +(y2 = new Ctor(y2));
  if (!x2.d || !y2.d || !x2.d[0] || !y2.d[0])
    return new Ctor(mathpow(+x2, yn));
  x2 = new Ctor(x2);
  if (x2.eq(1))
    return x2;
  pr = Ctor.precision;
  rm = Ctor.rounding;
  if (y2.eq(1))
    return finalise(x2, pr, rm);
  e2 = mathfloor(y2.e / LOG_BASE);
  if (e2 >= y2.d.length - 1 && (k2 = yn < 0 ? -yn : yn) <= MAX_SAFE_INTEGER$1) {
    r2 = intPow(Ctor, x2, k2, pr);
    return y2.s < 0 ? new Ctor(1).div(r2) : finalise(r2, pr, rm);
  }
  s2 = x2.s;
  if (s2 < 0) {
    if (e2 < y2.d.length - 1)
      return new Ctor(NaN);
    if ((y2.d[e2] & 1) == 0)
      s2 = 1;
    if (x2.e == 0 && x2.d[0] == 1 && x2.d.length == 1) {
      x2.s = s2;
      return x2;
    }
  }
  k2 = mathpow(+x2, yn);
  e2 = k2 == 0 || !isFinite(k2) ? mathfloor(yn * (Math.log("0." + digitsToString(x2.d)) / Math.LN10 + x2.e + 1)) : new Ctor(k2 + "").e;
  if (e2 > Ctor.maxE + 1 || e2 < Ctor.minE - 1)
    return new Ctor(e2 > 0 ? s2 / 0 : 0);
  external = false;
  Ctor.rounding = x2.s = 1;
  k2 = Math.min(12, (e2 + "").length);
  r2 = naturalExponential(y2.times(naturalLogarithm(x2, pr + k2)), pr);
  if (r2.d) {
    r2 = finalise(r2, pr + 5, 1);
    if (checkRoundingDigits(r2.d, pr, rm)) {
      e2 = pr + 10;
      r2 = finalise(naturalExponential(y2.times(naturalLogarithm(x2, e2 + k2)), e2), e2 + 5, 1);
      if (+digitsToString(r2.d).slice(pr + 1, pr + 15) + 1 == 1e14) {
        r2 = finalise(r2, pr + 1, 0);
      }
    }
  }
  r2.s = s2;
  external = true;
  Ctor.rounding = rm;
  return finalise(r2, pr, rm);
};
P$1.toPrecision = function(sd, rm) {
  var str, x2 = this, Ctor = x2.constructor;
  if (sd === void 0) {
    str = finiteToString(x2, x2.e <= Ctor.toExpNeg || x2.e >= Ctor.toExpPos);
  } else {
    checkInt32(sd, 1, MAX_DIGITS);
    if (rm === void 0)
      rm = Ctor.rounding;
    else
      checkInt32(rm, 0, 8);
    x2 = finalise(new Ctor(x2), sd, rm);
    str = finiteToString(x2, sd <= x2.e || x2.e <= Ctor.toExpNeg, sd);
  }
  return x2.isNeg() && !x2.isZero() ? "-" + str : str;
};
P$1.toSignificantDigits = P$1.toSD = function(sd, rm) {
  var x2 = this, Ctor = x2.constructor;
  if (sd === void 0) {
    sd = Ctor.precision;
    rm = Ctor.rounding;
  } else {
    checkInt32(sd, 1, MAX_DIGITS);
    if (rm === void 0)
      rm = Ctor.rounding;
    else
      checkInt32(rm, 0, 8);
  }
  return finalise(new Ctor(x2), sd, rm);
};
P$1.toString = function() {
  var x2 = this, Ctor = x2.constructor, str = finiteToString(x2, x2.e <= Ctor.toExpNeg || x2.e >= Ctor.toExpPos);
  return x2.isNeg() && !x2.isZero() ? "-" + str : str;
};
P$1.truncated = P$1.trunc = function() {
  return finalise(new this.constructor(this), this.e + 1, 1);
};
P$1.valueOf = P$1.toJSON = function() {
  var x2 = this, Ctor = x2.constructor, str = finiteToString(x2, x2.e <= Ctor.toExpNeg || x2.e >= Ctor.toExpPos);
  return x2.isNeg() ? "-" + str : str;
};
function digitsToString(d2) {
  var i2, k2, ws, indexOfLastWord = d2.length - 1, str = "", w2 = d2[0];
  if (indexOfLastWord > 0) {
    str += w2;
    for (i2 = 1; i2 < indexOfLastWord; i2++) {
      ws = d2[i2] + "";
      k2 = LOG_BASE - ws.length;
      if (k2)
        str += getZeroString(k2);
      str += ws;
    }
    w2 = d2[i2];
    ws = w2 + "";
    k2 = LOG_BASE - ws.length;
    if (k2)
      str += getZeroString(k2);
  } else if (w2 === 0) {
    return "0";
  }
  for (; w2 % 10 === 0; )
    w2 /= 10;
  return str + w2;
}
function checkInt32(i2, min3, max3) {
  if (i2 !== ~~i2 || i2 < min3 || i2 > max3) {
    throw Error(invalidArgument + i2);
  }
}
function checkRoundingDigits(d2, i2, rm, repeating) {
  var di, k2, r2, rd;
  for (k2 = d2[0]; k2 >= 10; k2 /= 10)
    --i2;
  if (--i2 < 0) {
    i2 += LOG_BASE;
    di = 0;
  } else {
    di = Math.ceil((i2 + 1) / LOG_BASE);
    i2 %= LOG_BASE;
  }
  k2 = mathpow(10, LOG_BASE - i2);
  rd = d2[di] % k2 | 0;
  if (repeating == null) {
    if (i2 < 3) {
      if (i2 == 0)
        rd = rd / 100 | 0;
      else if (i2 == 1)
        rd = rd / 10 | 0;
      r2 = rm < 4 && rd == 99999 || rm > 3 && rd == 49999 || rd == 5e4 || rd == 0;
    } else {
      r2 = (rm < 4 && rd + 1 == k2 || rm > 3 && rd + 1 == k2 / 2) && (d2[di + 1] / k2 / 100 | 0) == mathpow(10, i2 - 2) - 1 || (rd == k2 / 2 || rd == 0) && (d2[di + 1] / k2 / 100 | 0) == 0;
    }
  } else {
    if (i2 < 4) {
      if (i2 == 0)
        rd = rd / 1e3 | 0;
      else if (i2 == 1)
        rd = rd / 100 | 0;
      else if (i2 == 2)
        rd = rd / 10 | 0;
      r2 = (repeating || rm < 4) && rd == 9999 || !repeating && rm > 3 && rd == 4999;
    } else {
      r2 = ((repeating || rm < 4) && rd + 1 == k2 || !repeating && rm > 3 && rd + 1 == k2 / 2) && (d2[di + 1] / k2 / 1e3 | 0) == mathpow(10, i2 - 3) - 1;
    }
  }
  return r2;
}
function convertBase(str, baseIn, baseOut) {
  var j2, arr = [0], arrL, i2 = 0, strL = str.length;
  for (; i2 < strL; ) {
    for (arrL = arr.length; arrL--; )
      arr[arrL] *= baseIn;
    arr[0] += NUMERALS.indexOf(str.charAt(i2++));
    for (j2 = 0; j2 < arr.length; j2++) {
      if (arr[j2] > baseOut - 1) {
        if (arr[j2 + 1] === void 0)
          arr[j2 + 1] = 0;
        arr[j2 + 1] += arr[j2] / baseOut | 0;
        arr[j2] %= baseOut;
      }
    }
  }
  return arr.reverse();
}
function cosine(Ctor, x2) {
  var k2, len2, y2;
  if (x2.isZero())
    return x2;
  len2 = x2.d.length;
  if (len2 < 32) {
    k2 = Math.ceil(len2 / 3);
    y2 = (1 / tinyPow(4, k2)).toString();
  } else {
    k2 = 16;
    y2 = "2.3283064365386962890625e-10";
  }
  Ctor.precision += k2;
  x2 = taylorSeries(Ctor, 1, x2.times(y2), new Ctor(1));
  for (var i2 = k2; i2--; ) {
    var cos2x = x2.times(x2);
    x2 = cos2x.times(cos2x).minus(cos2x).times(8).plus(1);
  }
  Ctor.precision -= k2;
  return x2;
}
var divide = function() {
  function multiplyInteger(x2, k2, base2) {
    var temp, carry2 = 0, i2 = x2.length;
    for (x2 = x2.slice(); i2--; ) {
      temp = x2[i2] * k2 + carry2;
      x2[i2] = temp % base2 | 0;
      carry2 = temp / base2 | 0;
    }
    if (carry2)
      x2.unshift(carry2);
    return x2;
  }
  function compare2(a2, b2, aL, bL) {
    var i2, r2;
    if (aL != bL) {
      r2 = aL > bL ? 1 : -1;
    } else {
      for (i2 = r2 = 0; i2 < aL; i2++) {
        if (a2[i2] != b2[i2]) {
          r2 = a2[i2] > b2[i2] ? 1 : -1;
          break;
        }
      }
    }
    return r2;
  }
  function subtract2(a2, b2, aL, base2) {
    var i2 = 0;
    for (; aL--; ) {
      a2[aL] -= i2;
      i2 = a2[aL] < b2[aL] ? 1 : 0;
      a2[aL] = i2 * base2 + a2[aL] - b2[aL];
    }
    for (; !a2[0] && a2.length > 1; )
      a2.shift();
  }
  return function(x2, y2, pr, rm, dp, base2) {
    var cmp, e2, i2, k2, logBase, more, prod, prodL, q2, qd, rem, remL, rem0, sd, t3, xi, xL, yd0, yL, yz, Ctor = x2.constructor, sign2 = x2.s == y2.s ? 1 : -1, xd = x2.d, yd = y2.d;
    if (!xd || !xd[0] || !yd || !yd[0]) {
      return new Ctor(
        // Return NaN if either NaN, or both Infinity or 0.
        !x2.s || !y2.s || (xd ? yd && xd[0] == yd[0] : !yd) ? NaN : (
          // Return ±0 if x is 0 or y is ±Infinity, or return ±Infinity as y is 0.
          xd && xd[0] == 0 || !yd ? sign2 * 0 : sign2 / 0
        )
      );
    }
    if (base2) {
      logBase = 1;
      e2 = x2.e - y2.e;
    } else {
      base2 = BASE;
      logBase = LOG_BASE;
      e2 = mathfloor(x2.e / logBase) - mathfloor(y2.e / logBase);
    }
    yL = yd.length;
    xL = xd.length;
    q2 = new Ctor(sign2);
    qd = q2.d = [];
    for (i2 = 0; yd[i2] == (xd[i2] || 0); i2++)
      ;
    if (yd[i2] > (xd[i2] || 0))
      e2--;
    if (pr == null) {
      sd = pr = Ctor.precision;
      rm = Ctor.rounding;
    } else if (dp) {
      sd = pr + (x2.e - y2.e) + 1;
    } else {
      sd = pr;
    }
    if (sd < 0) {
      qd.push(1);
      more = true;
    } else {
      sd = sd / logBase + 2 | 0;
      i2 = 0;
      if (yL == 1) {
        k2 = 0;
        yd = yd[0];
        sd++;
        for (; (i2 < xL || k2) && sd--; i2++) {
          t3 = k2 * base2 + (xd[i2] || 0);
          qd[i2] = t3 / yd | 0;
          k2 = t3 % yd | 0;
        }
        more = k2 || i2 < xL;
      } else {
        k2 = base2 / (yd[0] + 1) | 0;
        if (k2 > 1) {
          yd = multiplyInteger(yd, k2, base2);
          xd = multiplyInteger(xd, k2, base2);
          yL = yd.length;
          xL = xd.length;
        }
        xi = yL;
        rem = xd.slice(0, yL);
        remL = rem.length;
        for (; remL < yL; )
          rem[remL++] = 0;
        yz = yd.slice();
        yz.unshift(0);
        yd0 = yd[0];
        if (yd[1] >= base2 / 2)
          ++yd0;
        do {
          k2 = 0;
          cmp = compare2(yd, rem, yL, remL);
          if (cmp < 0) {
            rem0 = rem[0];
            if (yL != remL)
              rem0 = rem0 * base2 + (rem[1] || 0);
            k2 = rem0 / yd0 | 0;
            if (k2 > 1) {
              if (k2 >= base2)
                k2 = base2 - 1;
              prod = multiplyInteger(yd, k2, base2);
              prodL = prod.length;
              remL = rem.length;
              cmp = compare2(prod, rem, prodL, remL);
              if (cmp == 1) {
                k2--;
                subtract2(prod, yL < prodL ? yz : yd, prodL, base2);
              }
            } else {
              if (k2 == 0)
                cmp = k2 = 1;
              prod = yd.slice();
            }
            prodL = prod.length;
            if (prodL < remL)
              prod.unshift(0);
            subtract2(rem, prod, remL, base2);
            if (cmp == -1) {
              remL = rem.length;
              cmp = compare2(yd, rem, yL, remL);
              if (cmp < 1) {
                k2++;
                subtract2(rem, yL < remL ? yz : yd, remL, base2);
              }
            }
            remL = rem.length;
          } else if (cmp === 0) {
            k2++;
            rem = [0];
          }
          qd[i2++] = k2;
          if (cmp && rem[0]) {
            rem[remL++] = xd[xi] || 0;
          } else {
            rem = [xd[xi]];
            remL = 1;
          }
        } while ((xi++ < xL || rem[0] !== void 0) && sd--);
        more = rem[0] !== void 0;
      }
      if (!qd[0])
        qd.shift();
    }
    if (logBase == 1) {
      q2.e = e2;
      inexact = more;
    } else {
      for (i2 = 1, k2 = qd[0]; k2 >= 10; k2 /= 10)
        i2++;
      q2.e = i2 + e2 * logBase - 1;
      finalise(q2, dp ? pr + q2.e + 1 : pr, rm, more);
    }
    return q2;
  };
}();
function finalise(x2, sd, rm, isTruncated) {
  var digits, i2, j2, k2, rd, roundUp, w2, xd, xdi, Ctor = x2.constructor;
  out:
    if (sd != null) {
      xd = x2.d;
      if (!xd)
        return x2;
      for (digits = 1, k2 = xd[0]; k2 >= 10; k2 /= 10)
        digits++;
      i2 = sd - digits;
      if (i2 < 0) {
        i2 += LOG_BASE;
        j2 = sd;
        w2 = xd[xdi = 0];
        rd = w2 / mathpow(10, digits - j2 - 1) % 10 | 0;
      } else {
        xdi = Math.ceil((i2 + 1) / LOG_BASE);
        k2 = xd.length;
        if (xdi >= k2) {
          if (isTruncated) {
            for (; k2++ <= xdi; )
              xd.push(0);
            w2 = rd = 0;
            digits = 1;
            i2 %= LOG_BASE;
            j2 = i2 - LOG_BASE + 1;
          } else {
            break out;
          }
        } else {
          w2 = k2 = xd[xdi];
          for (digits = 1; k2 >= 10; k2 /= 10)
            digits++;
          i2 %= LOG_BASE;
          j2 = i2 - LOG_BASE + digits;
          rd = j2 < 0 ? 0 : w2 / mathpow(10, digits - j2 - 1) % 10 | 0;
        }
      }
      isTruncated = isTruncated || sd < 0 || xd[xdi + 1] !== void 0 || (j2 < 0 ? w2 : w2 % mathpow(10, digits - j2 - 1));
      roundUp = rm < 4 ? (rd || isTruncated) && (rm == 0 || rm == (x2.s < 0 ? 3 : 2)) : rd > 5 || rd == 5 && (rm == 4 || isTruncated || rm == 6 && // Check whether the digit to the left of the rounding digit is odd.
      (i2 > 0 ? j2 > 0 ? w2 / mathpow(10, digits - j2) : 0 : xd[xdi - 1]) % 10 & 1 || rm == (x2.s < 0 ? 8 : 7));
      if (sd < 1 || !xd[0]) {
        xd.length = 0;
        if (roundUp) {
          sd -= x2.e + 1;
          xd[0] = mathpow(10, (LOG_BASE - sd % LOG_BASE) % LOG_BASE);
          x2.e = -sd || 0;
        } else {
          xd[0] = x2.e = 0;
        }
        return x2;
      }
      if (i2 == 0) {
        xd.length = xdi;
        k2 = 1;
        xdi--;
      } else {
        xd.length = xdi + 1;
        k2 = mathpow(10, LOG_BASE - i2);
        xd[xdi] = j2 > 0 ? (w2 / mathpow(10, digits - j2) % mathpow(10, j2) | 0) * k2 : 0;
      }
      if (roundUp) {
        for (; ; ) {
          if (xdi == 0) {
            for (i2 = 1, j2 = xd[0]; j2 >= 10; j2 /= 10)
              i2++;
            j2 = xd[0] += k2;
            for (k2 = 1; j2 >= 10; j2 /= 10)
              k2++;
            if (i2 != k2) {
              x2.e++;
              if (xd[0] == BASE)
                xd[0] = 1;
            }
            break;
          } else {
            xd[xdi] += k2;
            if (xd[xdi] != BASE)
              break;
            xd[xdi--] = 0;
            k2 = 1;
          }
        }
      }
      for (i2 = xd.length; xd[--i2] === 0; )
        xd.pop();
    }
  if (external) {
    if (x2.e > Ctor.maxE) {
      x2.d = null;
      x2.e = NaN;
    } else if (x2.e < Ctor.minE) {
      x2.e = 0;
      x2.d = [0];
    }
  }
  return x2;
}
function finiteToString(x2, isExp, sd) {
  if (!x2.isFinite())
    return nonFiniteToString(x2);
  var k2, e2 = x2.e, str = digitsToString(x2.d), len2 = str.length;
  if (isExp) {
    if (sd && (k2 = sd - len2) > 0) {
      str = str.charAt(0) + "." + str.slice(1) + getZeroString(k2);
    } else if (len2 > 1) {
      str = str.charAt(0) + "." + str.slice(1);
    }
    str = str + (x2.e < 0 ? "e" : "e+") + x2.e;
  } else if (e2 < 0) {
    str = "0." + getZeroString(-e2 - 1) + str;
    if (sd && (k2 = sd - len2) > 0)
      str += getZeroString(k2);
  } else if (e2 >= len2) {
    str += getZeroString(e2 + 1 - len2);
    if (sd && (k2 = sd - e2 - 1) > 0)
      str = str + "." + getZeroString(k2);
  } else {
    if ((k2 = e2 + 1) < len2)
      str = str.slice(0, k2) + "." + str.slice(k2);
    if (sd && (k2 = sd - len2) > 0) {
      if (e2 + 1 === len2)
        str += ".";
      str += getZeroString(k2);
    }
  }
  return str;
}
function getBase10Exponent(digits, e2) {
  var w2 = digits[0];
  for (e2 *= LOG_BASE; w2 >= 10; w2 /= 10)
    e2++;
  return e2;
}
function getLn10(Ctor, sd, pr) {
  if (sd > LN10_PRECISION) {
    external = true;
    if (pr)
      Ctor.precision = pr;
    throw Error(precisionLimitExceeded);
  }
  return finalise(new Ctor(LN10), sd, 1, true);
}
function getPi(Ctor, sd, rm) {
  if (sd > PI_PRECISION)
    throw Error(precisionLimitExceeded);
  return finalise(new Ctor(PI$9), sd, rm, true);
}
function getPrecision$1(digits) {
  var w2 = digits.length - 1, len2 = w2 * LOG_BASE + 1;
  w2 = digits[w2];
  if (w2) {
    for (; w2 % 10 == 0; w2 /= 10)
      len2--;
    for (w2 = digits[0]; w2 >= 10; w2 /= 10)
      len2++;
  }
  return len2;
}
function getZeroString(k2) {
  var zs = "";
  for (; k2--; )
    zs += "0";
  return zs;
}
function intPow(Ctor, x2, n2, pr) {
  var isTruncated, r2 = new Ctor(1), k2 = Math.ceil(pr / LOG_BASE + 4);
  external = false;
  for (; ; ) {
    if (n2 % 2) {
      r2 = r2.times(x2);
      if (truncate(r2.d, k2))
        isTruncated = true;
    }
    n2 = mathfloor(n2 / 2);
    if (n2 === 0) {
      n2 = r2.d.length - 1;
      if (isTruncated && r2.d[n2] === 0)
        ++r2.d[n2];
      break;
    }
    x2 = x2.times(x2);
    truncate(x2.d, k2);
  }
  external = true;
  return r2;
}
function isOdd(n2) {
  return n2.d[n2.d.length - 1] & 1;
}
function maxOrMin(Ctor, args, ltgt) {
  var y2, x2 = new Ctor(args[0]), i2 = 0;
  for (; ++i2 < args.length; ) {
    y2 = new Ctor(args[i2]);
    if (!y2.s) {
      x2 = y2;
      break;
    } else if (x2[ltgt](y2)) {
      x2 = y2;
    }
  }
  return x2;
}
function naturalExponential(x2, sd) {
  var denominator, guard, j2, pow2, sum2, t3, wpr, rep = 0, i2 = 0, k2 = 0, Ctor = x2.constructor, rm = Ctor.rounding, pr = Ctor.precision;
  if (!x2.d || !x2.d[0] || x2.e > 17) {
    return new Ctor(x2.d ? !x2.d[0] ? 1 : x2.s < 0 ? 0 : 1 / 0 : x2.s ? x2.s < 0 ? 0 : x2 : 0 / 0);
  }
  if (sd == null) {
    external = false;
    wpr = pr;
  } else {
    wpr = sd;
  }
  t3 = new Ctor(0.03125);
  while (x2.e > -2) {
    x2 = x2.times(t3);
    k2 += 5;
  }
  guard = Math.log(mathpow(2, k2)) / Math.LN10 * 2 + 5 | 0;
  wpr += guard;
  denominator = pow2 = sum2 = new Ctor(1);
  Ctor.precision = wpr;
  for (; ; ) {
    pow2 = finalise(pow2.times(x2), wpr, 1);
    denominator = denominator.times(++i2);
    t3 = sum2.plus(divide(pow2, denominator, wpr, 1));
    if (digitsToString(t3.d).slice(0, wpr) === digitsToString(sum2.d).slice(0, wpr)) {
      j2 = k2;
      while (j2--)
        sum2 = finalise(sum2.times(sum2), wpr, 1);
      if (sd == null) {
        if (rep < 3 && checkRoundingDigits(sum2.d, wpr - guard, rm, rep)) {
          Ctor.precision = wpr += 10;
          denominator = pow2 = t3 = new Ctor(1);
          i2 = 0;
          rep++;
        } else {
          return finalise(sum2, Ctor.precision = pr, rm, external = true);
        }
      } else {
        Ctor.precision = pr;
        return sum2;
      }
    }
    sum2 = t3;
  }
}
function naturalLogarithm(y2, sd) {
  var c2, c0, denominator, e2, numerator, rep, sum2, t3, wpr, x1, x2, n2 = 1, guard = 10, x3 = y2, xd = x3.d, Ctor = x3.constructor, rm = Ctor.rounding, pr = Ctor.precision;
  if (x3.s < 0 || !xd || !xd[0] || !x3.e && xd[0] == 1 && xd.length == 1) {
    return new Ctor(xd && !xd[0] ? -1 / 0 : x3.s != 1 ? NaN : xd ? 0 : x3);
  }
  if (sd == null) {
    external = false;
    wpr = pr;
  } else {
    wpr = sd;
  }
  Ctor.precision = wpr += guard;
  c2 = digitsToString(xd);
  c0 = c2.charAt(0);
  if (Math.abs(e2 = x3.e) < 15e14) {
    while (c0 < 7 && c0 != 1 || c0 == 1 && c2.charAt(1) > 3) {
      x3 = x3.times(y2);
      c2 = digitsToString(x3.d);
      c0 = c2.charAt(0);
      n2++;
    }
    e2 = x3.e;
    if (c0 > 1) {
      x3 = new Ctor("0." + c2);
      e2++;
    } else {
      x3 = new Ctor(c0 + "." + c2.slice(1));
    }
  } else {
    t3 = getLn10(Ctor, wpr + 2, pr).times(e2 + "");
    x3 = naturalLogarithm(new Ctor(c0 + "." + c2.slice(1)), wpr - guard).plus(t3);
    Ctor.precision = pr;
    return sd == null ? finalise(x3, pr, rm, external = true) : x3;
  }
  x1 = x3;
  sum2 = numerator = x3 = divide(x3.minus(1), x3.plus(1), wpr, 1);
  x2 = finalise(x3.times(x3), wpr, 1);
  denominator = 3;
  for (; ; ) {
    numerator = finalise(numerator.times(x2), wpr, 1);
    t3 = sum2.plus(divide(numerator, new Ctor(denominator), wpr, 1));
    if (digitsToString(t3.d).slice(0, wpr) === digitsToString(sum2.d).slice(0, wpr)) {
      sum2 = sum2.times(2);
      if (e2 !== 0)
        sum2 = sum2.plus(getLn10(Ctor, wpr + 2, pr).times(e2 + ""));
      sum2 = divide(sum2, new Ctor(n2), wpr, 1);
      if (sd == null) {
        if (checkRoundingDigits(sum2.d, wpr - guard, rm, rep)) {
          Ctor.precision = wpr += guard;
          t3 = numerator = x3 = divide(x1.minus(1), x1.plus(1), wpr, 1);
          x2 = finalise(x3.times(x3), wpr, 1);
          denominator = rep = 1;
        } else {
          return finalise(sum2, Ctor.precision = pr, rm, external = true);
        }
      } else {
        Ctor.precision = pr;
        return sum2;
      }
    }
    sum2 = t3;
    denominator += 2;
  }
}
function nonFiniteToString(x2) {
  return String(x2.s * x2.s / 0);
}
function parseDecimal(x2, str) {
  var e2, i2, len2;
  if ((e2 = str.indexOf(".")) > -1)
    str = str.replace(".", "");
  if ((i2 = str.search(/e/i)) > 0) {
    if (e2 < 0)
      e2 = i2;
    e2 += +str.slice(i2 + 1);
    str = str.substring(0, i2);
  } else if (e2 < 0) {
    e2 = str.length;
  }
  for (i2 = 0; str.charCodeAt(i2) === 48; i2++)
    ;
  for (len2 = str.length; str.charCodeAt(len2 - 1) === 48; --len2)
    ;
  str = str.slice(i2, len2);
  if (str) {
    len2 -= i2;
    x2.e = e2 = e2 - i2 - 1;
    x2.d = [];
    i2 = (e2 + 1) % LOG_BASE;
    if (e2 < 0)
      i2 += LOG_BASE;
    if (i2 < len2) {
      if (i2)
        x2.d.push(+str.slice(0, i2));
      for (len2 -= LOG_BASE; i2 < len2; )
        x2.d.push(+str.slice(i2, i2 += LOG_BASE));
      str = str.slice(i2);
      i2 = LOG_BASE - str.length;
    } else {
      i2 -= len2;
    }
    for (; i2--; )
      str += "0";
    x2.d.push(+str);
    if (external) {
      if (x2.e > x2.constructor.maxE) {
        x2.d = null;
        x2.e = NaN;
      } else if (x2.e < x2.constructor.minE) {
        x2.e = 0;
        x2.d = [0];
      }
    }
  } else {
    x2.e = 0;
    x2.d = [0];
  }
  return x2;
}
function parseOther(x2, str) {
  var base2, Ctor, divisor, i2, isFloat, len2, p2, xd, xe;
  if (str.indexOf("_") > -1) {
    str = str.replace(/(\d)_(?=\d)/g, "$1");
    if (isDecimal.test(str))
      return parseDecimal(x2, str);
  } else if (str === "Infinity" || str === "NaN") {
    if (!+str)
      x2.s = NaN;
    x2.e = NaN;
    x2.d = null;
    return x2;
  }
  if (isHex$1.test(str)) {
    base2 = 16;
    str = str.toLowerCase();
  } else if (isBinary.test(str)) {
    base2 = 2;
  } else if (isOctal.test(str)) {
    base2 = 8;
  } else {
    throw Error(invalidArgument + str);
  }
  i2 = str.search(/p/i);
  if (i2 > 0) {
    p2 = +str.slice(i2 + 1);
    str = str.substring(2, i2);
  } else {
    str = str.slice(2);
  }
  i2 = str.indexOf(".");
  isFloat = i2 >= 0;
  Ctor = x2.constructor;
  if (isFloat) {
    str = str.replace(".", "");
    len2 = str.length;
    i2 = len2 - i2;
    divisor = intPow(Ctor, new Ctor(base2), i2, i2 * 2);
  }
  xd = convertBase(str, base2, BASE);
  xe = xd.length - 1;
  for (i2 = xe; xd[i2] === 0; --i2)
    xd.pop();
  if (i2 < 0)
    return new Ctor(x2.s * 0);
  x2.e = getBase10Exponent(xd, xe);
  x2.d = xd;
  external = false;
  if (isFloat)
    x2 = divide(x2, divisor, len2 * 4);
  if (p2)
    x2 = x2.times(Math.abs(p2) < 54 ? mathpow(2, p2) : Decimal.pow(2, p2));
  external = true;
  return x2;
}
function sine(Ctor, x2) {
  var k2, len2 = x2.d.length;
  if (len2 < 3) {
    return x2.isZero() ? x2 : taylorSeries(Ctor, 2, x2, x2);
  }
  k2 = 1.4 * Math.sqrt(len2);
  k2 = k2 > 16 ? 16 : k2 | 0;
  x2 = x2.times(1 / tinyPow(5, k2));
  x2 = taylorSeries(Ctor, 2, x2, x2);
  var sin2_x, d5 = new Ctor(5), d16 = new Ctor(16), d20 = new Ctor(20);
  for (; k2--; ) {
    sin2_x = x2.times(x2);
    x2 = x2.times(d5.plus(sin2_x.times(d16.times(sin2_x).minus(d20))));
  }
  return x2;
}
function taylorSeries(Ctor, n2, x2, y2, isHyperbolic) {
  var j2, t3, u2, x22, pr = Ctor.precision, k2 = Math.ceil(pr / LOG_BASE);
  external = false;
  x22 = x2.times(x2);
  u2 = new Ctor(y2);
  for (; ; ) {
    t3 = divide(u2.times(x22), new Ctor(n2++ * n2++), pr, 1);
    u2 = isHyperbolic ? y2.plus(t3) : y2.minus(t3);
    y2 = divide(t3.times(x22), new Ctor(n2++ * n2++), pr, 1);
    t3 = u2.plus(y2);
    if (t3.d[k2] !== void 0) {
      for (j2 = k2; t3.d[j2] === u2.d[j2] && j2--; )
        ;
      if (j2 == -1)
        break;
    }
    j2 = u2;
    u2 = y2;
    y2 = t3;
    t3 = j2;
  }
  external = true;
  t3.d.length = k2 + 1;
  return t3;
}
function tinyPow(b2, e2) {
  var n2 = b2;
  while (--e2)
    n2 *= b2;
  return n2;
}
function toLessThanHalfPi(Ctor, x2) {
  var t3, isNeg = x2.s < 0, pi = getPi(Ctor, Ctor.precision, 1), halfPi = pi.times(0.5);
  x2 = x2.abs();
  if (x2.lte(halfPi)) {
    quadrant = isNeg ? 4 : 1;
    return x2;
  }
  t3 = x2.divToInt(pi);
  if (t3.isZero()) {
    quadrant = isNeg ? 3 : 2;
  } else {
    x2 = x2.minus(t3.times(pi));
    if (x2.lte(halfPi)) {
      quadrant = isOdd(t3) ? isNeg ? 2 : 3 : isNeg ? 4 : 1;
      return x2;
    }
    quadrant = isOdd(t3) ? isNeg ? 1 : 4 : isNeg ? 3 : 2;
  }
  return x2.minus(pi).abs();
}
function toStringBinary(x2, baseOut, sd, rm) {
  var base2, e2, i2, k2, len2, roundUp, str, xd, y2, Ctor = x2.constructor, isExp = sd !== void 0;
  if (isExp) {
    checkInt32(sd, 1, MAX_DIGITS);
    if (rm === void 0)
      rm = Ctor.rounding;
    else
      checkInt32(rm, 0, 8);
  } else {
    sd = Ctor.precision;
    rm = Ctor.rounding;
  }
  if (!x2.isFinite()) {
    str = nonFiniteToString(x2);
  } else {
    str = finiteToString(x2);
    i2 = str.indexOf(".");
    if (isExp) {
      base2 = 2;
      if (baseOut == 16) {
        sd = sd * 4 - 3;
      } else if (baseOut == 8) {
        sd = sd * 3 - 2;
      }
    } else {
      base2 = baseOut;
    }
    if (i2 >= 0) {
      str = str.replace(".", "");
      y2 = new Ctor(1);
      y2.e = str.length - i2;
      y2.d = convertBase(finiteToString(y2), 10, base2);
      y2.e = y2.d.length;
    }
    xd = convertBase(str, 10, base2);
    e2 = len2 = xd.length;
    for (; xd[--len2] == 0; )
      xd.pop();
    if (!xd[0]) {
      str = isExp ? "0p+0" : "0";
    } else {
      if (i2 < 0) {
        e2--;
      } else {
        x2 = new Ctor(x2);
        x2.d = xd;
        x2.e = e2;
        x2 = divide(x2, y2, sd, rm, 0, base2);
        xd = x2.d;
        e2 = x2.e;
        roundUp = inexact;
      }
      i2 = xd[sd];
      k2 = base2 / 2;
      roundUp = roundUp || xd[sd + 1] !== void 0;
      roundUp = rm < 4 ? (i2 !== void 0 || roundUp) && (rm === 0 || rm === (x2.s < 0 ? 3 : 2)) : i2 > k2 || i2 === k2 && (rm === 4 || roundUp || rm === 6 && xd[sd - 1] & 1 || rm === (x2.s < 0 ? 8 : 7));
      xd.length = sd;
      if (roundUp) {
        for (; ++xd[--sd] > base2 - 1; ) {
          xd[sd] = 0;
          if (!sd) {
            ++e2;
            xd.unshift(1);
          }
        }
      }
      for (len2 = xd.length; !xd[len2 - 1]; --len2)
        ;
      for (i2 = 0, str = ""; i2 < len2; i2++)
        str += NUMERALS.charAt(xd[i2]);
      if (isExp) {
        if (len2 > 1) {
          if (baseOut == 16 || baseOut == 8) {
            i2 = baseOut == 16 ? 4 : 3;
            for (--len2; len2 % i2; len2++)
              str += "0";
            xd = convertBase(str, base2, baseOut);
            for (len2 = xd.length; !xd[len2 - 1]; --len2)
              ;
            for (i2 = 1, str = "1."; i2 < len2; i2++)
              str += NUMERALS.charAt(xd[i2]);
          } else {
            str = str.charAt(0) + "." + str.slice(1);
          }
        }
        str = str + (e2 < 0 ? "p" : "p+") + e2;
      } else if (e2 < 0) {
        for (; ++e2; )
          str = "0" + str;
        str = "0." + str;
      } else {
        if (++e2 > len2)
          for (e2 -= len2; e2--; )
            str += "0";
        else if (e2 < len2)
          str = str.slice(0, e2) + "." + str.slice(e2);
      }
    }
    str = (baseOut == 16 ? "0x" : baseOut == 2 ? "0b" : baseOut == 8 ? "0o" : "") + str;
  }
  return x2.s < 0 ? "-" + str : str;
}
function truncate(arr, len2) {
  if (arr.length > len2) {
    arr.length = len2;
    return true;
  }
}
function abs$4(x2) {
  return new this(x2).abs();
}
function acos(x2) {
  return new this(x2).acos();
}
function acosh(x2) {
  return new this(x2).acosh();
}
function add$4(x2, y2) {
  return new this(x2).plus(y2);
}
function asin(x2) {
  return new this(x2).asin();
}
function asinh(x2) {
  return new this(x2).asinh();
}
function atan(x2) {
  return new this(x2).atan();
}
function atanh(x2) {
  return new this(x2).atanh();
}
function atan2(y2, x2) {
  y2 = new this(y2);
  x2 = new this(x2);
  var r2, pr = this.precision, rm = this.rounding, wpr = pr + 4;
  if (!y2.s || !x2.s) {
    r2 = new this(NaN);
  } else if (!y2.d && !x2.d) {
    r2 = getPi(this, wpr, 1).times(x2.s > 0 ? 0.25 : 0.75);
    r2.s = y2.s;
  } else if (!x2.d || y2.isZero()) {
    r2 = x2.s < 0 ? getPi(this, pr, rm) : new this(0);
    r2.s = y2.s;
  } else if (!y2.d || x2.isZero()) {
    r2 = getPi(this, wpr, 1).times(0.5);
    r2.s = y2.s;
  } else if (x2.s < 0) {
    this.precision = wpr;
    this.rounding = 1;
    r2 = this.atan(divide(y2, x2, wpr, 1));
    x2 = getPi(this, wpr, 1);
    this.precision = pr;
    this.rounding = rm;
    r2 = y2.s < 0 ? r2.minus(x2) : r2.plus(x2);
  } else {
    r2 = this.atan(divide(y2, x2, wpr, 1));
  }
  return r2;
}
function cbrt(x2) {
  return new this(x2).cbrt();
}
function ceil$1(x2) {
  return finalise(x2 = new this(x2), x2.e + 1, 2);
}
function clamp$1(x2, min3, max3) {
  return new this(x2).clamp(min3, max3);
}
function config(obj) {
  if (!obj || typeof obj !== "object")
    throw Error(decimalError + "Object expected");
  var i2, p2, v4, useDefaults = obj.defaults === true, ps = [
    "precision",
    1,
    MAX_DIGITS,
    "rounding",
    0,
    8,
    "toExpNeg",
    -EXP_LIMIT,
    0,
    "toExpPos",
    0,
    EXP_LIMIT,
    "maxE",
    0,
    EXP_LIMIT,
    "minE",
    -EXP_LIMIT,
    0,
    "modulo",
    0,
    9
  ];
  for (i2 = 0; i2 < ps.length; i2 += 3) {
    if (p2 = ps[i2], useDefaults)
      this[p2] = DEFAULTS$1[p2];
    if ((v4 = obj[p2]) !== void 0) {
      if (mathfloor(v4) === v4 && v4 >= ps[i2 + 1] && v4 <= ps[i2 + 2])
        this[p2] = v4;
      else
        throw Error(invalidArgument + p2 + ": " + v4);
    }
  }
  if (p2 = "crypto", useDefaults)
    this[p2] = DEFAULTS$1[p2];
  if ((v4 = obj[p2]) !== void 0) {
    if (v4 === true || v4 === false || v4 === 0 || v4 === 1) {
      if (v4) {
        if (typeof crypto != "undefined" && crypto && (crypto.getRandomValues || crypto.randomBytes)) {
          this[p2] = true;
        } else {
          throw Error(cryptoUnavailable);
        }
      } else {
        this[p2] = false;
      }
    } else {
      throw Error(invalidArgument + p2 + ": " + v4);
    }
  }
  return this;
}
function cos(x2) {
  return new this(x2).cos();
}
function cosh(x2) {
  return new this(x2).cosh();
}
function clone$8(obj) {
  var i2, p2, ps;
  function Decimal2(v4) {
    var e2, i3, t3, x2 = this;
    if (!(x2 instanceof Decimal2))
      return new Decimal2(v4);
    x2.constructor = Decimal2;
    if (isDecimalInstance(v4)) {
      x2.s = v4.s;
      if (external) {
        if (!v4.d || v4.e > Decimal2.maxE) {
          x2.e = NaN;
          x2.d = null;
        } else if (v4.e < Decimal2.minE) {
          x2.e = 0;
          x2.d = [0];
        } else {
          x2.e = v4.e;
          x2.d = v4.d.slice();
        }
      } else {
        x2.e = v4.e;
        x2.d = v4.d ? v4.d.slice() : v4.d;
      }
      return;
    }
    t3 = typeof v4;
    if (t3 === "number") {
      if (v4 === 0) {
        x2.s = 1 / v4 < 0 ? -1 : 1;
        x2.e = 0;
        x2.d = [0];
        return;
      }
      if (v4 < 0) {
        v4 = -v4;
        x2.s = -1;
      } else {
        x2.s = 1;
      }
      if (v4 === ~~v4 && v4 < 1e7) {
        for (e2 = 0, i3 = v4; i3 >= 10; i3 /= 10)
          e2++;
        if (external) {
          if (e2 > Decimal2.maxE) {
            x2.e = NaN;
            x2.d = null;
          } else if (e2 < Decimal2.minE) {
            x2.e = 0;
            x2.d = [0];
          } else {
            x2.e = e2;
            x2.d = [v4];
          }
        } else {
          x2.e = e2;
          x2.d = [v4];
        }
        return;
      } else if (v4 * 0 !== 0) {
        if (!v4)
          x2.s = NaN;
        x2.e = NaN;
        x2.d = null;
        return;
      }
      return parseDecimal(x2, v4.toString());
    } else if (t3 !== "string") {
      throw Error(invalidArgument + v4);
    }
    if ((i3 = v4.charCodeAt(0)) === 45) {
      v4 = v4.slice(1);
      x2.s = -1;
    } else {
      if (i3 === 43)
        v4 = v4.slice(1);
      x2.s = 1;
    }
    return isDecimal.test(v4) ? parseDecimal(x2, v4) : parseOther(x2, v4);
  }
  Decimal2.prototype = P$1;
  Decimal2.ROUND_UP = 0;
  Decimal2.ROUND_DOWN = 1;
  Decimal2.ROUND_CEIL = 2;
  Decimal2.ROUND_FLOOR = 3;
  Decimal2.ROUND_HALF_UP = 4;
  Decimal2.ROUND_HALF_DOWN = 5;
  Decimal2.ROUND_HALF_EVEN = 6;
  Decimal2.ROUND_HALF_CEIL = 7;
  Decimal2.ROUND_HALF_FLOOR = 8;
  Decimal2.EUCLID = 9;
  Decimal2.config = Decimal2.set = config;
  Decimal2.clone = clone$8;
  Decimal2.isDecimal = isDecimalInstance;
  Decimal2.abs = abs$4;
  Decimal2.acos = acos;
  Decimal2.acosh = acosh;
  Decimal2.add = add$4;
  Decimal2.asin = asin;
  Decimal2.asinh = asinh;
  Decimal2.atan = atan;
  Decimal2.atanh = atanh;
  Decimal2.atan2 = atan2;
  Decimal2.cbrt = cbrt;
  Decimal2.ceil = ceil$1;
  Decimal2.clamp = clamp$1;
  Decimal2.cos = cos;
  Decimal2.cosh = cosh;
  Decimal2.div = div;
  Decimal2.exp = exp;
  Decimal2.floor = floor$2;
  Decimal2.hypot = hypot;
  Decimal2.ln = ln$1;
  Decimal2.log = log;
  Decimal2.log10 = log10;
  Decimal2.log2 = log2;
  Decimal2.max = max$9;
  Decimal2.min = min$a;
  Decimal2.mod = mod$3;
  Decimal2.mul = mul$1;
  Decimal2.pow = pow;
  Decimal2.random = random;
  Decimal2.round = round$6;
  Decimal2.sign = sign$2;
  Decimal2.sin = sin;
  Decimal2.sinh = sinh;
  Decimal2.sqrt = sqrt;
  Decimal2.sub = sub$1;
  Decimal2.sum = sum$1;
  Decimal2.tan = tan;
  Decimal2.tanh = tanh;
  Decimal2.trunc = trunc$1;
  if (obj === void 0)
    obj = {};
  if (obj) {
    if (obj.defaults !== true) {
      ps = ["precision", "rounding", "toExpNeg", "toExpPos", "maxE", "minE", "modulo", "crypto"];
      for (i2 = 0; i2 < ps.length; )
        if (!obj.hasOwnProperty(p2 = ps[i2++]))
          obj[p2] = this[p2];
    }
  }
  Decimal2.config(obj);
  return Decimal2;
}
function div(x2, y2) {
  return new this(x2).div(y2);
}
function exp(x2) {
  return new this(x2).exp();
}
function floor$2(x2) {
  return finalise(x2 = new this(x2), x2.e + 1, 3);
}
function hypot() {
  var i2, n2, t3 = new this(0);
  external = false;
  for (i2 = 0; i2 < arguments.length; ) {
    n2 = new this(arguments[i2++]);
    if (!n2.d) {
      if (n2.s) {
        external = true;
        return new this(1 / 0);
      }
      t3 = n2;
    } else if (t3.d) {
      t3 = t3.plus(n2.times(n2));
    }
  }
  external = true;
  return t3.sqrt();
}
function isDecimalInstance(obj) {
  return obj instanceof Decimal || obj && obj.toStringTag === tag || false;
}
function ln$1(x2) {
  return new this(x2).ln();
}
function log(x2, y2) {
  return new this(x2).log(y2);
}
function log2(x2) {
  return new this(x2).log(2);
}
function log10(x2) {
  return new this(x2).log(10);
}
function max$9() {
  return maxOrMin(this, arguments, "lt");
}
function min$a() {
  return maxOrMin(this, arguments, "gt");
}
function mod$3(x2, y2) {
  return new this(x2).mod(y2);
}
function mul$1(x2, y2) {
  return new this(x2).mul(y2);
}
function pow(x2, y2) {
  return new this(x2).pow(y2);
}
function random(sd) {
  var d2, e2, k2, n2, i2 = 0, r2 = new this(1), rd = [];
  if (sd === void 0)
    sd = this.precision;
  else
    checkInt32(sd, 1, MAX_DIGITS);
  k2 = Math.ceil(sd / LOG_BASE);
  if (!this.crypto) {
    for (; i2 < k2; )
      rd[i2++] = Math.random() * 1e7 | 0;
  } else if (crypto.getRandomValues) {
    d2 = crypto.getRandomValues(new Uint32Array(k2));
    for (; i2 < k2; ) {
      n2 = d2[i2];
      if (n2 >= 429e7) {
        d2[i2] = crypto.getRandomValues(new Uint32Array(1))[0];
      } else {
        rd[i2++] = n2 % 1e7;
      }
    }
  } else if (crypto.randomBytes) {
    d2 = crypto.randomBytes(k2 *= 4);
    for (; i2 < k2; ) {
      n2 = d2[i2] + (d2[i2 + 1] << 8) + (d2[i2 + 2] << 16) + ((d2[i2 + 3] & 127) << 24);
      if (n2 >= 214e7) {
        crypto.randomBytes(4).copy(d2, i2);
      } else {
        rd.push(n2 % 1e7);
        i2 += 4;
      }
    }
    i2 = k2 / 4;
  } else {
    throw Error(cryptoUnavailable);
  }
  k2 = rd[--i2];
  sd %= LOG_BASE;
  if (k2 && sd) {
    n2 = mathpow(10, LOG_BASE - sd);
    rd[i2] = (k2 / n2 | 0) * n2;
  }
  for (; rd[i2] === 0; i2--)
    rd.pop();
  if (i2 < 0) {
    e2 = 0;
    rd = [0];
  } else {
    e2 = -1;
    for (; rd[0] === 0; e2 -= LOG_BASE)
      rd.shift();
    for (k2 = 1, n2 = rd[0]; n2 >= 10; n2 /= 10)
      k2++;
    if (k2 < LOG_BASE)
      e2 -= LOG_BASE - k2;
  }
  r2.e = e2;
  r2.d = rd;
  return r2;
}
function round$6(x2) {
  return finalise(x2 = new this(x2), x2.e + 1, this.rounding);
}
function sign$2(x2) {
  x2 = new this(x2);
  return x2.d ? x2.d[0] ? x2.s : 0 * x2.s : x2.s || NaN;
}
function sin(x2) {
  return new this(x2).sin();
}
function sinh(x2) {
  return new this(x2).sinh();
}
function sqrt(x2) {
  return new this(x2).sqrt();
}
function sub$1(x2, y2) {
  return new this(x2).sub(y2);
}
function sum$1() {
  var i2 = 0, args = arguments, x2 = new this(args[i2]);
  external = false;
  for (; x2.s && ++i2 < args.length; )
    x2 = x2.plus(args[i2]);
  external = true;
  return finalise(x2, this.precision, this.rounding);
}
function tan(x2) {
  return new this(x2).tan();
}
function tanh(x2) {
  return new this(x2).tanh();
}
function trunc$1(x2) {
  return finalise(x2 = new this(x2), x2.e + 1, 1);
}
P$1[Symbol.for("nodejs.util.inspect.custom")] = P$1.toString;
P$1[Symbol.toStringTag] = "Decimal";
var Decimal = P$1.constructor = clone$8(DEFAULTS$1);
LN10 = new Decimal(LN10);
PI$9 = new Decimal(PI$9);
//! moment.js
//! version : 2.30.1
//! authors : Tim Wood, Iskren Chernev, Moment.js contributors
//! license : MIT
//! momentjs.com
var hookCallback$1;
function hooks$2() {
  return hookCallback$1.apply(null, arguments);
}
function setHookCallback$1(callback) {
  hookCallback$1 = callback;
}
function isArray$7(input) {
  return input instanceof Array || Object.prototype.toString.call(input) === "[object Array]";
}
function isObject$m(input) {
  return input != null && Object.prototype.toString.call(input) === "[object Object]";
}
function hasOwnProp$1(a2, b2) {
  return Object.prototype.hasOwnProperty.call(a2, b2);
}
function isObjectEmpty$1(obj) {
  if (Object.getOwnPropertyNames) {
    return Object.getOwnPropertyNames(obj).length === 0;
  } else {
    var k2;
    for (k2 in obj) {
      if (hasOwnProp$1(obj, k2)) {
        return false;
      }
    }
    return true;
  }
}
function isUndefined$3(input) {
  return input === void 0;
}
function isNumber$4(input) {
  return typeof input === "number" || Object.prototype.toString.call(input) === "[object Number]";
}
function isDate$3(input) {
  return input instanceof Date || Object.prototype.toString.call(input) === "[object Date]";
}
function map$3(arr, fn) {
  var res = [], i2, arrLen = arr.length;
  for (i2 = 0; i2 < arrLen; ++i2) {
    res.push(fn(arr[i2], i2));
  }
  return res;
}
function extend$6(a2, b2) {
  for (var i2 in b2) {
    if (hasOwnProp$1(b2, i2)) {
      a2[i2] = b2[i2];
    }
  }
  if (hasOwnProp$1(b2, "toString")) {
    a2.toString = b2.toString;
  }
  if (hasOwnProp$1(b2, "valueOf")) {
    a2.valueOf = b2.valueOf;
  }
  return a2;
}
function createUTC$1(input, format2, locale2, strict) {
  return createLocalOrUTC$1(input, format2, locale2, strict, true).utc();
}
function defaultParsingFlags$1() {
  return {
    empty: false,
    unusedTokens: [],
    unusedInput: [],
    overflow: -2,
    charsLeftOver: 0,
    nullInput: false,
    invalidEra: null,
    invalidMonth: null,
    invalidFormat: false,
    userInvalidated: false,
    iso: false,
    parsedDateParts: [],
    era: null,
    meridiem: null,
    rfc2822: false,
    weekdayMismatch: false
  };
}
function getParsingFlags$1(m2) {
  if (m2._pf == null) {
    m2._pf = defaultParsingFlags$1();
  }
  return m2._pf;
}
var some$1;
if (Array.prototype.some) {
  some$1 = Array.prototype.some;
} else {
  some$1 = function(fun) {
    var t3 = Object(this), len2 = t3.length >>> 0, i2;
    for (i2 = 0; i2 < len2; i2++) {
      if (i2 in t3 && fun.call(this, t3[i2], i2, t3)) {
        return true;
      }
    }
    return false;
  };
}
function isValid$3(m2) {
  var flags3 = null, parsedParts = false, isNowValid = m2._d && !isNaN(m2._d.getTime());
  if (isNowValid) {
    flags3 = getParsingFlags$1(m2);
    parsedParts = some$1.call(flags3.parsedDateParts, function(i2) {
      return i2 != null;
    });
    isNowValid = flags3.overflow < 0 && !flags3.empty && !flags3.invalidEra && !flags3.invalidMonth && !flags3.invalidWeekday && !flags3.weekdayMismatch && !flags3.nullInput && !flags3.invalidFormat && !flags3.userInvalidated && (!flags3.meridiem || flags3.meridiem && parsedParts);
    if (m2._strict) {
      isNowValid = isNowValid && flags3.charsLeftOver === 0 && flags3.unusedTokens.length === 0 && flags3.bigHour === void 0;
    }
  }
  if (Object.isFrozen == null || !Object.isFrozen(m2)) {
    m2._isValid = isNowValid;
  } else {
    return isNowValid;
  }
  return m2._isValid;
}
function createInvalid$2(flags3) {
  var m2 = createUTC$1(NaN);
  if (flags3 != null) {
    extend$6(getParsingFlags$1(m2), flags3);
  } else {
    getParsingFlags$1(m2).userInvalidated = true;
  }
  return m2;
}
var momentProperties$1 = hooks$2.momentProperties = [], updateInProgress$1 = false;
function copyConfig$1(to2, from2) {
  var i2, prop, val, momentPropertiesLen = momentProperties$1.length;
  if (!isUndefined$3(from2._isAMomentObject)) {
    to2._isAMomentObject = from2._isAMomentObject;
  }
  if (!isUndefined$3(from2._i)) {
    to2._i = from2._i;
  }
  if (!isUndefined$3(from2._f)) {
    to2._f = from2._f;
  }
  if (!isUndefined$3(from2._l)) {
    to2._l = from2._l;
  }
  if (!isUndefined$3(from2._strict)) {
    to2._strict = from2._strict;
  }
  if (!isUndefined$3(from2._tzm)) {
    to2._tzm = from2._tzm;
  }
  if (!isUndefined$3(from2._isUTC)) {
    to2._isUTC = from2._isUTC;
  }
  if (!isUndefined$3(from2._offset)) {
    to2._offset = from2._offset;
  }
  if (!isUndefined$3(from2._pf)) {
    to2._pf = getParsingFlags$1(from2);
  }
  if (!isUndefined$3(from2._locale)) {
    to2._locale = from2._locale;
  }
  if (momentPropertiesLen > 0) {
    for (i2 = 0; i2 < momentPropertiesLen; i2++) {
      prop = momentProperties$1[i2];
      val = from2[prop];
      if (!isUndefined$3(val)) {
        to2[prop] = val;
      }
    }
  }
  return to2;
}
function Moment$1(config2) {
  copyConfig$1(this, config2);
  this._d = new Date(config2._d != null ? config2._d.getTime() : NaN);
  if (!this.isValid()) {
    this._d = /* @__PURE__ */ new Date(NaN);
  }
  if (updateInProgress$1 === false) {
    updateInProgress$1 = true;
    hooks$2.updateOffset(this);
    updateInProgress$1 = false;
  }
}
function isMoment$1(obj) {
  return obj instanceof Moment$1 || obj != null && obj._isAMomentObject != null;
}
function warn$2(msg) {
  if (hooks$2.suppressDeprecationWarnings === false && typeof console !== "undefined" && console.warn) {
    console.warn("Deprecation warning: " + msg);
  }
}
function deprecate$1(msg, fn) {
  var firstTime = true;
  return extend$6(function() {
    if (hooks$2.deprecationHandler != null) {
      hooks$2.deprecationHandler(null, msg);
    }
    if (firstTime) {
      var args = [], arg, i2, key, argLen = arguments.length;
      for (i2 = 0; i2 < argLen; i2++) {
        arg = "";
        if (typeof arguments[i2] === "object") {
          arg += "\n[" + i2 + "] ";
          for (key in arguments[0]) {
            if (hasOwnProp$1(arguments[0], key)) {
              arg += key + ": " + arguments[0][key] + ", ";
            }
          }
          arg = arg.slice(0, -2);
        } else {
          arg = arguments[i2];
        }
        args.push(arg);
      }
      warn$2(
        msg + "\nArguments: " + Array.prototype.slice.call(args).join("") + "\n" + new Error().stack
      );
      firstTime = false;
    }
    return fn.apply(this, arguments);
  }, fn);
}
var deprecations$1 = {};
function deprecateSimple$1(name, msg) {
  if (hooks$2.deprecationHandler != null) {
    hooks$2.deprecationHandler(name, msg);
  }
  if (!deprecations$1[name]) {
    warn$2(msg);
    deprecations$1[name] = true;
  }
}
hooks$2.suppressDeprecationWarnings = false;
hooks$2.deprecationHandler = null;
function isFunction$5(input) {
  return typeof Function !== "undefined" && input instanceof Function || Object.prototype.toString.call(input) === "[object Function]";
}
function set$5(config2) {
  var prop, i2;
  for (i2 in config2) {
    if (hasOwnProp$1(config2, i2)) {
      prop = config2[i2];
      if (isFunction$5(prop)) {
        this[i2] = prop;
      } else {
        this["_" + i2] = prop;
      }
    }
  }
  this._config = config2;
  this._dayOfMonthOrdinalParseLenient = new RegExp(
    (this._dayOfMonthOrdinalParse.source || this._ordinalParse.source) + "|" + /\d{1,2}/.source
  );
}
function mergeConfigs$1(parentConfig, childConfig) {
  var res = extend$6({}, parentConfig), prop;
  for (prop in childConfig) {
    if (hasOwnProp$1(childConfig, prop)) {
      if (isObject$m(parentConfig[prop]) && isObject$m(childConfig[prop])) {
        res[prop] = {};
        extend$6(res[prop], parentConfig[prop]);
        extend$6(res[prop], childConfig[prop]);
      } else if (childConfig[prop] != null) {
        res[prop] = childConfig[prop];
      } else {
        delete res[prop];
      }
    }
  }
  for (prop in parentConfig) {
    if (hasOwnProp$1(parentConfig, prop) && !hasOwnProp$1(childConfig, prop) && isObject$m(parentConfig[prop])) {
      res[prop] = extend$6({}, res[prop]);
    }
  }
  return res;
}
function Locale$1(config2) {
  if (config2 != null) {
    this.set(config2);
  }
}
var keys$5;
if (Object.keys) {
  keys$5 = Object.keys;
} else {
  keys$5 = function(obj) {
    var i2, res = [];
    for (i2 in obj) {
      if (hasOwnProp$1(obj, i2)) {
        res.push(i2);
      }
    }
    return res;
  };
}
var defaultCalendar$1 = {
  sameDay: "[Today at] LT",
  nextDay: "[Tomorrow at] LT",
  nextWeek: "dddd [at] LT",
  lastDay: "[Yesterday at] LT",
  lastWeek: "[Last] dddd [at] LT",
  sameElse: "L"
};
function calendar$2(key, mom, now2) {
  var output2 = this._calendar[key] || this._calendar["sameElse"];
  return isFunction$5(output2) ? output2.call(mom, now2) : output2;
}
function zeroFill$1(number, targetLength, forceSign) {
  var absNumber = "" + Math.abs(number), zerosToFill = targetLength - absNumber.length, sign2 = number >= 0;
  return (sign2 ? forceSign ? "+" : "" : "-") + Math.pow(10, Math.max(0, zerosToFill)).toString().substr(1) + absNumber;
}
var formattingTokens$1 = /(\[[^\[]*\])|(\\)?([Hh]mm(ss)?|Mo|MM?M?M?|Do|DDDo|DD?D?D?|ddd?d?|do?|w[o|w]?|W[o|W]?|Qo?|N{1,5}|YYYYYY|YYYYY|YYYY|YY|y{2,4}|yo?|gg(ggg?)?|GG(GGG?)?|e|E|a|A|hh?|HH?|kk?|mm?|ss?|S{1,9}|x|X|zz?|ZZ?|.)/g, localFormattingTokens$1 = /(\[[^\[]*\])|(\\)?(LTS|LT|LL?L?L?|l{1,4})/g, formatFunctions$1 = {}, formatTokenFunctions$1 = {};
function addFormatToken$1(token2, padded, ordinal2, callback) {
  var func = callback;
  if (typeof callback === "string") {
    func = function() {
      return this[callback]();
    };
  }
  if (token2) {
    formatTokenFunctions$1[token2] = func;
  }
  if (padded) {
    formatTokenFunctions$1[padded[0]] = function() {
      return zeroFill$1(func.apply(this, arguments), padded[1], padded[2]);
    };
  }
  if (ordinal2) {
    formatTokenFunctions$1[ordinal2] = function() {
      return this.localeData().ordinal(
        func.apply(this, arguments),
        token2
      );
    };
  }
}
function removeFormattingTokens$1(input) {
  if (input.match(/\[[\s\S]/)) {
    return input.replace(/^\[|\]$/g, "");
  }
  return input.replace(/\\/g, "");
}
function makeFormatFunction$1(format2) {
  var array2 = format2.match(formattingTokens$1), i2, length;
  for (i2 = 0, length = array2.length; i2 < length; i2++) {
    if (formatTokenFunctions$1[array2[i2]]) {
      array2[i2] = formatTokenFunctions$1[array2[i2]];
    } else {
      array2[i2] = removeFormattingTokens$1(array2[i2]);
    }
  }
  return function(mom) {
    var output2 = "", i3;
    for (i3 = 0; i3 < length; i3++) {
      output2 += isFunction$5(array2[i3]) ? array2[i3].call(mom, format2) : array2[i3];
    }
    return output2;
  };
}
function formatMoment$1(m2, format2) {
  if (!m2.isValid()) {
    return m2.localeData().invalidDate();
  }
  format2 = expandFormat$1(format2, m2.localeData());
  formatFunctions$1[format2] = formatFunctions$1[format2] || makeFormatFunction$1(format2);
  return formatFunctions$1[format2](m2);
}
function expandFormat$1(format2, locale2) {
  var i2 = 5;
  function replaceLongDateFormatTokens(input) {
    return locale2.longDateFormat(input) || input;
  }
  localFormattingTokens$1.lastIndex = 0;
  while (i2 >= 0 && localFormattingTokens$1.test(format2)) {
    format2 = format2.replace(
      localFormattingTokens$1,
      replaceLongDateFormatTokens
    );
    localFormattingTokens$1.lastIndex = 0;
    i2 -= 1;
  }
  return format2;
}
var defaultLongDateFormat$1 = {
  LTS: "h:mm:ss A",
  LT: "h:mm A",
  L: "MM/DD/YYYY",
  LL: "MMMM D, YYYY",
  LLL: "MMMM D, YYYY h:mm A",
  LLLL: "dddd, MMMM D, YYYY h:mm A"
};
function longDateFormat$1(key) {
  var format2 = this._longDateFormat[key], formatUpper = this._longDateFormat[key.toUpperCase()];
  if (format2 || !formatUpper) {
    return format2;
  }
  this._longDateFormat[key] = formatUpper.match(formattingTokens$1).map(function(tok) {
    if (tok === "MMMM" || tok === "MM" || tok === "DD" || tok === "dddd") {
      return tok.slice(1);
    }
    return tok;
  }).join("");
  return this._longDateFormat[key];
}
var defaultInvalidDate$1 = "Invalid date";
function invalidDate$1() {
  return this._invalidDate;
}
var defaultOrdinal$1 = "%d", defaultDayOfMonthOrdinalParse$1 = /\d{1,2}/;
function ordinal$1(number) {
  return this._ordinal.replace("%d", number);
}
var defaultRelativeTime$1 = {
  future: "in %s",
  past: "%s ago",
  s: "a few seconds",
  ss: "%d seconds",
  m: "a minute",
  mm: "%d minutes",
  h: "an hour",
  hh: "%d hours",
  d: "a day",
  dd: "%d days",
  w: "a week",
  ww: "%d weeks",
  M: "a month",
  MM: "%d months",
  y: "a year",
  yy: "%d years"
};
function relativeTime$2(number, withoutSuffix, string, isFuture) {
  var output2 = this._relativeTime[string];
  return isFunction$5(output2) ? output2(number, withoutSuffix, string, isFuture) : output2.replace(/%d/i, number);
}
function pastFuture$1(diff2, output2) {
  var format2 = this._relativeTime[diff2 > 0 ? "future" : "past"];
  return isFunction$5(format2) ? format2(output2) : format2.replace(/%s/i, output2);
}
var aliases$1 = {
  D: "date",
  dates: "date",
  date: "date",
  d: "day",
  days: "day",
  day: "day",
  e: "weekday",
  weekdays: "weekday",
  weekday: "weekday",
  E: "isoWeekday",
  isoweekdays: "isoWeekday",
  isoweekday: "isoWeekday",
  DDD: "dayOfYear",
  dayofyears: "dayOfYear",
  dayofyear: "dayOfYear",
  h: "hour",
  hours: "hour",
  hour: "hour",
  ms: "millisecond",
  milliseconds: "millisecond",
  millisecond: "millisecond",
  m: "minute",
  minutes: "minute",
  minute: "minute",
  M: "month",
  months: "month",
  month: "month",
  Q: "quarter",
  quarters: "quarter",
  quarter: "quarter",
  s: "second",
  seconds: "second",
  second: "second",
  gg: "weekYear",
  weekyears: "weekYear",
  weekyear: "weekYear",
  GG: "isoWeekYear",
  isoweekyears: "isoWeekYear",
  isoweekyear: "isoWeekYear",
  w: "week",
  weeks: "week",
  week: "week",
  W: "isoWeek",
  isoweeks: "isoWeek",
  isoweek: "isoWeek",
  y: "year",
  years: "year",
  year: "year"
};
function normalizeUnits$1(units) {
  return typeof units === "string" ? aliases$1[units] || aliases$1[units.toLowerCase()] : void 0;
}
function normalizeObjectUnits$1(inputObject) {
  var normalizedInput = {}, normalizedProp, prop;
  for (prop in inputObject) {
    if (hasOwnProp$1(inputObject, prop)) {
      normalizedProp = normalizeUnits$1(prop);
      if (normalizedProp) {
        normalizedInput[normalizedProp] = inputObject[prop];
      }
    }
  }
  return normalizedInput;
}
var priorities$1 = {
  date: 9,
  day: 11,
  weekday: 11,
  isoWeekday: 11,
  dayOfYear: 4,
  hour: 13,
  millisecond: 16,
  minute: 14,
  month: 8,
  quarter: 7,
  second: 15,
  weekYear: 1,
  isoWeekYear: 1,
  week: 5,
  isoWeek: 5,
  year: 1
};
function getPrioritizedUnits$1(unitsObj) {
  var units = [], u2;
  for (u2 in unitsObj) {
    if (hasOwnProp$1(unitsObj, u2)) {
      units.push({ unit: u2, priority: priorities$1[u2] });
    }
  }
  units.sort(function(a2, b2) {
    return a2.priority - b2.priority;
  });
  return units;
}
var match1$1 = /\d/, match2$1 = /\d\d/, match3$1 = /\d{3}/, match4$1 = /\d{4}/, match6$1 = /[+-]?\d{6}/, match1to2$1 = /\d\d?/, match3to4$1 = /\d\d\d\d?/, match5to6$1 = /\d\d\d\d\d\d?/, match1to3$1 = /\d{1,3}/, match1to4$1 = /\d{1,4}/, match1to6$1 = /[+-]?\d{1,6}/, matchUnsigned$1 = /\d+/, matchSigned$1 = /[+-]?\d+/, matchOffset$1 = /Z|[+-]\d\d:?\d\d/gi, matchShortOffset$1 = /Z|[+-]\d\d(?::?\d\d)?/gi, matchTimestamp$1 = /[+-]?\d+(\.\d{1,3})?/, matchWord$1 = /[0-9]{0,256}['a-z\u00A0-\u05FF\u0700-\uD7FF\uF900-\uFDCF\uFDF0-\uFF07\uFF10-\uFFEF]{1,256}|[\u0600-\u06FF\/]{1,256}(\s*?[\u0600-\u06FF]{1,256}){1,2}/i, match1to2NoLeadingZero$1 = /^[1-9]\d?/, match1to2HasZero$1 = /^([1-9]\d|\d)/, regexes$1;
regexes$1 = {};
function addRegexToken$1(token2, regex2, strictRegex) {
  regexes$1[token2] = isFunction$5(regex2) ? regex2 : function(isStrict, localeData2) {
    return isStrict && strictRegex ? strictRegex : regex2;
  };
}
function getParseRegexForToken$1(token2, config2) {
  if (!hasOwnProp$1(regexes$1, token2)) {
    return new RegExp(unescapeFormat$1(token2));
  }
  return regexes$1[token2](config2._strict, config2._locale);
}
function unescapeFormat$1(s2) {
  return regexEscape$1(
    s2.replace("\\", "").replace(
      /\\(\[)|\\(\])|\[([^\]\[]*)\]|\\(.)/g,
      function(matched, p1, p2, p3, p4) {
        return p1 || p2 || p3 || p4;
      }
    )
  );
}
function regexEscape$1(s2) {
  return s2.replace(/[-\/\\^$*+?.()|[\]{}]/g, "\\$&");
}
function absFloor$1(number) {
  if (number < 0) {
    return Math.ceil(number) || 0;
  } else {
    return Math.floor(number);
  }
}
function toInt$1(argumentForCoercion) {
  var coercedNumber = +argumentForCoercion, value = 0;
  if (coercedNumber !== 0 && isFinite(coercedNumber)) {
    value = absFloor$1(coercedNumber);
  }
  return value;
}
var tokens$1 = {};
function addParseToken$1(token2, callback) {
  var i2, func = callback, tokenLen;
  if (typeof token2 === "string") {
    token2 = [token2];
  }
  if (isNumber$4(callback)) {
    func = function(input, array2) {
      array2[callback] = toInt$1(input);
    };
  }
  tokenLen = token2.length;
  for (i2 = 0; i2 < tokenLen; i2++) {
    tokens$1[token2[i2]] = func;
  }
}
function addWeekParseToken$1(token2, callback) {
  addParseToken$1(token2, function(input, array2, config2, token3) {
    config2._w = config2._w || {};
    callback(input, config2._w, config2, token3);
  });
}
function addTimeToArrayFromToken$1(token2, input, config2) {
  if (input != null && hasOwnProp$1(tokens$1, token2)) {
    tokens$1[token2](input, config2._a, config2, token2);
  }
}
function isLeapYear$1(year) {
  return year % 4 === 0 && year % 100 !== 0 || year % 400 === 0;
}
var YEAR$1 = 0, MONTH$1 = 1, DATE$1 = 2, HOUR$1 = 3, MINUTE$1 = 4, SECOND$1 = 5, MILLISECOND$1 = 6, WEEK$1 = 7, WEEKDAY$1 = 8;
addFormatToken$1("Y", 0, 0, function() {
  var y2 = this.year();
  return y2 <= 9999 ? zeroFill$1(y2, 4) : "+" + y2;
});
addFormatToken$1(0, ["YY", 2], 0, function() {
  return this.year() % 100;
});
addFormatToken$1(0, ["YYYY", 4], 0, "year");
addFormatToken$1(0, ["YYYYY", 5], 0, "year");
addFormatToken$1(0, ["YYYYYY", 6, true], 0, "year");
addRegexToken$1("Y", matchSigned$1);
addRegexToken$1("YY", match1to2$1, match2$1);
addRegexToken$1("YYYY", match1to4$1, match4$1);
addRegexToken$1("YYYYY", match1to6$1, match6$1);
addRegexToken$1("YYYYYY", match1to6$1, match6$1);
addParseToken$1(["YYYYY", "YYYYYY"], YEAR$1);
addParseToken$1("YYYY", function(input, array2) {
  array2[YEAR$1] = input.length === 2 ? hooks$2.parseTwoDigitYear(input) : toInt$1(input);
});
addParseToken$1("YY", function(input, array2) {
  array2[YEAR$1] = hooks$2.parseTwoDigitYear(input);
});
addParseToken$1("Y", function(input, array2) {
  array2[YEAR$1] = parseInt(input, 10);
});
function daysInYear$1(year) {
  return isLeapYear$1(year) ? 366 : 365;
}
hooks$2.parseTwoDigitYear = function(input) {
  return toInt$1(input) + (toInt$1(input) > 68 ? 1900 : 2e3);
};
var getSetYear$1 = makeGetSet$1("FullYear", true);
function getIsLeapYear$1() {
  return isLeapYear$1(this.year());
}
function makeGetSet$1(unit, keepTime) {
  return function(value) {
    if (value != null) {
      set$1$2(this, unit, value);
      hooks$2.updateOffset(this, keepTime);
      return this;
    } else {
      return get$4(this, unit);
    }
  };
}
function get$4(mom, unit) {
  if (!mom.isValid()) {
    return NaN;
  }
  var d2 = mom._d, isUTC = mom._isUTC;
  switch (unit) {
    case "Milliseconds":
      return isUTC ? d2.getUTCMilliseconds() : d2.getMilliseconds();
    case "Seconds":
      return isUTC ? d2.getUTCSeconds() : d2.getSeconds();
    case "Minutes":
      return isUTC ? d2.getUTCMinutes() : d2.getMinutes();
    case "Hours":
      return isUTC ? d2.getUTCHours() : d2.getHours();
    case "Date":
      return isUTC ? d2.getUTCDate() : d2.getDate();
    case "Day":
      return isUTC ? d2.getUTCDay() : d2.getDay();
    case "Month":
      return isUTC ? d2.getUTCMonth() : d2.getMonth();
    case "FullYear":
      return isUTC ? d2.getUTCFullYear() : d2.getFullYear();
    default:
      return NaN;
  }
}
function set$1$2(mom, unit, value) {
  var d2, isUTC, year, month, date;
  if (!mom.isValid() || isNaN(value)) {
    return;
  }
  d2 = mom._d;
  isUTC = mom._isUTC;
  switch (unit) {
    case "Milliseconds":
      return void (isUTC ? d2.setUTCMilliseconds(value) : d2.setMilliseconds(value));
    case "Seconds":
      return void (isUTC ? d2.setUTCSeconds(value) : d2.setSeconds(value));
    case "Minutes":
      return void (isUTC ? d2.setUTCMinutes(value) : d2.setMinutes(value));
    case "Hours":
      return void (isUTC ? d2.setUTCHours(value) : d2.setHours(value));
    case "Date":
      return void (isUTC ? d2.setUTCDate(value) : d2.setDate(value));
    case "FullYear":
      break;
    default:
      return;
  }
  year = value;
  month = mom.month();
  date = mom.date();
  date = date === 29 && month === 1 && !isLeapYear$1(year) ? 28 : date;
  void (isUTC ? d2.setUTCFullYear(year, month, date) : d2.setFullYear(year, month, date));
}
function stringGet$1(units) {
  units = normalizeUnits$1(units);
  if (isFunction$5(this[units])) {
    return this[units]();
  }
  return this;
}
function stringSet$1(units, value) {
  if (typeof units === "object") {
    units = normalizeObjectUnits$1(units);
    var prioritized = getPrioritizedUnits$1(units), i2, prioritizedLen = prioritized.length;
    for (i2 = 0; i2 < prioritizedLen; i2++) {
      this[prioritized[i2].unit](units[prioritized[i2].unit]);
    }
  } else {
    units = normalizeUnits$1(units);
    if (isFunction$5(this[units])) {
      return this[units](value);
    }
  }
  return this;
}
function mod$2(n2, x2) {
  return (n2 % x2 + x2) % x2;
}
var indexOf$4;
if (Array.prototype.indexOf) {
  indexOf$4 = Array.prototype.indexOf;
} else {
  indexOf$4 = function(o2) {
    var i2;
    for (i2 = 0; i2 < this.length; ++i2) {
      if (this[i2] === o2) {
        return i2;
      }
    }
    return -1;
  };
}
function daysInMonth$1(year, month) {
  if (isNaN(year) || isNaN(month)) {
    return NaN;
  }
  var modMonth = mod$2(month, 12);
  year += (month - modMonth) / 12;
  return modMonth === 1 ? isLeapYear$1(year) ? 29 : 28 : 31 - modMonth % 7 % 2;
}
addFormatToken$1("M", ["MM", 2], "Mo", function() {
  return this.month() + 1;
});
addFormatToken$1("MMM", 0, 0, function(format2) {
  return this.localeData().monthsShort(this, format2);
});
addFormatToken$1("MMMM", 0, 0, function(format2) {
  return this.localeData().months(this, format2);
});
addRegexToken$1("M", match1to2$1, match1to2NoLeadingZero$1);
addRegexToken$1("MM", match1to2$1, match2$1);
addRegexToken$1("MMM", function(isStrict, locale2) {
  return locale2.monthsShortRegex(isStrict);
});
addRegexToken$1("MMMM", function(isStrict, locale2) {
  return locale2.monthsRegex(isStrict);
});
addParseToken$1(["M", "MM"], function(input, array2) {
  array2[MONTH$1] = toInt$1(input) - 1;
});
addParseToken$1(["MMM", "MMMM"], function(input, array2, config2, token2) {
  var month = config2._locale.monthsParse(input, token2, config2._strict);
  if (month != null) {
    array2[MONTH$1] = month;
  } else {
    getParsingFlags$1(config2).invalidMonth = input;
  }
});
var defaultLocaleMonths$1 = "January_February_March_April_May_June_July_August_September_October_November_December".split(
  "_"
), defaultLocaleMonthsShort$1 = "Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec".split("_"), MONTHS_IN_FORMAT$1 = /D[oD]?(\[[^\[\]]*\]|\s)+MMMM?/, defaultMonthsShortRegex$1 = matchWord$1, defaultMonthsRegex$1 = matchWord$1;
function localeMonths$1(m2, format2) {
  if (!m2) {
    return isArray$7(this._months) ? this._months : this._months["standalone"];
  }
  return isArray$7(this._months) ? this._months[m2.month()] : this._months[(this._months.isFormat || MONTHS_IN_FORMAT$1).test(format2) ? "format" : "standalone"][m2.month()];
}
function localeMonthsShort$1(m2, format2) {
  if (!m2) {
    return isArray$7(this._monthsShort) ? this._monthsShort : this._monthsShort["standalone"];
  }
  return isArray$7(this._monthsShort) ? this._monthsShort[m2.month()] : this._monthsShort[MONTHS_IN_FORMAT$1.test(format2) ? "format" : "standalone"][m2.month()];
}
function handleStrictParse$2(monthName, format2, strict) {
  var i2, ii, mom, llc = monthName.toLocaleLowerCase();
  if (!this._monthsParse) {
    this._monthsParse = [];
    this._longMonthsParse = [];
    this._shortMonthsParse = [];
    for (i2 = 0; i2 < 12; ++i2) {
      mom = createUTC$1([2e3, i2]);
      this._shortMonthsParse[i2] = this.monthsShort(
        mom,
        ""
      ).toLocaleLowerCase();
      this._longMonthsParse[i2] = this.months(mom, "").toLocaleLowerCase();
    }
  }
  if (strict) {
    if (format2 === "MMM") {
      ii = indexOf$4.call(this._shortMonthsParse, llc);
      return ii !== -1 ? ii : null;
    } else {
      ii = indexOf$4.call(this._longMonthsParse, llc);
      return ii !== -1 ? ii : null;
    }
  } else {
    if (format2 === "MMM") {
      ii = indexOf$4.call(this._shortMonthsParse, llc);
      if (ii !== -1) {
        return ii;
      }
      ii = indexOf$4.call(this._longMonthsParse, llc);
      return ii !== -1 ? ii : null;
    } else {
      ii = indexOf$4.call(this._longMonthsParse, llc);
      if (ii !== -1) {
        return ii;
      }
      ii = indexOf$4.call(this._shortMonthsParse, llc);
      return ii !== -1 ? ii : null;
    }
  }
}
function localeMonthsParse$1(monthName, format2, strict) {
  var i2, mom, regex2;
  if (this._monthsParseExact) {
    return handleStrictParse$2.call(this, monthName, format2, strict);
  }
  if (!this._monthsParse) {
    this._monthsParse = [];
    this._longMonthsParse = [];
    this._shortMonthsParse = [];
  }
  for (i2 = 0; i2 < 12; i2++) {
    mom = createUTC$1([2e3, i2]);
    if (strict && !this._longMonthsParse[i2]) {
      this._longMonthsParse[i2] = new RegExp(
        "^" + this.months(mom, "").replace(".", "") + "$",
        "i"
      );
      this._shortMonthsParse[i2] = new RegExp(
        "^" + this.monthsShort(mom, "").replace(".", "") + "$",
        "i"
      );
    }
    if (!strict && !this._monthsParse[i2]) {
      regex2 = "^" + this.months(mom, "") + "|^" + this.monthsShort(mom, "");
      this._monthsParse[i2] = new RegExp(regex2.replace(".", ""), "i");
    }
    if (strict && format2 === "MMMM" && this._longMonthsParse[i2].test(monthName)) {
      return i2;
    } else if (strict && format2 === "MMM" && this._shortMonthsParse[i2].test(monthName)) {
      return i2;
    } else if (!strict && this._monthsParse[i2].test(monthName)) {
      return i2;
    }
  }
}
function setMonth$1(mom, value) {
  if (!mom.isValid()) {
    return mom;
  }
  if (typeof value === "string") {
    if (/^\d+$/.test(value)) {
      value = toInt$1(value);
    } else {
      value = mom.localeData().monthsParse(value);
      if (!isNumber$4(value)) {
        return mom;
      }
    }
  }
  var month = value, date = mom.date();
  date = date < 29 ? date : Math.min(date, daysInMonth$1(mom.year(), month));
  void (mom._isUTC ? mom._d.setUTCMonth(month, date) : mom._d.setMonth(month, date));
  return mom;
}
function getSetMonth$1(value) {
  if (value != null) {
    setMonth$1(this, value);
    hooks$2.updateOffset(this, true);
    return this;
  } else {
    return get$4(this, "Month");
  }
}
function getDaysInMonth$1() {
  return daysInMonth$1(this.year(), this.month());
}
function monthsShortRegex$1(isStrict) {
  if (this._monthsParseExact) {
    if (!hasOwnProp$1(this, "_monthsRegex")) {
      computeMonthsParse$1.call(this);
    }
    if (isStrict) {
      return this._monthsShortStrictRegex;
    } else {
      return this._monthsShortRegex;
    }
  } else {
    if (!hasOwnProp$1(this, "_monthsShortRegex")) {
      this._monthsShortRegex = defaultMonthsShortRegex$1;
    }
    return this._monthsShortStrictRegex && isStrict ? this._monthsShortStrictRegex : this._monthsShortRegex;
  }
}
function monthsRegex$1(isStrict) {
  if (this._monthsParseExact) {
    if (!hasOwnProp$1(this, "_monthsRegex")) {
      computeMonthsParse$1.call(this);
    }
    if (isStrict) {
      return this._monthsStrictRegex;
    } else {
      return this._monthsRegex;
    }
  } else {
    if (!hasOwnProp$1(this, "_monthsRegex")) {
      this._monthsRegex = defaultMonthsRegex$1;
    }
    return this._monthsStrictRegex && isStrict ? this._monthsStrictRegex : this._monthsRegex;
  }
}
function computeMonthsParse$1() {
  function cmpLenRev(a2, b2) {
    return b2.length - a2.length;
  }
  var shortPieces = [], longPieces = [], mixedPieces = [], i2, mom, shortP, longP;
  for (i2 = 0; i2 < 12; i2++) {
    mom = createUTC$1([2e3, i2]);
    shortP = regexEscape$1(this.monthsShort(mom, ""));
    longP = regexEscape$1(this.months(mom, ""));
    shortPieces.push(shortP);
    longPieces.push(longP);
    mixedPieces.push(longP);
    mixedPieces.push(shortP);
  }
  shortPieces.sort(cmpLenRev);
  longPieces.sort(cmpLenRev);
  mixedPieces.sort(cmpLenRev);
  this._monthsRegex = new RegExp("^(" + mixedPieces.join("|") + ")", "i");
  this._monthsShortRegex = this._monthsRegex;
  this._monthsStrictRegex = new RegExp(
    "^(" + longPieces.join("|") + ")",
    "i"
  );
  this._monthsShortStrictRegex = new RegExp(
    "^(" + shortPieces.join("|") + ")",
    "i"
  );
}
function createDate$1(y2, m2, d2, h2, M2, s2, ms) {
  var date;
  if (y2 < 100 && y2 >= 0) {
    date = new Date(y2 + 400, m2, d2, h2, M2, s2, ms);
    if (isFinite(date.getFullYear())) {
      date.setFullYear(y2);
    }
  } else {
    date = new Date(y2, m2, d2, h2, M2, s2, ms);
  }
  return date;
}
function createUTCDate$1(y2) {
  var date, args;
  if (y2 < 100 && y2 >= 0) {
    args = Array.prototype.slice.call(arguments);
    args[0] = y2 + 400;
    date = new Date(Date.UTC.apply(null, args));
    if (isFinite(date.getUTCFullYear())) {
      date.setUTCFullYear(y2);
    }
  } else {
    date = new Date(Date.UTC.apply(null, arguments));
  }
  return date;
}
function firstWeekOffset$1(year, dow, doy) {
  var fwd = 7 + dow - doy, fwdlw = (7 + createUTCDate$1(year, 0, fwd).getUTCDay() - dow) % 7;
  return -fwdlw + fwd - 1;
}
function dayOfYearFromWeeks$1(year, week, weekday, dow, doy) {
  var localWeekday = (7 + weekday - dow) % 7, weekOffset = firstWeekOffset$1(year, dow, doy), dayOfYear = 1 + 7 * (week - 1) + localWeekday + weekOffset, resYear, resDayOfYear;
  if (dayOfYear <= 0) {
    resYear = year - 1;
    resDayOfYear = daysInYear$1(resYear) + dayOfYear;
  } else if (dayOfYear > daysInYear$1(year)) {
    resYear = year + 1;
    resDayOfYear = dayOfYear - daysInYear$1(year);
  } else {
    resYear = year;
    resDayOfYear = dayOfYear;
  }
  return {
    year: resYear,
    dayOfYear: resDayOfYear
  };
}
function weekOfYear$1(mom, dow, doy) {
  var weekOffset = firstWeekOffset$1(mom.year(), dow, doy), week = Math.floor((mom.dayOfYear() - weekOffset - 1) / 7) + 1, resWeek, resYear;
  if (week < 1) {
    resYear = mom.year() - 1;
    resWeek = week + weeksInYear$1(resYear, dow, doy);
  } else if (week > weeksInYear$1(mom.year(), dow, doy)) {
    resWeek = week - weeksInYear$1(mom.year(), dow, doy);
    resYear = mom.year() + 1;
  } else {
    resYear = mom.year();
    resWeek = week;
  }
  return {
    week: resWeek,
    year: resYear
  };
}
function weeksInYear$1(year, dow, doy) {
  var weekOffset = firstWeekOffset$1(year, dow, doy), weekOffsetNext = firstWeekOffset$1(year + 1, dow, doy);
  return (daysInYear$1(year) - weekOffset + weekOffsetNext) / 7;
}
addFormatToken$1("w", ["ww", 2], "wo", "week");
addFormatToken$1("W", ["WW", 2], "Wo", "isoWeek");
addRegexToken$1("w", match1to2$1, match1to2NoLeadingZero$1);
addRegexToken$1("ww", match1to2$1, match2$1);
addRegexToken$1("W", match1to2$1, match1to2NoLeadingZero$1);
addRegexToken$1("WW", match1to2$1, match2$1);
addWeekParseToken$1(
  ["w", "ww", "W", "WW"],
  function(input, week, config2, token2) {
    week[token2.substr(0, 1)] = toInt$1(input);
  }
);
function localeWeek$1(mom) {
  return weekOfYear$1(mom, this._week.dow, this._week.doy).week;
}
var defaultLocaleWeek$1 = {
  dow: 0,
  // Sunday is the first day of the week.
  doy: 6
  // The week that contains Jan 6th is the first week of the year.
};
function localeFirstDayOfWeek$1() {
  return this._week.dow;
}
function localeFirstDayOfYear$1() {
  return this._week.doy;
}
function getSetWeek$1(input) {
  var week = this.localeData().week(this);
  return input == null ? week : this.add((input - week) * 7, "d");
}
function getSetISOWeek$1(input) {
  var week = weekOfYear$1(this, 1, 4).week;
  return input == null ? week : this.add((input - week) * 7, "d");
}
addFormatToken$1("d", 0, "do", "day");
addFormatToken$1("dd", 0, 0, function(format2) {
  return this.localeData().weekdaysMin(this, format2);
});
addFormatToken$1("ddd", 0, 0, function(format2) {
  return this.localeData().weekdaysShort(this, format2);
});
addFormatToken$1("dddd", 0, 0, function(format2) {
  return this.localeData().weekdays(this, format2);
});
addFormatToken$1("e", 0, 0, "weekday");
addFormatToken$1("E", 0, 0, "isoWeekday");
addRegexToken$1("d", match1to2$1);
addRegexToken$1("e", match1to2$1);
addRegexToken$1("E", match1to2$1);
addRegexToken$1("dd", function(isStrict, locale2) {
  return locale2.weekdaysMinRegex(isStrict);
});
addRegexToken$1("ddd", function(isStrict, locale2) {
  return locale2.weekdaysShortRegex(isStrict);
});
addRegexToken$1("dddd", function(isStrict, locale2) {
  return locale2.weekdaysRegex(isStrict);
});
addWeekParseToken$1(["dd", "ddd", "dddd"], function(input, week, config2, token2) {
  var weekday = config2._locale.weekdaysParse(input, token2, config2._strict);
  if (weekday != null) {
    week.d = weekday;
  } else {
    getParsingFlags$1(config2).invalidWeekday = input;
  }
});
addWeekParseToken$1(["d", "e", "E"], function(input, week, config2, token2) {
  week[token2] = toInt$1(input);
});
function parseWeekday$1(input, locale2) {
  if (typeof input !== "string") {
    return input;
  }
  if (!isNaN(input)) {
    return parseInt(input, 10);
  }
  input = locale2.weekdaysParse(input);
  if (typeof input === "number") {
    return input;
  }
  return null;
}
function parseIsoWeekday$1(input, locale2) {
  if (typeof input === "string") {
    return locale2.weekdaysParse(input) % 7 || 7;
  }
  return isNaN(input) ? null : input;
}
function shiftWeekdays$1(ws, n2) {
  return ws.slice(n2, 7).concat(ws.slice(0, n2));
}
var defaultLocaleWeekdays$1 = "Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday".split("_"), defaultLocaleWeekdaysShort$1 = "Sun_Mon_Tue_Wed_Thu_Fri_Sat".split("_"), defaultLocaleWeekdaysMin$1 = "Su_Mo_Tu_We_Th_Fr_Sa".split("_"), defaultWeekdaysRegex$1 = matchWord$1, defaultWeekdaysShortRegex$1 = matchWord$1, defaultWeekdaysMinRegex$1 = matchWord$1;
function localeWeekdays$1(m2, format2) {
  var weekdays = isArray$7(this._weekdays) ? this._weekdays : this._weekdays[m2 && m2 !== true && this._weekdays.isFormat.test(format2) ? "format" : "standalone"];
  return m2 === true ? shiftWeekdays$1(weekdays, this._week.dow) : m2 ? weekdays[m2.day()] : weekdays;
}
function localeWeekdaysShort$1(m2) {
  return m2 === true ? shiftWeekdays$1(this._weekdaysShort, this._week.dow) : m2 ? this._weekdaysShort[m2.day()] : this._weekdaysShort;
}
function localeWeekdaysMin$1(m2) {
  return m2 === true ? shiftWeekdays$1(this._weekdaysMin, this._week.dow) : m2 ? this._weekdaysMin[m2.day()] : this._weekdaysMin;
}
function handleStrictParse$1$1(weekdayName, format2, strict) {
  var i2, ii, mom, llc = weekdayName.toLocaleLowerCase();
  if (!this._weekdaysParse) {
    this._weekdaysParse = [];
    this._shortWeekdaysParse = [];
    this._minWeekdaysParse = [];
    for (i2 = 0; i2 < 7; ++i2) {
      mom = createUTC$1([2e3, 1]).day(i2);
      this._minWeekdaysParse[i2] = this.weekdaysMin(
        mom,
        ""
      ).toLocaleLowerCase();
      this._shortWeekdaysParse[i2] = this.weekdaysShort(
        mom,
        ""
      ).toLocaleLowerCase();
      this._weekdaysParse[i2] = this.weekdays(mom, "").toLocaleLowerCase();
    }
  }
  if (strict) {
    if (format2 === "dddd") {
      ii = indexOf$4.call(this._weekdaysParse, llc);
      return ii !== -1 ? ii : null;
    } else if (format2 === "ddd") {
      ii = indexOf$4.call(this._shortWeekdaysParse, llc);
      return ii !== -1 ? ii : null;
    } else {
      ii = indexOf$4.call(this._minWeekdaysParse, llc);
      return ii !== -1 ? ii : null;
    }
  } else {
    if (format2 === "dddd") {
      ii = indexOf$4.call(this._weekdaysParse, llc);
      if (ii !== -1) {
        return ii;
      }
      ii = indexOf$4.call(this._shortWeekdaysParse, llc);
      if (ii !== -1) {
        return ii;
      }
      ii = indexOf$4.call(this._minWeekdaysParse, llc);
      return ii !== -1 ? ii : null;
    } else if (format2 === "ddd") {
      ii = indexOf$4.call(this._shortWeekdaysParse, llc);
      if (ii !== -1) {
        return ii;
      }
      ii = indexOf$4.call(this._weekdaysParse, llc);
      if (ii !== -1) {
        return ii;
      }
      ii = indexOf$4.call(this._minWeekdaysParse, llc);
      return ii !== -1 ? ii : null;
    } else {
      ii = indexOf$4.call(this._minWeekdaysParse, llc);
      if (ii !== -1) {
        return ii;
      }
      ii = indexOf$4.call(this._weekdaysParse, llc);
      if (ii !== -1) {
        return ii;
      }
      ii = indexOf$4.call(this._shortWeekdaysParse, llc);
      return ii !== -1 ? ii : null;
    }
  }
}
function localeWeekdaysParse$1(weekdayName, format2, strict) {
  var i2, mom, regex2;
  if (this._weekdaysParseExact) {
    return handleStrictParse$1$1.call(this, weekdayName, format2, strict);
  }
  if (!this._weekdaysParse) {
    this._weekdaysParse = [];
    this._minWeekdaysParse = [];
    this._shortWeekdaysParse = [];
    this._fullWeekdaysParse = [];
  }
  for (i2 = 0; i2 < 7; i2++) {
    mom = createUTC$1([2e3, 1]).day(i2);
    if (strict && !this._fullWeekdaysParse[i2]) {
      this._fullWeekdaysParse[i2] = new RegExp(
        "^" + this.weekdays(mom, "").replace(".", "\\.?") + "$",
        "i"
      );
      this._shortWeekdaysParse[i2] = new RegExp(
        "^" + this.weekdaysShort(mom, "").replace(".", "\\.?") + "$",
        "i"
      );
      this._minWeekdaysParse[i2] = new RegExp(
        "^" + this.weekdaysMin(mom, "").replace(".", "\\.?") + "$",
        "i"
      );
    }
    if (!this._weekdaysParse[i2]) {
      regex2 = "^" + this.weekdays(mom, "") + "|^" + this.weekdaysShort(mom, "") + "|^" + this.weekdaysMin(mom, "");
      this._weekdaysParse[i2] = new RegExp(regex2.replace(".", ""), "i");
    }
    if (strict && format2 === "dddd" && this._fullWeekdaysParse[i2].test(weekdayName)) {
      return i2;
    } else if (strict && format2 === "ddd" && this._shortWeekdaysParse[i2].test(weekdayName)) {
      return i2;
    } else if (strict && format2 === "dd" && this._minWeekdaysParse[i2].test(weekdayName)) {
      return i2;
    } else if (!strict && this._weekdaysParse[i2].test(weekdayName)) {
      return i2;
    }
  }
}
function getSetDayOfWeek$1(input) {
  if (!this.isValid()) {
    return input != null ? this : NaN;
  }
  var day = get$4(this, "Day");
  if (input != null) {
    input = parseWeekday$1(input, this.localeData());
    return this.add(input - day, "d");
  } else {
    return day;
  }
}
function getSetLocaleDayOfWeek$1(input) {
  if (!this.isValid()) {
    return input != null ? this : NaN;
  }
  var weekday = (this.day() + 7 - this.localeData()._week.dow) % 7;
  return input == null ? weekday : this.add(input - weekday, "d");
}
function getSetISODayOfWeek$1(input) {
  if (!this.isValid()) {
    return input != null ? this : NaN;
  }
  if (input != null) {
    var weekday = parseIsoWeekday$1(input, this.localeData());
    return this.day(this.day() % 7 ? weekday : weekday - 7);
  } else {
    return this.day() || 7;
  }
}
function weekdaysRegex$1(isStrict) {
  if (this._weekdaysParseExact) {
    if (!hasOwnProp$1(this, "_weekdaysRegex")) {
      computeWeekdaysParse$1.call(this);
    }
    if (isStrict) {
      return this._weekdaysStrictRegex;
    } else {
      return this._weekdaysRegex;
    }
  } else {
    if (!hasOwnProp$1(this, "_weekdaysRegex")) {
      this._weekdaysRegex = defaultWeekdaysRegex$1;
    }
    return this._weekdaysStrictRegex && isStrict ? this._weekdaysStrictRegex : this._weekdaysRegex;
  }
}
function weekdaysShortRegex$1(isStrict) {
  if (this._weekdaysParseExact) {
    if (!hasOwnProp$1(this, "_weekdaysRegex")) {
      computeWeekdaysParse$1.call(this);
    }
    if (isStrict) {
      return this._weekdaysShortStrictRegex;
    } else {
      return this._weekdaysShortRegex;
    }
  } else {
    if (!hasOwnProp$1(this, "_weekdaysShortRegex")) {
      this._weekdaysShortRegex = defaultWeekdaysShortRegex$1;
    }
    return this._weekdaysShortStrictRegex && isStrict ? this._weekdaysShortStrictRegex : this._weekdaysShortRegex;
  }
}
function weekdaysMinRegex$1(isStrict) {
  if (this._weekdaysParseExact) {
    if (!hasOwnProp$1(this, "_weekdaysRegex")) {
      computeWeekdaysParse$1.call(this);
    }
    if (isStrict) {
      return this._weekdaysMinStrictRegex;
    } else {
      return this._weekdaysMinRegex;
    }
  } else {
    if (!hasOwnProp$1(this, "_weekdaysMinRegex")) {
      this._weekdaysMinRegex = defaultWeekdaysMinRegex$1;
    }
    return this._weekdaysMinStrictRegex && isStrict ? this._weekdaysMinStrictRegex : this._weekdaysMinRegex;
  }
}
function computeWeekdaysParse$1() {
  function cmpLenRev(a2, b2) {
    return b2.length - a2.length;
  }
  var minPieces = [], shortPieces = [], longPieces = [], mixedPieces = [], i2, mom, minp, shortp, longp;
  for (i2 = 0; i2 < 7; i2++) {
    mom = createUTC$1([2e3, 1]).day(i2);
    minp = regexEscape$1(this.weekdaysMin(mom, ""));
    shortp = regexEscape$1(this.weekdaysShort(mom, ""));
    longp = regexEscape$1(this.weekdays(mom, ""));
    minPieces.push(minp);
    shortPieces.push(shortp);
    longPieces.push(longp);
    mixedPieces.push(minp);
    mixedPieces.push(shortp);
    mixedPieces.push(longp);
  }
  minPieces.sort(cmpLenRev);
  shortPieces.sort(cmpLenRev);
  longPieces.sort(cmpLenRev);
  mixedPieces.sort(cmpLenRev);
  this._weekdaysRegex = new RegExp("^(" + mixedPieces.join("|") + ")", "i");
  this._weekdaysShortRegex = this._weekdaysRegex;
  this._weekdaysMinRegex = this._weekdaysRegex;
  this._weekdaysStrictRegex = new RegExp(
    "^(" + longPieces.join("|") + ")",
    "i"
  );
  this._weekdaysShortStrictRegex = new RegExp(
    "^(" + shortPieces.join("|") + ")",
    "i"
  );
  this._weekdaysMinStrictRegex = new RegExp(
    "^(" + minPieces.join("|") + ")",
    "i"
  );
}
function hFormat$1() {
  return this.hours() % 12 || 12;
}
function kFormat$1() {
  return this.hours() || 24;
}
addFormatToken$1("H", ["HH", 2], 0, "hour");
addFormatToken$1("h", ["hh", 2], 0, hFormat$1);
addFormatToken$1("k", ["kk", 2], 0, kFormat$1);
addFormatToken$1("hmm", 0, 0, function() {
  return "" + hFormat$1.apply(this) + zeroFill$1(this.minutes(), 2);
});
addFormatToken$1("hmmss", 0, 0, function() {
  return "" + hFormat$1.apply(this) + zeroFill$1(this.minutes(), 2) + zeroFill$1(this.seconds(), 2);
});
addFormatToken$1("Hmm", 0, 0, function() {
  return "" + this.hours() + zeroFill$1(this.minutes(), 2);
});
addFormatToken$1("Hmmss", 0, 0, function() {
  return "" + this.hours() + zeroFill$1(this.minutes(), 2) + zeroFill$1(this.seconds(), 2);
});
function meridiem$1(token2, lowercase) {
  addFormatToken$1(token2, 0, 0, function() {
    return this.localeData().meridiem(
      this.hours(),
      this.minutes(),
      lowercase
    );
  });
}
meridiem$1("a", true);
meridiem$1("A", false);
function matchMeridiem$1(isStrict, locale2) {
  return locale2._meridiemParse;
}
addRegexToken$1("a", matchMeridiem$1);
addRegexToken$1("A", matchMeridiem$1);
addRegexToken$1("H", match1to2$1, match1to2HasZero$1);
addRegexToken$1("h", match1to2$1, match1to2NoLeadingZero$1);
addRegexToken$1("k", match1to2$1, match1to2NoLeadingZero$1);
addRegexToken$1("HH", match1to2$1, match2$1);
addRegexToken$1("hh", match1to2$1, match2$1);
addRegexToken$1("kk", match1to2$1, match2$1);
addRegexToken$1("hmm", match3to4$1);
addRegexToken$1("hmmss", match5to6$1);
addRegexToken$1("Hmm", match3to4$1);
addRegexToken$1("Hmmss", match5to6$1);
addParseToken$1(["H", "HH"], HOUR$1);
addParseToken$1(["k", "kk"], function(input, array2, config2) {
  var kInput = toInt$1(input);
  array2[HOUR$1] = kInput === 24 ? 0 : kInput;
});
addParseToken$1(["a", "A"], function(input, array2, config2) {
  config2._isPm = config2._locale.isPM(input);
  config2._meridiem = input;
});
addParseToken$1(["h", "hh"], function(input, array2, config2) {
  array2[HOUR$1] = toInt$1(input);
  getParsingFlags$1(config2).bigHour = true;
});
addParseToken$1("hmm", function(input, array2, config2) {
  var pos = input.length - 2;
  array2[HOUR$1] = toInt$1(input.substr(0, pos));
  array2[MINUTE$1] = toInt$1(input.substr(pos));
  getParsingFlags$1(config2).bigHour = true;
});
addParseToken$1("hmmss", function(input, array2, config2) {
  var pos1 = input.length - 4, pos2 = input.length - 2;
  array2[HOUR$1] = toInt$1(input.substr(0, pos1));
  array2[MINUTE$1] = toInt$1(input.substr(pos1, 2));
  array2[SECOND$1] = toInt$1(input.substr(pos2));
  getParsingFlags$1(config2).bigHour = true;
});
addParseToken$1("Hmm", function(input, array2, config2) {
  var pos = input.length - 2;
  array2[HOUR$1] = toInt$1(input.substr(0, pos));
  array2[MINUTE$1] = toInt$1(input.substr(pos));
});
addParseToken$1("Hmmss", function(input, array2, config2) {
  var pos1 = input.length - 4, pos2 = input.length - 2;
  array2[HOUR$1] = toInt$1(input.substr(0, pos1));
  array2[MINUTE$1] = toInt$1(input.substr(pos1, 2));
  array2[SECOND$1] = toInt$1(input.substr(pos2));
});
function localeIsPM$1(input) {
  return (input + "").toLowerCase().charAt(0) === "p";
}
var defaultLocaleMeridiemParse$1 = /[ap]\.?m?\.?/i, getSetHour$1 = makeGetSet$1("Hours", true);
function localeMeridiem$1(hours2, minutes2, isLower) {
  if (hours2 > 11) {
    return isLower ? "pm" : "PM";
  } else {
    return isLower ? "am" : "AM";
  }
}
var baseConfig$1 = {
  calendar: defaultCalendar$1,
  longDateFormat: defaultLongDateFormat$1,
  invalidDate: defaultInvalidDate$1,
  ordinal: defaultOrdinal$1,
  dayOfMonthOrdinalParse: defaultDayOfMonthOrdinalParse$1,
  relativeTime: defaultRelativeTime$1,
  months: defaultLocaleMonths$1,
  monthsShort: defaultLocaleMonthsShort$1,
  week: defaultLocaleWeek$1,
  weekdays: defaultLocaleWeekdays$1,
  weekdaysMin: defaultLocaleWeekdaysMin$1,
  weekdaysShort: defaultLocaleWeekdaysShort$1,
  meridiemParse: defaultLocaleMeridiemParse$1
};
var locales$1 = {}, localeFamilies$1 = {}, globalLocale$1;
function commonPrefix$1(arr1, arr2) {
  var i2, minl = Math.min(arr1.length, arr2.length);
  for (i2 = 0; i2 < minl; i2 += 1) {
    if (arr1[i2] !== arr2[i2]) {
      return i2;
    }
  }
  return minl;
}
function normalizeLocale$1(key) {
  return key ? key.toLowerCase().replace("_", "-") : key;
}
function chooseLocale$1(names) {
  var i2 = 0, j2, next, locale2, split2;
  while (i2 < names.length) {
    split2 = normalizeLocale$1(names[i2]).split("-");
    j2 = split2.length;
    next = normalizeLocale$1(names[i2 + 1]);
    next = next ? next.split("-") : null;
    while (j2 > 0) {
      locale2 = loadLocale$1(split2.slice(0, j2).join("-"));
      if (locale2) {
        return locale2;
      }
      if (next && next.length >= j2 && commonPrefix$1(split2, next) >= j2 - 1) {
        break;
      }
      j2--;
    }
    i2++;
  }
  return globalLocale$1;
}
function isLocaleNameSane$1(name) {
  return !!(name && name.match("^[^/\\\\]*$"));
}
function loadLocale$1(name) {
  var oldLocale = null, aliasedRequire;
  if (locales$1[name] === void 0 && typeof module !== "undefined" && module && module.exports && isLocaleNameSane$1(name)) {
    try {
      oldLocale = globalLocale$1._abbr;
      aliasedRequire = require;
      aliasedRequire("./locale/" + name);
      getSetGlobalLocale$1(oldLocale);
    } catch (e2) {
      locales$1[name] = null;
    }
  }
  return locales$1[name];
}
function getSetGlobalLocale$1(key, values2) {
  var data2;
  if (key) {
    if (isUndefined$3(values2)) {
      data2 = getLocale$1(key);
    } else {
      data2 = defineLocale$1(key, values2);
    }
    if (data2) {
      globalLocale$1 = data2;
    } else {
      if (typeof console !== "undefined" && console.warn) {
        console.warn(
          "Locale " + key + " not found. Did you forget to load it?"
        );
      }
    }
  }
  return globalLocale$1._abbr;
}
function defineLocale$1(name, config2) {
  if (config2 !== null) {
    var locale2, parentConfig = baseConfig$1;
    config2.abbr = name;
    if (locales$1[name] != null) {
      deprecateSimple$1(
        "defineLocaleOverride",
        "use moment.updateLocale(localeName, config) to change an existing locale. moment.defineLocale(localeName, config) should only be used for creating a new locale See http://momentjs.com/guides/#/warnings/define-locale/ for more info."
      );
      parentConfig = locales$1[name]._config;
    } else if (config2.parentLocale != null) {
      if (locales$1[config2.parentLocale] != null) {
        parentConfig = locales$1[config2.parentLocale]._config;
      } else {
        locale2 = loadLocale$1(config2.parentLocale);
        if (locale2 != null) {
          parentConfig = locale2._config;
        } else {
          if (!localeFamilies$1[config2.parentLocale]) {
            localeFamilies$1[config2.parentLocale] = [];
          }
          localeFamilies$1[config2.parentLocale].push({
            name,
            config: config2
          });
          return null;
        }
      }
    }
    locales$1[name] = new Locale$1(mergeConfigs$1(parentConfig, config2));
    if (localeFamilies$1[name]) {
      localeFamilies$1[name].forEach(function(x2) {
        defineLocale$1(x2.name, x2.config);
      });
    }
    getSetGlobalLocale$1(name);
    return locales$1[name];
  } else {
    delete locales$1[name];
    return null;
  }
}
function updateLocale$1(name, config2) {
  if (config2 != null) {
    var locale2, tmpLocale, parentConfig = baseConfig$1;
    if (locales$1[name] != null && locales$1[name].parentLocale != null) {
      locales$1[name].set(mergeConfigs$1(locales$1[name]._config, config2));
    } else {
      tmpLocale = loadLocale$1(name);
      if (tmpLocale != null) {
        parentConfig = tmpLocale._config;
      }
      config2 = mergeConfigs$1(parentConfig, config2);
      if (tmpLocale == null) {
        config2.abbr = name;
      }
      locale2 = new Locale$1(config2);
      locale2.parentLocale = locales$1[name];
      locales$1[name] = locale2;
    }
    getSetGlobalLocale$1(name);
  } else {
    if (locales$1[name] != null) {
      if (locales$1[name].parentLocale != null) {
        locales$1[name] = locales$1[name].parentLocale;
        if (name === getSetGlobalLocale$1()) {
          getSetGlobalLocale$1(name);
        }
      } else if (locales$1[name] != null) {
        delete locales$1[name];
      }
    }
  }
  return locales$1[name];
}
function getLocale$1(key) {
  var locale2;
  if (key && key._locale && key._locale._abbr) {
    key = key._locale._abbr;
  }
  if (!key) {
    return globalLocale$1;
  }
  if (!isArray$7(key)) {
    locale2 = loadLocale$1(key);
    if (locale2) {
      return locale2;
    }
    key = [key];
  }
  return chooseLocale$1(key);
}
function listLocales$1() {
  return keys$5(locales$1);
}
function checkOverflow$1(m2) {
  var overflow2, a2 = m2._a;
  if (a2 && getParsingFlags$1(m2).overflow === -2) {
    overflow2 = a2[MONTH$1] < 0 || a2[MONTH$1] > 11 ? MONTH$1 : a2[DATE$1] < 1 || a2[DATE$1] > daysInMonth$1(a2[YEAR$1], a2[MONTH$1]) ? DATE$1 : a2[HOUR$1] < 0 || a2[HOUR$1] > 24 || a2[HOUR$1] === 24 && (a2[MINUTE$1] !== 0 || a2[SECOND$1] !== 0 || a2[MILLISECOND$1] !== 0) ? HOUR$1 : a2[MINUTE$1] < 0 || a2[MINUTE$1] > 59 ? MINUTE$1 : a2[SECOND$1] < 0 || a2[SECOND$1] > 59 ? SECOND$1 : a2[MILLISECOND$1] < 0 || a2[MILLISECOND$1] > 999 ? MILLISECOND$1 : -1;
    if (getParsingFlags$1(m2)._overflowDayOfYear && (overflow2 < YEAR$1 || overflow2 > DATE$1)) {
      overflow2 = DATE$1;
    }
    if (getParsingFlags$1(m2)._overflowWeeks && overflow2 === -1) {
      overflow2 = WEEK$1;
    }
    if (getParsingFlags$1(m2)._overflowWeekday && overflow2 === -1) {
      overflow2 = WEEKDAY$1;
    }
    getParsingFlags$1(m2).overflow = overflow2;
  }
  return m2;
}
var extendedIsoRegex$1 = /^\s*((?:[+-]\d{6}|\d{4})-(?:\d\d-\d\d|W\d\d-\d|W\d\d|\d\d\d|\d\d))(?:(T| )(\d\d(?::\d\d(?::\d\d(?:[.,]\d+)?)?)?)([+-]\d\d(?::?\d\d)?|\s*Z)?)?$/, basicIsoRegex$1 = /^\s*((?:[+-]\d{6}|\d{4})(?:\d\d\d\d|W\d\d\d|W\d\d|\d\d\d|\d\d|))(?:(T| )(\d\d(?:\d\d(?:\d\d(?:[.,]\d+)?)?)?)([+-]\d\d(?::?\d\d)?|\s*Z)?)?$/, tzRegex$1 = /Z|[+-]\d\d(?::?\d\d)?/, isoDates$1 = [
  ["YYYYYY-MM-DD", /[+-]\d{6}-\d\d-\d\d/],
  ["YYYY-MM-DD", /\d{4}-\d\d-\d\d/],
  ["GGGG-[W]WW-E", /\d{4}-W\d\d-\d/],
  ["GGGG-[W]WW", /\d{4}-W\d\d/, false],
  ["YYYY-DDD", /\d{4}-\d{3}/],
  ["YYYY-MM", /\d{4}-\d\d/, false],
  ["YYYYYYMMDD", /[+-]\d{10}/],
  ["YYYYMMDD", /\d{8}/],
  ["GGGG[W]WWE", /\d{4}W\d{3}/],
  ["GGGG[W]WW", /\d{4}W\d{2}/, false],
  ["YYYYDDD", /\d{7}/],
  ["YYYYMM", /\d{6}/, false],
  ["YYYY", /\d{4}/, false]
], isoTimes$1 = [
  ["HH:mm:ss.SSSS", /\d\d:\d\d:\d\d\.\d+/],
  ["HH:mm:ss,SSSS", /\d\d:\d\d:\d\d,\d+/],
  ["HH:mm:ss", /\d\d:\d\d:\d\d/],
  ["HH:mm", /\d\d:\d\d/],
  ["HHmmss.SSSS", /\d\d\d\d\d\d\.\d+/],
  ["HHmmss,SSSS", /\d\d\d\d\d\d,\d+/],
  ["HHmmss", /\d\d\d\d\d\d/],
  ["HHmm", /\d\d\d\d/],
  ["HH", /\d\d/]
], aspNetJsonRegex$1 = /^\/?Date\((-?\d+)/i, rfc2822$1 = /^(?:(Mon|Tue|Wed|Thu|Fri|Sat|Sun),?\s)?(\d{1,2})\s(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s(\d{2,4})\s(\d\d):(\d\d)(?::(\d\d))?\s(?:(UT|GMT|[ECMP][SD]T)|([Zz])|([+-]\d{4}))$/, obsOffsets$1 = {
  UT: 0,
  GMT: 0,
  EDT: -4 * 60,
  EST: -5 * 60,
  CDT: -5 * 60,
  CST: -6 * 60,
  MDT: -6 * 60,
  MST: -7 * 60,
  PDT: -7 * 60,
  PST: -8 * 60
};
function configFromISO$1(config2) {
  var i2, l2, string = config2._i, match5 = extendedIsoRegex$1.exec(string) || basicIsoRegex$1.exec(string), allowTime, dateFormat, timeFormat, tzFormat, isoDatesLen = isoDates$1.length, isoTimesLen = isoTimes$1.length;
  if (match5) {
    getParsingFlags$1(config2).iso = true;
    for (i2 = 0, l2 = isoDatesLen; i2 < l2; i2++) {
      if (isoDates$1[i2][1].exec(match5[1])) {
        dateFormat = isoDates$1[i2][0];
        allowTime = isoDates$1[i2][2] !== false;
        break;
      }
    }
    if (dateFormat == null) {
      config2._isValid = false;
      return;
    }
    if (match5[3]) {
      for (i2 = 0, l2 = isoTimesLen; i2 < l2; i2++) {
        if (isoTimes$1[i2][1].exec(match5[3])) {
          timeFormat = (match5[2] || " ") + isoTimes$1[i2][0];
          break;
        }
      }
      if (timeFormat == null) {
        config2._isValid = false;
        return;
      }
    }
    if (!allowTime && timeFormat != null) {
      config2._isValid = false;
      return;
    }
    if (match5[4]) {
      if (tzRegex$1.exec(match5[4])) {
        tzFormat = "Z";
      } else {
        config2._isValid = false;
        return;
      }
    }
    config2._f = dateFormat + (timeFormat || "") + (tzFormat || "");
    configFromStringAndFormat$1(config2);
  } else {
    config2._isValid = false;
  }
}
function extractFromRFC2822Strings$1(yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr) {
  var result = [
    untruncateYear$1(yearStr),
    defaultLocaleMonthsShort$1.indexOf(monthStr),
    parseInt(dayStr, 10),
    parseInt(hourStr, 10),
    parseInt(minuteStr, 10)
  ];
  if (secondStr) {
    result.push(parseInt(secondStr, 10));
  }
  return result;
}
function untruncateYear$1(yearStr) {
  var year = parseInt(yearStr, 10);
  if (year <= 49) {
    return 2e3 + year;
  } else if (year <= 999) {
    return 1900 + year;
  }
  return year;
}
function preprocessRFC2822$1(s2) {
  return s2.replace(/\([^()]*\)|[\n\t]/g, " ").replace(/(\s\s+)/g, " ").replace(/^\s\s*/, "").replace(/\s\s*$/, "");
}
function checkWeekday$1(weekdayStr, parsedInput, config2) {
  if (weekdayStr) {
    var weekdayProvided = defaultLocaleWeekdaysShort$1.indexOf(weekdayStr), weekdayActual = new Date(
      parsedInput[0],
      parsedInput[1],
      parsedInput[2]
    ).getDay();
    if (weekdayProvided !== weekdayActual) {
      getParsingFlags$1(config2).weekdayMismatch = true;
      config2._isValid = false;
      return false;
    }
  }
  return true;
}
function calculateOffset$1(obsOffset, militaryOffset, numOffset) {
  if (obsOffset) {
    return obsOffsets$1[obsOffset];
  } else if (militaryOffset) {
    return 0;
  } else {
    var hm = parseInt(numOffset, 10), m2 = hm % 100, h2 = (hm - m2) / 100;
    return h2 * 60 + m2;
  }
}
function configFromRFC2822$1(config2) {
  var match5 = rfc2822$1.exec(preprocessRFC2822$1(config2._i)), parsedArray;
  if (match5) {
    parsedArray = extractFromRFC2822Strings$1(
      match5[4],
      match5[3],
      match5[2],
      match5[5],
      match5[6],
      match5[7]
    );
    if (!checkWeekday$1(match5[1], parsedArray, config2)) {
      return;
    }
    config2._a = parsedArray;
    config2._tzm = calculateOffset$1(match5[8], match5[9], match5[10]);
    config2._d = createUTCDate$1.apply(null, config2._a);
    config2._d.setUTCMinutes(config2._d.getUTCMinutes() - config2._tzm);
    getParsingFlags$1(config2).rfc2822 = true;
  } else {
    config2._isValid = false;
  }
}
function configFromString$1(config2) {
  var matched = aspNetJsonRegex$1.exec(config2._i);
  if (matched !== null) {
    config2._d = /* @__PURE__ */ new Date(+matched[1]);
    return;
  }
  configFromISO$1(config2);
  if (config2._isValid === false) {
    delete config2._isValid;
  } else {
    return;
  }
  configFromRFC2822$1(config2);
  if (config2._isValid === false) {
    delete config2._isValid;
  } else {
    return;
  }
  if (config2._strict) {
    config2._isValid = false;
  } else {
    hooks$2.createFromInputFallback(config2);
  }
}
hooks$2.createFromInputFallback = deprecate$1(
  "value provided is not in a recognized RFC2822 or ISO format. moment construction falls back to js Date(), which is not reliable across all browsers and versions. Non RFC2822/ISO date formats are discouraged. Please refer to http://momentjs.com/guides/#/warnings/js-date/ for more info.",
  function(config2) {
    config2._d = /* @__PURE__ */ new Date(config2._i + (config2._useUTC ? " UTC" : ""));
  }
);
function defaults$6(a2, b2, c2) {
  if (a2 != null) {
    return a2;
  }
  if (b2 != null) {
    return b2;
  }
  return c2;
}
function currentDateArray$1(config2) {
  var nowValue = new Date(hooks$2.now());
  if (config2._useUTC) {
    return [
      nowValue.getUTCFullYear(),
      nowValue.getUTCMonth(),
      nowValue.getUTCDate()
    ];
  }
  return [nowValue.getFullYear(), nowValue.getMonth(), nowValue.getDate()];
}
function configFromArray$1(config2) {
  var i2, date, input = [], currentDate, expectedWeekday, yearToUse;
  if (config2._d) {
    return;
  }
  currentDate = currentDateArray$1(config2);
  if (config2._w && config2._a[DATE$1] == null && config2._a[MONTH$1] == null) {
    dayOfYearFromWeekInfo$1(config2);
  }
  if (config2._dayOfYear != null) {
    yearToUse = defaults$6(config2._a[YEAR$1], currentDate[YEAR$1]);
    if (config2._dayOfYear > daysInYear$1(yearToUse) || config2._dayOfYear === 0) {
      getParsingFlags$1(config2)._overflowDayOfYear = true;
    }
    date = createUTCDate$1(yearToUse, 0, config2._dayOfYear);
    config2._a[MONTH$1] = date.getUTCMonth();
    config2._a[DATE$1] = date.getUTCDate();
  }
  for (i2 = 0; i2 < 3 && config2._a[i2] == null; ++i2) {
    config2._a[i2] = input[i2] = currentDate[i2];
  }
  for (; i2 < 7; i2++) {
    config2._a[i2] = input[i2] = config2._a[i2] == null ? i2 === 2 ? 1 : 0 : config2._a[i2];
  }
  if (config2._a[HOUR$1] === 24 && config2._a[MINUTE$1] === 0 && config2._a[SECOND$1] === 0 && config2._a[MILLISECOND$1] === 0) {
    config2._nextDay = true;
    config2._a[HOUR$1] = 0;
  }
  config2._d = (config2._useUTC ? createUTCDate$1 : createDate$1).apply(
    null,
    input
  );
  expectedWeekday = config2._useUTC ? config2._d.getUTCDay() : config2._d.getDay();
  if (config2._tzm != null) {
    config2._d.setUTCMinutes(config2._d.getUTCMinutes() - config2._tzm);
  }
  if (config2._nextDay) {
    config2._a[HOUR$1] = 24;
  }
  if (config2._w && typeof config2._w.d !== "undefined" && config2._w.d !== expectedWeekday) {
    getParsingFlags$1(config2).weekdayMismatch = true;
  }
}
function dayOfYearFromWeekInfo$1(config2) {
  var w2, weekYear, week, weekday, dow, doy, temp, weekdayOverflow, curWeek;
  w2 = config2._w;
  if (w2.GG != null || w2.W != null || w2.E != null) {
    dow = 1;
    doy = 4;
    weekYear = defaults$6(
      w2.GG,
      config2._a[YEAR$1],
      weekOfYear$1(createLocal$1(), 1, 4).year
    );
    week = defaults$6(w2.W, 1);
    weekday = defaults$6(w2.E, 1);
    if (weekday < 1 || weekday > 7) {
      weekdayOverflow = true;
    }
  } else {
    dow = config2._locale._week.dow;
    doy = config2._locale._week.doy;
    curWeek = weekOfYear$1(createLocal$1(), dow, doy);
    weekYear = defaults$6(w2.gg, config2._a[YEAR$1], curWeek.year);
    week = defaults$6(w2.w, curWeek.week);
    if (w2.d != null) {
      weekday = w2.d;
      if (weekday < 0 || weekday > 6) {
        weekdayOverflow = true;
      }
    } else if (w2.e != null) {
      weekday = w2.e + dow;
      if (w2.e < 0 || w2.e > 6) {
        weekdayOverflow = true;
      }
    } else {
      weekday = dow;
    }
  }
  if (week < 1 || week > weeksInYear$1(weekYear, dow, doy)) {
    getParsingFlags$1(config2)._overflowWeeks = true;
  } else if (weekdayOverflow != null) {
    getParsingFlags$1(config2)._overflowWeekday = true;
  } else {
    temp = dayOfYearFromWeeks$1(weekYear, week, weekday, dow, doy);
    config2._a[YEAR$1] = temp.year;
    config2._dayOfYear = temp.dayOfYear;
  }
}
hooks$2.ISO_8601 = function() {
};
hooks$2.RFC_2822 = function() {
};
function configFromStringAndFormat$1(config2) {
  if (config2._f === hooks$2.ISO_8601) {
    configFromISO$1(config2);
    return;
  }
  if (config2._f === hooks$2.RFC_2822) {
    configFromRFC2822$1(config2);
    return;
  }
  config2._a = [];
  getParsingFlags$1(config2).empty = true;
  var string = "" + config2._i, i2, parsedInput, tokens2, token2, skipped, stringLength = string.length, totalParsedInputLength = 0, era, tokenLen;
  tokens2 = expandFormat$1(config2._f, config2._locale).match(formattingTokens$1) || [];
  tokenLen = tokens2.length;
  for (i2 = 0; i2 < tokenLen; i2++) {
    token2 = tokens2[i2];
    parsedInput = (string.match(getParseRegexForToken$1(token2, config2)) || [])[0];
    if (parsedInput) {
      skipped = string.substr(0, string.indexOf(parsedInput));
      if (skipped.length > 0) {
        getParsingFlags$1(config2).unusedInput.push(skipped);
      }
      string = string.slice(
        string.indexOf(parsedInput) + parsedInput.length
      );
      totalParsedInputLength += parsedInput.length;
    }
    if (formatTokenFunctions$1[token2]) {
      if (parsedInput) {
        getParsingFlags$1(config2).empty = false;
      } else {
        getParsingFlags$1(config2).unusedTokens.push(token2);
      }
      addTimeToArrayFromToken$1(token2, parsedInput, config2);
    } else if (config2._strict && !parsedInput) {
      getParsingFlags$1(config2).unusedTokens.push(token2);
    }
  }
  getParsingFlags$1(config2).charsLeftOver = stringLength - totalParsedInputLength;
  if (string.length > 0) {
    getParsingFlags$1(config2).unusedInput.push(string);
  }
  if (config2._a[HOUR$1] <= 12 && getParsingFlags$1(config2).bigHour === true && config2._a[HOUR$1] > 0) {
    getParsingFlags$1(config2).bigHour = void 0;
  }
  getParsingFlags$1(config2).parsedDateParts = config2._a.slice(0);
  getParsingFlags$1(config2).meridiem = config2._meridiem;
  config2._a[HOUR$1] = meridiemFixWrap$1(
    config2._locale,
    config2._a[HOUR$1],
    config2._meridiem
  );
  era = getParsingFlags$1(config2).era;
  if (era !== null) {
    config2._a[YEAR$1] = config2._locale.erasConvertYear(era, config2._a[YEAR$1]);
  }
  configFromArray$1(config2);
  checkOverflow$1(config2);
}
function meridiemFixWrap$1(locale2, hour, meridiem2) {
  var isPm;
  if (meridiem2 == null) {
    return hour;
  }
  if (locale2.meridiemHour != null) {
    return locale2.meridiemHour(hour, meridiem2);
  } else if (locale2.isPM != null) {
    isPm = locale2.isPM(meridiem2);
    if (isPm && hour < 12) {
      hour += 12;
    }
    if (!isPm && hour === 12) {
      hour = 0;
    }
    return hour;
  } else {
    return hour;
  }
}
function configFromStringAndArray$1(config2) {
  var tempConfig, bestMoment, scoreToBeat, i2, currentScore, validFormatFound, bestFormatIsValid = false, configfLen = config2._f.length;
  if (configfLen === 0) {
    getParsingFlags$1(config2).invalidFormat = true;
    config2._d = /* @__PURE__ */ new Date(NaN);
    return;
  }
  for (i2 = 0; i2 < configfLen; i2++) {
    currentScore = 0;
    validFormatFound = false;
    tempConfig = copyConfig$1({}, config2);
    if (config2._useUTC != null) {
      tempConfig._useUTC = config2._useUTC;
    }
    tempConfig._f = config2._f[i2];
    configFromStringAndFormat$1(tempConfig);
    if (isValid$3(tempConfig)) {
      validFormatFound = true;
    }
    currentScore += getParsingFlags$1(tempConfig).charsLeftOver;
    currentScore += getParsingFlags$1(tempConfig).unusedTokens.length * 10;
    getParsingFlags$1(tempConfig).score = currentScore;
    if (!bestFormatIsValid) {
      if (scoreToBeat == null || currentScore < scoreToBeat || validFormatFound) {
        scoreToBeat = currentScore;
        bestMoment = tempConfig;
        if (validFormatFound) {
          bestFormatIsValid = true;
        }
      }
    } else {
      if (currentScore < scoreToBeat) {
        scoreToBeat = currentScore;
        bestMoment = tempConfig;
      }
    }
  }
  extend$6(config2, bestMoment || tempConfig);
}
function configFromObject$1(config2) {
  if (config2._d) {
    return;
  }
  var i2 = normalizeObjectUnits$1(config2._i), dayOrDate = i2.day === void 0 ? i2.date : i2.day;
  config2._a = map$3(
    [i2.year, i2.month, dayOrDate, i2.hour, i2.minute, i2.second, i2.millisecond],
    function(obj) {
      return obj && parseInt(obj, 10);
    }
  );
  configFromArray$1(config2);
}
function createFromConfig$1(config2) {
  var res = new Moment$1(checkOverflow$1(prepareConfig$1(config2)));
  if (res._nextDay) {
    res.add(1, "d");
    res._nextDay = void 0;
  }
  return res;
}
function prepareConfig$1(config2) {
  var input = config2._i, format2 = config2._f;
  config2._locale = config2._locale || getLocale$1(config2._l);
  if (input === null || format2 === void 0 && input === "") {
    return createInvalid$2({ nullInput: true });
  }
  if (typeof input === "string") {
    config2._i = input = config2._locale.preparse(input);
  }
  if (isMoment$1(input)) {
    return new Moment$1(checkOverflow$1(input));
  } else if (isDate$3(input)) {
    config2._d = input;
  } else if (isArray$7(format2)) {
    configFromStringAndArray$1(config2);
  } else if (format2) {
    configFromStringAndFormat$1(config2);
  } else {
    configFromInput$1(config2);
  }
  if (!isValid$3(config2)) {
    config2._d = null;
  }
  return config2;
}
function configFromInput$1(config2) {
  var input = config2._i;
  if (isUndefined$3(input)) {
    config2._d = new Date(hooks$2.now());
  } else if (isDate$3(input)) {
    config2._d = new Date(input.valueOf());
  } else if (typeof input === "string") {
    configFromString$1(config2);
  } else if (isArray$7(input)) {
    config2._a = map$3(input.slice(0), function(obj) {
      return parseInt(obj, 10);
    });
    configFromArray$1(config2);
  } else if (isObject$m(input)) {
    configFromObject$1(config2);
  } else if (isNumber$4(input)) {
    config2._d = new Date(input);
  } else {
    hooks$2.createFromInputFallback(config2);
  }
}
function createLocalOrUTC$1(input, format2, locale2, strict, isUTC) {
  var c2 = {};
  if (format2 === true || format2 === false) {
    strict = format2;
    format2 = void 0;
  }
  if (locale2 === true || locale2 === false) {
    strict = locale2;
    locale2 = void 0;
  }
  if (isObject$m(input) && isObjectEmpty$1(input) || isArray$7(input) && input.length === 0) {
    input = void 0;
  }
  c2._isAMomentObject = true;
  c2._useUTC = c2._isUTC = isUTC;
  c2._l = locale2;
  c2._i = input;
  c2._f = format2;
  c2._strict = strict;
  return createFromConfig$1(c2);
}
function createLocal$1(input, format2, locale2, strict) {
  return createLocalOrUTC$1(input, format2, locale2, strict, false);
}
var prototypeMin$1 = deprecate$1(
  "moment().min is deprecated, use moment.max instead. http://momentjs.com/guides/#/warnings/min-max/",
  function() {
    var other = createLocal$1.apply(null, arguments);
    if (this.isValid() && other.isValid()) {
      return other < this ? this : other;
    } else {
      return createInvalid$2();
    }
  }
), prototypeMax$1 = deprecate$1(
  "moment().max is deprecated, use moment.min instead. http://momentjs.com/guides/#/warnings/min-max/",
  function() {
    var other = createLocal$1.apply(null, arguments);
    if (this.isValid() && other.isValid()) {
      return other > this ? this : other;
    } else {
      return createInvalid$2();
    }
  }
);
function pickBy$1(fn, moments) {
  var res, i2;
  if (moments.length === 1 && isArray$7(moments[0])) {
    moments = moments[0];
  }
  if (!moments.length) {
    return createLocal$1();
  }
  res = moments[0];
  for (i2 = 1; i2 < moments.length; ++i2) {
    if (!moments[i2].isValid() || moments[i2][fn](res)) {
      res = moments[i2];
    }
  }
  return res;
}
function min$9() {
  var args = [].slice.call(arguments, 0);
  return pickBy$1("isBefore", args);
}
function max$8() {
  var args = [].slice.call(arguments, 0);
  return pickBy$1("isAfter", args);
}
var now$2 = function() {
  return Date.now ? Date.now() : +/* @__PURE__ */ new Date();
};
var ordering$1 = [
  "year",
  "quarter",
  "month",
  "week",
  "day",
  "hour",
  "minute",
  "second",
  "millisecond"
];
function isDurationValid$1(m2) {
  var key, unitHasDecimal = false, i2, orderLen = ordering$1.length;
  for (key in m2) {
    if (hasOwnProp$1(m2, key) && !(indexOf$4.call(ordering$1, key) !== -1 && (m2[key] == null || !isNaN(m2[key])))) {
      return false;
    }
  }
  for (i2 = 0; i2 < orderLen; ++i2) {
    if (m2[ordering$1[i2]]) {
      if (unitHasDecimal) {
        return false;
      }
      if (parseFloat(m2[ordering$1[i2]]) !== toInt$1(m2[ordering$1[i2]])) {
        unitHasDecimal = true;
      }
    }
  }
  return true;
}
function isValid$1$1() {
  return this._isValid;
}
function createInvalid$1$1() {
  return createDuration$1(NaN);
}
function Duration$1(duration2) {
  var normalizedInput = normalizeObjectUnits$1(duration2), years2 = normalizedInput.year || 0, quarters = normalizedInput.quarter || 0, months2 = normalizedInput.month || 0, weeks2 = normalizedInput.week || normalizedInput.isoWeek || 0, days2 = normalizedInput.day || 0, hours2 = normalizedInput.hour || 0, minutes2 = normalizedInput.minute || 0, seconds2 = normalizedInput.second || 0, milliseconds2 = normalizedInput.millisecond || 0;
  this._isValid = isDurationValid$1(normalizedInput);
  this._milliseconds = +milliseconds2 + seconds2 * 1e3 + // 1000
  minutes2 * 6e4 + // 1000 * 60
  hours2 * 1e3 * 60 * 60;
  this._days = +days2 + weeks2 * 7;
  this._months = +months2 + quarters * 3 + years2 * 12;
  this._data = {};
  this._locale = getLocale$1();
  this._bubble();
}
function isDuration$1(obj) {
  return obj instanceof Duration$1;
}
function absRound$1(number) {
  if (number < 0) {
    return Math.round(-1 * number) * -1;
  } else {
    return Math.round(number);
  }
}
function compareArrays$1(array1, array2, dontConvert) {
  var len2 = Math.min(array1.length, array2.length), lengthDiff = Math.abs(array1.length - array2.length), diffs = 0, i2;
  for (i2 = 0; i2 < len2; i2++) {
    if (dontConvert && array1[i2] !== array2[i2] || !dontConvert && toInt$1(array1[i2]) !== toInt$1(array2[i2])) {
      diffs++;
    }
  }
  return diffs + lengthDiff;
}
function offset$1(token2, separator) {
  addFormatToken$1(token2, 0, 0, function() {
    var offset2 = this.utcOffset(), sign2 = "+";
    if (offset2 < 0) {
      offset2 = -offset2;
      sign2 = "-";
    }
    return sign2 + zeroFill$1(~~(offset2 / 60), 2) + separator + zeroFill$1(~~offset2 % 60, 2);
  });
}
offset$1("Z", ":");
offset$1("ZZ", "");
addRegexToken$1("Z", matchShortOffset$1);
addRegexToken$1("ZZ", matchShortOffset$1);
addParseToken$1(["Z", "ZZ"], function(input, array2, config2) {
  config2._useUTC = true;
  config2._tzm = offsetFromString$1(matchShortOffset$1, input);
});
var chunkOffset$1 = /([\+\-]|\d\d)/gi;
function offsetFromString$1(matcher, string) {
  var matches = (string || "").match(matcher), chunk, parts, minutes2;
  if (matches === null) {
    return null;
  }
  chunk = matches[matches.length - 1] || [];
  parts = (chunk + "").match(chunkOffset$1) || ["-", 0, 0];
  minutes2 = +(parts[1] * 60) + toInt$1(parts[2]);
  return minutes2 === 0 ? 0 : parts[0] === "+" ? minutes2 : -minutes2;
}
function cloneWithOffset$1(input, model) {
  var res, diff2;
  if (model._isUTC) {
    res = model.clone();
    diff2 = (isMoment$1(input) || isDate$3(input) ? input.valueOf() : createLocal$1(input).valueOf()) - res.valueOf();
    res._d.setTime(res._d.valueOf() + diff2);
    hooks$2.updateOffset(res, false);
    return res;
  } else {
    return createLocal$1(input).local();
  }
}
function getDateOffset$1(m2) {
  return -Math.round(m2._d.getTimezoneOffset());
}
hooks$2.updateOffset = function() {
};
function getSetOffset$1(input, keepLocalTime, keepMinutes) {
  var offset2 = this._offset || 0, localAdjust;
  if (!this.isValid()) {
    return input != null ? this : NaN;
  }
  if (input != null) {
    if (typeof input === "string") {
      input = offsetFromString$1(matchShortOffset$1, input);
      if (input === null) {
        return this;
      }
    } else if (Math.abs(input) < 16 && !keepMinutes) {
      input = input * 60;
    }
    if (!this._isUTC && keepLocalTime) {
      localAdjust = getDateOffset$1(this);
    }
    this._offset = input;
    this._isUTC = true;
    if (localAdjust != null) {
      this.add(localAdjust, "m");
    }
    if (offset2 !== input) {
      if (!keepLocalTime || this._changeInProgress) {
        addSubtract$2(
          this,
          createDuration$1(input - offset2, "m"),
          1,
          false
        );
      } else if (!this._changeInProgress) {
        this._changeInProgress = true;
        hooks$2.updateOffset(this, true);
        this._changeInProgress = null;
      }
    }
    return this;
  } else {
    return this._isUTC ? offset2 : getDateOffset$1(this);
  }
}
function getSetZone$1(input, keepLocalTime) {
  if (input != null) {
    if (typeof input !== "string") {
      input = -input;
    }
    this.utcOffset(input, keepLocalTime);
    return this;
  } else {
    return -this.utcOffset();
  }
}
function setOffsetToUTC$1(keepLocalTime) {
  return this.utcOffset(0, keepLocalTime);
}
function setOffsetToLocal$1(keepLocalTime) {
  if (this._isUTC) {
    this.utcOffset(0, keepLocalTime);
    this._isUTC = false;
    if (keepLocalTime) {
      this.subtract(getDateOffset$1(this), "m");
    }
  }
  return this;
}
function setOffsetToParsedOffset$1() {
  if (this._tzm != null) {
    this.utcOffset(this._tzm, false, true);
  } else if (typeof this._i === "string") {
    var tZone = offsetFromString$1(matchOffset$1, this._i);
    if (tZone != null) {
      this.utcOffset(tZone);
    } else {
      this.utcOffset(0, true);
    }
  }
  return this;
}
function hasAlignedHourOffset$1(input) {
  if (!this.isValid()) {
    return false;
  }
  input = input ? createLocal$1(input).utcOffset() : 0;
  return (this.utcOffset() - input) % 60 === 0;
}
function isDaylightSavingTime$1() {
  return this.utcOffset() > this.clone().month(0).utcOffset() || this.utcOffset() > this.clone().month(5).utcOffset();
}
function isDaylightSavingTimeShifted$1() {
  if (!isUndefined$3(this._isDSTShifted)) {
    return this._isDSTShifted;
  }
  var c2 = {}, other;
  copyConfig$1(c2, this);
  c2 = prepareConfig$1(c2);
  if (c2._a) {
    other = c2._isUTC ? createUTC$1(c2._a) : createLocal$1(c2._a);
    this._isDSTShifted = this.isValid() && compareArrays$1(c2._a, other.toArray()) > 0;
  } else {
    this._isDSTShifted = false;
  }
  return this._isDSTShifted;
}
function isLocal$1() {
  return this.isValid() ? !this._isUTC : false;
}
function isUtcOffset$1() {
  return this.isValid() ? this._isUTC : false;
}
function isUtc$1() {
  return this.isValid() ? this._isUTC && this._offset === 0 : false;
}
var aspNetRegex$1 = /^(-|\+)?(?:(\d*)[. ])?(\d+):(\d+)(?::(\d+)(\.\d*)?)?$/, isoRegex$1 = /^(-|\+)?P(?:([-+]?[0-9,.]*)Y)?(?:([-+]?[0-9,.]*)M)?(?:([-+]?[0-9,.]*)W)?(?:([-+]?[0-9,.]*)D)?(?:T(?:([-+]?[0-9,.]*)H)?(?:([-+]?[0-9,.]*)M)?(?:([-+]?[0-9,.]*)S)?)?$/;
function createDuration$1(input, key) {
  var duration2 = input, match5 = null, sign2, ret, diffRes;
  if (isDuration$1(input)) {
    duration2 = {
      ms: input._milliseconds,
      d: input._days,
      M: input._months
    };
  } else if (isNumber$4(input) || !isNaN(+input)) {
    duration2 = {};
    if (key) {
      duration2[key] = +input;
    } else {
      duration2.milliseconds = +input;
    }
  } else if (match5 = aspNetRegex$1.exec(input)) {
    sign2 = match5[1] === "-" ? -1 : 1;
    duration2 = {
      y: 0,
      d: toInt$1(match5[DATE$1]) * sign2,
      h: toInt$1(match5[HOUR$1]) * sign2,
      m: toInt$1(match5[MINUTE$1]) * sign2,
      s: toInt$1(match5[SECOND$1]) * sign2,
      ms: toInt$1(absRound$1(match5[MILLISECOND$1] * 1e3)) * sign2
      // the millisecond decimal point is included in the match
    };
  } else if (match5 = isoRegex$1.exec(input)) {
    sign2 = match5[1] === "-" ? -1 : 1;
    duration2 = {
      y: parseIso$1(match5[2], sign2),
      M: parseIso$1(match5[3], sign2),
      w: parseIso$1(match5[4], sign2),
      d: parseIso$1(match5[5], sign2),
      h: parseIso$1(match5[6], sign2),
      m: parseIso$1(match5[7], sign2),
      s: parseIso$1(match5[8], sign2)
    };
  } else if (duration2 == null) {
    duration2 = {};
  } else if (typeof duration2 === "object" && ("from" in duration2 || "to" in duration2)) {
    diffRes = momentsDifference$1(
      createLocal$1(duration2.from),
      createLocal$1(duration2.to)
    );
    duration2 = {};
    duration2.ms = diffRes.milliseconds;
    duration2.M = diffRes.months;
  }
  ret = new Duration$1(duration2);
  if (isDuration$1(input) && hasOwnProp$1(input, "_locale")) {
    ret._locale = input._locale;
  }
  if (isDuration$1(input) && hasOwnProp$1(input, "_isValid")) {
    ret._isValid = input._isValid;
  }
  return ret;
}
createDuration$1.fn = Duration$1.prototype;
createDuration$1.invalid = createInvalid$1$1;
function parseIso$1(inp, sign2) {
  var res = inp && parseFloat(inp.replace(",", "."));
  return (isNaN(res) ? 0 : res) * sign2;
}
function positiveMomentsDifference$1(base2, other) {
  var res = {};
  res.months = other.month() - base2.month() + (other.year() - base2.year()) * 12;
  if (base2.clone().add(res.months, "M").isAfter(other)) {
    --res.months;
  }
  res.milliseconds = +other - +base2.clone().add(res.months, "M");
  return res;
}
function momentsDifference$1(base2, other) {
  var res;
  if (!(base2.isValid() && other.isValid())) {
    return { milliseconds: 0, months: 0 };
  }
  other = cloneWithOffset$1(other, base2);
  if (base2.isBefore(other)) {
    res = positiveMomentsDifference$1(base2, other);
  } else {
    res = positiveMomentsDifference$1(other, base2);
    res.milliseconds = -res.milliseconds;
    res.months = -res.months;
  }
  return res;
}
function createAdder$1(direction2, name) {
  return function(val, period) {
    var dur, tmp;
    if (period !== null && !isNaN(+period)) {
      deprecateSimple$1(
        name,
        "moment()." + name + "(period, number) is deprecated. Please use moment()." + name + "(number, period). See http://momentjs.com/guides/#/warnings/add-inverted-param/ for more info."
      );
      tmp = val;
      val = period;
      period = tmp;
    }
    dur = createDuration$1(val, period);
    addSubtract$2(this, dur, direction2);
    return this;
  };
}
function addSubtract$2(mom, duration2, isAdding, updateOffset) {
  var milliseconds2 = duration2._milliseconds, days2 = absRound$1(duration2._days), months2 = absRound$1(duration2._months);
  if (!mom.isValid()) {
    return;
  }
  updateOffset = updateOffset == null ? true : updateOffset;
  if (months2) {
    setMonth$1(mom, get$4(mom, "Month") + months2 * isAdding);
  }
  if (days2) {
    set$1$2(mom, "Date", get$4(mom, "Date") + days2 * isAdding);
  }
  if (milliseconds2) {
    mom._d.setTime(mom._d.valueOf() + milliseconds2 * isAdding);
  }
  if (updateOffset) {
    hooks$2.updateOffset(mom, days2 || months2);
  }
}
var add$3 = createAdder$1(1, "add"), subtract$2 = createAdder$1(-1, "subtract");
function isString$3(input) {
  return typeof input === "string" || input instanceof String;
}
function isMomentInput$1(input) {
  return isMoment$1(input) || isDate$3(input) || isString$3(input) || isNumber$4(input) || isNumberOrStringArray$1(input) || isMomentInputObject$1(input) || input === null || input === void 0;
}
function isMomentInputObject$1(input) {
  var objectTest = isObject$m(input) && !isObjectEmpty$1(input), propertyTest = false, properties = [
    "years",
    "year",
    "y",
    "months",
    "month",
    "M",
    "days",
    "day",
    "d",
    "dates",
    "date",
    "D",
    "hours",
    "hour",
    "h",
    "minutes",
    "minute",
    "m",
    "seconds",
    "second",
    "s",
    "milliseconds",
    "millisecond",
    "ms"
  ], i2, property, propertyLen = properties.length;
  for (i2 = 0; i2 < propertyLen; i2 += 1) {
    property = properties[i2];
    propertyTest = propertyTest || hasOwnProp$1(input, property);
  }
  return objectTest && propertyTest;
}
function isNumberOrStringArray$1(input) {
  var arrayTest = isArray$7(input), dataTypeTest = false;
  if (arrayTest) {
    dataTypeTest = input.filter(function(item) {
      return !isNumber$4(item) && isString$3(input);
    }).length === 0;
  }
  return arrayTest && dataTypeTest;
}
function isCalendarSpec$1(input) {
  var objectTest = isObject$m(input) && !isObjectEmpty$1(input), propertyTest = false, properties = [
    "sameDay",
    "nextDay",
    "lastDay",
    "nextWeek",
    "lastWeek",
    "sameElse"
  ], i2, property;
  for (i2 = 0; i2 < properties.length; i2 += 1) {
    property = properties[i2];
    propertyTest = propertyTest || hasOwnProp$1(input, property);
  }
  return objectTest && propertyTest;
}
function getCalendarFormat$1(myMoment, now2) {
  var diff2 = myMoment.diff(now2, "days", true);
  return diff2 < -6 ? "sameElse" : diff2 < -1 ? "lastWeek" : diff2 < 0 ? "lastDay" : diff2 < 1 ? "sameDay" : diff2 < 2 ? "nextDay" : diff2 < 7 ? "nextWeek" : "sameElse";
}
function calendar$1$1(time2, formats) {
  if (arguments.length === 1) {
    if (!arguments[0]) {
      time2 = void 0;
      formats = void 0;
    } else if (isMomentInput$1(arguments[0])) {
      time2 = arguments[0];
      formats = void 0;
    } else if (isCalendarSpec$1(arguments[0])) {
      formats = arguments[0];
      time2 = void 0;
    }
  }
  var now2 = time2 || createLocal$1(), sod = cloneWithOffset$1(now2, this).startOf("day"), format2 = hooks$2.calendarFormat(this, sod) || "sameElse", output2 = formats && (isFunction$5(formats[format2]) ? formats[format2].call(this, now2) : formats[format2]);
  return this.format(
    output2 || this.localeData().calendar(format2, this, createLocal$1(now2))
  );
}
function clone$7() {
  return new Moment$1(this);
}
function isAfter$1(input, units) {
  var localInput = isMoment$1(input) ? input : createLocal$1(input);
  if (!(this.isValid() && localInput.isValid())) {
    return false;
  }
  units = normalizeUnits$1(units) || "millisecond";
  if (units === "millisecond") {
    return this.valueOf() > localInput.valueOf();
  } else {
    return localInput.valueOf() < this.clone().startOf(units).valueOf();
  }
}
function isBefore$1(input, units) {
  var localInput = isMoment$1(input) ? input : createLocal$1(input);
  if (!(this.isValid() && localInput.isValid())) {
    return false;
  }
  units = normalizeUnits$1(units) || "millisecond";
  if (units === "millisecond") {
    return this.valueOf() < localInput.valueOf();
  } else {
    return this.clone().endOf(units).valueOf() < localInput.valueOf();
  }
}
function isBetween$1(from2, to2, units, inclusivity) {
  var localFrom = isMoment$1(from2) ? from2 : createLocal$1(from2), localTo = isMoment$1(to2) ? to2 : createLocal$1(to2);
  if (!(this.isValid() && localFrom.isValid() && localTo.isValid())) {
    return false;
  }
  inclusivity = inclusivity || "()";
  return (inclusivity[0] === "(" ? this.isAfter(localFrom, units) : !this.isBefore(localFrom, units)) && (inclusivity[1] === ")" ? this.isBefore(localTo, units) : !this.isAfter(localTo, units));
}
function isSame$1(input, units) {
  var localInput = isMoment$1(input) ? input : createLocal$1(input), inputMs;
  if (!(this.isValid() && localInput.isValid())) {
    return false;
  }
  units = normalizeUnits$1(units) || "millisecond";
  if (units === "millisecond") {
    return this.valueOf() === localInput.valueOf();
  } else {
    inputMs = localInput.valueOf();
    return this.clone().startOf(units).valueOf() <= inputMs && inputMs <= this.clone().endOf(units).valueOf();
  }
}
function isSameOrAfter$1(input, units) {
  return this.isSame(input, units) || this.isAfter(input, units);
}
function isSameOrBefore$1(input, units) {
  return this.isSame(input, units) || this.isBefore(input, units);
}
function diff$3(input, units, asFloat) {
  var that, zoneDelta, output2;
  if (!this.isValid()) {
    return NaN;
  }
  that = cloneWithOffset$1(input, this);
  if (!that.isValid()) {
    return NaN;
  }
  zoneDelta = (that.utcOffset() - this.utcOffset()) * 6e4;
  units = normalizeUnits$1(units);
  switch (units) {
    case "year":
      output2 = monthDiff$1(this, that) / 12;
      break;
    case "month":
      output2 = monthDiff$1(this, that);
      break;
    case "quarter":
      output2 = monthDiff$1(this, that) / 3;
      break;
    case "second":
      output2 = (this - that) / 1e3;
      break;
    case "minute":
      output2 = (this - that) / 6e4;
      break;
    case "hour":
      output2 = (this - that) / 36e5;
      break;
    case "day":
      output2 = (this - that - zoneDelta) / 864e5;
      break;
    case "week":
      output2 = (this - that - zoneDelta) / 6048e5;
      break;
    default:
      output2 = this - that;
  }
  return asFloat ? output2 : absFloor$1(output2);
}
function monthDiff$1(a2, b2) {
  if (a2.date() < b2.date()) {
    return -monthDiff$1(b2, a2);
  }
  var wholeMonthDiff = (b2.year() - a2.year()) * 12 + (b2.month() - a2.month()), anchor = a2.clone().add(wholeMonthDiff, "months"), anchor2, adjust;
  if (b2 - anchor < 0) {
    anchor2 = a2.clone().add(wholeMonthDiff - 1, "months");
    adjust = (b2 - anchor) / (anchor - anchor2);
  } else {
    anchor2 = a2.clone().add(wholeMonthDiff + 1, "months");
    adjust = (b2 - anchor) / (anchor2 - anchor);
  }
  return -(wholeMonthDiff + adjust) || 0;
}
hooks$2.defaultFormat = "YYYY-MM-DDTHH:mm:ssZ";
hooks$2.defaultFormatUtc = "YYYY-MM-DDTHH:mm:ss[Z]";
function toString$f() {
  return this.clone().locale("en").format("ddd MMM DD YYYY HH:mm:ss [GMT]ZZ");
}
function toISOString$2(keepOffset) {
  if (!this.isValid()) {
    return null;
  }
  var utc2 = keepOffset !== true, m2 = utc2 ? this.clone().utc() : this;
  if (m2.year() < 0 || m2.year() > 9999) {
    return formatMoment$1(
      m2,
      utc2 ? "YYYYYY-MM-DD[T]HH:mm:ss.SSS[Z]" : "YYYYYY-MM-DD[T]HH:mm:ss.SSSZ"
    );
  }
  if (isFunction$5(Date.prototype.toISOString)) {
    if (utc2) {
      return this.toDate().toISOString();
    } else {
      return new Date(this.valueOf() + this.utcOffset() * 60 * 1e3).toISOString().replace("Z", formatMoment$1(m2, "Z"));
    }
  }
  return formatMoment$1(
    m2,
    utc2 ? "YYYY-MM-DD[T]HH:mm:ss.SSS[Z]" : "YYYY-MM-DD[T]HH:mm:ss.SSSZ"
  );
}
function inspect$1() {
  if (!this.isValid()) {
    return "moment.invalid(/* " + this._i + " */)";
  }
  var func = "moment", zone = "", prefix, year, datetime, suffix2;
  if (!this.isLocal()) {
    func = this.utcOffset() === 0 ? "moment.utc" : "moment.parseZone";
    zone = "Z";
  }
  prefix = "[" + func + '("]';
  year = 0 <= this.year() && this.year() <= 9999 ? "YYYY" : "YYYYYY";
  datetime = "-MM-DD[T]HH:mm:ss.SSS";
  suffix2 = zone + '[")]';
  return this.format(prefix + year + datetime + suffix2);
}
function format$2(inputString) {
  if (!inputString) {
    inputString = this.isUtc() ? hooks$2.defaultFormatUtc : hooks$2.defaultFormat;
  }
  var output2 = formatMoment$1(this, inputString);
  return this.localeData().postformat(output2);
}
function from$1(time2, withoutSuffix) {
  if (this.isValid() && (isMoment$1(time2) && time2.isValid() || createLocal$1(time2).isValid())) {
    return createDuration$1({ to: this, from: time2 }).locale(this.locale()).humanize(!withoutSuffix);
  } else {
    return this.localeData().invalidDate();
  }
}
function fromNow$1(withoutSuffix) {
  return this.from(createLocal$1(), withoutSuffix);
}
function to$1(time2, withoutSuffix) {
  if (this.isValid() && (isMoment$1(time2) && time2.isValid() || createLocal$1(time2).isValid())) {
    return createDuration$1({ from: this, to: time2 }).locale(this.locale()).humanize(!withoutSuffix);
  } else {
    return this.localeData().invalidDate();
  }
}
function toNow$1(withoutSuffix) {
  return this.to(createLocal$1(), withoutSuffix);
}
function locale$1(key) {
  var newLocaleData;
  if (key === void 0) {
    return this._locale._abbr;
  } else {
    newLocaleData = getLocale$1(key);
    if (newLocaleData != null) {
      this._locale = newLocaleData;
    }
    return this;
  }
}
var lang$1 = deprecate$1(
  "moment().lang() is deprecated. Instead, use moment().localeData() to get the language configuration. Use moment().locale() to change languages.",
  function(key) {
    if (key === void 0) {
      return this.localeData();
    } else {
      return this.locale(key);
    }
  }
);
function localeData$1() {
  return this._locale;
}
var MS_PER_SECOND$1 = 1e3, MS_PER_MINUTE$1 = 60 * MS_PER_SECOND$1, MS_PER_HOUR$1 = 60 * MS_PER_MINUTE$1, MS_PER_400_YEARS$1 = (365 * 400 + 97) * 24 * MS_PER_HOUR$1;
function mod$1$1(dividend, divisor) {
  return (dividend % divisor + divisor) % divisor;
}
function localStartOfDate$1(y2, m2, d2) {
  if (y2 < 100 && y2 >= 0) {
    return new Date(y2 + 400, m2, d2) - MS_PER_400_YEARS$1;
  } else {
    return new Date(y2, m2, d2).valueOf();
  }
}
function utcStartOfDate$1(y2, m2, d2) {
  if (y2 < 100 && y2 >= 0) {
    return Date.UTC(y2 + 400, m2, d2) - MS_PER_400_YEARS$1;
  } else {
    return Date.UTC(y2, m2, d2);
  }
}
function startOf$1(units) {
  var time2, startOfDate;
  units = normalizeUnits$1(units);
  if (units === void 0 || units === "millisecond" || !this.isValid()) {
    return this;
  }
  startOfDate = this._isUTC ? utcStartOfDate$1 : localStartOfDate$1;
  switch (units) {
    case "year":
      time2 = startOfDate(this.year(), 0, 1);
      break;
    case "quarter":
      time2 = startOfDate(
        this.year(),
        this.month() - this.month() % 3,
        1
      );
      break;
    case "month":
      time2 = startOfDate(this.year(), this.month(), 1);
      break;
    case "week":
      time2 = startOfDate(
        this.year(),
        this.month(),
        this.date() - this.weekday()
      );
      break;
    case "isoWeek":
      time2 = startOfDate(
        this.year(),
        this.month(),
        this.date() - (this.isoWeekday() - 1)
      );
      break;
    case "day":
    case "date":
      time2 = startOfDate(this.year(), this.month(), this.date());
      break;
    case "hour":
      time2 = this._d.valueOf();
      time2 -= mod$1$1(
        time2 + (this._isUTC ? 0 : this.utcOffset() * MS_PER_MINUTE$1),
        MS_PER_HOUR$1
      );
      break;
    case "minute":
      time2 = this._d.valueOf();
      time2 -= mod$1$1(time2, MS_PER_MINUTE$1);
      break;
    case "second":
      time2 = this._d.valueOf();
      time2 -= mod$1$1(time2, MS_PER_SECOND$1);
      break;
  }
  this._d.setTime(time2);
  hooks$2.updateOffset(this, true);
  return this;
}
function endOf$1(units) {
  var time2, startOfDate;
  units = normalizeUnits$1(units);
  if (units === void 0 || units === "millisecond" || !this.isValid()) {
    return this;
  }
  startOfDate = this._isUTC ? utcStartOfDate$1 : localStartOfDate$1;
  switch (units) {
    case "year":
      time2 = startOfDate(this.year() + 1, 0, 1) - 1;
      break;
    case "quarter":
      time2 = startOfDate(
        this.year(),
        this.month() - this.month() % 3 + 3,
        1
      ) - 1;
      break;
    case "month":
      time2 = startOfDate(this.year(), this.month() + 1, 1) - 1;
      break;
    case "week":
      time2 = startOfDate(
        this.year(),
        this.month(),
        this.date() - this.weekday() + 7
      ) - 1;
      break;
    case "isoWeek":
      time2 = startOfDate(
        this.year(),
        this.month(),
        this.date() - (this.isoWeekday() - 1) + 7
      ) - 1;
      break;
    case "day":
    case "date":
      time2 = startOfDate(this.year(), this.month(), this.date() + 1) - 1;
      break;
    case "hour":
      time2 = this._d.valueOf();
      time2 += MS_PER_HOUR$1 - mod$1$1(
        time2 + (this._isUTC ? 0 : this.utcOffset() * MS_PER_MINUTE$1),
        MS_PER_HOUR$1
      ) - 1;
      break;
    case "minute":
      time2 = this._d.valueOf();
      time2 += MS_PER_MINUTE$1 - mod$1$1(time2, MS_PER_MINUTE$1) - 1;
      break;
    case "second":
      time2 = this._d.valueOf();
      time2 += MS_PER_SECOND$1 - mod$1$1(time2, MS_PER_SECOND$1) - 1;
      break;
  }
  this._d.setTime(time2);
  hooks$2.updateOffset(this, true);
  return this;
}
function valueOf$2() {
  return this._d.valueOf() - (this._offset || 0) * 6e4;
}
function unix$1() {
  return Math.floor(this.valueOf() / 1e3);
}
function toDate$1() {
  return new Date(this.valueOf());
}
function toArray$3() {
  var m2 = this;
  return [
    m2.year(),
    m2.month(),
    m2.date(),
    m2.hour(),
    m2.minute(),
    m2.second(),
    m2.millisecond()
  ];
}
function toObject$6() {
  var m2 = this;
  return {
    years: m2.year(),
    months: m2.month(),
    date: m2.date(),
    hours: m2.hours(),
    minutes: m2.minutes(),
    seconds: m2.seconds(),
    milliseconds: m2.milliseconds()
  };
}
function toJSON$1() {
  return this.isValid() ? this.toISOString() : null;
}
function isValid$2$1() {
  return isValid$3(this);
}
function parsingFlags$1() {
  return extend$6({}, getParsingFlags$1(this));
}
function invalidAt$1() {
  return getParsingFlags$1(this).overflow;
}
function creationData$1() {
  return {
    input: this._i,
    format: this._f,
    locale: this._locale,
    isUTC: this._isUTC,
    strict: this._strict
  };
}
addFormatToken$1("N", 0, 0, "eraAbbr");
addFormatToken$1("NN", 0, 0, "eraAbbr");
addFormatToken$1("NNN", 0, 0, "eraAbbr");
addFormatToken$1("NNNN", 0, 0, "eraName");
addFormatToken$1("NNNNN", 0, 0, "eraNarrow");
addFormatToken$1("y", ["y", 1], "yo", "eraYear");
addFormatToken$1("y", ["yy", 2], 0, "eraYear");
addFormatToken$1("y", ["yyy", 3], 0, "eraYear");
addFormatToken$1("y", ["yyyy", 4], 0, "eraYear");
addRegexToken$1("N", matchEraAbbr$1);
addRegexToken$1("NN", matchEraAbbr$1);
addRegexToken$1("NNN", matchEraAbbr$1);
addRegexToken$1("NNNN", matchEraName$1);
addRegexToken$1("NNNNN", matchEraNarrow$1);
addParseToken$1(
  ["N", "NN", "NNN", "NNNN", "NNNNN"],
  function(input, array2, config2, token2) {
    var era = config2._locale.erasParse(input, token2, config2._strict);
    if (era) {
      getParsingFlags$1(config2).era = era;
    } else {
      getParsingFlags$1(config2).invalidEra = input;
    }
  }
);
addRegexToken$1("y", matchUnsigned$1);
addRegexToken$1("yy", matchUnsigned$1);
addRegexToken$1("yyy", matchUnsigned$1);
addRegexToken$1("yyyy", matchUnsigned$1);
addRegexToken$1("yo", matchEraYearOrdinal$1);
addParseToken$1(["y", "yy", "yyy", "yyyy"], YEAR$1);
addParseToken$1(["yo"], function(input, array2, config2, token2) {
  var match5;
  if (config2._locale._eraYearOrdinalRegex) {
    match5 = input.match(config2._locale._eraYearOrdinalRegex);
  }
  if (config2._locale.eraYearOrdinalParse) {
    array2[YEAR$1] = config2._locale.eraYearOrdinalParse(input, match5);
  } else {
    array2[YEAR$1] = parseInt(input, 10);
  }
});
function localeEras$1(m2, format2) {
  var i2, l2, date, eras = this._eras || getLocale$1("en")._eras;
  for (i2 = 0, l2 = eras.length; i2 < l2; ++i2) {
    switch (typeof eras[i2].since) {
      case "string":
        date = hooks$2(eras[i2].since).startOf("day");
        eras[i2].since = date.valueOf();
        break;
    }
    switch (typeof eras[i2].until) {
      case "undefined":
        eras[i2].until = Infinity;
        break;
      case "string":
        date = hooks$2(eras[i2].until).startOf("day").valueOf();
        eras[i2].until = date.valueOf();
        break;
    }
  }
  return eras;
}
function localeErasParse$1(eraName, format2, strict) {
  var i2, l2, eras = this.eras(), name, abbr, narrow;
  eraName = eraName.toUpperCase();
  for (i2 = 0, l2 = eras.length; i2 < l2; ++i2) {
    name = eras[i2].name.toUpperCase();
    abbr = eras[i2].abbr.toUpperCase();
    narrow = eras[i2].narrow.toUpperCase();
    if (strict) {
      switch (format2) {
        case "N":
        case "NN":
        case "NNN":
          if (abbr === eraName) {
            return eras[i2];
          }
          break;
        case "NNNN":
          if (name === eraName) {
            return eras[i2];
          }
          break;
        case "NNNNN":
          if (narrow === eraName) {
            return eras[i2];
          }
          break;
      }
    } else if ([name, abbr, narrow].indexOf(eraName) >= 0) {
      return eras[i2];
    }
  }
}
function localeErasConvertYear$1(era, year) {
  var dir3 = era.since <= era.until ? 1 : -1;
  if (year === void 0) {
    return hooks$2(era.since).year();
  } else {
    return hooks$2(era.since).year() + (year - era.offset) * dir3;
  }
}
function getEraName$1() {
  var i2, l2, val, eras = this.localeData().eras();
  for (i2 = 0, l2 = eras.length; i2 < l2; ++i2) {
    val = this.clone().startOf("day").valueOf();
    if (eras[i2].since <= val && val <= eras[i2].until) {
      return eras[i2].name;
    }
    if (eras[i2].until <= val && val <= eras[i2].since) {
      return eras[i2].name;
    }
  }
  return "";
}
function getEraNarrow$1() {
  var i2, l2, val, eras = this.localeData().eras();
  for (i2 = 0, l2 = eras.length; i2 < l2; ++i2) {
    val = this.clone().startOf("day").valueOf();
    if (eras[i2].since <= val && val <= eras[i2].until) {
      return eras[i2].narrow;
    }
    if (eras[i2].until <= val && val <= eras[i2].since) {
      return eras[i2].narrow;
    }
  }
  return "";
}
function getEraAbbr$1() {
  var i2, l2, val, eras = this.localeData().eras();
  for (i2 = 0, l2 = eras.length; i2 < l2; ++i2) {
    val = this.clone().startOf("day").valueOf();
    if (eras[i2].since <= val && val <= eras[i2].until) {
      return eras[i2].abbr;
    }
    if (eras[i2].until <= val && val <= eras[i2].since) {
      return eras[i2].abbr;
    }
  }
  return "";
}
function getEraYear$1() {
  var i2, l2, dir3, val, eras = this.localeData().eras();
  for (i2 = 0, l2 = eras.length; i2 < l2; ++i2) {
    dir3 = eras[i2].since <= eras[i2].until ? 1 : -1;
    val = this.clone().startOf("day").valueOf();
    if (eras[i2].since <= val && val <= eras[i2].until || eras[i2].until <= val && val <= eras[i2].since) {
      return (this.year() - hooks$2(eras[i2].since).year()) * dir3 + eras[i2].offset;
    }
  }
  return this.year();
}
function erasNameRegex$1(isStrict) {
  if (!hasOwnProp$1(this, "_erasNameRegex")) {
    computeErasParse$1.call(this);
  }
  return isStrict ? this._erasNameRegex : this._erasRegex;
}
function erasAbbrRegex$1(isStrict) {
  if (!hasOwnProp$1(this, "_erasAbbrRegex")) {
    computeErasParse$1.call(this);
  }
  return isStrict ? this._erasAbbrRegex : this._erasRegex;
}
function erasNarrowRegex$1(isStrict) {
  if (!hasOwnProp$1(this, "_erasNarrowRegex")) {
    computeErasParse$1.call(this);
  }
  return isStrict ? this._erasNarrowRegex : this._erasRegex;
}
function matchEraAbbr$1(isStrict, locale2) {
  return locale2.erasAbbrRegex(isStrict);
}
function matchEraName$1(isStrict, locale2) {
  return locale2.erasNameRegex(isStrict);
}
function matchEraNarrow$1(isStrict, locale2) {
  return locale2.erasNarrowRegex(isStrict);
}
function matchEraYearOrdinal$1(isStrict, locale2) {
  return locale2._eraYearOrdinalRegex || matchUnsigned$1;
}
function computeErasParse$1() {
  var abbrPieces = [], namePieces = [], narrowPieces = [], mixedPieces = [], i2, l2, erasName, erasAbbr, erasNarrow, eras = this.eras();
  for (i2 = 0, l2 = eras.length; i2 < l2; ++i2) {
    erasName = regexEscape$1(eras[i2].name);
    erasAbbr = regexEscape$1(eras[i2].abbr);
    erasNarrow = regexEscape$1(eras[i2].narrow);
    namePieces.push(erasName);
    abbrPieces.push(erasAbbr);
    narrowPieces.push(erasNarrow);
    mixedPieces.push(erasName);
    mixedPieces.push(erasAbbr);
    mixedPieces.push(erasNarrow);
  }
  this._erasRegex = new RegExp("^(" + mixedPieces.join("|") + ")", "i");
  this._erasNameRegex = new RegExp("^(" + namePieces.join("|") + ")", "i");
  this._erasAbbrRegex = new RegExp("^(" + abbrPieces.join("|") + ")", "i");
  this._erasNarrowRegex = new RegExp(
    "^(" + narrowPieces.join("|") + ")",
    "i"
  );
}
addFormatToken$1(0, ["gg", 2], 0, function() {
  return this.weekYear() % 100;
});
addFormatToken$1(0, ["GG", 2], 0, function() {
  return this.isoWeekYear() % 100;
});
function addWeekYearFormatToken$1(token2, getter) {
  addFormatToken$1(0, [token2, token2.length], 0, getter);
}
addWeekYearFormatToken$1("gggg", "weekYear");
addWeekYearFormatToken$1("ggggg", "weekYear");
addWeekYearFormatToken$1("GGGG", "isoWeekYear");
addWeekYearFormatToken$1("GGGGG", "isoWeekYear");
addRegexToken$1("G", matchSigned$1);
addRegexToken$1("g", matchSigned$1);
addRegexToken$1("GG", match1to2$1, match2$1);
addRegexToken$1("gg", match1to2$1, match2$1);
addRegexToken$1("GGGG", match1to4$1, match4$1);
addRegexToken$1("gggg", match1to4$1, match4$1);
addRegexToken$1("GGGGG", match1to6$1, match6$1);
addRegexToken$1("ggggg", match1to6$1, match6$1);
addWeekParseToken$1(
  ["gggg", "ggggg", "GGGG", "GGGGG"],
  function(input, week, config2, token2) {
    week[token2.substr(0, 2)] = toInt$1(input);
  }
);
addWeekParseToken$1(["gg", "GG"], function(input, week, config2, token2) {
  week[token2] = hooks$2.parseTwoDigitYear(input);
});
function getSetWeekYear$1(input) {
  return getSetWeekYearHelper$1.call(
    this,
    input,
    this.week(),
    this.weekday() + this.localeData()._week.dow,
    this.localeData()._week.dow,
    this.localeData()._week.doy
  );
}
function getSetISOWeekYear$1(input) {
  return getSetWeekYearHelper$1.call(
    this,
    input,
    this.isoWeek(),
    this.isoWeekday(),
    1,
    4
  );
}
function getISOWeeksInYear$1() {
  return weeksInYear$1(this.year(), 1, 4);
}
function getISOWeeksInISOWeekYear$1() {
  return weeksInYear$1(this.isoWeekYear(), 1, 4);
}
function getWeeksInYear$1() {
  var weekInfo = this.localeData()._week;
  return weeksInYear$1(this.year(), weekInfo.dow, weekInfo.doy);
}
function getWeeksInWeekYear$1() {
  var weekInfo = this.localeData()._week;
  return weeksInYear$1(this.weekYear(), weekInfo.dow, weekInfo.doy);
}
function getSetWeekYearHelper$1(input, week, weekday, dow, doy) {
  var weeksTarget;
  if (input == null) {
    return weekOfYear$1(this, dow, doy).year;
  } else {
    weeksTarget = weeksInYear$1(input, dow, doy);
    if (week > weeksTarget) {
      week = weeksTarget;
    }
    return setWeekAll$1.call(this, input, week, weekday, dow, doy);
  }
}
function setWeekAll$1(weekYear, week, weekday, dow, doy) {
  var dayOfYearData = dayOfYearFromWeeks$1(weekYear, week, weekday, dow, doy), date = createUTCDate$1(dayOfYearData.year, 0, dayOfYearData.dayOfYear);
  this.year(date.getUTCFullYear());
  this.month(date.getUTCMonth());
  this.date(date.getUTCDate());
  return this;
}
addFormatToken$1("Q", 0, "Qo", "quarter");
addRegexToken$1("Q", match1$1);
addParseToken$1("Q", function(input, array2) {
  array2[MONTH$1] = (toInt$1(input) - 1) * 3;
});
function getSetQuarter$1(input) {
  return input == null ? Math.ceil((this.month() + 1) / 3) : this.month((input - 1) * 3 + this.month() % 3);
}
addFormatToken$1("D", ["DD", 2], "Do", "date");
addRegexToken$1("D", match1to2$1, match1to2NoLeadingZero$1);
addRegexToken$1("DD", match1to2$1, match2$1);
addRegexToken$1("Do", function(isStrict, locale2) {
  return isStrict ? locale2._dayOfMonthOrdinalParse || locale2._ordinalParse : locale2._dayOfMonthOrdinalParseLenient;
});
addParseToken$1(["D", "DD"], DATE$1);
addParseToken$1("Do", function(input, array2) {
  array2[DATE$1] = toInt$1(input.match(match1to2$1)[0]);
});
var getSetDayOfMonth$1 = makeGetSet$1("Date", true);
addFormatToken$1("DDD", ["DDDD", 3], "DDDo", "dayOfYear");
addRegexToken$1("DDD", match1to3$1);
addRegexToken$1("DDDD", match3$1);
addParseToken$1(["DDD", "DDDD"], function(input, array2, config2) {
  config2._dayOfYear = toInt$1(input);
});
function getSetDayOfYear$1(input) {
  var dayOfYear = Math.round(
    (this.clone().startOf("day") - this.clone().startOf("year")) / 864e5
  ) + 1;
  return input == null ? dayOfYear : this.add(input - dayOfYear, "d");
}
addFormatToken$1("m", ["mm", 2], 0, "minute");
addRegexToken$1("m", match1to2$1, match1to2HasZero$1);
addRegexToken$1("mm", match1to2$1, match2$1);
addParseToken$1(["m", "mm"], MINUTE$1);
var getSetMinute$1 = makeGetSet$1("Minutes", false);
addFormatToken$1("s", ["ss", 2], 0, "second");
addRegexToken$1("s", match1to2$1, match1to2HasZero$1);
addRegexToken$1("ss", match1to2$1, match2$1);
addParseToken$1(["s", "ss"], SECOND$1);
var getSetSecond$1 = makeGetSet$1("Seconds", false);
addFormatToken$1("S", 0, 0, function() {
  return ~~(this.millisecond() / 100);
});
addFormatToken$1(0, ["SS", 2], 0, function() {
  return ~~(this.millisecond() / 10);
});
addFormatToken$1(0, ["SSS", 3], 0, "millisecond");
addFormatToken$1(0, ["SSSS", 4], 0, function() {
  return this.millisecond() * 10;
});
addFormatToken$1(0, ["SSSSS", 5], 0, function() {
  return this.millisecond() * 100;
});
addFormatToken$1(0, ["SSSSSS", 6], 0, function() {
  return this.millisecond() * 1e3;
});
addFormatToken$1(0, ["SSSSSSS", 7], 0, function() {
  return this.millisecond() * 1e4;
});
addFormatToken$1(0, ["SSSSSSSS", 8], 0, function() {
  return this.millisecond() * 1e5;
});
addFormatToken$1(0, ["SSSSSSSSS", 9], 0, function() {
  return this.millisecond() * 1e6;
});
addRegexToken$1("S", match1to3$1, match1$1);
addRegexToken$1("SS", match1to3$1, match2$1);
addRegexToken$1("SSS", match1to3$1, match3$1);
var token$1, getSetMillisecond$1;
for (token$1 = "SSSS"; token$1.length <= 9; token$1 += "S") {
  addRegexToken$1(token$1, matchUnsigned$1);
}
function parseMs$1(input, array2) {
  array2[MILLISECOND$1] = toInt$1(("0." + input) * 1e3);
}
for (token$1 = "S"; token$1.length <= 9; token$1 += "S") {
  addParseToken$1(token$1, parseMs$1);
}
getSetMillisecond$1 = makeGetSet$1("Milliseconds", false);
addFormatToken$1("z", 0, 0, "zoneAbbr");
addFormatToken$1("zz", 0, 0, "zoneName");
function getZoneAbbr$1() {
  return this._isUTC ? "UTC" : "";
}
function getZoneName$1() {
  return this._isUTC ? "Coordinated Universal Time" : "";
}
var proto$4 = Moment$1.prototype;
proto$4.add = add$3;
proto$4.calendar = calendar$1$1;
proto$4.clone = clone$7;
proto$4.diff = diff$3;
proto$4.endOf = endOf$1;
proto$4.format = format$2;
proto$4.from = from$1;
proto$4.fromNow = fromNow$1;
proto$4.to = to$1;
proto$4.toNow = toNow$1;
proto$4.get = stringGet$1;
proto$4.invalidAt = invalidAt$1;
proto$4.isAfter = isAfter$1;
proto$4.isBefore = isBefore$1;
proto$4.isBetween = isBetween$1;
proto$4.isSame = isSame$1;
proto$4.isSameOrAfter = isSameOrAfter$1;
proto$4.isSameOrBefore = isSameOrBefore$1;
proto$4.isValid = isValid$2$1;
proto$4.lang = lang$1;
proto$4.locale = locale$1;
proto$4.localeData = localeData$1;
proto$4.max = prototypeMax$1;
proto$4.min = prototypeMin$1;
proto$4.parsingFlags = parsingFlags$1;
proto$4.set = stringSet$1;
proto$4.startOf = startOf$1;
proto$4.subtract = subtract$2;
proto$4.toArray = toArray$3;
proto$4.toObject = toObject$6;
proto$4.toDate = toDate$1;
proto$4.toISOString = toISOString$2;
proto$4.inspect = inspect$1;
if (typeof Symbol !== "undefined" && Symbol.for != null) {
  proto$4[Symbol.for("nodejs.util.inspect.custom")] = function() {
    return "Moment<" + this.format() + ">";
  };
}
proto$4.toJSON = toJSON$1;
proto$4.toString = toString$f;
proto$4.unix = unix$1;
proto$4.valueOf = valueOf$2;
proto$4.creationData = creationData$1;
proto$4.eraName = getEraName$1;
proto$4.eraNarrow = getEraNarrow$1;
proto$4.eraAbbr = getEraAbbr$1;
proto$4.eraYear = getEraYear$1;
proto$4.year = getSetYear$1;
proto$4.isLeapYear = getIsLeapYear$1;
proto$4.weekYear = getSetWeekYear$1;
proto$4.isoWeekYear = getSetISOWeekYear$1;
proto$4.quarter = proto$4.quarters = getSetQuarter$1;
proto$4.month = getSetMonth$1;
proto$4.daysInMonth = getDaysInMonth$1;
proto$4.week = proto$4.weeks = getSetWeek$1;
proto$4.isoWeek = proto$4.isoWeeks = getSetISOWeek$1;
proto$4.weeksInYear = getWeeksInYear$1;
proto$4.weeksInWeekYear = getWeeksInWeekYear$1;
proto$4.isoWeeksInYear = getISOWeeksInYear$1;
proto$4.isoWeeksInISOWeekYear = getISOWeeksInISOWeekYear$1;
proto$4.date = getSetDayOfMonth$1;
proto$4.day = proto$4.days = getSetDayOfWeek$1;
proto$4.weekday = getSetLocaleDayOfWeek$1;
proto$4.isoWeekday = getSetISODayOfWeek$1;
proto$4.dayOfYear = getSetDayOfYear$1;
proto$4.hour = proto$4.hours = getSetHour$1;
proto$4.minute = proto$4.minutes = getSetMinute$1;
proto$4.second = proto$4.seconds = getSetSecond$1;
proto$4.millisecond = proto$4.milliseconds = getSetMillisecond$1;
proto$4.utcOffset = getSetOffset$1;
proto$4.utc = setOffsetToUTC$1;
proto$4.local = setOffsetToLocal$1;
proto$4.parseZone = setOffsetToParsedOffset$1;
proto$4.hasAlignedHourOffset = hasAlignedHourOffset$1;
proto$4.isDST = isDaylightSavingTime$1;
proto$4.isLocal = isLocal$1;
proto$4.isUtcOffset = isUtcOffset$1;
proto$4.isUtc = isUtc$1;
proto$4.isUTC = isUtc$1;
proto$4.zoneAbbr = getZoneAbbr$1;
proto$4.zoneName = getZoneName$1;
proto$4.dates = deprecate$1(
  "dates accessor is deprecated. Use date instead.",
  getSetDayOfMonth$1
);
proto$4.months = deprecate$1(
  "months accessor is deprecated. Use month instead",
  getSetMonth$1
);
proto$4.years = deprecate$1(
  "years accessor is deprecated. Use year instead",
  getSetYear$1
);
proto$4.zone = deprecate$1(
  "moment().zone is deprecated, use moment().utcOffset instead. http://momentjs.com/guides/#/warnings/zone/",
  getSetZone$1
);
proto$4.isDSTShifted = deprecate$1(
  "isDSTShifted is deprecated. See http://momentjs.com/guides/#/warnings/dst-shifted/ for more information",
  isDaylightSavingTimeShifted$1
);
function createUnix$1(input) {
  return createLocal$1(input * 1e3);
}
function createInZone$1() {
  return createLocal$1.apply(null, arguments).parseZone();
}
function preParsePostFormat$1(string) {
  return string;
}
var proto$1$1 = Locale$1.prototype;
proto$1$1.calendar = calendar$2;
proto$1$1.longDateFormat = longDateFormat$1;
proto$1$1.invalidDate = invalidDate$1;
proto$1$1.ordinal = ordinal$1;
proto$1$1.preparse = preParsePostFormat$1;
proto$1$1.postformat = preParsePostFormat$1;
proto$1$1.relativeTime = relativeTime$2;
proto$1$1.pastFuture = pastFuture$1;
proto$1$1.set = set$5;
proto$1$1.eras = localeEras$1;
proto$1$1.erasParse = localeErasParse$1;
proto$1$1.erasConvertYear = localeErasConvertYear$1;
proto$1$1.erasAbbrRegex = erasAbbrRegex$1;
proto$1$1.erasNameRegex = erasNameRegex$1;
proto$1$1.erasNarrowRegex = erasNarrowRegex$1;
proto$1$1.months = localeMonths$1;
proto$1$1.monthsShort = localeMonthsShort$1;
proto$1$1.monthsParse = localeMonthsParse$1;
proto$1$1.monthsRegex = monthsRegex$1;
proto$1$1.monthsShortRegex = monthsShortRegex$1;
proto$1$1.week = localeWeek$1;
proto$1$1.firstDayOfYear = localeFirstDayOfYear$1;
proto$1$1.firstDayOfWeek = localeFirstDayOfWeek$1;
proto$1$1.weekdays = localeWeekdays$1;
proto$1$1.weekdaysMin = localeWeekdaysMin$1;
proto$1$1.weekdaysShort = localeWeekdaysShort$1;
proto$1$1.weekdaysParse = localeWeekdaysParse$1;
proto$1$1.weekdaysRegex = weekdaysRegex$1;
proto$1$1.weekdaysShortRegex = weekdaysShortRegex$1;
proto$1$1.weekdaysMinRegex = weekdaysMinRegex$1;
proto$1$1.isPM = localeIsPM$1;
proto$1$1.meridiem = localeMeridiem$1;
function get$1$2(format2, index2, field, setter) {
  var locale2 = getLocale$1(), utc2 = createUTC$1().set(setter, index2);
  return locale2[field](utc2, format2);
}
function listMonthsImpl$1(format2, index2, field) {
  if (isNumber$4(format2)) {
    index2 = format2;
    format2 = void 0;
  }
  format2 = format2 || "";
  if (index2 != null) {
    return get$1$2(format2, index2, field, "month");
  }
  var i2, out2 = [];
  for (i2 = 0; i2 < 12; i2++) {
    out2[i2] = get$1$2(format2, i2, field, "month");
  }
  return out2;
}
function listWeekdaysImpl$1(localeSorted, format2, index2, field) {
  if (typeof localeSorted === "boolean") {
    if (isNumber$4(format2)) {
      index2 = format2;
      format2 = void 0;
    }
    format2 = format2 || "";
  } else {
    format2 = localeSorted;
    index2 = format2;
    localeSorted = false;
    if (isNumber$4(format2)) {
      index2 = format2;
      format2 = void 0;
    }
    format2 = format2 || "";
  }
  var locale2 = getLocale$1(), shift = localeSorted ? locale2._week.dow : 0, i2, out2 = [];
  if (index2 != null) {
    return get$1$2(format2, (index2 + shift) % 7, field, "day");
  }
  for (i2 = 0; i2 < 7; i2++) {
    out2[i2] = get$1$2(format2, (i2 + shift) % 7, field, "day");
  }
  return out2;
}
function listMonths$1(format2, index2) {
  return listMonthsImpl$1(format2, index2, "months");
}
function listMonthsShort$1(format2, index2) {
  return listMonthsImpl$1(format2, index2, "monthsShort");
}
function listWeekdays$1(localeSorted, format2, index2) {
  return listWeekdaysImpl$1(localeSorted, format2, index2, "weekdays");
}
function listWeekdaysShort$1(localeSorted, format2, index2) {
  return listWeekdaysImpl$1(localeSorted, format2, index2, "weekdaysShort");
}
function listWeekdaysMin$1(localeSorted, format2, index2) {
  return listWeekdaysImpl$1(localeSorted, format2, index2, "weekdaysMin");
}
getSetGlobalLocale$1("en", {
  eras: [
    {
      since: "0001-01-01",
      until: Infinity,
      offset: 1,
      name: "Anno Domini",
      narrow: "AD",
      abbr: "AD"
    },
    {
      since: "0000-12-31",
      until: -Infinity,
      offset: 1,
      name: "Before Christ",
      narrow: "BC",
      abbr: "BC"
    }
  ],
  dayOfMonthOrdinalParse: /\d{1,2}(th|st|nd|rd)/,
  ordinal: function(number) {
    var b2 = number % 10, output2 = toInt$1(number % 100 / 10) === 1 ? "th" : b2 === 1 ? "st" : b2 === 2 ? "nd" : b2 === 3 ? "rd" : "th";
    return number + output2;
  }
});
hooks$2.lang = deprecate$1(
  "moment.lang is deprecated. Use moment.locale instead.",
  getSetGlobalLocale$1
);
hooks$2.langData = deprecate$1(
  "moment.langData is deprecated. Use moment.localeData instead.",
  getLocale$1
);
var mathAbs$4 = Math.abs;
function abs$3() {
  var data2 = this._data;
  this._milliseconds = mathAbs$4(this._milliseconds);
  this._days = mathAbs$4(this._days);
  this._months = mathAbs$4(this._months);
  data2.milliseconds = mathAbs$4(data2.milliseconds);
  data2.seconds = mathAbs$4(data2.seconds);
  data2.minutes = mathAbs$4(data2.minutes);
  data2.hours = mathAbs$4(data2.hours);
  data2.months = mathAbs$4(data2.months);
  data2.years = mathAbs$4(data2.years);
  return this;
}
function addSubtract$1$1(duration2, input, value, direction2) {
  var other = createDuration$1(input, value);
  duration2._milliseconds += direction2 * other._milliseconds;
  duration2._days += direction2 * other._days;
  duration2._months += direction2 * other._months;
  return duration2._bubble();
}
function add$1$1(input, value) {
  return addSubtract$1$1(this, input, value, 1);
}
function subtract$1$1(input, value) {
  return addSubtract$1$1(this, input, value, -1);
}
function absCeil$1(number) {
  if (number < 0) {
    return Math.floor(number);
  } else {
    return Math.ceil(number);
  }
}
function bubble$1() {
  var milliseconds2 = this._milliseconds, days2 = this._days, months2 = this._months, data2 = this._data, seconds2, minutes2, hours2, years2, monthsFromDays;
  if (!(milliseconds2 >= 0 && days2 >= 0 && months2 >= 0 || milliseconds2 <= 0 && days2 <= 0 && months2 <= 0)) {
    milliseconds2 += absCeil$1(monthsToDays$1(months2) + days2) * 864e5;
    days2 = 0;
    months2 = 0;
  }
  data2.milliseconds = milliseconds2 % 1e3;
  seconds2 = absFloor$1(milliseconds2 / 1e3);
  data2.seconds = seconds2 % 60;
  minutes2 = absFloor$1(seconds2 / 60);
  data2.minutes = minutes2 % 60;
  hours2 = absFloor$1(minutes2 / 60);
  data2.hours = hours2 % 24;
  days2 += absFloor$1(hours2 / 24);
  monthsFromDays = absFloor$1(daysToMonths$1(days2));
  months2 += monthsFromDays;
  days2 -= absCeil$1(monthsToDays$1(monthsFromDays));
  years2 = absFloor$1(months2 / 12);
  months2 %= 12;
  data2.days = days2;
  data2.months = months2;
  data2.years = years2;
  return this;
}
function daysToMonths$1(days2) {
  return days2 * 4800 / 146097;
}
function monthsToDays$1(months2) {
  return months2 * 146097 / 4800;
}
function as$1(units) {
  if (!this.isValid()) {
    return NaN;
  }
  var days2, months2, milliseconds2 = this._milliseconds;
  units = normalizeUnits$1(units);
  if (units === "month" || units === "quarter" || units === "year") {
    days2 = this._days + milliseconds2 / 864e5;
    months2 = this._months + daysToMonths$1(days2);
    switch (units) {
      case "month":
        return months2;
      case "quarter":
        return months2 / 3;
      case "year":
        return months2 / 12;
    }
  } else {
    days2 = this._days + Math.round(monthsToDays$1(this._months));
    switch (units) {
      case "week":
        return days2 / 7 + milliseconds2 / 6048e5;
      case "day":
        return days2 + milliseconds2 / 864e5;
      case "hour":
        return days2 * 24 + milliseconds2 / 36e5;
      case "minute":
        return days2 * 1440 + milliseconds2 / 6e4;
      case "second":
        return days2 * 86400 + milliseconds2 / 1e3;
      case "millisecond":
        return Math.floor(days2 * 864e5) + milliseconds2;
      default:
        throw new Error("Unknown unit " + units);
    }
  }
}
function makeAs$1(alias) {
  return function() {
    return this.as(alias);
  };
}
var asMilliseconds$1 = makeAs$1("ms"), asSeconds$1 = makeAs$1("s"), asMinutes$1 = makeAs$1("m"), asHours$1 = makeAs$1("h"), asDays$1 = makeAs$1("d"), asWeeks$1 = makeAs$1("w"), asMonths$1 = makeAs$1("M"), asQuarters$1 = makeAs$1("Q"), asYears$1 = makeAs$1("y"), valueOf$1$1 = asMilliseconds$1;
function clone$1$2() {
  return createDuration$1(this);
}
function get$2$1(units) {
  units = normalizeUnits$1(units);
  return this.isValid() ? this[units + "s"]() : NaN;
}
function makeGetter$1(name) {
  return function() {
    return this.isValid() ? this._data[name] : NaN;
  };
}
var milliseconds$1 = makeGetter$1("milliseconds"), seconds$1 = makeGetter$1("seconds"), minutes$1 = makeGetter$1("minutes"), hours$1 = makeGetter$1("hours"), days$2 = makeGetter$1("days"), months$2 = makeGetter$1("months"), years$1 = makeGetter$1("years");
function weeks$1() {
  return absFloor$1(this.days() / 7);
}
var round$5 = Math.round, thresholds$1 = {
  ss: 44,
  // a few seconds to seconds
  s: 45,
  // seconds to minute
  m: 45,
  // minutes to hour
  h: 22,
  // hours to day
  d: 26,
  // days to month/week
  w: null,
  // weeks to month
  M: 11
  // months to year
};
function substituteTimeAgo$1(string, number, withoutSuffix, isFuture, locale2) {
  return locale2.relativeTime(number || 1, !!withoutSuffix, string, isFuture);
}
function relativeTime$1$1(posNegDuration, withoutSuffix, thresholds2, locale2) {
  var duration2 = createDuration$1(posNegDuration).abs(), seconds2 = round$5(duration2.as("s")), minutes2 = round$5(duration2.as("m")), hours2 = round$5(duration2.as("h")), days2 = round$5(duration2.as("d")), months2 = round$5(duration2.as("M")), weeks2 = round$5(duration2.as("w")), years2 = round$5(duration2.as("y")), a2 = seconds2 <= thresholds2.ss && ["s", seconds2] || seconds2 < thresholds2.s && ["ss", seconds2] || minutes2 <= 1 && ["m"] || minutes2 < thresholds2.m && ["mm", minutes2] || hours2 <= 1 && ["h"] || hours2 < thresholds2.h && ["hh", hours2] || days2 <= 1 && ["d"] || days2 < thresholds2.d && ["dd", days2];
  if (thresholds2.w != null) {
    a2 = a2 || weeks2 <= 1 && ["w"] || weeks2 < thresholds2.w && ["ww", weeks2];
  }
  a2 = a2 || months2 <= 1 && ["M"] || months2 < thresholds2.M && ["MM", months2] || years2 <= 1 && ["y"] || ["yy", years2];
  a2[2] = withoutSuffix;
  a2[3] = +posNegDuration > 0;
  a2[4] = locale2;
  return substituteTimeAgo$1.apply(null, a2);
}
function getSetRelativeTimeRounding$1(roundingFunction) {
  if (roundingFunction === void 0) {
    return round$5;
  }
  if (typeof roundingFunction === "function") {
    round$5 = roundingFunction;
    return true;
  }
  return false;
}
function getSetRelativeTimeThreshold$1(threshold, limit) {
  if (thresholds$1[threshold] === void 0) {
    return false;
  }
  if (limit === void 0) {
    return thresholds$1[threshold];
  }
  thresholds$1[threshold] = limit;
  if (threshold === "s") {
    thresholds$1.ss = limit - 1;
  }
  return true;
}
function humanize$1(argWithSuffix, argThresholds) {
  if (!this.isValid()) {
    return this.localeData().invalidDate();
  }
  var withSuffix = false, th = thresholds$1, locale2, output2;
  if (typeof argWithSuffix === "object") {
    argThresholds = argWithSuffix;
    argWithSuffix = false;
  }
  if (typeof argWithSuffix === "boolean") {
    withSuffix = argWithSuffix;
  }
  if (typeof argThresholds === "object") {
    th = Object.assign({}, thresholds$1, argThresholds);
    if (argThresholds.s != null && argThresholds.ss == null) {
      th.ss = argThresholds.s - 1;
    }
  }
  locale2 = this.localeData();
  output2 = relativeTime$1$1(this, !withSuffix, th, locale2);
  if (withSuffix) {
    output2 = locale2.pastFuture(+this, output2);
  }
  return locale2.postformat(output2);
}
var abs$1$1 = Math.abs;
function sign$1(x2) {
  return (x2 > 0) - (x2 < 0) || +x2;
}
function toISOString$1$1() {
  if (!this.isValid()) {
    return this.localeData().invalidDate();
  }
  var seconds2 = abs$1$1(this._milliseconds) / 1e3, days2 = abs$1$1(this._days), months2 = abs$1$1(this._months), minutes2, hours2, years2, s2, total = this.asSeconds(), totalSign, ymSign, daysSign, hmsSign;
  if (!total) {
    return "P0D";
  }
  minutes2 = absFloor$1(seconds2 / 60);
  hours2 = absFloor$1(minutes2 / 60);
  seconds2 %= 60;
  minutes2 %= 60;
  years2 = absFloor$1(months2 / 12);
  months2 %= 12;
  s2 = seconds2 ? seconds2.toFixed(3).replace(/\.?0+$/, "") : "";
  totalSign = total < 0 ? "-" : "";
  ymSign = sign$1(this._months) !== sign$1(total) ? "-" : "";
  daysSign = sign$1(this._days) !== sign$1(total) ? "-" : "";
  hmsSign = sign$1(this._milliseconds) !== sign$1(total) ? "-" : "";
  return totalSign + "P" + (years2 ? ymSign + years2 + "Y" : "") + (months2 ? ymSign + months2 + "M" : "") + (days2 ? daysSign + days2 + "D" : "") + (hours2 || minutes2 || seconds2 ? "T" : "") + (hours2 ? hmsSign + hours2 + "H" : "") + (minutes2 ? hmsSign + minutes2 + "M" : "") + (seconds2 ? hmsSign + s2 + "S" : "");
}
var proto$2$1 = Duration$1.prototype;
proto$2$1.isValid = isValid$1$1;
proto$2$1.abs = abs$3;
proto$2$1.add = add$1$1;
proto$2$1.subtract = subtract$1$1;
proto$2$1.as = as$1;
proto$2$1.asMilliseconds = asMilliseconds$1;
proto$2$1.asSeconds = asSeconds$1;
proto$2$1.asMinutes = asMinutes$1;
proto$2$1.asHours = asHours$1;
proto$2$1.asDays = asDays$1;
proto$2$1.asWeeks = asWeeks$1;
proto$2$1.asMonths = asMonths$1;
proto$2$1.asQuarters = asQuarters$1;
proto$2$1.asYears = asYears$1;
proto$2$1.valueOf = valueOf$1$1;
proto$2$1._bubble = bubble$1;
proto$2$1.clone = clone$1$2;
proto$2$1.get = get$2$1;
proto$2$1.milliseconds = milliseconds$1;
proto$2$1.seconds = seconds$1;
proto$2$1.minutes = minutes$1;
proto$2$1.hours = hours$1;
proto$2$1.days = days$2;
proto$2$1.weeks = weeks$1;
proto$2$1.months = months$2;
proto$2$1.years = years$1;
proto$2$1.humanize = humanize$1;
proto$2$1.toISOString = toISOString$1$1;
proto$2$1.toString = toISOString$1$1;
proto$2$1.toJSON = toISOString$1$1;
proto$2$1.locale = locale$1;
proto$2$1.localeData = localeData$1;
proto$2$1.toIsoString = deprecate$1(
  "toIsoString() is deprecated. Please use toISOString() instead (notice the capitals)",
  toISOString$1$1
);
proto$2$1.lang = lang$1;
addFormatToken$1("X", 0, 0, "unix");
addFormatToken$1("x", 0, 0, "valueOf");
addRegexToken$1("x", matchSigned$1);
addRegexToken$1("X", matchTimestamp$1);
addParseToken$1("X", function(input, array2, config2) {
  config2._d = new Date(parseFloat(input) * 1e3);
});
addParseToken$1("x", function(input, array2, config2) {
  config2._d = new Date(toInt$1(input));
});
//! moment.js
hooks$2.version = "2.30.1";
setHookCallback$1(createLocal$1);
hooks$2.fn = proto$4;
hooks$2.min = min$9;
hooks$2.max = max$8;
hooks$2.now = now$2;
hooks$2.utc = createUTC$1;
hooks$2.unix = createUnix$1;
hooks$2.months = listMonths$1;
hooks$2.isDate = isDate$3;
hooks$2.locale = getSetGlobalLocale$1;
hooks$2.invalid = createInvalid$2;
hooks$2.duration = createDuration$1;
hooks$2.isMoment = isMoment$1;
hooks$2.weekdays = listWeekdays$1;
hooks$2.parseZone = createInZone$1;
hooks$2.localeData = getLocale$1;
hooks$2.isDuration = isDuration$1;
hooks$2.monthsShort = listMonthsShort$1;
hooks$2.weekdaysMin = listWeekdaysMin$1;
hooks$2.defineLocale = defineLocale$1;
hooks$2.updateLocale = updateLocale$1;
hooks$2.locales = listLocales$1;
hooks$2.weekdaysShort = listWeekdaysShort$1;
hooks$2.normalizeUnits = normalizeUnits$1;
hooks$2.relativeTimeRounding = getSetRelativeTimeRounding$1;
hooks$2.relativeTimeThreshold = getSetRelativeTimeThreshold$1;
hooks$2.calendarFormat = getCalendarFormat$1;
hooks$2.prototype = proto$4;
hooks$2.HTML5_FMT = {
  DATETIME_LOCAL: "YYYY-MM-DDTHH:mm",
  // <input type="datetime-local" />
  DATETIME_LOCAL_SECONDS: "YYYY-MM-DDTHH:mm:ss",
  // <input type="datetime-local" step="1" />
  DATETIME_LOCAL_MS: "YYYY-MM-DDTHH:mm:ss.SSS",
  // <input type="datetime-local" step="0.001" />
  DATE: "YYYY-MM-DD",
  // <input type="date" />
  TIME: "HH:mm",
  // <input type="time" />
  TIME_SECONDS: "HH:mm:ss",
  // <input type="time" step="1" />
  TIME_MS: "HH:mm:ss.SSS",
  // <input type="time" step="0.001" />
  WEEK: "GGGG-[W]WW",
  // <input type="week" />
  MONTH: "YYYY-MM"
  // <input type="month" />
};
let nanoid = (size = 21) => crypto.getRandomValues(new Uint8Array(size)).reduce((id, byte2) => {
  byte2 &= 63;
  if (byte2 < 36) {
    id += byte2.toString(36);
  } else if (byte2 < 62) {
    id += (byte2 - 26).toString(36).toUpperCase();
  } else if (byte2 > 62) {
    id += "-";
  } else {
    id += "_";
  }
  return id;
}, "");
/*! js-cookie v3.0.5 | MIT */
function assign$2(target) {
  for (var i2 = 1; i2 < arguments.length; i2++) {
    var source = arguments[i2];
    for (var key in source) {
      target[key] = source[key];
    }
  }
  return target;
}
var defaultConverter = {
  read: function(value) {
    if (value[0] === '"') {
      value = value.slice(1, -1);
    }
    return value.replace(/(%[\dA-F]{2})+/gi, decodeURIComponent);
  },
  write: function(value) {
    return encodeURIComponent(value).replace(
      /%(2[346BF]|3[AC-F]|40|5[BDE]|60|7[BCD])/g,
      decodeURIComponent
    );
  }
};
function init$4(converter, defaultAttributes) {
  function set2(name, value, attributes) {
    if (typeof document === "undefined") {
      return;
    }
    attributes = assign$2({}, defaultAttributes, attributes);
    if (typeof attributes.expires === "number") {
      attributes.expires = new Date(Date.now() + attributes.expires * 864e5);
    }
    if (attributes.expires) {
      attributes.expires = attributes.expires.toUTCString();
    }
    name = encodeURIComponent(name).replace(/%(2[346B]|5E|60|7C)/g, decodeURIComponent).replace(/[()]/g, escape);
    var stringifiedAttributes = "";
    for (var attributeName in attributes) {
      if (!attributes[attributeName]) {
        continue;
      }
      stringifiedAttributes += "; " + attributeName;
      if (attributes[attributeName] === true) {
        continue;
      }
      stringifiedAttributes += "=" + attributes[attributeName].split(";")[0];
    }
    return document.cookie = name + "=" + converter.write(value, name) + stringifiedAttributes;
  }
  function get4(name) {
    if (typeof document === "undefined" || arguments.length && !name) {
      return;
    }
    var cookies2 = document.cookie ? document.cookie.split("; ") : [];
    var jar = {};
    for (var i2 = 0; i2 < cookies2.length; i2++) {
      var parts = cookies2[i2].split("=");
      var value = parts.slice(1).join("=");
      try {
        var found = decodeURIComponent(parts[0]);
        jar[found] = converter.read(value, found);
        if (name === found) {
          break;
        }
      } catch (e2) {
      }
    }
    return name ? jar[name] : jar;
  }
  return Object.create(
    {
      set: set2,
      get: get4,
      remove: function(name, attributes) {
        set2(
          name,
          "",
          assign$2({}, attributes, {
            expires: -1
          })
        );
      },
      withAttributes: function(attributes) {
        return init$4(this.converter, assign$2({}, this.attributes, attributes));
      },
      withConverter: function(converter2) {
        return init$4(assign$2({}, this.converter, converter2), this.attributes);
      }
    },
    {
      attributes: { value: Object.freeze(defaultAttributes) },
      converter: { value: Object.freeze(converter) }
    }
  );
}
var api = init$4(defaultConverter, { path: "/" });
function bind$9(fn, thisArg) {
  return function wrap() {
    return fn.apply(thisArg, arguments);
  };
}
const { toString: toString$e } = Object.prototype;
const { getPrototypeOf: getPrototypeOf$3 } = Object;
const kindOf = ((cache) => (thing) => {
  const str = toString$e.call(thing);
  return cache[str] || (cache[str] = str.slice(8, -1).toLowerCase());
})(/* @__PURE__ */ Object.create(null));
const kindOfTest = (type2) => {
  type2 = type2.toLowerCase();
  return (thing) => kindOf(thing) === type2;
};
const typeOfTest = (type2) => (thing) => typeof thing === type2;
const { isArray: isArray$6 } = Array;
const isUndefined$2 = typeOfTest("undefined");
function isBuffer$1(val) {
  return val !== null && !isUndefined$2(val) && val.constructor !== null && !isUndefined$2(val.constructor) && isFunction$4(val.constructor.isBuffer) && val.constructor.isBuffer(val);
}
const isArrayBuffer = kindOfTest("ArrayBuffer");
function isArrayBufferView(val) {
  let result;
  if (typeof ArrayBuffer !== "undefined" && ArrayBuffer.isView) {
    result = ArrayBuffer.isView(val);
  } else {
    result = val && val.buffer && isArrayBuffer(val.buffer);
  }
  return result;
}
const isString$2 = typeOfTest("string");
const isFunction$4 = typeOfTest("function");
const isNumber$3 = typeOfTest("number");
const isObject$l = (thing) => thing !== null && typeof thing === "object";
const isBoolean = (thing) => thing === true || thing === false;
const isPlainObject$2 = (val) => {
  if (kindOf(val) !== "object") {
    return false;
  }
  const prototype2 = getPrototypeOf$3(val);
  return (prototype2 === null || prototype2 === Object.prototype || Object.getPrototypeOf(prototype2) === null) && !(Symbol.toStringTag in val) && !(Symbol.iterator in val);
};
const isDate$2 = kindOfTest("Date");
const isFile = kindOfTest("File");
const isBlob = kindOfTest("Blob");
const isFileList = kindOfTest("FileList");
const isStream = (val) => isObject$l(val) && isFunction$4(val.pipe);
const isFormData = (thing) => {
  let kind;
  return thing && (typeof FormData === "function" && thing instanceof FormData || isFunction$4(thing.append) && ((kind = kindOf(thing)) === "formdata" || // detect form-data instance
  kind === "object" && isFunction$4(thing.toString) && thing.toString() === "[object FormData]"));
};
const isURLSearchParams = kindOfTest("URLSearchParams");
const trim$2 = (str) => str.trim ? str.trim() : str.replace(/^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/g, "");
function forEach$1(obj, fn, { allOwnKeys = false } = {}) {
  if (obj === null || typeof obj === "undefined") {
    return;
  }
  let i2;
  let l2;
  if (typeof obj !== "object") {
    obj = [obj];
  }
  if (isArray$6(obj)) {
    for (i2 = 0, l2 = obj.length; i2 < l2; i2++) {
      fn.call(null, obj[i2], i2, obj);
    }
  } else {
    const keys4 = allOwnKeys ? Object.getOwnPropertyNames(obj) : Object.keys(obj);
    const len2 = keys4.length;
    let key;
    for (i2 = 0; i2 < len2; i2++) {
      key = keys4[i2];
      fn.call(null, obj[key], key, obj);
    }
  }
}
function findKey(obj, key) {
  key = key.toLowerCase();
  const keys4 = Object.keys(obj);
  let i2 = keys4.length;
  let _key;
  while (i2-- > 0) {
    _key = keys4[i2];
    if (key === _key.toLowerCase()) {
      return _key;
    }
  }
  return null;
}
const _global = (() => {
  if (typeof globalThis !== "undefined")
    return globalThis;
  return typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : global;
})();
const isContextDefined = (context) => !isUndefined$2(context) && context !== _global;
function merge$2() {
  const { caseless } = isContextDefined(this) && this || {};
  const result = {};
  const assignValue = (val, key) => {
    const targetKey = caseless && findKey(result, key) || key;
    if (isPlainObject$2(result[targetKey]) && isPlainObject$2(val)) {
      result[targetKey] = merge$2(result[targetKey], val);
    } else if (isPlainObject$2(val)) {
      result[targetKey] = merge$2({}, val);
    } else if (isArray$6(val)) {
      result[targetKey] = val.slice();
    } else {
      result[targetKey] = val;
    }
  };
  for (let i2 = 0, l2 = arguments.length; i2 < l2; i2++) {
    arguments[i2] && forEach$1(arguments[i2], assignValue);
  }
  return result;
}
const extend$5 = (a2, b2, thisArg, { allOwnKeys } = {}) => {
  forEach$1(b2, (val, key) => {
    if (thisArg && isFunction$4(val)) {
      a2[key] = bind$9(val, thisArg);
    } else {
      a2[key] = val;
    }
  }, { allOwnKeys });
  return a2;
};
const stripBOM = (content2) => {
  if (content2.charCodeAt(0) === 65279) {
    content2 = content2.slice(1);
  }
  return content2;
};
const inherits$1 = (constructor, superConstructor, props, descriptors2) => {
  constructor.prototype = Object.create(superConstructor.prototype, descriptors2);
  constructor.prototype.constructor = constructor;
  Object.defineProperty(constructor, "super", {
    value: superConstructor.prototype
  });
  props && Object.assign(constructor.prototype, props);
};
const toFlatObject = (sourceObj, destObj, filter3, propFilter) => {
  let props;
  let i2;
  let prop;
  const merged = {};
  destObj = destObj || {};
  if (sourceObj == null)
    return destObj;
  do {
    props = Object.getOwnPropertyNames(sourceObj);
    i2 = props.length;
    while (i2-- > 0) {
      prop = props[i2];
      if ((!propFilter || propFilter(prop, sourceObj, destObj)) && !merged[prop]) {
        destObj[prop] = sourceObj[prop];
        merged[prop] = true;
      }
    }
    sourceObj = filter3 !== false && getPrototypeOf$3(sourceObj);
  } while (sourceObj && (!filter3 || filter3(sourceObj, destObj)) && sourceObj !== Object.prototype);
  return destObj;
};
const endsWith = (str, searchString, position2) => {
  str = String(str);
  if (position2 === void 0 || position2 > str.length) {
    position2 = str.length;
  }
  position2 -= searchString.length;
  const lastIndex = str.indexOf(searchString, position2);
  return lastIndex !== -1 && lastIndex === position2;
};
const toArray$2 = (thing) => {
  if (!thing)
    return null;
  if (isArray$6(thing))
    return thing;
  let i2 = thing.length;
  if (!isNumber$3(i2))
    return null;
  const arr = new Array(i2);
  while (i2-- > 0) {
    arr[i2] = thing[i2];
  }
  return arr;
};
const isTypedArray$1 = ((TypedArray2) => {
  return (thing) => {
    return TypedArray2 && thing instanceof TypedArray2;
  };
})(typeof Uint8Array !== "undefined" && getPrototypeOf$3(Uint8Array));
const forEachEntry = (obj, fn) => {
  const generator = obj && obj[Symbol.iterator];
  const iterator2 = generator.call(obj);
  let result;
  while ((result = iterator2.next()) && !result.done) {
    const pair = result.value;
    fn.call(obj, pair[0], pair[1]);
  }
};
const matchAll = (regExp, str) => {
  let matches;
  const arr = [];
  while ((matches = regExp.exec(str)) !== null) {
    arr.push(matches);
  }
  return arr;
};
const isHTMLForm = kindOfTest("HTMLFormElement");
const toCamelCase$1 = (str) => {
  return str.toLowerCase().replace(
    /[-_\s]([a-z\d])(\w*)/g,
    function replacer(m2, p1, p2) {
      return p1.toUpperCase() + p2;
    }
  );
};
const hasOwnProperty$3 = (({ hasOwnProperty: hasOwnProperty2 }) => (obj, prop) => hasOwnProperty2.call(obj, prop))(Object.prototype);
const isRegExp$2 = kindOfTest("RegExp");
const reduceDescriptors = (obj, reducer) => {
  const descriptors2 = Object.getOwnPropertyDescriptors(obj);
  const reducedDescriptors = {};
  forEach$1(descriptors2, (descriptor, name) => {
    let ret;
    if ((ret = reducer(descriptor, name, obj)) !== false) {
      reducedDescriptors[name] = ret || descriptor;
    }
  });
  Object.defineProperties(obj, reducedDescriptors);
};
const freezeMethods = (obj) => {
  reduceDescriptors(obj, (descriptor, name) => {
    if (isFunction$4(obj) && ["arguments", "caller", "callee"].indexOf(name) !== -1) {
      return false;
    }
    const value = obj[name];
    if (!isFunction$4(value))
      return;
    descriptor.enumerable = false;
    if ("writable" in descriptor) {
      descriptor.writable = false;
      return;
    }
    if (!descriptor.set) {
      descriptor.set = () => {
        throw Error("Can not rewrite read-only method '" + name + "'");
      };
    }
  });
};
const toObjectSet = (arrayOrString, delimiter) => {
  const obj = {};
  const define2 = (arr) => {
    arr.forEach((value) => {
      obj[value] = true;
    });
  };
  isArray$6(arrayOrString) ? define2(arrayOrString) : define2(String(arrayOrString).split(delimiter));
  return obj;
};
const noop$3 = () => {
};
const toFiniteNumber = (value, defaultValue) => {
  value = +value;
  return Number.isFinite(value) ? value : defaultValue;
};
const ALPHA = "abcdefghijklmnopqrstuvwxyz";
const DIGIT = "0123456789";
const ALPHABET = {
  DIGIT,
  ALPHA,
  ALPHA_DIGIT: ALPHA + ALPHA.toUpperCase() + DIGIT
};
const generateString = (size = 16, alphabet = ALPHABET.ALPHA_DIGIT) => {
  let str = "";
  const { length } = alphabet;
  while (size--) {
    str += alphabet[Math.random() * length | 0];
  }
  return str;
};
function isSpecCompliantForm(thing) {
  return !!(thing && isFunction$4(thing.append) && thing[Symbol.toStringTag] === "FormData" && thing[Symbol.iterator]);
}
const toJSONObject = (obj) => {
  const stack = new Array(10);
  const visit = (source, i2) => {
    if (isObject$l(source)) {
      if (stack.indexOf(source) >= 0) {
        return;
      }
      if (!("toJSON" in source)) {
        stack[i2] = source;
        const target = isArray$6(source) ? [] : {};
        forEach$1(source, (value, key) => {
          const reducedValue = visit(value, i2 + 1);
          !isUndefined$2(reducedValue) && (target[key] = reducedValue);
        });
        stack[i2] = void 0;
        return target;
      }
    }
    return source;
  };
  return visit(obj, 0);
};
const isAsyncFn = kindOfTest("AsyncFunction");
const isThenable$1 = (thing) => thing && (isObject$l(thing) || isFunction$4(thing)) && isFunction$4(thing.then) && isFunction$4(thing.catch);
const utils$4 = {
  isArray: isArray$6,
  isArrayBuffer,
  isBuffer: isBuffer$1,
  isFormData,
  isArrayBufferView,
  isString: isString$2,
  isNumber: isNumber$3,
  isBoolean,
  isObject: isObject$l,
  isPlainObject: isPlainObject$2,
  isUndefined: isUndefined$2,
  isDate: isDate$2,
  isFile,
  isBlob,
  isRegExp: isRegExp$2,
  isFunction: isFunction$4,
  isStream,
  isURLSearchParams,
  isTypedArray: isTypedArray$1,
  isFileList,
  forEach: forEach$1,
  merge: merge$2,
  extend: extend$5,
  trim: trim$2,
  stripBOM,
  inherits: inherits$1,
  toFlatObject,
  kindOf,
  kindOfTest,
  endsWith,
  toArray: toArray$2,
  forEachEntry,
  matchAll,
  isHTMLForm,
  hasOwnProperty: hasOwnProperty$3,
  hasOwnProp: hasOwnProperty$3,
  // an alias to avoid ESLint no-prototype-builtins detection
  reduceDescriptors,
  freezeMethods,
  toObjectSet,
  toCamelCase: toCamelCase$1,
  noop: noop$3,
  toFiniteNumber,
  findKey,
  global: _global,
  isContextDefined,
  ALPHABET,
  generateString,
  isSpecCompliantForm,
  toJSONObject,
  isAsyncFn,
  isThenable: isThenable$1
};
function AxiosError(message, code, config2, request, response) {
  Error.call(this);
  if (Error.captureStackTrace) {
    Error.captureStackTrace(this, this.constructor);
  } else {
    this.stack = new Error().stack;
  }
  this.message = message;
  this.name = "AxiosError";
  code && (this.code = code);
  config2 && (this.config = config2);
  request && (this.request = request);
  response && (this.response = response);
}
utils$4.inherits(AxiosError, Error, {
  toJSON: function toJSON() {
    return {
      // Standard
      message: this.message,
      name: this.name,
      // Microsoft
      description: this.description,
      number: this.number,
      // Mozilla
      fileName: this.fileName,
      lineNumber: this.lineNumber,
      columnNumber: this.columnNumber,
      stack: this.stack,
      // Axios
      config: utils$4.toJSONObject(this.config),
      code: this.code,
      status: this.response && this.response.status ? this.response.status : null
    };
  }
});
const prototype$1 = AxiosError.prototype;
const descriptors$1 = {};
[
  "ERR_BAD_OPTION_VALUE",
  "ERR_BAD_OPTION",
  "ECONNABORTED",
  "ETIMEDOUT",
  "ERR_NETWORK",
  "ERR_FR_TOO_MANY_REDIRECTS",
  "ERR_DEPRECATED",
  "ERR_BAD_RESPONSE",
  "ERR_BAD_REQUEST",
  "ERR_CANCELED",
  "ERR_NOT_SUPPORT",
  "ERR_INVALID_URL"
  // eslint-disable-next-line func-names
].forEach((code) => {
  descriptors$1[code] = { value: code };
});
Object.defineProperties(AxiosError, descriptors$1);
Object.defineProperty(prototype$1, "isAxiosError", { value: true });
AxiosError.from = (error, code, config2, request, response, customProps) => {
  const axiosError = Object.create(prototype$1);
  utils$4.toFlatObject(error, axiosError, function filter3(obj) {
    return obj !== Error.prototype;
  }, (prop) => {
    return prop !== "isAxiosError";
  });
  AxiosError.call(axiosError, error.message, code, config2, request, response);
  axiosError.cause = error;
  axiosError.name = error.name;
  customProps && Object.assign(axiosError, customProps);
  return axiosError;
};
const httpAdapter = null;
function isVisitable(thing) {
  return utils$4.isPlainObject(thing) || utils$4.isArray(thing);
}
function removeBrackets(key) {
  return utils$4.endsWith(key, "[]") ? key.slice(0, -2) : key;
}
function renderKey(path, key, dots) {
  if (!path)
    return key;
  return path.concat(key).map(function each2(token2, i2) {
    token2 = removeBrackets(token2);
    return !dots && i2 ? "[" + token2 + "]" : token2;
  }).join(dots ? "." : "");
}
function isFlatArray(arr) {
  return utils$4.isArray(arr) && !arr.some(isVisitable);
}
const predicates = utils$4.toFlatObject(utils$4, {}, null, function filter(prop) {
  return /^is[A-Z]/.test(prop);
});
function toFormData(obj, formData, options) {
  if (!utils$4.isObject(obj)) {
    throw new TypeError("target must be an object");
  }
  formData = formData || new FormData();
  options = utils$4.toFlatObject(options, {
    metaTokens: true,
    dots: false,
    indexes: false
  }, false, function defined(option, source) {
    return !utils$4.isUndefined(source[option]);
  });
  const metaTokens = options.metaTokens;
  const visitor = options.visitor || defaultVisitor;
  const dots = options.dots;
  const indexes = options.indexes;
  const _Blob = options.Blob || typeof Blob !== "undefined" && Blob;
  const useBlob = _Blob && utils$4.isSpecCompliantForm(formData);
  if (!utils$4.isFunction(visitor)) {
    throw new TypeError("visitor must be a function");
  }
  function convertValue(value) {
    if (value === null)
      return "";
    if (utils$4.isDate(value)) {
      return value.toISOString();
    }
    if (!useBlob && utils$4.isBlob(value)) {
      throw new AxiosError("Blob is not supported. Use a Buffer instead.");
    }
    if (utils$4.isArrayBuffer(value) || utils$4.isTypedArray(value)) {
      return useBlob && typeof Blob === "function" ? new Blob([value]) : Buffer.from(value);
    }
    return value;
  }
  function defaultVisitor(value, key, path) {
    let arr = value;
    if (value && !path && typeof value === "object") {
      if (utils$4.endsWith(key, "{}")) {
        key = metaTokens ? key : key.slice(0, -2);
        value = JSON.stringify(value);
      } else if (utils$4.isArray(value) && isFlatArray(value) || (utils$4.isFileList(value) || utils$4.endsWith(key, "[]")) && (arr = utils$4.toArray(value))) {
        key = removeBrackets(key);
        arr.forEach(function each2(el, index2) {
          !(utils$4.isUndefined(el) || el === null) && formData.append(
            // eslint-disable-next-line no-nested-ternary
            indexes === true ? renderKey([key], index2, dots) : indexes === null ? key : key + "[]",
            convertValue(el)
          );
        });
        return false;
      }
    }
    if (isVisitable(value)) {
      return true;
    }
    formData.append(renderKey(path, key, dots), convertValue(value));
    return false;
  }
  const stack = [];
  const exposedHelpers = Object.assign(predicates, {
    defaultVisitor,
    convertValue,
    isVisitable
  });
  function build(value, path) {
    if (utils$4.isUndefined(value))
      return;
    if (stack.indexOf(value) !== -1) {
      throw Error("Circular reference detected in " + path.join("."));
    }
    stack.push(value);
    utils$4.forEach(value, function each2(el, key) {
      const result = !(utils$4.isUndefined(el) || el === null) && visitor.call(
        formData,
        el,
        utils$4.isString(key) ? key.trim() : key,
        path,
        exposedHelpers
      );
      if (result === true) {
        build(el, path ? path.concat(key) : [key]);
      }
    });
    stack.pop();
  }
  if (!utils$4.isObject(obj)) {
    throw new TypeError("data must be an object");
  }
  build(obj);
  return formData;
}
function encode$3(str) {
  const charMap = {
    "!": "%21",
    "'": "%27",
    "(": "%28",
    ")": "%29",
    "~": "%7E",
    "%20": "+",
    "%00": "\0"
  };
  return encodeURIComponent(str).replace(/[!'()~]|%20|%00/g, function replacer(match5) {
    return charMap[match5];
  });
}
function AxiosURLSearchParams(params, options) {
  this._pairs = [];
  params && toFormData(params, this, options);
}
const prototype = AxiosURLSearchParams.prototype;
prototype.append = function append(name, value) {
  this._pairs.push([name, value]);
};
prototype.toString = function toString(encoder2) {
  const _encode = encoder2 ? function(value) {
    return encoder2.call(this, value, encode$3);
  } : encode$3;
  return this._pairs.map(function each2(pair) {
    return _encode(pair[0]) + "=" + _encode(pair[1]);
  }, "").join("&");
};
function encode$2(val) {
  return encodeURIComponent(val).replace(/%3A/gi, ":").replace(/%24/g, "$").replace(/%2C/gi, ",").replace(/%20/g, "+").replace(/%5B/gi, "[").replace(/%5D/gi, "]");
}
function buildURL(url, params, options) {
  if (!params) {
    return url;
  }
  const _encode = options && options.encode || encode$2;
  const serializeFn = options && options.serialize;
  let serializedParams;
  if (serializeFn) {
    serializedParams = serializeFn(params, options);
  } else {
    serializedParams = utils$4.isURLSearchParams(params) ? params.toString() : new AxiosURLSearchParams(params, options).toString(_encode);
  }
  if (serializedParams) {
    const hashmarkIndex = url.indexOf("#");
    if (hashmarkIndex !== -1) {
      url = url.slice(0, hashmarkIndex);
    }
    url += (url.indexOf("?") === -1 ? "?" : "&") + serializedParams;
  }
  return url;
}
class InterceptorManager {
  constructor() {
    this.handlers = [];
  }
  /**
   * Add a new interceptor to the stack
   *
   * @param {Function} fulfilled The function to handle `then` for a `Promise`
   * @param {Function} rejected The function to handle `reject` for a `Promise`
   *
   * @return {Number} An ID used to remove interceptor later
   */
  use(fulfilled, rejected, options) {
    this.handlers.push({
      fulfilled,
      rejected,
      synchronous: options ? options.synchronous : false,
      runWhen: options ? options.runWhen : null
    });
    return this.handlers.length - 1;
  }
  /**
   * Remove an interceptor from the stack
   *
   * @param {Number} id The ID that was returned by `use`
   *
   * @returns {Boolean} `true` if the interceptor was removed, `false` otherwise
   */
  eject(id) {
    if (this.handlers[id]) {
      this.handlers[id] = null;
    }
  }
  /**
   * Clear all interceptors from the stack
   *
   * @returns {void}
   */
  clear() {
    if (this.handlers) {
      this.handlers = [];
    }
  }
  /**
   * Iterate over all the registered interceptors
   *
   * This method is particularly useful for skipping over any
   * interceptors that may have become `null` calling `eject`.
   *
   * @param {Function} fn The function to call for each interceptor
   *
   * @returns {void}
   */
  forEach(fn) {
    utils$4.forEach(this.handlers, function forEachHandler(h2) {
      if (h2 !== null) {
        fn(h2);
      }
    });
  }
}
const InterceptorManager$1 = InterceptorManager;
const transitionalDefaults = {
  silentJSONParsing: true,
  forcedJSONParsing: true,
  clarifyTimeoutError: false
};
const URLSearchParams$1 = typeof URLSearchParams !== "undefined" ? URLSearchParams : AxiosURLSearchParams;
const FormData$1 = typeof FormData !== "undefined" ? FormData : null;
const Blob$1 = typeof Blob !== "undefined" ? Blob : null;
const platform$2 = {
  isBrowser: true,
  classes: {
    URLSearchParams: URLSearchParams$1,
    FormData: FormData$1,
    Blob: Blob$1
  },
  protocols: ["http", "https", "file", "blob", "url", "data"]
};
const hasBrowserEnv = typeof window !== "undefined" && typeof document !== "undefined";
const hasStandardBrowserEnv = ((product) => {
  return hasBrowserEnv && ["ReactNative", "NativeScript", "NS"].indexOf(product) < 0;
})(typeof navigator !== "undefined" && navigator.product);
const hasStandardBrowserWebWorkerEnv = (() => {
  return typeof WorkerGlobalScope !== "undefined" && // eslint-disable-next-line no-undef
  self instanceof WorkerGlobalScope && typeof self.importScripts === "function";
})();
const utils$3 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  hasBrowserEnv,
  hasStandardBrowserEnv,
  hasStandardBrowserWebWorkerEnv
}, Symbol.toStringTag, { value: "Module" }));
const platform$1 = {
  ...utils$3,
  ...platform$2
};
function toURLEncodedForm(data2, options) {
  return toFormData(data2, new platform$1.classes.URLSearchParams(), Object.assign({
    visitor: function(value, key, path, helpers) {
      if (platform$1.isNode && utils$4.isBuffer(value)) {
        this.append(key, value.toString("base64"));
        return false;
      }
      return helpers.defaultVisitor.apply(this, arguments);
    }
  }, options));
}
function parsePropPath(name) {
  return utils$4.matchAll(/\w+|\[(\w*)]/g, name).map((match5) => {
    return match5[0] === "[]" ? "" : match5[1] || match5[0];
  });
}
function arrayToObject(arr) {
  const obj = {};
  const keys4 = Object.keys(arr);
  let i2;
  const len2 = keys4.length;
  let key;
  for (i2 = 0; i2 < len2; i2++) {
    key = keys4[i2];
    obj[key] = arr[key];
  }
  return obj;
}
function formDataToJSON(formData) {
  function buildPath2(path, value, target, index2) {
    let name = path[index2++];
    if (name === "__proto__")
      return true;
    const isNumericKey = Number.isFinite(+name);
    const isLast = index2 >= path.length;
    name = !name && utils$4.isArray(target) ? target.length : name;
    if (isLast) {
      if (utils$4.hasOwnProp(target, name)) {
        target[name] = [target[name], value];
      } else {
        target[name] = value;
      }
      return !isNumericKey;
    }
    if (!target[name] || !utils$4.isObject(target[name])) {
      target[name] = [];
    }
    const result = buildPath2(path, value, target[name], index2);
    if (result && utils$4.isArray(target[name])) {
      target[name] = arrayToObject(target[name]);
    }
    return !isNumericKey;
  }
  if (utils$4.isFormData(formData) && utils$4.isFunction(formData.entries)) {
    const obj = {};
    utils$4.forEachEntry(formData, (name, value) => {
      buildPath2(parsePropPath(name), value, obj, 0);
    });
    return obj;
  }
  return null;
}
function stringifySafely(rawValue, parser, encoder2) {
  if (utils$4.isString(rawValue)) {
    try {
      (parser || JSON.parse)(rawValue);
      return utils$4.trim(rawValue);
    } catch (e2) {
      if (e2.name !== "SyntaxError") {
        throw e2;
      }
    }
  }
  return (encoder2 || JSON.stringify)(rawValue);
}
const defaults$4 = {
  transitional: transitionalDefaults,
  adapter: ["xhr", "http"],
  transformRequest: [function transformRequest(data2, headers) {
    const contentType = headers.getContentType() || "";
    const hasJSONContentType = contentType.indexOf("application/json") > -1;
    const isObjectPayload = utils$4.isObject(data2);
    if (isObjectPayload && utils$4.isHTMLForm(data2)) {
      data2 = new FormData(data2);
    }
    const isFormData2 = utils$4.isFormData(data2);
    if (isFormData2) {
      return hasJSONContentType ? JSON.stringify(formDataToJSON(data2)) : data2;
    }
    if (utils$4.isArrayBuffer(data2) || utils$4.isBuffer(data2) || utils$4.isStream(data2) || utils$4.isFile(data2) || utils$4.isBlob(data2)) {
      return data2;
    }
    if (utils$4.isArrayBufferView(data2)) {
      return data2.buffer;
    }
    if (utils$4.isURLSearchParams(data2)) {
      headers.setContentType("application/x-www-form-urlencoded;charset=utf-8", false);
      return data2.toString();
    }
    let isFileList2;
    if (isObjectPayload) {
      if (contentType.indexOf("application/x-www-form-urlencoded") > -1) {
        return toURLEncodedForm(data2, this.formSerializer).toString();
      }
      if ((isFileList2 = utils$4.isFileList(data2)) || contentType.indexOf("multipart/form-data") > -1) {
        const _FormData = this.env && this.env.FormData;
        return toFormData(
          isFileList2 ? { "files[]": data2 } : data2,
          _FormData && new _FormData(),
          this.formSerializer
        );
      }
    }
    if (isObjectPayload || hasJSONContentType) {
      headers.setContentType("application/json", false);
      return stringifySafely(data2);
    }
    return data2;
  }],
  transformResponse: [function transformResponse(data2) {
    const transitional2 = this.transitional || defaults$4.transitional;
    const forcedJSONParsing = transitional2 && transitional2.forcedJSONParsing;
    const JSONRequested = this.responseType === "json";
    if (data2 && utils$4.isString(data2) && (forcedJSONParsing && !this.responseType || JSONRequested)) {
      const silentJSONParsing = transitional2 && transitional2.silentJSONParsing;
      const strictJSONParsing = !silentJSONParsing && JSONRequested;
      try {
        return JSON.parse(data2);
      } catch (e2) {
        if (strictJSONParsing) {
          if (e2.name === "SyntaxError") {
            throw AxiosError.from(e2, AxiosError.ERR_BAD_RESPONSE, this, null, this.response);
          }
          throw e2;
        }
      }
    }
    return data2;
  }],
  /**
   * A timeout in milliseconds to abort a request. If set to 0 (default) a
   * timeout is not created.
   */
  timeout: 0,
  xsrfCookieName: "XSRF-TOKEN",
  xsrfHeaderName: "X-XSRF-TOKEN",
  maxContentLength: -1,
  maxBodyLength: -1,
  env: {
    FormData: platform$1.classes.FormData,
    Blob: platform$1.classes.Blob
  },
  validateStatus: function validateStatus(status) {
    return status >= 200 && status < 300;
  },
  headers: {
    common: {
      "Accept": "application/json, text/plain, */*",
      "Content-Type": void 0
    }
  }
};
utils$4.forEach(["delete", "get", "head", "post", "put", "patch"], (method) => {
  defaults$4.headers[method] = {};
});
const defaults$5 = defaults$4;
const ignoreDuplicateOf = utils$4.toObjectSet([
  "age",
  "authorization",
  "content-length",
  "content-type",
  "etag",
  "expires",
  "from",
  "host",
  "if-modified-since",
  "if-unmodified-since",
  "last-modified",
  "location",
  "max-forwards",
  "proxy-authorization",
  "referer",
  "retry-after",
  "user-agent"
]);
const parseHeaders = (rawHeaders) => {
  const parsed = {};
  let key;
  let val;
  let i2;
  rawHeaders && rawHeaders.split("\n").forEach(function parser(line) {
    i2 = line.indexOf(":");
    key = line.substring(0, i2).trim().toLowerCase();
    val = line.substring(i2 + 1).trim();
    if (!key || parsed[key] && ignoreDuplicateOf[key]) {
      return;
    }
    if (key === "set-cookie") {
      if (parsed[key]) {
        parsed[key].push(val);
      } else {
        parsed[key] = [val];
      }
    } else {
      parsed[key] = parsed[key] ? parsed[key] + ", " + val : val;
    }
  });
  return parsed;
};
const $internals = Symbol("internals");
function normalizeHeader(header) {
  return header && String(header).trim().toLowerCase();
}
function normalizeValue(value) {
  if (value === false || value == null) {
    return value;
  }
  return utils$4.isArray(value) ? value.map(normalizeValue) : String(value);
}
function parseTokens(str) {
  const tokens2 = /* @__PURE__ */ Object.create(null);
  const tokensRE = /([^\s,;=]+)\s*(?:=\s*([^,;]+))?/g;
  let match5;
  while (match5 = tokensRE.exec(str)) {
    tokens2[match5[1]] = match5[2];
  }
  return tokens2;
}
const isValidHeaderName = (str) => /^[-_a-zA-Z0-9^`|~,!#$%&'*+.]+$/.test(str.trim());
function matchHeaderValue(context, value, header, filter3, isHeaderNameFilter) {
  if (utils$4.isFunction(filter3)) {
    return filter3.call(this, value, header);
  }
  if (isHeaderNameFilter) {
    value = header;
  }
  if (!utils$4.isString(value))
    return;
  if (utils$4.isString(filter3)) {
    return value.indexOf(filter3) !== -1;
  }
  if (utils$4.isRegExp(filter3)) {
    return filter3.test(value);
  }
}
function formatHeader(header) {
  return header.trim().toLowerCase().replace(/([a-z\d])(\w*)/g, (w2, char, str) => {
    return char.toUpperCase() + str;
  });
}
function buildAccessors(obj, header) {
  const accessorName = utils$4.toCamelCase(" " + header);
  ["get", "set", "has"].forEach((methodName) => {
    Object.defineProperty(obj, methodName + accessorName, {
      value: function(arg1, arg2, arg3) {
        return this[methodName].call(this, header, arg1, arg2, arg3);
      },
      configurable: true
    });
  });
}
class AxiosHeaders {
  constructor(headers) {
    headers && this.set(headers);
  }
  set(header, valueOrRewrite, rewrite) {
    const self2 = this;
    function setHeader(_value, _header, _rewrite) {
      const lHeader = normalizeHeader(_header);
      if (!lHeader) {
        throw new Error("header name must be a non-empty string");
      }
      const key = utils$4.findKey(self2, lHeader);
      if (!key || self2[key] === void 0 || _rewrite === true || _rewrite === void 0 && self2[key] !== false) {
        self2[key || _header] = normalizeValue(_value);
      }
    }
    const setHeaders = (headers, _rewrite) => utils$4.forEach(headers, (_value, _header) => setHeader(_value, _header, _rewrite));
    if (utils$4.isPlainObject(header) || header instanceof this.constructor) {
      setHeaders(header, valueOrRewrite);
    } else if (utils$4.isString(header) && (header = header.trim()) && !isValidHeaderName(header)) {
      setHeaders(parseHeaders(header), valueOrRewrite);
    } else {
      header != null && setHeader(valueOrRewrite, header, rewrite);
    }
    return this;
  }
  get(header, parser) {
    header = normalizeHeader(header);
    if (header) {
      const key = utils$4.findKey(this, header);
      if (key) {
        const value = this[key];
        if (!parser) {
          return value;
        }
        if (parser === true) {
          return parseTokens(value);
        }
        if (utils$4.isFunction(parser)) {
          return parser.call(this, value, key);
        }
        if (utils$4.isRegExp(parser)) {
          return parser.exec(value);
        }
        throw new TypeError("parser must be boolean|regexp|function");
      }
    }
  }
  has(header, matcher) {
    header = normalizeHeader(header);
    if (header) {
      const key = utils$4.findKey(this, header);
      return !!(key && this[key] !== void 0 && (!matcher || matchHeaderValue(this, this[key], key, matcher)));
    }
    return false;
  }
  delete(header, matcher) {
    const self2 = this;
    let deleted = false;
    function deleteHeader(_header) {
      _header = normalizeHeader(_header);
      if (_header) {
        const key = utils$4.findKey(self2, _header);
        if (key && (!matcher || matchHeaderValue(self2, self2[key], key, matcher))) {
          delete self2[key];
          deleted = true;
        }
      }
    }
    if (utils$4.isArray(header)) {
      header.forEach(deleteHeader);
    } else {
      deleteHeader(header);
    }
    return deleted;
  }
  clear(matcher) {
    const keys4 = Object.keys(this);
    let i2 = keys4.length;
    let deleted = false;
    while (i2--) {
      const key = keys4[i2];
      if (!matcher || matchHeaderValue(this, this[key], key, matcher, true)) {
        delete this[key];
        deleted = true;
      }
    }
    return deleted;
  }
  normalize(format2) {
    const self2 = this;
    const headers = {};
    utils$4.forEach(this, (value, header) => {
      const key = utils$4.findKey(headers, header);
      if (key) {
        self2[key] = normalizeValue(value);
        delete self2[header];
        return;
      }
      const normalized = format2 ? formatHeader(header) : String(header).trim();
      if (normalized !== header) {
        delete self2[header];
      }
      self2[normalized] = normalizeValue(value);
      headers[normalized] = true;
    });
    return this;
  }
  concat(...targets) {
    return this.constructor.concat(this, ...targets);
  }
  toJSON(asStrings) {
    const obj = /* @__PURE__ */ Object.create(null);
    utils$4.forEach(this, (value, header) => {
      value != null && value !== false && (obj[header] = asStrings && utils$4.isArray(value) ? value.join(", ") : value);
    });
    return obj;
  }
  [Symbol.iterator]() {
    return Object.entries(this.toJSON())[Symbol.iterator]();
  }
  toString() {
    return Object.entries(this.toJSON()).map(([header, value]) => header + ": " + value).join("\n");
  }
  get [Symbol.toStringTag]() {
    return "AxiosHeaders";
  }
  static from(thing) {
    return thing instanceof this ? thing : new this(thing);
  }
  static concat(first, ...targets) {
    const computed = new this(first);
    targets.forEach((target) => computed.set(target));
    return computed;
  }
  static accessor(header) {
    const internals = this[$internals] = this[$internals] = {
      accessors: {}
    };
    const accessors = internals.accessors;
    const prototype2 = this.prototype;
    function defineAccessor(_header) {
      const lHeader = normalizeHeader(_header);
      if (!accessors[lHeader]) {
        buildAccessors(prototype2, _header);
        accessors[lHeader] = true;
      }
    }
    utils$4.isArray(header) ? header.forEach(defineAccessor) : defineAccessor(header);
    return this;
  }
}
AxiosHeaders.accessor(["Content-Type", "Content-Length", "Accept", "Accept-Encoding", "User-Agent", "Authorization"]);
utils$4.reduceDescriptors(AxiosHeaders.prototype, ({ value }, key) => {
  let mapped = key[0].toUpperCase() + key.slice(1);
  return {
    get: () => value,
    set(headerValue) {
      this[mapped] = headerValue;
    }
  };
});
utils$4.freezeMethods(AxiosHeaders);
const AxiosHeaders$1 = AxiosHeaders;
function transformData(fns, response) {
  const config2 = this || defaults$5;
  const context = response || config2;
  const headers = AxiosHeaders$1.from(context.headers);
  let data2 = context.data;
  utils$4.forEach(fns, function transform3(fn) {
    data2 = fn.call(config2, data2, headers.normalize(), response ? response.status : void 0);
  });
  headers.normalize();
  return data2;
}
function isCancel(value) {
  return !!(value && value.__CANCEL__);
}
function CanceledError(message, config2, request) {
  AxiosError.call(this, message == null ? "canceled" : message, AxiosError.ERR_CANCELED, config2, request);
  this.name = "CanceledError";
}
utils$4.inherits(CanceledError, AxiosError, {
  __CANCEL__: true
});
function settle(resolve2, reject2, response) {
  const validateStatus2 = response.config.validateStatus;
  if (!response.status || !validateStatus2 || validateStatus2(response.status)) {
    resolve2(response);
  } else {
    reject2(new AxiosError(
      "Request failed with status code " + response.status,
      [AxiosError.ERR_BAD_REQUEST, AxiosError.ERR_BAD_RESPONSE][Math.floor(response.status / 100) - 4],
      response.config,
      response.request,
      response
    ));
  }
}
const cookies = platform$1.hasStandardBrowserEnv ? (
  // Standard browser envs support document.cookie
  {
    write(name, value, expires, path, domain, secure) {
      const cookie = [name + "=" + encodeURIComponent(value)];
      utils$4.isNumber(expires) && cookie.push("expires=" + new Date(expires).toGMTString());
      utils$4.isString(path) && cookie.push("path=" + path);
      utils$4.isString(domain) && cookie.push("domain=" + domain);
      secure === true && cookie.push("secure");
      document.cookie = cookie.join("; ");
    },
    read(name) {
      const match5 = document.cookie.match(new RegExp("(^|;\\s*)(" + name + ")=([^;]*)"));
      return match5 ? decodeURIComponent(match5[3]) : null;
    },
    remove(name) {
      this.write(name, "", Date.now() - 864e5);
    }
  }
) : (
  // Non-standard browser env (web workers, react-native) lack needed support.
  {
    write() {
    },
    read() {
      return null;
    },
    remove() {
    }
  }
);
function isAbsoluteURL(url) {
  return /^([a-z][a-z\d+\-.]*:)?\/\//i.test(url);
}
function combineURLs(baseURL, relativeURL) {
  return relativeURL ? baseURL.replace(/\/?\/$/, "") + "/" + relativeURL.replace(/^\/+/, "") : baseURL;
}
function buildFullPath(baseURL, requestedURL) {
  if (baseURL && !isAbsoluteURL(requestedURL)) {
    return combineURLs(baseURL, requestedURL);
  }
  return requestedURL;
}
const isURLSameOrigin = platform$1.hasStandardBrowserEnv ? (
  // Standard browser envs have full support of the APIs needed to test
  // whether the request URL is of the same origin as current location.
  function standardBrowserEnv() {
    const msie = /(msie|trident)/i.test(navigator.userAgent);
    const urlParsingNode = document.createElement("a");
    let originURL;
    function resolveURL(url) {
      let href = url;
      if (msie) {
        urlParsingNode.setAttribute("href", href);
        href = urlParsingNode.href;
      }
      urlParsingNode.setAttribute("href", href);
      return {
        href: urlParsingNode.href,
        protocol: urlParsingNode.protocol ? urlParsingNode.protocol.replace(/:$/, "") : "",
        host: urlParsingNode.host,
        search: urlParsingNode.search ? urlParsingNode.search.replace(/^\?/, "") : "",
        hash: urlParsingNode.hash ? urlParsingNode.hash.replace(/^#/, "") : "",
        hostname: urlParsingNode.hostname,
        port: urlParsingNode.port,
        pathname: urlParsingNode.pathname.charAt(0) === "/" ? urlParsingNode.pathname : "/" + urlParsingNode.pathname
      };
    }
    originURL = resolveURL(window.location.href);
    return function isURLSameOrigin2(requestURL) {
      const parsed = utils$4.isString(requestURL) ? resolveURL(requestURL) : requestURL;
      return parsed.protocol === originURL.protocol && parsed.host === originURL.host;
    };
  }()
) : (
  // Non standard browser envs (web workers, react-native) lack needed support.
  function nonStandardBrowserEnv() {
    return function isURLSameOrigin2() {
      return true;
    };
  }()
);
function parseProtocol(url) {
  const match5 = /^([-+\w]{1,25})(:?\/\/|:)/.exec(url);
  return match5 && match5[1] || "";
}
function speedometer(samplesCount, min3) {
  samplesCount = samplesCount || 10;
  const bytes = new Array(samplesCount);
  const timestamps = new Array(samplesCount);
  let head = 0;
  let tail = 0;
  let firstSampleTS;
  min3 = min3 !== void 0 ? min3 : 1e3;
  return function push2(chunkLength) {
    const now2 = Date.now();
    const startedAt = timestamps[tail];
    if (!firstSampleTS) {
      firstSampleTS = now2;
    }
    bytes[head] = chunkLength;
    timestamps[head] = now2;
    let i2 = tail;
    let bytesCount = 0;
    while (i2 !== head) {
      bytesCount += bytes[i2++];
      i2 = i2 % samplesCount;
    }
    head = (head + 1) % samplesCount;
    if (head === tail) {
      tail = (tail + 1) % samplesCount;
    }
    if (now2 - firstSampleTS < min3) {
      return;
    }
    const passed = startedAt && now2 - startedAt;
    return passed ? Math.round(bytesCount * 1e3 / passed) : void 0;
  };
}
function progressEventReducer(listener, isDownloadStream) {
  let bytesNotified = 0;
  const _speedometer = speedometer(50, 250);
  return (e2) => {
    const loaded = e2.loaded;
    const total = e2.lengthComputable ? e2.total : void 0;
    const progressBytes = loaded - bytesNotified;
    const rate = _speedometer(progressBytes);
    const inRange = loaded <= total;
    bytesNotified = loaded;
    const data2 = {
      loaded,
      total,
      progress: total ? loaded / total : void 0,
      bytes: progressBytes,
      rate: rate ? rate : void 0,
      estimated: rate && total && inRange ? (total - loaded) / rate : void 0,
      event: e2
    };
    data2[isDownloadStream ? "download" : "upload"] = true;
    listener(data2);
  };
}
const isXHRAdapterSupported = typeof XMLHttpRequest !== "undefined";
const xhrAdapter = isXHRAdapterSupported && function(config2) {
  return new Promise(function dispatchXhrRequest(resolve2, reject2) {
    let requestData = config2.data;
    const requestHeaders = AxiosHeaders$1.from(config2.headers).normalize();
    let { responseType, withXSRFToken } = config2;
    let onCanceled;
    function done() {
      if (config2.cancelToken) {
        config2.cancelToken.unsubscribe(onCanceled);
      }
      if (config2.signal) {
        config2.signal.removeEventListener("abort", onCanceled);
      }
    }
    let contentType;
    if (utils$4.isFormData(requestData)) {
      if (platform$1.hasStandardBrowserEnv || platform$1.hasStandardBrowserWebWorkerEnv) {
        requestHeaders.setContentType(false);
      } else if ((contentType = requestHeaders.getContentType()) !== false) {
        const [type2, ...tokens2] = contentType ? contentType.split(";").map((token2) => token2.trim()).filter(Boolean) : [];
        requestHeaders.setContentType([type2 || "multipart/form-data", ...tokens2].join("; "));
      }
    }
    let request = new XMLHttpRequest();
    if (config2.auth) {
      const username = config2.auth.username || "";
      const password = config2.auth.password ? unescape(encodeURIComponent(config2.auth.password)) : "";
      requestHeaders.set("Authorization", "Basic " + btoa(username + ":" + password));
    }
    const fullPath = buildFullPath(config2.baseURL, config2.url);
    request.open(config2.method.toUpperCase(), buildURL(fullPath, config2.params, config2.paramsSerializer), true);
    request.timeout = config2.timeout;
    function onloadend() {
      if (!request) {
        return;
      }
      const responseHeaders = AxiosHeaders$1.from(
        "getAllResponseHeaders" in request && request.getAllResponseHeaders()
      );
      const responseData = !responseType || responseType === "text" || responseType === "json" ? request.responseText : request.response;
      const response = {
        data: responseData,
        status: request.status,
        statusText: request.statusText,
        headers: responseHeaders,
        config: config2,
        request
      };
      settle(function _resolve(value) {
        resolve2(value);
        done();
      }, function _reject(err2) {
        reject2(err2);
        done();
      }, response);
      request = null;
    }
    if ("onloadend" in request) {
      request.onloadend = onloadend;
    } else {
      request.onreadystatechange = function handleLoad() {
        if (!request || request.readyState !== 4) {
          return;
        }
        if (request.status === 0 && !(request.responseURL && request.responseURL.indexOf("file:") === 0)) {
          return;
        }
        setTimeout(onloadend);
      };
    }
    request.onabort = function handleAbort() {
      if (!request) {
        return;
      }
      reject2(new AxiosError("Request aborted", AxiosError.ECONNABORTED, config2, request));
      request = null;
    };
    request.onerror = function handleError() {
      reject2(new AxiosError("Network Error", AxiosError.ERR_NETWORK, config2, request));
      request = null;
    };
    request.ontimeout = function handleTimeout() {
      let timeoutErrorMessage = config2.timeout ? "timeout of " + config2.timeout + "ms exceeded" : "timeout exceeded";
      const transitional2 = config2.transitional || transitionalDefaults;
      if (config2.timeoutErrorMessage) {
        timeoutErrorMessage = config2.timeoutErrorMessage;
      }
      reject2(new AxiosError(
        timeoutErrorMessage,
        transitional2.clarifyTimeoutError ? AxiosError.ETIMEDOUT : AxiosError.ECONNABORTED,
        config2,
        request
      ));
      request = null;
    };
    if (platform$1.hasStandardBrowserEnv) {
      withXSRFToken && utils$4.isFunction(withXSRFToken) && (withXSRFToken = withXSRFToken(config2));
      if (withXSRFToken || withXSRFToken !== false && isURLSameOrigin(fullPath)) {
        const xsrfValue = config2.xsrfHeaderName && config2.xsrfCookieName && cookies.read(config2.xsrfCookieName);
        if (xsrfValue) {
          requestHeaders.set(config2.xsrfHeaderName, xsrfValue);
        }
      }
    }
    requestData === void 0 && requestHeaders.setContentType(null);
    if ("setRequestHeader" in request) {
      utils$4.forEach(requestHeaders.toJSON(), function setRequestHeader(val, key) {
        request.setRequestHeader(key, val);
      });
    }
    if (!utils$4.isUndefined(config2.withCredentials)) {
      request.withCredentials = !!config2.withCredentials;
    }
    if (responseType && responseType !== "json") {
      request.responseType = config2.responseType;
    }
    if (typeof config2.onDownloadProgress === "function") {
      request.addEventListener("progress", progressEventReducer(config2.onDownloadProgress, true));
    }
    if (typeof config2.onUploadProgress === "function" && request.upload) {
      request.upload.addEventListener("progress", progressEventReducer(config2.onUploadProgress));
    }
    if (config2.cancelToken || config2.signal) {
      onCanceled = (cancel) => {
        if (!request) {
          return;
        }
        reject2(!cancel || cancel.type ? new CanceledError(null, config2, request) : cancel);
        request.abort();
        request = null;
      };
      config2.cancelToken && config2.cancelToken.subscribe(onCanceled);
      if (config2.signal) {
        config2.signal.aborted ? onCanceled() : config2.signal.addEventListener("abort", onCanceled);
      }
    }
    const protocol = parseProtocol(fullPath);
    if (protocol && platform$1.protocols.indexOf(protocol) === -1) {
      reject2(new AxiosError("Unsupported protocol " + protocol + ":", AxiosError.ERR_BAD_REQUEST, config2));
      return;
    }
    request.send(requestData || null);
  });
};
const knownAdapters = {
  http: httpAdapter,
  xhr: xhrAdapter
};
utils$4.forEach(knownAdapters, (fn, value) => {
  if (fn) {
    try {
      Object.defineProperty(fn, "name", { value });
    } catch (e2) {
    }
    Object.defineProperty(fn, "adapterName", { value });
  }
});
const renderReason = (reason) => `- ${reason}`;
const isResolvedHandle = (adapter) => utils$4.isFunction(adapter) || adapter === null || adapter === false;
const adapters = {
  getAdapter: (adapters2) => {
    adapters2 = utils$4.isArray(adapters2) ? adapters2 : [adapters2];
    const { length } = adapters2;
    let nameOrAdapter;
    let adapter;
    const rejectedReasons = {};
    for (let i2 = 0; i2 < length; i2++) {
      nameOrAdapter = adapters2[i2];
      let id;
      adapter = nameOrAdapter;
      if (!isResolvedHandle(nameOrAdapter)) {
        adapter = knownAdapters[(id = String(nameOrAdapter)).toLowerCase()];
        if (adapter === void 0) {
          throw new AxiosError(`Unknown adapter '${id}'`);
        }
      }
      if (adapter) {
        break;
      }
      rejectedReasons[id || "#" + i2] = adapter;
    }
    if (!adapter) {
      const reasons = Object.entries(rejectedReasons).map(
        ([id, state]) => `adapter ${id} ` + (state === false ? "is not supported by the environment" : "is not available in the build")
      );
      let s2 = length ? reasons.length > 1 ? "since :\n" + reasons.map(renderReason).join("\n") : " " + renderReason(reasons[0]) : "as no adapter specified";
      throw new AxiosError(
        `There is no suitable adapter to dispatch the request ` + s2,
        "ERR_NOT_SUPPORT"
      );
    }
    return adapter;
  },
  adapters: knownAdapters
};
function throwIfCancellationRequested(config2) {
  if (config2.cancelToken) {
    config2.cancelToken.throwIfRequested();
  }
  if (config2.signal && config2.signal.aborted) {
    throw new CanceledError(null, config2);
  }
}
function dispatchRequest(config2) {
  throwIfCancellationRequested(config2);
  config2.headers = AxiosHeaders$1.from(config2.headers);
  config2.data = transformData.call(
    config2,
    config2.transformRequest
  );
  if (["post", "put", "patch"].indexOf(config2.method) !== -1) {
    config2.headers.setContentType("application/x-www-form-urlencoded", false);
  }
  const adapter = adapters.getAdapter(config2.adapter || defaults$5.adapter);
  return adapter(config2).then(function onAdapterResolution(response) {
    throwIfCancellationRequested(config2);
    response.data = transformData.call(
      config2,
      config2.transformResponse,
      response
    );
    response.headers = AxiosHeaders$1.from(response.headers);
    return response;
  }, function onAdapterRejection(reason) {
    if (!isCancel(reason)) {
      throwIfCancellationRequested(config2);
      if (reason && reason.response) {
        reason.response.data = transformData.call(
          config2,
          config2.transformResponse,
          reason.response
        );
        reason.response.headers = AxiosHeaders$1.from(reason.response.headers);
      }
    }
    return Promise.reject(reason);
  });
}
const headersToObject = (thing) => thing instanceof AxiosHeaders$1 ? thing.toJSON() : thing;
function mergeConfig(config1, config2) {
  config2 = config2 || {};
  const config3 = {};
  function getMergedValue(target, source, caseless) {
    if (utils$4.isPlainObject(target) && utils$4.isPlainObject(source)) {
      return utils$4.merge.call({ caseless }, target, source);
    } else if (utils$4.isPlainObject(source)) {
      return utils$4.merge({}, source);
    } else if (utils$4.isArray(source)) {
      return source.slice();
    }
    return source;
  }
  function mergeDeepProperties(a2, b2, caseless) {
    if (!utils$4.isUndefined(b2)) {
      return getMergedValue(a2, b2, caseless);
    } else if (!utils$4.isUndefined(a2)) {
      return getMergedValue(void 0, a2, caseless);
    }
  }
  function valueFromConfig2(a2, b2) {
    if (!utils$4.isUndefined(b2)) {
      return getMergedValue(void 0, b2);
    }
  }
  function defaultToConfig2(a2, b2) {
    if (!utils$4.isUndefined(b2)) {
      return getMergedValue(void 0, b2);
    } else if (!utils$4.isUndefined(a2)) {
      return getMergedValue(void 0, a2);
    }
  }
  function mergeDirectKeys(a2, b2, prop) {
    if (prop in config2) {
      return getMergedValue(a2, b2);
    } else if (prop in config1) {
      return getMergedValue(void 0, a2);
    }
  }
  const mergeMap = {
    url: valueFromConfig2,
    method: valueFromConfig2,
    data: valueFromConfig2,
    baseURL: defaultToConfig2,
    transformRequest: defaultToConfig2,
    transformResponse: defaultToConfig2,
    paramsSerializer: defaultToConfig2,
    timeout: defaultToConfig2,
    timeoutMessage: defaultToConfig2,
    withCredentials: defaultToConfig2,
    withXSRFToken: defaultToConfig2,
    adapter: defaultToConfig2,
    responseType: defaultToConfig2,
    xsrfCookieName: defaultToConfig2,
    xsrfHeaderName: defaultToConfig2,
    onUploadProgress: defaultToConfig2,
    onDownloadProgress: defaultToConfig2,
    decompress: defaultToConfig2,
    maxContentLength: defaultToConfig2,
    maxBodyLength: defaultToConfig2,
    beforeRedirect: defaultToConfig2,
    transport: defaultToConfig2,
    httpAgent: defaultToConfig2,
    httpsAgent: defaultToConfig2,
    cancelToken: defaultToConfig2,
    socketPath: defaultToConfig2,
    responseEncoding: defaultToConfig2,
    validateStatus: mergeDirectKeys,
    headers: (a2, b2) => mergeDeepProperties(headersToObject(a2), headersToObject(b2), true)
  };
  utils$4.forEach(Object.keys(Object.assign({}, config1, config2)), function computeConfigValue(prop) {
    const merge2 = mergeMap[prop] || mergeDeepProperties;
    const configValue = merge2(config1[prop], config2[prop], prop);
    utils$4.isUndefined(configValue) && merge2 !== mergeDirectKeys || (config3[prop] = configValue);
  });
  return config3;
}
const VERSION = "1.6.7";
const validators$1 = {};
["object", "boolean", "number", "function", "string", "symbol"].forEach((type2, i2) => {
  validators$1[type2] = function validator2(thing) {
    return typeof thing === type2 || "a" + (i2 < 1 ? "n " : " ") + type2;
  };
});
const deprecatedWarnings = {};
validators$1.transitional = function transitional(validator2, version2, message) {
  function formatMessage(opt, desc) {
    return "[Axios v" + VERSION + "] Transitional option '" + opt + "'" + desc + (message ? ". " + message : "");
  }
  return (value, opt, opts) => {
    if (validator2 === false) {
      throw new AxiosError(
        formatMessage(opt, " has been removed" + (version2 ? " in " + version2 : "")),
        AxiosError.ERR_DEPRECATED
      );
    }
    if (version2 && !deprecatedWarnings[opt]) {
      deprecatedWarnings[opt] = true;
      console.warn(
        formatMessage(
          opt,
          " has been deprecated since v" + version2 + " and will be removed in the near future"
        )
      );
    }
    return validator2 ? validator2(value, opt, opts) : true;
  };
};
function assertOptions(options, schema, allowUnknown) {
  if (typeof options !== "object") {
    throw new AxiosError("options must be an object", AxiosError.ERR_BAD_OPTION_VALUE);
  }
  const keys4 = Object.keys(options);
  let i2 = keys4.length;
  while (i2-- > 0) {
    const opt = keys4[i2];
    const validator2 = schema[opt];
    if (validator2) {
      const value = options[opt];
      const result = value === void 0 || validator2(value, opt, options);
      if (result !== true) {
        throw new AxiosError("option " + opt + " must be " + result, AxiosError.ERR_BAD_OPTION_VALUE);
      }
      continue;
    }
    if (allowUnknown !== true) {
      throw new AxiosError("Unknown option " + opt, AxiosError.ERR_BAD_OPTION);
    }
  }
}
const validator = {
  assertOptions,
  validators: validators$1
};
const validators = validator.validators;
class Axios {
  constructor(instanceConfig) {
    this.defaults = instanceConfig;
    this.interceptors = {
      request: new InterceptorManager$1(),
      response: new InterceptorManager$1()
    };
  }
  /**
   * Dispatch a request
   *
   * @param {String|Object} configOrUrl The config specific for this request (merged with this.defaults)
   * @param {?Object} config
   *
   * @returns {Promise} The Promise to be fulfilled
   */
  async request(configOrUrl, config2) {
    try {
      return await this._request(configOrUrl, config2);
    } catch (err2) {
      if (err2 instanceof Error) {
        let dummy;
        Error.captureStackTrace ? Error.captureStackTrace(dummy = {}) : dummy = new Error();
        const stack = dummy.stack ? dummy.stack.replace(/^.+\n/, "") : "";
        if (!err2.stack) {
          err2.stack = stack;
        } else if (stack && !String(err2.stack).endsWith(stack.replace(/^.+\n.+\n/, ""))) {
          err2.stack += "\n" + stack;
        }
      }
      throw err2;
    }
  }
  _request(configOrUrl, config2) {
    if (typeof configOrUrl === "string") {
      config2 = config2 || {};
      config2.url = configOrUrl;
    } else {
      config2 = configOrUrl || {};
    }
    config2 = mergeConfig(this.defaults, config2);
    const { transitional: transitional2, paramsSerializer, headers } = config2;
    if (transitional2 !== void 0) {
      validator.assertOptions(transitional2, {
        silentJSONParsing: validators.transitional(validators.boolean),
        forcedJSONParsing: validators.transitional(validators.boolean),
        clarifyTimeoutError: validators.transitional(validators.boolean)
      }, false);
    }
    if (paramsSerializer != null) {
      if (utils$4.isFunction(paramsSerializer)) {
        config2.paramsSerializer = {
          serialize: paramsSerializer
        };
      } else {
        validator.assertOptions(paramsSerializer, {
          encode: validators.function,
          serialize: validators.function
        }, true);
      }
    }
    config2.method = (config2.method || this.defaults.method || "get").toLowerCase();
    let contextHeaders = headers && utils$4.merge(
      headers.common,
      headers[config2.method]
    );
    headers && utils$4.forEach(
      ["delete", "get", "head", "post", "put", "patch", "common"],
      (method) => {
        delete headers[method];
      }
    );
    config2.headers = AxiosHeaders$1.concat(contextHeaders, headers);
    const requestInterceptorChain = [];
    let synchronousRequestInterceptors = true;
    this.interceptors.request.forEach(function unshiftRequestInterceptors(interceptor) {
      if (typeof interceptor.runWhen === "function" && interceptor.runWhen(config2) === false) {
        return;
      }
      synchronousRequestInterceptors = synchronousRequestInterceptors && interceptor.synchronous;
      requestInterceptorChain.unshift(interceptor.fulfilled, interceptor.rejected);
    });
    const responseInterceptorChain = [];
    this.interceptors.response.forEach(function pushResponseInterceptors(interceptor) {
      responseInterceptorChain.push(interceptor.fulfilled, interceptor.rejected);
    });
    let promise2;
    let i2 = 0;
    let len2;
    if (!synchronousRequestInterceptors) {
      const chain = [dispatchRequest.bind(this), void 0];
      chain.unshift.apply(chain, requestInterceptorChain);
      chain.push.apply(chain, responseInterceptorChain);
      len2 = chain.length;
      promise2 = Promise.resolve(config2);
      while (i2 < len2) {
        promise2 = promise2.then(chain[i2++], chain[i2++]);
      }
      return promise2;
    }
    len2 = requestInterceptorChain.length;
    let newConfig = config2;
    i2 = 0;
    while (i2 < len2) {
      const onFulfilled = requestInterceptorChain[i2++];
      const onRejected = requestInterceptorChain[i2++];
      try {
        newConfig = onFulfilled(newConfig);
      } catch (error) {
        onRejected.call(this, error);
        break;
      }
    }
    try {
      promise2 = dispatchRequest.call(this, newConfig);
    } catch (error) {
      return Promise.reject(error);
    }
    i2 = 0;
    len2 = responseInterceptorChain.length;
    while (i2 < len2) {
      promise2 = promise2.then(responseInterceptorChain[i2++], responseInterceptorChain[i2++]);
    }
    return promise2;
  }
  getUri(config2) {
    config2 = mergeConfig(this.defaults, config2);
    const fullPath = buildFullPath(config2.baseURL, config2.url);
    return buildURL(fullPath, config2.params, config2.paramsSerializer);
  }
}
utils$4.forEach(["delete", "get", "head", "options"], function forEachMethodNoData(method) {
  Axios.prototype[method] = function(url, config2) {
    return this.request(mergeConfig(config2 || {}, {
      method,
      url,
      data: (config2 || {}).data
    }));
  };
});
utils$4.forEach(["post", "put", "patch"], function forEachMethodWithData(method) {
  function generateHTTPMethod(isForm) {
    return function httpMethod(url, data2, config2) {
      return this.request(mergeConfig(config2 || {}, {
        method,
        headers: isForm ? {
          "Content-Type": "multipart/form-data"
        } : {},
        url,
        data: data2
      }));
    };
  }
  Axios.prototype[method] = generateHTTPMethod();
  Axios.prototype[method + "Form"] = generateHTTPMethod(true);
});
const Axios$1 = Axios;
class CancelToken {
  constructor(executor) {
    if (typeof executor !== "function") {
      throw new TypeError("executor must be a function.");
    }
    let resolvePromise;
    this.promise = new Promise(function promiseExecutor(resolve2) {
      resolvePromise = resolve2;
    });
    const token2 = this;
    this.promise.then((cancel) => {
      if (!token2._listeners)
        return;
      let i2 = token2._listeners.length;
      while (i2-- > 0) {
        token2._listeners[i2](cancel);
      }
      token2._listeners = null;
    });
    this.promise.then = (onfulfilled) => {
      let _resolve;
      const promise2 = new Promise((resolve2) => {
        token2.subscribe(resolve2);
        _resolve = resolve2;
      }).then(onfulfilled);
      promise2.cancel = function reject2() {
        token2.unsubscribe(_resolve);
      };
      return promise2;
    };
    executor(function cancel(message, config2, request) {
      if (token2.reason) {
        return;
      }
      token2.reason = new CanceledError(message, config2, request);
      resolvePromise(token2.reason);
    });
  }
  /**
   * Throws a `CanceledError` if cancellation has been requested.
   */
  throwIfRequested() {
    if (this.reason) {
      throw this.reason;
    }
  }
  /**
   * Subscribe to the cancel signal
   */
  subscribe(listener) {
    if (this.reason) {
      listener(this.reason);
      return;
    }
    if (this._listeners) {
      this._listeners.push(listener);
    } else {
      this._listeners = [listener];
    }
  }
  /**
   * Unsubscribe from the cancel signal
   */
  unsubscribe(listener) {
    if (!this._listeners) {
      return;
    }
    const index2 = this._listeners.indexOf(listener);
    if (index2 !== -1) {
      this._listeners.splice(index2, 1);
    }
  }
  /**
   * Returns an object that contains a new `CancelToken` and a function that, when called,
   * cancels the `CancelToken`.
   */
  static source() {
    let cancel;
    const token2 = new CancelToken(function executor(c2) {
      cancel = c2;
    });
    return {
      token: token2,
      cancel
    };
  }
}
const CancelToken$1 = CancelToken;
function spread(callback) {
  return function wrap(arr) {
    return callback.apply(null, arr);
  };
}
function isAxiosError(payload) {
  return utils$4.isObject(payload) && payload.isAxiosError === true;
}
const HttpStatusCode = {
  Continue: 100,
  SwitchingProtocols: 101,
  Processing: 102,
  EarlyHints: 103,
  Ok: 200,
  Created: 201,
  Accepted: 202,
  NonAuthoritativeInformation: 203,
  NoContent: 204,
  ResetContent: 205,
  PartialContent: 206,
  MultiStatus: 207,
  AlreadyReported: 208,
  ImUsed: 226,
  MultipleChoices: 300,
  MovedPermanently: 301,
  Found: 302,
  SeeOther: 303,
  NotModified: 304,
  UseProxy: 305,
  Unused: 306,
  TemporaryRedirect: 307,
  PermanentRedirect: 308,
  BadRequest: 400,
  Unauthorized: 401,
  PaymentRequired: 402,
  Forbidden: 403,
  NotFound: 404,
  MethodNotAllowed: 405,
  NotAcceptable: 406,
  ProxyAuthenticationRequired: 407,
  RequestTimeout: 408,
  Conflict: 409,
  Gone: 410,
  LengthRequired: 411,
  PreconditionFailed: 412,
  PayloadTooLarge: 413,
  UriTooLong: 414,
  UnsupportedMediaType: 415,
  RangeNotSatisfiable: 416,
  ExpectationFailed: 417,
  ImATeapot: 418,
  MisdirectedRequest: 421,
  UnprocessableEntity: 422,
  Locked: 423,
  FailedDependency: 424,
  TooEarly: 425,
  UpgradeRequired: 426,
  PreconditionRequired: 428,
  TooManyRequests: 429,
  RequestHeaderFieldsTooLarge: 431,
  UnavailableForLegalReasons: 451,
  InternalServerError: 500,
  NotImplemented: 501,
  BadGateway: 502,
  ServiceUnavailable: 503,
  GatewayTimeout: 504,
  HttpVersionNotSupported: 505,
  VariantAlsoNegotiates: 506,
  InsufficientStorage: 507,
  LoopDetected: 508,
  NotExtended: 510,
  NetworkAuthenticationRequired: 511
};
Object.entries(HttpStatusCode).forEach(([key, value]) => {
  HttpStatusCode[value] = key;
});
const HttpStatusCode$1 = HttpStatusCode;
function createInstance(defaultConfig) {
  const context = new Axios$1(defaultConfig);
  const instance = bind$9(Axios$1.prototype.request, context);
  utils$4.extend(instance, Axios$1.prototype, context, { allOwnKeys: true });
  utils$4.extend(instance, context, null, { allOwnKeys: true });
  instance.create = function create4(instanceConfig) {
    return createInstance(mergeConfig(defaultConfig, instanceConfig));
  };
  return instance;
}
const axios = createInstance(defaults$5);
axios.Axios = Axios$1;
axios.CanceledError = CanceledError;
axios.CancelToken = CancelToken$1;
axios.isCancel = isCancel;
axios.VERSION = VERSION;
axios.toFormData = toFormData;
axios.AxiosError = AxiosError;
axios.Cancel = axios.CanceledError;
axios.all = function all(promises) {
  return Promise.all(promises);
};
axios.spread = spread;
axios.isAxiosError = isAxiosError;
axios.mergeConfig = mergeConfig;
axios.AxiosHeaders = AxiosHeaders$1;
axios.formToJSON = (thing) => formDataToJSON(utils$4.isHTMLForm(thing) ? new FormData(thing) : thing);
axios.getAdapter = adapters.getAdapter;
axios.HttpStatusCode = HttpStatusCode$1;
axios.default = axios;
const axios$1 = axios;
/*! pako 2.1.0 https://github.com/nodeca/pako @license (MIT AND Zlib) */
const Z_FIXED$1 = 4;
const Z_BINARY = 0;
const Z_TEXT = 1;
const Z_UNKNOWN$1 = 2;
function zero$1(buf) {
  let len2 = buf.length;
  while (--len2 >= 0) {
    buf[len2] = 0;
  }
}
const STORED_BLOCK = 0;
const STATIC_TREES = 1;
const DYN_TREES = 2;
const MIN_MATCH$1 = 3;
const MAX_MATCH$1 = 258;
const LENGTH_CODES$1 = 29;
const LITERALS$1 = 256;
const L_CODES$1 = LITERALS$1 + 1 + LENGTH_CODES$1;
const D_CODES$1 = 30;
const BL_CODES$1 = 19;
const HEAP_SIZE$1 = 2 * L_CODES$1 + 1;
const MAX_BITS$1 = 15;
const Buf_size = 16;
const MAX_BL_BITS = 7;
const END_BLOCK = 256;
const REP_3_6 = 16;
const REPZ_3_10 = 17;
const REPZ_11_138 = 18;
const extra_lbits = (
  /* extra bits for each length code */
  new Uint8Array([0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 5, 5, 0])
);
const extra_dbits = (
  /* extra bits for each distance code */
  new Uint8Array([0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13])
);
const extra_blbits = (
  /* extra bits for each bit length code */
  new Uint8Array([0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 7])
);
const bl_order = new Uint8Array([16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15]);
const DIST_CODE_LEN = 512;
const static_ltree = new Array((L_CODES$1 + 2) * 2);
zero$1(static_ltree);
const static_dtree = new Array(D_CODES$1 * 2);
zero$1(static_dtree);
const _dist_code = new Array(DIST_CODE_LEN);
zero$1(_dist_code);
const _length_code = new Array(MAX_MATCH$1 - MIN_MATCH$1 + 1);
zero$1(_length_code);
const base_length = new Array(LENGTH_CODES$1);
zero$1(base_length);
const base_dist = new Array(D_CODES$1);
zero$1(base_dist);
function StaticTreeDesc(static_tree, extra_bits, extra_base, elems, max_length) {
  this.static_tree = static_tree;
  this.extra_bits = extra_bits;
  this.extra_base = extra_base;
  this.elems = elems;
  this.max_length = max_length;
  this.has_stree = static_tree && static_tree.length;
}
let static_l_desc;
let static_d_desc;
let static_bl_desc;
function TreeDesc(dyn_tree, stat_desc) {
  this.dyn_tree = dyn_tree;
  this.max_code = 0;
  this.stat_desc = stat_desc;
}
const d_code = (dist2) => {
  return dist2 < 256 ? _dist_code[dist2] : _dist_code[256 + (dist2 >>> 7)];
};
const put_short = (s2, w2) => {
  s2.pending_buf[s2.pending++] = w2 & 255;
  s2.pending_buf[s2.pending++] = w2 >>> 8 & 255;
};
const send_bits = (s2, value, length) => {
  if (s2.bi_valid > Buf_size - length) {
    s2.bi_buf |= value << s2.bi_valid & 65535;
    put_short(s2, s2.bi_buf);
    s2.bi_buf = value >> Buf_size - s2.bi_valid;
    s2.bi_valid += length - Buf_size;
  } else {
    s2.bi_buf |= value << s2.bi_valid & 65535;
    s2.bi_valid += length;
  }
};
const send_code = (s2, c2, tree) => {
  send_bits(
    s2,
    tree[c2 * 2],
    tree[c2 * 2 + 1]
    /*.Len*/
  );
};
const bi_reverse = (code, len2) => {
  let res = 0;
  do {
    res |= code & 1;
    code >>>= 1;
    res <<= 1;
  } while (--len2 > 0);
  return res >>> 1;
};
const bi_flush = (s2) => {
  if (s2.bi_valid === 16) {
    put_short(s2, s2.bi_buf);
    s2.bi_buf = 0;
    s2.bi_valid = 0;
  } else if (s2.bi_valid >= 8) {
    s2.pending_buf[s2.pending++] = s2.bi_buf & 255;
    s2.bi_buf >>= 8;
    s2.bi_valid -= 8;
  }
};
const gen_bitlen = (s2, desc) => {
  const tree = desc.dyn_tree;
  const max_code = desc.max_code;
  const stree = desc.stat_desc.static_tree;
  const has_stree = desc.stat_desc.has_stree;
  const extra = desc.stat_desc.extra_bits;
  const base2 = desc.stat_desc.extra_base;
  const max_length = desc.stat_desc.max_length;
  let h2;
  let n2, m2;
  let bits2;
  let xbits;
  let f2;
  let overflow2 = 0;
  for (bits2 = 0; bits2 <= MAX_BITS$1; bits2++) {
    s2.bl_count[bits2] = 0;
  }
  tree[s2.heap[s2.heap_max] * 2 + 1] = 0;
  for (h2 = s2.heap_max + 1; h2 < HEAP_SIZE$1; h2++) {
    n2 = s2.heap[h2];
    bits2 = tree[tree[n2 * 2 + 1] * 2 + 1] + 1;
    if (bits2 > max_length) {
      bits2 = max_length;
      overflow2++;
    }
    tree[n2 * 2 + 1] = bits2;
    if (n2 > max_code) {
      continue;
    }
    s2.bl_count[bits2]++;
    xbits = 0;
    if (n2 >= base2) {
      xbits = extra[n2 - base2];
    }
    f2 = tree[n2 * 2];
    s2.opt_len += f2 * (bits2 + xbits);
    if (has_stree) {
      s2.static_len += f2 * (stree[n2 * 2 + 1] + xbits);
    }
  }
  if (overflow2 === 0) {
    return;
  }
  do {
    bits2 = max_length - 1;
    while (s2.bl_count[bits2] === 0) {
      bits2--;
    }
    s2.bl_count[bits2]--;
    s2.bl_count[bits2 + 1] += 2;
    s2.bl_count[max_length]--;
    overflow2 -= 2;
  } while (overflow2 > 0);
  for (bits2 = max_length; bits2 !== 0; bits2--) {
    n2 = s2.bl_count[bits2];
    while (n2 !== 0) {
      m2 = s2.heap[--h2];
      if (m2 > max_code) {
        continue;
      }
      if (tree[m2 * 2 + 1] !== bits2) {
        s2.opt_len += (bits2 - tree[m2 * 2 + 1]) * tree[m2 * 2];
        tree[m2 * 2 + 1] = bits2;
      }
      n2--;
    }
  }
};
const gen_codes = (tree, max_code, bl_count) => {
  const next_code = new Array(MAX_BITS$1 + 1);
  let code = 0;
  let bits2;
  let n2;
  for (bits2 = 1; bits2 <= MAX_BITS$1; bits2++) {
    code = code + bl_count[bits2 - 1] << 1;
    next_code[bits2] = code;
  }
  for (n2 = 0; n2 <= max_code; n2++) {
    let len2 = tree[n2 * 2 + 1];
    if (len2 === 0) {
      continue;
    }
    tree[n2 * 2] = bi_reverse(next_code[len2]++, len2);
  }
};
const tr_static_init = () => {
  let n2;
  let bits2;
  let length;
  let code;
  let dist2;
  const bl_count = new Array(MAX_BITS$1 + 1);
  length = 0;
  for (code = 0; code < LENGTH_CODES$1 - 1; code++) {
    base_length[code] = length;
    for (n2 = 0; n2 < 1 << extra_lbits[code]; n2++) {
      _length_code[length++] = code;
    }
  }
  _length_code[length - 1] = code;
  dist2 = 0;
  for (code = 0; code < 16; code++) {
    base_dist[code] = dist2;
    for (n2 = 0; n2 < 1 << extra_dbits[code]; n2++) {
      _dist_code[dist2++] = code;
    }
  }
  dist2 >>= 7;
  for (; code < D_CODES$1; code++) {
    base_dist[code] = dist2 << 7;
    for (n2 = 0; n2 < 1 << extra_dbits[code] - 7; n2++) {
      _dist_code[256 + dist2++] = code;
    }
  }
  for (bits2 = 0; bits2 <= MAX_BITS$1; bits2++) {
    bl_count[bits2] = 0;
  }
  n2 = 0;
  while (n2 <= 143) {
    static_ltree[n2 * 2 + 1] = 8;
    n2++;
    bl_count[8]++;
  }
  while (n2 <= 255) {
    static_ltree[n2 * 2 + 1] = 9;
    n2++;
    bl_count[9]++;
  }
  while (n2 <= 279) {
    static_ltree[n2 * 2 + 1] = 7;
    n2++;
    bl_count[7]++;
  }
  while (n2 <= 287) {
    static_ltree[n2 * 2 + 1] = 8;
    n2++;
    bl_count[8]++;
  }
  gen_codes(static_ltree, L_CODES$1 + 1, bl_count);
  for (n2 = 0; n2 < D_CODES$1; n2++) {
    static_dtree[n2 * 2 + 1] = 5;
    static_dtree[n2 * 2] = bi_reverse(n2, 5);
  }
  static_l_desc = new StaticTreeDesc(static_ltree, extra_lbits, LITERALS$1 + 1, L_CODES$1, MAX_BITS$1);
  static_d_desc = new StaticTreeDesc(static_dtree, extra_dbits, 0, D_CODES$1, MAX_BITS$1);
  static_bl_desc = new StaticTreeDesc(new Array(0), extra_blbits, 0, BL_CODES$1, MAX_BL_BITS);
};
const init_block = (s2) => {
  let n2;
  for (n2 = 0; n2 < L_CODES$1; n2++) {
    s2.dyn_ltree[n2 * 2] = 0;
  }
  for (n2 = 0; n2 < D_CODES$1; n2++) {
    s2.dyn_dtree[n2 * 2] = 0;
  }
  for (n2 = 0; n2 < BL_CODES$1; n2++) {
    s2.bl_tree[n2 * 2] = 0;
  }
  s2.dyn_ltree[END_BLOCK * 2] = 1;
  s2.opt_len = s2.static_len = 0;
  s2.sym_next = s2.matches = 0;
};
const bi_windup = (s2) => {
  if (s2.bi_valid > 8) {
    put_short(s2, s2.bi_buf);
  } else if (s2.bi_valid > 0) {
    s2.pending_buf[s2.pending++] = s2.bi_buf;
  }
  s2.bi_buf = 0;
  s2.bi_valid = 0;
};
const smaller = (tree, n2, m2, depth) => {
  const _n2 = n2 * 2;
  const _m2 = m2 * 2;
  return tree[_n2] < tree[_m2] || tree[_n2] === tree[_m2] && depth[n2] <= depth[m2];
};
const pqdownheap = (s2, tree, k2) => {
  const v4 = s2.heap[k2];
  let j2 = k2 << 1;
  while (j2 <= s2.heap_len) {
    if (j2 < s2.heap_len && smaller(tree, s2.heap[j2 + 1], s2.heap[j2], s2.depth)) {
      j2++;
    }
    if (smaller(tree, v4, s2.heap[j2], s2.depth)) {
      break;
    }
    s2.heap[k2] = s2.heap[j2];
    k2 = j2;
    j2 <<= 1;
  }
  s2.heap[k2] = v4;
};
const compress_block = (s2, ltree, dtree) => {
  let dist2;
  let lc2;
  let sx = 0;
  let code;
  let extra;
  if (s2.sym_next !== 0) {
    do {
      dist2 = s2.pending_buf[s2.sym_buf + sx++] & 255;
      dist2 += (s2.pending_buf[s2.sym_buf + sx++] & 255) << 8;
      lc2 = s2.pending_buf[s2.sym_buf + sx++];
      if (dist2 === 0) {
        send_code(s2, lc2, ltree);
      } else {
        code = _length_code[lc2];
        send_code(s2, code + LITERALS$1 + 1, ltree);
        extra = extra_lbits[code];
        if (extra !== 0) {
          lc2 -= base_length[code];
          send_bits(s2, lc2, extra);
        }
        dist2--;
        code = d_code(dist2);
        send_code(s2, code, dtree);
        extra = extra_dbits[code];
        if (extra !== 0) {
          dist2 -= base_dist[code];
          send_bits(s2, dist2, extra);
        }
      }
    } while (sx < s2.sym_next);
  }
  send_code(s2, END_BLOCK, ltree);
};
const build_tree = (s2, desc) => {
  const tree = desc.dyn_tree;
  const stree = desc.stat_desc.static_tree;
  const has_stree = desc.stat_desc.has_stree;
  const elems = desc.stat_desc.elems;
  let n2, m2;
  let max_code = -1;
  let node2;
  s2.heap_len = 0;
  s2.heap_max = HEAP_SIZE$1;
  for (n2 = 0; n2 < elems; n2++) {
    if (tree[n2 * 2] !== 0) {
      s2.heap[++s2.heap_len] = max_code = n2;
      s2.depth[n2] = 0;
    } else {
      tree[n2 * 2 + 1] = 0;
    }
  }
  while (s2.heap_len < 2) {
    node2 = s2.heap[++s2.heap_len] = max_code < 2 ? ++max_code : 0;
    tree[node2 * 2] = 1;
    s2.depth[node2] = 0;
    s2.opt_len--;
    if (has_stree) {
      s2.static_len -= stree[node2 * 2 + 1];
    }
  }
  desc.max_code = max_code;
  for (n2 = s2.heap_len >> 1; n2 >= 1; n2--) {
    pqdownheap(s2, tree, n2);
  }
  node2 = elems;
  do {
    n2 = s2.heap[
      1
      /*SMALLEST*/
    ];
    s2.heap[
      1
      /*SMALLEST*/
    ] = s2.heap[s2.heap_len--];
    pqdownheap(
      s2,
      tree,
      1
      /*SMALLEST*/
    );
    m2 = s2.heap[
      1
      /*SMALLEST*/
    ];
    s2.heap[--s2.heap_max] = n2;
    s2.heap[--s2.heap_max] = m2;
    tree[node2 * 2] = tree[n2 * 2] + tree[m2 * 2];
    s2.depth[node2] = (s2.depth[n2] >= s2.depth[m2] ? s2.depth[n2] : s2.depth[m2]) + 1;
    tree[n2 * 2 + 1] = tree[m2 * 2 + 1] = node2;
    s2.heap[
      1
      /*SMALLEST*/
    ] = node2++;
    pqdownheap(
      s2,
      tree,
      1
      /*SMALLEST*/
    );
  } while (s2.heap_len >= 2);
  s2.heap[--s2.heap_max] = s2.heap[
    1
    /*SMALLEST*/
  ];
  gen_bitlen(s2, desc);
  gen_codes(tree, max_code, s2.bl_count);
};
const scan_tree = (s2, tree, max_code) => {
  let n2;
  let prevlen = -1;
  let curlen;
  let nextlen = tree[0 * 2 + 1];
  let count2 = 0;
  let max_count = 7;
  let min_count = 4;
  if (nextlen === 0) {
    max_count = 138;
    min_count = 3;
  }
  tree[(max_code + 1) * 2 + 1] = 65535;
  for (n2 = 0; n2 <= max_code; n2++) {
    curlen = nextlen;
    nextlen = tree[(n2 + 1) * 2 + 1];
    if (++count2 < max_count && curlen === nextlen) {
      continue;
    } else if (count2 < min_count) {
      s2.bl_tree[curlen * 2] += count2;
    } else if (curlen !== 0) {
      if (curlen !== prevlen) {
        s2.bl_tree[curlen * 2]++;
      }
      s2.bl_tree[REP_3_6 * 2]++;
    } else if (count2 <= 10) {
      s2.bl_tree[REPZ_3_10 * 2]++;
    } else {
      s2.bl_tree[REPZ_11_138 * 2]++;
    }
    count2 = 0;
    prevlen = curlen;
    if (nextlen === 0) {
      max_count = 138;
      min_count = 3;
    } else if (curlen === nextlen) {
      max_count = 6;
      min_count = 3;
    } else {
      max_count = 7;
      min_count = 4;
    }
  }
};
const send_tree = (s2, tree, max_code) => {
  let n2;
  let prevlen = -1;
  let curlen;
  let nextlen = tree[0 * 2 + 1];
  let count2 = 0;
  let max_count = 7;
  let min_count = 4;
  if (nextlen === 0) {
    max_count = 138;
    min_count = 3;
  }
  for (n2 = 0; n2 <= max_code; n2++) {
    curlen = nextlen;
    nextlen = tree[(n2 + 1) * 2 + 1];
    if (++count2 < max_count && curlen === nextlen) {
      continue;
    } else if (count2 < min_count) {
      do {
        send_code(s2, curlen, s2.bl_tree);
      } while (--count2 !== 0);
    } else if (curlen !== 0) {
      if (curlen !== prevlen) {
        send_code(s2, curlen, s2.bl_tree);
        count2--;
      }
      send_code(s2, REP_3_6, s2.bl_tree);
      send_bits(s2, count2 - 3, 2);
    } else if (count2 <= 10) {
      send_code(s2, REPZ_3_10, s2.bl_tree);
      send_bits(s2, count2 - 3, 3);
    } else {
      send_code(s2, REPZ_11_138, s2.bl_tree);
      send_bits(s2, count2 - 11, 7);
    }
    count2 = 0;
    prevlen = curlen;
    if (nextlen === 0) {
      max_count = 138;
      min_count = 3;
    } else if (curlen === nextlen) {
      max_count = 6;
      min_count = 3;
    } else {
      max_count = 7;
      min_count = 4;
    }
  }
};
const build_bl_tree = (s2) => {
  let max_blindex;
  scan_tree(s2, s2.dyn_ltree, s2.l_desc.max_code);
  scan_tree(s2, s2.dyn_dtree, s2.d_desc.max_code);
  build_tree(s2, s2.bl_desc);
  for (max_blindex = BL_CODES$1 - 1; max_blindex >= 3; max_blindex--) {
    if (s2.bl_tree[bl_order[max_blindex] * 2 + 1] !== 0) {
      break;
    }
  }
  s2.opt_len += 3 * (max_blindex + 1) + 5 + 5 + 4;
  return max_blindex;
};
const send_all_trees = (s2, lcodes, dcodes, blcodes) => {
  let rank2;
  send_bits(s2, lcodes - 257, 5);
  send_bits(s2, dcodes - 1, 5);
  send_bits(s2, blcodes - 4, 4);
  for (rank2 = 0; rank2 < blcodes; rank2++) {
    send_bits(s2, s2.bl_tree[bl_order[rank2] * 2 + 1], 3);
  }
  send_tree(s2, s2.dyn_ltree, lcodes - 1);
  send_tree(s2, s2.dyn_dtree, dcodes - 1);
};
const detect_data_type = (s2) => {
  let block_mask = 4093624447;
  let n2;
  for (n2 = 0; n2 <= 31; n2++, block_mask >>>= 1) {
    if (block_mask & 1 && s2.dyn_ltree[n2 * 2] !== 0) {
      return Z_BINARY;
    }
  }
  if (s2.dyn_ltree[9 * 2] !== 0 || s2.dyn_ltree[10 * 2] !== 0 || s2.dyn_ltree[13 * 2] !== 0) {
    return Z_TEXT;
  }
  for (n2 = 32; n2 < LITERALS$1; n2++) {
    if (s2.dyn_ltree[n2 * 2] !== 0) {
      return Z_TEXT;
    }
  }
  return Z_BINARY;
};
let static_init_done = false;
const _tr_init$1 = (s2) => {
  if (!static_init_done) {
    tr_static_init();
    static_init_done = true;
  }
  s2.l_desc = new TreeDesc(s2.dyn_ltree, static_l_desc);
  s2.d_desc = new TreeDesc(s2.dyn_dtree, static_d_desc);
  s2.bl_desc = new TreeDesc(s2.bl_tree, static_bl_desc);
  s2.bi_buf = 0;
  s2.bi_valid = 0;
  init_block(s2);
};
const _tr_stored_block$1 = (s2, buf, stored_len, last) => {
  send_bits(s2, (STORED_BLOCK << 1) + (last ? 1 : 0), 3);
  bi_windup(s2);
  put_short(s2, stored_len);
  put_short(s2, ~stored_len);
  if (stored_len) {
    s2.pending_buf.set(s2.window.subarray(buf, buf + stored_len), s2.pending);
  }
  s2.pending += stored_len;
};
const _tr_align$1 = (s2) => {
  send_bits(s2, STATIC_TREES << 1, 3);
  send_code(s2, END_BLOCK, static_ltree);
  bi_flush(s2);
};
const _tr_flush_block$1 = (s2, buf, stored_len, last) => {
  let opt_lenb, static_lenb;
  let max_blindex = 0;
  if (s2.level > 0) {
    if (s2.strm.data_type === Z_UNKNOWN$1) {
      s2.strm.data_type = detect_data_type(s2);
    }
    build_tree(s2, s2.l_desc);
    build_tree(s2, s2.d_desc);
    max_blindex = build_bl_tree(s2);
    opt_lenb = s2.opt_len + 3 + 7 >>> 3;
    static_lenb = s2.static_len + 3 + 7 >>> 3;
    if (static_lenb <= opt_lenb) {
      opt_lenb = static_lenb;
    }
  } else {
    opt_lenb = static_lenb = stored_len + 5;
  }
  if (stored_len + 4 <= opt_lenb && buf !== -1) {
    _tr_stored_block$1(s2, buf, stored_len, last);
  } else if (s2.strategy === Z_FIXED$1 || static_lenb === opt_lenb) {
    send_bits(s2, (STATIC_TREES << 1) + (last ? 1 : 0), 3);
    compress_block(s2, static_ltree, static_dtree);
  } else {
    send_bits(s2, (DYN_TREES << 1) + (last ? 1 : 0), 3);
    send_all_trees(s2, s2.l_desc.max_code + 1, s2.d_desc.max_code + 1, max_blindex + 1);
    compress_block(s2, s2.dyn_ltree, s2.dyn_dtree);
  }
  init_block(s2);
  if (last) {
    bi_windup(s2);
  }
};
const _tr_tally$1 = (s2, dist2, lc2) => {
  s2.pending_buf[s2.sym_buf + s2.sym_next++] = dist2;
  s2.pending_buf[s2.sym_buf + s2.sym_next++] = dist2 >> 8;
  s2.pending_buf[s2.sym_buf + s2.sym_next++] = lc2;
  if (dist2 === 0) {
    s2.dyn_ltree[lc2 * 2]++;
  } else {
    s2.matches++;
    dist2--;
    s2.dyn_ltree[(_length_code[lc2] + LITERALS$1 + 1) * 2]++;
    s2.dyn_dtree[d_code(dist2) * 2]++;
  }
  return s2.sym_next === s2.sym_end;
};
var _tr_init_1 = _tr_init$1;
var _tr_stored_block_1 = _tr_stored_block$1;
var _tr_flush_block_1 = _tr_flush_block$1;
var _tr_tally_1 = _tr_tally$1;
var _tr_align_1 = _tr_align$1;
var trees = {
  _tr_init: _tr_init_1,
  _tr_stored_block: _tr_stored_block_1,
  _tr_flush_block: _tr_flush_block_1,
  _tr_tally: _tr_tally_1,
  _tr_align: _tr_align_1
};
const adler32 = (adler2, buf, len2, pos) => {
  let s1 = adler2 & 65535 | 0, s2 = adler2 >>> 16 & 65535 | 0, n2 = 0;
  while (len2 !== 0) {
    n2 = len2 > 2e3 ? 2e3 : len2;
    len2 -= n2;
    do {
      s1 = s1 + buf[pos++] | 0;
      s2 = s2 + s1 | 0;
    } while (--n2);
    s1 %= 65521;
    s2 %= 65521;
  }
  return s1 | s2 << 16 | 0;
};
var adler32_1 = adler32;
const makeTable = () => {
  let c2, table = [];
  for (var n2 = 0; n2 < 256; n2++) {
    c2 = n2;
    for (var k2 = 0; k2 < 8; k2++) {
      c2 = c2 & 1 ? 3988292384 ^ c2 >>> 1 : c2 >>> 1;
    }
    table[n2] = c2;
  }
  return table;
};
const crcTable = new Uint32Array(makeTable());
const crc32 = (crc, buf, len2, pos) => {
  const t3 = crcTable;
  const end2 = pos + len2;
  crc ^= -1;
  for (let i2 = pos; i2 < end2; i2++) {
    crc = crc >>> 8 ^ t3[(crc ^ buf[i2]) & 255];
  }
  return crc ^ -1;
};
var crc32_1 = crc32;
var messages = {
  2: "need dictionary",
  /* Z_NEED_DICT       2  */
  1: "stream end",
  /* Z_STREAM_END      1  */
  0: "",
  /* Z_OK              0  */
  "-1": "file error",
  /* Z_ERRNO         (-1) */
  "-2": "stream error",
  /* Z_STREAM_ERROR  (-2) */
  "-3": "data error",
  /* Z_DATA_ERROR    (-3) */
  "-4": "insufficient memory",
  /* Z_MEM_ERROR     (-4) */
  "-5": "buffer error",
  /* Z_BUF_ERROR     (-5) */
  "-6": "incompatible version"
  /* Z_VERSION_ERROR (-6) */
};
var constants$2$1 = {
  /* Allowed flush values; see deflate() and inflate() below for details */
  Z_NO_FLUSH: 0,
  Z_PARTIAL_FLUSH: 1,
  Z_SYNC_FLUSH: 2,
  Z_FULL_FLUSH: 3,
  Z_FINISH: 4,
  Z_BLOCK: 5,
  Z_TREES: 6,
  /* Return codes for the compression/decompression functions. Negative values
  * are errors, positive values are used for special but normal events.
  */
  Z_OK: 0,
  Z_STREAM_END: 1,
  Z_NEED_DICT: 2,
  Z_ERRNO: -1,
  Z_STREAM_ERROR: -2,
  Z_DATA_ERROR: -3,
  Z_MEM_ERROR: -4,
  Z_BUF_ERROR: -5,
  //Z_VERSION_ERROR: -6,
  /* compression levels */
  Z_NO_COMPRESSION: 0,
  Z_BEST_SPEED: 1,
  Z_BEST_COMPRESSION: 9,
  Z_DEFAULT_COMPRESSION: -1,
  Z_FILTERED: 1,
  Z_HUFFMAN_ONLY: 2,
  Z_RLE: 3,
  Z_FIXED: 4,
  Z_DEFAULT_STRATEGY: 0,
  /* Possible values of the data_type field (though see inflate()) */
  Z_BINARY: 0,
  Z_TEXT: 1,
  //Z_ASCII:                1, // = Z_TEXT (deprecated)
  Z_UNKNOWN: 2,
  /* The deflate compression method */
  Z_DEFLATED: 8
  //Z_NULL:                 null // Use -1 or null inline, depending on var type
};
const { _tr_init, _tr_stored_block, _tr_flush_block, _tr_tally, _tr_align } = trees;
const {
  Z_NO_FLUSH: Z_NO_FLUSH$2,
  Z_PARTIAL_FLUSH,
  Z_FULL_FLUSH: Z_FULL_FLUSH$1,
  Z_FINISH: Z_FINISH$3,
  Z_BLOCK: Z_BLOCK$1,
  Z_OK: Z_OK$3,
  Z_STREAM_END: Z_STREAM_END$3,
  Z_STREAM_ERROR: Z_STREAM_ERROR$2,
  Z_DATA_ERROR: Z_DATA_ERROR$2,
  Z_BUF_ERROR: Z_BUF_ERROR$1,
  Z_DEFAULT_COMPRESSION: Z_DEFAULT_COMPRESSION$1,
  Z_FILTERED,
  Z_HUFFMAN_ONLY,
  Z_RLE,
  Z_FIXED,
  Z_DEFAULT_STRATEGY: Z_DEFAULT_STRATEGY$1,
  Z_UNKNOWN,
  Z_DEFLATED: Z_DEFLATED$2
} = constants$2$1;
const MAX_MEM_LEVEL = 9;
const MAX_WBITS$1 = 15;
const DEF_MEM_LEVEL = 8;
const LENGTH_CODES = 29;
const LITERALS = 256;
const L_CODES = LITERALS + 1 + LENGTH_CODES;
const D_CODES = 30;
const BL_CODES = 19;
const HEAP_SIZE = 2 * L_CODES + 1;
const MAX_BITS = 15;
const MIN_MATCH = 3;
const MAX_MATCH = 258;
const MIN_LOOKAHEAD = MAX_MATCH + MIN_MATCH + 1;
const PRESET_DICT = 32;
const INIT_STATE = 42;
const GZIP_STATE = 57;
const EXTRA_STATE = 69;
const NAME_STATE = 73;
const COMMENT_STATE = 91;
const HCRC_STATE = 103;
const BUSY_STATE = 113;
const FINISH_STATE = 666;
const BS_NEED_MORE = 1;
const BS_BLOCK_DONE = 2;
const BS_FINISH_STARTED = 3;
const BS_FINISH_DONE = 4;
const OS_CODE = 3;
const err$1 = (strm, errorCode) => {
  strm.msg = messages[errorCode];
  return errorCode;
};
const rank = (f2) => {
  return f2 * 2 - (f2 > 4 ? 9 : 0);
};
const zero = (buf) => {
  let len2 = buf.length;
  while (--len2 >= 0) {
    buf[len2] = 0;
  }
};
const slide_hash = (s2) => {
  let n2, m2;
  let p2;
  let wsize = s2.w_size;
  n2 = s2.hash_size;
  p2 = n2;
  do {
    m2 = s2.head[--p2];
    s2.head[p2] = m2 >= wsize ? m2 - wsize : 0;
  } while (--n2);
  n2 = wsize;
  p2 = n2;
  do {
    m2 = s2.prev[--p2];
    s2.prev[p2] = m2 >= wsize ? m2 - wsize : 0;
  } while (--n2);
};
let HASH_ZLIB = (s2, prev, data2) => (prev << s2.hash_shift ^ data2) & s2.hash_mask;
let HASH = HASH_ZLIB;
const flush_pending = (strm) => {
  const s2 = strm.state;
  let len2 = s2.pending;
  if (len2 > strm.avail_out) {
    len2 = strm.avail_out;
  }
  if (len2 === 0) {
    return;
  }
  strm.output.set(s2.pending_buf.subarray(s2.pending_out, s2.pending_out + len2), strm.next_out);
  strm.next_out += len2;
  s2.pending_out += len2;
  strm.total_out += len2;
  strm.avail_out -= len2;
  s2.pending -= len2;
  if (s2.pending === 0) {
    s2.pending_out = 0;
  }
};
const flush_block_only = (s2, last) => {
  _tr_flush_block(s2, s2.block_start >= 0 ? s2.block_start : -1, s2.strstart - s2.block_start, last);
  s2.block_start = s2.strstart;
  flush_pending(s2.strm);
};
const put_byte = (s2, b2) => {
  s2.pending_buf[s2.pending++] = b2;
};
const putShortMSB = (s2, b2) => {
  s2.pending_buf[s2.pending++] = b2 >>> 8 & 255;
  s2.pending_buf[s2.pending++] = b2 & 255;
};
const read_buf = (strm, buf, start2, size) => {
  let len2 = strm.avail_in;
  if (len2 > size) {
    len2 = size;
  }
  if (len2 === 0) {
    return 0;
  }
  strm.avail_in -= len2;
  buf.set(strm.input.subarray(strm.next_in, strm.next_in + len2), start2);
  if (strm.state.wrap === 1) {
    strm.adler = adler32_1(strm.adler, buf, len2, start2);
  } else if (strm.state.wrap === 2) {
    strm.adler = crc32_1(strm.adler, buf, len2, start2);
  }
  strm.next_in += len2;
  strm.total_in += len2;
  return len2;
};
const longest_match = (s2, cur_match) => {
  let chain_length = s2.max_chain_length;
  let scan = s2.strstart;
  let match5;
  let len2;
  let best_len = s2.prev_length;
  let nice_match = s2.nice_match;
  const limit = s2.strstart > s2.w_size - MIN_LOOKAHEAD ? s2.strstart - (s2.w_size - MIN_LOOKAHEAD) : 0;
  const _win = s2.window;
  const wmask = s2.w_mask;
  const prev = s2.prev;
  const strend = s2.strstart + MAX_MATCH;
  let scan_end1 = _win[scan + best_len - 1];
  let scan_end = _win[scan + best_len];
  if (s2.prev_length >= s2.good_match) {
    chain_length >>= 2;
  }
  if (nice_match > s2.lookahead) {
    nice_match = s2.lookahead;
  }
  do {
    match5 = cur_match;
    if (_win[match5 + best_len] !== scan_end || _win[match5 + best_len - 1] !== scan_end1 || _win[match5] !== _win[scan] || _win[++match5] !== _win[scan + 1]) {
      continue;
    }
    scan += 2;
    match5++;
    do {
    } while (_win[++scan] === _win[++match5] && _win[++scan] === _win[++match5] && _win[++scan] === _win[++match5] && _win[++scan] === _win[++match5] && _win[++scan] === _win[++match5] && _win[++scan] === _win[++match5] && _win[++scan] === _win[++match5] && _win[++scan] === _win[++match5] && scan < strend);
    len2 = MAX_MATCH - (strend - scan);
    scan = strend - MAX_MATCH;
    if (len2 > best_len) {
      s2.match_start = cur_match;
      best_len = len2;
      if (len2 >= nice_match) {
        break;
      }
      scan_end1 = _win[scan + best_len - 1];
      scan_end = _win[scan + best_len];
    }
  } while ((cur_match = prev[cur_match & wmask]) > limit && --chain_length !== 0);
  if (best_len <= s2.lookahead) {
    return best_len;
  }
  return s2.lookahead;
};
const fill_window = (s2) => {
  const _w_size = s2.w_size;
  let n2, more, str;
  do {
    more = s2.window_size - s2.lookahead - s2.strstart;
    if (s2.strstart >= _w_size + (_w_size - MIN_LOOKAHEAD)) {
      s2.window.set(s2.window.subarray(_w_size, _w_size + _w_size - more), 0);
      s2.match_start -= _w_size;
      s2.strstart -= _w_size;
      s2.block_start -= _w_size;
      if (s2.insert > s2.strstart) {
        s2.insert = s2.strstart;
      }
      slide_hash(s2);
      more += _w_size;
    }
    if (s2.strm.avail_in === 0) {
      break;
    }
    n2 = read_buf(s2.strm, s2.window, s2.strstart + s2.lookahead, more);
    s2.lookahead += n2;
    if (s2.lookahead + s2.insert >= MIN_MATCH) {
      str = s2.strstart - s2.insert;
      s2.ins_h = s2.window[str];
      s2.ins_h = HASH(s2, s2.ins_h, s2.window[str + 1]);
      while (s2.insert) {
        s2.ins_h = HASH(s2, s2.ins_h, s2.window[str + MIN_MATCH - 1]);
        s2.prev[str & s2.w_mask] = s2.head[s2.ins_h];
        s2.head[s2.ins_h] = str;
        str++;
        s2.insert--;
        if (s2.lookahead + s2.insert < MIN_MATCH) {
          break;
        }
      }
    }
  } while (s2.lookahead < MIN_LOOKAHEAD && s2.strm.avail_in !== 0);
};
const deflate_stored = (s2, flush) => {
  let min_block = s2.pending_buf_size - 5 > s2.w_size ? s2.w_size : s2.pending_buf_size - 5;
  let len2, left, have, last = 0;
  let used = s2.strm.avail_in;
  do {
    len2 = 65535;
    have = s2.bi_valid + 42 >> 3;
    if (s2.strm.avail_out < have) {
      break;
    }
    have = s2.strm.avail_out - have;
    left = s2.strstart - s2.block_start;
    if (len2 > left + s2.strm.avail_in) {
      len2 = left + s2.strm.avail_in;
    }
    if (len2 > have) {
      len2 = have;
    }
    if (len2 < min_block && (len2 === 0 && flush !== Z_FINISH$3 || flush === Z_NO_FLUSH$2 || len2 !== left + s2.strm.avail_in)) {
      break;
    }
    last = flush === Z_FINISH$3 && len2 === left + s2.strm.avail_in ? 1 : 0;
    _tr_stored_block(s2, 0, 0, last);
    s2.pending_buf[s2.pending - 4] = len2;
    s2.pending_buf[s2.pending - 3] = len2 >> 8;
    s2.pending_buf[s2.pending - 2] = ~len2;
    s2.pending_buf[s2.pending - 1] = ~len2 >> 8;
    flush_pending(s2.strm);
    if (left) {
      if (left > len2) {
        left = len2;
      }
      s2.strm.output.set(s2.window.subarray(s2.block_start, s2.block_start + left), s2.strm.next_out);
      s2.strm.next_out += left;
      s2.strm.avail_out -= left;
      s2.strm.total_out += left;
      s2.block_start += left;
      len2 -= left;
    }
    if (len2) {
      read_buf(s2.strm, s2.strm.output, s2.strm.next_out, len2);
      s2.strm.next_out += len2;
      s2.strm.avail_out -= len2;
      s2.strm.total_out += len2;
    }
  } while (last === 0);
  used -= s2.strm.avail_in;
  if (used) {
    if (used >= s2.w_size) {
      s2.matches = 2;
      s2.window.set(s2.strm.input.subarray(s2.strm.next_in - s2.w_size, s2.strm.next_in), 0);
      s2.strstart = s2.w_size;
      s2.insert = s2.strstart;
    } else {
      if (s2.window_size - s2.strstart <= used) {
        s2.strstart -= s2.w_size;
        s2.window.set(s2.window.subarray(s2.w_size, s2.w_size + s2.strstart), 0);
        if (s2.matches < 2) {
          s2.matches++;
        }
        if (s2.insert > s2.strstart) {
          s2.insert = s2.strstart;
        }
      }
      s2.window.set(s2.strm.input.subarray(s2.strm.next_in - used, s2.strm.next_in), s2.strstart);
      s2.strstart += used;
      s2.insert += used > s2.w_size - s2.insert ? s2.w_size - s2.insert : used;
    }
    s2.block_start = s2.strstart;
  }
  if (s2.high_water < s2.strstart) {
    s2.high_water = s2.strstart;
  }
  if (last) {
    return BS_FINISH_DONE;
  }
  if (flush !== Z_NO_FLUSH$2 && flush !== Z_FINISH$3 && s2.strm.avail_in === 0 && s2.strstart === s2.block_start) {
    return BS_BLOCK_DONE;
  }
  have = s2.window_size - s2.strstart;
  if (s2.strm.avail_in > have && s2.block_start >= s2.w_size) {
    s2.block_start -= s2.w_size;
    s2.strstart -= s2.w_size;
    s2.window.set(s2.window.subarray(s2.w_size, s2.w_size + s2.strstart), 0);
    if (s2.matches < 2) {
      s2.matches++;
    }
    have += s2.w_size;
    if (s2.insert > s2.strstart) {
      s2.insert = s2.strstart;
    }
  }
  if (have > s2.strm.avail_in) {
    have = s2.strm.avail_in;
  }
  if (have) {
    read_buf(s2.strm, s2.window, s2.strstart, have);
    s2.strstart += have;
    s2.insert += have > s2.w_size - s2.insert ? s2.w_size - s2.insert : have;
  }
  if (s2.high_water < s2.strstart) {
    s2.high_water = s2.strstart;
  }
  have = s2.bi_valid + 42 >> 3;
  have = s2.pending_buf_size - have > 65535 ? 65535 : s2.pending_buf_size - have;
  min_block = have > s2.w_size ? s2.w_size : have;
  left = s2.strstart - s2.block_start;
  if (left >= min_block || (left || flush === Z_FINISH$3) && flush !== Z_NO_FLUSH$2 && s2.strm.avail_in === 0 && left <= have) {
    len2 = left > have ? have : left;
    last = flush === Z_FINISH$3 && s2.strm.avail_in === 0 && len2 === left ? 1 : 0;
    _tr_stored_block(s2, s2.block_start, len2, last);
    s2.block_start += len2;
    flush_pending(s2.strm);
  }
  return last ? BS_FINISH_STARTED : BS_NEED_MORE;
};
const deflate_fast = (s2, flush) => {
  let hash_head;
  let bflush;
  for (; ; ) {
    if (s2.lookahead < MIN_LOOKAHEAD) {
      fill_window(s2);
      if (s2.lookahead < MIN_LOOKAHEAD && flush === Z_NO_FLUSH$2) {
        return BS_NEED_MORE;
      }
      if (s2.lookahead === 0) {
        break;
      }
    }
    hash_head = 0;
    if (s2.lookahead >= MIN_MATCH) {
      s2.ins_h = HASH(s2, s2.ins_h, s2.window[s2.strstart + MIN_MATCH - 1]);
      hash_head = s2.prev[s2.strstart & s2.w_mask] = s2.head[s2.ins_h];
      s2.head[s2.ins_h] = s2.strstart;
    }
    if (hash_head !== 0 && s2.strstart - hash_head <= s2.w_size - MIN_LOOKAHEAD) {
      s2.match_length = longest_match(s2, hash_head);
    }
    if (s2.match_length >= MIN_MATCH) {
      bflush = _tr_tally(s2, s2.strstart - s2.match_start, s2.match_length - MIN_MATCH);
      s2.lookahead -= s2.match_length;
      if (s2.match_length <= s2.max_lazy_match && s2.lookahead >= MIN_MATCH) {
        s2.match_length--;
        do {
          s2.strstart++;
          s2.ins_h = HASH(s2, s2.ins_h, s2.window[s2.strstart + MIN_MATCH - 1]);
          hash_head = s2.prev[s2.strstart & s2.w_mask] = s2.head[s2.ins_h];
          s2.head[s2.ins_h] = s2.strstart;
        } while (--s2.match_length !== 0);
        s2.strstart++;
      } else {
        s2.strstart += s2.match_length;
        s2.match_length = 0;
        s2.ins_h = s2.window[s2.strstart];
        s2.ins_h = HASH(s2, s2.ins_h, s2.window[s2.strstart + 1]);
      }
    } else {
      bflush = _tr_tally(s2, 0, s2.window[s2.strstart]);
      s2.lookahead--;
      s2.strstart++;
    }
    if (bflush) {
      flush_block_only(s2, false);
      if (s2.strm.avail_out === 0) {
        return BS_NEED_MORE;
      }
    }
  }
  s2.insert = s2.strstart < MIN_MATCH - 1 ? s2.strstart : MIN_MATCH - 1;
  if (flush === Z_FINISH$3) {
    flush_block_only(s2, true);
    if (s2.strm.avail_out === 0) {
      return BS_FINISH_STARTED;
    }
    return BS_FINISH_DONE;
  }
  if (s2.sym_next) {
    flush_block_only(s2, false);
    if (s2.strm.avail_out === 0) {
      return BS_NEED_MORE;
    }
  }
  return BS_BLOCK_DONE;
};
const deflate_slow = (s2, flush) => {
  let hash_head;
  let bflush;
  let max_insert;
  for (; ; ) {
    if (s2.lookahead < MIN_LOOKAHEAD) {
      fill_window(s2);
      if (s2.lookahead < MIN_LOOKAHEAD && flush === Z_NO_FLUSH$2) {
        return BS_NEED_MORE;
      }
      if (s2.lookahead === 0) {
        break;
      }
    }
    hash_head = 0;
    if (s2.lookahead >= MIN_MATCH) {
      s2.ins_h = HASH(s2, s2.ins_h, s2.window[s2.strstart + MIN_MATCH - 1]);
      hash_head = s2.prev[s2.strstart & s2.w_mask] = s2.head[s2.ins_h];
      s2.head[s2.ins_h] = s2.strstart;
    }
    s2.prev_length = s2.match_length;
    s2.prev_match = s2.match_start;
    s2.match_length = MIN_MATCH - 1;
    if (hash_head !== 0 && s2.prev_length < s2.max_lazy_match && s2.strstart - hash_head <= s2.w_size - MIN_LOOKAHEAD) {
      s2.match_length = longest_match(s2, hash_head);
      if (s2.match_length <= 5 && (s2.strategy === Z_FILTERED || s2.match_length === MIN_MATCH && s2.strstart - s2.match_start > 4096)) {
        s2.match_length = MIN_MATCH - 1;
      }
    }
    if (s2.prev_length >= MIN_MATCH && s2.match_length <= s2.prev_length) {
      max_insert = s2.strstart + s2.lookahead - MIN_MATCH;
      bflush = _tr_tally(s2, s2.strstart - 1 - s2.prev_match, s2.prev_length - MIN_MATCH);
      s2.lookahead -= s2.prev_length - 1;
      s2.prev_length -= 2;
      do {
        if (++s2.strstart <= max_insert) {
          s2.ins_h = HASH(s2, s2.ins_h, s2.window[s2.strstart + MIN_MATCH - 1]);
          hash_head = s2.prev[s2.strstart & s2.w_mask] = s2.head[s2.ins_h];
          s2.head[s2.ins_h] = s2.strstart;
        }
      } while (--s2.prev_length !== 0);
      s2.match_available = 0;
      s2.match_length = MIN_MATCH - 1;
      s2.strstart++;
      if (bflush) {
        flush_block_only(s2, false);
        if (s2.strm.avail_out === 0) {
          return BS_NEED_MORE;
        }
      }
    } else if (s2.match_available) {
      bflush = _tr_tally(s2, 0, s2.window[s2.strstart - 1]);
      if (bflush) {
        flush_block_only(s2, false);
      }
      s2.strstart++;
      s2.lookahead--;
      if (s2.strm.avail_out === 0) {
        return BS_NEED_MORE;
      }
    } else {
      s2.match_available = 1;
      s2.strstart++;
      s2.lookahead--;
    }
  }
  if (s2.match_available) {
    bflush = _tr_tally(s2, 0, s2.window[s2.strstart - 1]);
    s2.match_available = 0;
  }
  s2.insert = s2.strstart < MIN_MATCH - 1 ? s2.strstart : MIN_MATCH - 1;
  if (flush === Z_FINISH$3) {
    flush_block_only(s2, true);
    if (s2.strm.avail_out === 0) {
      return BS_FINISH_STARTED;
    }
    return BS_FINISH_DONE;
  }
  if (s2.sym_next) {
    flush_block_only(s2, false);
    if (s2.strm.avail_out === 0) {
      return BS_NEED_MORE;
    }
  }
  return BS_BLOCK_DONE;
};
const deflate_rle = (s2, flush) => {
  let bflush;
  let prev;
  let scan, strend;
  const _win = s2.window;
  for (; ; ) {
    if (s2.lookahead <= MAX_MATCH) {
      fill_window(s2);
      if (s2.lookahead <= MAX_MATCH && flush === Z_NO_FLUSH$2) {
        return BS_NEED_MORE;
      }
      if (s2.lookahead === 0) {
        break;
      }
    }
    s2.match_length = 0;
    if (s2.lookahead >= MIN_MATCH && s2.strstart > 0) {
      scan = s2.strstart - 1;
      prev = _win[scan];
      if (prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan]) {
        strend = s2.strstart + MAX_MATCH;
        do {
        } while (prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && prev === _win[++scan] && scan < strend);
        s2.match_length = MAX_MATCH - (strend - scan);
        if (s2.match_length > s2.lookahead) {
          s2.match_length = s2.lookahead;
        }
      }
    }
    if (s2.match_length >= MIN_MATCH) {
      bflush = _tr_tally(s2, 1, s2.match_length - MIN_MATCH);
      s2.lookahead -= s2.match_length;
      s2.strstart += s2.match_length;
      s2.match_length = 0;
    } else {
      bflush = _tr_tally(s2, 0, s2.window[s2.strstart]);
      s2.lookahead--;
      s2.strstart++;
    }
    if (bflush) {
      flush_block_only(s2, false);
      if (s2.strm.avail_out === 0) {
        return BS_NEED_MORE;
      }
    }
  }
  s2.insert = 0;
  if (flush === Z_FINISH$3) {
    flush_block_only(s2, true);
    if (s2.strm.avail_out === 0) {
      return BS_FINISH_STARTED;
    }
    return BS_FINISH_DONE;
  }
  if (s2.sym_next) {
    flush_block_only(s2, false);
    if (s2.strm.avail_out === 0) {
      return BS_NEED_MORE;
    }
  }
  return BS_BLOCK_DONE;
};
const deflate_huff = (s2, flush) => {
  let bflush;
  for (; ; ) {
    if (s2.lookahead === 0) {
      fill_window(s2);
      if (s2.lookahead === 0) {
        if (flush === Z_NO_FLUSH$2) {
          return BS_NEED_MORE;
        }
        break;
      }
    }
    s2.match_length = 0;
    bflush = _tr_tally(s2, 0, s2.window[s2.strstart]);
    s2.lookahead--;
    s2.strstart++;
    if (bflush) {
      flush_block_only(s2, false);
      if (s2.strm.avail_out === 0) {
        return BS_NEED_MORE;
      }
    }
  }
  s2.insert = 0;
  if (flush === Z_FINISH$3) {
    flush_block_only(s2, true);
    if (s2.strm.avail_out === 0) {
      return BS_FINISH_STARTED;
    }
    return BS_FINISH_DONE;
  }
  if (s2.sym_next) {
    flush_block_only(s2, false);
    if (s2.strm.avail_out === 0) {
      return BS_NEED_MORE;
    }
  }
  return BS_BLOCK_DONE;
};
function Config(good_length, max_lazy, nice_length, max_chain, func) {
  this.good_length = good_length;
  this.max_lazy = max_lazy;
  this.nice_length = nice_length;
  this.max_chain = max_chain;
  this.func = func;
}
const configuration_table = [
  /*      good lazy nice chain */
  new Config(0, 0, 0, 0, deflate_stored),
  /* 0 store only */
  new Config(4, 4, 8, 4, deflate_fast),
  /* 1 max speed, no lazy matches */
  new Config(4, 5, 16, 8, deflate_fast),
  /* 2 */
  new Config(4, 6, 32, 32, deflate_fast),
  /* 3 */
  new Config(4, 4, 16, 16, deflate_slow),
  /* 4 lazy matches */
  new Config(8, 16, 32, 32, deflate_slow),
  /* 5 */
  new Config(8, 16, 128, 128, deflate_slow),
  /* 6 */
  new Config(8, 32, 128, 256, deflate_slow),
  /* 7 */
  new Config(32, 128, 258, 1024, deflate_slow),
  /* 8 */
  new Config(32, 258, 258, 4096, deflate_slow)
  /* 9 max compression */
];
const lm_init = (s2) => {
  s2.window_size = 2 * s2.w_size;
  zero(s2.head);
  s2.max_lazy_match = configuration_table[s2.level].max_lazy;
  s2.good_match = configuration_table[s2.level].good_length;
  s2.nice_match = configuration_table[s2.level].nice_length;
  s2.max_chain_length = configuration_table[s2.level].max_chain;
  s2.strstart = 0;
  s2.block_start = 0;
  s2.lookahead = 0;
  s2.insert = 0;
  s2.match_length = s2.prev_length = MIN_MATCH - 1;
  s2.match_available = 0;
  s2.ins_h = 0;
};
function DeflateState() {
  this.strm = null;
  this.status = 0;
  this.pending_buf = null;
  this.pending_buf_size = 0;
  this.pending_out = 0;
  this.pending = 0;
  this.wrap = 0;
  this.gzhead = null;
  this.gzindex = 0;
  this.method = Z_DEFLATED$2;
  this.last_flush = -1;
  this.w_size = 0;
  this.w_bits = 0;
  this.w_mask = 0;
  this.window = null;
  this.window_size = 0;
  this.prev = null;
  this.head = null;
  this.ins_h = 0;
  this.hash_size = 0;
  this.hash_bits = 0;
  this.hash_mask = 0;
  this.hash_shift = 0;
  this.block_start = 0;
  this.match_length = 0;
  this.prev_match = 0;
  this.match_available = 0;
  this.strstart = 0;
  this.match_start = 0;
  this.lookahead = 0;
  this.prev_length = 0;
  this.max_chain_length = 0;
  this.max_lazy_match = 0;
  this.level = 0;
  this.strategy = 0;
  this.good_match = 0;
  this.nice_match = 0;
  this.dyn_ltree = new Uint16Array(HEAP_SIZE * 2);
  this.dyn_dtree = new Uint16Array((2 * D_CODES + 1) * 2);
  this.bl_tree = new Uint16Array((2 * BL_CODES + 1) * 2);
  zero(this.dyn_ltree);
  zero(this.dyn_dtree);
  zero(this.bl_tree);
  this.l_desc = null;
  this.d_desc = null;
  this.bl_desc = null;
  this.bl_count = new Uint16Array(MAX_BITS + 1);
  this.heap = new Uint16Array(2 * L_CODES + 1);
  zero(this.heap);
  this.heap_len = 0;
  this.heap_max = 0;
  this.depth = new Uint16Array(2 * L_CODES + 1);
  zero(this.depth);
  this.sym_buf = 0;
  this.lit_bufsize = 0;
  this.sym_next = 0;
  this.sym_end = 0;
  this.opt_len = 0;
  this.static_len = 0;
  this.matches = 0;
  this.insert = 0;
  this.bi_buf = 0;
  this.bi_valid = 0;
}
const deflateStateCheck = (strm) => {
  if (!strm) {
    return 1;
  }
  const s2 = strm.state;
  if (!s2 || s2.strm !== strm || s2.status !== INIT_STATE && //#ifdef GZIP
  s2.status !== GZIP_STATE && //#endif
  s2.status !== EXTRA_STATE && s2.status !== NAME_STATE && s2.status !== COMMENT_STATE && s2.status !== HCRC_STATE && s2.status !== BUSY_STATE && s2.status !== FINISH_STATE) {
    return 1;
  }
  return 0;
};
const deflateResetKeep = (strm) => {
  if (deflateStateCheck(strm)) {
    return err$1(strm, Z_STREAM_ERROR$2);
  }
  strm.total_in = strm.total_out = 0;
  strm.data_type = Z_UNKNOWN;
  const s2 = strm.state;
  s2.pending = 0;
  s2.pending_out = 0;
  if (s2.wrap < 0) {
    s2.wrap = -s2.wrap;
  }
  s2.status = //#ifdef GZIP
  s2.wrap === 2 ? GZIP_STATE : (
    //#endif
    s2.wrap ? INIT_STATE : BUSY_STATE
  );
  strm.adler = s2.wrap === 2 ? 0 : 1;
  s2.last_flush = -2;
  _tr_init(s2);
  return Z_OK$3;
};
const deflateReset = (strm) => {
  const ret = deflateResetKeep(strm);
  if (ret === Z_OK$3) {
    lm_init(strm.state);
  }
  return ret;
};
const deflateSetHeader = (strm, head) => {
  if (deflateStateCheck(strm) || strm.state.wrap !== 2) {
    return Z_STREAM_ERROR$2;
  }
  strm.state.gzhead = head;
  return Z_OK$3;
};
const deflateInit2 = (strm, level, method, windowBits, memLevel, strategy) => {
  if (!strm) {
    return Z_STREAM_ERROR$2;
  }
  let wrap = 1;
  if (level === Z_DEFAULT_COMPRESSION$1) {
    level = 6;
  }
  if (windowBits < 0) {
    wrap = 0;
    windowBits = -windowBits;
  } else if (windowBits > 15) {
    wrap = 2;
    windowBits -= 16;
  }
  if (memLevel < 1 || memLevel > MAX_MEM_LEVEL || method !== Z_DEFLATED$2 || windowBits < 8 || windowBits > 15 || level < 0 || level > 9 || strategy < 0 || strategy > Z_FIXED || windowBits === 8 && wrap !== 1) {
    return err$1(strm, Z_STREAM_ERROR$2);
  }
  if (windowBits === 8) {
    windowBits = 9;
  }
  const s2 = new DeflateState();
  strm.state = s2;
  s2.strm = strm;
  s2.status = INIT_STATE;
  s2.wrap = wrap;
  s2.gzhead = null;
  s2.w_bits = windowBits;
  s2.w_size = 1 << s2.w_bits;
  s2.w_mask = s2.w_size - 1;
  s2.hash_bits = memLevel + 7;
  s2.hash_size = 1 << s2.hash_bits;
  s2.hash_mask = s2.hash_size - 1;
  s2.hash_shift = ~~((s2.hash_bits + MIN_MATCH - 1) / MIN_MATCH);
  s2.window = new Uint8Array(s2.w_size * 2);
  s2.head = new Uint16Array(s2.hash_size);
  s2.prev = new Uint16Array(s2.w_size);
  s2.lit_bufsize = 1 << memLevel + 6;
  s2.pending_buf_size = s2.lit_bufsize * 4;
  s2.pending_buf = new Uint8Array(s2.pending_buf_size);
  s2.sym_buf = s2.lit_bufsize;
  s2.sym_end = (s2.lit_bufsize - 1) * 3;
  s2.level = level;
  s2.strategy = strategy;
  s2.method = method;
  return deflateReset(strm);
};
const deflateInit = (strm, level) => {
  return deflateInit2(strm, level, Z_DEFLATED$2, MAX_WBITS$1, DEF_MEM_LEVEL, Z_DEFAULT_STRATEGY$1);
};
const deflate$2 = (strm, flush) => {
  if (deflateStateCheck(strm) || flush > Z_BLOCK$1 || flush < 0) {
    return strm ? err$1(strm, Z_STREAM_ERROR$2) : Z_STREAM_ERROR$2;
  }
  const s2 = strm.state;
  if (!strm.output || strm.avail_in !== 0 && !strm.input || s2.status === FINISH_STATE && flush !== Z_FINISH$3) {
    return err$1(strm, strm.avail_out === 0 ? Z_BUF_ERROR$1 : Z_STREAM_ERROR$2);
  }
  const old_flush = s2.last_flush;
  s2.last_flush = flush;
  if (s2.pending !== 0) {
    flush_pending(strm);
    if (strm.avail_out === 0) {
      s2.last_flush = -1;
      return Z_OK$3;
    }
  } else if (strm.avail_in === 0 && rank(flush) <= rank(old_flush) && flush !== Z_FINISH$3) {
    return err$1(strm, Z_BUF_ERROR$1);
  }
  if (s2.status === FINISH_STATE && strm.avail_in !== 0) {
    return err$1(strm, Z_BUF_ERROR$1);
  }
  if (s2.status === INIT_STATE && s2.wrap === 0) {
    s2.status = BUSY_STATE;
  }
  if (s2.status === INIT_STATE) {
    let header = Z_DEFLATED$2 + (s2.w_bits - 8 << 4) << 8;
    let level_flags = -1;
    if (s2.strategy >= Z_HUFFMAN_ONLY || s2.level < 2) {
      level_flags = 0;
    } else if (s2.level < 6) {
      level_flags = 1;
    } else if (s2.level === 6) {
      level_flags = 2;
    } else {
      level_flags = 3;
    }
    header |= level_flags << 6;
    if (s2.strstart !== 0) {
      header |= PRESET_DICT;
    }
    header += 31 - header % 31;
    putShortMSB(s2, header);
    if (s2.strstart !== 0) {
      putShortMSB(s2, strm.adler >>> 16);
      putShortMSB(s2, strm.adler & 65535);
    }
    strm.adler = 1;
    s2.status = BUSY_STATE;
    flush_pending(strm);
    if (s2.pending !== 0) {
      s2.last_flush = -1;
      return Z_OK$3;
    }
  }
  if (s2.status === GZIP_STATE) {
    strm.adler = 0;
    put_byte(s2, 31);
    put_byte(s2, 139);
    put_byte(s2, 8);
    if (!s2.gzhead) {
      put_byte(s2, 0);
      put_byte(s2, 0);
      put_byte(s2, 0);
      put_byte(s2, 0);
      put_byte(s2, 0);
      put_byte(s2, s2.level === 9 ? 2 : s2.strategy >= Z_HUFFMAN_ONLY || s2.level < 2 ? 4 : 0);
      put_byte(s2, OS_CODE);
      s2.status = BUSY_STATE;
      flush_pending(strm);
      if (s2.pending !== 0) {
        s2.last_flush = -1;
        return Z_OK$3;
      }
    } else {
      put_byte(
        s2,
        (s2.gzhead.text ? 1 : 0) + (s2.gzhead.hcrc ? 2 : 0) + (!s2.gzhead.extra ? 0 : 4) + (!s2.gzhead.name ? 0 : 8) + (!s2.gzhead.comment ? 0 : 16)
      );
      put_byte(s2, s2.gzhead.time & 255);
      put_byte(s2, s2.gzhead.time >> 8 & 255);
      put_byte(s2, s2.gzhead.time >> 16 & 255);
      put_byte(s2, s2.gzhead.time >> 24 & 255);
      put_byte(s2, s2.level === 9 ? 2 : s2.strategy >= Z_HUFFMAN_ONLY || s2.level < 2 ? 4 : 0);
      put_byte(s2, s2.gzhead.os & 255);
      if (s2.gzhead.extra && s2.gzhead.extra.length) {
        put_byte(s2, s2.gzhead.extra.length & 255);
        put_byte(s2, s2.gzhead.extra.length >> 8 & 255);
      }
      if (s2.gzhead.hcrc) {
        strm.adler = crc32_1(strm.adler, s2.pending_buf, s2.pending, 0);
      }
      s2.gzindex = 0;
      s2.status = EXTRA_STATE;
    }
  }
  if (s2.status === EXTRA_STATE) {
    if (s2.gzhead.extra) {
      let beg = s2.pending;
      let left = (s2.gzhead.extra.length & 65535) - s2.gzindex;
      while (s2.pending + left > s2.pending_buf_size) {
        let copy2 = s2.pending_buf_size - s2.pending;
        s2.pending_buf.set(s2.gzhead.extra.subarray(s2.gzindex, s2.gzindex + copy2), s2.pending);
        s2.pending = s2.pending_buf_size;
        if (s2.gzhead.hcrc && s2.pending > beg) {
          strm.adler = crc32_1(strm.adler, s2.pending_buf, s2.pending - beg, beg);
        }
        s2.gzindex += copy2;
        flush_pending(strm);
        if (s2.pending !== 0) {
          s2.last_flush = -1;
          return Z_OK$3;
        }
        beg = 0;
        left -= copy2;
      }
      let gzhead_extra = new Uint8Array(s2.gzhead.extra);
      s2.pending_buf.set(gzhead_extra.subarray(s2.gzindex, s2.gzindex + left), s2.pending);
      s2.pending += left;
      if (s2.gzhead.hcrc && s2.pending > beg) {
        strm.adler = crc32_1(strm.adler, s2.pending_buf, s2.pending - beg, beg);
      }
      s2.gzindex = 0;
    }
    s2.status = NAME_STATE;
  }
  if (s2.status === NAME_STATE) {
    if (s2.gzhead.name) {
      let beg = s2.pending;
      let val;
      do {
        if (s2.pending === s2.pending_buf_size) {
          if (s2.gzhead.hcrc && s2.pending > beg) {
            strm.adler = crc32_1(strm.adler, s2.pending_buf, s2.pending - beg, beg);
          }
          flush_pending(strm);
          if (s2.pending !== 0) {
            s2.last_flush = -1;
            return Z_OK$3;
          }
          beg = 0;
        }
        if (s2.gzindex < s2.gzhead.name.length) {
          val = s2.gzhead.name.charCodeAt(s2.gzindex++) & 255;
        } else {
          val = 0;
        }
        put_byte(s2, val);
      } while (val !== 0);
      if (s2.gzhead.hcrc && s2.pending > beg) {
        strm.adler = crc32_1(strm.adler, s2.pending_buf, s2.pending - beg, beg);
      }
      s2.gzindex = 0;
    }
    s2.status = COMMENT_STATE;
  }
  if (s2.status === COMMENT_STATE) {
    if (s2.gzhead.comment) {
      let beg = s2.pending;
      let val;
      do {
        if (s2.pending === s2.pending_buf_size) {
          if (s2.gzhead.hcrc && s2.pending > beg) {
            strm.adler = crc32_1(strm.adler, s2.pending_buf, s2.pending - beg, beg);
          }
          flush_pending(strm);
          if (s2.pending !== 0) {
            s2.last_flush = -1;
            return Z_OK$3;
          }
          beg = 0;
        }
        if (s2.gzindex < s2.gzhead.comment.length) {
          val = s2.gzhead.comment.charCodeAt(s2.gzindex++) & 255;
        } else {
          val = 0;
        }
        put_byte(s2, val);
      } while (val !== 0);
      if (s2.gzhead.hcrc && s2.pending > beg) {
        strm.adler = crc32_1(strm.adler, s2.pending_buf, s2.pending - beg, beg);
      }
    }
    s2.status = HCRC_STATE;
  }
  if (s2.status === HCRC_STATE) {
    if (s2.gzhead.hcrc) {
      if (s2.pending + 2 > s2.pending_buf_size) {
        flush_pending(strm);
        if (s2.pending !== 0) {
          s2.last_flush = -1;
          return Z_OK$3;
        }
      }
      put_byte(s2, strm.adler & 255);
      put_byte(s2, strm.adler >> 8 & 255);
      strm.adler = 0;
    }
    s2.status = BUSY_STATE;
    flush_pending(strm);
    if (s2.pending !== 0) {
      s2.last_flush = -1;
      return Z_OK$3;
    }
  }
  if (strm.avail_in !== 0 || s2.lookahead !== 0 || flush !== Z_NO_FLUSH$2 && s2.status !== FINISH_STATE) {
    let bstate = s2.level === 0 ? deflate_stored(s2, flush) : s2.strategy === Z_HUFFMAN_ONLY ? deflate_huff(s2, flush) : s2.strategy === Z_RLE ? deflate_rle(s2, flush) : configuration_table[s2.level].func(s2, flush);
    if (bstate === BS_FINISH_STARTED || bstate === BS_FINISH_DONE) {
      s2.status = FINISH_STATE;
    }
    if (bstate === BS_NEED_MORE || bstate === BS_FINISH_STARTED) {
      if (strm.avail_out === 0) {
        s2.last_flush = -1;
      }
      return Z_OK$3;
    }
    if (bstate === BS_BLOCK_DONE) {
      if (flush === Z_PARTIAL_FLUSH) {
        _tr_align(s2);
      } else if (flush !== Z_BLOCK$1) {
        _tr_stored_block(s2, 0, 0, false);
        if (flush === Z_FULL_FLUSH$1) {
          zero(s2.head);
          if (s2.lookahead === 0) {
            s2.strstart = 0;
            s2.block_start = 0;
            s2.insert = 0;
          }
        }
      }
      flush_pending(strm);
      if (strm.avail_out === 0) {
        s2.last_flush = -1;
        return Z_OK$3;
      }
    }
  }
  if (flush !== Z_FINISH$3) {
    return Z_OK$3;
  }
  if (s2.wrap <= 0) {
    return Z_STREAM_END$3;
  }
  if (s2.wrap === 2) {
    put_byte(s2, strm.adler & 255);
    put_byte(s2, strm.adler >> 8 & 255);
    put_byte(s2, strm.adler >> 16 & 255);
    put_byte(s2, strm.adler >> 24 & 255);
    put_byte(s2, strm.total_in & 255);
    put_byte(s2, strm.total_in >> 8 & 255);
    put_byte(s2, strm.total_in >> 16 & 255);
    put_byte(s2, strm.total_in >> 24 & 255);
  } else {
    putShortMSB(s2, strm.adler >>> 16);
    putShortMSB(s2, strm.adler & 65535);
  }
  flush_pending(strm);
  if (s2.wrap > 0) {
    s2.wrap = -s2.wrap;
  }
  return s2.pending !== 0 ? Z_OK$3 : Z_STREAM_END$3;
};
const deflateEnd = (strm) => {
  if (deflateStateCheck(strm)) {
    return Z_STREAM_ERROR$2;
  }
  const status = strm.state.status;
  strm.state = null;
  return status === BUSY_STATE ? err$1(strm, Z_DATA_ERROR$2) : Z_OK$3;
};
const deflateSetDictionary = (strm, dictionary) => {
  let dictLength = dictionary.length;
  if (deflateStateCheck(strm)) {
    return Z_STREAM_ERROR$2;
  }
  const s2 = strm.state;
  const wrap = s2.wrap;
  if (wrap === 2 || wrap === 1 && s2.status !== INIT_STATE || s2.lookahead) {
    return Z_STREAM_ERROR$2;
  }
  if (wrap === 1) {
    strm.adler = adler32_1(strm.adler, dictionary, dictLength, 0);
  }
  s2.wrap = 0;
  if (dictLength >= s2.w_size) {
    if (wrap === 0) {
      zero(s2.head);
      s2.strstart = 0;
      s2.block_start = 0;
      s2.insert = 0;
    }
    let tmpDict = new Uint8Array(s2.w_size);
    tmpDict.set(dictionary.subarray(dictLength - s2.w_size, dictLength), 0);
    dictionary = tmpDict;
    dictLength = s2.w_size;
  }
  const avail = strm.avail_in;
  const next = strm.next_in;
  const input = strm.input;
  strm.avail_in = dictLength;
  strm.next_in = 0;
  strm.input = dictionary;
  fill_window(s2);
  while (s2.lookahead >= MIN_MATCH) {
    let str = s2.strstart;
    let n2 = s2.lookahead - (MIN_MATCH - 1);
    do {
      s2.ins_h = HASH(s2, s2.ins_h, s2.window[str + MIN_MATCH - 1]);
      s2.prev[str & s2.w_mask] = s2.head[s2.ins_h];
      s2.head[s2.ins_h] = str;
      str++;
    } while (--n2);
    s2.strstart = str;
    s2.lookahead = MIN_MATCH - 1;
    fill_window(s2);
  }
  s2.strstart += s2.lookahead;
  s2.block_start = s2.strstart;
  s2.insert = s2.lookahead;
  s2.lookahead = 0;
  s2.match_length = s2.prev_length = MIN_MATCH - 1;
  s2.match_available = 0;
  strm.next_in = next;
  strm.input = input;
  strm.avail_in = avail;
  s2.wrap = wrap;
  return Z_OK$3;
};
var deflateInit_1 = deflateInit;
var deflateInit2_1 = deflateInit2;
var deflateReset_1 = deflateReset;
var deflateResetKeep_1 = deflateResetKeep;
var deflateSetHeader_1 = deflateSetHeader;
var deflate_2$1 = deflate$2;
var deflateEnd_1 = deflateEnd;
var deflateSetDictionary_1 = deflateSetDictionary;
var deflateInfo = "pako deflate (from Nodeca project)";
var deflate_1$2 = {
  deflateInit: deflateInit_1,
  deflateInit2: deflateInit2_1,
  deflateReset: deflateReset_1,
  deflateResetKeep: deflateResetKeep_1,
  deflateSetHeader: deflateSetHeader_1,
  deflate: deflate_2$1,
  deflateEnd: deflateEnd_1,
  deflateSetDictionary: deflateSetDictionary_1,
  deflateInfo
};
const _has = (obj, key) => {
  return Object.prototype.hasOwnProperty.call(obj, key);
};
var assign$1 = function(obj) {
  const sources = Array.prototype.slice.call(arguments, 1);
  while (sources.length) {
    const source = sources.shift();
    if (!source) {
      continue;
    }
    if (typeof source !== "object") {
      throw new TypeError(source + "must be non-object");
    }
    for (const p2 in source) {
      if (_has(source, p2)) {
        obj[p2] = source[p2];
      }
    }
  }
  return obj;
};
var flattenChunks = (chunks) => {
  let len2 = 0;
  for (let i2 = 0, l2 = chunks.length; i2 < l2; i2++) {
    len2 += chunks[i2].length;
  }
  const result = new Uint8Array(len2);
  for (let i2 = 0, pos = 0, l2 = chunks.length; i2 < l2; i2++) {
    let chunk = chunks[i2];
    result.set(chunk, pos);
    pos += chunk.length;
  }
  return result;
};
var common = {
  assign: assign$1,
  flattenChunks
};
let STR_APPLY_UIA_OK = true;
try {
  String.fromCharCode.apply(null, new Uint8Array(1));
} catch (__) {
  STR_APPLY_UIA_OK = false;
}
const _utf8len = new Uint8Array(256);
for (let q2 = 0; q2 < 256; q2++) {
  _utf8len[q2] = q2 >= 252 ? 6 : q2 >= 248 ? 5 : q2 >= 240 ? 4 : q2 >= 224 ? 3 : q2 >= 192 ? 2 : 1;
}
_utf8len[254] = _utf8len[254] = 1;
var string2buf = (str) => {
  if (typeof TextEncoder === "function" && TextEncoder.prototype.encode) {
    return new TextEncoder().encode(str);
  }
  let buf, c2, c22, m_pos, i2, str_len = str.length, buf_len = 0;
  for (m_pos = 0; m_pos < str_len; m_pos++) {
    c2 = str.charCodeAt(m_pos);
    if ((c2 & 64512) === 55296 && m_pos + 1 < str_len) {
      c22 = str.charCodeAt(m_pos + 1);
      if ((c22 & 64512) === 56320) {
        c2 = 65536 + (c2 - 55296 << 10) + (c22 - 56320);
        m_pos++;
      }
    }
    buf_len += c2 < 128 ? 1 : c2 < 2048 ? 2 : c2 < 65536 ? 3 : 4;
  }
  buf = new Uint8Array(buf_len);
  for (i2 = 0, m_pos = 0; i2 < buf_len; m_pos++) {
    c2 = str.charCodeAt(m_pos);
    if ((c2 & 64512) === 55296 && m_pos + 1 < str_len) {
      c22 = str.charCodeAt(m_pos + 1);
      if ((c22 & 64512) === 56320) {
        c2 = 65536 + (c2 - 55296 << 10) + (c22 - 56320);
        m_pos++;
      }
    }
    if (c2 < 128) {
      buf[i2++] = c2;
    } else if (c2 < 2048) {
      buf[i2++] = 192 | c2 >>> 6;
      buf[i2++] = 128 | c2 & 63;
    } else if (c2 < 65536) {
      buf[i2++] = 224 | c2 >>> 12;
      buf[i2++] = 128 | c2 >>> 6 & 63;
      buf[i2++] = 128 | c2 & 63;
    } else {
      buf[i2++] = 240 | c2 >>> 18;
      buf[i2++] = 128 | c2 >>> 12 & 63;
      buf[i2++] = 128 | c2 >>> 6 & 63;
      buf[i2++] = 128 | c2 & 63;
    }
  }
  return buf;
};
const buf2binstring = (buf, len2) => {
  if (len2 < 65534) {
    if (buf.subarray && STR_APPLY_UIA_OK) {
      return String.fromCharCode.apply(null, buf.length === len2 ? buf : buf.subarray(0, len2));
    }
  }
  let result = "";
  for (let i2 = 0; i2 < len2; i2++) {
    result += String.fromCharCode(buf[i2]);
  }
  return result;
};
var buf2string = (buf, max3) => {
  const len2 = max3 || buf.length;
  if (typeof TextDecoder === "function" && TextDecoder.prototype.decode) {
    return new TextDecoder().decode(buf.subarray(0, max3));
  }
  let i2, out2;
  const utf16buf = new Array(len2 * 2);
  for (out2 = 0, i2 = 0; i2 < len2; ) {
    let c2 = buf[i2++];
    if (c2 < 128) {
      utf16buf[out2++] = c2;
      continue;
    }
    let c_len = _utf8len[c2];
    if (c_len > 4) {
      utf16buf[out2++] = 65533;
      i2 += c_len - 1;
      continue;
    }
    c2 &= c_len === 2 ? 31 : c_len === 3 ? 15 : 7;
    while (c_len > 1 && i2 < len2) {
      c2 = c2 << 6 | buf[i2++] & 63;
      c_len--;
    }
    if (c_len > 1) {
      utf16buf[out2++] = 65533;
      continue;
    }
    if (c2 < 65536) {
      utf16buf[out2++] = c2;
    } else {
      c2 -= 65536;
      utf16buf[out2++] = 55296 | c2 >> 10 & 1023;
      utf16buf[out2++] = 56320 | c2 & 1023;
    }
  }
  return buf2binstring(utf16buf, out2);
};
var utf8border = (buf, max3) => {
  max3 = max3 || buf.length;
  if (max3 > buf.length) {
    max3 = buf.length;
  }
  let pos = max3 - 1;
  while (pos >= 0 && (buf[pos] & 192) === 128) {
    pos--;
  }
  if (pos < 0) {
    return max3;
  }
  if (pos === 0) {
    return max3;
  }
  return pos + _utf8len[buf[pos]] > max3 ? pos : max3;
};
var strings = {
  string2buf,
  buf2string,
  utf8border
};
function ZStream() {
  this.input = null;
  this.next_in = 0;
  this.avail_in = 0;
  this.total_in = 0;
  this.output = null;
  this.next_out = 0;
  this.avail_out = 0;
  this.total_out = 0;
  this.msg = "";
  this.state = null;
  this.data_type = 2;
  this.adler = 0;
}
var zstream = ZStream;
const toString$1$1 = Object.prototype.toString;
const {
  Z_NO_FLUSH: Z_NO_FLUSH$1,
  Z_SYNC_FLUSH,
  Z_FULL_FLUSH,
  Z_FINISH: Z_FINISH$2,
  Z_OK: Z_OK$2,
  Z_STREAM_END: Z_STREAM_END$2,
  Z_DEFAULT_COMPRESSION,
  Z_DEFAULT_STRATEGY,
  Z_DEFLATED: Z_DEFLATED$1
} = constants$2$1;
function Deflate$1(options) {
  this.options = common.assign({
    level: Z_DEFAULT_COMPRESSION,
    method: Z_DEFLATED$1,
    chunkSize: 16384,
    windowBits: 15,
    memLevel: 8,
    strategy: Z_DEFAULT_STRATEGY
  }, options || {});
  let opt = this.options;
  if (opt.raw && opt.windowBits > 0) {
    opt.windowBits = -opt.windowBits;
  } else if (opt.gzip && opt.windowBits > 0 && opt.windowBits < 16) {
    opt.windowBits += 16;
  }
  this.err = 0;
  this.msg = "";
  this.ended = false;
  this.chunks = [];
  this.strm = new zstream();
  this.strm.avail_out = 0;
  let status = deflate_1$2.deflateInit2(
    this.strm,
    opt.level,
    opt.method,
    opt.windowBits,
    opt.memLevel,
    opt.strategy
  );
  if (status !== Z_OK$2) {
    throw new Error(messages[status]);
  }
  if (opt.header) {
    deflate_1$2.deflateSetHeader(this.strm, opt.header);
  }
  if (opt.dictionary) {
    let dict;
    if (typeof opt.dictionary === "string") {
      dict = strings.string2buf(opt.dictionary);
    } else if (toString$1$1.call(opt.dictionary) === "[object ArrayBuffer]") {
      dict = new Uint8Array(opt.dictionary);
    } else {
      dict = opt.dictionary;
    }
    status = deflate_1$2.deflateSetDictionary(this.strm, dict);
    if (status !== Z_OK$2) {
      throw new Error(messages[status]);
    }
    this._dict_set = true;
  }
}
Deflate$1.prototype.push = function(data2, flush_mode) {
  const strm = this.strm;
  const chunkSize = this.options.chunkSize;
  let status, _flush_mode;
  if (this.ended) {
    return false;
  }
  if (flush_mode === ~~flush_mode)
    _flush_mode = flush_mode;
  else
    _flush_mode = flush_mode === true ? Z_FINISH$2 : Z_NO_FLUSH$1;
  if (typeof data2 === "string") {
    strm.input = strings.string2buf(data2);
  } else if (toString$1$1.call(data2) === "[object ArrayBuffer]") {
    strm.input = new Uint8Array(data2);
  } else {
    strm.input = data2;
  }
  strm.next_in = 0;
  strm.avail_in = strm.input.length;
  for (; ; ) {
    if (strm.avail_out === 0) {
      strm.output = new Uint8Array(chunkSize);
      strm.next_out = 0;
      strm.avail_out = chunkSize;
    }
    if ((_flush_mode === Z_SYNC_FLUSH || _flush_mode === Z_FULL_FLUSH) && strm.avail_out <= 6) {
      this.onData(strm.output.subarray(0, strm.next_out));
      strm.avail_out = 0;
      continue;
    }
    status = deflate_1$2.deflate(strm, _flush_mode);
    if (status === Z_STREAM_END$2) {
      if (strm.next_out > 0) {
        this.onData(strm.output.subarray(0, strm.next_out));
      }
      status = deflate_1$2.deflateEnd(this.strm);
      this.onEnd(status);
      this.ended = true;
      return status === Z_OK$2;
    }
    if (strm.avail_out === 0) {
      this.onData(strm.output);
      continue;
    }
    if (_flush_mode > 0 && strm.next_out > 0) {
      this.onData(strm.output.subarray(0, strm.next_out));
      strm.avail_out = 0;
      continue;
    }
    if (strm.avail_in === 0)
      break;
  }
  return true;
};
Deflate$1.prototype.onData = function(chunk) {
  this.chunks.push(chunk);
};
Deflate$1.prototype.onEnd = function(status) {
  if (status === Z_OK$2) {
    this.result = common.flattenChunks(this.chunks);
  }
  this.chunks = [];
  this.err = status;
  this.msg = this.strm.msg;
};
function deflate$1(input, options) {
  const deflator = new Deflate$1(options);
  deflator.push(input, true);
  if (deflator.err) {
    throw deflator.msg || messages[deflator.err];
  }
  return deflator.result;
}
function deflateRaw$1(input, options) {
  options = options || {};
  options.raw = true;
  return deflate$1(input, options);
}
function gzip$1(input, options) {
  options = options || {};
  options.gzip = true;
  return deflate$1(input, options);
}
var Deflate_1$1 = Deflate$1;
var deflate_2 = deflate$1;
var deflateRaw_1$1 = deflateRaw$1;
var gzip_1$1 = gzip$1;
var constants$1$1 = constants$2$1;
var deflate_1$1 = {
  Deflate: Deflate_1$1,
  deflate: deflate_2,
  deflateRaw: deflateRaw_1$1,
  gzip: gzip_1$1,
  constants: constants$1$1
};
const BAD$1 = 16209;
const TYPE$1 = 16191;
var inffast = function inflate_fast(strm, start2) {
  let _in;
  let last;
  let _out;
  let beg;
  let end2;
  let dmax;
  let wsize;
  let whave;
  let wnext;
  let s_window;
  let hold;
  let bits2;
  let lcode;
  let dcode;
  let lmask;
  let dmask;
  let here;
  let op2;
  let len2;
  let dist2;
  let from2;
  let from_source;
  let input, output2;
  const state = strm.state;
  _in = strm.next_in;
  input = strm.input;
  last = _in + (strm.avail_in - 5);
  _out = strm.next_out;
  output2 = strm.output;
  beg = _out - (start2 - strm.avail_out);
  end2 = _out + (strm.avail_out - 257);
  dmax = state.dmax;
  wsize = state.wsize;
  whave = state.whave;
  wnext = state.wnext;
  s_window = state.window;
  hold = state.hold;
  bits2 = state.bits;
  lcode = state.lencode;
  dcode = state.distcode;
  lmask = (1 << state.lenbits) - 1;
  dmask = (1 << state.distbits) - 1;
  top:
    do {
      if (bits2 < 15) {
        hold += input[_in++] << bits2;
        bits2 += 8;
        hold += input[_in++] << bits2;
        bits2 += 8;
      }
      here = lcode[hold & lmask];
      dolen:
        for (; ; ) {
          op2 = here >>> 24;
          hold >>>= op2;
          bits2 -= op2;
          op2 = here >>> 16 & 255;
          if (op2 === 0) {
            output2[_out++] = here & 65535;
          } else if (op2 & 16) {
            len2 = here & 65535;
            op2 &= 15;
            if (op2) {
              if (bits2 < op2) {
                hold += input[_in++] << bits2;
                bits2 += 8;
              }
              len2 += hold & (1 << op2) - 1;
              hold >>>= op2;
              bits2 -= op2;
            }
            if (bits2 < 15) {
              hold += input[_in++] << bits2;
              bits2 += 8;
              hold += input[_in++] << bits2;
              bits2 += 8;
            }
            here = dcode[hold & dmask];
            dodist:
              for (; ; ) {
                op2 = here >>> 24;
                hold >>>= op2;
                bits2 -= op2;
                op2 = here >>> 16 & 255;
                if (op2 & 16) {
                  dist2 = here & 65535;
                  op2 &= 15;
                  if (bits2 < op2) {
                    hold += input[_in++] << bits2;
                    bits2 += 8;
                    if (bits2 < op2) {
                      hold += input[_in++] << bits2;
                      bits2 += 8;
                    }
                  }
                  dist2 += hold & (1 << op2) - 1;
                  if (dist2 > dmax) {
                    strm.msg = "invalid distance too far back";
                    state.mode = BAD$1;
                    break top;
                  }
                  hold >>>= op2;
                  bits2 -= op2;
                  op2 = _out - beg;
                  if (dist2 > op2) {
                    op2 = dist2 - op2;
                    if (op2 > whave) {
                      if (state.sane) {
                        strm.msg = "invalid distance too far back";
                        state.mode = BAD$1;
                        break top;
                      }
                    }
                    from2 = 0;
                    from_source = s_window;
                    if (wnext === 0) {
                      from2 += wsize - op2;
                      if (op2 < len2) {
                        len2 -= op2;
                        do {
                          output2[_out++] = s_window[from2++];
                        } while (--op2);
                        from2 = _out - dist2;
                        from_source = output2;
                      }
                    } else if (wnext < op2) {
                      from2 += wsize + wnext - op2;
                      op2 -= wnext;
                      if (op2 < len2) {
                        len2 -= op2;
                        do {
                          output2[_out++] = s_window[from2++];
                        } while (--op2);
                        from2 = 0;
                        if (wnext < len2) {
                          op2 = wnext;
                          len2 -= op2;
                          do {
                            output2[_out++] = s_window[from2++];
                          } while (--op2);
                          from2 = _out - dist2;
                          from_source = output2;
                        }
                      }
                    } else {
                      from2 += wnext - op2;
                      if (op2 < len2) {
                        len2 -= op2;
                        do {
                          output2[_out++] = s_window[from2++];
                        } while (--op2);
                        from2 = _out - dist2;
                        from_source = output2;
                      }
                    }
                    while (len2 > 2) {
                      output2[_out++] = from_source[from2++];
                      output2[_out++] = from_source[from2++];
                      output2[_out++] = from_source[from2++];
                      len2 -= 3;
                    }
                    if (len2) {
                      output2[_out++] = from_source[from2++];
                      if (len2 > 1) {
                        output2[_out++] = from_source[from2++];
                      }
                    }
                  } else {
                    from2 = _out - dist2;
                    do {
                      output2[_out++] = output2[from2++];
                      output2[_out++] = output2[from2++];
                      output2[_out++] = output2[from2++];
                      len2 -= 3;
                    } while (len2 > 2);
                    if (len2) {
                      output2[_out++] = output2[from2++];
                      if (len2 > 1) {
                        output2[_out++] = output2[from2++];
                      }
                    }
                  }
                } else if ((op2 & 64) === 0) {
                  here = dcode[(here & 65535) + (hold & (1 << op2) - 1)];
                  continue dodist;
                } else {
                  strm.msg = "invalid distance code";
                  state.mode = BAD$1;
                  break top;
                }
                break;
              }
          } else if ((op2 & 64) === 0) {
            here = lcode[(here & 65535) + (hold & (1 << op2) - 1)];
            continue dolen;
          } else if (op2 & 32) {
            state.mode = TYPE$1;
            break top;
          } else {
            strm.msg = "invalid literal/length code";
            state.mode = BAD$1;
            break top;
          }
          break;
        }
    } while (_in < last && _out < end2);
  len2 = bits2 >> 3;
  _in -= len2;
  bits2 -= len2 << 3;
  hold &= (1 << bits2) - 1;
  strm.next_in = _in;
  strm.next_out = _out;
  strm.avail_in = _in < last ? 5 + (last - _in) : 5 - (_in - last);
  strm.avail_out = _out < end2 ? 257 + (end2 - _out) : 257 - (_out - end2);
  state.hold = hold;
  state.bits = bits2;
  return;
};
const MAXBITS = 15;
const ENOUGH_LENS$1 = 852;
const ENOUGH_DISTS$1 = 592;
const CODES$1 = 0;
const LENS$1 = 1;
const DISTS$1 = 2;
const lbase = new Uint16Array([
  /* Length codes 257..285 base */
  3,
  4,
  5,
  6,
  7,
  8,
  9,
  10,
  11,
  13,
  15,
  17,
  19,
  23,
  27,
  31,
  35,
  43,
  51,
  59,
  67,
  83,
  99,
  115,
  131,
  163,
  195,
  227,
  258,
  0,
  0
]);
const lext = new Uint8Array([
  /* Length codes 257..285 extra */
  16,
  16,
  16,
  16,
  16,
  16,
  16,
  16,
  17,
  17,
  17,
  17,
  18,
  18,
  18,
  18,
  19,
  19,
  19,
  19,
  20,
  20,
  20,
  20,
  21,
  21,
  21,
  21,
  16,
  72,
  78
]);
const dbase = new Uint16Array([
  /* Distance codes 0..29 base */
  1,
  2,
  3,
  4,
  5,
  7,
  9,
  13,
  17,
  25,
  33,
  49,
  65,
  97,
  129,
  193,
  257,
  385,
  513,
  769,
  1025,
  1537,
  2049,
  3073,
  4097,
  6145,
  8193,
  12289,
  16385,
  24577,
  0,
  0
]);
const dext = new Uint8Array([
  /* Distance codes 0..29 extra */
  16,
  16,
  16,
  16,
  17,
  17,
  18,
  18,
  19,
  19,
  20,
  20,
  21,
  21,
  22,
  22,
  23,
  23,
  24,
  24,
  25,
  25,
  26,
  26,
  27,
  27,
  28,
  28,
  29,
  29,
  64,
  64
]);
const inflate_table = (type2, lens, lens_index, codes, table, table_index, work, opts) => {
  const bits2 = opts.bits;
  let len2 = 0;
  let sym = 0;
  let min3 = 0, max3 = 0;
  let root2 = 0;
  let curr = 0;
  let drop = 0;
  let left = 0;
  let used = 0;
  let huff = 0;
  let incr;
  let fill2;
  let low;
  let mask;
  let next;
  let base2 = null;
  let match5;
  const count2 = new Uint16Array(MAXBITS + 1);
  const offs = new Uint16Array(MAXBITS + 1);
  let extra = null;
  let here_bits, here_op, here_val;
  for (len2 = 0; len2 <= MAXBITS; len2++) {
    count2[len2] = 0;
  }
  for (sym = 0; sym < codes; sym++) {
    count2[lens[lens_index + sym]]++;
  }
  root2 = bits2;
  for (max3 = MAXBITS; max3 >= 1; max3--) {
    if (count2[max3] !== 0) {
      break;
    }
  }
  if (root2 > max3) {
    root2 = max3;
  }
  if (max3 === 0) {
    table[table_index++] = 1 << 24 | 64 << 16 | 0;
    table[table_index++] = 1 << 24 | 64 << 16 | 0;
    opts.bits = 1;
    return 0;
  }
  for (min3 = 1; min3 < max3; min3++) {
    if (count2[min3] !== 0) {
      break;
    }
  }
  if (root2 < min3) {
    root2 = min3;
  }
  left = 1;
  for (len2 = 1; len2 <= MAXBITS; len2++) {
    left <<= 1;
    left -= count2[len2];
    if (left < 0) {
      return -1;
    }
  }
  if (left > 0 && (type2 === CODES$1 || max3 !== 1)) {
    return -1;
  }
  offs[1] = 0;
  for (len2 = 1; len2 < MAXBITS; len2++) {
    offs[len2 + 1] = offs[len2] + count2[len2];
  }
  for (sym = 0; sym < codes; sym++) {
    if (lens[lens_index + sym] !== 0) {
      work[offs[lens[lens_index + sym]]++] = sym;
    }
  }
  if (type2 === CODES$1) {
    base2 = extra = work;
    match5 = 20;
  } else if (type2 === LENS$1) {
    base2 = lbase;
    extra = lext;
    match5 = 257;
  } else {
    base2 = dbase;
    extra = dext;
    match5 = 0;
  }
  huff = 0;
  sym = 0;
  len2 = min3;
  next = table_index;
  curr = root2;
  drop = 0;
  low = -1;
  used = 1 << root2;
  mask = used - 1;
  if (type2 === LENS$1 && used > ENOUGH_LENS$1 || type2 === DISTS$1 && used > ENOUGH_DISTS$1) {
    return 1;
  }
  for (; ; ) {
    here_bits = len2 - drop;
    if (work[sym] + 1 < match5) {
      here_op = 0;
      here_val = work[sym];
    } else if (work[sym] >= match5) {
      here_op = extra[work[sym] - match5];
      here_val = base2[work[sym] - match5];
    } else {
      here_op = 32 + 64;
      here_val = 0;
    }
    incr = 1 << len2 - drop;
    fill2 = 1 << curr;
    min3 = fill2;
    do {
      fill2 -= incr;
      table[next + (huff >> drop) + fill2] = here_bits << 24 | here_op << 16 | here_val | 0;
    } while (fill2 !== 0);
    incr = 1 << len2 - 1;
    while (huff & incr) {
      incr >>= 1;
    }
    if (incr !== 0) {
      huff &= incr - 1;
      huff += incr;
    } else {
      huff = 0;
    }
    sym++;
    if (--count2[len2] === 0) {
      if (len2 === max3) {
        break;
      }
      len2 = lens[lens_index + work[sym]];
    }
    if (len2 > root2 && (huff & mask) !== low) {
      if (drop === 0) {
        drop = root2;
      }
      next += min3;
      curr = len2 - drop;
      left = 1 << curr;
      while (curr + drop < max3) {
        left -= count2[curr + drop];
        if (left <= 0) {
          break;
        }
        curr++;
        left <<= 1;
      }
      used += 1 << curr;
      if (type2 === LENS$1 && used > ENOUGH_LENS$1 || type2 === DISTS$1 && used > ENOUGH_DISTS$1) {
        return 1;
      }
      low = huff & mask;
      table[low] = root2 << 24 | curr << 16 | next - table_index | 0;
    }
  }
  if (huff !== 0) {
    table[next + huff] = len2 - drop << 24 | 64 << 16 | 0;
  }
  opts.bits = root2;
  return 0;
};
var inftrees = inflate_table;
const CODES = 0;
const LENS = 1;
const DISTS = 2;
const {
  Z_FINISH: Z_FINISH$1,
  Z_BLOCK,
  Z_TREES,
  Z_OK: Z_OK$1,
  Z_STREAM_END: Z_STREAM_END$1,
  Z_NEED_DICT: Z_NEED_DICT$1,
  Z_STREAM_ERROR: Z_STREAM_ERROR$1,
  Z_DATA_ERROR: Z_DATA_ERROR$1,
  Z_MEM_ERROR: Z_MEM_ERROR$1,
  Z_BUF_ERROR,
  Z_DEFLATED
} = constants$2$1;
const HEAD = 16180;
const FLAGS = 16181;
const TIME = 16182;
const OS = 16183;
const EXLEN = 16184;
const EXTRA = 16185;
const NAME = 16186;
const COMMENT = 16187;
const HCRC = 16188;
const DICTID = 16189;
const DICT = 16190;
const TYPE = 16191;
const TYPEDO = 16192;
const STORED = 16193;
const COPY_ = 16194;
const COPY = 16195;
const TABLE = 16196;
const LENLENS = 16197;
const CODELENS = 16198;
const LEN_ = 16199;
const LEN = 16200;
const LENEXT = 16201;
const DIST = 16202;
const DISTEXT = 16203;
const MATCH$2 = 16204;
const LIT = 16205;
const CHECK = 16206;
const LENGTH = 16207;
const DONE = 16208;
const BAD = 16209;
const MEM = 16210;
const SYNC = 16211;
const ENOUGH_LENS = 852;
const ENOUGH_DISTS = 592;
const MAX_WBITS = 15;
const DEF_WBITS = MAX_WBITS;
const zswap32 = (q2) => {
  return (q2 >>> 24 & 255) + (q2 >>> 8 & 65280) + ((q2 & 65280) << 8) + ((q2 & 255) << 24);
};
function InflateState() {
  this.strm = null;
  this.mode = 0;
  this.last = false;
  this.wrap = 0;
  this.havedict = false;
  this.flags = 0;
  this.dmax = 0;
  this.check = 0;
  this.total = 0;
  this.head = null;
  this.wbits = 0;
  this.wsize = 0;
  this.whave = 0;
  this.wnext = 0;
  this.window = null;
  this.hold = 0;
  this.bits = 0;
  this.length = 0;
  this.offset = 0;
  this.extra = 0;
  this.lencode = null;
  this.distcode = null;
  this.lenbits = 0;
  this.distbits = 0;
  this.ncode = 0;
  this.nlen = 0;
  this.ndist = 0;
  this.have = 0;
  this.next = null;
  this.lens = new Uint16Array(320);
  this.work = new Uint16Array(288);
  this.lendyn = null;
  this.distdyn = null;
  this.sane = 0;
  this.back = 0;
  this.was = 0;
}
const inflateStateCheck = (strm) => {
  if (!strm) {
    return 1;
  }
  const state = strm.state;
  if (!state || state.strm !== strm || state.mode < HEAD || state.mode > SYNC) {
    return 1;
  }
  return 0;
};
const inflateResetKeep = (strm) => {
  if (inflateStateCheck(strm)) {
    return Z_STREAM_ERROR$1;
  }
  const state = strm.state;
  strm.total_in = strm.total_out = state.total = 0;
  strm.msg = "";
  if (state.wrap) {
    strm.adler = state.wrap & 1;
  }
  state.mode = HEAD;
  state.last = 0;
  state.havedict = 0;
  state.flags = -1;
  state.dmax = 32768;
  state.head = null;
  state.hold = 0;
  state.bits = 0;
  state.lencode = state.lendyn = new Int32Array(ENOUGH_LENS);
  state.distcode = state.distdyn = new Int32Array(ENOUGH_DISTS);
  state.sane = 1;
  state.back = -1;
  return Z_OK$1;
};
const inflateReset = (strm) => {
  if (inflateStateCheck(strm)) {
    return Z_STREAM_ERROR$1;
  }
  const state = strm.state;
  state.wsize = 0;
  state.whave = 0;
  state.wnext = 0;
  return inflateResetKeep(strm);
};
const inflateReset2 = (strm, windowBits) => {
  let wrap;
  if (inflateStateCheck(strm)) {
    return Z_STREAM_ERROR$1;
  }
  const state = strm.state;
  if (windowBits < 0) {
    wrap = 0;
    windowBits = -windowBits;
  } else {
    wrap = (windowBits >> 4) + 5;
    if (windowBits < 48) {
      windowBits &= 15;
    }
  }
  if (windowBits && (windowBits < 8 || windowBits > 15)) {
    return Z_STREAM_ERROR$1;
  }
  if (state.window !== null && state.wbits !== windowBits) {
    state.window = null;
  }
  state.wrap = wrap;
  state.wbits = windowBits;
  return inflateReset(strm);
};
const inflateInit2 = (strm, windowBits) => {
  if (!strm) {
    return Z_STREAM_ERROR$1;
  }
  const state = new InflateState();
  strm.state = state;
  state.strm = strm;
  state.window = null;
  state.mode = HEAD;
  const ret = inflateReset2(strm, windowBits);
  if (ret !== Z_OK$1) {
    strm.state = null;
  }
  return ret;
};
const inflateInit = (strm) => {
  return inflateInit2(strm, DEF_WBITS);
};
let virgin = true;
let lenfix, distfix;
const fixedtables = (state) => {
  if (virgin) {
    lenfix = new Int32Array(512);
    distfix = new Int32Array(32);
    let sym = 0;
    while (sym < 144) {
      state.lens[sym++] = 8;
    }
    while (sym < 256) {
      state.lens[sym++] = 9;
    }
    while (sym < 280) {
      state.lens[sym++] = 7;
    }
    while (sym < 288) {
      state.lens[sym++] = 8;
    }
    inftrees(LENS, state.lens, 0, 288, lenfix, 0, state.work, { bits: 9 });
    sym = 0;
    while (sym < 32) {
      state.lens[sym++] = 5;
    }
    inftrees(DISTS, state.lens, 0, 32, distfix, 0, state.work, { bits: 5 });
    virgin = false;
  }
  state.lencode = lenfix;
  state.lenbits = 9;
  state.distcode = distfix;
  state.distbits = 5;
};
const updatewindow = (strm, src, end2, copy2) => {
  let dist2;
  const state = strm.state;
  if (state.window === null) {
    state.wsize = 1 << state.wbits;
    state.wnext = 0;
    state.whave = 0;
    state.window = new Uint8Array(state.wsize);
  }
  if (copy2 >= state.wsize) {
    state.window.set(src.subarray(end2 - state.wsize, end2), 0);
    state.wnext = 0;
    state.whave = state.wsize;
  } else {
    dist2 = state.wsize - state.wnext;
    if (dist2 > copy2) {
      dist2 = copy2;
    }
    state.window.set(src.subarray(end2 - copy2, end2 - copy2 + dist2), state.wnext);
    copy2 -= dist2;
    if (copy2) {
      state.window.set(src.subarray(end2 - copy2, end2), 0);
      state.wnext = copy2;
      state.whave = state.wsize;
    } else {
      state.wnext += dist2;
      if (state.wnext === state.wsize) {
        state.wnext = 0;
      }
      if (state.whave < state.wsize) {
        state.whave += dist2;
      }
    }
  }
  return 0;
};
const inflate$2 = (strm, flush) => {
  let state;
  let input, output2;
  let next;
  let put;
  let have, left;
  let hold;
  let bits2;
  let _in, _out;
  let copy2;
  let from2;
  let from_source;
  let here = 0;
  let here_bits, here_op, here_val;
  let last_bits, last_op, last_val;
  let len2;
  let ret;
  const hbuf = new Uint8Array(4);
  let opts;
  let n2;
  const order = (
    /* permutation of code lengths */
    new Uint8Array([16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15])
  );
  if (inflateStateCheck(strm) || !strm.output || !strm.input && strm.avail_in !== 0) {
    return Z_STREAM_ERROR$1;
  }
  state = strm.state;
  if (state.mode === TYPE) {
    state.mode = TYPEDO;
  }
  put = strm.next_out;
  output2 = strm.output;
  left = strm.avail_out;
  next = strm.next_in;
  input = strm.input;
  have = strm.avail_in;
  hold = state.hold;
  bits2 = state.bits;
  _in = have;
  _out = left;
  ret = Z_OK$1;
  inf_leave:
    for (; ; ) {
      switch (state.mode) {
        case HEAD:
          if (state.wrap === 0) {
            state.mode = TYPEDO;
            break;
          }
          while (bits2 < 16) {
            if (have === 0) {
              break inf_leave;
            }
            have--;
            hold += input[next++] << bits2;
            bits2 += 8;
          }
          if (state.wrap & 2 && hold === 35615) {
            if (state.wbits === 0) {
              state.wbits = 15;
            }
            state.check = 0;
            hbuf[0] = hold & 255;
            hbuf[1] = hold >>> 8 & 255;
            state.check = crc32_1(state.check, hbuf, 2, 0);
            hold = 0;
            bits2 = 0;
            state.mode = FLAGS;
            break;
          }
          if (state.head) {
            state.head.done = false;
          }
          if (!(state.wrap & 1) || /* check if zlib header allowed */
          (((hold & 255) << 8) + (hold >> 8)) % 31) {
            strm.msg = "incorrect header check";
            state.mode = BAD;
            break;
          }
          if ((hold & 15) !== Z_DEFLATED) {
            strm.msg = "unknown compression method";
            state.mode = BAD;
            break;
          }
          hold >>>= 4;
          bits2 -= 4;
          len2 = (hold & 15) + 8;
          if (state.wbits === 0) {
            state.wbits = len2;
          }
          if (len2 > 15 || len2 > state.wbits) {
            strm.msg = "invalid window size";
            state.mode = BAD;
            break;
          }
          state.dmax = 1 << state.wbits;
          state.flags = 0;
          strm.adler = state.check = 1;
          state.mode = hold & 512 ? DICTID : TYPE;
          hold = 0;
          bits2 = 0;
          break;
        case FLAGS:
          while (bits2 < 16) {
            if (have === 0) {
              break inf_leave;
            }
            have--;
            hold += input[next++] << bits2;
            bits2 += 8;
          }
          state.flags = hold;
          if ((state.flags & 255) !== Z_DEFLATED) {
            strm.msg = "unknown compression method";
            state.mode = BAD;
            break;
          }
          if (state.flags & 57344) {
            strm.msg = "unknown header flags set";
            state.mode = BAD;
            break;
          }
          if (state.head) {
            state.head.text = hold >> 8 & 1;
          }
          if (state.flags & 512 && state.wrap & 4) {
            hbuf[0] = hold & 255;
            hbuf[1] = hold >>> 8 & 255;
            state.check = crc32_1(state.check, hbuf, 2, 0);
          }
          hold = 0;
          bits2 = 0;
          state.mode = TIME;
        case TIME:
          while (bits2 < 32) {
            if (have === 0) {
              break inf_leave;
            }
            have--;
            hold += input[next++] << bits2;
            bits2 += 8;
          }
          if (state.head) {
            state.head.time = hold;
          }
          if (state.flags & 512 && state.wrap & 4) {
            hbuf[0] = hold & 255;
            hbuf[1] = hold >>> 8 & 255;
            hbuf[2] = hold >>> 16 & 255;
            hbuf[3] = hold >>> 24 & 255;
            state.check = crc32_1(state.check, hbuf, 4, 0);
          }
          hold = 0;
          bits2 = 0;
          state.mode = OS;
        case OS:
          while (bits2 < 16) {
            if (have === 0) {
              break inf_leave;
            }
            have--;
            hold += input[next++] << bits2;
            bits2 += 8;
          }
          if (state.head) {
            state.head.xflags = hold & 255;
            state.head.os = hold >> 8;
          }
          if (state.flags & 512 && state.wrap & 4) {
            hbuf[0] = hold & 255;
            hbuf[1] = hold >>> 8 & 255;
            state.check = crc32_1(state.check, hbuf, 2, 0);
          }
          hold = 0;
          bits2 = 0;
          state.mode = EXLEN;
        case EXLEN:
          if (state.flags & 1024) {
            while (bits2 < 16) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits2;
              bits2 += 8;
            }
            state.length = hold;
            if (state.head) {
              state.head.extra_len = hold;
            }
            if (state.flags & 512 && state.wrap & 4) {
              hbuf[0] = hold & 255;
              hbuf[1] = hold >>> 8 & 255;
              state.check = crc32_1(state.check, hbuf, 2, 0);
            }
            hold = 0;
            bits2 = 0;
          } else if (state.head) {
            state.head.extra = null;
          }
          state.mode = EXTRA;
        case EXTRA:
          if (state.flags & 1024) {
            copy2 = state.length;
            if (copy2 > have) {
              copy2 = have;
            }
            if (copy2) {
              if (state.head) {
                len2 = state.head.extra_len - state.length;
                if (!state.head.extra) {
                  state.head.extra = new Uint8Array(state.head.extra_len);
                }
                state.head.extra.set(
                  input.subarray(
                    next,
                    // extra field is limited to 65536 bytes
                    // - no need for additional size check
                    next + copy2
                  ),
                  /*len + copy > state.head.extra_max - len ? state.head.extra_max : copy,*/
                  len2
                );
              }
              if (state.flags & 512 && state.wrap & 4) {
                state.check = crc32_1(state.check, input, copy2, next);
              }
              have -= copy2;
              next += copy2;
              state.length -= copy2;
            }
            if (state.length) {
              break inf_leave;
            }
          }
          state.length = 0;
          state.mode = NAME;
        case NAME:
          if (state.flags & 2048) {
            if (have === 0) {
              break inf_leave;
            }
            copy2 = 0;
            do {
              len2 = input[next + copy2++];
              if (state.head && len2 && state.length < 65536) {
                state.head.name += String.fromCharCode(len2);
              }
            } while (len2 && copy2 < have);
            if (state.flags & 512 && state.wrap & 4) {
              state.check = crc32_1(state.check, input, copy2, next);
            }
            have -= copy2;
            next += copy2;
            if (len2) {
              break inf_leave;
            }
          } else if (state.head) {
            state.head.name = null;
          }
          state.length = 0;
          state.mode = COMMENT;
        case COMMENT:
          if (state.flags & 4096) {
            if (have === 0) {
              break inf_leave;
            }
            copy2 = 0;
            do {
              len2 = input[next + copy2++];
              if (state.head && len2 && state.length < 65536) {
                state.head.comment += String.fromCharCode(len2);
              }
            } while (len2 && copy2 < have);
            if (state.flags & 512 && state.wrap & 4) {
              state.check = crc32_1(state.check, input, copy2, next);
            }
            have -= copy2;
            next += copy2;
            if (len2) {
              break inf_leave;
            }
          } else if (state.head) {
            state.head.comment = null;
          }
          state.mode = HCRC;
        case HCRC:
          if (state.flags & 512) {
            while (bits2 < 16) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits2;
              bits2 += 8;
            }
            if (state.wrap & 4 && hold !== (state.check & 65535)) {
              strm.msg = "header crc mismatch";
              state.mode = BAD;
              break;
            }
            hold = 0;
            bits2 = 0;
          }
          if (state.head) {
            state.head.hcrc = state.flags >> 9 & 1;
            state.head.done = true;
          }
          strm.adler = state.check = 0;
          state.mode = TYPE;
          break;
        case DICTID:
          while (bits2 < 32) {
            if (have === 0) {
              break inf_leave;
            }
            have--;
            hold += input[next++] << bits2;
            bits2 += 8;
          }
          strm.adler = state.check = zswap32(hold);
          hold = 0;
          bits2 = 0;
          state.mode = DICT;
        case DICT:
          if (state.havedict === 0) {
            strm.next_out = put;
            strm.avail_out = left;
            strm.next_in = next;
            strm.avail_in = have;
            state.hold = hold;
            state.bits = bits2;
            return Z_NEED_DICT$1;
          }
          strm.adler = state.check = 1;
          state.mode = TYPE;
        case TYPE:
          if (flush === Z_BLOCK || flush === Z_TREES) {
            break inf_leave;
          }
        case TYPEDO:
          if (state.last) {
            hold >>>= bits2 & 7;
            bits2 -= bits2 & 7;
            state.mode = CHECK;
            break;
          }
          while (bits2 < 3) {
            if (have === 0) {
              break inf_leave;
            }
            have--;
            hold += input[next++] << bits2;
            bits2 += 8;
          }
          state.last = hold & 1;
          hold >>>= 1;
          bits2 -= 1;
          switch (hold & 3) {
            case 0:
              state.mode = STORED;
              break;
            case 1:
              fixedtables(state);
              state.mode = LEN_;
              if (flush === Z_TREES) {
                hold >>>= 2;
                bits2 -= 2;
                break inf_leave;
              }
              break;
            case 2:
              state.mode = TABLE;
              break;
            case 3:
              strm.msg = "invalid block type";
              state.mode = BAD;
          }
          hold >>>= 2;
          bits2 -= 2;
          break;
        case STORED:
          hold >>>= bits2 & 7;
          bits2 -= bits2 & 7;
          while (bits2 < 32) {
            if (have === 0) {
              break inf_leave;
            }
            have--;
            hold += input[next++] << bits2;
            bits2 += 8;
          }
          if ((hold & 65535) !== (hold >>> 16 ^ 65535)) {
            strm.msg = "invalid stored block lengths";
            state.mode = BAD;
            break;
          }
          state.length = hold & 65535;
          hold = 0;
          bits2 = 0;
          state.mode = COPY_;
          if (flush === Z_TREES) {
            break inf_leave;
          }
        case COPY_:
          state.mode = COPY;
        case COPY:
          copy2 = state.length;
          if (copy2) {
            if (copy2 > have) {
              copy2 = have;
            }
            if (copy2 > left) {
              copy2 = left;
            }
            if (copy2 === 0) {
              break inf_leave;
            }
            output2.set(input.subarray(next, next + copy2), put);
            have -= copy2;
            next += copy2;
            left -= copy2;
            put += copy2;
            state.length -= copy2;
            break;
          }
          state.mode = TYPE;
          break;
        case TABLE:
          while (bits2 < 14) {
            if (have === 0) {
              break inf_leave;
            }
            have--;
            hold += input[next++] << bits2;
            bits2 += 8;
          }
          state.nlen = (hold & 31) + 257;
          hold >>>= 5;
          bits2 -= 5;
          state.ndist = (hold & 31) + 1;
          hold >>>= 5;
          bits2 -= 5;
          state.ncode = (hold & 15) + 4;
          hold >>>= 4;
          bits2 -= 4;
          if (state.nlen > 286 || state.ndist > 30) {
            strm.msg = "too many length or distance symbols";
            state.mode = BAD;
            break;
          }
          state.have = 0;
          state.mode = LENLENS;
        case LENLENS:
          while (state.have < state.ncode) {
            while (bits2 < 3) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits2;
              bits2 += 8;
            }
            state.lens[order[state.have++]] = hold & 7;
            hold >>>= 3;
            bits2 -= 3;
          }
          while (state.have < 19) {
            state.lens[order[state.have++]] = 0;
          }
          state.lencode = state.lendyn;
          state.lenbits = 7;
          opts = { bits: state.lenbits };
          ret = inftrees(CODES, state.lens, 0, 19, state.lencode, 0, state.work, opts);
          state.lenbits = opts.bits;
          if (ret) {
            strm.msg = "invalid code lengths set";
            state.mode = BAD;
            break;
          }
          state.have = 0;
          state.mode = CODELENS;
        case CODELENS:
          while (state.have < state.nlen + state.ndist) {
            for (; ; ) {
              here = state.lencode[hold & (1 << state.lenbits) - 1];
              here_bits = here >>> 24;
              here_op = here >>> 16 & 255;
              here_val = here & 65535;
              if (here_bits <= bits2) {
                break;
              }
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits2;
              bits2 += 8;
            }
            if (here_val < 16) {
              hold >>>= here_bits;
              bits2 -= here_bits;
              state.lens[state.have++] = here_val;
            } else {
              if (here_val === 16) {
                n2 = here_bits + 2;
                while (bits2 < n2) {
                  if (have === 0) {
                    break inf_leave;
                  }
                  have--;
                  hold += input[next++] << bits2;
                  bits2 += 8;
                }
                hold >>>= here_bits;
                bits2 -= here_bits;
                if (state.have === 0) {
                  strm.msg = "invalid bit length repeat";
                  state.mode = BAD;
                  break;
                }
                len2 = state.lens[state.have - 1];
                copy2 = 3 + (hold & 3);
                hold >>>= 2;
                bits2 -= 2;
              } else if (here_val === 17) {
                n2 = here_bits + 3;
                while (bits2 < n2) {
                  if (have === 0) {
                    break inf_leave;
                  }
                  have--;
                  hold += input[next++] << bits2;
                  bits2 += 8;
                }
                hold >>>= here_bits;
                bits2 -= here_bits;
                len2 = 0;
                copy2 = 3 + (hold & 7);
                hold >>>= 3;
                bits2 -= 3;
              } else {
                n2 = here_bits + 7;
                while (bits2 < n2) {
                  if (have === 0) {
                    break inf_leave;
                  }
                  have--;
                  hold += input[next++] << bits2;
                  bits2 += 8;
                }
                hold >>>= here_bits;
                bits2 -= here_bits;
                len2 = 0;
                copy2 = 11 + (hold & 127);
                hold >>>= 7;
                bits2 -= 7;
              }
              if (state.have + copy2 > state.nlen + state.ndist) {
                strm.msg = "invalid bit length repeat";
                state.mode = BAD;
                break;
              }
              while (copy2--) {
                state.lens[state.have++] = len2;
              }
            }
          }
          if (state.mode === BAD) {
            break;
          }
          if (state.lens[256] === 0) {
            strm.msg = "invalid code -- missing end-of-block";
            state.mode = BAD;
            break;
          }
          state.lenbits = 9;
          opts = { bits: state.lenbits };
          ret = inftrees(LENS, state.lens, 0, state.nlen, state.lencode, 0, state.work, opts);
          state.lenbits = opts.bits;
          if (ret) {
            strm.msg = "invalid literal/lengths set";
            state.mode = BAD;
            break;
          }
          state.distbits = 6;
          state.distcode = state.distdyn;
          opts = { bits: state.distbits };
          ret = inftrees(DISTS, state.lens, state.nlen, state.ndist, state.distcode, 0, state.work, opts);
          state.distbits = opts.bits;
          if (ret) {
            strm.msg = "invalid distances set";
            state.mode = BAD;
            break;
          }
          state.mode = LEN_;
          if (flush === Z_TREES) {
            break inf_leave;
          }
        case LEN_:
          state.mode = LEN;
        case LEN:
          if (have >= 6 && left >= 258) {
            strm.next_out = put;
            strm.avail_out = left;
            strm.next_in = next;
            strm.avail_in = have;
            state.hold = hold;
            state.bits = bits2;
            inffast(strm, _out);
            put = strm.next_out;
            output2 = strm.output;
            left = strm.avail_out;
            next = strm.next_in;
            input = strm.input;
            have = strm.avail_in;
            hold = state.hold;
            bits2 = state.bits;
            if (state.mode === TYPE) {
              state.back = -1;
            }
            break;
          }
          state.back = 0;
          for (; ; ) {
            here = state.lencode[hold & (1 << state.lenbits) - 1];
            here_bits = here >>> 24;
            here_op = here >>> 16 & 255;
            here_val = here & 65535;
            if (here_bits <= bits2) {
              break;
            }
            if (have === 0) {
              break inf_leave;
            }
            have--;
            hold += input[next++] << bits2;
            bits2 += 8;
          }
          if (here_op && (here_op & 240) === 0) {
            last_bits = here_bits;
            last_op = here_op;
            last_val = here_val;
            for (; ; ) {
              here = state.lencode[last_val + ((hold & (1 << last_bits + last_op) - 1) >> last_bits)];
              here_bits = here >>> 24;
              here_op = here >>> 16 & 255;
              here_val = here & 65535;
              if (last_bits + here_bits <= bits2) {
                break;
              }
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits2;
              bits2 += 8;
            }
            hold >>>= last_bits;
            bits2 -= last_bits;
            state.back += last_bits;
          }
          hold >>>= here_bits;
          bits2 -= here_bits;
          state.back += here_bits;
          state.length = here_val;
          if (here_op === 0) {
            state.mode = LIT;
            break;
          }
          if (here_op & 32) {
            state.back = -1;
            state.mode = TYPE;
            break;
          }
          if (here_op & 64) {
            strm.msg = "invalid literal/length code";
            state.mode = BAD;
            break;
          }
          state.extra = here_op & 15;
          state.mode = LENEXT;
        case LENEXT:
          if (state.extra) {
            n2 = state.extra;
            while (bits2 < n2) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits2;
              bits2 += 8;
            }
            state.length += hold & (1 << state.extra) - 1;
            hold >>>= state.extra;
            bits2 -= state.extra;
            state.back += state.extra;
          }
          state.was = state.length;
          state.mode = DIST;
        case DIST:
          for (; ; ) {
            here = state.distcode[hold & (1 << state.distbits) - 1];
            here_bits = here >>> 24;
            here_op = here >>> 16 & 255;
            here_val = here & 65535;
            if (here_bits <= bits2) {
              break;
            }
            if (have === 0) {
              break inf_leave;
            }
            have--;
            hold += input[next++] << bits2;
            bits2 += 8;
          }
          if ((here_op & 240) === 0) {
            last_bits = here_bits;
            last_op = here_op;
            last_val = here_val;
            for (; ; ) {
              here = state.distcode[last_val + ((hold & (1 << last_bits + last_op) - 1) >> last_bits)];
              here_bits = here >>> 24;
              here_op = here >>> 16 & 255;
              here_val = here & 65535;
              if (last_bits + here_bits <= bits2) {
                break;
              }
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits2;
              bits2 += 8;
            }
            hold >>>= last_bits;
            bits2 -= last_bits;
            state.back += last_bits;
          }
          hold >>>= here_bits;
          bits2 -= here_bits;
          state.back += here_bits;
          if (here_op & 64) {
            strm.msg = "invalid distance code";
            state.mode = BAD;
            break;
          }
          state.offset = here_val;
          state.extra = here_op & 15;
          state.mode = DISTEXT;
        case DISTEXT:
          if (state.extra) {
            n2 = state.extra;
            while (bits2 < n2) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits2;
              bits2 += 8;
            }
            state.offset += hold & (1 << state.extra) - 1;
            hold >>>= state.extra;
            bits2 -= state.extra;
            state.back += state.extra;
          }
          if (state.offset > state.dmax) {
            strm.msg = "invalid distance too far back";
            state.mode = BAD;
            break;
          }
          state.mode = MATCH$2;
        case MATCH$2:
          if (left === 0) {
            break inf_leave;
          }
          copy2 = _out - left;
          if (state.offset > copy2) {
            copy2 = state.offset - copy2;
            if (copy2 > state.whave) {
              if (state.sane) {
                strm.msg = "invalid distance too far back";
                state.mode = BAD;
                break;
              }
            }
            if (copy2 > state.wnext) {
              copy2 -= state.wnext;
              from2 = state.wsize - copy2;
            } else {
              from2 = state.wnext - copy2;
            }
            if (copy2 > state.length) {
              copy2 = state.length;
            }
            from_source = state.window;
          } else {
            from_source = output2;
            from2 = put - state.offset;
            copy2 = state.length;
          }
          if (copy2 > left) {
            copy2 = left;
          }
          left -= copy2;
          state.length -= copy2;
          do {
            output2[put++] = from_source[from2++];
          } while (--copy2);
          if (state.length === 0) {
            state.mode = LEN;
          }
          break;
        case LIT:
          if (left === 0) {
            break inf_leave;
          }
          output2[put++] = state.length;
          left--;
          state.mode = LEN;
          break;
        case CHECK:
          if (state.wrap) {
            while (bits2 < 32) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold |= input[next++] << bits2;
              bits2 += 8;
            }
            _out -= left;
            strm.total_out += _out;
            state.total += _out;
            if (state.wrap & 4 && _out) {
              strm.adler = state.check = /*UPDATE_CHECK(state.check, put - _out, _out);*/
              state.flags ? crc32_1(state.check, output2, _out, put - _out) : adler32_1(state.check, output2, _out, put - _out);
            }
            _out = left;
            if (state.wrap & 4 && (state.flags ? hold : zswap32(hold)) !== state.check) {
              strm.msg = "incorrect data check";
              state.mode = BAD;
              break;
            }
            hold = 0;
            bits2 = 0;
          }
          state.mode = LENGTH;
        case LENGTH:
          if (state.wrap && state.flags) {
            while (bits2 < 32) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits2;
              bits2 += 8;
            }
            if (state.wrap & 4 && hold !== (state.total & 4294967295)) {
              strm.msg = "incorrect length check";
              state.mode = BAD;
              break;
            }
            hold = 0;
            bits2 = 0;
          }
          state.mode = DONE;
        case DONE:
          ret = Z_STREAM_END$1;
          break inf_leave;
        case BAD:
          ret = Z_DATA_ERROR$1;
          break inf_leave;
        case MEM:
          return Z_MEM_ERROR$1;
        case SYNC:
        default:
          return Z_STREAM_ERROR$1;
      }
    }
  strm.next_out = put;
  strm.avail_out = left;
  strm.next_in = next;
  strm.avail_in = have;
  state.hold = hold;
  state.bits = bits2;
  if (state.wsize || _out !== strm.avail_out && state.mode < BAD && (state.mode < CHECK || flush !== Z_FINISH$1)) {
    if (updatewindow(strm, strm.output, strm.next_out, _out - strm.avail_out))
      ;
  }
  _in -= strm.avail_in;
  _out -= strm.avail_out;
  strm.total_in += _in;
  strm.total_out += _out;
  state.total += _out;
  if (state.wrap & 4 && _out) {
    strm.adler = state.check = /*UPDATE_CHECK(state.check, strm.next_out - _out, _out);*/
    state.flags ? crc32_1(state.check, output2, _out, strm.next_out - _out) : adler32_1(state.check, output2, _out, strm.next_out - _out);
  }
  strm.data_type = state.bits + (state.last ? 64 : 0) + (state.mode === TYPE ? 128 : 0) + (state.mode === LEN_ || state.mode === COPY_ ? 256 : 0);
  if ((_in === 0 && _out === 0 || flush === Z_FINISH$1) && ret === Z_OK$1) {
    ret = Z_BUF_ERROR;
  }
  return ret;
};
const inflateEnd = (strm) => {
  if (inflateStateCheck(strm)) {
    return Z_STREAM_ERROR$1;
  }
  let state = strm.state;
  if (state.window) {
    state.window = null;
  }
  strm.state = null;
  return Z_OK$1;
};
const inflateGetHeader = (strm, head) => {
  if (inflateStateCheck(strm)) {
    return Z_STREAM_ERROR$1;
  }
  const state = strm.state;
  if ((state.wrap & 2) === 0) {
    return Z_STREAM_ERROR$1;
  }
  state.head = head;
  head.done = false;
  return Z_OK$1;
};
const inflateSetDictionary = (strm, dictionary) => {
  const dictLength = dictionary.length;
  let state;
  let dictid;
  let ret;
  if (inflateStateCheck(strm)) {
    return Z_STREAM_ERROR$1;
  }
  state = strm.state;
  if (state.wrap !== 0 && state.mode !== DICT) {
    return Z_STREAM_ERROR$1;
  }
  if (state.mode === DICT) {
    dictid = 1;
    dictid = adler32_1(dictid, dictionary, dictLength, 0);
    if (dictid !== state.check) {
      return Z_DATA_ERROR$1;
    }
  }
  ret = updatewindow(strm, dictionary, dictLength, dictLength);
  if (ret) {
    state.mode = MEM;
    return Z_MEM_ERROR$1;
  }
  state.havedict = 1;
  return Z_OK$1;
};
var inflateReset_1 = inflateReset;
var inflateReset2_1 = inflateReset2;
var inflateResetKeep_1 = inflateResetKeep;
var inflateInit_1 = inflateInit;
var inflateInit2_1 = inflateInit2;
var inflate_2$1 = inflate$2;
var inflateEnd_1 = inflateEnd;
var inflateGetHeader_1 = inflateGetHeader;
var inflateSetDictionary_1 = inflateSetDictionary;
var inflateInfo = "pako inflate (from Nodeca project)";
var inflate_1$2 = {
  inflateReset: inflateReset_1,
  inflateReset2: inflateReset2_1,
  inflateResetKeep: inflateResetKeep_1,
  inflateInit: inflateInit_1,
  inflateInit2: inflateInit2_1,
  inflate: inflate_2$1,
  inflateEnd: inflateEnd_1,
  inflateGetHeader: inflateGetHeader_1,
  inflateSetDictionary: inflateSetDictionary_1,
  inflateInfo
};
function GZheader() {
  this.text = 0;
  this.time = 0;
  this.xflags = 0;
  this.os = 0;
  this.extra = null;
  this.extra_len = 0;
  this.name = "";
  this.comment = "";
  this.hcrc = 0;
  this.done = false;
}
var gzheader = GZheader;
const toString$d = Object.prototype.toString;
const {
  Z_NO_FLUSH,
  Z_FINISH,
  Z_OK,
  Z_STREAM_END,
  Z_NEED_DICT,
  Z_STREAM_ERROR,
  Z_DATA_ERROR,
  Z_MEM_ERROR
} = constants$2$1;
function Inflate$1(options) {
  this.options = common.assign({
    chunkSize: 1024 * 64,
    windowBits: 15,
    to: ""
  }, options || {});
  const opt = this.options;
  if (opt.raw && opt.windowBits >= 0 && opt.windowBits < 16) {
    opt.windowBits = -opt.windowBits;
    if (opt.windowBits === 0) {
      opt.windowBits = -15;
    }
  }
  if (opt.windowBits >= 0 && opt.windowBits < 16 && !(options && options.windowBits)) {
    opt.windowBits += 32;
  }
  if (opt.windowBits > 15 && opt.windowBits < 48) {
    if ((opt.windowBits & 15) === 0) {
      opt.windowBits |= 15;
    }
  }
  this.err = 0;
  this.msg = "";
  this.ended = false;
  this.chunks = [];
  this.strm = new zstream();
  this.strm.avail_out = 0;
  let status = inflate_1$2.inflateInit2(
    this.strm,
    opt.windowBits
  );
  if (status !== Z_OK) {
    throw new Error(messages[status]);
  }
  this.header = new gzheader();
  inflate_1$2.inflateGetHeader(this.strm, this.header);
  if (opt.dictionary) {
    if (typeof opt.dictionary === "string") {
      opt.dictionary = strings.string2buf(opt.dictionary);
    } else if (toString$d.call(opt.dictionary) === "[object ArrayBuffer]") {
      opt.dictionary = new Uint8Array(opt.dictionary);
    }
    if (opt.raw) {
      status = inflate_1$2.inflateSetDictionary(this.strm, opt.dictionary);
      if (status !== Z_OK) {
        throw new Error(messages[status]);
      }
    }
  }
}
Inflate$1.prototype.push = function(data2, flush_mode) {
  const strm = this.strm;
  const chunkSize = this.options.chunkSize;
  const dictionary = this.options.dictionary;
  let status, _flush_mode, last_avail_out;
  if (this.ended)
    return false;
  if (flush_mode === ~~flush_mode)
    _flush_mode = flush_mode;
  else
    _flush_mode = flush_mode === true ? Z_FINISH : Z_NO_FLUSH;
  if (toString$d.call(data2) === "[object ArrayBuffer]") {
    strm.input = new Uint8Array(data2);
  } else {
    strm.input = data2;
  }
  strm.next_in = 0;
  strm.avail_in = strm.input.length;
  for (; ; ) {
    if (strm.avail_out === 0) {
      strm.output = new Uint8Array(chunkSize);
      strm.next_out = 0;
      strm.avail_out = chunkSize;
    }
    status = inflate_1$2.inflate(strm, _flush_mode);
    if (status === Z_NEED_DICT && dictionary) {
      status = inflate_1$2.inflateSetDictionary(strm, dictionary);
      if (status === Z_OK) {
        status = inflate_1$2.inflate(strm, _flush_mode);
      } else if (status === Z_DATA_ERROR) {
        status = Z_NEED_DICT;
      }
    }
    while (strm.avail_in > 0 && status === Z_STREAM_END && strm.state.wrap > 0 && data2[strm.next_in] !== 0) {
      inflate_1$2.inflateReset(strm);
      status = inflate_1$2.inflate(strm, _flush_mode);
    }
    switch (status) {
      case Z_STREAM_ERROR:
      case Z_DATA_ERROR:
      case Z_NEED_DICT:
      case Z_MEM_ERROR:
        this.onEnd(status);
        this.ended = true;
        return false;
    }
    last_avail_out = strm.avail_out;
    if (strm.next_out) {
      if (strm.avail_out === 0 || status === Z_STREAM_END) {
        if (this.options.to === "string") {
          let next_out_utf8 = strings.utf8border(strm.output, strm.next_out);
          let tail = strm.next_out - next_out_utf8;
          let utf8str = strings.buf2string(strm.output, next_out_utf8);
          strm.next_out = tail;
          strm.avail_out = chunkSize - tail;
          if (tail)
            strm.output.set(strm.output.subarray(next_out_utf8, next_out_utf8 + tail), 0);
          this.onData(utf8str);
        } else {
          this.onData(strm.output.length === strm.next_out ? strm.output : strm.output.subarray(0, strm.next_out));
        }
      }
    }
    if (status === Z_OK && last_avail_out === 0)
      continue;
    if (status === Z_STREAM_END) {
      status = inflate_1$2.inflateEnd(this.strm);
      this.onEnd(status);
      this.ended = true;
      return true;
    }
    if (strm.avail_in === 0)
      break;
  }
  return true;
};
Inflate$1.prototype.onData = function(chunk) {
  this.chunks.push(chunk);
};
Inflate$1.prototype.onEnd = function(status) {
  if (status === Z_OK) {
    if (this.options.to === "string") {
      this.result = this.chunks.join("");
    } else {
      this.result = common.flattenChunks(this.chunks);
    }
  }
  this.chunks = [];
  this.err = status;
  this.msg = this.strm.msg;
};
function inflate$1(input, options) {
  const inflator = new Inflate$1(options);
  inflator.push(input);
  if (inflator.err)
    throw inflator.msg || messages[inflator.err];
  return inflator.result;
}
function inflateRaw$1(input, options) {
  options = options || {};
  options.raw = true;
  return inflate$1(input, options);
}
var Inflate_1$1 = Inflate$1;
var inflate_2 = inflate$1;
var inflateRaw_1$1 = inflateRaw$1;
var ungzip$1 = inflate$1;
var constants$3 = constants$2$1;
var inflate_1$1 = {
  Inflate: Inflate_1$1,
  inflate: inflate_2,
  inflateRaw: inflateRaw_1$1,
  ungzip: ungzip$1,
  constants: constants$3
};
const { Deflate, deflate, deflateRaw, gzip } = deflate_1$1;
const { Inflate, inflate, inflateRaw, ungzip } = inflate_1$1;
var Deflate_1 = Deflate;
var deflate_1 = deflate;
var deflateRaw_1 = deflateRaw;
var gzip_1 = gzip;
var Inflate_1 = Inflate;
var inflate_1 = inflate;
var inflateRaw_1 = inflateRaw;
var ungzip_1 = ungzip;
var constants_1 = constants$2$1;
var pako = {
  Deflate: Deflate_1,
  deflate: deflate_1,
  deflateRaw: deflateRaw_1,
  gzip: gzip_1,
  Inflate: Inflate_1,
  inflate: inflate_1,
  inflateRaw: inflateRaw_1,
  ungzip: ungzip_1,
  constants: constants_1
};
var BI_RM = "0123456789abcdefghijklmnopqrstuvwxyz";
function int2char(n2) {
  return BI_RM.charAt(n2);
}
function op_and(x2, y2) {
  return x2 & y2;
}
function op_or(x2, y2) {
  return x2 | y2;
}
function op_xor(x2, y2) {
  return x2 ^ y2;
}
function op_andnot(x2, y2) {
  return x2 & ~y2;
}
function lbit(x2) {
  if (x2 == 0) {
    return -1;
  }
  var r2 = 0;
  if ((x2 & 65535) == 0) {
    x2 >>= 16;
    r2 += 16;
  }
  if ((x2 & 255) == 0) {
    x2 >>= 8;
    r2 += 8;
  }
  if ((x2 & 15) == 0) {
    x2 >>= 4;
    r2 += 4;
  }
  if ((x2 & 3) == 0) {
    x2 >>= 2;
    r2 += 2;
  }
  if ((x2 & 1) == 0) {
    ++r2;
  }
  return r2;
}
function cbit(x2) {
  var r2 = 0;
  while (x2 != 0) {
    x2 &= x2 - 1;
    ++r2;
  }
  return r2;
}
var b64map = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
var b64pad = "=";
function hex2b64(h2) {
  var i2;
  var c2;
  var ret = "";
  for (i2 = 0; i2 + 3 <= h2.length; i2 += 3) {
    c2 = parseInt(h2.substring(i2, i2 + 3), 16);
    ret += b64map.charAt(c2 >> 6) + b64map.charAt(c2 & 63);
  }
  if (i2 + 1 == h2.length) {
    c2 = parseInt(h2.substring(i2, i2 + 1), 16);
    ret += b64map.charAt(c2 << 2);
  } else if (i2 + 2 == h2.length) {
    c2 = parseInt(h2.substring(i2, i2 + 2), 16);
    ret += b64map.charAt(c2 >> 2) + b64map.charAt((c2 & 3) << 4);
  }
  while ((ret.length & 3) > 0) {
    ret += b64pad;
  }
  return ret;
}
function b64tohex(s2) {
  var ret = "";
  var i2;
  var k2 = 0;
  var slop = 0;
  for (i2 = 0; i2 < s2.length; ++i2) {
    if (s2.charAt(i2) == b64pad) {
      break;
    }
    var v4 = b64map.indexOf(s2.charAt(i2));
    if (v4 < 0) {
      continue;
    }
    if (k2 == 0) {
      ret += int2char(v4 >> 2);
      slop = v4 & 3;
      k2 = 1;
    } else if (k2 == 1) {
      ret += int2char(slop << 2 | v4 >> 4);
      slop = v4 & 15;
      k2 = 2;
    } else if (k2 == 2) {
      ret += int2char(slop);
      ret += int2char(v4 >> 2);
      slop = v4 & 3;
      k2 = 3;
    } else {
      ret += int2char(slop << 2 | v4 >> 4);
      ret += int2char(v4 & 15);
      k2 = 0;
    }
  }
  if (k2 == 1) {
    ret += int2char(slop << 2);
  }
  return ret;
}
var decoder$1;
var Hex = {
  decode: function(a2) {
    var i2;
    if (decoder$1 === void 0) {
      var hex = "0123456789ABCDEF";
      var ignore = " \f\n\r	 \u2028\u2029";
      decoder$1 = {};
      for (i2 = 0; i2 < 16; ++i2) {
        decoder$1[hex.charAt(i2)] = i2;
      }
      hex = hex.toLowerCase();
      for (i2 = 10; i2 < 16; ++i2) {
        decoder$1[hex.charAt(i2)] = i2;
      }
      for (i2 = 0; i2 < ignore.length; ++i2) {
        decoder$1[ignore.charAt(i2)] = -1;
      }
    }
    var out2 = [];
    var bits2 = 0;
    var char_count = 0;
    for (i2 = 0; i2 < a2.length; ++i2) {
      var c2 = a2.charAt(i2);
      if (c2 == "=") {
        break;
      }
      c2 = decoder$1[c2];
      if (c2 == -1) {
        continue;
      }
      if (c2 === void 0) {
        throw new Error("Illegal character at offset " + i2);
      }
      bits2 |= c2;
      if (++char_count >= 2) {
        out2[out2.length] = bits2;
        bits2 = 0;
        char_count = 0;
      } else {
        bits2 <<= 4;
      }
    }
    if (char_count) {
      throw new Error("Hex encoding incomplete: 4 bits missing");
    }
    return out2;
  }
};
var decoder;
var Base64 = {
  decode: function(a2) {
    var i2;
    if (decoder === void 0) {
      var b64 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
      var ignore = "= \f\n\r	 \u2028\u2029";
      decoder = /* @__PURE__ */ Object.create(null);
      for (i2 = 0; i2 < 64; ++i2) {
        decoder[b64.charAt(i2)] = i2;
      }
      decoder["-"] = 62;
      decoder["_"] = 63;
      for (i2 = 0; i2 < ignore.length; ++i2) {
        decoder[ignore.charAt(i2)] = -1;
      }
    }
    var out2 = [];
    var bits2 = 0;
    var char_count = 0;
    for (i2 = 0; i2 < a2.length; ++i2) {
      var c2 = a2.charAt(i2);
      if (c2 == "=") {
        break;
      }
      c2 = decoder[c2];
      if (c2 == -1) {
        continue;
      }
      if (c2 === void 0) {
        throw new Error("Illegal character at offset " + i2);
      }
      bits2 |= c2;
      if (++char_count >= 4) {
        out2[out2.length] = bits2 >> 16;
        out2[out2.length] = bits2 >> 8 & 255;
        out2[out2.length] = bits2 & 255;
        bits2 = 0;
        char_count = 0;
      } else {
        bits2 <<= 6;
      }
    }
    switch (char_count) {
      case 1:
        throw new Error("Base64 encoding incomplete: at least 2 bits missing");
      case 2:
        out2[out2.length] = bits2 >> 10;
        break;
      case 3:
        out2[out2.length] = bits2 >> 16;
        out2[out2.length] = bits2 >> 8 & 255;
        break;
    }
    return out2;
  },
  re: /-----BEGIN [^-]+-----([A-Za-z0-9+\/=\s]+)-----END [^-]+-----|begin-base64[^\n]+\n([A-Za-z0-9+\/=\s]+)====/,
  unarmor: function(a2) {
    var m2 = Base64.re.exec(a2);
    if (m2) {
      if (m2[1]) {
        a2 = m2[1];
      } else if (m2[2]) {
        a2 = m2[2];
      } else {
        throw new Error("RegExp out of sync");
      }
    }
    return Base64.decode(a2);
  }
};
var max$7 = 1e13;
var Int10 = (
  /** @class */
  function() {
    function Int102(value) {
      this.buf = [+value || 0];
    }
    Int102.prototype.mulAdd = function(m2, c2) {
      var b2 = this.buf;
      var l2 = b2.length;
      var i2;
      var t3;
      for (i2 = 0; i2 < l2; ++i2) {
        t3 = b2[i2] * m2 + c2;
        if (t3 < max$7) {
          c2 = 0;
        } else {
          c2 = 0 | t3 / max$7;
          t3 -= c2 * max$7;
        }
        b2[i2] = t3;
      }
      if (c2 > 0) {
        b2[i2] = c2;
      }
    };
    Int102.prototype.sub = function(c2) {
      var b2 = this.buf;
      var l2 = b2.length;
      var i2;
      var t3;
      for (i2 = 0; i2 < l2; ++i2) {
        t3 = b2[i2] - c2;
        if (t3 < 0) {
          t3 += max$7;
          c2 = 1;
        } else {
          c2 = 0;
        }
        b2[i2] = t3;
      }
      while (b2[b2.length - 1] === 0) {
        b2.pop();
      }
    };
    Int102.prototype.toString = function(base2) {
      if ((base2 || 10) != 10) {
        throw new Error("only base 10 is supported");
      }
      var b2 = this.buf;
      var s2 = b2[b2.length - 1].toString();
      for (var i2 = b2.length - 2; i2 >= 0; --i2) {
        s2 += (max$7 + b2[i2]).toString().substring(1);
      }
      return s2;
    };
    Int102.prototype.valueOf = function() {
      var b2 = this.buf;
      var v4 = 0;
      for (var i2 = b2.length - 1; i2 >= 0; --i2) {
        v4 = v4 * max$7 + b2[i2];
      }
      return v4;
    };
    Int102.prototype.simplify = function() {
      var b2 = this.buf;
      return b2.length == 1 ? b2[0] : this;
    };
    return Int102;
  }()
);
var ellipsis = "…";
var reTimeS = /^(\d\d)(0[1-9]|1[0-2])(0[1-9]|[12]\d|3[01])([01]\d|2[0-3])(?:([0-5]\d)(?:([0-5]\d)(?:[.,](\d{1,3}))?)?)?(Z|[-+](?:[0]\d|1[0-2])([0-5]\d)?)?$/;
var reTimeL = /^(\d\d\d\d)(0[1-9]|1[0-2])(0[1-9]|[12]\d|3[01])([01]\d|2[0-3])(?:([0-5]\d)(?:([0-5]\d)(?:[.,](\d{1,3}))?)?)?(Z|[-+](?:[0]\d|1[0-2])([0-5]\d)?)?$/;
function stringCut(str, len2) {
  if (str.length > len2) {
    str = str.substring(0, len2) + ellipsis;
  }
  return str;
}
var Stream = (
  /** @class */
  function() {
    function Stream2(enc, pos) {
      this.hexDigits = "0123456789ABCDEF";
      if (enc instanceof Stream2) {
        this.enc = enc.enc;
        this.pos = enc.pos;
      } else {
        this.enc = enc;
        this.pos = pos;
      }
    }
    Stream2.prototype.get = function(pos) {
      if (pos === void 0) {
        pos = this.pos++;
      }
      if (pos >= this.enc.length) {
        throw new Error("Requesting byte offset ".concat(pos, " on a stream of length ").concat(this.enc.length));
      }
      return "string" === typeof this.enc ? this.enc.charCodeAt(pos) : this.enc[pos];
    };
    Stream2.prototype.hexByte = function(b2) {
      return this.hexDigits.charAt(b2 >> 4 & 15) + this.hexDigits.charAt(b2 & 15);
    };
    Stream2.prototype.hexDump = function(start2, end2, raw) {
      var s2 = "";
      for (var i2 = start2; i2 < end2; ++i2) {
        s2 += this.hexByte(this.get(i2));
        if (raw !== true) {
          switch (i2 & 15) {
            case 7:
              s2 += "  ";
              break;
            case 15:
              s2 += "\n";
              break;
            default:
              s2 += " ";
          }
        }
      }
      return s2;
    };
    Stream2.prototype.isASCII = function(start2, end2) {
      for (var i2 = start2; i2 < end2; ++i2) {
        var c2 = this.get(i2);
        if (c2 < 32 || c2 > 176) {
          return false;
        }
      }
      return true;
    };
    Stream2.prototype.parseStringISO = function(start2, end2) {
      var s2 = "";
      for (var i2 = start2; i2 < end2; ++i2) {
        s2 += String.fromCharCode(this.get(i2));
      }
      return s2;
    };
    Stream2.prototype.parseStringUTF = function(start2, end2) {
      var s2 = "";
      for (var i2 = start2; i2 < end2; ) {
        var c2 = this.get(i2++);
        if (c2 < 128) {
          s2 += String.fromCharCode(c2);
        } else if (c2 > 191 && c2 < 224) {
          s2 += String.fromCharCode((c2 & 31) << 6 | this.get(i2++) & 63);
        } else {
          s2 += String.fromCharCode((c2 & 15) << 12 | (this.get(i2++) & 63) << 6 | this.get(i2++) & 63);
        }
      }
      return s2;
    };
    Stream2.prototype.parseStringBMP = function(start2, end2) {
      var str = "";
      var hi;
      var lo;
      for (var i2 = start2; i2 < end2; ) {
        hi = this.get(i2++);
        lo = this.get(i2++);
        str += String.fromCharCode(hi << 8 | lo);
      }
      return str;
    };
    Stream2.prototype.parseTime = function(start2, end2, shortYear) {
      var s2 = this.parseStringISO(start2, end2);
      var m2 = (shortYear ? reTimeS : reTimeL).exec(s2);
      if (!m2) {
        return "Unrecognized time: " + s2;
      }
      if (shortYear) {
        m2[1] = +m2[1];
        m2[1] += +m2[1] < 70 ? 2e3 : 1900;
      }
      s2 = m2[1] + "-" + m2[2] + "-" + m2[3] + " " + m2[4];
      if (m2[5]) {
        s2 += ":" + m2[5];
        if (m2[6]) {
          s2 += ":" + m2[6];
          if (m2[7]) {
            s2 += "." + m2[7];
          }
        }
      }
      if (m2[8]) {
        s2 += " UTC";
        if (m2[8] != "Z") {
          s2 += m2[8];
          if (m2[9]) {
            s2 += ":" + m2[9];
          }
        }
      }
      return s2;
    };
    Stream2.prototype.parseInteger = function(start2, end2) {
      var v4 = this.get(start2);
      var neg = v4 > 127;
      var pad2 = neg ? 255 : 0;
      var len2;
      var s2 = "";
      while (v4 == pad2 && ++start2 < end2) {
        v4 = this.get(start2);
      }
      len2 = end2 - start2;
      if (len2 === 0) {
        return neg ? -1 : 0;
      }
      if (len2 > 4) {
        s2 = v4;
        len2 <<= 3;
        while (((+s2 ^ pad2) & 128) == 0) {
          s2 = +s2 << 1;
          --len2;
        }
        s2 = "(" + len2 + " bit)\n";
      }
      if (neg) {
        v4 = v4 - 256;
      }
      var n2 = new Int10(v4);
      for (var i2 = start2 + 1; i2 < end2; ++i2) {
        n2.mulAdd(256, this.get(i2));
      }
      return s2 + n2.toString();
    };
    Stream2.prototype.parseBitString = function(start2, end2, maxLength) {
      var unusedBit = this.get(start2);
      var lenBit = (end2 - start2 - 1 << 3) - unusedBit;
      var intro = "(" + lenBit + " bit)\n";
      var s2 = "";
      for (var i2 = start2 + 1; i2 < end2; ++i2) {
        var b2 = this.get(i2);
        var skip = i2 == end2 - 1 ? unusedBit : 0;
        for (var j2 = 7; j2 >= skip; --j2) {
          s2 += b2 >> j2 & 1 ? "1" : "0";
        }
        if (s2.length > maxLength) {
          return intro + stringCut(s2, maxLength);
        }
      }
      return intro + s2;
    };
    Stream2.prototype.parseOctetString = function(start2, end2, maxLength) {
      if (this.isASCII(start2, end2)) {
        return stringCut(this.parseStringISO(start2, end2), maxLength);
      }
      var len2 = end2 - start2;
      var s2 = "(" + len2 + " byte)\n";
      maxLength /= 2;
      if (len2 > maxLength) {
        end2 = start2 + maxLength;
      }
      for (var i2 = start2; i2 < end2; ++i2) {
        s2 += this.hexByte(this.get(i2));
      }
      if (len2 > maxLength) {
        s2 += ellipsis;
      }
      return s2;
    };
    Stream2.prototype.parseOID = function(start2, end2, maxLength) {
      var s2 = "";
      var n2 = new Int10();
      var bits2 = 0;
      for (var i2 = start2; i2 < end2; ++i2) {
        var v4 = this.get(i2);
        n2.mulAdd(128, v4 & 127);
        bits2 += 7;
        if (!(v4 & 128)) {
          if (s2 === "") {
            n2 = n2.simplify();
            if (n2 instanceof Int10) {
              n2.sub(80);
              s2 = "2." + n2.toString();
            } else {
              var m2 = n2 < 80 ? n2 < 40 ? 0 : 1 : 2;
              s2 = m2 + "." + (n2 - m2 * 40);
            }
          } else {
            s2 += "." + n2.toString();
          }
          if (s2.length > maxLength) {
            return stringCut(s2, maxLength);
          }
          n2 = new Int10();
          bits2 = 0;
        }
      }
      if (bits2 > 0) {
        s2 += ".incomplete";
      }
      return s2;
    };
    return Stream2;
  }()
);
var ASN1 = (
  /** @class */
  function() {
    function ASN12(stream, header, length, tag2, sub2) {
      if (!(tag2 instanceof ASN1Tag)) {
        throw new Error("Invalid tag value.");
      }
      this.stream = stream;
      this.header = header;
      this.length = length;
      this.tag = tag2;
      this.sub = sub2;
    }
    ASN12.prototype.typeName = function() {
      switch (this.tag.tagClass) {
        case 0:
          switch (this.tag.tagNumber) {
            case 0:
              return "EOC";
            case 1:
              return "BOOLEAN";
            case 2:
              return "INTEGER";
            case 3:
              return "BIT_STRING";
            case 4:
              return "OCTET_STRING";
            case 5:
              return "NULL";
            case 6:
              return "OBJECT_IDENTIFIER";
            case 7:
              return "ObjectDescriptor";
            case 8:
              return "EXTERNAL";
            case 9:
              return "REAL";
            case 10:
              return "ENUMERATED";
            case 11:
              return "EMBEDDED_PDV";
            case 12:
              return "UTF8String";
            case 16:
              return "SEQUENCE";
            case 17:
              return "SET";
            case 18:
              return "NumericString";
            case 19:
              return "PrintableString";
            case 20:
              return "TeletexString";
            case 21:
              return "VideotexString";
            case 22:
              return "IA5String";
            case 23:
              return "UTCTime";
            case 24:
              return "GeneralizedTime";
            case 25:
              return "GraphicString";
            case 26:
              return "VisibleString";
            case 27:
              return "GeneralString";
            case 28:
              return "UniversalString";
            case 30:
              return "BMPString";
          }
          return "Universal_" + this.tag.tagNumber.toString();
        case 1:
          return "Application_" + this.tag.tagNumber.toString();
        case 2:
          return "[" + this.tag.tagNumber.toString() + "]";
        case 3:
          return "Private_" + this.tag.tagNumber.toString();
      }
    };
    ASN12.prototype.content = function(maxLength) {
      if (this.tag === void 0) {
        return null;
      }
      if (maxLength === void 0) {
        maxLength = Infinity;
      }
      var content2 = this.posContent();
      var len2 = Math.abs(this.length);
      if (!this.tag.isUniversal()) {
        if (this.sub !== null) {
          return "(" + this.sub.length + " elem)";
        }
        return this.stream.parseOctetString(content2, content2 + len2, maxLength);
      }
      switch (this.tag.tagNumber) {
        case 1:
          return this.stream.get(content2) === 0 ? "false" : "true";
        case 2:
          return this.stream.parseInteger(content2, content2 + len2);
        case 3:
          return this.sub ? "(" + this.sub.length + " elem)" : this.stream.parseBitString(content2, content2 + len2, maxLength);
        case 4:
          return this.sub ? "(" + this.sub.length + " elem)" : this.stream.parseOctetString(content2, content2 + len2, maxLength);
        case 6:
          return this.stream.parseOID(content2, content2 + len2, maxLength);
        case 16:
        case 17:
          if (this.sub !== null) {
            return "(" + this.sub.length + " elem)";
          } else {
            return "(no elem)";
          }
        case 12:
          return stringCut(this.stream.parseStringUTF(content2, content2 + len2), maxLength);
        case 18:
        case 19:
        case 20:
        case 21:
        case 22:
        case 26:
          return stringCut(this.stream.parseStringISO(content2, content2 + len2), maxLength);
        case 30:
          return stringCut(this.stream.parseStringBMP(content2, content2 + len2), maxLength);
        case 23:
        case 24:
          return this.stream.parseTime(content2, content2 + len2, this.tag.tagNumber == 23);
      }
      return null;
    };
    ASN12.prototype.toString = function() {
      return this.typeName() + "@" + this.stream.pos + "[header:" + this.header + ",length:" + this.length + ",sub:" + (this.sub === null ? "null" : this.sub.length) + "]";
    };
    ASN12.prototype.toPrettyString = function(indent) {
      if (indent === void 0) {
        indent = "";
      }
      var s2 = indent + this.typeName() + " @" + this.stream.pos;
      if (this.length >= 0) {
        s2 += "+";
      }
      s2 += this.length;
      if (this.tag.tagConstructed) {
        s2 += " (constructed)";
      } else if (this.tag.isUniversal() && (this.tag.tagNumber == 3 || this.tag.tagNumber == 4) && this.sub !== null) {
        s2 += " (encapsulates)";
      }
      s2 += "\n";
      if (this.sub !== null) {
        indent += "  ";
        for (var i2 = 0, max3 = this.sub.length; i2 < max3; ++i2) {
          s2 += this.sub[i2].toPrettyString(indent);
        }
      }
      return s2;
    };
    ASN12.prototype.posStart = function() {
      return this.stream.pos;
    };
    ASN12.prototype.posContent = function() {
      return this.stream.pos + this.header;
    };
    ASN12.prototype.posEnd = function() {
      return this.stream.pos + this.header + Math.abs(this.length);
    };
    ASN12.prototype.toHexString = function() {
      return this.stream.hexDump(this.posStart(), this.posEnd(), true);
    };
    ASN12.decodeLength = function(stream) {
      var buf = stream.get();
      var len2 = buf & 127;
      if (len2 == buf) {
        return len2;
      }
      if (len2 > 6) {
        throw new Error("Length over 48 bits not supported at position " + (stream.pos - 1));
      }
      if (len2 === 0) {
        return null;
      }
      buf = 0;
      for (var i2 = 0; i2 < len2; ++i2) {
        buf = buf * 256 + stream.get();
      }
      return buf;
    };
    ASN12.prototype.getHexStringValue = function() {
      var hexString = this.toHexString();
      var offset2 = this.header * 2;
      var length = this.length * 2;
      return hexString.substr(offset2, length);
    };
    ASN12.decode = function(str) {
      var stream;
      if (!(str instanceof Stream)) {
        stream = new Stream(str, 0);
      } else {
        stream = str;
      }
      var streamStart = new Stream(stream);
      var tag2 = new ASN1Tag(stream);
      var len2 = ASN12.decodeLength(stream);
      var start2 = stream.pos;
      var header = start2 - streamStart.pos;
      var sub2 = null;
      var getSub = function() {
        var ret = [];
        if (len2 !== null) {
          var end2 = start2 + len2;
          while (stream.pos < end2) {
            ret[ret.length] = ASN12.decode(stream);
          }
          if (stream.pos != end2) {
            throw new Error("Content size is not correct for container starting at offset " + start2);
          }
        } else {
          try {
            for (; ; ) {
              var s2 = ASN12.decode(stream);
              if (s2.tag.isEOC()) {
                break;
              }
              ret[ret.length] = s2;
            }
            len2 = start2 - stream.pos;
          } catch (e2) {
            throw new Error("Exception while decoding undefined length content: " + e2);
          }
        }
        return ret;
      };
      if (tag2.tagConstructed) {
        sub2 = getSub();
      } else if (tag2.isUniversal() && (tag2.tagNumber == 3 || tag2.tagNumber == 4)) {
        try {
          if (tag2.tagNumber == 3) {
            if (stream.get() != 0) {
              throw new Error("BIT STRINGs with unused bits cannot encapsulate.");
            }
          }
          sub2 = getSub();
          for (var i2 = 0; i2 < sub2.length; ++i2) {
            if (sub2[i2].tag.isEOC()) {
              throw new Error("EOC is not supposed to be actual content.");
            }
          }
        } catch (e2) {
          sub2 = null;
        }
      }
      if (sub2 === null) {
        if (len2 === null) {
          throw new Error("We can't skip over an invalid tag with undefined length at offset " + start2);
        }
        stream.pos = start2 + Math.abs(len2);
      }
      return new ASN12(streamStart, header, len2, tag2, sub2);
    };
    return ASN12;
  }()
);
var ASN1Tag = (
  /** @class */
  function() {
    function ASN1Tag2(stream) {
      var buf = stream.get();
      this.tagClass = buf >> 6;
      this.tagConstructed = (buf & 32) !== 0;
      this.tagNumber = buf & 31;
      if (this.tagNumber == 31) {
        var n2 = new Int10();
        do {
          buf = stream.get();
          n2.mulAdd(128, buf & 127);
        } while (buf & 128);
        this.tagNumber = n2.simplify();
      }
    }
    ASN1Tag2.prototype.isUniversal = function() {
      return this.tagClass === 0;
    };
    ASN1Tag2.prototype.isEOC = function() {
      return this.tagClass === 0 && this.tagNumber === 0;
    };
    return ASN1Tag2;
  }()
);
var dbits;
var canary = 244837814094590;
var j_lm = (canary & 16777215) == 15715070;
var lowprimes = [2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83, 89, 97, 101, 103, 107, 109, 113, 127, 131, 137, 139, 149, 151, 157, 163, 167, 173, 179, 181, 191, 193, 197, 199, 211, 223, 227, 229, 233, 239, 241, 251, 257, 263, 269, 271, 277, 281, 283, 293, 307, 311, 313, 317, 331, 337, 347, 349, 353, 359, 367, 373, 379, 383, 389, 397, 401, 409, 419, 421, 431, 433, 439, 443, 449, 457, 461, 463, 467, 479, 487, 491, 499, 503, 509, 521, 523, 541, 547, 557, 563, 569, 571, 577, 587, 593, 599, 601, 607, 613, 617, 619, 631, 641, 643, 647, 653, 659, 661, 673, 677, 683, 691, 701, 709, 719, 727, 733, 739, 743, 751, 757, 761, 769, 773, 787, 797, 809, 811, 821, 823, 827, 829, 839, 853, 857, 859, 863, 877, 881, 883, 887, 907, 911, 919, 929, 937, 941, 947, 953, 967, 971, 977, 983, 991, 997];
var lplim = (1 << 26) / lowprimes[lowprimes.length - 1];
var BigInteger = (
  /** @class */
  function() {
    function BigInteger2(a2, b2, c2) {
      if (a2 != null) {
        if ("number" == typeof a2) {
          this.fromNumber(a2, b2, c2);
        } else if (b2 == null && "string" != typeof a2) {
          this.fromString(a2, 256);
        } else {
          this.fromString(a2, b2);
        }
      }
    }
    BigInteger2.prototype.toString = function(b2) {
      if (this.s < 0) {
        return "-" + this.negate().toString(b2);
      }
      var k2;
      if (b2 == 16) {
        k2 = 4;
      } else if (b2 == 8) {
        k2 = 3;
      } else if (b2 == 2) {
        k2 = 1;
      } else if (b2 == 32) {
        k2 = 5;
      } else if (b2 == 4) {
        k2 = 2;
      } else {
        return this.toRadix(b2);
      }
      var km = (1 << k2) - 1;
      var d2;
      var m2 = false;
      var r2 = "";
      var i2 = this.t;
      var p2 = this.DB - i2 * this.DB % k2;
      if (i2-- > 0) {
        if (p2 < this.DB && (d2 = this[i2] >> p2) > 0) {
          m2 = true;
          r2 = int2char(d2);
        }
        while (i2 >= 0) {
          if (p2 < k2) {
            d2 = (this[i2] & (1 << p2) - 1) << k2 - p2;
            d2 |= this[--i2] >> (p2 += this.DB - k2);
          } else {
            d2 = this[i2] >> (p2 -= k2) & km;
            if (p2 <= 0) {
              p2 += this.DB;
              --i2;
            }
          }
          if (d2 > 0) {
            m2 = true;
          }
          if (m2) {
            r2 += int2char(d2);
          }
        }
      }
      return m2 ? r2 : "0";
    };
    BigInteger2.prototype.negate = function() {
      var r2 = nbi();
      BigInteger2.ZERO.subTo(this, r2);
      return r2;
    };
    BigInteger2.prototype.abs = function() {
      return this.s < 0 ? this.negate() : this;
    };
    BigInteger2.prototype.compareTo = function(a2) {
      var r2 = this.s - a2.s;
      if (r2 != 0) {
        return r2;
      }
      var i2 = this.t;
      r2 = i2 - a2.t;
      if (r2 != 0) {
        return this.s < 0 ? -r2 : r2;
      }
      while (--i2 >= 0) {
        if ((r2 = this[i2] - a2[i2]) != 0) {
          return r2;
        }
      }
      return 0;
    };
    BigInteger2.prototype.bitLength = function() {
      if (this.t <= 0) {
        return 0;
      }
      return this.DB * (this.t - 1) + nbits(this[this.t - 1] ^ this.s & this.DM);
    };
    BigInteger2.prototype.mod = function(a2) {
      var r2 = nbi();
      this.abs().divRemTo(a2, null, r2);
      if (this.s < 0 && r2.compareTo(BigInteger2.ZERO) > 0) {
        a2.subTo(r2, r2);
      }
      return r2;
    };
    BigInteger2.prototype.modPowInt = function(e2, m2) {
      var z2;
      if (e2 < 256 || m2.isEven()) {
        z2 = new Classic(m2);
      } else {
        z2 = new Montgomery(m2);
      }
      return this.exp(e2, z2);
    };
    BigInteger2.prototype.clone = function() {
      var r2 = nbi();
      this.copyTo(r2);
      return r2;
    };
    BigInteger2.prototype.intValue = function() {
      if (this.s < 0) {
        if (this.t == 1) {
          return this[0] - this.DV;
        } else if (this.t == 0) {
          return -1;
        }
      } else if (this.t == 1) {
        return this[0];
      } else if (this.t == 0) {
        return 0;
      }
      return (this[1] & (1 << 32 - this.DB) - 1) << this.DB | this[0];
    };
    BigInteger2.prototype.byteValue = function() {
      return this.t == 0 ? this.s : this[0] << 24 >> 24;
    };
    BigInteger2.prototype.shortValue = function() {
      return this.t == 0 ? this.s : this[0] << 16 >> 16;
    };
    BigInteger2.prototype.signum = function() {
      if (this.s < 0) {
        return -1;
      } else if (this.t <= 0 || this.t == 1 && this[0] <= 0) {
        return 0;
      } else {
        return 1;
      }
    };
    BigInteger2.prototype.toByteArray = function() {
      var i2 = this.t;
      var r2 = [];
      r2[0] = this.s;
      var p2 = this.DB - i2 * this.DB % 8;
      var d2;
      var k2 = 0;
      if (i2-- > 0) {
        if (p2 < this.DB && (d2 = this[i2] >> p2) != (this.s & this.DM) >> p2) {
          r2[k2++] = d2 | this.s << this.DB - p2;
        }
        while (i2 >= 0) {
          if (p2 < 8) {
            d2 = (this[i2] & (1 << p2) - 1) << 8 - p2;
            d2 |= this[--i2] >> (p2 += this.DB - 8);
          } else {
            d2 = this[i2] >> (p2 -= 8) & 255;
            if (p2 <= 0) {
              p2 += this.DB;
              --i2;
            }
          }
          if ((d2 & 128) != 0) {
            d2 |= -256;
          }
          if (k2 == 0 && (this.s & 128) != (d2 & 128)) {
            ++k2;
          }
          if (k2 > 0 || d2 != this.s) {
            r2[k2++] = d2;
          }
        }
      }
      return r2;
    };
    BigInteger2.prototype.equals = function(a2) {
      return this.compareTo(a2) == 0;
    };
    BigInteger2.prototype.min = function(a2) {
      return this.compareTo(a2) < 0 ? this : a2;
    };
    BigInteger2.prototype.max = function(a2) {
      return this.compareTo(a2) > 0 ? this : a2;
    };
    BigInteger2.prototype.and = function(a2) {
      var r2 = nbi();
      this.bitwiseTo(a2, op_and, r2);
      return r2;
    };
    BigInteger2.prototype.or = function(a2) {
      var r2 = nbi();
      this.bitwiseTo(a2, op_or, r2);
      return r2;
    };
    BigInteger2.prototype.xor = function(a2) {
      var r2 = nbi();
      this.bitwiseTo(a2, op_xor, r2);
      return r2;
    };
    BigInteger2.prototype.andNot = function(a2) {
      var r2 = nbi();
      this.bitwiseTo(a2, op_andnot, r2);
      return r2;
    };
    BigInteger2.prototype.not = function() {
      var r2 = nbi();
      for (var i2 = 0; i2 < this.t; ++i2) {
        r2[i2] = this.DM & ~this[i2];
      }
      r2.t = this.t;
      r2.s = ~this.s;
      return r2;
    };
    BigInteger2.prototype.shiftLeft = function(n2) {
      var r2 = nbi();
      if (n2 < 0) {
        this.rShiftTo(-n2, r2);
      } else {
        this.lShiftTo(n2, r2);
      }
      return r2;
    };
    BigInteger2.prototype.shiftRight = function(n2) {
      var r2 = nbi();
      if (n2 < 0) {
        this.lShiftTo(-n2, r2);
      } else {
        this.rShiftTo(n2, r2);
      }
      return r2;
    };
    BigInteger2.prototype.getLowestSetBit = function() {
      for (var i2 = 0; i2 < this.t; ++i2) {
        if (this[i2] != 0) {
          return i2 * this.DB + lbit(this[i2]);
        }
      }
      if (this.s < 0) {
        return this.t * this.DB;
      }
      return -1;
    };
    BigInteger2.prototype.bitCount = function() {
      var r2 = 0;
      var x2 = this.s & this.DM;
      for (var i2 = 0; i2 < this.t; ++i2) {
        r2 += cbit(this[i2] ^ x2);
      }
      return r2;
    };
    BigInteger2.prototype.testBit = function(n2) {
      var j2 = Math.floor(n2 / this.DB);
      if (j2 >= this.t) {
        return this.s != 0;
      }
      return (this[j2] & 1 << n2 % this.DB) != 0;
    };
    BigInteger2.prototype.setBit = function(n2) {
      return this.changeBit(n2, op_or);
    };
    BigInteger2.prototype.clearBit = function(n2) {
      return this.changeBit(n2, op_andnot);
    };
    BigInteger2.prototype.flipBit = function(n2) {
      return this.changeBit(n2, op_xor);
    };
    BigInteger2.prototype.add = function(a2) {
      var r2 = nbi();
      this.addTo(a2, r2);
      return r2;
    };
    BigInteger2.prototype.subtract = function(a2) {
      var r2 = nbi();
      this.subTo(a2, r2);
      return r2;
    };
    BigInteger2.prototype.multiply = function(a2) {
      var r2 = nbi();
      this.multiplyTo(a2, r2);
      return r2;
    };
    BigInteger2.prototype.divide = function(a2) {
      var r2 = nbi();
      this.divRemTo(a2, r2, null);
      return r2;
    };
    BigInteger2.prototype.remainder = function(a2) {
      var r2 = nbi();
      this.divRemTo(a2, null, r2);
      return r2;
    };
    BigInteger2.prototype.divideAndRemainder = function(a2) {
      var q2 = nbi();
      var r2 = nbi();
      this.divRemTo(a2, q2, r2);
      return [q2, r2];
    };
    BigInteger2.prototype.modPow = function(e2, m2) {
      var i2 = e2.bitLength();
      var k2;
      var r2 = nbv(1);
      var z2;
      if (i2 <= 0) {
        return r2;
      } else if (i2 < 18) {
        k2 = 1;
      } else if (i2 < 48) {
        k2 = 3;
      } else if (i2 < 144) {
        k2 = 4;
      } else if (i2 < 768) {
        k2 = 5;
      } else {
        k2 = 6;
      }
      if (i2 < 8) {
        z2 = new Classic(m2);
      } else if (m2.isEven()) {
        z2 = new Barrett(m2);
      } else {
        z2 = new Montgomery(m2);
      }
      var g2 = [];
      var n2 = 3;
      var k1 = k2 - 1;
      var km = (1 << k2) - 1;
      g2[1] = z2.convert(this);
      if (k2 > 1) {
        var g22 = nbi();
        z2.sqrTo(g2[1], g22);
        while (n2 <= km) {
          g2[n2] = nbi();
          z2.mulTo(g22, g2[n2 - 2], g2[n2]);
          n2 += 2;
        }
      }
      var j2 = e2.t - 1;
      var w2;
      var is1 = true;
      var r22 = nbi();
      var t3;
      i2 = nbits(e2[j2]) - 1;
      while (j2 >= 0) {
        if (i2 >= k1) {
          w2 = e2[j2] >> i2 - k1 & km;
        } else {
          w2 = (e2[j2] & (1 << i2 + 1) - 1) << k1 - i2;
          if (j2 > 0) {
            w2 |= e2[j2 - 1] >> this.DB + i2 - k1;
          }
        }
        n2 = k2;
        while ((w2 & 1) == 0) {
          w2 >>= 1;
          --n2;
        }
        if ((i2 -= n2) < 0) {
          i2 += this.DB;
          --j2;
        }
        if (is1) {
          g2[w2].copyTo(r2);
          is1 = false;
        } else {
          while (n2 > 1) {
            z2.sqrTo(r2, r22);
            z2.sqrTo(r22, r2);
            n2 -= 2;
          }
          if (n2 > 0) {
            z2.sqrTo(r2, r22);
          } else {
            t3 = r2;
            r2 = r22;
            r22 = t3;
          }
          z2.mulTo(r22, g2[w2], r2);
        }
        while (j2 >= 0 && (e2[j2] & 1 << i2) == 0) {
          z2.sqrTo(r2, r22);
          t3 = r2;
          r2 = r22;
          r22 = t3;
          if (--i2 < 0) {
            i2 = this.DB - 1;
            --j2;
          }
        }
      }
      return z2.revert(r2);
    };
    BigInteger2.prototype.modInverse = function(m2) {
      var ac = m2.isEven();
      if (this.isEven() && ac || m2.signum() == 0) {
        return BigInteger2.ZERO;
      }
      var u2 = m2.clone();
      var v4 = this.clone();
      var a2 = nbv(1);
      var b2 = nbv(0);
      var c2 = nbv(0);
      var d2 = nbv(1);
      while (u2.signum() != 0) {
        while (u2.isEven()) {
          u2.rShiftTo(1, u2);
          if (ac) {
            if (!a2.isEven() || !b2.isEven()) {
              a2.addTo(this, a2);
              b2.subTo(m2, b2);
            }
            a2.rShiftTo(1, a2);
          } else if (!b2.isEven()) {
            b2.subTo(m2, b2);
          }
          b2.rShiftTo(1, b2);
        }
        while (v4.isEven()) {
          v4.rShiftTo(1, v4);
          if (ac) {
            if (!c2.isEven() || !d2.isEven()) {
              c2.addTo(this, c2);
              d2.subTo(m2, d2);
            }
            c2.rShiftTo(1, c2);
          } else if (!d2.isEven()) {
            d2.subTo(m2, d2);
          }
          d2.rShiftTo(1, d2);
        }
        if (u2.compareTo(v4) >= 0) {
          u2.subTo(v4, u2);
          if (ac) {
            a2.subTo(c2, a2);
          }
          b2.subTo(d2, b2);
        } else {
          v4.subTo(u2, v4);
          if (ac) {
            c2.subTo(a2, c2);
          }
          d2.subTo(b2, d2);
        }
      }
      if (v4.compareTo(BigInteger2.ONE) != 0) {
        return BigInteger2.ZERO;
      }
      if (d2.compareTo(m2) >= 0) {
        return d2.subtract(m2);
      }
      if (d2.signum() < 0) {
        d2.addTo(m2, d2);
      } else {
        return d2;
      }
      if (d2.signum() < 0) {
        return d2.add(m2);
      } else {
        return d2;
      }
    };
    BigInteger2.prototype.pow = function(e2) {
      return this.exp(e2, new NullExp());
    };
    BigInteger2.prototype.gcd = function(a2) {
      var x2 = this.s < 0 ? this.negate() : this.clone();
      var y2 = a2.s < 0 ? a2.negate() : a2.clone();
      if (x2.compareTo(y2) < 0) {
        var t3 = x2;
        x2 = y2;
        y2 = t3;
      }
      var i2 = x2.getLowestSetBit();
      var g2 = y2.getLowestSetBit();
      if (g2 < 0) {
        return x2;
      }
      if (i2 < g2) {
        g2 = i2;
      }
      if (g2 > 0) {
        x2.rShiftTo(g2, x2);
        y2.rShiftTo(g2, y2);
      }
      while (x2.signum() > 0) {
        if ((i2 = x2.getLowestSetBit()) > 0) {
          x2.rShiftTo(i2, x2);
        }
        if ((i2 = y2.getLowestSetBit()) > 0) {
          y2.rShiftTo(i2, y2);
        }
        if (x2.compareTo(y2) >= 0) {
          x2.subTo(y2, x2);
          x2.rShiftTo(1, x2);
        } else {
          y2.subTo(x2, y2);
          y2.rShiftTo(1, y2);
        }
      }
      if (g2 > 0) {
        y2.lShiftTo(g2, y2);
      }
      return y2;
    };
    BigInteger2.prototype.isProbablePrime = function(t3) {
      var i2;
      var x2 = this.abs();
      if (x2.t == 1 && x2[0] <= lowprimes[lowprimes.length - 1]) {
        for (i2 = 0; i2 < lowprimes.length; ++i2) {
          if (x2[0] == lowprimes[i2]) {
            return true;
          }
        }
        return false;
      }
      if (x2.isEven()) {
        return false;
      }
      i2 = 1;
      while (i2 < lowprimes.length) {
        var m2 = lowprimes[i2];
        var j2 = i2 + 1;
        while (j2 < lowprimes.length && m2 < lplim) {
          m2 *= lowprimes[j2++];
        }
        m2 = x2.modInt(m2);
        while (i2 < j2) {
          if (m2 % lowprimes[i2++] == 0) {
            return false;
          }
        }
      }
      return x2.millerRabin(t3);
    };
    BigInteger2.prototype.copyTo = function(r2) {
      for (var i2 = this.t - 1; i2 >= 0; --i2) {
        r2[i2] = this[i2];
      }
      r2.t = this.t;
      r2.s = this.s;
    };
    BigInteger2.prototype.fromInt = function(x2) {
      this.t = 1;
      this.s = x2 < 0 ? -1 : 0;
      if (x2 > 0) {
        this[0] = x2;
      } else if (x2 < -1) {
        this[0] = x2 + this.DV;
      } else {
        this.t = 0;
      }
    };
    BigInteger2.prototype.fromString = function(s2, b2) {
      var k2;
      if (b2 == 16) {
        k2 = 4;
      } else if (b2 == 8) {
        k2 = 3;
      } else if (b2 == 256) {
        k2 = 8;
      } else if (b2 == 2) {
        k2 = 1;
      } else if (b2 == 32) {
        k2 = 5;
      } else if (b2 == 4) {
        k2 = 2;
      } else {
        this.fromRadix(s2, b2);
        return;
      }
      this.t = 0;
      this.s = 0;
      var i2 = s2.length;
      var mi = false;
      var sh = 0;
      while (--i2 >= 0) {
        var x2 = k2 == 8 ? +s2[i2] & 255 : intAt(s2, i2);
        if (x2 < 0) {
          if (s2.charAt(i2) == "-") {
            mi = true;
          }
          continue;
        }
        mi = false;
        if (sh == 0) {
          this[this.t++] = x2;
        } else if (sh + k2 > this.DB) {
          this[this.t - 1] |= (x2 & (1 << this.DB - sh) - 1) << sh;
          this[this.t++] = x2 >> this.DB - sh;
        } else {
          this[this.t - 1] |= x2 << sh;
        }
        sh += k2;
        if (sh >= this.DB) {
          sh -= this.DB;
        }
      }
      if (k2 == 8 && (+s2[0] & 128) != 0) {
        this.s = -1;
        if (sh > 0) {
          this[this.t - 1] |= (1 << this.DB - sh) - 1 << sh;
        }
      }
      this.clamp();
      if (mi) {
        BigInteger2.ZERO.subTo(this, this);
      }
    };
    BigInteger2.prototype.clamp = function() {
      var c2 = this.s & this.DM;
      while (this.t > 0 && this[this.t - 1] == c2) {
        --this.t;
      }
    };
    BigInteger2.prototype.dlShiftTo = function(n2, r2) {
      var i2;
      for (i2 = this.t - 1; i2 >= 0; --i2) {
        r2[i2 + n2] = this[i2];
      }
      for (i2 = n2 - 1; i2 >= 0; --i2) {
        r2[i2] = 0;
      }
      r2.t = this.t + n2;
      r2.s = this.s;
    };
    BigInteger2.prototype.drShiftTo = function(n2, r2) {
      for (var i2 = n2; i2 < this.t; ++i2) {
        r2[i2 - n2] = this[i2];
      }
      r2.t = Math.max(this.t - n2, 0);
      r2.s = this.s;
    };
    BigInteger2.prototype.lShiftTo = function(n2, r2) {
      var bs = n2 % this.DB;
      var cbs = this.DB - bs;
      var bm = (1 << cbs) - 1;
      var ds = Math.floor(n2 / this.DB);
      var c2 = this.s << bs & this.DM;
      for (var i2 = this.t - 1; i2 >= 0; --i2) {
        r2[i2 + ds + 1] = this[i2] >> cbs | c2;
        c2 = (this[i2] & bm) << bs;
      }
      for (var i2 = ds - 1; i2 >= 0; --i2) {
        r2[i2] = 0;
      }
      r2[ds] = c2;
      r2.t = this.t + ds + 1;
      r2.s = this.s;
      r2.clamp();
    };
    BigInteger2.prototype.rShiftTo = function(n2, r2) {
      r2.s = this.s;
      var ds = Math.floor(n2 / this.DB);
      if (ds >= this.t) {
        r2.t = 0;
        return;
      }
      var bs = n2 % this.DB;
      var cbs = this.DB - bs;
      var bm = (1 << bs) - 1;
      r2[0] = this[ds] >> bs;
      for (var i2 = ds + 1; i2 < this.t; ++i2) {
        r2[i2 - ds - 1] |= (this[i2] & bm) << cbs;
        r2[i2 - ds] = this[i2] >> bs;
      }
      if (bs > 0) {
        r2[this.t - ds - 1] |= (this.s & bm) << cbs;
      }
      r2.t = this.t - ds;
      r2.clamp();
    };
    BigInteger2.prototype.subTo = function(a2, r2) {
      var i2 = 0;
      var c2 = 0;
      var m2 = Math.min(a2.t, this.t);
      while (i2 < m2) {
        c2 += this[i2] - a2[i2];
        r2[i2++] = c2 & this.DM;
        c2 >>= this.DB;
      }
      if (a2.t < this.t) {
        c2 -= a2.s;
        while (i2 < this.t) {
          c2 += this[i2];
          r2[i2++] = c2 & this.DM;
          c2 >>= this.DB;
        }
        c2 += this.s;
      } else {
        c2 += this.s;
        while (i2 < a2.t) {
          c2 -= a2[i2];
          r2[i2++] = c2 & this.DM;
          c2 >>= this.DB;
        }
        c2 -= a2.s;
      }
      r2.s = c2 < 0 ? -1 : 0;
      if (c2 < -1) {
        r2[i2++] = this.DV + c2;
      } else if (c2 > 0) {
        r2[i2++] = c2;
      }
      r2.t = i2;
      r2.clamp();
    };
    BigInteger2.prototype.multiplyTo = function(a2, r2) {
      var x2 = this.abs();
      var y2 = a2.abs();
      var i2 = x2.t;
      r2.t = i2 + y2.t;
      while (--i2 >= 0) {
        r2[i2] = 0;
      }
      for (i2 = 0; i2 < y2.t; ++i2) {
        r2[i2 + x2.t] = x2.am(0, y2[i2], r2, i2, 0, x2.t);
      }
      r2.s = 0;
      r2.clamp();
      if (this.s != a2.s) {
        BigInteger2.ZERO.subTo(r2, r2);
      }
    };
    BigInteger2.prototype.squareTo = function(r2) {
      var x2 = this.abs();
      var i2 = r2.t = 2 * x2.t;
      while (--i2 >= 0) {
        r2[i2] = 0;
      }
      for (i2 = 0; i2 < x2.t - 1; ++i2) {
        var c2 = x2.am(i2, x2[i2], r2, 2 * i2, 0, 1);
        if ((r2[i2 + x2.t] += x2.am(i2 + 1, 2 * x2[i2], r2, 2 * i2 + 1, c2, x2.t - i2 - 1)) >= x2.DV) {
          r2[i2 + x2.t] -= x2.DV;
          r2[i2 + x2.t + 1] = 1;
        }
      }
      if (r2.t > 0) {
        r2[r2.t - 1] += x2.am(i2, x2[i2], r2, 2 * i2, 0, 1);
      }
      r2.s = 0;
      r2.clamp();
    };
    BigInteger2.prototype.divRemTo = function(m2, q2, r2) {
      var pm = m2.abs();
      if (pm.t <= 0) {
        return;
      }
      var pt3 = this.abs();
      if (pt3.t < pm.t) {
        if (q2 != null) {
          q2.fromInt(0);
        }
        if (r2 != null) {
          this.copyTo(r2);
        }
        return;
      }
      if (r2 == null) {
        r2 = nbi();
      }
      var y2 = nbi();
      var ts = this.s;
      var ms = m2.s;
      var nsh = this.DB - nbits(pm[pm.t - 1]);
      if (nsh > 0) {
        pm.lShiftTo(nsh, y2);
        pt3.lShiftTo(nsh, r2);
      } else {
        pm.copyTo(y2);
        pt3.copyTo(r2);
      }
      var ys = y2.t;
      var y0 = y2[ys - 1];
      if (y0 == 0) {
        return;
      }
      var yt2 = y0 * (1 << this.F1) + (ys > 1 ? y2[ys - 2] >> this.F2 : 0);
      var d1 = this.FV / yt2;
      var d2 = (1 << this.F1) / yt2;
      var e2 = 1 << this.F2;
      var i2 = r2.t;
      var j2 = i2 - ys;
      var t3 = q2 == null ? nbi() : q2;
      y2.dlShiftTo(j2, t3);
      if (r2.compareTo(t3) >= 0) {
        r2[r2.t++] = 1;
        r2.subTo(t3, r2);
      }
      BigInteger2.ONE.dlShiftTo(ys, t3);
      t3.subTo(y2, y2);
      while (y2.t < ys) {
        y2[y2.t++] = 0;
      }
      while (--j2 >= 0) {
        var qd = r2[--i2] == y0 ? this.DM : Math.floor(r2[i2] * d1 + (r2[i2 - 1] + e2) * d2);
        if ((r2[i2] += y2.am(0, qd, r2, j2, 0, ys)) < qd) {
          y2.dlShiftTo(j2, t3);
          r2.subTo(t3, r2);
          while (r2[i2] < --qd) {
            r2.subTo(t3, r2);
          }
        }
      }
      if (q2 != null) {
        r2.drShiftTo(ys, q2);
        if (ts != ms) {
          BigInteger2.ZERO.subTo(q2, q2);
        }
      }
      r2.t = ys;
      r2.clamp();
      if (nsh > 0) {
        r2.rShiftTo(nsh, r2);
      }
      if (ts < 0) {
        BigInteger2.ZERO.subTo(r2, r2);
      }
    };
    BigInteger2.prototype.invDigit = function() {
      if (this.t < 1) {
        return 0;
      }
      var x2 = this[0];
      if ((x2 & 1) == 0) {
        return 0;
      }
      var y2 = x2 & 3;
      y2 = y2 * (2 - (x2 & 15) * y2) & 15;
      y2 = y2 * (2 - (x2 & 255) * y2) & 255;
      y2 = y2 * (2 - ((x2 & 65535) * y2 & 65535)) & 65535;
      y2 = y2 * (2 - x2 * y2 % this.DV) % this.DV;
      return y2 > 0 ? this.DV - y2 : -y2;
    };
    BigInteger2.prototype.isEven = function() {
      return (this.t > 0 ? this[0] & 1 : this.s) == 0;
    };
    BigInteger2.prototype.exp = function(e2, z2) {
      if (e2 > 4294967295 || e2 < 1) {
        return BigInteger2.ONE;
      }
      var r2 = nbi();
      var r22 = nbi();
      var g2 = z2.convert(this);
      var i2 = nbits(e2) - 1;
      g2.copyTo(r2);
      while (--i2 >= 0) {
        z2.sqrTo(r2, r22);
        if ((e2 & 1 << i2) > 0) {
          z2.mulTo(r22, g2, r2);
        } else {
          var t3 = r2;
          r2 = r22;
          r22 = t3;
        }
      }
      return z2.revert(r2);
    };
    BigInteger2.prototype.chunkSize = function(r2) {
      return Math.floor(Math.LN2 * this.DB / Math.log(r2));
    };
    BigInteger2.prototype.toRadix = function(b2) {
      if (b2 == null) {
        b2 = 10;
      }
      if (this.signum() == 0 || b2 < 2 || b2 > 36) {
        return "0";
      }
      var cs = this.chunkSize(b2);
      var a2 = Math.pow(b2, cs);
      var d2 = nbv(a2);
      var y2 = nbi();
      var z2 = nbi();
      var r2 = "";
      this.divRemTo(d2, y2, z2);
      while (y2.signum() > 0) {
        r2 = (a2 + z2.intValue()).toString(b2).substr(1) + r2;
        y2.divRemTo(d2, y2, z2);
      }
      return z2.intValue().toString(b2) + r2;
    };
    BigInteger2.prototype.fromRadix = function(s2, b2) {
      this.fromInt(0);
      if (b2 == null) {
        b2 = 10;
      }
      var cs = this.chunkSize(b2);
      var d2 = Math.pow(b2, cs);
      var mi = false;
      var j2 = 0;
      var w2 = 0;
      for (var i2 = 0; i2 < s2.length; ++i2) {
        var x2 = intAt(s2, i2);
        if (x2 < 0) {
          if (s2.charAt(i2) == "-" && this.signum() == 0) {
            mi = true;
          }
          continue;
        }
        w2 = b2 * w2 + x2;
        if (++j2 >= cs) {
          this.dMultiply(d2);
          this.dAddOffset(w2, 0);
          j2 = 0;
          w2 = 0;
        }
      }
      if (j2 > 0) {
        this.dMultiply(Math.pow(b2, j2));
        this.dAddOffset(w2, 0);
      }
      if (mi) {
        BigInteger2.ZERO.subTo(this, this);
      }
    };
    BigInteger2.prototype.fromNumber = function(a2, b2, c2) {
      if ("number" == typeof b2) {
        if (a2 < 2) {
          this.fromInt(1);
        } else {
          this.fromNumber(a2, c2);
          if (!this.testBit(a2 - 1)) {
            this.bitwiseTo(BigInteger2.ONE.shiftLeft(a2 - 1), op_or, this);
          }
          if (this.isEven()) {
            this.dAddOffset(1, 0);
          }
          while (!this.isProbablePrime(b2)) {
            this.dAddOffset(2, 0);
            if (this.bitLength() > a2) {
              this.subTo(BigInteger2.ONE.shiftLeft(a2 - 1), this);
            }
          }
        }
      } else {
        var x2 = [];
        var t3 = a2 & 7;
        x2.length = (a2 >> 3) + 1;
        b2.nextBytes(x2);
        if (t3 > 0) {
          x2[0] &= (1 << t3) - 1;
        } else {
          x2[0] = 0;
        }
        this.fromString(x2, 256);
      }
    };
    BigInteger2.prototype.bitwiseTo = function(a2, op2, r2) {
      var i2;
      var f2;
      var m2 = Math.min(a2.t, this.t);
      for (i2 = 0; i2 < m2; ++i2) {
        r2[i2] = op2(this[i2], a2[i2]);
      }
      if (a2.t < this.t) {
        f2 = a2.s & this.DM;
        for (i2 = m2; i2 < this.t; ++i2) {
          r2[i2] = op2(this[i2], f2);
        }
        r2.t = this.t;
      } else {
        f2 = this.s & this.DM;
        for (i2 = m2; i2 < a2.t; ++i2) {
          r2[i2] = op2(f2, a2[i2]);
        }
        r2.t = a2.t;
      }
      r2.s = op2(this.s, a2.s);
      r2.clamp();
    };
    BigInteger2.prototype.changeBit = function(n2, op2) {
      var r2 = BigInteger2.ONE.shiftLeft(n2);
      this.bitwiseTo(r2, op2, r2);
      return r2;
    };
    BigInteger2.prototype.addTo = function(a2, r2) {
      var i2 = 0;
      var c2 = 0;
      var m2 = Math.min(a2.t, this.t);
      while (i2 < m2) {
        c2 += this[i2] + a2[i2];
        r2[i2++] = c2 & this.DM;
        c2 >>= this.DB;
      }
      if (a2.t < this.t) {
        c2 += a2.s;
        while (i2 < this.t) {
          c2 += this[i2];
          r2[i2++] = c2 & this.DM;
          c2 >>= this.DB;
        }
        c2 += this.s;
      } else {
        c2 += this.s;
        while (i2 < a2.t) {
          c2 += a2[i2];
          r2[i2++] = c2 & this.DM;
          c2 >>= this.DB;
        }
        c2 += a2.s;
      }
      r2.s = c2 < 0 ? -1 : 0;
      if (c2 > 0) {
        r2[i2++] = c2;
      } else if (c2 < -1) {
        r2[i2++] = this.DV + c2;
      }
      r2.t = i2;
      r2.clamp();
    };
    BigInteger2.prototype.dMultiply = function(n2) {
      this[this.t] = this.am(0, n2 - 1, this, 0, 0, this.t);
      ++this.t;
      this.clamp();
    };
    BigInteger2.prototype.dAddOffset = function(n2, w2) {
      if (n2 == 0) {
        return;
      }
      while (this.t <= w2) {
        this[this.t++] = 0;
      }
      this[w2] += n2;
      while (this[w2] >= this.DV) {
        this[w2] -= this.DV;
        if (++w2 >= this.t) {
          this[this.t++] = 0;
        }
        ++this[w2];
      }
    };
    BigInteger2.prototype.multiplyLowerTo = function(a2, n2, r2) {
      var i2 = Math.min(this.t + a2.t, n2);
      r2.s = 0;
      r2.t = i2;
      while (i2 > 0) {
        r2[--i2] = 0;
      }
      for (var j2 = r2.t - this.t; i2 < j2; ++i2) {
        r2[i2 + this.t] = this.am(0, a2[i2], r2, i2, 0, this.t);
      }
      for (var j2 = Math.min(a2.t, n2); i2 < j2; ++i2) {
        this.am(0, a2[i2], r2, i2, 0, n2 - i2);
      }
      r2.clamp();
    };
    BigInteger2.prototype.multiplyUpperTo = function(a2, n2, r2) {
      --n2;
      var i2 = r2.t = this.t + a2.t - n2;
      r2.s = 0;
      while (--i2 >= 0) {
        r2[i2] = 0;
      }
      for (i2 = Math.max(n2 - this.t, 0); i2 < a2.t; ++i2) {
        r2[this.t + i2 - n2] = this.am(n2 - i2, a2[i2], r2, 0, 0, this.t + i2 - n2);
      }
      r2.clamp();
      r2.drShiftTo(1, r2);
    };
    BigInteger2.prototype.modInt = function(n2) {
      if (n2 <= 0) {
        return 0;
      }
      var d2 = this.DV % n2;
      var r2 = this.s < 0 ? n2 - 1 : 0;
      if (this.t > 0) {
        if (d2 == 0) {
          r2 = this[0] % n2;
        } else {
          for (var i2 = this.t - 1; i2 >= 0; --i2) {
            r2 = (d2 * r2 + this[i2]) % n2;
          }
        }
      }
      return r2;
    };
    BigInteger2.prototype.millerRabin = function(t3) {
      var n1 = this.subtract(BigInteger2.ONE);
      var k2 = n1.getLowestSetBit();
      if (k2 <= 0) {
        return false;
      }
      var r2 = n1.shiftRight(k2);
      t3 = t3 + 1 >> 1;
      if (t3 > lowprimes.length) {
        t3 = lowprimes.length;
      }
      var a2 = nbi();
      for (var i2 = 0; i2 < t3; ++i2) {
        a2.fromInt(lowprimes[Math.floor(Math.random() * lowprimes.length)]);
        var y2 = a2.modPow(r2, this);
        if (y2.compareTo(BigInteger2.ONE) != 0 && y2.compareTo(n1) != 0) {
          var j2 = 1;
          while (j2++ < k2 && y2.compareTo(n1) != 0) {
            y2 = y2.modPowInt(2, this);
            if (y2.compareTo(BigInteger2.ONE) == 0) {
              return false;
            }
          }
          if (y2.compareTo(n1) != 0) {
            return false;
          }
        }
      }
      return true;
    };
    BigInteger2.prototype.square = function() {
      var r2 = nbi();
      this.squareTo(r2);
      return r2;
    };
    BigInteger2.prototype.gcda = function(a2, callback) {
      var x2 = this.s < 0 ? this.negate() : this.clone();
      var y2 = a2.s < 0 ? a2.negate() : a2.clone();
      if (x2.compareTo(y2) < 0) {
        var t3 = x2;
        x2 = y2;
        y2 = t3;
      }
      var i2 = x2.getLowestSetBit();
      var g2 = y2.getLowestSetBit();
      if (g2 < 0) {
        callback(x2);
        return;
      }
      if (i2 < g2) {
        g2 = i2;
      }
      if (g2 > 0) {
        x2.rShiftTo(g2, x2);
        y2.rShiftTo(g2, y2);
      }
      var gcda1 = function() {
        if ((i2 = x2.getLowestSetBit()) > 0) {
          x2.rShiftTo(i2, x2);
        }
        if ((i2 = y2.getLowestSetBit()) > 0) {
          y2.rShiftTo(i2, y2);
        }
        if (x2.compareTo(y2) >= 0) {
          x2.subTo(y2, x2);
          x2.rShiftTo(1, x2);
        } else {
          y2.subTo(x2, y2);
          y2.rShiftTo(1, y2);
        }
        if (!(x2.signum() > 0)) {
          if (g2 > 0) {
            y2.lShiftTo(g2, y2);
          }
          setTimeout(function() {
            callback(y2);
          }, 0);
        } else {
          setTimeout(gcda1, 0);
        }
      };
      setTimeout(gcda1, 10);
    };
    BigInteger2.prototype.fromNumberAsync = function(a2, b2, c2, callback) {
      if ("number" == typeof b2) {
        if (a2 < 2) {
          this.fromInt(1);
        } else {
          this.fromNumber(a2, c2);
          if (!this.testBit(a2 - 1)) {
            this.bitwiseTo(BigInteger2.ONE.shiftLeft(a2 - 1), op_or, this);
          }
          if (this.isEven()) {
            this.dAddOffset(1, 0);
          }
          var bnp_1 = this;
          var bnpfn1_1 = function() {
            bnp_1.dAddOffset(2, 0);
            if (bnp_1.bitLength() > a2) {
              bnp_1.subTo(BigInteger2.ONE.shiftLeft(a2 - 1), bnp_1);
            }
            if (bnp_1.isProbablePrime(b2)) {
              setTimeout(function() {
                callback();
              }, 0);
            } else {
              setTimeout(bnpfn1_1, 0);
            }
          };
          setTimeout(bnpfn1_1, 0);
        }
      } else {
        var x2 = [];
        var t3 = a2 & 7;
        x2.length = (a2 >> 3) + 1;
        b2.nextBytes(x2);
        if (t3 > 0) {
          x2[0] &= (1 << t3) - 1;
        } else {
          x2[0] = 0;
        }
        this.fromString(x2, 256);
      }
    };
    return BigInteger2;
  }()
);
var NullExp = (
  /** @class */
  function() {
    function NullExp2() {
    }
    NullExp2.prototype.convert = function(x2) {
      return x2;
    };
    NullExp2.prototype.revert = function(x2) {
      return x2;
    };
    NullExp2.prototype.mulTo = function(x2, y2, r2) {
      x2.multiplyTo(y2, r2);
    };
    NullExp2.prototype.sqrTo = function(x2, r2) {
      x2.squareTo(r2);
    };
    return NullExp2;
  }()
);
var Classic = (
  /** @class */
  function() {
    function Classic2(m2) {
      this.m = m2;
    }
    Classic2.prototype.convert = function(x2) {
      if (x2.s < 0 || x2.compareTo(this.m) >= 0) {
        return x2.mod(this.m);
      } else {
        return x2;
      }
    };
    Classic2.prototype.revert = function(x2) {
      return x2;
    };
    Classic2.prototype.reduce = function(x2) {
      x2.divRemTo(this.m, null, x2);
    };
    Classic2.prototype.mulTo = function(x2, y2, r2) {
      x2.multiplyTo(y2, r2);
      this.reduce(r2);
    };
    Classic2.prototype.sqrTo = function(x2, r2) {
      x2.squareTo(r2);
      this.reduce(r2);
    };
    return Classic2;
  }()
);
var Montgomery = (
  /** @class */
  function() {
    function Montgomery2(m2) {
      this.m = m2;
      this.mp = m2.invDigit();
      this.mpl = this.mp & 32767;
      this.mph = this.mp >> 15;
      this.um = (1 << m2.DB - 15) - 1;
      this.mt2 = 2 * m2.t;
    }
    Montgomery2.prototype.convert = function(x2) {
      var r2 = nbi();
      x2.abs().dlShiftTo(this.m.t, r2);
      r2.divRemTo(this.m, null, r2);
      if (x2.s < 0 && r2.compareTo(BigInteger.ZERO) > 0) {
        this.m.subTo(r2, r2);
      }
      return r2;
    };
    Montgomery2.prototype.revert = function(x2) {
      var r2 = nbi();
      x2.copyTo(r2);
      this.reduce(r2);
      return r2;
    };
    Montgomery2.prototype.reduce = function(x2) {
      while (x2.t <= this.mt2) {
        x2[x2.t++] = 0;
      }
      for (var i2 = 0; i2 < this.m.t; ++i2) {
        var j2 = x2[i2] & 32767;
        var u0 = j2 * this.mpl + ((j2 * this.mph + (x2[i2] >> 15) * this.mpl & this.um) << 15) & x2.DM;
        j2 = i2 + this.m.t;
        x2[j2] += this.m.am(0, u0, x2, i2, 0, this.m.t);
        while (x2[j2] >= x2.DV) {
          x2[j2] -= x2.DV;
          x2[++j2]++;
        }
      }
      x2.clamp();
      x2.drShiftTo(this.m.t, x2);
      if (x2.compareTo(this.m) >= 0) {
        x2.subTo(this.m, x2);
      }
    };
    Montgomery2.prototype.mulTo = function(x2, y2, r2) {
      x2.multiplyTo(y2, r2);
      this.reduce(r2);
    };
    Montgomery2.prototype.sqrTo = function(x2, r2) {
      x2.squareTo(r2);
      this.reduce(r2);
    };
    return Montgomery2;
  }()
);
var Barrett = (
  /** @class */
  function() {
    function Barrett2(m2) {
      this.m = m2;
      this.r2 = nbi();
      this.q3 = nbi();
      BigInteger.ONE.dlShiftTo(2 * m2.t, this.r2);
      this.mu = this.r2.divide(m2);
    }
    Barrett2.prototype.convert = function(x2) {
      if (x2.s < 0 || x2.t > 2 * this.m.t) {
        return x2.mod(this.m);
      } else if (x2.compareTo(this.m) < 0) {
        return x2;
      } else {
        var r2 = nbi();
        x2.copyTo(r2);
        this.reduce(r2);
        return r2;
      }
    };
    Barrett2.prototype.revert = function(x2) {
      return x2;
    };
    Barrett2.prototype.reduce = function(x2) {
      x2.drShiftTo(this.m.t - 1, this.r2);
      if (x2.t > this.m.t + 1) {
        x2.t = this.m.t + 1;
        x2.clamp();
      }
      this.mu.multiplyUpperTo(this.r2, this.m.t + 1, this.q3);
      this.m.multiplyLowerTo(this.q3, this.m.t + 1, this.r2);
      while (x2.compareTo(this.r2) < 0) {
        x2.dAddOffset(1, this.m.t + 1);
      }
      x2.subTo(this.r2, x2);
      while (x2.compareTo(this.m) >= 0) {
        x2.subTo(this.m, x2);
      }
    };
    Barrett2.prototype.mulTo = function(x2, y2, r2) {
      x2.multiplyTo(y2, r2);
      this.reduce(r2);
    };
    Barrett2.prototype.sqrTo = function(x2, r2) {
      x2.squareTo(r2);
      this.reduce(r2);
    };
    return Barrett2;
  }()
);
function nbi() {
  return new BigInteger(null);
}
function parseBigInt(str, r2) {
  return new BigInteger(str, r2);
}
var inBrowser = typeof navigator !== "undefined";
if (inBrowser && j_lm && navigator.appName == "Microsoft Internet Explorer") {
  BigInteger.prototype.am = function am2(i2, x2, w2, j2, c2, n2) {
    var xl = x2 & 32767;
    var xh = x2 >> 15;
    while (--n2 >= 0) {
      var l2 = this[i2] & 32767;
      var h2 = this[i2++] >> 15;
      var m2 = xh * l2 + h2 * xl;
      l2 = xl * l2 + ((m2 & 32767) << 15) + w2[j2] + (c2 & 1073741823);
      c2 = (l2 >>> 30) + (m2 >>> 15) + xh * h2 + (c2 >>> 30);
      w2[j2++] = l2 & 1073741823;
    }
    return c2;
  };
  dbits = 30;
} else if (inBrowser && j_lm && navigator.appName != "Netscape") {
  BigInteger.prototype.am = function am1(i2, x2, w2, j2, c2, n2) {
    while (--n2 >= 0) {
      var v4 = x2 * this[i2++] + w2[j2] + c2;
      c2 = Math.floor(v4 / 67108864);
      w2[j2++] = v4 & 67108863;
    }
    return c2;
  };
  dbits = 26;
} else {
  BigInteger.prototype.am = function am3(i2, x2, w2, j2, c2, n2) {
    var xl = x2 & 16383;
    var xh = x2 >> 14;
    while (--n2 >= 0) {
      var l2 = this[i2] & 16383;
      var h2 = this[i2++] >> 14;
      var m2 = xh * l2 + h2 * xl;
      l2 = xl * l2 + ((m2 & 16383) << 14) + w2[j2] + c2;
      c2 = (l2 >> 28) + (m2 >> 14) + xh * h2;
      w2[j2++] = l2 & 268435455;
    }
    return c2;
  };
  dbits = 28;
}
BigInteger.prototype.DB = dbits;
BigInteger.prototype.DM = (1 << dbits) - 1;
BigInteger.prototype.DV = 1 << dbits;
var BI_FP = 52;
BigInteger.prototype.FV = Math.pow(2, BI_FP);
BigInteger.prototype.F1 = BI_FP - dbits;
BigInteger.prototype.F2 = 2 * dbits - BI_FP;
var BI_RC = [];
var rr;
var vv;
rr = "0".charCodeAt(0);
for (vv = 0; vv <= 9; ++vv) {
  BI_RC[rr++] = vv;
}
rr = "a".charCodeAt(0);
for (vv = 10; vv < 36; ++vv) {
  BI_RC[rr++] = vv;
}
rr = "A".charCodeAt(0);
for (vv = 10; vv < 36; ++vv) {
  BI_RC[rr++] = vv;
}
function intAt(s2, i2) {
  var c2 = BI_RC[s2.charCodeAt(i2)];
  return c2 == null ? -1 : c2;
}
function nbv(i2) {
  var r2 = nbi();
  r2.fromInt(i2);
  return r2;
}
function nbits(x2) {
  var r2 = 1;
  var t3;
  if ((t3 = x2 >>> 16) != 0) {
    x2 = t3;
    r2 += 16;
  }
  if ((t3 = x2 >> 8) != 0) {
    x2 = t3;
    r2 += 8;
  }
  if ((t3 = x2 >> 4) != 0) {
    x2 = t3;
    r2 += 4;
  }
  if ((t3 = x2 >> 2) != 0) {
    x2 = t3;
    r2 += 2;
  }
  if ((t3 = x2 >> 1) != 0) {
    x2 = t3;
    r2 += 1;
  }
  return r2;
}
BigInteger.ZERO = nbv(0);
BigInteger.ONE = nbv(1);
var Arcfour = (
  /** @class */
  function() {
    function Arcfour2() {
      this.i = 0;
      this.j = 0;
      this.S = [];
    }
    Arcfour2.prototype.init = function(key) {
      var i2;
      var j2;
      var t3;
      for (i2 = 0; i2 < 256; ++i2) {
        this.S[i2] = i2;
      }
      j2 = 0;
      for (i2 = 0; i2 < 256; ++i2) {
        j2 = j2 + this.S[i2] + key[i2 % key.length] & 255;
        t3 = this.S[i2];
        this.S[i2] = this.S[j2];
        this.S[j2] = t3;
      }
      this.i = 0;
      this.j = 0;
    };
    Arcfour2.prototype.next = function() {
      var t3;
      this.i = this.i + 1 & 255;
      this.j = this.j + this.S[this.i] & 255;
      t3 = this.S[this.i];
      this.S[this.i] = this.S[this.j];
      this.S[this.j] = t3;
      return this.S[t3 + this.S[this.i] & 255];
    };
    return Arcfour2;
  }()
);
function prng_newstate() {
  return new Arcfour();
}
var rng_psize = 256;
var rng_state;
var rng_pool = null;
var rng_pptr;
if (rng_pool == null) {
  rng_pool = [];
  rng_pptr = 0;
  var t$1 = void 0;
  if (typeof window !== "undefined" && window.crypto && window.crypto.getRandomValues) {
    var z$2 = new Uint32Array(256);
    window.crypto.getRandomValues(z$2);
    for (t$1 = 0; t$1 < z$2.length; ++t$1) {
      rng_pool[rng_pptr++] = z$2[t$1] & 255;
    }
  }
  var count$1 = 0;
  var onMouseMoveListener_1 = function(ev) {
    count$1 = count$1 || 0;
    if (count$1 >= 256 || rng_pptr >= rng_psize) {
      if (window.removeEventListener) {
        window.removeEventListener("mousemove", onMouseMoveListener_1, false);
      } else if (window.detachEvent) {
        window.detachEvent("onmousemove", onMouseMoveListener_1);
      }
      return;
    }
    try {
      var mouseCoordinates = ev.x + ev.y;
      rng_pool[rng_pptr++] = mouseCoordinates & 255;
      count$1 += 1;
    } catch (e2) {
    }
  };
  if (typeof window !== "undefined") {
    if (window.addEventListener) {
      window.addEventListener("mousemove", onMouseMoveListener_1, false);
    } else if (window.attachEvent) {
      window.attachEvent("onmousemove", onMouseMoveListener_1);
    }
  }
}
function rng_get_byte() {
  if (rng_state == null) {
    rng_state = prng_newstate();
    while (rng_pptr < rng_psize) {
      var random2 = Math.floor(65536 * Math.random());
      rng_pool[rng_pptr++] = random2 & 255;
    }
    rng_state.init(rng_pool);
    for (rng_pptr = 0; rng_pptr < rng_pool.length; ++rng_pptr) {
      rng_pool[rng_pptr] = 0;
    }
    rng_pptr = 0;
  }
  return rng_state.next();
}
var SecureRandom = (
  /** @class */
  function() {
    function SecureRandom2() {
    }
    SecureRandom2.prototype.nextBytes = function(ba) {
      for (var i2 = 0; i2 < ba.length; ++i2) {
        ba[i2] = rng_get_byte();
      }
    };
    return SecureRandom2;
  }()
);
function pkcs1pad1(s2, n2) {
  if (n2 < s2.length + 22) {
    console.error("Message too long for RSA");
    return null;
  }
  var len2 = n2 - s2.length - 6;
  var filler = "";
  for (var f2 = 0; f2 < len2; f2 += 2) {
    filler += "ff";
  }
  var m2 = "0001" + filler + "00" + s2;
  return parseBigInt(m2, 16);
}
function pkcs1pad2(s2, n2) {
  if (n2 < s2.length + 11) {
    console.error("Message too long for RSA");
    return null;
  }
  var ba = [];
  var i2 = s2.length - 1;
  while (i2 >= 0 && n2 > 0) {
    var c2 = s2.charCodeAt(i2--);
    if (c2 < 128) {
      ba[--n2] = c2;
    } else if (c2 > 127 && c2 < 2048) {
      ba[--n2] = c2 & 63 | 128;
      ba[--n2] = c2 >> 6 | 192;
    } else {
      ba[--n2] = c2 & 63 | 128;
      ba[--n2] = c2 >> 6 & 63 | 128;
      ba[--n2] = c2 >> 12 | 224;
    }
  }
  ba[--n2] = 0;
  var rng = new SecureRandom();
  var x2 = [];
  while (n2 > 2) {
    x2[0] = 0;
    while (x2[0] == 0) {
      rng.nextBytes(x2);
    }
    ba[--n2] = x2[0];
  }
  ba[--n2] = 2;
  ba[--n2] = 0;
  return new BigInteger(ba);
}
var RSAKey = (
  /** @class */
  function() {
    function RSAKey2() {
      this.n = null;
      this.e = 0;
      this.d = null;
      this.p = null;
      this.q = null;
      this.dmp1 = null;
      this.dmq1 = null;
      this.coeff = null;
    }
    RSAKey2.prototype.doPublic = function(x2) {
      return x2.modPowInt(this.e, this.n);
    };
    RSAKey2.prototype.doPrivate = function(x2) {
      if (this.p == null || this.q == null) {
        return x2.modPow(this.d, this.n);
      }
      var xp = x2.mod(this.p).modPow(this.dmp1, this.p);
      var xq = x2.mod(this.q).modPow(this.dmq1, this.q);
      while (xp.compareTo(xq) < 0) {
        xp = xp.add(this.p);
      }
      return xp.subtract(xq).multiply(this.coeff).mod(this.p).multiply(this.q).add(xq);
    };
    RSAKey2.prototype.setPublic = function(N2, E2) {
      if (N2 != null && E2 != null && N2.length > 0 && E2.length > 0) {
        this.n = parseBigInt(N2, 16);
        this.e = parseInt(E2, 16);
      } else {
        console.error("Invalid RSA public key");
      }
    };
    RSAKey2.prototype.encrypt = function(text2) {
      var maxLength = this.n.bitLength() + 7 >> 3;
      var m2 = pkcs1pad2(text2, maxLength);
      if (m2 == null) {
        return null;
      }
      var c2 = this.doPublic(m2);
      if (c2 == null) {
        return null;
      }
      var h2 = c2.toString(16);
      var length = h2.length;
      for (var i2 = 0; i2 < maxLength * 2 - length; i2++) {
        h2 = "0" + h2;
      }
      return h2;
    };
    RSAKey2.prototype.setPrivate = function(N2, E2, D2) {
      if (N2 != null && E2 != null && N2.length > 0 && E2.length > 0) {
        this.n = parseBigInt(N2, 16);
        this.e = parseInt(E2, 16);
        this.d = parseBigInt(D2, 16);
      } else {
        console.error("Invalid RSA private key");
      }
    };
    RSAKey2.prototype.setPrivateEx = function(N2, E2, D2, P2, Q2, DP, DQ, C2) {
      if (N2 != null && E2 != null && N2.length > 0 && E2.length > 0) {
        this.n = parseBigInt(N2, 16);
        this.e = parseInt(E2, 16);
        this.d = parseBigInt(D2, 16);
        this.p = parseBigInt(P2, 16);
        this.q = parseBigInt(Q2, 16);
        this.dmp1 = parseBigInt(DP, 16);
        this.dmq1 = parseBigInt(DQ, 16);
        this.coeff = parseBigInt(C2, 16);
      } else {
        console.error("Invalid RSA private key");
      }
    };
    RSAKey2.prototype.generate = function(B3, E2) {
      var rng = new SecureRandom();
      var qs = B3 >> 1;
      this.e = parseInt(E2, 16);
      var ee2 = new BigInteger(E2, 16);
      for (; ; ) {
        for (; ; ) {
          this.p = new BigInteger(B3 - qs, 1, rng);
          if (this.p.subtract(BigInteger.ONE).gcd(ee2).compareTo(BigInteger.ONE) == 0 && this.p.isProbablePrime(10)) {
            break;
          }
        }
        for (; ; ) {
          this.q = new BigInteger(qs, 1, rng);
          if (this.q.subtract(BigInteger.ONE).gcd(ee2).compareTo(BigInteger.ONE) == 0 && this.q.isProbablePrime(10)) {
            break;
          }
        }
        if (this.p.compareTo(this.q) <= 0) {
          var t3 = this.p;
          this.p = this.q;
          this.q = t3;
        }
        var p1 = this.p.subtract(BigInteger.ONE);
        var q1 = this.q.subtract(BigInteger.ONE);
        var phi = p1.multiply(q1);
        if (phi.gcd(ee2).compareTo(BigInteger.ONE) == 0) {
          this.n = this.p.multiply(this.q);
          this.d = ee2.modInverse(phi);
          this.dmp1 = this.d.mod(p1);
          this.dmq1 = this.d.mod(q1);
          this.coeff = this.q.modInverse(this.p);
          break;
        }
      }
    };
    RSAKey2.prototype.decrypt = function(ctext) {
      var c2 = parseBigInt(ctext, 16);
      var m2 = this.doPrivate(c2);
      if (m2 == null) {
        return null;
      }
      return pkcs1unpad2(m2, this.n.bitLength() + 7 >> 3);
    };
    RSAKey2.prototype.generateAsync = function(B3, E2, callback) {
      var rng = new SecureRandom();
      var qs = B3 >> 1;
      this.e = parseInt(E2, 16);
      var ee2 = new BigInteger(E2, 16);
      var rsa = this;
      var loop1 = function() {
        var loop4 = function() {
          if (rsa.p.compareTo(rsa.q) <= 0) {
            var t3 = rsa.p;
            rsa.p = rsa.q;
            rsa.q = t3;
          }
          var p1 = rsa.p.subtract(BigInteger.ONE);
          var q1 = rsa.q.subtract(BigInteger.ONE);
          var phi = p1.multiply(q1);
          if (phi.gcd(ee2).compareTo(BigInteger.ONE) == 0) {
            rsa.n = rsa.p.multiply(rsa.q);
            rsa.d = ee2.modInverse(phi);
            rsa.dmp1 = rsa.d.mod(p1);
            rsa.dmq1 = rsa.d.mod(q1);
            rsa.coeff = rsa.q.modInverse(rsa.p);
            setTimeout(function() {
              callback();
            }, 0);
          } else {
            setTimeout(loop1, 0);
          }
        };
        var loop3 = function() {
          rsa.q = nbi();
          rsa.q.fromNumberAsync(qs, 1, rng, function() {
            rsa.q.subtract(BigInteger.ONE).gcda(ee2, function(r2) {
              if (r2.compareTo(BigInteger.ONE) == 0 && rsa.q.isProbablePrime(10)) {
                setTimeout(loop4, 0);
              } else {
                setTimeout(loop3, 0);
              }
            });
          });
        };
        var loop2 = function() {
          rsa.p = nbi();
          rsa.p.fromNumberAsync(B3 - qs, 1, rng, function() {
            rsa.p.subtract(BigInteger.ONE).gcda(ee2, function(r2) {
              if (r2.compareTo(BigInteger.ONE) == 0 && rsa.p.isProbablePrime(10)) {
                setTimeout(loop3, 0);
              } else {
                setTimeout(loop2, 0);
              }
            });
          });
        };
        setTimeout(loop2, 0);
      };
      setTimeout(loop1, 0);
    };
    RSAKey2.prototype.sign = function(text2, digestMethod, digestName) {
      var header = getDigestHeader(digestName);
      var digest = header + digestMethod(text2).toString();
      var m2 = pkcs1pad1(digest, this.n.bitLength() / 4);
      if (m2 == null) {
        return null;
      }
      var c2 = this.doPrivate(m2);
      if (c2 == null) {
        return null;
      }
      var h2 = c2.toString(16);
      if ((h2.length & 1) == 0) {
        return h2;
      } else {
        return "0" + h2;
      }
    };
    RSAKey2.prototype.verify = function(text2, signature, digestMethod) {
      var c2 = parseBigInt(signature, 16);
      var m2 = this.doPublic(c2);
      if (m2 == null) {
        return null;
      }
      var unpadded = m2.toString(16).replace(/^1f+00/, "");
      var digest = removeDigestHeader(unpadded);
      return digest == digestMethod(text2).toString();
    };
    return RSAKey2;
  }()
);
function pkcs1unpad2(d2, n2) {
  var b2 = d2.toByteArray();
  var i2 = 0;
  while (i2 < b2.length && b2[i2] == 0) {
    ++i2;
  }
  if (b2.length - i2 != n2 - 1 || b2[i2] != 2) {
    return null;
  }
  ++i2;
  while (b2[i2] != 0) {
    if (++i2 >= b2.length) {
      return null;
    }
  }
  var ret = "";
  while (++i2 < b2.length) {
    var c2 = b2[i2] & 255;
    if (c2 < 128) {
      ret += String.fromCharCode(c2);
    } else if (c2 > 191 && c2 < 224) {
      ret += String.fromCharCode((c2 & 31) << 6 | b2[i2 + 1] & 63);
      ++i2;
    } else {
      ret += String.fromCharCode((c2 & 15) << 12 | (b2[i2 + 1] & 63) << 6 | b2[i2 + 2] & 63);
      i2 += 2;
    }
  }
  return ret;
}
var DIGEST_HEADERS = {
  md2: "3020300c06082a864886f70d020205000410",
  md5: "3020300c06082a864886f70d020505000410",
  sha1: "3021300906052b0e03021a05000414",
  sha224: "302d300d06096086480165030402040500041c",
  sha256: "3031300d060960864801650304020105000420",
  sha384: "3041300d060960864801650304020205000430",
  sha512: "3051300d060960864801650304020305000440",
  ripemd160: "3021300906052b2403020105000414"
};
function getDigestHeader(name) {
  return DIGEST_HEADERS[name] || "";
}
function removeDigestHeader(str) {
  for (var name_1 in DIGEST_HEADERS) {
    if (DIGEST_HEADERS.hasOwnProperty(name_1)) {
      var header = DIGEST_HEADERS[name_1];
      var len2 = header.length;
      if (str.substr(0, len2) == header) {
        return str.substr(len2);
      }
    }
  }
  return str;
}
/*!
Copyright (c) 2011, Yahoo! Inc. All rights reserved.
Code licensed under the BSD License:
http://developer.yahoo.com/yui/license.html
version: 2.9.0
*/
var YAHOO = {};
YAHOO.lang = {
  /**
   * Utility to set up the prototype, constructor and superclass properties to
   * support an inheritance strategy that can chain constructors and methods.
   * Static members will not be inherited.
   *
   * @method extend
   * @static
   * @param {Function} subc   the object to modify
   * @param {Function} superc the object to inherit
   * @param {Object} overrides  additional properties/methods to add to the
   *                              subclass prototype.  These will override the
   *                              matching items obtained from the superclass
   *                              if present.
   */
  extend: function(subc, superc, overrides) {
    if (!superc || !subc) {
      throw new Error("YAHOO.lang.extend failed, please check that all dependencies are included.");
    }
    var F2 = function() {
    };
    F2.prototype = superc.prototype;
    subc.prototype = new F2();
    subc.prototype.constructor = subc;
    subc.superclass = superc.prototype;
    if (superc.prototype.constructor == Object.prototype.constructor) {
      superc.prototype.constructor = superc;
    }
    if (overrides) {
      var i2;
      for (i2 in overrides) {
        subc.prototype[i2] = overrides[i2];
      }
      var _IEEnumFix = function() {
      }, ADD = ["toString", "valueOf"];
      try {
        if (/MSIE/.test(navigator.userAgent)) {
          _IEEnumFix = function(r2, s2) {
            for (i2 = 0; i2 < ADD.length; i2 = i2 + 1) {
              var fname = ADD[i2], f2 = s2[fname];
              if (typeof f2 === "function" && f2 != Object.prototype[fname]) {
                r2[fname] = f2;
              }
            }
          };
        }
      } catch (ex) {
      }
      _IEEnumFix(subc.prototype, overrides);
    }
  }
};
/**
 * @fileOverview
 * @name asn1-1.0.js
 * @author Kenji Urushima kenji.urushima@gmail.com
 * @version asn1 1.0.13 (2017-Jun-02)
 * @since jsrsasign 2.1
 * @license <a href="https://kjur.github.io/jsrsasign/license/">MIT License</a>
 */
var KJUR = {};
if (typeof KJUR.asn1 == "undefined" || !KJUR.asn1)
  KJUR.asn1 = {};
KJUR.asn1.ASN1Util = new function() {
  this.integerToByteHex = function(i2) {
    var h2 = i2.toString(16);
    if (h2.length % 2 == 1)
      h2 = "0" + h2;
    return h2;
  };
  this.bigIntToMinTwosComplementsHex = function(bigIntegerValue) {
    var h2 = bigIntegerValue.toString(16);
    if (h2.substr(0, 1) != "-") {
      if (h2.length % 2 == 1) {
        h2 = "0" + h2;
      } else {
        if (!h2.match(/^[0-7]/)) {
          h2 = "00" + h2;
        }
      }
    } else {
      var hPos = h2.substr(1);
      var xorLen = hPos.length;
      if (xorLen % 2 == 1) {
        xorLen += 1;
      } else {
        if (!h2.match(/^[0-7]/)) {
          xorLen += 2;
        }
      }
      var hMask = "";
      for (var i2 = 0; i2 < xorLen; i2++) {
        hMask += "f";
      }
      var biMask = new BigInteger(hMask, 16);
      var biNeg = biMask.xor(bigIntegerValue).add(BigInteger.ONE);
      h2 = biNeg.toString(16).replace(/^-/, "");
    }
    return h2;
  };
  this.getPEMStringFromHex = function(dataHex, pemHeader) {
    return hextopem(dataHex, pemHeader);
  };
  this.newObject = function(param) {
    var _KJUR = KJUR, _KJUR_asn1 = _KJUR.asn1, _DERBoolean = _KJUR_asn1.DERBoolean, _DERInteger = _KJUR_asn1.DERInteger, _DERBitString = _KJUR_asn1.DERBitString, _DEROctetString = _KJUR_asn1.DEROctetString, _DERNull = _KJUR_asn1.DERNull, _DERObjectIdentifier = _KJUR_asn1.DERObjectIdentifier, _DEREnumerated = _KJUR_asn1.DEREnumerated, _DERUTF8String = _KJUR_asn1.DERUTF8String, _DERNumericString = _KJUR_asn1.DERNumericString, _DERPrintableString = _KJUR_asn1.DERPrintableString, _DERTeletexString = _KJUR_asn1.DERTeletexString, _DERIA5String = _KJUR_asn1.DERIA5String, _DERUTCTime = _KJUR_asn1.DERUTCTime, _DERGeneralizedTime = _KJUR_asn1.DERGeneralizedTime, _DERSequence = _KJUR_asn1.DERSequence, _DERSet = _KJUR_asn1.DERSet, _DERTaggedObject = _KJUR_asn1.DERTaggedObject, _newObject = _KJUR_asn1.ASN1Util.newObject;
    var keys4 = Object.keys(param);
    if (keys4.length != 1)
      throw "key of param shall be only one.";
    var key = keys4[0];
    if (":bool:int:bitstr:octstr:null:oid:enum:utf8str:numstr:prnstr:telstr:ia5str:utctime:gentime:seq:set:tag:".indexOf(":" + key + ":") == -1)
      throw "undefined key: " + key;
    if (key == "bool")
      return new _DERBoolean(param[key]);
    if (key == "int")
      return new _DERInteger(param[key]);
    if (key == "bitstr")
      return new _DERBitString(param[key]);
    if (key == "octstr")
      return new _DEROctetString(param[key]);
    if (key == "null")
      return new _DERNull(param[key]);
    if (key == "oid")
      return new _DERObjectIdentifier(param[key]);
    if (key == "enum")
      return new _DEREnumerated(param[key]);
    if (key == "utf8str")
      return new _DERUTF8String(param[key]);
    if (key == "numstr")
      return new _DERNumericString(param[key]);
    if (key == "prnstr")
      return new _DERPrintableString(param[key]);
    if (key == "telstr")
      return new _DERTeletexString(param[key]);
    if (key == "ia5str")
      return new _DERIA5String(param[key]);
    if (key == "utctime")
      return new _DERUTCTime(param[key]);
    if (key == "gentime")
      return new _DERGeneralizedTime(param[key]);
    if (key == "seq") {
      var paramList = param[key];
      var a2 = [];
      for (var i2 = 0; i2 < paramList.length; i2++) {
        var asn1Obj = _newObject(paramList[i2]);
        a2.push(asn1Obj);
      }
      return new _DERSequence({ "array": a2 });
    }
    if (key == "set") {
      var paramList = param[key];
      var a2 = [];
      for (var i2 = 0; i2 < paramList.length; i2++) {
        var asn1Obj = _newObject(paramList[i2]);
        a2.push(asn1Obj);
      }
      return new _DERSet({ "array": a2 });
    }
    if (key == "tag") {
      var tagParam = param[key];
      if (Object.prototype.toString.call(tagParam) === "[object Array]" && tagParam.length == 3) {
        var obj = _newObject(tagParam[2]);
        return new _DERTaggedObject({
          tag: tagParam[0],
          explicit: tagParam[1],
          obj
        });
      } else {
        var newParam = {};
        if (tagParam.explicit !== void 0)
          newParam.explicit = tagParam.explicit;
        if (tagParam.tag !== void 0)
          newParam.tag = tagParam.tag;
        if (tagParam.obj === void 0)
          throw "obj shall be specified for 'tag'.";
        newParam.obj = _newObject(tagParam.obj);
        return new _DERTaggedObject(newParam);
      }
    }
  };
  this.jsonToASN1HEX = function(param) {
    var asn1Obj = this.newObject(param);
    return asn1Obj.getEncodedHex();
  };
}();
KJUR.asn1.ASN1Util.oidHexToInt = function(hex) {
  var s2 = "";
  var i01 = parseInt(hex.substr(0, 2), 16);
  var i0 = Math.floor(i01 / 40);
  var i1 = i01 % 40;
  var s2 = i0 + "." + i1;
  var binbuf = "";
  for (var i2 = 2; i2 < hex.length; i2 += 2) {
    var value = parseInt(hex.substr(i2, 2), 16);
    var bin = ("00000000" + value.toString(2)).slice(-8);
    binbuf = binbuf + bin.substr(1, 7);
    if (bin.substr(0, 1) == "0") {
      var bi = new BigInteger(binbuf, 2);
      s2 = s2 + "." + bi.toString(10);
      binbuf = "";
    }
  }
  return s2;
};
KJUR.asn1.ASN1Util.oidIntToHex = function(oidString) {
  var itox = function(i3) {
    var h3 = i3.toString(16);
    if (h3.length == 1)
      h3 = "0" + h3;
    return h3;
  };
  var roidtox = function(roid) {
    var h3 = "";
    var bi = new BigInteger(roid, 10);
    var b2 = bi.toString(2);
    var padLen = 7 - b2.length % 7;
    if (padLen == 7)
      padLen = 0;
    var bPad = "";
    for (var i3 = 0; i3 < padLen; i3++)
      bPad += "0";
    b2 = bPad + b2;
    for (var i3 = 0; i3 < b2.length - 1; i3 += 7) {
      var b8 = b2.substr(i3, 7);
      if (i3 != b2.length - 7)
        b8 = "1" + b8;
      h3 += itox(parseInt(b8, 2));
    }
    return h3;
  };
  if (!oidString.match(/^[0-9.]+$/)) {
    throw "malformed oid string: " + oidString;
  }
  var h2 = "";
  var a2 = oidString.split(".");
  var i0 = parseInt(a2[0]) * 40 + parseInt(a2[1]);
  h2 += itox(i0);
  a2.splice(0, 2);
  for (var i2 = 0; i2 < a2.length; i2++) {
    h2 += roidtox(a2[i2]);
  }
  return h2;
};
KJUR.asn1.ASN1Object = function() {
  var hV = "";
  this.getLengthHexFromValue = function() {
    if (typeof this.hV == "undefined" || this.hV == null) {
      throw "this.hV is null or undefined.";
    }
    if (this.hV.length % 2 == 1) {
      throw "value hex must be even length: n=" + hV.length + ",v=" + this.hV;
    }
    var n2 = this.hV.length / 2;
    var hN = n2.toString(16);
    if (hN.length % 2 == 1) {
      hN = "0" + hN;
    }
    if (n2 < 128) {
      return hN;
    } else {
      var hNlen = hN.length / 2;
      if (hNlen > 15) {
        throw "ASN.1 length too long to represent by 8x: n = " + n2.toString(16);
      }
      var head = 128 + hNlen;
      return head.toString(16) + hN;
    }
  };
  this.getEncodedHex = function() {
    if (this.hTLV == null || this.isModified) {
      this.hV = this.getFreshValueHex();
      this.hL = this.getLengthHexFromValue();
      this.hTLV = this.hT + this.hL + this.hV;
      this.isModified = false;
    }
    return this.hTLV;
  };
  this.getValueHex = function() {
    this.getEncodedHex();
    return this.hV;
  };
  this.getFreshValueHex = function() {
    return "";
  };
};
KJUR.asn1.DERAbstractString = function(params) {
  KJUR.asn1.DERAbstractString.superclass.constructor.call(this);
  this.getString = function() {
    return this.s;
  };
  this.setString = function(newS) {
    this.hTLV = null;
    this.isModified = true;
    this.s = newS;
    this.hV = stohex(this.s);
  };
  this.setStringHex = function(newHexString) {
    this.hTLV = null;
    this.isModified = true;
    this.s = null;
    this.hV = newHexString;
  };
  this.getFreshValueHex = function() {
    return this.hV;
  };
  if (typeof params != "undefined") {
    if (typeof params == "string") {
      this.setString(params);
    } else if (typeof params["str"] != "undefined") {
      this.setString(params["str"]);
    } else if (typeof params["hex"] != "undefined") {
      this.setStringHex(params["hex"]);
    }
  }
};
YAHOO.lang.extend(KJUR.asn1.DERAbstractString, KJUR.asn1.ASN1Object);
KJUR.asn1.DERAbstractTime = function(params) {
  KJUR.asn1.DERAbstractTime.superclass.constructor.call(this);
  this.localDateToUTC = function(d2) {
    utc = d2.getTime() + d2.getTimezoneOffset() * 6e4;
    var utcDate = new Date(utc);
    return utcDate;
  };
  this.formatDate = function(dateObject, type2, withMillis) {
    var pad2 = this.zeroPadding;
    var d2 = this.localDateToUTC(dateObject);
    var year = String(d2.getFullYear());
    if (type2 == "utc")
      year = year.substr(2, 2);
    var month = pad2(String(d2.getMonth() + 1), 2);
    var day = pad2(String(d2.getDate()), 2);
    var hour = pad2(String(d2.getHours()), 2);
    var min3 = pad2(String(d2.getMinutes()), 2);
    var sec = pad2(String(d2.getSeconds()), 2);
    var s2 = year + month + day + hour + min3 + sec;
    if (withMillis === true) {
      var millis = d2.getMilliseconds();
      if (millis != 0) {
        var sMillis = pad2(String(millis), 3);
        sMillis = sMillis.replace(/[0]+$/, "");
        s2 = s2 + "." + sMillis;
      }
    }
    return s2 + "Z";
  };
  this.zeroPadding = function(s2, len2) {
    if (s2.length >= len2)
      return s2;
    return new Array(len2 - s2.length + 1).join("0") + s2;
  };
  this.getString = function() {
    return this.s;
  };
  this.setString = function(newS) {
    this.hTLV = null;
    this.isModified = true;
    this.s = newS;
    this.hV = stohex(newS);
  };
  this.setByDateValue = function(year, month, day, hour, min3, sec) {
    var dateObject = new Date(Date.UTC(year, month - 1, day, hour, min3, sec, 0));
    this.setByDate(dateObject);
  };
  this.getFreshValueHex = function() {
    return this.hV;
  };
};
YAHOO.lang.extend(KJUR.asn1.DERAbstractTime, KJUR.asn1.ASN1Object);
KJUR.asn1.DERAbstractStructured = function(params) {
  KJUR.asn1.DERAbstractString.superclass.constructor.call(this);
  this.setByASN1ObjectArray = function(asn1ObjectArray) {
    this.hTLV = null;
    this.isModified = true;
    this.asn1Array = asn1ObjectArray;
  };
  this.appendASN1Object = function(asn1Object) {
    this.hTLV = null;
    this.isModified = true;
    this.asn1Array.push(asn1Object);
  };
  this.asn1Array = new Array();
  if (typeof params != "undefined") {
    if (typeof params["array"] != "undefined") {
      this.asn1Array = params["array"];
    }
  }
};
YAHOO.lang.extend(KJUR.asn1.DERAbstractStructured, KJUR.asn1.ASN1Object);
KJUR.asn1.DERBoolean = function() {
  KJUR.asn1.DERBoolean.superclass.constructor.call(this);
  this.hT = "01";
  this.hTLV = "0101ff";
};
YAHOO.lang.extend(KJUR.asn1.DERBoolean, KJUR.asn1.ASN1Object);
KJUR.asn1.DERInteger = function(params) {
  KJUR.asn1.DERInteger.superclass.constructor.call(this);
  this.hT = "02";
  this.setByBigInteger = function(bigIntegerValue) {
    this.hTLV = null;
    this.isModified = true;
    this.hV = KJUR.asn1.ASN1Util.bigIntToMinTwosComplementsHex(bigIntegerValue);
  };
  this.setByInteger = function(intValue) {
    var bi = new BigInteger(String(intValue), 10);
    this.setByBigInteger(bi);
  };
  this.setValueHex = function(newHexString) {
    this.hV = newHexString;
  };
  this.getFreshValueHex = function() {
    return this.hV;
  };
  if (typeof params != "undefined") {
    if (typeof params["bigint"] != "undefined") {
      this.setByBigInteger(params["bigint"]);
    } else if (typeof params["int"] != "undefined") {
      this.setByInteger(params["int"]);
    } else if (typeof params == "number") {
      this.setByInteger(params);
    } else if (typeof params["hex"] != "undefined") {
      this.setValueHex(params["hex"]);
    }
  }
};
YAHOO.lang.extend(KJUR.asn1.DERInteger, KJUR.asn1.ASN1Object);
KJUR.asn1.DERBitString = function(params) {
  if (params !== void 0 && typeof params.obj !== "undefined") {
    var o2 = KJUR.asn1.ASN1Util.newObject(params.obj);
    params.hex = "00" + o2.getEncodedHex();
  }
  KJUR.asn1.DERBitString.superclass.constructor.call(this);
  this.hT = "03";
  this.setHexValueIncludingUnusedBits = function(newHexStringIncludingUnusedBits) {
    this.hTLV = null;
    this.isModified = true;
    this.hV = newHexStringIncludingUnusedBits;
  };
  this.setUnusedBitsAndHexValue = function(unusedBits, hValue) {
    if (unusedBits < 0 || 7 < unusedBits) {
      throw "unused bits shall be from 0 to 7: u = " + unusedBits;
    }
    var hUnusedBits = "0" + unusedBits;
    this.hTLV = null;
    this.isModified = true;
    this.hV = hUnusedBits + hValue;
  };
  this.setByBinaryString = function(binaryString) {
    binaryString = binaryString.replace(/0+$/, "");
    var unusedBits = 8 - binaryString.length % 8;
    if (unusedBits == 8)
      unusedBits = 0;
    for (var i2 = 0; i2 <= unusedBits; i2++) {
      binaryString += "0";
    }
    var h2 = "";
    for (var i2 = 0; i2 < binaryString.length - 1; i2 += 8) {
      var b2 = binaryString.substr(i2, 8);
      var x2 = parseInt(b2, 2).toString(16);
      if (x2.length == 1)
        x2 = "0" + x2;
      h2 += x2;
    }
    this.hTLV = null;
    this.isModified = true;
    this.hV = "0" + unusedBits + h2;
  };
  this.setByBooleanArray = function(booleanArray) {
    var s2 = "";
    for (var i2 = 0; i2 < booleanArray.length; i2++) {
      if (booleanArray[i2] == true) {
        s2 += "1";
      } else {
        s2 += "0";
      }
    }
    this.setByBinaryString(s2);
  };
  this.newFalseArray = function(nLength) {
    var a2 = new Array(nLength);
    for (var i2 = 0; i2 < nLength; i2++) {
      a2[i2] = false;
    }
    return a2;
  };
  this.getFreshValueHex = function() {
    return this.hV;
  };
  if (typeof params != "undefined") {
    if (typeof params == "string" && params.toLowerCase().match(/^[0-9a-f]+$/)) {
      this.setHexValueIncludingUnusedBits(params);
    } else if (typeof params["hex"] != "undefined") {
      this.setHexValueIncludingUnusedBits(params["hex"]);
    } else if (typeof params["bin"] != "undefined") {
      this.setByBinaryString(params["bin"]);
    } else if (typeof params["array"] != "undefined") {
      this.setByBooleanArray(params["array"]);
    }
  }
};
YAHOO.lang.extend(KJUR.asn1.DERBitString, KJUR.asn1.ASN1Object);
KJUR.asn1.DEROctetString = function(params) {
  if (params !== void 0 && typeof params.obj !== "undefined") {
    var o2 = KJUR.asn1.ASN1Util.newObject(params.obj);
    params.hex = o2.getEncodedHex();
  }
  KJUR.asn1.DEROctetString.superclass.constructor.call(this, params);
  this.hT = "04";
};
YAHOO.lang.extend(KJUR.asn1.DEROctetString, KJUR.asn1.DERAbstractString);
KJUR.asn1.DERNull = function() {
  KJUR.asn1.DERNull.superclass.constructor.call(this);
  this.hT = "05";
  this.hTLV = "0500";
};
YAHOO.lang.extend(KJUR.asn1.DERNull, KJUR.asn1.ASN1Object);
KJUR.asn1.DERObjectIdentifier = function(params) {
  var itox = function(i2) {
    var h2 = i2.toString(16);
    if (h2.length == 1)
      h2 = "0" + h2;
    return h2;
  };
  var roidtox = function(roid) {
    var h2 = "";
    var bi = new BigInteger(roid, 10);
    var b2 = bi.toString(2);
    var padLen = 7 - b2.length % 7;
    if (padLen == 7)
      padLen = 0;
    var bPad = "";
    for (var i2 = 0; i2 < padLen; i2++)
      bPad += "0";
    b2 = bPad + b2;
    for (var i2 = 0; i2 < b2.length - 1; i2 += 7) {
      var b8 = b2.substr(i2, 7);
      if (i2 != b2.length - 7)
        b8 = "1" + b8;
      h2 += itox(parseInt(b8, 2));
    }
    return h2;
  };
  KJUR.asn1.DERObjectIdentifier.superclass.constructor.call(this);
  this.hT = "06";
  this.setValueHex = function(newHexString) {
    this.hTLV = null;
    this.isModified = true;
    this.s = null;
    this.hV = newHexString;
  };
  this.setValueOidString = function(oidString) {
    if (!oidString.match(/^[0-9.]+$/)) {
      throw "malformed oid string: " + oidString;
    }
    var h2 = "";
    var a2 = oidString.split(".");
    var i0 = parseInt(a2[0]) * 40 + parseInt(a2[1]);
    h2 += itox(i0);
    a2.splice(0, 2);
    for (var i2 = 0; i2 < a2.length; i2++) {
      h2 += roidtox(a2[i2]);
    }
    this.hTLV = null;
    this.isModified = true;
    this.s = null;
    this.hV = h2;
  };
  this.setValueName = function(oidName) {
    var oid = KJUR.asn1.x509.OID.name2oid(oidName);
    if (oid !== "") {
      this.setValueOidString(oid);
    } else {
      throw "DERObjectIdentifier oidName undefined: " + oidName;
    }
  };
  this.getFreshValueHex = function() {
    return this.hV;
  };
  if (params !== void 0) {
    if (typeof params === "string") {
      if (params.match(/^[0-2].[0-9.]+$/)) {
        this.setValueOidString(params);
      } else {
        this.setValueName(params);
      }
    } else if (params.oid !== void 0) {
      this.setValueOidString(params.oid);
    } else if (params.hex !== void 0) {
      this.setValueHex(params.hex);
    } else if (params.name !== void 0) {
      this.setValueName(params.name);
    }
  }
};
YAHOO.lang.extend(KJUR.asn1.DERObjectIdentifier, KJUR.asn1.ASN1Object);
KJUR.asn1.DEREnumerated = function(params) {
  KJUR.asn1.DEREnumerated.superclass.constructor.call(this);
  this.hT = "0a";
  this.setByBigInteger = function(bigIntegerValue) {
    this.hTLV = null;
    this.isModified = true;
    this.hV = KJUR.asn1.ASN1Util.bigIntToMinTwosComplementsHex(bigIntegerValue);
  };
  this.setByInteger = function(intValue) {
    var bi = new BigInteger(String(intValue), 10);
    this.setByBigInteger(bi);
  };
  this.setValueHex = function(newHexString) {
    this.hV = newHexString;
  };
  this.getFreshValueHex = function() {
    return this.hV;
  };
  if (typeof params != "undefined") {
    if (typeof params["int"] != "undefined") {
      this.setByInteger(params["int"]);
    } else if (typeof params == "number") {
      this.setByInteger(params);
    } else if (typeof params["hex"] != "undefined") {
      this.setValueHex(params["hex"]);
    }
  }
};
YAHOO.lang.extend(KJUR.asn1.DEREnumerated, KJUR.asn1.ASN1Object);
KJUR.asn1.DERUTF8String = function(params) {
  KJUR.asn1.DERUTF8String.superclass.constructor.call(this, params);
  this.hT = "0c";
};
YAHOO.lang.extend(KJUR.asn1.DERUTF8String, KJUR.asn1.DERAbstractString);
KJUR.asn1.DERNumericString = function(params) {
  KJUR.asn1.DERNumericString.superclass.constructor.call(this, params);
  this.hT = "12";
};
YAHOO.lang.extend(KJUR.asn1.DERNumericString, KJUR.asn1.DERAbstractString);
KJUR.asn1.DERPrintableString = function(params) {
  KJUR.asn1.DERPrintableString.superclass.constructor.call(this, params);
  this.hT = "13";
};
YAHOO.lang.extend(KJUR.asn1.DERPrintableString, KJUR.asn1.DERAbstractString);
KJUR.asn1.DERTeletexString = function(params) {
  KJUR.asn1.DERTeletexString.superclass.constructor.call(this, params);
  this.hT = "14";
};
YAHOO.lang.extend(KJUR.asn1.DERTeletexString, KJUR.asn1.DERAbstractString);
KJUR.asn1.DERIA5String = function(params) {
  KJUR.asn1.DERIA5String.superclass.constructor.call(this, params);
  this.hT = "16";
};
YAHOO.lang.extend(KJUR.asn1.DERIA5String, KJUR.asn1.DERAbstractString);
KJUR.asn1.DERUTCTime = function(params) {
  KJUR.asn1.DERUTCTime.superclass.constructor.call(this, params);
  this.hT = "17";
  this.setByDate = function(dateObject) {
    this.hTLV = null;
    this.isModified = true;
    this.date = dateObject;
    this.s = this.formatDate(this.date, "utc");
    this.hV = stohex(this.s);
  };
  this.getFreshValueHex = function() {
    if (typeof this.date == "undefined" && typeof this.s == "undefined") {
      this.date = /* @__PURE__ */ new Date();
      this.s = this.formatDate(this.date, "utc");
      this.hV = stohex(this.s);
    }
    return this.hV;
  };
  if (params !== void 0) {
    if (params.str !== void 0) {
      this.setString(params.str);
    } else if (typeof params == "string" && params.match(/^[0-9]{12}Z$/)) {
      this.setString(params);
    } else if (params.hex !== void 0) {
      this.setStringHex(params.hex);
    } else if (params.date !== void 0) {
      this.setByDate(params.date);
    }
  }
};
YAHOO.lang.extend(KJUR.asn1.DERUTCTime, KJUR.asn1.DERAbstractTime);
KJUR.asn1.DERGeneralizedTime = function(params) {
  KJUR.asn1.DERGeneralizedTime.superclass.constructor.call(this, params);
  this.hT = "18";
  this.withMillis = false;
  this.setByDate = function(dateObject) {
    this.hTLV = null;
    this.isModified = true;
    this.date = dateObject;
    this.s = this.formatDate(this.date, "gen", this.withMillis);
    this.hV = stohex(this.s);
  };
  this.getFreshValueHex = function() {
    if (this.date === void 0 && this.s === void 0) {
      this.date = /* @__PURE__ */ new Date();
      this.s = this.formatDate(this.date, "gen", this.withMillis);
      this.hV = stohex(this.s);
    }
    return this.hV;
  };
  if (params !== void 0) {
    if (params.str !== void 0) {
      this.setString(params.str);
    } else if (typeof params == "string" && params.match(/^[0-9]{14}Z$/)) {
      this.setString(params);
    } else if (params.hex !== void 0) {
      this.setStringHex(params.hex);
    } else if (params.date !== void 0) {
      this.setByDate(params.date);
    }
    if (params.millis === true) {
      this.withMillis = true;
    }
  }
};
YAHOO.lang.extend(KJUR.asn1.DERGeneralizedTime, KJUR.asn1.DERAbstractTime);
KJUR.asn1.DERSequence = function(params) {
  KJUR.asn1.DERSequence.superclass.constructor.call(this, params);
  this.hT = "30";
  this.getFreshValueHex = function() {
    var h2 = "";
    for (var i2 = 0; i2 < this.asn1Array.length; i2++) {
      var asn1Obj = this.asn1Array[i2];
      h2 += asn1Obj.getEncodedHex();
    }
    this.hV = h2;
    return this.hV;
  };
};
YAHOO.lang.extend(KJUR.asn1.DERSequence, KJUR.asn1.DERAbstractStructured);
KJUR.asn1.DERSet = function(params) {
  KJUR.asn1.DERSet.superclass.constructor.call(this, params);
  this.hT = "31";
  this.sortFlag = true;
  this.getFreshValueHex = function() {
    var a2 = new Array();
    for (var i2 = 0; i2 < this.asn1Array.length; i2++) {
      var asn1Obj = this.asn1Array[i2];
      a2.push(asn1Obj.getEncodedHex());
    }
    if (this.sortFlag == true)
      a2.sort();
    this.hV = a2.join("");
    return this.hV;
  };
  if (typeof params != "undefined") {
    if (typeof params.sortflag != "undefined" && params.sortflag == false)
      this.sortFlag = false;
  }
};
YAHOO.lang.extend(KJUR.asn1.DERSet, KJUR.asn1.DERAbstractStructured);
KJUR.asn1.DERTaggedObject = function(params) {
  KJUR.asn1.DERTaggedObject.superclass.constructor.call(this);
  this.hT = "a0";
  this.hV = "";
  this.isExplicit = true;
  this.asn1Object = null;
  this.setASN1Object = function(isExplicitFlag, tagNoHex, asn1Object) {
    this.hT = tagNoHex;
    this.isExplicit = isExplicitFlag;
    this.asn1Object = asn1Object;
    if (this.isExplicit) {
      this.hV = this.asn1Object.getEncodedHex();
      this.hTLV = null;
      this.isModified = true;
    } else {
      this.hV = null;
      this.hTLV = asn1Object.getEncodedHex();
      this.hTLV = this.hTLV.replace(/^../, tagNoHex);
      this.isModified = false;
    }
  };
  this.getFreshValueHex = function() {
    return this.hV;
  };
  if (typeof params != "undefined") {
    if (typeof params["tag"] != "undefined") {
      this.hT = params["tag"];
    }
    if (typeof params["explicit"] != "undefined") {
      this.isExplicit = params["explicit"];
    }
    if (typeof params["obj"] != "undefined") {
      this.asn1Object = params["obj"];
      this.setASN1Object(this.isExplicit, this.hT, this.asn1Object);
    }
  }
};
YAHOO.lang.extend(KJUR.asn1.DERTaggedObject, KJUR.asn1.ASN1Object);
var __extends$2 = globalThis && globalThis.__extends || function() {
  var extendStatics2 = function(d2, b2) {
    extendStatics2 = Object.setPrototypeOf || { __proto__: [] } instanceof Array && function(d3, b3) {
      d3.__proto__ = b3;
    } || function(d3, b3) {
      for (var p2 in b3)
        if (Object.prototype.hasOwnProperty.call(b3, p2))
          d3[p2] = b3[p2];
    };
    return extendStatics2(d2, b2);
  };
  return function(d2, b2) {
    if (typeof b2 !== "function" && b2 !== null)
      throw new TypeError("Class extends value " + String(b2) + " is not a constructor or null");
    extendStatics2(d2, b2);
    function __() {
      this.constructor = d2;
    }
    d2.prototype = b2 === null ? Object.create(b2) : (__.prototype = b2.prototype, new __());
  };
}();
var JSEncryptRSAKey = (
  /** @class */
  function(_super) {
    __extends$2(JSEncryptRSAKey2, _super);
    function JSEncryptRSAKey2(key) {
      var _this = _super.call(this) || this;
      if (key) {
        if (typeof key === "string") {
          _this.parseKey(key);
        } else if (JSEncryptRSAKey2.hasPrivateKeyProperty(key) || JSEncryptRSAKey2.hasPublicKeyProperty(key)) {
          _this.parsePropertiesFrom(key);
        }
      }
      return _this;
    }
    JSEncryptRSAKey2.prototype.parseKey = function(pem) {
      try {
        var modulus = 0;
        var public_exponent = 0;
        var reHex = /^\s*(?:[0-9A-Fa-f][0-9A-Fa-f]\s*)+$/;
        var der = reHex.test(pem) ? Hex.decode(pem) : Base64.unarmor(pem);
        var asn1 = ASN1.decode(der);
        if (asn1.sub.length === 3) {
          asn1 = asn1.sub[2].sub[0];
        }
        if (asn1.sub.length === 9) {
          modulus = asn1.sub[1].getHexStringValue();
          this.n = parseBigInt(modulus, 16);
          public_exponent = asn1.sub[2].getHexStringValue();
          this.e = parseInt(public_exponent, 16);
          var private_exponent = asn1.sub[3].getHexStringValue();
          this.d = parseBigInt(private_exponent, 16);
          var prime1 = asn1.sub[4].getHexStringValue();
          this.p = parseBigInt(prime1, 16);
          var prime2 = asn1.sub[5].getHexStringValue();
          this.q = parseBigInt(prime2, 16);
          var exponent1 = asn1.sub[6].getHexStringValue();
          this.dmp1 = parseBigInt(exponent1, 16);
          var exponent2 = asn1.sub[7].getHexStringValue();
          this.dmq1 = parseBigInt(exponent2, 16);
          var coefficient = asn1.sub[8].getHexStringValue();
          this.coeff = parseBigInt(coefficient, 16);
        } else if (asn1.sub.length === 2) {
          if (asn1.sub[0].sub) {
            var bit_string = asn1.sub[1];
            var sequence = bit_string.sub[0];
            modulus = sequence.sub[0].getHexStringValue();
            this.n = parseBigInt(modulus, 16);
            public_exponent = sequence.sub[1].getHexStringValue();
            this.e = parseInt(public_exponent, 16);
          } else {
            modulus = asn1.sub[0].getHexStringValue();
            this.n = parseBigInt(modulus, 16);
            public_exponent = asn1.sub[1].getHexStringValue();
            this.e = parseInt(public_exponent, 16);
          }
        } else {
          return false;
        }
        return true;
      } catch (ex) {
        return false;
      }
    };
    JSEncryptRSAKey2.prototype.getPrivateBaseKey = function() {
      var options = {
        array: [
          new KJUR.asn1.DERInteger({ int: 0 }),
          new KJUR.asn1.DERInteger({ bigint: this.n }),
          new KJUR.asn1.DERInteger({ int: this.e }),
          new KJUR.asn1.DERInteger({ bigint: this.d }),
          new KJUR.asn1.DERInteger({ bigint: this.p }),
          new KJUR.asn1.DERInteger({ bigint: this.q }),
          new KJUR.asn1.DERInteger({ bigint: this.dmp1 }),
          new KJUR.asn1.DERInteger({ bigint: this.dmq1 }),
          new KJUR.asn1.DERInteger({ bigint: this.coeff })
        ]
      };
      var seq = new KJUR.asn1.DERSequence(options);
      return seq.getEncodedHex();
    };
    JSEncryptRSAKey2.prototype.getPrivateBaseKeyB64 = function() {
      return hex2b64(this.getPrivateBaseKey());
    };
    JSEncryptRSAKey2.prototype.getPublicBaseKey = function() {
      var first_sequence = new KJUR.asn1.DERSequence({
        array: [
          new KJUR.asn1.DERObjectIdentifier({ oid: "1.2.840.113549.1.1.1" }),
          new KJUR.asn1.DERNull()
        ]
      });
      var second_sequence = new KJUR.asn1.DERSequence({
        array: [
          new KJUR.asn1.DERInteger({ bigint: this.n }),
          new KJUR.asn1.DERInteger({ int: this.e })
        ]
      });
      var bit_string = new KJUR.asn1.DERBitString({
        hex: "00" + second_sequence.getEncodedHex()
      });
      var seq = new KJUR.asn1.DERSequence({
        array: [first_sequence, bit_string]
      });
      return seq.getEncodedHex();
    };
    JSEncryptRSAKey2.prototype.getPublicBaseKeyB64 = function() {
      return hex2b64(this.getPublicBaseKey());
    };
    JSEncryptRSAKey2.wordwrap = function(str, width) {
      width = width || 64;
      if (!str) {
        return str;
      }
      var regex2 = "(.{1," + width + "})( +|$\n?)|(.{1," + width + "})";
      return str.match(RegExp(regex2, "g")).join("\n");
    };
    JSEncryptRSAKey2.prototype.getPrivateKey = function() {
      var key = "-----BEGIN RSA PRIVATE KEY-----\n";
      key += JSEncryptRSAKey2.wordwrap(this.getPrivateBaseKeyB64()) + "\n";
      key += "-----END RSA PRIVATE KEY-----";
      return key;
    };
    JSEncryptRSAKey2.prototype.getPublicKey = function() {
      var key = "-----BEGIN PUBLIC KEY-----\n";
      key += JSEncryptRSAKey2.wordwrap(this.getPublicBaseKeyB64()) + "\n";
      key += "-----END PUBLIC KEY-----";
      return key;
    };
    JSEncryptRSAKey2.hasPublicKeyProperty = function(obj) {
      obj = obj || {};
      return obj.hasOwnProperty("n") && obj.hasOwnProperty("e");
    };
    JSEncryptRSAKey2.hasPrivateKeyProperty = function(obj) {
      obj = obj || {};
      return obj.hasOwnProperty("n") && obj.hasOwnProperty("e") && obj.hasOwnProperty("d") && obj.hasOwnProperty("p") && obj.hasOwnProperty("q") && obj.hasOwnProperty("dmp1") && obj.hasOwnProperty("dmq1") && obj.hasOwnProperty("coeff");
    };
    JSEncryptRSAKey2.prototype.parsePropertiesFrom = function(obj) {
      this.n = obj.n;
      this.e = obj.e;
      if (obj.hasOwnProperty("d")) {
        this.d = obj.d;
        this.p = obj.p;
        this.q = obj.q;
        this.dmp1 = obj.dmp1;
        this.dmq1 = obj.dmq1;
        this.coeff = obj.coeff;
      }
    };
    return JSEncryptRSAKey2;
  }(RSAKey)
);
var _a$2;
var version$3 = typeof process !== "undefined" ? (_a$2 = process.env) === null || _a$2 === void 0 ? void 0 : _a$2.npm_package_version : void 0;
var JSEncrypt = (
  /** @class */
  function() {
    function JSEncrypt2(options) {
      if (options === void 0) {
        options = {};
      }
      options = options || {};
      this.default_key_size = options.default_key_size ? parseInt(options.default_key_size, 10) : 1024;
      this.default_public_exponent = options.default_public_exponent || "010001";
      this.log = options.log || false;
      this.key = null;
    }
    JSEncrypt2.prototype.setKey = function(key) {
      if (this.log && this.key) {
        console.warn("A key was already set, overriding existing.");
      }
      this.key = new JSEncryptRSAKey(key);
    };
    JSEncrypt2.prototype.setPrivateKey = function(privkey) {
      this.setKey(privkey);
    };
    JSEncrypt2.prototype.setPublicKey = function(pubkey) {
      this.setKey(pubkey);
    };
    JSEncrypt2.prototype.decrypt = function(str) {
      try {
        return this.getKey().decrypt(b64tohex(str));
      } catch (ex) {
        return false;
      }
    };
    JSEncrypt2.prototype.encrypt = function(str) {
      try {
        return hex2b64(this.getKey().encrypt(str));
      } catch (ex) {
        return false;
      }
    };
    JSEncrypt2.prototype.sign = function(str, digestMethod, digestName) {
      try {
        return hex2b64(this.getKey().sign(str, digestMethod, digestName));
      } catch (ex) {
        return false;
      }
    };
    JSEncrypt2.prototype.verify = function(str, signature, digestMethod) {
      try {
        return this.getKey().verify(str, b64tohex(signature), digestMethod);
      } catch (ex) {
        return false;
      }
    };
    JSEncrypt2.prototype.getKey = function(cb) {
      if (!this.key) {
        this.key = new JSEncryptRSAKey();
        if (cb && {}.toString.call(cb) === "[object Function]") {
          this.key.generateAsync(this.default_key_size, this.default_public_exponent, cb);
          return;
        }
        this.key.generate(this.default_key_size, this.default_public_exponent);
      }
      return this.key;
    };
    JSEncrypt2.prototype.getPrivateKey = function() {
      return this.getKey().getPrivateKey();
    };
    JSEncrypt2.prototype.getPrivateKeyB64 = function() {
      return this.getKey().getPrivateBaseKeyB64();
    };
    JSEncrypt2.prototype.getPublicKey = function() {
      return this.getKey().getPublicKey();
    };
    JSEncrypt2.prototype.getPublicKeyB64 = function() {
      return this.getKey().getPublicBaseKeyB64();
    };
    JSEncrypt2.version = version$3;
    return JSEncrypt2;
  }()
);
var commonjsGlobal = typeof globalThis !== "undefined" ? globalThis : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : {};
function getDefaultExportFromCjs(x2) {
  return x2 && x2.__esModule && Object.prototype.hasOwnProperty.call(x2, "default") ? x2["default"] : x2;
}
function getAugmentedNamespace(n2) {
  if (n2.__esModule)
    return n2;
  var f2 = n2.default;
  if (typeof f2 == "function") {
    var a2 = function a3() {
      if (this instanceof a3) {
        return Reflect.construct(f2, arguments, this.constructor);
      }
      return f2.apply(this, arguments);
    };
    a2.prototype = f2.prototype;
  } else
    a2 = {};
  Object.defineProperty(a2, "__esModule", { value: true });
  Object.keys(n2).forEach(function(k2) {
    var d2 = Object.getOwnPropertyDescriptor(n2, k2);
    Object.defineProperty(a2, k2, d2.get ? d2 : {
      enumerable: true,
      get: function() {
        return n2[k2];
      }
    });
  });
  return a2;
}
var cryptoJs = { exports: {} };
function commonjsRequire(path) {
  throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
}
var core = { exports: {} };
const __viteBrowserExternal = {};
const __viteBrowserExternal$1 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: __viteBrowserExternal
}, Symbol.toStringTag, { value: "Module" }));
const require$$0$1 = /* @__PURE__ */ getAugmentedNamespace(__viteBrowserExternal$1);
var hasRequiredCore;
function requireCore() {
  if (hasRequiredCore)
    return core.exports;
  hasRequiredCore = 1;
  (function(module2, exports) {
    (function(root2, factory) {
      {
        module2.exports = factory();
      }
    })(commonjsGlobal, function() {
      var CryptoJS2 = CryptoJS2 || function(Math2, undefined$12) {
        var crypto2;
        if (typeof window !== "undefined" && window.crypto) {
          crypto2 = window.crypto;
        }
        if (typeof self !== "undefined" && self.crypto) {
          crypto2 = self.crypto;
        }
        if (typeof globalThis !== "undefined" && globalThis.crypto) {
          crypto2 = globalThis.crypto;
        }
        if (!crypto2 && typeof window !== "undefined" && window.msCrypto) {
          crypto2 = window.msCrypto;
        }
        if (!crypto2 && typeof commonjsGlobal !== "undefined" && commonjsGlobal.crypto) {
          crypto2 = commonjsGlobal.crypto;
        }
        if (!crypto2 && typeof commonjsRequire === "function") {
          try {
            crypto2 = require$$0$1;
          } catch (err2) {
          }
        }
        var cryptoSecureRandomInt = function() {
          if (crypto2) {
            if (typeof crypto2.getRandomValues === "function") {
              try {
                return crypto2.getRandomValues(new Uint32Array(1))[0];
              } catch (err2) {
              }
            }
            if (typeof crypto2.randomBytes === "function") {
              try {
                return crypto2.randomBytes(4).readInt32LE();
              } catch (err2) {
              }
            }
          }
          throw new Error("Native crypto module could not be used to get secure random number.");
        };
        var create4 = Object.create || function() {
          function F2() {
          }
          return function(obj) {
            var subtype;
            F2.prototype = obj;
            subtype = new F2();
            F2.prototype = null;
            return subtype;
          };
        }();
        var C2 = {};
        var C_lib = C2.lib = {};
        var Base = C_lib.Base = function() {
          return {
            /**
             * Creates a new object that inherits from this object.
             *
             * @param {Object} overrides Properties to copy into the new object.
             *
             * @return {Object} The new object.
             *
             * @static
             *
             * @example
             *
             *     var MyType = CryptoJS.lib.Base.extend({
             *         field: 'value',
             *
             *         method: function () {
             *         }
             *     });
             */
            extend: function(overrides) {
              var subtype = create4(this);
              if (overrides) {
                subtype.mixIn(overrides);
              }
              if (!subtype.hasOwnProperty("init") || this.init === subtype.init) {
                subtype.init = function() {
                  subtype.$super.init.apply(this, arguments);
                };
              }
              subtype.init.prototype = subtype;
              subtype.$super = this;
              return subtype;
            },
            /**
             * Extends this object and runs the init method.
             * Arguments to create() will be passed to init().
             *
             * @return {Object} The new object.
             *
             * @static
             *
             * @example
             *
             *     var instance = MyType.create();
             */
            create: function() {
              var instance = this.extend();
              instance.init.apply(instance, arguments);
              return instance;
            },
            /**
             * Initializes a newly created object.
             * Override this method to add some logic when your objects are created.
             *
             * @example
             *
             *     var MyType = CryptoJS.lib.Base.extend({
             *         init: function () {
             *             // ...
             *         }
             *     });
             */
            init: function() {
            },
            /**
             * Copies properties into this object.
             *
             * @param {Object} properties The properties to mix in.
             *
             * @example
             *
             *     MyType.mixIn({
             *         field: 'value'
             *     });
             */
            mixIn: function(properties) {
              for (var propertyName in properties) {
                if (properties.hasOwnProperty(propertyName)) {
                  this[propertyName] = properties[propertyName];
                }
              }
              if (properties.hasOwnProperty("toString")) {
                this.toString = properties.toString;
              }
            },
            /**
             * Creates a copy of this object.
             *
             * @return {Object} The clone.
             *
             * @example
             *
             *     var clone = instance.clone();
             */
            clone: function() {
              return this.init.prototype.extend(this);
            }
          };
        }();
        var WordArray = C_lib.WordArray = Base.extend({
          /**
           * Initializes a newly created word array.
           *
           * @param {Array} words (Optional) An array of 32-bit words.
           * @param {number} sigBytes (Optional) The number of significant bytes in the words.
           *
           * @example
           *
           *     var wordArray = CryptoJS.lib.WordArray.create();
           *     var wordArray = CryptoJS.lib.WordArray.create([0x00010203, 0x04050607]);
           *     var wordArray = CryptoJS.lib.WordArray.create([0x00010203, 0x04050607], 6);
           */
          init: function(words, sigBytes) {
            words = this.words = words || [];
            if (sigBytes != undefined$12) {
              this.sigBytes = sigBytes;
            } else {
              this.sigBytes = words.length * 4;
            }
          },
          /**
           * Converts this word array to a string.
           *
           * @param {Encoder} encoder (Optional) The encoding strategy to use. Default: CryptoJS.enc.Hex
           *
           * @return {string} The stringified word array.
           *
           * @example
           *
           *     var string = wordArray + '';
           *     var string = wordArray.toString();
           *     var string = wordArray.toString(CryptoJS.enc.Utf8);
           */
          toString: function(encoder2) {
            return (encoder2 || Hex2).stringify(this);
          },
          /**
           * Concatenates a word array to this word array.
           *
           * @param {WordArray} wordArray The word array to append.
           *
           * @return {WordArray} This word array.
           *
           * @example
           *
           *     wordArray1.concat(wordArray2);
           */
          concat: function(wordArray) {
            var thisWords = this.words;
            var thatWords = wordArray.words;
            var thisSigBytes = this.sigBytes;
            var thatSigBytes = wordArray.sigBytes;
            this.clamp();
            if (thisSigBytes % 4) {
              for (var i2 = 0; i2 < thatSigBytes; i2++) {
                var thatByte = thatWords[i2 >>> 2] >>> 24 - i2 % 4 * 8 & 255;
                thisWords[thisSigBytes + i2 >>> 2] |= thatByte << 24 - (thisSigBytes + i2) % 4 * 8;
              }
            } else {
              for (var j2 = 0; j2 < thatSigBytes; j2 += 4) {
                thisWords[thisSigBytes + j2 >>> 2] = thatWords[j2 >>> 2];
              }
            }
            this.sigBytes += thatSigBytes;
            return this;
          },
          /**
           * Removes insignificant bits.
           *
           * @example
           *
           *     wordArray.clamp();
           */
          clamp: function() {
            var words = this.words;
            var sigBytes = this.sigBytes;
            words[sigBytes >>> 2] &= 4294967295 << 32 - sigBytes % 4 * 8;
            words.length = Math2.ceil(sigBytes / 4);
          },
          /**
           * Creates a copy of this word array.
           *
           * @return {WordArray} The clone.
           *
           * @example
           *
           *     var clone = wordArray.clone();
           */
          clone: function() {
            var clone2 = Base.clone.call(this);
            clone2.words = this.words.slice(0);
            return clone2;
          },
          /**
           * Creates a word array filled with random bytes.
           *
           * @param {number} nBytes The number of random bytes to generate.
           *
           * @return {WordArray} The random word array.
           *
           * @static
           *
           * @example
           *
           *     var wordArray = CryptoJS.lib.WordArray.random(16);
           */
          random: function(nBytes) {
            var words = [];
            for (var i2 = 0; i2 < nBytes; i2 += 4) {
              words.push(cryptoSecureRandomInt());
            }
            return new WordArray.init(words, nBytes);
          }
        });
        var C_enc = C2.enc = {};
        var Hex2 = C_enc.Hex = {
          /**
           * Converts a word array to a hex string.
           *
           * @param {WordArray} wordArray The word array.
           *
           * @return {string} The hex string.
           *
           * @static
           *
           * @example
           *
           *     var hexString = CryptoJS.enc.Hex.stringify(wordArray);
           */
          stringify: function(wordArray) {
            var words = wordArray.words;
            var sigBytes = wordArray.sigBytes;
            var hexChars = [];
            for (var i2 = 0; i2 < sigBytes; i2++) {
              var bite = words[i2 >>> 2] >>> 24 - i2 % 4 * 8 & 255;
              hexChars.push((bite >>> 4).toString(16));
              hexChars.push((bite & 15).toString(16));
            }
            return hexChars.join("");
          },
          /**
           * Converts a hex string to a word array.
           *
           * @param {string} hexStr The hex string.
           *
           * @return {WordArray} The word array.
           *
           * @static
           *
           * @example
           *
           *     var wordArray = CryptoJS.enc.Hex.parse(hexString);
           */
          parse: function(hexStr) {
            var hexStrLength = hexStr.length;
            var words = [];
            for (var i2 = 0; i2 < hexStrLength; i2 += 2) {
              words[i2 >>> 3] |= parseInt(hexStr.substr(i2, 2), 16) << 24 - i2 % 8 * 4;
            }
            return new WordArray.init(words, hexStrLength / 2);
          }
        };
        var Latin1 = C_enc.Latin1 = {
          /**
           * Converts a word array to a Latin1 string.
           *
           * @param {WordArray} wordArray The word array.
           *
           * @return {string} The Latin1 string.
           *
           * @static
           *
           * @example
           *
           *     var latin1String = CryptoJS.enc.Latin1.stringify(wordArray);
           */
          stringify: function(wordArray) {
            var words = wordArray.words;
            var sigBytes = wordArray.sigBytes;
            var latin1Chars = [];
            for (var i2 = 0; i2 < sigBytes; i2++) {
              var bite = words[i2 >>> 2] >>> 24 - i2 % 4 * 8 & 255;
              latin1Chars.push(String.fromCharCode(bite));
            }
            return latin1Chars.join("");
          },
          /**
           * Converts a Latin1 string to a word array.
           *
           * @param {string} latin1Str The Latin1 string.
           *
           * @return {WordArray} The word array.
           *
           * @static
           *
           * @example
           *
           *     var wordArray = CryptoJS.enc.Latin1.parse(latin1String);
           */
          parse: function(latin1Str) {
            var latin1StrLength = latin1Str.length;
            var words = [];
            for (var i2 = 0; i2 < latin1StrLength; i2++) {
              words[i2 >>> 2] |= (latin1Str.charCodeAt(i2) & 255) << 24 - i2 % 4 * 8;
            }
            return new WordArray.init(words, latin1StrLength);
          }
        };
        var Utf8 = C_enc.Utf8 = {
          /**
           * Converts a word array to a UTF-8 string.
           *
           * @param {WordArray} wordArray The word array.
           *
           * @return {string} The UTF-8 string.
           *
           * @static
           *
           * @example
           *
           *     var utf8String = CryptoJS.enc.Utf8.stringify(wordArray);
           */
          stringify: function(wordArray) {
            try {
              return decodeURIComponent(escape(Latin1.stringify(wordArray)));
            } catch (e2) {
              throw new Error("Malformed UTF-8 data");
            }
          },
          /**
           * Converts a UTF-8 string to a word array.
           *
           * @param {string} utf8Str The UTF-8 string.
           *
           * @return {WordArray} The word array.
           *
           * @static
           *
           * @example
           *
           *     var wordArray = CryptoJS.enc.Utf8.parse(utf8String);
           */
          parse: function(utf8Str) {
            return Latin1.parse(unescape(encodeURIComponent(utf8Str)));
          }
        };
        var BufferedBlockAlgorithm = C_lib.BufferedBlockAlgorithm = Base.extend({
          /**
           * Resets this block algorithm's data buffer to its initial state.
           *
           * @example
           *
           *     bufferedBlockAlgorithm.reset();
           */
          reset: function() {
            this._data = new WordArray.init();
            this._nDataBytes = 0;
          },
          /**
           * Adds new data to this block algorithm's buffer.
           *
           * @param {WordArray|string} data The data to append. Strings are converted to a WordArray using UTF-8.
           *
           * @example
           *
           *     bufferedBlockAlgorithm._append('data');
           *     bufferedBlockAlgorithm._append(wordArray);
           */
          _append: function(data2) {
            if (typeof data2 == "string") {
              data2 = Utf8.parse(data2);
            }
            this._data.concat(data2);
            this._nDataBytes += data2.sigBytes;
          },
          /**
           * Processes available data blocks.
           *
           * This method invokes _doProcessBlock(offset), which must be implemented by a concrete subtype.
           *
           * @param {boolean} doFlush Whether all blocks and partial blocks should be processed.
           *
           * @return {WordArray} The processed data.
           *
           * @example
           *
           *     var processedData = bufferedBlockAlgorithm._process();
           *     var processedData = bufferedBlockAlgorithm._process(!!'flush');
           */
          _process: function(doFlush) {
            var processedWords;
            var data2 = this._data;
            var dataWords = data2.words;
            var dataSigBytes = data2.sigBytes;
            var blockSize = this.blockSize;
            var blockSizeBytes = blockSize * 4;
            var nBlocksReady = dataSigBytes / blockSizeBytes;
            if (doFlush) {
              nBlocksReady = Math2.ceil(nBlocksReady);
            } else {
              nBlocksReady = Math2.max((nBlocksReady | 0) - this._minBufferSize, 0);
            }
            var nWordsReady = nBlocksReady * blockSize;
            var nBytesReady = Math2.min(nWordsReady * 4, dataSigBytes);
            if (nWordsReady) {
              for (var offset2 = 0; offset2 < nWordsReady; offset2 += blockSize) {
                this._doProcessBlock(dataWords, offset2);
              }
              processedWords = dataWords.splice(0, nWordsReady);
              data2.sigBytes -= nBytesReady;
            }
            return new WordArray.init(processedWords, nBytesReady);
          },
          /**
           * Creates a copy of this object.
           *
           * @return {Object} The clone.
           *
           * @example
           *
           *     var clone = bufferedBlockAlgorithm.clone();
           */
          clone: function() {
            var clone2 = Base.clone.call(this);
            clone2._data = this._data.clone();
            return clone2;
          },
          _minBufferSize: 0
        });
        C_lib.Hasher = BufferedBlockAlgorithm.extend({
          /**
           * Configuration options.
           */
          cfg: Base.extend(),
          /**
           * Initializes a newly created hasher.
           *
           * @param {Object} cfg (Optional) The configuration options to use for this hash computation.
           *
           * @example
           *
           *     var hasher = CryptoJS.algo.SHA256.create();
           */
          init: function(cfg) {
            this.cfg = this.cfg.extend(cfg);
            this.reset();
          },
          /**
           * Resets this hasher to its initial state.
           *
           * @example
           *
           *     hasher.reset();
           */
          reset: function() {
            BufferedBlockAlgorithm.reset.call(this);
            this._doReset();
          },
          /**
           * Updates this hasher with a message.
           *
           * @param {WordArray|string} messageUpdate The message to append.
           *
           * @return {Hasher} This hasher.
           *
           * @example
           *
           *     hasher.update('message');
           *     hasher.update(wordArray);
           */
          update: function(messageUpdate) {
            this._append(messageUpdate);
            this._process();
            return this;
          },
          /**
           * Finalizes the hash computation.
           * Note that the finalize operation is effectively a destructive, read-once operation.
           *
           * @param {WordArray|string} messageUpdate (Optional) A final message update.
           *
           * @return {WordArray} The hash.
           *
           * @example
           *
           *     var hash = hasher.finalize();
           *     var hash = hasher.finalize('message');
           *     var hash = hasher.finalize(wordArray);
           */
          finalize: function(messageUpdate) {
            if (messageUpdate) {
              this._append(messageUpdate);
            }
            var hash = this._doFinalize();
            return hash;
          },
          blockSize: 512 / 32,
          /**
           * Creates a shortcut function to a hasher's object interface.
           *
           * @param {Hasher} hasher The hasher to create a helper for.
           *
           * @return {Function} The shortcut function.
           *
           * @static
           *
           * @example
           *
           *     var SHA256 = CryptoJS.lib.Hasher._createHelper(CryptoJS.algo.SHA256);
           */
          _createHelper: function(hasher) {
            return function(message, cfg) {
              return new hasher.init(cfg).finalize(message);
            };
          },
          /**
           * Creates a shortcut function to the HMAC's object interface.
           *
           * @param {Hasher} hasher The hasher to use in this HMAC helper.
           *
           * @return {Function} The shortcut function.
           *
           * @static
           *
           * @example
           *
           *     var HmacSHA256 = CryptoJS.lib.Hasher._createHmacHelper(CryptoJS.algo.SHA256);
           */
          _createHmacHelper: function(hasher) {
            return function(message, key) {
              return new C_algo.HMAC.init(hasher, key).finalize(message);
            };
          }
        });
        var C_algo = C2.algo = {};
        return C2;
      }(Math);
      return CryptoJS2;
    });
  })(core);
  return core.exports;
}
var x64Core = { exports: {} };
var hasRequiredX64Core;
function requireX64Core() {
  if (hasRequiredX64Core)
    return x64Core.exports;
  hasRequiredX64Core = 1;
  (function(module2, exports) {
    (function(root2, factory) {
      {
        module2.exports = factory(requireCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function(undefined$12) {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var Base = C_lib.Base;
        var X32WordArray = C_lib.WordArray;
        var C_x64 = C2.x64 = {};
        C_x64.Word = Base.extend({
          /**
           * Initializes a newly created 64-bit word.
           *
           * @param {number} high The high 32 bits.
           * @param {number} low The low 32 bits.
           *
           * @example
           *
           *     var x64Word = CryptoJS.x64.Word.create(0x00010203, 0x04050607);
           */
          init: function(high, low) {
            this.high = high;
            this.low = low;
          }
          /**
           * Bitwise NOTs this word.
           *
           * @return {X64Word} A new x64-Word object after negating.
           *
           * @example
           *
           *     var negated = x64Word.not();
           */
          // not: function () {
          // var high = ~this.high;
          // var low = ~this.low;
          // return X64Word.create(high, low);
          // },
          /**
           * Bitwise ANDs this word with the passed word.
           *
           * @param {X64Word} word The x64-Word to AND with this word.
           *
           * @return {X64Word} A new x64-Word object after ANDing.
           *
           * @example
           *
           *     var anded = x64Word.and(anotherX64Word);
           */
          // and: function (word) {
          // var high = this.high & word.high;
          // var low = this.low & word.low;
          // return X64Word.create(high, low);
          // },
          /**
           * Bitwise ORs this word with the passed word.
           *
           * @param {X64Word} word The x64-Word to OR with this word.
           *
           * @return {X64Word} A new x64-Word object after ORing.
           *
           * @example
           *
           *     var ored = x64Word.or(anotherX64Word);
           */
          // or: function (word) {
          // var high = this.high | word.high;
          // var low = this.low | word.low;
          // return X64Word.create(high, low);
          // },
          /**
           * Bitwise XORs this word with the passed word.
           *
           * @param {X64Word} word The x64-Word to XOR with this word.
           *
           * @return {X64Word} A new x64-Word object after XORing.
           *
           * @example
           *
           *     var xored = x64Word.xor(anotherX64Word);
           */
          // xor: function (word) {
          // var high = this.high ^ word.high;
          // var low = this.low ^ word.low;
          // return X64Word.create(high, low);
          // },
          /**
           * Shifts this word n bits to the left.
           *
           * @param {number} n The number of bits to shift.
           *
           * @return {X64Word} A new x64-Word object after shifting.
           *
           * @example
           *
           *     var shifted = x64Word.shiftL(25);
           */
          // shiftL: function (n) {
          // if (n < 32) {
          // var high = (this.high << n) | (this.low >>> (32 - n));
          // var low = this.low << n;
          // } else {
          // var high = this.low << (n - 32);
          // var low = 0;
          // }
          // return X64Word.create(high, low);
          // },
          /**
           * Shifts this word n bits to the right.
           *
           * @param {number} n The number of bits to shift.
           *
           * @return {X64Word} A new x64-Word object after shifting.
           *
           * @example
           *
           *     var shifted = x64Word.shiftR(7);
           */
          // shiftR: function (n) {
          // if (n < 32) {
          // var low = (this.low >>> n) | (this.high << (32 - n));
          // var high = this.high >>> n;
          // } else {
          // var low = this.high >>> (n - 32);
          // var high = 0;
          // }
          // return X64Word.create(high, low);
          // },
          /**
           * Rotates this word n bits to the left.
           *
           * @param {number} n The number of bits to rotate.
           *
           * @return {X64Word} A new x64-Word object after rotating.
           *
           * @example
           *
           *     var rotated = x64Word.rotL(25);
           */
          // rotL: function (n) {
          // return this.shiftL(n).or(this.shiftR(64 - n));
          // },
          /**
           * Rotates this word n bits to the right.
           *
           * @param {number} n The number of bits to rotate.
           *
           * @return {X64Word} A new x64-Word object after rotating.
           *
           * @example
           *
           *     var rotated = x64Word.rotR(7);
           */
          // rotR: function (n) {
          // return this.shiftR(n).or(this.shiftL(64 - n));
          // },
          /**
           * Adds this word with the passed word.
           *
           * @param {X64Word} word The x64-Word to add with this word.
           *
           * @return {X64Word} A new x64-Word object after adding.
           *
           * @example
           *
           *     var added = x64Word.add(anotherX64Word);
           */
          // add: function (word) {
          // var low = (this.low + word.low) | 0;
          // var carry = (low >>> 0) < (this.low >>> 0) ? 1 : 0;
          // var high = (this.high + word.high + carry) | 0;
          // return X64Word.create(high, low);
          // }
        });
        C_x64.WordArray = Base.extend({
          /**
           * Initializes a newly created word array.
           *
           * @param {Array} words (Optional) An array of CryptoJS.x64.Word objects.
           * @param {number} sigBytes (Optional) The number of significant bytes in the words.
           *
           * @example
           *
           *     var wordArray = CryptoJS.x64.WordArray.create();
           *
           *     var wordArray = CryptoJS.x64.WordArray.create([
           *         CryptoJS.x64.Word.create(0x00010203, 0x04050607),
           *         CryptoJS.x64.Word.create(0x18191a1b, 0x1c1d1e1f)
           *     ]);
           *
           *     var wordArray = CryptoJS.x64.WordArray.create([
           *         CryptoJS.x64.Word.create(0x00010203, 0x04050607),
           *         CryptoJS.x64.Word.create(0x18191a1b, 0x1c1d1e1f)
           *     ], 10);
           */
          init: function(words, sigBytes) {
            words = this.words = words || [];
            if (sigBytes != undefined$12) {
              this.sigBytes = sigBytes;
            } else {
              this.sigBytes = words.length * 8;
            }
          },
          /**
           * Converts this 64-bit word array to a 32-bit word array.
           *
           * @return {CryptoJS.lib.WordArray} This word array's data as a 32-bit word array.
           *
           * @example
           *
           *     var x32WordArray = x64WordArray.toX32();
           */
          toX32: function() {
            var x64Words = this.words;
            var x64WordsLength = x64Words.length;
            var x32Words = [];
            for (var i2 = 0; i2 < x64WordsLength; i2++) {
              var x64Word = x64Words[i2];
              x32Words.push(x64Word.high);
              x32Words.push(x64Word.low);
            }
            return X32WordArray.create(x32Words, this.sigBytes);
          },
          /**
           * Creates a copy of this word array.
           *
           * @return {X64WordArray} The clone.
           *
           * @example
           *
           *     var clone = x64WordArray.clone();
           */
          clone: function() {
            var clone2 = Base.clone.call(this);
            var words = clone2.words = this.words.slice(0);
            var wordsLength = words.length;
            for (var i2 = 0; i2 < wordsLength; i2++) {
              words[i2] = words[i2].clone();
            }
            return clone2;
          }
        });
      })();
      return CryptoJS2;
    });
  })(x64Core);
  return x64Core.exports;
}
var libTypedarrays = { exports: {} };
var hasRequiredLibTypedarrays;
function requireLibTypedarrays() {
  if (hasRequiredLibTypedarrays)
    return libTypedarrays.exports;
  hasRequiredLibTypedarrays = 1;
  (function(module2, exports) {
    (function(root2, factory) {
      {
        module2.exports = factory(requireCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        if (typeof ArrayBuffer != "function") {
          return;
        }
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var WordArray = C_lib.WordArray;
        var superInit = WordArray.init;
        var subInit = WordArray.init = function(typedArray) {
          if (typedArray instanceof ArrayBuffer) {
            typedArray = new Uint8Array(typedArray);
          }
          if (typedArray instanceof Int8Array || typeof Uint8ClampedArray !== "undefined" && typedArray instanceof Uint8ClampedArray || typedArray instanceof Int16Array || typedArray instanceof Uint16Array || typedArray instanceof Int32Array || typedArray instanceof Uint32Array || typedArray instanceof Float32Array || typedArray instanceof Float64Array) {
            typedArray = new Uint8Array(typedArray.buffer, typedArray.byteOffset, typedArray.byteLength);
          }
          if (typedArray instanceof Uint8Array) {
            var typedArrayByteLength = typedArray.byteLength;
            var words = [];
            for (var i2 = 0; i2 < typedArrayByteLength; i2++) {
              words[i2 >>> 2] |= typedArray[i2] << 24 - i2 % 4 * 8;
            }
            superInit.call(this, words, typedArrayByteLength);
          } else {
            superInit.apply(this, arguments);
          }
        };
        subInit.prototype = WordArray;
      })();
      return CryptoJS2.lib.WordArray;
    });
  })(libTypedarrays);
  return libTypedarrays.exports;
}
var encUtf16 = { exports: {} };
var hasRequiredEncUtf16;
function requireEncUtf16() {
  if (hasRequiredEncUtf16)
    return encUtf16.exports;
  hasRequiredEncUtf16 = 1;
  (function(module2, exports) {
    (function(root2, factory) {
      {
        module2.exports = factory(requireCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var WordArray = C_lib.WordArray;
        var C_enc = C2.enc;
        C_enc.Utf16 = C_enc.Utf16BE = {
          /**
           * Converts a word array to a UTF-16 BE string.
           *
           * @param {WordArray} wordArray The word array.
           *
           * @return {string} The UTF-16 BE string.
           *
           * @static
           *
           * @example
           *
           *     var utf16String = CryptoJS.enc.Utf16.stringify(wordArray);
           */
          stringify: function(wordArray) {
            var words = wordArray.words;
            var sigBytes = wordArray.sigBytes;
            var utf16Chars = [];
            for (var i2 = 0; i2 < sigBytes; i2 += 2) {
              var codePoint = words[i2 >>> 2] >>> 16 - i2 % 4 * 8 & 65535;
              utf16Chars.push(String.fromCharCode(codePoint));
            }
            return utf16Chars.join("");
          },
          /**
           * Converts a UTF-16 BE string to a word array.
           *
           * @param {string} utf16Str The UTF-16 BE string.
           *
           * @return {WordArray} The word array.
           *
           * @static
           *
           * @example
           *
           *     var wordArray = CryptoJS.enc.Utf16.parse(utf16String);
           */
          parse: function(utf16Str) {
            var utf16StrLength = utf16Str.length;
            var words = [];
            for (var i2 = 0; i2 < utf16StrLength; i2++) {
              words[i2 >>> 1] |= utf16Str.charCodeAt(i2) << 16 - i2 % 2 * 16;
            }
            return WordArray.create(words, utf16StrLength * 2);
          }
        };
        C_enc.Utf16LE = {
          /**
           * Converts a word array to a UTF-16 LE string.
           *
           * @param {WordArray} wordArray The word array.
           *
           * @return {string} The UTF-16 LE string.
           *
           * @static
           *
           * @example
           *
           *     var utf16Str = CryptoJS.enc.Utf16LE.stringify(wordArray);
           */
          stringify: function(wordArray) {
            var words = wordArray.words;
            var sigBytes = wordArray.sigBytes;
            var utf16Chars = [];
            for (var i2 = 0; i2 < sigBytes; i2 += 2) {
              var codePoint = swapEndian(words[i2 >>> 2] >>> 16 - i2 % 4 * 8 & 65535);
              utf16Chars.push(String.fromCharCode(codePoint));
            }
            return utf16Chars.join("");
          },
          /**
           * Converts a UTF-16 LE string to a word array.
           *
           * @param {string} utf16Str The UTF-16 LE string.
           *
           * @return {WordArray} The word array.
           *
           * @static
           *
           * @example
           *
           *     var wordArray = CryptoJS.enc.Utf16LE.parse(utf16Str);
           */
          parse: function(utf16Str) {
            var utf16StrLength = utf16Str.length;
            var words = [];
            for (var i2 = 0; i2 < utf16StrLength; i2++) {
              words[i2 >>> 1] |= swapEndian(utf16Str.charCodeAt(i2) << 16 - i2 % 2 * 16);
            }
            return WordArray.create(words, utf16StrLength * 2);
          }
        };
        function swapEndian(word) {
          return word << 8 & 4278255360 | word >>> 8 & 16711935;
        }
      })();
      return CryptoJS2.enc.Utf16;
    });
  })(encUtf16);
  return encUtf16.exports;
}
var encBase64 = { exports: {} };
var hasRequiredEncBase64;
function requireEncBase64() {
  if (hasRequiredEncBase64)
    return encBase64.exports;
  hasRequiredEncBase64 = 1;
  (function(module2, exports) {
    (function(root2, factory) {
      {
        module2.exports = factory(requireCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var WordArray = C_lib.WordArray;
        var C_enc = C2.enc;
        C_enc.Base64 = {
          /**
           * Converts a word array to a Base64 string.
           *
           * @param {WordArray} wordArray The word array.
           *
           * @return {string} The Base64 string.
           *
           * @static
           *
           * @example
           *
           *     var base64String = CryptoJS.enc.Base64.stringify(wordArray);
           */
          stringify: function(wordArray) {
            var words = wordArray.words;
            var sigBytes = wordArray.sigBytes;
            var map2 = this._map;
            wordArray.clamp();
            var base64Chars = [];
            for (var i2 = 0; i2 < sigBytes; i2 += 3) {
              var byte1 = words[i2 >>> 2] >>> 24 - i2 % 4 * 8 & 255;
              var byte2 = words[i2 + 1 >>> 2] >>> 24 - (i2 + 1) % 4 * 8 & 255;
              var byte3 = words[i2 + 2 >>> 2] >>> 24 - (i2 + 2) % 4 * 8 & 255;
              var triplet = byte1 << 16 | byte2 << 8 | byte3;
              for (var j2 = 0; j2 < 4 && i2 + j2 * 0.75 < sigBytes; j2++) {
                base64Chars.push(map2.charAt(triplet >>> 6 * (3 - j2) & 63));
              }
            }
            var paddingChar = map2.charAt(64);
            if (paddingChar) {
              while (base64Chars.length % 4) {
                base64Chars.push(paddingChar);
              }
            }
            return base64Chars.join("");
          },
          /**
           * Converts a Base64 string to a word array.
           *
           * @param {string} base64Str The Base64 string.
           *
           * @return {WordArray} The word array.
           *
           * @static
           *
           * @example
           *
           *     var wordArray = CryptoJS.enc.Base64.parse(base64String);
           */
          parse: function(base64Str) {
            var base64StrLength = base64Str.length;
            var map2 = this._map;
            var reverseMap = this._reverseMap;
            if (!reverseMap) {
              reverseMap = this._reverseMap = [];
              for (var j2 = 0; j2 < map2.length; j2++) {
                reverseMap[map2.charCodeAt(j2)] = j2;
              }
            }
            var paddingChar = map2.charAt(64);
            if (paddingChar) {
              var paddingIndex = base64Str.indexOf(paddingChar);
              if (paddingIndex !== -1) {
                base64StrLength = paddingIndex;
              }
            }
            return parseLoop(base64Str, base64StrLength, reverseMap);
          },
          _map: "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/="
        };
        function parseLoop(base64Str, base64StrLength, reverseMap) {
          var words = [];
          var nBytes = 0;
          for (var i2 = 0; i2 < base64StrLength; i2++) {
            if (i2 % 4) {
              var bits1 = reverseMap[base64Str.charCodeAt(i2 - 1)] << i2 % 4 * 2;
              var bits2 = reverseMap[base64Str.charCodeAt(i2)] >>> 6 - i2 % 4 * 2;
              var bitsCombined = bits1 | bits2;
              words[nBytes >>> 2] |= bitsCombined << 24 - nBytes % 4 * 8;
              nBytes++;
            }
          }
          return WordArray.create(words, nBytes);
        }
      })();
      return CryptoJS2.enc.Base64;
    });
  })(encBase64);
  return encBase64.exports;
}
var encBase64url = { exports: {} };
var hasRequiredEncBase64url;
function requireEncBase64url() {
  if (hasRequiredEncBase64url)
    return encBase64url.exports;
  hasRequiredEncBase64url = 1;
  (function(module2, exports) {
    (function(root2, factory) {
      {
        module2.exports = factory(requireCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var WordArray = C_lib.WordArray;
        var C_enc = C2.enc;
        C_enc.Base64url = {
          /**
           * Converts a word array to a Base64url string.
           *
           * @param {WordArray} wordArray The word array.
           *
           * @param {boolean} urlSafe Whether to use url safe
           *
           * @return {string} The Base64url string.
           *
           * @static
           *
           * @example
           *
           *     var base64String = CryptoJS.enc.Base64url.stringify(wordArray);
           */
          stringify: function(wordArray, urlSafe) {
            if (urlSafe === void 0) {
              urlSafe = true;
            }
            var words = wordArray.words;
            var sigBytes = wordArray.sigBytes;
            var map2 = urlSafe ? this._safe_map : this._map;
            wordArray.clamp();
            var base64Chars = [];
            for (var i2 = 0; i2 < sigBytes; i2 += 3) {
              var byte1 = words[i2 >>> 2] >>> 24 - i2 % 4 * 8 & 255;
              var byte2 = words[i2 + 1 >>> 2] >>> 24 - (i2 + 1) % 4 * 8 & 255;
              var byte3 = words[i2 + 2 >>> 2] >>> 24 - (i2 + 2) % 4 * 8 & 255;
              var triplet = byte1 << 16 | byte2 << 8 | byte3;
              for (var j2 = 0; j2 < 4 && i2 + j2 * 0.75 < sigBytes; j2++) {
                base64Chars.push(map2.charAt(triplet >>> 6 * (3 - j2) & 63));
              }
            }
            var paddingChar = map2.charAt(64);
            if (paddingChar) {
              while (base64Chars.length % 4) {
                base64Chars.push(paddingChar);
              }
            }
            return base64Chars.join("");
          },
          /**
           * Converts a Base64url string to a word array.
           *
           * @param {string} base64Str The Base64url string.
           *
           * @param {boolean} urlSafe Whether to use url safe
           *
           * @return {WordArray} The word array.
           *
           * @static
           *
           * @example
           *
           *     var wordArray = CryptoJS.enc.Base64url.parse(base64String);
           */
          parse: function(base64Str, urlSafe) {
            if (urlSafe === void 0) {
              urlSafe = true;
            }
            var base64StrLength = base64Str.length;
            var map2 = urlSafe ? this._safe_map : this._map;
            var reverseMap = this._reverseMap;
            if (!reverseMap) {
              reverseMap = this._reverseMap = [];
              for (var j2 = 0; j2 < map2.length; j2++) {
                reverseMap[map2.charCodeAt(j2)] = j2;
              }
            }
            var paddingChar = map2.charAt(64);
            if (paddingChar) {
              var paddingIndex = base64Str.indexOf(paddingChar);
              if (paddingIndex !== -1) {
                base64StrLength = paddingIndex;
              }
            }
            return parseLoop(base64Str, base64StrLength, reverseMap);
          },
          _map: "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=",
          _safe_map: "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_"
        };
        function parseLoop(base64Str, base64StrLength, reverseMap) {
          var words = [];
          var nBytes = 0;
          for (var i2 = 0; i2 < base64StrLength; i2++) {
            if (i2 % 4) {
              var bits1 = reverseMap[base64Str.charCodeAt(i2 - 1)] << i2 % 4 * 2;
              var bits2 = reverseMap[base64Str.charCodeAt(i2)] >>> 6 - i2 % 4 * 2;
              var bitsCombined = bits1 | bits2;
              words[nBytes >>> 2] |= bitsCombined << 24 - nBytes % 4 * 8;
              nBytes++;
            }
          }
          return WordArray.create(words, nBytes);
        }
      })();
      return CryptoJS2.enc.Base64url;
    });
  })(encBase64url);
  return encBase64url.exports;
}
var md5 = { exports: {} };
var hasRequiredMd5;
function requireMd5() {
  if (hasRequiredMd5)
    return md5.exports;
  hasRequiredMd5 = 1;
  (function(module2, exports) {
    (function(root2, factory) {
      {
        module2.exports = factory(requireCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function(Math2) {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var WordArray = C_lib.WordArray;
        var Hasher = C_lib.Hasher;
        var C_algo = C2.algo;
        var T2 = [];
        (function() {
          for (var i2 = 0; i2 < 64; i2++) {
            T2[i2] = Math2.abs(Math2.sin(i2 + 1)) * 4294967296 | 0;
          }
        })();
        var MD5 = C_algo.MD5 = Hasher.extend({
          _doReset: function() {
            this._hash = new WordArray.init([
              1732584193,
              4023233417,
              2562383102,
              271733878
            ]);
          },
          _doProcessBlock: function(M2, offset2) {
            for (var i2 = 0; i2 < 16; i2++) {
              var offset_i = offset2 + i2;
              var M_offset_i = M2[offset_i];
              M2[offset_i] = (M_offset_i << 8 | M_offset_i >>> 24) & 16711935 | (M_offset_i << 24 | M_offset_i >>> 8) & 4278255360;
            }
            var H4 = this._hash.words;
            var M_offset_0 = M2[offset2 + 0];
            var M_offset_1 = M2[offset2 + 1];
            var M_offset_2 = M2[offset2 + 2];
            var M_offset_3 = M2[offset2 + 3];
            var M_offset_4 = M2[offset2 + 4];
            var M_offset_5 = M2[offset2 + 5];
            var M_offset_6 = M2[offset2 + 6];
            var M_offset_7 = M2[offset2 + 7];
            var M_offset_8 = M2[offset2 + 8];
            var M_offset_9 = M2[offset2 + 9];
            var M_offset_10 = M2[offset2 + 10];
            var M_offset_11 = M2[offset2 + 11];
            var M_offset_12 = M2[offset2 + 12];
            var M_offset_13 = M2[offset2 + 13];
            var M_offset_14 = M2[offset2 + 14];
            var M_offset_15 = M2[offset2 + 15];
            var a2 = H4[0];
            var b2 = H4[1];
            var c2 = H4[2];
            var d2 = H4[3];
            a2 = FF(a2, b2, c2, d2, M_offset_0, 7, T2[0]);
            d2 = FF(d2, a2, b2, c2, M_offset_1, 12, T2[1]);
            c2 = FF(c2, d2, a2, b2, M_offset_2, 17, T2[2]);
            b2 = FF(b2, c2, d2, a2, M_offset_3, 22, T2[3]);
            a2 = FF(a2, b2, c2, d2, M_offset_4, 7, T2[4]);
            d2 = FF(d2, a2, b2, c2, M_offset_5, 12, T2[5]);
            c2 = FF(c2, d2, a2, b2, M_offset_6, 17, T2[6]);
            b2 = FF(b2, c2, d2, a2, M_offset_7, 22, T2[7]);
            a2 = FF(a2, b2, c2, d2, M_offset_8, 7, T2[8]);
            d2 = FF(d2, a2, b2, c2, M_offset_9, 12, T2[9]);
            c2 = FF(c2, d2, a2, b2, M_offset_10, 17, T2[10]);
            b2 = FF(b2, c2, d2, a2, M_offset_11, 22, T2[11]);
            a2 = FF(a2, b2, c2, d2, M_offset_12, 7, T2[12]);
            d2 = FF(d2, a2, b2, c2, M_offset_13, 12, T2[13]);
            c2 = FF(c2, d2, a2, b2, M_offset_14, 17, T2[14]);
            b2 = FF(b2, c2, d2, a2, M_offset_15, 22, T2[15]);
            a2 = GG(a2, b2, c2, d2, M_offset_1, 5, T2[16]);
            d2 = GG(d2, a2, b2, c2, M_offset_6, 9, T2[17]);
            c2 = GG(c2, d2, a2, b2, M_offset_11, 14, T2[18]);
            b2 = GG(b2, c2, d2, a2, M_offset_0, 20, T2[19]);
            a2 = GG(a2, b2, c2, d2, M_offset_5, 5, T2[20]);
            d2 = GG(d2, a2, b2, c2, M_offset_10, 9, T2[21]);
            c2 = GG(c2, d2, a2, b2, M_offset_15, 14, T2[22]);
            b2 = GG(b2, c2, d2, a2, M_offset_4, 20, T2[23]);
            a2 = GG(a2, b2, c2, d2, M_offset_9, 5, T2[24]);
            d2 = GG(d2, a2, b2, c2, M_offset_14, 9, T2[25]);
            c2 = GG(c2, d2, a2, b2, M_offset_3, 14, T2[26]);
            b2 = GG(b2, c2, d2, a2, M_offset_8, 20, T2[27]);
            a2 = GG(a2, b2, c2, d2, M_offset_13, 5, T2[28]);
            d2 = GG(d2, a2, b2, c2, M_offset_2, 9, T2[29]);
            c2 = GG(c2, d2, a2, b2, M_offset_7, 14, T2[30]);
            b2 = GG(b2, c2, d2, a2, M_offset_12, 20, T2[31]);
            a2 = HH(a2, b2, c2, d2, M_offset_5, 4, T2[32]);
            d2 = HH(d2, a2, b2, c2, M_offset_8, 11, T2[33]);
            c2 = HH(c2, d2, a2, b2, M_offset_11, 16, T2[34]);
            b2 = HH(b2, c2, d2, a2, M_offset_14, 23, T2[35]);
            a2 = HH(a2, b2, c2, d2, M_offset_1, 4, T2[36]);
            d2 = HH(d2, a2, b2, c2, M_offset_4, 11, T2[37]);
            c2 = HH(c2, d2, a2, b2, M_offset_7, 16, T2[38]);
            b2 = HH(b2, c2, d2, a2, M_offset_10, 23, T2[39]);
            a2 = HH(a2, b2, c2, d2, M_offset_13, 4, T2[40]);
            d2 = HH(d2, a2, b2, c2, M_offset_0, 11, T2[41]);
            c2 = HH(c2, d2, a2, b2, M_offset_3, 16, T2[42]);
            b2 = HH(b2, c2, d2, a2, M_offset_6, 23, T2[43]);
            a2 = HH(a2, b2, c2, d2, M_offset_9, 4, T2[44]);
            d2 = HH(d2, a2, b2, c2, M_offset_12, 11, T2[45]);
            c2 = HH(c2, d2, a2, b2, M_offset_15, 16, T2[46]);
            b2 = HH(b2, c2, d2, a2, M_offset_2, 23, T2[47]);
            a2 = II(a2, b2, c2, d2, M_offset_0, 6, T2[48]);
            d2 = II(d2, a2, b2, c2, M_offset_7, 10, T2[49]);
            c2 = II(c2, d2, a2, b2, M_offset_14, 15, T2[50]);
            b2 = II(b2, c2, d2, a2, M_offset_5, 21, T2[51]);
            a2 = II(a2, b2, c2, d2, M_offset_12, 6, T2[52]);
            d2 = II(d2, a2, b2, c2, M_offset_3, 10, T2[53]);
            c2 = II(c2, d2, a2, b2, M_offset_10, 15, T2[54]);
            b2 = II(b2, c2, d2, a2, M_offset_1, 21, T2[55]);
            a2 = II(a2, b2, c2, d2, M_offset_8, 6, T2[56]);
            d2 = II(d2, a2, b2, c2, M_offset_15, 10, T2[57]);
            c2 = II(c2, d2, a2, b2, M_offset_6, 15, T2[58]);
            b2 = II(b2, c2, d2, a2, M_offset_13, 21, T2[59]);
            a2 = II(a2, b2, c2, d2, M_offset_4, 6, T2[60]);
            d2 = II(d2, a2, b2, c2, M_offset_11, 10, T2[61]);
            c2 = II(c2, d2, a2, b2, M_offset_2, 15, T2[62]);
            b2 = II(b2, c2, d2, a2, M_offset_9, 21, T2[63]);
            H4[0] = H4[0] + a2 | 0;
            H4[1] = H4[1] + b2 | 0;
            H4[2] = H4[2] + c2 | 0;
            H4[3] = H4[3] + d2 | 0;
          },
          _doFinalize: function() {
            var data2 = this._data;
            var dataWords = data2.words;
            var nBitsTotal = this._nDataBytes * 8;
            var nBitsLeft = data2.sigBytes * 8;
            dataWords[nBitsLeft >>> 5] |= 128 << 24 - nBitsLeft % 32;
            var nBitsTotalH = Math2.floor(nBitsTotal / 4294967296);
            var nBitsTotalL = nBitsTotal;
            dataWords[(nBitsLeft + 64 >>> 9 << 4) + 15] = (nBitsTotalH << 8 | nBitsTotalH >>> 24) & 16711935 | (nBitsTotalH << 24 | nBitsTotalH >>> 8) & 4278255360;
            dataWords[(nBitsLeft + 64 >>> 9 << 4) + 14] = (nBitsTotalL << 8 | nBitsTotalL >>> 24) & 16711935 | (nBitsTotalL << 24 | nBitsTotalL >>> 8) & 4278255360;
            data2.sigBytes = (dataWords.length + 1) * 4;
            this._process();
            var hash = this._hash;
            var H4 = hash.words;
            for (var i2 = 0; i2 < 4; i2++) {
              var H_i = H4[i2];
              H4[i2] = (H_i << 8 | H_i >>> 24) & 16711935 | (H_i << 24 | H_i >>> 8) & 4278255360;
            }
            return hash;
          },
          clone: function() {
            var clone2 = Hasher.clone.call(this);
            clone2._hash = this._hash.clone();
            return clone2;
          }
        });
        function FF(a2, b2, c2, d2, x2, s2, t3) {
          var n2 = a2 + (b2 & c2 | ~b2 & d2) + x2 + t3;
          return (n2 << s2 | n2 >>> 32 - s2) + b2;
        }
        function GG(a2, b2, c2, d2, x2, s2, t3) {
          var n2 = a2 + (b2 & d2 | c2 & ~d2) + x2 + t3;
          return (n2 << s2 | n2 >>> 32 - s2) + b2;
        }
        function HH(a2, b2, c2, d2, x2, s2, t3) {
          var n2 = a2 + (b2 ^ c2 ^ d2) + x2 + t3;
          return (n2 << s2 | n2 >>> 32 - s2) + b2;
        }
        function II(a2, b2, c2, d2, x2, s2, t3) {
          var n2 = a2 + (c2 ^ (b2 | ~d2)) + x2 + t3;
          return (n2 << s2 | n2 >>> 32 - s2) + b2;
        }
        C2.MD5 = Hasher._createHelper(MD5);
        C2.HmacMD5 = Hasher._createHmacHelper(MD5);
      })(Math);
      return CryptoJS2.MD5;
    });
  })(md5);
  return md5.exports;
}
var sha1 = { exports: {} };
var hasRequiredSha1;
function requireSha1() {
  if (hasRequiredSha1)
    return sha1.exports;
  hasRequiredSha1 = 1;
  (function(module2, exports) {
    (function(root2, factory) {
      {
        module2.exports = factory(requireCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var WordArray = C_lib.WordArray;
        var Hasher = C_lib.Hasher;
        var C_algo = C2.algo;
        var W2 = [];
        var SHA1 = C_algo.SHA1 = Hasher.extend({
          _doReset: function() {
            this._hash = new WordArray.init([
              1732584193,
              4023233417,
              2562383102,
              271733878,
              3285377520
            ]);
          },
          _doProcessBlock: function(M2, offset2) {
            var H4 = this._hash.words;
            var a2 = H4[0];
            var b2 = H4[1];
            var c2 = H4[2];
            var d2 = H4[3];
            var e2 = H4[4];
            for (var i2 = 0; i2 < 80; i2++) {
              if (i2 < 16) {
                W2[i2] = M2[offset2 + i2] | 0;
              } else {
                var n2 = W2[i2 - 3] ^ W2[i2 - 8] ^ W2[i2 - 14] ^ W2[i2 - 16];
                W2[i2] = n2 << 1 | n2 >>> 31;
              }
              var t3 = (a2 << 5 | a2 >>> 27) + e2 + W2[i2];
              if (i2 < 20) {
                t3 += (b2 & c2 | ~b2 & d2) + 1518500249;
              } else if (i2 < 40) {
                t3 += (b2 ^ c2 ^ d2) + 1859775393;
              } else if (i2 < 60) {
                t3 += (b2 & c2 | b2 & d2 | c2 & d2) - 1894007588;
              } else {
                t3 += (b2 ^ c2 ^ d2) - 899497514;
              }
              e2 = d2;
              d2 = c2;
              c2 = b2 << 30 | b2 >>> 2;
              b2 = a2;
              a2 = t3;
            }
            H4[0] = H4[0] + a2 | 0;
            H4[1] = H4[1] + b2 | 0;
            H4[2] = H4[2] + c2 | 0;
            H4[3] = H4[3] + d2 | 0;
            H4[4] = H4[4] + e2 | 0;
          },
          _doFinalize: function() {
            var data2 = this._data;
            var dataWords = data2.words;
            var nBitsTotal = this._nDataBytes * 8;
            var nBitsLeft = data2.sigBytes * 8;
            dataWords[nBitsLeft >>> 5] |= 128 << 24 - nBitsLeft % 32;
            dataWords[(nBitsLeft + 64 >>> 9 << 4) + 14] = Math.floor(nBitsTotal / 4294967296);
            dataWords[(nBitsLeft + 64 >>> 9 << 4) + 15] = nBitsTotal;
            data2.sigBytes = dataWords.length * 4;
            this._process();
            return this._hash;
          },
          clone: function() {
            var clone2 = Hasher.clone.call(this);
            clone2._hash = this._hash.clone();
            return clone2;
          }
        });
        C2.SHA1 = Hasher._createHelper(SHA1);
        C2.HmacSHA1 = Hasher._createHmacHelper(SHA1);
      })();
      return CryptoJS2.SHA1;
    });
  })(sha1);
  return sha1.exports;
}
var sha256 = { exports: {} };
var hasRequiredSha256;
function requireSha256() {
  if (hasRequiredSha256)
    return sha256.exports;
  hasRequiredSha256 = 1;
  (function(module2, exports) {
    (function(root2, factory) {
      {
        module2.exports = factory(requireCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function(Math2) {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var WordArray = C_lib.WordArray;
        var Hasher = C_lib.Hasher;
        var C_algo = C2.algo;
        var H4 = [];
        var K2 = [];
        (function() {
          function isPrime(n3) {
            var sqrtN = Math2.sqrt(n3);
            for (var factor = 2; factor <= sqrtN; factor++) {
              if (!(n3 % factor)) {
                return false;
              }
            }
            return true;
          }
          function getFractionalBits(n3) {
            return (n3 - (n3 | 0)) * 4294967296 | 0;
          }
          var n2 = 2;
          var nPrime = 0;
          while (nPrime < 64) {
            if (isPrime(n2)) {
              if (nPrime < 8) {
                H4[nPrime] = getFractionalBits(Math2.pow(n2, 1 / 2));
              }
              K2[nPrime] = getFractionalBits(Math2.pow(n2, 1 / 3));
              nPrime++;
            }
            n2++;
          }
        })();
        var W2 = [];
        var SHA256 = C_algo.SHA256 = Hasher.extend({
          _doReset: function() {
            this._hash = new WordArray.init(H4.slice(0));
          },
          _doProcessBlock: function(M2, offset2) {
            var H5 = this._hash.words;
            var a2 = H5[0];
            var b2 = H5[1];
            var c2 = H5[2];
            var d2 = H5[3];
            var e2 = H5[4];
            var f2 = H5[5];
            var g2 = H5[6];
            var h2 = H5[7];
            for (var i2 = 0; i2 < 64; i2++) {
              if (i2 < 16) {
                W2[i2] = M2[offset2 + i2] | 0;
              } else {
                var gamma0x = W2[i2 - 15];
                var gamma0 = (gamma0x << 25 | gamma0x >>> 7) ^ (gamma0x << 14 | gamma0x >>> 18) ^ gamma0x >>> 3;
                var gamma1x = W2[i2 - 2];
                var gamma1 = (gamma1x << 15 | gamma1x >>> 17) ^ (gamma1x << 13 | gamma1x >>> 19) ^ gamma1x >>> 10;
                W2[i2] = gamma0 + W2[i2 - 7] + gamma1 + W2[i2 - 16];
              }
              var ch = e2 & f2 ^ ~e2 & g2;
              var maj = a2 & b2 ^ a2 & c2 ^ b2 & c2;
              var sigma0 = (a2 << 30 | a2 >>> 2) ^ (a2 << 19 | a2 >>> 13) ^ (a2 << 10 | a2 >>> 22);
              var sigma1 = (e2 << 26 | e2 >>> 6) ^ (e2 << 21 | e2 >>> 11) ^ (e2 << 7 | e2 >>> 25);
              var t1 = h2 + sigma1 + ch + K2[i2] + W2[i2];
              var t22 = sigma0 + maj;
              h2 = g2;
              g2 = f2;
              f2 = e2;
              e2 = d2 + t1 | 0;
              d2 = c2;
              c2 = b2;
              b2 = a2;
              a2 = t1 + t22 | 0;
            }
            H5[0] = H5[0] + a2 | 0;
            H5[1] = H5[1] + b2 | 0;
            H5[2] = H5[2] + c2 | 0;
            H5[3] = H5[3] + d2 | 0;
            H5[4] = H5[4] + e2 | 0;
            H5[5] = H5[5] + f2 | 0;
            H5[6] = H5[6] + g2 | 0;
            H5[7] = H5[7] + h2 | 0;
          },
          _doFinalize: function() {
            var data2 = this._data;
            var dataWords = data2.words;
            var nBitsTotal = this._nDataBytes * 8;
            var nBitsLeft = data2.sigBytes * 8;
            dataWords[nBitsLeft >>> 5] |= 128 << 24 - nBitsLeft % 32;
            dataWords[(nBitsLeft + 64 >>> 9 << 4) + 14] = Math2.floor(nBitsTotal / 4294967296);
            dataWords[(nBitsLeft + 64 >>> 9 << 4) + 15] = nBitsTotal;
            data2.sigBytes = dataWords.length * 4;
            this._process();
            return this._hash;
          },
          clone: function() {
            var clone2 = Hasher.clone.call(this);
            clone2._hash = this._hash.clone();
            return clone2;
          }
        });
        C2.SHA256 = Hasher._createHelper(SHA256);
        C2.HmacSHA256 = Hasher._createHmacHelper(SHA256);
      })(Math);
      return CryptoJS2.SHA256;
    });
  })(sha256);
  return sha256.exports;
}
var sha224 = { exports: {} };
var hasRequiredSha224;
function requireSha224() {
  if (hasRequiredSha224)
    return sha224.exports;
  hasRequiredSha224 = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireSha256());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var WordArray = C_lib.WordArray;
        var C_algo = C2.algo;
        var SHA256 = C_algo.SHA256;
        var SHA224 = C_algo.SHA224 = SHA256.extend({
          _doReset: function() {
            this._hash = new WordArray.init([
              3238371032,
              914150663,
              812702999,
              4144912697,
              4290775857,
              1750603025,
              1694076839,
              3204075428
            ]);
          },
          _doFinalize: function() {
            var hash = SHA256._doFinalize.call(this);
            hash.sigBytes -= 4;
            return hash;
          }
        });
        C2.SHA224 = SHA256._createHelper(SHA224);
        C2.HmacSHA224 = SHA256._createHmacHelper(SHA224);
      })();
      return CryptoJS2.SHA224;
    });
  })(sha224);
  return sha224.exports;
}
var sha512 = { exports: {} };
var hasRequiredSha512;
function requireSha512() {
  if (hasRequiredSha512)
    return sha512.exports;
  hasRequiredSha512 = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireX64Core());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var Hasher = C_lib.Hasher;
        var C_x64 = C2.x64;
        var X64Word = C_x64.Word;
        var X64WordArray = C_x64.WordArray;
        var C_algo = C2.algo;
        function X64Word_create() {
          return X64Word.create.apply(X64Word, arguments);
        }
        var K2 = [
          X64Word_create(1116352408, 3609767458),
          X64Word_create(1899447441, 602891725),
          X64Word_create(3049323471, 3964484399),
          X64Word_create(3921009573, 2173295548),
          X64Word_create(961987163, 4081628472),
          X64Word_create(1508970993, 3053834265),
          X64Word_create(2453635748, 2937671579),
          X64Word_create(2870763221, 3664609560),
          X64Word_create(3624381080, 2734883394),
          X64Word_create(310598401, 1164996542),
          X64Word_create(607225278, 1323610764),
          X64Word_create(1426881987, 3590304994),
          X64Word_create(1925078388, 4068182383),
          X64Word_create(2162078206, 991336113),
          X64Word_create(2614888103, 633803317),
          X64Word_create(3248222580, 3479774868),
          X64Word_create(3835390401, 2666613458),
          X64Word_create(4022224774, 944711139),
          X64Word_create(264347078, 2341262773),
          X64Word_create(604807628, 2007800933),
          X64Word_create(770255983, 1495990901),
          X64Word_create(1249150122, 1856431235),
          X64Word_create(1555081692, 3175218132),
          X64Word_create(1996064986, 2198950837),
          X64Word_create(2554220882, 3999719339),
          X64Word_create(2821834349, 766784016),
          X64Word_create(2952996808, 2566594879),
          X64Word_create(3210313671, 3203337956),
          X64Word_create(3336571891, 1034457026),
          X64Word_create(3584528711, 2466948901),
          X64Word_create(113926993, 3758326383),
          X64Word_create(338241895, 168717936),
          X64Word_create(666307205, 1188179964),
          X64Word_create(773529912, 1546045734),
          X64Word_create(1294757372, 1522805485),
          X64Word_create(1396182291, 2643833823),
          X64Word_create(1695183700, 2343527390),
          X64Word_create(1986661051, 1014477480),
          X64Word_create(2177026350, 1206759142),
          X64Word_create(2456956037, 344077627),
          X64Word_create(2730485921, 1290863460),
          X64Word_create(2820302411, 3158454273),
          X64Word_create(3259730800, 3505952657),
          X64Word_create(3345764771, 106217008),
          X64Word_create(3516065817, 3606008344),
          X64Word_create(3600352804, 1432725776),
          X64Word_create(4094571909, 1467031594),
          X64Word_create(275423344, 851169720),
          X64Word_create(430227734, 3100823752),
          X64Word_create(506948616, 1363258195),
          X64Word_create(659060556, 3750685593),
          X64Word_create(883997877, 3785050280),
          X64Word_create(958139571, 3318307427),
          X64Word_create(1322822218, 3812723403),
          X64Word_create(1537002063, 2003034995),
          X64Word_create(1747873779, 3602036899),
          X64Word_create(1955562222, 1575990012),
          X64Word_create(2024104815, 1125592928),
          X64Word_create(2227730452, 2716904306),
          X64Word_create(2361852424, 442776044),
          X64Word_create(2428436474, 593698344),
          X64Word_create(2756734187, 3733110249),
          X64Word_create(3204031479, 2999351573),
          X64Word_create(3329325298, 3815920427),
          X64Word_create(3391569614, 3928383900),
          X64Word_create(3515267271, 566280711),
          X64Word_create(3940187606, 3454069534),
          X64Word_create(4118630271, 4000239992),
          X64Word_create(116418474, 1914138554),
          X64Word_create(174292421, 2731055270),
          X64Word_create(289380356, 3203993006),
          X64Word_create(460393269, 320620315),
          X64Word_create(685471733, 587496836),
          X64Word_create(852142971, 1086792851),
          X64Word_create(1017036298, 365543100),
          X64Word_create(1126000580, 2618297676),
          X64Word_create(1288033470, 3409855158),
          X64Word_create(1501505948, 4234509866),
          X64Word_create(1607167915, 987167468),
          X64Word_create(1816402316, 1246189591)
        ];
        var W2 = [];
        (function() {
          for (var i2 = 0; i2 < 80; i2++) {
            W2[i2] = X64Word_create();
          }
        })();
        var SHA512 = C_algo.SHA512 = Hasher.extend({
          _doReset: function() {
            this._hash = new X64WordArray.init([
              new X64Word.init(1779033703, 4089235720),
              new X64Word.init(3144134277, 2227873595),
              new X64Word.init(1013904242, 4271175723),
              new X64Word.init(2773480762, 1595750129),
              new X64Word.init(1359893119, 2917565137),
              new X64Word.init(2600822924, 725511199),
              new X64Word.init(528734635, 4215389547),
              new X64Word.init(1541459225, 327033209)
            ]);
          },
          _doProcessBlock: function(M2, offset2) {
            var H4 = this._hash.words;
            var H0 = H4[0];
            var H1 = H4[1];
            var H22 = H4[2];
            var H32 = H4[3];
            var H42 = H4[4];
            var H5 = H4[5];
            var H6 = H4[6];
            var H7 = H4[7];
            var H0h = H0.high;
            var H0l = H0.low;
            var H1h = H1.high;
            var H1l = H1.low;
            var H2h = H22.high;
            var H2l = H22.low;
            var H3h = H32.high;
            var H3l = H32.low;
            var H4h = H42.high;
            var H4l = H42.low;
            var H5h = H5.high;
            var H5l = H5.low;
            var H6h = H6.high;
            var H6l = H6.low;
            var H7h = H7.high;
            var H7l = H7.low;
            var ah = H0h;
            var al = H0l;
            var bh = H1h;
            var bl = H1l;
            var ch = H2h;
            var cl = H2l;
            var dh = H3h;
            var dl = H3l;
            var eh = H4h;
            var el = H4l;
            var fh = H5h;
            var fl2 = H5l;
            var gh = H6h;
            var gl = H6l;
            var hh = H7h;
            var hl = H7l;
            for (var i2 = 0; i2 < 80; i2++) {
              var Wil;
              var Wih;
              var Wi = W2[i2];
              if (i2 < 16) {
                Wih = Wi.high = M2[offset2 + i2 * 2] | 0;
                Wil = Wi.low = M2[offset2 + i2 * 2 + 1] | 0;
              } else {
                var gamma0x = W2[i2 - 15];
                var gamma0xh = gamma0x.high;
                var gamma0xl = gamma0x.low;
                var gamma0h = (gamma0xh >>> 1 | gamma0xl << 31) ^ (gamma0xh >>> 8 | gamma0xl << 24) ^ gamma0xh >>> 7;
                var gamma0l = (gamma0xl >>> 1 | gamma0xh << 31) ^ (gamma0xl >>> 8 | gamma0xh << 24) ^ (gamma0xl >>> 7 | gamma0xh << 25);
                var gamma1x = W2[i2 - 2];
                var gamma1xh = gamma1x.high;
                var gamma1xl = gamma1x.low;
                var gamma1h = (gamma1xh >>> 19 | gamma1xl << 13) ^ (gamma1xh << 3 | gamma1xl >>> 29) ^ gamma1xh >>> 6;
                var gamma1l = (gamma1xl >>> 19 | gamma1xh << 13) ^ (gamma1xl << 3 | gamma1xh >>> 29) ^ (gamma1xl >>> 6 | gamma1xh << 26);
                var Wi7 = W2[i2 - 7];
                var Wi7h = Wi7.high;
                var Wi7l = Wi7.low;
                var Wi16 = W2[i2 - 16];
                var Wi16h = Wi16.high;
                var Wi16l = Wi16.low;
                Wil = gamma0l + Wi7l;
                Wih = gamma0h + Wi7h + (Wil >>> 0 < gamma0l >>> 0 ? 1 : 0);
                Wil = Wil + gamma1l;
                Wih = Wih + gamma1h + (Wil >>> 0 < gamma1l >>> 0 ? 1 : 0);
                Wil = Wil + Wi16l;
                Wih = Wih + Wi16h + (Wil >>> 0 < Wi16l >>> 0 ? 1 : 0);
                Wi.high = Wih;
                Wi.low = Wil;
              }
              var chh = eh & fh ^ ~eh & gh;
              var chl = el & fl2 ^ ~el & gl;
              var majh = ah & bh ^ ah & ch ^ bh & ch;
              var majl = al & bl ^ al & cl ^ bl & cl;
              var sigma0h = (ah >>> 28 | al << 4) ^ (ah << 30 | al >>> 2) ^ (ah << 25 | al >>> 7);
              var sigma0l = (al >>> 28 | ah << 4) ^ (al << 30 | ah >>> 2) ^ (al << 25 | ah >>> 7);
              var sigma1h = (eh >>> 14 | el << 18) ^ (eh >>> 18 | el << 14) ^ (eh << 23 | el >>> 9);
              var sigma1l = (el >>> 14 | eh << 18) ^ (el >>> 18 | eh << 14) ^ (el << 23 | eh >>> 9);
              var Ki = K2[i2];
              var Kih = Ki.high;
              var Kil = Ki.low;
              var t1l = hl + sigma1l;
              var t1h = hh + sigma1h + (t1l >>> 0 < hl >>> 0 ? 1 : 0);
              var t1l = t1l + chl;
              var t1h = t1h + chh + (t1l >>> 0 < chl >>> 0 ? 1 : 0);
              var t1l = t1l + Kil;
              var t1h = t1h + Kih + (t1l >>> 0 < Kil >>> 0 ? 1 : 0);
              var t1l = t1l + Wil;
              var t1h = t1h + Wih + (t1l >>> 0 < Wil >>> 0 ? 1 : 0);
              var t2l = sigma0l + majl;
              var t2h = sigma0h + majh + (t2l >>> 0 < sigma0l >>> 0 ? 1 : 0);
              hh = gh;
              hl = gl;
              gh = fh;
              gl = fl2;
              fh = eh;
              fl2 = el;
              el = dl + t1l | 0;
              eh = dh + t1h + (el >>> 0 < dl >>> 0 ? 1 : 0) | 0;
              dh = ch;
              dl = cl;
              ch = bh;
              cl = bl;
              bh = ah;
              bl = al;
              al = t1l + t2l | 0;
              ah = t1h + t2h + (al >>> 0 < t1l >>> 0 ? 1 : 0) | 0;
            }
            H0l = H0.low = H0l + al;
            H0.high = H0h + ah + (H0l >>> 0 < al >>> 0 ? 1 : 0);
            H1l = H1.low = H1l + bl;
            H1.high = H1h + bh + (H1l >>> 0 < bl >>> 0 ? 1 : 0);
            H2l = H22.low = H2l + cl;
            H22.high = H2h + ch + (H2l >>> 0 < cl >>> 0 ? 1 : 0);
            H3l = H32.low = H3l + dl;
            H32.high = H3h + dh + (H3l >>> 0 < dl >>> 0 ? 1 : 0);
            H4l = H42.low = H4l + el;
            H42.high = H4h + eh + (H4l >>> 0 < el >>> 0 ? 1 : 0);
            H5l = H5.low = H5l + fl2;
            H5.high = H5h + fh + (H5l >>> 0 < fl2 >>> 0 ? 1 : 0);
            H6l = H6.low = H6l + gl;
            H6.high = H6h + gh + (H6l >>> 0 < gl >>> 0 ? 1 : 0);
            H7l = H7.low = H7l + hl;
            H7.high = H7h + hh + (H7l >>> 0 < hl >>> 0 ? 1 : 0);
          },
          _doFinalize: function() {
            var data2 = this._data;
            var dataWords = data2.words;
            var nBitsTotal = this._nDataBytes * 8;
            var nBitsLeft = data2.sigBytes * 8;
            dataWords[nBitsLeft >>> 5] |= 128 << 24 - nBitsLeft % 32;
            dataWords[(nBitsLeft + 128 >>> 10 << 5) + 30] = Math.floor(nBitsTotal / 4294967296);
            dataWords[(nBitsLeft + 128 >>> 10 << 5) + 31] = nBitsTotal;
            data2.sigBytes = dataWords.length * 4;
            this._process();
            var hash = this._hash.toX32();
            return hash;
          },
          clone: function() {
            var clone2 = Hasher.clone.call(this);
            clone2._hash = this._hash.clone();
            return clone2;
          },
          blockSize: 1024 / 32
        });
        C2.SHA512 = Hasher._createHelper(SHA512);
        C2.HmacSHA512 = Hasher._createHmacHelper(SHA512);
      })();
      return CryptoJS2.SHA512;
    });
  })(sha512);
  return sha512.exports;
}
var sha384 = { exports: {} };
var hasRequiredSha384;
function requireSha384() {
  if (hasRequiredSha384)
    return sha384.exports;
  hasRequiredSha384 = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireX64Core(), requireSha512());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_x64 = C2.x64;
        var X64Word = C_x64.Word;
        var X64WordArray = C_x64.WordArray;
        var C_algo = C2.algo;
        var SHA512 = C_algo.SHA512;
        var SHA384 = C_algo.SHA384 = SHA512.extend({
          _doReset: function() {
            this._hash = new X64WordArray.init([
              new X64Word.init(3418070365, 3238371032),
              new X64Word.init(1654270250, 914150663),
              new X64Word.init(2438529370, 812702999),
              new X64Word.init(355462360, 4144912697),
              new X64Word.init(1731405415, 4290775857),
              new X64Word.init(2394180231, 1750603025),
              new X64Word.init(3675008525, 1694076839),
              new X64Word.init(1203062813, 3204075428)
            ]);
          },
          _doFinalize: function() {
            var hash = SHA512._doFinalize.call(this);
            hash.sigBytes -= 16;
            return hash;
          }
        });
        C2.SHA384 = SHA512._createHelper(SHA384);
        C2.HmacSHA384 = SHA512._createHmacHelper(SHA384);
      })();
      return CryptoJS2.SHA384;
    });
  })(sha384);
  return sha384.exports;
}
var sha3 = { exports: {} };
var hasRequiredSha3;
function requireSha3() {
  if (hasRequiredSha3)
    return sha3.exports;
  hasRequiredSha3 = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireX64Core());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function(Math2) {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var WordArray = C_lib.WordArray;
        var Hasher = C_lib.Hasher;
        var C_x64 = C2.x64;
        var X64Word = C_x64.Word;
        var C_algo = C2.algo;
        var RHO_OFFSETS = [];
        var PI_INDEXES = [];
        var ROUND_CONSTANTS = [];
        (function() {
          var x2 = 1, y2 = 0;
          for (var t3 = 0; t3 < 24; t3++) {
            RHO_OFFSETS[x2 + 5 * y2] = (t3 + 1) * (t3 + 2) / 2 % 64;
            var newX = y2 % 5;
            var newY = (2 * x2 + 3 * y2) % 5;
            x2 = newX;
            y2 = newY;
          }
          for (var x2 = 0; x2 < 5; x2++) {
            for (var y2 = 0; y2 < 5; y2++) {
              PI_INDEXES[x2 + 5 * y2] = y2 + (2 * x2 + 3 * y2) % 5 * 5;
            }
          }
          var LFSR = 1;
          for (var i2 = 0; i2 < 24; i2++) {
            var roundConstantMsw = 0;
            var roundConstantLsw = 0;
            for (var j2 = 0; j2 < 7; j2++) {
              if (LFSR & 1) {
                var bitPosition = (1 << j2) - 1;
                if (bitPosition < 32) {
                  roundConstantLsw ^= 1 << bitPosition;
                } else {
                  roundConstantMsw ^= 1 << bitPosition - 32;
                }
              }
              if (LFSR & 128) {
                LFSR = LFSR << 1 ^ 113;
              } else {
                LFSR <<= 1;
              }
            }
            ROUND_CONSTANTS[i2] = X64Word.create(roundConstantMsw, roundConstantLsw);
          }
        })();
        var T2 = [];
        (function() {
          for (var i2 = 0; i2 < 25; i2++) {
            T2[i2] = X64Word.create();
          }
        })();
        var SHA3 = C_algo.SHA3 = Hasher.extend({
          /**
           * Configuration options.
           *
           * @property {number} outputLength
           *   The desired number of bits in the output hash.
           *   Only values permitted are: 224, 256, 384, 512.
           *   Default: 512
           */
          cfg: Hasher.cfg.extend({
            outputLength: 512
          }),
          _doReset: function() {
            var state = this._state = [];
            for (var i2 = 0; i2 < 25; i2++) {
              state[i2] = new X64Word.init();
            }
            this.blockSize = (1600 - 2 * this.cfg.outputLength) / 32;
          },
          _doProcessBlock: function(M2, offset2) {
            var state = this._state;
            var nBlockSizeLanes = this.blockSize / 2;
            for (var i2 = 0; i2 < nBlockSizeLanes; i2++) {
              var M2i = M2[offset2 + 2 * i2];
              var M2i1 = M2[offset2 + 2 * i2 + 1];
              M2i = (M2i << 8 | M2i >>> 24) & 16711935 | (M2i << 24 | M2i >>> 8) & 4278255360;
              M2i1 = (M2i1 << 8 | M2i1 >>> 24) & 16711935 | (M2i1 << 24 | M2i1 >>> 8) & 4278255360;
              var lane = state[i2];
              lane.high ^= M2i1;
              lane.low ^= M2i;
            }
            for (var round2 = 0; round2 < 24; round2++) {
              for (var x2 = 0; x2 < 5; x2++) {
                var tMsw = 0, tLsw = 0;
                for (var y2 = 0; y2 < 5; y2++) {
                  var lane = state[x2 + 5 * y2];
                  tMsw ^= lane.high;
                  tLsw ^= lane.low;
                }
                var Tx = T2[x2];
                Tx.high = tMsw;
                Tx.low = tLsw;
              }
              for (var x2 = 0; x2 < 5; x2++) {
                var Tx4 = T2[(x2 + 4) % 5];
                var Tx1 = T2[(x2 + 1) % 5];
                var Tx1Msw = Tx1.high;
                var Tx1Lsw = Tx1.low;
                var tMsw = Tx4.high ^ (Tx1Msw << 1 | Tx1Lsw >>> 31);
                var tLsw = Tx4.low ^ (Tx1Lsw << 1 | Tx1Msw >>> 31);
                for (var y2 = 0; y2 < 5; y2++) {
                  var lane = state[x2 + 5 * y2];
                  lane.high ^= tMsw;
                  lane.low ^= tLsw;
                }
              }
              for (var laneIndex = 1; laneIndex < 25; laneIndex++) {
                var tMsw;
                var tLsw;
                var lane = state[laneIndex];
                var laneMsw = lane.high;
                var laneLsw = lane.low;
                var rhoOffset = RHO_OFFSETS[laneIndex];
                if (rhoOffset < 32) {
                  tMsw = laneMsw << rhoOffset | laneLsw >>> 32 - rhoOffset;
                  tLsw = laneLsw << rhoOffset | laneMsw >>> 32 - rhoOffset;
                } else {
                  tMsw = laneLsw << rhoOffset - 32 | laneMsw >>> 64 - rhoOffset;
                  tLsw = laneMsw << rhoOffset - 32 | laneLsw >>> 64 - rhoOffset;
                }
                var TPiLane = T2[PI_INDEXES[laneIndex]];
                TPiLane.high = tMsw;
                TPiLane.low = tLsw;
              }
              var T0 = T2[0];
              var state0 = state[0];
              T0.high = state0.high;
              T0.low = state0.low;
              for (var x2 = 0; x2 < 5; x2++) {
                for (var y2 = 0; y2 < 5; y2++) {
                  var laneIndex = x2 + 5 * y2;
                  var lane = state[laneIndex];
                  var TLane = T2[laneIndex];
                  var Tx1Lane = T2[(x2 + 1) % 5 + 5 * y2];
                  var Tx2Lane = T2[(x2 + 2) % 5 + 5 * y2];
                  lane.high = TLane.high ^ ~Tx1Lane.high & Tx2Lane.high;
                  lane.low = TLane.low ^ ~Tx1Lane.low & Tx2Lane.low;
                }
              }
              var lane = state[0];
              var roundConstant = ROUND_CONSTANTS[round2];
              lane.high ^= roundConstant.high;
              lane.low ^= roundConstant.low;
            }
          },
          _doFinalize: function() {
            var data2 = this._data;
            var dataWords = data2.words;
            this._nDataBytes * 8;
            var nBitsLeft = data2.sigBytes * 8;
            var blockSizeBits = this.blockSize * 32;
            dataWords[nBitsLeft >>> 5] |= 1 << 24 - nBitsLeft % 32;
            dataWords[(Math2.ceil((nBitsLeft + 1) / blockSizeBits) * blockSizeBits >>> 5) - 1] |= 128;
            data2.sigBytes = dataWords.length * 4;
            this._process();
            var state = this._state;
            var outputLengthBytes = this.cfg.outputLength / 8;
            var outputLengthLanes = outputLengthBytes / 8;
            var hashWords = [];
            for (var i2 = 0; i2 < outputLengthLanes; i2++) {
              var lane = state[i2];
              var laneMsw = lane.high;
              var laneLsw = lane.low;
              laneMsw = (laneMsw << 8 | laneMsw >>> 24) & 16711935 | (laneMsw << 24 | laneMsw >>> 8) & 4278255360;
              laneLsw = (laneLsw << 8 | laneLsw >>> 24) & 16711935 | (laneLsw << 24 | laneLsw >>> 8) & 4278255360;
              hashWords.push(laneLsw);
              hashWords.push(laneMsw);
            }
            return new WordArray.init(hashWords, outputLengthBytes);
          },
          clone: function() {
            var clone2 = Hasher.clone.call(this);
            var state = clone2._state = this._state.slice(0);
            for (var i2 = 0; i2 < 25; i2++) {
              state[i2] = state[i2].clone();
            }
            return clone2;
          }
        });
        C2.SHA3 = Hasher._createHelper(SHA3);
        C2.HmacSHA3 = Hasher._createHmacHelper(SHA3);
      })(Math);
      return CryptoJS2.SHA3;
    });
  })(sha3);
  return sha3.exports;
}
var ripemd160 = { exports: {} };
var hasRequiredRipemd160;
function requireRipemd160() {
  if (hasRequiredRipemd160)
    return ripemd160.exports;
  hasRequiredRipemd160 = 1;
  (function(module2, exports) {
    (function(root2, factory) {
      {
        module2.exports = factory(requireCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      /** @preserve
      			(c) 2012 by Cédric Mesnil. All rights reserved.
      
      			Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
      
      			    - Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
      			    - Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
      
      			THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
      			*/
      (function(Math2) {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var WordArray = C_lib.WordArray;
        var Hasher = C_lib.Hasher;
        var C_algo = C2.algo;
        var _zl = WordArray.create([
          0,
          1,
          2,
          3,
          4,
          5,
          6,
          7,
          8,
          9,
          10,
          11,
          12,
          13,
          14,
          15,
          7,
          4,
          13,
          1,
          10,
          6,
          15,
          3,
          12,
          0,
          9,
          5,
          2,
          14,
          11,
          8,
          3,
          10,
          14,
          4,
          9,
          15,
          8,
          1,
          2,
          7,
          0,
          6,
          13,
          11,
          5,
          12,
          1,
          9,
          11,
          10,
          0,
          8,
          12,
          4,
          13,
          3,
          7,
          15,
          14,
          5,
          6,
          2,
          4,
          0,
          5,
          9,
          7,
          12,
          2,
          10,
          14,
          1,
          3,
          8,
          11,
          6,
          15,
          13
        ]);
        var _zr = WordArray.create([
          5,
          14,
          7,
          0,
          9,
          2,
          11,
          4,
          13,
          6,
          15,
          8,
          1,
          10,
          3,
          12,
          6,
          11,
          3,
          7,
          0,
          13,
          5,
          10,
          14,
          15,
          8,
          12,
          4,
          9,
          1,
          2,
          15,
          5,
          1,
          3,
          7,
          14,
          6,
          9,
          11,
          8,
          12,
          2,
          10,
          0,
          4,
          13,
          8,
          6,
          4,
          1,
          3,
          11,
          15,
          0,
          5,
          12,
          2,
          13,
          9,
          7,
          10,
          14,
          12,
          15,
          10,
          4,
          1,
          5,
          8,
          7,
          6,
          2,
          13,
          14,
          0,
          3,
          9,
          11
        ]);
        var _sl = WordArray.create([
          11,
          14,
          15,
          12,
          5,
          8,
          7,
          9,
          11,
          13,
          14,
          15,
          6,
          7,
          9,
          8,
          7,
          6,
          8,
          13,
          11,
          9,
          7,
          15,
          7,
          12,
          15,
          9,
          11,
          7,
          13,
          12,
          11,
          13,
          6,
          7,
          14,
          9,
          13,
          15,
          14,
          8,
          13,
          6,
          5,
          12,
          7,
          5,
          11,
          12,
          14,
          15,
          14,
          15,
          9,
          8,
          9,
          14,
          5,
          6,
          8,
          6,
          5,
          12,
          9,
          15,
          5,
          11,
          6,
          8,
          13,
          12,
          5,
          12,
          13,
          14,
          11,
          8,
          5,
          6
        ]);
        var _sr = WordArray.create([
          8,
          9,
          9,
          11,
          13,
          15,
          15,
          5,
          7,
          7,
          8,
          11,
          14,
          14,
          12,
          6,
          9,
          13,
          15,
          7,
          12,
          8,
          9,
          11,
          7,
          7,
          12,
          7,
          6,
          15,
          13,
          11,
          9,
          7,
          15,
          11,
          8,
          6,
          6,
          14,
          12,
          13,
          5,
          14,
          13,
          13,
          7,
          5,
          15,
          5,
          8,
          11,
          14,
          14,
          6,
          14,
          6,
          9,
          12,
          9,
          12,
          5,
          15,
          8,
          8,
          5,
          12,
          9,
          12,
          5,
          14,
          6,
          8,
          13,
          6,
          5,
          15,
          13,
          11,
          11
        ]);
        var _hl = WordArray.create([0, 1518500249, 1859775393, 2400959708, 2840853838]);
        var _hr = WordArray.create([1352829926, 1548603684, 1836072691, 2053994217, 0]);
        var RIPEMD160 = C_algo.RIPEMD160 = Hasher.extend({
          _doReset: function() {
            this._hash = WordArray.create([1732584193, 4023233417, 2562383102, 271733878, 3285377520]);
          },
          _doProcessBlock: function(M2, offset2) {
            for (var i2 = 0; i2 < 16; i2++) {
              var offset_i = offset2 + i2;
              var M_offset_i = M2[offset_i];
              M2[offset_i] = (M_offset_i << 8 | M_offset_i >>> 24) & 16711935 | (M_offset_i << 24 | M_offset_i >>> 8) & 4278255360;
            }
            var H4 = this._hash.words;
            var hl = _hl.words;
            var hr = _hr.words;
            var zl = _zl.words;
            var zr = _zr.words;
            var sl = _sl.words;
            var sr = _sr.words;
            var al, bl, cl, dl, el;
            var ar, br, cr, dr, er;
            ar = al = H4[0];
            br = bl = H4[1];
            cr = cl = H4[2];
            dr = dl = H4[3];
            er = el = H4[4];
            var t3;
            for (var i2 = 0; i2 < 80; i2 += 1) {
              t3 = al + M2[offset2 + zl[i2]] | 0;
              if (i2 < 16) {
                t3 += f1(bl, cl, dl) + hl[0];
              } else if (i2 < 32) {
                t3 += f2(bl, cl, dl) + hl[1];
              } else if (i2 < 48) {
                t3 += f3(bl, cl, dl) + hl[2];
              } else if (i2 < 64) {
                t3 += f4(bl, cl, dl) + hl[3];
              } else {
                t3 += f5(bl, cl, dl) + hl[4];
              }
              t3 = t3 | 0;
              t3 = rotl(t3, sl[i2]);
              t3 = t3 + el | 0;
              al = el;
              el = dl;
              dl = rotl(cl, 10);
              cl = bl;
              bl = t3;
              t3 = ar + M2[offset2 + zr[i2]] | 0;
              if (i2 < 16) {
                t3 += f5(br, cr, dr) + hr[0];
              } else if (i2 < 32) {
                t3 += f4(br, cr, dr) + hr[1];
              } else if (i2 < 48) {
                t3 += f3(br, cr, dr) + hr[2];
              } else if (i2 < 64) {
                t3 += f2(br, cr, dr) + hr[3];
              } else {
                t3 += f1(br, cr, dr) + hr[4];
              }
              t3 = t3 | 0;
              t3 = rotl(t3, sr[i2]);
              t3 = t3 + er | 0;
              ar = er;
              er = dr;
              dr = rotl(cr, 10);
              cr = br;
              br = t3;
            }
            t3 = H4[1] + cl + dr | 0;
            H4[1] = H4[2] + dl + er | 0;
            H4[2] = H4[3] + el + ar | 0;
            H4[3] = H4[4] + al + br | 0;
            H4[4] = H4[0] + bl + cr | 0;
            H4[0] = t3;
          },
          _doFinalize: function() {
            var data2 = this._data;
            var dataWords = data2.words;
            var nBitsTotal = this._nDataBytes * 8;
            var nBitsLeft = data2.sigBytes * 8;
            dataWords[nBitsLeft >>> 5] |= 128 << 24 - nBitsLeft % 32;
            dataWords[(nBitsLeft + 64 >>> 9 << 4) + 14] = (nBitsTotal << 8 | nBitsTotal >>> 24) & 16711935 | (nBitsTotal << 24 | nBitsTotal >>> 8) & 4278255360;
            data2.sigBytes = (dataWords.length + 1) * 4;
            this._process();
            var hash = this._hash;
            var H4 = hash.words;
            for (var i2 = 0; i2 < 5; i2++) {
              var H_i = H4[i2];
              H4[i2] = (H_i << 8 | H_i >>> 24) & 16711935 | (H_i << 24 | H_i >>> 8) & 4278255360;
            }
            return hash;
          },
          clone: function() {
            var clone2 = Hasher.clone.call(this);
            clone2._hash = this._hash.clone();
            return clone2;
          }
        });
        function f1(x2, y2, z2) {
          return x2 ^ y2 ^ z2;
        }
        function f2(x2, y2, z2) {
          return x2 & y2 | ~x2 & z2;
        }
        function f3(x2, y2, z2) {
          return (x2 | ~y2) ^ z2;
        }
        function f4(x2, y2, z2) {
          return x2 & z2 | y2 & ~z2;
        }
        function f5(x2, y2, z2) {
          return x2 ^ (y2 | ~z2);
        }
        function rotl(x2, n2) {
          return x2 << n2 | x2 >>> 32 - n2;
        }
        C2.RIPEMD160 = Hasher._createHelper(RIPEMD160);
        C2.HmacRIPEMD160 = Hasher._createHmacHelper(RIPEMD160);
      })();
      return CryptoJS2.RIPEMD160;
    });
  })(ripemd160);
  return ripemd160.exports;
}
var hmac = { exports: {} };
var hasRequiredHmac;
function requireHmac() {
  if (hasRequiredHmac)
    return hmac.exports;
  hasRequiredHmac = 1;
  (function(module2, exports) {
    (function(root2, factory) {
      {
        module2.exports = factory(requireCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var Base = C_lib.Base;
        var C_enc = C2.enc;
        var Utf8 = C_enc.Utf8;
        var C_algo = C2.algo;
        C_algo.HMAC = Base.extend({
          /**
           * Initializes a newly created HMAC.
           *
           * @param {Hasher} hasher The hash algorithm to use.
           * @param {WordArray|string} key The secret key.
           *
           * @example
           *
           *     var hmacHasher = CryptoJS.algo.HMAC.create(CryptoJS.algo.SHA256, key);
           */
          init: function(hasher, key) {
            hasher = this._hasher = new hasher.init();
            if (typeof key == "string") {
              key = Utf8.parse(key);
            }
            var hasherBlockSize = hasher.blockSize;
            var hasherBlockSizeBytes = hasherBlockSize * 4;
            if (key.sigBytes > hasherBlockSizeBytes) {
              key = hasher.finalize(key);
            }
            key.clamp();
            var oKey = this._oKey = key.clone();
            var iKey = this._iKey = key.clone();
            var oKeyWords = oKey.words;
            var iKeyWords = iKey.words;
            for (var i2 = 0; i2 < hasherBlockSize; i2++) {
              oKeyWords[i2] ^= 1549556828;
              iKeyWords[i2] ^= 909522486;
            }
            oKey.sigBytes = iKey.sigBytes = hasherBlockSizeBytes;
            this.reset();
          },
          /**
           * Resets this HMAC to its initial state.
           *
           * @example
           *
           *     hmacHasher.reset();
           */
          reset: function() {
            var hasher = this._hasher;
            hasher.reset();
            hasher.update(this._iKey);
          },
          /**
           * Updates this HMAC with a message.
           *
           * @param {WordArray|string} messageUpdate The message to append.
           *
           * @return {HMAC} This HMAC instance.
           *
           * @example
           *
           *     hmacHasher.update('message');
           *     hmacHasher.update(wordArray);
           */
          update: function(messageUpdate) {
            this._hasher.update(messageUpdate);
            return this;
          },
          /**
           * Finalizes the HMAC computation.
           * Note that the finalize operation is effectively a destructive, read-once operation.
           *
           * @param {WordArray|string} messageUpdate (Optional) A final message update.
           *
           * @return {WordArray} The HMAC.
           *
           * @example
           *
           *     var hmac = hmacHasher.finalize();
           *     var hmac = hmacHasher.finalize('message');
           *     var hmac = hmacHasher.finalize(wordArray);
           */
          finalize: function(messageUpdate) {
            var hasher = this._hasher;
            var innerHash = hasher.finalize(messageUpdate);
            hasher.reset();
            var hmac2 = hasher.finalize(this._oKey.clone().concat(innerHash));
            return hmac2;
          }
        });
      })();
    });
  })(hmac);
  return hmac.exports;
}
var pbkdf2 = { exports: {} };
var hasRequiredPbkdf2;
function requirePbkdf2() {
  if (hasRequiredPbkdf2)
    return pbkdf2.exports;
  hasRequiredPbkdf2 = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireSha256(), requireHmac());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var Base = C_lib.Base;
        var WordArray = C_lib.WordArray;
        var C_algo = C2.algo;
        var SHA256 = C_algo.SHA256;
        var HMAC = C_algo.HMAC;
        var PBKDF2 = C_algo.PBKDF2 = Base.extend({
          /**
           * Configuration options.
           *
           * @property {number} keySize The key size in words to generate. Default: 4 (128 bits)
           * @property {Hasher} hasher The hasher to use. Default: SHA256
           * @property {number} iterations The number of iterations to perform. Default: 250000
           */
          cfg: Base.extend({
            keySize: 128 / 32,
            hasher: SHA256,
            iterations: 25e4
          }),
          /**
           * Initializes a newly created key derivation function.
           *
           * @param {Object} cfg (Optional) The configuration options to use for the derivation.
           *
           * @example
           *
           *     var kdf = CryptoJS.algo.PBKDF2.create();
           *     var kdf = CryptoJS.algo.PBKDF2.create({ keySize: 8 });
           *     var kdf = CryptoJS.algo.PBKDF2.create({ keySize: 8, iterations: 1000 });
           */
          init: function(cfg) {
            this.cfg = this.cfg.extend(cfg);
          },
          /**
           * Computes the Password-Based Key Derivation Function 2.
           *
           * @param {WordArray|string} password The password.
           * @param {WordArray|string} salt A salt.
           *
           * @return {WordArray} The derived key.
           *
           * @example
           *
           *     var key = kdf.compute(password, salt);
           */
          compute: function(password, salt) {
            var cfg = this.cfg;
            var hmac2 = HMAC.create(cfg.hasher, password);
            var derivedKey = WordArray.create();
            var blockIndex = WordArray.create([1]);
            var derivedKeyWords = derivedKey.words;
            var blockIndexWords = blockIndex.words;
            var keySize = cfg.keySize;
            var iterations = cfg.iterations;
            while (derivedKeyWords.length < keySize) {
              var block = hmac2.update(salt).finalize(blockIndex);
              hmac2.reset();
              var blockWords = block.words;
              var blockWordsLength = blockWords.length;
              var intermediate = block;
              for (var i2 = 1; i2 < iterations; i2++) {
                intermediate = hmac2.finalize(intermediate);
                hmac2.reset();
                var intermediateWords = intermediate.words;
                for (var j2 = 0; j2 < blockWordsLength; j2++) {
                  blockWords[j2] ^= intermediateWords[j2];
                }
              }
              derivedKey.concat(block);
              blockIndexWords[0]++;
            }
            derivedKey.sigBytes = keySize * 4;
            return derivedKey;
          }
        });
        C2.PBKDF2 = function(password, salt, cfg) {
          return PBKDF2.create(cfg).compute(password, salt);
        };
      })();
      return CryptoJS2.PBKDF2;
    });
  })(pbkdf2);
  return pbkdf2.exports;
}
var evpkdf = { exports: {} };
var hasRequiredEvpkdf;
function requireEvpkdf() {
  if (hasRequiredEvpkdf)
    return evpkdf.exports;
  hasRequiredEvpkdf = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireSha1(), requireHmac());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var Base = C_lib.Base;
        var WordArray = C_lib.WordArray;
        var C_algo = C2.algo;
        var MD5 = C_algo.MD5;
        var EvpKDF = C_algo.EvpKDF = Base.extend({
          /**
           * Configuration options.
           *
           * @property {number} keySize The key size in words to generate. Default: 4 (128 bits)
           * @property {Hasher} hasher The hash algorithm to use. Default: MD5
           * @property {number} iterations The number of iterations to perform. Default: 1
           */
          cfg: Base.extend({
            keySize: 128 / 32,
            hasher: MD5,
            iterations: 1
          }),
          /**
           * Initializes a newly created key derivation function.
           *
           * @param {Object} cfg (Optional) The configuration options to use for the derivation.
           *
           * @example
           *
           *     var kdf = CryptoJS.algo.EvpKDF.create();
           *     var kdf = CryptoJS.algo.EvpKDF.create({ keySize: 8 });
           *     var kdf = CryptoJS.algo.EvpKDF.create({ keySize: 8, iterations: 1000 });
           */
          init: function(cfg) {
            this.cfg = this.cfg.extend(cfg);
          },
          /**
           * Derives a key from a password.
           *
           * @param {WordArray|string} password The password.
           * @param {WordArray|string} salt A salt.
           *
           * @return {WordArray} The derived key.
           *
           * @example
           *
           *     var key = kdf.compute(password, salt);
           */
          compute: function(password, salt) {
            var block;
            var cfg = this.cfg;
            var hasher = cfg.hasher.create();
            var derivedKey = WordArray.create();
            var derivedKeyWords = derivedKey.words;
            var keySize = cfg.keySize;
            var iterations = cfg.iterations;
            while (derivedKeyWords.length < keySize) {
              if (block) {
                hasher.update(block);
              }
              block = hasher.update(password).finalize(salt);
              hasher.reset();
              for (var i2 = 1; i2 < iterations; i2++) {
                block = hasher.finalize(block);
                hasher.reset();
              }
              derivedKey.concat(block);
            }
            derivedKey.sigBytes = keySize * 4;
            return derivedKey;
          }
        });
        C2.EvpKDF = function(password, salt, cfg) {
          return EvpKDF.create(cfg).compute(password, salt);
        };
      })();
      return CryptoJS2.EvpKDF;
    });
  })(evpkdf);
  return evpkdf.exports;
}
var cipherCore = { exports: {} };
var hasRequiredCipherCore;
function requireCipherCore() {
  if (hasRequiredCipherCore)
    return cipherCore.exports;
  hasRequiredCipherCore = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireEvpkdf());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      CryptoJS2.lib.Cipher || function(undefined$12) {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var Base = C_lib.Base;
        var WordArray = C_lib.WordArray;
        var BufferedBlockAlgorithm = C_lib.BufferedBlockAlgorithm;
        var C_enc = C2.enc;
        C_enc.Utf8;
        var Base642 = C_enc.Base64;
        var C_algo = C2.algo;
        var EvpKDF = C_algo.EvpKDF;
        var Cipher = C_lib.Cipher = BufferedBlockAlgorithm.extend({
          /**
           * Configuration options.
           *
           * @property {WordArray} iv The IV to use for this operation.
           */
          cfg: Base.extend(),
          /**
           * Creates this cipher in encryption mode.
           *
           * @param {WordArray} key The key.
           * @param {Object} cfg (Optional) The configuration options to use for this operation.
           *
           * @return {Cipher} A cipher instance.
           *
           * @static
           *
           * @example
           *
           *     var cipher = CryptoJS.algo.AES.createEncryptor(keyWordArray, { iv: ivWordArray });
           */
          createEncryptor: function(key, cfg) {
            return this.create(this._ENC_XFORM_MODE, key, cfg);
          },
          /**
           * Creates this cipher in decryption mode.
           *
           * @param {WordArray} key The key.
           * @param {Object} cfg (Optional) The configuration options to use for this operation.
           *
           * @return {Cipher} A cipher instance.
           *
           * @static
           *
           * @example
           *
           *     var cipher = CryptoJS.algo.AES.createDecryptor(keyWordArray, { iv: ivWordArray });
           */
          createDecryptor: function(key, cfg) {
            return this.create(this._DEC_XFORM_MODE, key, cfg);
          },
          /**
           * Initializes a newly created cipher.
           *
           * @param {number} xformMode Either the encryption or decryption transormation mode constant.
           * @param {WordArray} key The key.
           * @param {Object} cfg (Optional) The configuration options to use for this operation.
           *
           * @example
           *
           *     var cipher = CryptoJS.algo.AES.create(CryptoJS.algo.AES._ENC_XFORM_MODE, keyWordArray, { iv: ivWordArray });
           */
          init: function(xformMode, key, cfg) {
            this.cfg = this.cfg.extend(cfg);
            this._xformMode = xformMode;
            this._key = key;
            this.reset();
          },
          /**
           * Resets this cipher to its initial state.
           *
           * @example
           *
           *     cipher.reset();
           */
          reset: function() {
            BufferedBlockAlgorithm.reset.call(this);
            this._doReset();
          },
          /**
           * Adds data to be encrypted or decrypted.
           *
           * @param {WordArray|string} dataUpdate The data to encrypt or decrypt.
           *
           * @return {WordArray} The data after processing.
           *
           * @example
           *
           *     var encrypted = cipher.process('data');
           *     var encrypted = cipher.process(wordArray);
           */
          process: function(dataUpdate) {
            this._append(dataUpdate);
            return this._process();
          },
          /**
           * Finalizes the encryption or decryption process.
           * Note that the finalize operation is effectively a destructive, read-once operation.
           *
           * @param {WordArray|string} dataUpdate The final data to encrypt or decrypt.
           *
           * @return {WordArray} The data after final processing.
           *
           * @example
           *
           *     var encrypted = cipher.finalize();
           *     var encrypted = cipher.finalize('data');
           *     var encrypted = cipher.finalize(wordArray);
           */
          finalize: function(dataUpdate) {
            if (dataUpdate) {
              this._append(dataUpdate);
            }
            var finalProcessedData = this._doFinalize();
            return finalProcessedData;
          },
          keySize: 128 / 32,
          ivSize: 128 / 32,
          _ENC_XFORM_MODE: 1,
          _DEC_XFORM_MODE: 2,
          /**
           * Creates shortcut functions to a cipher's object interface.
           *
           * @param {Cipher} cipher The cipher to create a helper for.
           *
           * @return {Object} An object with encrypt and decrypt shortcut functions.
           *
           * @static
           *
           * @example
           *
           *     var AES = CryptoJS.lib.Cipher._createHelper(CryptoJS.algo.AES);
           */
          _createHelper: function() {
            function selectCipherStrategy(key) {
              if (typeof key == "string") {
                return PasswordBasedCipher;
              } else {
                return SerializableCipher;
              }
            }
            return function(cipher) {
              return {
                encrypt: function(message, key, cfg) {
                  return selectCipherStrategy(key).encrypt(cipher, message, key, cfg);
                },
                decrypt: function(ciphertext, key, cfg) {
                  return selectCipherStrategy(key).decrypt(cipher, ciphertext, key, cfg);
                }
              };
            };
          }()
        });
        C_lib.StreamCipher = Cipher.extend({
          _doFinalize: function() {
            var finalProcessedBlocks = this._process(true);
            return finalProcessedBlocks;
          },
          blockSize: 1
        });
        var C_mode = C2.mode = {};
        var BlockCipherMode = C_lib.BlockCipherMode = Base.extend({
          /**
           * Creates this mode for encryption.
           *
           * @param {Cipher} cipher A block cipher instance.
           * @param {Array} iv The IV words.
           *
           * @static
           *
           * @example
           *
           *     var mode = CryptoJS.mode.CBC.createEncryptor(cipher, iv.words);
           */
          createEncryptor: function(cipher, iv) {
            return this.Encryptor.create(cipher, iv);
          },
          /**
           * Creates this mode for decryption.
           *
           * @param {Cipher} cipher A block cipher instance.
           * @param {Array} iv The IV words.
           *
           * @static
           *
           * @example
           *
           *     var mode = CryptoJS.mode.CBC.createDecryptor(cipher, iv.words);
           */
          createDecryptor: function(cipher, iv) {
            return this.Decryptor.create(cipher, iv);
          },
          /**
           * Initializes a newly created mode.
           *
           * @param {Cipher} cipher A block cipher instance.
           * @param {Array} iv The IV words.
           *
           * @example
           *
           *     var mode = CryptoJS.mode.CBC.Encryptor.create(cipher, iv.words);
           */
          init: function(cipher, iv) {
            this._cipher = cipher;
            this._iv = iv;
          }
        });
        var CBC = C_mode.CBC = function() {
          var CBC2 = BlockCipherMode.extend();
          CBC2.Encryptor = CBC2.extend({
            /**
             * Processes the data block at offset.
             *
             * @param {Array} words The data words to operate on.
             * @param {number} offset The offset where the block starts.
             *
             * @example
             *
             *     mode.processBlock(data.words, offset);
             */
            processBlock: function(words, offset2) {
              var cipher = this._cipher;
              var blockSize = cipher.blockSize;
              xorBlock.call(this, words, offset2, blockSize);
              cipher.encryptBlock(words, offset2);
              this._prevBlock = words.slice(offset2, offset2 + blockSize);
            }
          });
          CBC2.Decryptor = CBC2.extend({
            /**
             * Processes the data block at offset.
             *
             * @param {Array} words The data words to operate on.
             * @param {number} offset The offset where the block starts.
             *
             * @example
             *
             *     mode.processBlock(data.words, offset);
             */
            processBlock: function(words, offset2) {
              var cipher = this._cipher;
              var blockSize = cipher.blockSize;
              var thisBlock = words.slice(offset2, offset2 + blockSize);
              cipher.decryptBlock(words, offset2);
              xorBlock.call(this, words, offset2, blockSize);
              this._prevBlock = thisBlock;
            }
          });
          function xorBlock(words, offset2, blockSize) {
            var block;
            var iv = this._iv;
            if (iv) {
              block = iv;
              this._iv = undefined$12;
            } else {
              block = this._prevBlock;
            }
            for (var i2 = 0; i2 < blockSize; i2++) {
              words[offset2 + i2] ^= block[i2];
            }
          }
          return CBC2;
        }();
        var C_pad = C2.pad = {};
        var Pkcs7 = C_pad.Pkcs7 = {
          /**
           * Pads data using the algorithm defined in PKCS #5/7.
           *
           * @param {WordArray} data The data to pad.
           * @param {number} blockSize The multiple that the data should be padded to.
           *
           * @static
           *
           * @example
           *
           *     CryptoJS.pad.Pkcs7.pad(wordArray, 4);
           */
          pad: function(data2, blockSize) {
            var blockSizeBytes = blockSize * 4;
            var nPaddingBytes = blockSizeBytes - data2.sigBytes % blockSizeBytes;
            var paddingWord = nPaddingBytes << 24 | nPaddingBytes << 16 | nPaddingBytes << 8 | nPaddingBytes;
            var paddingWords = [];
            for (var i2 = 0; i2 < nPaddingBytes; i2 += 4) {
              paddingWords.push(paddingWord);
            }
            var padding = WordArray.create(paddingWords, nPaddingBytes);
            data2.concat(padding);
          },
          /**
           * Unpads data that had been padded using the algorithm defined in PKCS #5/7.
           *
           * @param {WordArray} data The data to unpad.
           *
           * @static
           *
           * @example
           *
           *     CryptoJS.pad.Pkcs7.unpad(wordArray);
           */
          unpad: function(data2) {
            var nPaddingBytes = data2.words[data2.sigBytes - 1 >>> 2] & 255;
            data2.sigBytes -= nPaddingBytes;
          }
        };
        C_lib.BlockCipher = Cipher.extend({
          /**
           * Configuration options.
           *
           * @property {Mode} mode The block mode to use. Default: CBC
           * @property {Padding} padding The padding strategy to use. Default: Pkcs7
           */
          cfg: Cipher.cfg.extend({
            mode: CBC,
            padding: Pkcs7
          }),
          reset: function() {
            var modeCreator;
            Cipher.reset.call(this);
            var cfg = this.cfg;
            var iv = cfg.iv;
            var mode2 = cfg.mode;
            if (this._xformMode == this._ENC_XFORM_MODE) {
              modeCreator = mode2.createEncryptor;
            } else {
              modeCreator = mode2.createDecryptor;
              this._minBufferSize = 1;
            }
            if (this._mode && this._mode.__creator == modeCreator) {
              this._mode.init(this, iv && iv.words);
            } else {
              this._mode = modeCreator.call(mode2, this, iv && iv.words);
              this._mode.__creator = modeCreator;
            }
          },
          _doProcessBlock: function(words, offset2) {
            this._mode.processBlock(words, offset2);
          },
          _doFinalize: function() {
            var finalProcessedBlocks;
            var padding = this.cfg.padding;
            if (this._xformMode == this._ENC_XFORM_MODE) {
              padding.pad(this._data, this.blockSize);
              finalProcessedBlocks = this._process(true);
            } else {
              finalProcessedBlocks = this._process(true);
              padding.unpad(finalProcessedBlocks);
            }
            return finalProcessedBlocks;
          },
          blockSize: 128 / 32
        });
        var CipherParams = C_lib.CipherParams = Base.extend({
          /**
           * Initializes a newly created cipher params object.
           *
           * @param {Object} cipherParams An object with any of the possible cipher parameters.
           *
           * @example
           *
           *     var cipherParams = CryptoJS.lib.CipherParams.create({
           *         ciphertext: ciphertextWordArray,
           *         key: keyWordArray,
           *         iv: ivWordArray,
           *         salt: saltWordArray,
           *         algorithm: CryptoJS.algo.AES,
           *         mode: CryptoJS.mode.CBC,
           *         padding: CryptoJS.pad.PKCS7,
           *         blockSize: 4,
           *         formatter: CryptoJS.format.OpenSSL
           *     });
           */
          init: function(cipherParams) {
            this.mixIn(cipherParams);
          },
          /**
           * Converts this cipher params object to a string.
           *
           * @param {Format} formatter (Optional) The formatting strategy to use.
           *
           * @return {string} The stringified cipher params.
           *
           * @throws Error If neither the formatter nor the default formatter is set.
           *
           * @example
           *
           *     var string = cipherParams + '';
           *     var string = cipherParams.toString();
           *     var string = cipherParams.toString(CryptoJS.format.OpenSSL);
           */
          toString: function(formatter) {
            return (formatter || this.formatter).stringify(this);
          }
        });
        var C_format = C2.format = {};
        var OpenSSLFormatter = C_format.OpenSSL = {
          /**
           * Converts a cipher params object to an OpenSSL-compatible string.
           *
           * @param {CipherParams} cipherParams The cipher params object.
           *
           * @return {string} The OpenSSL-compatible string.
           *
           * @static
           *
           * @example
           *
           *     var openSSLString = CryptoJS.format.OpenSSL.stringify(cipherParams);
           */
          stringify: function(cipherParams) {
            var wordArray;
            var ciphertext = cipherParams.ciphertext;
            var salt = cipherParams.salt;
            if (salt) {
              wordArray = WordArray.create([1398893684, 1701076831]).concat(salt).concat(ciphertext);
            } else {
              wordArray = ciphertext;
            }
            return wordArray.toString(Base642);
          },
          /**
           * Converts an OpenSSL-compatible string to a cipher params object.
           *
           * @param {string} openSSLStr The OpenSSL-compatible string.
           *
           * @return {CipherParams} The cipher params object.
           *
           * @static
           *
           * @example
           *
           *     var cipherParams = CryptoJS.format.OpenSSL.parse(openSSLString);
           */
          parse: function(openSSLStr) {
            var salt;
            var ciphertext = Base642.parse(openSSLStr);
            var ciphertextWords = ciphertext.words;
            if (ciphertextWords[0] == 1398893684 && ciphertextWords[1] == 1701076831) {
              salt = WordArray.create(ciphertextWords.slice(2, 4));
              ciphertextWords.splice(0, 4);
              ciphertext.sigBytes -= 16;
            }
            return CipherParams.create({ ciphertext, salt });
          }
        };
        var SerializableCipher = C_lib.SerializableCipher = Base.extend({
          /**
           * Configuration options.
           *
           * @property {Formatter} format The formatting strategy to convert cipher param objects to and from a string. Default: OpenSSL
           */
          cfg: Base.extend({
            format: OpenSSLFormatter
          }),
          /**
           * Encrypts a message.
           *
           * @param {Cipher} cipher The cipher algorithm to use.
           * @param {WordArray|string} message The message to encrypt.
           * @param {WordArray} key The key.
           * @param {Object} cfg (Optional) The configuration options to use for this operation.
           *
           * @return {CipherParams} A cipher params object.
           *
           * @static
           *
           * @example
           *
           *     var ciphertextParams = CryptoJS.lib.SerializableCipher.encrypt(CryptoJS.algo.AES, message, key);
           *     var ciphertextParams = CryptoJS.lib.SerializableCipher.encrypt(CryptoJS.algo.AES, message, key, { iv: iv });
           *     var ciphertextParams = CryptoJS.lib.SerializableCipher.encrypt(CryptoJS.algo.AES, message, key, { iv: iv, format: CryptoJS.format.OpenSSL });
           */
          encrypt: function(cipher, message, key, cfg) {
            cfg = this.cfg.extend(cfg);
            var encryptor = cipher.createEncryptor(key, cfg);
            var ciphertext = encryptor.finalize(message);
            var cipherCfg = encryptor.cfg;
            return CipherParams.create({
              ciphertext,
              key,
              iv: cipherCfg.iv,
              algorithm: cipher,
              mode: cipherCfg.mode,
              padding: cipherCfg.padding,
              blockSize: cipher.blockSize,
              formatter: cfg.format
            });
          },
          /**
           * Decrypts serialized ciphertext.
           *
           * @param {Cipher} cipher The cipher algorithm to use.
           * @param {CipherParams|string} ciphertext The ciphertext to decrypt.
           * @param {WordArray} key The key.
           * @param {Object} cfg (Optional) The configuration options to use for this operation.
           *
           * @return {WordArray} The plaintext.
           *
           * @static
           *
           * @example
           *
           *     var plaintext = CryptoJS.lib.SerializableCipher.decrypt(CryptoJS.algo.AES, formattedCiphertext, key, { iv: iv, format: CryptoJS.format.OpenSSL });
           *     var plaintext = CryptoJS.lib.SerializableCipher.decrypt(CryptoJS.algo.AES, ciphertextParams, key, { iv: iv, format: CryptoJS.format.OpenSSL });
           */
          decrypt: function(cipher, ciphertext, key, cfg) {
            cfg = this.cfg.extend(cfg);
            ciphertext = this._parse(ciphertext, cfg.format);
            var plaintext = cipher.createDecryptor(key, cfg).finalize(ciphertext.ciphertext);
            return plaintext;
          },
          /**
           * Converts serialized ciphertext to CipherParams,
           * else assumed CipherParams already and returns ciphertext unchanged.
           *
           * @param {CipherParams|string} ciphertext The ciphertext.
           * @param {Formatter} format The formatting strategy to use to parse serialized ciphertext.
           *
           * @return {CipherParams} The unserialized ciphertext.
           *
           * @static
           *
           * @example
           *
           *     var ciphertextParams = CryptoJS.lib.SerializableCipher._parse(ciphertextStringOrParams, format);
           */
          _parse: function(ciphertext, format2) {
            if (typeof ciphertext == "string") {
              return format2.parse(ciphertext, this);
            } else {
              return ciphertext;
            }
          }
        });
        var C_kdf = C2.kdf = {};
        var OpenSSLKdf = C_kdf.OpenSSL = {
          /**
           * Derives a key and IV from a password.
           *
           * @param {string} password The password to derive from.
           * @param {number} keySize The size in words of the key to generate.
           * @param {number} ivSize The size in words of the IV to generate.
           * @param {WordArray|string} salt (Optional) A 64-bit salt to use. If omitted, a salt will be generated randomly.
           *
           * @return {CipherParams} A cipher params object with the key, IV, and salt.
           *
           * @static
           *
           * @example
           *
           *     var derivedParams = CryptoJS.kdf.OpenSSL.execute('Password', 256/32, 128/32);
           *     var derivedParams = CryptoJS.kdf.OpenSSL.execute('Password', 256/32, 128/32, 'saltsalt');
           */
          execute: function(password, keySize, ivSize, salt, hasher) {
            if (!salt) {
              salt = WordArray.random(64 / 8);
            }
            if (!hasher) {
              var key = EvpKDF.create({ keySize: keySize + ivSize }).compute(password, salt);
            } else {
              var key = EvpKDF.create({ keySize: keySize + ivSize, hasher }).compute(password, salt);
            }
            var iv = WordArray.create(key.words.slice(keySize), ivSize * 4);
            key.sigBytes = keySize * 4;
            return CipherParams.create({ key, iv, salt });
          }
        };
        var PasswordBasedCipher = C_lib.PasswordBasedCipher = SerializableCipher.extend({
          /**
           * Configuration options.
           *
           * @property {KDF} kdf The key derivation function to use to generate a key and IV from a password. Default: OpenSSL
           */
          cfg: SerializableCipher.cfg.extend({
            kdf: OpenSSLKdf
          }),
          /**
           * Encrypts a message using a password.
           *
           * @param {Cipher} cipher The cipher algorithm to use.
           * @param {WordArray|string} message The message to encrypt.
           * @param {string} password The password.
           * @param {Object} cfg (Optional) The configuration options to use for this operation.
           *
           * @return {CipherParams} A cipher params object.
           *
           * @static
           *
           * @example
           *
           *     var ciphertextParams = CryptoJS.lib.PasswordBasedCipher.encrypt(CryptoJS.algo.AES, message, 'password');
           *     var ciphertextParams = CryptoJS.lib.PasswordBasedCipher.encrypt(CryptoJS.algo.AES, message, 'password', { format: CryptoJS.format.OpenSSL });
           */
          encrypt: function(cipher, message, password, cfg) {
            cfg = this.cfg.extend(cfg);
            var derivedParams = cfg.kdf.execute(password, cipher.keySize, cipher.ivSize, cfg.salt, cfg.hasher);
            cfg.iv = derivedParams.iv;
            var ciphertext = SerializableCipher.encrypt.call(this, cipher, message, derivedParams.key, cfg);
            ciphertext.mixIn(derivedParams);
            return ciphertext;
          },
          /**
           * Decrypts serialized ciphertext using a password.
           *
           * @param {Cipher} cipher The cipher algorithm to use.
           * @param {CipherParams|string} ciphertext The ciphertext to decrypt.
           * @param {string} password The password.
           * @param {Object} cfg (Optional) The configuration options to use for this operation.
           *
           * @return {WordArray} The plaintext.
           *
           * @static
           *
           * @example
           *
           *     var plaintext = CryptoJS.lib.PasswordBasedCipher.decrypt(CryptoJS.algo.AES, formattedCiphertext, 'password', { format: CryptoJS.format.OpenSSL });
           *     var plaintext = CryptoJS.lib.PasswordBasedCipher.decrypt(CryptoJS.algo.AES, ciphertextParams, 'password', { format: CryptoJS.format.OpenSSL });
           */
          decrypt: function(cipher, ciphertext, password, cfg) {
            cfg = this.cfg.extend(cfg);
            ciphertext = this._parse(ciphertext, cfg.format);
            var derivedParams = cfg.kdf.execute(password, cipher.keySize, cipher.ivSize, ciphertext.salt, cfg.hasher);
            cfg.iv = derivedParams.iv;
            var plaintext = SerializableCipher.decrypt.call(this, cipher, ciphertext, derivedParams.key, cfg);
            return plaintext;
          }
        });
      }();
    });
  })(cipherCore);
  return cipherCore.exports;
}
var modeCfb = { exports: {} };
var hasRequiredModeCfb;
function requireModeCfb() {
  if (hasRequiredModeCfb)
    return modeCfb.exports;
  hasRequiredModeCfb = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      CryptoJS2.mode.CFB = function() {
        var CFB2 = CryptoJS2.lib.BlockCipherMode.extend();
        CFB2.Encryptor = CFB2.extend({
          processBlock: function(words, offset2) {
            var cipher = this._cipher;
            var blockSize = cipher.blockSize;
            generateKeystreamAndEncrypt.call(this, words, offset2, blockSize, cipher);
            this._prevBlock = words.slice(offset2, offset2 + blockSize);
          }
        });
        CFB2.Decryptor = CFB2.extend({
          processBlock: function(words, offset2) {
            var cipher = this._cipher;
            var blockSize = cipher.blockSize;
            var thisBlock = words.slice(offset2, offset2 + blockSize);
            generateKeystreamAndEncrypt.call(this, words, offset2, blockSize, cipher);
            this._prevBlock = thisBlock;
          }
        });
        function generateKeystreamAndEncrypt(words, offset2, blockSize, cipher) {
          var keystream;
          var iv = this._iv;
          if (iv) {
            keystream = iv.slice(0);
            this._iv = void 0;
          } else {
            keystream = this._prevBlock;
          }
          cipher.encryptBlock(keystream, 0);
          for (var i2 = 0; i2 < blockSize; i2++) {
            words[offset2 + i2] ^= keystream[i2];
          }
        }
        return CFB2;
      }();
      return CryptoJS2.mode.CFB;
    });
  })(modeCfb);
  return modeCfb.exports;
}
var modeCtr = { exports: {} };
var hasRequiredModeCtr;
function requireModeCtr() {
  if (hasRequiredModeCtr)
    return modeCtr.exports;
  hasRequiredModeCtr = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      CryptoJS2.mode.CTR = function() {
        var CTR = CryptoJS2.lib.BlockCipherMode.extend();
        var Encryptor = CTR.Encryptor = CTR.extend({
          processBlock: function(words, offset2) {
            var cipher = this._cipher;
            var blockSize = cipher.blockSize;
            var iv = this._iv;
            var counter2 = this._counter;
            if (iv) {
              counter2 = this._counter = iv.slice(0);
              this._iv = void 0;
            }
            var keystream = counter2.slice(0);
            cipher.encryptBlock(keystream, 0);
            counter2[blockSize - 1] = counter2[blockSize - 1] + 1 | 0;
            for (var i2 = 0; i2 < blockSize; i2++) {
              words[offset2 + i2] ^= keystream[i2];
            }
          }
        });
        CTR.Decryptor = Encryptor;
        return CTR;
      }();
      return CryptoJS2.mode.CTR;
    });
  })(modeCtr);
  return modeCtr.exports;
}
var modeCtrGladman = { exports: {} };
var hasRequiredModeCtrGladman;
function requireModeCtrGladman() {
  if (hasRequiredModeCtrGladman)
    return modeCtrGladman.exports;
  hasRequiredModeCtrGladman = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      /** @preserve
       * Counter block mode compatible with  Dr Brian Gladman fileenc.c
       * derived from CryptoJS.mode.CTR
       * Jan Hruby jhruby.web@gmail.com
       */
      CryptoJS2.mode.CTRGladman = function() {
        var CTRGladman = CryptoJS2.lib.BlockCipherMode.extend();
        function incWord(word) {
          if ((word >> 24 & 255) === 255) {
            var b1 = word >> 16 & 255;
            var b2 = word >> 8 & 255;
            var b3 = word & 255;
            if (b1 === 255) {
              b1 = 0;
              if (b2 === 255) {
                b2 = 0;
                if (b3 === 255) {
                  b3 = 0;
                } else {
                  ++b3;
                }
              } else {
                ++b2;
              }
            } else {
              ++b1;
            }
            word = 0;
            word += b1 << 16;
            word += b2 << 8;
            word += b3;
          } else {
            word += 1 << 24;
          }
          return word;
        }
        function incCounter(counter2) {
          if ((counter2[0] = incWord(counter2[0])) === 0) {
            counter2[1] = incWord(counter2[1]);
          }
          return counter2;
        }
        var Encryptor = CTRGladman.Encryptor = CTRGladman.extend({
          processBlock: function(words, offset2) {
            var cipher = this._cipher;
            var blockSize = cipher.blockSize;
            var iv = this._iv;
            var counter2 = this._counter;
            if (iv) {
              counter2 = this._counter = iv.slice(0);
              this._iv = void 0;
            }
            incCounter(counter2);
            var keystream = counter2.slice(0);
            cipher.encryptBlock(keystream, 0);
            for (var i2 = 0; i2 < blockSize; i2++) {
              words[offset2 + i2] ^= keystream[i2];
            }
          }
        });
        CTRGladman.Decryptor = Encryptor;
        return CTRGladman;
      }();
      return CryptoJS2.mode.CTRGladman;
    });
  })(modeCtrGladman);
  return modeCtrGladman.exports;
}
var modeOfb = { exports: {} };
var hasRequiredModeOfb;
function requireModeOfb() {
  if (hasRequiredModeOfb)
    return modeOfb.exports;
  hasRequiredModeOfb = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      CryptoJS2.mode.OFB = function() {
        var OFB = CryptoJS2.lib.BlockCipherMode.extend();
        var Encryptor = OFB.Encryptor = OFB.extend({
          processBlock: function(words, offset2) {
            var cipher = this._cipher;
            var blockSize = cipher.blockSize;
            var iv = this._iv;
            var keystream = this._keystream;
            if (iv) {
              keystream = this._keystream = iv.slice(0);
              this._iv = void 0;
            }
            cipher.encryptBlock(keystream, 0);
            for (var i2 = 0; i2 < blockSize; i2++) {
              words[offset2 + i2] ^= keystream[i2];
            }
          }
        });
        OFB.Decryptor = Encryptor;
        return OFB;
      }();
      return CryptoJS2.mode.OFB;
    });
  })(modeOfb);
  return modeOfb.exports;
}
var modeEcb = { exports: {} };
var hasRequiredModeEcb;
function requireModeEcb() {
  if (hasRequiredModeEcb)
    return modeEcb.exports;
  hasRequiredModeEcb = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      CryptoJS2.mode.ECB = function() {
        var ECB = CryptoJS2.lib.BlockCipherMode.extend();
        ECB.Encryptor = ECB.extend({
          processBlock: function(words, offset2) {
            this._cipher.encryptBlock(words, offset2);
          }
        });
        ECB.Decryptor = ECB.extend({
          processBlock: function(words, offset2) {
            this._cipher.decryptBlock(words, offset2);
          }
        });
        return ECB;
      }();
      return CryptoJS2.mode.ECB;
    });
  })(modeEcb);
  return modeEcb.exports;
}
var padAnsix923 = { exports: {} };
var hasRequiredPadAnsix923;
function requirePadAnsix923() {
  if (hasRequiredPadAnsix923)
    return padAnsix923.exports;
  hasRequiredPadAnsix923 = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      CryptoJS2.pad.AnsiX923 = {
        pad: function(data2, blockSize) {
          var dataSigBytes = data2.sigBytes;
          var blockSizeBytes = blockSize * 4;
          var nPaddingBytes = blockSizeBytes - dataSigBytes % blockSizeBytes;
          var lastBytePos = dataSigBytes + nPaddingBytes - 1;
          data2.clamp();
          data2.words[lastBytePos >>> 2] |= nPaddingBytes << 24 - lastBytePos % 4 * 8;
          data2.sigBytes += nPaddingBytes;
        },
        unpad: function(data2) {
          var nPaddingBytes = data2.words[data2.sigBytes - 1 >>> 2] & 255;
          data2.sigBytes -= nPaddingBytes;
        }
      };
      return CryptoJS2.pad.Ansix923;
    });
  })(padAnsix923);
  return padAnsix923.exports;
}
var padIso10126 = { exports: {} };
var hasRequiredPadIso10126;
function requirePadIso10126() {
  if (hasRequiredPadIso10126)
    return padIso10126.exports;
  hasRequiredPadIso10126 = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      CryptoJS2.pad.Iso10126 = {
        pad: function(data2, blockSize) {
          var blockSizeBytes = blockSize * 4;
          var nPaddingBytes = blockSizeBytes - data2.sigBytes % blockSizeBytes;
          data2.concat(CryptoJS2.lib.WordArray.random(nPaddingBytes - 1)).concat(CryptoJS2.lib.WordArray.create([nPaddingBytes << 24], 1));
        },
        unpad: function(data2) {
          var nPaddingBytes = data2.words[data2.sigBytes - 1 >>> 2] & 255;
          data2.sigBytes -= nPaddingBytes;
        }
      };
      return CryptoJS2.pad.Iso10126;
    });
  })(padIso10126);
  return padIso10126.exports;
}
var padIso97971 = { exports: {} };
var hasRequiredPadIso97971;
function requirePadIso97971() {
  if (hasRequiredPadIso97971)
    return padIso97971.exports;
  hasRequiredPadIso97971 = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      CryptoJS2.pad.Iso97971 = {
        pad: function(data2, blockSize) {
          data2.concat(CryptoJS2.lib.WordArray.create([2147483648], 1));
          CryptoJS2.pad.ZeroPadding.pad(data2, blockSize);
        },
        unpad: function(data2) {
          CryptoJS2.pad.ZeroPadding.unpad(data2);
          data2.sigBytes--;
        }
      };
      return CryptoJS2.pad.Iso97971;
    });
  })(padIso97971);
  return padIso97971.exports;
}
var padZeropadding = { exports: {} };
var hasRequiredPadZeropadding;
function requirePadZeropadding() {
  if (hasRequiredPadZeropadding)
    return padZeropadding.exports;
  hasRequiredPadZeropadding = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      CryptoJS2.pad.ZeroPadding = {
        pad: function(data2, blockSize) {
          var blockSizeBytes = blockSize * 4;
          data2.clamp();
          data2.sigBytes += blockSizeBytes - (data2.sigBytes % blockSizeBytes || blockSizeBytes);
        },
        unpad: function(data2) {
          var dataWords = data2.words;
          var i2 = data2.sigBytes - 1;
          for (var i2 = data2.sigBytes - 1; i2 >= 0; i2--) {
            if (dataWords[i2 >>> 2] >>> 24 - i2 % 4 * 8 & 255) {
              data2.sigBytes = i2 + 1;
              break;
            }
          }
        }
      };
      return CryptoJS2.pad.ZeroPadding;
    });
  })(padZeropadding);
  return padZeropadding.exports;
}
var padNopadding = { exports: {} };
var hasRequiredPadNopadding;
function requirePadNopadding() {
  if (hasRequiredPadNopadding)
    return padNopadding.exports;
  hasRequiredPadNopadding = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      CryptoJS2.pad.NoPadding = {
        pad: function() {
        },
        unpad: function() {
        }
      };
      return CryptoJS2.pad.NoPadding;
    });
  })(padNopadding);
  return padNopadding.exports;
}
var formatHex = { exports: {} };
var hasRequiredFormatHex;
function requireFormatHex() {
  if (hasRequiredFormatHex)
    return formatHex.exports;
  hasRequiredFormatHex = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function(undefined$12) {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var CipherParams = C_lib.CipherParams;
        var C_enc = C2.enc;
        var Hex2 = C_enc.Hex;
        var C_format = C2.format;
        C_format.Hex = {
          /**
           * Converts the ciphertext of a cipher params object to a hexadecimally encoded string.
           *
           * @param {CipherParams} cipherParams The cipher params object.
           *
           * @return {string} The hexadecimally encoded string.
           *
           * @static
           *
           * @example
           *
           *     var hexString = CryptoJS.format.Hex.stringify(cipherParams);
           */
          stringify: function(cipherParams) {
            return cipherParams.ciphertext.toString(Hex2);
          },
          /**
           * Converts a hexadecimally encoded ciphertext string to a cipher params object.
           *
           * @param {string} input The hexadecimally encoded string.
           *
           * @return {CipherParams} The cipher params object.
           *
           * @static
           *
           * @example
           *
           *     var cipherParams = CryptoJS.format.Hex.parse(hexString);
           */
          parse: function(input) {
            var ciphertext = Hex2.parse(input);
            return CipherParams.create({ ciphertext });
          }
        };
      })();
      return CryptoJS2.format.Hex;
    });
  })(formatHex);
  return formatHex.exports;
}
var aes = { exports: {} };
var hasRequiredAes;
function requireAes() {
  if (hasRequiredAes)
    return aes.exports;
  hasRequiredAes = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireEncBase64(), requireMd5(), requireEvpkdf(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var BlockCipher = C_lib.BlockCipher;
        var C_algo = C2.algo;
        var SBOX = [];
        var INV_SBOX = [];
        var SUB_MIX_0 = [];
        var SUB_MIX_1 = [];
        var SUB_MIX_2 = [];
        var SUB_MIX_3 = [];
        var INV_SUB_MIX_0 = [];
        var INV_SUB_MIX_1 = [];
        var INV_SUB_MIX_2 = [];
        var INV_SUB_MIX_3 = [];
        (function() {
          var d2 = [];
          for (var i2 = 0; i2 < 256; i2++) {
            if (i2 < 128) {
              d2[i2] = i2 << 1;
            } else {
              d2[i2] = i2 << 1 ^ 283;
            }
          }
          var x2 = 0;
          var xi = 0;
          for (var i2 = 0; i2 < 256; i2++) {
            var sx = xi ^ xi << 1 ^ xi << 2 ^ xi << 3 ^ xi << 4;
            sx = sx >>> 8 ^ sx & 255 ^ 99;
            SBOX[x2] = sx;
            INV_SBOX[sx] = x2;
            var x22 = d2[x2];
            var x4 = d2[x22];
            var x8 = d2[x4];
            var t3 = d2[sx] * 257 ^ sx * 16843008;
            SUB_MIX_0[x2] = t3 << 24 | t3 >>> 8;
            SUB_MIX_1[x2] = t3 << 16 | t3 >>> 16;
            SUB_MIX_2[x2] = t3 << 8 | t3 >>> 24;
            SUB_MIX_3[x2] = t3;
            var t3 = x8 * 16843009 ^ x4 * 65537 ^ x22 * 257 ^ x2 * 16843008;
            INV_SUB_MIX_0[sx] = t3 << 24 | t3 >>> 8;
            INV_SUB_MIX_1[sx] = t3 << 16 | t3 >>> 16;
            INV_SUB_MIX_2[sx] = t3 << 8 | t3 >>> 24;
            INV_SUB_MIX_3[sx] = t3;
            if (!x2) {
              x2 = xi = 1;
            } else {
              x2 = x22 ^ d2[d2[d2[x8 ^ x22]]];
              xi ^= d2[d2[xi]];
            }
          }
        })();
        var RCON = [0, 1, 2, 4, 8, 16, 32, 64, 128, 27, 54];
        var AES = C_algo.AES = BlockCipher.extend({
          _doReset: function() {
            var t3;
            if (this._nRounds && this._keyPriorReset === this._key) {
              return;
            }
            var key = this._keyPriorReset = this._key;
            var keyWords = key.words;
            var keySize = key.sigBytes / 4;
            var nRounds = this._nRounds = keySize + 6;
            var ksRows = (nRounds + 1) * 4;
            var keySchedule = this._keySchedule = [];
            for (var ksRow = 0; ksRow < ksRows; ksRow++) {
              if (ksRow < keySize) {
                keySchedule[ksRow] = keyWords[ksRow];
              } else {
                t3 = keySchedule[ksRow - 1];
                if (!(ksRow % keySize)) {
                  t3 = t3 << 8 | t3 >>> 24;
                  t3 = SBOX[t3 >>> 24] << 24 | SBOX[t3 >>> 16 & 255] << 16 | SBOX[t3 >>> 8 & 255] << 8 | SBOX[t3 & 255];
                  t3 ^= RCON[ksRow / keySize | 0] << 24;
                } else if (keySize > 6 && ksRow % keySize == 4) {
                  t3 = SBOX[t3 >>> 24] << 24 | SBOX[t3 >>> 16 & 255] << 16 | SBOX[t3 >>> 8 & 255] << 8 | SBOX[t3 & 255];
                }
                keySchedule[ksRow] = keySchedule[ksRow - keySize] ^ t3;
              }
            }
            var invKeySchedule = this._invKeySchedule = [];
            for (var invKsRow = 0; invKsRow < ksRows; invKsRow++) {
              var ksRow = ksRows - invKsRow;
              if (invKsRow % 4) {
                var t3 = keySchedule[ksRow];
              } else {
                var t3 = keySchedule[ksRow - 4];
              }
              if (invKsRow < 4 || ksRow <= 4) {
                invKeySchedule[invKsRow] = t3;
              } else {
                invKeySchedule[invKsRow] = INV_SUB_MIX_0[SBOX[t3 >>> 24]] ^ INV_SUB_MIX_1[SBOX[t3 >>> 16 & 255]] ^ INV_SUB_MIX_2[SBOX[t3 >>> 8 & 255]] ^ INV_SUB_MIX_3[SBOX[t3 & 255]];
              }
            }
          },
          encryptBlock: function(M2, offset2) {
            this._doCryptBlock(M2, offset2, this._keySchedule, SUB_MIX_0, SUB_MIX_1, SUB_MIX_2, SUB_MIX_3, SBOX);
          },
          decryptBlock: function(M2, offset2) {
            var t3 = M2[offset2 + 1];
            M2[offset2 + 1] = M2[offset2 + 3];
            M2[offset2 + 3] = t3;
            this._doCryptBlock(M2, offset2, this._invKeySchedule, INV_SUB_MIX_0, INV_SUB_MIX_1, INV_SUB_MIX_2, INV_SUB_MIX_3, INV_SBOX);
            var t3 = M2[offset2 + 1];
            M2[offset2 + 1] = M2[offset2 + 3];
            M2[offset2 + 3] = t3;
          },
          _doCryptBlock: function(M2, offset2, keySchedule, SUB_MIX_02, SUB_MIX_12, SUB_MIX_22, SUB_MIX_32, SBOX2) {
            var nRounds = this._nRounds;
            var s0 = M2[offset2] ^ keySchedule[0];
            var s1 = M2[offset2 + 1] ^ keySchedule[1];
            var s2 = M2[offset2 + 2] ^ keySchedule[2];
            var s3 = M2[offset2 + 3] ^ keySchedule[3];
            var ksRow = 4;
            for (var round2 = 1; round2 < nRounds; round2++) {
              var t0 = SUB_MIX_02[s0 >>> 24] ^ SUB_MIX_12[s1 >>> 16 & 255] ^ SUB_MIX_22[s2 >>> 8 & 255] ^ SUB_MIX_32[s3 & 255] ^ keySchedule[ksRow++];
              var t1 = SUB_MIX_02[s1 >>> 24] ^ SUB_MIX_12[s2 >>> 16 & 255] ^ SUB_MIX_22[s3 >>> 8 & 255] ^ SUB_MIX_32[s0 & 255] ^ keySchedule[ksRow++];
              var t22 = SUB_MIX_02[s2 >>> 24] ^ SUB_MIX_12[s3 >>> 16 & 255] ^ SUB_MIX_22[s0 >>> 8 & 255] ^ SUB_MIX_32[s1 & 255] ^ keySchedule[ksRow++];
              var t3 = SUB_MIX_02[s3 >>> 24] ^ SUB_MIX_12[s0 >>> 16 & 255] ^ SUB_MIX_22[s1 >>> 8 & 255] ^ SUB_MIX_32[s2 & 255] ^ keySchedule[ksRow++];
              s0 = t0;
              s1 = t1;
              s2 = t22;
              s3 = t3;
            }
            var t0 = (SBOX2[s0 >>> 24] << 24 | SBOX2[s1 >>> 16 & 255] << 16 | SBOX2[s2 >>> 8 & 255] << 8 | SBOX2[s3 & 255]) ^ keySchedule[ksRow++];
            var t1 = (SBOX2[s1 >>> 24] << 24 | SBOX2[s2 >>> 16 & 255] << 16 | SBOX2[s3 >>> 8 & 255] << 8 | SBOX2[s0 & 255]) ^ keySchedule[ksRow++];
            var t22 = (SBOX2[s2 >>> 24] << 24 | SBOX2[s3 >>> 16 & 255] << 16 | SBOX2[s0 >>> 8 & 255] << 8 | SBOX2[s1 & 255]) ^ keySchedule[ksRow++];
            var t3 = (SBOX2[s3 >>> 24] << 24 | SBOX2[s0 >>> 16 & 255] << 16 | SBOX2[s1 >>> 8 & 255] << 8 | SBOX2[s2 & 255]) ^ keySchedule[ksRow++];
            M2[offset2] = t0;
            M2[offset2 + 1] = t1;
            M2[offset2 + 2] = t22;
            M2[offset2 + 3] = t3;
          },
          keySize: 256 / 32
        });
        C2.AES = BlockCipher._createHelper(AES);
      })();
      return CryptoJS2.AES;
    });
  })(aes);
  return aes.exports;
}
var tripledes = { exports: {} };
var hasRequiredTripledes;
function requireTripledes() {
  if (hasRequiredTripledes)
    return tripledes.exports;
  hasRequiredTripledes = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireEncBase64(), requireMd5(), requireEvpkdf(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var WordArray = C_lib.WordArray;
        var BlockCipher = C_lib.BlockCipher;
        var C_algo = C2.algo;
        var PC1 = [
          57,
          49,
          41,
          33,
          25,
          17,
          9,
          1,
          58,
          50,
          42,
          34,
          26,
          18,
          10,
          2,
          59,
          51,
          43,
          35,
          27,
          19,
          11,
          3,
          60,
          52,
          44,
          36,
          63,
          55,
          47,
          39,
          31,
          23,
          15,
          7,
          62,
          54,
          46,
          38,
          30,
          22,
          14,
          6,
          61,
          53,
          45,
          37,
          29,
          21,
          13,
          5,
          28,
          20,
          12,
          4
        ];
        var PC2 = [
          14,
          17,
          11,
          24,
          1,
          5,
          3,
          28,
          15,
          6,
          21,
          10,
          23,
          19,
          12,
          4,
          26,
          8,
          16,
          7,
          27,
          20,
          13,
          2,
          41,
          52,
          31,
          37,
          47,
          55,
          30,
          40,
          51,
          45,
          33,
          48,
          44,
          49,
          39,
          56,
          34,
          53,
          46,
          42,
          50,
          36,
          29,
          32
        ];
        var BIT_SHIFTS = [1, 2, 4, 6, 8, 10, 12, 14, 15, 17, 19, 21, 23, 25, 27, 28];
        var SBOX_P = [
          {
            0: 8421888,
            268435456: 32768,
            536870912: 8421378,
            805306368: 2,
            1073741824: 512,
            1342177280: 8421890,
            1610612736: 8389122,
            1879048192: 8388608,
            2147483648: 514,
            2415919104: 8389120,
            2684354560: 33280,
            2952790016: 8421376,
            3221225472: 32770,
            3489660928: 8388610,
            3758096384: 0,
            4026531840: 33282,
            134217728: 0,
            402653184: 8421890,
            671088640: 33282,
            939524096: 32768,
            1207959552: 8421888,
            1476395008: 512,
            1744830464: 8421378,
            2013265920: 2,
            2281701376: 8389120,
            2550136832: 33280,
            2818572288: 8421376,
            3087007744: 8389122,
            3355443200: 8388610,
            3623878656: 32770,
            3892314112: 514,
            4160749568: 8388608,
            1: 32768,
            268435457: 2,
            536870913: 8421888,
            805306369: 8388608,
            1073741825: 8421378,
            1342177281: 33280,
            1610612737: 512,
            1879048193: 8389122,
            2147483649: 8421890,
            2415919105: 8421376,
            2684354561: 8388610,
            2952790017: 33282,
            3221225473: 514,
            3489660929: 8389120,
            3758096385: 32770,
            4026531841: 0,
            134217729: 8421890,
            402653185: 8421376,
            671088641: 8388608,
            939524097: 512,
            1207959553: 32768,
            1476395009: 8388610,
            1744830465: 2,
            2013265921: 33282,
            2281701377: 32770,
            2550136833: 8389122,
            2818572289: 514,
            3087007745: 8421888,
            3355443201: 8389120,
            3623878657: 0,
            3892314113: 33280,
            4160749569: 8421378
          },
          {
            0: 1074282512,
            16777216: 16384,
            33554432: 524288,
            50331648: 1074266128,
            67108864: 1073741840,
            83886080: 1074282496,
            100663296: 1073758208,
            117440512: 16,
            134217728: 540672,
            150994944: 1073758224,
            167772160: 1073741824,
            184549376: 540688,
            201326592: 524304,
            218103808: 0,
            234881024: 16400,
            251658240: 1074266112,
            8388608: 1073758208,
            25165824: 540688,
            41943040: 16,
            58720256: 1073758224,
            75497472: 1074282512,
            92274688: 1073741824,
            109051904: 524288,
            125829120: 1074266128,
            142606336: 524304,
            159383552: 0,
            176160768: 16384,
            192937984: 1074266112,
            209715200: 1073741840,
            226492416: 540672,
            243269632: 1074282496,
            260046848: 16400,
            268435456: 0,
            285212672: 1074266128,
            301989888: 1073758224,
            318767104: 1074282496,
            335544320: 1074266112,
            352321536: 16,
            369098752: 540688,
            385875968: 16384,
            402653184: 16400,
            419430400: 524288,
            436207616: 524304,
            452984832: 1073741840,
            469762048: 540672,
            486539264: 1073758208,
            503316480: 1073741824,
            520093696: 1074282512,
            276824064: 540688,
            293601280: 524288,
            310378496: 1074266112,
            327155712: 16384,
            343932928: 1073758208,
            360710144: 1074282512,
            377487360: 16,
            394264576: 1073741824,
            411041792: 1074282496,
            427819008: 1073741840,
            444596224: 1073758224,
            461373440: 524304,
            478150656: 0,
            494927872: 16400,
            511705088: 1074266128,
            528482304: 540672
          },
          {
            0: 260,
            1048576: 0,
            2097152: 67109120,
            3145728: 65796,
            4194304: 65540,
            5242880: 67108868,
            6291456: 67174660,
            7340032: 67174400,
            8388608: 67108864,
            9437184: 67174656,
            10485760: 65792,
            11534336: 67174404,
            12582912: 67109124,
            13631488: 65536,
            14680064: 4,
            15728640: 256,
            524288: 67174656,
            1572864: 67174404,
            2621440: 0,
            3670016: 67109120,
            4718592: 67108868,
            5767168: 65536,
            6815744: 65540,
            7864320: 260,
            8912896: 4,
            9961472: 256,
            11010048: 67174400,
            12058624: 65796,
            13107200: 65792,
            14155776: 67109124,
            15204352: 67174660,
            16252928: 67108864,
            16777216: 67174656,
            17825792: 65540,
            18874368: 65536,
            19922944: 67109120,
            20971520: 256,
            22020096: 67174660,
            23068672: 67108868,
            24117248: 0,
            25165824: 67109124,
            26214400: 67108864,
            27262976: 4,
            28311552: 65792,
            29360128: 67174400,
            30408704: 260,
            31457280: 65796,
            32505856: 67174404,
            17301504: 67108864,
            18350080: 260,
            19398656: 67174656,
            20447232: 0,
            21495808: 65540,
            22544384: 67109120,
            23592960: 256,
            24641536: 67174404,
            25690112: 65536,
            26738688: 67174660,
            27787264: 65796,
            28835840: 67108868,
            29884416: 67109124,
            30932992: 67174400,
            31981568: 4,
            33030144: 65792
          },
          {
            0: 2151682048,
            65536: 2147487808,
            131072: 4198464,
            196608: 2151677952,
            262144: 0,
            327680: 4198400,
            393216: 2147483712,
            458752: 4194368,
            524288: 2147483648,
            589824: 4194304,
            655360: 64,
            720896: 2147487744,
            786432: 2151678016,
            851968: 4160,
            917504: 4096,
            983040: 2151682112,
            32768: 2147487808,
            98304: 64,
            163840: 2151678016,
            229376: 2147487744,
            294912: 4198400,
            360448: 2151682112,
            425984: 0,
            491520: 2151677952,
            557056: 4096,
            622592: 2151682048,
            688128: 4194304,
            753664: 4160,
            819200: 2147483648,
            884736: 4194368,
            950272: 4198464,
            1015808: 2147483712,
            1048576: 4194368,
            1114112: 4198400,
            1179648: 2147483712,
            1245184: 0,
            1310720: 4160,
            1376256: 2151678016,
            1441792: 2151682048,
            1507328: 2147487808,
            1572864: 2151682112,
            1638400: 2147483648,
            1703936: 2151677952,
            1769472: 4198464,
            1835008: 2147487744,
            1900544: 4194304,
            1966080: 64,
            2031616: 4096,
            1081344: 2151677952,
            1146880: 2151682112,
            1212416: 0,
            1277952: 4198400,
            1343488: 4194368,
            1409024: 2147483648,
            1474560: 2147487808,
            1540096: 64,
            1605632: 2147483712,
            1671168: 4096,
            1736704: 2147487744,
            1802240: 2151678016,
            1867776: 4160,
            1933312: 2151682048,
            1998848: 4194304,
            2064384: 4198464
          },
          {
            0: 128,
            4096: 17039360,
            8192: 262144,
            12288: 536870912,
            16384: 537133184,
            20480: 16777344,
            24576: 553648256,
            28672: 262272,
            32768: 16777216,
            36864: 537133056,
            40960: 536871040,
            45056: 553910400,
            49152: 553910272,
            53248: 0,
            57344: 17039488,
            61440: 553648128,
            2048: 17039488,
            6144: 553648256,
            10240: 128,
            14336: 17039360,
            18432: 262144,
            22528: 537133184,
            26624: 553910272,
            30720: 536870912,
            34816: 537133056,
            38912: 0,
            43008: 553910400,
            47104: 16777344,
            51200: 536871040,
            55296: 553648128,
            59392: 16777216,
            63488: 262272,
            65536: 262144,
            69632: 128,
            73728: 536870912,
            77824: 553648256,
            81920: 16777344,
            86016: 553910272,
            90112: 537133184,
            94208: 16777216,
            98304: 553910400,
            102400: 553648128,
            106496: 17039360,
            110592: 537133056,
            114688: 262272,
            118784: 536871040,
            122880: 0,
            126976: 17039488,
            67584: 553648256,
            71680: 16777216,
            75776: 17039360,
            79872: 537133184,
            83968: 536870912,
            88064: 17039488,
            92160: 128,
            96256: 553910272,
            100352: 262272,
            104448: 553910400,
            108544: 0,
            112640: 553648128,
            116736: 16777344,
            120832: 262144,
            124928: 537133056,
            129024: 536871040
          },
          {
            0: 268435464,
            256: 8192,
            512: 270532608,
            768: 270540808,
            1024: 268443648,
            1280: 2097152,
            1536: 2097160,
            1792: 268435456,
            2048: 0,
            2304: 268443656,
            2560: 2105344,
            2816: 8,
            3072: 270532616,
            3328: 2105352,
            3584: 8200,
            3840: 270540800,
            128: 270532608,
            384: 270540808,
            640: 8,
            896: 2097152,
            1152: 2105352,
            1408: 268435464,
            1664: 268443648,
            1920: 8200,
            2176: 2097160,
            2432: 8192,
            2688: 268443656,
            2944: 270532616,
            3200: 0,
            3456: 270540800,
            3712: 2105344,
            3968: 268435456,
            4096: 268443648,
            4352: 270532616,
            4608: 270540808,
            4864: 8200,
            5120: 2097152,
            5376: 268435456,
            5632: 268435464,
            5888: 2105344,
            6144: 2105352,
            6400: 0,
            6656: 8,
            6912: 270532608,
            7168: 8192,
            7424: 268443656,
            7680: 270540800,
            7936: 2097160,
            4224: 8,
            4480: 2105344,
            4736: 2097152,
            4992: 268435464,
            5248: 268443648,
            5504: 8200,
            5760: 270540808,
            6016: 270532608,
            6272: 270540800,
            6528: 270532616,
            6784: 8192,
            7040: 2105352,
            7296: 2097160,
            7552: 0,
            7808: 268435456,
            8064: 268443656
          },
          {
            0: 1048576,
            16: 33555457,
            32: 1024,
            48: 1049601,
            64: 34604033,
            80: 0,
            96: 1,
            112: 34603009,
            128: 33555456,
            144: 1048577,
            160: 33554433,
            176: 34604032,
            192: 34603008,
            208: 1025,
            224: 1049600,
            240: 33554432,
            8: 34603009,
            24: 0,
            40: 33555457,
            56: 34604032,
            72: 1048576,
            88: 33554433,
            104: 33554432,
            120: 1025,
            136: 1049601,
            152: 33555456,
            168: 34603008,
            184: 1048577,
            200: 1024,
            216: 34604033,
            232: 1,
            248: 1049600,
            256: 33554432,
            272: 1048576,
            288: 33555457,
            304: 34603009,
            320: 1048577,
            336: 33555456,
            352: 34604032,
            368: 1049601,
            384: 1025,
            400: 34604033,
            416: 1049600,
            432: 1,
            448: 0,
            464: 34603008,
            480: 33554433,
            496: 1024,
            264: 1049600,
            280: 33555457,
            296: 34603009,
            312: 1,
            328: 33554432,
            344: 1048576,
            360: 1025,
            376: 34604032,
            392: 33554433,
            408: 34603008,
            424: 0,
            440: 34604033,
            456: 1049601,
            472: 1024,
            488: 33555456,
            504: 1048577
          },
          {
            0: 134219808,
            1: 131072,
            2: 134217728,
            3: 32,
            4: 131104,
            5: 134350880,
            6: 134350848,
            7: 2048,
            8: 134348800,
            9: 134219776,
            10: 133120,
            11: 134348832,
            12: 2080,
            13: 0,
            14: 134217760,
            15: 133152,
            2147483648: 2048,
            2147483649: 134350880,
            2147483650: 134219808,
            2147483651: 134217728,
            2147483652: 134348800,
            2147483653: 133120,
            2147483654: 133152,
            2147483655: 32,
            2147483656: 134217760,
            2147483657: 2080,
            2147483658: 131104,
            2147483659: 134350848,
            2147483660: 0,
            2147483661: 134348832,
            2147483662: 134219776,
            2147483663: 131072,
            16: 133152,
            17: 134350848,
            18: 32,
            19: 2048,
            20: 134219776,
            21: 134217760,
            22: 134348832,
            23: 131072,
            24: 0,
            25: 131104,
            26: 134348800,
            27: 134219808,
            28: 134350880,
            29: 133120,
            30: 2080,
            31: 134217728,
            2147483664: 131072,
            2147483665: 2048,
            2147483666: 134348832,
            2147483667: 133152,
            2147483668: 32,
            2147483669: 134348800,
            2147483670: 134217728,
            2147483671: 134219808,
            2147483672: 134350880,
            2147483673: 134217760,
            2147483674: 134219776,
            2147483675: 0,
            2147483676: 133120,
            2147483677: 2080,
            2147483678: 131104,
            2147483679: 134350848
          }
        ];
        var SBOX_MASK = [
          4160749569,
          528482304,
          33030144,
          2064384,
          129024,
          8064,
          504,
          2147483679
        ];
        var DES = C_algo.DES = BlockCipher.extend({
          _doReset: function() {
            var key = this._key;
            var keyWords = key.words;
            var keyBits = [];
            for (var i2 = 0; i2 < 56; i2++) {
              var keyBitPos = PC1[i2] - 1;
              keyBits[i2] = keyWords[keyBitPos >>> 5] >>> 31 - keyBitPos % 32 & 1;
            }
            var subKeys = this._subKeys = [];
            for (var nSubKey = 0; nSubKey < 16; nSubKey++) {
              var subKey = subKeys[nSubKey] = [];
              var bitShift = BIT_SHIFTS[nSubKey];
              for (var i2 = 0; i2 < 24; i2++) {
                subKey[i2 / 6 | 0] |= keyBits[(PC2[i2] - 1 + bitShift) % 28] << 31 - i2 % 6;
                subKey[4 + (i2 / 6 | 0)] |= keyBits[28 + (PC2[i2 + 24] - 1 + bitShift) % 28] << 31 - i2 % 6;
              }
              subKey[0] = subKey[0] << 1 | subKey[0] >>> 31;
              for (var i2 = 1; i2 < 7; i2++) {
                subKey[i2] = subKey[i2] >>> (i2 - 1) * 4 + 3;
              }
              subKey[7] = subKey[7] << 5 | subKey[7] >>> 27;
            }
            var invSubKeys = this._invSubKeys = [];
            for (var i2 = 0; i2 < 16; i2++) {
              invSubKeys[i2] = subKeys[15 - i2];
            }
          },
          encryptBlock: function(M2, offset2) {
            this._doCryptBlock(M2, offset2, this._subKeys);
          },
          decryptBlock: function(M2, offset2) {
            this._doCryptBlock(M2, offset2, this._invSubKeys);
          },
          _doCryptBlock: function(M2, offset2, subKeys) {
            this._lBlock = M2[offset2];
            this._rBlock = M2[offset2 + 1];
            exchangeLR.call(this, 4, 252645135);
            exchangeLR.call(this, 16, 65535);
            exchangeRL.call(this, 2, 858993459);
            exchangeRL.call(this, 8, 16711935);
            exchangeLR.call(this, 1, 1431655765);
            for (var round2 = 0; round2 < 16; round2++) {
              var subKey = subKeys[round2];
              var lBlock = this._lBlock;
              var rBlock = this._rBlock;
              var f2 = 0;
              for (var i2 = 0; i2 < 8; i2++) {
                f2 |= SBOX_P[i2][((rBlock ^ subKey[i2]) & SBOX_MASK[i2]) >>> 0];
              }
              this._lBlock = rBlock;
              this._rBlock = lBlock ^ f2;
            }
            var t3 = this._lBlock;
            this._lBlock = this._rBlock;
            this._rBlock = t3;
            exchangeLR.call(this, 1, 1431655765);
            exchangeRL.call(this, 8, 16711935);
            exchangeRL.call(this, 2, 858993459);
            exchangeLR.call(this, 16, 65535);
            exchangeLR.call(this, 4, 252645135);
            M2[offset2] = this._lBlock;
            M2[offset2 + 1] = this._rBlock;
          },
          keySize: 64 / 32,
          ivSize: 64 / 32,
          blockSize: 64 / 32
        });
        function exchangeLR(offset2, mask) {
          var t3 = (this._lBlock >>> offset2 ^ this._rBlock) & mask;
          this._rBlock ^= t3;
          this._lBlock ^= t3 << offset2;
        }
        function exchangeRL(offset2, mask) {
          var t3 = (this._rBlock >>> offset2 ^ this._lBlock) & mask;
          this._lBlock ^= t3;
          this._rBlock ^= t3 << offset2;
        }
        C2.DES = BlockCipher._createHelper(DES);
        var TripleDES = C_algo.TripleDES = BlockCipher.extend({
          _doReset: function() {
            var key = this._key;
            var keyWords = key.words;
            if (keyWords.length !== 2 && keyWords.length !== 4 && keyWords.length < 6) {
              throw new Error("Invalid key length - 3DES requires the key length to be 64, 128, 192 or >192.");
            }
            var key1 = keyWords.slice(0, 2);
            var key2 = keyWords.length < 4 ? keyWords.slice(0, 2) : keyWords.slice(2, 4);
            var key3 = keyWords.length < 6 ? keyWords.slice(0, 2) : keyWords.slice(4, 6);
            this._des1 = DES.createEncryptor(WordArray.create(key1));
            this._des2 = DES.createEncryptor(WordArray.create(key2));
            this._des3 = DES.createEncryptor(WordArray.create(key3));
          },
          encryptBlock: function(M2, offset2) {
            this._des1.encryptBlock(M2, offset2);
            this._des2.decryptBlock(M2, offset2);
            this._des3.encryptBlock(M2, offset2);
          },
          decryptBlock: function(M2, offset2) {
            this._des3.decryptBlock(M2, offset2);
            this._des2.encryptBlock(M2, offset2);
            this._des1.decryptBlock(M2, offset2);
          },
          keySize: 192 / 32,
          ivSize: 64 / 32,
          blockSize: 64 / 32
        });
        C2.TripleDES = BlockCipher._createHelper(TripleDES);
      })();
      return CryptoJS2.TripleDES;
    });
  })(tripledes);
  return tripledes.exports;
}
var rc4 = { exports: {} };
var hasRequiredRc4;
function requireRc4() {
  if (hasRequiredRc4)
    return rc4.exports;
  hasRequiredRc4 = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireEncBase64(), requireMd5(), requireEvpkdf(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var StreamCipher = C_lib.StreamCipher;
        var C_algo = C2.algo;
        var RC4 = C_algo.RC4 = StreamCipher.extend({
          _doReset: function() {
            var key = this._key;
            var keyWords = key.words;
            var keySigBytes = key.sigBytes;
            var S2 = this._S = [];
            for (var i2 = 0; i2 < 256; i2++) {
              S2[i2] = i2;
            }
            for (var i2 = 0, j2 = 0; i2 < 256; i2++) {
              var keyByteIndex = i2 % keySigBytes;
              var keyByte = keyWords[keyByteIndex >>> 2] >>> 24 - keyByteIndex % 4 * 8 & 255;
              j2 = (j2 + S2[i2] + keyByte) % 256;
              var t3 = S2[i2];
              S2[i2] = S2[j2];
              S2[j2] = t3;
            }
            this._i = this._j = 0;
          },
          _doProcessBlock: function(M2, offset2) {
            M2[offset2] ^= generateKeystreamWord.call(this);
          },
          keySize: 256 / 32,
          ivSize: 0
        });
        function generateKeystreamWord() {
          var S2 = this._S;
          var i2 = this._i;
          var j2 = this._j;
          var keystreamWord = 0;
          for (var n2 = 0; n2 < 4; n2++) {
            i2 = (i2 + 1) % 256;
            j2 = (j2 + S2[i2]) % 256;
            var t3 = S2[i2];
            S2[i2] = S2[j2];
            S2[j2] = t3;
            keystreamWord |= S2[(S2[i2] + S2[j2]) % 256] << 24 - n2 * 8;
          }
          this._i = i2;
          this._j = j2;
          return keystreamWord;
        }
        C2.RC4 = StreamCipher._createHelper(RC4);
        var RC4Drop = C_algo.RC4Drop = RC4.extend({
          /**
           * Configuration options.
           *
           * @property {number} drop The number of keystream words to drop. Default 192
           */
          cfg: RC4.cfg.extend({
            drop: 192
          }),
          _doReset: function() {
            RC4._doReset.call(this);
            for (var i2 = this.cfg.drop; i2 > 0; i2--) {
              generateKeystreamWord.call(this);
            }
          }
        });
        C2.RC4Drop = StreamCipher._createHelper(RC4Drop);
      })();
      return CryptoJS2.RC4;
    });
  })(rc4);
  return rc4.exports;
}
var rabbit = { exports: {} };
var hasRequiredRabbit;
function requireRabbit() {
  if (hasRequiredRabbit)
    return rabbit.exports;
  hasRequiredRabbit = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireEncBase64(), requireMd5(), requireEvpkdf(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var StreamCipher = C_lib.StreamCipher;
        var C_algo = C2.algo;
        var S2 = [];
        var C_ = [];
        var G2 = [];
        var Rabbit = C_algo.Rabbit = StreamCipher.extend({
          _doReset: function() {
            var K2 = this._key.words;
            var iv = this.cfg.iv;
            for (var i2 = 0; i2 < 4; i2++) {
              K2[i2] = (K2[i2] << 8 | K2[i2] >>> 24) & 16711935 | (K2[i2] << 24 | K2[i2] >>> 8) & 4278255360;
            }
            var X2 = this._X = [
              K2[0],
              K2[3] << 16 | K2[2] >>> 16,
              K2[1],
              K2[0] << 16 | K2[3] >>> 16,
              K2[2],
              K2[1] << 16 | K2[0] >>> 16,
              K2[3],
              K2[2] << 16 | K2[1] >>> 16
            ];
            var C3 = this._C = [
              K2[2] << 16 | K2[2] >>> 16,
              K2[0] & 4294901760 | K2[1] & 65535,
              K2[3] << 16 | K2[3] >>> 16,
              K2[1] & 4294901760 | K2[2] & 65535,
              K2[0] << 16 | K2[0] >>> 16,
              K2[2] & 4294901760 | K2[3] & 65535,
              K2[1] << 16 | K2[1] >>> 16,
              K2[3] & 4294901760 | K2[0] & 65535
            ];
            this._b = 0;
            for (var i2 = 0; i2 < 4; i2++) {
              nextState.call(this);
            }
            for (var i2 = 0; i2 < 8; i2++) {
              C3[i2] ^= X2[i2 + 4 & 7];
            }
            if (iv) {
              var IV = iv.words;
              var IV_0 = IV[0];
              var IV_1 = IV[1];
              var i0 = (IV_0 << 8 | IV_0 >>> 24) & 16711935 | (IV_0 << 24 | IV_0 >>> 8) & 4278255360;
              var i22 = (IV_1 << 8 | IV_1 >>> 24) & 16711935 | (IV_1 << 24 | IV_1 >>> 8) & 4278255360;
              var i1 = i0 >>> 16 | i22 & 4294901760;
              var i3 = i22 << 16 | i0 & 65535;
              C3[0] ^= i0;
              C3[1] ^= i1;
              C3[2] ^= i22;
              C3[3] ^= i3;
              C3[4] ^= i0;
              C3[5] ^= i1;
              C3[6] ^= i22;
              C3[7] ^= i3;
              for (var i2 = 0; i2 < 4; i2++) {
                nextState.call(this);
              }
            }
          },
          _doProcessBlock: function(M2, offset2) {
            var X2 = this._X;
            nextState.call(this);
            S2[0] = X2[0] ^ X2[5] >>> 16 ^ X2[3] << 16;
            S2[1] = X2[2] ^ X2[7] >>> 16 ^ X2[5] << 16;
            S2[2] = X2[4] ^ X2[1] >>> 16 ^ X2[7] << 16;
            S2[3] = X2[6] ^ X2[3] >>> 16 ^ X2[1] << 16;
            for (var i2 = 0; i2 < 4; i2++) {
              S2[i2] = (S2[i2] << 8 | S2[i2] >>> 24) & 16711935 | (S2[i2] << 24 | S2[i2] >>> 8) & 4278255360;
              M2[offset2 + i2] ^= S2[i2];
            }
          },
          blockSize: 128 / 32,
          ivSize: 64 / 32
        });
        function nextState() {
          var X2 = this._X;
          var C3 = this._C;
          for (var i2 = 0; i2 < 8; i2++) {
            C_[i2] = C3[i2];
          }
          C3[0] = C3[0] + 1295307597 + this._b | 0;
          C3[1] = C3[1] + 3545052371 + (C3[0] >>> 0 < C_[0] >>> 0 ? 1 : 0) | 0;
          C3[2] = C3[2] + 886263092 + (C3[1] >>> 0 < C_[1] >>> 0 ? 1 : 0) | 0;
          C3[3] = C3[3] + 1295307597 + (C3[2] >>> 0 < C_[2] >>> 0 ? 1 : 0) | 0;
          C3[4] = C3[4] + 3545052371 + (C3[3] >>> 0 < C_[3] >>> 0 ? 1 : 0) | 0;
          C3[5] = C3[5] + 886263092 + (C3[4] >>> 0 < C_[4] >>> 0 ? 1 : 0) | 0;
          C3[6] = C3[6] + 1295307597 + (C3[5] >>> 0 < C_[5] >>> 0 ? 1 : 0) | 0;
          C3[7] = C3[7] + 3545052371 + (C3[6] >>> 0 < C_[6] >>> 0 ? 1 : 0) | 0;
          this._b = C3[7] >>> 0 < C_[7] >>> 0 ? 1 : 0;
          for (var i2 = 0; i2 < 8; i2++) {
            var gx = X2[i2] + C3[i2];
            var ga = gx & 65535;
            var gb = gx >>> 16;
            var gh = ((ga * ga >>> 17) + ga * gb >>> 15) + gb * gb;
            var gl = ((gx & 4294901760) * gx | 0) + ((gx & 65535) * gx | 0);
            G2[i2] = gh ^ gl;
          }
          X2[0] = G2[0] + (G2[7] << 16 | G2[7] >>> 16) + (G2[6] << 16 | G2[6] >>> 16) | 0;
          X2[1] = G2[1] + (G2[0] << 8 | G2[0] >>> 24) + G2[7] | 0;
          X2[2] = G2[2] + (G2[1] << 16 | G2[1] >>> 16) + (G2[0] << 16 | G2[0] >>> 16) | 0;
          X2[3] = G2[3] + (G2[2] << 8 | G2[2] >>> 24) + G2[1] | 0;
          X2[4] = G2[4] + (G2[3] << 16 | G2[3] >>> 16) + (G2[2] << 16 | G2[2] >>> 16) | 0;
          X2[5] = G2[5] + (G2[4] << 8 | G2[4] >>> 24) + G2[3] | 0;
          X2[6] = G2[6] + (G2[5] << 16 | G2[5] >>> 16) + (G2[4] << 16 | G2[4] >>> 16) | 0;
          X2[7] = G2[7] + (G2[6] << 8 | G2[6] >>> 24) + G2[5] | 0;
        }
        C2.Rabbit = StreamCipher._createHelper(Rabbit);
      })();
      return CryptoJS2.Rabbit;
    });
  })(rabbit);
  return rabbit.exports;
}
var rabbitLegacy = { exports: {} };
var hasRequiredRabbitLegacy;
function requireRabbitLegacy() {
  if (hasRequiredRabbitLegacy)
    return rabbitLegacy.exports;
  hasRequiredRabbitLegacy = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireEncBase64(), requireMd5(), requireEvpkdf(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var StreamCipher = C_lib.StreamCipher;
        var C_algo = C2.algo;
        var S2 = [];
        var C_ = [];
        var G2 = [];
        var RabbitLegacy = C_algo.RabbitLegacy = StreamCipher.extend({
          _doReset: function() {
            var K2 = this._key.words;
            var iv = this.cfg.iv;
            var X2 = this._X = [
              K2[0],
              K2[3] << 16 | K2[2] >>> 16,
              K2[1],
              K2[0] << 16 | K2[3] >>> 16,
              K2[2],
              K2[1] << 16 | K2[0] >>> 16,
              K2[3],
              K2[2] << 16 | K2[1] >>> 16
            ];
            var C3 = this._C = [
              K2[2] << 16 | K2[2] >>> 16,
              K2[0] & 4294901760 | K2[1] & 65535,
              K2[3] << 16 | K2[3] >>> 16,
              K2[1] & 4294901760 | K2[2] & 65535,
              K2[0] << 16 | K2[0] >>> 16,
              K2[2] & 4294901760 | K2[3] & 65535,
              K2[1] << 16 | K2[1] >>> 16,
              K2[3] & 4294901760 | K2[0] & 65535
            ];
            this._b = 0;
            for (var i2 = 0; i2 < 4; i2++) {
              nextState.call(this);
            }
            for (var i2 = 0; i2 < 8; i2++) {
              C3[i2] ^= X2[i2 + 4 & 7];
            }
            if (iv) {
              var IV = iv.words;
              var IV_0 = IV[0];
              var IV_1 = IV[1];
              var i0 = (IV_0 << 8 | IV_0 >>> 24) & 16711935 | (IV_0 << 24 | IV_0 >>> 8) & 4278255360;
              var i22 = (IV_1 << 8 | IV_1 >>> 24) & 16711935 | (IV_1 << 24 | IV_1 >>> 8) & 4278255360;
              var i1 = i0 >>> 16 | i22 & 4294901760;
              var i3 = i22 << 16 | i0 & 65535;
              C3[0] ^= i0;
              C3[1] ^= i1;
              C3[2] ^= i22;
              C3[3] ^= i3;
              C3[4] ^= i0;
              C3[5] ^= i1;
              C3[6] ^= i22;
              C3[7] ^= i3;
              for (var i2 = 0; i2 < 4; i2++) {
                nextState.call(this);
              }
            }
          },
          _doProcessBlock: function(M2, offset2) {
            var X2 = this._X;
            nextState.call(this);
            S2[0] = X2[0] ^ X2[5] >>> 16 ^ X2[3] << 16;
            S2[1] = X2[2] ^ X2[7] >>> 16 ^ X2[5] << 16;
            S2[2] = X2[4] ^ X2[1] >>> 16 ^ X2[7] << 16;
            S2[3] = X2[6] ^ X2[3] >>> 16 ^ X2[1] << 16;
            for (var i2 = 0; i2 < 4; i2++) {
              S2[i2] = (S2[i2] << 8 | S2[i2] >>> 24) & 16711935 | (S2[i2] << 24 | S2[i2] >>> 8) & 4278255360;
              M2[offset2 + i2] ^= S2[i2];
            }
          },
          blockSize: 128 / 32,
          ivSize: 64 / 32
        });
        function nextState() {
          var X2 = this._X;
          var C3 = this._C;
          for (var i2 = 0; i2 < 8; i2++) {
            C_[i2] = C3[i2];
          }
          C3[0] = C3[0] + 1295307597 + this._b | 0;
          C3[1] = C3[1] + 3545052371 + (C3[0] >>> 0 < C_[0] >>> 0 ? 1 : 0) | 0;
          C3[2] = C3[2] + 886263092 + (C3[1] >>> 0 < C_[1] >>> 0 ? 1 : 0) | 0;
          C3[3] = C3[3] + 1295307597 + (C3[2] >>> 0 < C_[2] >>> 0 ? 1 : 0) | 0;
          C3[4] = C3[4] + 3545052371 + (C3[3] >>> 0 < C_[3] >>> 0 ? 1 : 0) | 0;
          C3[5] = C3[5] + 886263092 + (C3[4] >>> 0 < C_[4] >>> 0 ? 1 : 0) | 0;
          C3[6] = C3[6] + 1295307597 + (C3[5] >>> 0 < C_[5] >>> 0 ? 1 : 0) | 0;
          C3[7] = C3[7] + 3545052371 + (C3[6] >>> 0 < C_[6] >>> 0 ? 1 : 0) | 0;
          this._b = C3[7] >>> 0 < C_[7] >>> 0 ? 1 : 0;
          for (var i2 = 0; i2 < 8; i2++) {
            var gx = X2[i2] + C3[i2];
            var ga = gx & 65535;
            var gb = gx >>> 16;
            var gh = ((ga * ga >>> 17) + ga * gb >>> 15) + gb * gb;
            var gl = ((gx & 4294901760) * gx | 0) + ((gx & 65535) * gx | 0);
            G2[i2] = gh ^ gl;
          }
          X2[0] = G2[0] + (G2[7] << 16 | G2[7] >>> 16) + (G2[6] << 16 | G2[6] >>> 16) | 0;
          X2[1] = G2[1] + (G2[0] << 8 | G2[0] >>> 24) + G2[7] | 0;
          X2[2] = G2[2] + (G2[1] << 16 | G2[1] >>> 16) + (G2[0] << 16 | G2[0] >>> 16) | 0;
          X2[3] = G2[3] + (G2[2] << 8 | G2[2] >>> 24) + G2[1] | 0;
          X2[4] = G2[4] + (G2[3] << 16 | G2[3] >>> 16) + (G2[2] << 16 | G2[2] >>> 16) | 0;
          X2[5] = G2[5] + (G2[4] << 8 | G2[4] >>> 24) + G2[3] | 0;
          X2[6] = G2[6] + (G2[5] << 16 | G2[5] >>> 16) + (G2[4] << 16 | G2[4] >>> 16) | 0;
          X2[7] = G2[7] + (G2[6] << 8 | G2[6] >>> 24) + G2[5] | 0;
        }
        C2.RabbitLegacy = StreamCipher._createHelper(RabbitLegacy);
      })();
      return CryptoJS2.RabbitLegacy;
    });
  })(rabbitLegacy);
  return rabbitLegacy.exports;
}
var blowfish = { exports: {} };
var hasRequiredBlowfish;
function requireBlowfish() {
  if (hasRequiredBlowfish)
    return blowfish.exports;
  hasRequiredBlowfish = 1;
  (function(module2, exports) {
    (function(root2, factory, undef) {
      {
        module2.exports = factory(requireCore(), requireEncBase64(), requireMd5(), requireEvpkdf(), requireCipherCore());
      }
    })(commonjsGlobal, function(CryptoJS2) {
      (function() {
        var C2 = CryptoJS2;
        var C_lib = C2.lib;
        var BlockCipher = C_lib.BlockCipher;
        var C_algo = C2.algo;
        const N2 = 16;
        const ORIG_P = [
          608135816,
          2242054355,
          320440878,
          57701188,
          2752067618,
          698298832,
          137296536,
          3964562569,
          1160258022,
          953160567,
          3193202383,
          887688300,
          3232508343,
          3380367581,
          1065670069,
          3041331479,
          2450970073,
          2306472731
        ];
        const ORIG_S = [
          [
            3509652390,
            2564797868,
            805139163,
            3491422135,
            3101798381,
            1780907670,
            3128725573,
            4046225305,
            614570311,
            3012652279,
            134345442,
            2240740374,
            1667834072,
            1901547113,
            2757295779,
            4103290238,
            227898511,
            1921955416,
            1904987480,
            2182433518,
            2069144605,
            3260701109,
            2620446009,
            720527379,
            3318853667,
            677414384,
            3393288472,
            3101374703,
            2390351024,
            1614419982,
            1822297739,
            2954791486,
            3608508353,
            3174124327,
            2024746970,
            1432378464,
            3864339955,
            2857741204,
            1464375394,
            1676153920,
            1439316330,
            715854006,
            3033291828,
            289532110,
            2706671279,
            2087905683,
            3018724369,
            1668267050,
            732546397,
            1947742710,
            3462151702,
            2609353502,
            2950085171,
            1814351708,
            2050118529,
            680887927,
            999245976,
            1800124847,
            3300911131,
            1713906067,
            1641548236,
            4213287313,
            1216130144,
            1575780402,
            4018429277,
            3917837745,
            3693486850,
            3949271944,
            596196993,
            3549867205,
            258830323,
            2213823033,
            772490370,
            2760122372,
            1774776394,
            2652871518,
            566650946,
            4142492826,
            1728879713,
            2882767088,
            1783734482,
            3629395816,
            2517608232,
            2874225571,
            1861159788,
            326777828,
            3124490320,
            2130389656,
            2716951837,
            967770486,
            1724537150,
            2185432712,
            2364442137,
            1164943284,
            2105845187,
            998989502,
            3765401048,
            2244026483,
            1075463327,
            1455516326,
            1322494562,
            910128902,
            469688178,
            1117454909,
            936433444,
            3490320968,
            3675253459,
            1240580251,
            122909385,
            2157517691,
            634681816,
            4142456567,
            3825094682,
            3061402683,
            2540495037,
            79693498,
            3249098678,
            1084186820,
            1583128258,
            426386531,
            1761308591,
            1047286709,
            322548459,
            995290223,
            1845252383,
            2603652396,
            3431023940,
            2942221577,
            3202600964,
            3727903485,
            1712269319,
            422464435,
            3234572375,
            1170764815,
            3523960633,
            3117677531,
            1434042557,
            442511882,
            3600875718,
            1076654713,
            1738483198,
            4213154764,
            2393238008,
            3677496056,
            1014306527,
            4251020053,
            793779912,
            2902807211,
            842905082,
            4246964064,
            1395751752,
            1040244610,
            2656851899,
            3396308128,
            445077038,
            3742853595,
            3577915638,
            679411651,
            2892444358,
            2354009459,
            1767581616,
            3150600392,
            3791627101,
            3102740896,
            284835224,
            4246832056,
            1258075500,
            768725851,
            2589189241,
            3069724005,
            3532540348,
            1274779536,
            3789419226,
            2764799539,
            1660621633,
            3471099624,
            4011903706,
            913787905,
            3497959166,
            737222580,
            2514213453,
            2928710040,
            3937242737,
            1804850592,
            3499020752,
            2949064160,
            2386320175,
            2390070455,
            2415321851,
            4061277028,
            2290661394,
            2416832540,
            1336762016,
            1754252060,
            3520065937,
            3014181293,
            791618072,
            3188594551,
            3933548030,
            2332172193,
            3852520463,
            3043980520,
            413987798,
            3465142937,
            3030929376,
            4245938359,
            2093235073,
            3534596313,
            375366246,
            2157278981,
            2479649556,
            555357303,
            3870105701,
            2008414854,
            3344188149,
            4221384143,
            3956125452,
            2067696032,
            3594591187,
            2921233993,
            2428461,
            544322398,
            577241275,
            1471733935,
            610547355,
            4027169054,
            1432588573,
            1507829418,
            2025931657,
            3646575487,
            545086370,
            48609733,
            2200306550,
            1653985193,
            298326376,
            1316178497,
            3007786442,
            2064951626,
            458293330,
            2589141269,
            3591329599,
            3164325604,
            727753846,
            2179363840,
            146436021,
            1461446943,
            4069977195,
            705550613,
            3059967265,
            3887724982,
            4281599278,
            3313849956,
            1404054877,
            2845806497,
            146425753,
            1854211946
          ],
          [
            1266315497,
            3048417604,
            3681880366,
            3289982499,
            290971e4,
            1235738493,
            2632868024,
            2414719590,
            3970600049,
            1771706367,
            1449415276,
            3266420449,
            422970021,
            1963543593,
            2690192192,
            3826793022,
            1062508698,
            1531092325,
            1804592342,
            2583117782,
            2714934279,
            4024971509,
            1294809318,
            4028980673,
            1289560198,
            2221992742,
            1669523910,
            35572830,
            157838143,
            1052438473,
            1016535060,
            1802137761,
            1753167236,
            1386275462,
            3080475397,
            2857371447,
            1040679964,
            2145300060,
            2390574316,
            1461121720,
            2956646967,
            4031777805,
            4028374788,
            33600511,
            2920084762,
            1018524850,
            629373528,
            3691585981,
            3515945977,
            2091462646,
            2486323059,
            586499841,
            988145025,
            935516892,
            3367335476,
            2599673255,
            2839830854,
            265290510,
            3972581182,
            2759138881,
            3795373465,
            1005194799,
            847297441,
            406762289,
            1314163512,
            1332590856,
            1866599683,
            4127851711,
            750260880,
            613907577,
            1450815602,
            3165620655,
            3734664991,
            3650291728,
            3012275730,
            3704569646,
            1427272223,
            778793252,
            1343938022,
            2676280711,
            2052605720,
            1946737175,
            3164576444,
            3914038668,
            3967478842,
            3682934266,
            1661551462,
            3294938066,
            4011595847,
            840292616,
            3712170807,
            616741398,
            312560963,
            711312465,
            1351876610,
            322626781,
            1910503582,
            271666773,
            2175563734,
            1594956187,
            70604529,
            3617834859,
            1007753275,
            1495573769,
            4069517037,
            2549218298,
            2663038764,
            504708206,
            2263041392,
            3941167025,
            2249088522,
            1514023603,
            1998579484,
            1312622330,
            694541497,
            2582060303,
            2151582166,
            1382467621,
            776784248,
            2618340202,
            3323268794,
            2497899128,
            2784771155,
            503983604,
            4076293799,
            907881277,
            423175695,
            432175456,
            1378068232,
            4145222326,
            3954048622,
            3938656102,
            3820766613,
            2793130115,
            2977904593,
            26017576,
            3274890735,
            3194772133,
            1700274565,
            1756076034,
            4006520079,
            3677328699,
            720338349,
            1533947780,
            354530856,
            688349552,
            3973924725,
            1637815568,
            332179504,
            3949051286,
            53804574,
            2852348879,
            3044236432,
            1282449977,
            3583942155,
            3416972820,
            4006381244,
            1617046695,
            2628476075,
            3002303598,
            1686838959,
            431878346,
            2686675385,
            1700445008,
            1080580658,
            1009431731,
            832498133,
            3223435511,
            2605976345,
            2271191193,
            2516031870,
            1648197032,
            4164389018,
            2548247927,
            300782431,
            375919233,
            238389289,
            3353747414,
            2531188641,
            2019080857,
            1475708069,
            455242339,
            2609103871,
            448939670,
            3451063019,
            1395535956,
            2413381860,
            1841049896,
            1491858159,
            885456874,
            4264095073,
            4001119347,
            1565136089,
            3898914787,
            1108368660,
            540939232,
            1173283510,
            2745871338,
            3681308437,
            4207628240,
            3343053890,
            4016749493,
            1699691293,
            1103962373,
            3625875870,
            2256883143,
            3830138730,
            1031889488,
            3479347698,
            1535977030,
            4236805024,
            3251091107,
            2132092099,
            1774941330,
            1199868427,
            1452454533,
            157007616,
            2904115357,
            342012276,
            595725824,
            1480756522,
            206960106,
            497939518,
            591360097,
            863170706,
            2375253569,
            3596610801,
            1814182875,
            2094937945,
            3421402208,
            1082520231,
            3463918190,
            2785509508,
            435703966,
            3908032597,
            1641649973,
            2842273706,
            3305899714,
            1510255612,
            2148256476,
            2655287854,
            3276092548,
            4258621189,
            236887753,
            3681803219,
            274041037,
            1734335097,
            3815195456,
            3317970021,
            1899903192,
            1026095262,
            4050517792,
            356393447,
            2410691914,
            3873677099,
            3682840055
          ],
          [
            3913112168,
            2491498743,
            4132185628,
            2489919796,
            1091903735,
            1979897079,
            3170134830,
            3567386728,
            3557303409,
            857797738,
            1136121015,
            1342202287,
            507115054,
            2535736646,
            337727348,
            3213592640,
            1301675037,
            2528481711,
            1895095763,
            1721773893,
            3216771564,
            62756741,
            2142006736,
            835421444,
            2531993523,
            1442658625,
            3659876326,
            2882144922,
            676362277,
            1392781812,
            170690266,
            3921047035,
            1759253602,
            3611846912,
            1745797284,
            664899054,
            1329594018,
            3901205900,
            3045908486,
            2062866102,
            2865634940,
            3543621612,
            3464012697,
            1080764994,
            553557557,
            3656615353,
            3996768171,
            991055499,
            499776247,
            1265440854,
            648242737,
            3940784050,
            980351604,
            3713745714,
            1749149687,
            3396870395,
            4211799374,
            3640570775,
            1161844396,
            3125318951,
            1431517754,
            545492359,
            4268468663,
            3499529547,
            1437099964,
            2702547544,
            3433638243,
            2581715763,
            2787789398,
            1060185593,
            1593081372,
            2418618748,
            4260947970,
            69676912,
            2159744348,
            86519011,
            2512459080,
            3838209314,
            1220612927,
            3339683548,
            133810670,
            1090789135,
            1078426020,
            1569222167,
            845107691,
            3583754449,
            4072456591,
            1091646820,
            628848692,
            1613405280,
            3757631651,
            526609435,
            236106946,
            48312990,
            2942717905,
            3402727701,
            1797494240,
            859738849,
            992217954,
            4005476642,
            2243076622,
            3870952857,
            3732016268,
            765654824,
            3490871365,
            2511836413,
            1685915746,
            3888969200,
            1414112111,
            2273134842,
            3281911079,
            4080962846,
            172450625,
            2569994100,
            980381355,
            4109958455,
            2819808352,
            2716589560,
            2568741196,
            3681446669,
            3329971472,
            1835478071,
            660984891,
            3704678404,
            4045999559,
            3422617507,
            3040415634,
            1762651403,
            1719377915,
            3470491036,
            2693910283,
            3642056355,
            3138596744,
            1364962596,
            2073328063,
            1983633131,
            926494387,
            3423689081,
            2150032023,
            4096667949,
            1749200295,
            3328846651,
            309677260,
            2016342300,
            1779581495,
            3079819751,
            111262694,
            1274766160,
            443224088,
            298511866,
            1025883608,
            3806446537,
            1145181785,
            168956806,
            3641502830,
            3584813610,
            1689216846,
            3666258015,
            3200248200,
            1692713982,
            2646376535,
            4042768518,
            1618508792,
            1610833997,
            3523052358,
            4130873264,
            2001055236,
            3610705100,
            2202168115,
            4028541809,
            2961195399,
            1006657119,
            2006996926,
            3186142756,
            1430667929,
            3210227297,
            1314452623,
            4074634658,
            4101304120,
            2273951170,
            1399257539,
            3367210612,
            3027628629,
            1190975929,
            2062231137,
            2333990788,
            2221543033,
            2438960610,
            1181637006,
            548689776,
            2362791313,
            3372408396,
            3104550113,
            3145860560,
            296247880,
            1970579870,
            3078560182,
            3769228297,
            1714227617,
            3291629107,
            3898220290,
            166772364,
            1251581989,
            493813264,
            448347421,
            195405023,
            2709975567,
            677966185,
            3703036547,
            1463355134,
            2715995803,
            1338867538,
            1343315457,
            2802222074,
            2684532164,
            233230375,
            2599980071,
            2000651841,
            3277868038,
            1638401717,
            4028070440,
            3237316320,
            6314154,
            819756386,
            300326615,
            590932579,
            1405279636,
            3267499572,
            3150704214,
            2428286686,
            3959192993,
            3461946742,
            1862657033,
            1266418056,
            963775037,
            2089974820,
            2263052895,
            1917689273,
            448879540,
            3550394620,
            3981727096,
            150775221,
            3627908307,
            1303187396,
            508620638,
            2975983352,
            2726630617,
            1817252668,
            1876281319,
            1457606340,
            908771278,
            3720792119,
            3617206836,
            2455994898,
            1729034894,
            1080033504
          ],
          [
            976866871,
            3556439503,
            2881648439,
            1522871579,
            1555064734,
            1336096578,
            3548522304,
            2579274686,
            3574697629,
            3205460757,
            3593280638,
            3338716283,
            3079412587,
            564236357,
            2993598910,
            1781952180,
            1464380207,
            3163844217,
            3332601554,
            1699332808,
            1393555694,
            1183702653,
            3581086237,
            1288719814,
            691649499,
            2847557200,
            2895455976,
            3193889540,
            2717570544,
            1781354906,
            1676643554,
            2592534050,
            3230253752,
            1126444790,
            2770207658,
            2633158820,
            2210423226,
            2615765581,
            2414155088,
            3127139286,
            673620729,
            2805611233,
            1269405062,
            4015350505,
            3341807571,
            4149409754,
            1057255273,
            2012875353,
            2162469141,
            2276492801,
            2601117357,
            993977747,
            3918593370,
            2654263191,
            753973209,
            36408145,
            2530585658,
            25011837,
            3520020182,
            2088578344,
            530523599,
            2918365339,
            1524020338,
            1518925132,
            3760827505,
            3759777254,
            1202760957,
            3985898139,
            3906192525,
            674977740,
            4174734889,
            2031300136,
            2019492241,
            3983892565,
            4153806404,
            3822280332,
            352677332,
            2297720250,
            60907813,
            90501309,
            3286998549,
            1016092578,
            2535922412,
            2839152426,
            457141659,
            509813237,
            4120667899,
            652014361,
            1966332200,
            2975202805,
            55981186,
            2327461051,
            676427537,
            3255491064,
            2882294119,
            3433927263,
            1307055953,
            942726286,
            933058658,
            2468411793,
            3933900994,
            4215176142,
            1361170020,
            2001714738,
            2830558078,
            3274259782,
            1222529897,
            1679025792,
            2729314320,
            3714953764,
            1770335741,
            151462246,
            3013232138,
            1682292957,
            1483529935,
            471910574,
            1539241949,
            458788160,
            3436315007,
            1807016891,
            3718408830,
            978976581,
            1043663428,
            3165965781,
            1927990952,
            4200891579,
            2372276910,
            3208408903,
            3533431907,
            1412390302,
            2931980059,
            4132332400,
            1947078029,
            3881505623,
            4168226417,
            2941484381,
            1077988104,
            1320477388,
            886195818,
            18198404,
            3786409e3,
            2509781533,
            112762804,
            3463356488,
            1866414978,
            891333506,
            18488651,
            661792760,
            1628790961,
            3885187036,
            3141171499,
            876946877,
            2693282273,
            1372485963,
            791857591,
            2686433993,
            3759982718,
            3167212022,
            3472953795,
            2716379847,
            445679433,
            3561995674,
            3504004811,
            3574258232,
            54117162,
            3331405415,
            2381918588,
            3769707343,
            4154350007,
            1140177722,
            4074052095,
            668550556,
            3214352940,
            367459370,
            261225585,
            2610173221,
            4209349473,
            3468074219,
            3265815641,
            314222801,
            3066103646,
            3808782860,
            282218597,
            3406013506,
            3773591054,
            379116347,
            1285071038,
            846784868,
            2669647154,
            3771962079,
            3550491691,
            2305946142,
            453669953,
            1268987020,
            3317592352,
            3279303384,
            3744833421,
            2610507566,
            3859509063,
            266596637,
            3847019092,
            517658769,
            3462560207,
            3443424879,
            370717030,
            4247526661,
            2224018117,
            4143653529,
            4112773975,
            2788324899,
            2477274417,
            1456262402,
            2901442914,
            1517677493,
            1846949527,
            2295493580,
            3734397586,
            2176403920,
            1280348187,
            1908823572,
            3871786941,
            846861322,
            1172426758,
            3287448474,
            3383383037,
            1655181056,
            3139813346,
            901632758,
            1897031941,
            2986607138,
            3066810236,
            3447102507,
            1393639104,
            373351379,
            950779232,
            625454576,
            3124240540,
            4148612726,
            2007998917,
            544563296,
            2244738638,
            2330496472,
            2058025392,
            1291430526,
            424198748,
            50039436,
            29584100,
            3605783033,
            2429876329,
            2791104160,
            1057563949,
            3255363231,
            3075367218,
            3463963227,
            1469046755,
            985887462
          ]
        ];
        var BLOWFISH_CTX = {
          pbox: [],
          sbox: []
        };
        function F2(ctx, x2) {
          let a2 = x2 >> 24 & 255;
          let b2 = x2 >> 16 & 255;
          let c2 = x2 >> 8 & 255;
          let d2 = x2 & 255;
          let y2 = ctx.sbox[0][a2] + ctx.sbox[1][b2];
          y2 = y2 ^ ctx.sbox[2][c2];
          y2 = y2 + ctx.sbox[3][d2];
          return y2;
        }
        function BlowFish_Encrypt(ctx, left, right) {
          let Xl = left;
          let Xr = right;
          let temp;
          for (let i2 = 0; i2 < N2; ++i2) {
            Xl = Xl ^ ctx.pbox[i2];
            Xr = F2(ctx, Xl) ^ Xr;
            temp = Xl;
            Xl = Xr;
            Xr = temp;
          }
          temp = Xl;
          Xl = Xr;
          Xr = temp;
          Xr = Xr ^ ctx.pbox[N2];
          Xl = Xl ^ ctx.pbox[N2 + 1];
          return { left: Xl, right: Xr };
        }
        function BlowFish_Decrypt(ctx, left, right) {
          let Xl = left;
          let Xr = right;
          let temp;
          for (let i2 = N2 + 1; i2 > 1; --i2) {
            Xl = Xl ^ ctx.pbox[i2];
            Xr = F2(ctx, Xl) ^ Xr;
            temp = Xl;
            Xl = Xr;
            Xr = temp;
          }
          temp = Xl;
          Xl = Xr;
          Xr = temp;
          Xr = Xr ^ ctx.pbox[1];
          Xl = Xl ^ ctx.pbox[0];
          return { left: Xl, right: Xr };
        }
        function BlowFishInit(ctx, key, keysize) {
          for (let Row = 0; Row < 4; Row++) {
            ctx.sbox[Row] = [];
            for (let Col = 0; Col < 256; Col++) {
              ctx.sbox[Row][Col] = ORIG_S[Row][Col];
            }
          }
          let keyIndex = 0;
          for (let index2 = 0; index2 < N2 + 2; index2++) {
            ctx.pbox[index2] = ORIG_P[index2] ^ key[keyIndex];
            keyIndex++;
            if (keyIndex >= keysize) {
              keyIndex = 0;
            }
          }
          let Data1 = 0;
          let Data2 = 0;
          let res = 0;
          for (let i2 = 0; i2 < N2 + 2; i2 += 2) {
            res = BlowFish_Encrypt(ctx, Data1, Data2);
            Data1 = res.left;
            Data2 = res.right;
            ctx.pbox[i2] = Data1;
            ctx.pbox[i2 + 1] = Data2;
          }
          for (let i2 = 0; i2 < 4; i2++) {
            for (let j2 = 0; j2 < 256; j2 += 2) {
              res = BlowFish_Encrypt(ctx, Data1, Data2);
              Data1 = res.left;
              Data2 = res.right;
              ctx.sbox[i2][j2] = Data1;
              ctx.sbox[i2][j2 + 1] = Data2;
            }
          }
          return true;
        }
        var Blowfish = C_algo.Blowfish = BlockCipher.extend({
          _doReset: function() {
            if (this._keyPriorReset === this._key) {
              return;
            }
            var key = this._keyPriorReset = this._key;
            var keyWords = key.words;
            var keySize = key.sigBytes / 4;
            BlowFishInit(BLOWFISH_CTX, keyWords, keySize);
          },
          encryptBlock: function(M2, offset2) {
            var res = BlowFish_Encrypt(BLOWFISH_CTX, M2[offset2], M2[offset2 + 1]);
            M2[offset2] = res.left;
            M2[offset2 + 1] = res.right;
          },
          decryptBlock: function(M2, offset2) {
            var res = BlowFish_Decrypt(BLOWFISH_CTX, M2[offset2], M2[offset2 + 1]);
            M2[offset2] = res.left;
            M2[offset2 + 1] = res.right;
          },
          blockSize: 64 / 32,
          keySize: 128 / 32,
          ivSize: 64 / 32
        });
        C2.Blowfish = BlockCipher._createHelper(Blowfish);
      })();
      return CryptoJS2.Blowfish;
    });
  })(blowfish);
  return blowfish.exports;
}
(function(module2, exports) {
  (function(root2, factory, undef) {
    {
      module2.exports = factory(requireCore(), requireX64Core(), requireLibTypedarrays(), requireEncUtf16(), requireEncBase64(), requireEncBase64url(), requireMd5(), requireSha1(), requireSha256(), requireSha224(), requireSha512(), requireSha384(), requireSha3(), requireRipemd160(), requireHmac(), requirePbkdf2(), requireEvpkdf(), requireCipherCore(), requireModeCfb(), requireModeCtr(), requireModeCtrGladman(), requireModeOfb(), requireModeEcb(), requirePadAnsix923(), requirePadIso10126(), requirePadIso97971(), requirePadZeropadding(), requirePadNopadding(), requireFormatHex(), requireAes(), requireTripledes(), requireRc4(), requireRabbit(), requireRabbitLegacy(), requireBlowfish());
    }
  })(commonjsGlobal, function(CryptoJS2) {
    return CryptoJS2;
  });
})(cryptoJs);
var cryptoJsExports = cryptoJs.exports;
const CryptoJS = /* @__PURE__ */ getDefaultExportFromCjs(cryptoJsExports);
/*! xlsx.js (C) 2013-present SheetJS -- http://sheetjs.com */
var XLSX = {};
XLSX.version = "0.18.5";
var current_ansi = 1252;
var VALID_ANSI = [874, 932, 936, 949, 950, 1250, 1251, 1252, 1253, 1254, 1255, 1256, 1257, 1258, 1e4];
var CS2CP = {
  /*::[*/
  0: 1252,
  /* ANSI */
  /*::[*/
  1: 65001,
  /* DEFAULT */
  /*::[*/
  2: 65001,
  /* SYMBOL */
  /*::[*/
  77: 1e4,
  /* MAC */
  /*::[*/
  128: 932,
  /* SHIFTJIS */
  /*::[*/
  129: 949,
  /* HANGUL */
  /*::[*/
  130: 1361,
  /* JOHAB */
  /*::[*/
  134: 936,
  /* GB2312 */
  /*::[*/
  136: 950,
  /* CHINESEBIG5 */
  /*::[*/
  161: 1253,
  /* GREEK */
  /*::[*/
  162: 1254,
  /* TURKISH */
  /*::[*/
  163: 1258,
  /* VIETNAMESE */
  /*::[*/
  177: 1255,
  /* HEBREW */
  /*::[*/
  178: 1256,
  /* ARABIC */
  /*::[*/
  186: 1257,
  /* BALTIC */
  /*::[*/
  204: 1251,
  /* RUSSIAN */
  /*::[*/
  222: 874,
  /* THAI */
  /*::[*/
  238: 1250,
  /* EASTEUROPE */
  /*::[*/
  255: 1252,
  /* OEM */
  /*::[*/
  69: 6969
  /* MISC */
};
var set_ansi = function(cp) {
  if (VALID_ANSI.indexOf(cp) == -1)
    return;
  current_ansi = CS2CP[0] = cp;
};
function reset_ansi() {
  set_ansi(1252);
}
var set_cp = function(cp) {
  set_ansi(cp);
};
function reset_cp() {
  set_cp(1200);
  reset_ansi();
}
function char_codes(data2) {
  var o2 = [];
  for (var i2 = 0, len2 = data2.length; i2 < len2; ++i2)
    o2[i2] = data2.charCodeAt(i2);
  return o2;
}
function utf16leread(data2) {
  var o2 = [];
  for (var i2 = 0; i2 < data2.length >> 1; ++i2)
    o2[i2] = String.fromCharCode(data2.charCodeAt(2 * i2) + (data2.charCodeAt(2 * i2 + 1) << 8));
  return o2.join("");
}
function utf16beread(data2) {
  var o2 = [];
  for (var i2 = 0; i2 < data2.length >> 1; ++i2)
    o2[i2] = String.fromCharCode(data2.charCodeAt(2 * i2 + 1) + (data2.charCodeAt(2 * i2) << 8));
  return o2.join("");
}
var debom = function(data2) {
  var c1 = data2.charCodeAt(0), c2 = data2.charCodeAt(1);
  if (c1 == 255 && c2 == 254)
    return utf16leread(data2.slice(2));
  if (c1 == 254 && c2 == 255)
    return utf16beread(data2.slice(2));
  if (c1 == 65279)
    return data2.slice(1);
  return data2;
};
var _getchar = function _gc1(x2) {
  return String.fromCharCode(x2);
};
var _getansi = function _ga1(x2) {
  return String.fromCharCode(x2);
};
var $cptable;
var Base64_map = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
function Base64_encode(input) {
  var o2 = "";
  var c1 = 0, c2 = 0, c3 = 0, e1 = 0, e2 = 0, e3 = 0, e4 = 0;
  for (var i2 = 0; i2 < input.length; ) {
    c1 = input.charCodeAt(i2++);
    e1 = c1 >> 2;
    c2 = input.charCodeAt(i2++);
    e2 = (c1 & 3) << 4 | c2 >> 4;
    c3 = input.charCodeAt(i2++);
    e3 = (c2 & 15) << 2 | c3 >> 6;
    e4 = c3 & 63;
    if (isNaN(c2)) {
      e3 = e4 = 64;
    } else if (isNaN(c3)) {
      e4 = 64;
    }
    o2 += Base64_map.charAt(e1) + Base64_map.charAt(e2) + Base64_map.charAt(e3) + Base64_map.charAt(e4);
  }
  return o2;
}
function Base64_decode(input) {
  var o2 = "";
  var c1 = 0, c2 = 0, c3 = 0, e1 = 0, e2 = 0, e3 = 0, e4 = 0;
  input = input.replace(/[^\w\+\/\=]/g, "");
  for (var i2 = 0; i2 < input.length; ) {
    e1 = Base64_map.indexOf(input.charAt(i2++));
    e2 = Base64_map.indexOf(input.charAt(i2++));
    c1 = e1 << 2 | e2 >> 4;
    o2 += String.fromCharCode(c1);
    e3 = Base64_map.indexOf(input.charAt(i2++));
    c2 = (e2 & 15) << 4 | e3 >> 2;
    if (e3 !== 64) {
      o2 += String.fromCharCode(c2);
    }
    e4 = Base64_map.indexOf(input.charAt(i2++));
    c3 = (e3 & 3) << 6 | e4;
    if (e4 !== 64) {
      o2 += String.fromCharCode(c3);
    }
  }
  return o2;
}
var has_buf = /* @__PURE__ */ function() {
  return typeof Buffer !== "undefined" && typeof process !== "undefined" && typeof process.versions !== "undefined" && !!process.versions.node;
}();
var Buffer_from = /* @__PURE__ */ function() {
  if (typeof Buffer !== "undefined") {
    var nbfs = !Buffer.from;
    if (!nbfs)
      try {
        Buffer.from("foo", "utf8");
      } catch (e2) {
        nbfs = true;
      }
    return nbfs ? function(buf, enc) {
      return enc ? new Buffer(buf, enc) : new Buffer(buf);
    } : Buffer.from.bind(Buffer);
  }
  return function() {
  };
}();
function new_raw_buf(len2) {
  if (has_buf)
    return Buffer.alloc ? Buffer.alloc(len2) : new Buffer(len2);
  return typeof Uint8Array != "undefined" ? new Uint8Array(len2) : new Array(len2);
}
function new_unsafe_buf(len2) {
  if (has_buf)
    return Buffer.allocUnsafe ? Buffer.allocUnsafe(len2) : new Buffer(len2);
  return typeof Uint8Array != "undefined" ? new Uint8Array(len2) : new Array(len2);
}
var s2a = function s2a2(s2) {
  if (has_buf)
    return Buffer_from(s2, "binary");
  return s2.split("").map(function(x2) {
    return x2.charCodeAt(0) & 255;
  });
};
function s2ab(s2) {
  if (typeof ArrayBuffer === "undefined")
    return s2a(s2);
  var buf = new ArrayBuffer(s2.length), view = new Uint8Array(buf);
  for (var i2 = 0; i2 != s2.length; ++i2)
    view[i2] = s2.charCodeAt(i2) & 255;
  return buf;
}
function a2s(data2) {
  if (Array.isArray(data2))
    return data2.map(function(c2) {
      return String.fromCharCode(c2);
    }).join("");
  var o2 = [];
  for (var i2 = 0; i2 < data2.length; ++i2)
    o2[i2] = String.fromCharCode(data2[i2]);
  return o2.join("");
}
function a2u(data2) {
  if (typeof Uint8Array === "undefined")
    throw new Error("Unsupported");
  return new Uint8Array(data2);
}
function ab2a(data2) {
  if (typeof ArrayBuffer == "undefined")
    throw new Error("Unsupported");
  if (data2 instanceof ArrayBuffer)
    return ab2a(new Uint8Array(data2));
  var o2 = new Array(data2.length);
  for (var i2 = 0; i2 < data2.length; ++i2)
    o2[i2] = data2[i2];
  return o2;
}
var bconcat = has_buf ? function(bufs) {
  return Buffer.concat(bufs.map(function(buf) {
    return Buffer.isBuffer(buf) ? buf : Buffer_from(buf);
  }));
} : function(bufs) {
  if (typeof Uint8Array !== "undefined") {
    var i2 = 0, maxlen = 0;
    for (i2 = 0; i2 < bufs.length; ++i2)
      maxlen += bufs[i2].length;
    var o2 = new Uint8Array(maxlen);
    var len2 = 0;
    for (i2 = 0, maxlen = 0; i2 < bufs.length; maxlen += len2, ++i2) {
      len2 = bufs[i2].length;
      if (bufs[i2] instanceof Uint8Array)
        o2.set(bufs[i2], maxlen);
      else if (typeof bufs[i2] == "string") {
        throw "wtf";
      } else
        o2.set(new Uint8Array(bufs[i2]), maxlen);
    }
    return o2;
  }
  return [].concat.apply([], bufs.map(function(buf) {
    return Array.isArray(buf) ? buf : [].slice.call(buf);
  }));
};
function utf8decode(content2) {
  var out2 = [], widx = 0, L2 = content2.length + 250;
  var o2 = new_raw_buf(content2.length + 255);
  for (var ridx = 0; ridx < content2.length; ++ridx) {
    var c2 = content2.charCodeAt(ridx);
    if (c2 < 128)
      o2[widx++] = c2;
    else if (c2 < 2048) {
      o2[widx++] = 192 | c2 >> 6 & 31;
      o2[widx++] = 128 | c2 & 63;
    } else if (c2 >= 55296 && c2 < 57344) {
      c2 = (c2 & 1023) + 64;
      var d2 = content2.charCodeAt(++ridx) & 1023;
      o2[widx++] = 240 | c2 >> 8 & 7;
      o2[widx++] = 128 | c2 >> 2 & 63;
      o2[widx++] = 128 | d2 >> 6 & 15 | (c2 & 3) << 4;
      o2[widx++] = 128 | d2 & 63;
    } else {
      o2[widx++] = 224 | c2 >> 12 & 15;
      o2[widx++] = 128 | c2 >> 6 & 63;
      o2[widx++] = 128 | c2 & 63;
    }
    if (widx > L2) {
      out2.push(o2.slice(0, widx));
      widx = 0;
      o2 = new_raw_buf(65535);
      L2 = 65530;
    }
  }
  out2.push(o2.slice(0, widx));
  return bconcat(out2);
}
var chr0 = /\u0000/g, chr1 = /[\u0001-\u0006]/g;
function _strrev(x2) {
  var o2 = "", i2 = x2.length - 1;
  while (i2 >= 0)
    o2 += x2.charAt(i2--);
  return o2;
}
function pad0(v4, d2) {
  var t3 = "" + v4;
  return t3.length >= d2 ? t3 : fill("0", d2 - t3.length) + t3;
}
function pad_(v4, d2) {
  var t3 = "" + v4;
  return t3.length >= d2 ? t3 : fill(" ", d2 - t3.length) + t3;
}
function rpad_(v4, d2) {
  var t3 = "" + v4;
  return t3.length >= d2 ? t3 : t3 + fill(" ", d2 - t3.length);
}
function pad0r1(v4, d2) {
  var t3 = "" + Math.round(v4);
  return t3.length >= d2 ? t3 : fill("0", d2 - t3.length) + t3;
}
function pad0r2(v4, d2) {
  var t3 = "" + v4;
  return t3.length >= d2 ? t3 : fill("0", d2 - t3.length) + t3;
}
var p2_32 = /* @__PURE__ */ Math.pow(2, 32);
function pad0r(v4, d2) {
  if (v4 > p2_32 || v4 < -p2_32)
    return pad0r1(v4, d2);
  var i2 = Math.round(v4);
  return pad0r2(i2, d2);
}
function SSF_isgeneral(s2, i2) {
  i2 = i2 || 0;
  return s2.length >= 7 + i2 && (s2.charCodeAt(i2) | 32) === 103 && (s2.charCodeAt(i2 + 1) | 32) === 101 && (s2.charCodeAt(i2 + 2) | 32) === 110 && (s2.charCodeAt(i2 + 3) | 32) === 101 && (s2.charCodeAt(i2 + 4) | 32) === 114 && (s2.charCodeAt(i2 + 5) | 32) === 97 && (s2.charCodeAt(i2 + 6) | 32) === 108;
}
var days$1 = [
  ["Sun", "Sunday"],
  ["Mon", "Monday"],
  ["Tue", "Tuesday"],
  ["Wed", "Wednesday"],
  ["Thu", "Thursday"],
  ["Fri", "Friday"],
  ["Sat", "Saturday"]
];
var months$1 = [
  ["J", "Jan", "January"],
  ["F", "Feb", "February"],
  ["M", "Mar", "March"],
  ["A", "Apr", "April"],
  ["M", "May", "May"],
  ["J", "Jun", "June"],
  ["J", "Jul", "July"],
  ["A", "Aug", "August"],
  ["S", "Sep", "September"],
  ["O", "Oct", "October"],
  ["N", "Nov", "November"],
  ["D", "Dec", "December"]
];
function SSF_init_table(t3) {
  if (!t3)
    t3 = {};
  t3[0] = "General";
  t3[1] = "0";
  t3[2] = "0.00";
  t3[3] = "#,##0";
  t3[4] = "#,##0.00";
  t3[9] = "0%";
  t3[10] = "0.00%";
  t3[11] = "0.00E+00";
  t3[12] = "# ?/?";
  t3[13] = "# ??/??";
  t3[14] = "m/d/yy";
  t3[15] = "d-mmm-yy";
  t3[16] = "d-mmm";
  t3[17] = "mmm-yy";
  t3[18] = "h:mm AM/PM";
  t3[19] = "h:mm:ss AM/PM";
  t3[20] = "h:mm";
  t3[21] = "h:mm:ss";
  t3[22] = "m/d/yy h:mm";
  t3[37] = "#,##0 ;(#,##0)";
  t3[38] = "#,##0 ;[Red](#,##0)";
  t3[39] = "#,##0.00;(#,##0.00)";
  t3[40] = "#,##0.00;[Red](#,##0.00)";
  t3[45] = "mm:ss";
  t3[46] = "[h]:mm:ss";
  t3[47] = "mmss.0";
  t3[48] = "##0.0E+0";
  t3[49] = "@";
  t3[56] = '"上午/下午 "hh"時"mm"分"ss"秒 "';
  return t3;
}
var table_fmt = {
  0: "General",
  1: "0",
  2: "0.00",
  3: "#,##0",
  4: "#,##0.00",
  9: "0%",
  10: "0.00%",
  11: "0.00E+00",
  12: "# ?/?",
  13: "# ??/??",
  14: "m/d/yy",
  15: "d-mmm-yy",
  16: "d-mmm",
  17: "mmm-yy",
  18: "h:mm AM/PM",
  19: "h:mm:ss AM/PM",
  20: "h:mm",
  21: "h:mm:ss",
  22: "m/d/yy h:mm",
  37: "#,##0 ;(#,##0)",
  38: "#,##0 ;[Red](#,##0)",
  39: "#,##0.00;(#,##0.00)",
  40: "#,##0.00;[Red](#,##0.00)",
  45: "mm:ss",
  46: "[h]:mm:ss",
  47: "mmss.0",
  48: "##0.0E+0",
  49: "@",
  56: '"上午/下午 "hh"時"mm"分"ss"秒 "'
};
var SSF_default_map = {
  5: 37,
  6: 38,
  7: 39,
  8: 40,
  //  5 -> 37 ...  8 -> 40
  23: 0,
  24: 0,
  25: 0,
  26: 0,
  // 23 ->  0 ... 26 ->  0
  27: 14,
  28: 14,
  29: 14,
  30: 14,
  31: 14,
  // 27 -> 14 ... 31 -> 14
  50: 14,
  51: 14,
  52: 14,
  53: 14,
  54: 14,
  // 50 -> 14 ... 58 -> 14
  55: 14,
  56: 14,
  57: 14,
  58: 14,
  59: 1,
  60: 2,
  61: 3,
  62: 4,
  // 59 ->  1 ... 62 ->  4
  67: 9,
  68: 10,
  // 67 ->  9 ... 68 -> 10
  69: 12,
  70: 13,
  71: 14,
  // 69 -> 12 ... 71 -> 14
  72: 14,
  73: 15,
  74: 16,
  75: 17,
  // 72 -> 14 ... 75 -> 17
  76: 20,
  77: 21,
  78: 22,
  // 76 -> 20 ... 78 -> 22
  79: 45,
  80: 46,
  81: 47,
  // 79 -> 45 ... 81 -> 47
  82: 0
  // 82 ->  0 ... 65536 -> 0 (omitted)
};
var SSF_default_str = {
  //  5 -- Currency,   0 decimal, black negative
  5: '"$"#,##0_);\\("$"#,##0\\)',
  63: '"$"#,##0_);\\("$"#,##0\\)',
  //  6 -- Currency,   0 decimal, red   negative
  6: '"$"#,##0_);[Red]\\("$"#,##0\\)',
  64: '"$"#,##0_);[Red]\\("$"#,##0\\)',
  //  7 -- Currency,   2 decimal, black negative
  7: '"$"#,##0.00_);\\("$"#,##0.00\\)',
  65: '"$"#,##0.00_);\\("$"#,##0.00\\)',
  //  8 -- Currency,   2 decimal, red   negative
  8: '"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',
  66: '"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',
  // 41 -- Accounting, 0 decimal, No Symbol
  41: '_(* #,##0_);_(* \\(#,##0\\);_(* "-"_);_(@_)',
  // 42 -- Accounting, 0 decimal, $  Symbol
  42: '_("$"* #,##0_);_("$"* \\(#,##0\\);_("$"* "-"_);_(@_)',
  // 43 -- Accounting, 2 decimal, No Symbol
  43: '_(* #,##0.00_);_(* \\(#,##0.00\\);_(* "-"??_);_(@_)',
  // 44 -- Accounting, 2 decimal, $  Symbol
  44: '_("$"* #,##0.00_);_("$"* \\(#,##0.00\\);_("$"* "-"??_);_(@_)'
};
function SSF_frac(x2, D2, mixed) {
  var sgn = x2 < 0 ? -1 : 1;
  var B3 = x2 * sgn;
  var P_2 = 0, P_1 = 1, P2 = 0;
  var Q_2 = 1, Q_1 = 0, Q2 = 0;
  var A2 = Math.floor(B3);
  while (Q_1 < D2) {
    A2 = Math.floor(B3);
    P2 = A2 * P_1 + P_2;
    Q2 = A2 * Q_1 + Q_2;
    if (B3 - A2 < 5e-8)
      break;
    B3 = 1 / (B3 - A2);
    P_2 = P_1;
    P_1 = P2;
    Q_2 = Q_1;
    Q_1 = Q2;
  }
  if (Q2 > D2) {
    if (Q_1 > D2) {
      Q2 = Q_2;
      P2 = P_2;
    } else {
      Q2 = Q_1;
      P2 = P_1;
    }
  }
  if (!mixed)
    return [0, sgn * P2, Q2];
  var q2 = Math.floor(sgn * P2 / Q2);
  return [q2, sgn * P2 - q2 * Q2, Q2];
}
function SSF_parse_date_code(v4, opts, b2) {
  if (v4 > 2958465 || v4 < 0)
    return null;
  var date = v4 | 0, time2 = Math.floor(86400 * (v4 - date)), dow = 0;
  var dout = [];
  var out2 = { D: date, T: time2, u: 86400 * (v4 - date) - time2, y: 0, m: 0, d: 0, H: 0, M: 0, S: 0, q: 0 };
  if (Math.abs(out2.u) < 1e-6)
    out2.u = 0;
  if (opts && opts.date1904)
    date += 1462;
  if (out2.u > 0.9999) {
    out2.u = 0;
    if (++time2 == 86400) {
      out2.T = time2 = 0;
      ++date;
      ++out2.D;
    }
  }
  if (date === 60) {
    dout = b2 ? [1317, 10, 29] : [1900, 2, 29];
    dow = 3;
  } else if (date === 0) {
    dout = b2 ? [1317, 8, 29] : [1900, 1, 0];
    dow = 6;
  } else {
    if (date > 60)
      --date;
    var d2 = new Date(1900, 0, 1);
    d2.setDate(d2.getDate() + date - 1);
    dout = [d2.getFullYear(), d2.getMonth() + 1, d2.getDate()];
    dow = d2.getDay();
    if (date < 60)
      dow = (dow + 6) % 7;
    if (b2)
      dow = SSF_fix_hijri(d2, dout);
  }
  out2.y = dout[0];
  out2.m = dout[1];
  out2.d = dout[2];
  out2.S = time2 % 60;
  time2 = Math.floor(time2 / 60);
  out2.M = time2 % 60;
  time2 = Math.floor(time2 / 60);
  out2.H = time2;
  out2.q = dow;
  return out2;
}
var SSFbasedate = /* @__PURE__ */ new Date(1899, 11, 31, 0, 0, 0);
var SSFdnthresh = /* @__PURE__ */ SSFbasedate.getTime();
var SSFbase1904 = /* @__PURE__ */ new Date(1900, 2, 1, 0, 0, 0);
function datenum_local(v4, date1904) {
  var epoch = /* @__PURE__ */ v4.getTime();
  if (date1904)
    epoch -= 1461 * 24 * 60 * 60 * 1e3;
  else if (v4 >= SSFbase1904)
    epoch += 24 * 60 * 60 * 1e3;
  return (epoch - (SSFdnthresh + (/* @__PURE__ */ v4.getTimezoneOffset() - /* @__PURE__ */ SSFbasedate.getTimezoneOffset()) * 6e4)) / (24 * 60 * 60 * 1e3);
}
function SSF_strip_decimal(o2) {
  return o2.indexOf(".") == -1 ? o2 : o2.replace(/(?:\.0*|(\.\d*[1-9])0+)$/, "$1");
}
function SSF_normalize_exp(o2) {
  if (o2.indexOf("E") == -1)
    return o2;
  return o2.replace(/(?:\.0*|(\.\d*[1-9])0+)[Ee]/, "$1E").replace(/(E[+-])(\d)$/, "$10$2");
}
function SSF_small_exp(v4) {
  var w2 = v4 < 0 ? 12 : 11;
  var o2 = SSF_strip_decimal(v4.toFixed(12));
  if (o2.length <= w2)
    return o2;
  o2 = v4.toPrecision(10);
  if (o2.length <= w2)
    return o2;
  return v4.toExponential(5);
}
function SSF_large_exp(v4) {
  var o2 = SSF_strip_decimal(v4.toFixed(11));
  return o2.length > (v4 < 0 ? 12 : 11) || o2 === "0" || o2 === "-0" ? v4.toPrecision(6) : o2;
}
function SSF_general_num(v4) {
  var V2 = Math.floor(Math.log(Math.abs(v4)) * Math.LOG10E), o2;
  if (V2 >= -4 && V2 <= -1)
    o2 = v4.toPrecision(10 + V2);
  else if (Math.abs(V2) <= 9)
    o2 = SSF_small_exp(v4);
  else if (V2 === 10)
    o2 = v4.toFixed(10).substr(0, 12);
  else
    o2 = SSF_large_exp(v4);
  return SSF_strip_decimal(SSF_normalize_exp(o2.toUpperCase()));
}
function SSF_general(v4, opts) {
  switch (typeof v4) {
    case "string":
      return v4;
    case "boolean":
      return v4 ? "TRUE" : "FALSE";
    case "number":
      return (v4 | 0) === v4 ? v4.toString(10) : SSF_general_num(v4);
    case "undefined":
      return "";
    case "object":
      if (v4 == null)
        return "";
      if (v4 instanceof Date)
        return SSF_format(14, datenum_local(v4, opts && opts.date1904), opts);
  }
  throw new Error("unsupported value in General format: " + v4);
}
function SSF_fix_hijri(date, o2) {
  o2[0] -= 581;
  var dow = date.getDay();
  if (date < 60)
    dow = (dow + 6) % 7;
  return dow;
}
function SSF_write_date(type2, fmt, val, ss0) {
  var o2 = "", ss = 0, tt2 = 0, y2 = val.y, out2, outl = 0;
  switch (type2) {
    case 98:
      y2 = val.y + 543;
    case 121:
      switch (fmt.length) {
        case 1:
        case 2:
          out2 = y2 % 100;
          outl = 2;
          break;
        default:
          out2 = y2 % 1e4;
          outl = 4;
          break;
      }
      break;
    case 109:
      switch (fmt.length) {
        case 1:
        case 2:
          out2 = val.m;
          outl = fmt.length;
          break;
        case 3:
          return months$1[val.m - 1][1];
        case 5:
          return months$1[val.m - 1][0];
        default:
          return months$1[val.m - 1][2];
      }
      break;
    case 100:
      switch (fmt.length) {
        case 1:
        case 2:
          out2 = val.d;
          outl = fmt.length;
          break;
        case 3:
          return days$1[val.q][0];
        default:
          return days$1[val.q][1];
      }
      break;
    case 104:
      switch (fmt.length) {
        case 1:
        case 2:
          out2 = 1 + (val.H + 11) % 12;
          outl = fmt.length;
          break;
        default:
          throw "bad hour format: " + fmt;
      }
      break;
    case 72:
      switch (fmt.length) {
        case 1:
        case 2:
          out2 = val.H;
          outl = fmt.length;
          break;
        default:
          throw "bad hour format: " + fmt;
      }
      break;
    case 77:
      switch (fmt.length) {
        case 1:
        case 2:
          out2 = val.M;
          outl = fmt.length;
          break;
        default:
          throw "bad minute format: " + fmt;
      }
      break;
    case 115:
      if (fmt != "s" && fmt != "ss" && fmt != ".0" && fmt != ".00" && fmt != ".000")
        throw "bad second format: " + fmt;
      if (val.u === 0 && (fmt == "s" || fmt == "ss"))
        return pad0(val.S, fmt.length);
      if (ss0 >= 2)
        tt2 = ss0 === 3 ? 1e3 : 100;
      else
        tt2 = ss0 === 1 ? 10 : 1;
      ss = Math.round(tt2 * (val.S + val.u));
      if (ss >= 60 * tt2)
        ss = 0;
      if (fmt === "s")
        return ss === 0 ? "0" : "" + ss / tt2;
      o2 = pad0(ss, 2 + ss0);
      if (fmt === "ss")
        return o2.substr(0, 2);
      return "." + o2.substr(2, fmt.length - 1);
    case 90:
      switch (fmt) {
        case "[h]":
        case "[hh]":
          out2 = val.D * 24 + val.H;
          break;
        case "[m]":
        case "[mm]":
          out2 = (val.D * 24 + val.H) * 60 + val.M;
          break;
        case "[s]":
        case "[ss]":
          out2 = ((val.D * 24 + val.H) * 60 + val.M) * 60 + Math.round(val.S + val.u);
          break;
        default:
          throw "bad abstime format: " + fmt;
      }
      outl = fmt.length === 3 ? 1 : 2;
      break;
    case 101:
      out2 = y2;
      outl = 1;
      break;
  }
  var outstr = outl > 0 ? pad0(out2, outl) : "";
  return outstr;
}
function commaify(s2) {
  var w2 = 3;
  if (s2.length <= w2)
    return s2;
  var j2 = s2.length % w2, o2 = s2.substr(0, j2);
  for (; j2 != s2.length; j2 += w2)
    o2 += (o2.length > 0 ? "," : "") + s2.substr(j2, w2);
  return o2;
}
var pct1 = /%/g;
function write_num_pct(type2, fmt, val) {
  var sfmt = fmt.replace(pct1, ""), mul3 = fmt.length - sfmt.length;
  return write_num(type2, sfmt, val * Math.pow(10, 2 * mul3)) + fill("%", mul3);
}
function write_num_cm(type2, fmt, val) {
  var idx = fmt.length - 1;
  while (fmt.charCodeAt(idx - 1) === 44)
    --idx;
  return write_num(type2, fmt.substr(0, idx), val / Math.pow(10, 3 * (fmt.length - idx)));
}
function write_num_exp(fmt, val) {
  var o2;
  var idx = fmt.indexOf("E") - fmt.indexOf(".") - 1;
  if (fmt.match(/^#+0.0E\+0$/)) {
    if (val == 0)
      return "0.0E+0";
    else if (val < 0)
      return "-" + write_num_exp(fmt, -val);
    var period = fmt.indexOf(".");
    if (period === -1)
      period = fmt.indexOf("E");
    var ee2 = Math.floor(Math.log(val) * Math.LOG10E) % period;
    if (ee2 < 0)
      ee2 += period;
    o2 = (val / Math.pow(10, ee2)).toPrecision(idx + 1 + (period + ee2) % period);
    if (o2.indexOf("e") === -1) {
      var fakee = Math.floor(Math.log(val) * Math.LOG10E);
      if (o2.indexOf(".") === -1)
        o2 = o2.charAt(0) + "." + o2.substr(1) + "E+" + (fakee - o2.length + ee2);
      else
        o2 += "E+" + (fakee - ee2);
      while (o2.substr(0, 2) === "0.") {
        o2 = o2.charAt(0) + o2.substr(2, period) + "." + o2.substr(2 + period);
        o2 = o2.replace(/^0+([1-9])/, "$1").replace(/^0+\./, "0.");
      }
      o2 = o2.replace(/\+-/, "-");
    }
    o2 = o2.replace(/^([+-]?)(\d*)\.(\d*)[Ee]/, function($$, $1, $2, $3) {
      return $1 + $2 + $3.substr(0, (period + ee2) % period) + "." + $3.substr(ee2) + "E";
    });
  } else
    o2 = val.toExponential(idx);
  if (fmt.match(/E\+00$/) && o2.match(/e[+-]\d$/))
    o2 = o2.substr(0, o2.length - 1) + "0" + o2.charAt(o2.length - 1);
  if (fmt.match(/E\-/) && o2.match(/e\+/))
    o2 = o2.replace(/e\+/, "e");
  return o2.replace("e", "E");
}
var frac1 = /# (\?+)( ?)\/( ?)(\d+)/;
function write_num_f1(r2, aval, sign2) {
  var den = parseInt(r2[4], 10), rr2 = Math.round(aval * den), base2 = Math.floor(rr2 / den);
  var myn = rr2 - base2 * den, myd = den;
  return sign2 + (base2 === 0 ? "" : "" + base2) + " " + (myn === 0 ? fill(" ", r2[1].length + 1 + r2[4].length) : pad_(myn, r2[1].length) + r2[2] + "/" + r2[3] + pad0(myd, r2[4].length));
}
function write_num_f2(r2, aval, sign2) {
  return sign2 + (aval === 0 ? "" : "" + aval) + fill(" ", r2[1].length + 2 + r2[4].length);
}
var dec1 = /^#*0*\.([0#]+)/;
var closeparen = /\).*[0#]/;
var phone = /\(###\) ###\\?-####/;
function hashq(str) {
  var o2 = "", cc;
  for (var i2 = 0; i2 != str.length; ++i2)
    switch (cc = str.charCodeAt(i2)) {
      case 35:
        break;
      case 63:
        o2 += " ";
        break;
      case 48:
        o2 += "0";
        break;
      default:
        o2 += String.fromCharCode(cc);
    }
  return o2;
}
function rnd(val, d2) {
  var dd = Math.pow(10, d2);
  return "" + Math.round(val * dd) / dd;
}
function dec(val, d2) {
  var _frac = val - Math.floor(val), dd = Math.pow(10, d2);
  if (d2 < ("" + Math.round(_frac * dd)).length)
    return 0;
  return Math.round(_frac * dd);
}
function carry(val, d2) {
  if (d2 < ("" + Math.round((val - Math.floor(val)) * Math.pow(10, d2))).length) {
    return 1;
  }
  return 0;
}
function flr(val) {
  if (val < 2147483647 && val > -2147483648)
    return "" + (val >= 0 ? val | 0 : val - 1 | 0);
  return "" + Math.floor(val);
}
function write_num_flt(type2, fmt, val) {
  if (type2.charCodeAt(0) === 40 && !fmt.match(closeparen)) {
    var ffmt = fmt.replace(/\( */, "").replace(/ \)/, "").replace(/\)/, "");
    if (val >= 0)
      return write_num_flt("n", ffmt, val);
    return "(" + write_num_flt("n", ffmt, -val) + ")";
  }
  if (fmt.charCodeAt(fmt.length - 1) === 44)
    return write_num_cm(type2, fmt, val);
  if (fmt.indexOf("%") !== -1)
    return write_num_pct(type2, fmt, val);
  if (fmt.indexOf("E") !== -1)
    return write_num_exp(fmt, val);
  if (fmt.charCodeAt(0) === 36)
    return "$" + write_num_flt(type2, fmt.substr(fmt.charAt(1) == " " ? 2 : 1), val);
  var o2;
  var r2, ri, ff, aval = Math.abs(val), sign2 = val < 0 ? "-" : "";
  if (fmt.match(/^00+$/))
    return sign2 + pad0r(aval, fmt.length);
  if (fmt.match(/^[#?]+$/)) {
    o2 = pad0r(val, 0);
    if (o2 === "0")
      o2 = "";
    return o2.length > fmt.length ? o2 : hashq(fmt.substr(0, fmt.length - o2.length)) + o2;
  }
  if (r2 = fmt.match(frac1))
    return write_num_f1(r2, aval, sign2);
  if (fmt.match(/^#+0+$/))
    return sign2 + pad0r(aval, fmt.length - fmt.indexOf("0"));
  if (r2 = fmt.match(dec1)) {
    o2 = rnd(val, r2[1].length).replace(/^([^\.]+)$/, "$1." + hashq(r2[1])).replace(/\.$/, "." + hashq(r2[1])).replace(/\.(\d*)$/, function($$, $1) {
      return "." + $1 + fill("0", hashq(
        /*::(*/
        r2[1]
      ).length - $1.length);
    });
    return fmt.indexOf("0.") !== -1 ? o2 : o2.replace(/^0\./, ".");
  }
  fmt = fmt.replace(/^#+([0.])/, "$1");
  if (r2 = fmt.match(/^(0*)\.(#*)$/)) {
    return sign2 + rnd(aval, r2[2].length).replace(/\.(\d*[1-9])0*$/, ".$1").replace(/^(-?\d*)$/, "$1.").replace(/^0\./, r2[1].length ? "0." : ".");
  }
  if (r2 = fmt.match(/^#{1,3},##0(\.?)$/))
    return sign2 + commaify(pad0r(aval, 0));
  if (r2 = fmt.match(/^#,##0\.([#0]*0)$/)) {
    return val < 0 ? "-" + write_num_flt(type2, fmt, -val) : commaify("" + (Math.floor(val) + carry(val, r2[1].length))) + "." + pad0(dec(val, r2[1].length), r2[1].length);
  }
  if (r2 = fmt.match(/^#,#*,#0/))
    return write_num_flt(type2, fmt.replace(/^#,#*,/, ""), val);
  if (r2 = fmt.match(/^([0#]+)(\\?-([0#]+))+$/)) {
    o2 = _strrev(write_num_flt(type2, fmt.replace(/[\\-]/g, ""), val));
    ri = 0;
    return _strrev(_strrev(fmt.replace(/\\/g, "")).replace(/[0#]/g, function(x3) {
      return ri < o2.length ? o2.charAt(ri++) : x3 === "0" ? "0" : "";
    }));
  }
  if (fmt.match(phone)) {
    o2 = write_num_flt(type2, "##########", val);
    return "(" + o2.substr(0, 3) + ") " + o2.substr(3, 3) + "-" + o2.substr(6);
  }
  var oa = "";
  if (r2 = fmt.match(/^([#0?]+)( ?)\/( ?)([#0?]+)/)) {
    ri = Math.min(
      /*::String(*/
      r2[4].length,
      7
    );
    ff = SSF_frac(aval, Math.pow(10, ri) - 1, false);
    o2 = "" + sign2;
    oa = write_num(
      "n",
      /*::String(*/
      r2[1],
      ff[1]
    );
    if (oa.charAt(oa.length - 1) == " ")
      oa = oa.substr(0, oa.length - 1) + "0";
    o2 += oa + /*::String(*/
    r2[2] + "/" + /*::String(*/
    r2[3];
    oa = rpad_(ff[2], ri);
    if (oa.length < r2[4].length)
      oa = hashq(r2[4].substr(r2[4].length - oa.length)) + oa;
    o2 += oa;
    return o2;
  }
  if (r2 = fmt.match(/^# ([#0?]+)( ?)\/( ?)([#0?]+)/)) {
    ri = Math.min(Math.max(r2[1].length, r2[4].length), 7);
    ff = SSF_frac(aval, Math.pow(10, ri) - 1, true);
    return sign2 + (ff[0] || (ff[1] ? "" : "0")) + " " + (ff[1] ? pad_(ff[1], ri) + r2[2] + "/" + r2[3] + rpad_(ff[2], ri) : fill(" ", 2 * ri + 1 + r2[2].length + r2[3].length));
  }
  if (r2 = fmt.match(/^[#0?]+$/)) {
    o2 = pad0r(val, 0);
    if (fmt.length <= o2.length)
      return o2;
    return hashq(fmt.substr(0, fmt.length - o2.length)) + o2;
  }
  if (r2 = fmt.match(/^([#0?]+)\.([#0]+)$/)) {
    o2 = "" + val.toFixed(Math.min(r2[2].length, 10)).replace(/([^0])0+$/, "$1");
    ri = o2.indexOf(".");
    var lres = fmt.indexOf(".") - ri, rres = fmt.length - o2.length - lres;
    return hashq(fmt.substr(0, lres) + o2 + fmt.substr(fmt.length - rres));
  }
  if (r2 = fmt.match(/^00,000\.([#0]*0)$/)) {
    ri = dec(val, r2[1].length);
    return val < 0 ? "-" + write_num_flt(type2, fmt, -val) : commaify(flr(val)).replace(/^\d,\d{3}$/, "0$&").replace(/^\d*$/, function($$) {
      return "00," + ($$.length < 3 ? pad0(0, 3 - $$.length) : "") + $$;
    }) + "." + pad0(ri, r2[1].length);
  }
  switch (fmt) {
    case "###,##0.00":
      return write_num_flt(type2, "#,##0.00", val);
    case "###,###":
    case "##,###":
    case "#,###":
      var x2 = commaify(pad0r(aval, 0));
      return x2 !== "0" ? sign2 + x2 : "";
    case "###,###.00":
      return write_num_flt(type2, "###,##0.00", val).replace(/^0\./, ".");
    case "#,###.00":
      return write_num_flt(type2, "#,##0.00", val).replace(/^0\./, ".");
  }
  throw new Error("unsupported format |" + fmt + "|");
}
function write_num_cm2(type2, fmt, val) {
  var idx = fmt.length - 1;
  while (fmt.charCodeAt(idx - 1) === 44)
    --idx;
  return write_num(type2, fmt.substr(0, idx), val / Math.pow(10, 3 * (fmt.length - idx)));
}
function write_num_pct2(type2, fmt, val) {
  var sfmt = fmt.replace(pct1, ""), mul3 = fmt.length - sfmt.length;
  return write_num(type2, sfmt, val * Math.pow(10, 2 * mul3)) + fill("%", mul3);
}
function write_num_exp2(fmt, val) {
  var o2;
  var idx = fmt.indexOf("E") - fmt.indexOf(".") - 1;
  if (fmt.match(/^#+0.0E\+0$/)) {
    if (val == 0)
      return "0.0E+0";
    else if (val < 0)
      return "-" + write_num_exp2(fmt, -val);
    var period = fmt.indexOf(".");
    if (period === -1)
      period = fmt.indexOf("E");
    var ee2 = Math.floor(Math.log(val) * Math.LOG10E) % period;
    if (ee2 < 0)
      ee2 += period;
    o2 = (val / Math.pow(10, ee2)).toPrecision(idx + 1 + (period + ee2) % period);
    if (!o2.match(/[Ee]/)) {
      var fakee = Math.floor(Math.log(val) * Math.LOG10E);
      if (o2.indexOf(".") === -1)
        o2 = o2.charAt(0) + "." + o2.substr(1) + "E+" + (fakee - o2.length + ee2);
      else
        o2 += "E+" + (fakee - ee2);
      o2 = o2.replace(/\+-/, "-");
    }
    o2 = o2.replace(/^([+-]?)(\d*)\.(\d*)[Ee]/, function($$, $1, $2, $3) {
      return $1 + $2 + $3.substr(0, (period + ee2) % period) + "." + $3.substr(ee2) + "E";
    });
  } else
    o2 = val.toExponential(idx);
  if (fmt.match(/E\+00$/) && o2.match(/e[+-]\d$/))
    o2 = o2.substr(0, o2.length - 1) + "0" + o2.charAt(o2.length - 1);
  if (fmt.match(/E\-/) && o2.match(/e\+/))
    o2 = o2.replace(/e\+/, "e");
  return o2.replace("e", "E");
}
function write_num_int(type2, fmt, val) {
  if (type2.charCodeAt(0) === 40 && !fmt.match(closeparen)) {
    var ffmt = fmt.replace(/\( */, "").replace(/ \)/, "").replace(/\)/, "");
    if (val >= 0)
      return write_num_int("n", ffmt, val);
    return "(" + write_num_int("n", ffmt, -val) + ")";
  }
  if (fmt.charCodeAt(fmt.length - 1) === 44)
    return write_num_cm2(type2, fmt, val);
  if (fmt.indexOf("%") !== -1)
    return write_num_pct2(type2, fmt, val);
  if (fmt.indexOf("E") !== -1)
    return write_num_exp2(fmt, val);
  if (fmt.charCodeAt(0) === 36)
    return "$" + write_num_int(type2, fmt.substr(fmt.charAt(1) == " " ? 2 : 1), val);
  var o2;
  var r2, ri, ff, aval = Math.abs(val), sign2 = val < 0 ? "-" : "";
  if (fmt.match(/^00+$/))
    return sign2 + pad0(aval, fmt.length);
  if (fmt.match(/^[#?]+$/)) {
    o2 = "" + val;
    if (val === 0)
      o2 = "";
    return o2.length > fmt.length ? o2 : hashq(fmt.substr(0, fmt.length - o2.length)) + o2;
  }
  if (r2 = fmt.match(frac1))
    return write_num_f2(r2, aval, sign2);
  if (fmt.match(/^#+0+$/))
    return sign2 + pad0(aval, fmt.length - fmt.indexOf("0"));
  if (r2 = fmt.match(dec1)) {
    o2 = ("" + val).replace(/^([^\.]+)$/, "$1." + hashq(r2[1])).replace(/\.$/, "." + hashq(r2[1]));
    o2 = o2.replace(/\.(\d*)$/, function($$, $1) {
      return "." + $1 + fill("0", hashq(r2[1]).length - $1.length);
    });
    return fmt.indexOf("0.") !== -1 ? o2 : o2.replace(/^0\./, ".");
  }
  fmt = fmt.replace(/^#+([0.])/, "$1");
  if (r2 = fmt.match(/^(0*)\.(#*)$/)) {
    return sign2 + ("" + aval).replace(/\.(\d*[1-9])0*$/, ".$1").replace(/^(-?\d*)$/, "$1.").replace(/^0\./, r2[1].length ? "0." : ".");
  }
  if (r2 = fmt.match(/^#{1,3},##0(\.?)$/))
    return sign2 + commaify("" + aval);
  if (r2 = fmt.match(/^#,##0\.([#0]*0)$/)) {
    return val < 0 ? "-" + write_num_int(type2, fmt, -val) : commaify("" + val) + "." + fill("0", r2[1].length);
  }
  if (r2 = fmt.match(/^#,#*,#0/))
    return write_num_int(type2, fmt.replace(/^#,#*,/, ""), val);
  if (r2 = fmt.match(/^([0#]+)(\\?-([0#]+))+$/)) {
    o2 = _strrev(write_num_int(type2, fmt.replace(/[\\-]/g, ""), val));
    ri = 0;
    return _strrev(_strrev(fmt.replace(/\\/g, "")).replace(/[0#]/g, function(x3) {
      return ri < o2.length ? o2.charAt(ri++) : x3 === "0" ? "0" : "";
    }));
  }
  if (fmt.match(phone)) {
    o2 = write_num_int(type2, "##########", val);
    return "(" + o2.substr(0, 3) + ") " + o2.substr(3, 3) + "-" + o2.substr(6);
  }
  var oa = "";
  if (r2 = fmt.match(/^([#0?]+)( ?)\/( ?)([#0?]+)/)) {
    ri = Math.min(
      /*::String(*/
      r2[4].length,
      7
    );
    ff = SSF_frac(aval, Math.pow(10, ri) - 1, false);
    o2 = "" + sign2;
    oa = write_num(
      "n",
      /*::String(*/
      r2[1],
      ff[1]
    );
    if (oa.charAt(oa.length - 1) == " ")
      oa = oa.substr(0, oa.length - 1) + "0";
    o2 += oa + /*::String(*/
    r2[2] + "/" + /*::String(*/
    r2[3];
    oa = rpad_(ff[2], ri);
    if (oa.length < r2[4].length)
      oa = hashq(r2[4].substr(r2[4].length - oa.length)) + oa;
    o2 += oa;
    return o2;
  }
  if (r2 = fmt.match(/^# ([#0?]+)( ?)\/( ?)([#0?]+)/)) {
    ri = Math.min(Math.max(r2[1].length, r2[4].length), 7);
    ff = SSF_frac(aval, Math.pow(10, ri) - 1, true);
    return sign2 + (ff[0] || (ff[1] ? "" : "0")) + " " + (ff[1] ? pad_(ff[1], ri) + r2[2] + "/" + r2[3] + rpad_(ff[2], ri) : fill(" ", 2 * ri + 1 + r2[2].length + r2[3].length));
  }
  if (r2 = fmt.match(/^[#0?]+$/)) {
    o2 = "" + val;
    if (fmt.length <= o2.length)
      return o2;
    return hashq(fmt.substr(0, fmt.length - o2.length)) + o2;
  }
  if (r2 = fmt.match(/^([#0]+)\.([#0]+)$/)) {
    o2 = "" + val.toFixed(Math.min(r2[2].length, 10)).replace(/([^0])0+$/, "$1");
    ri = o2.indexOf(".");
    var lres = fmt.indexOf(".") - ri, rres = fmt.length - o2.length - lres;
    return hashq(fmt.substr(0, lres) + o2 + fmt.substr(fmt.length - rres));
  }
  if (r2 = fmt.match(/^00,000\.([#0]*0)$/)) {
    return val < 0 ? "-" + write_num_int(type2, fmt, -val) : commaify("" + val).replace(/^\d,\d{3}$/, "0$&").replace(/^\d*$/, function($$) {
      return "00," + ($$.length < 3 ? pad0(0, 3 - $$.length) : "") + $$;
    }) + "." + pad0(0, r2[1].length);
  }
  switch (fmt) {
    case "###,###":
    case "##,###":
    case "#,###":
      var x2 = commaify("" + aval);
      return x2 !== "0" ? sign2 + x2 : "";
    default:
      if (fmt.match(/\.[0#?]*$/))
        return write_num_int(type2, fmt.slice(0, fmt.lastIndexOf(".")), val) + hashq(fmt.slice(fmt.lastIndexOf(".")));
  }
  throw new Error("unsupported format |" + fmt + "|");
}
function write_num(type2, fmt, val) {
  return (val | 0) === val ? write_num_int(type2, fmt, val) : write_num_flt(type2, fmt, val);
}
function SSF_split_fmt(fmt) {
  var out2 = [];
  var in_str = false;
  for (var i2 = 0, j2 = 0; i2 < fmt.length; ++i2)
    switch (
      /*cc=*/
      fmt.charCodeAt(i2)
    ) {
      case 34:
        in_str = !in_str;
        break;
      case 95:
      case 42:
      case 92:
        ++i2;
        break;
      case 59:
        out2[out2.length] = fmt.substr(j2, i2 - j2);
        j2 = i2 + 1;
    }
  out2[out2.length] = fmt.substr(j2);
  if (in_str === true)
    throw new Error("Format |" + fmt + "| unterminated string ");
  return out2;
}
var SSF_abstime = /\[[HhMmSs\u0E0A\u0E19\u0E17]*\]/;
function fmt_is_date(fmt) {
  var i2 = 0, c2 = "", o2 = "";
  while (i2 < fmt.length) {
    switch (c2 = fmt.charAt(i2)) {
      case "G":
        if (SSF_isgeneral(fmt, i2))
          i2 += 6;
        i2++;
        break;
      case '"':
        for (
          ;
          /*cc=*/
          fmt.charCodeAt(++i2) !== 34 && i2 < fmt.length;
        ) {
        }
        ++i2;
        break;
      case "\\":
        i2 += 2;
        break;
      case "_":
        i2 += 2;
        break;
      case "@":
        ++i2;
        break;
      case "B":
      case "b":
        if (fmt.charAt(i2 + 1) === "1" || fmt.charAt(i2 + 1) === "2")
          return true;
      case "M":
      case "D":
      case "Y":
      case "H":
      case "S":
      case "E":
      case "m":
      case "d":
      case "y":
      case "h":
      case "s":
      case "e":
      case "g":
        return true;
      case "A":
      case "a":
      case "上":
        if (fmt.substr(i2, 3).toUpperCase() === "A/P")
          return true;
        if (fmt.substr(i2, 5).toUpperCase() === "AM/PM")
          return true;
        if (fmt.substr(i2, 5).toUpperCase() === "上午/下午")
          return true;
        ++i2;
        break;
      case "[":
        o2 = c2;
        while (fmt.charAt(i2++) !== "]" && i2 < fmt.length)
          o2 += fmt.charAt(i2);
        if (o2.match(SSF_abstime))
          return true;
        break;
      case ".":
      case "0":
      case "#":
        while (i2 < fmt.length && ("0#?.,E+-%".indexOf(c2 = fmt.charAt(++i2)) > -1 || c2 == "\\" && fmt.charAt(i2 + 1) == "-" && "0#".indexOf(fmt.charAt(i2 + 2)) > -1)) {
        }
        break;
      case "?":
        while (fmt.charAt(++i2) === c2) {
        }
        break;
      case "*":
        ++i2;
        if (fmt.charAt(i2) == " " || fmt.charAt(i2) == "*")
          ++i2;
        break;
      case "(":
      case ")":
        ++i2;
        break;
      case "1":
      case "2":
      case "3":
      case "4":
      case "5":
      case "6":
      case "7":
      case "8":
      case "9":
        while (i2 < fmt.length && "0123456789".indexOf(fmt.charAt(++i2)) > -1) {
        }
        break;
      case " ":
        ++i2;
        break;
      default:
        ++i2;
        break;
    }
  }
  return false;
}
function eval_fmt(fmt, v4, opts, flen) {
  var out2 = [], o2 = "", i2 = 0, c2 = "", lst = "t", dt2, j2, cc;
  var hr = "H";
  while (i2 < fmt.length) {
    switch (c2 = fmt.charAt(i2)) {
      case "G":
        if (!SSF_isgeneral(fmt, i2))
          throw new Error("unrecognized character " + c2 + " in " + fmt);
        out2[out2.length] = { t: "G", v: "General" };
        i2 += 7;
        break;
      case '"':
        for (o2 = ""; (cc = fmt.charCodeAt(++i2)) !== 34 && i2 < fmt.length; )
          o2 += String.fromCharCode(cc);
        out2[out2.length] = { t: "t", v: o2 };
        ++i2;
        break;
      case "\\":
        var w2 = fmt.charAt(++i2), t3 = w2 === "(" || w2 === ")" ? w2 : "t";
        out2[out2.length] = { t: t3, v: w2 };
        ++i2;
        break;
      case "_":
        out2[out2.length] = { t: "t", v: " " };
        i2 += 2;
        break;
      case "@":
        out2[out2.length] = { t: "T", v: v4 };
        ++i2;
        break;
      case "B":
      case "b":
        if (fmt.charAt(i2 + 1) === "1" || fmt.charAt(i2 + 1) === "2") {
          if (dt2 == null) {
            dt2 = SSF_parse_date_code(v4, opts, fmt.charAt(i2 + 1) === "2");
            if (dt2 == null)
              return "";
          }
          out2[out2.length] = { t: "X", v: fmt.substr(i2, 2) };
          lst = c2;
          i2 += 2;
          break;
        }
      case "M":
      case "D":
      case "Y":
      case "H":
      case "S":
      case "E":
        c2 = c2.toLowerCase();
      case "m":
      case "d":
      case "y":
      case "h":
      case "s":
      case "e":
      case "g":
        if (v4 < 0)
          return "";
        if (dt2 == null) {
          dt2 = SSF_parse_date_code(v4, opts);
          if (dt2 == null)
            return "";
        }
        o2 = c2;
        while (++i2 < fmt.length && fmt.charAt(i2).toLowerCase() === c2)
          o2 += c2;
        if (c2 === "m" && lst.toLowerCase() === "h")
          c2 = "M";
        if (c2 === "h")
          c2 = hr;
        out2[out2.length] = { t: c2, v: o2 };
        lst = c2;
        break;
      case "A":
      case "a":
      case "上":
        var q2 = { t: c2, v: c2 };
        if (dt2 == null)
          dt2 = SSF_parse_date_code(v4, opts);
        if (fmt.substr(i2, 3).toUpperCase() === "A/P") {
          if (dt2 != null)
            q2.v = dt2.H >= 12 ? "P" : "A";
          q2.t = "T";
          hr = "h";
          i2 += 3;
        } else if (fmt.substr(i2, 5).toUpperCase() === "AM/PM") {
          if (dt2 != null)
            q2.v = dt2.H >= 12 ? "PM" : "AM";
          q2.t = "T";
          i2 += 5;
          hr = "h";
        } else if (fmt.substr(i2, 5).toUpperCase() === "上午/下午") {
          if (dt2 != null)
            q2.v = dt2.H >= 12 ? "下午" : "上午";
          q2.t = "T";
          i2 += 5;
          hr = "h";
        } else {
          q2.t = "t";
          ++i2;
        }
        if (dt2 == null && q2.t === "T")
          return "";
        out2[out2.length] = q2;
        lst = c2;
        break;
      case "[":
        o2 = c2;
        while (fmt.charAt(i2++) !== "]" && i2 < fmt.length)
          o2 += fmt.charAt(i2);
        if (o2.slice(-1) !== "]")
          throw 'unterminated "[" block: |' + o2 + "|";
        if (o2.match(SSF_abstime)) {
          if (dt2 == null) {
            dt2 = SSF_parse_date_code(v4, opts);
            if (dt2 == null)
              return "";
          }
          out2[out2.length] = { t: "Z", v: o2.toLowerCase() };
          lst = o2.charAt(1);
        } else if (o2.indexOf("$") > -1) {
          o2 = (o2.match(/\$([^-\[\]]*)/) || [])[1] || "$";
          if (!fmt_is_date(fmt))
            out2[out2.length] = { t: "t", v: o2 };
        }
        break;
      case ".":
        if (dt2 != null) {
          o2 = c2;
          while (++i2 < fmt.length && (c2 = fmt.charAt(i2)) === "0")
            o2 += c2;
          out2[out2.length] = { t: "s", v: o2 };
          break;
        }
      case "0":
      case "#":
        o2 = c2;
        while (++i2 < fmt.length && "0#?.,E+-%".indexOf(c2 = fmt.charAt(i2)) > -1)
          o2 += c2;
        out2[out2.length] = { t: "n", v: o2 };
        break;
      case "?":
        o2 = c2;
        while (fmt.charAt(++i2) === c2)
          o2 += c2;
        out2[out2.length] = { t: c2, v: o2 };
        lst = c2;
        break;
      case "*":
        ++i2;
        if (fmt.charAt(i2) == " " || fmt.charAt(i2) == "*")
          ++i2;
        break;
      case "(":
      case ")":
        out2[out2.length] = { t: flen === 1 ? "t" : c2, v: c2 };
        ++i2;
        break;
      case "1":
      case "2":
      case "3":
      case "4":
      case "5":
      case "6":
      case "7":
      case "8":
      case "9":
        o2 = c2;
        while (i2 < fmt.length && "0123456789".indexOf(fmt.charAt(++i2)) > -1)
          o2 += fmt.charAt(i2);
        out2[out2.length] = { t: "D", v: o2 };
        break;
      case " ":
        out2[out2.length] = { t: c2, v: c2 };
        ++i2;
        break;
      case "$":
        out2[out2.length] = { t: "t", v: "$" };
        ++i2;
        break;
      default:
        if (",$-+/():!^&'~{}<>=€acfijklopqrtuvwxzP".indexOf(c2) === -1)
          throw new Error("unrecognized character " + c2 + " in " + fmt);
        out2[out2.length] = { t: "t", v: c2 };
        ++i2;
        break;
    }
  }
  var bt2 = 0, ss0 = 0, ssm;
  for (i2 = out2.length - 1, lst = "t"; i2 >= 0; --i2) {
    switch (out2[i2].t) {
      case "h":
      case "H":
        out2[i2].t = hr;
        lst = "h";
        if (bt2 < 1)
          bt2 = 1;
        break;
      case "s":
        if (ssm = out2[i2].v.match(/\.0+$/))
          ss0 = Math.max(ss0, ssm[0].length - 1);
        if (bt2 < 3)
          bt2 = 3;
      case "d":
      case "y":
      case "M":
      case "e":
        lst = out2[i2].t;
        break;
      case "m":
        if (lst === "s") {
          out2[i2].t = "M";
          if (bt2 < 2)
            bt2 = 2;
        }
        break;
      case "X":
        break;
      case "Z":
        if (bt2 < 1 && out2[i2].v.match(/[Hh]/))
          bt2 = 1;
        if (bt2 < 2 && out2[i2].v.match(/[Mm]/))
          bt2 = 2;
        if (bt2 < 3 && out2[i2].v.match(/[Ss]/))
          bt2 = 3;
    }
  }
  switch (bt2) {
    case 0:
      break;
    case 1:
      if (dt2.u >= 0.5) {
        dt2.u = 0;
        ++dt2.S;
      }
      if (dt2.S >= 60) {
        dt2.S = 0;
        ++dt2.M;
      }
      if (dt2.M >= 60) {
        dt2.M = 0;
        ++dt2.H;
      }
      break;
    case 2:
      if (dt2.u >= 0.5) {
        dt2.u = 0;
        ++dt2.S;
      }
      if (dt2.S >= 60) {
        dt2.S = 0;
        ++dt2.M;
      }
      break;
  }
  var nstr = "", jj;
  for (i2 = 0; i2 < out2.length; ++i2) {
    switch (out2[i2].t) {
      case "t":
      case "T":
      case " ":
      case "D":
        break;
      case "X":
        out2[i2].v = "";
        out2[i2].t = ";";
        break;
      case "d":
      case "m":
      case "y":
      case "h":
      case "H":
      case "M":
      case "s":
      case "e":
      case "b":
      case "Z":
        out2[i2].v = SSF_write_date(out2[i2].t.charCodeAt(0), out2[i2].v, dt2, ss0);
        out2[i2].t = "t";
        break;
      case "n":
      case "?":
        jj = i2 + 1;
        while (out2[jj] != null && ((c2 = out2[jj].t) === "?" || c2 === "D" || (c2 === " " || c2 === "t") && out2[jj + 1] != null && (out2[jj + 1].t === "?" || out2[jj + 1].t === "t" && out2[jj + 1].v === "/") || out2[i2].t === "(" && (c2 === " " || c2 === "n" || c2 === ")") || c2 === "t" && (out2[jj].v === "/" || out2[jj].v === " " && out2[jj + 1] != null && out2[jj + 1].t == "?"))) {
          out2[i2].v += out2[jj].v;
          out2[jj] = { v: "", t: ";" };
          ++jj;
        }
        nstr += out2[i2].v;
        i2 = jj - 1;
        break;
      case "G":
        out2[i2].t = "t";
        out2[i2].v = SSF_general(v4, opts);
        break;
    }
  }
  var vv2 = "", myv, ostr;
  if (nstr.length > 0) {
    if (nstr.charCodeAt(0) == 40) {
      myv = v4 < 0 && nstr.charCodeAt(0) === 45 ? -v4 : v4;
      ostr = write_num("n", nstr, myv);
    } else {
      myv = v4 < 0 && flen > 1 ? -v4 : v4;
      ostr = write_num("n", nstr, myv);
      if (myv < 0 && out2[0] && out2[0].t == "t") {
        ostr = ostr.substr(1);
        out2[0].v = "-" + out2[0].v;
      }
    }
    jj = ostr.length - 1;
    var decpt = out2.length;
    for (i2 = 0; i2 < out2.length; ++i2)
      if (out2[i2] != null && out2[i2].t != "t" && out2[i2].v.indexOf(".") > -1) {
        decpt = i2;
        break;
      }
    var lasti = out2.length;
    if (decpt === out2.length && ostr.indexOf("E") === -1) {
      for (i2 = out2.length - 1; i2 >= 0; --i2) {
        if (out2[i2] == null || "n?".indexOf(out2[i2].t) === -1)
          continue;
        if (jj >= out2[i2].v.length - 1) {
          jj -= out2[i2].v.length;
          out2[i2].v = ostr.substr(jj + 1, out2[i2].v.length);
        } else if (jj < 0)
          out2[i2].v = "";
        else {
          out2[i2].v = ostr.substr(0, jj + 1);
          jj = -1;
        }
        out2[i2].t = "t";
        lasti = i2;
      }
      if (jj >= 0 && lasti < out2.length)
        out2[lasti].v = ostr.substr(0, jj + 1) + out2[lasti].v;
    } else if (decpt !== out2.length && ostr.indexOf("E") === -1) {
      jj = ostr.indexOf(".") - 1;
      for (i2 = decpt; i2 >= 0; --i2) {
        if (out2[i2] == null || "n?".indexOf(out2[i2].t) === -1)
          continue;
        j2 = out2[i2].v.indexOf(".") > -1 && i2 === decpt ? out2[i2].v.indexOf(".") - 1 : out2[i2].v.length - 1;
        vv2 = out2[i2].v.substr(j2 + 1);
        for (; j2 >= 0; --j2) {
          if (jj >= 0 && (out2[i2].v.charAt(j2) === "0" || out2[i2].v.charAt(j2) === "#"))
            vv2 = ostr.charAt(jj--) + vv2;
        }
        out2[i2].v = vv2;
        out2[i2].t = "t";
        lasti = i2;
      }
      if (jj >= 0 && lasti < out2.length)
        out2[lasti].v = ostr.substr(0, jj + 1) + out2[lasti].v;
      jj = ostr.indexOf(".") + 1;
      for (i2 = decpt; i2 < out2.length; ++i2) {
        if (out2[i2] == null || "n?(".indexOf(out2[i2].t) === -1 && i2 !== decpt)
          continue;
        j2 = out2[i2].v.indexOf(".") > -1 && i2 === decpt ? out2[i2].v.indexOf(".") + 1 : 0;
        vv2 = out2[i2].v.substr(0, j2);
        for (; j2 < out2[i2].v.length; ++j2) {
          if (jj < ostr.length)
            vv2 += ostr.charAt(jj++);
        }
        out2[i2].v = vv2;
        out2[i2].t = "t";
        lasti = i2;
      }
    }
  }
  for (i2 = 0; i2 < out2.length; ++i2)
    if (out2[i2] != null && "n?".indexOf(out2[i2].t) > -1) {
      myv = flen > 1 && v4 < 0 && i2 > 0 && out2[i2 - 1].v === "-" ? -v4 : v4;
      out2[i2].v = write_num(out2[i2].t, out2[i2].v, myv);
      out2[i2].t = "t";
    }
  var retval = "";
  for (i2 = 0; i2 !== out2.length; ++i2)
    if (out2[i2] != null)
      retval += out2[i2].v;
  return retval;
}
var cfregex2 = /\[(=|>[=]?|<[>=]?)(-?\d+(?:\.\d*)?)\]/;
function chkcond(v4, rr2) {
  if (rr2 == null)
    return false;
  var thresh = parseFloat(rr2[2]);
  switch (rr2[1]) {
    case "=":
      if (v4 == thresh)
        return true;
      break;
    case ">":
      if (v4 > thresh)
        return true;
      break;
    case "<":
      if (v4 < thresh)
        return true;
      break;
    case "<>":
      if (v4 != thresh)
        return true;
      break;
    case ">=":
      if (v4 >= thresh)
        return true;
      break;
    case "<=":
      if (v4 <= thresh)
        return true;
      break;
  }
  return false;
}
function choose_fmt(f2, v4) {
  var fmt = SSF_split_fmt(f2);
  var l2 = fmt.length, lat = fmt[l2 - 1].indexOf("@");
  if (l2 < 4 && lat > -1)
    --l2;
  if (fmt.length > 4)
    throw new Error("cannot find right format for |" + fmt.join("|") + "|");
  if (typeof v4 !== "number")
    return [4, fmt.length === 4 || lat > -1 ? fmt[fmt.length - 1] : "@"];
  switch (fmt.length) {
    case 1:
      fmt = lat > -1 ? ["General", "General", "General", fmt[0]] : [fmt[0], fmt[0], fmt[0], "@"];
      break;
    case 2:
      fmt = lat > -1 ? [fmt[0], fmt[0], fmt[0], fmt[1]] : [fmt[0], fmt[1], fmt[0], "@"];
      break;
    case 3:
      fmt = lat > -1 ? [fmt[0], fmt[1], fmt[0], fmt[2]] : [fmt[0], fmt[1], fmt[2], "@"];
      break;
  }
  var ff = v4 > 0 ? fmt[0] : v4 < 0 ? fmt[1] : fmt[2];
  if (fmt[0].indexOf("[") === -1 && fmt[1].indexOf("[") === -1)
    return [l2, ff];
  if (fmt[0].match(/\[[=<>]/) != null || fmt[1].match(/\[[=<>]/) != null) {
    var m1 = fmt[0].match(cfregex2);
    var m2 = fmt[1].match(cfregex2);
    return chkcond(v4, m1) ? [l2, fmt[0]] : chkcond(v4, m2) ? [l2, fmt[1]] : [l2, fmt[m1 != null && m2 != null ? 2 : 1]];
  }
  return [l2, ff];
}
function SSF_format(fmt, v4, o2) {
  if (o2 == null)
    o2 = {};
  var sfmt = "";
  switch (typeof fmt) {
    case "string":
      if (fmt == "m/d/yy" && o2.dateNF)
        sfmt = o2.dateNF;
      else
        sfmt = fmt;
      break;
    case "number":
      if (fmt == 14 && o2.dateNF)
        sfmt = o2.dateNF;
      else
        sfmt = (o2.table != null ? o2.table : table_fmt)[fmt];
      if (sfmt == null)
        sfmt = o2.table && o2.table[SSF_default_map[fmt]] || table_fmt[SSF_default_map[fmt]];
      if (sfmt == null)
        sfmt = SSF_default_str[fmt] || "General";
      break;
  }
  if (SSF_isgeneral(sfmt, 0))
    return SSF_general(v4, o2);
  if (v4 instanceof Date)
    v4 = datenum_local(v4, o2.date1904);
  var f2 = choose_fmt(sfmt, v4);
  if (SSF_isgeneral(f2[1]))
    return SSF_general(v4, o2);
  if (v4 === true)
    v4 = "TRUE";
  else if (v4 === false)
    v4 = "FALSE";
  else if (v4 === "" || v4 == null)
    return "";
  return eval_fmt(f2[1], v4, o2, f2[0]);
}
function SSF_load(fmt, idx) {
  if (typeof idx != "number") {
    idx = +idx || -1;
    for (var i2 = 0; i2 < 392; ++i2) {
      if (table_fmt[i2] == void 0) {
        if (idx < 0)
          idx = i2;
        continue;
      }
      if (table_fmt[i2] == fmt) {
        idx = i2;
        break;
      }
    }
    if (idx < 0)
      idx = 391;
  }
  table_fmt[idx] = fmt;
  return idx;
}
function SSF_load_table(tbl) {
  for (var i2 = 0; i2 != 392; ++i2)
    if (tbl[i2] !== void 0)
      SSF_load(tbl[i2], i2);
}
function make_ssf() {
  table_fmt = SSF_init_table();
}
var SSFImplicit = {
  "5": '"$"#,##0_);\\("$"#,##0\\)',
  "6": '"$"#,##0_);[Red]\\("$"#,##0\\)',
  "7": '"$"#,##0.00_);\\("$"#,##0.00\\)',
  "8": '"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',
  "23": "General",
  "24": "General",
  "25": "General",
  "26": "General",
  "27": "m/d/yy",
  "28": "m/d/yy",
  "29": "m/d/yy",
  "30": "m/d/yy",
  "31": "m/d/yy",
  "32": "h:mm:ss",
  "33": "h:mm:ss",
  "34": "h:mm:ss",
  "35": "h:mm:ss",
  "36": "m/d/yy",
  "41": '_(* #,##0_);_(* (#,##0);_(* "-"_);_(@_)',
  "42": '_("$"* #,##0_);_("$"* (#,##0);_("$"* "-"_);_(@_)',
  "43": '_(* #,##0.00_);_(* (#,##0.00);_(* "-"??_);_(@_)',
  "44": '_("$"* #,##0.00_);_("$"* (#,##0.00);_("$"* "-"??_);_(@_)',
  "50": "m/d/yy",
  "51": "m/d/yy",
  "52": "m/d/yy",
  "53": "m/d/yy",
  "54": "m/d/yy",
  "55": "m/d/yy",
  "56": "m/d/yy",
  "57": "m/d/yy",
  "58": "m/d/yy",
  "59": "0",
  "60": "0.00",
  "61": "#,##0",
  "62": "#,##0.00",
  "63": '"$"#,##0_);\\("$"#,##0\\)',
  "64": '"$"#,##0_);[Red]\\("$"#,##0\\)',
  "65": '"$"#,##0.00_);\\("$"#,##0.00\\)',
  "66": '"$"#,##0.00_);[Red]\\("$"#,##0.00\\)',
  "67": "0%",
  "68": "0.00%",
  "69": "# ?/?",
  "70": "# ??/??",
  "71": "m/d/yy",
  "72": "m/d/yy",
  "73": "d-mmm-yy",
  "74": "d-mmm",
  "75": "mmm-yy",
  "76": "h:mm",
  "77": "h:mm:ss",
  "78": "m/d/yy h:mm",
  "79": "mm:ss",
  "80": "[h]:mm:ss",
  "81": "mmss.0"
};
var dateNFregex = /[dD]+|[mM]+|[yYeE]+|[Hh]+|[Ss]+/g;
function dateNF_regex(dateNF) {
  var fmt = typeof dateNF == "number" ? table_fmt[dateNF] : dateNF;
  fmt = fmt.replace(dateNFregex, "(\\d+)");
  return new RegExp("^" + fmt + "$");
}
function dateNF_fix(str, dateNF, match5) {
  var Y2 = -1, m2 = -1, d2 = -1, H4 = -1, M2 = -1, S2 = -1;
  (dateNF.match(dateNFregex) || []).forEach(function(n2, i2) {
    var v4 = parseInt(match5[i2 + 1], 10);
    switch (n2.toLowerCase().charAt(0)) {
      case "y":
        Y2 = v4;
        break;
      case "d":
        d2 = v4;
        break;
      case "h":
        H4 = v4;
        break;
      case "s":
        S2 = v4;
        break;
      case "m":
        if (H4 >= 0)
          M2 = v4;
        else
          m2 = v4;
        break;
    }
  });
  if (S2 >= 0 && M2 == -1 && m2 >= 0) {
    M2 = m2;
    m2 = -1;
  }
  var datestr = ("" + (Y2 >= 0 ? Y2 : (/* @__PURE__ */ new Date()).getFullYear())).slice(-4) + "-" + ("00" + (m2 >= 1 ? m2 : 1)).slice(-2) + "-" + ("00" + (d2 >= 1 ? d2 : 1)).slice(-2);
  if (datestr.length == 7)
    datestr = "0" + datestr;
  if (datestr.length == 8)
    datestr = "20" + datestr;
  var timestr = ("00" + (H4 >= 0 ? H4 : 0)).slice(-2) + ":" + ("00" + (M2 >= 0 ? M2 : 0)).slice(-2) + ":" + ("00" + (S2 >= 0 ? S2 : 0)).slice(-2);
  if (H4 == -1 && M2 == -1 && S2 == -1)
    return datestr;
  if (Y2 == -1 && m2 == -1 && d2 == -1)
    return timestr;
  return datestr + "T" + timestr;
}
var CRC32 = /* @__PURE__ */ function() {
  var CRC322 = {};
  CRC322.version = "1.2.0";
  function signed_crc_table() {
    var c2 = 0, table = new Array(256);
    for (var n2 = 0; n2 != 256; ++n2) {
      c2 = n2;
      c2 = c2 & 1 ? -306674912 ^ c2 >>> 1 : c2 >>> 1;
      c2 = c2 & 1 ? -306674912 ^ c2 >>> 1 : c2 >>> 1;
      c2 = c2 & 1 ? -306674912 ^ c2 >>> 1 : c2 >>> 1;
      c2 = c2 & 1 ? -306674912 ^ c2 >>> 1 : c2 >>> 1;
      c2 = c2 & 1 ? -306674912 ^ c2 >>> 1 : c2 >>> 1;
      c2 = c2 & 1 ? -306674912 ^ c2 >>> 1 : c2 >>> 1;
      c2 = c2 & 1 ? -306674912 ^ c2 >>> 1 : c2 >>> 1;
      c2 = c2 & 1 ? -306674912 ^ c2 >>> 1 : c2 >>> 1;
      table[n2] = c2;
    }
    return typeof Int32Array !== "undefined" ? new Int32Array(table) : table;
  }
  var T0 = signed_crc_table();
  function slice_by_16_tables(T10) {
    var c2 = 0, v4 = 0, n2 = 0, table = typeof Int32Array !== "undefined" ? new Int32Array(4096) : new Array(4096);
    for (n2 = 0; n2 != 256; ++n2)
      table[n2] = T10[n2];
    for (n2 = 0; n2 != 256; ++n2) {
      v4 = T10[n2];
      for (c2 = 256 + n2; c2 < 4096; c2 += 256)
        v4 = table[c2] = v4 >>> 8 ^ T10[v4 & 255];
    }
    var out2 = [];
    for (n2 = 1; n2 != 16; ++n2)
      out2[n2 - 1] = typeof Int32Array !== "undefined" ? table.subarray(n2 * 256, n2 * 256 + 256) : table.slice(n2 * 256, n2 * 256 + 256);
    return out2;
  }
  var TT = slice_by_16_tables(T0);
  var T1 = TT[0], T2 = TT[1], T3 = TT[2], T4 = TT[3], T5 = TT[4];
  var T6 = TT[5], T7 = TT[6], T8 = TT[7], T9 = TT[8], Ta = TT[9];
  var Tb = TT[10], Tc = TT[11], Td = TT[12], Te = TT[13], Tf = TT[14];
  function crc32_bstr(bstr, seed) {
    var C2 = seed ^ -1;
    for (var i2 = 0, L2 = bstr.length; i2 < L2; )
      C2 = C2 >>> 8 ^ T0[(C2 ^ bstr.charCodeAt(i2++)) & 255];
    return ~C2;
  }
  function crc32_buf(B3, seed) {
    var C2 = seed ^ -1, L2 = B3.length - 15, i2 = 0;
    for (; i2 < L2; )
      C2 = Tf[B3[i2++] ^ C2 & 255] ^ Te[B3[i2++] ^ C2 >> 8 & 255] ^ Td[B3[i2++] ^ C2 >> 16 & 255] ^ Tc[B3[i2++] ^ C2 >>> 24] ^ Tb[B3[i2++]] ^ Ta[B3[i2++]] ^ T9[B3[i2++]] ^ T8[B3[i2++]] ^ T7[B3[i2++]] ^ T6[B3[i2++]] ^ T5[B3[i2++]] ^ T4[B3[i2++]] ^ T3[B3[i2++]] ^ T2[B3[i2++]] ^ T1[B3[i2++]] ^ T0[B3[i2++]];
    L2 += 15;
    while (i2 < L2)
      C2 = C2 >>> 8 ^ T0[(C2 ^ B3[i2++]) & 255];
    return ~C2;
  }
  function crc32_str(str, seed) {
    var C2 = seed ^ -1;
    for (var i2 = 0, L2 = str.length, c2 = 0, d2 = 0; i2 < L2; ) {
      c2 = str.charCodeAt(i2++);
      if (c2 < 128) {
        C2 = C2 >>> 8 ^ T0[(C2 ^ c2) & 255];
      } else if (c2 < 2048) {
        C2 = C2 >>> 8 ^ T0[(C2 ^ (192 | c2 >> 6 & 31)) & 255];
        C2 = C2 >>> 8 ^ T0[(C2 ^ (128 | c2 & 63)) & 255];
      } else if (c2 >= 55296 && c2 < 57344) {
        c2 = (c2 & 1023) + 64;
        d2 = str.charCodeAt(i2++) & 1023;
        C2 = C2 >>> 8 ^ T0[(C2 ^ (240 | c2 >> 8 & 7)) & 255];
        C2 = C2 >>> 8 ^ T0[(C2 ^ (128 | c2 >> 2 & 63)) & 255];
        C2 = C2 >>> 8 ^ T0[(C2 ^ (128 | d2 >> 6 & 15 | (c2 & 3) << 4)) & 255];
        C2 = C2 >>> 8 ^ T0[(C2 ^ (128 | d2 & 63)) & 255];
      } else {
        C2 = C2 >>> 8 ^ T0[(C2 ^ (224 | c2 >> 12 & 15)) & 255];
        C2 = C2 >>> 8 ^ T0[(C2 ^ (128 | c2 >> 6 & 63)) & 255];
        C2 = C2 >>> 8 ^ T0[(C2 ^ (128 | c2 & 63)) & 255];
      }
    }
    return ~C2;
  }
  CRC322.table = T0;
  CRC322.bstr = crc32_bstr;
  CRC322.buf = crc32_buf;
  CRC322.str = crc32_str;
  return CRC322;
}();
var CFB = /* @__PURE__ */ function _CFB() {
  var exports = {};
  exports.version = "1.2.1";
  function namecmp(l2, r2) {
    var L2 = l2.split("/"), R2 = r2.split("/");
    for (var i3 = 0, c2 = 0, Z2 = Math.min(L2.length, R2.length); i3 < Z2; ++i3) {
      if (c2 = L2[i3].length - R2[i3].length)
        return c2;
      if (L2[i3] != R2[i3])
        return L2[i3] < R2[i3] ? -1 : 1;
    }
    return L2.length - R2.length;
  }
  function dirname(p2) {
    if (p2.charAt(p2.length - 1) == "/")
      return p2.slice(0, -1).indexOf("/") === -1 ? p2 : dirname(p2.slice(0, -1));
    var c2 = p2.lastIndexOf("/");
    return c2 === -1 ? p2 : p2.slice(0, c2 + 1);
  }
  function filename(p2) {
    if (p2.charAt(p2.length - 1) == "/")
      return filename(p2.slice(0, -1));
    var c2 = p2.lastIndexOf("/");
    return c2 === -1 ? p2 : p2.slice(c2 + 1);
  }
  function write_dos_date(buf, date) {
    if (typeof date === "string")
      date = new Date(date);
    var hms = date.getHours();
    hms = hms << 6 | date.getMinutes();
    hms = hms << 5 | date.getSeconds() >>> 1;
    buf.write_shift(2, hms);
    var ymd = date.getFullYear() - 1980;
    ymd = ymd << 4 | date.getMonth() + 1;
    ymd = ymd << 5 | date.getDate();
    buf.write_shift(2, ymd);
  }
  function parse_dos_date(buf) {
    var hms = buf.read_shift(2) & 65535;
    var ymd = buf.read_shift(2) & 65535;
    var val = /* @__PURE__ */ new Date();
    var d2 = ymd & 31;
    ymd >>>= 5;
    var m2 = ymd & 15;
    ymd >>>= 4;
    val.setMilliseconds(0);
    val.setFullYear(ymd + 1980);
    val.setMonth(m2 - 1);
    val.setDate(d2);
    var S2 = hms & 31;
    hms >>>= 5;
    var M2 = hms & 63;
    hms >>>= 6;
    val.setHours(hms);
    val.setMinutes(M2);
    val.setSeconds(S2 << 1);
    return val;
  }
  function parse_extra_field(blob) {
    prep_blob(blob, 0);
    var o2 = (
      /*::(*/
      {}
    );
    var flags3 = 0;
    while (blob.l <= blob.length - 4) {
      var type2 = blob.read_shift(2);
      var sz = blob.read_shift(2), tgt = blob.l + sz;
      var p2 = {};
      switch (type2) {
        case 21589:
          {
            flags3 = blob.read_shift(1);
            if (flags3 & 1)
              p2.mtime = blob.read_shift(4);
            if (sz > 5) {
              if (flags3 & 2)
                p2.atime = blob.read_shift(4);
              if (flags3 & 4)
                p2.ctime = blob.read_shift(4);
            }
            if (p2.mtime)
              p2.mt = new Date(p2.mtime * 1e3);
          }
          break;
      }
      blob.l = tgt;
      o2[type2] = p2;
    }
    return o2;
  }
  var fs;
  function get_fs() {
    return fs || (fs = {});
  }
  function parse2(file, options) {
    if (file[0] == 80 && file[1] == 75)
      return parse_zip2(file, options);
    if ((file[0] | 32) == 109 && (file[1] | 32) == 105)
      return parse_mad(file, options);
    if (file.length < 512)
      throw new Error("CFB file size " + file.length + " < 512");
    var mver = 3;
    var ssz = 512;
    var nmfs = 0;
    var difat_sec_cnt = 0;
    var dir_start = 0;
    var minifat_start = 0;
    var difat_start = 0;
    var fat_addrs = [];
    var blob = (
      /*::(*/
      file.slice(0, 512)
    );
    prep_blob(blob, 0);
    var mv = check_get_mver(blob);
    mver = mv[0];
    switch (mver) {
      case 3:
        ssz = 512;
        break;
      case 4:
        ssz = 4096;
        break;
      case 0:
        if (mv[1] == 0)
          return parse_zip2(file, options);
      default:
        throw new Error("Major Version: Expected 3 or 4 saw " + mver);
    }
    if (ssz !== 512) {
      blob = /*::(*/
      file.slice(0, ssz);
      prep_blob(
        blob,
        28
        /* blob.l */
      );
    }
    var header = file.slice(0, ssz);
    check_shifts(blob, mver);
    var dir_cnt = blob.read_shift(4, "i");
    if (mver === 3 && dir_cnt !== 0)
      throw new Error("# Directory Sectors: Expected 0 saw " + dir_cnt);
    blob.l += 4;
    dir_start = blob.read_shift(4, "i");
    blob.l += 4;
    blob.chk("00100000", "Mini Stream Cutoff Size: ");
    minifat_start = blob.read_shift(4, "i");
    nmfs = blob.read_shift(4, "i");
    difat_start = blob.read_shift(4, "i");
    difat_sec_cnt = blob.read_shift(4, "i");
    for (var q3 = -1, j2 = 0; j2 < 109; ++j2) {
      q3 = blob.read_shift(4, "i");
      if (q3 < 0)
        break;
      fat_addrs[j2] = q3;
    }
    var sectors = sectorify(file, ssz);
    sleuth_fat(difat_start, difat_sec_cnt, sectors, ssz, fat_addrs);
    var sector_list = make_sector_list(sectors, dir_start, fat_addrs, ssz);
    sector_list[dir_start].name = "!Directory";
    if (nmfs > 0 && minifat_start !== ENDOFCHAIN)
      sector_list[minifat_start].name = "!MiniFAT";
    sector_list[fat_addrs[0]].name = "!FAT";
    sector_list.fat_addrs = fat_addrs;
    sector_list.ssz = ssz;
    var files = {}, Paths = [], FileIndex = [], FullPaths = [];
    read_directory(dir_start, sector_list, sectors, Paths, nmfs, files, FileIndex, minifat_start);
    build_full_paths(FileIndex, FullPaths, Paths);
    Paths.shift();
    var o2 = {
      FileIndex,
      FullPaths
    };
    if (options && options.raw)
      o2.raw = { header, sectors };
    return o2;
  }
  function check_get_mver(blob) {
    if (blob[blob.l] == 80 && blob[blob.l + 1] == 75)
      return [0, 0];
    blob.chk(HEADER_SIGNATURE, "Header Signature: ");
    blob.l += 16;
    var mver = blob.read_shift(2, "u");
    return [blob.read_shift(2, "u"), mver];
  }
  function check_shifts(blob, mver) {
    var shift = 9;
    blob.l += 2;
    switch (shift = blob.read_shift(2)) {
      case 9:
        if (mver != 3)
          throw new Error("Sector Shift: Expected 9 saw " + shift);
        break;
      case 12:
        if (mver != 4)
          throw new Error("Sector Shift: Expected 12 saw " + shift);
        break;
      default:
        throw new Error("Sector Shift: Expected 9 or 12 saw " + shift);
    }
    blob.chk("0600", "Mini Sector Shift: ");
    blob.chk("000000000000", "Reserved: ");
  }
  function sectorify(file, ssz) {
    var nsectors = Math.ceil(file.length / ssz) - 1;
    var sectors = [];
    for (var i3 = 1; i3 < nsectors; ++i3)
      sectors[i3 - 1] = file.slice(i3 * ssz, (i3 + 1) * ssz);
    sectors[nsectors - 1] = file.slice(nsectors * ssz);
    return sectors;
  }
  function build_full_paths(FI, FP, Paths) {
    var i3 = 0, L2 = 0, R2 = 0, C2 = 0, j2 = 0, pl = Paths.length;
    var dad = [], q3 = [];
    for (; i3 < pl; ++i3) {
      dad[i3] = q3[i3] = i3;
      FP[i3] = Paths[i3];
    }
    for (; j2 < q3.length; ++j2) {
      i3 = q3[j2];
      L2 = FI[i3].L;
      R2 = FI[i3].R;
      C2 = FI[i3].C;
      if (dad[i3] === i3) {
        if (L2 !== -1 && dad[L2] !== L2)
          dad[i3] = dad[L2];
        if (R2 !== -1 && dad[R2] !== R2)
          dad[i3] = dad[R2];
      }
      if (C2 !== -1)
        dad[C2] = i3;
      if (L2 !== -1 && i3 != dad[i3]) {
        dad[L2] = dad[i3];
        if (q3.lastIndexOf(L2) < j2)
          q3.push(L2);
      }
      if (R2 !== -1 && i3 != dad[i3]) {
        dad[R2] = dad[i3];
        if (q3.lastIndexOf(R2) < j2)
          q3.push(R2);
      }
    }
    for (i3 = 1; i3 < pl; ++i3)
      if (dad[i3] === i3) {
        if (R2 !== -1 && dad[R2] !== R2)
          dad[i3] = dad[R2];
        else if (L2 !== -1 && dad[L2] !== L2)
          dad[i3] = dad[L2];
      }
    for (i3 = 1; i3 < pl; ++i3) {
      if (FI[i3].type === 0)
        continue;
      j2 = i3;
      if (j2 != dad[j2])
        do {
          j2 = dad[j2];
          FP[i3] = FP[j2] + "/" + FP[i3];
        } while (j2 !== 0 && -1 !== dad[j2] && j2 != dad[j2]);
      dad[i3] = -1;
    }
    FP[0] += "/";
    for (i3 = 1; i3 < pl; ++i3) {
      if (FI[i3].type !== 2)
        FP[i3] += "/";
    }
  }
  function get_mfat_entry(entry, payload, mini) {
    var start2 = entry.start, size = entry.size;
    var o2 = [];
    var idx = start2;
    while (mini && size > 0 && idx >= 0) {
      o2.push(payload.slice(idx * MSSZ, idx * MSSZ + MSSZ));
      size -= MSSZ;
      idx = __readInt32LE(mini, idx * 4);
    }
    if (o2.length === 0)
      return new_buf(0);
    return bconcat(o2).slice(0, entry.size);
  }
  function sleuth_fat(idx, cnt, sectors, ssz, fat_addrs) {
    var q3 = ENDOFCHAIN;
    if (idx === ENDOFCHAIN) {
      if (cnt !== 0)
        throw new Error("DIFAT chain shorter than expected");
    } else if (idx !== -1) {
      var sector = sectors[idx], m2 = (ssz >>> 2) - 1;
      if (!sector)
        return;
      for (var i3 = 0; i3 < m2; ++i3) {
        if ((q3 = __readInt32LE(sector, i3 * 4)) === ENDOFCHAIN)
          break;
        fat_addrs.push(q3);
      }
      sleuth_fat(__readInt32LE(sector, ssz - 4), cnt - 1, sectors, ssz, fat_addrs);
    }
  }
  function get_sector_list(sectors, start2, fat_addrs, ssz, chkd) {
    var buf = [], buf_chain = [];
    if (!chkd)
      chkd = [];
    var modulus = ssz - 1, j2 = 0, jj = 0;
    for (j2 = start2; j2 >= 0; ) {
      chkd[j2] = true;
      buf[buf.length] = j2;
      buf_chain.push(sectors[j2]);
      var addr = fat_addrs[Math.floor(j2 * 4 / ssz)];
      jj = j2 * 4 & modulus;
      if (ssz < 4 + jj)
        throw new Error("FAT boundary crossed: " + j2 + " 4 " + ssz);
      if (!sectors[addr])
        break;
      j2 = __readInt32LE(sectors[addr], jj);
    }
    return { nodes: buf, data: __toBuffer([buf_chain]) };
  }
  function make_sector_list(sectors, dir_start, fat_addrs, ssz) {
    var sl = sectors.length, sector_list = [];
    var chkd = [], buf = [], buf_chain = [];
    var modulus = ssz - 1, i3 = 0, j2 = 0, k2 = 0, jj = 0;
    for (i3 = 0; i3 < sl; ++i3) {
      buf = [];
      k2 = i3 + dir_start;
      if (k2 >= sl)
        k2 -= sl;
      if (chkd[k2])
        continue;
      buf_chain = [];
      var seen2 = [];
      for (j2 = k2; j2 >= 0; ) {
        seen2[j2] = true;
        chkd[j2] = true;
        buf[buf.length] = j2;
        buf_chain.push(sectors[j2]);
        var addr = fat_addrs[Math.floor(j2 * 4 / ssz)];
        jj = j2 * 4 & modulus;
        if (ssz < 4 + jj)
          throw new Error("FAT boundary crossed: " + j2 + " 4 " + ssz);
        if (!sectors[addr])
          break;
        j2 = __readInt32LE(sectors[addr], jj);
        if (seen2[j2])
          break;
      }
      sector_list[k2] = { nodes: buf, data: __toBuffer([buf_chain]) };
    }
    return sector_list;
  }
  function read_directory(dir_start, sector_list, sectors, Paths, nmfs, files, FileIndex, mini) {
    var minifat_store = 0, pl = Paths.length ? 2 : 0;
    var sector = sector_list[dir_start].data;
    var i3 = 0, namelen = 0, name;
    for (; i3 < sector.length; i3 += 128) {
      var blob = (
        /*::(*/
        sector.slice(i3, i3 + 128)
      );
      prep_blob(blob, 64);
      namelen = blob.read_shift(2);
      name = __utf16le(blob, 0, namelen - pl);
      Paths.push(name);
      var o2 = {
        name,
        type: blob.read_shift(1),
        color: blob.read_shift(1),
        L: blob.read_shift(4, "i"),
        R: blob.read_shift(4, "i"),
        C: blob.read_shift(4, "i"),
        clsid: blob.read_shift(16),
        state: blob.read_shift(4, "i"),
        start: 0,
        size: 0
      };
      var ctime = blob.read_shift(2) + blob.read_shift(2) + blob.read_shift(2) + blob.read_shift(2);
      if (ctime !== 0)
        o2.ct = read_date(blob, blob.l - 8);
      var mtime = blob.read_shift(2) + blob.read_shift(2) + blob.read_shift(2) + blob.read_shift(2);
      if (mtime !== 0)
        o2.mt = read_date(blob, blob.l - 8);
      o2.start = blob.read_shift(4, "i");
      o2.size = blob.read_shift(4, "i");
      if (o2.size < 0 && o2.start < 0) {
        o2.size = o2.type = 0;
        o2.start = ENDOFCHAIN;
        o2.name = "";
      }
      if (o2.type === 5) {
        minifat_store = o2.start;
        if (nmfs > 0 && minifat_store !== ENDOFCHAIN)
          sector_list[minifat_store].name = "!StreamData";
      } else if (o2.size >= 4096) {
        o2.storage = "fat";
        if (sector_list[o2.start] === void 0)
          sector_list[o2.start] = get_sector_list(sectors, o2.start, sector_list.fat_addrs, sector_list.ssz);
        sector_list[o2.start].name = o2.name;
        o2.content = sector_list[o2.start].data.slice(0, o2.size);
      } else {
        o2.storage = "minifat";
        if (o2.size < 0)
          o2.size = 0;
        else if (minifat_store !== ENDOFCHAIN && o2.start !== ENDOFCHAIN && sector_list[minifat_store]) {
          o2.content = get_mfat_entry(o2, sector_list[minifat_store].data, (sector_list[mini] || {}).data);
        }
      }
      if (o2.content)
        prep_blob(o2.content, 0);
      files[name] = o2;
      FileIndex.push(o2);
    }
  }
  function read_date(blob, offset2) {
    return new Date((__readUInt32LE(blob, offset2 + 4) / 1e7 * Math.pow(2, 32) + __readUInt32LE(blob, offset2) / 1e7 - 11644473600) * 1e3);
  }
  function read_file(filename2, options) {
    get_fs();
    return parse2(fs.readFileSync(filename2), options);
  }
  function read(blob, options) {
    var type2 = options && options.type;
    if (!type2) {
      if (has_buf && Buffer.isBuffer(blob))
        type2 = "buffer";
    }
    switch (type2 || "base64") {
      case "file":
        return read_file(blob, options);
      case "base64":
        return parse2(s2a(Base64_decode(blob)), options);
      case "binary":
        return parse2(s2a(blob), options);
    }
    return parse2(
      /*::typeof blob == 'string' ? new Buffer(blob, 'utf-8') : */
      blob,
      options
    );
  }
  function init_cfb(cfb, opts) {
    var o2 = opts || {}, root2 = o2.root || "Root Entry";
    if (!cfb.FullPaths)
      cfb.FullPaths = [];
    if (!cfb.FileIndex)
      cfb.FileIndex = [];
    if (cfb.FullPaths.length !== cfb.FileIndex.length)
      throw new Error("inconsistent CFB structure");
    if (cfb.FullPaths.length === 0) {
      cfb.FullPaths[0] = root2 + "/";
      cfb.FileIndex[0] = { name: root2, type: 5 };
    }
    if (o2.CLSID)
      cfb.FileIndex[0].clsid = o2.CLSID;
    seed_cfb(cfb);
  }
  function seed_cfb(cfb) {
    var nm = "Sh33tJ5";
    if (CFB.find(cfb, "/" + nm))
      return;
    var p2 = new_buf(4);
    p2[0] = 55;
    p2[1] = p2[3] = 50;
    p2[2] = 54;
    cfb.FileIndex.push({ name: nm, type: 2, content: p2, size: 4, L: 69, R: 69, C: 69 });
    cfb.FullPaths.push(cfb.FullPaths[0] + nm);
    rebuild_cfb(cfb);
  }
  function rebuild_cfb(cfb, f2) {
    init_cfb(cfb);
    var gc = false, s2 = false;
    for (var i3 = cfb.FullPaths.length - 1; i3 >= 0; --i3) {
      var _file = cfb.FileIndex[i3];
      switch (_file.type) {
        case 0:
          if (s2)
            gc = true;
          else {
            cfb.FileIndex.pop();
            cfb.FullPaths.pop();
          }
          break;
        case 1:
        case 2:
        case 5:
          s2 = true;
          if (isNaN(_file.R * _file.L * _file.C))
            gc = true;
          if (_file.R > -1 && _file.L > -1 && _file.R == _file.L)
            gc = true;
          break;
        default:
          gc = true;
          break;
      }
    }
    if (!gc && !f2)
      return;
    var now2 = new Date(1987, 1, 19), j2 = 0;
    var fullPaths = Object.create ? /* @__PURE__ */ Object.create(null) : {};
    var data2 = [];
    for (i3 = 0; i3 < cfb.FullPaths.length; ++i3) {
      fullPaths[cfb.FullPaths[i3]] = true;
      if (cfb.FileIndex[i3].type === 0)
        continue;
      data2.push([cfb.FullPaths[i3], cfb.FileIndex[i3]]);
    }
    for (i3 = 0; i3 < data2.length; ++i3) {
      var dad = dirname(data2[i3][0]);
      s2 = fullPaths[dad];
      if (!s2) {
        data2.push([dad, {
          name: filename(dad).replace("/", ""),
          type: 1,
          clsid: HEADER_CLSID,
          ct: now2,
          mt: now2,
          content: null
        }]);
        fullPaths[dad] = true;
      }
    }
    data2.sort(function(x2, y2) {
      return namecmp(x2[0], y2[0]);
    });
    cfb.FullPaths = [];
    cfb.FileIndex = [];
    for (i3 = 0; i3 < data2.length; ++i3) {
      cfb.FullPaths[i3] = data2[i3][0];
      cfb.FileIndex[i3] = data2[i3][1];
    }
    for (i3 = 0; i3 < data2.length; ++i3) {
      var elt = cfb.FileIndex[i3];
      var nm = cfb.FullPaths[i3];
      elt.name = filename(nm).replace("/", "");
      elt.L = elt.R = elt.C = -(elt.color = 1);
      elt.size = elt.content ? elt.content.length : 0;
      elt.start = 0;
      elt.clsid = elt.clsid || HEADER_CLSID;
      if (i3 === 0) {
        elt.C = data2.length > 1 ? 1 : -1;
        elt.size = 0;
        elt.type = 5;
      } else if (nm.slice(-1) == "/") {
        for (j2 = i3 + 1; j2 < data2.length; ++j2)
          if (dirname(cfb.FullPaths[j2]) == nm)
            break;
        elt.C = j2 >= data2.length ? -1 : j2;
        for (j2 = i3 + 1; j2 < data2.length; ++j2)
          if (dirname(cfb.FullPaths[j2]) == dirname(nm))
            break;
        elt.R = j2 >= data2.length ? -1 : j2;
        elt.type = 1;
      } else {
        if (dirname(cfb.FullPaths[i3 + 1] || "") == dirname(nm))
          elt.R = i3 + 1;
        elt.type = 2;
      }
    }
  }
  function _write(cfb, options) {
    var _opts = options || {};
    if (_opts.fileType == "mad")
      return write_mad(cfb, _opts);
    rebuild_cfb(cfb);
    switch (_opts.fileType) {
      case "zip":
        return write_zip2(cfb, _opts);
    }
    var L2 = function(cfb2) {
      var mini_size = 0, fat_size = 0;
      for (var i4 = 0; i4 < cfb2.FileIndex.length; ++i4) {
        var file2 = cfb2.FileIndex[i4];
        if (!file2.content)
          continue;
        var flen2 = file2.content.length;
        if (flen2 > 0) {
          if (flen2 < 4096)
            mini_size += flen2 + 63 >> 6;
          else
            fat_size += flen2 + 511 >> 9;
        }
      }
      var dir_cnt = cfb2.FullPaths.length + 3 >> 2;
      var mini_cnt = mini_size + 7 >> 3;
      var mfat_cnt = mini_size + 127 >> 7;
      var fat_base = mini_cnt + fat_size + dir_cnt + mfat_cnt;
      var fat_cnt = fat_base + 127 >> 7;
      var difat_cnt = fat_cnt <= 109 ? 0 : Math.ceil((fat_cnt - 109) / 127);
      while (fat_base + fat_cnt + difat_cnt + 127 >> 7 > fat_cnt)
        difat_cnt = ++fat_cnt <= 109 ? 0 : Math.ceil((fat_cnt - 109) / 127);
      var L3 = [1, difat_cnt, fat_cnt, mfat_cnt, dir_cnt, fat_size, mini_size, 0];
      cfb2.FileIndex[0].size = mini_size << 6;
      L3[7] = (cfb2.FileIndex[0].start = L3[0] + L3[1] + L3[2] + L3[3] + L3[4] + L3[5]) + (L3[6] + 7 >> 3);
      return L3;
    }(cfb);
    var o2 = new_buf(L2[7] << 9);
    var i3 = 0, T2 = 0;
    {
      for (i3 = 0; i3 < 8; ++i3)
        o2.write_shift(1, HEADER_SIG[i3]);
      for (i3 = 0; i3 < 8; ++i3)
        o2.write_shift(2, 0);
      o2.write_shift(2, 62);
      o2.write_shift(2, 3);
      o2.write_shift(2, 65534);
      o2.write_shift(2, 9);
      o2.write_shift(2, 6);
      for (i3 = 0; i3 < 3; ++i3)
        o2.write_shift(2, 0);
      o2.write_shift(4, 0);
      o2.write_shift(4, L2[2]);
      o2.write_shift(4, L2[0] + L2[1] + L2[2] + L2[3] - 1);
      o2.write_shift(4, 0);
      o2.write_shift(4, 1 << 12);
      o2.write_shift(4, L2[3] ? L2[0] + L2[1] + L2[2] - 1 : ENDOFCHAIN);
      o2.write_shift(4, L2[3]);
      o2.write_shift(-4, L2[1] ? L2[0] - 1 : ENDOFCHAIN);
      o2.write_shift(4, L2[1]);
      for (i3 = 0; i3 < 109; ++i3)
        o2.write_shift(-4, i3 < L2[2] ? L2[1] + i3 : -1);
    }
    if (L2[1]) {
      for (T2 = 0; T2 < L2[1]; ++T2) {
        for (; i3 < 236 + T2 * 127; ++i3)
          o2.write_shift(-4, i3 < L2[2] ? L2[1] + i3 : -1);
        o2.write_shift(-4, T2 === L2[1] - 1 ? ENDOFCHAIN : T2 + 1);
      }
    }
    var chainit = function(w2) {
      for (T2 += w2; i3 < T2 - 1; ++i3)
        o2.write_shift(-4, i3 + 1);
      if (w2) {
        ++i3;
        o2.write_shift(-4, ENDOFCHAIN);
      }
    };
    T2 = i3 = 0;
    for (T2 += L2[1]; i3 < T2; ++i3)
      o2.write_shift(-4, consts.DIFSECT);
    for (T2 += L2[2]; i3 < T2; ++i3)
      o2.write_shift(-4, consts.FATSECT);
    chainit(L2[3]);
    chainit(L2[4]);
    var j2 = 0, flen = 0;
    var file = cfb.FileIndex[0];
    for (; j2 < cfb.FileIndex.length; ++j2) {
      file = cfb.FileIndex[j2];
      if (!file.content)
        continue;
      flen = file.content.length;
      if (flen < 4096)
        continue;
      file.start = T2;
      chainit(flen + 511 >> 9);
    }
    chainit(L2[6] + 7 >> 3);
    while (o2.l & 511)
      o2.write_shift(-4, consts.ENDOFCHAIN);
    T2 = i3 = 0;
    for (j2 = 0; j2 < cfb.FileIndex.length; ++j2) {
      file = cfb.FileIndex[j2];
      if (!file.content)
        continue;
      flen = file.content.length;
      if (!flen || flen >= 4096)
        continue;
      file.start = T2;
      chainit(flen + 63 >> 6);
    }
    while (o2.l & 511)
      o2.write_shift(-4, consts.ENDOFCHAIN);
    for (i3 = 0; i3 < L2[4] << 2; ++i3) {
      var nm = cfb.FullPaths[i3];
      if (!nm || nm.length === 0) {
        for (j2 = 0; j2 < 17; ++j2)
          o2.write_shift(4, 0);
        for (j2 = 0; j2 < 3; ++j2)
          o2.write_shift(4, -1);
        for (j2 = 0; j2 < 12; ++j2)
          o2.write_shift(4, 0);
        continue;
      }
      file = cfb.FileIndex[i3];
      if (i3 === 0)
        file.start = file.size ? file.start - 1 : ENDOFCHAIN;
      var _nm = i3 === 0 && _opts.root || file.name;
      flen = 2 * (_nm.length + 1);
      o2.write_shift(64, _nm, "utf16le");
      o2.write_shift(2, flen);
      o2.write_shift(1, file.type);
      o2.write_shift(1, file.color);
      o2.write_shift(-4, file.L);
      o2.write_shift(-4, file.R);
      o2.write_shift(-4, file.C);
      if (!file.clsid)
        for (j2 = 0; j2 < 4; ++j2)
          o2.write_shift(4, 0);
      else
        o2.write_shift(16, file.clsid, "hex");
      o2.write_shift(4, file.state || 0);
      o2.write_shift(4, 0);
      o2.write_shift(4, 0);
      o2.write_shift(4, 0);
      o2.write_shift(4, 0);
      o2.write_shift(4, file.start);
      o2.write_shift(4, file.size);
      o2.write_shift(4, 0);
    }
    for (i3 = 1; i3 < cfb.FileIndex.length; ++i3) {
      file = cfb.FileIndex[i3];
      if (file.size >= 4096) {
        o2.l = file.start + 1 << 9;
        if (has_buf && Buffer.isBuffer(file.content)) {
          file.content.copy(o2, o2.l, 0, file.size);
          o2.l += file.size + 511 & -512;
        } else {
          for (j2 = 0; j2 < file.size; ++j2)
            o2.write_shift(1, file.content[j2]);
          for (; j2 & 511; ++j2)
            o2.write_shift(1, 0);
        }
      }
    }
    for (i3 = 1; i3 < cfb.FileIndex.length; ++i3) {
      file = cfb.FileIndex[i3];
      if (file.size > 0 && file.size < 4096) {
        if (has_buf && Buffer.isBuffer(file.content)) {
          file.content.copy(o2, o2.l, 0, file.size);
          o2.l += file.size + 63 & -64;
        } else {
          for (j2 = 0; j2 < file.size; ++j2)
            o2.write_shift(1, file.content[j2]);
          for (; j2 & 63; ++j2)
            o2.write_shift(1, 0);
        }
      }
    }
    if (has_buf) {
      o2.l = o2.length;
    } else {
      while (o2.l < o2.length)
        o2.write_shift(1, 0);
    }
    return o2;
  }
  function find2(cfb, path) {
    var UCFullPaths = cfb.FullPaths.map(function(x2) {
      return x2.toUpperCase();
    });
    var UCPaths = UCFullPaths.map(function(x2) {
      var y2 = x2.split("/");
      return y2[y2.length - (x2.slice(-1) == "/" ? 2 : 1)];
    });
    var k2 = false;
    if (path.charCodeAt(0) === 47) {
      k2 = true;
      path = UCFullPaths[0].slice(0, -1) + path;
    } else
      k2 = path.indexOf("/") !== -1;
    var UCPath = path.toUpperCase();
    var w2 = k2 === true ? UCFullPaths.indexOf(UCPath) : UCPaths.indexOf(UCPath);
    if (w2 !== -1)
      return cfb.FileIndex[w2];
    var m2 = !UCPath.match(chr1);
    UCPath = UCPath.replace(chr0, "");
    if (m2)
      UCPath = UCPath.replace(chr1, "!");
    for (w2 = 0; w2 < UCFullPaths.length; ++w2) {
      if ((m2 ? UCFullPaths[w2].replace(chr1, "!") : UCFullPaths[w2]).replace(chr0, "") == UCPath)
        return cfb.FileIndex[w2];
      if ((m2 ? UCPaths[w2].replace(chr1, "!") : UCPaths[w2]).replace(chr0, "") == UCPath)
        return cfb.FileIndex[w2];
    }
    return null;
  }
  var MSSZ = 64;
  var ENDOFCHAIN = -2;
  var HEADER_SIGNATURE = "d0cf11e0a1b11ae1";
  var HEADER_SIG = [208, 207, 17, 224, 161, 177, 26, 225];
  var HEADER_CLSID = "00000000000000000000000000000000";
  var consts = {
    /* 2.1 Compund File Sector Numbers and Types */
    MAXREGSECT: -6,
    DIFSECT: -4,
    FATSECT: -3,
    ENDOFCHAIN,
    FREESECT: -1,
    /* 2.2 Compound File Header */
    HEADER_SIGNATURE,
    HEADER_MINOR_VERSION: "3e00",
    MAXREGSID: -6,
    NOSTREAM: -1,
    HEADER_CLSID,
    /* 2.6.1 Compound File Directory Entry */
    EntryTypes: ["unknown", "storage", "stream", "lockbytes", "property", "root"]
  };
  function write_file(cfb, filename2, options) {
    get_fs();
    var o2 = _write(cfb, options);
    fs.writeFileSync(filename2, o2);
  }
  function a2s2(o2) {
    var out2 = new Array(o2.length);
    for (var i3 = 0; i3 < o2.length; ++i3)
      out2[i3] = String.fromCharCode(o2[i3]);
    return out2.join("");
  }
  function write3(cfb, options) {
    var o2 = _write(cfb, options);
    switch (options && options.type || "buffer") {
      case "file":
        get_fs();
        fs.writeFileSync(options.filename, o2);
        return o2;
      case "binary":
        return typeof o2 == "string" ? o2 : a2s2(o2);
      case "base64":
        return Base64_encode(typeof o2 == "string" ? o2 : a2s2(o2));
      case "buffer":
        if (has_buf)
          return Buffer.isBuffer(o2) ? o2 : Buffer_from(o2);
      case "array":
        return typeof o2 == "string" ? s2a(o2) : o2;
    }
    return o2;
  }
  var _zlib;
  function use_zlib(zlib) {
    try {
      var InflateRaw = zlib.InflateRaw;
      var InflRaw = new InflateRaw();
      InflRaw._processChunk(new Uint8Array([3, 0]), InflRaw._finishFlushFlag);
      if (InflRaw.bytesRead)
        _zlib = zlib;
      else
        throw new Error("zlib does not expose bytesRead");
    } catch (e2) {
      console.error("cannot use native zlib: " + (e2.message || e2));
    }
  }
  function _inflateRawSync(payload, usz) {
    if (!_zlib)
      return _inflate(payload, usz);
    var InflateRaw = _zlib.InflateRaw;
    var InflRaw = new InflateRaw();
    var out2 = InflRaw._processChunk(payload.slice(payload.l), InflRaw._finishFlushFlag);
    payload.l += InflRaw.bytesRead;
    return out2;
  }
  function _deflateRawSync(payload) {
    return _zlib ? _zlib.deflateRawSync(payload) : _deflate(payload);
  }
  var CLEN_ORDER = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15];
  var LEN_LN = [3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31, 35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258];
  var DST_LN = [1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193, 257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145, 8193, 12289, 16385, 24577];
  function bit_swap_8(n2) {
    var t3 = (n2 << 1 | n2 << 11) & 139536 | (n2 << 5 | n2 << 15) & 558144;
    return (t3 >> 16 | t3 >> 8 | t3) & 255;
  }
  var use_typed_arrays = typeof Uint8Array !== "undefined";
  var bitswap8 = use_typed_arrays ? new Uint8Array(1 << 8) : [];
  for (var q2 = 0; q2 < 1 << 8; ++q2)
    bitswap8[q2] = bit_swap_8(q2);
  function bit_swap_n(n2, b2) {
    var rev2 = bitswap8[n2 & 255];
    if (b2 <= 8)
      return rev2 >>> 8 - b2;
    rev2 = rev2 << 8 | bitswap8[n2 >> 8 & 255];
    if (b2 <= 16)
      return rev2 >>> 16 - b2;
    rev2 = rev2 << 8 | bitswap8[n2 >> 16 & 255];
    return rev2 >>> 24 - b2;
  }
  function read_bits_2(buf, bl) {
    var w2 = bl & 7, h2 = bl >>> 3;
    return (buf[h2] | (w2 <= 6 ? 0 : buf[h2 + 1] << 8)) >>> w2 & 3;
  }
  function read_bits_3(buf, bl) {
    var w2 = bl & 7, h2 = bl >>> 3;
    return (buf[h2] | (w2 <= 5 ? 0 : buf[h2 + 1] << 8)) >>> w2 & 7;
  }
  function read_bits_4(buf, bl) {
    var w2 = bl & 7, h2 = bl >>> 3;
    return (buf[h2] | (w2 <= 4 ? 0 : buf[h2 + 1] << 8)) >>> w2 & 15;
  }
  function read_bits_5(buf, bl) {
    var w2 = bl & 7, h2 = bl >>> 3;
    return (buf[h2] | (w2 <= 3 ? 0 : buf[h2 + 1] << 8)) >>> w2 & 31;
  }
  function read_bits_7(buf, bl) {
    var w2 = bl & 7, h2 = bl >>> 3;
    return (buf[h2] | (w2 <= 1 ? 0 : buf[h2 + 1] << 8)) >>> w2 & 127;
  }
  function read_bits_n(buf, bl, n2) {
    var w2 = bl & 7, h2 = bl >>> 3, f2 = (1 << n2) - 1;
    var v4 = buf[h2] >>> w2;
    if (n2 < 8 - w2)
      return v4 & f2;
    v4 |= buf[h2 + 1] << 8 - w2;
    if (n2 < 16 - w2)
      return v4 & f2;
    v4 |= buf[h2 + 2] << 16 - w2;
    if (n2 < 24 - w2)
      return v4 & f2;
    v4 |= buf[h2 + 3] << 24 - w2;
    return v4 & f2;
  }
  function write_bits_3(buf, bl, v4) {
    var w2 = bl & 7, h2 = bl >>> 3;
    if (w2 <= 5)
      buf[h2] |= (v4 & 7) << w2;
    else {
      buf[h2] |= v4 << w2 & 255;
      buf[h2 + 1] = (v4 & 7) >> 8 - w2;
    }
    return bl + 3;
  }
  function write_bits_1(buf, bl, v4) {
    var w2 = bl & 7, h2 = bl >>> 3;
    v4 = (v4 & 1) << w2;
    buf[h2] |= v4;
    return bl + 1;
  }
  function write_bits_8(buf, bl, v4) {
    var w2 = bl & 7, h2 = bl >>> 3;
    v4 <<= w2;
    buf[h2] |= v4 & 255;
    v4 >>>= 8;
    buf[h2 + 1] = v4;
    return bl + 8;
  }
  function write_bits_16(buf, bl, v4) {
    var w2 = bl & 7, h2 = bl >>> 3;
    v4 <<= w2;
    buf[h2] |= v4 & 255;
    v4 >>>= 8;
    buf[h2 + 1] = v4 & 255;
    buf[h2 + 2] = v4 >>> 8;
    return bl + 16;
  }
  function realloc(b2, sz) {
    var L2 = b2.length, M2 = 2 * L2 > sz ? 2 * L2 : sz + 5, i3 = 0;
    if (L2 >= sz)
      return b2;
    if (has_buf) {
      var o2 = new_unsafe_buf(M2);
      if (b2.copy)
        b2.copy(o2);
      else
        for (; i3 < b2.length; ++i3)
          o2[i3] = b2[i3];
      return o2;
    } else if (use_typed_arrays) {
      var a2 = new Uint8Array(M2);
      if (a2.set)
        a2.set(b2);
      else
        for (; i3 < L2; ++i3)
          a2[i3] = b2[i3];
      return a2;
    }
    b2.length = M2;
    return b2;
  }
  function zero_fill_array(n2) {
    var o2 = new Array(n2);
    for (var i3 = 0; i3 < n2; ++i3)
      o2[i3] = 0;
    return o2;
  }
  function build_tree2(clens, cmap, MAX) {
    var maxlen = 1, w2 = 0, i3 = 0, j2 = 0, ccode = 0, L2 = clens.length;
    var bl_count = use_typed_arrays ? new Uint16Array(32) : zero_fill_array(32);
    for (i3 = 0; i3 < 32; ++i3)
      bl_count[i3] = 0;
    for (i3 = L2; i3 < MAX; ++i3)
      clens[i3] = 0;
    L2 = clens.length;
    var ctree = use_typed_arrays ? new Uint16Array(L2) : zero_fill_array(L2);
    for (i3 = 0; i3 < L2; ++i3) {
      bl_count[w2 = clens[i3]]++;
      if (maxlen < w2)
        maxlen = w2;
      ctree[i3] = 0;
    }
    bl_count[0] = 0;
    for (i3 = 1; i3 <= maxlen; ++i3)
      bl_count[i3 + 16] = ccode = ccode + bl_count[i3 - 1] << 1;
    for (i3 = 0; i3 < L2; ++i3) {
      ccode = clens[i3];
      if (ccode != 0)
        ctree[i3] = bl_count[ccode + 16]++;
    }
    var cleni = 0;
    for (i3 = 0; i3 < L2; ++i3) {
      cleni = clens[i3];
      if (cleni != 0) {
        ccode = bit_swap_n(ctree[i3], maxlen) >> maxlen - cleni;
        for (j2 = (1 << maxlen + 4 - cleni) - 1; j2 >= 0; --j2)
          cmap[ccode | j2 << cleni] = cleni & 15 | i3 << 4;
      }
    }
    return maxlen;
  }
  var fix_lmap = use_typed_arrays ? new Uint16Array(512) : zero_fill_array(512);
  var fix_dmap = use_typed_arrays ? new Uint16Array(32) : zero_fill_array(32);
  if (!use_typed_arrays) {
    for (var i2 = 0; i2 < 512; ++i2)
      fix_lmap[i2] = 0;
    for (i2 = 0; i2 < 32; ++i2)
      fix_dmap[i2] = 0;
  }
  (function() {
    var dlens = [];
    var i3 = 0;
    for (; i3 < 32; i3++)
      dlens.push(5);
    build_tree2(dlens, fix_dmap, 32);
    var clens = [];
    i3 = 0;
    for (; i3 <= 143; i3++)
      clens.push(8);
    for (; i3 <= 255; i3++)
      clens.push(9);
    for (; i3 <= 279; i3++)
      clens.push(7);
    for (; i3 <= 287; i3++)
      clens.push(8);
    build_tree2(clens, fix_lmap, 288);
  })();
  var _deflateRaw = /* @__PURE__ */ function _deflateRawIIFE() {
    var DST_LN_RE = use_typed_arrays ? new Uint8Array(32768) : [];
    var j2 = 0, k2 = 0;
    for (; j2 < DST_LN.length - 1; ++j2) {
      for (; k2 < DST_LN[j2 + 1]; ++k2)
        DST_LN_RE[k2] = j2;
    }
    for (; k2 < 32768; ++k2)
      DST_LN_RE[k2] = 29;
    var LEN_LN_RE = use_typed_arrays ? new Uint8Array(259) : [];
    for (j2 = 0, k2 = 0; j2 < LEN_LN.length - 1; ++j2) {
      for (; k2 < LEN_LN[j2 + 1]; ++k2)
        LEN_LN_RE[k2] = j2;
    }
    function write_stored(data2, out2) {
      var boff = 0;
      while (boff < data2.length) {
        var L2 = Math.min(65535, data2.length - boff);
        var h2 = boff + L2 == data2.length;
        out2.write_shift(1, +h2);
        out2.write_shift(2, L2);
        out2.write_shift(2, ~L2 & 65535);
        while (L2-- > 0)
          out2[out2.l++] = data2[boff++];
      }
      return out2.l;
    }
    function write_huff_fixed(data2, out2) {
      var bl = 0;
      var boff = 0;
      var addrs = use_typed_arrays ? new Uint16Array(32768) : [];
      while (boff < data2.length) {
        var L2 = (
          /* data.length - boff; */
          Math.min(65535, data2.length - boff)
        );
        if (L2 < 10) {
          bl = write_bits_3(out2, bl, +!!(boff + L2 == data2.length));
          if (bl & 7)
            bl += 8 - (bl & 7);
          out2.l = bl / 8 | 0;
          out2.write_shift(2, L2);
          out2.write_shift(2, ~L2 & 65535);
          while (L2-- > 0)
            out2[out2.l++] = data2[boff++];
          bl = out2.l * 8;
          continue;
        }
        bl = write_bits_3(out2, bl, +!!(boff + L2 == data2.length) + 2);
        var hash = 0;
        while (L2-- > 0) {
          var d2 = data2[boff];
          hash = (hash << 5 ^ d2) & 32767;
          var match5 = -1, mlen = 0;
          if (match5 = addrs[hash]) {
            match5 |= boff & ~32767;
            if (match5 > boff)
              match5 -= 32768;
            if (match5 < boff)
              while (data2[match5 + mlen] == data2[boff + mlen] && mlen < 250)
                ++mlen;
          }
          if (mlen > 2) {
            d2 = LEN_LN_RE[mlen];
            if (d2 <= 22)
              bl = write_bits_8(out2, bl, bitswap8[d2 + 1] >> 1) - 1;
            else {
              write_bits_8(out2, bl, 3);
              bl += 5;
              write_bits_8(out2, bl, bitswap8[d2 - 23] >> 5);
              bl += 3;
            }
            var len_eb = d2 < 8 ? 0 : d2 - 4 >> 2;
            if (len_eb > 0) {
              write_bits_16(out2, bl, mlen - LEN_LN[d2]);
              bl += len_eb;
            }
            d2 = DST_LN_RE[boff - match5];
            bl = write_bits_8(out2, bl, bitswap8[d2] >> 3);
            bl -= 3;
            var dst_eb = d2 < 4 ? 0 : d2 - 2 >> 1;
            if (dst_eb > 0) {
              write_bits_16(out2, bl, boff - match5 - DST_LN[d2]);
              bl += dst_eb;
            }
            for (var q3 = 0; q3 < mlen; ++q3) {
              addrs[hash] = boff & 32767;
              hash = (hash << 5 ^ data2[boff]) & 32767;
              ++boff;
            }
            L2 -= mlen - 1;
          } else {
            if (d2 <= 143)
              d2 = d2 + 48;
            else
              bl = write_bits_1(out2, bl, 1);
            bl = write_bits_8(out2, bl, bitswap8[d2]);
            addrs[hash] = boff & 32767;
            ++boff;
          }
        }
        bl = write_bits_8(out2, bl, 0) - 1;
      }
      out2.l = (bl + 7) / 8 | 0;
      return out2.l;
    }
    return function _deflateRaw2(data2, out2) {
      if (data2.length < 8)
        return write_stored(data2, out2);
      return write_huff_fixed(data2, out2);
    };
  }();
  function _deflate(data2) {
    var buf = new_buf(50 + Math.floor(data2.length * 1.1));
    var off = _deflateRaw(data2, buf);
    return buf.slice(0, off);
  }
  var dyn_lmap = use_typed_arrays ? new Uint16Array(32768) : zero_fill_array(32768);
  var dyn_dmap = use_typed_arrays ? new Uint16Array(32768) : zero_fill_array(32768);
  var dyn_cmap = use_typed_arrays ? new Uint16Array(128) : zero_fill_array(128);
  var dyn_len_1 = 1, dyn_len_2 = 1;
  function dyn(data2, boff) {
    var _HLIT = read_bits_5(data2, boff) + 257;
    boff += 5;
    var _HDIST = read_bits_5(data2, boff) + 1;
    boff += 5;
    var _HCLEN = read_bits_4(data2, boff) + 4;
    boff += 4;
    var w2 = 0;
    var clens = use_typed_arrays ? new Uint8Array(19) : zero_fill_array(19);
    var ctree = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
    var maxlen = 1;
    var bl_count = use_typed_arrays ? new Uint8Array(8) : zero_fill_array(8);
    var next_code = use_typed_arrays ? new Uint8Array(8) : zero_fill_array(8);
    var L2 = clens.length;
    for (var i3 = 0; i3 < _HCLEN; ++i3) {
      clens[CLEN_ORDER[i3]] = w2 = read_bits_3(data2, boff);
      if (maxlen < w2)
        maxlen = w2;
      bl_count[w2]++;
      boff += 3;
    }
    var ccode = 0;
    bl_count[0] = 0;
    for (i3 = 1; i3 <= maxlen; ++i3)
      next_code[i3] = ccode = ccode + bl_count[i3 - 1] << 1;
    for (i3 = 0; i3 < L2; ++i3)
      if ((ccode = clens[i3]) != 0)
        ctree[i3] = next_code[ccode]++;
    var cleni = 0;
    for (i3 = 0; i3 < L2; ++i3) {
      cleni = clens[i3];
      if (cleni != 0) {
        ccode = bitswap8[ctree[i3]] >> 8 - cleni;
        for (var j2 = (1 << 7 - cleni) - 1; j2 >= 0; --j2)
          dyn_cmap[ccode | j2 << cleni] = cleni & 7 | i3 << 3;
      }
    }
    var hcodes = [];
    maxlen = 1;
    for (; hcodes.length < _HLIT + _HDIST; ) {
      ccode = dyn_cmap[read_bits_7(data2, boff)];
      boff += ccode & 7;
      switch (ccode >>>= 3) {
        case 16:
          w2 = 3 + read_bits_2(data2, boff);
          boff += 2;
          ccode = hcodes[hcodes.length - 1];
          while (w2-- > 0)
            hcodes.push(ccode);
          break;
        case 17:
          w2 = 3 + read_bits_3(data2, boff);
          boff += 3;
          while (w2-- > 0)
            hcodes.push(0);
          break;
        case 18:
          w2 = 11 + read_bits_7(data2, boff);
          boff += 7;
          while (w2-- > 0)
            hcodes.push(0);
          break;
        default:
          hcodes.push(ccode);
          if (maxlen < ccode)
            maxlen = ccode;
          break;
      }
    }
    var h1 = hcodes.slice(0, _HLIT), h2 = hcodes.slice(_HLIT);
    for (i3 = _HLIT; i3 < 286; ++i3)
      h1[i3] = 0;
    for (i3 = _HDIST; i3 < 30; ++i3)
      h2[i3] = 0;
    dyn_len_1 = build_tree2(h1, dyn_lmap, 286);
    dyn_len_2 = build_tree2(h2, dyn_dmap, 30);
    return boff;
  }
  function inflate2(data2, usz) {
    if (data2[0] == 3 && !(data2[1] & 3)) {
      return [new_raw_buf(usz), 2];
    }
    var boff = 0;
    var header = 0;
    var outbuf = new_unsafe_buf(usz ? usz : 1 << 18);
    var woff = 0;
    var OL = outbuf.length >>> 0;
    var max_len_1 = 0, max_len_2 = 0;
    while ((header & 1) == 0) {
      header = read_bits_3(data2, boff);
      boff += 3;
      if (header >>> 1 == 0) {
        if (boff & 7)
          boff += 8 - (boff & 7);
        var sz = data2[boff >>> 3] | data2[(boff >>> 3) + 1] << 8;
        boff += 32;
        if (sz > 0) {
          if (!usz && OL < woff + sz) {
            outbuf = realloc(outbuf, woff + sz);
            OL = outbuf.length;
          }
          while (sz-- > 0) {
            outbuf[woff++] = data2[boff >>> 3];
            boff += 8;
          }
        }
        continue;
      } else if (header >> 1 == 1) {
        max_len_1 = 9;
        max_len_2 = 5;
      } else {
        boff = dyn(data2, boff);
        max_len_1 = dyn_len_1;
        max_len_2 = dyn_len_2;
      }
      for (; ; ) {
        if (!usz && OL < woff + 32767) {
          outbuf = realloc(outbuf, woff + 32767);
          OL = outbuf.length;
        }
        var bits2 = read_bits_n(data2, boff, max_len_1);
        var code = header >>> 1 == 1 ? fix_lmap[bits2] : dyn_lmap[bits2];
        boff += code & 15;
        code >>>= 4;
        if ((code >>> 8 & 255) === 0)
          outbuf[woff++] = code;
        else if (code == 256)
          break;
        else {
          code -= 257;
          var len_eb = code < 8 ? 0 : code - 4 >> 2;
          if (len_eb > 5)
            len_eb = 0;
          var tgt = woff + LEN_LN[code];
          if (len_eb > 0) {
            tgt += read_bits_n(data2, boff, len_eb);
            boff += len_eb;
          }
          bits2 = read_bits_n(data2, boff, max_len_2);
          code = header >>> 1 == 1 ? fix_dmap[bits2] : dyn_dmap[bits2];
          boff += code & 15;
          code >>>= 4;
          var dst_eb = code < 4 ? 0 : code - 2 >> 1;
          var dst = DST_LN[code];
          if (dst_eb > 0) {
            dst += read_bits_n(data2, boff, dst_eb);
            boff += dst_eb;
          }
          if (!usz && OL < tgt) {
            outbuf = realloc(outbuf, tgt + 100);
            OL = outbuf.length;
          }
          while (woff < tgt) {
            outbuf[woff] = outbuf[woff - dst];
            ++woff;
          }
        }
      }
    }
    if (usz)
      return [outbuf, boff + 7 >>> 3];
    return [outbuf.slice(0, woff), boff + 7 >>> 3];
  }
  function _inflate(payload, usz) {
    var data2 = payload.slice(payload.l || 0);
    var out2 = inflate2(data2, usz);
    payload.l += out2[1];
    return out2[0];
  }
  function warn_or_throw(wrn, msg) {
    if (wrn) {
      if (typeof console !== "undefined")
        console.error(msg);
    } else
      throw new Error(msg);
  }
  function parse_zip2(file, options) {
    var blob = (
      /*::(*/
      file
    );
    prep_blob(blob, 0);
    var FileIndex = [], FullPaths = [];
    var o2 = {
      FileIndex,
      FullPaths
    };
    init_cfb(o2, { root: options.root });
    var i3 = blob.length - 4;
    while ((blob[i3] != 80 || blob[i3 + 1] != 75 || blob[i3 + 2] != 5 || blob[i3 + 3] != 6) && i3 >= 0)
      --i3;
    blob.l = i3 + 4;
    blob.l += 4;
    var fcnt = blob.read_shift(2);
    blob.l += 6;
    var start_cd = blob.read_shift(4);
    blob.l = start_cd;
    for (i3 = 0; i3 < fcnt; ++i3) {
      blob.l += 20;
      var csz = blob.read_shift(4);
      var usz = blob.read_shift(4);
      var namelen = blob.read_shift(2);
      var efsz = blob.read_shift(2);
      var fcsz = blob.read_shift(2);
      blob.l += 8;
      var offset2 = blob.read_shift(4);
      var EF = parse_extra_field(
        /*::(*/
        blob.slice(blob.l + namelen, blob.l + namelen + efsz)
        /*:: :any)*/
      );
      blob.l += namelen + efsz + fcsz;
      var L2 = blob.l;
      blob.l = offset2 + 4;
      parse_local_file(blob, csz, usz, o2, EF);
      blob.l = L2;
    }
    return o2;
  }
  function parse_local_file(blob, csz, usz, o2, EF) {
    blob.l += 2;
    var flags3 = blob.read_shift(2);
    var meth = blob.read_shift(2);
    var date = parse_dos_date(blob);
    if (flags3 & 8257)
      throw new Error("Unsupported ZIP encryption");
    var crc322 = blob.read_shift(4);
    var _csz = blob.read_shift(4);
    var _usz = blob.read_shift(4);
    var namelen = blob.read_shift(2);
    var efsz = blob.read_shift(2);
    var name = "";
    for (var i3 = 0; i3 < namelen; ++i3)
      name += String.fromCharCode(blob[blob.l++]);
    if (efsz) {
      var ef = parse_extra_field(
        /*::(*/
        blob.slice(blob.l, blob.l + efsz)
        /*:: :any)*/
      );
      if ((ef[21589] || {}).mt)
        date = ef[21589].mt;
      if (((EF || {})[21589] || {}).mt)
        date = EF[21589].mt;
    }
    blob.l += efsz;
    var data2 = blob.slice(blob.l, blob.l + _csz);
    switch (meth) {
      case 8:
        data2 = _inflateRawSync(blob, _usz);
        break;
      case 0:
        break;
      default:
        throw new Error("Unsupported ZIP Compression method " + meth);
    }
    var wrn = false;
    if (flags3 & 8) {
      crc322 = blob.read_shift(4);
      if (crc322 == 134695760) {
        crc322 = blob.read_shift(4);
        wrn = true;
      }
      _csz = blob.read_shift(4);
      _usz = blob.read_shift(4);
    }
    if (_csz != csz)
      warn_or_throw(wrn, "Bad compressed size: " + csz + " != " + _csz);
    if (_usz != usz)
      warn_or_throw(wrn, "Bad uncompressed size: " + usz + " != " + _usz);
    cfb_add(o2, name, data2, { unsafe: true, mt: date });
  }
  function write_zip2(cfb, options) {
    var _opts = options || {};
    var out2 = [], cdirs = [];
    var o2 = new_buf(1);
    var method = _opts.compression ? 8 : 0, flags3 = 0;
    var i3 = 0, j2 = 0;
    var start_cd = 0, fcnt = 0;
    var root2 = cfb.FullPaths[0], fp = root2, fi = cfb.FileIndex[0];
    var crcs = [];
    var sz_cd = 0;
    for (i3 = 1; i3 < cfb.FullPaths.length; ++i3) {
      fp = cfb.FullPaths[i3].slice(root2.length);
      fi = cfb.FileIndex[i3];
      if (!fi.size || !fi.content || fp == "Sh33tJ5")
        continue;
      var start2 = start_cd;
      var namebuf = new_buf(fp.length);
      for (j2 = 0; j2 < fp.length; ++j2)
        namebuf.write_shift(1, fp.charCodeAt(j2) & 127);
      namebuf = namebuf.slice(0, namebuf.l);
      crcs[fcnt] = CRC32.buf(
        /*::((*/
        fi.content,
        0
      );
      var outbuf = fi.content;
      if (method == 8)
        outbuf = _deflateRawSync(outbuf);
      o2 = new_buf(30);
      o2.write_shift(4, 67324752);
      o2.write_shift(2, 20);
      o2.write_shift(2, flags3);
      o2.write_shift(2, method);
      if (fi.mt)
        write_dos_date(o2, fi.mt);
      else
        o2.write_shift(4, 0);
      o2.write_shift(-4, crcs[fcnt]);
      o2.write_shift(4, outbuf.length);
      o2.write_shift(
        4,
        /*::(*/
        fi.content.length
      );
      o2.write_shift(2, namebuf.length);
      o2.write_shift(2, 0);
      start_cd += o2.length;
      out2.push(o2);
      start_cd += namebuf.length;
      out2.push(namebuf);
      start_cd += outbuf.length;
      out2.push(outbuf);
      o2 = new_buf(46);
      o2.write_shift(4, 33639248);
      o2.write_shift(2, 0);
      o2.write_shift(2, 20);
      o2.write_shift(2, flags3);
      o2.write_shift(2, method);
      o2.write_shift(4, 0);
      o2.write_shift(-4, crcs[fcnt]);
      o2.write_shift(4, outbuf.length);
      o2.write_shift(
        4,
        /*::(*/
        fi.content.length
      );
      o2.write_shift(2, namebuf.length);
      o2.write_shift(2, 0);
      o2.write_shift(2, 0);
      o2.write_shift(2, 0);
      o2.write_shift(2, 0);
      o2.write_shift(4, 0);
      o2.write_shift(4, start2);
      sz_cd += o2.l;
      cdirs.push(o2);
      sz_cd += namebuf.length;
      cdirs.push(namebuf);
      ++fcnt;
    }
    o2 = new_buf(22);
    o2.write_shift(4, 101010256);
    o2.write_shift(2, 0);
    o2.write_shift(2, 0);
    o2.write_shift(2, fcnt);
    o2.write_shift(2, fcnt);
    o2.write_shift(4, sz_cd);
    o2.write_shift(4, start_cd);
    o2.write_shift(2, 0);
    return bconcat([bconcat(out2), bconcat(cdirs), o2]);
  }
  var ContentTypeMap = {
    "htm": "text/html",
    "xml": "text/xml",
    "gif": "image/gif",
    "jpg": "image/jpeg",
    "png": "image/png",
    "mso": "application/x-mso",
    "thmx": "application/vnd.ms-officetheme",
    "sh33tj5": "application/octet-stream"
  };
  function get_content_type(fi, fp) {
    if (fi.ctype)
      return fi.ctype;
    var ext = fi.name || "", m2 = ext.match(/\.([^\.]+)$/);
    if (m2 && ContentTypeMap[m2[1]])
      return ContentTypeMap[m2[1]];
    if (fp) {
      m2 = (ext = fp).match(/[\.\\]([^\.\\])+$/);
      if (m2 && ContentTypeMap[m2[1]])
        return ContentTypeMap[m2[1]];
    }
    return "application/octet-stream";
  }
  function write_base64_76(bstr) {
    var data2 = Base64_encode(bstr);
    var o2 = [];
    for (var i3 = 0; i3 < data2.length; i3 += 76)
      o2.push(data2.slice(i3, i3 + 76));
    return o2.join("\r\n") + "\r\n";
  }
  function write_quoted_printable(text2) {
    var encoded = text2.replace(/[\x00-\x08\x0B\x0C\x0E-\x1F\x7E-\xFF=]/g, function(c2) {
      var w2 = c2.charCodeAt(0).toString(16).toUpperCase();
      return "=" + (w2.length == 1 ? "0" + w2 : w2);
    });
    encoded = encoded.replace(/ $/mg, "=20").replace(/\t$/mg, "=09");
    if (encoded.charAt(0) == "\n")
      encoded = "=0D" + encoded.slice(1);
    encoded = encoded.replace(/\r(?!\n)/mg, "=0D").replace(/\n\n/mg, "\n=0A").replace(/([^\r\n])\n/mg, "$1=0A");
    var o2 = [], split2 = encoded.split("\r\n");
    for (var si = 0; si < split2.length; ++si) {
      var str = split2[si];
      if (str.length == 0) {
        o2.push("");
        continue;
      }
      for (var i3 = 0; i3 < str.length; ) {
        var end2 = 76;
        var tmp = str.slice(i3, i3 + end2);
        if (tmp.charAt(end2 - 1) == "=")
          end2--;
        else if (tmp.charAt(end2 - 2) == "=")
          end2 -= 2;
        else if (tmp.charAt(end2 - 3) == "=")
          end2 -= 3;
        tmp = str.slice(i3, i3 + end2);
        i3 += end2;
        if (i3 < str.length)
          tmp += "=";
        o2.push(tmp);
      }
    }
    return o2.join("\r\n");
  }
  function parse_quoted_printable(data2) {
    var o2 = [];
    for (var di = 0; di < data2.length; ++di) {
      var line = data2[di];
      while (di <= data2.length && line.charAt(line.length - 1) == "=")
        line = line.slice(0, line.length - 1) + data2[++di];
      o2.push(line);
    }
    for (var oi = 0; oi < o2.length; ++oi)
      o2[oi] = o2[oi].replace(/[=][0-9A-Fa-f]{2}/g, function($$) {
        return String.fromCharCode(parseInt($$.slice(1), 16));
      });
    return s2a(o2.join("\r\n"));
  }
  function parse_mime(cfb, data2, root2) {
    var fname = "", cte = "", ctype = "", fdata;
    var di = 0;
    for (; di < 10; ++di) {
      var line = data2[di];
      if (!line || line.match(/^\s*$/))
        break;
      var m2 = line.match(/^(.*?):\s*([^\s].*)$/);
      if (m2)
        switch (m2[1].toLowerCase()) {
          case "content-location":
            fname = m2[2].trim();
            break;
          case "content-type":
            ctype = m2[2].trim();
            break;
          case "content-transfer-encoding":
            cte = m2[2].trim();
            break;
        }
    }
    ++di;
    switch (cte.toLowerCase()) {
      case "base64":
        fdata = s2a(Base64_decode(data2.slice(di).join("")));
        break;
      case "quoted-printable":
        fdata = parse_quoted_printable(data2.slice(di));
        break;
      default:
        throw new Error("Unsupported Content-Transfer-Encoding " + cte);
    }
    var file = cfb_add(cfb, fname.slice(root2.length), fdata, { unsafe: true });
    if (ctype)
      file.ctype = ctype;
  }
  function parse_mad(file, options) {
    if (a2s2(file.slice(0, 13)).toLowerCase() != "mime-version:")
      throw new Error("Unsupported MAD header");
    var root2 = options && options.root || "";
    var data2 = (has_buf && Buffer.isBuffer(file) ? file.toString("binary") : a2s2(file)).split("\r\n");
    var di = 0, row = "";
    for (di = 0; di < data2.length; ++di) {
      row = data2[di];
      if (!/^Content-Location:/i.test(row))
        continue;
      row = row.slice(row.indexOf("file"));
      if (!root2)
        root2 = row.slice(0, row.lastIndexOf("/") + 1);
      if (row.slice(0, root2.length) == root2)
        continue;
      while (root2.length > 0) {
        root2 = root2.slice(0, root2.length - 1);
        root2 = root2.slice(0, root2.lastIndexOf("/") + 1);
        if (row.slice(0, root2.length) == root2)
          break;
      }
    }
    var mboundary = (data2[1] || "").match(/boundary="(.*?)"/);
    if (!mboundary)
      throw new Error("MAD cannot find boundary");
    var boundary = "--" + (mboundary[1] || "");
    var FileIndex = [], FullPaths = [];
    var o2 = {
      FileIndex,
      FullPaths
    };
    init_cfb(o2);
    var start_di, fcnt = 0;
    for (di = 0; di < data2.length; ++di) {
      var line = data2[di];
      if (line !== boundary && line !== boundary + "--")
        continue;
      if (fcnt++)
        parse_mime(o2, data2.slice(start_di, di), root2);
      start_di = di;
    }
    return o2;
  }
  function write_mad(cfb, options) {
    var opts = options || {};
    var boundary = opts.boundary || "SheetJS";
    boundary = "------=" + boundary;
    var out2 = [
      "MIME-Version: 1.0",
      'Content-Type: multipart/related; boundary="' + boundary.slice(2) + '"',
      "",
      "",
      ""
    ];
    var root2 = cfb.FullPaths[0], fp = root2, fi = cfb.FileIndex[0];
    for (var i3 = 1; i3 < cfb.FullPaths.length; ++i3) {
      fp = cfb.FullPaths[i3].slice(root2.length);
      fi = cfb.FileIndex[i3];
      if (!fi.size || !fi.content || fp == "Sh33tJ5")
        continue;
      fp = fp.replace(/[\x00-\x08\x0B\x0C\x0E-\x1F\x7E-\xFF]/g, function(c2) {
        return "_x" + c2.charCodeAt(0).toString(16) + "_";
      }).replace(/[\u0080-\uFFFF]/g, function(u2) {
        return "_u" + u2.charCodeAt(0).toString(16) + "_";
      });
      var ca = fi.content;
      var cstr = has_buf && Buffer.isBuffer(ca) ? ca.toString("binary") : a2s2(ca);
      var dispcnt = 0, L2 = Math.min(1024, cstr.length), cc = 0;
      for (var csl = 0; csl <= L2; ++csl)
        if ((cc = cstr.charCodeAt(csl)) >= 32 && cc < 128)
          ++dispcnt;
      var qp = dispcnt >= L2 * 4 / 5;
      out2.push(boundary);
      out2.push("Content-Location: " + (opts.root || "file:///C:/SheetJS/") + fp);
      out2.push("Content-Transfer-Encoding: " + (qp ? "quoted-printable" : "base64"));
      out2.push("Content-Type: " + get_content_type(fi, fp));
      out2.push("");
      out2.push(qp ? write_quoted_printable(cstr) : write_base64_76(cstr));
    }
    out2.push(boundary + "--\r\n");
    return out2.join("\r\n");
  }
  function cfb_new(opts) {
    var o2 = {};
    init_cfb(o2, opts);
    return o2;
  }
  function cfb_add(cfb, name, content2, opts) {
    var unsafe = opts && opts.unsafe;
    if (!unsafe)
      init_cfb(cfb);
    var file = !unsafe && CFB.find(cfb, name);
    if (!file) {
      var fpath = cfb.FullPaths[0];
      if (name.slice(0, fpath.length) == fpath)
        fpath = name;
      else {
        if (fpath.slice(-1) != "/")
          fpath += "/";
        fpath = (fpath + name).replace("//", "/");
      }
      file = { name: filename(name), type: 2 };
      cfb.FileIndex.push(file);
      cfb.FullPaths.push(fpath);
      if (!unsafe)
        CFB.utils.cfb_gc(cfb);
    }
    file.content = content2;
    file.size = content2 ? content2.length : 0;
    if (opts) {
      if (opts.CLSID)
        file.clsid = opts.CLSID;
      if (opts.mt)
        file.mt = opts.mt;
      if (opts.ct)
        file.ct = opts.ct;
    }
    return file;
  }
  function cfb_del(cfb, name) {
    init_cfb(cfb);
    var file = CFB.find(cfb, name);
    if (file) {
      for (var j2 = 0; j2 < cfb.FileIndex.length; ++j2)
        if (cfb.FileIndex[j2] == file) {
          cfb.FileIndex.splice(j2, 1);
          cfb.FullPaths.splice(j2, 1);
          return true;
        }
    }
    return false;
  }
  function cfb_mov(cfb, old_name, new_name) {
    init_cfb(cfb);
    var file = CFB.find(cfb, old_name);
    if (file) {
      for (var j2 = 0; j2 < cfb.FileIndex.length; ++j2)
        if (cfb.FileIndex[j2] == file) {
          cfb.FileIndex[j2].name = filename(new_name);
          cfb.FullPaths[j2] = new_name;
          return true;
        }
    }
    return false;
  }
  function cfb_gc(cfb) {
    rebuild_cfb(cfb, true);
  }
  exports.find = find2;
  exports.read = read;
  exports.parse = parse2;
  exports.write = write3;
  exports.writeFile = write_file;
  exports.utils = {
    cfb_new,
    cfb_add,
    cfb_del,
    cfb_mov,
    cfb_gc,
    ReadShift,
    CheckField,
    prep_blob,
    bconcat,
    use_zlib,
    _deflateRaw: _deflate,
    _inflateRaw: _inflate,
    consts
  };
  return exports;
}();
function blobify(data2) {
  if (typeof data2 === "string")
    return s2ab(data2);
  if (Array.isArray(data2))
    return a2u(data2);
  return data2;
}
function write_dl(fname, payload, enc) {
  if (typeof Deno !== "undefined") {
    if (enc && typeof payload == "string")
      switch (enc) {
        case "utf8":
          payload = new TextEncoder(enc).encode(payload);
          break;
        case "binary":
          payload = s2ab(payload);
          break;
        default:
          throw new Error("Unsupported encoding " + enc);
      }
    return Deno.writeFileSync(fname, payload);
  }
  var data2 = enc == "utf8" ? utf8write(payload) : payload;
  if (typeof IE_SaveFile !== "undefined")
    return IE_SaveFile(data2, fname);
  if (typeof Blob !== "undefined") {
    var blob = new Blob([blobify(data2)], { type: "application/octet-stream" });
    if (typeof navigator !== "undefined" && navigator.msSaveBlob)
      return navigator.msSaveBlob(blob, fname);
    if (typeof saveAs !== "undefined")
      return saveAs(blob, fname);
    if (typeof URL !== "undefined" && typeof document !== "undefined" && document.createElement && URL.createObjectURL) {
      var url = URL.createObjectURL(blob);
      if (typeof chrome === "object" && typeof (chrome.downloads || {}).download == "function") {
        if (URL.revokeObjectURL && typeof setTimeout !== "undefined")
          setTimeout(function() {
            URL.revokeObjectURL(url);
          }, 6e4);
        return chrome.downloads.download({ url, filename: fname, saveAs: true });
      }
      var a2 = document.createElement("a");
      if (a2.download != null) {
        a2.download = fname;
        a2.href = url;
        document.body.appendChild(a2);
        a2.click();
        document.body.removeChild(a2);
        if (URL.revokeObjectURL && typeof setTimeout !== "undefined")
          setTimeout(function() {
            URL.revokeObjectURL(url);
          }, 6e4);
        return url;
      }
    }
  }
  if (typeof $ !== "undefined" && typeof File !== "undefined" && typeof Folder !== "undefined")
    try {
      var out2 = File(fname);
      out2.open("w");
      out2.encoding = "binary";
      if (Array.isArray(payload))
        payload = a2s(payload);
      out2.write(payload);
      out2.close();
      return payload;
    } catch (e2) {
      if (!e2.message || !e2.message.match(/onstruct/))
        throw e2;
    }
  throw new Error("cannot save file " + fname);
}
function read_binary(path) {
  if (typeof Deno !== "undefined")
    return Deno.readFileSync(path);
  if (typeof $ !== "undefined" && typeof File !== "undefined" && typeof Folder !== "undefined")
    try {
      var infile = File(path);
      infile.open("r");
      infile.encoding = "binary";
      var data2 = infile.read();
      infile.close();
      return data2;
    } catch (e2) {
      if (!e2.message || !e2.message.match(/onstruct/))
        throw e2;
    }
  throw new Error("Cannot access file " + path);
}
function keys$4(o2) {
  var ks = Object.keys(o2), o22 = [];
  for (var i2 = 0; i2 < ks.length; ++i2)
    if (Object.prototype.hasOwnProperty.call(o2, ks[i2]))
      o22.push(ks[i2]);
  return o22;
}
function evert_key(obj, key) {
  var o2 = [], K2 = keys$4(obj);
  for (var i2 = 0; i2 !== K2.length; ++i2)
    if (o2[obj[K2[i2]][key]] == null)
      o2[obj[K2[i2]][key]] = K2[i2];
  return o2;
}
function evert(obj) {
  var o2 = [], K2 = keys$4(obj);
  for (var i2 = 0; i2 !== K2.length; ++i2)
    o2[obj[K2[i2]]] = K2[i2];
  return o2;
}
function evert_num(obj) {
  var o2 = [], K2 = keys$4(obj);
  for (var i2 = 0; i2 !== K2.length; ++i2)
    o2[obj[K2[i2]]] = parseInt(K2[i2], 10);
  return o2;
}
function evert_arr(obj) {
  var o2 = [], K2 = keys$4(obj);
  for (var i2 = 0; i2 !== K2.length; ++i2) {
    if (o2[obj[K2[i2]]] == null)
      o2[obj[K2[i2]]] = [];
    o2[obj[K2[i2]]].push(K2[i2]);
  }
  return o2;
}
var basedate = /* @__PURE__ */ new Date(1899, 11, 30, 0, 0, 0);
function datenum(v4, date1904) {
  var epoch = /* @__PURE__ */ v4.getTime();
  if (date1904)
    epoch -= 1462 * 24 * 60 * 60 * 1e3;
  var dnthresh2 = /* @__PURE__ */ basedate.getTime() + (/* @__PURE__ */ v4.getTimezoneOffset() - /* @__PURE__ */ basedate.getTimezoneOffset()) * 6e4;
  return (epoch - dnthresh2) / (24 * 60 * 60 * 1e3);
}
var refdate = /* @__PURE__ */ new Date();
var dnthresh = /* @__PURE__ */ basedate.getTime() + (/* @__PURE__ */ refdate.getTimezoneOffset() - /* @__PURE__ */ basedate.getTimezoneOffset()) * 6e4;
var refoffset = /* @__PURE__ */ refdate.getTimezoneOffset();
function numdate(v4) {
  var out2 = /* @__PURE__ */ new Date();
  out2.setTime(v4 * 24 * 60 * 60 * 1e3 + dnthresh);
  if (out2.getTimezoneOffset() !== refoffset) {
    out2.setTime(out2.getTime() + (out2.getTimezoneOffset() - refoffset) * 6e4);
  }
  return out2;
}
function parse_isodur(s2) {
  var sec = 0, mt2 = 0, time2 = false;
  var m2 = s2.match(/P([0-9\.]+Y)?([0-9\.]+M)?([0-9\.]+D)?T([0-9\.]+H)?([0-9\.]+M)?([0-9\.]+S)?/);
  if (!m2)
    throw new Error("|" + s2 + "| is not an ISO8601 Duration");
  for (var i2 = 1; i2 != m2.length; ++i2) {
    if (!m2[i2])
      continue;
    mt2 = 1;
    if (i2 > 3)
      time2 = true;
    switch (m2[i2].slice(m2[i2].length - 1)) {
      case "Y":
        throw new Error("Unsupported ISO Duration Field: " + m2[i2].slice(m2[i2].length - 1));
      case "D":
        mt2 *= 24;
      case "H":
        mt2 *= 60;
      case "M":
        if (!time2)
          throw new Error("Unsupported ISO Duration Field: M");
        else
          mt2 *= 60;
    }
    sec += mt2 * parseInt(m2[i2], 10);
  }
  return sec;
}
var good_pd_date_1 = /* @__PURE__ */ new Date("2017-02-19T19:06:09.000Z");
var good_pd_date = /* @__PURE__ */ isNaN(/* @__PURE__ */ good_pd_date_1.getFullYear()) ? /* @__PURE__ */ new Date("2/19/17") : good_pd_date_1;
var good_pd = /* @__PURE__ */ good_pd_date.getFullYear() == 2017;
function parseDate$1(str, fixdate) {
  var d2 = new Date(str);
  if (good_pd) {
    if (fixdate > 0)
      d2.setTime(d2.getTime() + d2.getTimezoneOffset() * 60 * 1e3);
    else if (fixdate < 0)
      d2.setTime(d2.getTime() - d2.getTimezoneOffset() * 60 * 1e3);
    return d2;
  }
  if (str instanceof Date)
    return str;
  if (good_pd_date.getFullYear() == 1917 && !isNaN(d2.getFullYear())) {
    var s2 = d2.getFullYear();
    if (str.indexOf("" + s2) > -1)
      return d2;
    d2.setFullYear(d2.getFullYear() + 100);
    return d2;
  }
  var n2 = str.match(/\d+/g) || ["2017", "2", "19", "0", "0", "0"];
  var out2 = new Date(+n2[0], +n2[1] - 1, +n2[2], +n2[3] || 0, +n2[4] || 0, +n2[5] || 0);
  if (str.indexOf("Z") > -1)
    out2 = new Date(out2.getTime() - out2.getTimezoneOffset() * 60 * 1e3);
  return out2;
}
function cc2str(arr, debomit) {
  if (has_buf && Buffer.isBuffer(arr)) {
    if (debomit) {
      if (arr[0] == 255 && arr[1] == 254)
        return utf8write(arr.slice(2).toString("utf16le"));
      if (arr[1] == 254 && arr[2] == 255)
        return utf8write(utf16beread(arr.slice(2).toString("binary")));
    }
    return arr.toString("binary");
  }
  if (typeof TextDecoder !== "undefined")
    try {
      if (debomit) {
        if (arr[0] == 255 && arr[1] == 254)
          return utf8write(new TextDecoder("utf-16le").decode(arr.slice(2)));
        if (arr[0] == 254 && arr[1] == 255)
          return utf8write(new TextDecoder("utf-16be").decode(arr.slice(2)));
      }
      var rev2 = {
        "€": "",
        "‚": "",
        "ƒ": "",
        "„": "",
        "…": "",
        "†": "",
        "‡": "",
        "ˆ": "",
        "‰": "",
        "Š": "",
        "‹": "",
        "Œ": "",
        "Ž": "",
        "‘": "",
        "’": "",
        "“": "",
        "”": "",
        "•": "",
        "–": "",
        "—": "",
        "˜": "",
        "™": "",
        "š": "",
        "›": "",
        "œ": "",
        "ž": "",
        "Ÿ": ""
      };
      if (Array.isArray(arr))
        arr = new Uint8Array(arr);
      return new TextDecoder("latin1").decode(arr).replace(/[€‚ƒ„…†‡ˆ‰Š‹ŒŽ‘’“”•–—˜™š›œžŸ]/g, function(c2) {
        return rev2[c2] || c2;
      });
    } catch (e2) {
    }
  var o2 = [];
  for (var i2 = 0; i2 != arr.length; ++i2)
    o2.push(String.fromCharCode(arr[i2]));
  return o2.join("");
}
function dup(o2) {
  if (typeof JSON != "undefined" && !Array.isArray(o2))
    return JSON.parse(JSON.stringify(o2));
  if (typeof o2 != "object" || o2 == null)
    return o2;
  if (o2 instanceof Date)
    return new Date(o2.getTime());
  var out2 = {};
  for (var k2 in o2)
    if (Object.prototype.hasOwnProperty.call(o2, k2))
      out2[k2] = dup(o2[k2]);
  return out2;
}
function fill(c2, l2) {
  var o2 = "";
  while (o2.length < l2)
    o2 += c2;
  return o2;
}
function fuzzynum(s2) {
  var v4 = Number(s2);
  if (!isNaN(v4))
    return isFinite(v4) ? v4 : NaN;
  if (!/\d/.test(s2))
    return v4;
  var wt2 = 1;
  var ss = s2.replace(/([\d]),([\d])/g, "$1$2").replace(/[$]/g, "").replace(/[%]/g, function() {
    wt2 *= 100;
    return "";
  });
  if (!isNaN(v4 = Number(ss)))
    return v4 / wt2;
  ss = ss.replace(/[(](.*)[)]/, function($$, $1) {
    wt2 = -wt2;
    return $1;
  });
  if (!isNaN(v4 = Number(ss)))
    return v4 / wt2;
  return v4;
}
var lower_months = ["january", "february", "march", "april", "may", "june", "july", "august", "september", "october", "november", "december"];
function fuzzydate(s2) {
  var o2 = new Date(s2), n2 = /* @__PURE__ */ new Date(NaN);
  var y2 = o2.getYear(), m2 = o2.getMonth(), d2 = o2.getDate();
  if (isNaN(d2))
    return n2;
  var lower = s2.toLowerCase();
  if (lower.match(/jan|feb|mar|apr|may|jun|jul|aug|sep|oct|nov|dec/)) {
    lower = lower.replace(/[^a-z]/g, "").replace(/([^a-z]|^)[ap]m?([^a-z]|$)/, "");
    if (lower.length > 3 && lower_months.indexOf(lower) == -1)
      return n2;
  } else if (lower.match(/[a-z]/))
    return n2;
  if (y2 < 0 || y2 > 8099)
    return n2;
  if ((m2 > 0 || d2 > 1) && y2 != 101)
    return o2;
  if (s2.match(/[^-0-9:,\/\\]/))
    return n2;
  return o2;
}
var split_regex = /* @__PURE__ */ function() {
  var safe_split_regex = "abacaba".split(/(:?b)/i).length == 5;
  return function split_regex2(str, re2, def) {
    if (safe_split_regex || typeof re2 == "string")
      return str.split(re2);
    var p2 = str.split(re2), o2 = [p2[0]];
    for (var i2 = 1; i2 < p2.length; ++i2) {
      o2.push(def);
      o2.push(p2[i2]);
    }
    return o2;
  };
}();
function getdatastr(data2) {
  if (!data2)
    return null;
  if (data2.content && data2.type)
    return cc2str(data2.content, true);
  if (data2.data)
    return debom(data2.data);
  if (data2.asNodeBuffer && has_buf)
    return debom(data2.asNodeBuffer().toString("binary"));
  if (data2.asBinary)
    return debom(data2.asBinary());
  if (data2._data && data2._data.getContent)
    return debom(cc2str(Array.prototype.slice.call(data2._data.getContent(), 0)));
  return null;
}
function getdatabin(data2) {
  if (!data2)
    return null;
  if (data2.data)
    return char_codes(data2.data);
  if (data2.asNodeBuffer && has_buf)
    return data2.asNodeBuffer();
  if (data2._data && data2._data.getContent) {
    var o2 = data2._data.getContent();
    if (typeof o2 == "string")
      return char_codes(o2);
    return Array.prototype.slice.call(o2);
  }
  if (data2.content && data2.type)
    return data2.content;
  return null;
}
function getdata(data2) {
  return data2 && data2.name.slice(-4) === ".bin" ? getdatabin(data2) : getdatastr(data2);
}
function safegetzipfile(zip, file) {
  var k2 = zip.FullPaths || keys$4(zip.files);
  var f2 = file.toLowerCase().replace(/[\/]/g, "\\"), g2 = f2.replace(/\\/g, "/");
  for (var i2 = 0; i2 < k2.length; ++i2) {
    var n2 = k2[i2].replace(/^Root Entry[\/]/, "").toLowerCase();
    if (f2 == n2 || g2 == n2)
      return zip.files ? zip.files[k2[i2]] : zip.FileIndex[i2];
  }
  return null;
}
function getzipfile(zip, file) {
  var o2 = safegetzipfile(zip, file);
  if (o2 == null)
    throw new Error("Cannot find file " + file + " in zip");
  return o2;
}
function getzipdata(zip, file, safe) {
  if (!safe)
    return getdata(getzipfile(zip, file));
  if (!file)
    return null;
  try {
    return getzipdata(zip, file);
  } catch (e2) {
    return null;
  }
}
function getzipstr(zip, file, safe) {
  if (!safe)
    return getdatastr(getzipfile(zip, file));
  if (!file)
    return null;
  try {
    return getzipstr(zip, file);
  } catch (e2) {
    return null;
  }
}
function getzipbin(zip, file, safe) {
  if (!safe)
    return getdatabin(getzipfile(zip, file));
  if (!file)
    return null;
  try {
    return getzipbin(zip, file);
  } catch (e2) {
    return null;
  }
}
function zipentries(zip) {
  var k2 = zip.FullPaths || keys$4(zip.files), o2 = [];
  for (var i2 = 0; i2 < k2.length; ++i2)
    if (k2[i2].slice(-1) != "/")
      o2.push(k2[i2].replace(/^Root Entry[\/]/, ""));
  return o2.sort();
}
function zip_add_file(zip, path, content2) {
  if (zip.FullPaths) {
    if (typeof content2 == "string") {
      var res;
      if (has_buf)
        res = Buffer_from(content2);
      else
        res = utf8decode(content2);
      return CFB.utils.cfb_add(zip, path, res);
    }
    CFB.utils.cfb_add(zip, path, content2);
  } else
    zip.file(path, content2);
}
function zip_new() {
  return CFB.utils.cfb_new();
}
function zip_read(d2, o2) {
  switch (o2.type) {
    case "base64":
      return CFB.read(d2, { type: "base64" });
    case "binary":
      return CFB.read(d2, { type: "binary" });
    case "buffer":
    case "array":
      return CFB.read(d2, { type: "buffer" });
  }
  throw new Error("Unrecognized type " + o2.type);
}
function resolve_path(path, base2) {
  if (path.charAt(0) == "/")
    return path.slice(1);
  var result = base2.split("/");
  if (base2.slice(-1) != "/")
    result.pop();
  var target = path.split("/");
  while (target.length !== 0) {
    var step = target.shift();
    if (step === "..")
      result.pop();
    else if (step !== ".")
      result.push(step);
  }
  return result.join("/");
}
var XML_HEADER = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>\r\n';
var attregexg = /([^"\s?>\/]+)\s*=\s*((?:")([^"]*)(?:")|(?:')([^']*)(?:')|([^'">\s]+))/g;
var tagregex1 = /<[\/\?]?[a-zA-Z0-9:_-]+(?:\s+[^"\s?>\/]+\s*=\s*(?:"[^"]*"|'[^']*'|[^'">\s=]+))*\s*[\/\?]?>/mg, tagregex2 = /<[^>]*>/g;
var tagregex = /* @__PURE__ */ XML_HEADER.match(tagregex1) ? tagregex1 : tagregex2;
var nsregex = /<\w*:/, nsregex2 = /<(\/?)\w+:/;
function parsexmltag(tag2, skip_root, skip_LC) {
  var z2 = {};
  var eq = 0, c2 = 0;
  for (; eq !== tag2.length; ++eq)
    if ((c2 = tag2.charCodeAt(eq)) === 32 || c2 === 10 || c2 === 13)
      break;
  if (!skip_root)
    z2[0] = tag2.slice(0, eq);
  if (eq === tag2.length)
    return z2;
  var m2 = tag2.match(attregexg), j2 = 0, v4 = "", i2 = 0, q2 = "", cc = "", quot = 1;
  if (m2)
    for (i2 = 0; i2 != m2.length; ++i2) {
      cc = m2[i2];
      for (c2 = 0; c2 != cc.length; ++c2)
        if (cc.charCodeAt(c2) === 61)
          break;
      q2 = cc.slice(0, c2).trim();
      while (cc.charCodeAt(c2 + 1) == 32)
        ++c2;
      quot = (eq = cc.charCodeAt(c2 + 1)) == 34 || eq == 39 ? 1 : 0;
      v4 = cc.slice(c2 + 1 + quot, cc.length - quot);
      for (j2 = 0; j2 != q2.length; ++j2)
        if (q2.charCodeAt(j2) === 58)
          break;
      if (j2 === q2.length) {
        if (q2.indexOf("_") > 0)
          q2 = q2.slice(0, q2.indexOf("_"));
        z2[q2] = v4;
        if (!skip_LC)
          z2[q2.toLowerCase()] = v4;
      } else {
        var k2 = (j2 === 5 && q2.slice(0, 5) === "xmlns" ? "xmlns" : "") + q2.slice(j2 + 1);
        if (z2[k2] && q2.slice(j2 - 3, j2) == "ext")
          continue;
        z2[k2] = v4;
        if (!skip_LC)
          z2[k2.toLowerCase()] = v4;
      }
    }
  return z2;
}
function strip_ns(x2) {
  return x2.replace(nsregex2, "<$1");
}
var encodings$1 = {
  "&quot;": '"',
  "&apos;": "'",
  "&gt;": ">",
  "&lt;": "<",
  "&amp;": "&"
};
var rencoding = /* @__PURE__ */ evert(encodings$1);
var unescapexml = /* @__PURE__ */ function() {
  var encregex = /&(?:quot|apos|gt|lt|amp|#x?([\da-fA-F]+));/ig, coderegex = /_x([\da-fA-F]{4})_/ig;
  return function unescapexml2(text2) {
    var s2 = text2 + "", i2 = s2.indexOf("<![CDATA[");
    if (i2 == -1)
      return s2.replace(encregex, function($$, $1) {
        return encodings$1[$$] || String.fromCharCode(parseInt($1, $$.indexOf("x") > -1 ? 16 : 10)) || $$;
      }).replace(coderegex, function(m2, c2) {
        return String.fromCharCode(parseInt(c2, 16));
      });
    var j2 = s2.indexOf("]]>");
    return unescapexml2(s2.slice(0, i2)) + s2.slice(i2 + 9, j2) + unescapexml2(s2.slice(j2 + 3));
  };
}();
var decregex = /[&<>'"]/g, charegex = /[\u0000-\u0008\u000b-\u001f]/g;
function escapexml(text2) {
  var s2 = text2 + "";
  return s2.replace(decregex, function(y2) {
    return rencoding[y2];
  }).replace(charegex, function(s3) {
    return "_x" + ("000" + s3.charCodeAt(0).toString(16)).slice(-4) + "_";
  });
}
function escapexmltag(text2) {
  return escapexml(text2).replace(/ /g, "_x0020_");
}
var htmlcharegex = /[\u0000-\u001f]/g;
function escapehtml(text2) {
  var s2 = text2 + "";
  return s2.replace(decregex, function(y2) {
    return rencoding[y2];
  }).replace(/\n/g, "<br/>").replace(htmlcharegex, function(s3) {
    return "&#x" + ("000" + s3.charCodeAt(0).toString(16)).slice(-4) + ";";
  });
}
function escapexlml(text2) {
  var s2 = text2 + "";
  return s2.replace(decregex, function(y2) {
    return rencoding[y2];
  }).replace(htmlcharegex, function(s3) {
    return "&#x" + s3.charCodeAt(0).toString(16).toUpperCase() + ";";
  });
}
var xlml_fixstr = /* @__PURE__ */ function() {
  var entregex = /&#(\d+);/g;
  function entrepl($$, $1) {
    return String.fromCharCode(parseInt($1, 10));
  }
  return function xlml_fixstr2(str) {
    return str.replace(entregex, entrepl);
  };
}();
function xlml_unfixstr(str) {
  return str.replace(/(\r\n|[\r\n])/g, "&#10;");
}
function parsexmlbool(value) {
  switch (value) {
    case 1:
    case true:
    case "1":
    case "true":
    case "TRUE":
      return true;
    default:
      return false;
  }
}
function utf8reada(orig) {
  var out2 = "", i2 = 0, c2 = 0, d2 = 0, e2 = 0, f2 = 0, w2 = 0;
  while (i2 < orig.length) {
    c2 = orig.charCodeAt(i2++);
    if (c2 < 128) {
      out2 += String.fromCharCode(c2);
      continue;
    }
    d2 = orig.charCodeAt(i2++);
    if (c2 > 191 && c2 < 224) {
      f2 = (c2 & 31) << 6;
      f2 |= d2 & 63;
      out2 += String.fromCharCode(f2);
      continue;
    }
    e2 = orig.charCodeAt(i2++);
    if (c2 < 240) {
      out2 += String.fromCharCode((c2 & 15) << 12 | (d2 & 63) << 6 | e2 & 63);
      continue;
    }
    f2 = orig.charCodeAt(i2++);
    w2 = ((c2 & 7) << 18 | (d2 & 63) << 12 | (e2 & 63) << 6 | f2 & 63) - 65536;
    out2 += String.fromCharCode(55296 + (w2 >>> 10 & 1023));
    out2 += String.fromCharCode(56320 + (w2 & 1023));
  }
  return out2;
}
function utf8readb(data2) {
  var out2 = new_raw_buf(2 * data2.length), w2, i2, j2 = 1, k2 = 0, ww = 0, c2;
  for (i2 = 0; i2 < data2.length; i2 += j2) {
    j2 = 1;
    if ((c2 = data2.charCodeAt(i2)) < 128)
      w2 = c2;
    else if (c2 < 224) {
      w2 = (c2 & 31) * 64 + (data2.charCodeAt(i2 + 1) & 63);
      j2 = 2;
    } else if (c2 < 240) {
      w2 = (c2 & 15) * 4096 + (data2.charCodeAt(i2 + 1) & 63) * 64 + (data2.charCodeAt(i2 + 2) & 63);
      j2 = 3;
    } else {
      j2 = 4;
      w2 = (c2 & 7) * 262144 + (data2.charCodeAt(i2 + 1) & 63) * 4096 + (data2.charCodeAt(i2 + 2) & 63) * 64 + (data2.charCodeAt(i2 + 3) & 63);
      w2 -= 65536;
      ww = 55296 + (w2 >>> 10 & 1023);
      w2 = 56320 + (w2 & 1023);
    }
    if (ww !== 0) {
      out2[k2++] = ww & 255;
      out2[k2++] = ww >>> 8;
      ww = 0;
    }
    out2[k2++] = w2 % 256;
    out2[k2++] = w2 >>> 8;
  }
  return out2.slice(0, k2).toString("ucs2");
}
function utf8readc(data2) {
  return Buffer_from(data2, "binary").toString("utf8");
}
var utf8corpus = "foo bar bazâð£";
var utf8read = has_buf && (/* @__PURE__ */ utf8readc(utf8corpus) == /* @__PURE__ */ utf8reada(utf8corpus) && utf8readc || /* @__PURE__ */ utf8readb(utf8corpus) == /* @__PURE__ */ utf8reada(utf8corpus) && utf8readb) || utf8reada;
var utf8write = has_buf ? function(data2) {
  return Buffer_from(data2, "utf8").toString("binary");
} : function(orig) {
  var out2 = [], i2 = 0, c2 = 0, d2 = 0;
  while (i2 < orig.length) {
    c2 = orig.charCodeAt(i2++);
    switch (true) {
      case c2 < 128:
        out2.push(String.fromCharCode(c2));
        break;
      case c2 < 2048:
        out2.push(String.fromCharCode(192 + (c2 >> 6)));
        out2.push(String.fromCharCode(128 + (c2 & 63)));
        break;
      case (c2 >= 55296 && c2 < 57344):
        c2 -= 55296;
        d2 = orig.charCodeAt(i2++) - 56320 + (c2 << 10);
        out2.push(String.fromCharCode(240 + (d2 >> 18 & 7)));
        out2.push(String.fromCharCode(144 + (d2 >> 12 & 63)));
        out2.push(String.fromCharCode(128 + (d2 >> 6 & 63)));
        out2.push(String.fromCharCode(128 + (d2 & 63)));
        break;
      default:
        out2.push(String.fromCharCode(224 + (c2 >> 12)));
        out2.push(String.fromCharCode(128 + (c2 >> 6 & 63)));
        out2.push(String.fromCharCode(128 + (c2 & 63)));
    }
  }
  return out2.join("");
};
var matchtag = /* @__PURE__ */ function() {
  var mtcache = {};
  return function matchtag2(f2, g2) {
    var t3 = f2 + "|" + (g2 || "");
    if (mtcache[t3])
      return mtcache[t3];
    return mtcache[t3] = new RegExp("<(?:\\w+:)?" + f2 + '(?: xml:space="preserve")?(?:[^>]*)>([\\s\\S]*?)</(?:\\w+:)?' + f2 + ">", g2 || "");
  };
}();
var htmldecode = /* @__PURE__ */ function() {
  var entities = [
    ["nbsp", " "],
    ["middot", "·"],
    ["quot", '"'],
    ["apos", "'"],
    ["gt", ">"],
    ["lt", "<"],
    ["amp", "&"]
  ].map(function(x2) {
    return [new RegExp("&" + x2[0] + ";", "ig"), x2[1]];
  });
  return function htmldecode2(str) {
    var o2 = str.replace(/^[\t\n\r ]+/, "").replace(/[\t\n\r ]+$/, "").replace(/>\s+/g, ">").replace(/\s+</g, "<").replace(/[\t\n\r ]+/g, " ").replace(/<\s*[bB][rR]\s*\/?>/g, "\n").replace(/<[^>]*>/g, "");
    for (var i2 = 0; i2 < entities.length; ++i2)
      o2 = o2.replace(entities[i2][0], entities[i2][1]);
    return o2;
  };
}();
var vtregex = /* @__PURE__ */ function() {
  var vt_cache = {};
  return function vt_regex(bt2) {
    if (vt_cache[bt2] !== void 0)
      return vt_cache[bt2];
    return vt_cache[bt2] = new RegExp("<(?:vt:)?" + bt2 + ">([\\s\\S]*?)</(?:vt:)?" + bt2 + ">", "g");
  };
}();
var vtvregex = /<\/?(?:vt:)?variant>/g, vtmregex = /<(?:vt:)([^>]*)>([\s\S]*)</;
function parseVector(data2, opts) {
  var h2 = parsexmltag(data2);
  var matches = data2.match(vtregex(h2.baseType)) || [];
  var res = [];
  if (matches.length != h2.size) {
    if (opts.WTF)
      throw new Error("unexpected vector length " + matches.length + " != " + h2.size);
    return res;
  }
  matches.forEach(function(x2) {
    var v4 = x2.replace(vtvregex, "").match(vtmregex);
    if (v4)
      res.push({ v: utf8read(v4[2]), t: v4[1] });
  });
  return res;
}
var wtregex = /(^\s|\s$|\n)/;
function writetag(f2, g2) {
  return "<" + f2 + (g2.match(wtregex) ? ' xml:space="preserve"' : "") + ">" + g2 + "</" + f2 + ">";
}
function wxt_helper(h2) {
  return keys$4(h2).map(function(k2) {
    return " " + k2 + '="' + h2[k2] + '"';
  }).join("");
}
function writextag(f2, g2, h2) {
  return "<" + f2 + (h2 != null ? wxt_helper(h2) : "") + (g2 != null ? (g2.match(wtregex) ? ' xml:space="preserve"' : "") + ">" + g2 + "</" + f2 : "/") + ">";
}
function write_w3cdtf(d2, t3) {
  try {
    return d2.toISOString().replace(/\.\d*/, "");
  } catch (e2) {
    if (t3)
      throw e2;
  }
  return "";
}
function write_vt(s2, xlsx) {
  switch (typeof s2) {
    case "string":
      var o2 = writextag("vt:lpwstr", escapexml(s2));
      if (xlsx)
        o2 = o2.replace(/&quot;/g, "_x0022_");
      return o2;
    case "number":
      return writextag((s2 | 0) == s2 ? "vt:i4" : "vt:r8", escapexml(String(s2)));
    case "boolean":
      return writextag("vt:bool", s2 ? "true" : "false");
  }
  if (s2 instanceof Date)
    return writextag("vt:filetime", write_w3cdtf(s2));
  throw new Error("Unable to serialize " + s2);
}
function xlml_normalize(d2) {
  if (has_buf && /*::typeof Buffer !== "undefined" && d != null && d instanceof Buffer &&*/
  Buffer.isBuffer(d2))
    return d2.toString("utf8");
  if (typeof d2 === "string")
    return d2;
  if (typeof Uint8Array !== "undefined" && d2 instanceof Uint8Array)
    return utf8read(a2s(ab2a(d2)));
  throw new Error("Bad input format: expected Buffer or string");
}
var xlmlregex = /<(\/?)([^\s?><!\/:]*:|)([^\s?<>:\/]+)(?:[\s?:\/][^>]*)?>/mg;
var XMLNS$1 = {
  CORE_PROPS: "http://schemas.openxmlformats.org/package/2006/metadata/core-properties",
  CUST_PROPS: "http://schemas.openxmlformats.org/officeDocument/2006/custom-properties",
  EXT_PROPS: "http://schemas.openxmlformats.org/officeDocument/2006/extended-properties",
  CT: "http://schemas.openxmlformats.org/package/2006/content-types",
  RELS: "http://schemas.openxmlformats.org/package/2006/relationships",
  TCMNT: "http://schemas.microsoft.com/office/spreadsheetml/2018/threadedcomments",
  "dc": "http://purl.org/dc/elements/1.1/",
  "dcterms": "http://purl.org/dc/terms/",
  "dcmitype": "http://purl.org/dc/dcmitype/",
  "mx": "http://schemas.microsoft.com/office/mac/excel/2008/main",
  "r": "http://schemas.openxmlformats.org/officeDocument/2006/relationships",
  "sjs": "http://schemas.openxmlformats.org/package/2006/sheetjs/core-properties",
  "vt": "http://schemas.openxmlformats.org/officeDocument/2006/docPropsVTypes",
  "xsi": "http://www.w3.org/2001/XMLSchema-instance",
  "xsd": "http://www.w3.org/2001/XMLSchema"
};
var XMLNS_main = [
  "http://schemas.openxmlformats.org/spreadsheetml/2006/main",
  "http://purl.oclc.org/ooxml/spreadsheetml/main",
  "http://schemas.microsoft.com/office/excel/2006/main",
  "http://schemas.microsoft.com/office/excel/2006/2"
];
var XLMLNS = {
  "o": "urn:schemas-microsoft-com:office:office",
  "x": "urn:schemas-microsoft-com:office:excel",
  "ss": "urn:schemas-microsoft-com:office:spreadsheet",
  "dt": "uuid:C2F41010-65B3-11d1-A29F-00AA00C14882",
  "mv": "http://macVmlSchemaUri",
  "v": "urn:schemas-microsoft-com:vml",
  "html": "http://www.w3.org/TR/REC-html40"
};
function read_double_le(b2, idx) {
  var s2 = 1 - 2 * (b2[idx + 7] >>> 7);
  var e2 = ((b2[idx + 7] & 127) << 4) + (b2[idx + 6] >>> 4 & 15);
  var m2 = b2[idx + 6] & 15;
  for (var i2 = 5; i2 >= 0; --i2)
    m2 = m2 * 256 + b2[idx + i2];
  if (e2 == 2047)
    return m2 == 0 ? s2 * Infinity : NaN;
  if (e2 == 0)
    e2 = -1022;
  else {
    e2 -= 1023;
    m2 += Math.pow(2, 52);
  }
  return s2 * Math.pow(2, e2 - 52) * m2;
}
function write_double_le(b2, v4, idx) {
  var bs = (v4 < 0 || 1 / v4 == -Infinity ? 1 : 0) << 7, e2 = 0, m2 = 0;
  var av = bs ? -v4 : v4;
  if (!isFinite(av)) {
    e2 = 2047;
    m2 = isNaN(v4) ? 26985 : 0;
  } else if (av == 0)
    e2 = m2 = 0;
  else {
    e2 = Math.floor(Math.log(av) / Math.LN2);
    m2 = av * Math.pow(2, 52 - e2);
    if (e2 <= -1023 && (!isFinite(m2) || m2 < Math.pow(2, 52))) {
      e2 = -1022;
    } else {
      m2 -= Math.pow(2, 52);
      e2 += 1023;
    }
  }
  for (var i2 = 0; i2 <= 5; ++i2, m2 /= 256)
    b2[idx + i2] = m2 & 255;
  b2[idx + 6] = (e2 & 15) << 4 | m2 & 15;
  b2[idx + 7] = e2 >> 4 | bs;
}
var ___toBuffer = function(bufs) {
  var x2 = [], w2 = 10240;
  for (var i2 = 0; i2 < bufs[0].length; ++i2)
    if (bufs[0][i2])
      for (var j2 = 0, L2 = bufs[0][i2].length; j2 < L2; j2 += w2)
        x2.push.apply(x2, bufs[0][i2].slice(j2, j2 + w2));
  return x2;
};
var __toBuffer = has_buf ? function(bufs) {
  return bufs[0].length > 0 && Buffer.isBuffer(bufs[0][0]) ? Buffer.concat(bufs[0].map(function(x2) {
    return Buffer.isBuffer(x2) ? x2 : Buffer_from(x2);
  })) : ___toBuffer(bufs);
} : ___toBuffer;
var ___utf16le = function(b2, s2, e2) {
  var ss = [];
  for (var i2 = s2; i2 < e2; i2 += 2)
    ss.push(String.fromCharCode(__readUInt16LE(b2, i2)));
  return ss.join("").replace(chr0, "");
};
var __utf16le = has_buf ? function(b2, s2, e2) {
  if (!Buffer.isBuffer(b2))
    return ___utf16le(b2, s2, e2);
  return b2.toString("utf16le", s2, e2).replace(chr0, "");
} : ___utf16le;
var ___hexlify = function(b2, s2, l2) {
  var ss = [];
  for (var i2 = s2; i2 < s2 + l2; ++i2)
    ss.push(("0" + b2[i2].toString(16)).slice(-2));
  return ss.join("");
};
var __hexlify = has_buf ? function(b2, s2, l2) {
  return Buffer.isBuffer(b2) ? b2.toString("hex", s2, s2 + l2) : ___hexlify(b2, s2, l2);
} : ___hexlify;
var ___utf8 = function(b2, s2, e2) {
  var ss = [];
  for (var i2 = s2; i2 < e2; i2++)
    ss.push(String.fromCharCode(__readUInt8(b2, i2)));
  return ss.join("");
};
var __utf8 = has_buf ? function utf8_b(b2, s2, e2) {
  return Buffer.isBuffer(b2) ? b2.toString("utf8", s2, e2) : ___utf8(b2, s2, e2);
} : ___utf8;
var ___lpstr = function(b2, i2) {
  var len2 = __readUInt32LE(b2, i2);
  return len2 > 0 ? __utf8(b2, i2 + 4, i2 + 4 + len2 - 1) : "";
};
var __lpstr = ___lpstr;
var ___cpstr = function(b2, i2) {
  var len2 = __readUInt32LE(b2, i2);
  return len2 > 0 ? __utf8(b2, i2 + 4, i2 + 4 + len2 - 1) : "";
};
var __cpstr = ___cpstr;
var ___lpwstr = function(b2, i2) {
  var len2 = 2 * __readUInt32LE(b2, i2);
  return len2 > 0 ? __utf8(b2, i2 + 4, i2 + 4 + len2 - 1) : "";
};
var __lpwstr = ___lpwstr;
var ___lpp4 = function lpp4_(b2, i2) {
  var len2 = __readUInt32LE(b2, i2);
  return len2 > 0 ? __utf16le(b2, i2 + 4, i2 + 4 + len2) : "";
};
var __lpp4 = ___lpp4;
var ___8lpp4 = function(b2, i2) {
  var len2 = __readUInt32LE(b2, i2);
  return len2 > 0 ? __utf8(b2, i2 + 4, i2 + 4 + len2) : "";
};
var __8lpp4 = ___8lpp4;
var ___double = function(b2, idx) {
  return read_double_le(b2, idx);
};
var __double = ___double;
var is_buf = function is_buf_a(a2) {
  return Array.isArray(a2) || typeof Uint8Array !== "undefined" && a2 instanceof Uint8Array;
};
if (has_buf) {
  __lpstr = function lpstr_b(b2, i2) {
    if (!Buffer.isBuffer(b2))
      return ___lpstr(b2, i2);
    var len2 = b2.readUInt32LE(i2);
    return len2 > 0 ? b2.toString("utf8", i2 + 4, i2 + 4 + len2 - 1) : "";
  };
  __cpstr = function cpstr_b(b2, i2) {
    if (!Buffer.isBuffer(b2))
      return ___cpstr(b2, i2);
    var len2 = b2.readUInt32LE(i2);
    return len2 > 0 ? b2.toString("utf8", i2 + 4, i2 + 4 + len2 - 1) : "";
  };
  __lpwstr = function lpwstr_b(b2, i2) {
    if (!Buffer.isBuffer(b2))
      return ___lpwstr(b2, i2);
    var len2 = 2 * b2.readUInt32LE(i2);
    return b2.toString("utf16le", i2 + 4, i2 + 4 + len2 - 1);
  };
  __lpp4 = function lpp4_b(b2, i2) {
    if (!Buffer.isBuffer(b2))
      return ___lpp4(b2, i2);
    var len2 = b2.readUInt32LE(i2);
    return b2.toString("utf16le", i2 + 4, i2 + 4 + len2);
  };
  __8lpp4 = function lpp4_8b(b2, i2) {
    if (!Buffer.isBuffer(b2))
      return ___8lpp4(b2, i2);
    var len2 = b2.readUInt32LE(i2);
    return b2.toString("utf8", i2 + 4, i2 + 4 + len2);
  };
  __double = function double_(b2, i2) {
    if (Buffer.isBuffer(b2))
      return b2.readDoubleLE(i2);
    return ___double(b2, i2);
  };
  is_buf = function is_buf_b(a2) {
    return Buffer.isBuffer(a2) || Array.isArray(a2) || typeof Uint8Array !== "undefined" && a2 instanceof Uint8Array;
  };
}
var __readUInt8 = function(b2, idx) {
  return b2[idx];
};
var __readUInt16LE = function(b2, idx) {
  return b2[idx + 1] * (1 << 8) + b2[idx];
};
var __readInt16LE = function(b2, idx) {
  var u2 = b2[idx + 1] * (1 << 8) + b2[idx];
  return u2 < 32768 ? u2 : (65535 - u2 + 1) * -1;
};
var __readUInt32LE = function(b2, idx) {
  return b2[idx + 3] * (1 << 24) + (b2[idx + 2] << 16) + (b2[idx + 1] << 8) + b2[idx];
};
var __readInt32LE = function(b2, idx) {
  return b2[idx + 3] << 24 | b2[idx + 2] << 16 | b2[idx + 1] << 8 | b2[idx];
};
var __readInt32BE = function(b2, idx) {
  return b2[idx] << 24 | b2[idx + 1] << 16 | b2[idx + 2] << 8 | b2[idx + 3];
};
function ReadShift(size, t3) {
  var o2 = "", oI, oR, oo = [], w2, vv2, i2, loc;
  switch (t3) {
    case "dbcs":
      loc = this.l;
      if (has_buf && Buffer.isBuffer(this))
        o2 = this.slice(this.l, this.l + 2 * size).toString("utf16le");
      else
        for (i2 = 0; i2 < size; ++i2) {
          o2 += String.fromCharCode(__readUInt16LE(this, loc));
          loc += 2;
        }
      size *= 2;
      break;
    case "utf8":
      o2 = __utf8(this, this.l, this.l + size);
      break;
    case "utf16le":
      size *= 2;
      o2 = __utf16le(this, this.l, this.l + size);
      break;
    case "wstr":
      return ReadShift.call(this, size, "dbcs");
    case "lpstr-ansi":
      o2 = __lpstr(this, this.l);
      size = 4 + __readUInt32LE(this, this.l);
      break;
    case "lpstr-cp":
      o2 = __cpstr(this, this.l);
      size = 4 + __readUInt32LE(this, this.l);
      break;
    case "lpwstr":
      o2 = __lpwstr(this, this.l);
      size = 4 + 2 * __readUInt32LE(this, this.l);
      break;
    case "lpp4":
      size = 4 + __readUInt32LE(this, this.l);
      o2 = __lpp4(this, this.l);
      if (size & 2)
        size += 2;
      break;
    case "8lpp4":
      size = 4 + __readUInt32LE(this, this.l);
      o2 = __8lpp4(this, this.l);
      if (size & 3)
        size += 4 - (size & 3);
      break;
    case "cstr":
      size = 0;
      o2 = "";
      while ((w2 = __readUInt8(this, this.l + size++)) !== 0)
        oo.push(_getchar(w2));
      o2 = oo.join("");
      break;
    case "_wstr":
      size = 0;
      o2 = "";
      while ((w2 = __readUInt16LE(this, this.l + size)) !== 0) {
        oo.push(_getchar(w2));
        size += 2;
      }
      size += 2;
      o2 = oo.join("");
      break;
    case "dbcs-cont":
      o2 = "";
      loc = this.l;
      for (i2 = 0; i2 < size; ++i2) {
        if (this.lens && this.lens.indexOf(loc) !== -1) {
          w2 = __readUInt8(this, loc);
          this.l = loc + 1;
          vv2 = ReadShift.call(this, size - i2, w2 ? "dbcs-cont" : "sbcs-cont");
          return oo.join("") + vv2;
        }
        oo.push(_getchar(__readUInt16LE(this, loc)));
        loc += 2;
      }
      o2 = oo.join("");
      size *= 2;
      break;
    case "cpstr":
    case "sbcs-cont":
      o2 = "";
      loc = this.l;
      for (i2 = 0; i2 != size; ++i2) {
        if (this.lens && this.lens.indexOf(loc) !== -1) {
          w2 = __readUInt8(this, loc);
          this.l = loc + 1;
          vv2 = ReadShift.call(this, size - i2, w2 ? "dbcs-cont" : "sbcs-cont");
          return oo.join("") + vv2;
        }
        oo.push(_getchar(__readUInt8(this, loc)));
        loc += 1;
      }
      o2 = oo.join("");
      break;
    default:
      switch (size) {
        case 1:
          oI = __readUInt8(this, this.l);
          this.l++;
          return oI;
        case 2:
          oI = (t3 === "i" ? __readInt16LE : __readUInt16LE)(this, this.l);
          this.l += 2;
          return oI;
        case 4:
        case -4:
          if (t3 === "i" || (this[this.l + 3] & 128) === 0) {
            oI = (size > 0 ? __readInt32LE : __readInt32BE)(this, this.l);
            this.l += 4;
            return oI;
          } else {
            oR = __readUInt32LE(this, this.l);
            this.l += 4;
          }
          return oR;
        case 8:
        case -8:
          if (t3 === "f") {
            if (size == 8)
              oR = __double(this, this.l);
            else
              oR = __double([this[this.l + 7], this[this.l + 6], this[this.l + 5], this[this.l + 4], this[this.l + 3], this[this.l + 2], this[this.l + 1], this[this.l + 0]], 0);
            this.l += 8;
            return oR;
          } else
            size = 8;
        case 16:
          o2 = __hexlify(this, this.l, size);
          break;
      }
  }
  this.l += size;
  return o2;
}
var __writeUInt32LE = function(b2, val, idx) {
  b2[idx] = val & 255;
  b2[idx + 1] = val >>> 8 & 255;
  b2[idx + 2] = val >>> 16 & 255;
  b2[idx + 3] = val >>> 24 & 255;
};
var __writeInt32LE = function(b2, val, idx) {
  b2[idx] = val & 255;
  b2[idx + 1] = val >> 8 & 255;
  b2[idx + 2] = val >> 16 & 255;
  b2[idx + 3] = val >> 24 & 255;
};
var __writeUInt16LE = function(b2, val, idx) {
  b2[idx] = val & 255;
  b2[idx + 1] = val >>> 8 & 255;
};
function WriteShift(t3, val, f2) {
  var size = 0, i2 = 0;
  if (f2 === "dbcs") {
    for (i2 = 0; i2 != val.length; ++i2)
      __writeUInt16LE(this, val.charCodeAt(i2), this.l + 2 * i2);
    size = 2 * val.length;
  } else if (f2 === "sbcs") {
    {
      val = val.replace(/[^\x00-\x7F]/g, "_");
      for (i2 = 0; i2 != val.length; ++i2)
        this[this.l + i2] = val.charCodeAt(i2) & 255;
    }
    size = val.length;
  } else if (f2 === "hex") {
    for (; i2 < t3; ++i2) {
      this[this.l++] = parseInt(val.slice(2 * i2, 2 * i2 + 2), 16) || 0;
    }
    return this;
  } else if (f2 === "utf16le") {
    var end2 = Math.min(this.l + t3, this.length);
    for (i2 = 0; i2 < Math.min(val.length, t3); ++i2) {
      var cc = val.charCodeAt(i2);
      this[this.l++] = cc & 255;
      this[this.l++] = cc >> 8;
    }
    while (this.l < end2)
      this[this.l++] = 0;
    return this;
  } else
    switch (t3) {
      case 1:
        size = 1;
        this[this.l] = val & 255;
        break;
      case 2:
        size = 2;
        this[this.l] = val & 255;
        val >>>= 8;
        this[this.l + 1] = val & 255;
        break;
      case 3:
        size = 3;
        this[this.l] = val & 255;
        val >>>= 8;
        this[this.l + 1] = val & 255;
        val >>>= 8;
        this[this.l + 2] = val & 255;
        break;
      case 4:
        size = 4;
        __writeUInt32LE(this, val, this.l);
        break;
      case 8:
        size = 8;
        if (f2 === "f") {
          write_double_le(this, val, this.l);
          break;
        }
      case 16:
        break;
      case -4:
        size = 4;
        __writeInt32LE(this, val, this.l);
        break;
    }
  this.l += size;
  return this;
}
function CheckField(hexstr, fld) {
  var m2 = __hexlify(this, this.l, hexstr.length >> 1);
  if (m2 !== hexstr)
    throw new Error(fld + "Expected " + hexstr + " saw " + m2);
  this.l += hexstr.length >> 1;
}
function prep_blob(blob, pos) {
  blob.l = pos;
  blob.read_shift = /*::(*/
  ReadShift;
  blob.chk = CheckField;
  blob.write_shift = WriteShift;
}
function parsenoop(blob, length) {
  blob.l += length;
}
function new_buf(sz) {
  var o2 = new_raw_buf(sz);
  prep_blob(o2, 0);
  return o2;
}
function recordhopper(data2, cb, opts) {
  if (!data2)
    return;
  var tmpbyte, cntbyte, length;
  prep_blob(data2, data2.l || 0);
  var L2 = data2.length, RT = 0, tgt = 0;
  while (data2.l < L2) {
    RT = data2.read_shift(1);
    if (RT & 128)
      RT = (RT & 127) + ((data2.read_shift(1) & 127) << 7);
    var R2 = XLSBRecordEnum[RT] || XLSBRecordEnum[65535];
    tmpbyte = data2.read_shift(1);
    length = tmpbyte & 127;
    for (cntbyte = 1; cntbyte < 4 && tmpbyte & 128; ++cntbyte)
      length += ((tmpbyte = data2.read_shift(1)) & 127) << 7 * cntbyte;
    tgt = data2.l + length;
    var d2 = R2.f && R2.f(data2, length, opts);
    data2.l = tgt;
    if (cb(d2, R2, RT))
      return;
  }
}
function buf_array() {
  var bufs = [], blksz = has_buf ? 256 : 2048;
  var newblk = function ba_newblk(sz) {
    var o2 = new_buf(sz);
    prep_blob(o2, 0);
    return o2;
  };
  var curbuf = newblk(blksz);
  var endbuf = function ba_endbuf() {
    if (!curbuf)
      return;
    if (curbuf.length > curbuf.l) {
      curbuf = curbuf.slice(0, curbuf.l);
      curbuf.l = curbuf.length;
    }
    if (curbuf.length > 0)
      bufs.push(curbuf);
    curbuf = null;
  };
  var next = function ba_next(sz) {
    if (curbuf && sz < curbuf.length - curbuf.l)
      return curbuf;
    endbuf();
    return curbuf = newblk(Math.max(sz + 1, blksz));
  };
  var end2 = function ba_end() {
    endbuf();
    return bconcat(bufs);
  };
  var push2 = function ba_push(buf) {
    endbuf();
    curbuf = buf;
    if (curbuf.l == null)
      curbuf.l = curbuf.length;
    next(blksz);
  };
  return { next, push: push2, end: end2, _bufs: bufs };
}
function write_record(ba, type2, payload, length) {
  var t3 = +type2, l2;
  if (isNaN(t3))
    return;
  if (!length)
    length = XLSBRecordEnum[t3].p || (payload || []).length || 0;
  l2 = 1 + (t3 >= 128 ? 1 : 0) + 1;
  if (length >= 128)
    ++l2;
  if (length >= 16384)
    ++l2;
  if (length >= 2097152)
    ++l2;
  var o2 = ba.next(l2);
  if (t3 <= 127)
    o2.write_shift(1, t3);
  else {
    o2.write_shift(1, (t3 & 127) + 128);
    o2.write_shift(1, t3 >> 7);
  }
  for (var i2 = 0; i2 != 4; ++i2) {
    if (length >= 128) {
      o2.write_shift(1, (length & 127) + 128);
      length >>= 7;
    } else {
      o2.write_shift(1, length);
      break;
    }
  }
  if (
    /*:: length != null &&*/
    length > 0 && is_buf(payload)
  )
    ba.push(payload);
}
function shift_cell_xls(cell, tgt, opts) {
  var out2 = dup(cell);
  if (tgt.s) {
    if (out2.cRel)
      out2.c += tgt.s.c;
    if (out2.rRel)
      out2.r += tgt.s.r;
  } else {
    if (out2.cRel)
      out2.c += tgt.c;
    if (out2.rRel)
      out2.r += tgt.r;
  }
  if (!opts || opts.biff < 12) {
    while (out2.c >= 256)
      out2.c -= 256;
    while (out2.r >= 65536)
      out2.r -= 65536;
  }
  return out2;
}
function shift_range_xls(cell, range2, opts) {
  var out2 = dup(cell);
  out2.s = shift_cell_xls(out2.s, range2.s, opts);
  out2.e = shift_cell_xls(out2.e, range2.s, opts);
  return out2;
}
function encode_cell_xls(c2, biff) {
  if (c2.cRel && c2.c < 0) {
    c2 = dup(c2);
    while (c2.c < 0)
      c2.c += biff > 8 ? 16384 : 256;
  }
  if (c2.rRel && c2.r < 0) {
    c2 = dup(c2);
    while (c2.r < 0)
      c2.r += biff > 8 ? 1048576 : biff > 5 ? 65536 : 16384;
  }
  var s2 = encode_cell(c2);
  if (!c2.cRel && c2.cRel != null)
    s2 = fix_col(s2);
  if (!c2.rRel && c2.rRel != null)
    s2 = fix_row(s2);
  return s2;
}
function encode_range_xls(r2, opts) {
  if (r2.s.r == 0 && !r2.s.rRel) {
    if (r2.e.r == (opts.biff >= 12 ? 1048575 : opts.biff >= 8 ? 65536 : 16384) && !r2.e.rRel) {
      return (r2.s.cRel ? "" : "$") + encode_col(r2.s.c) + ":" + (r2.e.cRel ? "" : "$") + encode_col(r2.e.c);
    }
  }
  if (r2.s.c == 0 && !r2.s.cRel) {
    if (r2.e.c == (opts.biff >= 12 ? 16383 : 255) && !r2.e.cRel) {
      return (r2.s.rRel ? "" : "$") + encode_row(r2.s.r) + ":" + (r2.e.rRel ? "" : "$") + encode_row(r2.e.r);
    }
  }
  return encode_cell_xls(r2.s, opts.biff) + ":" + encode_cell_xls(r2.e, opts.biff);
}
function decode_row(rowstr) {
  return parseInt(unfix_row(rowstr), 10) - 1;
}
function encode_row(row) {
  return "" + (row + 1);
}
function fix_row(cstr) {
  return cstr.replace(/([A-Z]|^)(\d+)$/, "$1$$$2");
}
function unfix_row(cstr) {
  return cstr.replace(/\$(\d+)$/, "$1");
}
function decode_col(colstr) {
  var c2 = unfix_col(colstr), d2 = 0, i2 = 0;
  for (; i2 !== c2.length; ++i2)
    d2 = 26 * d2 + c2.charCodeAt(i2) - 64;
  return d2 - 1;
}
function encode_col(col) {
  if (col < 0)
    throw new Error("invalid column " + col);
  var s2 = "";
  for (++col; col; col = Math.floor((col - 1) / 26))
    s2 = String.fromCharCode((col - 1) % 26 + 65) + s2;
  return s2;
}
function fix_col(cstr) {
  return cstr.replace(/^([A-Z])/, "$$$1");
}
function unfix_col(cstr) {
  return cstr.replace(/^\$([A-Z])/, "$1");
}
function split_cell(cstr) {
  return cstr.replace(/(\$?[A-Z]*)(\$?\d*)/, "$1,$2").split(",");
}
function decode_cell(cstr) {
  var R2 = 0, C2 = 0;
  for (var i2 = 0; i2 < cstr.length; ++i2) {
    var cc = cstr.charCodeAt(i2);
    if (cc >= 48 && cc <= 57)
      R2 = 10 * R2 + (cc - 48);
    else if (cc >= 65 && cc <= 90)
      C2 = 26 * C2 + (cc - 64);
  }
  return { c: C2 - 1, r: R2 - 1 };
}
function encode_cell(cell) {
  var col = cell.c + 1;
  var s2 = "";
  for (; col; col = (col - 1) / 26 | 0)
    s2 = String.fromCharCode((col - 1) % 26 + 65) + s2;
  return s2 + (cell.r + 1);
}
function decode_range(range2) {
  var idx = range2.indexOf(":");
  if (idx == -1)
    return { s: decode_cell(range2), e: decode_cell(range2) };
  return { s: decode_cell(range2.slice(0, idx)), e: decode_cell(range2.slice(idx + 1)) };
}
function encode_range(cs, ce2) {
  if (typeof ce2 === "undefined" || typeof ce2 === "number") {
    return encode_range(cs.s, cs.e);
  }
  if (typeof cs !== "string")
    cs = encode_cell(cs);
  if (typeof ce2 !== "string")
    ce2 = encode_cell(ce2);
  return cs == ce2 ? cs : cs + ":" + ce2;
}
function safe_decode_range(range2) {
  var o2 = { s: { c: 0, r: 0 }, e: { c: 0, r: 0 } };
  var idx = 0, i2 = 0, cc = 0;
  var len2 = range2.length;
  for (idx = 0; i2 < len2; ++i2) {
    if ((cc = range2.charCodeAt(i2) - 64) < 1 || cc > 26)
      break;
    idx = 26 * idx + cc;
  }
  o2.s.c = --idx;
  for (idx = 0; i2 < len2; ++i2) {
    if ((cc = range2.charCodeAt(i2) - 48) < 0 || cc > 9)
      break;
    idx = 10 * idx + cc;
  }
  o2.s.r = --idx;
  if (i2 === len2 || cc != 10) {
    o2.e.c = o2.s.c;
    o2.e.r = o2.s.r;
    return o2;
  }
  ++i2;
  for (idx = 0; i2 != len2; ++i2) {
    if ((cc = range2.charCodeAt(i2) - 64) < 1 || cc > 26)
      break;
    idx = 26 * idx + cc;
  }
  o2.e.c = --idx;
  for (idx = 0; i2 != len2; ++i2) {
    if ((cc = range2.charCodeAt(i2) - 48) < 0 || cc > 9)
      break;
    idx = 10 * idx + cc;
  }
  o2.e.r = --idx;
  return o2;
}
function safe_format_cell(cell, v4) {
  var q2 = cell.t == "d" && v4 instanceof Date;
  if (cell.z != null)
    try {
      return cell.w = SSF_format(cell.z, q2 ? datenum(v4) : v4);
    } catch (e2) {
    }
  try {
    return cell.w = SSF_format((cell.XF || {}).numFmtId || (q2 ? 14 : 0), q2 ? datenum(v4) : v4);
  } catch (e2) {
    return "" + v4;
  }
}
function format_cell(cell, v4, o2) {
  if (cell == null || cell.t == null || cell.t == "z")
    return "";
  if (cell.w !== void 0)
    return cell.w;
  if (cell.t == "d" && !cell.z && o2 && o2.dateNF)
    cell.z = o2.dateNF;
  if (cell.t == "e")
    return BErr[cell.v] || cell.v;
  if (v4 == void 0)
    return safe_format_cell(cell, cell.v);
  return safe_format_cell(cell, v4);
}
function sheet_to_workbook(sheet, opts) {
  var n2 = opts && opts.sheet ? opts.sheet : "Sheet1";
  var sheets = {};
  sheets[n2] = sheet;
  return { SheetNames: [n2], Sheets: sheets };
}
function sheet_add_aoa(_ws, data2, opts) {
  var o2 = opts || {};
  var dense = _ws ? Array.isArray(_ws) : o2.dense;
  var ws = _ws || (dense ? [] : {});
  var _R = 0, _C = 0;
  if (ws && o2.origin != null) {
    if (typeof o2.origin == "number")
      _R = o2.origin;
    else {
      var _origin = typeof o2.origin == "string" ? decode_cell(o2.origin) : o2.origin;
      _R = _origin.r;
      _C = _origin.c;
    }
    if (!ws["!ref"])
      ws["!ref"] = "A1:A1";
  }
  var range2 = { s: { c: 1e7, r: 1e7 }, e: { c: 0, r: 0 } };
  if (ws["!ref"]) {
    var _range = safe_decode_range(ws["!ref"]);
    range2.s.c = _range.s.c;
    range2.s.r = _range.s.r;
    range2.e.c = Math.max(range2.e.c, _range.e.c);
    range2.e.r = Math.max(range2.e.r, _range.e.r);
    if (_R == -1)
      range2.e.r = _R = _range.e.r + 1;
  }
  for (var R2 = 0; R2 != data2.length; ++R2) {
    if (!data2[R2])
      continue;
    if (!Array.isArray(data2[R2]))
      throw new Error("aoa_to_sheet expects an array of arrays");
    for (var C2 = 0; C2 != data2[R2].length; ++C2) {
      if (typeof data2[R2][C2] === "undefined")
        continue;
      var cell = { v: data2[R2][C2] };
      var __R = _R + R2, __C = _C + C2;
      if (range2.s.r > __R)
        range2.s.r = __R;
      if (range2.s.c > __C)
        range2.s.c = __C;
      if (range2.e.r < __R)
        range2.e.r = __R;
      if (range2.e.c < __C)
        range2.e.c = __C;
      if (data2[R2][C2] && typeof data2[R2][C2] === "object" && !Array.isArray(data2[R2][C2]) && !(data2[R2][C2] instanceof Date))
        cell = data2[R2][C2];
      else {
        if (Array.isArray(cell.v)) {
          cell.f = data2[R2][C2][1];
          cell.v = cell.v[0];
        }
        if (cell.v === null) {
          if (cell.f)
            cell.t = "n";
          else if (o2.nullError) {
            cell.t = "e";
            cell.v = 0;
          } else if (!o2.sheetStubs)
            continue;
          else
            cell.t = "z";
        } else if (typeof cell.v === "number")
          cell.t = "n";
        else if (typeof cell.v === "boolean")
          cell.t = "b";
        else if (cell.v instanceof Date) {
          cell.z = o2.dateNF || table_fmt[14];
          if (o2.cellDates) {
            cell.t = "d";
            cell.w = SSF_format(cell.z, datenum(cell.v));
          } else {
            cell.t = "n";
            cell.v = datenum(cell.v);
            cell.w = SSF_format(cell.z, cell.v);
          }
        } else
          cell.t = "s";
      }
      if (dense) {
        if (!ws[__R])
          ws[__R] = [];
        if (ws[__R][__C] && ws[__R][__C].z)
          cell.z = ws[__R][__C].z;
        ws[__R][__C] = cell;
      } else {
        var cell_ref = encode_cell({ c: __C, r: __R });
        if (ws[cell_ref] && ws[cell_ref].z)
          cell.z = ws[cell_ref].z;
        ws[cell_ref] = cell;
      }
    }
  }
  if (range2.s.c < 1e7)
    ws["!ref"] = encode_range(range2);
  return ws;
}
function aoa_to_sheet(data2, opts) {
  return sheet_add_aoa(null, data2, opts);
}
function parse_Int32LE(data2) {
  return data2.read_shift(4, "i");
}
function write_UInt32LE(x2, o2) {
  if (!o2)
    o2 = new_buf(4);
  o2.write_shift(4, x2);
  return o2;
}
function parse_XLWideString(data2) {
  var cchCharacters = data2.read_shift(4);
  return cchCharacters === 0 ? "" : data2.read_shift(cchCharacters, "dbcs");
}
function write_XLWideString(data2, o2) {
  var _null = false;
  if (o2 == null) {
    _null = true;
    o2 = new_buf(4 + 2 * data2.length);
  }
  o2.write_shift(4, data2.length);
  if (data2.length > 0)
    o2.write_shift(0, data2, "dbcs");
  return _null ? o2.slice(0, o2.l) : o2;
}
function parse_StrRun(data2) {
  return { ich: data2.read_shift(2), ifnt: data2.read_shift(2) };
}
function write_StrRun(run2, o2) {
  if (!o2)
    o2 = new_buf(4);
  o2.write_shift(2, run2.ich || 0);
  o2.write_shift(2, run2.ifnt || 0);
  return o2;
}
function parse_RichStr(data2, length) {
  var start2 = data2.l;
  var flags3 = data2.read_shift(1);
  var str = parse_XLWideString(data2);
  var rgsStrRun = [];
  var z2 = { t: str, h: str };
  if ((flags3 & 1) !== 0) {
    var dwSizeStrRun = data2.read_shift(4);
    for (var i2 = 0; i2 != dwSizeStrRun; ++i2)
      rgsStrRun.push(parse_StrRun(data2));
    z2.r = rgsStrRun;
  } else
    z2.r = [{ ich: 0, ifnt: 0 }];
  data2.l = start2 + length;
  return z2;
}
function write_RichStr(str, o2) {
  var _null = false;
  if (o2 == null) {
    _null = true;
    o2 = new_buf(15 + 4 * str.t.length);
  }
  o2.write_shift(1, 0);
  write_XLWideString(str.t, o2);
  return _null ? o2.slice(0, o2.l) : o2;
}
var parse_BrtCommentText = parse_RichStr;
function write_BrtCommentText(str, o2) {
  var _null = false;
  if (o2 == null) {
    _null = true;
    o2 = new_buf(23 + 4 * str.t.length);
  }
  o2.write_shift(1, 1);
  write_XLWideString(str.t, o2);
  o2.write_shift(4, 1);
  write_StrRun({ ich: 0, ifnt: 0 }, o2);
  return _null ? o2.slice(0, o2.l) : o2;
}
function parse_XLSBCell(data2) {
  var col = data2.read_shift(4);
  var iStyleRef = data2.read_shift(2);
  iStyleRef += data2.read_shift(1) << 16;
  data2.l++;
  return { c: col, iStyleRef };
}
function write_XLSBCell(cell, o2) {
  if (o2 == null)
    o2 = new_buf(8);
  o2.write_shift(-4, cell.c);
  o2.write_shift(3, cell.iStyleRef || cell.s);
  o2.write_shift(1, 0);
  return o2;
}
function parse_XLSBShortCell(data2) {
  var iStyleRef = data2.read_shift(2);
  iStyleRef += data2.read_shift(1) << 16;
  data2.l++;
  return { c: -1, iStyleRef };
}
function write_XLSBShortCell(cell, o2) {
  if (o2 == null)
    o2 = new_buf(4);
  o2.write_shift(3, cell.iStyleRef || cell.s);
  o2.write_shift(1, 0);
  return o2;
}
var parse_XLSBCodeName = parse_XLWideString;
var write_XLSBCodeName = write_XLWideString;
function parse_XLNullableWideString(data2) {
  var cchCharacters = data2.read_shift(4);
  return cchCharacters === 0 || cchCharacters === 4294967295 ? "" : data2.read_shift(cchCharacters, "dbcs");
}
function write_XLNullableWideString(data2, o2) {
  var _null = false;
  if (o2 == null) {
    _null = true;
    o2 = new_buf(127);
  }
  o2.write_shift(4, data2.length > 0 ? data2.length : 4294967295);
  if (data2.length > 0)
    o2.write_shift(0, data2, "dbcs");
  return _null ? o2.slice(0, o2.l) : o2;
}
var parse_XLNameWideString = parse_XLWideString;
var parse_RelID = parse_XLNullableWideString;
var write_RelID = write_XLNullableWideString;
function parse_RkNumber(data2) {
  var b2 = data2.slice(data2.l, data2.l + 4);
  var fX100 = b2[0] & 1, fInt = b2[0] & 2;
  data2.l += 4;
  var RK = fInt === 0 ? __double([0, 0, 0, 0, b2[0] & 252, b2[1], b2[2], b2[3]], 0) : __readInt32LE(b2, 0) >> 2;
  return fX100 ? RK / 100 : RK;
}
function write_RkNumber(data2, o2) {
  if (o2 == null)
    o2 = new_buf(4);
  var fX100 = 0, fInt = 0, d100 = data2 * 100;
  if (data2 == (data2 | 0) && data2 >= -(1 << 29) && data2 < 1 << 29) {
    fInt = 1;
  } else if (d100 == (d100 | 0) && d100 >= -(1 << 29) && d100 < 1 << 29) {
    fInt = 1;
    fX100 = 1;
  }
  if (fInt)
    o2.write_shift(-4, ((fX100 ? d100 : data2) << 2) + (fX100 + 2));
  else
    throw new Error("unsupported RkNumber " + data2);
}
function parse_RfX(data2) {
  var cell = { s: {}, e: {} };
  cell.s.r = data2.read_shift(4);
  cell.e.r = data2.read_shift(4);
  cell.s.c = data2.read_shift(4);
  cell.e.c = data2.read_shift(4);
  return cell;
}
function write_RfX(r2, o2) {
  if (!o2)
    o2 = new_buf(16);
  o2.write_shift(4, r2.s.r);
  o2.write_shift(4, r2.e.r);
  o2.write_shift(4, r2.s.c);
  o2.write_shift(4, r2.e.c);
  return o2;
}
var parse_UncheckedRfX = parse_RfX;
var write_UncheckedRfX = write_RfX;
function parse_Xnum(data2) {
  if (data2.length - data2.l < 8)
    throw "XLS Xnum Buffer underflow";
  return data2.read_shift(8, "f");
}
function write_Xnum(data2, o2) {
  return (o2 || new_buf(8)).write_shift(8, data2, "f");
}
function parse_BrtColor(data2) {
  var out2 = {};
  var d2 = data2.read_shift(1);
  var xColorType = d2 >>> 1;
  var index2 = data2.read_shift(1);
  var nTS = data2.read_shift(2, "i");
  var bR = data2.read_shift(1);
  var bG = data2.read_shift(1);
  var bB = data2.read_shift(1);
  data2.l++;
  switch (xColorType) {
    case 0:
      out2.auto = 1;
      break;
    case 1:
      out2.index = index2;
      var icv = XLSIcv[index2];
      if (icv)
        out2.rgb = rgb2Hex(icv);
      break;
    case 2:
      out2.rgb = rgb2Hex([bR, bG, bB]);
      break;
    case 3:
      out2.theme = index2;
      break;
  }
  if (nTS != 0)
    out2.tint = nTS > 0 ? nTS / 32767 : nTS / 32768;
  return out2;
}
function write_BrtColor(color2, o2) {
  if (!o2)
    o2 = new_buf(8);
  if (!color2 || color2.auto) {
    o2.write_shift(4, 0);
    o2.write_shift(4, 0);
    return o2;
  }
  if (color2.index != null) {
    o2.write_shift(1, 2);
    o2.write_shift(1, color2.index);
  } else if (color2.theme != null) {
    o2.write_shift(1, 6);
    o2.write_shift(1, color2.theme);
  } else {
    o2.write_shift(1, 5);
    o2.write_shift(1, 0);
  }
  var nTS = color2.tint || 0;
  if (nTS > 0)
    nTS *= 32767;
  else if (nTS < 0)
    nTS *= 32768;
  o2.write_shift(2, nTS);
  if (!color2.rgb || color2.theme != null) {
    o2.write_shift(2, 0);
    o2.write_shift(1, 0);
    o2.write_shift(1, 0);
  } else {
    var rgb2 = color2.rgb || "FFFFFF";
    if (typeof rgb2 == "number")
      rgb2 = ("000000" + rgb2.toString(16)).slice(-6);
    o2.write_shift(1, parseInt(rgb2.slice(0, 2), 16));
    o2.write_shift(1, parseInt(rgb2.slice(2, 4), 16));
    o2.write_shift(1, parseInt(rgb2.slice(4, 6), 16));
    o2.write_shift(1, 255);
  }
  return o2;
}
function parse_FontFlags(data2) {
  var d2 = data2.read_shift(1);
  data2.l++;
  var out2 = {
    fBold: d2 & 1,
    fItalic: d2 & 2,
    fUnderline: d2 & 4,
    fStrikeout: d2 & 8,
    fOutline: d2 & 16,
    fShadow: d2 & 32,
    fCondense: d2 & 64,
    fExtend: d2 & 128
  };
  return out2;
}
function write_FontFlags(font, o2) {
  if (!o2)
    o2 = new_buf(2);
  var grbit = (font.italic ? 2 : 0) | (font.strike ? 8 : 0) | (font.outline ? 16 : 0) | (font.shadow ? 32 : 0) | (font.condense ? 64 : 0) | (font.extend ? 128 : 0);
  o2.write_shift(1, grbit);
  o2.write_shift(1, 0);
  return o2;
}
function parse_ClipboardFormatOrString(o2, w2) {
  var ClipFmt = { 2: "BITMAP", 3: "METAFILEPICT", 8: "DIB", 14: "ENHMETAFILE" };
  var m2 = o2.read_shift(4);
  switch (m2) {
    case 0:
      return "";
    case 4294967295:
    case 4294967294:
      return ClipFmt[o2.read_shift(4)] || "";
  }
  if (m2 > 400)
    throw new Error("Unsupported Clipboard: " + m2.toString(16));
  o2.l -= 4;
  return o2.read_shift(0, w2 == 1 ? "lpstr" : "lpwstr");
}
function parse_ClipboardFormatOrAnsiString(o2) {
  return parse_ClipboardFormatOrString(o2, 1);
}
function parse_ClipboardFormatOrUnicodeString(o2) {
  return parse_ClipboardFormatOrString(o2, 2);
}
var VT_I2 = 2;
var VT_I4 = 3;
var VT_BOOL = 11;
var VT_VARIANT = 12;
var VT_UI4 = 19;
var VT_FILETIME = 64;
var VT_BLOB = 65;
var VT_CF = 71;
var VT_VECTOR_VARIANT = 4108;
var VT_VECTOR_LPSTR = 4126;
var VT_STRING = 80;
var VT_USTR = 81;
var VT_CUSTOM = [VT_STRING, VT_USTR];
var DocSummaryPIDDSI = {
  /*::[*/
  1: { n: "CodePage", t: VT_I2 },
  /*::[*/
  2: { n: "Category", t: VT_STRING },
  /*::[*/
  3: { n: "PresentationFormat", t: VT_STRING },
  /*::[*/
  4: { n: "ByteCount", t: VT_I4 },
  /*::[*/
  5: { n: "LineCount", t: VT_I4 },
  /*::[*/
  6: { n: "ParagraphCount", t: VT_I4 },
  /*::[*/
  7: { n: "SlideCount", t: VT_I4 },
  /*::[*/
  8: { n: "NoteCount", t: VT_I4 },
  /*::[*/
  9: { n: "HiddenCount", t: VT_I4 },
  /*::[*/
  10: { n: "MultimediaClipCount", t: VT_I4 },
  /*::[*/
  11: { n: "ScaleCrop", t: VT_BOOL },
  /*::[*/
  12: {
    n: "HeadingPairs",
    t: VT_VECTOR_VARIANT
    /* VT_VECTOR | VT_VARIANT */
  },
  /*::[*/
  13: {
    n: "TitlesOfParts",
    t: VT_VECTOR_LPSTR
    /* VT_VECTOR | VT_LPSTR */
  },
  /*::[*/
  14: { n: "Manager", t: VT_STRING },
  /*::[*/
  15: { n: "Company", t: VT_STRING },
  /*::[*/
  16: { n: "LinksUpToDate", t: VT_BOOL },
  /*::[*/
  17: { n: "CharacterCount", t: VT_I4 },
  /*::[*/
  19: { n: "SharedDoc", t: VT_BOOL },
  /*::[*/
  22: { n: "HyperlinksChanged", t: VT_BOOL },
  /*::[*/
  23: { n: "AppVersion", t: VT_I4, p: "version" },
  /*::[*/
  24: { n: "DigSig", t: VT_BLOB },
  /*::[*/
  26: { n: "ContentType", t: VT_STRING },
  /*::[*/
  27: { n: "ContentStatus", t: VT_STRING },
  /*::[*/
  28: { n: "Language", t: VT_STRING },
  /*::[*/
  29: { n: "Version", t: VT_STRING },
  /*::[*/
  255: {},
  /* [MS-OLEPS] 2.18 */
  /*::[*/
  2147483648: { n: "Locale", t: VT_UI4 },
  /*::[*/
  2147483651: { n: "Behavior", t: VT_UI4 },
  /*::[*/
  1919054434: {}
};
var SummaryPIDSI = {
  /*::[*/
  1: { n: "CodePage", t: VT_I2 },
  /*::[*/
  2: { n: "Title", t: VT_STRING },
  /*::[*/
  3: { n: "Subject", t: VT_STRING },
  /*::[*/
  4: { n: "Author", t: VT_STRING },
  /*::[*/
  5: { n: "Keywords", t: VT_STRING },
  /*::[*/
  6: { n: "Comments", t: VT_STRING },
  /*::[*/
  7: { n: "Template", t: VT_STRING },
  /*::[*/
  8: { n: "LastAuthor", t: VT_STRING },
  /*::[*/
  9: { n: "RevNumber", t: VT_STRING },
  /*::[*/
  10: { n: "EditTime", t: VT_FILETIME },
  /*::[*/
  11: { n: "LastPrinted", t: VT_FILETIME },
  /*::[*/
  12: { n: "CreatedDate", t: VT_FILETIME },
  /*::[*/
  13: { n: "ModifiedDate", t: VT_FILETIME },
  /*::[*/
  14: { n: "PageCount", t: VT_I4 },
  /*::[*/
  15: { n: "WordCount", t: VT_I4 },
  /*::[*/
  16: { n: "CharCount", t: VT_I4 },
  /*::[*/
  17: { n: "Thumbnail", t: VT_CF },
  /*::[*/
  18: { n: "Application", t: VT_STRING },
  /*::[*/
  19: { n: "DocSecurity", t: VT_I4 },
  /*::[*/
  255: {},
  /* [MS-OLEPS] 2.18 */
  /*::[*/
  2147483648: { n: "Locale", t: VT_UI4 },
  /*::[*/
  2147483651: { n: "Behavior", t: VT_UI4 },
  /*::[*/
  1919054434: {}
};
var CountryEnum = {
  /*::[*/
  1: "US",
  // United States
  /*::[*/
  2: "CA",
  // Canada
  /*::[*/
  3: "",
  // Latin America (except Brazil)
  /*::[*/
  7: "RU",
  // Russia
  /*::[*/
  20: "EG",
  // Egypt
  /*::[*/
  30: "GR",
  // Greece
  /*::[*/
  31: "NL",
  // Netherlands
  /*::[*/
  32: "BE",
  // Belgium
  /*::[*/
  33: "FR",
  // France
  /*::[*/
  34: "ES",
  // Spain
  /*::[*/
  36: "HU",
  // Hungary
  /*::[*/
  39: "IT",
  // Italy
  /*::[*/
  41: "CH",
  // Switzerland
  /*::[*/
  43: "AT",
  // Austria
  /*::[*/
  44: "GB",
  // United Kingdom
  /*::[*/
  45: "DK",
  // Denmark
  /*::[*/
  46: "SE",
  // Sweden
  /*::[*/
  47: "NO",
  // Norway
  /*::[*/
  48: "PL",
  // Poland
  /*::[*/
  49: "DE",
  // Germany
  /*::[*/
  52: "MX",
  // Mexico
  /*::[*/
  55: "BR",
  // Brazil
  /*::[*/
  61: "AU",
  // Australia
  /*::[*/
  64: "NZ",
  // New Zealand
  /*::[*/
  66: "TH",
  // Thailand
  /*::[*/
  81: "JP",
  // Japan
  /*::[*/
  82: "KR",
  // Korea
  /*::[*/
  84: "VN",
  // Viet Nam
  /*::[*/
  86: "CN",
  // China
  /*::[*/
  90: "TR",
  // Turkey
  /*::[*/
  105: "JS",
  // Ramastan
  /*::[*/
  213: "DZ",
  // Algeria
  /*::[*/
  216: "MA",
  // Morocco
  /*::[*/
  218: "LY",
  // Libya
  /*::[*/
  351: "PT",
  // Portugal
  /*::[*/
  354: "IS",
  // Iceland
  /*::[*/
  358: "FI",
  // Finland
  /*::[*/
  420: "CZ",
  // Czech Republic
  /*::[*/
  886: "TW",
  // Taiwan
  /*::[*/
  961: "LB",
  // Lebanon
  /*::[*/
  962: "JO",
  // Jordan
  /*::[*/
  963: "SY",
  // Syria
  /*::[*/
  964: "IQ",
  // Iraq
  /*::[*/
  965: "KW",
  // Kuwait
  /*::[*/
  966: "SA",
  // Saudi Arabia
  /*::[*/
  971: "AE",
  // United Arab Emirates
  /*::[*/
  972: "IL",
  // Israel
  /*::[*/
  974: "QA",
  // Qatar
  /*::[*/
  981: "IR",
  // Iran
  /*::[*/
  65535: "US"
  // United States
};
var XLSFillPattern = [
  null,
  "solid",
  "mediumGray",
  "darkGray",
  "lightGray",
  "darkHorizontal",
  "darkVertical",
  "darkDown",
  "darkUp",
  "darkGrid",
  "darkTrellis",
  "lightHorizontal",
  "lightVertical",
  "lightDown",
  "lightUp",
  "lightGrid",
  "lightTrellis",
  "gray125",
  "gray0625"
];
function rgbify(arr) {
  return arr.map(function(x2) {
    return [x2 >> 16 & 255, x2 >> 8 & 255, x2 & 255];
  });
}
var _XLSIcv = /* @__PURE__ */ rgbify([
  /* Color Constants */
  0,
  16777215,
  16711680,
  65280,
  255,
  16776960,
  16711935,
  65535,
  /* Overridable Defaults */
  0,
  16777215,
  16711680,
  65280,
  255,
  16776960,
  16711935,
  65535,
  8388608,
  32768,
  128,
  8421376,
  8388736,
  32896,
  12632256,
  8421504,
  10066431,
  10040166,
  16777164,
  13434879,
  6684774,
  16744576,
  26316,
  13421823,
  128,
  16711935,
  16776960,
  65535,
  8388736,
  8388608,
  32896,
  255,
  52479,
  13434879,
  13434828,
  16777113,
  10079487,
  16751052,
  13408767,
  16764057,
  3368703,
  3394764,
  10079232,
  16763904,
  16750848,
  16737792,
  6710937,
  9868950,
  13158,
  3381606,
  13056,
  3355392,
  10040064,
  10040166,
  3355545,
  3355443,
  /* Other entries to appease BIFF8/12 */
  16777215,
  /* 0x40 icvForeground ?? */
  0,
  /* 0x41 icvBackground ?? */
  0,
  /* 0x42 icvFrame ?? */
  0,
  /* 0x43 icv3D ?? */
  0,
  /* 0x44 icv3DText ?? */
  0,
  /* 0x45 icv3DHilite ?? */
  0,
  /* 0x46 icv3DShadow ?? */
  0,
  /* 0x47 icvHilite ?? */
  0,
  /* 0x48 icvCtlText ?? */
  0,
  /* 0x49 icvCtlScrl ?? */
  0,
  /* 0x4A icvCtlInv ?? */
  0,
  /* 0x4B icvCtlBody ?? */
  0,
  /* 0x4C icvCtlFrame ?? */
  0,
  /* 0x4D icvCtlFore ?? */
  0,
  /* 0x4E icvCtlBack ?? */
  0,
  /* 0x4F icvCtlNeutral */
  0,
  /* 0x50 icvInfoBk ?? */
  0
  /* 0x51 icvInfoText ?? */
]);
var XLSIcv = /* @__PURE__ */ dup(_XLSIcv);
var BErr = {
  /*::[*/
  0: "#NULL!",
  /*::[*/
  7: "#DIV/0!",
  /*::[*/
  15: "#VALUE!",
  /*::[*/
  23: "#REF!",
  /*::[*/
  29: "#NAME?",
  /*::[*/
  36: "#NUM!",
  /*::[*/
  42: "#N/A",
  /*::[*/
  43: "#GETTING_DATA",
  /*::[*/
  255: "#WTF?"
};
var RBErr = {
  "#NULL!": 0,
  "#DIV/0!": 7,
  "#VALUE!": 15,
  "#REF!": 23,
  "#NAME?": 29,
  "#NUM!": 36,
  "#N/A": 42,
  "#GETTING_DATA": 43,
  "#WTF?": 255
};
var ct2type = {
  /* Workbook */
  "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml": "workbooks",
  "application/vnd.ms-excel.sheet.macroEnabled.main+xml": "workbooks",
  "application/vnd.ms-excel.sheet.binary.macroEnabled.main": "workbooks",
  "application/vnd.ms-excel.addin.macroEnabled.main+xml": "workbooks",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.template.main+xml": "workbooks",
  /* Worksheet */
  "application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml": "sheets",
  "application/vnd.ms-excel.worksheet": "sheets",
  "application/vnd.ms-excel.binIndexWs": "TODO",
  /* Binary Index */
  /* Chartsheet */
  "application/vnd.openxmlformats-officedocument.spreadsheetml.chartsheet+xml": "charts",
  "application/vnd.ms-excel.chartsheet": "charts",
  /* Macrosheet */
  "application/vnd.ms-excel.macrosheet+xml": "macros",
  "application/vnd.ms-excel.macrosheet": "macros",
  "application/vnd.ms-excel.intlmacrosheet": "TODO",
  "application/vnd.ms-excel.binIndexMs": "TODO",
  /* Binary Index */
  /* Dialogsheet */
  "application/vnd.openxmlformats-officedocument.spreadsheetml.dialogsheet+xml": "dialogs",
  "application/vnd.ms-excel.dialogsheet": "dialogs",
  /* Shared Strings */
  "application/vnd.openxmlformats-officedocument.spreadsheetml.sharedStrings+xml": "strs",
  "application/vnd.ms-excel.sharedStrings": "strs",
  /* Styles */
  "application/vnd.openxmlformats-officedocument.spreadsheetml.styles+xml": "styles",
  "application/vnd.ms-excel.styles": "styles",
  /* File Properties */
  "application/vnd.openxmlformats-package.core-properties+xml": "coreprops",
  "application/vnd.openxmlformats-officedocument.custom-properties+xml": "custprops",
  "application/vnd.openxmlformats-officedocument.extended-properties+xml": "extprops",
  /* Custom Data Properties */
  "application/vnd.openxmlformats-officedocument.customXmlProperties+xml": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.customProperty": "TODO",
  /* Comments */
  "application/vnd.openxmlformats-officedocument.spreadsheetml.comments+xml": "comments",
  "application/vnd.ms-excel.comments": "comments",
  "application/vnd.ms-excel.threadedcomments+xml": "threadedcomments",
  "application/vnd.ms-excel.person+xml": "people",
  /* Metadata (Stock/Geography and Dynamic Array) */
  "application/vnd.openxmlformats-officedocument.spreadsheetml.sheetMetadata+xml": "metadata",
  "application/vnd.ms-excel.sheetMetadata": "metadata",
  /* PivotTable */
  "application/vnd.ms-excel.pivotTable": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.pivotTable+xml": "TODO",
  /* Chart Objects */
  "application/vnd.openxmlformats-officedocument.drawingml.chart+xml": "TODO",
  /* Chart Colors */
  "application/vnd.ms-office.chartcolorstyle+xml": "TODO",
  /* Chart Style */
  "application/vnd.ms-office.chartstyle+xml": "TODO",
  /* Chart Advanced */
  "application/vnd.ms-office.chartex+xml": "TODO",
  /* Calculation Chain */
  "application/vnd.ms-excel.calcChain": "calcchains",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.calcChain+xml": "calcchains",
  /* Printer Settings */
  "application/vnd.openxmlformats-officedocument.spreadsheetml.printerSettings": "TODO",
  /* ActiveX */
  "application/vnd.ms-office.activeX": "TODO",
  "application/vnd.ms-office.activeX+xml": "TODO",
  /* Custom Toolbars */
  "application/vnd.ms-excel.attachedToolbars": "TODO",
  /* External Data Connections */
  "application/vnd.ms-excel.connections": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.connections+xml": "TODO",
  /* External Links */
  "application/vnd.ms-excel.externalLink": "links",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.externalLink+xml": "links",
  /* PivotCache */
  "application/vnd.ms-excel.pivotCacheDefinition": "TODO",
  "application/vnd.ms-excel.pivotCacheRecords": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.pivotCacheDefinition+xml": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.pivotCacheRecords+xml": "TODO",
  /* Query Table */
  "application/vnd.ms-excel.queryTable": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.queryTable+xml": "TODO",
  /* Shared Workbook */
  "application/vnd.ms-excel.userNames": "TODO",
  "application/vnd.ms-excel.revisionHeaders": "TODO",
  "application/vnd.ms-excel.revisionLog": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.revisionHeaders+xml": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.revisionLog+xml": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.userNames+xml": "TODO",
  /* Single Cell Table */
  "application/vnd.ms-excel.tableSingleCells": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.tableSingleCells+xml": "TODO",
  /* Slicer */
  "application/vnd.ms-excel.slicer": "TODO",
  "application/vnd.ms-excel.slicerCache": "TODO",
  "application/vnd.ms-excel.slicer+xml": "TODO",
  "application/vnd.ms-excel.slicerCache+xml": "TODO",
  /* Sort Map */
  "application/vnd.ms-excel.wsSortMap": "TODO",
  /* Table */
  "application/vnd.ms-excel.table": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.table+xml": "TODO",
  /* Themes */
  "application/vnd.openxmlformats-officedocument.theme+xml": "themes",
  /* Theme Override */
  "application/vnd.openxmlformats-officedocument.themeOverride+xml": "TODO",
  /* Timeline */
  "application/vnd.ms-excel.Timeline+xml": "TODO",
  /* verify */
  "application/vnd.ms-excel.TimelineCache+xml": "TODO",
  /* verify */
  /* VBA */
  "application/vnd.ms-office.vbaProject": "vba",
  "application/vnd.ms-office.vbaProjectSignature": "TODO",
  /* Volatile Dependencies */
  "application/vnd.ms-office.volatileDependencies": "TODO",
  "application/vnd.openxmlformats-officedocument.spreadsheetml.volatileDependencies+xml": "TODO",
  /* Control Properties */
  "application/vnd.ms-excel.controlproperties+xml": "TODO",
  /* Data Model */
  "application/vnd.openxmlformats-officedocument.model+data": "TODO",
  /* Survey */
  "application/vnd.ms-excel.Survey+xml": "TODO",
  /* Drawing */
  "application/vnd.openxmlformats-officedocument.drawing+xml": "drawings",
  "application/vnd.openxmlformats-officedocument.drawingml.chartshapes+xml": "TODO",
  "application/vnd.openxmlformats-officedocument.drawingml.diagramColors+xml": "TODO",
  "application/vnd.openxmlformats-officedocument.drawingml.diagramData+xml": "TODO",
  "application/vnd.openxmlformats-officedocument.drawingml.diagramLayout+xml": "TODO",
  "application/vnd.openxmlformats-officedocument.drawingml.diagramStyle+xml": "TODO",
  /* VML */
  "application/vnd.openxmlformats-officedocument.vmlDrawing": "TODO",
  "application/vnd.openxmlformats-package.relationships+xml": "rels",
  "application/vnd.openxmlformats-officedocument.oleObject": "TODO",
  /* Image */
  "image/png": "TODO",
  "sheet": "js"
};
var CT_LIST = {
  workbooks: {
    xlsx: "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml",
    xlsm: "application/vnd.ms-excel.sheet.macroEnabled.main+xml",
    xlsb: "application/vnd.ms-excel.sheet.binary.macroEnabled.main",
    xlam: "application/vnd.ms-excel.addin.macroEnabled.main+xml",
    xltx: "application/vnd.openxmlformats-officedocument.spreadsheetml.template.main+xml"
  },
  strs: {
    /* Shared Strings */
    xlsx: "application/vnd.openxmlformats-officedocument.spreadsheetml.sharedStrings+xml",
    xlsb: "application/vnd.ms-excel.sharedStrings"
  },
  comments: {
    /* Comments */
    xlsx: "application/vnd.openxmlformats-officedocument.spreadsheetml.comments+xml",
    xlsb: "application/vnd.ms-excel.comments"
  },
  sheets: {
    /* Worksheet */
    xlsx: "application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml",
    xlsb: "application/vnd.ms-excel.worksheet"
  },
  charts: {
    /* Chartsheet */
    xlsx: "application/vnd.openxmlformats-officedocument.spreadsheetml.chartsheet+xml",
    xlsb: "application/vnd.ms-excel.chartsheet"
  },
  dialogs: {
    /* Dialogsheet */
    xlsx: "application/vnd.openxmlformats-officedocument.spreadsheetml.dialogsheet+xml",
    xlsb: "application/vnd.ms-excel.dialogsheet"
  },
  macros: {
    /* Macrosheet (Excel 4.0 Macros) */
    xlsx: "application/vnd.ms-excel.macrosheet+xml",
    xlsb: "application/vnd.ms-excel.macrosheet"
  },
  metadata: {
    /* Metadata (Stock/Geography and Dynamic Array) */
    xlsx: "application/vnd.openxmlformats-officedocument.spreadsheetml.sheetMetadata+xml",
    xlsb: "application/vnd.ms-excel.sheetMetadata"
  },
  styles: {
    /* Styles */
    xlsx: "application/vnd.openxmlformats-officedocument.spreadsheetml.styles+xml",
    xlsb: "application/vnd.ms-excel.styles"
  }
};
function new_ct() {
  return {
    workbooks: [],
    sheets: [],
    charts: [],
    dialogs: [],
    macros: [],
    rels: [],
    strs: [],
    comments: [],
    threadedcomments: [],
    links: [],
    coreprops: [],
    extprops: [],
    custprops: [],
    themes: [],
    styles: [],
    calcchains: [],
    vba: [],
    drawings: [],
    metadata: [],
    people: [],
    TODO: [],
    xmlns: ""
  };
}
function parse_ct(data2) {
  var ct2 = new_ct();
  if (!data2 || !data2.match)
    return ct2;
  var ctext = {};
  (data2.match(tagregex) || []).forEach(function(x2) {
    var y2 = parsexmltag(x2);
    switch (y2[0].replace(nsregex, "<")) {
      case "<?xml":
        break;
      case "<Types":
        ct2.xmlns = y2["xmlns" + (y2[0].match(/<(\w+):/) || ["", ""])[1]];
        break;
      case "<Default":
        ctext[y2.Extension] = y2.ContentType;
        break;
      case "<Override":
        if (ct2[ct2type[y2.ContentType]] !== void 0)
          ct2[ct2type[y2.ContentType]].push(y2.PartName);
        break;
    }
  });
  if (ct2.xmlns !== XMLNS$1.CT)
    throw new Error("Unknown Namespace: " + ct2.xmlns);
  ct2.calcchain = ct2.calcchains.length > 0 ? ct2.calcchains[0] : "";
  ct2.sst = ct2.strs.length > 0 ? ct2.strs[0] : "";
  ct2.style = ct2.styles.length > 0 ? ct2.styles[0] : "";
  ct2.defaults = ctext;
  delete ct2.calcchains;
  return ct2;
}
function write_ct(ct2, opts) {
  var type2ct = evert_arr(ct2type);
  var o2 = [], v4;
  o2[o2.length] = XML_HEADER;
  o2[o2.length] = writextag("Types", null, {
    "xmlns": XMLNS$1.CT,
    "xmlns:xsd": XMLNS$1.xsd,
    "xmlns:xsi": XMLNS$1.xsi
  });
  o2 = o2.concat([
    ["xml", "application/xml"],
    ["bin", "application/vnd.ms-excel.sheet.binary.macroEnabled.main"],
    ["vml", "application/vnd.openxmlformats-officedocument.vmlDrawing"],
    ["data", "application/vnd.openxmlformats-officedocument.model+data"],
    /* from test files */
    ["bmp", "image/bmp"],
    ["png", "image/png"],
    ["gif", "image/gif"],
    ["emf", "image/x-emf"],
    ["wmf", "image/x-wmf"],
    ["jpg", "image/jpeg"],
    ["jpeg", "image/jpeg"],
    ["tif", "image/tiff"],
    ["tiff", "image/tiff"],
    ["pdf", "application/pdf"],
    ["rels", "application/vnd.openxmlformats-package.relationships+xml"]
  ].map(function(x2) {
    return writextag("Default", null, { "Extension": x2[0], "ContentType": x2[1] });
  }));
  var f1 = function(w2) {
    if (ct2[w2] && ct2[w2].length > 0) {
      v4 = ct2[w2][0];
      o2[o2.length] = writextag("Override", null, {
        "PartName": (v4[0] == "/" ? "" : "/") + v4,
        "ContentType": CT_LIST[w2][opts.bookType] || CT_LIST[w2]["xlsx"]
      });
    }
  };
  var f2 = function(w2) {
    (ct2[w2] || []).forEach(function(v5) {
      o2[o2.length] = writextag("Override", null, {
        "PartName": (v5[0] == "/" ? "" : "/") + v5,
        "ContentType": CT_LIST[w2][opts.bookType] || CT_LIST[w2]["xlsx"]
      });
    });
  };
  var f3 = function(t3) {
    (ct2[t3] || []).forEach(function(v5) {
      o2[o2.length] = writextag("Override", null, {
        "PartName": (v5[0] == "/" ? "" : "/") + v5,
        "ContentType": type2ct[t3][0]
      });
    });
  };
  f1("workbooks");
  f2("sheets");
  f2("charts");
  f3("themes");
  ["strs", "styles"].forEach(f1);
  ["coreprops", "extprops", "custprops"].forEach(f3);
  f3("vba");
  f3("comments");
  f3("threadedcomments");
  f3("drawings");
  f2("metadata");
  f3("people");
  if (o2.length > 2) {
    o2[o2.length] = "</Types>";
    o2[1] = o2[1].replace("/>", ">");
  }
  return o2.join("");
}
var RELS = {
  WB: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument",
  SHEET: "http://sheetjs.openxmlformats.org/officeDocument/2006/relationships/officeDocument",
  HLINK: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/hyperlink",
  VML: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/vmlDrawing",
  XPATH: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/externalLinkPath",
  XMISS: "http://schemas.microsoft.com/office/2006/relationships/xlExternalLinkPath/xlPathMissing",
  XLINK: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/externalLink",
  CXML: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/customXml",
  CXMLP: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/customXmlProps",
  CMNT: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/comments",
  CORE_PROPS: "http://schemas.openxmlformats.org/package/2006/relationships/metadata/core-properties",
  EXT_PROPS: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/extended-properties",
  CUST_PROPS: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/custom-properties",
  SST: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/sharedStrings",
  STY: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/styles",
  THEME: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/theme",
  CHART: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/chart",
  CHARTEX: "http://schemas.microsoft.com/office/2014/relationships/chartEx",
  CS: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/chartsheet",
  WS: [
    "http://schemas.openxmlformats.org/officeDocument/2006/relationships/worksheet",
    "http://purl.oclc.org/ooxml/officeDocument/relationships/worksheet"
  ],
  DS: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/dialogsheet",
  MS: "http://schemas.microsoft.com/office/2006/relationships/xlMacrosheet",
  IMG: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/image",
  DRAW: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/drawing",
  XLMETA: "http://schemas.openxmlformats.org/officeDocument/2006/relationships/sheetMetadata",
  TCMNT: "http://schemas.microsoft.com/office/2017/10/relationships/threadedComment",
  PEOPLE: "http://schemas.microsoft.com/office/2017/10/relationships/person",
  VBA: "http://schemas.microsoft.com/office/2006/relationships/vbaProject"
};
function get_rels_path(file) {
  var n2 = file.lastIndexOf("/");
  return file.slice(0, n2 + 1) + "_rels/" + file.slice(n2 + 1) + ".rels";
}
function parse_rels(data2, currentFilePath) {
  var rels = { "!id": {} };
  if (!data2)
    return rels;
  if (currentFilePath.charAt(0) !== "/") {
    currentFilePath = "/" + currentFilePath;
  }
  var hash = {};
  (data2.match(tagregex) || []).forEach(function(x2) {
    var y2 = parsexmltag(x2);
    if (y2[0] === "<Relationship") {
      var rel = {};
      rel.Type = y2.Type;
      rel.Target = y2.Target;
      rel.Id = y2.Id;
      if (y2.TargetMode)
        rel.TargetMode = y2.TargetMode;
      var canonictarget = y2.TargetMode === "External" ? y2.Target : resolve_path(y2.Target, currentFilePath);
      rels[canonictarget] = rel;
      hash[y2.Id] = rel;
    }
  });
  rels["!id"] = hash;
  return rels;
}
function write_rels(rels) {
  var o2 = [XML_HEADER, writextag("Relationships", null, {
    //'xmlns:ns0': XMLNS.RELS,
    "xmlns": XMLNS$1.RELS
  })];
  keys$4(rels["!id"]).forEach(function(rid) {
    o2[o2.length] = writextag("Relationship", null, rels["!id"][rid]);
  });
  if (o2.length > 2) {
    o2[o2.length] = "</Relationships>";
    o2[1] = o2[1].replace("/>", ">");
  }
  return o2.join("");
}
function add_rels(rels, rId, f2, type2, relobj, targetmode) {
  if (!relobj)
    relobj = {};
  if (!rels["!id"])
    rels["!id"] = {};
  if (!rels["!idx"])
    rels["!idx"] = 1;
  if (rId < 0)
    for (rId = rels["!idx"]; rels["!id"]["rId" + rId]; ++rId) {
    }
  rels["!idx"] = rId + 1;
  relobj.Id = "rId" + rId;
  relobj.Type = type2;
  relobj.Target = f2;
  if (targetmode)
    relobj.TargetMode = targetmode;
  else if ([RELS.HLINK, RELS.XPATH, RELS.XMISS].indexOf(relobj.Type) > -1)
    relobj.TargetMode = "External";
  if (rels["!id"][relobj.Id])
    throw new Error("Cannot rewrite rId " + rId);
  rels["!id"][relobj.Id] = relobj;
  rels[("/" + relobj.Target).replace("//", "/")] = relobj;
  return rId;
}
var CT_ODS = "application/vnd.oasis.opendocument.spreadsheet";
function parse_manifest(d2, opts) {
  var str = xlml_normalize(d2);
  var Rn;
  var FEtag;
  while (Rn = xlmlregex.exec(str))
    switch (Rn[3]) {
      case "manifest":
        break;
      case "file-entry":
        FEtag = parsexmltag(Rn[0], false);
        if (FEtag.path == "/" && FEtag.type !== CT_ODS)
          throw new Error("This OpenDocument is not a spreadsheet");
        break;
      case "encryption-data":
      case "algorithm":
      case "start-key-generation":
      case "key-derivation":
        throw new Error("Unsupported ODS Encryption");
      default:
        if (opts && opts.WTF)
          throw Rn;
    }
}
function write_manifest(manifest) {
  var o2 = [XML_HEADER];
  o2.push('<manifest:manifest xmlns:manifest="urn:oasis:names:tc:opendocument:xmlns:manifest:1.0" manifest:version="1.2">\n');
  o2.push('  <manifest:file-entry manifest:full-path="/" manifest:version="1.2" manifest:media-type="application/vnd.oasis.opendocument.spreadsheet"/>\n');
  for (var i2 = 0; i2 < manifest.length; ++i2)
    o2.push('  <manifest:file-entry manifest:full-path="' + manifest[i2][0] + '" manifest:media-type="' + manifest[i2][1] + '"/>\n');
  o2.push("</manifest:manifest>");
  return o2.join("");
}
function write_rdf_type(file, res, tag2) {
  return [
    '  <rdf:Description rdf:about="' + file + '">\n',
    '    <rdf:type rdf:resource="http://docs.oasis-open.org/ns/office/1.2/meta/' + (tag2 || "odf") + "#" + res + '"/>\n',
    "  </rdf:Description>\n"
  ].join("");
}
function write_rdf_has(base2, file) {
  return [
    '  <rdf:Description rdf:about="' + base2 + '">\n',
    '    <ns0:hasPart xmlns:ns0="http://docs.oasis-open.org/ns/office/1.2/meta/pkg#" rdf:resource="' + file + '"/>\n',
    "  </rdf:Description>\n"
  ].join("");
}
function write_rdf(rdf) {
  var o2 = [XML_HEADER];
  o2.push('<rdf:RDF xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#">\n');
  for (var i2 = 0; i2 != rdf.length; ++i2) {
    o2.push(write_rdf_type(rdf[i2][0], rdf[i2][1]));
    o2.push(write_rdf_has("", rdf[i2][0]));
  }
  o2.push(write_rdf_type("", "Document", "pkg"));
  o2.push("</rdf:RDF>");
  return o2.join("");
}
function write_meta_ods() {
  return '<office:document-meta xmlns:office="urn:oasis:names:tc:opendocument:xmlns:office:1.0" xmlns:meta="urn:oasis:names:tc:opendocument:xmlns:meta:1.0" xmlns:dc="http://purl.org/dc/elements/1.1/" xmlns:xlink="http://www.w3.org/1999/xlink" office:version="1.2"><office:meta><meta:generator>SheetJS ' + XLSX.version + "</meta:generator></office:meta></office:document-meta>";
}
var CORE_PROPS = [
  ["cp:category", "Category"],
  ["cp:contentStatus", "ContentStatus"],
  ["cp:keywords", "Keywords"],
  ["cp:lastModifiedBy", "LastAuthor"],
  ["cp:lastPrinted", "LastPrinted"],
  ["cp:revision", "RevNumber"],
  ["cp:version", "Version"],
  ["dc:creator", "Author"],
  ["dc:description", "Comments"],
  ["dc:identifier", "Identifier"],
  ["dc:language", "Language"],
  ["dc:subject", "Subject"],
  ["dc:title", "Title"],
  ["dcterms:created", "CreatedDate", "date"],
  ["dcterms:modified", "ModifiedDate", "date"]
];
var CORE_PROPS_REGEX = /* @__PURE__ */ function() {
  var r2 = new Array(CORE_PROPS.length);
  for (var i2 = 0; i2 < CORE_PROPS.length; ++i2) {
    var f2 = CORE_PROPS[i2];
    var g2 = "(?:" + f2[0].slice(0, f2[0].indexOf(":")) + ":)" + f2[0].slice(f2[0].indexOf(":") + 1);
    r2[i2] = new RegExp("<" + g2 + "[^>]*>([\\s\\S]*?)</" + g2 + ">");
  }
  return r2;
}();
function parse_core_props(data2) {
  var p2 = {};
  data2 = utf8read(data2);
  for (var i2 = 0; i2 < CORE_PROPS.length; ++i2) {
    var f2 = CORE_PROPS[i2], cur = data2.match(CORE_PROPS_REGEX[i2]);
    if (cur != null && cur.length > 0)
      p2[f2[1]] = unescapexml(cur[1]);
    if (f2[2] === "date" && p2[f2[1]])
      p2[f2[1]] = parseDate$1(p2[f2[1]]);
  }
  return p2;
}
function cp_doit(f2, g2, h2, o2, p2) {
  if (p2[f2] != null || g2 == null || g2 === "")
    return;
  p2[f2] = g2;
  g2 = escapexml(g2);
  o2[o2.length] = h2 ? writextag(f2, g2, h2) : writetag(f2, g2);
}
function write_core_props(cp, _opts) {
  var opts = _opts || {};
  var o2 = [XML_HEADER, writextag("cp:coreProperties", null, {
    //'xmlns': XMLNS.CORE_PROPS,
    "xmlns:cp": XMLNS$1.CORE_PROPS,
    "xmlns:dc": XMLNS$1.dc,
    "xmlns:dcterms": XMLNS$1.dcterms,
    "xmlns:dcmitype": XMLNS$1.dcmitype,
    "xmlns:xsi": XMLNS$1.xsi
  })], p2 = {};
  if (!cp && !opts.Props)
    return o2.join("");
  if (cp) {
    if (cp.CreatedDate != null)
      cp_doit("dcterms:created", typeof cp.CreatedDate === "string" ? cp.CreatedDate : write_w3cdtf(cp.CreatedDate, opts.WTF), { "xsi:type": "dcterms:W3CDTF" }, o2, p2);
    if (cp.ModifiedDate != null)
      cp_doit("dcterms:modified", typeof cp.ModifiedDate === "string" ? cp.ModifiedDate : write_w3cdtf(cp.ModifiedDate, opts.WTF), { "xsi:type": "dcterms:W3CDTF" }, o2, p2);
  }
  for (var i2 = 0; i2 != CORE_PROPS.length; ++i2) {
    var f2 = CORE_PROPS[i2];
    var v4 = opts.Props && opts.Props[f2[1]] != null ? opts.Props[f2[1]] : cp ? cp[f2[1]] : null;
    if (v4 === true)
      v4 = "1";
    else if (v4 === false)
      v4 = "0";
    else if (typeof v4 == "number")
      v4 = String(v4);
    if (v4 != null)
      cp_doit(f2[0], v4, null, o2, p2);
  }
  if (o2.length > 2) {
    o2[o2.length] = "</cp:coreProperties>";
    o2[1] = o2[1].replace("/>", ">");
  }
  return o2.join("");
}
var EXT_PROPS = [
  ["Application", "Application", "string"],
  ["AppVersion", "AppVersion", "string"],
  ["Company", "Company", "string"],
  ["DocSecurity", "DocSecurity", "string"],
  ["Manager", "Manager", "string"],
  ["HyperlinksChanged", "HyperlinksChanged", "bool"],
  ["SharedDoc", "SharedDoc", "bool"],
  ["LinksUpToDate", "LinksUpToDate", "bool"],
  ["ScaleCrop", "ScaleCrop", "bool"],
  ["HeadingPairs", "HeadingPairs", "raw"],
  ["TitlesOfParts", "TitlesOfParts", "raw"]
];
var PseudoPropsPairs = [
  "Worksheets",
  "SheetNames",
  "NamedRanges",
  "DefinedNames",
  "Chartsheets",
  "ChartNames"
];
function load_props_pairs(HP, TOP, props, opts) {
  var v4 = [];
  if (typeof HP == "string")
    v4 = parseVector(HP, opts);
  else
    for (var j2 = 0; j2 < HP.length; ++j2)
      v4 = v4.concat(HP[j2].map(function(hp) {
        return { v: hp };
      }));
  var parts = typeof TOP == "string" ? parseVector(TOP, opts).map(function(x2) {
    return x2.v;
  }) : TOP;
  var idx = 0, len2 = 0;
  if (parts.length > 0)
    for (var i2 = 0; i2 !== v4.length; i2 += 2) {
      len2 = +v4[i2 + 1].v;
      switch (v4[i2].v) {
        case "Worksheets":
        case "工作表":
        case "Листы":
        case "أوراق العمل":
        case "ワークシート":
        case "גליונות עבודה":
        case "Arbeitsblätter":
        case "Çalışma Sayfaları":
        case "Feuilles de calcul":
        case "Fogli di lavoro":
        case "Folhas de cálculo":
        case "Planilhas":
        case "Regneark":
        case "Hojas de cálculo":
        case "Werkbladen":
          props.Worksheets = len2;
          props.SheetNames = parts.slice(idx, idx + len2);
          break;
        case "Named Ranges":
        case "Rangos con nombre":
        case "名前付き一覧":
        case "Benannte Bereiche":
        case "Navngivne områder":
          props.NamedRanges = len2;
          props.DefinedNames = parts.slice(idx, idx + len2);
          break;
        case "Charts":
        case "Diagramme":
          props.Chartsheets = len2;
          props.ChartNames = parts.slice(idx, idx + len2);
          break;
      }
      idx += len2;
    }
}
function parse_ext_props(data2, p2, opts) {
  var q2 = {};
  if (!p2)
    p2 = {};
  data2 = utf8read(data2);
  EXT_PROPS.forEach(function(f2) {
    var xml2 = (data2.match(matchtag(f2[0])) || [])[1];
    switch (f2[2]) {
      case "string":
        if (xml2)
          p2[f2[1]] = unescapexml(xml2);
        break;
      case "bool":
        p2[f2[1]] = xml2 === "true";
        break;
      case "raw":
        var cur = data2.match(new RegExp("<" + f2[0] + "[^>]*>([\\s\\S]*?)</" + f2[0] + ">"));
        if (cur && cur.length > 0)
          q2[f2[1]] = cur[1];
        break;
    }
  });
  if (q2.HeadingPairs && q2.TitlesOfParts)
    load_props_pairs(q2.HeadingPairs, q2.TitlesOfParts, p2, opts);
  return p2;
}
function write_ext_props(cp) {
  var o2 = [], W2 = writextag;
  if (!cp)
    cp = {};
  cp.Application = "SheetJS";
  o2[o2.length] = XML_HEADER;
  o2[o2.length] = writextag("Properties", null, {
    "xmlns": XMLNS$1.EXT_PROPS,
    "xmlns:vt": XMLNS$1.vt
  });
  EXT_PROPS.forEach(function(f2) {
    if (cp[f2[1]] === void 0)
      return;
    var v4;
    switch (f2[2]) {
      case "string":
        v4 = escapexml(String(cp[f2[1]]));
        break;
      case "bool":
        v4 = cp[f2[1]] ? "true" : "false";
        break;
    }
    if (v4 !== void 0)
      o2[o2.length] = W2(f2[0], v4);
  });
  o2[o2.length] = W2("HeadingPairs", W2("vt:vector", W2("vt:variant", "<vt:lpstr>Worksheets</vt:lpstr>") + W2("vt:variant", W2("vt:i4", String(cp.Worksheets))), { size: 2, baseType: "variant" }));
  o2[o2.length] = W2("TitlesOfParts", W2("vt:vector", cp.SheetNames.map(function(s2) {
    return "<vt:lpstr>" + escapexml(s2) + "</vt:lpstr>";
  }).join(""), { size: cp.Worksheets, baseType: "lpstr" }));
  if (o2.length > 2) {
    o2[o2.length] = "</Properties>";
    o2[1] = o2[1].replace("/>", ">");
  }
  return o2.join("");
}
var custregex = /<[^>]+>[^<]*/g;
function parse_cust_props(data2, opts) {
  var p2 = {}, name = "";
  var m2 = data2.match(custregex);
  if (m2)
    for (var i2 = 0; i2 != m2.length; ++i2) {
      var x2 = m2[i2], y2 = parsexmltag(x2);
      switch (y2[0]) {
        case "<?xml":
          break;
        case "<Properties":
          break;
        case "<property":
          name = unescapexml(y2.name);
          break;
        case "</property>":
          name = null;
          break;
        default:
          if (x2.indexOf("<vt:") === 0) {
            var toks = x2.split(">");
            var type2 = toks[0].slice(4), text2 = toks[1];
            switch (type2) {
              case "lpstr":
              case "bstr":
              case "lpwstr":
                p2[name] = unescapexml(text2);
                break;
              case "bool":
                p2[name] = parsexmlbool(text2);
                break;
              case "i1":
              case "i2":
              case "i4":
              case "i8":
              case "int":
              case "uint":
                p2[name] = parseInt(text2, 10);
                break;
              case "r4":
              case "r8":
              case "decimal":
                p2[name] = parseFloat(text2);
                break;
              case "filetime":
              case "date":
                p2[name] = parseDate$1(text2);
                break;
              case "cy":
              case "error":
                p2[name] = unescapexml(text2);
                break;
              default:
                if (type2.slice(-1) == "/")
                  break;
                if (opts.WTF && typeof console !== "undefined")
                  console.warn("Unexpected", x2, type2, toks);
            }
          } else if (x2.slice(0, 2) === "</")
            ;
          else if (opts.WTF)
            throw new Error(x2);
      }
    }
  return p2;
}
function write_cust_props(cp) {
  var o2 = [XML_HEADER, writextag("Properties", null, {
    "xmlns": XMLNS$1.CUST_PROPS,
    "xmlns:vt": XMLNS$1.vt
  })];
  if (!cp)
    return o2.join("");
  var pid = 1;
  keys$4(cp).forEach(function custprop(k2) {
    ++pid;
    o2[o2.length] = writextag("property", write_vt(cp[k2], true), {
      "fmtid": "{D5CDD505-2E9C-101B-9397-08002B2CF9AE}",
      "pid": pid,
      "name": escapexml(k2)
    });
  });
  if (o2.length > 2) {
    o2[o2.length] = "</Properties>";
    o2[1] = o2[1].replace("/>", ">");
  }
  return o2.join("");
}
var XLMLDocPropsMap = {
  Title: "Title",
  Subject: "Subject",
  Author: "Author",
  Keywords: "Keywords",
  Comments: "Description",
  LastAuthor: "LastAuthor",
  RevNumber: "Revision",
  Application: "AppName",
  /* TotalTime: 'TotalTime', */
  LastPrinted: "LastPrinted",
  CreatedDate: "Created",
  ModifiedDate: "LastSaved",
  /* Pages */
  /* Words */
  /* Characters */
  Category: "Category",
  /* PresentationFormat */
  Manager: "Manager",
  Company: "Company",
  /* Guid */
  /* HyperlinkBase */
  /* Bytes */
  /* Lines */
  /* Paragraphs */
  /* CharactersWithSpaces */
  AppVersion: "Version",
  ContentStatus: "ContentStatus",
  /* NOTE: missing from schema */
  Identifier: "Identifier",
  /* NOTE: missing from schema */
  Language: "Language"
  /* NOTE: missing from schema */
};
var evert_XLMLDPM;
function xlml_set_prop(Props, tag2, val) {
  if (!evert_XLMLDPM)
    evert_XLMLDPM = evert(XLMLDocPropsMap);
  tag2 = evert_XLMLDPM[tag2] || tag2;
  Props[tag2] = val;
}
function xlml_write_docprops(Props, opts) {
  var o2 = [];
  keys$4(XLMLDocPropsMap).map(function(m2) {
    for (var i2 = 0; i2 < CORE_PROPS.length; ++i2)
      if (CORE_PROPS[i2][1] == m2)
        return CORE_PROPS[i2];
    for (i2 = 0; i2 < EXT_PROPS.length; ++i2)
      if (EXT_PROPS[i2][1] == m2)
        return EXT_PROPS[i2];
    throw m2;
  }).forEach(function(p2) {
    if (Props[p2[1]] == null)
      return;
    var m2 = opts && opts.Props && opts.Props[p2[1]] != null ? opts.Props[p2[1]] : Props[p2[1]];
    switch (p2[2]) {
      case "date":
        m2 = new Date(m2).toISOString().replace(/\.\d*Z/, "Z");
        break;
    }
    if (typeof m2 == "number")
      m2 = String(m2);
    else if (m2 === true || m2 === false) {
      m2 = m2 ? "1" : "0";
    } else if (m2 instanceof Date)
      m2 = new Date(m2).toISOString().replace(/\.\d*Z/, "");
    o2.push(writetag(XLMLDocPropsMap[p2[1]] || p2[1], m2));
  });
  return writextag("DocumentProperties", o2.join(""), { xmlns: XLMLNS.o });
}
function xlml_write_custprops(Props, Custprops) {
  var BLACKLIST = ["Worksheets", "SheetNames"];
  var T2 = "CustomDocumentProperties";
  var o2 = [];
  if (Props)
    keys$4(Props).forEach(function(k2) {
      if (!Object.prototype.hasOwnProperty.call(Props, k2))
        return;
      for (var i2 = 0; i2 < CORE_PROPS.length; ++i2)
        if (k2 == CORE_PROPS[i2][1])
          return;
      for (i2 = 0; i2 < EXT_PROPS.length; ++i2)
        if (k2 == EXT_PROPS[i2][1])
          return;
      for (i2 = 0; i2 < BLACKLIST.length; ++i2)
        if (k2 == BLACKLIST[i2])
          return;
      var m2 = Props[k2];
      var t3 = "string";
      if (typeof m2 == "number") {
        t3 = "float";
        m2 = String(m2);
      } else if (m2 === true || m2 === false) {
        t3 = "boolean";
        m2 = m2 ? "1" : "0";
      } else
        m2 = String(m2);
      o2.push(writextag(escapexmltag(k2), m2, { "dt:dt": t3 }));
    });
  if (Custprops)
    keys$4(Custprops).forEach(function(k2) {
      if (!Object.prototype.hasOwnProperty.call(Custprops, k2))
        return;
      if (Props && Object.prototype.hasOwnProperty.call(Props, k2))
        return;
      var m2 = Custprops[k2];
      var t3 = "string";
      if (typeof m2 == "number") {
        t3 = "float";
        m2 = String(m2);
      } else if (m2 === true || m2 === false) {
        t3 = "boolean";
        m2 = m2 ? "1" : "0";
      } else if (m2 instanceof Date) {
        t3 = "dateTime.tz";
        m2 = m2.toISOString();
      } else
        m2 = String(m2);
      o2.push(writextag(escapexmltag(k2), m2, { "dt:dt": t3 }));
    });
  return "<" + T2 + ' xmlns="' + XLMLNS.o + '">' + o2.join("") + "</" + T2 + ">";
}
function parse_FILETIME(blob) {
  var dwLowDateTime = blob.read_shift(4), dwHighDateTime = blob.read_shift(4);
  return new Date((dwHighDateTime / 1e7 * Math.pow(2, 32) + dwLowDateTime / 1e7 - 11644473600) * 1e3).toISOString().replace(/\.000/, "");
}
function write_FILETIME(time2) {
  var date = typeof time2 == "string" ? new Date(Date.parse(time2)) : time2;
  var t3 = date.getTime() / 1e3 + 11644473600;
  var l2 = t3 % Math.pow(2, 32), h2 = (t3 - l2) / Math.pow(2, 32);
  l2 *= 1e7;
  h2 *= 1e7;
  var w2 = l2 / Math.pow(2, 32) | 0;
  if (w2 > 0) {
    l2 = l2 % Math.pow(2, 32);
    h2 += w2;
  }
  var o2 = new_buf(8);
  o2.write_shift(4, l2);
  o2.write_shift(4, h2);
  return o2;
}
function parse_lpstr(blob, type2, pad2) {
  var start2 = blob.l;
  var str = blob.read_shift(0, "lpstr-cp");
  if (pad2)
    while (blob.l - start2 & 3)
      ++blob.l;
  return str;
}
function parse_lpwstr(blob, type2, pad2) {
  var str = blob.read_shift(0, "lpwstr");
  if (pad2)
    blob.l += 4 - (str.length + 1 & 3) & 3;
  return str;
}
function parse_VtStringBase(blob, stringType, pad2) {
  if (stringType === 31)
    return parse_lpwstr(blob);
  return parse_lpstr(blob, stringType, pad2);
}
function parse_VtString(blob, t3, pad2) {
  return parse_VtStringBase(blob, t3, pad2 === false ? 0 : 4);
}
function parse_VtUnalignedString(blob, t3) {
  if (!t3)
    throw new Error("VtUnalignedString must have positive length");
  return parse_VtStringBase(blob, t3, 0);
}
function parse_VtVecLpwstrValue(blob) {
  var length = blob.read_shift(4);
  var ret = [];
  for (var i2 = 0; i2 != length; ++i2) {
    var start2 = blob.l;
    ret[i2] = blob.read_shift(0, "lpwstr").replace(chr0, "");
    if (blob.l - start2 & 2)
      blob.l += 2;
  }
  return ret;
}
function parse_VtVecUnalignedLpstrValue(blob) {
  var length = blob.read_shift(4);
  var ret = [];
  for (var i2 = 0; i2 != length; ++i2)
    ret[i2] = blob.read_shift(0, "lpstr-cp").replace(chr0, "");
  return ret;
}
function parse_VtHeadingPair(blob) {
  var start2 = blob.l;
  var headingString = parse_TypedPropertyValue(blob, VT_USTR);
  if (blob[blob.l] == 0 && blob[blob.l + 1] == 0 && blob.l - start2 & 2)
    blob.l += 2;
  var headerParts = parse_TypedPropertyValue(blob, VT_I4);
  return [headingString, headerParts];
}
function parse_VtVecHeadingPairValue(blob) {
  var cElements = blob.read_shift(4);
  var out2 = [];
  for (var i2 = 0; i2 < cElements / 2; ++i2)
    out2.push(parse_VtHeadingPair(blob));
  return out2;
}
function parse_dictionary(blob, CodePage) {
  var cnt = blob.read_shift(4);
  var dict = {};
  for (var j2 = 0; j2 != cnt; ++j2) {
    var pid = blob.read_shift(4);
    var len2 = blob.read_shift(4);
    dict[pid] = blob.read_shift(len2, CodePage === 1200 ? "utf16le" : "utf8").replace(chr0, "").replace(chr1, "!");
    if (CodePage === 1200 && len2 % 2)
      blob.l += 2;
  }
  if (blob.l & 3)
    blob.l = blob.l >> 2 + 1 << 2;
  return dict;
}
function parse_BLOB(blob) {
  var size = blob.read_shift(4);
  var bytes = blob.slice(blob.l, blob.l + size);
  blob.l += size;
  if ((size & 3) > 0)
    blob.l += 4 - (size & 3) & 3;
  return bytes;
}
function parse_ClipboardData(blob) {
  var o2 = {};
  o2.Size = blob.read_shift(4);
  blob.l += o2.Size + 3 - (o2.Size - 1) % 4;
  return o2;
}
function parse_TypedPropertyValue(blob, type2, _opts) {
  var t3 = blob.read_shift(2), ret, opts = _opts || {};
  blob.l += 2;
  if (type2 !== VT_VARIANT) {
    if (t3 !== type2 && VT_CUSTOM.indexOf(type2) === -1 && !((type2 & 65534) == 4126 && (t3 & 65534) == 4126))
      throw new Error("Expected type " + type2 + " saw " + t3);
  }
  switch (type2 === VT_VARIANT ? t3 : type2) {
    case 2:
      ret = blob.read_shift(2, "i");
      if (!opts.raw)
        blob.l += 2;
      return ret;
    case 3:
      ret = blob.read_shift(4, "i");
      return ret;
    case 11:
      return blob.read_shift(4) !== 0;
    case 19:
      ret = blob.read_shift(4);
      return ret;
    case 30:
      return parse_lpstr(blob, t3, 4).replace(chr0, "");
    case 31:
      return parse_lpwstr(blob);
    case 64:
      return parse_FILETIME(blob);
    case 65:
      return parse_BLOB(blob);
    case 71:
      return parse_ClipboardData(blob);
    case 80:
      return parse_VtString(blob, t3, !opts.raw).replace(chr0, "");
    case 81:
      return parse_VtUnalignedString(
        blob,
        t3
        /*, 4*/
      ).replace(chr0, "");
    case 4108:
      return parse_VtVecHeadingPairValue(blob);
    case 4126:
    case 4127:
      return t3 == 4127 ? parse_VtVecLpwstrValue(blob) : parse_VtVecUnalignedLpstrValue(blob);
    default:
      throw new Error("TypedPropertyValue unrecognized type " + type2 + " " + t3);
  }
}
function write_TypedPropertyValue(type2, value) {
  var o2 = new_buf(4), p2 = new_buf(4);
  o2.write_shift(4, type2 == 80 ? 31 : type2);
  switch (type2) {
    case 3:
      p2.write_shift(-4, value);
      break;
    case 5:
      p2 = new_buf(8);
      p2.write_shift(8, value, "f");
      break;
    case 11:
      p2.write_shift(4, value ? 1 : 0);
      break;
    case 64:
      p2 = write_FILETIME(value);
      break;
    case 31:
    case 80:
      p2 = new_buf(4 + 2 * (value.length + 1) + (value.length % 2 ? 0 : 2));
      p2.write_shift(4, value.length + 1);
      p2.write_shift(0, value, "dbcs");
      while (p2.l != p2.length)
        p2.write_shift(1, 0);
      break;
    default:
      throw new Error("TypedPropertyValue unrecognized type " + type2 + " " + value);
  }
  return bconcat([o2, p2]);
}
function parse_PropertySet(blob, PIDSI) {
  var start_addr = blob.l;
  var size = blob.read_shift(4);
  var NumProps = blob.read_shift(4);
  var Props = [], i2 = 0;
  var CodePage = 0;
  var Dictionary = -1, DictObj = {};
  for (i2 = 0; i2 != NumProps; ++i2) {
    var PropID = blob.read_shift(4);
    var Offset = blob.read_shift(4);
    Props[i2] = [PropID, Offset + start_addr];
  }
  Props.sort(function(x2, y2) {
    return x2[1] - y2[1];
  });
  var PropH = {};
  for (i2 = 0; i2 != NumProps; ++i2) {
    if (blob.l !== Props[i2][1]) {
      var fail = true;
      if (i2 > 0 && PIDSI)
        switch (PIDSI[Props[i2 - 1][0]].t) {
          case 2:
            if (blob.l + 2 === Props[i2][1]) {
              blob.l += 2;
              fail = false;
            }
            break;
          case 80:
            if (blob.l <= Props[i2][1]) {
              blob.l = Props[i2][1];
              fail = false;
            }
            break;
          case 4108:
            if (blob.l <= Props[i2][1]) {
              blob.l = Props[i2][1];
              fail = false;
            }
            break;
        }
      if ((!PIDSI || i2 == 0) && blob.l <= Props[i2][1]) {
        fail = false;
        blob.l = Props[i2][1];
      }
      if (fail)
        throw new Error("Read Error: Expected address " + Props[i2][1] + " at " + blob.l + " :" + i2);
    }
    if (PIDSI) {
      var piddsi = PIDSI[Props[i2][0]];
      PropH[piddsi.n] = parse_TypedPropertyValue(blob, piddsi.t, { raw: true });
      if (piddsi.p === "version")
        PropH[piddsi.n] = String(PropH[piddsi.n] >> 16) + "." + ("0000" + String(PropH[piddsi.n] & 65535)).slice(-4);
      if (piddsi.n == "CodePage")
        switch (PropH[piddsi.n]) {
          case 0:
            PropH[piddsi.n] = 1252;
          case 874:
          case 932:
          case 936:
          case 949:
          case 950:
          case 1250:
          case 1251:
          case 1253:
          case 1254:
          case 1255:
          case 1256:
          case 1257:
          case 1258:
          case 1e4:
          case 1200:
          case 1201:
          case 1252:
          case 65e3:
          case -536:
          case 65001:
          case -535:
            set_cp(CodePage = PropH[piddsi.n] >>> 0 & 65535);
            break;
          default:
            throw new Error("Unsupported CodePage: " + PropH[piddsi.n]);
        }
    } else {
      if (Props[i2][0] === 1) {
        CodePage = PropH.CodePage = parse_TypedPropertyValue(blob, VT_I2);
        set_cp(CodePage);
        if (Dictionary !== -1) {
          var oldpos = blob.l;
          blob.l = Props[Dictionary][1];
          DictObj = parse_dictionary(blob, CodePage);
          blob.l = oldpos;
        }
      } else if (Props[i2][0] === 0) {
        if (CodePage === 0) {
          Dictionary = i2;
          blob.l = Props[i2 + 1][1];
          continue;
        }
        DictObj = parse_dictionary(blob, CodePage);
      } else {
        var name = DictObj[Props[i2][0]];
        var val;
        switch (blob[blob.l]) {
          case 65:
            blob.l += 4;
            val = parse_BLOB(blob);
            break;
          case 30:
            blob.l += 4;
            val = parse_VtString(blob, blob[blob.l - 4]).replace(/\u0000+$/, "");
            break;
          case 31:
            blob.l += 4;
            val = parse_VtString(blob, blob[blob.l - 4]).replace(/\u0000+$/, "");
            break;
          case 3:
            blob.l += 4;
            val = blob.read_shift(4, "i");
            break;
          case 19:
            blob.l += 4;
            val = blob.read_shift(4);
            break;
          case 5:
            blob.l += 4;
            val = blob.read_shift(8, "f");
            break;
          case 11:
            blob.l += 4;
            val = parsebool(blob, 4);
            break;
          case 64:
            blob.l += 4;
            val = parseDate$1(parse_FILETIME(blob));
            break;
          default:
            throw new Error("unparsed value: " + blob[blob.l]);
        }
        PropH[name] = val;
      }
    }
  }
  blob.l = start_addr + size;
  return PropH;
}
var XLSPSSkip = ["CodePage", "Thumbnail", "_PID_LINKBASE", "_PID_HLINKS", "SystemIdentifier", "FMTID"];
function guess_property_type(val) {
  switch (typeof val) {
    case "boolean":
      return 11;
    case "number":
      return (val | 0) == val ? 3 : 5;
    case "string":
      return 31;
    case "object":
      if (val instanceof Date)
        return 64;
      break;
  }
  return -1;
}
function write_PropertySet(entries, RE, PIDSI) {
  var hdr = new_buf(8), piao = [], prop = [];
  var sz = 8, i2 = 0;
  var pr = new_buf(8), pio = new_buf(8);
  pr.write_shift(4, 2);
  pr.write_shift(4, 1200);
  pio.write_shift(4, 1);
  prop.push(pr);
  piao.push(pio);
  sz += 8 + pr.length;
  if (!RE) {
    pio = new_buf(8);
    pio.write_shift(4, 0);
    piao.unshift(pio);
    var bufs = [new_buf(4)];
    bufs[0].write_shift(4, entries.length);
    for (i2 = 0; i2 < entries.length; ++i2) {
      var value = entries[i2][0];
      pr = new_buf(4 + 4 + 2 * (value.length + 1) + (value.length % 2 ? 0 : 2));
      pr.write_shift(4, i2 + 2);
      pr.write_shift(4, value.length + 1);
      pr.write_shift(0, value, "dbcs");
      while (pr.l != pr.length)
        pr.write_shift(1, 0);
      bufs.push(pr);
    }
    pr = bconcat(bufs);
    prop.unshift(pr);
    sz += 8 + pr.length;
  }
  for (i2 = 0; i2 < entries.length; ++i2) {
    if (RE && !RE[entries[i2][0]])
      continue;
    if (XLSPSSkip.indexOf(entries[i2][0]) > -1 || PseudoPropsPairs.indexOf(entries[i2][0]) > -1)
      continue;
    if (entries[i2][1] == null)
      continue;
    var val = entries[i2][1], idx = 0;
    if (RE) {
      idx = +RE[entries[i2][0]];
      var pinfo = PIDSI[idx];
      if (pinfo.p == "version" && typeof val == "string") {
        var arr = val.split(".");
        val = (+arr[0] << 16) + (+arr[1] || 0);
      }
      pr = write_TypedPropertyValue(pinfo.t, val);
    } else {
      var T2 = guess_property_type(val);
      if (T2 == -1) {
        T2 = 31;
        val = String(val);
      }
      pr = write_TypedPropertyValue(T2, val);
    }
    prop.push(pr);
    pio = new_buf(8);
    pio.write_shift(4, !RE ? 2 + i2 : idx);
    piao.push(pio);
    sz += 8 + pr.length;
  }
  var w2 = 8 * (prop.length + 1);
  for (i2 = 0; i2 < prop.length; ++i2) {
    piao[i2].write_shift(4, w2);
    w2 += prop[i2].length;
  }
  hdr.write_shift(4, sz);
  hdr.write_shift(4, prop.length);
  return bconcat([hdr].concat(piao).concat(prop));
}
function parse_PropertySetStream(file, PIDSI, clsid) {
  var blob = file.content;
  if (!blob)
    return {};
  prep_blob(blob, 0);
  var NumSets, FMTID0, FMTID1, Offset0, Offset1 = 0;
  blob.chk("feff", "Byte Order: ");
  blob.read_shift(2);
  var SystemIdentifier = blob.read_shift(4);
  var CLSID = blob.read_shift(16);
  if (CLSID !== CFB.utils.consts.HEADER_CLSID && CLSID !== clsid)
    throw new Error("Bad PropertySet CLSID " + CLSID);
  NumSets = blob.read_shift(4);
  if (NumSets !== 1 && NumSets !== 2)
    throw new Error("Unrecognized #Sets: " + NumSets);
  FMTID0 = blob.read_shift(16);
  Offset0 = blob.read_shift(4);
  if (NumSets === 1 && Offset0 !== blob.l)
    throw new Error("Length mismatch: " + Offset0 + " !== " + blob.l);
  else if (NumSets === 2) {
    FMTID1 = blob.read_shift(16);
    Offset1 = blob.read_shift(4);
  }
  var PSet0 = parse_PropertySet(blob, PIDSI);
  var rval = { SystemIdentifier };
  for (var y2 in PSet0)
    rval[y2] = PSet0[y2];
  rval.FMTID = FMTID0;
  if (NumSets === 1)
    return rval;
  if (Offset1 - blob.l == 2)
    blob.l += 2;
  if (blob.l !== Offset1)
    throw new Error("Length mismatch 2: " + blob.l + " !== " + Offset1);
  var PSet1;
  try {
    PSet1 = parse_PropertySet(blob, null);
  } catch (e2) {
  }
  for (y2 in PSet1)
    rval[y2] = PSet1[y2];
  rval.FMTID = [FMTID0, FMTID1];
  return rval;
}
function write_PropertySetStream(entries, clsid, RE, PIDSI, entries2, clsid2) {
  var hdr = new_buf(entries2 ? 68 : 48);
  var bufs = [hdr];
  hdr.write_shift(2, 65534);
  hdr.write_shift(2, 0);
  hdr.write_shift(4, 842412599);
  hdr.write_shift(16, CFB.utils.consts.HEADER_CLSID, "hex");
  hdr.write_shift(4, entries2 ? 2 : 1);
  hdr.write_shift(16, clsid, "hex");
  hdr.write_shift(4, entries2 ? 68 : 48);
  var ps0 = write_PropertySet(entries, RE, PIDSI);
  bufs.push(ps0);
  if (entries2) {
    var ps1 = write_PropertySet(entries2, null, null);
    hdr.write_shift(16, clsid2, "hex");
    hdr.write_shift(4, 68 + ps0.length);
    bufs.push(ps1);
  }
  return bconcat(bufs);
}
function parsenoop2(blob, length) {
  blob.read_shift(length);
  return null;
}
function writezeroes(n2, o2) {
  if (!o2)
    o2 = new_buf(n2);
  for (var j2 = 0; j2 < n2; ++j2)
    o2.write_shift(1, 0);
  return o2;
}
function parslurp(blob, length, cb) {
  var arr = [], target = blob.l + length;
  while (blob.l < target)
    arr.push(cb(blob, target - blob.l));
  if (target !== blob.l)
    throw new Error("Slurp error");
  return arr;
}
function parsebool(blob, length) {
  return blob.read_shift(length) === 1;
}
function writebool(v4, o2) {
  if (!o2)
    o2 = new_buf(2);
  o2.write_shift(2, +!!v4);
  return o2;
}
function parseuint16(blob) {
  return blob.read_shift(2, "u");
}
function writeuint16(v4, o2) {
  if (!o2)
    o2 = new_buf(2);
  o2.write_shift(2, v4);
  return o2;
}
function parseuint16a(blob, length) {
  return parslurp(blob, length, parseuint16);
}
function parse_Bes(blob) {
  var v4 = blob.read_shift(1), t3 = blob.read_shift(1);
  return t3 === 1 ? v4 : v4 === 1;
}
function write_Bes(v4, t3, o2) {
  if (!o2)
    o2 = new_buf(2);
  o2.write_shift(1, t3 == "e" ? +v4 : +!!v4);
  o2.write_shift(1, t3 == "e" ? 1 : 0);
  return o2;
}
function parse_ShortXLUnicodeString(blob, length, opts) {
  var cch = blob.read_shift(opts && opts.biff >= 12 ? 2 : 1);
  var encoding = "sbcs-cont";
  if (opts && opts.biff >= 8)
    ;
  if (!opts || opts.biff == 8) {
    var fHighByte = blob.read_shift(1);
    if (fHighByte) {
      encoding = "dbcs-cont";
    }
  } else if (opts.biff == 12) {
    encoding = "wstr";
  }
  if (opts.biff >= 2 && opts.biff <= 5)
    encoding = "cpstr";
  var o2 = cch ? blob.read_shift(cch, encoding) : "";
  return o2;
}
function parse_XLUnicodeRichExtendedString(blob) {
  var cch = blob.read_shift(2), flags3 = blob.read_shift(1);
  var fExtSt = flags3 & 4, fRichSt = flags3 & 8;
  var width = 1 + (flags3 & 1);
  var cRun = 0, cbExtRst;
  var z2 = {};
  if (fRichSt)
    cRun = blob.read_shift(2);
  if (fExtSt)
    cbExtRst = blob.read_shift(4);
  var encoding = width == 2 ? "dbcs-cont" : "sbcs-cont";
  var msg = cch === 0 ? "" : blob.read_shift(cch, encoding);
  if (fRichSt)
    blob.l += 4 * cRun;
  if (fExtSt)
    blob.l += cbExtRst;
  z2.t = msg;
  if (!fRichSt) {
    z2.raw = "<t>" + z2.t + "</t>";
    z2.r = z2.t;
  }
  return z2;
}
function write_XLUnicodeRichExtendedString(xlstr) {
  var str = xlstr.t || "";
  var hdr = new_buf(3 + 0);
  hdr.write_shift(2, str.length);
  hdr.write_shift(1, 0 | 1);
  var otext = new_buf(2 * str.length);
  otext.write_shift(2 * str.length, str, "utf16le");
  var out2 = [hdr, otext];
  return bconcat(out2);
}
function parse_XLUnicodeStringNoCch(blob, cch, opts) {
  var retval;
  if (opts) {
    if (opts.biff >= 2 && opts.biff <= 5)
      return blob.read_shift(cch, "cpstr");
    if (opts.biff >= 12)
      return blob.read_shift(cch, "dbcs-cont");
  }
  var fHighByte = blob.read_shift(1);
  if (fHighByte === 0) {
    retval = blob.read_shift(cch, "sbcs-cont");
  } else {
    retval = blob.read_shift(cch, "dbcs-cont");
  }
  return retval;
}
function parse_XLUnicodeString(blob, length, opts) {
  var cch = blob.read_shift(opts && opts.biff == 2 ? 1 : 2);
  if (cch === 0) {
    blob.l++;
    return "";
  }
  return parse_XLUnicodeStringNoCch(blob, cch, opts);
}
function parse_XLUnicodeString2(blob, length, opts) {
  if (opts.biff > 5)
    return parse_XLUnicodeString(blob, length, opts);
  var cch = blob.read_shift(1);
  if (cch === 0) {
    blob.l++;
    return "";
  }
  return blob.read_shift(cch, opts.biff <= 4 || !blob.lens ? "cpstr" : "sbcs-cont");
}
function write_XLUnicodeString(str, opts, o2) {
  if (!o2)
    o2 = new_buf(3 + 2 * str.length);
  o2.write_shift(2, str.length);
  o2.write_shift(1, 1);
  o2.write_shift(31, str, "utf16le");
  return o2;
}
function parse_ControlInfo(blob) {
  var flags3 = blob.read_shift(1);
  blob.l++;
  var accel = blob.read_shift(2);
  blob.l += 2;
  return [flags3, accel];
}
function parse_URLMoniker(blob) {
  var len2 = blob.read_shift(4), start2 = blob.l;
  var extra = false;
  if (len2 > 24) {
    blob.l += len2 - 24;
    if (blob.read_shift(16) === "795881f43b1d7f48af2c825dc4852763")
      extra = true;
    blob.l = start2;
  }
  var url = blob.read_shift((extra ? len2 - 24 : len2) >> 1, "utf16le").replace(chr0, "");
  if (extra)
    blob.l += 24;
  return url;
}
function parse_FileMoniker(blob) {
  var cAnti = blob.read_shift(2);
  var preamble = "";
  while (cAnti-- > 0)
    preamble += "../";
  var ansiPath = blob.read_shift(0, "lpstr-ansi");
  blob.l += 2;
  if (blob.read_shift(2) != 57005)
    throw new Error("Bad FileMoniker");
  var sz = blob.read_shift(4);
  if (sz === 0)
    return preamble + ansiPath.replace(/\\/g, "/");
  var bytes = blob.read_shift(4);
  if (blob.read_shift(2) != 3)
    throw new Error("Bad FileMoniker");
  var unicodePath = blob.read_shift(bytes >> 1, "utf16le").replace(chr0, "");
  return preamble + unicodePath;
}
function parse_HyperlinkMoniker(blob, length) {
  var clsid = blob.read_shift(16);
  switch (clsid) {
    case "e0c9ea79f9bace118c8200aa004ba90b":
      return parse_URLMoniker(blob);
    case "0303000000000000c000000000000046":
      return parse_FileMoniker(blob);
    default:
      throw new Error("Unsupported Moniker " + clsid);
  }
}
function parse_HyperlinkString(blob) {
  var len2 = blob.read_shift(4);
  var o2 = len2 > 0 ? blob.read_shift(len2, "utf16le").replace(chr0, "") : "";
  return o2;
}
function write_HyperlinkString(str, o2) {
  if (!o2)
    o2 = new_buf(6 + str.length * 2);
  o2.write_shift(4, 1 + str.length);
  for (var i2 = 0; i2 < str.length; ++i2)
    o2.write_shift(2, str.charCodeAt(i2));
  o2.write_shift(2, 0);
  return o2;
}
function parse_Hyperlink(blob, length) {
  var end2 = blob.l + length;
  var sVer = blob.read_shift(4);
  if (sVer !== 2)
    throw new Error("Unrecognized streamVersion: " + sVer);
  var flags3 = blob.read_shift(2);
  blob.l += 2;
  var displayName, targetFrameName, moniker, oleMoniker, Loc = "", guid2, fileTime;
  if (flags3 & 16)
    displayName = parse_HyperlinkString(blob, end2 - blob.l);
  if (flags3 & 128)
    targetFrameName = parse_HyperlinkString(blob, end2 - blob.l);
  if ((flags3 & 257) === 257)
    moniker = parse_HyperlinkString(blob, end2 - blob.l);
  if ((flags3 & 257) === 1)
    oleMoniker = parse_HyperlinkMoniker(blob, end2 - blob.l);
  if (flags3 & 8)
    Loc = parse_HyperlinkString(blob, end2 - blob.l);
  if (flags3 & 32)
    guid2 = blob.read_shift(16);
  if (flags3 & 64)
    fileTime = parse_FILETIME(
      blob
      /*, 8*/
    );
  blob.l = end2;
  var target = targetFrameName || moniker || oleMoniker || "";
  if (target && Loc)
    target += "#" + Loc;
  if (!target)
    target = "#" + Loc;
  if (flags3 & 2 && target.charAt(0) == "/" && target.charAt(1) != "/")
    target = "file://" + target;
  var out2 = { Target: target };
  if (guid2)
    out2.guid = guid2;
  if (fileTime)
    out2.time = fileTime;
  if (displayName)
    out2.Tooltip = displayName;
  return out2;
}
function write_Hyperlink(hl) {
  var out2 = new_buf(512), i2 = 0;
  var Target = hl.Target;
  if (Target.slice(0, 7) == "file://")
    Target = Target.slice(7);
  var hashidx = Target.indexOf("#");
  var F2 = hashidx > -1 ? 31 : 23;
  switch (Target.charAt(0)) {
    case "#":
      F2 = 28;
      break;
    case ".":
      F2 &= ~2;
      break;
  }
  out2.write_shift(4, 2);
  out2.write_shift(4, F2);
  var data2 = [8, 6815827, 6619237, 4849780, 83];
  for (i2 = 0; i2 < data2.length; ++i2)
    out2.write_shift(4, data2[i2]);
  if (F2 == 28) {
    Target = Target.slice(1);
    write_HyperlinkString(Target, out2);
  } else if (F2 & 2) {
    data2 = "e0 c9 ea 79 f9 ba ce 11 8c 82 00 aa 00 4b a9 0b".split(" ");
    for (i2 = 0; i2 < data2.length; ++i2)
      out2.write_shift(1, parseInt(data2[i2], 16));
    var Pretarget = hashidx > -1 ? Target.slice(0, hashidx) : Target;
    out2.write_shift(4, 2 * (Pretarget.length + 1));
    for (i2 = 0; i2 < Pretarget.length; ++i2)
      out2.write_shift(2, Pretarget.charCodeAt(i2));
    out2.write_shift(2, 0);
    if (F2 & 8)
      write_HyperlinkString(hashidx > -1 ? Target.slice(hashidx + 1) : "", out2);
  } else {
    data2 = "03 03 00 00 00 00 00 00 c0 00 00 00 00 00 00 46".split(" ");
    for (i2 = 0; i2 < data2.length; ++i2)
      out2.write_shift(1, parseInt(data2[i2], 16));
    var P2 = 0;
    while (Target.slice(P2 * 3, P2 * 3 + 3) == "../" || Target.slice(P2 * 3, P2 * 3 + 3) == "..\\")
      ++P2;
    out2.write_shift(2, P2);
    out2.write_shift(4, Target.length - 3 * P2 + 1);
    for (i2 = 0; i2 < Target.length - 3 * P2; ++i2)
      out2.write_shift(1, Target.charCodeAt(i2 + 3 * P2) & 255);
    out2.write_shift(1, 0);
    out2.write_shift(2, 65535);
    out2.write_shift(2, 57005);
    for (i2 = 0; i2 < 6; ++i2)
      out2.write_shift(4, 0);
  }
  return out2.slice(0, out2.l);
}
function parse_LongRGBA(blob) {
  var r2 = blob.read_shift(1), g2 = blob.read_shift(1), b2 = blob.read_shift(1), a2 = blob.read_shift(1);
  return [r2, g2, b2, a2];
}
function parse_LongRGB(blob, length) {
  var x2 = parse_LongRGBA(blob);
  x2[3] = 0;
  return x2;
}
function parse_XLSCell(blob) {
  var rw = blob.read_shift(2);
  var col = blob.read_shift(2);
  var ixfe = blob.read_shift(2);
  return { r: rw, c: col, ixfe };
}
function write_XLSCell(R2, C2, ixfe, o2) {
  if (!o2)
    o2 = new_buf(6);
  o2.write_shift(2, R2);
  o2.write_shift(2, C2);
  o2.write_shift(2, ixfe || 0);
  return o2;
}
function parse_frtHeader(blob) {
  var rt2 = blob.read_shift(2);
  var flags3 = blob.read_shift(2);
  blob.l += 8;
  return { type: rt2, flags: flags3 };
}
function parse_OptXLUnicodeString(blob, length, opts) {
  return length === 0 ? "" : parse_XLUnicodeString2(blob, length, opts);
}
function parse_XTI(blob, length, opts) {
  var w2 = opts.biff > 8 ? 4 : 2;
  var iSupBook = blob.read_shift(w2), itabFirst = blob.read_shift(w2, "i"), itabLast = blob.read_shift(w2, "i");
  return [iSupBook, itabFirst, itabLast];
}
function parse_RkRec(blob) {
  var ixfe = blob.read_shift(2);
  var RK = parse_RkNumber(blob);
  return [ixfe, RK];
}
function parse_AddinUdf(blob, length, opts) {
  blob.l += 4;
  length -= 4;
  var l2 = blob.l + length;
  var udfName = parse_ShortXLUnicodeString(blob, length, opts);
  var cb = blob.read_shift(2);
  l2 -= blob.l;
  if (cb !== l2)
    throw new Error("Malformed AddinUdf: padding = " + l2 + " != " + cb);
  blob.l += cb;
  return udfName;
}
function parse_Ref8U(blob) {
  var rwFirst = blob.read_shift(2);
  var rwLast = blob.read_shift(2);
  var colFirst = blob.read_shift(2);
  var colLast = blob.read_shift(2);
  return { s: { c: colFirst, r: rwFirst }, e: { c: colLast, r: rwLast } };
}
function write_Ref8U(r2, o2) {
  if (!o2)
    o2 = new_buf(8);
  o2.write_shift(2, r2.s.r);
  o2.write_shift(2, r2.e.r);
  o2.write_shift(2, r2.s.c);
  o2.write_shift(2, r2.e.c);
  return o2;
}
function parse_RefU(blob) {
  var rwFirst = blob.read_shift(2);
  var rwLast = blob.read_shift(2);
  var colFirst = blob.read_shift(1);
  var colLast = blob.read_shift(1);
  return { s: { c: colFirst, r: rwFirst }, e: { c: colLast, r: rwLast } };
}
var parse_Ref = parse_RefU;
function parse_FtCmo(blob) {
  blob.l += 4;
  var ot2 = blob.read_shift(2);
  var id = blob.read_shift(2);
  var flags3 = blob.read_shift(2);
  blob.l += 12;
  return [id, ot2, flags3];
}
function parse_FtNts(blob) {
  var out2 = {};
  blob.l += 4;
  blob.l += 16;
  out2.fSharedNote = blob.read_shift(2);
  blob.l += 4;
  return out2;
}
function parse_FtCf(blob) {
  var out2 = {};
  blob.l += 4;
  blob.cf = blob.read_shift(2);
  return out2;
}
function parse_FtSkip(blob) {
  blob.l += 2;
  blob.l += blob.read_shift(2);
}
var FtTab = {
  /*::[*/
  0: parse_FtSkip,
  /* FtEnd */
  /*::[*/
  4: parse_FtSkip,
  /* FtMacro */
  /*::[*/
  5: parse_FtSkip,
  /* FtButton */
  /*::[*/
  6: parse_FtSkip,
  /* FtGmo */
  /*::[*/
  7: parse_FtCf,
  /* FtCf */
  /*::[*/
  8: parse_FtSkip,
  /* FtPioGrbit */
  /*::[*/
  9: parse_FtSkip,
  /* FtPictFmla */
  /*::[*/
  10: parse_FtSkip,
  /* FtCbls */
  /*::[*/
  11: parse_FtSkip,
  /* FtRbo */
  /*::[*/
  12: parse_FtSkip,
  /* FtSbs */
  /*::[*/
  13: parse_FtNts,
  /* FtNts */
  /*::[*/
  14: parse_FtSkip,
  /* FtSbsFmla */
  /*::[*/
  15: parse_FtSkip,
  /* FtGboData */
  /*::[*/
  16: parse_FtSkip,
  /* FtEdoData */
  /*::[*/
  17: parse_FtSkip,
  /* FtRboData */
  /*::[*/
  18: parse_FtSkip,
  /* FtCblsData */
  /*::[*/
  19: parse_FtSkip,
  /* FtLbsData */
  /*::[*/
  20: parse_FtSkip,
  /* FtCblsFmla */
  /*::[*/
  21: parse_FtCmo
};
function parse_FtArray(blob, length) {
  var tgt = blob.l + length;
  var fts = [];
  while (blob.l < tgt) {
    var ft2 = blob.read_shift(2);
    blob.l -= 2;
    try {
      fts.push(FtTab[ft2](blob, tgt - blob.l));
    } catch (e2) {
      blob.l = tgt;
      return fts;
    }
  }
  if (blob.l != tgt)
    blob.l = tgt;
  return fts;
}
function parse_BOF(blob, length) {
  var o2 = { BIFFVer: 0, dt: 0 };
  o2.BIFFVer = blob.read_shift(2);
  length -= 2;
  if (length >= 2) {
    o2.dt = blob.read_shift(2);
    blob.l -= 2;
  }
  switch (o2.BIFFVer) {
    case 1536:
    case 1280:
    case 1024:
    case 768:
    case 512:
    case 2:
    case 7:
      break;
    default:
      if (length > 6)
        throw new Error("Unexpected BIFF Ver " + o2.BIFFVer);
  }
  blob.read_shift(length);
  return o2;
}
function write_BOF(wb, t3, o2) {
  var h2 = 1536, w2 = 16;
  switch (o2.bookType) {
    case "biff8":
      break;
    case "biff5":
      h2 = 1280;
      w2 = 8;
      break;
    case "biff4":
      h2 = 4;
      w2 = 6;
      break;
    case "biff3":
      h2 = 3;
      w2 = 6;
      break;
    case "biff2":
      h2 = 2;
      w2 = 4;
      break;
    case "xla":
      break;
    default:
      throw new Error("unsupported BIFF version");
  }
  var out2 = new_buf(w2);
  out2.write_shift(2, h2);
  out2.write_shift(2, t3);
  if (w2 > 4)
    out2.write_shift(2, 29282);
  if (w2 > 6)
    out2.write_shift(2, 1997);
  if (w2 > 8) {
    out2.write_shift(2, 49161);
    out2.write_shift(2, 1);
    out2.write_shift(2, 1798);
    out2.write_shift(2, 0);
  }
  return out2;
}
function parse_InterfaceHdr(blob, length) {
  if (length === 0)
    return 1200;
  if (blob.read_shift(2) !== 1200)
    ;
  return 1200;
}
function parse_WriteAccess(blob, length, opts) {
  if (opts.enc) {
    blob.l += length;
    return "";
  }
  var l2 = blob.l;
  var UserName = parse_XLUnicodeString2(blob, 0, opts);
  blob.read_shift(length + l2 - blob.l);
  return UserName;
}
function write_WriteAccess(s2, opts) {
  var b8 = !opts || opts.biff == 8;
  var o2 = new_buf(b8 ? 112 : 54);
  o2.write_shift(opts.biff == 8 ? 2 : 1, 7);
  if (b8)
    o2.write_shift(1, 0);
  o2.write_shift(4, 859007059);
  o2.write_shift(4, 5458548 | (b8 ? 0 : 536870912));
  while (o2.l < o2.length)
    o2.write_shift(1, b8 ? 0 : 32);
  return o2;
}
function parse_WsBool(blob, length, opts) {
  var flags3 = opts && opts.biff == 8 || length == 2 ? blob.read_shift(2) : (blob.l += length, 0);
  return { fDialog: flags3 & 16, fBelow: flags3 & 64, fRight: flags3 & 128 };
}
function parse_BoundSheet8(blob, length, opts) {
  var pos = blob.read_shift(4);
  var hidden = blob.read_shift(1) & 3;
  var dt2 = blob.read_shift(1);
  switch (dt2) {
    case 0:
      dt2 = "Worksheet";
      break;
    case 1:
      dt2 = "Macrosheet";
      break;
    case 2:
      dt2 = "Chartsheet";
      break;
    case 6:
      dt2 = "VBAModule";
      break;
  }
  var name = parse_ShortXLUnicodeString(blob, 0, opts);
  if (name.length === 0)
    name = "Sheet1";
  return { pos, hs: hidden, dt: dt2, name };
}
function write_BoundSheet8(data2, opts) {
  var w2 = !opts || opts.biff >= 8 ? 2 : 1;
  var o2 = new_buf(8 + w2 * data2.name.length);
  o2.write_shift(4, data2.pos);
  o2.write_shift(1, data2.hs || 0);
  o2.write_shift(1, data2.dt);
  o2.write_shift(1, data2.name.length);
  if (opts.biff >= 8)
    o2.write_shift(1, 1);
  o2.write_shift(w2 * data2.name.length, data2.name, opts.biff < 8 ? "sbcs" : "utf16le");
  var out2 = o2.slice(0, o2.l);
  out2.l = o2.l;
  return out2;
}
function parse_SST(blob, length) {
  var end2 = blob.l + length;
  var cnt = blob.read_shift(4);
  var ucnt = blob.read_shift(4);
  var strs2 = [];
  for (var i2 = 0; i2 != ucnt && blob.l < end2; ++i2) {
    strs2.push(parse_XLUnicodeRichExtendedString(blob));
  }
  strs2.Count = cnt;
  strs2.Unique = ucnt;
  return strs2;
}
function write_SST(sst, opts) {
  var header = new_buf(8);
  header.write_shift(4, sst.Count);
  header.write_shift(4, sst.Unique);
  var strs2 = [];
  for (var j2 = 0; j2 < sst.length; ++j2)
    strs2[j2] = write_XLUnicodeRichExtendedString(sst[j2]);
  var o2 = bconcat([header].concat(strs2));
  o2.parts = [header.length].concat(strs2.map(function(str) {
    return str.length;
  }));
  return o2;
}
function parse_ExtSST(blob, length) {
  var extsst = {};
  extsst.dsst = blob.read_shift(2);
  blob.l += length - 2;
  return extsst;
}
function parse_Row(blob) {
  var z2 = {};
  z2.r = blob.read_shift(2);
  z2.c = blob.read_shift(2);
  z2.cnt = blob.read_shift(2) - z2.c;
  var miyRw = blob.read_shift(2);
  blob.l += 4;
  var flags3 = blob.read_shift(1);
  blob.l += 3;
  if (flags3 & 7)
    z2.level = flags3 & 7;
  if (flags3 & 32)
    z2.hidden = true;
  if (flags3 & 64)
    z2.hpt = miyRw / 20;
  return z2;
}
function parse_ForceFullCalculation(blob) {
  var header = parse_frtHeader(blob);
  if (header.type != 2211)
    throw new Error("Invalid Future Record " + header.type);
  var fullcalc = blob.read_shift(4);
  return fullcalc !== 0;
}
function parse_RecalcId(blob) {
  blob.read_shift(2);
  return blob.read_shift(4);
}
function parse_DefaultRowHeight(blob, length, opts) {
  var f2 = 0;
  if (!(opts && opts.biff == 2)) {
    f2 = blob.read_shift(2);
  }
  var miyRw = blob.read_shift(2);
  if (opts && opts.biff == 2) {
    f2 = 1 - (miyRw >> 15);
    miyRw &= 32767;
  }
  var fl2 = { Unsynced: f2 & 1, DyZero: (f2 & 2) >> 1, ExAsc: (f2 & 4) >> 2, ExDsc: (f2 & 8) >> 3 };
  return [fl2, miyRw];
}
function parse_Window1(blob) {
  var xWn = blob.read_shift(2), yWn = blob.read_shift(2), dxWn = blob.read_shift(2), dyWn = blob.read_shift(2);
  var flags3 = blob.read_shift(2), iTabCur = blob.read_shift(2), iTabFirst = blob.read_shift(2);
  var ctabSel = blob.read_shift(2), wTabRatio = blob.read_shift(2);
  return {
    Pos: [xWn, yWn],
    Dim: [dxWn, dyWn],
    Flags: flags3,
    CurTab: iTabCur,
    FirstTab: iTabFirst,
    Selected: ctabSel,
    TabRatio: wTabRatio
  };
}
function write_Window1() {
  var o2 = new_buf(18);
  o2.write_shift(2, 0);
  o2.write_shift(2, 0);
  o2.write_shift(2, 29280);
  o2.write_shift(2, 17600);
  o2.write_shift(2, 56);
  o2.write_shift(2, 0);
  o2.write_shift(2, 0);
  o2.write_shift(2, 1);
  o2.write_shift(2, 500);
  return o2;
}
function parse_Window2(blob, length, opts) {
  if (opts && opts.biff >= 2 && opts.biff < 5)
    return {};
  var f2 = blob.read_shift(2);
  return { RTL: f2 & 64 };
}
function write_Window2(view) {
  var o2 = new_buf(18), f2 = 1718;
  if (view && view.RTL)
    f2 |= 64;
  o2.write_shift(2, f2);
  o2.write_shift(4, 0);
  o2.write_shift(4, 64);
  o2.write_shift(4, 0);
  o2.write_shift(4, 0);
  return o2;
}
function parse_Pane() {
}
function parse_Font(blob, length, opts) {
  var o2 = {
    dyHeight: blob.read_shift(2),
    fl: blob.read_shift(2)
  };
  switch (opts && opts.biff || 8) {
    case 2:
      break;
    case 3:
    case 4:
      blob.l += 2;
      break;
    default:
      blob.l += 10;
      break;
  }
  o2.name = parse_ShortXLUnicodeString(blob, 0, opts);
  return o2;
}
function write_Font(data2, opts) {
  var name = data2.name || "Arial";
  var b5 = opts && opts.biff == 5, w2 = b5 ? 15 + name.length : 16 + 2 * name.length;
  var o2 = new_buf(w2);
  o2.write_shift(2, (data2.sz || 12) * 20);
  o2.write_shift(4, 0);
  o2.write_shift(2, 400);
  o2.write_shift(4, 0);
  o2.write_shift(2, 0);
  o2.write_shift(1, name.length);
  if (!b5)
    o2.write_shift(1, 1);
  o2.write_shift((b5 ? 1 : 2) * name.length, name, b5 ? "sbcs" : "utf16le");
  return o2;
}
function parse_LabelSst(blob) {
  var cell = parse_XLSCell(blob);
  cell.isst = blob.read_shift(4);
  return cell;
}
function write_LabelSst(R2, C2, v4, os) {
  var o2 = new_buf(10);
  write_XLSCell(R2, C2, os, o2);
  o2.write_shift(4, v4);
  return o2;
}
function parse_Label(blob, length, opts) {
  if (opts.biffguess && opts.biff == 2)
    opts.biff = 5;
  var target = blob.l + length;
  var cell = parse_XLSCell(blob);
  if (opts.biff == 2)
    blob.l++;
  var str = parse_XLUnicodeString(blob, target - blob.l, opts);
  cell.val = str;
  return cell;
}
function write_Label(R2, C2, v4, os, opts) {
  var b8 = !opts || opts.biff == 8;
  var o2 = new_buf(6 + 2 + +b8 + (1 + b8) * v4.length);
  write_XLSCell(R2, C2, os, o2);
  o2.write_shift(2, v4.length);
  if (b8)
    o2.write_shift(1, 1);
  o2.write_shift((1 + b8) * v4.length, v4, b8 ? "utf16le" : "sbcs");
  return o2;
}
function parse_Format(blob, length, opts) {
  var numFmtId = blob.read_shift(2);
  var fmtstr = parse_XLUnicodeString2(blob, 0, opts);
  return [numFmtId, fmtstr];
}
function write_Format(i2, f2, opts, o2) {
  var b5 = opts && opts.biff == 5;
  if (!o2)
    o2 = new_buf(b5 ? 3 + f2.length : 5 + 2 * f2.length);
  o2.write_shift(2, i2);
  o2.write_shift(b5 ? 1 : 2, f2.length);
  if (!b5)
    o2.write_shift(1, 1);
  o2.write_shift((b5 ? 1 : 2) * f2.length, f2, b5 ? "sbcs" : "utf16le");
  var out2 = o2.length > o2.l ? o2.slice(0, o2.l) : o2;
  if (out2.l == null)
    out2.l = out2.length;
  return out2;
}
var parse_BIFF2Format = parse_XLUnicodeString2;
function parse_Dimensions(blob, length, opts) {
  var end2 = blob.l + length;
  var w2 = opts.biff == 8 || !opts.biff ? 4 : 2;
  var r2 = blob.read_shift(w2), R2 = blob.read_shift(w2);
  var c2 = blob.read_shift(2), C2 = blob.read_shift(2);
  blob.l = end2;
  return { s: { r: r2, c: c2 }, e: { r: R2, c: C2 } };
}
function write_Dimensions(range2, opts) {
  var w2 = opts.biff == 8 || !opts.biff ? 4 : 2;
  var o2 = new_buf(2 * w2 + 6);
  o2.write_shift(w2, range2.s.r);
  o2.write_shift(w2, range2.e.r + 1);
  o2.write_shift(2, range2.s.c);
  o2.write_shift(2, range2.e.c + 1);
  o2.write_shift(2, 0);
  return o2;
}
function parse_RK(blob) {
  var rw = blob.read_shift(2), col = blob.read_shift(2);
  var rkrec = parse_RkRec(blob);
  return { r: rw, c: col, ixfe: rkrec[0], rknum: rkrec[1] };
}
function parse_MulRk(blob, length) {
  var target = blob.l + length - 2;
  var rw = blob.read_shift(2), col = blob.read_shift(2);
  var rkrecs = [];
  while (blob.l < target)
    rkrecs.push(parse_RkRec(blob));
  if (blob.l !== target)
    throw new Error("MulRK read error");
  var lastcol = blob.read_shift(2);
  if (rkrecs.length != lastcol - col + 1)
    throw new Error("MulRK length mismatch");
  return { r: rw, c: col, C: lastcol, rkrec: rkrecs };
}
function parse_MulBlank(blob, length) {
  var target = blob.l + length - 2;
  var rw = blob.read_shift(2), col = blob.read_shift(2);
  var ixfes = [];
  while (blob.l < target)
    ixfes.push(blob.read_shift(2));
  if (blob.l !== target)
    throw new Error("MulBlank read error");
  var lastcol = blob.read_shift(2);
  if (ixfes.length != lastcol - col + 1)
    throw new Error("MulBlank length mismatch");
  return { r: rw, c: col, C: lastcol, ixfe: ixfes };
}
function parse_CellStyleXF(blob, length, style, opts) {
  var o2 = {};
  var a2 = blob.read_shift(4), b2 = blob.read_shift(4);
  var c2 = blob.read_shift(4), d2 = blob.read_shift(2);
  o2.patternType = XLSFillPattern[c2 >> 26];
  if (!opts.cellStyles)
    return o2;
  o2.alc = a2 & 7;
  o2.fWrap = a2 >> 3 & 1;
  o2.alcV = a2 >> 4 & 7;
  o2.fJustLast = a2 >> 7 & 1;
  o2.trot = a2 >> 8 & 255;
  o2.cIndent = a2 >> 16 & 15;
  o2.fShrinkToFit = a2 >> 20 & 1;
  o2.iReadOrder = a2 >> 22 & 2;
  o2.fAtrNum = a2 >> 26 & 1;
  o2.fAtrFnt = a2 >> 27 & 1;
  o2.fAtrAlc = a2 >> 28 & 1;
  o2.fAtrBdr = a2 >> 29 & 1;
  o2.fAtrPat = a2 >> 30 & 1;
  o2.fAtrProt = a2 >> 31 & 1;
  o2.dgLeft = b2 & 15;
  o2.dgRight = b2 >> 4 & 15;
  o2.dgTop = b2 >> 8 & 15;
  o2.dgBottom = b2 >> 12 & 15;
  o2.icvLeft = b2 >> 16 & 127;
  o2.icvRight = b2 >> 23 & 127;
  o2.grbitDiag = b2 >> 30 & 3;
  o2.icvTop = c2 & 127;
  o2.icvBottom = c2 >> 7 & 127;
  o2.icvDiag = c2 >> 14 & 127;
  o2.dgDiag = c2 >> 21 & 15;
  o2.icvFore = d2 & 127;
  o2.icvBack = d2 >> 7 & 127;
  o2.fsxButton = d2 >> 14 & 1;
  return o2;
}
function parse_XF(blob, length, opts) {
  var o2 = {};
  o2.ifnt = blob.read_shift(2);
  o2.numFmtId = blob.read_shift(2);
  o2.flags = blob.read_shift(2);
  o2.fStyle = o2.flags >> 2 & 1;
  length -= 6;
  o2.data = parse_CellStyleXF(blob, length, o2.fStyle, opts);
  return o2;
}
function write_XF(data2, ixfeP, opts, o2) {
  var b5 = opts && opts.biff == 5;
  if (!o2)
    o2 = new_buf(b5 ? 16 : 20);
  o2.write_shift(2, 0);
  if (data2.style) {
    o2.write_shift(2, data2.numFmtId || 0);
    o2.write_shift(2, 65524);
  } else {
    o2.write_shift(2, data2.numFmtId || 0);
    o2.write_shift(2, ixfeP << 4);
  }
  var f2 = 0;
  if (data2.numFmtId > 0 && b5)
    f2 |= 1024;
  o2.write_shift(4, f2);
  o2.write_shift(4, 0);
  if (!b5)
    o2.write_shift(4, 0);
  o2.write_shift(2, 0);
  return o2;
}
function parse_Guts(blob) {
  blob.l += 4;
  var out2 = [blob.read_shift(2), blob.read_shift(2)];
  if (out2[0] !== 0)
    out2[0]--;
  if (out2[1] !== 0)
    out2[1]--;
  if (out2[0] > 7 || out2[1] > 7)
    throw new Error("Bad Gutters: " + out2.join("|"));
  return out2;
}
function write_Guts(guts) {
  var o2 = new_buf(8);
  o2.write_shift(4, 0);
  o2.write_shift(2, guts[0] ? guts[0] + 1 : 0);
  o2.write_shift(2, guts[1] ? guts[1] + 1 : 0);
  return o2;
}
function parse_BoolErr(blob, length, opts) {
  var cell = parse_XLSCell(blob);
  if (opts.biff == 2 || length == 9)
    ++blob.l;
  var val = parse_Bes(blob);
  cell.val = val;
  cell.t = val === true || val === false ? "b" : "e";
  return cell;
}
function write_BoolErr(R2, C2, v4, os, opts, t3) {
  var o2 = new_buf(8);
  write_XLSCell(R2, C2, os, o2);
  write_Bes(v4, t3, o2);
  return o2;
}
function parse_Number(blob, length, opts) {
  if (opts.biffguess && opts.biff == 2)
    opts.biff = 5;
  var cell = parse_XLSCell(blob);
  var xnum = parse_Xnum(blob);
  cell.val = xnum;
  return cell;
}
function write_Number(R2, C2, v4, os) {
  var o2 = new_buf(14);
  write_XLSCell(R2, C2, os, o2);
  write_Xnum(v4, o2);
  return o2;
}
var parse_XLHeaderFooter = parse_OptXLUnicodeString;
function parse_SupBook(blob, length, opts) {
  var end2 = blob.l + length;
  var ctab = blob.read_shift(2);
  var cch = blob.read_shift(2);
  opts.sbcch = cch;
  if (cch == 1025 || cch == 14849)
    return [cch, ctab];
  if (cch < 1 || cch > 255)
    throw new Error("Unexpected SupBook type: " + cch);
  var virtPath = parse_XLUnicodeStringNoCch(blob, cch);
  var rgst = [];
  while (end2 > blob.l)
    rgst.push(parse_XLUnicodeString(blob));
  return [cch, ctab, virtPath, rgst];
}
function parse_ExternName(blob, length, opts) {
  var flags3 = blob.read_shift(2);
  var body;
  var o2 = {
    fBuiltIn: flags3 & 1,
    fWantAdvise: flags3 >>> 1 & 1,
    fWantPict: flags3 >>> 2 & 1,
    fOle: flags3 >>> 3 & 1,
    fOleLink: flags3 >>> 4 & 1,
    cf: flags3 >>> 5 & 1023,
    fIcon: flags3 >>> 15 & 1
  };
  if (opts.sbcch === 14849)
    body = parse_AddinUdf(blob, length - 2, opts);
  o2.body = body || blob.read_shift(length - 2);
  if (typeof body === "string")
    o2.Name = body;
  return o2;
}
var XLSLblBuiltIn = [
  "_xlnm.Consolidate_Area",
  "_xlnm.Auto_Open",
  "_xlnm.Auto_Close",
  "_xlnm.Extract",
  "_xlnm.Database",
  "_xlnm.Criteria",
  "_xlnm.Print_Area",
  "_xlnm.Print_Titles",
  "_xlnm.Recorder",
  "_xlnm.Data_Form",
  "_xlnm.Auto_Activate",
  "_xlnm.Auto_Deactivate",
  "_xlnm.Sheet_Title",
  "_xlnm._FilterDatabase"
];
function parse_Lbl(blob, length, opts) {
  var target = blob.l + length;
  var flags3 = blob.read_shift(2);
  var chKey = blob.read_shift(1);
  var cch = blob.read_shift(1);
  var cce = blob.read_shift(opts && opts.biff == 2 ? 1 : 2);
  var itab = 0;
  if (!opts || opts.biff >= 5) {
    if (opts.biff != 5)
      blob.l += 2;
    itab = blob.read_shift(2);
    if (opts.biff == 5)
      blob.l += 2;
    blob.l += 4;
  }
  var name = parse_XLUnicodeStringNoCch(blob, cch, opts);
  if (flags3 & 32)
    name = XLSLblBuiltIn[name.charCodeAt(0)];
  var npflen = target - blob.l;
  if (opts && opts.biff == 2)
    --npflen;
  var rgce = target == blob.l || cce === 0 || !(npflen > 0) ? [] : parse_NameParsedFormula(blob, npflen, opts, cce);
  return {
    chKey,
    Name: name,
    itab,
    rgce
  };
}
function parse_ExternSheet(blob, length, opts) {
  if (opts.biff < 8)
    return parse_BIFF5ExternSheet(blob, length, opts);
  var o2 = [], target = blob.l + length, len2 = blob.read_shift(opts.biff > 8 ? 4 : 2);
  while (len2-- !== 0)
    o2.push(parse_XTI(blob, opts.biff > 8 ? 12 : 6, opts));
  if (blob.l != target)
    throw new Error("Bad ExternSheet: " + blob.l + " != " + target);
  return o2;
}
function parse_BIFF5ExternSheet(blob, length, opts) {
  if (blob[blob.l + 1] == 3)
    blob[blob.l]++;
  var o2 = parse_ShortXLUnicodeString(blob, length, opts);
  return o2.charCodeAt(0) == 3 ? o2.slice(1) : o2;
}
function parse_NameCmt(blob, length, opts) {
  if (opts.biff < 8) {
    blob.l += length;
    return;
  }
  var cchName = blob.read_shift(2);
  var cchComment = blob.read_shift(2);
  var name = parse_XLUnicodeStringNoCch(blob, cchName, opts);
  var comment = parse_XLUnicodeStringNoCch(blob, cchComment, opts);
  return [name, comment];
}
function parse_ShrFmla(blob, length, opts) {
  var ref = parse_RefU(blob);
  blob.l++;
  var cUse = blob.read_shift(1);
  length -= 8;
  return [parse_SharedParsedFormula(blob, length, opts), cUse, ref];
}
function parse_Array(blob, length, opts) {
  var ref = parse_Ref(blob);
  switch (opts.biff) {
    case 2:
      blob.l++;
      length -= 7;
      break;
    case 3:
    case 4:
      blob.l += 2;
      length -= 8;
      break;
    default:
      blob.l += 6;
      length -= 12;
  }
  return [ref, parse_ArrayParsedFormula(blob, length, opts)];
}
function parse_MTRSettings(blob) {
  var fMTREnabled = blob.read_shift(4) !== 0;
  var fUserSetThreadCount = blob.read_shift(4) !== 0;
  var cUserThreadCount = blob.read_shift(4);
  return [fMTREnabled, fUserSetThreadCount, cUserThreadCount];
}
function parse_NoteSh(blob, length, opts) {
  if (opts.biff < 8)
    return;
  var row = blob.read_shift(2), col = blob.read_shift(2);
  var flags3 = blob.read_shift(2), idObj = blob.read_shift(2);
  var stAuthor = parse_XLUnicodeString2(blob, 0, opts);
  if (opts.biff < 8)
    blob.read_shift(1);
  return [{ r: row, c: col }, stAuthor, idObj, flags3];
}
function parse_Note(blob, length, opts) {
  return parse_NoteSh(blob, length, opts);
}
function parse_MergeCells(blob, length) {
  var merges = [];
  var cmcs = blob.read_shift(2);
  while (cmcs--)
    merges.push(parse_Ref8U(blob));
  return merges;
}
function write_MergeCells(merges) {
  var o2 = new_buf(2 + merges.length * 8);
  o2.write_shift(2, merges.length);
  for (var i2 = 0; i2 < merges.length; ++i2)
    write_Ref8U(merges[i2], o2);
  return o2;
}
function parse_Obj(blob, length, opts) {
  if (opts && opts.biff < 8)
    return parse_BIFF5Obj(blob, length, opts);
  var cmo = parse_FtCmo(blob);
  var fts = parse_FtArray(blob, length - 22, cmo[1]);
  return { cmo, ft: fts };
}
var parse_BIFF5OT = {
  8: function(blob, length) {
    var tgt = blob.l + length;
    blob.l += 10;
    var cf = blob.read_shift(2);
    blob.l += 4;
    blob.l += 2;
    blob.l += 2;
    blob.l += 2;
    blob.l += 4;
    var cchName = blob.read_shift(1);
    blob.l += cchName;
    blob.l = tgt;
    return { fmt: cf };
  }
};
function parse_BIFF5Obj(blob, length, opts) {
  blob.l += 4;
  var ot2 = blob.read_shift(2);
  var id = blob.read_shift(2);
  var grbit = blob.read_shift(2);
  blob.l += 2;
  blob.l += 2;
  blob.l += 2;
  blob.l += 2;
  blob.l += 2;
  blob.l += 2;
  blob.l += 2;
  blob.l += 2;
  blob.l += 2;
  blob.l += 6;
  length -= 36;
  var fts = [];
  fts.push((parse_BIFF5OT[ot2] || parsenoop)(blob, length, opts));
  return { cmo: [id, ot2, grbit], ft: fts };
}
function parse_TxO(blob, length, opts) {
  var s2 = blob.l;
  var texts = "";
  try {
    blob.l += 4;
    var ot2 = (opts.lastobj || { cmo: [0, 0] }).cmo[1];
    var controlInfo;
    if ([0, 5, 7, 11, 12, 14].indexOf(ot2) == -1)
      blob.l += 6;
    else
      controlInfo = parse_ControlInfo(blob, 6, opts);
    var cchText = blob.read_shift(2);
    blob.read_shift(2);
    parseuint16(blob, 2);
    var len2 = blob.read_shift(2);
    blob.l += len2;
    for (var i2 = 1; i2 < blob.lens.length - 1; ++i2) {
      if (blob.l - s2 != blob.lens[i2])
        throw new Error("TxO: bad continue record");
      var hdr = blob[blob.l];
      var t3 = parse_XLUnicodeStringNoCch(blob, blob.lens[i2 + 1] - blob.lens[i2] - 1);
      texts += t3;
      if (texts.length >= (hdr ? cchText : 2 * cchText))
        break;
    }
    if (texts.length !== cchText && texts.length !== cchText * 2) {
      throw new Error("cchText: " + cchText + " != " + texts.length);
    }
    blob.l = s2 + length;
    return { t: texts };
  } catch (e2) {
    blob.l = s2 + length;
    return { t: texts };
  }
}
function parse_HLink(blob, length) {
  var ref = parse_Ref8U(blob);
  blob.l += 16;
  var hlink = parse_Hyperlink(blob, length - 24);
  return [ref, hlink];
}
function write_HLink(hl) {
  var O2 = new_buf(24);
  var ref = decode_cell(hl[0]);
  O2.write_shift(2, ref.r);
  O2.write_shift(2, ref.r);
  O2.write_shift(2, ref.c);
  O2.write_shift(2, ref.c);
  var clsid = "d0 c9 ea 79 f9 ba ce 11 8c 82 00 aa 00 4b a9 0b".split(" ");
  for (var i2 = 0; i2 < 16; ++i2)
    O2.write_shift(1, parseInt(clsid[i2], 16));
  return bconcat([O2, write_Hyperlink(hl[1])]);
}
function parse_HLinkTooltip(blob, length) {
  blob.read_shift(2);
  var ref = parse_Ref8U(blob);
  var wzTooltip = blob.read_shift((length - 10) / 2, "dbcs-cont");
  wzTooltip = wzTooltip.replace(chr0, "");
  return [ref, wzTooltip];
}
function write_HLinkTooltip(hl) {
  var TT = hl[1].Tooltip;
  var O2 = new_buf(10 + 2 * (TT.length + 1));
  O2.write_shift(2, 2048);
  var ref = decode_cell(hl[0]);
  O2.write_shift(2, ref.r);
  O2.write_shift(2, ref.r);
  O2.write_shift(2, ref.c);
  O2.write_shift(2, ref.c);
  for (var i2 = 0; i2 < TT.length; ++i2)
    O2.write_shift(2, TT.charCodeAt(i2));
  O2.write_shift(2, 0);
  return O2;
}
function parse_Country(blob) {
  var o2 = [0, 0], d2;
  d2 = blob.read_shift(2);
  o2[0] = CountryEnum[d2] || d2;
  d2 = blob.read_shift(2);
  o2[1] = CountryEnum[d2] || d2;
  return o2;
}
function write_Country(o2) {
  if (!o2)
    o2 = new_buf(4);
  o2.write_shift(2, 1);
  o2.write_shift(2, 1);
  return o2;
}
function parse_ClrtClient(blob) {
  var ccv = blob.read_shift(2);
  var o2 = [];
  while (ccv-- > 0)
    o2.push(parse_LongRGB(blob));
  return o2;
}
function parse_Palette(blob) {
  var ccv = blob.read_shift(2);
  var o2 = [];
  while (ccv-- > 0)
    o2.push(parse_LongRGB(blob));
  return o2;
}
function parse_XFCRC(blob) {
  blob.l += 2;
  var o2 = { cxfs: 0, crc: 0 };
  o2.cxfs = blob.read_shift(2);
  o2.crc = blob.read_shift(4);
  return o2;
}
function parse_ColInfo(blob, length, opts) {
  if (!opts.cellStyles)
    return parsenoop(blob, length);
  var w2 = opts && opts.biff >= 12 ? 4 : 2;
  var colFirst = blob.read_shift(w2);
  var colLast = blob.read_shift(w2);
  var coldx = blob.read_shift(w2);
  var ixfe = blob.read_shift(w2);
  var flags3 = blob.read_shift(2);
  if (w2 == 2)
    blob.l += 2;
  var o2 = { s: colFirst, e: colLast, w: coldx, ixfe, flags: flags3 };
  if (opts.biff >= 5 || !opts.biff)
    o2.level = flags3 >> 8 & 7;
  return o2;
}
function write_ColInfo(col, idx) {
  var o2 = new_buf(12);
  o2.write_shift(2, idx);
  o2.write_shift(2, idx);
  o2.write_shift(2, col.width * 256);
  o2.write_shift(2, 0);
  var f2 = 0;
  if (col.hidden)
    f2 |= 1;
  o2.write_shift(1, f2);
  f2 = col.level || 0;
  o2.write_shift(1, f2);
  o2.write_shift(2, 0);
  return o2;
}
function parse_Setup(blob, length) {
  var o2 = {};
  if (length < 32)
    return o2;
  blob.l += 16;
  o2.header = parse_Xnum(blob);
  o2.footer = parse_Xnum(blob);
  blob.l += 2;
  return o2;
}
function parse_ShtProps(blob, length, opts) {
  var def = { area: false };
  if (opts.biff != 5) {
    blob.l += length;
    return def;
  }
  var d2 = blob.read_shift(1);
  blob.l += 3;
  if (d2 & 16)
    def.area = true;
  return def;
}
function write_RRTabId(n2) {
  var out2 = new_buf(2 * n2);
  for (var i2 = 0; i2 < n2; ++i2)
    out2.write_shift(2, i2 + 1);
  return out2;
}
var parse_Blank = parse_XLSCell;
var parse_Scl = parseuint16a;
var parse_String = parse_XLUnicodeString;
function parse_ImData(blob) {
  var cf = blob.read_shift(2);
  var env2 = blob.read_shift(2);
  var lcb = blob.read_shift(4);
  var o2 = { fmt: cf, env: env2, len: lcb, data: blob.slice(blob.l, blob.l + lcb) };
  blob.l += lcb;
  return o2;
}
function parse_BIFF2STR(blob, length, opts) {
  if (opts.biffguess && opts.biff == 5)
    opts.biff = 2;
  var cell = parse_XLSCell(blob);
  ++blob.l;
  var str = parse_XLUnicodeString2(blob, length - 7, opts);
  cell.t = "str";
  cell.val = str;
  return cell;
}
function parse_BIFF2NUM(blob) {
  var cell = parse_XLSCell(blob);
  ++blob.l;
  var num = parse_Xnum(blob);
  cell.t = "n";
  cell.val = num;
  return cell;
}
function write_BIFF2NUM(r2, c2, val) {
  var out2 = new_buf(15);
  write_BIFF2Cell(out2, r2, c2);
  out2.write_shift(8, val, "f");
  return out2;
}
function parse_BIFF2INT(blob) {
  var cell = parse_XLSCell(blob);
  ++blob.l;
  var num = blob.read_shift(2);
  cell.t = "n";
  cell.val = num;
  return cell;
}
function write_BIFF2INT(r2, c2, val) {
  var out2 = new_buf(9);
  write_BIFF2Cell(out2, r2, c2);
  out2.write_shift(2, val);
  return out2;
}
function parse_BIFF2STRING(blob) {
  var cch = blob.read_shift(1);
  if (cch === 0) {
    blob.l++;
    return "";
  }
  return blob.read_shift(cch, "sbcs-cont");
}
function parse_BIFF2FONTXTRA(blob, length) {
  blob.l += 6;
  blob.l += 2;
  blob.l += 1;
  blob.l += 3;
  blob.l += 1;
  blob.l += length - 13;
}
function parse_RString(blob, length, opts) {
  var end2 = blob.l + length;
  var cell = parse_XLSCell(blob);
  var cch = blob.read_shift(2);
  var str = parse_XLUnicodeStringNoCch(blob, cch, opts);
  blob.l = end2;
  cell.t = "str";
  cell.val = str;
  return cell;
}
var DBF_SUPPORTED_VERSIONS = [2, 3, 48, 49, 131, 139, 140, 245];
var DBF = /* @__PURE__ */ function() {
  var dbf_codepage_map = {
    /* Code Pages Supported by Visual FoxPro */
    /*::[*/
    1: 437,
    /*::[*/
    2: 850,
    /*::[*/
    3: 1252,
    /*::[*/
    4: 1e4,
    /*::[*/
    100: 852,
    /*::[*/
    101: 866,
    /*::[*/
    102: 865,
    /*::[*/
    103: 861,
    /*::[*/
    104: 895,
    /*::[*/
    105: 620,
    /*::[*/
    106: 737,
    /*::[*/
    107: 857,
    /*::[*/
    120: 950,
    /*::[*/
    121: 949,
    /*::[*/
    122: 936,
    /*::[*/
    123: 932,
    /*::[*/
    124: 874,
    /*::[*/
    125: 1255,
    /*::[*/
    126: 1256,
    /*::[*/
    150: 10007,
    /*::[*/
    151: 10029,
    /*::[*/
    152: 10006,
    /*::[*/
    200: 1250,
    /*::[*/
    201: 1251,
    /*::[*/
    202: 1254,
    /*::[*/
    203: 1253,
    /* shapefile DBF extension */
    /*::[*/
    0: 20127,
    /*::[*/
    8: 865,
    /*::[*/
    9: 437,
    /*::[*/
    10: 850,
    /*::[*/
    11: 437,
    /*::[*/
    13: 437,
    /*::[*/
    14: 850,
    /*::[*/
    15: 437,
    /*::[*/
    16: 850,
    /*::[*/
    17: 437,
    /*::[*/
    18: 850,
    /*::[*/
    19: 932,
    /*::[*/
    20: 850,
    /*::[*/
    21: 437,
    /*::[*/
    22: 850,
    /*::[*/
    23: 865,
    /*::[*/
    24: 437,
    /*::[*/
    25: 437,
    /*::[*/
    26: 850,
    /*::[*/
    27: 437,
    /*::[*/
    28: 863,
    /*::[*/
    29: 850,
    /*::[*/
    31: 852,
    /*::[*/
    34: 852,
    /*::[*/
    35: 852,
    /*::[*/
    36: 860,
    /*::[*/
    37: 850,
    /*::[*/
    38: 866,
    /*::[*/
    55: 850,
    /*::[*/
    64: 852,
    /*::[*/
    77: 936,
    /*::[*/
    78: 949,
    /*::[*/
    79: 950,
    /*::[*/
    80: 874,
    /*::[*/
    87: 1252,
    /*::[*/
    88: 1252,
    /*::[*/
    89: 1252,
    /*::[*/
    108: 863,
    /*::[*/
    134: 737,
    /*::[*/
    135: 852,
    /*::[*/
    136: 857,
    /*::[*/
    204: 1257,
    /*::[*/
    255: 16969
  };
  var dbf_reverse_map = evert({
    /*::[*/
    1: 437,
    /*::[*/
    2: 850,
    /*::[*/
    3: 1252,
    /*::[*/
    4: 1e4,
    /*::[*/
    100: 852,
    /*::[*/
    101: 866,
    /*::[*/
    102: 865,
    /*::[*/
    103: 861,
    /*::[*/
    104: 895,
    /*::[*/
    105: 620,
    /*::[*/
    106: 737,
    /*::[*/
    107: 857,
    /*::[*/
    120: 950,
    /*::[*/
    121: 949,
    /*::[*/
    122: 936,
    /*::[*/
    123: 932,
    /*::[*/
    124: 874,
    /*::[*/
    125: 1255,
    /*::[*/
    126: 1256,
    /*::[*/
    150: 10007,
    /*::[*/
    151: 10029,
    /*::[*/
    152: 10006,
    /*::[*/
    200: 1250,
    /*::[*/
    201: 1251,
    /*::[*/
    202: 1254,
    /*::[*/
    203: 1253,
    /*::[*/
    0: 20127
  });
  function dbf_to_aoa(buf, opts) {
    var out2 = [];
    var d2 = new_raw_buf(1);
    switch (opts.type) {
      case "base64":
        d2 = s2a(Base64_decode(buf));
        break;
      case "binary":
        d2 = s2a(buf);
        break;
      case "buffer":
      case "array":
        d2 = buf;
        break;
    }
    prep_blob(d2, 0);
    var ft2 = d2.read_shift(1);
    var memo = !!(ft2 & 136);
    var vfp = false, l7 = false;
    switch (ft2) {
      case 2:
        break;
      case 3:
        break;
      case 48:
        vfp = true;
        memo = true;
        break;
      case 49:
        vfp = true;
        memo = true;
        break;
      case 131:
        break;
      case 139:
        break;
      case 140:
        l7 = true;
        break;
      case 245:
        break;
      default:
        throw new Error("DBF Unsupported Version: " + ft2.toString(16));
    }
    var nrow = 0, fpos = 521;
    if (ft2 == 2)
      nrow = d2.read_shift(2);
    d2.l += 3;
    if (ft2 != 2)
      nrow = d2.read_shift(4);
    if (nrow > 1048576)
      nrow = 1e6;
    if (ft2 != 2)
      fpos = d2.read_shift(2);
    var rlen = d2.read_shift(2);
    var current_cp = opts.codepage || 1252;
    if (ft2 != 2) {
      d2.l += 16;
      d2.read_shift(1);
      if (d2[d2.l] !== 0)
        current_cp = dbf_codepage_map[d2[d2.l]];
      d2.l += 1;
      d2.l += 2;
    }
    if (l7)
      d2.l += 36;
    var fields = [], field = {};
    var hend = Math.min(d2.length, ft2 == 2 ? 521 : fpos - 10 - (vfp ? 264 : 0));
    var ww = l7 ? 32 : 11;
    while (d2.l < hend && d2[d2.l] != 13) {
      field = {};
      field.name = $cptable.utils.decode(current_cp, d2.slice(d2.l, d2.l + ww)).replace(/[\u0000\r\n].*$/g, "");
      d2.l += ww;
      field.type = String.fromCharCode(d2.read_shift(1));
      if (ft2 != 2 && !l7)
        field.offset = d2.read_shift(4);
      field.len = d2.read_shift(1);
      if (ft2 == 2)
        field.offset = d2.read_shift(2);
      field.dec = d2.read_shift(1);
      if (field.name.length)
        fields.push(field);
      if (ft2 != 2)
        d2.l += l7 ? 13 : 14;
      switch (field.type) {
        case "B":
          if ((!vfp || field.len != 8) && opts.WTF)
            console.log("Skipping " + field.name + ":" + field.type);
          break;
        case "G":
        case "P":
          if (opts.WTF)
            console.log("Skipping " + field.name + ":" + field.type);
          break;
        case "+":
        case "0":
        case "@":
        case "C":
        case "D":
        case "F":
        case "I":
        case "L":
        case "M":
        case "N":
        case "O":
        case "T":
        case "Y":
          break;
        default:
          throw new Error("Unknown Field Type: " + field.type);
      }
    }
    if (d2[d2.l] !== 13)
      d2.l = fpos - 1;
    if (d2.read_shift(1) !== 13)
      throw new Error("DBF Terminator not found " + d2.l + " " + d2[d2.l]);
    d2.l = fpos;
    var R2 = 0, C2 = 0;
    out2[0] = [];
    for (C2 = 0; C2 != fields.length; ++C2)
      out2[0][C2] = fields[C2].name;
    while (nrow-- > 0) {
      if (d2[d2.l] === 42) {
        d2.l += rlen;
        continue;
      }
      ++d2.l;
      out2[++R2] = [];
      C2 = 0;
      for (C2 = 0; C2 != fields.length; ++C2) {
        var dd = d2.slice(d2.l, d2.l + fields[C2].len);
        d2.l += fields[C2].len;
        prep_blob(dd, 0);
        var s2 = $cptable.utils.decode(current_cp, dd);
        switch (fields[C2].type) {
          case "C":
            if (s2.trim().length)
              out2[R2][C2] = s2.replace(/\s+$/, "");
            break;
          case "D":
            if (s2.length === 8)
              out2[R2][C2] = new Date(+s2.slice(0, 4), +s2.slice(4, 6) - 1, +s2.slice(6, 8));
            else
              out2[R2][C2] = s2;
            break;
          case "F":
            out2[R2][C2] = parseFloat(s2.trim());
            break;
          case "+":
          case "I":
            out2[R2][C2] = l7 ? dd.read_shift(-4, "i") ^ 2147483648 : dd.read_shift(4, "i");
            break;
          case "L":
            switch (s2.trim().toUpperCase()) {
              case "Y":
              case "T":
                out2[R2][C2] = true;
                break;
              case "N":
              case "F":
                out2[R2][C2] = false;
                break;
              case "":
              case "?":
                break;
              default:
                throw new Error("DBF Unrecognized L:|" + s2 + "|");
            }
            break;
          case "M":
            if (!memo)
              throw new Error("DBF Unexpected MEMO for type " + ft2.toString(16));
            out2[R2][C2] = "##MEMO##" + (l7 ? parseInt(s2.trim(), 10) : dd.read_shift(4));
            break;
          case "N":
            s2 = s2.replace(/\u0000/g, "").trim();
            if (s2 && s2 != ".")
              out2[R2][C2] = +s2 || 0;
            break;
          case "@":
            out2[R2][C2] = new Date(dd.read_shift(-8, "f") - 621356832e5);
            break;
          case "T":
            out2[R2][C2] = new Date((dd.read_shift(4) - 2440588) * 864e5 + dd.read_shift(4));
            break;
          case "Y":
            out2[R2][C2] = dd.read_shift(4, "i") / 1e4 + dd.read_shift(4, "i") / 1e4 * Math.pow(2, 32);
            break;
          case "O":
            out2[R2][C2] = -dd.read_shift(-8, "f");
            break;
          case "B":
            if (vfp && fields[C2].len == 8) {
              out2[R2][C2] = dd.read_shift(8, "f");
              break;
            }
          case "G":
          case "P":
            dd.l += fields[C2].len;
            break;
          case "0":
            if (fields[C2].name === "_NullFlags")
              break;
          default:
            throw new Error("DBF Unsupported data type " + fields[C2].type);
        }
      }
    }
    if (ft2 != 2) {
      if (d2.l < d2.length && d2[d2.l++] != 26)
        throw new Error("DBF EOF Marker missing " + (d2.l - 1) + " of " + d2.length + " " + d2[d2.l - 1].toString(16));
    }
    if (opts && opts.sheetRows)
      out2 = out2.slice(0, opts.sheetRows);
    opts.DBF = fields;
    return out2;
  }
  function dbf_to_sheet(buf, opts) {
    var o2 = opts || {};
    if (!o2.dateNF)
      o2.dateNF = "yyyymmdd";
    var ws = aoa_to_sheet(dbf_to_aoa(buf, o2), o2);
    ws["!cols"] = o2.DBF.map(function(field) {
      return {
        wch: field.len,
        DBF: field
      };
    });
    delete o2.DBF;
    return ws;
  }
  function dbf_to_workbook(buf, opts) {
    try {
      return sheet_to_workbook(dbf_to_sheet(buf, opts), opts);
    } catch (e2) {
      if (opts && opts.WTF)
        throw e2;
    }
    return { SheetNames: [], Sheets: {} };
  }
  var _RLEN = { "B": 8, "C": 250, "L": 1, "D": 8, "?": 0, "": 0 };
  function sheet_to_dbf(ws, opts) {
    var o2 = opts || {};
    if (+o2.codepage >= 0)
      set_cp(+o2.codepage);
    if (o2.type == "string")
      throw new Error("Cannot write DBF to JS string");
    var ba = buf_array();
    var aoa = sheet_to_json(ws, { header: 1, raw: true, cellDates: true });
    var headers = aoa[0], data2 = aoa.slice(1), cols = ws["!cols"] || [];
    var i2 = 0, j2 = 0, hcnt = 0, rlen = 1;
    for (i2 = 0; i2 < headers.length; ++i2) {
      if (((cols[i2] || {}).DBF || {}).name) {
        headers[i2] = cols[i2].DBF.name;
        ++hcnt;
        continue;
      }
      if (headers[i2] == null)
        continue;
      ++hcnt;
      if (typeof headers[i2] === "number")
        headers[i2] = headers[i2].toString(10);
      if (typeof headers[i2] !== "string")
        throw new Error("DBF Invalid column name " + headers[i2] + " |" + typeof headers[i2] + "|");
      if (headers.indexOf(headers[i2]) !== i2) {
        for (j2 = 0; j2 < 1024; ++j2)
          if (headers.indexOf(headers[i2] + "_" + j2) == -1) {
            headers[i2] += "_" + j2;
            break;
          }
      }
    }
    var range2 = safe_decode_range(ws["!ref"]);
    var coltypes = [];
    var colwidths = [];
    var coldecimals = [];
    for (i2 = 0; i2 <= range2.e.c - range2.s.c; ++i2) {
      var guess = "", _guess = "", maxlen = 0;
      var col = [];
      for (j2 = 0; j2 < data2.length; ++j2) {
        if (data2[j2][i2] != null)
          col.push(data2[j2][i2]);
      }
      if (col.length == 0 || headers[i2] == null) {
        coltypes[i2] = "?";
        continue;
      }
      for (j2 = 0; j2 < col.length; ++j2) {
        switch (typeof col[j2]) {
          case "number":
            _guess = "B";
            break;
          case "string":
            _guess = "C";
            break;
          case "boolean":
            _guess = "L";
            break;
          case "object":
            _guess = col[j2] instanceof Date ? "D" : "C";
            break;
          default:
            _guess = "C";
        }
        maxlen = Math.max(maxlen, String(col[j2]).length);
        guess = guess && guess != _guess ? "C" : _guess;
      }
      if (maxlen > 250)
        maxlen = 250;
      _guess = ((cols[i2] || {}).DBF || {}).type;
      if (_guess == "C") {
        if (cols[i2].DBF.len > maxlen)
          maxlen = cols[i2].DBF.len;
      }
      if (guess == "B" && _guess == "N") {
        guess = "N";
        coldecimals[i2] = cols[i2].DBF.dec;
        maxlen = cols[i2].DBF.len;
      }
      colwidths[i2] = guess == "C" || _guess == "N" ? maxlen : _RLEN[guess] || 0;
      rlen += colwidths[i2];
      coltypes[i2] = guess;
    }
    var h2 = ba.next(32);
    h2.write_shift(4, 318902576);
    h2.write_shift(4, data2.length);
    h2.write_shift(2, 296 + 32 * hcnt);
    h2.write_shift(2, rlen);
    for (i2 = 0; i2 < 4; ++i2)
      h2.write_shift(4, 0);
    h2.write_shift(4, 0 | (+dbf_reverse_map[
      /*::String(*/
      current_ansi
      /*::)*/
    ] || 3) << 8);
    for (i2 = 0, j2 = 0; i2 < headers.length; ++i2) {
      if (headers[i2] == null)
        continue;
      var hf = ba.next(32);
      var _f = (headers[i2].slice(-10) + "\0\0\0\0\0\0\0\0\0\0\0").slice(0, 11);
      hf.write_shift(1, _f, "sbcs");
      hf.write_shift(1, coltypes[i2] == "?" ? "C" : coltypes[i2], "sbcs");
      hf.write_shift(4, j2);
      hf.write_shift(1, colwidths[i2] || _RLEN[coltypes[i2]] || 0);
      hf.write_shift(1, coldecimals[i2] || 0);
      hf.write_shift(1, 2);
      hf.write_shift(4, 0);
      hf.write_shift(1, 0);
      hf.write_shift(4, 0);
      hf.write_shift(4, 0);
      j2 += colwidths[i2] || _RLEN[coltypes[i2]] || 0;
    }
    var hb = ba.next(264);
    hb.write_shift(4, 13);
    for (i2 = 0; i2 < 65; ++i2)
      hb.write_shift(4, 0);
    for (i2 = 0; i2 < data2.length; ++i2) {
      var rout = ba.next(rlen);
      rout.write_shift(1, 0);
      for (j2 = 0; j2 < headers.length; ++j2) {
        if (headers[j2] == null)
          continue;
        switch (coltypes[j2]) {
          case "L":
            rout.write_shift(1, data2[i2][j2] == null ? 63 : data2[i2][j2] ? 84 : 70);
            break;
          case "B":
            rout.write_shift(8, data2[i2][j2] || 0, "f");
            break;
          case "N":
            var _n = "0";
            if (typeof data2[i2][j2] == "number")
              _n = data2[i2][j2].toFixed(coldecimals[j2] || 0);
            for (hcnt = 0; hcnt < colwidths[j2] - _n.length; ++hcnt)
              rout.write_shift(1, 32);
            rout.write_shift(1, _n, "sbcs");
            break;
          case "D":
            if (!data2[i2][j2])
              rout.write_shift(8, "00000000", "sbcs");
            else {
              rout.write_shift(4, ("0000" + data2[i2][j2].getFullYear()).slice(-4), "sbcs");
              rout.write_shift(2, ("00" + (data2[i2][j2].getMonth() + 1)).slice(-2), "sbcs");
              rout.write_shift(2, ("00" + data2[i2][j2].getDate()).slice(-2), "sbcs");
            }
            break;
          case "C":
            var _s = String(data2[i2][j2] != null ? data2[i2][j2] : "").slice(0, colwidths[j2]);
            rout.write_shift(1, _s, "sbcs");
            for (hcnt = 0; hcnt < colwidths[j2] - _s.length; ++hcnt)
              rout.write_shift(1, 32);
            break;
        }
      }
    }
    ba.next(1).write_shift(1, 26);
    return ba.end();
  }
  return {
    to_workbook: dbf_to_workbook,
    to_sheet: dbf_to_sheet,
    from_sheet: sheet_to_dbf
  };
}();
var SYLK = /* @__PURE__ */ function() {
  var sylk_escapes = {
    AA: "À",
    BA: "Á",
    CA: "Â",
    DA: 195,
    HA: "Ä",
    JA: 197,
    AE: "È",
    BE: "É",
    CE: "Ê",
    HE: "Ë",
    AI: "Ì",
    BI: "Í",
    CI: "Î",
    HI: "Ï",
    AO: "Ò",
    BO: "Ó",
    CO: "Ô",
    DO: 213,
    HO: "Ö",
    AU: "Ù",
    BU: "Ú",
    CU: "Û",
    HU: "Ü",
    Aa: "à",
    Ba: "á",
    Ca: "â",
    Da: 227,
    Ha: "ä",
    Ja: 229,
    Ae: "è",
    Be: "é",
    Ce: "ê",
    He: "ë",
    Ai: "ì",
    Bi: "í",
    Ci: "î",
    Hi: "ï",
    Ao: "ò",
    Bo: "ó",
    Co: "ô",
    Do: 245,
    Ho: "ö",
    Au: "ù",
    Bu: "ú",
    Cu: "û",
    Hu: "ü",
    KC: "Ç",
    Kc: "ç",
    q: "æ",
    z: "œ",
    a: "Æ",
    j: "Œ",
    DN: 209,
    Dn: 241,
    Hy: 255,
    S: 169,
    c: 170,
    R: 174,
    "B ": 180,
    /*::[*/
    0: 176,
    /*::[*/
    1: 177,
    /*::[*/
    2: 178,
    /*::[*/
    3: 179,
    /*::[*/
    5: 181,
    /*::[*/
    6: 182,
    /*::[*/
    7: 183,
    Q: 185,
    k: 186,
    b: 208,
    i: 216,
    l: 222,
    s: 240,
    y: 248,
    "!": 161,
    '"': 162,
    "#": 163,
    "(": 164,
    "%": 165,
    "'": 167,
    "H ": 168,
    "+": 171,
    ";": 187,
    "<": 188,
    "=": 189,
    ">": 190,
    "?": 191,
    "{": 223
  };
  var sylk_char_regex = new RegExp("\x1BN(" + keys$4(sylk_escapes).join("|").replace(/\|\|\|/, "|\\||").replace(/([?()+])/g, "\\$1") + "|\\|)", "gm");
  var sylk_char_fn = function(_2, $1) {
    var o2 = sylk_escapes[$1];
    return typeof o2 == "number" ? _getansi(o2) : o2;
  };
  var decode_sylk_char = function($$, $1, $2) {
    var newcc = $1.charCodeAt(0) - 32 << 4 | $2.charCodeAt(0) - 48;
    return newcc == 59 ? $$ : _getansi(newcc);
  };
  sylk_escapes["|"] = 254;
  function sylk_to_aoa(d2, opts) {
    switch (opts.type) {
      case "base64":
        return sylk_to_aoa_str(Base64_decode(d2), opts);
      case "binary":
        return sylk_to_aoa_str(d2, opts);
      case "buffer":
        return sylk_to_aoa_str(has_buf && Buffer.isBuffer(d2) ? d2.toString("binary") : a2s(d2), opts);
      case "array":
        return sylk_to_aoa_str(cc2str(d2), opts);
    }
    throw new Error("Unrecognized type " + opts.type);
  }
  function sylk_to_aoa_str(str, opts) {
    var records = str.split(/[\n\r]+/), R2 = -1, C2 = -1, ri = 0, rj = 0, arr = [];
    var formats = [];
    var next_cell_format = null;
    var sht = {}, rowinfo = [], colinfo = [], cw = [];
    var Mval = 0, j2;
    if (+opts.codepage >= 0)
      set_cp(+opts.codepage);
    for (; ri !== records.length; ++ri) {
      Mval = 0;
      var rstr = records[ri].trim().replace(/\x1B([\x20-\x2F])([\x30-\x3F])/g, decode_sylk_char).replace(sylk_char_regex, sylk_char_fn);
      var record = rstr.replace(/;;/g, "\0").split(";").map(function(x2) {
        return x2.replace(/\u0000/g, ";");
      });
      var RT = record[0], val;
      if (rstr.length > 0)
        switch (RT) {
          case "ID":
            break;
          case "E":
            break;
          case "B":
            break;
          case "O":
            break;
          case "W":
            break;
          case "P":
            if (record[1].charAt(0) == "P")
              formats.push(rstr.slice(3).replace(/;;/g, ";"));
            break;
          case "C":
            var C_seen_K = false, C_seen_X = false, C_seen_S = false, C_seen_E = false, _R = -1, _C = -1;
            for (rj = 1; rj < record.length; ++rj)
              switch (record[rj].charAt(0)) {
                case "A":
                  break;
                case "X":
                  C2 = parseInt(record[rj].slice(1)) - 1;
                  C_seen_X = true;
                  break;
                case "Y":
                  R2 = parseInt(record[rj].slice(1)) - 1;
                  if (!C_seen_X)
                    C2 = 0;
                  for (j2 = arr.length; j2 <= R2; ++j2)
                    arr[j2] = [];
                  break;
                case "K":
                  val = record[rj].slice(1);
                  if (val.charAt(0) === '"')
                    val = val.slice(1, val.length - 1);
                  else if (val === "TRUE")
                    val = true;
                  else if (val === "FALSE")
                    val = false;
                  else if (!isNaN(fuzzynum(val))) {
                    val = fuzzynum(val);
                    if (next_cell_format !== null && fmt_is_date(next_cell_format))
                      val = numdate(val);
                  } else if (!isNaN(fuzzydate(val).getDate())) {
                    val = parseDate$1(val);
                  }
                  C_seen_K = true;
                  break;
                case "E":
                  C_seen_E = true;
                  var formula = rc_to_a1(record[rj].slice(1), { r: R2, c: C2 });
                  arr[R2][C2] = [arr[R2][C2], formula];
                  break;
                case "S":
                  C_seen_S = true;
                  arr[R2][C2] = [arr[R2][C2], "S5S"];
                  break;
                case "G":
                  break;
                case "R":
                  _R = parseInt(record[rj].slice(1)) - 1;
                  break;
                case "C":
                  _C = parseInt(record[rj].slice(1)) - 1;
                  break;
                default:
                  if (opts && opts.WTF)
                    throw new Error("SYLK bad record " + rstr);
              }
            if (C_seen_K) {
              if (arr[R2][C2] && arr[R2][C2].length == 2)
                arr[R2][C2][0] = val;
              else
                arr[R2][C2] = val;
              next_cell_format = null;
            }
            if (C_seen_S) {
              if (C_seen_E)
                throw new Error("SYLK shared formula cannot have own formula");
              var shrbase = _R > -1 && arr[_R][_C];
              if (!shrbase || !shrbase[1])
                throw new Error("SYLK shared formula cannot find base");
              arr[R2][C2][1] = shift_formula_str(shrbase[1], { r: R2 - _R, c: C2 - _C });
            }
            break;
          case "F":
            var F_seen = 0;
            for (rj = 1; rj < record.length; ++rj)
              switch (record[rj].charAt(0)) {
                case "X":
                  C2 = parseInt(record[rj].slice(1)) - 1;
                  ++F_seen;
                  break;
                case "Y":
                  R2 = parseInt(record[rj].slice(1)) - 1;
                  for (j2 = arr.length; j2 <= R2; ++j2)
                    arr[j2] = [];
                  break;
                case "M":
                  Mval = parseInt(record[rj].slice(1)) / 20;
                  break;
                case "F":
                  break;
                case "G":
                  break;
                case "P":
                  next_cell_format = formats[parseInt(record[rj].slice(1))];
                  break;
                case "S":
                  break;
                case "D":
                  break;
                case "N":
                  break;
                case "W":
                  cw = record[rj].slice(1).split(" ");
                  for (j2 = parseInt(cw[0], 10); j2 <= parseInt(cw[1], 10); ++j2) {
                    Mval = parseInt(cw[2], 10);
                    colinfo[j2 - 1] = Mval === 0 ? { hidden: true } : { wch: Mval };
                    process_col(colinfo[j2 - 1]);
                  }
                  break;
                case "C":
                  C2 = parseInt(record[rj].slice(1)) - 1;
                  if (!colinfo[C2])
                    colinfo[C2] = {};
                  break;
                case "R":
                  R2 = parseInt(record[rj].slice(1)) - 1;
                  if (!rowinfo[R2])
                    rowinfo[R2] = {};
                  if (Mval > 0) {
                    rowinfo[R2].hpt = Mval;
                    rowinfo[R2].hpx = pt2px(Mval);
                  } else if (Mval === 0)
                    rowinfo[R2].hidden = true;
                  break;
                default:
                  if (opts && opts.WTF)
                    throw new Error("SYLK bad record " + rstr);
              }
            if (F_seen < 1)
              next_cell_format = null;
            break;
          default:
            if (opts && opts.WTF)
              throw new Error("SYLK bad record " + rstr);
        }
    }
    if (rowinfo.length > 0)
      sht["!rows"] = rowinfo;
    if (colinfo.length > 0)
      sht["!cols"] = colinfo;
    if (opts && opts.sheetRows)
      arr = arr.slice(0, opts.sheetRows);
    return [arr, sht];
  }
  function sylk_to_sheet(d2, opts) {
    var aoasht = sylk_to_aoa(d2, opts);
    var aoa = aoasht[0], ws = aoasht[1];
    var o2 = aoa_to_sheet(aoa, opts);
    keys$4(ws).forEach(function(k2) {
      o2[k2] = ws[k2];
    });
    return o2;
  }
  function sylk_to_workbook(d2, opts) {
    return sheet_to_workbook(sylk_to_sheet(d2, opts), opts);
  }
  function write_ws_cell_sylk(cell, ws, R2, C2) {
    var o2 = "C;Y" + (R2 + 1) + ";X" + (C2 + 1) + ";K";
    switch (cell.t) {
      case "n":
        o2 += cell.v || 0;
        if (cell.f && !cell.F)
          o2 += ";E" + a1_to_rc(cell.f, { r: R2, c: C2 });
        break;
      case "b":
        o2 += cell.v ? "TRUE" : "FALSE";
        break;
      case "e":
        o2 += cell.w || cell.v;
        break;
      case "d":
        o2 += '"' + (cell.w || cell.v) + '"';
        break;
      case "s":
        o2 += '"' + cell.v.replace(/"/g, "").replace(/;/g, ";;") + '"';
        break;
    }
    return o2;
  }
  function write_ws_cols_sylk(out2, cols) {
    cols.forEach(function(col, i2) {
      var rec = "F;W" + (i2 + 1) + " " + (i2 + 1) + " ";
      if (col.hidden)
        rec += "0";
      else {
        if (typeof col.width == "number" && !col.wpx)
          col.wpx = width2px(col.width);
        if (typeof col.wpx == "number" && !col.wch)
          col.wch = px2char(col.wpx);
        if (typeof col.wch == "number")
          rec += Math.round(col.wch);
      }
      if (rec.charAt(rec.length - 1) != " ")
        out2.push(rec);
    });
  }
  function write_ws_rows_sylk(out2, rows) {
    rows.forEach(function(row, i2) {
      var rec = "F;";
      if (row.hidden)
        rec += "M0;";
      else if (row.hpt)
        rec += "M" + 20 * row.hpt + ";";
      else if (row.hpx)
        rec += "M" + 20 * px2pt(row.hpx) + ";";
      if (rec.length > 2)
        out2.push(rec + "R" + (i2 + 1));
    });
  }
  function sheet_to_sylk(ws, opts) {
    var preamble = ["ID;PWXL;N;E"], o2 = [];
    var r2 = safe_decode_range(ws["!ref"]), cell;
    var dense = Array.isArray(ws);
    var RS = "\r\n";
    preamble.push("P;PGeneral");
    preamble.push("F;P0;DG0G8;M255");
    if (ws["!cols"])
      write_ws_cols_sylk(preamble, ws["!cols"]);
    if (ws["!rows"])
      write_ws_rows_sylk(preamble, ws["!rows"]);
    preamble.push("B;Y" + (r2.e.r - r2.s.r + 1) + ";X" + (r2.e.c - r2.s.c + 1) + ";D" + [r2.s.c, r2.s.r, r2.e.c, r2.e.r].join(" "));
    for (var R2 = r2.s.r; R2 <= r2.e.r; ++R2) {
      for (var C2 = r2.s.c; C2 <= r2.e.c; ++C2) {
        var coord = encode_cell({ r: R2, c: C2 });
        cell = dense ? (ws[R2] || [])[C2] : ws[coord];
        if (!cell || cell.v == null && (!cell.f || cell.F))
          continue;
        o2.push(write_ws_cell_sylk(cell, ws, R2, C2));
      }
    }
    return preamble.join(RS) + RS + o2.join(RS) + RS + "E" + RS;
  }
  return {
    to_workbook: sylk_to_workbook,
    to_sheet: sylk_to_sheet,
    from_sheet: sheet_to_sylk
  };
}();
var DIF = /* @__PURE__ */ function() {
  function dif_to_aoa(d2, opts) {
    switch (opts.type) {
      case "base64":
        return dif_to_aoa_str(Base64_decode(d2), opts);
      case "binary":
        return dif_to_aoa_str(d2, opts);
      case "buffer":
        return dif_to_aoa_str(has_buf && Buffer.isBuffer(d2) ? d2.toString("binary") : a2s(d2), opts);
      case "array":
        return dif_to_aoa_str(cc2str(d2), opts);
    }
    throw new Error("Unrecognized type " + opts.type);
  }
  function dif_to_aoa_str(str, opts) {
    var records = str.split("\n"), R2 = -1, C2 = -1, ri = 0, arr = [];
    for (; ri !== records.length; ++ri) {
      if (records[ri].trim() === "BOT") {
        arr[++R2] = [];
        C2 = 0;
        continue;
      }
      if (R2 < 0)
        continue;
      var metadata = records[ri].trim().split(",");
      var type2 = metadata[0], value = metadata[1];
      ++ri;
      var data2 = records[ri] || "";
      while ((data2.match(/["]/g) || []).length & 1 && ri < records.length - 1)
        data2 += "\n" + records[++ri];
      data2 = data2.trim();
      switch (+type2) {
        case -1:
          if (data2 === "BOT") {
            arr[++R2] = [];
            C2 = 0;
            continue;
          } else if (data2 !== "EOD")
            throw new Error("Unrecognized DIF special command " + data2);
          break;
        case 0:
          if (data2 === "TRUE")
            arr[R2][C2] = true;
          else if (data2 === "FALSE")
            arr[R2][C2] = false;
          else if (!isNaN(fuzzynum(value)))
            arr[R2][C2] = fuzzynum(value);
          else if (!isNaN(fuzzydate(value).getDate()))
            arr[R2][C2] = parseDate$1(value);
          else
            arr[R2][C2] = value;
          ++C2;
          break;
        case 1:
          data2 = data2.slice(1, data2.length - 1);
          data2 = data2.replace(/""/g, '"');
          if (data2 && data2.match(/^=".*"$/))
            data2 = data2.slice(2, -1);
          arr[R2][C2++] = data2 !== "" ? data2 : null;
          break;
      }
      if (data2 === "EOD")
        break;
    }
    if (opts && opts.sheetRows)
      arr = arr.slice(0, opts.sheetRows);
    return arr;
  }
  function dif_to_sheet(str, opts) {
    return aoa_to_sheet(dif_to_aoa(str, opts), opts);
  }
  function dif_to_workbook(str, opts) {
    return sheet_to_workbook(dif_to_sheet(str, opts), opts);
  }
  var sheet_to_dif = /* @__PURE__ */ function() {
    var push_field = function pf(o2, topic, v4, n2, s2) {
      o2.push(topic);
      o2.push(v4 + "," + n2);
      o2.push('"' + s2.replace(/"/g, '""') + '"');
    };
    var push_value = function po(o2, type2, v4, s2) {
      o2.push(type2 + "," + v4);
      o2.push(type2 == 1 ? '"' + s2.replace(/"/g, '""') + '"' : s2);
    };
    return function sheet_to_dif2(ws) {
      var o2 = [];
      var r2 = safe_decode_range(ws["!ref"]), cell;
      var dense = Array.isArray(ws);
      push_field(o2, "TABLE", 0, 1, "sheetjs");
      push_field(o2, "VECTORS", 0, r2.e.r - r2.s.r + 1, "");
      push_field(o2, "TUPLES", 0, r2.e.c - r2.s.c + 1, "");
      push_field(o2, "DATA", 0, 0, "");
      for (var R2 = r2.s.r; R2 <= r2.e.r; ++R2) {
        push_value(o2, -1, 0, "BOT");
        for (var C2 = r2.s.c; C2 <= r2.e.c; ++C2) {
          var coord = encode_cell({ r: R2, c: C2 });
          cell = dense ? (ws[R2] || [])[C2] : ws[coord];
          if (!cell) {
            push_value(o2, 1, 0, "");
            continue;
          }
          switch (cell.t) {
            case "n":
              var val = cell.w;
              if (!val && cell.v != null)
                val = cell.v;
              if (val == null) {
                if (cell.f && !cell.F)
                  push_value(o2, 1, 0, "=" + cell.f);
                else
                  push_value(o2, 1, 0, "");
              } else
                push_value(o2, 0, val, "V");
              break;
            case "b":
              push_value(o2, 0, cell.v ? 1 : 0, cell.v ? "TRUE" : "FALSE");
              break;
            case "s":
              push_value(o2, 1, 0, isNaN(cell.v) ? cell.v : '="' + cell.v + '"');
              break;
            case "d":
              if (!cell.w)
                cell.w = SSF_format(cell.z || table_fmt[14], datenum(parseDate$1(cell.v)));
              push_value(o2, 0, cell.w, "V");
              break;
            default:
              push_value(o2, 1, 0, "");
          }
        }
      }
      push_value(o2, -1, 0, "EOD");
      var RS = "\r\n";
      var oo = o2.join(RS);
      return oo;
    };
  }();
  return {
    to_workbook: dif_to_workbook,
    to_sheet: dif_to_sheet,
    from_sheet: sheet_to_dif
  };
}();
var ETH = /* @__PURE__ */ function() {
  function decode2(s2) {
    return s2.replace(/\\b/g, "\\").replace(/\\c/g, ":").replace(/\\n/g, "\n");
  }
  function encode4(s2) {
    return s2.replace(/\\/g, "\\b").replace(/:/g, "\\c").replace(/\n/g, "\\n");
  }
  function eth_to_aoa(str, opts) {
    var records = str.split("\n"), R2 = -1, C2 = -1, ri = 0, arr = [];
    for (; ri !== records.length; ++ri) {
      var record = records[ri].trim().split(":");
      if (record[0] !== "cell")
        continue;
      var addr = decode_cell(record[1]);
      if (arr.length <= addr.r) {
        for (R2 = arr.length; R2 <= addr.r; ++R2)
          if (!arr[R2])
            arr[R2] = [];
      }
      R2 = addr.r;
      C2 = addr.c;
      switch (record[2]) {
        case "t":
          arr[R2][C2] = decode2(record[3]);
          break;
        case "v":
          arr[R2][C2] = +record[3];
          break;
        case "vtf":
          var _f = record[record.length - 1];
        case "vtc":
          switch (record[3]) {
            case "nl":
              arr[R2][C2] = +record[4] ? true : false;
              break;
            default:
              arr[R2][C2] = +record[4];
              break;
          }
          if (record[2] == "vtf")
            arr[R2][C2] = [arr[R2][C2], _f];
      }
    }
    if (opts && opts.sheetRows)
      arr = arr.slice(0, opts.sheetRows);
    return arr;
  }
  function eth_to_sheet(d2, opts) {
    return aoa_to_sheet(eth_to_aoa(d2, opts), opts);
  }
  function eth_to_workbook(d2, opts) {
    return sheet_to_workbook(eth_to_sheet(d2, opts), opts);
  }
  var header = [
    "socialcalc:version:1.5",
    "MIME-Version: 1.0",
    "Content-Type: multipart/mixed; boundary=SocialCalcSpreadsheetControlSave"
  ].join("\n");
  var sep = [
    "--SocialCalcSpreadsheetControlSave",
    "Content-type: text/plain; charset=UTF-8"
  ].join("\n") + "\n";
  var meta = [
    "# SocialCalc Spreadsheet Control Save",
    "part:sheet"
  ].join("\n");
  var end2 = "--SocialCalcSpreadsheetControlSave--";
  function sheet_to_eth_data(ws) {
    if (!ws || !ws["!ref"])
      return "";
    var o2 = [], oo = [], cell, coord = "";
    var r2 = decode_range(ws["!ref"]);
    var dense = Array.isArray(ws);
    for (var R2 = r2.s.r; R2 <= r2.e.r; ++R2) {
      for (var C2 = r2.s.c; C2 <= r2.e.c; ++C2) {
        coord = encode_cell({ r: R2, c: C2 });
        cell = dense ? (ws[R2] || [])[C2] : ws[coord];
        if (!cell || cell.v == null || cell.t === "z")
          continue;
        oo = ["cell", coord, "t"];
        switch (cell.t) {
          case "s":
          case "str":
            oo.push(encode4(cell.v));
            break;
          case "n":
            if (!cell.f) {
              oo[2] = "v";
              oo[3] = cell.v;
            } else {
              oo[2] = "vtf";
              oo[3] = "n";
              oo[4] = cell.v;
              oo[5] = encode4(cell.f);
            }
            break;
          case "b":
            oo[2] = "vt" + (cell.f ? "f" : "c");
            oo[3] = "nl";
            oo[4] = cell.v ? "1" : "0";
            oo[5] = encode4(cell.f || (cell.v ? "TRUE" : "FALSE"));
            break;
          case "d":
            var t3 = datenum(parseDate$1(cell.v));
            oo[2] = "vtc";
            oo[3] = "nd";
            oo[4] = "" + t3;
            oo[5] = cell.w || SSF_format(cell.z || table_fmt[14], t3);
            break;
          case "e":
            continue;
        }
        o2.push(oo.join(":"));
      }
    }
    o2.push("sheet:c:" + (r2.e.c - r2.s.c + 1) + ":r:" + (r2.e.r - r2.s.r + 1) + 