package com.bokesoft.distro.tech.yigosupport.extension.cache;

import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.bokesoft.distro.tech.yigosupport.extension.utils.yigo.SessionUtils;
import com.bokesoft.yigo.cache.CacheFactory;
import com.bokesoft.yigo.cache.ICache;
import com.bokesoft.yigo.struct.datatable.DataTable;

/**
 * 通过 SQL 语句 和 参数 缓存数据库查询结果, 提供可缓存的数据库查询功能, 通过分布式缓存检查数据的有效性, 支持集群环境
 */
public class SqlQueryCache {
	private static Logger log = LoggerFactory.getLogger(SqlQueryCache.class);
	public static final String SQL_CACHE_KEY_PREFIX = "Yee_SqlQueyCache";
	
	private static RequestContext requestContext = null;
	/**
	 * 集成到环境的配置接口, 以从环境重获得请求上下文, 实现一个请求过程中对分布式缓存的访问优化
	 * @param requestContext
	 */
	public static void setRequestContext(RequestContext requestContext) {
		SqlQueryCache.requestContext = requestContext;
	}
	
	private String cacheName;
	
	private Map<String, Object> localCache = new ConcurrentHashMap<>();
	private Map<String, Long> localQueryTimeMap = new ConcurrentHashMap<>();

	/**
	 * 一般不会直接构造 SqlQueryCache, 而是应该通过 {@link SqlQueryCacheManager} 创建.
	 * @param cacheName
	 */
	protected SqlQueryCache(String cacheName) {
		this.cacheName = cacheName;
	}
	
	/**
	 * 执行 SQL 语句并缓存查询结果数据, 在缓存未清除(见 {@link #clear()})前重复执行查询不需要重复访问数据库.
	 * @param <T> 缓存的数据类型
	 * @param sql SQL 语句
	 * @param params SQL 语句参数
	 * @param wrapper 用于将 SQL 语句查询出来的 DataTable 转化为需要缓存的 Java 对象
	 * @param decorator 用于在每次缓存数据返回给应用代码前进行额外的处理
	 * @return
	 */
	public <T> T basicQuery(String sql, Object[] params, DataWrapper<T> wrapper, DataDecorator<T> decorator){
	    String sqlCacheKey = getSqlCahceKey(sql, params);
	    String lastQueryTimeCacheKey = getLastQueryTimeKey(sqlCacheKey);
	    
	    Request request = null;
	    if (null!=SqlQueryCache.requestContext) {
	    	request = SqlQueryCache.requestContext.getRequest();
	    }
	    
	    Long remoteTime = null;
	    if (null!=request){
	        remoteTime = (Long)request.getAttribute(sqlCacheKey);
	    }
	   
	    ICache<Long> yigoCache = null;
	    if (null==remoteTime){
	        yigoCache = getYigoCache();
	        remoteTime = yigoCache.get(lastQueryTimeCacheKey);
	        if (null!=request){
	        	request.setAttribute(sqlCacheKey, remoteTime);
	        }
	    }
	   
	    boolean needOverwriteCache;    //是否覆盖远程缓存中的时间戳
	    if(null==remoteTime){
	        needOverwriteCache = true;
	    }else{
	        //最后查询时间缓存在数据修改后会被清除, 所以只要有值, 就说明从那时起到现在没有经过修改
	        needOverwriteCache = false;
	    }
	   
	    boolean needDBQuery;    //是否需要重新查询数据库
	    if (!localCache.containsKey(sqlCacheKey)){
	        needDBQuery=true;
	    }else{
	        Long localTime = localQueryTimeMap.get(sqlCacheKey);
	       
	        if (null==localTime){
	            needDBQuery=true;    //并发场景额外控制
	            log.error("并发冲突 - localCache 和 localQueryTimeMap 中根据 Key '{}' 获取的值不一致, 将重新查询数据", sqlCacheKey);
	        }else{
	            if (null==remoteTime){
	                needDBQuery=true;
	            }else if(!localTime.equals(remoteTime)){
	                needDBQuery=true;
	            }else{
	                needDBQuery=false;
	            }
	        }
	    }
	   
	    if (needOverwriteCache){
	        remoteTime = System.currentTimeMillis();
	    }
	   
	    @SuppressWarnings("unchecked")
	    T cacheData = (T)localCache.get(sqlCacheKey);
	   
	    if (needDBQuery || (null==localCache)){
	        synchronized(this){
	            @SuppressWarnings("unchecked")
	            T tmp = (T)localCache.get(sqlCacheKey);
	            cacheData = tmp;
	           
	            if (needDBQuery || (null==localCache)){
	                log.info("重新读取 SqlQueryCache: {}, PARAMS={}, CacheKey={}",
	                            this.cacheName, StringUtils.join(params, ","), sqlCacheKey);
	                DataTable dbData = getQueryDataTable(sql, params);
	                cacheData = wrapper.build(dbData);
	                localQueryTimeMap.put(sqlCacheKey, remoteTime);
	                localCache.put(sqlCacheKey, cacheData);
	            }else{
	                if(log.isDebugEnabled()){
	                    log.debug("命中 SqlCacheCache: {}, PARAMS={}, CacheKey={} .",
	                            this.cacheName, StringUtils.join(params, ","), sqlCacheKey);
	                }
	            }
	        }
	    }else{
	        if(log.isDebugEnabled()){
	            log.debug("命中 SqlCacheCache: {}, PARAMS={}, CacheKey={} .",
	                    this.cacheName, StringUtils.join(params, ","), sqlCacheKey);
	        }
	    }
	   
	    if (needOverwriteCache){
	        log.info("更新时间戳标记 SqlQueryCache: {}, lastQueryTimeCacheKey={}", this.cacheName, lastQueryTimeCacheKey);
	        if (yigoCache==null){
	            yigoCache=getYigoCache();
	        }
	        yigoCache.put(lastQueryTimeCacheKey, remoteTime);
	        if (null!=request){
	        	request.setAttribute(sqlCacheKey, remoteTime);
	        }
	    }
	   
	    return decorator.fetch(cacheData);
	}

	/**
	 * 针对 Key-Value 结构的查询数据缓存, 提供对 {@link #basicQuery(String, Object[], DataWrapper, DataDecorator)} 方法
	 * 的简化封装.
	 * @param <T> Value 的数据类型
	 * @param sql SQL 语句, 其中必须包括 keyField 和 valueField 指定的两个字段
	 * @param params SQL 语句参数
	 * @param keyField 查询结果中作为 Key 的字段名, 字段数据类型需要是 String 类型
	 * @param valueField 查询结果中作为 Value 的字段名
	 * @param typeClass 查询结果 Value 的数据类型
	 * @return
	 */
	public <T> Map<String, T> queryMap(String sql, Object[] params, String keyField, String valueField, Class<T> typeClass){
	    return basicQuery(sql, params, data -> {
	        FieldReader<T> reader = getFieldReader(typeClass);
	       
	        Map<String, T> cacheData = new HashMap<>();
	        data.beforeFirst();
	        while(data.next()){
	        	String key = data.getString(keyField);
	            T value = reader.get(data, valueField);
	            if (null!=value){
	                cacheData.put(key, value);
	            }
	        }
	        return cacheData;
	    }, cachedData -> {
	        return Collections.unmodifiableMap(cachedData);
	    });
	}
	
	private DataTable getQueryDataTable(String sql, Object[] params) {
		return SessionUtils.processWithContext(null, ctx->{
			DataTable dt = ctx.getDBManager().execPrepareQuery(sql, params);
			return dt;
		});
	}

	private String getLastQueryTimeKey(String sqlCacheKey) {
		return "LastQueryTime:"+sqlCacheKey;
	}

	private String getSqlCahceKey(String sql, Object[] params) {
		return sql+":"+StringUtils.join(params, ",");
	}

	/**
	 * 清除当前 Cache
	 */
	public void clear() {
	    log.info("清除 SqlCacheCache: {}", this.cacheName);
	    synchronized(this){
	        Set<String> keys=new HashSet<>();
	        keys.addAll(localCache.keySet());
	       
	        ICache<Long> yigoCache = getYigoCache();
	        keys.addAll(yigoCache.getKeys());
	       
	        int size = keys.size();
	        if (size>0){
	            log.debug("清除 SqlCacheCache: {}, 共 {} 项 ...", this.cacheName, size);
				//FIXME 不能直接使用 getYigoCache().clear(), 会清除所有的缓存信息, 包括 session 信息
	            for(String key: keys){
	                yigoCache.remove(key);
	            }
	            log.debug("清除 SqlCacheCache 完成: {}, 共 {} 项 .", this.cacheName, size);
	        }
	        localCache.clear();
	        localQueryTimeMap.clear();
	    }
	}
	
	private ICache<Long> getYigoCache() {
		String cacheKey = SQL_CACHE_KEY_PREFIX +":"+ this.cacheName;
		ICache<Long> cache = CacheFactory.getInstance().createCache(cacheKey);
		return cache;
	}
	
	/**
	 * 抽象从数据库查询结果 {@link DataTable} 构建缓存数据对象的过程
	 * @param <T>
	 */
	public static interface DataWrapper<T> {
	    /**
	     * 通过查询出现的数据构造用于缓存的数据对象
	     * (注意: 此方法执行时是在 "synchornized(this)" 代码块中, 因此对同一个 Cache 不存在并发问题)
	     */
	    public T build(DataTable data);
	}

	/**
	 * 抽象每次从缓存读取数据后的额外修饰处理(例如将数据副本作为结果返回)
	 * @param <T>
	 */
	public static interface DataDecorator<T> {
	    /**
	     * 每次从缓存中取到数据后的处理(简化起见, 取出的数据与缓存中的数据类型保持一致)
	     */
	    public T fetch(T cachedData);
	}

	/**
	 * 辅助性接口, 用于 queryMap 等方法中根据类型读取 DataTable 字段值
	 * @param <T>
	 */
	public static interface FieldReader<T> {
	    public T get(DataTable data, String columnKey);
	}

	@SuppressWarnings({"rawtypes", "unchecked"})
	private static final <T> FieldReader<T> getFieldReader(Class<T> type){
	    if (type.isAssignableFrom(String.class)){
	        return new FieldReader(){
	            public String get(DataTable data, String columnKey){
	                return data.getString(columnKey);
	            }
	        };
	    }else if (type.isAssignableFrom(Integer.class)){
	        return new FieldReader(){
	            public Integer get(DataTable data, String columnKey){
	                return data.getInt(columnKey);
	            }
	        };
	    }else if (type.isAssignableFrom(Long.class)){
	        return new FieldReader(){
	            public Long get(DataTable data, String columnKey){
	                return data.getLong(columnKey);
	            }
	        };
	    }else if (type.isAssignableFrom(Number.class)){
	        return new FieldReader(){
	            public Number get(DataTable data, String columnKey){
	                return data.getNumeric(columnKey);
	            }
	        };
	    }else if (type.isAssignableFrom(Date.class)){
	        return new FieldReader(){
	            public Date get(DataTable data, String columnKey){
	                return data.getDateTime(columnKey);
	            }
	        };
	    }
	    throw new UnsupportedOperationException("不支持的数据类型: "+type.getName());
	}

	/**
	 * 请求上下文对象, 用于获取当前请求的 {@link Request} 对象
	 */
	public static interface RequestContext {
		public Request getRequest();
	}
	/**
	 * 抽象请求对象, 主要目的是为了优化缓存访问 - 一次请求之中只需要一次查询缓存
	 */
	public static interface Request {
		public Object getAttribute(String key);
		public void setAttribute(String key, Object value);
	}
	
}
