package com.bokesoft.distro.tech.yigosupport.extension.utils.yigo;


import com.bokesoft.yes.common.util.SerializeUtil;
import com.bokesoft.yes.tools.cache.ICacheCallback;
import com.bokesoft.yigo.cache.ICache;
import com.bokesoft.yigo.mid.file.util.AttachmentUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.reflect.MethodUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 此类适用yigo-2.2.2以上版本
 * ICache在 yigo-2.2.2 以后统一package为 com.bokesoft.yigo.cache.ICache
 * 在yigo-2.2.2之前,package为com.bokesoft.yes.cache.ICache
 * 所以早期yigo在使用此类,会出现nosuchclass的问题
 */
public class CacheFactoryWrapUtil{
    /**
     * 获取CacheFactory的方法名
     */
    private static final String METHOD_INSTANCE = "getInstance";
    /**
     * 创建cache的方法名
     */
    private static final String METHOD_CREATE_CACHE = "createCache";
    /**
     * 2.2.2~2.2.6版本,Class#packag路径
     */
    private static final String CACHE_FACTORY_CLASS_NAME_2 = "com.bokesoft.yes.tools.cache.CacheFactory";
    /**
     * 3.0.0版本以上,Class#packag路径
     */
    private static final String CACHE_FACTORY_CLASS_NAME_3 = "com.bokesoft.yigo.cache.CacheFactory";
    /**
     * cache 大对象转为附件的大小限度
     */
    private static final int COMMON_LIMIT_SIZE = 1024;

    private static final Map<String, ICache<?>> CACHE_MAP = new ConcurrentHashMap<>();

    private static final Class<?> CACHE_FACTORY_CLASS = getCacheFactoryClass();

    private static final String STORE_DIR_DATE_FORMAT = "yyyy/MM/dd";

    /**
     * 考虑适用yigo版本（2.2.2~3.1.x）内,兼容式获取 ICache 对象
     * @param cacheKey 参数
     * @return {@link ICache}
     */
    @SuppressWarnings("unchecked")
    public static <V> ICache<V> getCache(String cacheKey) {
        return (ICache<V>) CACHE_MAP.computeIfAbsent(cacheKey, (val) ->
                buildCache(cacheKey)
        );
    }

    /**
     * 考虑适用yigo版本（2.2.2~3.1.x）内,兼容式获取 ICache 对象
     * @param cacheKey 参数
     * @return {@link ICache}
     */
    public static <V> ICache<V> getBigDataCache(String cacheKey,String formKey) {
        return getBigDataCache(cacheKey,formKey,COMMON_LIMIT_SIZE);
    }

    /**
     * 考虑适用yigo版本（2.2.2~3.1.x）内,兼容式获取 ICache 对象
     * @param cacheKey 参数
     * @return {@link ICache}
     */
    public static <V> ICache<V> getBigDataCache(String cacheKey,String formKey,int limitSize) {
        if(StringUtils.isBlank(formKey)){
            throw new RuntimeException("使用BigDataCache必须输入有效Fomrkey!");
        }
        ICache<V> result = getCache(cacheKey);
        return new WrapperCache<V>(result,cacheKey,formKey,limitSize);
    }

    private static <V> ICache<V> buildCache(String cacheKey) {
        try {
            Object iCacheFactoryObj = MethodUtils.invokeStaticMethod(CACHE_FACTORY_CLASS, METHOD_INSTANCE);
            @SuppressWarnings("unchecked")
            ICache<V> result = (ICache<V>) MethodUtils.invokeMethod(iCacheFactoryObj,METHOD_CREATE_CACHE, cacheKey);
            return result;
        } catch (Exception e) {
            return ExceptionUtils.rethrow(e);
        }
    }

    private static Class<?> getCacheFactoryClass() {
        try {
            Class<?> factoryClass;
            try {
                factoryClass = Class.forName(CACHE_FACTORY_CLASS_NAME_3);
            } catch (ClassNotFoundException e) {
                factoryClass = Class.forName(CACHE_FACTORY_CLASS_NAME_2);
            }
            return factoryClass;
        }catch (ClassNotFoundException e){
            return ExceptionUtils.rethrow(e);
        }
    }


    @SuppressWarnings({"rawtypes", "unchecked"}) /* Cache 中可能放 V, 也可能放 String(PROTOCOL_ATTACHMENT+fileUrl) */
    public static class WrapperCache<V> implements ICache<V> {
        private static final String PROTOCOL_ATTACHMENT = "yigo-attachment:";
        private Logger logger = LoggerFactory.getLogger(this.getClass());

        private ICache innerCache;
        private String cacheKey;
        private String formKey;
        private int limitSize;

        public WrapperCache (ICache cache, String cacheKey, String formKey,int limitSize){
            this.innerCache = cache;
            this.cacheKey = cacheKey;
            this.formKey = formKey;
            this.limitSize = limitSize;
        }

        @Override
        public void put(String key, V value) {
            // 如果是原类数据,直接使用原本缓存实例存放数据
            if( value.getClass().isPrimitive()) {
                innerCache.put(key,value);
                return;
            }
            // 如果是数据计算小于1kb,直接使用原本缓存实例存放数据
            byte[] bytes = SerializeUtil.serialize(value);
            if( bytes.length < limitSize ) {
                innerCache.put(key,value);
                return;
            }
            String fileUrl = SessionUtils.processWithContext("", (ctx) -> {
                String dir = cacheKey+"/"+DateFormatUtils.format(new Date(),STORE_DIR_DATE_FORMAT);
                long oid = System.currentTimeMillis();
                String url = AttachmentUtil.newProvider("", ctx.getVE()).
                        upload(ctx, key, formKey, oid, "", dir, bytes, true);
                return url;
            });
            logger.debug("缓存(name='{}',key='{}') 存放数据大于 {}, 使用附件模式记录, 附件路径 '{}'",cacheKey,key,limitSize,fileUrl);
            innerCache.put(key, PROTOCOL_ATTACHMENT+fileUrl);
        }

        @Override
        public V get(String key) {
            Object result = innerCache.get(key);
            if(result instanceof String){
                if(result.toString().startsWith(PROTOCOL_ATTACHMENT)){

                    String fileUrl = result.toString().substring(PROTOCOL_ATTACHMENT.length());
                    byte[] data = SessionUtils.processWithContext("",(ctx)->{
                        byte[] bytes= (byte[]) AttachmentUtil.newProvider("",ctx.getVE()).
                                download(ctx,formKey,fileUrl);
                        return bytes;
                    });
                    logger.debug("缓存(name='{}',key='{}') 存放内容为大数据, 附件路径为 '{}'",cacheKey,key,result);
                    return (V) SerializeUtil.unSerialize(data);
                }
            }
            return (V) result;
        }

        @Override
        public boolean contains(String key) {
            return innerCache.contains(key);
        }

        @Override
        public void remove(String key) {
            innerCache.remove(key);
        }

        @Override
        public void removeAll(List<String> keyList) {
            innerCache.removeAll(keyList);
        }

        @Override
        public Set<String> getKeys() {
            return innerCache.getKeys();
        }

        @Override
        public void clear() {
            innerCache.clear();
        }

        @Override
        public long size() {
            return innerCache.size();
        }

        @Override
        public boolean isEmpty() {
            return innerCache.isEmpty();
        }

        @Override
        public Map<String, V> getAll(List<String> keyList) {
            Map<String, V> result = new HashMap();
            if(null != keyList && !(keyList.isEmpty())){
                for(String key:keyList){
                    result.put(key, this.get(key));
                }
            }
            return result;
        }

        @Override
        public Long[] findGroupIds(Object[][] objects) {
            return innerCache.findGroupIds(objects);
        }

        @Override
        public Boolean insertGroupIds(Object[][] objects, Long[] longs) {
            return innerCache.insertGroupIds(objects,longs);
        }

        @Override
        public long incr(String s, long l, ICacheCallback iCacheCallback) {
            return innerCache.incr(s, l, iCacheCallback);
        }

        public Object getOriginalValue(String key){
            Object result = innerCache.get(key);
            return result;
        }
    }
}
