package com.bokesoft.distro.tech.yigosupport.extension.system;

import com.bokesoft.distro.tech.commons.basis.trace.TraceUtil;
import com.bokesoft.distro.tech.yigosupport.extension.system.dto.SystemInfoDTO;
import com.bokesoft.distro.tech.yigosupport.extension.system.request.SystemRequestHolder;
import com.bokesoft.yes.mid.ver.SystemInfo;
import com.bokesoft.yigo.mid.scheduler.QuartzManager;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.quartz.JobExecutionContext;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SchedulerFactory;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

public class SystemInfoService {

    public SystemInfoDTO getSystemInfo() {
        SystemInfoDTO info = new SystemInfoDTO();

        // Yigo版本
        info.setYigoVersion(getYigoVersion());

        // ERP版本
        info.setErpVersion(getErpVersion());

        // 当前请求信息
        info.setCurrentRequests(getRequests());

        // 当前正在执行的 QuartzJob 线程
        info.setExecutingQuartzJobs(getExecutingJobs());

        return info;
    }

    public List<String> getRequests() {
        HttpServletRequest curHttpReq = SystemRequestHolder.getCurrentRequest();
        Map<String, HttpServletRequest> currentRequests = SystemRequestHolder.getAliveRequests();
        List<String> requestStrList = new ArrayList<>();
        for (Map.Entry<String, HttpServletRequest> httpServletRequestEntry : currentRequests.entrySet()) {
            HttpServletRequest request = httpServletRequestEntry.getValue();
            // request是值引用地址寄存,所以直接用“==”
            if(!(request == curHttpReq)){
                // 请求记数,应该排除当前请求
                String requestURI = request.getRequestURI();
                String requestMethod = request.getMethod();
                String requestMsg = String.format("[正在执行的请求: %s, 请求方法: %s, TraceID: %s]", requestURI, requestMethod, TraceUtil.getTraceId());
                requestStrList.add(requestMsg);
            }
        }
        return requestStrList;
    }

    public List<String> getExecutingJobs() {
        try {
            QuartzManager quartzManager = QuartzManager.getInstance();
            SchedulerFactory schedulerFactory = (SchedulerFactory) FieldUtils.readField(
                    quartzManager, "schedulerFactory", true);
            Scheduler scheduler = schedulerFactory.getScheduler();
            // 获取所有 Job 名称
            return getExecutingJobMsgsStr(scheduler);
        } catch (SchedulerException | IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }

    private List<String> getExecutingJobMsgsStr(Scheduler scheduler) throws SchedulerException {
        List<String> jobMsgs = new ArrayList<>();
        for (JobExecutionContext currentlyExecutingJob : scheduler.getCurrentlyExecutingJobs()) {
            String jobMsg = getJobMsgsStr(currentlyExecutingJob);
            jobMsgs.add(jobMsg);
        }
        return jobMsgs;
    }

    private static String getJobMsgsStr(JobExecutionContext currentlyExecutingJob) {
        String jobName = currentlyExecutingJob.getJobDetail().getKey().getName();
        String jobGroup = currentlyExecutingJob.getJobDetail().getKey().getGroup();
        Date scheduledFireTime = currentlyExecutingJob.getScheduledFireTime();
        long jobRunTime = Duration.between(scheduledFireTime.toInstant(), new Date().toInstant()).toMillis();
        return String.format("[正在执行的 Job 名称: %s, 分组: %s, 运行时间: %d 毫秒]", jobName, jobGroup, jobRunTime);
    }

    private String getErpVersion() {
        try {
            Class<?> clazz = Class.forName("com.bokesoft.yes.util.ProjectSystemInfoUtil");

            Method getVersionMethod = clazz.getMethod("getVersion");
            Method getBuildIDMethod = clazz.getMethod("getBuildID");
            Method getBuildRevisionMethod = clazz.getMethod("getBuildRevision");

            String version = (String) getVersionMethod.invoke(null);
            String buildId = (String) getBuildIDMethod.invoke(null);
            String revision = (String) getBuildRevisionMethod.invoke(null);

            return version + "@" + buildId + "@" + revision;
        } catch (ClassNotFoundException e) {
            return "UNKNOWN";
        } catch (NoSuchMethodException | InvocationTargetException | IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }

    private String getYigoVersion() {
        return SystemInfo.getVersion() + "@" + SystemInfo.getBuildID();
    }

}
