package com.bokesoft.distro.tech.yigosupport.extension.dee;

import java.util.Date;
import java.util.List;

import org.apache.commons.lang3.StringUtils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bokesoft.distro.tech.yigosupport.extension.dee.impl.DTSUtils;
import com.bokesoft.distro.tech.yigosupport.extension.dee.impl.JSONUtils;
import com.bokesoft.distro.tech.yigosupport.extension.dee.impl.NodeItr;
import com.bokesoft.yes.dts.DataTransferService;
import com.bokesoft.yes.dts.data.DTSDataProvider;
import com.bokesoft.yes.dts.types.OperationTypes;
import com.bokesoft.yes.dts.types.TransactionType;
import com.bokesoft.yigo.mid.base.DefaultContext;

/**
 * 基于 Yigo 对 DEE 的支持，实现对数据写入到 Yigo 的支持
 */
public class DeeImporter {
	private JSONArray data;
	private DeeImporter() {
		//私有构造函数保证不会被直接 new 出 DeeImporter 对象
		this.data = new JSONArray();
	}
	
	/**
	 * 通过 JSON 数组 构造 DeeImporter
	 * @param data
	 * @return
	 */
	public static DeeImporter fromJSONArray(JSONArray data) {
		DeeImporter di = new DeeImporter();
		di.data = data;
		return di;
	}
	
	/**
	 * 通过 JSON 对象 构造 DeeImporter
	 * @param json
	 * @return
	 */
	public static DeeImporter fromJSONObject(JSONObject json) {
		DeeImporter di = new DeeImporter();
		JSONArray data = new JSONArray();
		data.add(json);
		di.data = data;
		return di;
	}
	
	/**
	 * 通过 Java 对象列表 构造 DeeImporter
	 * @param list
	 * @return
	 */
	public static <T> DeeImporter fromList(List<T> list) {
		DeeImporter di = new DeeImporter();
		
		JSONArray data = new JSONArray();
		
		for(T o: list) {
			JSONObject json = (JSONObject) JSON.toJSON(o);
			data.add(json);
		}
		
		di.data = data;
		return di;
	}

	/**
	 * 通过 Java 对象 构造 DeeImporter
	 * @param object
	 * @return
	 */
	public static DeeImporter fromObject(Object object) {
		JSONObject json = (JSONObject) JSON.toJSON(object);
		return fromJSONObject(json);
	}

	/**
	 * 获取 DeeImporter 内部通过 JSON 数组 方式存储的数据
	 * @return
	 */
	public JSONArray getData() {
		return data;
	}

	/**
	 * 重命名 JSON 对象节点
	 * @param parentJsonPath 通过 JSONPath 指定需要修改其子节点名字的 JSON 数据节点
	 * @param oldNodeName 子节点原名
	 * @param newNodeName 子节点修改后的新节点名
	 * @return
	 */
	public DeeImporter renameNode(String parentJsonPath, String oldNodeName, String newNodeName) {
		JSONUtils.foreachJSONPath(this.data, parentJsonPath, new NodeItr() {
			@Override
			public void process(JSONObject node) {
				Object child = node.get(oldNodeName);
				if (null!=child) {
					node.remove(oldNodeName);
					node.put(newNodeName, child);
				}
			}
		});

		return this;
	}
	
	/**
	 * 移除指定的 JSON 节点, 通常用于清除不需要的字段
	 * @param parentJsonPath 通过 JSONPath 指定需要其子节点内容需要合并为字符串的 JSON 数据节点
	 * @param nodeName 子节点名
	 * @return
	 */
	public DeeImporter removeNode(String parentJsonPath, String nodeName) {
		JSONUtils.foreachJSONPath(this.data, parentJsonPath, new NodeItr() {
			@Override
			public void process(JSONObject node) {
				node.remove(nodeName);
			}
		});

		return this;
	}

	/**
	 * 将 JSON 节点的内容从具有结构的对象合并为其JSON字符串形式
	 * @param parentJsonPath 通过 JSONPath 指定需要其子节点内容需要合并为字符串的 JSON 数据节点
	 * @param nodeName 子节点名
	 * @return
	 */
	public DeeImporter aggregateAsJSONString(String parentJsonPath, String nodeName) {
		JSONUtils.foreachJSONPath(this.data, parentJsonPath, new NodeItr() {
			@Override
			public void process(JSONObject node) {
				Object child = node.get(nodeName);
				if (null!=child) {
					if ( !(child instanceof Number)	//基本类型的数据, 不需要合并为字符串
							&& !(child instanceof String)
							&& !(child instanceof Date) ){
						node.put(nodeName, JSON.toJSONString(child));
					}
				}
			}
		});

		return this;
	}

	/**
	 * 配置主表的导入属性 - DataObjectKey, 参考 {@link DTSDataProvider#META_KEY_THING_NAME}
	 * @param thingName
	 * @return
	 */
	public DeeImporter configThingName(String thingName) {
		DTSUtils.configHeader(this.data, DTSDataProvider.META_KEY_THING_NAME, thingName);
		return this;
	}

	/**
	 * 配置主表的导入属性 - 表单 Key, 参考 {@link DTSDataProvider#META_KEY_FORMKEY}
	 * @param formKey
	 * @return
	 */
	public DeeImporter configFormKey(String formKey) {
		DTSUtils.configHeader(this.data, DTSDataProvider.META_KEY_FORMKEY, formKey);
		return this;
	}
	
	/**
	 * 配置主表的导入属性 - 当前数据记录的计划操作类型(insert update delete insert-or-update);
	 * 参考 {@link DTSDataProvider#META_KEY_OP_PLAN}
	 * @param opPlan
	 * @return
	 */
	public DeeImporter configOpPlan(OperationTypes opPlan) {
		String opName = DTSUtils.opPlan2Str(opPlan);
		return configOpPlan(opName);
	}
	
	/**
	 * 配置主表的导入属性 - 当前数据记录的计划操作类型(insert update delete insert-or-update);
	 * 参考 {@link DTSDataProvider#META_KEY_OP_PLAN}
	 * @param opPlanName
	 * @return
	 */
	public DeeImporter configOpPlan(String opPlanName) {
		DTSUtils.configHeader(this.data, DTSDataProvider.META_KEY_OP_PLAN, opPlanName);
		return this;
	}
	
	/**
	 * 配置主表的导入属性 - 数据记录的唯一标识字段名(可以指定多个字段), 参考 {@link DTSDataProvider#META_KEY_ID_FIELDS}
	 * @param idFields
	 * @return
	 */
	public DeeImporter configIdFields(String ... idFields) {
		String flds = StringUtils.join(idFields, ",");
		DTSUtils.configHeader(this.data, DTSDataProvider.META_KEY_ID_FIELDS, flds);
		return this;
	}

    /**
     * 配置主表的导入属性 - 需要返回的字段, 参考 {@link DTSDataProvider#META_KEY_RETURN}
     * @param field
     * @return
     */
    public DeeImporter configReturnFields(String ... field) {
        String fields = StringUtils.join(field, ",");
        DTSUtils.configHeader(this.data, DTSDataProvider.META_KEY_RETURN, fields);
        return this;
    }

	/**
	 * 配置主表的导入属性 - 用于控制是否自动删除未传入的明细记录, 其值为明细的数据定义名称(可以制定多个表名), "*" 代表所有;
	 * 参考 {@link DTSDataProvider#META_KEY_DELETE_UNKNOWN_DETAIL}
	 * @param tableKeys
	 * @return
	 */
	public DeeImporter configAutoDeleteUnknownDetail(String ... tableKeys) {
		String tbls = StringUtils.join(tableKeys, ",");
		DTSUtils.configHeader(this.data, DTSDataProvider.META_KEY_DELETE_UNKNOWN_DETAIL, tbls);
		return this;
	}
	
	/**
	 * 配置明细表的导入属性 - 当前数据记录的计划操作类型(insert update delete insert-or-update);
	 * 参考 {@link DTSDataProvider#META_KEY_OP_PLAN}
	 * @param tableName
	 * @param opPlan
	 * @return
	 */
	public DeeImporter configDetailOpPlan(String tableName, OperationTypes opPlan) {
		String opName = DTSUtils.opPlan2Str(opPlan);
		return configDetailOpPlan(tableName, opName);
	}
	
	/**
	 * 配置明细表的导入属性 - 当前数据记录的计划操作类型(insert update delete insert-or-update);
	 * 参考 {@link DTSDataProvider#META_KEY_OP_PLAN}
	 * @param tableName
	 * @param opPlanName
	 * @return
	 */
	public DeeImporter configDetailOpPlan(String tableName, String opPlanName) {
		DTSUtils.configDetail(tableName, this.data, DTSDataProvider.META_KEY_OP_PLAN, opPlanName);
		return this;
	}
	
	/**
	 * 配置明细表的导入属性 - 数据记录的唯一标识字段名(可以指定多个字段), 参考 {@link DTSDataProvider#META_KEY_ID_FIELDS}
	 * @param tableName
	 * @param idFields
	 * @return
	 */
	public DeeImporter configDetailIdFields(String tableName, String ... idFields) {
		String flds = StringUtils.join(idFields, ",");
		DTSUtils.configDetail(tableName, this.data, DTSDataProvider.META_KEY_ID_FIELDS, flds);
		return this;
	}
	
	/**
	 * 将多行数据保存到 Yigo(独立事务)
	 * @param ctx
     * @return
	 * @throws Throwable
	 */
	public org.json.JSONArray importData(DefaultContext ctx) throws Throwable {
		return _doImport(ctx, TransactionType.INDEPENDENT /* 独立事务 */);
    }

	/**
	 * 将多行数据保存到 Yigo(整体事务)
	 * @param ctx
	 * @throws Throwable
	 */
	public org.json.JSONArray saveData(DefaultContext ctx) throws Throwable {
		return _doImport(ctx, TransactionType.WHOLE /* 整体事务 */);
	}

	private org.json.JSONArray _doImport(DefaultContext ctx, int transType) throws Throwable {
		DTSDataProvider dtsDataBuilder = new DTSDataProvider();
        return (org.json.JSONArray)DataTransferService.
        		multiDataTransfer(ctx, dtsDataBuilder, this.data.toJSONString(), transType);
	}
	
	/**
	 * 通过自定义接口调用远程接口(要求: in/out 都是 JSONArray 格式的字符串)
	 * @param conn
	 * @return
	 * @throws Throwable
	 */
	public org.json.JSONArray callRemoteGateway(RawConnection conn) {
		String result = conn.post(this.data.toJSONString());
		return new org.json.JSONArray(result);
	}
	
	/**
	 * 远程接口调用的抽象
	 */
	public static interface RawConnection {
		/**
		 * 提交 DeeImport 数据并返回处理结果
		 * @param data 符合 DEE(Yigo 的 DTS)要求的 JSONArray 格式的字符串
		 * @return 处理结果, JSONArray 格式的字符串
		 */
		public String post(String data);
	}
}
