package com.bokesoft.yes.excel.parser;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Iterator;

import com.bokesoft.yes.excel.document.ExcelDataTable;
import com.bokesoft.yes.excel.document.ExcelDetailDataRow;
import com.bokesoft.yes.excel.document.ExcelExpandDataRow;
import com.bokesoft.yes.excel.document.IExcelDataRow;
import com.bokesoft.yes.excel.template.ExcelCell;
import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.excel.BaseExportExcelFunctionImpl;
import com.bokesoft.yigo.meta.dataobject.MetaColumn;
import com.bokesoft.yigo.parser.BaseFunImplCluster;

/**
 * 基于模板导出Excel用到的中间层公式
 * 
 * @author guowj
 *
 */
public class ExcelFunction extends BaseFunImplCluster {

	/**
	 * 列扩展的横向汇总公式，参数为【fieldIndex】
	 * <ul>
	 * <li>fieldIndex 扩展字段的单元格列号
	 * </ul>
	 * 
	 * @author guowj
	 */
	class SumExpandLandscapeImpl extends BaseExportExcelFunctionImpl{

		@Override
		public Object evalImpl(String name, ExcelContext context, Object[] args) throws Throwable {
			int fieldIndex = TypeConvertor.toInteger(args[0]);
		
			MetaColumn metaColumn = null;
			Object value = null;
			BigDecimal result = BigDecimal.ZERO;
			ExcelCell excelCell = context.getExcelRow().getCellByIndexInExcel(fieldIndex);
			
			Iterator<IExcelDataRow> rowList = context.getExcelExpandDataRow().iterator();
			IExcelDataRow erow = null;
			while(rowList.hasNext()) {
				erow = rowList.next();
				metaColumn = context.getMetaTable().get(excelCell.getDefinition());

				if(metaColumn != null) {
					value = context.getDataTable().getObject(erow.getRowIndex(), metaColumn.getKey());
					BigDecimal dValue = TypeConvertor.toBigDecimal(value);
					result = result.add(dValue);
				}
			}
			
			return result;
		}
		
	}
	
	/**
	 * 列扩展的纵向汇总公式，参数为【efi, expandFieldKey, columnKey】
	 * <ul>
	 * <li>efi 列扩展数据字段包含的多个value, 形如：{1,2,3}
	 * <li>expandFieldKey 列扩展的字段
	 * <li>columnKey 列扩展数据字段
	 * </ul>
	 * 
	 * @author guowj
	 */
	class SumExpandPortraitImpl extends BaseExportExcelFunctionImpl{

		@Override
		public Object evalImpl(String name, ExcelContext context, Object[] args) throws Throwable {
			
			String efi = TypeConvertor.toString(args[0]);
			if(efi.startsWith("{") && efi.endsWith("}")) {
				efi = efi.substring(1,efi.length()-1);
			}
			String[] expandFieldIndexArray = efi.split(",");
			int expandFieldIndex = TypeConvertor.toInteger(expandFieldIndexArray[context.getExpandCount()]);

			String expandFieldKey = TypeConvertor.toString(args[1]);
			String columnKey = TypeConvertor.toString(args[2]);
		
			BigDecimal result = BigDecimal.ZERO;
			
			// 同一列的值的列表
			ArrayList<IExcelDataRow> list = new ArrayList<IExcelDataRow>();
			
			String tableKey = context.getExcelRow().getTableKey();
			ExcelDataTable excelDataTable = context.getExcelDataTableMap().get(tableKey);
			Iterator<IExcelDataRow> itExcelDataRows = excelDataTable.iterator();
			ExcelExpandDataRow exrow = null;
			while(itExcelDataRows.hasNext()) {
				Object obj = null;
				exrow = (ExcelExpandDataRow)itExcelDataRows.next();
				Iterator<IExcelDataRow> rowList = exrow.iterator();
				IExcelDataRow erow = null;
				while(rowList.hasNext()) {
					erow = rowList.next();
					obj = context.getDocument().get(tableKey).getObject(erow.getRowIndex(), expandFieldKey);
					if(TypeConvertor.toInteger(obj) == expandFieldIndex) {
						list.add(erow);
					}
				}
			}
			
			// 计算同一列的值的总和
			for(IExcelDataRow detailRow : list) {
				Object value = context.getDocument().get(tableKey).getObject(detailRow.getRowIndex(), columnKey);
				BigDecimal dValue = TypeConvertor.toBigDecimal(value);
				result = result.add(dValue);
			}
			
			return result;
		}
		
	}
	
	/**
	 * 列扩展的区域汇总公式，参数为【 expandFieldKey, columnKey】
	 * <ul>
	 * <li>expandFieldKey 列扩展的字段
	 * <li>columnKey 列扩展数据字段
	 * </ul>
	 * 
	 */
	class SumExpandTotalImpl extends BaseExportExcelFunctionImpl{

		@Override
		public Object evalImpl(String name, ExcelContext context, Object[] args) throws Throwable {

			String expandFieldKey = TypeConvertor.toString(args[0]);
			String columnKey = TypeConvertor.toString(args[1]);
		
			BigDecimal result = BigDecimal.ZERO;
			
			// 同一列的值的列表
			ArrayList<IExcelDataRow> list = new ArrayList<IExcelDataRow>();
			
			String tableKey = context.getExcelRow().getTableKey();
			ExcelDataTable excelDataTable = context.getExcelDataTableMap().get(tableKey);
			Iterator<IExcelDataRow> itExcelDataRows = excelDataTable.iterator();
			ExcelExpandDataRow exrow = null;
			while(itExcelDataRows.hasNext()) {
				Object obj = null;
				exrow = (ExcelExpandDataRow)itExcelDataRows.next();
				Iterator<IExcelDataRow> rowList = exrow.iterator();
				IExcelDataRow erow = null;
				while(rowList.hasNext()) {
					erow = rowList.next();
					obj = context.getDocument().get(tableKey).getObject(erow.getRowIndex(), expandFieldKey);
					list.add(erow);
				}
			}
			
			// 计算同一列扩展区域内的值的总和
			for(IExcelDataRow detailRow : list) {
				Object value = context.getDocument().get(tableKey).getObject(detailRow.getRowIndex(), columnKey);
				BigDecimal dValue = TypeConvertor.toBigDecimal(value);
				result = result.add(dValue);
			}
			
			return result;
		}
		
	}
	
	/**
	 * 明细行的纵向汇总公式，参数为【fieldKey】
	 * <ul>
	 * <li>fieldKey 明细行字段
	 * </ul>
	 * 
	 * @author guowj
	 */
	class SumPortraitImpl extends BaseExportExcelFunctionImpl{

		@Override
		public Object evalImpl(String name, ExcelContext context, Object[] args) throws Throwable {
			String fieldKey = TypeConvertor.toString(args[0]);
			BigDecimal result = BigDecimal.ZERO;
			
			String tableKey = context.getExcelRow().getTableKey();
			ExcelDataTable excelDataTable = context.getExcelDataTableMap().get(tableKey);
			Iterator<IExcelDataRow> itExcelDataRows = excelDataTable.iterator();
			ExcelDetailDataRow detailRow = null;
			while(itExcelDataRows.hasNext()) {
				IExcelDataRow itExcelDataRow = itExcelDataRows.next();
				if(itExcelDataRow instanceof ExcelDetailDataRow) {
					detailRow = (ExcelDetailDataRow) itExcelDataRow;
					Object value = context.getDocument().get(tableKey).getObject(detailRow.getRowIndex(), fieldKey);
					BigDecimal dValue = TypeConvertor.toBigDecimal(value);
					result = result.add(dValue);
				}
			}
			
			
			return result;
		}
		
	}
	
	/**
	 * 明细行的横向汇总公式，参数为【fieldIndex】
	 * <ul>
	 * <li>fieldIndex 一组明细行字段的列号{1,2,3}
	 * </ul>
	 * 
	 * @author guowj
	 */
	class SumLandscapeImpl extends BaseExportExcelFunctionImpl{

		@Override
		public Object evalImpl(String name, ExcelContext context, Object[] args) throws Throwable {
			String fieldIndex = TypeConvertor.toString(args[0]);
			String[] fieldIndexArray = fieldIndex.split(",");
			BigDecimal result = BigDecimal.ZERO;
			
			for(String f : fieldIndexArray) {
				int iFieldIndex = TypeConvertor.toInteger(f);
				String definition = context.getExcelRow().getCellByIndexInExcel(iFieldIndex).getDefinition();
				
				ExcelDetailDataRow detailDataRow = context.getExcelDetailDataRow();
				Object value = context.getDataTable().getObject(detailDataRow.getRowIndex(), definition);
				BigDecimal dValue = TypeConvertor.toBigDecimal(value);
				result = result.add(dValue);
			}
			
			return result;
		}
		
	}
	
	/**
	 * 分组汇总行的汇总公式，参数为【definition】
	 * <ul>
	 * <li>definition 明细行字段
	 * </ul>
	 * 
	 * @author guowj
	 */
	class SumGroupImpl extends BaseExportExcelFunctionImpl{

		@Override
		public Object evalImpl(String name, ExcelContext context, Object[] args) throws Throwable {
			String definition = TypeConvertor.toString(args[0]);
			BigDecimal result = BigDecimal.ZERO;
			
			String tableKey = context.getExcelRow().getTableKey();
			ArrayList<IExcelDataRow> detailRows = context.getDetailRows();
			for(IExcelDataRow row : detailRows) {
				//该row不用强转成ExcelDetailDataRow，因为getRowIndex()是IExcelDataRow的接口方法
				Object value = context.getDocument().get(tableKey).getObject(row.getRowIndex(), definition);
				BigDecimal dValue = TypeConvertor.toBigDecimal(value);
				result = result.add(dValue);
			}
			
			return result;
		}
		
	}
	
	/**
	 * 填充分组行的上一级分组字段，参数为【definition】
	 * <ul>
	 * <li>definition 明细行字段
	 * </ul>
	 * 
	 * @author guowj
	 */
	class FillGroupImpl extends BaseExportExcelFunctionImpl{

		@Override
		public Object evalImpl(String name, ExcelContext context, Object[] args) throws Throwable {
			String definition = TypeConvertor.toString(args[0]);
			String result = "";
			
			String tableKey = context.getExcelRow().getTableKey();
			ArrayList<IExcelDataRow> detailRows = context.getDetailRows();
			// 分组行包含的每一个明细行的上一级分组字段是一样的
			IExcelDataRow row = detailRows.get(0);
			Object value = context.getDocument().get(tableKey).getObject(row.getRowIndex(), definition);

			result = TypeConvertor.toString(value);
			
			return result;
		}
		
	}

	@Override
	public Object[][] getImplTable() {
		return new Object[][] { 
			{ "SumExpandLandscape", new SumExpandLandscapeImpl() },
			{ "SumExpandPortrait", new SumExpandPortraitImpl() },
			{ "SumPortrait", new SumPortraitImpl() },
			{ "SumLandscape", new SumLandscapeImpl() },
			{ "SumGroup", new SumGroupImpl() },
			{ "FillGroup", new FillGroupImpl() },
			{ "SumExpandTotal", new SumExpandTotalImpl() },
//			{ "SumMixLandscape", new SumMixLandscapeImpl() }
		};
	}
}
