package com.bokesoft.yes.excel.document;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map.Entry;
import java.util.TreeMap;

import com.bokesoft.yes.common.struct.MultiKey;
import com.bokesoft.yes.common.struct.MultiKeyNode;
import com.bokesoft.yes.common.util.DBTypeUtil;
import com.bokesoft.yes.common.util.StringUtil;
import com.bokesoft.yes.excel.parser.ExcelContext;
import com.bokesoft.yes.excel.template.ExcelSheet;
import com.bokesoft.yes.excel.template.ExcelWorkbook;
import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.meta.dataobject.MetaColumn;
import com.bokesoft.yigo.meta.dataobject.MetaDataObject;
import com.bokesoft.yigo.meta.dataobject.MetaTable;
import com.bokesoft.yigo.struct.datatable.DataTable;
import com.bokesoft.yigo.struct.document.Document;

public class ExcelDocumentProcess {
	private ExcelWorkbook workbook;
	
	private Document document;
	
	private HashMap<String, ExcelDataTable> excelDataTableMap= new HashMap<String, ExcelDataTable>();
	
	private MetaDataObject metaDataObject;
	
	private ExcelContext excelContext;
	
	public ExcelDocumentProcess(MetaDataObject metaDataObject, ExcelWorkbook workbook, Document document, ExcelContext excelContext) {
		this.workbook = workbook;
		this.document = document;
		this.metaDataObject = metaDataObject;
		this.excelContext = excelContext;
	}
	
	public void process() {
		// 遍历所有表格行 将扩展行
		Iterator<ExcelSheet> excelSheets = workbook.iterator();
		ExcelSheet excelSheet = null;
		while(excelSheets.hasNext()) {
			excelSheet = excelSheets.next();
			ArrayList<ExcelGridInfo> gridInfoList = excelSheet.getGridInfoList();
			for (ExcelGridInfo info : gridInfoList) {
				processExcelGrid(excelSheet, document, info);
			}
		}
		excelContext.setExcelDataTableMap(excelDataTableMap);
	}
	
	private void processExcelGrid(ExcelSheet excelSheet, Document document, ExcelGridInfo info) {
		String tableKey = info.getTableKey();
		if (StringUtil.isBlankOrNull(tableKey)) {
			return;
		}
		
		MetaTable metaTable = metaDataObject.getMetaTable(tableKey);
		
		ArrayList<String> primaryFieldList = new ArrayList<String>();
		ArrayList<Integer> primaryFieldTypeList = new ArrayList<Integer>();
		
		ArrayList<Integer> groupFieldTypeList = new ArrayList<Integer>();
		
		for(String columnKey : info.getGroupFieldList()) {
			groupFieldTypeList.add(DBTypeUtil.dataType2JavaDataType(metaTable.get(columnKey).getDataType()));
		}
		
		ExcelDataTable excelDataTable = excelDataTableMap.get(tableKey);
		if(!excelDataTableMap.containsKey(tableKey)) {
			excelDataTable = new ExcelDataTable();
			excelDataTableMap.put(tableKey, excelDataTable);
		}
		
		// 循环所有数据集，通过多个关键字段的值组成键值 生成expendRow
		DataTable dataTable = document.get(tableKey);
		int rowCount = dataTable == null ? 0 : dataTable.size();
		
		Iterator<MetaColumn> itMetaColumns = metaTable.iterator();
		MetaColumn metaColumn = null;
		// 找到数据源配置对象 加入关键字
		while (itMetaColumns.hasNext()) {
			metaColumn = itMetaColumns.next();
			if (metaColumn.getIsPrimary()) {
				primaryFieldList.add(metaColumn.getBindingDBColumnName());
				primaryFieldTypeList.add(DBTypeUtil.dataType2JavaDataType(metaColumn.getDataType()));
			}
		}

		String[] primaryFields = primaryFieldList.toArray(new String[0]);
		Integer[] primaryFieldTypes = primaryFieldTypeList.toArray(new Integer[0]);
		
		for (int index = 0; index < rowCount; index++) {
			if (info.isColumnExpand()) {
				MultiKey multiKey = makeMultiKey(dataTable, index, primaryFields, primaryFieldTypes);
				ExcelExpandDataRow expandRow = excelDataTable.getExpandRow(multiKey);
				if (expandRow == null) {
					expandRow = new ExcelExpandDataRow();
					excelDataTable.addExpandDataRow(multiKey, expandRow);
				}
				
				ExcelDetailDataRow detailRow = new ExcelDetailDataRow();
				detailRow.setRowIndex(index);
				expandRow.addDetailRow(detailRow);
			} else if(info.isHasSum()) {
				String[] groupFields = info.getGroupFieldList().toArray(new String[0]);
				Integer[] groupFieldTypes = groupFieldTypeList.toArray(new Integer[0]);
				MultiKey multiKey = makeMultiKey(dataTable, index, groupFields, groupFieldTypes);
				ExcelGroupDataRow groupRow = excelDataTable.getGroupRow(multiKey);
				String sign = "";
				// 最内层，groupLevel是1
				Integer groupLevel = 1;
				if (groupRow == null) {
					groupRow = new ExcelGroupDataRow();
					// 由definition得到value，构造ExcelGroupDataRow中内外层一致的valueMap
					Iterator<Entry<Integer, GroupInfo>> itEntrySet = info.getGroupLevelMap().entrySet().iterator();
					while(itEntrySet.hasNext()) {
						Entry<Integer, GroupInfo> entrySet = itEntrySet.next();
						String definition = entrySet.getValue().getDefinition();
						Object value = dataTable.getObject(index, definition);
						groupRow.addValueMap(entrySet.getKey(), value);
					}	
					excelDataTable.addGroupDataRow(multiKey, groupRow);
					excelDataTable.addGroupDataRowToList(groupRow);
					
					// 构造用于区分不同分组的标识
					Object value = groupRow.getValueMap().get(groupLevel);
					sign = TypeConvertor.toString(groupLevel) + "#" + TypeConvertor.toString(value);
					// 添加最内层的头部分组行
					addHeadGroupRow(info, groupLevel, sign, groupRow);
				}else {
					// 为具有相同ExcelGroupDataRow的ExcelDetailDataRow构造sign
					ArrayList<IExcelDataRow> rList = groupRow.getDetailRows();
					ExcelDetailDataRow dRow = (ExcelDetailDataRow)groupRow.getRow(rList.size() - 1);
					sign = dRow.getSign(groupLevel);
				}
				// 添加明细行
				ExcelDetailDataRow detailRow = new ExcelDetailDataRow();
				detailRow.setRowIndex(index);
				detailRow.addSign(groupLevel, sign);
				detailRow.addDefinition(groupLevel, info.getGroupLevelMap().get(groupLevel).getDefinition());
				groupRow.addDetailRow(detailRow);
			}else {
				ExcelDetailDataRow detailRow = new ExcelDetailDataRow();
				detailRow.setRowIndex(index);
				excelDataTable.addDetailRow(detailRow);
			}
		}
		
//		 向最内层的头部分组行添加该分组的所有明细行，并且添加最内层的尾部分组行
		ArrayList<ExcelGroupDataRow> groupRows = excelDataTable.getGroupList();
		for(ExcelGroupDataRow groupRow : groupRows){
			Integer groupLevel = 1;
			addTailGroupRow(info, groupLevel, groupRow);
		}
		
		groupRowProcess(info, excelDataTable);		
	}
	
	/**
	 * 添加所有分组行（除了最内层的分组行）
	 * 
	 * @param info
	 * @param excelDataTable
	 */
	private void groupRowProcess(ExcelGridInfo info, ExcelDataTable excelDataTable) {
		
		Iterator<Entry<Integer, GroupInfo>> itEntrySet = info.getGroupLevelMap().entrySet().iterator();
		while(itEntrySet.hasNext()) {
			// groupLevel是升序的
			Entry<Integer, GroupInfo> entrySet = itEntrySet.next();
			Integer groupLevel = entrySet.getKey();
			boolean isMulGroup = info.getGroupLevelMap().entrySet().size() > 1;
			if(isMulGroup && groupLevel.intValue() == 1) {
				continue;
			}
			buildDetailRowAndGroupRow(groupLevel, info, excelDataTable, isMulGroup);	
		}
		
		if(excelDataTable.getSortList() != null) {
			for(ExcelGroupDataRow excelGroupDataRow : excelDataTable.getSortList()) {
				excelDataTable.addAllDetailRow(excelGroupDataRow);
			}
		}
	}
	
	private void buildDetailRowAndGroupRow(Integer groupLevel, ExcelGridInfo info, ExcelDataTable excelDataTable, boolean isMulGroup) {
		HashMap<String, ExcelGroupDataRow> sortMap = new HashMap<String, ExcelGroupDataRow>();
		ArrayList<ExcelGroupDataRow> sortList = new ArrayList<ExcelGroupDataRow>();
		
		ArrayList<ExcelGroupDataRow> collection = null;
		if(excelDataTable.getSortMap() == null) {
			collection = excelDataTable.getGroupList();
		}else {
			collection = excelDataTable.getSortList();
		}
		
		for(ExcelGroupDataRow groupRow : collection) {
			Object value = groupRow.getValueMap().get(groupLevel);
			String sign = TypeConvertor.toString(groupLevel) + "#" + TypeConvertor.toString(value);
			String mapKey = this.createMapKey(groupRow.valueMapIterator(), groupLevel, value);
			ExcelGroupDataRow gpRow = sortMap.get(mapKey);
			if(gpRow == null) {
				gpRow = new ExcelGroupDataRow();
				sortMap.put(mapKey, gpRow);
				sortList.add(gpRow);
				
				if(isMulGroup) {
					// 因为具有相同ExcelGroupDataRow的ExcelDetailDataRow，其外层的分组字段的value是一样的
					gpRow.addAllValueMap(groupRow.getValueMap());
					// 添加头部分组行
					addHeadGroupRow(info, groupLevel, sign, gpRow);
				}
			}
			Iterator<IExcelDataRow> rows = groupRow.iterator();
			while(rows.hasNext()) {
				IExcelDataRow r = rows.next();
				if(r.getType().equalsIgnoreCase("Detail")) {
					ExcelDetailDataRow row = (ExcelDetailDataRow)r;
					row.addSign(groupLevel, sign);
					row.addDefinition(groupLevel, info.getGroupLevelMap().get(groupLevel).getDefinition());
				}
				gpRow.addDetailRow(r);
			}
		}
		
		excelDataTable.setSortMap(sortMap);
		excelDataTable.setSortList(sortList);
		
		if(!isMulGroup) {
			return;
		}
		// 向头部分组行添加该分组的所有明细行，并且添加尾部分组行
		for(ExcelGroupDataRow gpRow : sortList) {
			addTailGroupRow(info, groupLevel, gpRow);
		}
	}
	
	private String createMapKey(Iterator<Entry<Integer, Object>> valueMapEntry, Integer currentLevel ,Object currentValue) {
		Entry<Integer, Object> entry = null;
		String mapKey = currentLevel + "#" +currentValue;
		while(valueMapEntry.hasNext()) {
			entry = valueMapEntry.next();
			Integer level = entry.getKey();
			Object value = entry.getValue();
			if(level.intValue() > currentLevel.intValue()) {
				mapKey = level + "#" + value + "-" + mapKey;
			}	
		}
		return mapKey;
	}
	
	/**
	 * 添加头部分组行
	 * 
	 * @param info 记录Excel模板中的表格信息
	 * @param groupLevel 分组级别
	 * @param sign 不同分组的标识
	 * @param groupRow 分组行
	 */
	private void addHeadGroupRow(ExcelGridInfo info, Integer groupLevel, String sign, ExcelGroupDataRow groupRow) {
		TreeMap<Integer, GroupInfo> groupLevelMap = info.getGroupLevelMap();
		GroupInfo groupInfo = groupLevelMap.get(groupLevel);
		if(groupInfo == null) {
			return;
		}
		String position = groupInfo.getGroupPosition();
		if("head".equalsIgnoreCase(position) || "all".equalsIgnoreCase(position)) {
			int groupHeadRowCount = groupInfo.getGroupHeadRowCount();
			for(int i = 1 ; i <= groupHeadRowCount; i++) {
				ExcelGroupDetailRow groupDetailRow = new ExcelGroupDetailRow();
				groupDetailRow.setGroupLevel(groupLevel);
				groupDetailRow.setExcelRow(groupInfo.getExcelRow());
				groupDetailRow.setSign(sign);
				groupDetailRow.setDefinition(groupInfo.getDefinition());
				groupRow.addDetailRow(groupDetailRow);
			}
		}
	}
	
	/**
	 * 向头部分组行添加该分组的所有明细行，并且添加尾部分组行
	 * 
	 * @param info 记录Excel模板中的表格信息
	 * @param groupLevel 分组级别
	 * @param groupRow 分组行
	 */
	private void addTailGroupRow(ExcelGridInfo info, Integer groupLevel, ExcelGroupDataRow groupRow) {
		
		Object value = groupRow.getValueMap().get(groupLevel);
		String sign = TypeConvertor.toString(groupLevel) + "#" + TypeConvertor.toString(value);
		
		// 向ExcelGroupDetailRow添加该分组包含的所有的明细行(除了尾部分组行，尾部分组行单独添加所有分组行)
		ArrayList<IExcelDataRow> groupDetailRows = new ArrayList<IExcelDataRow>();
		ArrayList<IExcelDataRow> detailRows = new ArrayList<IExcelDataRow>();
		Iterator<IExcelDataRow> rows = groupRow.iterator();
		// 将每个ExcelGroupDataRow中的IExcelDataRow进行分类，分为ExcelGroupDetailRow和ExcelDetailDataRow
		while(rows.hasNext()) {
			IExcelDataRow row = rows.next();
			if(row.getType().equalsIgnoreCase("GroupDetail")) {
				groupDetailRows.add(row);
			}else {
				detailRows.add(row);
			}
		}
		
		// 向头部分组行添加该分组包含的所有明细行
		for(IExcelDataRow r : groupDetailRows) {
			ExcelGroupDetailRow gdr = (ExcelGroupDetailRow)r;
			if(gdr.getSign().equalsIgnoreCase(sign)) {
				gdr.setDetailRows(detailRows);
			}
		}
		
		// 添加尾部分组行
		TreeMap<Integer, GroupInfo> groupLevelMap = info.getGroupLevelMap();
		GroupInfo groupInfo = groupLevelMap.get(groupLevel);
		String position = groupInfo.getGroupPosition();
		if("tail".equalsIgnoreCase(position) || "all".equalsIgnoreCase(position)) {
			int groupTailRowCount = groupInfo.getGroupTailRowCount();
			for(int i = 1 ; i <= groupTailRowCount; i++) {
				ExcelGroupDetailRow groupDetailRow = new ExcelGroupDetailRow();
				groupDetailRow.setGroupLevel(groupLevel);
				groupDetailRow.setExcelRow(groupInfo.getExcelRow());
				groupDetailRow.setSign(sign);
				groupDetailRow.setDefinition(groupInfo.getDefinition());
				// 尾部分组行单独添加该分组包含的所有明细行
				groupDetailRow.setDetailRows(detailRows);
				groupRow.addDetailRow(groupDetailRow);
			}
		}
	}
	
	private MultiKey makeMultiKey(DataTable table, int index, String[] fields, Integer[] types) {
		MultiKey value = new MultiKey();
		int length = fields.length;
		for ( int i = 0; i<length; ++i ) {
			value.addValue(new MultiKeyNode(types[i], table.getObject(index, fields[i])));
		}
		
		return value;
	}
	
	public ExcelDataTable getExcelDataTable(String gridKey) {
		return excelDataTableMap.get(gridKey);
	}
}
