package com.bokesoft.yes.excel.cmd.stamp.exportor;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFFormulaEvaluator;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import com.bokesoft.yes.common.util.StringUtil;
import com.bokesoft.yes.excel.cmd.normal.ExcelDataUtil;
import com.bokesoft.yes.excel.cmd.normal.ExcelStyleFactory;
import com.bokesoft.yes.excel.document.ExcelDataTable;
import com.bokesoft.yes.excel.document.ExcelDetailDataRow;
import com.bokesoft.yes.excel.document.ExcelDocumentProcess;
import com.bokesoft.yes.excel.document.ExcelExpandDataRow;
import com.bokesoft.yes.excel.document.ExcelGroupDetailRow;
import com.bokesoft.yes.excel.document.GroupContext;
import com.bokesoft.yes.excel.document.IExcelDataRow;
import com.bokesoft.yes.excel.parser.ExcelContext;
import com.bokesoft.yes.excel.template.ColumnExpandItem;
import com.bokesoft.yes.excel.template.ExcelCell;
import com.bokesoft.yes.excel.template.ExcelDisplay;
import com.bokesoft.yes.excel.template.ExcelRow;
import com.bokesoft.yes.excel.template.ExcelSheet;
import com.bokesoft.yes.excel.template.ExcelWorkbook;
import com.bokesoft.yes.excel.template.util.ExcelUtil;
import com.bokesoft.yes.excel.transformer.ExcelTransformerFactory;
import com.bokesoft.yes.excel.transformer.IExcelTransformer;
import com.bokesoft.yigo.common.def.ScriptType;
import com.bokesoft.yigo.common.util.TypeConvertor;
import com.bokesoft.yigo.meta.dataobject.MetaColumn;
import com.bokesoft.yigo.meta.dataobject.MetaDataObject;
import com.bokesoft.yigo.meta.dataobject.MetaTable;
import com.bokesoft.yigo.meta.exceltemplate.ExcelTemplateCellType;
import com.bokesoft.yigo.meta.exceltemplate.ExcelTemplateRowType;
import com.bokesoft.yigo.meta.exceltemplate.MetaExcelColumn;
import com.bokesoft.yigo.meta.exceltemplate.MetaExcelWorkbook;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.struct.datatable.DataTable;
import com.bokesoft.yigo.struct.document.Document;
import com.bokesoft.yigo.util.StampExcelUtil;

/**
 * 
 * 主要代码从 ExportExcelWithTemplate 提取，用于复用 
 *
 */
public class DocumentExportor4XmlTemplate implements IDocumentExportor {
	/** 上下文环境*/
	private DefaultContext context = null;
	/**　Excel导出的上下文环境*/
	private ExcelContext excelContext = null;
	/** Excel对象*/
	private Workbook workBook = null;
	/** Excel模板配置对象*/
	private MetaExcelWorkbook metaWorkbook = null;
	/** Excel模板处理对象*/
	private ExcelWorkbook virtualWorkbook = null;
	/** 值转换工具类*/
	private ExcelTransformerFactory transformerFactory = null;
	/** Excel单元格格式工具类*/
	private ExcelStyleFactory styleFactory = null;
	
	public DocumentExportor4XmlTemplate(MetaExcelWorkbook metaWorkbook, DefaultContext context) {
		this.context = context;
		this.metaWorkbook = metaWorkbook;
		this.workBook = new XSSFWorkbook();
		this.styleFactory = new ExcelStyleFactory(workBook);
		this.transformerFactory = new ExcelTransformerFactory();
	}

	public Workbook getWorkbook() {
		return this.workBook;
	}
	
	@Override
	public void export(Document document, String documentTitle) throws Throwable {
		this.excelContext = new ExcelContext(context);
		MetaDataObject metaDataObject = document.getMetaDataObject();

		context.setDocument(document);
		context.setFormKey(this.metaWorkbook.getFormKey());
		context.setDataObject(metaDataObject);
		excelContext.setDocument(document);
		// 通过配置文件生成Excel模型
		virtualWorkbook = new ExcelWorkbook(metaDataObject, metaWorkbook, context.getMidParser());
		virtualWorkbook.calcExpandColumn(context.getMidParser(), document, metaDataObject);
		// 处理明细数据集,扩展和汇总
		ExcelDocumentProcess excelDocumentProcess = new ExcelDocumentProcess(metaDataObject, virtualWorkbook, document, excelContext);
		excelDocumentProcess.process();
		
		HashMap<Integer, Integer> columnWidths = null;
		Iterator<ExcelSheet> itExeclSheet = virtualWorkbook.iterator();
		while (itExeclSheet.hasNext()) {
			columnWidths = new HashMap<Integer, Integer>();
			ExcelSheet excelSheet = itExeclSheet.next();
			Sheet sheet = ExcelUtil.getSheet(workBook, StringUtil.isBlankOrNull(documentTitle) ? excelSheet.getName() : documentTitle);

			int offset = 0;
			Iterator<ExcelRow> itExeclRow = excelSheet.iterator();
			while (itExeclRow.hasNext()) {
				ExcelRow excelRow = itExeclRow.next();
				int rowType = excelRow.getType();
				switch (rowType) {
				case ExcelTemplateRowType.Head:
				case ExcelTemplateRowType.Fix:
				case ExcelTemplateRowType.DetailHead:
					exportFixRow(document, metaDataObject, sheet, excelRow, offset, columnWidths);
					break;
				case ExcelTemplateRowType.Detail:
				case ExcelTemplateRowType.Total:
					// 导出明细数据，连带汇总数据一起
					offset = exportDetailRow(excelDocumentProcess, document, metaDataObject, sheet, excelRow, offset, columnWidths);
					break;
				}
			}
			//如设置了列宽进行取值，未设置列宽则默认最佳列宽
			Iterator<MetaExcelColumn> itMetaExeclColumn = excelSheet.columnIterator();
			while(itMetaExeclColumn.hasNext()){
				MetaExcelColumn metaExeclColumn = itMetaExeclColumn.next();
				int columnIndex = metaExeclColumn.getColumnIndex();
				int width = (int)(metaExeclColumn.getWidth()*256/7);
				columnWidths.put(columnIndex - 1, width);				
			}
			ExcelUtil.setSheetColumnWidth(sheet, columnWidths);
		}
	}

	/**
	 * 导出明细数据 并返回行偏移量
	 * @param document 
	 * @param metaForm
	 * @param sheet
	 * @param excelRow
	 * @return
	 * @throws Throwable 
	 */
	private int exportDetailRow(ExcelDocumentProcess process, Document document, MetaDataObject metaDataObject, Sheet sheet, ExcelRow excelRow, int offset, HashMap<Integer, Integer> columnWidths) throws Throwable {
		excelContext.setExcelRow(excelRow);
		String tableKey = excelRow.getTableKey();
		ExcelDataTable excelDataTable = process.getExcelDataTable(tableKey);
		if (excelDataTable == null) {
			// gridKey对应的表不存在
			return offset;
		}
		
		Row row = null;
		int startRowIndex = excelRow.getRowIndex() - 1;
		int firstDetailRowIndex = excelRow.getRowIndex();
		
		DataTable dataTable = document.get(tableKey);
		MetaTable metaTable = metaDataObject.getMetaTable(tableKey);
		excelContext.setDataTable(dataTable);
		excelContext.setMetaTable(metaTable);
		Iterator<IExcelDataRow> itExcelDataRows = excelDataTable.iterator();
		int rowCount = 0;
		// 记录动态合并单元格所需要的上下文
		HashMap<Integer, GroupContext> groupContextMap = new HashMap<Integer, GroupContext>();
		int groupContextCount = 0;
		
		if (excelRow.getType() == ExcelTemplateRowType.Total){
			int rowIndex = startRowIndex + offset;
			//是否存在Excel公式
			for(int i=0;i<excelRow.getCellCount();i++){
				ExcelCell cell = excelRow.getCell(i);
				if(cell.getSourceType().equalsIgnoreCase(ExcelTemplateCellType.STR_ExcelFormula)){
					cell.setExcelFormula(true);
					String formulaKey = cell.getDefinition();
					cell.setExcelFormulaKey(formulaKey);
					String excelFormula = calcExcelFormula(cell,formulaKey,itExcelDataRows,firstDetailRowIndex);
					cell.setDefinition(excelFormula);
				}
			}
					
			row = ExcelUtil.getRow(sheet, rowIndex);
			row.setHeightInPoints((short) excelRow.getHeight());
			itExcelDataRows = excelDataTable.iterator();
			IExcelDataRow excelDataRow = itExcelDataRows.next();
			String type = excelDataRow.getType();
			if ("Expand".equalsIgnoreCase(type)) {
				fillExpandData(row, (ExcelExpandDataRow)excelDataRow, dataTable, excelRow, metaTable, columnWidths);
			} else if ("Detail".equalsIgnoreCase(type)) {
				fillDetailData(sheet, rowIndex, (ExcelDetailDataRow)excelDataRow, dataTable, excelRow, metaTable, columnWidths, groupContextMap, groupContextCount);
			}else if ("GroupDetail".equalsIgnoreCase(type)) {
				fillGroupData(sheet, rowIndex, (ExcelGroupDetailRow)excelDataRow, dataTable, excelRow, metaTable, columnWidths, groupContextMap, groupContextCount);
			}
			
			offset++;
			rowCount++;
			groupContextCount++;
		}else{
			while(itExcelDataRows.hasNext()) {
				int rowIndex = startRowIndex + offset;
				row = ExcelUtil.getRow(sheet, rowIndex);
				row.setHeightInPoints((short) excelRow.getHeight());
				
				IExcelDataRow excelDataRow = itExcelDataRows.next();
				String type = excelDataRow.getType();
				if ("Expand".equalsIgnoreCase(type)) {
					fillExpandData(row, (ExcelExpandDataRow)excelDataRow, dataTable, excelRow, metaTable, columnWidths);
				} else if ("Detail".equalsIgnoreCase(type)) {
					fillDetailData(sheet, rowIndex, (ExcelDetailDataRow)excelDataRow, dataTable, excelRow, metaTable, columnWidths, groupContextMap, groupContextCount);
				}else if ("GroupDetail".equalsIgnoreCase(type)) {
					fillGroupData(sheet, rowIndex, (ExcelGroupDetailRow)excelDataRow, dataTable, excelRow, metaTable, columnWidths, groupContextMap, groupContextCount);
				}
				
				offset++;
				rowCount++;
				groupContextCount++;
			}
			
		}
		
		groupContextMap.clear();

		if (rowCount > 0) {
			offset -= 1;
		}
		return offset;
	}
	
	private String calcExcelFormula(ExcelCell cell, String formulaKey, Iterator<IExcelDataRow> itExcelDataRows, int rowIndex) {
		int colIndex = -1;
		ArrayList<Integer> rowIndexs = new ArrayList<>();
		ExcelWorkbook tempWorkbook = virtualWorkbook;
		Iterator<ExcelSheet> itExeclSheet = tempWorkbook.iterator();
		while (itExeclSheet.hasNext()) {
			ExcelSheet excelSheet = itExeclSheet.next();
			
			for(int i=0;i<excelSheet.getRowCount();i++){
				ExcelRow excelRow = excelSheet.getRow(i);
				int rowType = excelRow.getType();
				switch (rowType) {
				case ExcelTemplateRowType.Detail:
						for(int j=0;j<excelRow.getCellCount();j++){
							ExcelCell excelCell = excelRow.getCell(j);
							String key = excelCell.getDefinition();
							if (formulaKey.endsWith(key)){
								colIndex = excelCell.getCellIndex();
								break;
							}
						}

				}
			}
		}
		while (itExcelDataRows.hasNext()){
			ExcelDetailDataRow detailDataRow = (ExcelDetailDataRow)itExcelDataRows.next();
			int realRowIndex = detailDataRow.getRowIndex() + rowIndex;
			rowIndexs.add(realRowIndex);
		}
		colIndex = cell.getCellIndex();
		String excelNum = ExcelUtil.transToExcelNum(rowIndexs,colIndex);
		String excelFormula = "SUM(" + excelNum + ")";
		return excelFormula;
	}

	
	private void fillGroupData(Sheet sheet, int rowIndex, ExcelGroupDetailRow groupDetailDataRow, 
			DataTable dataTable, ExcelRow excelRow, MetaTable metaTable, HashMap<Integer, Integer> columnWidths, 
			HashMap<Integer, GroupContext> groupContextMap, int groupContextCount) throws Throwable {
		Iterator<ExcelCell> itExcelCell = null;
		ExcelCell excelCell = null;
		
		GroupContext groupContext = new GroupContext();
		groupContext.addDefinition(groupDetailDataRow.getGroupLevel(), groupDetailDataRow.getDefinition());
		
		groupContextMap.put(groupContextCount, groupContext);
		GroupContext lastGroupContext = groupContextMap.get(groupContextCount - 1);
		
		// 用于计算分组行汇总值
		excelContext.setDetailRows(groupDetailDataRow.getDetailRows());
		
		itExcelCell = groupDetailDataRow.getExcelRow().iterator();
		while(itExcelCell.hasNext()) {
			excelCell = itExcelCell.next();
			if(excelCell.getCellIndex() == -1) {
				continue;
			}
			fillCellData(dataTable, metaTable, sheet, rowIndex, excelCell, columnWidths, groupContext, lastGroupContext);
		}
	}
	
	private void fillDetailData(Sheet sheet, int rowIndex, ExcelDetailDataRow detailDataRow, 
			DataTable dataTable, ExcelRow excelRow, MetaTable metaTable, HashMap<Integer, Integer> columnWidths,
			HashMap<Integer, GroupContext> groupContextMap, int groupContextCount) throws Throwable {
		Iterator<ExcelCell> itExcelCell = null;
		ExcelCell excelCell = null;
		excelContext.setExcelDetailDataRow(detailDataRow);
		
		GroupContext groupContext = new GroupContext();
		groupContext.setDefinitionMap(detailDataRow.getDefinitionMap());
		
		groupContextMap.put(groupContextCount, groupContext);
		GroupContext lastGroupContext = groupContextMap.get(groupContextCount - 1);
		
		dataTable.setPos(detailDataRow.getRowIndex());
		itExcelCell = excelRow.iterator();
		while(itExcelCell.hasNext()) {
			excelCell = itExcelCell.next();
			fillCellData(dataTable, metaTable, sheet, rowIndex, excelCell, columnWidths, groupContext, lastGroupContext);
		}
	}
	
	private void fillExpandData(Row row, ExcelExpandDataRow expandDataRow, 
			DataTable dataTable, ExcelRow excelRow, MetaTable metaTable, HashMap<Integer, Integer> columnWidths) throws Throwable {
		Iterator<ExcelCell> itExcelCell = null;
		ExcelCell excelCell = null;
		Cell cell = null;
		
		String definition = null;
		IExcelDataRow refRow = null;
		IExcelTransformer transformer = null;
		
		MetaColumn metaColumn = null;
		
		itExcelCell = excelRow.iterator();
		while(itExcelCell.hasNext()) {
			excelCell = itExcelCell.next();
			definition = excelCell.getDefinition();
			String sourceType = excelCell.getSourceType();
			boolean wrapText = false;
			
			int colIndex = excelCell.getCellIndex() - 1;
			cell = ExcelUtil.getCell(row, colIndex);
			
			Object value = null;
			if (sourceType.equalsIgnoreCase("field")) {
				// 确定数据行
				if ( excelCell.getColumnExpandTarget() )  {
					refRow = findRowInExpandRow(dataTable, expandDataRow, excelCell);
				} else {
					refRow = expandDataRow.getRow(0);
				}
				
				if (refRow != null) {
					// 找到对应列
					metaColumn = metaTable.get(definition);
					value = dataTable.getObject(refRow.getRowIndex(), metaColumn.getKey());
					
					transformer = transformerFactory.getTransformer(context.getVE(), excelCell);
					if (transformer != null) {
						value = transformer.transform(excelCell, value);
					}
				}
			} else if (sourceType.equalsIgnoreCase("const")) {
				value = definition;
			} else if (sourceType.equalsIgnoreCase("formula")) {
				excelContext.setExcelExpandDataRow(expandDataRow);
				if (excelCell.getColumnExpandItems() != null && !excelCell.getColumnExpandItems().isEmpty()){
					excelContext.setExpandCount(excelCell.getColumnExpandItems().get(0).getExpandIndex());
				}
				value = excelContext.getMidParser().eval(ScriptType.Formula, definition, excelContext, null);
			}

			if (value != null) {
				//对一些0值的处理
				if (value instanceof BigDecimal) {
					BigDecimal tempValue = (BigDecimal) value;
					if (tempValue.compareTo(BigDecimal.ZERO) != 0) {
				ExcelUtil.setCellValue(cell, value);
			}
				} else {
					ExcelUtil.setCellValue(cell, value);
				}
			}
			
			// 设置单元格样式
			ExcelDisplay display = excelCell.getDisplay();
			if (display != null){
				wrapText = display.isWrapText();
			}
			if (wrapText){
				styleFactory.getCellStyle(excelCell).setWrapText(wrapText);
			}
			cell.setCellStyle(styleFactory.getCellStyle(excelCell));
			ExcelUtil.calcMaxColumnWidth(cell, columnWidths, colIndex);
		}
	}
	
	private IExcelDataRow findRowInExpandRow(DataTable dataTable, ExcelExpandDataRow expandDataRow, ExcelCell excelCell) {
		IExcelDataRow row = null;
		ExcelDetailDataRow tmpRow = null;
		ArrayList<ColumnExpandItem> columnExpandItems = excelCell.getColumnExpandItems();
		int size = expandDataRow.size();
		for ( int i = 0; i<size; ++i ) {
			tmpRow = (ExcelDetailDataRow) expandDataRow.getRow(i);
			ColumnExpandItem columnExpandItem = null;
			boolean found = true;
			Iterator<ColumnExpandItem> it = columnExpandItems.iterator();
			while ( it.hasNext() ) {
				columnExpandItem = it.next();
				Object value = dataTable.getObject(tmpRow.getRowIndex(), columnExpandItem.getKey());
				if ( !ExcelDataUtil.equals(columnExpandItem.getValue(), value) ) {
					found = false;
					break;
				}
			}
			if ( found ) {
				row = tmpRow;
				break;
			}
		}
		return row;
	}
	
	private void exportFixRow(Document document, MetaDataObject metaDataObject, Sheet sheet, ExcelRow excelRow, int offset, HashMap<Integer, Integer> columnWidths) throws Throwable {
		int rowIndex = excelRow.getRowIndex()+ offset - 1;
		Row row = ExcelUtil.getRow(sheet, rowIndex);
		row.setHeightInPoints((short) excelRow.getHeight());
		
		ExcelCell excelCell = null;
		excelContext.setExcelRow(excelRow);
		Iterator<ExcelCell> itExeclCell = excelRow.iterator();
		while (itExeclCell.hasNext()) {
			excelCell = itExeclCell.next();
			String tableKey = excelCell.getTableKey();
			
			MetaTable metaTable = metaDataObject.getMetaTable(tableKey);
			DataTable dataTable = document.get(tableKey);
			fillCellData(dataTable, metaTable, sheet, rowIndex, excelCell, columnWidths, null, null);
		}
	}
	
	private void fillCellData(DataTable dataTable, MetaTable metaTable, Sheet sheet, int rowIndex,
			ExcelCell excelCell, HashMap<Integer, Integer> columnWidths, GroupContext groupContext, GroupContext lastGroupContext) throws Throwable {
		int colIndex = excelCell.getCellIndex() - 1; 
		int columnSpan = excelCell.getMergedColumnSpan() - 1;
		int rowSpan = excelCell.getMergedRowSpan() - 1;
		boolean wrapText = false;
		
		String definition = excelCell.getDefinition();
		Row row = ExcelUtil.getRow(sheet, rowIndex);
		String sourceType = excelCell.getSourceType();
		
		Object value = null;
		if (sourceType.equalsIgnoreCase("field")) {
			if (excelCell.getColumnExpandTarget()) {
				value = excelCell.getExpandCaption() == null ? excelCell.getExpandValue() : excelCell.getExpandCaption();
			} else {
				value = dataTable.getObject(definition);
			}
		} else if (sourceType.equalsIgnoreCase("const")) {
			value = definition;
		} else if (sourceType.equalsIgnoreCase("formula")) {
			value = excelContext.getMidParser().eval(ScriptType.Formula, definition, excelContext, null);
			
			String[] formula = definition.split("\\(");
			String SumExpandDirect = formula[0];
			if("SumExpandPortrait".equalsIgnoreCase(SumExpandDirect)) {
				int expandCount = excelContext.getExpandCount();
				
				String para1 = formula[1].substring(formula[1].indexOf("{") + 1, formula[1].lastIndexOf("}"));
				String[] expandFieldIndex = para1.split(",");
				if(expandCount < expandFieldIndex.length - 1) {
					expandCount = expandCount + 1;
					excelContext.setExpandCount(expandCount);
				}else {
					excelContext.setExpandCount(0);
				}	
			}
		} else if (sourceType.equalsIgnoreCase("ExcelFormula")) {
			value = definition;
		} 

		// 设置单元格样式
		ExcelDisplay display = excelCell.getDisplay();
		if (display != null){
			wrapText = display.isWrapText();
		}
		CellStyle cellStyle = styleFactory.getCellStyle(excelCell);
		if (wrapText){
			cellStyle.setWrapText(wrapText);
		}
		
		//判断是否为跨行跨列的单元格
		Cell cell = ExcelUtil.getCell(row, colIndex);
		this.fillCellStyle(sheet, row, cell, rowIndex, colIndex, rowSpan, columnSpan, cellStyle);
		
		boolean isSpanCell = columnSpan > 0 || rowSpan > 0;
		if(isSpanCell){
			sheet.addMergedRegionUnsafe(new CellRangeAddress(rowIndex, rowIndex + rowSpan, colIndex, colIndex + columnSpan));
		}
		
		short formatType = StampExcelUtil.getExcelCellStyle(excelCell, workBook);
		if (formatType != -1) {
			cellStyle.setDataFormat(formatType);
		} 
		if (!StampExcelUtil.isNumericCell(excelCell)) {
			IExcelTransformer transformer = transformerFactory.getTransformer(context.getVE(), excelCell);
			if (transformer != null) {
				value = transformer.transform(excelCell, value);
			}
		}
		
	
		
		if (value != null){
			if (sourceType.equalsIgnoreCase("ExcelFormula")){
				cell.setCellFormula((String)value);
				XSSFFormulaEvaluator formulaEvaluator = (XSSFFormulaEvaluator)workBook.getCreationHelper().createFormulaEvaluator();
				formulaEvaluator.evaluateFormulaCell(cell);
			}else {
				//对一些0值的处理
				if (value instanceof BigDecimal) {
					BigDecimal tempValue = (BigDecimal) value;
					if (tempValue.compareTo(BigDecimal.ZERO) != 0) {
						ExcelUtil.setCellValue(cell, value);
					}
				} else {
					ExcelUtil.setCellValue(cell, value);
				}
			}
		}
		
		
		
		if(groupContext != null) {
			groupContext.putDisplayValue(excelCell.getCellIndex(), TypeConvertor.toString(value));
		}

		// 动态合并单元格，非明细行和分组行不能合并单元格，const类型不能合并单元格
		this.mergeGroupCell(sheet, excelCell, rowIndex, colIndex, groupContext, lastGroupContext, sourceType, definition, value);
		boolean bColumnMerged = (columnSpan > 0);
		if (!bColumnMerged && !sourceType.equalsIgnoreCase("ExcelFormula")) {
			// 未有橫向单元格合并时才计算最佳列宽 
			ExcelUtil.calcMaxColumnWidth(cell, columnWidths, colIndex);
		}
	}
	
	private void fillCellStyle( Sheet sheet, Row row , Cell cell, int rowIndex, int colIndex, int rowSpan, int columnSpan, CellStyle cellStyle){
		boolean isSpanCell = columnSpan > 0 || rowSpan > 0;
		if (!isSpanCell) {
			cell.setCellStyle(cellStyle);
			return;
		}
		Cell  tempCell = null;
		for (int c = 0; c <= columnSpan; c++) {
			for (int r = 0; r <= rowSpan; r++) {
				tempCell = ExcelUtil.getCell(sheet, rowIndex + r, colIndex + c);
				tempCell.setCellStyle(cellStyle);
			}
		}
	}
	
	
	//记录分组字段的起始合并行索引（Key:分组字段所在列索引， value:分组字段合并起始行索引）
	private Map<Integer, Integer> startMergeRowMap =  new HashMap<Integer, Integer>();
	
	//动态合并单元格，非明细行和分组行不能合并单元格，const类型不能合并单元格
	//1、第一次遇到当前单元格的值和上一行同一列的单元格值相同，在startMergeRowMap记录前一行的索引为合并的起始索引
	//2、当分组字段单元格出现const值，并且有被记录过起始的合并行索引：那么合并目标索引终止到当前行的前一行，移除startMergeRowMap中的记录
	//3、如果当前分组单元格值和上一行同位置单元格值不同，并且有被记录过起始的合并行索引：那么合并目标索引终止到当前行的前一行，移除startMergeRowMap中的记录
	private void mergeGroupCell(Sheet sheet,ExcelCell excelCell, int rowIndex, int colIndex, 
			GroupContext groupContext, GroupContext lastGroupContext, 
			String sourceType, String definition, Object value ){
		//groupContext为空：既不是分组行也不是明细行
		if(groupContext == null)
			return;
		//lastGroupContext为空：第一个分组行或者第一个明细行
		if(lastGroupContext == null)
			return;
		//当来源类型为常量，合并起始行里面有内容记录，那么合并起始行---当前行的前一个行，清楚合并起始行记录
		if(sourceType.equalsIgnoreCase("const")){
			if(startMergeRowMap.containsKey(colIndex)){
				sheet.addMergedRegion(new CellRangeAddress(startMergeRowMap.get(colIndex), rowIndex-1, colIndex, colIndex));
				startMergeRowMap.remove(colIndex);
			}
			return;
		}
			
		//当前行为分组行，但是单元的的definition是FillGroup(...)
		if(groupContext.getGroupLevel(definition) == null) {
			//非分组字段不能合并单元格,除非definition为FillGroup(...)函数
			String[] formula = definition.split("\\(");
			String FillGroup = formula[0];
			if(!"FillGroup".equalsIgnoreCase(FillGroup)) 
				return;
		}
		
		int cellIndex = excelCell.getCellIndex();
		String lastDisplayValue = lastGroupContext.getDisplayValue(cellIndex);
		if(lastDisplayValue == null)
			return;

		//如果当前分组单元格值和上一行同位置单元格值不同，并且有被记录过起始的合并行索引：那么合并目标索引终止到当前行的前一行，移除startMergeRowMap中的记
		String  curCellValue = TypeConvertor.toString(value);
		if(!curCellValue.equalsIgnoreCase(lastDisplayValue)){
			if(startMergeRowMap.containsKey(colIndex)){
				sheet.addMergedRegion(new CellRangeAddress(startMergeRowMap.get(colIndex), rowIndex-1, colIndex, colIndex));
				startMergeRowMap.remove(colIndex);
			}
			return;
	   }
		//保存当前列单元格合并起始行索引
	   if(!startMergeRowMap.containsKey(colIndex)){
		    startMergeRowMap.put(colIndex, rowIndex -1);
	   }
		
	}
}
