package com.bokesoft.yigo.mid.util;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import com.bokesoft.yes.struct.md5.RefFile;
import com.bokesoft.yes.struct.md5.RefProject;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.solution.MetaProjectCollection;
import com.bokesoft.yigo.meta.solution.MetaProjectProfile;
import com.bokesoft.yigo.mid.util.MD5Util;

public class MD5Generator {
		
	/**
	 * 服务器启动时,生成应用程序MD5文件
	 * 使用Apache开源项目commons-codec.jar
	 * 
	 * @param web_app_path 应用程序位置,即MD5文件的存放位置
	 * @param bin_path 应用程序文件夹
	 * @throws Throwable
	 */
	public static void generateServerAppMD5(String web_app_path) throws Throwable {
		String md5_path = web_app_path + "App.md5";
		File md5File = new File(md5_path);
		File file = new File(web_app_path, "bin");		
		
		FileOutputStream fos = new FileOutputStream(md5File);
		OutputStreamWriter osw = new OutputStreamWriter(fos,"UTF-8");
		BufferedWriter writer = new BufferedWriter(osw);

		// 生成MD5加密信息
		if (file.exists() && file.isDirectory()) {
			File[] files = file.listFiles(new FilenameFilter() {
				
				@Override
				public boolean accept(File dir, String name) {
					return name.endsWith(".jar");
				}
			});
			// 排个序
			Arrays.sort(files, new Comparator<File>() {

				@Override
				public int compare(File f1, File f2) {
					return f1.getName().compareTo(f2.getName());
				}
			});
			for (int i = 0; i < files.length; i++) {
				File appFile = files[i];
				
				// 可能存在主题文件夹 || 隐藏的文件
				if( appFile.isDirectory() || appFile.isHidden() )
					continue;
				
				String md5Str = MD5Util.getMD5(appFile);

				StringBuilder md5_msg = new StringBuilder();
				md5_msg.append(appFile.getName()).append("*").append(md5Str).append("\n");
				writer.write(md5_msg.toString());
				writer.flush();
			}
		}
		
		// 主题文件
		File themeFile = new File(web_app_path, "bin/theme");
		String bin_path = web_app_path + "bin";
			
		// 存在
		if( themeFile.exists() && themeFile.isDirectory() ) {
			generateTheme(themeFile, bin_path, writer);
		}
		
		if( writer != null ){
			writer.close();
		}
		if( osw != null ) {
			osw.close();
		}
		if( fos != null ) {
			fos.close();
		}
	}
	
//	/**
//	 * 生成文件的MD5,经过试验,两种方法生成的结果一致,换一种生成方式,避免创建多个buffer对象
// 	 * @param file 文件
//	 * @return MD5字符串
//	 * @throws Throwable
//	 */
//	private static String generateMD5(File file) throws Throwable {
////		byte[] buffer = new byte[4096];
//		FileInputStream in = new FileInputStream(file);
//	//	ByteArrayOutputStream baos = new ByteArrayOutputStream(4096);
//		
////		int len = 0;
////		while((len = is.read(buffer)) != -1) {
////			baos.write(buffer, 0, len);
////		}
//
//		String md5Str = DigestUtils.md5Hex(in);
////		baos.close();
//		in.close();
//		return md5Str;
//	}
	
	/**
	 * 生成主题文件的MD5,跳过隐藏文件
	 * @param themeFile
	 * @param bin_path
	 * @param writer
	 * @throws Throwable
	 */
	private static void generateTheme(File themeFile,String bin_path,BufferedWriter writer) throws Throwable {
		if( themeFile.isHidden() ) 
			return;
		if( themeFile.isDirectory() ) {
			File[] files = themeFile.listFiles();
			// 排个序
			Arrays.sort(files, new Comparator<File>() {

				@Override
				public int compare(File f1, File f2) {
					return f1.getName().compareTo(f2.getName());
				}
			});
			for( int i = 0;i < files.length;i++ ) {
				generateTheme(files[i], bin_path, writer);
			}
		} else {
			// 将'\'替换成'/'
			String relativePath = themeFile.getPath().substring(bin_path.length() + 1);
			relativePath = relativePath.replaceAll("\\\\", "/");
			
			String md5Str = MD5Util.getMD5(themeFile);
			
			StringBuilder md5_msg = new StringBuilder();
			md5_msg.append(relativePath).append("*").append(md5Str).append("\n");
			writer.write(md5_msg.toString());
			writer.flush();
		}
	}
	
	/**
	 * 生成服务端配置的MD5文件
	 * @param solutionPath 配置地址
	 * @param metaFactory 配置工厂
	 * @throws Throwable
	 */
	public static void generateServerConfigMD5(String solutionPath, IMetaFactory metaFactory) throws Throwable {

		// 截取地址末尾多余的'\'或者'/'
		
		// Windows
		if ( solutionPath.endsWith("\\") ) {
			solutionPath = solutionPath.substring(0, solutionPath.lastIndexOf("\\"));
		
		// Linux
		} else if ( solutionPath.endsWith("/") ) {
			solutionPath = solutionPath.substring(0, solutionPath.lastIndexOf("/"));
		}
		
		// 配置文件
		File file = new File(solutionPath);
		
		String md5Name = "Config.md5";

		// 服务端配置文件的md5位置
		String config_md5_path = solutionPath + "/" + md5Name;
		File config_md5 = new File(config_md5_path);

		FileOutputStream fos = new FileOutputStream(config_md5);
		OutputStreamWriter osw = new OutputStreamWriter(fos, "UTF-8");
		BufferedWriter writer = new BufferedWriter(osw);
	
		generateFileMD5(md5Name, file, solutionPath, writer);

		// 取出应用中外置的工程
		MetaProjectCollection metaProjectCollection = metaFactory.getSolution().getProjectCollection();		
		if (metaProjectCollection != null) {
			Iterator<MetaProjectProfile> itMetaProject = metaProjectCollection.iterator();
			MetaProjectProfile metaProjectProfile = null;
			String projectParentPath = null;
			String refPath = null;
			while (itMetaProject.hasNext()) {
				metaProjectProfile = itMetaProject.next();
				refPath = metaProjectProfile.getRefPath();
				if (refPath != null && !refPath.isEmpty()) {
					if (refPath.startsWith("/") || refPath.indexOf(":") > 0) {
						projectParentPath = refPath.substring(0 , refPath.indexOf(metaProjectProfile.getKey()) - 1);
						file = new File(refPath);
						generateFileMD5(md5Name, file, projectParentPath, writer);
					} else {
						file = new File(solutionPath, refPath);
						String path = file.getCanonicalPath();
						projectParentPath = path.substring(0, path.indexOf(metaProjectProfile.getKey())-1);
						file = new File(path);
						generateFileMD5(md5Name, file, projectParentPath, writer);
					}
				}
			}
		}

		if (writer != null) {
			writer.close();
		}
		if( osw != null ) {
			osw.close();
		}
		if( fos != null ) {
			fos.close();
		}
	}
	
	/**
	 * 生成配置的MD5,跳过隐藏的目录及Data文件夹
	 * @param md5Name 生成的MD5名称
	 * @param file 配置文件或者目录
	 * @param solutionPath 解决方案路径
	 * @param writer 写出流
	 * @throws Throwable
	 */
	private static void generateFileMD5(String md5Name,File file, String solutionPath, BufferedWriter writer) throws Throwable {
		if( file.isHidden() || file.getName().equalsIgnoreCase(md5Name)) 
			return;
		if( file.isDirectory() && file.getName().equals("Data") )
			return;
		if ( file.isDirectory() ) {
			File[] files = file.listFiles();
			// 排个序
			Arrays.sort(files, new Comparator<File>() {

				@Override
				public int compare(File f1, File f2) {
					return f1.getName().compareTo(f2.getName());
				}
			});
			for (int i = 0; i < files.length; i++) {
				generateFileMD5(md5Name,files[i], solutionPath, writer);
			}
		} else {
			// 将'\'替换成'/'
			String relativePath = file.getPath().substring(solutionPath.length() + 1);// 相对路径
			relativePath = relativePath.replaceAll("\\\\", "/");
			
			String md5Str = MD5Util.getMD5(file);
			
			StringBuilder md5_msg = new StringBuilder();
			md5_msg.append(relativePath).append("*").append(md5Str).append("\n");
			writer.write(md5_msg.toString());
			writer.flush();
		}
	}
	
	/**
	 * 根据APP所引用的工程定义生成MD5信息
	 * @param solutionPath 解决方法路径
	 * @param appKey 应用名称
	 * @param projects 应用所引用的工程定义
	 * @throws Throwable 
	 */
	public static void generateSpecialProjectsMD5(String solutionPath,String appKey,List<RefProject> projects) throws Throwable{
		// 截取地址末尾多余的'\'或者'/'
		// Windows
		if ( solutionPath.endsWith("\\") ) {
			solutionPath = solutionPath.substring(0, solutionPath.lastIndexOf("\\"));
		
		// Linux
		} else if ( solutionPath.endsWith("/") ) {
			solutionPath = solutionPath.substring(0, solutionPath.lastIndexOf("/"));
		}
		
		// MD5名称
		String md5Name = appKey + ".md5";
		
		// 服务端配置文件的md5位置
		String config_md5_path = solutionPath + "/" + md5Name;
		File config_md5 = new File(config_md5_path);

		FileOutputStream fos = new FileOutputStream(config_md5);
		OutputStreamWriter osw = new OutputStreamWriter(fos, "UTF-8");
		BufferedWriter writer = new BufferedWriter(osw);
		
		// 解决方案目录
		File solutionDir = new File(solutionPath);
		File[] files = solutionDir.listFiles();
		
		// 排个序
		Arrays.sort(files, new Comparator<File>() {

			@Override
			public int compare(File f1, File f2) {
				return f1.getName().compareTo(f2.getName());
			}
		});
		
		// 生成解决方案下的纯文件MD5
		for( File file : files ) {
			
			// 略过隐藏文件和目录 保留Resource文件夹
			if( file.isHidden() || (file.isDirectory() && !file.getName().equals("Resource")) )
				continue;
			
			// 略过MD5文件
			if( !file.isDirectory() ) {
				String prefix = file.getName().substring(file.getName().lastIndexOf(".") + 1);
				if( prefix.equalsIgnoreCase("md5") )
					continue;				
			}
			
			generateFileMD5(md5Name, file, solutionPath, writer);
		}
		
		// 记录下路径,防止重复生成
		List<String> paths = new ArrayList<String>();
				
		// 逐个生成MD5信息
		for( RefProject project : projects ) {
			
			// 工程目录
			String projectPath = solutionPath + "/" + project.getKey();
			File dir = new File(projectPath);
			
			// 生成整个project文件夹文件的md5
			if( project.isEmpty() ) {
				if( !dir.exists() ) 
					throw new RuntimeException("Dir " + projectPath + " not exists!");
				
				generateFileMD5(md5Name, dir, solutionPath, writer);				
			} 
			// 生成project文件夹中指定文件的md5
			else {
				
				// 生成特定文件
				for ( RefFile file : project.getFiles() ) {
					String filePath = solutionPath + "/" + project.getKey() + "/" + file.getPath();
					File fileDir = new File(filePath);
					if( !fileDir.exists() ) 
						throw new RuntimeException("Dir " + filePath + " not exists!");
					
					generateFileMD5(md5Name, fileDir, solutionPath, writer);
					
					// 记录下已经生成的
					paths.add(file.getPath());
				}
				
				// 生成其他project目录下纯文件的md5
				files = dir.listFiles();
				// 排个序
				Arrays.sort(files, new Comparator<File>() {

					@Override
					public int compare(File f1, File f2) {
						return f1.getName().compareTo(f2.getName());
					}
				});
				for( File file : files ) {
					
					// 略过隐藏文件和目录
					if( file.isHidden() || file.isDirectory() )
						continue;
					
					// 略过已经生成的
					if( paths.contains(file.getName()) )
						continue;
					
					generateFileMD5(md5Name, file, solutionPath, writer);
				}
			}
			
			paths.clear();
		}
		if (writer != null) {
			writer.close();
		}
		if( osw != null ) {
			osw.close();
		}
		if( fos != null ) {
			fos.close();
		}
	}
	
}
