package com.bokesoft.yigo.mid.scheduler;

import java.io.InputStream;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

import com.bokesoft.yes.mid.base.CoreSetting;
import com.bokesoft.yes.mid.base.ServerSetting;
import org.quartz.*;
import org.quartz.impl.StdSchedulerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 定时任务的管理类
 *
 * @author 陈瑞
 *
 */
public class QuartzManager {
	private static final Logger logger = LoggerFactory.getLogger(QuartzManager.class);
	private static StdSchedulerFactory schedulerFactory = new StdSchedulerFactory();


	private static QuartzManager instance = null;

	public static QuartzManager getInstance() {
		if( instance == null ){
			instance = new QuartzManager();
		}
		return instance;
	}

	QuartzManager(){}

	/**
	 * 判断当前yigo应用，是否支持quartz-schedule的管理
	 * @return
	 */
	public boolean isSupportManagement() {
		// 在server.propertis中未开启定时任务
		if(!ServerSetting.getInstance().isEnableScheduling()) {
			return false;
		}
		// 如果在core.propertis中没有配置dsn属性
		CoreSetting coreSetting = CoreSetting.getInstance();
		String dsn = coreSetting.getDSN();
		if (dsn == null || dsn.isEmpty()) {
			return false;
		}
		return  true;
	}


	/**
	 * 使用配置文件路径初始化
	 * @param filePath 配置文件地址
	 * @throws Throwable
	 */
	public void init(final String filePath) throws Throwable {
		schedulerFactory.initialize(filePath);
	}

	/**
	 * 使用配置文件流初始化
	 * @param fis 配置文件流
	 * @throws Throwable
	 */
	public void init(final InputStream fis) throws Throwable {
		if (fis != null) {
			schedulerFactory.initialize(fis);
		}
	}

	/**
	 * 添加一个定时任务调度
	 *
	 * @param jobName 任务的名称
	 * @param jobGroup 任务的组名
	 * @param triggerName 触发器的名称
	 * @param triggerGroup 触发器的组名
	 * @param job 任务类
	 * @param jobDescription 任务的描述
	 * @param delay 多久后开始
	 * @param repeatCount 重复次数,为-1表示永久循环
	 * @param repeatInterval 间隔
	 * @throws Throwable
	 */
	public void addJob(String jobName,String jobGroup,String triggerName,String triggerGroup,
			Job job, String jobDescription,long delay,int repeatCount, long repeatInterval) throws Throwable{
		if(!isSupportManagement()){
			throw new UnsupportedOperationException("Scheduling is disabled, not allow add job");
		}
		Scheduler scheduler = schedulerFactory.getScheduler();
		JobDetail jobDetail = JobBuilder.newJob(job.getClass())
				.withIdentity(jobName, jobGroup)
				.withDescription(jobDescription)
				.build();
		// 开始时间
		Date startTime = new Date(delay + System.currentTimeMillis());

		SimpleTrigger trigger = null;
		if( repeatCount == -1 ){
			trigger = (SimpleTrigger) TriggerBuilder.newTrigger()//
					.withIdentity(triggerName, triggerGroup)//
					.withSchedule(SimpleScheduleBuilder.simpleSchedule()//
							.withIntervalInMilliseconds(repeatInterval)//
							.repeatForever())//
					.startAt(startTime)
					.build();

		} else {
			trigger = (SimpleTrigger) TriggerBuilder.newTrigger()//
					.withIdentity(triggerName, triggerGroup)//
					.withSchedule(SimpleScheduleBuilder.simpleSchedule()//
							.withIntervalInMilliseconds(repeatInterval)//
							.withRepeatCount(repeatCount))//
					.startAt(startTime)//
					.build();//
		}

		HashSet<Trigger> set = new HashSet<Trigger>();
		set.add(trigger);
		logger.info("add job.name:{},group:{}," +
						"trigger.name:{},trigger.group:{},trigger.repeatInterval:{},trigger.repeatCount:{}"
				, jobName, jobGroup, triggerName, triggerGroup, repeatInterval, repeatCount);
		scheduler.scheduleJob(jobDetail, set, true);

//		scheduler.scheduleJob(jobDetail, trigger);
	}

	/**
	 * 添加一个定时任务调度
	 *
	 * @param jobName 任务的名称
	 * @param jobGroup 任务的组名
	 * @param triggerName 触发器的名称
	 * @param triggerGroup 触发器的组名
	 * @param jobClass 任务类
	 * @param jobDescription 任务的描述
	 * @param cronExp cron表达式
	 * @throws Throwable
	 */
	public void addJob(String jobName,String jobGroup,String triggerName, String triggerGroup,
						Class<? extends Job> jobClass,String jobDescription, String cronExp) throws SchedulerException {
		if(!isSupportManagement()){
			throw new UnsupportedOperationException("Scheduling is disabled, not allow add job");
		}
		Scheduler scheduler = schedulerFactory.getScheduler();
		JobDetail jobDetail = JobBuilder.newJob(jobClass)
				.withIdentity(jobName, jobGroup)
				.withDescription(jobDescription)
				.build();
		CronTrigger trigger = TriggerBuilder.newTrigger()
				.withIdentity(triggerName,triggerGroup)
				.withSchedule(CronScheduleBuilder.cronSchedule(cronExp))
				.build();
		HashSet<Trigger> set = new HashSet<Trigger>();
		set.add(trigger);
		scheduler.scheduleJob(jobDetail, set, true);
		logger.info("add job.name:{},group:{},trigger.name:{},trigger.group:{},trigger.cronexp:{}"
				, jobName, jobGroup, triggerName, triggerGroup, cronExp);
	}

	/**
	 * 删除任务
	 * @param name 任务名称
	 * @param group 任务的组名
	 * @return
	 * @throws Throwable
	 */
	public boolean deleteJob(String name,String group) throws Throwable {
		if(!isSupportManagement()){
			throw new UnsupportedOperationException("Scheduling is disabled, not allow delete job");
		}
		Scheduler scheduler = schedulerFactory.getScheduler();
		logger.info("delete job.name:{},job.group:{}",name,group);
		return scheduler.deleteJob(new JobKey(name, group));
	}

	/**
	 * 校验任务是否存在
	 * @param name 任务名称
	 * @param group 任务的组名
	 * @throws SchedulerException
	 */
	public static boolean checkJobExist(String name,String group) throws SchedulerException {
		Scheduler scheduler = schedulerFactory.getScheduler();
		return scheduler.checkExists(new JobKey(name,group));
	}

	/**
	 * 调度任务是否启动
	 * @throws Throwable
	 */
	public boolean isStarted() throws SchedulerException {
		Scheduler scheduler = schedulerFactory.getScheduler();
		return scheduler.isStarted();
	}

	/**
	 * 启动调度任务
	 * @throws Throwable
	 */
	public void start() throws SchedulerException {
		if(!isSupportManagement()){
			throw new UnsupportedOperationException("Scheduling is disabled, not allow start job");
		}
		Scheduler scheduler = schedulerFactory.getScheduler();
		prepareTraceJobListener(scheduler);
		logger.info("job start now.");
		scheduler.start();
	}

	/**
	 * 调度任务是否关闭了
	 * @throws SchedulerException
	 */
	public boolean isShutdown() throws SchedulerException{
		Scheduler scheduler = schedulerFactory.getScheduler();
		return scheduler.isShutdown();
	}

	/**
	 * 关闭调度任务
	 * @throws SchedulerException
	 */
	public void shutdown() throws SchedulerException{
		Scheduler scheduler = schedulerFactory.getScheduler();
		if( scheduler != null && !scheduler.isShutdown()) {
			scheduler.shutdown();
			logger.info("shutdown job now");
		}
	}

	/**
	 * 获取当前Quartz是否处于 standby 状态
	 * @throws SchedulerException
	 */
	public boolean isStandby() throws SchedulerException{
		Scheduler scheduler = schedulerFactory.getScheduler();
		return scheduler.isInStandbyMode();
	}

	/**
	 * 暂停调度任务,且忽略暂停之后的任务触发事件
	 * @throws SchedulerException
	 */
	public void standby() throws SchedulerException {
		if(!isSupportManagement()){
			throw new UnsupportedOperationException("Scheduling is disabled, not allow standby");
		}
		Scheduler scheduler = schedulerFactory.getScheduler();
		if( scheduler != null && !scheduler.isShutdown()) {
			scheduler.standby();
			logger.info("standby job now");
		}
	}

	/**
	 * 获取当前正在实行的任务job情况
	 * @return
	 * @throws SchedulerException
	 */
	public List<JobExecutionContext> getCurrentlyExecutingJobs() throws SchedulerException {
		Scheduler scheduler = schedulerFactory.getScheduler();
		return scheduler.getCurrentlyExecutingJobs();
	}

	/**
	 * 删除所有任务
	 *
	 * @throws Throwable
	 */
	public void clear() throws Throwable {
		Scheduler scheduler = schedulerFactory.getScheduler();
		if (scheduler != null) {
			scheduler.clear();
			logger.info("clear job now");
		}
	}

	/**
	 * 注册数据记录框架的切面监听处理程序
	 * @param scheduler
	 * @throws SchedulerException
	 */
	private void prepareTraceJobListener(Scheduler scheduler) throws SchedulerException {
		TraceFrameJobLisenter jobLisenter = new TraceFrameJobLisenter();
		ListenerManager manager = scheduler.getListenerManager();
		// 如果有相同Listener不再重复注册
		if( null == manager.getJobListener(jobLisenter.getName())){
			manager.addJobListener(jobLisenter);
			logger.info("add TraceFrameJobLisenter to job");
		}
	}
}
