package com.bokesoft.yigo.mid.log;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.json.JSONObject;

import com.bokesoft.yes.common.log.LogSvr;
import com.bokesoft.yes.mid.connection.dbmanager.BatchPsPara;
import com.bokesoft.yes.mid.connection.dbmanager.PSArgs;
import com.bokesoft.yigo.common.def.SystemDBField;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.bokesoft.yigo.mid.connection.DBType;
import com.bokesoft.yigo.mid.connection.IDBManager;
import com.bokesoft.yigo.mid.variant.VariantData;
import com.bokesoft.yigo.mid.variant.VariantKey;
import com.bokesoft.yigo.mid.variant.log.ILogArchiveIO;
import com.bokesoft.yigo.mid.variant.log.LogKey;

public class ArchiveProxy {
	/**
	 * 批量归档到ES的记录数
	 */
	private static final int BATCH_SIZE = 1000;
	
	/**
	 * 归档成功的记录数
	 */
	private List<LogKey> toDeleteKeys = new ArrayList<>();
	
	/**
	 * IO类型，目前支持ES
	 */
	private String type ;
	
	/**
	 * 构造函数
	 * @param type IO类型，目前支持ES
	 */
	public ArchiveProxy(String type) {
		super();
		this.type = type;
	}

	/**
	 * 归档指定日期之前的数据
	 * @param context
	 * @param date
	 * @throws Throwable
	 */
	public void archive(DefaultContext context, Date date) throws Throwable{
		DefaultContext newContext = null;
		try {
			newContext = new DefaultContext(context.getVE());
			// 查询数据库，并归档到目标存储位置
			loadAndArchive(newContext, date);
			
			//删除已经归档成功的记录
			deleteLog(newContext);
			
			newContext.commit();
			
			LogSvr.getInstance().info("bpm log archive completed...");
			
		} catch (Throwable e) {
			if (newContext != null) {
				newContext.rollback();
			}
			LogSvr.getInstance().error("bpm log archive error", e);
		}finally {
			if (newContext != null) {
				newContext.close();
			}
		}
	}
	
	private void loadAndArchive(DefaultContext context, Date date) throws Throwable {
		IDBManager dbManager = context.getDBManager();
		StringBuilder builder = new StringBuilder("select * from " 
				+ dbManager.keyWordEscape(SystemDBField.BPM_LOG) 
				+ " where "
				+ dbManager.keyWordEscape(SystemDBField.FINISHTIME)
				+ " < ");
		int dbType = dbManager.getDBType();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		String dateString = sdf.format(date);
		if (dbType == DBType.Oracle) {
			builder.append(" to_date(?,'yyyy-mm-dd hh24:mi:ss') ");
		} else {
			builder.append(" ? ");
		}
		String sql = builder.toString();
		PreparedStatement ps = null;
		ResultSet rs = null;
		List<LogData> logs = new ArrayList<>();
		try {
			ps = dbManager.preparedQueryStatement(sql);
			PSArgs args = new PSArgs();
			args.addStringArg(dateString);
			rs = dbManager.executeQuery(ps, sql, args);
			int count = 0;
			while(rs.next()) {
				LogData data = new LogData();
				data.populate(rs);
				logs.add(data);
				count ++;
				if (count % BATCH_SIZE == 0) {
					archiveLog(context, logs);
					logs.clear();
				}
			}
			
			if (logs.size() > 0) {
				archiveLog(context, logs);
				logs.clear();
			}
			
		} finally {
			if (ps != null) {
				ps.close();
			}
			if (rs != null) {
				rs.close();
			}
		}
	}
	
	private void loadAndArchive(DefaultContext context, Date start, Date end) throws Throwable {
		IDBManager dbManager = context.getDBManager();
		StringBuilder builder = new StringBuilder("select "
				+ dbManager.keyWordEscape(SystemDBField.INSTANCEID) + ", "
				+ dbManager.keyWordEscape(SystemDBField.LOGID)
				+ " from " 
				+ dbManager.keyWordEscape(SystemDBField.BPM_LOG)
				+ " where "
				+ dbManager.keyWordEscape(SystemDBField.FINISHTIME)
				+ " > ");
		int dbType = dbManager.getDBType();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		String startString = sdf.format(start);
		String endString = sdf.format(end);
		if (dbType == DBType.Oracle) {
			builder.append(" to_date(?,'yyyy-mm-dd hh24:mi:ss') and "
					+ SystemDBField.FINISHTIME
					+ " < to_date(?,'yyyy-mm-dd hh24:mi:ss') ");
		} else {
			builder.append(" ? and "
					+ SystemDBField.FINISHTIME
					+ " < ? ");
		}
		String sql = builder.toString();
		PreparedStatement ps = null;
		ResultSet rs = null;
		List<LogData> logs = new ArrayList<>();
		try {
			ps = dbManager.preparedQueryStatement(sql);
			PSArgs args = new PSArgs();
			args.addStringArg(startString);
			args.addStringArg(endString);
			rs = dbManager.executeQuery(ps, sql, args);
			int count = 0;
			while(rs.next()) {
				LogData data = new LogData();
				data.populate(rs);
				logs.add(data);
				count ++;
				if (count % BATCH_SIZE == 0) {
					archiveLog(context, logs);
					logs.clear();
				}
			}
			
			if (logs.size() > 0) {
				archiveLog(context, logs);
				logs.clear();
			}
			
		} finally {
			if (ps != null) {
				ps.close();
			}
			if (rs != null) {
				rs.close();
			}
		}
	}
	
	/**
	 * 归档指定时间区间的数据
	 * @param context
	 * @param start
	 * @param end
	 * @throws Throwable
	 */
	public void archive(DefaultContext context, Date start, Date end)throws Throwable{
		DefaultContext newContext = null;
		try {
			newContext = new DefaultContext(context.getVE());
			// 查询数据库，并归档到目标存储位置
			loadAndArchive(newContext, start, end);
			
			//删除已经归档成功的记录
			deleteLog(newContext);
			
			newContext.commit();
			LogSvr.getInstance().info("bpm log archive completed...");
			
		} catch (Throwable e) {
			if (newContext != null) {
				newContext.rollback();
			}
			LogSvr.getInstance().error("bpm log archive error", e);
		}finally {
			if (newContext != null) {
				newContext.close();
			}
		}
	}	
	
	private void archiveLog(DefaultContext context, List<LogData> logs) throws Throwable{
		String index = "bpm-log-archive";
		ILogArchiveIO io = LogArchiveIOFactory.getArchiveIO(context, type);
		List<VariantData> datas = new ArrayList<>();
		VariantData data;
		for (LogData log : logs) {
			long instanceID = log.getInstanceID();
			Integer logID = log.getLogID();
			LogKey logKey = new LogKey(instanceID, logID);
			Map<String,Object> options = new HashMap<>();
			options.put("index", index);
			VariantKey key = new VariantKey(logKey.toString(), options);
			data = new VariantData(key);
			data.setData(log.toJSON().toString());
			datas.add(data);
		}
		if (io != null) {
			List<LogKey> successKeys = io.archiveLog(context, datas);
			toDeleteKeys.addAll(successKeys);
		}
	}
	
	private void deleteLog(DefaultContext context) throws Throwable {
		IDBManager dbManager = context.getDBManager();
		String sql = "delete from " 
					+ dbManager.keyWordEscape(SystemDBField.BPM_LOG) 
					+ " where "
					+ dbManager.keyWordEscape(SystemDBField.INSTANCEID) 
					+ "=? and "
					+ dbManager.keyWordEscape(SystemDBField.LOGID)
					+ "=?";
		BatchPsPara bpp = new BatchPsPara(sql);
		for (LogKey logKey : toDeleteKeys) {
			long instanceID = logKey.getInstanceID();
			int logId = logKey.getLogId();
			PSArgs args = new PSArgs();
			args.addLongArg(instanceID);
			args.addIntArg(logId);
			bpp.putArgs(args);
		}
		dbManager.executeUpdate(bpp);
	}
	
	public List<LogData> getLogs(DefaultContext context, long instanceID) throws Throwable{
		List<LogData> list = new ArrayList<>();
		String index = "bpm-log-archive";
		ILogArchiveIO io = LogArchiveIOFactory.getArchiveIO(context, type);
		if (io != null) {
			List<VariantData> logs = io.loadLogs(context, instanceID, index);
			LogData data = null;
			if (logs != null) {
				for (VariantData log : logs) {
					String docString = log.getData();
					JSONObject object = new JSONObject(docString);
					if (object.has("data")) {
						String logString = object.getString("data");
						JSONObject logJson = new JSONObject(logString);
						data = new LogData();
						data.fromJSON(logJson);
						list.add(data);
					}
					
				}
			}
		}
		return list;
	}
}
