package com.bokesoft.yigo.meta.util;

import com.bokesoft.yigo.meta.base.IMetaResolver;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.factory.IMetaResolverFactory;
import com.bokesoft.yigo.meta.form.MetaForm;

import java.io.File;
import java.io.InputStream;

public class MetaResolverUtil {

    /**
     * 校验filePath中是否存在../../的情况，导致file的最终路径，不包含parentPath，即文件不在根目录下。
     *
     * @param parentPath 文件所在根目录
     * @param filePath   文件路径
     * @return 文件路径中是否包含parentPath, 即文件是否在根目录下，true代表在根目录下，false代表不在
     */
    private static boolean checkRealPath(String parentPath, String filePath) {
        try {
            if (parentPath != null && !parentPath.isEmpty() && filePath != null && !filePath.isEmpty()) {
                File parentFile = new File(parentPath);
                File realFile = new File(filePath);
                if (parentFile.exists() && realFile.exists()) {
                    String parentRealPath = parentFile.getCanonicalPath().replaceAll("\\\\", "/");
                    String realRealPath = realFile.getCanonicalPath().replaceAll("\\\\", "/");
                    return realRealPath.contains(parentRealPath);
                }
            }
            return false;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 仅加载Resource目录下的文件
     *
     * @param metaFactory 配置工厂
     * @param formKey     表单标识
     * @param filePath    Resource目录下文件的相对路径
     * @return Resource目录下文件内容
     * @throws Throwable 报错
     */
    public static InputStream loadResourceInputStream(IMetaFactory metaFactory, String formKey, String filePath) throws Throwable {
        String projectKey = "";
        if (formKey != null && !formKey.isEmpty()) {
            MetaForm metaForm = metaFactory.getMetaForm(formKey);
            projectKey = metaForm.getProjectKey();
        }
        return loadResourceInputStreamByProject(metaFactory, projectKey, filePath);
    }

    /**
     * 仅加载Resource目录下的文件
     *
     * @param metaFactory 配置工厂
     * @param project     表单所在工程的标识
     * @param filePath    Resource目录下文件的相对路径
     * @return Resource目录下文件内容
     * @throws Throwable 报错
     */
    public static InputStream loadResourceInputStreamByProject(IMetaFactory metaFactory, String project, String filePath) throws Throwable {
        String resourcePrefix = "/Resource";
        try {
            filePath = filePath.replaceAll("\\\\", "/");
            if (project != null && !project.isEmpty()) {
                IMetaResolverFactory metaResolverFactory = metaFactory.getMetaResolverFactory(project);
                IMetaResolver metaResolver = metaResolverFactory.newMetaResolver(resourcePrefix);
                InputStream inputStream = loadResourceInputStreamByResolver(metaResolver, filePath);
                if (inputStream == null) {
                    metaResolver = metaFactory.getMetaResolverFactory().newMetaResolver(resourcePrefix);
                    inputStream = loadResourceInputStreamByResolver(metaResolver, filePath);
                }
                return inputStream;
            } else {
                IMetaResolver metaResolver = metaFactory.getMetaResolverFactory().newMetaResolver(resourcePrefix);
                return loadResourceInputStreamByResolver(metaResolver, filePath);
            }
        } catch (UnsupportedOperationException e) {
            return metaFactory.loadResource(filePath, project);
        }
    }

    private static InputStream loadResourceInputStreamByResolver(IMetaResolver metaResolver, String filePath) throws Throwable {
        String newFilePath = filePath.replaceAll("Resource/", "");
        String parentRealPath = metaResolver.getPath("");
        String fileRealPath = metaResolver.getPath(newFilePath);
        if (!checkRealPath(parentRealPath, fileRealPath)) {//如果文件不在根目录下，则不予读取。
            return null;
        }
        return metaResolver.read(newFilePath, -1);
    }
}
