package com.bokesoft.yigo.meta.rights;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.Objects;
import java.util.Set;

import com.bokesoft.yes.common.util.Callback;
import com.bokesoft.yigo.meta.base.AbstractMetaObject;
import com.bokesoft.yigo.meta.base.IElementMerge;
import com.bokesoft.yigo.meta.base.IMetaEnv;
import com.bokesoft.yigo.meta.base.MetaException;
/**
 * 权限定义
 * @author 陈瑞
 *
 */
public class MetaRightsDefinition extends AbstractMetaObject implements IElementMerge<MetaRightsDefinition>{
	
	private MetaFormRightsCollection formRights = null;
	
	private MetaDictRightsCollection dictRights = null;
	
	private Boolean allRights = false;
	
	/**
	 * {@link #allRights}属性值和本solution不同的solution的标示集合
	 */
	private Set<String> differentSolutionKeys;
	/**
	 * {@link #allRights}属性值和本solution不同的表单权限集合
	 */
	private MetaFormRightsCollection differentFormRights = null;
	/**
	 * {@link #allRights}属性值和本solution不同的字典权限集合
	 */
	private MetaDictRightsCollection differentDictRights = null;
	/**
	 * 通过{@link #addFormRights(String, boolean)}接口添加权限，且allRights参数值和本solution同名属性值不同的表单的标示集合
	 */
	private Set<String> differentFormKeys;
	/**
	 * 通过{@link #addDictRights(String, boolean)}接口添加权限，且allRights参数值和本solution同名属性值不同的表单的标示集合
	 */
	private Set<String> differentItemKeys;
	
	public static final String TAG_NAME = "RightsDefinition";

	@Override
	public void getChildMetaObjects(LinkedList<Object> list) {
		if( formRights != null ) {
			list.add(formRights);
		}
		if( dictRights != null ) {
			list.add(dictRights);
		}
	}

	@Override
	public String getTagName() {
		return TAG_NAME;
	}

	@Override
	public AbstractMetaObject createChildMetaObject(IMetaEnv env, Object doc,
			Object node, String tagName, Object flag, int runType)
			throws MetaException {
		AbstractMetaObject resultMetaObject = null;
		if( MetaFormRightsCollection.TAG_NAME.equals(tagName) ) {
			formRights = new MetaFormRightsCollection();
			resultMetaObject = formRights;
		} else if( MetaDictRightsCollection.TAG_NAME.equals(tagName) ) {
			dictRights = new MetaDictRightsCollection();
			resultMetaObject = dictRights;
		}
		return resultMetaObject;
	}

	/**
	 * 获取设置的表单或者字典是否全部需要设置权限
	 * @return 标志
	 */
	public Boolean isAllRights() {
		return allRights;
	}

	/**
	 * 设置是否定义的表单或者字典是否全部需要设置权限
	 * @param allRights 标志
	 */
	public void setAllRights(Boolean allRights) {
		this.allRights = allRights;
	}

	/**
	 * 获取表单权限集合
	 * @return
	 */
	public MetaFormRightsCollection getFormRightsCollection() {
		return formRights;
	}

	/**
	 * 设置表单权限集合
	 * @param formRightsCollection
	 */
	public void setFormRightsCollection(
			MetaFormRightsCollection formRightsCollection) {
		this.formRights = formRightsCollection;
	}

	/**
	 * 获取字典权限集合
	 * @return
	 */
	public MetaDictRightsCollection getDictRightsCollection() {
		return dictRights;
	}

	/**
	 * 设置字典权限集合
	 * @param dictRightsCollection
	 */
	public void setDictRightsCollection(
			MetaDictRightsCollection dictRightsCollection) {
		this.dictRights = dictRightsCollection;
	}

	@Override
	public AbstractMetaObject clone() {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public AbstractMetaObject newInstance() {
		return new MetaRightsDefinition();
	}
	
	/**
	 * 是否有该表单所有权限
	 * @param formKey 表单标识
	 * @return 如果有formKey所指表单有所有权限则返回true，否则返回false
	 */
	public boolean hasAllFormRights(String formKey) {
		return hasAllFormRights(formKey, null);
	}
	
	/**
	 * 是否有该表单所有权限
	 * @param formKey 表单标识
	 * @param solutionKey 表单所在solution的标示
	 * @return 如果有formKey所指表单有所有权限则返回true，否则返回false
	 */
	public boolean hasAllFormRights(String formKey, String solutionKey) {
		if ((solutionKey != null && differentSolutionKeys != null && differentSolutionKeys.contains(solutionKey)) ||
				(differentFormKeys != null && differentFormKeys.contains(formKey))) {
			return innerHasAllFormRights(formKey, !allRights, differentFormRights);
		}
		return innerHasAllFormRights(formKey, allRights, formRights);
	}
	
	/**
	 * 是否有该表单所有权限
	 * @param formKey 表单标识
	 * @param allRigths 是否为白名单模式
	 * @param formRights 表单权限集合
	 * @return 如果有formKey所指表单有所有权限则返回true，否则返回false
	 */
	private boolean innerHasAllFormRights(String formKey, boolean allRights, MetaFormRightsCollection formRights) {
		if( allRights ) {
			return formRights == null ? false : formRights.contains(formKey);
		} else {
			return formRights == null ? true : !formRights.contains(formKey);
		}
	}
	
	/**
	 * 是否有该字典所有权限
	 * @param itemKey 字典标识
	 * @return 如果itemKey所指字典有所有权限则返回true，否则返回false
	 */
	public boolean hasAllDictRights(String itemKey) {
		return hasAllDictRights(itemKey, null);
	}
	
	/**
	 * 是否有该字典所有权限
	 * @param itemKey 字典标识
	 * @param solutionKey 字典所在solution的标示
	 * @return 如果itemKey所指字典有所有权限则返回true，否则返回false
	 */
	public boolean hasAllDictRights(String itemKey, String solutionKey) {
		if ((solutionKey != null && differentSolutionKeys != null && differentSolutionKeys.contains(solutionKey)) ||
				(differentItemKeys != null && differentItemKeys.contains(itemKey))) {
			return innerHasAllDictRights(itemKey, !allRights, differentDictRights);
		}
		return innerHasAllDictRights(itemKey, allRights, dictRights);
	}
	
	/**
	 * 是否有该字典所有权限
	 * @param itemKey 字典标识
	 * @param allRigths 是否为白名单模式
	 * @param dictRights 字典权限集合
	 * @return 如果itemKey所指字典有所有权限则返回true，否则返回false
	 */
	private boolean innerHasAllDictRights(String itemKey, boolean allRights, MetaDictRightsCollection dictRights) {
		if( allRights ) {
			return dictRights == null ? false : dictRights.contains(itemKey);
		} else {
			return dictRights == null ? true : !dictRights.contains(itemKey);
		}
	}

	@Override
	public void doPostProcess(int flag, Callback<AbstractMetaObject,Boolean> callback) throws MetaException {
		if( formRights != null ) {
			formRights.doPostProcess(flag, callback);
		}
		if( dictRights != null ) {
			dictRights.doPostProcess(flag, callback);
		}
		if (differentFormRights != null) {
			differentFormRights.doPostProcess(flag, callback);
		}
		if (differentDictRights != null) {
			differentDictRights.doPostProcess(flag, callback);
		}
	}

	@Override
	public MetaRightsDefinition toMerge(MetaRightsDefinition obj) {
		return toMerge(obj, null);
	}

	/**
	 * 合并同类型对象中的元素：一般适用于集合类型或成员为集合类型
	 * @param obj 同类型对象
	 * @param solutionKey obj所属的solution的标示
	 * @return 合并后对象
	 */
	public MetaRightsDefinition toMerge(MetaRightsDefinition obj, String solutionKey) {
		if (obj != null) {
			if (allRights != obj.allRights) {
				if (differentSolutionKeys == null) {
					differentSolutionKeys = new HashSet<>();
				}
				differentSolutionKeys.add(solutionKey);

				if (differentFormRights == null) {
					differentFormRights = new MetaFormRightsCollection();
				}
				differentFormRights.toMerge(obj.formRights);
				
				if (differentDictRights == null) {
					differentDictRights = new MetaDictRightsCollection();
				}
				differentDictRights.toMerge(obj.dictRights);
			} else {
				if(formRights == null) {
					formRights = new MetaFormRightsCollection();
				}
				formRights.toMerge(obj.formRights);

				if (dictRights == null) {
					dictRights = new MetaDictRightsCollection();
				}
				dictRights.toMerge(obj.dictRights);
			}
		}
		return this;
	}
	
	/**
	 * 新增表单权限
	 * @param formKey 表单标识
	 */
	public void addFormRights(String formKey) {
		addFormRights(formKey, allRights);
	}

	/**
	 * 新增表单权限
	 * @param formKey 表单标识
	 * @param allRights 是否是添加到白名单
	 */
	public void addFormRights(String formKey, boolean allRights) {
		if (allRights == this.allRights) {
			if (formRights == null) {
				formRights = new MetaFormRightsCollection();
			}
			innerAddFormRights(formKey, formRights);
		} else {
			if (differentFormKeys == null) {
				differentFormKeys = new HashSet<>();
			}
			differentFormKeys.add(formKey);
			if (differentFormRights == null) {
				differentFormRights = new MetaFormRightsCollection();
			}
			innerAddFormRights(formKey, differentFormRights);
		}
	}
	
	/**
	 * 新增表单权限
	 * @param formKey 表单标识
	 * @param formRights 表单权限集合
	 */
	private void innerAddFormRights(String formKey, MetaFormRightsCollection formRights) {
		if (formRights.contains(formKey)) {
			return;
		}
		MetaFormDefinition formDefinition = new MetaFormDefinition();
		formDefinition.setFormKey(formKey);
		formRights.add(formDefinition);
	}
	
	/**
	 * 移除表单权限
	 * @param formKey 表单标识
	 */
	public void removeFormRights(String formKey) {
		if (differentFormKeys != null && differentFormKeys.contains(formKey)) {
			innerRemoveFormRights(formKey, differentFormRights);
			differentFormKeys.remove(formKey);
		} else {
			if (!innerRemoveFormRights(formKey, formRights)) {
				innerRemoveFormRights(formKey, differentFormRights);
			}
		}
	}
	
	/**
	 * 移除表单权限
	 * @param formKey 表单标识
	 * @param formRights 表单权限集合
	 * @return 是否移除
	 */
	private boolean innerRemoveFormRights(String formKey, MetaFormRightsCollection formRights) {
		if(formRights == null) {
			return false;
		}
		for (int i = 0, size = formRights.size(); i < size; i++) {
			MetaFormDefinition formDefinition = formRights.get(i);
			if(Objects.equals(formDefinition.getFormKey(), formKey)) {
				formRights.remove(i);
				return true;
			}
		}
		return false;
	}
	
	/**
	 * 新增字典权限
	 * @param itemKey 字典标识
	 */
	public void addDictRights(String itemKey) {
		addDictRights(itemKey, allRights);
	}

	/**
	 * 新增字典权限
	 * @param itemKey 字典标识
	 * @param allRights 是否是添加到白名单
	 */
	public void addDictRights(String itemKey, boolean allRights) {
		if (allRights == this.allRights) {
			if (dictRights == null) {
				dictRights = new MetaDictRightsCollection();
			}
			innerAddDictRights(itemKey, dictRights);
		} else {
			if (differentItemKeys == null) {
				differentItemKeys = new HashSet<>();
			}
			differentItemKeys.add(itemKey);
			if (differentDictRights == null) {
				differentDictRights = new MetaDictRightsCollection();
			}
			innerAddDictRights(itemKey, differentDictRights);
		}
	}
	
	/**
	 * 新增字典权限
	 * @param itemKey 字典标识
	 * @param dictRights 字典权限集合
	 */
	private void innerAddDictRights(String itemKey, MetaDictRightsCollection dictRights) {
		if (dictRights.contains(itemKey)) {
			return;
		}
		MetaDictDefinition dictDefinition = new MetaDictDefinition();
		dictDefinition.setItemKey(itemKey);
		dictRights.add(dictDefinition);
	}
	
	/**
	 * 移除字典权限
	 * @param itemKey 字典标识
	 */
	public void removeDictRights(String itemKey) {
		if (differentItemKeys != null && differentItemKeys.contains(itemKey)) {
			innerRemoveDictRights(itemKey, differentDictRights);
			differentItemKeys.remove(itemKey);
		} else {
			if (!innerRemoveDictRights(itemKey, dictRights)) {
				innerRemoveDictRights(itemKey, differentDictRights);
			}
		}
	}
	
	/**
	 * 移除字典权限
	 * @param itemKey 字典标识
	 * @param dictRights 字典权限集合
	 * @return 是否移除
	 */
	private boolean innerRemoveDictRights(String itemKey, MetaDictRightsCollection dictRights) {
		if(dictRights == null) {
			return false;
		}
		for (int i = 0, size = dictRights.size(); i < size; i++) {
			MetaDictDefinition dictDefinition = dictRights.get(i);
			if(Objects.equals(dictDefinition.getItemKey(), itemKey)) {
				dictRights.remove(i);
				return true;
			}
		}
		return false;
	}
}
